/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    https://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.grails.plugins.databasemigration.command

import grails.dev.commands.ApplicationCommand
import groovy.sql.Sql
import org.grails.plugins.databasemigration.DatabaseMigrationException
import org.h2.Driver
import org.springframework.jdbc.datasource.DriverManagerDataSource
import spock.lang.AutoCleanup

import java.sql.Connection

class DbmDiffCommandSpec extends ApplicationContextDatabaseMigrationCommandSpec {

    @Override
    protected Class<ApplicationCommand> getCommandClass() {
        return DbmDiffCommand
    }

    @AutoCleanup
    Connection otherDbConnection

    @AutoCleanup
    Sql otherDbSql

    def setup() {
        def otherDbDataSource = new DriverManagerDataSource('jdbc:h2:mem:otherDb', 'sa', '')
        otherDbDataSource.driverClassName = Driver.name
        otherDbConnection = otherDbDataSource.connection
        otherDbSql = new Sql(otherDbConnection)
        otherDbSql.executeUpdate '''
CREATE TABLE book (id INT GENERATED BY DEFAULT AS IDENTITY NOT NULL, title VARCHAR(255) NOT NULL, CONSTRAINT PK_BOOK PRIMARY KEY (id))
'''
        sql.executeUpdate '''
CREATE TABLE book (id INT GENERATED BY DEFAULT AS IDENTITY NOT NULL, title VARCHAR(255) NOT NULL, price INT NOT NULL, CONSTRAINT PK_BOOK PRIMARY KEY (id));
CREATE TABLE author (id INT GENERATED BY DEFAULT AS IDENTITY NOT NULL, name VARCHAR(255) NOT NULL, CONSTRAINT PK_AUTHOR PRIMARY KEY (id));
'''
    }

    def "writes Change Log to update the database to STDOUT"() {
        when:
            command.handle(getExecutionContext('other'))

        then:
        String expected = extractOutput(output)
        expected =~ '''
databaseChangeLog = \\{

    changeSet\\(author: ".+?", id: ".+?"\\) \\{
        createTable\\(tableName: "AUTHOR"\\) \\{
            column\\(autoIncrement: "true", name: "ID", type: "INT"\\) \\{
                constraints\\(nullable:"false", primaryKey: "true", primaryKeyName: "PK_AUTHOR"\\)
            \\}

            column\\(name: "NAME", type: "VARCHAR\\(255\\)"\\) \\{
                constraints\\(nullable: "false"\\)
            \\}
        \\}
    \\}

    changeSet\\(author: ".+?", id: ".+?"\\) \\{
        addColumn\\(tableName: "BOOK"\\) \\{
            column\\(name: "PRICE", type: "INTEGER"\\) \\{
                constraints\\(nullable: "false"\\)
            \\}
        \\}
    \\}
\\}
'''.replaceAll(/\s/,"")
    }

    def "writes Change Log to update the database to a file given as arguments"() {
        given:
            def outputChangeLog = new File(changeLogLocation, 'diff.groovy')

        when:
            command.handle(getExecutionContext('other', outputChangeLog.name))

        then:
            outputChangeLog.text?.replaceAll(/\s/,"") =~ '''
databaseChangeLog = \\{

    changeSet\\(author: ".+?", id: ".+?"\\) \\{
        createTable\\(tableName: "AUTHOR"\\) \\{
            column\\(autoIncrement: "true", name: "ID", type: "INT"\\) \\{
                constraints\\(nullable:"false", primaryKey: "true", primaryKeyName: "PK_AUTHOR"\\)
            \\}

            column\\(name: "NAME", type: "VARCHAR\\(255\\)"\\) \\{
                constraints\\(nullable: "false"\\)
            \\}
        \\}
    \\}

    changeSet\\(author: ".+?", id: ".+?"\\) \\{
        addColumn\\(tableName: "BOOK"\\) \\{
            column\\(name: "PRICE", type: "INTEGER"\\) \\{
                constraints\\(nullable: "false"\\)
            \\}
        \\}
    \\}
\\}
'''.replaceAll(/\s/,"")
    }

    def "an error occurs if the otherEnv parameter is not specified"() {
        when:
            command.handle(getExecutionContext())

        then:
            def e = thrown(DatabaseMigrationException)
            e.message == 'You must specify the environment to diff against'
    }

    def "an error occurs if other environment and current environment is same"() {
        when:
            command.handle(getExecutionContext('test'))

        then:
            def e = thrown(DatabaseMigrationException)
            e.message == 'You must specify a different environment than the one the command is running in'
    }
}
