/*******************************************************************************
 *  PROJECT: PaletteParser
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2006 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <cppunit/ui/text/TestRunner.h>
#include <cppunit/extensions/HelperMacros.h>
#include <paletteparser/palette.h>
#include <iostream>
#include <sstream>
#include <fstream>

using namespace std;

namespace gcs
{
    bool colorscheme_debug = false;
}

class ParseSuite : public CppUnit::TestFixture
{
    CPPUNIT_TEST_SUITE(ParseSuite);
    CPPUNIT_TEST(testValidPalette);
    CPPUNIT_TEST(testParseColorValid);
    CPPUNIT_TEST(testParseColorInvalid);
    //CPPUNIT_TEST(testWithoutName);
    //CPPUNIT_TEST(testWithoutColumns);
    //CPPUNIT_TEST(testIgnoreComments);
    //CPPUNIT_TEST(testIgnoreEmptyLines);
    //CPPUNIT_TEST(testInvalidLine);
    CPPUNIT_TEST(testInvalidColor);
    //CPPUNIT_TEST(testVeryLongLine);
    //CPPUNIT_TEST(testPaletteOutOfOrder);
    //CPPUNIT_TEST(testPaletteOutOfOrder);
    CPPUNIT_TEST(testParseFile);
    // + test setting the values programmatically...
    CPPUNIT_TEST_SUITE_END();

    protected:
    void testValidPalette(void);
    void testParseColorValid(void);
    void testParseColorInvalid(void);
    void testWithoutName(void);
    void testWithoutColumns(void);
    void testIgnoreComments(void);
    void testIgnoreEmptyLines(void);
    void testInvalidLine(void);
    void testInvalidColor(void);
    void testVeryLongLine(void);
    void testPaletteOutOfOrder(void);
    void testParseFile(void);
};

void ParseSuite::testValidPalette(void)
{
    std::istringstream example_palette(
            "GIMP Palette\n"
            "Name: Bears\n"
            "#\n"
            "  8   8   8	grey3\n"
            " 68  44  44	Untitled\n"
            " 80   8  12	Untitled\n"
            " 72  56  56	Untitled\n"
            "104  84  68	Untitled\n"
            "116  96  80	Untitled\n"
            " 84  56  44	Untitled\n"
            "140 104  88	Untitled\n"
            "172 116  92	Untitled\n"
            " 68  44  56	Untitled\n"
            " 88  72  68	Untitled\n"
            "112  84  76	Untitled\n"
            "  8   8  28	Untitled");

    pp::Palette p;
    //example_palette >> p;
    p.parse(example_palette);
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Bears"), p.name());
    CPPUNIT_ASSERT_EQUAL((size_t) 0, p.columns());
    CPPUNIT_ASSERT_EQUAL((size_t) 13, p.colors().size());

    // test a few random color values
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("grey3"), p.colors()[0]->get_name());

    CPPUNIT_ASSERT_EQUAL((gint) 72, p.colors()[3]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 56, p.colors()[3]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 56, p.colors()[3]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), p.colors()[3]->get_name());

    CPPUNIT_ASSERT_EQUAL((gint) 172, p.colors()[8]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 116, p.colors()[8]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 92, p.colors()[8]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), p.colors()[8]->get_name());
}


void ParseSuite::testInvalidColor(void)
{
    std::istringstream example_palette(
            "GIMP Palette\n"
            "Name: Bears\n"
            "Columns: 8\n"
            "#\n"
            "  8   8   8	grey3\n"
            " 68  44  44	Untitled\n"
            " 80   8  12	Untitled\n"
            " 72  56  56	Untitled\n"
            "104  84  68	Untitled color\n"
            "354  96  80	Untitled\n"
            " 84  56  44	Untitled\n"
            "140 104  88	Untitled\n"
            "172 116  92	Untitled\n"
            " 68  44  56	Untitled\n"
            " 88  72  68	Untitled\n"
            "112  84  76	Untitled\n"
            "  8   8  28	Untitled");
    pp::Palette p;
    //example_palette2 >> p2;
    p.parse(example_palette);
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Bears"), p.name());
    CPPUNIT_ASSERT_EQUAL((size_t) 8, p.columns()); // this time with columns
    CPPUNIT_ASSERT_EQUAL((size_t) 12, p.colors().size());

    // test a few random color values
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 8, p.colors()[0]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("grey3"), p.colors()[0]->get_name());

    CPPUNIT_ASSERT_EQUAL((gint) 104, p.colors()[4]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 84, p.colors()[4]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 68, p.colors()[4]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled color"), p.colors()[4]->get_name());

    // it's down an extra line since we should skip the bad color between
    // there...
    CPPUNIT_ASSERT_EQUAL((gint) 68, p.colors()[8]->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 44, p.colors()[8]->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 56, p.colors()[8]->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), p.colors()[8]->get_name());

}

void ParseSuite::testParseFile(void)
{
    const char* filename = "/usr/local/share/agave/palettes/Tango-Palette.gpl";
    std::ifstream infile(filename);
    //cout << endl << "Now let's parse a system palette.  I know! " << filename << endl;
    pp::Palette p3;
    p3.parse(infile);
    //cout << p3;
    //return 0;
}


void ParseSuite::testParseColorValid(void)
{
#define LONG_DESC "this one has a really long name.  So long that it might not even fit into the buffer that is allocated for the name field while parsing it.  It would fit into the rgb_t field, but not while parsing this one has a really long name.  So long that it might not even fit into the buffer that is allocated for the name field while parsing it.  It would fit into the rgb_t field, but not while parsing. Maybe I just make the parsing buffer large enough so that I never have to worry about it or something...."

    const char* s1 = "  8   8   8	grey3\n";
    const char* s2 = " 68  44  44	Untitled\n";
    const char* s3 = " 80   8  12	Untitled\n";
    const char* s4 = " 72  56  56	Untitled\n";
    const char* s5 = "104  84  68	Untitled\n";
    const char* s6 = "116  96  80	Untitled\n";
    const char* s7 = " 84  56  44	Untitled\n";
    const char* s8 = "140 104  88	" LONG_DESC "\n";
    const char* s9 = "172 116  92          multiplespaceseparator\n";
    const char* s10 = " 68  44  56 Untitled\n";
    const char* s11 = " 88  72  68	descriptionwithspaceatend \n";
    const char* s12 = "112  84  76	multi-word description\n";
    const char* s13 = "  8   8  28	Untitled";
    const char* s14 = "  0   0   0	black"; // at limits
    const char* s15 = "255 255 255	white"; // hi limit

    pp::Palette p;
    pp::Palette::value_type rgb;
    rgb = p.parse_color(s1);
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("grey3"), rgb->get_name());

    rgb = p.parse_color(s2);
    CPPUNIT_ASSERT_EQUAL((gint) 68, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 44, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 44, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s3);
    CPPUNIT_ASSERT_EQUAL((gint) 80, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 12, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s4);
    CPPUNIT_ASSERT_EQUAL((gint) 72, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 56, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 56, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s5);
    CPPUNIT_ASSERT_EQUAL((gint) 104, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 84, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 68, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s6);
    CPPUNIT_ASSERT_EQUAL((gint) 116, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 96, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 80, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s7);
    CPPUNIT_ASSERT_EQUAL((gint) 84, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 56, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 44, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s8);
    CPPUNIT_ASSERT_EQUAL((gint) 140, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 104, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 88, rgb->get_blue());
    // just make sure the first X characters are the same
    CPPUNIT_ASSERT_EQUAL(Glib::ustring(LONG_DESC).substr(0, 100), rgb->get_name().substr(0, 100));

    rgb = p.parse_color(s9);
    CPPUNIT_ASSERT_EQUAL((gint) 172, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 116, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 92, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("multiplespaceseparator"), rgb->get_name());

    rgb = p.parse_color(s10);
    CPPUNIT_ASSERT_EQUAL((gint) 68, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 44, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 56, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s11);
    CPPUNIT_ASSERT_EQUAL((gint) 88, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 72, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 68, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("descriptionwithspaceatend"), rgb->get_name());

    rgb = p.parse_color(s12);
    CPPUNIT_ASSERT_EQUAL((gint) 112, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 84, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 76, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("multi-word description"), rgb->get_name());

    rgb = p.parse_color(s13);
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 8, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 28, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("Untitled"), rgb->get_name());

    rgb = p.parse_color(s14);
    CPPUNIT_ASSERT_EQUAL((gint) 0, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 0, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 0, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("black"), rgb->get_name());

    rgb = p.parse_color(s15);
    CPPUNIT_ASSERT_EQUAL((gint) 255, rgb->get_red());
    CPPUNIT_ASSERT_EQUAL((gint) 255, rgb->get_green());
    CPPUNIT_ASSERT_EQUAL((gint) 255, rgb->get_blue());
    CPPUNIT_ASSERT_EQUAL(Glib::ustring("white"), rgb->get_name());

}


void ParseSuite::testParseColorInvalid(void)
{
    const char* s1 = " -8   8   8	grey3\n";    // red negative
    const char* s2 = "  8  -8   8	grey3\n";    // green negative
    const char* s3 = "  8   8  -8	grey3\n";    // blue negative
    const char* s4 = " -8  -8  -8	grey3\n";    // all negative
    const char* s5 = "368  44  44	Untitled\n"; // red too large
    const char* s6 = " 80   8 512	Untitled\n"; // blue too large
    const char* s7 = " 72 856  56	Untitled\n"; // green too large
    const char* s8 = "404 584 668	Untitled\n"; // all too large
    const char* s9 = "116 96 80 Untitled\n"; // odd spacing -- should this fail?
    const char* s10 = " 84  56  44	Untitled";   // missing newline at end -- should this fail?
    const char* s11 = "  a  56  44	Untitled\n";   // red is not a number
    const char* s12 = " 84   b  44	Untitled\n";   // green is not a number
    const char* s13 = " 84  56   c	Untitled\n";   // blue is not a number
    const char* s14 = "  a   b   c	Untitled\n";   // all not numbers
    const char* s15 = "  &   ?   =	Untitled\n";   // other weird characters
    //"140 104  88	Untitled\n";
    //"172 116  92	Untitled\n";
    //" 68  44  56	Untitled\n";
    //" 88  72  68	Untitled\n";
    //"112  84  76	Untitled\n";
    //"  8   8  28	Untitled";

    pp::Palette p;
    CPPUNIT_ASSERT_THROW(p.parse_color(s1), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s2), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s3), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s4), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s5), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s6), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s7), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s8), pp::ParseError);
    //CPPUNIT_ASSERT_THROW(p.parse_color(s9), pp::ParseError);
    //CPPUNIT_ASSERT_THROW(p.parse_color(s10), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s11), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s12), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s13), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s14), pp::ParseError);
    CPPUNIT_ASSERT_THROW(p.parse_color(s15), pp::ParseError);
}


int main()
{
    CppUnit::TextUi::TestRunner runner;
    runner.addTest(ParseSuite::suite());
    return !runner.run();
}
