/* ----------------------------------------------------------------------   
* Copyright (C) 2010 ARM Limited. All rights reserved.   
*   
* $Date:        15. July 2011  
* $Revision: 	V1.0.10  
*   
* Project: 	    CMSIS DSP Library   
* Title:	    arm_cfft_radix4_init_q15.c   
*   
* Description:	Radix-4 Decimation in Frequency Q15 FFT & IFFT initialization function   
*   
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
* Version 1.0.10 2011/7/15 
*    Big Endian support added and Merged M0 and M3/M4 Source code.  
*   
* Version 1.0.3 2010/11/29  
*    Re-organized the CMSIS folders and updated documentation.   
*    
* Version 1.0.2 2010/11/11   
*    Documentation updated.    
*   
* Version 1.0.1 2010/10/05    
*    Production release and review comments incorporated.   
*   
* Version 1.0.0 2010/09/20    
*    Production release and review comments incorporated.   
*   
* Version 0.0.5  2010/04/26    
* 	 incorporated review comments and updated with latest CMSIS layer   
*   
* Version 0.0.3  2010/03/10    
*    Initial version   
* -------------------------------------------------------------------- */

#include "arm_math.h"
#include "arm_common_tables.h"

/**   
 * @ingroup groupTransforms   
 */


/**   
 * @addtogroup CFFT_CIFFT   
 * @{   
 */

/*   
* @brief  Twiddle factors Table   
*/

/**  
* \par   
* Example code for Q15 Twiddle factors Generation::   
* \par   
* <pre>for(i = 0; i< N; i++)   
* {   
*	twiddleCoefQ15[2*i]= cos(i * 2*PI/(float)N);   
*	twiddleCoefQ15[2*i+1]= sin(i * 2*PI/(float)N);   
* } </pre>   
* \par   
* where N = 1024	and PI = 3.14159265358979   
* \par   
* Cos and Sin values are interleaved fashion   
* \par   
* Convert Floating point to Q15(Fixed point 1.15):   
*	round(twiddleCoefQ15(i) * pow(2, 15))   
*   
*/

static const q15_t twiddleCoefQ15[4096*2] = {

0x7fff,	0x0,	0x7fff,	0x32,	0x7fff,	0x65,	0x7fff,	0x97,
	0x7fff,	0xc9,	0x7fff,	0xfb,	0x7fff,	0x12e,	0x7ffe,	0x160,
	0x7ffe,	0x192,	0x7ffd,	0x1c4,	0x7ffc,	0x1f7,	0x7ffb,	0x229,
	0x7ffa,	0x25b,	0x7ff9,	0x28d,	0x7ff8,	0x2c0,	0x7ff7,	0x2f2,
	0x7ff6,	0x324,	0x7ff5,	0x356,	0x7ff4,	0x389,	0x7ff2,	0x3bb,
	0x7ff1,	0x3ed,	0x7fef,	0x41f,	0x7fed,	0x452,	0x7fec,	0x484,
	0x7fea,	0x4b6,	0x7fe8,	0x4e8,	0x7fe6,	0x51b,	0x7fe4,	0x54d,
	0x7fe2,	0x57f,	0x7fe0,	0x5b1,	0x7fdd,	0x5e3,	0x7fdb,	0x616,
	0x7fd9,	0x648,	0x7fd6,	0x67a,	0x7fd3,	0x6ac,	0x7fd1,	0x6de,
	0x7fce,	0x711,	0x7fcb,	0x743,	0x7fc8,	0x775,	0x7fc5,	0x7a7,
	0x7fc2,	0x7d9,	0x7fbf,	0x80c,	0x7fbc,	0x83e,	0x7fb9,	0x870,
	0x7fb5,	0x8a2,	0x7fb2,	0x8d4,	0x7fae,	0x906,	0x7fab,	0x938,
	0x7fa7,	0x96b,	0x7fa3,	0x99d,	0x7fa0,	0x9cf,	0x7f9c,	0xa01,
	0x7f98,	0xa33,	0x7f94,	0xa65,	0x7f90,	0xa97,	0x7f8b,	0xac9,
	0x7f87,	0xafb,	0x7f83,	0xb2d,	0x7f7e,	0xb60,	0x7f7a,	0xb92,
	0x7f75,	0xbc4,	0x7f71,	0xbf6,	0x7f6c,	0xc28,	0x7f67,	0xc5a,
	0x7f62,	0xc8c,	0x7f5d,	0xcbe,	0x7f58,	0xcf0,	0x7f53,	0xd22,
	0x7f4e,	0xd54,	0x7f49,	0xd86,	0x7f43,	0xdb8,	0x7f3e,	0xdea,
	0x7f38,	0xe1c,	0x7f33,	0xe4e,	0x7f2d,	0xe80,	0x7f27,	0xeb2,
	0x7f22,	0xee4,	0x7f1c,	0xf15,	0x7f16,	0xf47,	0x7f10,	0xf79,
	0x7f0a,	0xfab,	0x7f03,	0xfdd,	0x7efd,	0x100f,	0x7ef7,	0x1041,
	0x7ef0,	0x1073,	0x7eea,	0x10a4,	0x7ee3,	0x10d6,	0x7edd,	0x1108,
	0x7ed6,	0x113a,	0x7ecf,	0x116c,	0x7ec8,	0x119e,	0x7ec1,	0x11cf,
	0x7eba,	0x1201,	0x7eb3,	0x1233,	0x7eac,	0x1265,	0x7ea5,	0x1296,
	0x7e9d,	0x12c8,	0x7e96,	0x12fa,	0x7e8e,	0x132b,	0x7e87,	0x135d,
	0x7e7f,	0x138f,	0x7e78,	0x13c1,	0x7e70,	0x13f2,	0x7e68,	0x1424,
	0x7e60,	0x1455,	0x7e58,	0x1487,	0x7e50,	0x14b9,	0x7e48,	0x14ea,
	0x7e3f,	0x151c,	0x7e37,	0x154d,	0x7e2f,	0x157f,	0x7e26,	0x15b1,
	0x7e1e,	0x15e2,	0x7e15,	0x1614,	0x7e0c,	0x1645,	0x7e03,	0x1677,
	0x7dfb,	0x16a8,	0x7df2,	0x16da,	0x7de9,	0x170b,	0x7de0,	0x173c,
	0x7dd6,	0x176e,	0x7dcd,	0x179f,	0x7dc4,	0x17d1,	0x7dba,	0x1802,
	0x7db1,	0x1833,	0x7da7,	0x1865,	0x7d9e,	0x1896,	0x7d94,	0x18c7,
	0x7d8a,	0x18f9,	0x7d81,	0x192a,	0x7d77,	0x195b,	0x7d6d,	0x198d,
	0x7d63,	0x19be,	0x7d58,	0x19ef,	0x7d4e,	0x1a20,	0x7d44,	0x1a51,
	0x7d3a,	0x1a83,	0x7d2f,	0x1ab4,	0x7d25,	0x1ae5,	0x7d1a,	0x1b16,
	0x7d0f,	0x1b47,	0x7d05,	0x1b78,	0x7cfa,	0x1ba9,	0x7cef,	0x1bda,
	0x7ce4,	0x1c0c,	0x7cd9,	0x1c3d,	0x7cce,	0x1c6e,	0x7cc2,	0x1c9f,
	0x7cb7,	0x1cd0,	0x7cac,	0x1d01,	0x7ca0,	0x1d31,	0x7c95,	0x1d62,
	0x7c89,	0x1d93,	0x7c7e,	0x1dc4,	0x7c72,	0x1df5,	0x7c66,	0x1e26,
	0x7c5a,	0x1e57,	0x7c4e,	0x1e88,	0x7c42,	0x1eb8,	0x7c36,	0x1ee9,
	0x7c2a,	0x1f1a,	0x7c1e,	0x1f4b,	0x7c11,	0x1f7b,	0x7c05,	0x1fac,
	0x7bf9,	0x1fdd,	0x7bec,	0x200e,	0x7bdf,	0x203e,	0x7bd3,	0x206f,
	0x7bc6,	0x209f,	0x7bb9,	0x20d0,	0x7bac,	0x2101,	0x7b9f,	0x2131,
	0x7b92,	0x2162,	0x7b85,	0x2192,	0x7b78,	0x21c3,	0x7b6a,	0x21f3,
	0x7b5d,	0x2224,	0x7b50,	0x2254,	0x7b42,	0x2284,	0x7b34,	0x22b5,
	0x7b27,	0x22e5,	0x7b19,	0x2316,	0x7b0b,	0x2346,	0x7afd,	0x2376,
	0x7aef,	0x23a7,	0x7ae1,	0x23d7,	0x7ad3,	0x2407,	0x7ac5,	0x2437,
	0x7ab7,	0x2467,	0x7aa8,	0x2498,	0x7a9a,	0x24c8,	0x7a8c,	0x24f8,
	0x7a7d,	0x2528,	0x7a6e,	0x2558,	0x7a60,	0x2588,	0x7a51,	0x25b8,
	0x7a42,	0x25e8,	0x7a33,	0x2618,	0x7a24,	0x2648,	0x7a15,	0x2678,
	0x7a06,	0x26a8,	0x79f7,	0x26d8,	0x79e7,	0x2708,	0x79d8,	0x2738,
	0x79c9,	0x2768,	0x79b9,	0x2797,	0x79aa,	0x27c7,	0x799a,	0x27f7,
	0x798a,	0x2827,	0x797a,	0x2856,	0x796a,	0x2886,	0x795b,	0x28b6,
	0x794a,	0x28e5,	0x793a,	0x2915,	0x792a,	0x2945,	0x791a,	0x2974,
	0x790a,	0x29a4,	0x78f9,	0x29d3,	0x78e9,	0x2a03,	0x78d8,	0x2a32,
	0x78c8,	0x2a62,	0x78b7,	0x2a91,	0x78a6,	0x2ac1,	0x7895,	0x2af0,
	0x7885,	0x2b1f,	0x7874,	0x2b4f,	0x7863,	0x2b7e,	0x7851,	0x2bad,
	0x7840,	0x2bdc,	0x782f,	0x2c0c,	0x781e,	0x2c3b,	0x780c,	0x2c6a,
	0x77fb,	0x2c99,	0x77e9,	0x2cc8,	0x77d8,	0x2cf7,	0x77c6,	0x2d26,
	0x77b4,	0x2d55,	0x77a2,	0x2d84,	0x7790,	0x2db3,	0x777e,	0x2de2,
	0x776c,	0x2e11,	0x775a,	0x2e40,	0x7748,	0x2e6f,	0x7736,	0x2e9e,
	0x7723,	0x2ecc,	0x7711,	0x2efb,	0x76fe,	0x2f2a,	0x76ec,	0x2f59,
	0x76d9,	0x2f87,	0x76c7,	0x2fb6,	0x76b4,	0x2fe5,	0x76a1,	0x3013,
	0x768e,	0x3042,	0x767b,	0x3070,	0x7668,	0x309f,	0x7655,	0x30cd,
	0x7642,	0x30fc,	0x762e,	0x312a,	0x761b,	0x3159,	0x7608,	0x3187,
	0x75f4,	0x31b5,	0x75e1,	0x31e4,	0x75cd,	0x3212,	0x75b9,	0x3240,
	0x75a6,	0x326e,	0x7592,	0x329d,	0x757e,	0x32cb,	0x756a,	0x32f9,
	0x7556,	0x3327,	0x7542,	0x3355,	0x752d,	0x3383,	0x7519,	0x33b1,
	0x7505,	0x33df,	0x74f0,	0x340d,	0x74dc,	0x343b,	0x74c7,	0x3469,
	0x74b3,	0x3497,	0x749e,	0x34c4,	0x7489,	0x34f2,	0x7475,	0x3520,
	0x7460,	0x354e,	0x744b,	0x357b,	0x7436,	0x35a9,	0x7421,	0x35d7,
	0x740b,	0x3604,	0x73f6,	0x3632,	0x73e1,	0x365f,	0x73cb,	0x368d,
	0x73b6,	0x36ba,	0x73a0,	0x36e8,	0x738b,	0x3715,	0x7375,	0x3742,
	0x735f,	0x3770,	0x734a,	0x379d,	0x7334,	0x37ca,	0x731e,	0x37f7,
	0x7308,	0x3825,	0x72f2,	0x3852,	0x72dc,	0x387f,	0x72c5,	0x38ac,
	0x72af,	0x38d9,	0x7299,	0x3906,	0x7282,	0x3933,	0x726c,	0x3960,
	0x7255,	0x398d,	0x723f,	0x39ba,	0x7228,	0x39e7,	0x7211,	0x3a13,
	0x71fa,	0x3a40,	0x71e3,	0x3a6d,	0x71cc,	0x3a9a,	0x71b5,	0x3ac6,
	0x719e,	0x3af3,	0x7187,	0x3b20,	0x7170,	0x3b4c,	0x7158,	0x3b79,
	0x7141,	0x3ba5,	0x712a,	0x3bd2,	0x7112,	0x3bfe,	0x70fa,	0x3c2a,
	0x70e3,	0x3c57,	0x70cb,	0x3c83,	0x70b3,	0x3caf,	0x709b,	0x3cdc,
	0x7083,	0x3d08,	0x706b,	0x3d34,	0x7053,	0x3d60,	0x703b,	0x3d8c,
	0x7023,	0x3db8,	0x700b,	0x3de4,	0x6ff2,	0x3e10,	0x6fda,	0x3e3c,
	0x6fc2,	0x3e68,	0x6fa9,	0x3e94,	0x6f90,	0x3ec0,	0x6f78,	0x3eec,
	0x6f5f,	0x3f17,	0x6f46,	0x3f43,	0x6f2d,	0x3f6f,	0x6f14,	0x3f9a,
	0x6efb,	0x3fc6,	0x6ee2,	0x3ff1,	0x6ec9,	0x401d,	0x6eb0,	0x4048,
	0x6e97,	0x4074,	0x6e7d,	0x409f,	0x6e64,	0x40cb,	0x6e4a,	0x40f6,
	0x6e31,	0x4121,	0x6e17,	0x414d,	0x6dfe,	0x4178,	0x6de4,	0x41a3,
	0x6dca,	0x41ce,	0x6db0,	0x41f9,	0x6d96,	0x4224,	0x6d7c,	0x424f,
	0x6d62,	0x427a,	0x6d48,	0x42a5,	0x6d2e,	0x42d0,	0x6d14,	0x42fb,
	0x6cf9,	0x4326,	0x6cdf,	0x4351,	0x6cc4,	0x437b,	0x6caa,	0x43a6,
	0x6c8f,	0x43d1,	0x6c75,	0x43fb,	0x6c5a,	0x4426,	0x6c3f,	0x4450,
	0x6c24,	0x447b,	0x6c09,	0x44a5,	0x6bee,	0x44d0,	0x6bd3,	0x44fa,
	0x6bb8,	0x4524,	0x6b9d,	0x454f,	0x6b82,	0x4579,	0x6b66,	0x45a3,
	0x6b4b,	0x45cd,	0x6b30,	0x45f7,	0x6b14,	0x4621,	0x6af8,	0x464b,
	0x6add,	0x4675,	0x6ac1,	0x469f,	0x6aa5,	0x46c9,	0x6a89,	0x46f3,
	0x6a6e,	0x471d,	0x6a52,	0x4747,	0x6a36,	0x4770,	0x6a1a,	0x479a,
	0x69fd,	0x47c4,	0x69e1,	0x47ed,	0x69c5,	0x4817,	0x69a9,	0x4840,
	0x698c,	0x486a,	0x6970,	0x4893,	0x6953,	0x48bd,	0x6937,	0x48e6,
	0x691a,	0x490f,	0x68fd,	0x4939,	0x68e0,	0x4962,	0x68c4,	0x498b,
	0x68a7,	0x49b4,	0x688a,	0x49dd,	0x686d,	0x4a06,	0x6850,	0x4a2f,
	0x6832,	0x4a58,	0x6815,	0x4a81,	0x67f8,	0x4aaa,	0x67da,	0x4ad3,
	0x67bd,	0x4afb,	0x67a0,	0x4b24,	0x6782,	0x4b4d,	0x6764,	0x4b75,
	0x6747,	0x4b9e,	0x6729,	0x4bc7,	0x670b,	0x4bef,	0x66ed,	0x4c17,
	0x66d0,	0x4c40,	0x66b2,	0x4c68,	0x6693,	0x4c91,	0x6675,	0x4cb9,
	0x6657,	0x4ce1,	0x6639,	0x4d09,	0x661b,	0x4d31,	0x65fc,	0x4d59,
	0x65de,	0x4d81,	0x65c0,	0x4da9,	0x65a1,	0x4dd1,	0x6582,	0x4df9,
	0x6564,	0x4e21,	0x6545,	0x4e49,	0x6526,	0x4e71,	0x6507,	0x4e98,
	0x64e9,	0x4ec0,	0x64ca,	0x4ee8,	0x64ab,	0x4f0f,	0x648b,	0x4f37,
	0x646c,	0x4f5e,	0x644d,	0x4f85,	0x642e,	0x4fad,	0x640f,	0x4fd4,
	0x63ef,	0x4ffb,	0x63d0,	0x5023,	0x63b0,	0x504a,	0x6391,	0x5071,
	0x6371,	0x5098,	0x6351,	0x50bf,	0x6332,	0x50e6,	0x6312,	0x510d,
	0x62f2,	0x5134,	0x62d2,	0x515b,	0x62b2,	0x5181,	0x6292,	0x51a8,
	0x6272,	0x51cf,	0x6252,	0x51f5,	0x6232,	0x521c,	0x6211,	0x5243,
	0x61f1,	0x5269,	0x61d1,	0x5290,	0x61b0,	0x52b6,	0x6190,	0x52dc,
	0x616f,	0x5303,	0x614e,	0x5329,	0x612e,	0x534f,	0x610d,	0x5375,
	0x60ec,	0x539b,	0x60cb,	0x53c1,	0x60aa,	0x53e7,	0x6089,	0x540d,
	0x6068,	0x5433,	0x6047,	0x5459,	0x6026,	0x547f,	0x6005,	0x54a4,
	0x5fe4,	0x54ca,	0x5fc2,	0x54f0,	0x5fa1,	0x5515,	0x5f80,	0x553b,
	0x5f5e,	0x5560,	0x5f3c,	0x5586,	0x5f1b,	0x55ab,	0x5ef9,	0x55d0,
	0x5ed7,	0x55f6,	0x5eb6,	0x561b,	0x5e94,	0x5640,	0x5e72,	0x5665,
	0x5e50,	0x568a,	0x5e2e,	0x56af,	0x5e0c,	0x56d4,	0x5dea,	0x56f9,
	0x5dc8,	0x571e,	0x5da5,	0x5743,	0x5d83,	0x5767,	0x5d61,	0x578c,
	0x5d3e,	0x57b1,	0x5d1c,	0x57d5,	0x5cf9,	0x57fa,	0x5cd7,	0x581e,
	0x5cb4,	0x5843,	0x5c91,	0x5867,	0x5c6f,	0x588c,	0x5c4c,	0x58b0,
	0x5c29,	0x58d4,	0x5c06,	0x58f8,	0x5be3,	0x591c,	0x5bc0,	0x5940,
	0x5b9d,	0x5964,	0x5b7a,	0x5988,	0x5b57,	0x59ac,	0x5b34,	0x59d0,
	0x5b10,	0x59f4,	0x5aed,	0x5a18,	0x5ac9,	0x5a3b,	0x5aa6,	0x5a5f,
	0x5a82,	0x5a82,	0x5a5f,	0x5aa6,	0x5a3b,	0x5ac9,	0x5a18,	0x5aed,
	0x59f4,	0x5b10,	0x59d0,	0x5b34,	0x59ac,	0x5b57,	0x5988,	0x5b7a,
	0x5964,	0x5b9d,	0x5940,	0x5bc0,	0x591c,	0x5be3,	0x58f8,	0x5c06,
	0x58d4,	0x5c29,	0x58b0,	0x5c4c,	0x588c,	0x5c6f,	0x5867,	0x5c91,
	0x5843,	0x5cb4,	0x581e,	0x5cd7,	0x57fa,	0x5cf9,	0x57d5,	0x5d1c,
	0x57b1,	0x5d3e,	0x578c,	0x5d61,	0x5767,	0x5d83,	0x5743,	0x5da5,
	0x571e,	0x5dc8,	0x56f9,	0x5dea,	0x56d4,	0x5e0c,	0x56af,	0x5e2e,
	0x568a,	0x5e50,	0x5665,	0x5e72,	0x5640,	0x5e94,	0x561b,	0x5eb6,
	0x55f6,	0x5ed7,	0x55d0,	0x5ef9,	0x55ab,	0x5f1b,	0x5586,	0x5f3c,
	0x5560,	0x5f5e,	0x553b,	0x5f80,	0x5515,	0x5fa1,	0x54f0,	0x5fc2,
	0x54ca,	0x5fe4,	0x54a4,	0x6005,	0x547f,	0x6026,	0x5459,	0x6047,
	0x5433,	0x6068,	0x540d,	0x6089,	0x53e7,	0x60aa,	0x53c1,	0x60cb,
	0x539b,	0x60ec,	0x5375,	0x610d,	0x534f,	0x612e,	0x5329,	0x614e,
	0x5303,	0x616f,	0x52dc,	0x6190,	0x52b6,	0x61b0,	0x5290,	0x61d1,
	0x5269,	0x61f1,	0x5243,	0x6211,	0x521c,	0x6232,	0x51f5,	0x6252,
	0x51cf,	0x6272,	0x51a8,	0x6292,	0x5181,	0x62b2,	0x515b,	0x62d2,
	0x5134,	0x62f2,	0x510d,	0x6312,	0x50e6,	0x6332,	0x50bf,	0x6351,
	0x5098,	0x6371,	0x5071,	0x6391,	0x504a,	0x63b0,	0x5023,	0x63d0,
	0x4ffb,	0x63ef,	0x4fd4,	0x640f,	0x4fad,	0x642e,	0x4f85,	0x644d,
	0x4f5e,	0x646c,	0x4f37,	0x648b,	0x4f0f,	0x64ab,	0x4ee8,	0x64ca,
	0x4ec0,	0x64e9,	0x4e98,	0x6507,	0x4e71,	0x6526,	0x4e49,	0x6545,
	0x4e21,	0x6564,	0x4df9,	0x6582,	0x4dd1,	0x65a1,	0x4da9,	0x65c0,
	0x4d81,	0x65de,	0x4d59,	0x65fc,	0x4d31,	0x661b,	0x4d09,	0x6639,
	0x4ce1,	0x6657,	0x4cb9,	0x6675,	0x4c91,	0x6693,	0x4c68,	0x66b2,
	0x4c40,	0x66d0,	0x4c17,	0x66ed,	0x4bef,	0x670b,	0x4bc7,	0x6729,
	0x4b9e,	0x6747,	0x4b75,	0x6764,	0x4b4d,	0x6782,	0x4b24,	0x67a0,
	0x4afb,	0x67bd,	0x4ad3,	0x67da,	0x4aaa,	0x67f8,	0x4a81,	0x6815,
	0x4a58,	0x6832,	0x4a2f,	0x6850,	0x4a06,	0x686d,	0x49dd,	0x688a,
	0x49b4,	0x68a7,	0x498b,	0x68c4,	0x4962,	0x68e0,	0x4939,	0x68fd,
	0x490f,	0x691a,	0x48e6,	0x6937,	0x48bd,	0x6953,	0x4893,	0x6970,
	0x486a,	0x698c,	0x4840,	0x69a9,	0x4817,	0x69c5,	0x47ed,	0x69e1,
	0x47c4,	0x69fd,	0x479a,	0x6a1a,	0x4770,	0x6a36,	0x4747,	0x6a52,
	0x471d,	0x6a6e,	0x46f3,	0x6a89,	0x46c9,	0x6aa5,	0x469f,	0x6ac1,
	0x4675,	0x6add,	0x464b,	0x6af8,	0x4621,	0x6b14,	0x45f7,	0x6b30,
	0x45cd,	0x6b4b,	0x45a3,	0x6b66,	0x4579,	0x6b82,	0x454f,	0x6b9d,
	0x4524,	0x6bb8,	0x44fa,	0x6bd3,	0x44d0,	0x6bee,	0x44a5,	0x6c09,
	0x447b,	0x6c24,	0x4450,	0x6c3f,	0x4426,	0x6c5a,	0x43fb,	0x6c75,
	0x43d1,	0x6c8f,	0x43a6,	0x6caa,	0x437b,	0x6cc4,	0x4351,	0x6cdf,
	0x4326,	0x6cf9,	0x42fb,	0x6d14,	0x42d0,	0x6d2e,	0x42a5,	0x6d48,
	0x427a,	0x6d62,	0x424f,	0x6d7c,	0x4224,	0x6d96,	0x41f9,	0x6db0,
	0x41ce,	0x6dca,	0x41a3,	0x6de4,	0x4178,	0x6dfe,	0x414d,	0x6e17,
	0x4121,	0x6e31,	0x40f6,	0x6e4a,	0x40cb,	0x6e64,	0x409f,	0x6e7d,
	0x4074,	0x6e97,	0x4048,	0x6eb0,	0x401d,	0x6ec9,	0x3ff1,	0x6ee2,
	0x3fc6,	0x6efb,	0x3f9a,	0x6f14,	0x3f6f,	0x6f2d,	0x3f43,	0x6f46,
	0x3f17,	0x6f5f,	0x3eec,	0x6f78,	0x3ec0,	0x6f90,	0x3e94,	0x6fa9,
	0x3e68,	0x6fc2,	0x3e3c,	0x6fda,	0x3e10,	0x6ff2,	0x3de4,	0x700b,
	0x3db8,	0x7023,	0x3d8c,	0x703b,	0x3d60,	0x7053,	0x3d34,	0x706b,
	0x3d08,	0x7083,	0x3cdc,	0x709b,	0x3caf,	0x70b3,	0x3c83,	0x70cb,
	0x3c57,	0x70e3,	0x3c2a,	0x70fa,	0x3bfe,	0x7112,	0x3bd2,	0x712a,
	0x3ba5,	0x7141,	0x3b79,	0x7158,	0x3b4c,	0x7170,	0x3b20,	0x7187,
	0x3af3,	0x719e,	0x3ac6,	0x71b5,	0x3a9a,	0x71cc,	0x3a6d,	0x71e3,
	0x3a40,	0x71fa,	0x3a13,	0x7211,	0x39e7,	0x7228,	0x39ba,	0x723f,
	0x398d,	0x7255,	0x3960,	0x726c,	0x3933,	0x7282,	0x3906,	0x7299,
	0x38d9,	0x72af,	0x38ac,	0x72c5,	0x387f,	0x72dc,	0x3852,	0x72f2,
	0x3825,	0x7308,	0x37f7,	0x731e,	0x37ca,	0x7334,	0x379d,	0x734a,
	0x3770,	0x735f,	0x3742,	0x7375,	0x3715,	0x738b,	0x36e8,	0x73a0,
	0x36ba,	0x73b6,	0x368d,	0x73cb,	0x365f,	0x73e1,	0x3632,	0x73f6,
	0x3604,	0x740b,	0x35d7,	0x7421,	0x35a9,	0x7436,	0x357b,	0x744b,
	0x354e,	0x7460,	0x3520,	0x7475,	0x34f2,	0x7489,	0x34c4,	0x749e,
	0x3497,	0x74b3,	0x3469,	0x74c7,	0x343b,	0x74dc,	0x340d,	0x74f0,
	0x33df,	0x7505,	0x33b1,	0x7519,	0x3383,	0x752d,	0x3355,	0x7542,
	0x3327,	0x7556,	0x32f9,	0x756a,	0x32cb,	0x757e,	0x329d,	0x7592,
	0x326e,	0x75a6,	0x3240,	0x75b9,	0x3212,	0x75cd,	0x31e4,	0x75e1,
	0x31b5,	0x75f4,	0x3187,	0x7608,	0x3159,	0x761b,	0x312a,	0x762e,
	0x30fc,	0x7642,	0x30cd,	0x7655,	0x309f,	0x7668,	0x3070,	0x767b,
	0x3042,	0x768e,	0x3013,	0x76a1,	0x2fe5,	0x76b4,	0x2fb6,	0x76c7,
	0x2f87,	0x76d9,	0x2f59,	0x76ec,	0x2f2a,	0x76fe,	0x2efb,	0x7711,
	0x2ecc,	0x7723,	0x2e9e,	0x7736,	0x2e6f,	0x7748,	0x2e40,	0x775a,
	0x2e11,	0x776c,	0x2de2,	0x777e,	0x2db3,	0x7790,	0x2d84,	0x77a2,
	0x2d55,	0x77b4,	0x2d26,	0x77c6,	0x2cf7,	0x77d8,	0x2cc8,	0x77e9,
	0x2c99,	0x77fb,	0x2c6a,	0x780c,	0x2c3b,	0x781e,	0x2c0c,	0x782f,
	0x2bdc,	0x7840,	0x2bad,	0x7851,	0x2b7e,	0x7863,	0x2b4f,	0x7874,
	0x2b1f,	0x7885,	0x2af0,	0x7895,	0x2ac1,	0x78a6,	0x2a91,	0x78b7,
	0x2a62,	0x78c8,	0x2a32,	0x78d8,	0x2a03,	0x78e9,	0x29d3,	0x78f9,
	0x29a4,	0x790a,	0x2974,	0x791a,	0x2945,	0x792a,	0x2915,	0x793a,
	0x28e5,	0x794a,	0x28b6,	0x795b,	0x2886,	0x796a,	0x2856,	0x797a,
	0x2827,	0x798a,	0x27f7,	0x799a,	0x27c7,	0x79aa,	0x2797,	0x79b9,
	0x2768,	0x79c9,	0x2738,	0x79d8,	0x2708,	0x79e7,	0x26d8,	0x79f7,
	0x26a8,	0x7a06,	0x2678,	0x7a15,	0x2648,	0x7a24,	0x2618,	0x7a33,
	0x25e8,	0x7a42,	0x25b8,	0x7a51,	0x2588,	0x7a60,	0x2558,	0x7a6e,
	0x2528,	0x7a7d,	0x24f8,	0x7a8c,	0x24c8,	0x7a9a,	0x2498,	0x7aa8,
	0x2467,	0x7ab7,	0x2437,	0x7ac5,	0x2407,	0x7ad3,	0x23d7,	0x7ae1,
	0x23a7,	0x7aef,	0x2376,	0x7afd,	0x2346,	0x7b0b,	0x2316,	0x7b19,
	0x22e5,	0x7b27,	0x22b5,	0x7b34,	0x2284,	0x7b42,	0x2254,	0x7b50,
	0x2224,	0x7b5d,	0x21f3,	0x7b6a,	0x21c3,	0x7b78,	0x2192,	0x7b85,
	0x2162,	0x7b92,	0x2131,	0x7b9f,	0x2101,	0x7bac,	0x20d0,	0x7bb9,
	0x209f,	0x7bc6,	0x206f,	0x7bd3,	0x203e,	0x7bdf,	0x200e,	0x7bec,
	0x1fdd,	0x7bf9,	0x1fac,	0x7c05,	0x1f7b,	0x7c11,	0x1f4b,	0x7c1e,
	0x1f1a,	0x7c2a,	0x1ee9,	0x7c36,	0x1eb8,	0x7c42,	0x1e88,	0x7c4e,
	0x1e57,	0x7c5a,	0x1e26,	0x7c66,	0x1df5,	0x7c72,	0x1dc4,	0x7c7e,
	0x1d93,	0x7c89,	0x1d62,	0x7c95,	0x1d31,	0x7ca0,	0x1d01,	0x7cac,
	0x1cd0,	0x7cb7,	0x1c9f,	0x7cc2,	0x1c6e,	0x7cce,	0x1c3d,	0x7cd9,
	0x1c0c,	0x7ce4,	0x1bda,	0x7cef,	0x1ba9,	0x7cfa,	0x1b78,	0x7d05,
	0x1b47,	0x7d0f,	0x1b16,	0x7d1a,	0x1ae5,	0x7d25,	0x1ab4,	0x7d2f,
	0x1a83,	0x7d3a,	0x1a51,	0x7d44,	0x1a20,	0x7d4e,	0x19ef,	0x7d58,
	0x19be,	0x7d63,	0x198d,	0x7d6d,	0x195b,	0x7d77,	0x192a,	0x7d81,
	0x18f9,	0x7d8a,	0x18c7,	0x7d94,	0x1896,	0x7d9e,	0x1865,	0x7da7,
	0x1833,	0x7db1,	0x1802,	0x7dba,	0x17d1,	0x7dc4,	0x179f,	0x7dcd,
	0x176e,	0x7dd6,	0x173c,	0x7de0,	0x170b,	0x7de9,	0x16da,	0x7df2,
	0x16a8,	0x7dfb,	0x1677,	0x7e03,	0x1645,	0x7e0c,	0x1614,	0x7e15,
	0x15e2,	0x7e1e,	0x15b1,	0x7e26,	0x157f,	0x7e2f,	0x154d,	0x7e37,
	0x151c,	0x7e3f,	0x14ea,	0x7e48,	0x14b9,	0x7e50,	0x1487,	0x7e58,
	0x1455,	0x7e60,	0x1424,	0x7e68,	0x13f2,	0x7e70,	0x13c1,	0x7e78,
	0x138f,	0x7e7f,	0x135d,	0x7e87,	0x132b,	0x7e8e,	0x12fa,	0x7e96,
	0x12c8,	0x7e9d,	0x1296,	0x7ea5,	0x1265,	0x7eac,	0x1233,	0x7eb3,
	0x1201,	0x7eba,	0x11cf,	0x7ec1,	0x119e,	0x7ec8,	0x116c,	0x7ecf,
	0x113a,	0x7ed6,	0x1108,	0x7edd,	0x10d6,	0x7ee3,	0x10a4,	0x7eea,
	0x1073,	0x7ef0,	0x1041,	0x7ef7,	0x100f,	0x7efd,	0xfdd,	0x7f03,
	0xfab,	0x7f0a,	0xf79,	0x7f10,	0xf47,	0x7f16,	0xf15,	0x7f1c,
	0xee4,	0x7f22,	0xeb2,	0x7f27,	0xe80,	0x7f2d,	0xe4e,	0x7f33,
	0xe1c,	0x7f38,	0xdea,	0x7f3e,	0xdb8,	0x7f43,	0xd86,	0x7f49,
	0xd54,	0x7f4e,	0xd22,	0x7f53,	0xcf0,	0x7f58,	0xcbe,	0x7f5d,
	0xc8c,	0x7f62,	0xc5a,	0x7f67,	0xc28,	0x7f6c,	0xbf6,	0x7f71,
	0xbc4,	0x7f75,	0xb92,	0x7f7a,	0xb60,	0x7f7e,	0xb2d,	0x7f83,
	0xafb,	0x7f87,	0xac9,	0x7f8b,	0xa97,	0x7f90,	0xa65,	0x7f94,
	0xa33,	0x7f98,	0xa01,	0x7f9c,	0x9cf,	0x7fa0,	0x99d,	0x7fa3,
	0x96b,	0x7fa7,	0x938,	0x7fab,	0x906,	0x7fae,	0x8d4,	0x7fb2,
	0x8a2,	0x7fb5,	0x870,	0x7fb9,	0x83e,	0x7fbc,	0x80c,	0x7fbf,
	0x7d9,	0x7fc2,	0x7a7,	0x7fc5,	0x775,	0x7fc8,	0x743,	0x7fcb,
	0x711,	0x7fce,	0x6de,	0x7fd1,	0x6ac,	0x7fd3,	0x67a,	0x7fd6,
	0x648,	0x7fd9,	0x616,	0x7fdb,	0x5e3,	0x7fdd,	0x5b1,	0x7fe0,
	0x57f,	0x7fe2,	0x54d,	0x7fe4,	0x51b,	0x7fe6,	0x4e8,	0x7fe8,
	0x4b6,	0x7fea,	0x484,	0x7fec,	0x452,	0x7fed,	0x41f,	0x7fef,
	0x3ed,	0x7ff1,	0x3bb,	0x7ff2,	0x389,	0x7ff4,	0x356,	0x7ff5,
	0x324,	0x7ff6,	0x2f2,	0x7ff7,	0x2c0,	0x7ff8,	0x28d,	0x7ff9,
	0x25b,	0x7ffa,	0x229,	0x7ffb,	0x1f7,	0x7ffc,	0x1c4,	0x7ffd,
	0x192,	0x7ffe,	0x160,	0x7ffe,	0x12e,	0x7fff,	0xfb,	0x7fff,
	0xc9,	0x7fff,	0x97,	0x7fff,	0x65,	0x7fff,	0x32,	0x7fff,
	0x0,	0x7fff,	0xffce,	0x7fff,	0xff9b,	0x7fff,	0xff69,	0x7fff,
	0xff37,	0x7fff,	0xff05,	0x7fff,	0xfed2,	0x7fff,	0xfea0,	0x7ffe,
	0xfe6e,	0x7ffe,	0xfe3c,	0x7ffd,	0xfe09,	0x7ffc,	0xfdd7,	0x7ffb,
	0xfda5,	0x7ffa,	0xfd73,	0x7ff9,	0xfd40,	0x7ff8,	0xfd0e,	0x7ff7,
	0xfcdc,	0x7ff6,	0xfcaa,	0x7ff5,	0xfc77,	0x7ff4,	0xfc45,	0x7ff2,
	0xfc13,	0x7ff1,	0xfbe1,	0x7fef,	0xfbae,	0x7fed,	0xfb7c,	0x7fec,
	0xfb4a,	0x7fea,	0xfb18,	0x7fe8,	0xfae5,	0x7fe6,	0xfab3,	0x7fe4,
	0xfa81,	0x7fe2,	0xfa4f,	0x7fe0,	0xfa1d,	0x7fdd,	0xf9ea,	0x7fdb,
	0xf9b8,	0x7fd9,	0xf986,	0x7fd6,	0xf954,	0x7fd3,	0xf922,	0x7fd1,
	0xf8ef,	0x7fce,	0xf8bd,	0x7fcb,	0xf88b,	0x7fc8,	0xf859,	0x7fc5,
	0xf827,	0x7fc2,	0xf7f4,	0x7fbf,	0xf7c2,	0x7fbc,	0xf790,	0x7fb9,
	0xf75e,	0x7fb5,	0xf72c,	0x7fb2,	0xf6fa,	0x7fae,	0xf6c8,	0x7fab,
	0xf695,	0x7fa7,	0xf663,	0x7fa3,	0xf631,	0x7fa0,	0xf5ff,	0x7f9c,
	0xf5cd,	0x7f98,	0xf59b,	0x7f94,	0xf569,	0x7f90,	0xf537,	0x7f8b,
	0xf505,	0x7f87,	0xf4d3,	0x7f83,	0xf4a0,	0x7f7e,	0xf46e,	0x7f7a,
	0xf43c,	0x7f75,	0xf40a,	0x7f71,	0xf3d8,	0x7f6c,	0xf3a6,	0x7f67,
	0xf374,	0x7f62,	0xf342,	0x7f5d,	0xf310,	0x7f58,	0xf2de,	0x7f53,
	0xf2ac,	0x7f4e,	0xf27a,	0x7f49,	0xf248,	0x7f43,	0xf216,	0x7f3e,
	0xf1e4,	0x7f38,	0xf1b2,	0x7f33,	0xf180,	0x7f2d,	0xf14e,	0x7f27,
	0xf11c,	0x7f22,	0xf0eb,	0x7f1c,	0xf0b9,	0x7f16,	0xf087,	0x7f10,
	0xf055,	0x7f0a,	0xf023,	0x7f03,	0xeff1,	0x7efd,	0xefbf,	0x7ef7,
	0xef8d,	0x7ef0,	0xef5c,	0x7eea,	0xef2a,	0x7ee3,	0xeef8,	0x7edd,
	0xeec6,	0x7ed6,	0xee94,	0x7ecf,	0xee62,	0x7ec8,	0xee31,	0x7ec1,
	0xedff,	0x7eba,	0xedcd,	0x7eb3,	0xed9b,	0x7eac,	0xed6a,	0x7ea5,
	0xed38,	0x7e9d,	0xed06,	0x7e96,	0xecd5,	0x7e8e,	0xeca3,	0x7e87,
	0xec71,	0x7e7f,	0xec3f,	0x7e78,	0xec0e,	0x7e70,	0xebdc,	0x7e68,
	0xebab,	0x7e60,	0xeb79,	0x7e58,	0xeb47,	0x7e50,	0xeb16,	0x7e48,
	0xeae4,	0x7e3f,	0xeab3,	0x7e37,	0xea81,	0x7e2f,	0xea4f,	0x7e26,
	0xea1e,	0x7e1e,	0xe9ec,	0x7e15,	0xe9bb,	0x7e0c,	0xe989,	0x7e03,
	0xe958,	0x7dfb,	0xe926,	0x7df2,	0xe8f5,	0x7de9,	0xe8c4,	0x7de0,
	0xe892,	0x7dd6,	0xe861,	0x7dcd,	0xe82f,	0x7dc4,	0xe7fe,	0x7dba,
	0xe7cd,	0x7db1,	0xe79b,	0x7da7,	0xe76a,	0x7d9e,	0xe739,	0x7d94,
	0xe707,	0x7d8a,	0xe6d6,	0x7d81,	0xe6a5,	0x7d77,	0xe673,	0x7d6d,
	0xe642,	0x7d63,	0xe611,	0x7d58,	0xe5e0,	0x7d4e,	0xe5af,	0x7d44,
	0xe57d,	0x7d3a,	0xe54c,	0x7d2f,	0xe51b,	0x7d25,	0xe4ea,	0x7d1a,
	0xe4b9,	0x7d0f,	0xe488,	0x7d05,	0xe457,	0x7cfa,	0xe426,	0x7cef,
	0xe3f4,	0x7ce4,	0xe3c3,	0x7cd9,	0xe392,	0x7cce,	0xe361,	0x7cc2,
	0xe330,	0x7cb7,	0xe2ff,	0x7cac,	0xe2cf,	0x7ca0,	0xe29e,	0x7c95,
	0xe26d,	0x7c89,	0xe23c,	0x7c7e,	0xe20b,	0x7c72,	0xe1da,	0x7c66,
	0xe1a9,	0x7c5a,	0xe178,	0x7c4e,	0xe148,	0x7c42,	0xe117,	0x7c36,
	0xe0e6,	0x7c2a,	0xe0b5,	0x7c1e,	0xe085,	0x7c11,	0xe054,	0x7c05,
	0xe023,	0x7bf9,	0xdff2,	0x7bec,	0xdfc2,	0x7bdf,	0xdf91,	0x7bd3,
	0xdf61,	0x7bc6,	0xdf30,	0x7bb9,	0xdeff,	0x7bac,	0xdecf,	0x7b9f,
	0xde9e,	0x7b92,	0xde6e,	0x7b85,	0xde3d,	0x7b78,	0xde0d,	0x7b6a,
	0xdddc,	0x7b5d,	0xddac,	0x7b50,	0xdd7c,	0x7b42,	0xdd4b,	0x7b34,
	0xdd1b,	0x7b27,	0xdcea,	0x7b19,	0xdcba,	0x7b0b,	0xdc8a,	0x7afd,
	0xdc59,	0x7aef,	0xdc29,	0x7ae1,	0xdbf9,	0x7ad3,	0xdbc9,	0x7ac5,
	0xdb99,	0x7ab7,	0xdb68,	0x7aa8,	0xdb38,	0x7a9a,	0xdb08,	0x7a8c,
	0xdad8,	0x7a7d,	0xdaa8,	0x7a6e,	0xda78,	0x7a60,	0xda48,	0x7a51,
	0xda18,	0x7a42,	0xd9e8,	0x7a33,	0xd9b8,	0x7a24,	0xd988,	0x7a15,
	0xd958,	0x7a06,	0xd928,	0x79f7,	0xd8f8,	0x79e7,	0xd8c8,	0x79d8,
	0xd898,	0x79c9,	0xd869,	0x79b9,	0xd839,	0x79aa,	0xd809,	0x799a,
	0xd7d9,	0x798a,	0xd7aa,	0x797a,	0xd77a,	0x796a,	0xd74a,	0x795b,
	0xd71b,	0x794a,	0xd6eb,	0x793a,	0xd6bb,	0x792a,	0xd68c,	0x791a,
	0xd65c,	0x790a,	0xd62d,	0x78f9,	0xd5fd,	0x78e9,	0xd5ce,	0x78d8,
	0xd59e,	0x78c8,	0xd56f,	0x78b7,	0xd53f,	0x78a6,	0xd510,	0x7895,
	0xd4e1,	0x7885,	0xd4b1,	0x7874,	0xd482,	0x7863,	0xd453,	0x7851,
	0xd424,	0x7840,	0xd3f4,	0x782f,	0xd3c5,	0x781e,	0xd396,	0x780c,
	0xd367,	0x77fb,	0xd338,	0x77e9,	0xd309,	0x77d8,	0xd2da,	0x77c6,
	0xd2ab,	0x77b4,	0xd27c,	0x77a2,	0xd24d,	0x7790,	0xd21e,	0x777e,
	0xd1ef,	0x776c,	0xd1c0,	0x775a,	0xd191,	0x7748,	0xd162,	0x7736,
	0xd134,	0x7723,	0xd105,	0x7711,	0xd0d6,	0x76fe,	0xd0a7,	0x76ec,
	0xd079,	0x76d9,	0xd04a,	0x76c7,	0xd01b,	0x76b4,	0xcfed,	0x76a1,
	0xcfbe,	0x768e,	0xcf90,	0x767b,	0xcf61,	0x7668,	0xcf33,	0x7655,
	0xcf04,	0x7642,	0xced6,	0x762e,	0xcea7,	0x761b,	0xce79,	0x7608,
	0xce4b,	0x75f4,	0xce1c,	0x75e1,	0xcdee,	0x75cd,	0xcdc0,	0x75b9,
	0xcd92,	0x75a6,	0xcd63,	0x7592,	0xcd35,	0x757e,	0xcd07,	0x756a,
	0xccd9,	0x7556,	0xccab,	0x7542,	0xcc7d,	0x752d,	0xcc4f,	0x7519,
	0xcc21,	0x7505,	0xcbf3,	0x74f0,	0xcbc5,	0x74dc,	0xcb97,	0x74c7,
	0xcb69,	0x74b3,	0xcb3c,	0x749e,	0xcb0e,	0x7489,	0xcae0,	0x7475,
	0xcab2,	0x7460,	0xca85,	0x744b,	0xca57,	0x7436,	0xca29,	0x7421,
	0xc9fc,	0x740b,	0xc9ce,	0x73f6,	0xc9a1,	0x73e1,	0xc973,	0x73cb,
	0xc946,	0x73b6,	0xc918,	0x73a0,	0xc8eb,	0x738b,	0xc8be,	0x7375,
	0xc890,	0x735f,	0xc863,	0x734a,	0xc836,	0x7334,	0xc809,	0x731e,
	0xc7db,	0x7308,	0xc7ae,	0x72f2,	0xc781,	0x72dc,	0xc754,	0x72c5,
	0xc727,	0x72af,	0xc6fa,	0x7299,	0xc6cd,	0x7282,	0xc6a0,	0x726c,
	0xc673,	0x7255,	0xc646,	0x723f,	0xc619,	0x7228,	0xc5ed,	0x7211,
	0xc5c0,	0x71fa,	0xc593,	0x71e3,	0xc566,	0x71cc,	0xc53a,	0x71b5,
	0xc50d,	0x719e,	0xc4e0,	0x7187,	0xc4b4,	0x7170,	0xc487,	0x7158,
	0xc45b,	0x7141,	0xc42e,	0x712a,	0xc402,	0x7112,	0xc3d6,	0x70fa,
	0xc3a9,	0x70e3,	0xc37d,	0x70cb,	0xc351,	0x70b3,	0xc324,	0x709b,
	0xc2f8,	0x7083,	0xc2cc,	0x706b,	0xc2a0,	0x7053,	0xc274,	0x703b,
	0xc248,	0x7023,	0xc21c,	0x700b,	0xc1f0,	0x6ff2,	0xc1c4,	0x6fda,
	0xc198,	0x6fc2,	0xc16c,	0x6fa9,	0xc140,	0x6f90,	0xc114,	0x6f78,
	0xc0e9,	0x6f5f,	0xc0bd,	0x6f46,	0xc091,	0x6f2d,	0xc066,	0x6f14,
	0xc03a,	0x6efb,	0xc00f,	0x6ee2,	0xbfe3,	0x6ec9,	0xbfb8,	0x6eb0,
	0xbf8c,	0x6e97,	0xbf61,	0x6e7d,	0xbf35,	0x6e64,	0xbf0a,	0x6e4a,
	0xbedf,	0x6e31,	0xbeb3,	0x6e17,	0xbe88,	0x6dfe,	0xbe5d,	0x6de4,
	0xbe32,	0x6dca,	0xbe07,	0x6db0,	0xbddc,	0x6d96,	0xbdb1,	0x6d7c,
	0xbd86,	0x6d62,	0xbd5b,	0x6d48,	0xbd30,	0x6d2e,	0xbd05,	0x6d14,
	0xbcda,	0x6cf9,	0xbcaf,	0x6cdf,	0xbc85,	0x6cc4,	0xbc5a,	0x6caa,
	0xbc2f,	0x6c8f,	0xbc05,	0x6c75,	0xbbda,	0x6c5a,	0xbbb0,	0x6c3f,
	0xbb85,	0x6c24,	0xbb5b,	0x6c09,	0xbb30,	0x6bee,	0xbb06,	0x6bd3,
	0xbadc,	0x6bb8,	0xbab1,	0x6b9d,	0xba87,	0x6b82,	0xba5d,	0x6b66,
	0xba33,	0x6b4b,	0xba09,	0x6b30,	0xb9df,	0x6b14,	0xb9b5,	0x6af8,
	0xb98b,	0x6add,	0xb961,	0x6ac1,	0xb937,	0x6aa5,	0xb90d,	0x6a89,
	0xb8e3,	0x6a6e,	0xb8b9,	0x6a52,	0xb890,	0x6a36,	0xb866,	0x6a1a,
	0xb83c,	0x69fd,	0xb813,	0x69e1,	0xb7e9,	0x69c5,	0xb7c0,	0x69a9,
	0xb796,	0x698c,	0xb76d,	0x6970,	0xb743,	0x6953,	0xb71a,	0x6937,
	0xb6f1,	0x691a,	0xb6c7,	0x68fd,	0xb69e,	0x68e0,	0xb675,	0x68c4,
	0xb64c,	0x68a7,	0xb623,	0x688a,	0xb5fa,	0x686d,	0xb5d1,	0x6850,
	0xb5a8,	0x6832,	0xb57f,	0x6815,	0xb556,	0x67f8,	0xb52d,	0x67da,
	0xb505,	0x67bd,	0xb4dc,	0x67a0,	0xb4b3,	0x6782,	0xb48b,	0x6764,
	0xb462,	0x6747,	0xb439,	0x6729,	0xb411,	0x670b,	0xb3e9,	0x66ed,
	0xb3c0,	0x66d0,	0xb398,	0x66b2,	0xb36f,	0x6693,	0xb347,	0x6675,
	0xb31f,	0x6657,	0xb2f7,	0x6639,	0xb2cf,	0x661b,	0xb2a7,	0x65fc,
	0xb27f,	0x65de,	0xb257,	0x65c0,	0xb22f,	0x65a1,	0xb207,	0x6582,
	0xb1df,	0x6564,	0xb1b7,	0x6545,	0xb18f,	0x6526,	0xb168,	0x6507,
	0xb140,	0x64e9,	0xb118,	0x64ca,	0xb0f1,	0x64ab,	0xb0c9,	0x648b,
	0xb0a2,	0x646c,	0xb07b,	0x644d,	0xb053,	0x642e,	0xb02c,	0x640f,
	0xb005,	0x63ef,	0xafdd,	0x63d0,	0xafb6,	0x63b0,	0xaf8f,	0x6391,
	0xaf68,	0x6371,	0xaf41,	0x6351,	0xaf1a,	0x6332,	0xaef3,	0x6312,
	0xaecc,	0x62f2,	0xaea5,	0x62d2,	0xae7f,	0x62b2,	0xae58,	0x6292,
	0xae31,	0x6272,	0xae0b,	0x6252,	0xade4,	0x6232,	0xadbd,	0x6211,
	0xad97,	0x61f1,	0xad70,	0x61d1,	0xad4a,	0x61b0,	0xad24,	0x6190,
	0xacfd,	0x616f,	0xacd7,	0x614e,	0xacb1,	0x612e,	0xac8b,	0x610d,
	0xac65,	0x60ec,	0xac3f,	0x60cb,	0xac19,	0x60aa,	0xabf3,	0x6089,
	0xabcd,	0x6068,	0xaba7,	0x6047,	0xab81,	0x6026,	0xab5c,	0x6005,
	0xab36,	0x5fe4,	0xab10,	0x5fc2,	0xaaeb,	0x5fa1,	0xaac5,	0x5f80,
	0xaaa0,	0x5f5e,	0xaa7a,	0x5f3c,	0xaa55,	0x5f1b,	0xaa30,	0x5ef9,
	0xaa0a,	0x5ed7,	0xa9e5,	0x5eb6,	0xa9c0,	0x5e94,	0xa99b,	0x5e72,
	0xa976,	0x5e50,	0xa951,	0x5e2e,	0xa92c,	0x5e0c,	0xa907,	0x5dea,
	0xa8e2,	0x5dc8,	0xa8bd,	0x5da5,	0xa899,	0x5d83,	0xa874,	0x5d61,
	0xa84f,	0x5d3e,	0xa82b,	0x5d1c,	0xa806,	0x5cf9,	0xa7e2,	0x5cd7,
	0xa7bd,	0x5cb4,	0xa799,	0x5c91,	0xa774,	0x5c6f,	0xa750,	0x5c4c,
	0xa72c,	0x5c29,	0xa708,	0x5c06,	0xa6e4,	0x5be3,	0xa6c0,	0x5bc0,
	0xa69c,	0x5b9d,	0xa678,	0x5b7a,	0xa654,	0x5b57,	0xa630,	0x5b34,
	0xa60c,	0x5b10,	0xa5e8,	0x5aed,	0xa5c5,	0x5ac9,	0xa5a1,	0x5aa6,
	0xa57e,	0x5a82,	0xa55a,	0x5a5f,	0xa537,	0x5a3b,	0xa513,	0x5a18,
	0xa4f0,	0x59f4,	0xa4cc,	0x59d0,	0xa4a9,	0x59ac,	0xa486,	0x5988,
	0xa463,	0x5964,	0xa440,	0x5940,	0xa41d,	0x591c,	0xa3fa,	0x58f8,
	0xa3d7,	0x58d4,	0xa3b4,	0x58b0,	0xa391,	0x588c,	0xa36f,	0x5867,
	0xa34c,	0x5843,	0xa329,	0x581e,	0xa307,	0x57fa,	0xa2e4,	0x57d5,
	0xa2c2,	0x57b1,	0xa29f,	0x578c,	0xa27d,	0x5767,	0xa25b,	0x5743,
	0xa238,	0x571e,	0xa216,	0x56f9,	0xa1f4,	0x56d4,	0xa1d2,	0x56af,
	0xa1b0,	0x568a,	0xa18e,	0x5665,	0xa16c,	0x5640,	0xa14a,	0x561b,
	0xa129,	0x55f6,	0xa107,	0x55d0,	0xa0e5,	0x55ab,	0xa0c4,	0x5586,
	0xa0a2,	0x5560,	0xa080,	0x553b,	0xa05f,	0x5515,	0xa03e,	0x54f0,
	0xa01c,	0x54ca,	0x9ffb,	0x54a4,	0x9fda,	0x547f,	0x9fb9,	0x5459,
	0x9f98,	0x5433,	0x9f77,	0x540d,	0x9f56,	0x53e7,	0x9f35,	0x53c1,
	0x9f14,	0x539b,	0x9ef3,	0x5375,	0x9ed2,	0x534f,	0x9eb2,	0x5329,
	0x9e91,	0x5303,	0x9e70,	0x52dc,	0x9e50,	0x52b6,	0x9e2f,	0x5290,
	0x9e0f,	0x5269,	0x9def,	0x5243,	0x9dce,	0x521c,	0x9dae,	0x51f5,
	0x9d8e,	0x51cf,	0x9d6e,	0x51a8,	0x9d4e,	0x5181,	0x9d2e,	0x515b,
	0x9d0e,	0x5134,	0x9cee,	0x510d,	0x9cce,	0x50e6,	0x9caf,	0x50bf,
	0x9c8f,	0x5098,	0x9c6f,	0x5071,	0x9c50,	0x504a,	0x9c30,	0x5023,
	0x9c11,	0x4ffb,	0x9bf1,	0x4fd4,	0x9bd2,	0x4fad,	0x9bb3,	0x4f85,
	0x9b94,	0x4f5e,	0x9b75,	0x4f37,	0x9b55,	0x4f0f,	0x9b36,	0x4ee8,
	0x9b17,	0x4ec0,	0x9af9,	0x4e98,	0x9ada,	0x4e71,	0x9abb,	0x4e49,
	0x9a9c,	0x4e21,	0x9a7e,	0x4df9,	0x9a5f,	0x4dd1,	0x9a40,	0x4da9,
	0x9a22,	0x4d81,	0x9a04,	0x4d59,	0x99e5,	0x4d31,	0x99c7,	0x4d09,
	0x99a9,	0x4ce1,	0x998b,	0x4cb9,	0x996d,	0x4c91,	0x994e,	0x4c68,
	0x9930,	0x4c40,	0x9913,	0x4c17,	0x98f5,	0x4bef,	0x98d7,	0x4bc7,
	0x98b9,	0x4b9e,	0x989c,	0x4b75,	0x987e,	0x4b4d,	0x9860,	0x4b24,
	0x9843,	0x4afb,	0x9826,	0x4ad3,	0x9808,	0x4aaa,	0x97eb,	0x4a81,
	0x97ce,	0x4a58,	0x97b0,	0x4a2f,	0x9793,	0x4a06,	0x9776,	0x49dd,
	0x9759,	0x49b4,	0x973c,	0x498b,	0x9720,	0x4962,	0x9703,	0x4939,
	0x96e6,	0x490f,	0x96c9,	0x48e6,	0x96ad,	0x48bd,	0x9690,	0x4893,
	0x9674,	0x486a,	0x9657,	0x4840,	0x963b,	0x4817,	0x961f,	0x47ed,
	0x9603,	0x47c4,	0x95e6,	0x479a,	0x95ca,	0x4770,	0x95ae,	0x4747,
	0x9592,	0x471d,	0x9577,	0x46f3,	0x955b,	0x46c9,	0x953f,	0x469f,
	0x9523,	0x4675,	0x9508,	0x464b,	0x94ec,	0x4621,	0x94d0,	0x45f7,
	0x94b5,	0x45cd,	0x949a,	0x45a3,	0x947e,	0x4579,	0x9463,	0x454f,
	0x9448,	0x4524,	0x942d,	0x44fa,	0x9412,	0x44d0,	0x93f7,	0x44a5,
	0x93dc,	0x447b,	0x93c1,	0x4450,	0x93a6,	0x4426,	0x938b,	0x43fb,
	0x9371,	0x43d1,	0x9356,	0x43a6,	0x933c,	0x437b,	0x9321,	0x4351,
	0x9307,	0x4326,	0x92ec,	0x42fb,	0x92d2,	0x42d0,	0x92b8,	0x42a5,
	0x929e,	0x427a,	0x9284,	0x424f,	0x926a,	0x4224,	0x9250,	0x41f9,
	0x9236,	0x41ce,	0x921c,	0x41a3,	0x9202,	0x4178,	0x91e9,	0x414d,
	0x91cf,	0x4121,	0x91b6,	0x40f6,	0x919c,	0x40cb,	0x9183,	0x409f,
	0x9169,	0x4074,	0x9150,	0x4048,	0x9137,	0x401d,	0x911e,	0x3ff1,
	0x9105,	0x3fc6,	0x90ec,	0x3f9a,	0x90d3,	0x3f6f,	0x90ba,	0x3f43,
	0x90a1,	0x3f17,	0x9088,	0x3eec,	0x9070,	0x3ec0,	0x9057,	0x3e94,
	0x903e,	0x3e68,	0x9026,	0x3e3c,	0x900e,	0x3e10,	0x8ff5,	0x3de4,
	0x8fdd,	0x3db8,	0x8fc5,	0x3d8c,	0x8fad,	0x3d60,	0x8f95,	0x3d34,
	0x8f7d,	0x3d08,	0x8f65,	0x3cdc,	0x8f4d,	0x3caf,	0x8f35,	0x3c83,
	0x8f1d,	0x3c57,	0x8f06,	0x3c2a,	0x8eee,	0x3bfe,	0x8ed6,	0x3bd2,
	0x8ebf,	0x3ba5,	0x8ea8,	0x3b79,	0x8e90,	0x3b4c,	0x8e79,	0x3b20,
	0x8e62,	0x3af3,	0x8e4b,	0x3ac6,	0x8e34,	0x3a9a,	0x8e1d,	0x3a6d,
	0x8e06,	0x3a40,	0x8def,	0x3a13,	0x8dd8,	0x39e7,	0x8dc1,	0x39ba,
	0x8dab,	0x398d,	0x8d94,	0x3960,	0x8d7e,	0x3933,	0x8d67,	0x3906,
	0x8d51,	0x38d9,	0x8d3b,	0x38ac,	0x8d24,	0x387f,	0x8d0e,	0x3852,
	0x8cf8,	0x3825,	0x8ce2,	0x37f7,	0x8ccc,	0x37ca,	0x8cb6,	0x379d,
	0x8ca1,	0x3770,	0x8c8b,	0x3742,	0x8c75,	0x3715,	0x8c60,	0x36e8,
	0x8c4a,	0x36ba,	0x8c35,	0x368d,	0x8c1f,	0x365f,	0x8c0a,	0x3632,
	0x8bf5,	0x3604,	0x8bdf,	0x35d7,	0x8bca,	0x35a9,	0x8bb5,	0x357b,
	0x8ba0,	0x354e,	0x8b8b,	0x3520,	0x8b77,	0x34f2,	0x8b62,	0x34c4,
	0x8b4d,	0x3497,	0x8b39,	0x3469,	0x8b24,	0x343b,	0x8b10,	0x340d,
	0x8afb,	0x33df,	0x8ae7,	0x33b1,	0x8ad3,	0x3383,	0x8abe,	0x3355,
	0x8aaa,	0x3327,	0x8a96,	0x32f9,	0x8a82,	0x32cb,	0x8a6e,	0x329d,
	0x8a5a,	0x326e,	0x8a47,	0x3240,	0x8a33,	0x3212,	0x8a1f,	0x31e4,
	0x8a0c,	0x31b5,	0x89f8,	0x3187,	0x89e5,	0x3159,	0x89d2,	0x312a,
	0x89be,	0x30fc,	0x89ab,	0x30cd,	0x8998,	0x309f,	0x8985,	0x3070,
	0x8972,	0x3042,	0x895f,	0x3013,	0x894c,	0x2fe5,	0x8939,	0x2fb6,
	0x8927,	0x2f87,	0x8914,	0x2f59,	0x8902,	0x2f2a,	0x88ef,	0x2efb,
	0x88dd,	0x2ecc,	0x88ca,	0x2e9e,	0x88b8,	0x2e6f,	0x88a6,	0x2e40,
	0x8894,	0x2e11,	0x8882,	0x2de2,	0x8870,	0x2db3,	0x885e,	0x2d84,
	0x884c,	0x2d55,	0x883a,	0x2d26,	0x8828,	0x2cf7,	0x8817,	0x2cc8,
	0x8805,	0x2c99,	0x87f4,	0x2c6a,	0x87e2,	0x2c3b,	0x87d1,	0x2c0c,
	0x87c0,	0x2bdc,	0x87af,	0x2bad,	0x879d,	0x2b7e,	0x878c,	0x2b4f,
	0x877b,	0x2b1f,	0x876b,	0x2af0,	0x875a,	0x2ac1,	0x8749,	0x2a91,
	0x8738,	0x2a62,	0x8728,	0x2a32,	0x8717,	0x2a03,	0x8707,	0x29d3,
	0x86f6,	0x29a4,	0x86e6,	0x2974,	0x86d6,	0x2945,	0x86c6,	0x2915,
	0x86b6,	0x28e5,	0x86a5,	0x28b6,	0x8696,	0x2886,	0x8686,	0x2856,
	0x8676,	0x2827,	0x8666,	0x27f7,	0x8656,	0x27c7,	0x8647,	0x2797,
	0x8637,	0x2768,	0x8628,	0x2738,	0x8619,	0x2708,	0x8609,	0x26d8,
	0x85fa,	0x26a8,	0x85eb,	0x2678,	0x85dc,	0x2648,	0x85cd,	0x2618,
	0x85be,	0x25e8,	0x85af,	0x25b8,	0x85a0,	0x2588,	0x8592,	0x2558,
	0x8583,	0x2528,	0x8574,	0x24f8,	0x8566,	0x24c8,	0x8558,	0x2498,
	0x8549,	0x2467,	0x853b,	0x2437,	0x852d,	0x2407,	0x851f,	0x23d7,
	0x8511,	0x23a7,	0x8503,	0x2376,	0x84f5,	0x2346,	0x84e7,	0x2316,
	0x84d9,	0x22e5,	0x84cc,	0x22b5,	0x84be,	0x2284,	0x84b0,	0x2254,
	0x84a3,	0x2224,	0x8496,	0x21f3,	0x8488,	0x21c3,	0x847b,	0x2192,
	0x846e,	0x2162,	0x8461,	0x2131,	0x8454,	0x2101,	0x8447,	0x20d0,
	0x843a,	0x209f,	0x842d,	0x206f,	0x8421,	0x203e,	0x8414,	0x200e,
	0x8407,	0x1fdd,	0x83fb,	0x1fac,	0x83ef,	0x1f7b,	0x83e2,	0x1f4b,
	0x83d6,	0x1f1a,	0x83ca,	0x1ee9,	0x83be,	0x1eb8,	0x83b2,	0x1e88,
	0x83a6,	0x1e57,	0x839a,	0x1e26,	0x838e,	0x1df5,	0x8382,	0x1dc4,
	0x8377,	0x1d93,	0x836b,	0x1d62,	0x8360,	0x1d31,	0x8354,	0x1d01,
	0x8349,	0x1cd0,	0x833e,	0x1c9f,	0x8332,	0x1c6e,	0x8327,	0x1c3d,
	0x831c,	0x1c0c,	0x8311,	0x1bda,	0x8306,	0x1ba9,	0x82fb,	0x1b78,
	0x82f1,	0x1b47,	0x82e6,	0x1b16,	0x82db,	0x1ae5,	0x82d1,	0x1ab4,
	0x82c6,	0x1a83,	0x82bc,	0x1a51,	0x82b2,	0x1a20,	0x82a8,	0x19ef,
	0x829d,	0x19be,	0x8293,	0x198d,	0x8289,	0x195b,	0x827f,	0x192a,
	0x8276,	0x18f9,	0x826c,	0x18c7,	0x8262,	0x1896,	0x8259,	0x1865,
	0x824f,	0x1833,	0x8246,	0x1802,	0x823c,	0x17d1,	0x8233,	0x179f,
	0x822a,	0x176e,	0x8220,	0x173c,	0x8217,	0x170b,	0x820e,	0x16da,
	0x8205,	0x16a8,	0x81fd,	0x1677,	0x81f4,	0x1645,	0x81eb,	0x1614,
	0x81e2,	0x15e2,	0x81da,	0x15b1,	0x81d1,	0x157f,	0x81c9,	0x154d,
	0x81c1,	0x151c,	0x81b8,	0x14ea,	0x81b0,	0x14b9,	0x81a8,	0x1487,
	0x81a0,	0x1455,	0x8198,	0x1424,	0x8190,	0x13f2,	0x8188,	0x13c1,
	0x8181,	0x138f,	0x8179,	0x135d,	0x8172,	0x132b,	0x816a,	0x12fa,
	0x8163,	0x12c8,	0x815b,	0x1296,	0x8154,	0x1265,	0x814d,	0x1233,
	0x8146,	0x1201,	0x813f,	0x11cf,	0x8138,	0x119e,	0x8131,	0x116c,
	0x812a,	0x113a,	0x8123,	0x1108,	0x811d,	0x10d6,	0x8116,	0x10a4,
	0x8110,	0x1073,	0x8109,	0x1041,	0x8103,	0x100f,	0x80fd,	0xfdd,
	0x80f6,	0xfab,	0x80f0,	0xf79,	0x80ea,	0xf47,	0x80e4,	0xf15,
	0x80de,	0xee4,	0x80d9,	0xeb2,	0x80d3,	0xe80,	0x80cd,	0xe4e,
	0x80c8,	0xe1c,	0x80c2,	0xdea,	0x80bd,	0xdb8,	0x80b7,	0xd86,
	0x80b2,	0xd54,	0x80ad,	0xd22,	0x80a8,	0xcf0,	0x80a3,	0xcbe,
	0x809e,	0xc8c,	0x8099,	0xc5a,	0x8094,	0xc28,	0x808f,	0xbf6,
	0x808b,	0xbc4,	0x8086,	0xb92,	0x8082,	0xb60,	0x807d,	0xb2d,
	0x8079,	0xafb,	0x8075,	0xac9,	0x8070,	0xa97,	0x806c,	0xa65,
	0x8068,	0xa33,	0x8064,	0xa01,	0x8060,	0x9cf,	0x805d,	0x99d,
	0x8059,	0x96b,	0x8055,	0x938,	0x8052,	0x906,	0x804e,	0x8d4,
	0x804b,	0x8a2,	0x8047,	0x870,	0x8044,	0x83e,	0x8041,	0x80c,
	0x803e,	0x7d9,	0x803b,	0x7a7,	0x8038,	0x775,	0x8035,	0x743,
	0x8032,	0x711,	0x802f,	0x6de,	0x802d,	0x6ac,	0x802a,	0x67a,
	0x8027,	0x648,	0x8025,	0x616,	0x8023,	0x5e3,	0x8020,	0x5b1,
	0x801e,	0x57f,	0x801c,	0x54d,	0x801a,	0x51b,	0x8018,	0x4e8,
	0x8016,	0x4b6,	0x8014,	0x484,	0x8013,	0x452,	0x8011,	0x41f,
	0x800f,	0x3ed,	0x800e,	0x3bb,	0x800c,	0x389,	0x800b,	0x356,
	0x800a,	0x324,	0x8009,	0x2f2,	0x8008,	0x2c0,	0x8007,	0x28d,
	0x8006,	0x25b,	0x8005,	0x229,	0x8004,	0x1f7,	0x8003,	0x1c4,
	0x8002,	0x192,	0x8002,	0x160,	0x8001,	0x12e,	0x8001,	0xfb,
	0x8001,	0xc9,	0x8000,	0x97,	0x8000,	0x65,	0x8000,	0x32,
	0x8000,	0x0,	0x8000,	0xffce,	0x8000,	0xff9b,	0x8000,	0xff69,
	0x8001,	0xff37,	0x8001,	0xff05,	0x8001,	0xfed2,	0x8002,	0xfea0,
	0x8002,	0xfe6e,	0x8003,	0xfe3c,	0x8004,	0xfe09,	0x8005,	0xfdd7,
	0x8006,	0xfda5,	0x8007,	0xfd73,	0x8008,	0xfd40,	0x8009,	0xfd0e,
	0x800a,	0xfcdc,	0x800b,	0xfcaa,	0x800c,	0xfc77,	0x800e,	0xfc45,
	0x800f,	0xfc13,	0x8011,	0xfbe1,	0x8013,	0xfbae,	0x8014,	0xfb7c,
	0x8016,	0xfb4a,	0x8018,	0xfb18,	0x801a,	0xfae5,	0x801c,	0xfab3,
	0x801e,	0xfa81,	0x8020,	0xfa4f,	0x8023,	0xfa1d,	0x8025,	0xf9ea,
	0x8027,	0xf9b8,	0x802a,	0xf986,	0x802d,	0xf954,	0x802f,	0xf922,
	0x8032,	0xf8ef,	0x8035,	0xf8bd,	0x8038,	0xf88b,	0x803b,	0xf859,
	0x803e,	0xf827,	0x8041,	0xf7f4,	0x8044,	0xf7c2,	0x8047,	0xf790,
	0x804b,	0xf75e,	0x804e,	0xf72c,	0x8052,	0xf6fa,	0x8055,	0xf6c8,
	0x8059,	0xf695,	0x805d,	0xf663,	0x8060,	0xf631,	0x8064,	0xf5ff,
	0x8068,	0xf5cd,	0x806c,	0xf59b,	0x8070,	0xf569,	0x8075,	0xf537,
	0x8079,	0xf505,	0x807d,	0xf4d3,	0x8082,	0xf4a0,	0x8086,	0xf46e,
	0x808b,	0xf43c,	0x808f,	0xf40a,	0x8094,	0xf3d8,	0x8099,	0xf3a6,
	0x809e,	0xf374,	0x80a3,	0xf342,	0x80a8,	0xf310,	0x80ad,	0xf2de,
	0x80b2,	0xf2ac,	0x80b7,	0xf27a,	0x80bd,	0xf248,	0x80c2,	0xf216,
	0x80c8,	0xf1e4,	0x80cd,	0xf1b2,	0x80d3,	0xf180,	0x80d9,	0xf14e,
	0x80de,	0xf11c,	0x80e4,	0xf0eb,	0x80ea,	0xf0b9,	0x80f0,	0xf087,
	0x80f6,	0xf055,	0x80fd,	0xf023,	0x8103,	0xeff1,	0x8109,	0xefbf,
	0x8110,	0xef8d,	0x8116,	0xef5c,	0x811d,	0xef2a,	0x8123,	0xeef8,
	0x812a,	0xeec6,	0x8131,	0xee94,	0x8138,	0xee62,	0x813f,	0xee31,
	0x8146,	0xedff,	0x814d,	0xedcd,	0x8154,	0xed9b,	0x815b,	0xed6a,
	0x8163,	0xed38,	0x816a,	0xed06,	0x8172,	0xecd5,	0x8179,	0xeca3,
	0x8181,	0xec71,	0x8188,	0xec3f,	0x8190,	0xec0e,	0x8198,	0xebdc,
	0x81a0,	0xebab,	0x81a8,	0xeb79,	0x81b0,	0xeb47,	0x81b8,	0xeb16,
	0x81c1,	0xeae4,	0x81c9,	0xeab3,	0x81d1,	0xea81,	0x81da,	0xea4f,
	0x81e2,	0xea1e,	0x81eb,	0xe9ec,	0x81f4,	0xe9bb,	0x81fd,	0xe989,
	0x8205,	0xe958,	0x820e,	0xe926,	0x8217,	0xe8f5,	0x8220,	0xe8c4,
	0x822a,	0xe892,	0x8233,	0xe861,	0x823c,	0xe82f,	0x8246,	0xe7fe,
	0x824f,	0xe7cd,	0x8259,	0xe79b,	0x8262,	0xe76a,	0x826c,	0xe739,
	0x8276,	0xe707,	0x827f,	0xe6d6,	0x8289,	0xe6a5,	0x8293,	0xe673,
	0x829d,	0xe642,	0x82a8,	0xe611,	0x82b2,	0xe5e0,	0x82bc,	0xe5af,
	0x82c6,	0xe57d,	0x82d1,	0xe54c,	0x82db,	0xe51b,	0x82e6,	0xe4ea,
	0x82f1,	0xe4b9,	0x82fb,	0xe488,	0x8306,	0xe457,	0x8311,	0xe426,
	0x831c,	0xe3f4,	0x8327,	0xe3c3,	0x8332,	0xe392,	0x833e,	0xe361,
	0x8349,	0xe330,	0x8354,	0xe2ff,	0x8360,	0xe2cf,	0x836b,	0xe29e,
	0x8377,	0xe26d,	0x8382,	0xe23c,	0x838e,	0xe20b,	0x839a,	0xe1da,
	0x83a6,	0xe1a9,	0x83b2,	0xe178,	0x83be,	0xe148,	0x83ca,	0xe117,
	0x83d6,	0xe0e6,	0x83e2,	0xe0b5,	0x83ef,	0xe085,	0x83fb,	0xe054,
	0x8407,	0xe023,	0x8414,	0xdff2,	0x8421,	0xdfc2,	0x842d,	0xdf91,
	0x843a,	0xdf61,	0x8447,	0xdf30,	0x8454,	0xdeff,	0x8461,	0xdecf,
	0x846e,	0xde9e,	0x847b,	0xde6e,	0x8488,	0xde3d,	0x8496,	0xde0d,
	0x84a3,	0xdddc,	0x84b0,	0xddac,	0x84be,	0xdd7c,	0x84cc,	0xdd4b,
	0x84d9,	0xdd1b,	0x84e7,	0xdcea,	0x84f5,	0xdcba,	0x8503,	0xdc8a,
	0x8511,	0xdc59,	0x851f,	0xdc29,	0x852d,	0xdbf9,	0x853b,	0xdbc9,
	0x8549,	0xdb99,	0x8558,	0xdb68,	0x8566,	0xdb38,	0x8574,	0xdb08,
	0x8583,	0xdad8,	0x8592,	0xdaa8,	0x85a0,	0xda78,	0x85af,	0xda48,
	0x85be,	0xda18,	0x85cd,	0xd9e8,	0x85dc,	0xd9b8,	0x85eb,	0xd988,
	0x85fa,	0xd958,	0x8609,	0xd928,	0x8619,	0xd8f8,	0x8628,	0xd8c8,
	0x8637,	0xd898,	0x8647,	0xd869,	0x8656,	0xd839,	0x8666,	0xd809,
	0x8676,	0xd7d9,	0x8686,	0xd7aa,	0x8696,	0xd77a,	0x86a5,	0xd74a,
	0x86b6,	0xd71b,	0x86c6,	0xd6eb,	0x86d6,	0xd6bb,	0x86e6,	0xd68c,
	0x86f6,	0xd65c,	0x8707,	0xd62d,	0x8717,	0xd5fd,	0x8728,	0xd5ce,
	0x8738,	0xd59e,	0x8749,	0xd56f,	0x875a,	0xd53f,	0x876b,	0xd510,
	0x877b,	0xd4e1,	0x878c,	0xd4b1,	0x879d,	0xd482,	0x87af,	0xd453,
	0x87c0,	0xd424,	0x87d1,	0xd3f4,	0x87e2,	0xd3c5,	0x87f4,	0xd396,
	0x8805,	0xd367,	0x8817,	0xd338,	0x8828,	0xd309,	0x883a,	0xd2da,
	0x884c,	0xd2ab,	0x885e,	0xd27c,	0x8870,	0xd24d,	0x8882,	0xd21e,
	0x8894,	0xd1ef,	0x88a6,	0xd1c0,	0x88b8,	0xd191,	0x88ca,	0xd162,
	0x88dd,	0xd134,	0x88ef,	0xd105,	0x8902,	0xd0d6,	0x8914,	0xd0a7,
	0x8927,	0xd079,	0x8939,	0xd04a,	0x894c,	0xd01b,	0x895f,	0xcfed,
	0x8972,	0xcfbe,	0x8985,	0xcf90,	0x8998,	0xcf61,	0x89ab,	0xcf33,
	0x89be,	0xcf04,	0x89d2,	0xced6,	0x89e5,	0xcea7,	0x89f8,	0xce79,
	0x8a0c,	0xce4b,	0x8a1f,	0xce1c,	0x8a33,	0xcdee,	0x8a47,	0xcdc0,
	0x8a5a,	0xcd92,	0x8a6e,	0xcd63,	0x8a82,	0xcd35,	0x8a96,	0xcd07,
	0x8aaa,	0xccd9,	0x8abe,	0xccab,	0x8ad3,	0xcc7d,	0x8ae7,	0xcc4f,
	0x8afb,	0xcc21,	0x8b10,	0xcbf3,	0x8b24,	0xcbc5,	0x8b39,	0xcb97,
	0x8b4d,	0xcb69,	0x8b62,	0xcb3c,	0x8b77,	0xcb0e,	0x8b8b,	0xcae0,
	0x8ba0,	0xcab2,	0x8bb5,	0xca85,	0x8bca,	0xca57,	0x8bdf,	0xca29,
	0x8bf5,	0xc9fc,	0x8c0a,	0xc9ce,	0x8c1f,	0xc9a1,	0x8c35,	0xc973,
	0x8c4a,	0xc946,	0x8c60,	0xc918,	0x8c75,	0xc8eb,	0x8c8b,	0xc8be,
	0x8ca1,	0xc890,	0x8cb6,	0xc863,	0x8ccc,	0xc836,	0x8ce2,	0xc809,
	0x8cf8,	0xc7db,	0x8d0e,	0xc7ae,	0x8d24,	0xc781,	0x8d3b,	0xc754,
	0x8d51,	0xc727,	0x8d67,	0xc6fa,	0x8d7e,	0xc6cd,	0x8d94,	0xc6a0,
	0x8dab,	0xc673,	0x8dc1,	0xc646,	0x8dd8,	0xc619,	0x8def,	0xc5ed,
	0x8e06,	0xc5c0,	0x8e1d,	0xc593,	0x8e34,	0xc566,	0x8e4b,	0xc53a,
	0x8e62,	0xc50d,	0x8e79,	0xc4e0,	0x8e90,	0xc4b4,	0x8ea8,	0xc487,
	0x8ebf,	0xc45b,	0x8ed6,	0xc42e,	0x8eee,	0xc402,	0x8f06,	0xc3d6,
	0x8f1d,	0xc3a9,	0x8f35,	0xc37d,	0x8f4d,	0xc351,	0x8f65,	0xc324,
	0x8f7d,	0xc2f8,	0x8f95,	0xc2cc,	0x8fad,	0xc2a0,	0x8fc5,	0xc274,
	0x8fdd,	0xc248,	0x8ff5,	0xc21c,	0x900e,	0xc1f0,	0x9026,	0xc1c4,
	0x903e,	0xc198,	0x9057,	0xc16c,	0x9070,	0xc140,	0x9088,	0xc114,
	0x90a1,	0xc0e9,	0x90ba,	0xc0bd,	0x90d3,	0xc091,	0x90ec,	0xc066,
	0x9105,	0xc03a,	0x911e,	0xc00f,	0x9137,	0xbfe3,	0x9150,	0xbfb8,
	0x9169,	0xbf8c,	0x9183,	0xbf61,	0x919c,	0xbf35,	0x91b6,	0xbf0a,
	0x91cf,	0xbedf,	0x91e9,	0xbeb3,	0x9202,	0xbe88,	0x921c,	0xbe5d,
	0x9236,	0xbe32,	0x9250,	0xbe07,	0x926a,	0xbddc,	0x9284,	0xbdb1,
	0x929e,	0xbd86,	0x92b8,	0xbd5b,	0x92d2,	0xbd30,	0x92ec,	0xbd05,
	0x9307,	0xbcda,	0x9321,	0xbcaf,	0x933c,	0xbc85,	0x9356,	0xbc5a,
	0x9371,	0xbc2f,	0x938b,	0xbc05,	0x93a6,	0xbbda,	0x93c1,	0xbbb0,
	0x93dc,	0xbb85,	0x93f7,	0xbb5b,	0x9412,	0xbb30,	0x942d,	0xbb06,
	0x9448,	0xbadc,	0x9463,	0xbab1,	0x947e,	0xba87,	0x949a,	0xba5d,
	0x94b5,	0xba33,	0x94d0,	0xba09,	0x94ec,	0xb9df,	0x9508,	0xb9b5,
	0x9523,	0xb98b,	0x953f,	0xb961,	0x955b,	0xb937,	0x9577,	0xb90d,
	0x9592,	0xb8e3,	0x95ae,	0xb8b9,	0x95ca,	0xb890,	0x95e6,	0xb866,
	0x9603,	0xb83c,	0x961f,	0xb813,	0x963b,	0xb7e9,	0x9657,	0xb7c0,
	0x9674,	0xb796,	0x9690,	0xb76d,	0x96ad,	0xb743,	0x96c9,	0xb71a,
	0x96e6,	0xb6f1,	0x9703,	0xb6c7,	0x9720,	0xb69e,	0x973c,	0xb675,
	0x9759,	0xb64c,	0x9776,	0xb623,	0x9793,	0xb5fa,	0x97b0,	0xb5d1,
	0x97ce,	0xb5a8,	0x97eb,	0xb57f,	0x9808,	0xb556,	0x9826,	0xb52d,
	0x9843,	0xb505,	0x9860,	0xb4dc,	0x987e,	0xb4b3,	0x989c,	0xb48b,
	0x98b9,	0xb462,	0x98d7,	0xb439,	0x98f5,	0xb411,	0x9913,	0xb3e9,
	0x9930,	0xb3c0,	0x994e,	0xb398,	0x996d,	0xb36f,	0x998b,	0xb347,
	0x99a9,	0xb31f,	0x99c7,	0xb2f7,	0x99e5,	0xb2cf,	0x9a04,	0xb2a7,
	0x9a22,	0xb27f,	0x9a40,	0xb257,	0x9a5f,	0xb22f,	0x9a7e,	0xb207,
	0x9a9c,	0xb1df,	0x9abb,	0xb1b7,	0x9ada,	0xb18f,	0x9af9,	0xb168,
	0x9b17,	0xb140,	0x9b36,	0xb118,	0x9b55,	0xb0f1,	0x9b75,	0xb0c9,
	0x9b94,	0xb0a2,	0x9bb3,	0xb07b,	0x9bd2,	0xb053,	0x9bf1,	0xb02c,
	0x9c11,	0xb005,	0x9c30,	0xafdd,	0x9c50,	0xafb6,	0x9c6f,	0xaf8f,
	0x9c8f,	0xaf68,	0x9caf,	0xaf41,	0x9cce,	0xaf1a,	0x9cee,	0xaef3,
	0x9d0e,	0xaecc,	0x9d2e,	0xaea5,	0x9d4e,	0xae7f,	0x9d6e,	0xae58,
	0x9d8e,	0xae31,	0x9dae,	0xae0b,	0x9dce,	0xade4,	0x9def,	0xadbd,
	0x9e0f,	0xad97,	0x9e2f,	0xad70,	0x9e50,	0xad4a,	0x9e70,	0xad24,
	0x9e91,	0xacfd,	0x9eb2,	0xacd7,	0x9ed2,	0xacb1,	0x9ef3,	0xac8b,
	0x9f14,	0xac65,	0x9f35,	0xac3f,	0x9f56,	0xac19,	0x9f77,	0xabf3,
	0x9f98,	0xabcd,	0x9fb9,	0xaba7,	0x9fda,	0xab81,	0x9ffb,	0xab5c,
	0xa01c,	0xab36,	0xa03e,	0xab10,	0xa05f,	0xaaeb,	0xa080,	0xaac5,
	0xa0a2,	0xaaa0,	0xa0c4,	0xaa7a,	0xa0e5,	0xaa55,	0xa107,	0xaa30,
	0xa129,	0xaa0a,	0xa14a,	0xa9e5,	0xa16c,	0xa9c0,	0xa18e,	0xa99b,
	0xa1b0,	0xa976,	0xa1d2,	0xa951,	0xa1f4,	0xa92c,	0xa216,	0xa907,
	0xa238,	0xa8e2,	0xa25b,	0xa8bd,	0xa27d,	0xa899,	0xa29f,	0xa874,
	0xa2c2,	0xa84f,	0xa2e4,	0xa82b,	0xa307,	0xa806,	0xa329,	0xa7e2,
	0xa34c,	0xa7bd,	0xa36f,	0xa799,	0xa391,	0xa774,	0xa3b4,	0xa750,
	0xa3d7,	0xa72c,	0xa3fa,	0xa708,	0xa41d,	0xa6e4,	0xa440,	0xa6c0,
	0xa463,	0xa69c,	0xa486,	0xa678,	0xa4a9,	0xa654,	0xa4cc,	0xa630,
	0xa4f0,	0xa60c,	0xa513,	0xa5e8,	0xa537,	0xa5c5,	0xa55a,	0xa5a1,
	0xa57e,	0xa57e,	0xa5a1,	0xa55a,	0xa5c5,	0xa537,	0xa5e8,	0xa513,
	0xa60c,	0xa4f0,	0xa630,	0xa4cc,	0xa654,	0xa4a9,	0xa678,	0xa486,
	0xa69c,	0xa463,	0xa6c0,	0xa440,	0xa6e4,	0xa41d,	0xa708,	0xa3fa,
	0xa72c,	0xa3d7,	0xa750,	0xa3b4,	0xa774,	0xa391,	0xa799,	0xa36f,
	0xa7bd,	0xa34c,	0xa7e2,	0xa329,	0xa806,	0xa307,	0xa82b,	0xa2e4,
	0xa84f,	0xa2c2,	0xa874,	0xa29f,	0xa899,	0xa27d,	0xa8bd,	0xa25b,
	0xa8e2,	0xa238,	0xa907,	0xa216,	0xa92c,	0xa1f4,	0xa951,	0xa1d2,
	0xa976,	0xa1b0,	0xa99b,	0xa18e,	0xa9c0,	0xa16c,	0xa9e5,	0xa14a,
	0xaa0a,	0xa129,	0xaa30,	0xa107,	0xaa55,	0xa0e5,	0xaa7a,	0xa0c4,
	0xaaa0,	0xa0a2,	0xaac5,	0xa080,	0xaaeb,	0xa05f,	0xab10,	0xa03e,
	0xab36,	0xa01c,	0xab5c,	0x9ffb,	0xab81,	0x9fda,	0xaba7,	0x9fb9,
	0xabcd,	0x9f98,	0xabf3,	0x9f77,	0xac19,	0x9f56,	0xac3f,	0x9f35,
	0xac65,	0x9f14,	0xac8b,	0x9ef3,	0xacb1,	0x9ed2,	0xacd7,	0x9eb2,
	0xacfd,	0x9e91,	0xad24,	0x9e70,	0xad4a,	0x9e50,	0xad70,	0x9e2f,
	0xad97,	0x9e0f,	0xadbd,	0x9def,	0xade4,	0x9dce,	0xae0b,	0x9dae,
	0xae31,	0x9d8e,	0xae58,	0x9d6e,	0xae7f,	0x9d4e,	0xaea5,	0x9d2e,
	0xaecc,	0x9d0e,	0xaef3,	0x9cee,	0xaf1a,	0x9cce,	0xaf41,	0x9caf,
	0xaf68,	0x9c8f,	0xaf8f,	0x9c6f,	0xafb6,	0x9c50,	0xafdd,	0x9c30,
	0xb005,	0x9c11,	0xb02c,	0x9bf1,	0xb053,	0x9bd2,	0xb07b,	0x9bb3,
	0xb0a2,	0x9b94,	0xb0c9,	0x9b75,	0xb0f1,	0x9b55,	0xb118,	0x9b36,
	0xb140,	0x9b17,	0xb168,	0x9af9,	0xb18f,	0x9ada,	0xb1b7,	0x9abb,
	0xb1df,	0x9a9c,	0xb207,	0x9a7e,	0xb22f,	0x9a5f,	0xb257,	0x9a40,
	0xb27f,	0x9a22,	0xb2a7,	0x9a04,	0xb2cf,	0x99e5,	0xb2f7,	0x99c7,
	0xb31f,	0x99a9,	0xb347,	0x998b,	0xb36f,	0x996d,	0xb398,	0x994e,
	0xb3c0,	0x9930,	0xb3e9,	0x9913,	0xb411,	0x98f5,	0xb439,	0x98d7,
	0xb462,	0x98b9,	0xb48b,	0x989c,	0xb4b3,	0x987e,	0xb4dc,	0x9860,
	0xb505,	0x9843,	0xb52d,	0x9826,	0xb556,	0x9808,	0xb57f,	0x97eb,
	0xb5a8,	0x97ce,	0xb5d1,	0x97b0,	0xb5fa,	0x9793,	0xb623,	0x9776,
	0xb64c,	0x9759,	0xb675,	0x973c,	0xb69e,	0x9720,	0xb6c7,	0x9703,
	0xb6f1,	0x96e6,	0xb71a,	0x96c9,	0xb743,	0x96ad,	0xb76d,	0x9690,
	0xb796,	0x9674,	0xb7c0,	0x9657,	0xb7e9,	0x963b,	0xb813,	0x961f,
	0xb83c,	0x9603,	0xb866,	0x95e6,	0xb890,	0x95ca,	0xb8b9,	0x95ae,
	0xb8e3,	0x9592,	0xb90d,	0x9577,	0xb937,	0x955b,	0xb961,	0x953f,
	0xb98b,	0x9523,	0xb9b5,	0x9508,	0xb9df,	0x94ec,	0xba09,	0x94d0,
	0xba33,	0x94b5,	0xba5d,	0x949a,	0xba87,	0x947e,	0xbab1,	0x9463,
	0xbadc,	0x9448,	0xbb06,	0x942d,	0xbb30,	0x9412,	0xbb5b,	0x93f7,
	0xbb85,	0x93dc,	0xbbb0,	0x93c1,	0xbbda,	0x93a6,	0xbc05,	0x938b,
	0xbc2f,	0x9371,	0xbc5a,	0x9356,	0xbc85,	0x933c,	0xbcaf,	0x9321,
	0xbcda,	0x9307,	0xbd05,	0x92ec,	0xbd30,	0x92d2,	0xbd5b,	0x92b8,
	0xbd86,	0x929e,	0xbdb1,	0x9284,	0xbddc,	0x926a,	0xbe07,	0x9250,
	0xbe32,	0x9236,	0xbe5d,	0x921c,	0xbe88,	0x9202,	0xbeb3,	0x91e9,
	0xbedf,	0x91cf,	0xbf0a,	0x91b6,	0xbf35,	0x919c,	0xbf61,	0x9183,
	0xbf8c,	0x9169,	0xbfb8,	0x9150,	0xbfe3,	0x9137,	0xc00f,	0x911e,
	0xc03a,	0x9105,	0xc066,	0x90ec,	0xc091,	0x90d3,	0xc0bd,	0x90ba,
	0xc0e9,	0x90a1,	0xc114,	0x9088,	0xc140,	0x9070,	0xc16c,	0x9057,
	0xc198,	0x903e,	0xc1c4,	0x9026,	0xc1f0,	0x900e,	0xc21c,	0x8ff5,
	0xc248,	0x8fdd,	0xc274,	0x8fc5,	0xc2a0,	0x8fad,	0xc2cc,	0x8f95,
	0xc2f8,	0x8f7d,	0xc324,	0x8f65,	0xc351,	0x8f4d,	0xc37d,	0x8f35,
	0xc3a9,	0x8f1d,	0xc3d6,	0x8f06,	0xc402,	0x8eee,	0xc42e,	0x8ed6,
	0xc45b,	0x8ebf,	0xc487,	0x8ea8,	0xc4b4,	0x8e90,	0xc4e0,	0x8e79,
	0xc50d,	0x8e62,	0xc53a,	0x8e4b,	0xc566,	0x8e34,	0xc593,	0x8e1d,
	0xc5c0,	0x8e06,	0xc5ed,	0x8def,	0xc619,	0x8dd8,	0xc646,	0x8dc1,
	0xc673,	0x8dab,	0xc6a0,	0x8d94,	0xc6cd,	0x8d7e,	0xc6fa,	0x8d67,
	0xc727,	0x8d51,	0xc754,	0x8d3b,	0xc781,	0x8d24,	0xc7ae,	0x8d0e,
	0xc7db,	0x8cf8,	0xc809,	0x8ce2,	0xc836,	0x8ccc,	0xc863,	0x8cb6,
	0xc890,	0x8ca1,	0xc8be,	0x8c8b,	0xc8eb,	0x8c75,	0xc918,	0x8c60,
	0xc946,	0x8c4a,	0xc973,	0x8c35,	0xc9a1,	0x8c1f,	0xc9ce,	0x8c0a,
	0xc9fc,	0x8bf5,	0xca29,	0x8bdf,	0xca57,	0x8bca,	0xca85,	0x8bb5,
	0xcab2,	0x8ba0,	0xcae0,	0x8b8b,	0xcb0e,	0x8b77,	0xcb3c,	0x8b62,
	0xcb69,	0x8b4d,	0xcb97,	0x8b39,	0xcbc5,	0x8b24,	0xcbf3,	0x8b10,
	0xcc21,	0x8afb,	0xcc4f,	0x8ae7,	0xcc7d,	0x8ad3,	0xccab,	0x8abe,
	0xccd9,	0x8aaa,	0xcd07,	0x8a96,	0xcd35,	0x8a82,	0xcd63,	0x8a6e,
	0xcd92,	0x8a5a,	0xcdc0,	0x8a47,	0xcdee,	0x8a33,	0xce1c,	0x8a1f,
	0xce4b,	0x8a0c,	0xce79,	0x89f8,	0xcea7,	0x89e5,	0xced6,	0x89d2,
	0xcf04,	0x89be,	0xcf33,	0x89ab,	0xcf61,	0x8998,	0xcf90,	0x8985,
	0xcfbe,	0x8972,	0xcfed,	0x895f,	0xd01b,	0x894c,	0xd04a,	0x8939,
	0xd079,	0x8927,	0xd0a7,	0x8914,	0xd0d6,	0x8902,	0xd105,	0x88ef,
	0xd134,	0x88dd,	0xd162,	0x88ca,	0xd191,	0x88b8,	0xd1c0,	0x88a6,
	0xd1ef,	0x8894,	0xd21e,	0x8882,	0xd24d,	0x8870,	0xd27c,	0x885e,
	0xd2ab,	0x884c,	0xd2da,	0x883a,	0xd309,	0x8828,	0xd338,	0x8817,
	0xd367,	0x8805,	0xd396,	0x87f4,	0xd3c5,	0x87e2,	0xd3f4,	0x87d1,
	0xd424,	0x87c0,	0xd453,	0x87af,	0xd482,	0x879d,	0xd4b1,	0x878c,
	0xd4e1,	0x877b,	0xd510,	0x876b,	0xd53f,	0x875a,	0xd56f,	0x8749,
	0xd59e,	0x8738,	0xd5ce,	0x8728,	0xd5fd,	0x8717,	0xd62d,	0x8707,
	0xd65c,	0x86f6,	0xd68c,	0x86e6,	0xd6bb,	0x86d6,	0xd6eb,	0x86c6,
	0xd71b,	0x86b6,	0xd74a,	0x86a5,	0xd77a,	0x8696,	0xd7aa,	0x8686,
	0xd7d9,	0x8676,	0xd809,	0x8666,	0xd839,	0x8656,	0xd869,	0x8647,
	0xd898,	0x8637,	0xd8c8,	0x8628,	0xd8f8,	0x8619,	0xd928,	0x8609,
	0xd958,	0x85fa,	0xd988,	0x85eb,	0xd9b8,	0x85dc,	0xd9e8,	0x85cd,
	0xda18,	0x85be,	0xda48,	0x85af,	0xda78,	0x85a0,	0xdaa8,	0x8592,
	0xdad8,	0x8583,	0xdb08,	0x8574,	0xdb38,	0x8566,	0xdb68,	0x8558,
	0xdb99,	0x8549,	0xdbc9,	0x853b,	0xdbf9,	0x852d,	0xdc29,	0x851f,
	0xdc59,	0x8511,	0xdc8a,	0x8503,	0xdcba,	0x84f5,	0xdcea,	0x84e7,
	0xdd1b,	0x84d9,	0xdd4b,	0x84cc,	0xdd7c,	0x84be,	0xddac,	0x84b0,
	0xdddc,	0x84a3,	0xde0d,	0x8496,	0xde3d,	0x8488,	0xde6e,	0x847b,
	0xde9e,	0x846e,	0xdecf,	0x8461,	0xdeff,	0x8454,	0xdf30,	0x8447,
	0xdf61,	0x843a,	0xdf91,	0x842d,	0xdfc2,	0x8421,	0xdff2,	0x8414,
	0xe023,	0x8407,	0xe054,	0x83fb,	0xe085,	0x83ef,	0xe0b5,	0x83e2,
	0xe0e6,	0x83d6,	0xe117,	0x83ca,	0xe148,	0x83be,	0xe178,	0x83b2,
	0xe1a9,	0x83a6,	0xe1da,	0x839a,	0xe20b,	0x838e,	0xe23c,	0x8382,
	0xe26d,	0x8377,	0xe29e,	0x836b,	0xe2cf,	0x8360,	0xe2ff,	0x8354,
	0xe330,	0x8349,	0xe361,	0x833e,	0xe392,	0x8332,	0xe3c3,	0x8327,
	0xe3f4,	0x831c,	0xe426,	0x8311,	0xe457,	0x8306,	0xe488,	0x82fb,
	0xe4b9,	0x82f1,	0xe4ea,	0x82e6,	0xe51b,	0x82db,	0xe54c,	0x82d1,
	0xe57d,	0x82c6,	0xe5af,	0x82bc,	0xe5e0,	0x82b2,	0xe611,	0x82a8,
	0xe642,	0x829d,	0xe673,	0x8293,	0xe6a5,	0x8289,	0xe6d6,	0x827f,
	0xe707,	0x8276,	0xe739,	0x826c,	0xe76a,	0x8262,	0xe79b,	0x8259,
	0xe7cd,	0x824f,	0xe7fe,	0x8246,	0xe82f,	0x823c,	0xe861,	0x8233,
	0xe892,	0x822a,	0xe8c4,	0x8220,	0xe8f5,	0x8217,	0xe926,	0x820e,
	0xe958,	0x8205,	0xe989,	0x81fd,	0xe9bb,	0x81f4,	0xe9ec,	0x81eb,
	0xea1e,	0x81e2,	0xea4f,	0x81da,	0xea81,	0x81d1,	0xeab3,	0x81c9,
	0xeae4,	0x81c1,	0xeb16,	0x81b8,	0xeb47,	0x81b0,	0xeb79,	0x81a8,
	0xebab,	0x81a0,	0xebdc,	0x8198,	0xec0e,	0x8190,	0xec3f,	0x8188,
	0xec71,	0x8181,	0xeca3,	0x8179,	0xecd5,	0x8172,	0xed06,	0x816a,
	0xed38,	0x8163,	0xed6a,	0x815b,	0xed9b,	0x8154,	0xedcd,	0x814d,
	0xedff,	0x8146,	0xee31,	0x813f,	0xee62,	0x8138,	0xee94,	0x8131,
	0xeec6,	0x812a,	0xeef8,	0x8123,	0xef2a,	0x811d,	0xef5c,	0x8116,
	0xef8d,	0x8110,	0xefbf,	0x8109,	0xeff1,	0x8103,	0xf023,	0x80fd,
	0xf055,	0x80f6,	0xf087,	0x80f0,	0xf0b9,	0x80ea,	0xf0eb,	0x80e4,
	0xf11c,	0x80de,	0xf14e,	0x80d9,	0xf180,	0x80d3,	0xf1b2,	0x80cd,
	0xf1e4,	0x80c8,	0xf216,	0x80c2,	0xf248,	0x80bd,	0xf27a,	0x80b7,
	0xf2ac,	0x80b2,	0xf2de,	0x80ad,	0xf310,	0x80a8,	0xf342,	0x80a3,
	0xf374,	0x809e,	0xf3a6,	0x8099,	0xf3d8,	0x8094,	0xf40a,	0x808f,
	0xf43c,	0x808b,	0xf46e,	0x8086,	0xf4a0,	0x8082,	0xf4d3,	0x807d,
	0xf505,	0x8079,	0xf537,	0x8075,	0xf569,	0x8070,	0xf59b,	0x806c,
	0xf5cd,	0x8068,	0xf5ff,	0x8064,	0xf631,	0x8060,	0xf663,	0x805d,
	0xf695,	0x8059,	0xf6c8,	0x8055,	0xf6fa,	0x8052,	0xf72c,	0x804e,
	0xf75e,	0x804b,	0xf790,	0x8047,	0xf7c2,	0x8044,	0xf7f4,	0x8041,
	0xf827,	0x803e,	0xf859,	0x803b,	0xf88b,	0x8038,	0xf8bd,	0x8035,
	0xf8ef,	0x8032,	0xf922,	0x802f,	0xf954,	0x802d,	0xf986,	0x802a,
	0xf9b8,	0x8027,	0xf9ea,	0x8025,	0xfa1d,	0x8023,	0xfa4f,	0x8020,
	0xfa81,	0x801e,	0xfab3,	0x801c,	0xfae5,	0x801a,	0xfb18,	0x8018,
	0xfb4a,	0x8016,	0xfb7c,	0x8014,	0xfbae,	0x8013,	0xfbe1,	0x8011,
	0xfc13,	0x800f,	0xfc45,	0x800e,	0xfc77,	0x800c,	0xfcaa,	0x800b,
	0xfcdc,	0x800a,	0xfd0e,	0x8009,	0xfd40,	0x8008,	0xfd73,	0x8007,
	0xfda5,	0x8006,	0xfdd7,	0x8005,	0xfe09,	0x8004,	0xfe3c,	0x8003,
	0xfe6e,	0x8002,	0xfea0,	0x8002,	0xfed2,	0x8001,	0xff05,	0x8001,
	0xff37,	0x8001,	0xff69,	0x8000,	0xff9b,	0x8000,	0xffce,	0x8000,
	0x0,	0x8000,	0x32,	0x8000,	0x65,	0x8000,	0x97,	0x8000,
	0xc9,	0x8001,	0xfb,	0x8001,	0x12e,	0x8001,	0x160,	0x8002,
	0x192,	0x8002,	0x1c4,	0x8003,	0x1f7,	0x8004,	0x229,	0x8005,
	0x25b,	0x8006,	0x28d,	0x8007,	0x2c0,	0x8008,	0x2f2,	0x8009,
	0x324,	0x800a,	0x356,	0x800b,	0x389,	0x800c,	0x3bb,	0x800e,
	0x3ed,	0x800f,	0x41f,	0x8011,	0x452,	0x8013,	0x484,	0x8014,
	0x4b6,	0x8016,	0x4e8,	0x8018,	0x51b,	0x801a,	0x54d,	0x801c,
	0x57f,	0x801e,	0x5b1,	0x8020,	0x5e3,	0x8023,	0x616,	0x8025,
	0x648,	0x8027,	0x67a,	0x802a,	0x6ac,	0x802d,	0x6de,	0x802f,
	0x711,	0x8032,	0x743,	0x8035,	0x775,	0x8038,	0x7a7,	0x803b,
	0x7d9,	0x803e,	0x80c,	0x8041,	0x83e,	0x8044,	0x870,	0x8047,
	0x8a2,	0x804b,	0x8d4,	0x804e,	0x906,	0x8052,	0x938,	0x8055,
	0x96b,	0x8059,	0x99d,	0x805d,	0x9cf,	0x8060,	0xa01,	0x8064,
	0xa33,	0x8068,	0xa65,	0x806c,	0xa97,	0x8070,	0xac9,	0x8075,
	0xafb,	0x8079,	0xb2d,	0x807d,	0xb60,	0x8082,	0xb92,	0x8086,
	0xbc4,	0x808b,	0xbf6,	0x808f,	0xc28,	0x8094,	0xc5a,	0x8099,
	0xc8c,	0x809e,	0xcbe,	0x80a3,	0xcf0,	0x80a8,	0xd22,	0x80ad,
	0xd54,	0x80b2,	0xd86,	0x80b7,	0xdb8,	0x80bd,	0xdea,	0x80c2,
	0xe1c,	0x80c8,	0xe4e,	0x80cd,	0xe80,	0x80d3,	0xeb2,	0x80d9,
	0xee4,	0x80de,	0xf15,	0x80e4,	0xf47,	0x80ea,	0xf79,	0x80f0,
	0xfab,	0x80f6,	0xfdd,	0x80fd,	0x100f,	0x8103,	0x1041,	0x8109,
	0x1073,	0x8110,	0x10a4,	0x8116,	0x10d6,	0x811d,	0x1108,	0x8123,
	0x113a,	0x812a,	0x116c,	0x8131,	0x119e,	0x8138,	0x11cf,	0x813f,
	0x1201,	0x8146,	0x1233,	0x814d,	0x1265,	0x8154,	0x1296,	0x815b,
	0x12c8,	0x8163,	0x12fa,	0x816a,	0x132b,	0x8172,	0x135d,	0x8179,
	0x138f,	0x8181,	0x13c1,	0x8188,	0x13f2,	0x8190,	0x1424,	0x8198,
	0x1455,	0x81a0,	0x1487,	0x81a8,	0x14b9,	0x81b0,	0x14ea,	0x81b8,
	0x151c,	0x81c1,	0x154d,	0x81c9,	0x157f,	0x81d1,	0x15b1,	0x81da,
	0x15e2,	0x81e2,	0x1614,	0x81eb,	0x1645,	0x81f4,	0x1677,	0x81fd,
	0x16a8,	0x8205,	0x16da,	0x820e,	0x170b,	0x8217,	0x173c,	0x8220,
	0x176e,	0x822a,	0x179f,	0x8233,	0x17d1,	0x823c,	0x1802,	0x8246,
	0x1833,	0x824f,	0x1865,	0x8259,	0x1896,	0x8262,	0x18c7,	0x826c,
	0x18f9,	0x8276,	0x192a,	0x827f,	0x195b,	0x8289,	0x198d,	0x8293,
	0x19be,	0x829d,	0x19ef,	0x82a8,	0x1a20,	0x82b2,	0x1a51,	0x82bc,
	0x1a83,	0x82c6,	0x1ab4,	0x82d1,	0x1ae5,	0x82db,	0x1b16,	0x82e6,
	0x1b47,	0x82f1,	0x1b78,	0x82fb,	0x1ba9,	0x8306,	0x1bda,	0x8311,
	0x1c0c,	0x831c,	0x1c3d,	0x8327,	0x1c6e,	0x8332,	0x1c9f,	0x833e,
	0x1cd0,	0x8349,	0x1d01,	0x8354,	0x1d31,	0x8360,	0x1d62,	0x836b,
	0x1d93,	0x8377,	0x1dc4,	0x8382,	0x1df5,	0x838e,	0x1e26,	0x839a,
	0x1e57,	0x83a6,	0x1e88,	0x83b2,	0x1eb8,	0x83be,	0x1ee9,	0x83ca,
	0x1f1a,	0x83d6,	0x1f4b,	0x83e2,	0x1f7b,	0x83ef,	0x1fac,	0x83fb,
	0x1fdd,	0x8407,	0x200e,	0x8414,	0x203e,	0x8421,	0x206f,	0x842d,
	0x209f,	0x843a,	0x20d0,	0x8447,	0x2101,	0x8454,	0x2131,	0x8461,
	0x2162,	0x846e,	0x2192,	0x847b,	0x21c3,	0x8488,	0x21f3,	0x8496,
	0x2224,	0x84a3,	0x2254,	0x84b0,	0x2284,	0x84be,	0x22b5,	0x84cc,
	0x22e5,	0x84d9,	0x2316,	0x84e7,	0x2346,	0x84f5,	0x2376,	0x8503,
	0x23a7,	0x8511,	0x23d7,	0x851f,	0x2407,	0x852d,	0x2437,	0x853b,
	0x2467,	0x8549,	0x2498,	0x8558,	0x24c8,	0x8566,	0x24f8,	0x8574,
	0x2528,	0x8583,	0x2558,	0x8592,	0x2588,	0x85a0,	0x25b8,	0x85af,
	0x25e8,	0x85be,	0x2618,	0x85cd,	0x2648,	0x85dc,	0x2678,	0x85eb,
	0x26a8,	0x85fa,	0x26d8,	0x8609,	0x2708,	0x8619,	0x2738,	0x8628,
	0x2768,	0x8637,	0x2797,	0x8647,	0x27c7,	0x8656,	0x27f7,	0x8666,
	0x2827,	0x8676,	0x2856,	0x8686,	0x2886,	0x8696,	0x28b6,	0x86a5,
	0x28e5,	0x86b6,	0x2915,	0x86c6,	0x2945,	0x86d6,	0x2974,	0x86e6,
	0x29a4,	0x86f6,	0x29d3,	0x8707,	0x2a03,	0x8717,	0x2a32,	0x8728,
	0x2a62,	0x8738,	0x2a91,	0x8749,	0x2ac1,	0x875a,	0x2af0,	0x876b,
	0x2b1f,	0x877b,	0x2b4f,	0x878c,	0x2b7e,	0x879d,	0x2bad,	0x87af,
	0x2bdc,	0x87c0,	0x2c0c,	0x87d1,	0x2c3b,	0x87e2,	0x2c6a,	0x87f4,
	0x2c99,	0x8805,	0x2cc8,	0x8817,	0x2cf7,	0x8828,	0x2d26,	0x883a,
	0x2d55,	0x884c,	0x2d84,	0x885e,	0x2db3,	0x8870,	0x2de2,	0x8882,
	0x2e11,	0x8894,	0x2e40,	0x88a6,	0x2e6f,	0x88b8,	0x2e9e,	0x88ca,
	0x2ecc,	0x88dd,	0x2efb,	0x88ef,	0x2f2a,	0x8902,	0x2f59,	0x8914,
	0x2f87,	0x8927,	0x2fb6,	0x8939,	0x2fe5,	0x894c,	0x3013,	0x895f,
	0x3042,	0x8972,	0x3070,	0x8985,	0x309f,	0x8998,	0x30cd,	0x89ab,
	0x30fc,	0x89be,	0x312a,	0x89d2,	0x3159,	0x89e5,	0x3187,	0x89f8,
	0x31b5,	0x8a0c,	0x31e4,	0x8a1f,	0x3212,	0x8a33,	0x3240,	0x8a47,
	0x326e,	0x8a5a,	0x329d,	0x8a6e,	0x32cb,	0x8a82,	0x32f9,	0x8a96,
	0x3327,	0x8aaa,	0x3355,	0x8abe,	0x3383,	0x8ad3,	0x33b1,	0x8ae7,
	0x33df,	0x8afb,	0x340d,	0x8b10,	0x343b,	0x8b24,	0x3469,	0x8b39,
	0x3497,	0x8b4d,	0x34c4,	0x8b62,	0x34f2,	0x8b77,	0x3520,	0x8b8b,
	0x354e,	0x8ba0,	0x357b,	0x8bb5,	0x35a9,	0x8bca,	0x35d7,	0x8bdf,
	0x3604,	0x8bf5,	0x3632,	0x8c0a,	0x365f,	0x8c1f,	0x368d,	0x8c35,
	0x36ba,	0x8c4a,	0x36e8,	0x8c60,	0x3715,	0x8c75,	0x3742,	0x8c8b,
	0x3770,	0x8ca1,	0x379d,	0x8cb6,	0x37ca,	0x8ccc,	0x37f7,	0x8ce2,
	0x3825,	0x8cf8,	0x3852,	0x8d0e,	0x387f,	0x8d24,	0x38ac,	0x8d3b,
	0x38d9,	0x8d51,	0x3906,	0x8d67,	0x3933,	0x8d7e,	0x3960,	0x8d94,
	0x398d,	0x8dab,	0x39ba,	0x8dc1,	0x39e7,	0x8dd8,	0x3a13,	0x8def,
	0x3a40,	0x8e06,	0x3a6d,	0x8e1d,	0x3a9a,	0x8e34,	0x3ac6,	0x8e4b,
	0x3af3,	0x8e62,	0x3b20,	0x8e79,	0x3b4c,	0x8e90,	0x3b79,	0x8ea8,
	0x3ba5,	0x8ebf,	0x3bd2,	0x8ed6,	0x3bfe,	0x8eee,	0x3c2a,	0x8f06,
	0x3c57,	0x8f1d,	0x3c83,	0x8f35,	0x3caf,	0x8f4d,	0x3cdc,	0x8f65,
	0x3d08,	0x8f7d,	0x3d34,	0x8f95,	0x3d60,	0x8fad,	0x3d8c,	0x8fc5,
	0x3db8,	0x8fdd,	0x3de4,	0x8ff5,	0x3e10,	0x900e,	0x3e3c,	0x9026,
	0x3e68,	0x903e,	0x3e94,	0x9057,	0x3ec0,	0x9070,	0x3eec,	0x9088,
	0x3f17,	0x90a1,	0x3f43,	0x90ba,	0x3f6f,	0x90d3,	0x3f9a,	0x90ec,
	0x3fc6,	0x9105,	0x3ff1,	0x911e,	0x401d,	0x9137,	0x4048,	0x9150,
	0x4074,	0x9169,	0x409f,	0x9183,	0x40cb,	0x919c,	0x40f6,	0x91b6,
	0x4121,	0x91cf,	0x414d,	0x91e9,	0x4178,	0x9202,	0x41a3,	0x921c,
	0x41ce,	0x9236,	0x41f9,	0x9250,	0x4224,	0x926a,	0x424f,	0x9284,
	0x427a,	0x929e,	0x42a5,	0x92b8,	0x42d0,	0x92d2,	0x42fb,	0x92ec,
	0x4326,	0x9307,	0x4351,	0x9321,	0x437b,	0x933c,	0x43a6,	0x9356,
	0x43d1,	0x9371,	0x43fb,	0x938b,	0x4426,	0x93a6,	0x4450,	0x93c1,
	0x447b,	0x93dc,	0x44a5,	0x93f7,	0x44d0,	0x9412,	0x44fa,	0x942d,
	0x4524,	0x9448,	0x454f,	0x9463,	0x4579,	0x947e,	0x45a3,	0x949a,
	0x45cd,	0x94b5,	0x45f7,	0x94d0,	0x4621,	0x94ec,	0x464b,	0x9508,
	0x4675,	0x9523,	0x469f,	0x953f,	0x46c9,	0x955b,	0x46f3,	0x9577,
	0x471d,	0x9592,	0x4747,	0x95ae,	0x4770,	0x95ca,	0x479a,	0x95e6,
	0x47c4,	0x9603,	0x47ed,	0x961f,	0x4817,	0x963b,	0x4840,	0x9657,
	0x486a,	0x9674,	0x4893,	0x9690,	0x48bd,	0x96ad,	0x48e6,	0x96c9,
	0x490f,	0x96e6,	0x4939,	0x9703,	0x4962,	0x9720,	0x498b,	0x973c,
	0x49b4,	0x9759,	0x49dd,	0x9776,	0x4a06,	0x9793,	0x4a2f,	0x97b0,
	0x4a58,	0x97ce,	0x4a81,	0x97eb,	0x4aaa,	0x9808,	0x4ad3,	0x9826,
	0x4afb,	0x9843,	0x4b24,	0x9860,	0x4b4d,	0x987e,	0x4b75,	0x989c,
	0x4b9e,	0x98b9,	0x4bc7,	0x98d7,	0x4bef,	0x98f5,	0x4c17,	0x9913,
	0x4c40,	0x9930,	0x4c68,	0x994e,	0x4c91,	0x996d,	0x4cb9,	0x998b,
	0x4ce1,	0x99a9,	0x4d09,	0x99c7,	0x4d31,	0x99e5,	0x4d59,	0x9a04,
	0x4d81,	0x9a22,	0x4da9,	0x9a40,	0x4dd1,	0x9a5f,	0x4df9,	0x9a7e,
	0x4e21,	0x9a9c,	0x4e49,	0x9abb,	0x4e71,	0x9ada,	0x4e98,	0x9af9,
	0x4ec0,	0x9b17,	0x4ee8,	0x9b36,	0x4f0f,	0x9b55,	0x4f37,	0x9b75,
	0x4f5e,	0x9b94,	0x4f85,	0x9bb3,	0x4fad,	0x9bd2,	0x4fd4,	0x9bf1,
	0x4ffb,	0x9c11,	0x5023,	0x9c30,	0x504a,	0x9c50,	0x5071,	0x9c6f,
	0x5098,	0x9c8f,	0x50bf,	0x9caf,	0x50e6,	0x9cce,	0x510d,	0x9cee,
	0x5134,	0x9d0e,	0x515b,	0x9d2e,	0x5181,	0x9d4e,	0x51a8,	0x9d6e,
	0x51cf,	0x9d8e,	0x51f5,	0x9dae,	0x521c,	0x9dce,	0x5243,	0x9def,
	0x5269,	0x9e0f,	0x5290,	0x9e2f,	0x52b6,	0x9e50,	0x52dc,	0x9e70,
	0x5303,	0x9e91,	0x5329,	0x9eb2,	0x534f,	0x9ed2,	0x5375,	0x9ef3,
	0x539b,	0x9f14,	0x53c1,	0x9f35,	0x53e7,	0x9f56,	0x540d,	0x9f77,
	0x5433,	0x9f98,	0x5459,	0x9fb9,	0x547f,	0x9fda,	0x54a4,	0x9ffb,
	0x54ca,	0xa01c,	0x54f0,	0xa03e,	0x5515,	0xa05f,	0x553b,	0xa080,
	0x5560,	0xa0a2,	0x5586,	0xa0c4,	0x55ab,	0xa0e5,	0x55d0,	0xa107,
	0x55f6,	0xa129,	0x561b,	0xa14a,	0x5640,	0xa16c,	0x5665,	0xa18e,
	0x568a,	0xa1b0,	0x56af,	0xa1d2,	0x56d4,	0xa1f4,	0x56f9,	0xa216,
	0x571e,	0xa238,	0x5743,	0xa25b,	0x5767,	0xa27d,	0x578c,	0xa29f,
	0x57b1,	0xa2c2,	0x57d5,	0xa2e4,	0x57fa,	0xa307,	0x581e,	0xa329,
	0x5843,	0xa34c,	0x5867,	0xa36f,	0x588c,	0xa391,	0x58b0,	0xa3b4,
	0x58d4,	0xa3d7,	0x58f8,	0xa3fa,	0x591c,	0xa41d,	0x5940,	0xa440,
	0x5964,	0xa463,	0x5988,	0xa486,	0x59ac,	0xa4a9,	0x59d0,	0xa4cc,
	0x59f4,	0xa4f0,	0x5a18,	0xa513,	0x5a3b,	0xa537,	0x5a5f,	0xa55a,
	0x5a82,	0xa57e,	0x5aa6,	0xa5a1,	0x5ac9,	0xa5c5,	0x5aed,	0xa5e8,
	0x5b10,	0xa60c,	0x5b34,	0xa630,	0x5b57,	0xa654,	0x5b7a,	0xa678,
	0x5b9d,	0xa69c,	0x5bc0,	0xa6c0,	0x5be3,	0xa6e4,	0x5c06,	0xa708,
	0x5c29,	0xa72c,	0x5c4c,	0xa750,	0x5c6f,	0xa774,	0x5c91,	0xa799,
	0x5cb4,	0xa7bd,	0x5cd7,	0xa7e2,	0x5cf9,	0xa806,	0x5d1c,	0xa82b,
	0x5d3e,	0xa84f,	0x5d61,	0xa874,	0x5d83,	0xa899,	0x5da5,	0xa8bd,
	0x5dc8,	0xa8e2,	0x5dea,	0xa907,	0x5e0c,	0xa92c,	0x5e2e,	0xa951,
	0x5e50,	0xa976,	0x5e72,	0xa99b,	0x5e94,	0xa9c0,	0x5eb6,	0xa9e5,
	0x5ed7,	0xaa0a,	0x5ef9,	0xaa30,	0x5f1b,	0xaa55,	0x5f3c,	0xaa7a,
	0x5f5e,	0xaaa0,	0x5f80,	0xaac5,	0x5fa1,	0xaaeb,	0x5fc2,	0xab10,
	0x5fe4,	0xab36,	0x6005,	0xab5c,	0x6026,	0xab81,	0x6047,	0xaba7,
	0x6068,	0xabcd,	0x6089,	0xabf3,	0x60aa,	0xac19,	0x60cb,	0xac3f,
	0x60ec,	0xac65,	0x610d,	0xac8b,	0x612e,	0xacb1,	0x614e,	0xacd7,
	0x616f,	0xacfd,	0x6190,	0xad24,	0x61b0,	0xad4a,	0x61d1,	0xad70,
	0x61f1,	0xad97,	0x6211,	0xadbd,	0x6232,	0xade4,	0x6252,	0xae0b,
	0x6272,	0xae31,	0x6292,	0xae58,	0x62b2,	0xae7f,	0x62d2,	0xaea5,
	0x62f2,	0xaecc,	0x6312,	0xaef3,	0x6332,	0xaf1a,	0x6351,	0xaf41,
	0x6371,	0xaf68,	0x6391,	0xaf8f,	0x63b0,	0xafb6,	0x63d0,	0xafdd,
	0x63ef,	0xb005,	0x640f,	0xb02c,	0x642e,	0xb053,	0x644d,	0xb07b,
	0x646c,	0xb0a2,	0x648b,	0xb0c9,	0x64ab,	0xb0f1,	0x64ca,	0xb118,
	0x64e9,	0xb140,	0x6507,	0xb168,	0x6526,	0xb18f,	0x6545,	0xb1b7,
	0x6564,	0xb1df,	0x6582,	0xb207,	0x65a1,	0xb22f,	0x65c0,	0xb257,
	0x65de,	0xb27f,	0x65fc,	0xb2a7,	0x661b,	0xb2cf,	0x6639,	0xb2f7,
	0x6657,	0xb31f,	0x6675,	0xb347,	0x6693,	0xb36f,	0x66b2,	0xb398,
	0x66d0,	0xb3c0,	0x66ed,	0xb3e9,	0x670b,	0xb411,	0x6729,	0xb439,
	0x6747,	0xb462,	0x6764,	0xb48b,	0x6782,	0xb4b3,	0x67a0,	0xb4dc,
	0x67bd,	0xb505,	0x67da,	0xb52d,	0x67f8,	0xb556,	0x6815,	0xb57f,
	0x6832,	0xb5a8,	0x6850,	0xb5d1,	0x686d,	0xb5fa,	0x688a,	0xb623,
	0x68a7,	0xb64c,	0x68c4,	0xb675,	0x68e0,	0xb69e,	0x68fd,	0xb6c7,
	0x691a,	0xb6f1,	0x6937,	0xb71a,	0x6953,	0xb743,	0x6970,	0xb76d,
	0x698c,	0xb796,	0x69a9,	0xb7c0,	0x69c5,	0xb7e9,	0x69e1,	0xb813,
	0x69fd,	0xb83c,	0x6a1a,	0xb866,	0x6a36,	0xb890,	0x6a52,	0xb8b9,
	0x6a6e,	0xb8e3,	0x6a89,	0xb90d,	0x6aa5,	0xb937,	0x6ac1,	0xb961,
	0x6add,	0xb98b,	0x6af8,	0xb9b5,	0x6b14,	0xb9df,	0x6b30,	0xba09,
	0x6b4b,	0xba33,	0x6b66,	0xba5d,	0x6b82,	0xba87,	0x6b9d,	0xbab1,
	0x6bb8,	0xbadc,	0x6bd3,	0xbb06,	0x6bee,	0xbb30,	0x6c09,	0xbb5b,
	0x6c24,	0xbb85,	0x6c3f,	0xbbb0,	0x6c5a,	0xbbda,	0x6c75,	0xbc05,
	0x6c8f,	0xbc2f,	0x6caa,	0xbc5a,	0x6cc4,	0xbc85,	0x6cdf,	0xbcaf,
	0x6cf9,	0xbcda,	0x6d14,	0xbd05,	0x6d2e,	0xbd30,	0x6d48,	0xbd5b,
	0x6d62,	0xbd86,	0x6d7c,	0xbdb1,	0x6d96,	0xbddc,	0x6db0,	0xbe07,
	0x6dca,	0xbe32,	0x6de4,	0xbe5d,	0x6dfe,	0xbe88,	0x6e17,	0xbeb3,
	0x6e31,	0xbedf,	0x6e4a,	0xbf0a,	0x6e64,	0xbf35,	0x6e7d,	0xbf61,
	0x6e97,	0xbf8c,	0x6eb0,	0xbfb8,	0x6ec9,	0xbfe3,	0x6ee2,	0xc00f,
	0x6efb,	0xc03a,	0x6f14,	0xc066,	0x6f2d,	0xc091,	0x6f46,	0xc0bd,
	0x6f5f,	0xc0e9,	0x6f78,	0xc114,	0x6f90,	0xc140,	0x6fa9,	0xc16c,
	0x6fc2,	0xc198,	0x6fda,	0xc1c4,	0x6ff2,	0xc1f0,	0x700b,	0xc21c,
	0x7023,	0xc248,	0x703b,	0xc274,	0x7053,	0xc2a0,	0x706b,	0xc2cc,
	0x7083,	0xc2f8,	0x709b,	0xc324,	0x70b3,	0xc351,	0x70cb,	0xc37d,
	0x70e3,	0xc3a9,	0x70fa,	0xc3d6,	0x7112,	0xc402,	0x712a,	0xc42e,
	0x7141,	0xc45b,	0x7158,	0xc487,	0x7170,	0xc4b4,	0x7187,	0xc4e0,
	0x719e,	0xc50d,	0x71b5,	0xc53a,	0x71cc,	0xc566,	0x71e3,	0xc593,
	0x71fa,	0xc5c0,	0x7211,	0xc5ed,	0x7228,	0xc619,	0x723f,	0xc646,
	0x7255,	0xc673,	0x726c,	0xc6a0,	0x7282,	0xc6cd,	0x7299,	0xc6fa,
	0x72af,	0xc727,	0x72c5,	0xc754,	0x72dc,	0xc781,	0x72f2,	0xc7ae,
	0x7308,	0xc7db,	0x731e,	0xc809,	0x7334,	0xc836,	0x734a,	0xc863,
	0x735f,	0xc890,	0x7375,	0xc8be,	0x738b,	0xc8eb,	0x73a0,	0xc918,
	0x73b6,	0xc946,	0x73cb,	0xc973,	0x73e1,	0xc9a1,	0x73f6,	0xc9ce,
	0x740b,	0xc9fc,	0x7421,	0xca29,	0x7436,	0xca57,	0x744b,	0xca85,
	0x7460,	0xcab2,	0x7475,	0xcae0,	0x7489,	0xcb0e,	0x749e,	0xcb3c,
	0x74b3,	0xcb69,	0x74c7,	0xcb97,	0x74dc,	0xcbc5,	0x74f0,	0xcbf3,
	0x7505,	0xcc21,	0x7519,	0xcc4f,	0x752d,	0xcc7d,	0x7542,	0xccab,
	0x7556,	0xccd9,	0x756a,	0xcd07,	0x757e,	0xcd35,	0x7592,	0xcd63,
	0x75a6,	0xcd92,	0x75b9,	0xcdc0,	0x75cd,	0xcdee,	0x75e1,	0xce1c,
	0x75f4,	0xce4b,	0x7608,	0xce79,	0x761b,	0xcea7,	0x762e,	0xced6,
	0x7642,	0xcf04,	0x7655,	0xcf33,	0x7668,	0xcf61,	0x767b,	0xcf90,
	0x768e,	0xcfbe,	0x76a1,	0xcfed,	0x76b4,	0xd01b,	0x76c7,	0xd04a,
	0x76d9,	0xd079,	0x76ec,	0xd0a7,	0x76fe,	0xd0d6,	0x7711,	0xd105,
	0x7723,	0xd134,	0x7736,	0xd162,	0x7748,	0xd191,	0x775a,	0xd1c0,
	0x776c,	0xd1ef,	0x777e,	0xd21e,	0x7790,	0xd24d,	0x77a2,	0xd27c,
	0x77b4,	0xd2ab,	0x77c6,	0xd2da,	0x77d8,	0xd309,	0x77e9,	0xd338,
	0x77fb,	0xd367,	0x780c,	0xd396,	0x781e,	0xd3c5,	0x782f,	0xd3f4,
	0x7840,	0xd424,	0x7851,	0xd453,	0x7863,	0xd482,	0x7874,	0xd4b1,
	0x7885,	0xd4e1,	0x7895,	0xd510,	0x78a6,	0xd53f,	0x78b7,	0xd56f,
	0x78c8,	0xd59e,	0x78d8,	0xd5ce,	0x78e9,	0xd5fd,	0x78f9,	0xd62d,
	0x790a,	0xd65c,	0x791a,	0xd68c,	0x792a,	0xd6bb,	0x793a,	0xd6eb,
	0x794a,	0xd71b,	0x795b,	0xd74a,	0x796a,	0xd77a,	0x797a,	0xd7aa,
	0x798a,	0xd7d9,	0x799a,	0xd809,	0x79aa,	0xd839,	0x79b9,	0xd869,
	0x79c9,	0xd898,	0x79d8,	0xd8c8,	0x79e7,	0xd8f8,	0x79f7,	0xd928,
	0x7a06,	0xd958,	0x7a15,	0xd988,	0x7a24,	0xd9b8,	0x7a33,	0xd9e8,
	0x7a42,	0xda18,	0x7a51,	0xda48,	0x7a60,	0xda78,	0x7a6e,	0xdaa8,
	0x7a7d,	0xdad8,	0x7a8c,	0xdb08,	0x7a9a,	0xdb38,	0x7aa8,	0xdb68,
	0x7ab7,	0xdb99,	0x7ac5,	0xdbc9,	0x7ad3,	0xdbf9,	0x7ae1,	0xdc29,
	0x7aef,	0xdc59,	0x7afd,	0xdc8a,	0x7b0b,	0xdcba,	0x7b19,	0xdcea,
	0x7b27,	0xdd1b,	0x7b34,	0xdd4b,	0x7b42,	0xdd7c,	0x7b50,	0xddac,
	0x7b5d,	0xdddc,	0x7b6a,	0xde0d,	0x7b78,	0xde3d,	0x7b85,	0xde6e,
	0x7b92,	0xde9e,	0x7b9f,	0xdecf,	0x7bac,	0xdeff,	0x7bb9,	0xdf30,
	0x7bc6,	0xdf61,	0x7bd3,	0xdf91,	0x7bdf,	0xdfc2,	0x7bec,	0xdff2,
	0x7bf9,	0xe023,	0x7c05,	0xe054,	0x7c11,	0xe085,	0x7c1e,	0xe0b5,
	0x7c2a,	0xe0e6,	0x7c36,	0xe117,	0x7c42,	0xe148,	0x7c4e,	0xe178,
	0x7c5a,	0xe1a9,	0x7c66,	0xe1da,	0x7c72,	0xe20b,	0x7c7e,	0xe23c,
	0x7c89,	0xe26d,	0x7c95,	0xe29e,	0x7ca0,	0xe2cf,	0x7cac,	0xe2ff,
	0x7cb7,	0xe330,	0x7cc2,	0xe361,	0x7cce,	0xe392,	0x7cd9,	0xe3c3,
	0x7ce4,	0xe3f4,	0x7cef,	0xe426,	0x7cfa,	0xe457,	0x7d05,	0xe488,
	0x7d0f,	0xe4b9,	0x7d1a,	0xe4ea,	0x7d25,	0xe51b,	0x7d2f,	0xe54c,
	0x7d3a,	0xe57d,	0x7d44,	0xe5af,	0x7d4e,	0xe5e0,	0x7d58,	0xe611,
	0x7d63,	0xe642,	0x7d6d,	0xe673,	0x7d77,	0xe6a5,	0x7d81,	0xe6d6,
	0x7d8a,	0xe707,	0x7d94,	0xe739,	0x7d9e,	0xe76a,	0x7da7,	0xe79b,
	0x7db1,	0xe7cd,	0x7dba,	0xe7fe,	0x7dc4,	0xe82f,	0x7dcd,	0xe861,
	0x7dd6,	0xe892,	0x7de0,	0xe8c4,	0x7de9,	0xe8f5,	0x7df2,	0xe926,
	0x7dfb,	0xe958,	0x7e03,	0xe989,	0x7e0c,	0xe9bb,	0x7e15,	0xe9ec,
	0x7e1e,	0xea1e,	0x7e26,	0xea4f,	0x7e2f,	0xea81,	0x7e37,	0xeab3,
	0x7e3f,	0xeae4,	0x7e48,	0xeb16,	0x7e50,	0xeb47,	0x7e58,	0xeb79,
	0x7e60,	0xebab,	0x7e68,	0xebdc,	0x7e70,	0xec0e,	0x7e78,	0xec3f,
	0x7e7f,	0xec71,	0x7e87,	0xeca3,	0x7e8e,	0xecd5,	0x7e96,	0xed06,
	0x7e9d,	0xed38,	0x7ea5,	0xed6a,	0x7eac,	0xed9b,	0x7eb3,	0xedcd,
	0x7eba,	0xedff,	0x7ec1,	0xee31,	0x7ec8,	0xee62,	0x7ecf,	0xee94,
	0x7ed6,	0xeec6,	0x7edd,	0xeef8,	0x7ee3,	0xef2a,	0x7eea,	0xef5c,
	0x7ef0,	0xef8d,	0x7ef7,	0xefbf,	0x7efd,	0xeff1,	0x7f03,	0xf023,
	0x7f0a,	0xf055,	0x7f10,	0xf087,	0x7f16,	0xf0b9,	0x7f1c,	0xf0eb,
	0x7f22,	0xf11c,	0x7f27,	0xf14e,	0x7f2d,	0xf180,	0x7f33,	0xf1b2,
	0x7f38,	0xf1e4,	0x7f3e,	0xf216,	0x7f43,	0xf248,	0x7f49,	0xf27a,
	0x7f4e,	0xf2ac,	0x7f53,	0xf2de,	0x7f58,	0xf310,	0x7f5d,	0xf342,
	0x7f62,	0xf374,	0x7f67,	0xf3a6,	0x7f6c,	0xf3d8,	0x7f71,	0xf40a,
	0x7f75,	0xf43c,	0x7f7a,	0xf46e,	0x7f7e,	0xf4a0,	0x7f83,	0xf4d3,
	0x7f87,	0xf505,	0x7f8b,	0xf537,	0x7f90,	0xf569,	0x7f94,	0xf59b,
	0x7f98,	0xf5cd,	0x7f9c,	0xf5ff,	0x7fa0,	0xf631,	0x7fa3,	0xf663,
	0x7fa7,	0xf695,	0x7fab,	0xf6c8,	0x7fae,	0xf6fa,	0x7fb2,	0xf72c,
	0x7fb5,	0xf75e,	0x7fb9,	0xf790,	0x7fbc,	0xf7c2,	0x7fbf,	0xf7f4,
	0x7fc2,	0xf827,	0x7fc5,	0xf859,	0x7fc8,	0xf88b,	0x7fcb,	0xf8bd,
	0x7fce,	0xf8ef,	0x7fd1,	0xf922,	0x7fd3,	0xf954,	0x7fd6,	0xf986,
	0x7fd9,	0xf9b8,	0x7fdb,	0xf9ea,	0x7fdd,	0xfa1d,	0x7fe0,	0xfa4f,
	0x7fe2,	0xfa81,	0x7fe4,	0xfab3,	0x7fe6,	0xfae5,	0x7fe8,	0xfb18,
	0x7fea,	0xfb4a,	0x7fec,	0xfb7c,	0x7fed,	0xfbae,	0x7fef,	0xfbe1,
	0x7ff1,	0xfc13,	0x7ff2,	0xfc45,	0x7ff4,	0xfc77,	0x7ff5,	0xfcaa,
	0x7ff6,	0xfcdc,	0x7ff7,	0xfd0e,	0x7ff8,	0xfd40,	0x7ff9,	0xfd73,
	0x7ffa,	0xfda5,	0x7ffb,	0xfdd7,	0x7ffc,	0xfe09,	0x7ffd,	0xfe3c,
	0x7ffe,	0xfe6e,	0x7ffe,	0xfea0,	0x7fff,	0xfed2,	0x7fff,	0xff05,
	0x7fff,	0xff37,	0x7fff,	0xff69,	0x7fff,	0xff9b,	0x7fff,	0xffce
};


/**   
* @brief Initialization function for the Q15 CFFT/CIFFT.  
* @param[in,out] *S             points to an instance of the Q15 CFFT/CIFFT structure.  
* @param[in]     fftLen         length of the FFT.  
* @param[in]     ifftFlag       flag that selects forward (ifftFlag=0) or inverse (ifftFlag=1) transform.  
* @param[in]     bitReverseFlag flag that enables (bitReverseFlag=1) or disables (bitReverseFlag=0) bit reversal of output.  
* @return        The function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLen</code> is not a supported value.  
*   
* \par Description:  
* \par   
* The parameter <code>ifftFlag</code> controls whether a forward or inverse transform is computed.   
* Set(=1) ifftFlag for calculation of CIFFT otherwise  CFFT is calculated  
* \par   
* The parameter <code>bitReverseFlag</code> controls whether output is in normal order or bit reversed order.   
* Set(=1) bitReverseFlag for output to be in normal order otherwise output is in bit reversed order.   
* \par   
* The parameter <code>fftLen</code>	Specifies length of CFFT/CIFFT process. Supported FFT Lengths are 16, 64, 256, 1024.   
* \par   
* This Function also initializes Twiddle factor table pointer and Bit reversal table pointer.   
*/

arm_status arm_cfft_radix4_init_q15(
  arm_cfft_radix4_instance_q15 * S,
  uint16_t fftLen,
  uint8_t ifftFlag,
  uint8_t bitReverseFlag)
{
  /*  Initialise the default arm status */
  arm_status status = ARM_MATH_SUCCESS;
  /*  Initialise the FFT length */
  S->fftLen = fftLen;
  /*  Initialise the Twiddle coefficient pointer */
  S->pTwiddle = (q15_t *) twiddleCoefQ15;
  /*  Initialise the Flag for selection of CFFT or CIFFT */
  S->ifftFlag = ifftFlag;
  /*  Initialise the Flag for calculation Bit reversal or not */
  S->bitReverseFlag = bitReverseFlag;

  /*  Initializations of structure parameters depending on the FFT length */
  switch (S->fftLen)
  {
  case 4096u:
    /*  Initializations of structure parameters for 4096 point FFT */

    /*  Initialise the twiddle coef modifier value */
    S->twidCoefModifier = 1u;
    /*  Initialise the bit reversal table modifier */
    S->bitRevFactor = 1u;
    /*  Initialise the bit reversal table pointer */
    S->pBitRevTable = armBitRevTable;

    break;
	
  case 1024u:
    /*  Initializations of structure parameters for 1024 point FFT */
    S->twidCoefModifier = 4u;
    S->bitRevFactor = 4u;
    S->pBitRevTable = &armBitRevTable[3];

    break;
	
  case 256u:
    /*  Initializations of structure parameters for 256 point FFT */
    S->twidCoefModifier = 16u;
    S->bitRevFactor = 16u;
    S->pBitRevTable = &armBitRevTable[15];

    break;
	
  case 64u:
    /*  Initializations of structure parameters for 64 point FFT */
    S->twidCoefModifier = 64u;
    S->bitRevFactor = 64u;
    S->pBitRevTable = &armBitRevTable[63];

    break;
	
  case 16u:
    /*  Initializations of structure parameters for 16 point FFT */
    S->twidCoefModifier = 256u;
    S->bitRevFactor = 256u;
    S->pBitRevTable = &armBitRevTable[255];

    break;

  default:
    /*  Reporting argument error if fftSize is not valid value */
    status = ARM_MATH_ARGUMENT_ERROR;
    break;
  }

  return (status);
}

/**   
 * @} end of CFFT_CIFFT group   
 */
