// Base32 implementation
//
// Copyright 2010 Google Inc.
// Author: Markus Gutschke
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include <string.h>

#include "base32.h"

int base32_decode(const uint8_t *encoded, uint8_t *result, int bufSize) {
  int buffer = 0;

  int bitsLeft = 0;

  int count = 0;

  const uint8_t *ptr;

  for (ptr = encoded; count < bufSize && *ptr; ++ptr) {
    // if(count%3 != 0){
    uint8_t ch = *ptr;

    if (ch == ' ' || ch == '\t' || ch == '\r' || ch == '\n' || ch == '-') {
      continue;
    }
    buffer <<= 5;

    // Deal with commonly mistyped characters
    if (ch == '0') {
      ch = 'O';
    } else if (ch == '1') {
      ch = 'L';
    } else if (ch == '8') {
      ch = 'B';
    }

    // Look up one base32 digit
    if ((ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z')) {
      ch = (ch & 0x1F) - 1;
    } else if (ch >= '2' && ch <= '7') {
      ch -= '2' - 26;
    } else {
      return -1;
    }

    buffer |= ch;
    bitsLeft += 5;
    if (bitsLeft >= 8) {
      result[count++] = buffer >> (bitsLeft - 8);
      bitsLeft -= 8;
    }
    // } else {
    // result[count++] = ' ';
    // --ptr;
    // }
  }
  if (count < bufSize) {
    result[count] = '\000';
  }

  return count;
}

int base32_encode(const uint8_t *data, int length, uint8_t *result, int bufSize) {
  if (length < 0 || length > (1 << 28)) {
    return -1;
  }
  int count = 0;

  if (length > 0) {
    int buffer = data[0];

    int next = 1;

    int bitsLeft = 8;

    while (count < bufSize && (bitsLeft > 0 || next < length)) {
      if (bitsLeft < 5) {
        if (next < length) {
          buffer <<= 8;
          buffer |= data[next++] & 0xFF;
          bitsLeft += 8;
        } else {
          int pad = 5 - bitsLeft;

          buffer <<= pad;
          bitsLeft += pad;
        }
      }
      int index = 0x1F & (buffer >> (bitsLeft - 5));

      bitsLeft -= 5;
      result[count++] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567"[index];
    }
  }
  if (count < bufSize) {
    result[count] = '\000';
  }
  return count;
}

int base32_clean(const uint8_t *data, int length, uint8_t *result) {
  int i = 0, j = 0;

  while (i <= length) {
    if (data[i] >= 'A' && data[i] <= 'Z') {
      result[j] = data[i];
      i++;
      j++;
    } else if (data[i] >= 'a' && data[i] <= 'z') {
      result[j] = data[i] - 32;
      i++;
      j++;
    } else if (data[i] >= '2' && data[i] <= '7') {
      result[j] = data[i];
      i++;
      j++;
    } else if (data[i] == '0') {
      result[j] = 'O';
      i++;
      j++;
    } else if (data[i] == '1') {
      result[j] = 'L';
      i++;
      j++;
    } else if (data[i] == '8') {
      result[j] = 'B';
      i++;
      j++;
    } else {
      i++;
    }
  }
  return j;
}
