/// <reference path="moduleNameResolver.ts"/>
/// <reference path="binder.ts"/>
/// <reference path="symbolWalker.ts" />

/* @internal */
namespace ts {
    const ambientModuleSymbolRegex = /^".+"$/;

    let nextSymbolId = 1;
    let nextNodeId = 1;
    let nextMergeId = 1;
    let nextFlowId = 1;

    export function getNodeId(node: Node): number {
        if (!node.id) {
            node.id = nextNodeId;
            nextNodeId++;
        }
        return node.id;
    }

    export function getSymbolId(symbol: Symbol): number {
        if (!symbol.id) {
            symbol.id = nextSymbolId;
            nextSymbolId++;
        }

        return symbol.id;
    }

    export function isInstantiatedModule(node: ModuleDeclaration, preserveConstEnums: boolean) {
        const moduleState = getModuleInstanceState(node);
        return moduleState === ModuleInstanceState.Instantiated ||
            (preserveConstEnums && moduleState === ModuleInstanceState.ConstEnumOnly);
    }

    export function createTypeChecker(host: TypeCheckerHost, produceDiagnostics: boolean): TypeChecker {
        // Cancellation that controls whether or not we can cancel in the middle of type checking.
        // In general cancelling is *not* safe for the type checker.  We might be in the middle of
        // computing something, and we will leave our internals in an inconsistent state.  Callers
        // who set the cancellation token should catch if a cancellation exception occurs, and
        // should throw away and create a new TypeChecker.
        //
        // Currently we only support setting the cancellation token when getting diagnostics.  This
        // is because diagnostics can be quite expensive, and we want to allow hosts to bail out if
        // they no longer need the information (for example, if the user started editing again).
        let cancellationToken: CancellationToken;
        let requestedExternalEmitHelpers: ExternalEmitHelpers;
        let externalHelpersModule: Symbol;

        // tslint:disable variable-name
        const Symbol = objectAllocator.getSymbolConstructor();
        const Type = objectAllocator.getTypeConstructor();
        const Signature = objectAllocator.getSignatureConstructor();
        // tslint:enable variable-name

        let typeCount = 0;
        let symbolCount = 0;
        let enumCount = 0;
        let symbolInstantiationDepth = 0;

        const emptySymbols = createSymbolTable();
        const identityMapper: (type: Type) => Type = identity;

        const compilerOptions = host.getCompilerOptions();
        const languageVersion = getEmitScriptTarget(compilerOptions);
        const modulekind = getEmitModuleKind(compilerOptions);
        const noUnusedIdentifiers = !!compilerOptions.noUnusedLocals || !!compilerOptions.noUnusedParameters;
        const allowSyntheticDefaultImports = getAllowSyntheticDefaultImports(compilerOptions);
        const strictNullChecks = getStrictOptionValue(compilerOptions, "strictNullChecks");
        const strictFunctionTypes = getStrictOptionValue(compilerOptions, "strictFunctionTypes");
        const strictPropertyInitialization = getStrictOptionValue(compilerOptions, "strictPropertyInitialization");
        const noImplicitAny = getStrictOptionValue(compilerOptions, "noImplicitAny");
        const noImplicitThis = getStrictOptionValue(compilerOptions, "noImplicitThis");

        const emitResolver = createResolver();
        const nodeBuilder = createNodeBuilder();

        const undefinedSymbol = createSymbol(SymbolFlags.Property, "undefined" as __String);
        undefinedSymbol.declarations = [];
        const argumentsSymbol = createSymbol(SymbolFlags.Property, "arguments" as __String);

        /** This will be set during calls to `getResolvedSignature` where services determines an apparent number of arguments greater than what is actually provided. */
        let apparentArgumentCount: number | undefined;

        // for public members that accept a Node or one of its subtypes, we must guard against
        // synthetic nodes created during transformations by calling `getParseTreeNode`.
        // for most of these, we perform the guard only on `checker` to avoid any possible
        // extra cost of calling `getParseTreeNode` when calling these functions from inside the
        // checker.
        const checker: TypeChecker = {
            getNodeCount: () => sum(host.getSourceFiles(), "nodeCount"),
            getIdentifierCount: () => sum(host.getSourceFiles(), "identifierCount"),
            getSymbolCount: () => sum(host.getSourceFiles(), "symbolCount") + symbolCount,
            getTypeCount: () => typeCount,
            isUndefinedSymbol: symbol => symbol === undefinedSymbol,
            isArgumentsSymbol: symbol => symbol === argumentsSymbol,
            isUnknownSymbol: symbol => symbol === unknownSymbol,
            getMergedSymbol,
            getDiagnostics,
            getGlobalDiagnostics,
            getTypeOfSymbolAtLocation: (symbol, location) => {
                location = getParseTreeNode(location);
                return location ? getTypeOfSymbolAtLocation(symbol, location) : unknownType;
            },
            getSymbolsOfParameterPropertyDeclaration: (parameter, parameterName) => {
                parameter = getParseTreeNode(parameter, isParameter);
                Debug.assert(parameter !== undefined, "Cannot get symbols of a synthetic parameter that cannot be resolved to a parse-tree node.");
                return getSymbolsOfParameterPropertyDeclaration(parameter, escapeLeadingUnderscores(parameterName));
            },
            getDeclaredTypeOfSymbol,
            getPropertiesOfType,
            getPropertyOfType: (type, name) => getPropertyOfType(type, escapeLeadingUnderscores(name)),
            getIndexInfoOfType,
            getSignaturesOfType,
            getIndexTypeOfType,
            getBaseTypes,
            getBaseTypeOfLiteralType,
            getWidenedType,
            getTypeFromTypeNode: node => {
                node = getParseTreeNode(node, isTypeNode);
                return node ? getTypeFromTypeNode(node) : unknownType;
            },
            getParameterType: getTypeAtPosition,
            getReturnTypeOfSignature,
            getNullableType,
            getNonNullableType,
            typeToTypeNode: nodeBuilder.typeToTypeNode,
            indexInfoToIndexSignatureDeclaration: nodeBuilder.indexInfoToIndexSignatureDeclaration,
            signatureToSignatureDeclaration: nodeBuilder.signatureToSignatureDeclaration,
            symbolToEntityName: nodeBuilder.symbolToEntityName,
            symbolToExpression: nodeBuilder.symbolToExpression,
            symbolToTypeParameterDeclarations: nodeBuilder.symbolToTypeParameterDeclarations,
            symbolToParameterDeclaration: nodeBuilder.symbolToParameterDeclaration,
            typeParameterToDeclaration: nodeBuilder.typeParameterToDeclaration,
            getSymbolsInScope: (location, meaning) => {
                location = getParseTreeNode(location);
                return location ? getSymbolsInScope(location, meaning) : [];
            },
            getSymbolAtLocation: node => {
                node = getParseTreeNode(node);
                return node ? getSymbolAtLocation(node) : undefined;
            },
            getShorthandAssignmentValueSymbol: node => {
                node = getParseTreeNode(node);
                return node ? getShorthandAssignmentValueSymbol(node) : undefined;
            },
            getExportSpecifierLocalTargetSymbol: node => {
                node = getParseTreeNode(node, isExportSpecifier);
                return node ? getExportSpecifierLocalTargetSymbol(node) : undefined;
            },
            getExportSymbolOfSymbol(symbol) {
                return getMergedSymbol(symbol.exportSymbol || symbol);
            },
            getTypeAtLocation: node => {
                node = getParseTreeNode(node);
                return node ? getTypeOfNode(node) : unknownType;
            },
            getPropertySymbolOfDestructuringAssignment: location => {
                location = getParseTreeNode(location, isIdentifier);
                return location ? getPropertySymbolOfDestructuringAssignment(location) : undefined;
            },
            signatureToString: (signature, enclosingDeclaration, flags, kind) => {
                return signatureToString(signature, getParseTreeNode(enclosingDeclaration), flags, kind);
            },
            typeToString: (type, enclosingDeclaration, flags) => {
                return typeToString(type, getParseTreeNode(enclosingDeclaration), flags);
            },
            symbolToString: (symbol, enclosingDeclaration, meaning, flags) => {
                return symbolToString(symbol, getParseTreeNode(enclosingDeclaration), meaning, flags);
            },
            typePredicateToString: (predicate, enclosingDeclaration, flags) => {
                return typePredicateToString(predicate, getParseTreeNode(enclosingDeclaration), flags);
            },
            writeSignature: (signature, enclosingDeclaration, flags, kind, writer) => {
                return signatureToString(signature, getParseTreeNode(enclosingDeclaration), flags, kind, writer);
            },
            writeType: (type, enclosingDeclaration, flags, writer) => {
                return typeToString(type, getParseTreeNode(enclosingDeclaration), flags, writer);
            },
            writeSymbol: (symbol, enclosingDeclaration, meaning, flags, writer) => {
                return symbolToString(symbol, getParseTreeNode(enclosingDeclaration), meaning, flags, writer);
            },
            writeTypePredicate: (predicate, enclosingDeclaration, flags, writer) => {
                return typePredicateToString(predicate, getParseTreeNode(enclosingDeclaration), flags, writer);
            },
            getSymbolDisplayBuilder, // TODO (weswigham): Remove once deprecation process is complete
            getAugmentedPropertiesOfType,
            getRootSymbols,
            getContextualType: node => {
                node = getParseTreeNode(node, isExpression);
                return node ? getContextualType(node) : undefined;
            },
            getContextualTypeForArgumentAtIndex: (node, argIndex) => {
                node = getParseTreeNode(node, isCallLikeExpression);
                return node && getContextualTypeForArgumentAtIndex(node, argIndex);
            },
            getContextualTypeForJsxAttribute: (node) => {
                node = getParseTreeNode(node, isJsxAttributeLike);
                return node && getContextualTypeForJsxAttribute(node);
            },
            isContextSensitive,
            getFullyQualifiedName,
            getResolvedSignature: (node, candidatesOutArray, theArgumentCount) => {
                node = getParseTreeNode(node, isCallLikeExpression);
                apparentArgumentCount = theArgumentCount;
                const res = node ? getResolvedSignature(node, candidatesOutArray) : undefined;
                apparentArgumentCount = undefined;
                return res;
            },
            getConstantValue: node => {
                node = getParseTreeNode(node, canHaveConstantValue);
                return node ? getConstantValue(node) : undefined;
            },
            isValidPropertyAccess: (node, propertyName) => {
                node = getParseTreeNode(node, isPropertyAccessOrQualifiedName);
                return !!node && isValidPropertyAccess(node, escapeLeadingUnderscores(propertyName));
            },
            isValidPropertyAccessForCompletions: (node, type, property) => {
                node = getParseTreeNode(node, isPropertyAccessExpression);
                return !!node && isValidPropertyAccessForCompletions(node, type, property);
            },
            getSignatureFromDeclaration: declaration => {
                declaration = getParseTreeNode(declaration, isFunctionLike);
                return declaration ? getSignatureFromDeclaration(declaration) : undefined;
            },
            isImplementationOfOverload: node => {
                const parsed = getParseTreeNode(node, isFunctionLike);
                return parsed ? isImplementationOfOverload(parsed) : undefined;
            },
            getImmediateAliasedSymbol: symbol => {
                Debug.assert((symbol.flags & SymbolFlags.Alias) !== 0, "Should only get Alias here.");
                const links = getSymbolLinks(symbol);
                if (!links.immediateTarget) {
                    const node = getDeclarationOfAliasSymbol(symbol);
                    Debug.assert(!!node);
                    links.immediateTarget = getTargetOfAliasDeclaration(node, /*dontRecursivelyResolve*/ true);
                }

                return links.immediateTarget;
            },
            getAliasedSymbol: resolveAlias,
            getEmitResolver,
            getExportsOfModule: getExportsOfModuleAsArray,
            getExportsAndPropertiesOfModule,
            getSymbolWalker: createGetSymbolWalker(
                getRestTypeOfSignature,
                getTypePredicateOfSignature,
                getReturnTypeOfSignature,
                getBaseTypes,
                resolveStructuredTypeMembers,
                getTypeOfSymbol,
                getResolvedSymbol,
                getIndexTypeOfStructuredType,
                getConstraintFromTypeParameter,
                getFirstIdentifier,
            ),
            getAmbientModules,
            getAllAttributesTypeFromJsxOpeningLikeElement: node => {
                node = getParseTreeNode(node, isJsxOpeningLikeElement);
                return node ? getAllAttributesTypeFromJsxOpeningLikeElement(node) : undefined;
            },
            getJsxIntrinsicTagNamesAt,
            isOptionalParameter: node => {
                node = getParseTreeNode(node, isParameter);
                return node ? isOptionalParameter(node) : false;
            },
            tryGetMemberInModuleExports: (name, symbol) => tryGetMemberInModuleExports(escapeLeadingUnderscores(name), symbol),
            tryGetMemberInModuleExportsAndProperties: (name, symbol) => tryGetMemberInModuleExportsAndProperties(escapeLeadingUnderscores(name), symbol),
            tryFindAmbientModuleWithoutAugmentations: moduleName => {
                // we deliberately exclude augmentations
                // since we are only interested in declarations of the module itself
                return tryFindAmbientModule(moduleName, /*withAugmentations*/ false);
            },
            getApparentType,
            getUnionType,
            createAnonymousType,
            createSignature,
            createSymbol,
            createIndexInfo,
            getAnyType: () => anyType,
            getStringType: () => stringType,
            getNumberType: () => numberType,
            createPromiseType,
            createArrayType,
            getBooleanType: () => booleanType,
            getVoidType: () => voidType,
            getUndefinedType: () => undefinedType,
            getNullType: () => nullType,
            getESSymbolType: () => esSymbolType,
            getNeverType: () => neverType,
            isSymbolAccessible,
            isArrayLikeType,
            getAllPossiblePropertiesOfTypes,
            getSuggestionForNonexistentProperty: (node, type) => getSuggestionForNonexistentProperty(node, type),
            getSuggestionForNonexistentSymbol: (location, name, meaning) => getSuggestionForNonexistentSymbol(location, escapeLeadingUnderscores(name), meaning),
            getBaseConstraintOfType,
            getDefaultFromTypeParameter: type => type && type.flags & TypeFlags.TypeParameter ? getDefaultFromTypeParameter(type as TypeParameter) : undefined,
            resolveName(name, location, meaning, excludeGlobals) {
                return resolveName(location, escapeLeadingUnderscores(name), meaning, /*nameNotFoundMessage*/ undefined, /*nameArg*/ undefined, /*isUse*/ false, excludeGlobals);
            },
            getJsxNamespace: n => unescapeLeadingUnderscores(getJsxNamespace(n)),
            getAccessibleSymbolChain,
            getTypePredicateOfSignature,
            resolveExternalModuleSymbol,
            tryGetThisTypeAt: node => {
                node = getParseTreeNode(node);
                return node && tryGetThisTypeAt(node);
            },
            getTypeArgumentConstraint: node => {
                node = getParseTreeNode(node, isTypeNode);
                return node && getTypeArgumentConstraint(node);
            },

            getSuggestionDiagnostics: file => suggestionDiagnostics.get(file.fileName) || emptyArray,
        };

        const tupleTypes: GenericType[] = [];
        const unionTypes = createMap<UnionType>();
        const intersectionTypes = createMap<IntersectionType>();
        const literalTypes = createMap<LiteralType>();
        const indexedAccessTypes = createMap<IndexedAccessType>();
        const evolvingArrayTypes: EvolvingArrayType[] = [];
        const undefinedProperties = createMap<Symbol>() as UnderscoreEscapedMap<Symbol>;

        const unknownSymbol = createSymbol(SymbolFlags.Property, "unknown" as __String);
        const resolvingSymbol = createSymbol(0, InternalSymbolName.Resolving);

        const anyType = createIntrinsicType(TypeFlags.Any, "any");
        const autoType = createIntrinsicType(TypeFlags.Any, "any");
        const wildcardType = createIntrinsicType(TypeFlags.Any, "any");
        const unknownType = createIntrinsicType(TypeFlags.Any, "unknown");
        const undefinedType = createIntrinsicType(TypeFlags.Undefined, "undefined");
        const undefinedWideningType = strictNullChecks ? undefinedType : createIntrinsicType(TypeFlags.Undefined | TypeFlags.ContainsWideningType, "undefined");
        const nullType = createIntrinsicType(TypeFlags.Null, "null");
        const nullWideningType = strictNullChecks ? nullType : createIntrinsicType(TypeFlags.Null | TypeFlags.ContainsWideningType, "null");
        const stringType = createIntrinsicType(TypeFlags.String, "string");
        const numberType = createIntrinsicType(TypeFlags.Number, "number");
        const trueType = createIntrinsicType(TypeFlags.BooleanLiteral, "true");
        const falseType = createIntrinsicType(TypeFlags.BooleanLiteral, "false");
        const booleanType = createBooleanType([trueType, falseType]);
        const esSymbolType = createIntrinsicType(TypeFlags.ESSymbol, "symbol");
        const voidType = createIntrinsicType(TypeFlags.Void, "void");
        const neverType = createIntrinsicType(TypeFlags.Never, "never");
        const silentNeverType = createIntrinsicType(TypeFlags.Never, "never");
        const implicitNeverType = createIntrinsicType(TypeFlags.Never, "never");
        const nonPrimitiveType = createIntrinsicType(TypeFlags.NonPrimitive, "object");

        const emptyObjectType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);

        const emptyTypeLiteralSymbol = createSymbol(SymbolFlags.TypeLiteral, InternalSymbolName.Type);
        emptyTypeLiteralSymbol.members = createSymbolTable();
        const emptyTypeLiteralType = createAnonymousType(emptyTypeLiteralSymbol, emptySymbols, emptyArray, emptyArray, undefined, undefined);

        const emptyGenericType = <GenericType><ObjectType>createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        emptyGenericType.instantiations = createMap<TypeReference>();

        const anyFunctionType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        // The anyFunctionType contains the anyFunctionType by definition. The flag is further propagated
        // in getPropagatingFlagsOfTypes, and it is checked in inferFromTypes.
        anyFunctionType.flags |= TypeFlags.ContainsAnyFunctionType;

        const noConstraintType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        const circularConstraintType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);
        const resolvingDefaultType = createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, undefined, undefined);

        const markerSuperType = <TypeParameter>createType(TypeFlags.TypeParameter);
        const markerSubType = <TypeParameter>createType(TypeFlags.TypeParameter);
        markerSubType.constraint = markerSuperType;
        const markerOtherType = <TypeParameter>createType(TypeFlags.TypeParameter);

        const noTypePredicate = createIdentifierTypePredicate("<<unresolved>>", 0, anyType);

        const anySignature = createSignature(undefined, undefined, undefined, emptyArray, anyType, /*resolvedTypePredicate*/ undefined, 0, /*hasRestParameter*/ false, /*hasLiteralTypes*/ false);
        const unknownSignature = createSignature(undefined, undefined, undefined, emptyArray, unknownType, /*resolvedTypePredicate*/ undefined, 0, /*hasRestParameter*/ false, /*hasLiteralTypes*/ false);
        const resolvingSignature = createSignature(undefined, undefined, undefined, emptyArray, anyType, /*resolvedTypePredicate*/ undefined, 0, /*hasRestParameter*/ false, /*hasLiteralTypes*/ false);
        const silentNeverSignature = createSignature(undefined, undefined, undefined, emptyArray, silentNeverType, /*resolvedTypePredicate*/ undefined, 0, /*hasRestParameter*/ false, /*hasLiteralTypes*/ false);

        const enumNumberIndexInfo = createIndexInfo(stringType, /*isReadonly*/ true);
        const jsObjectLiteralIndexInfo = createIndexInfo(anyType, /*isReadonly*/ false);

        const globals = createSymbolTable();
        const reverseMappedCache = createMap<Type | undefined>();
        let ambientModulesCache: Symbol[] | undefined;
        /**
         * List of every ambient module with a "*" wildcard.
         * Unlike other ambient modules, these can't be stored in `globals` because symbol tables only deal with exact matches.
         * This is only used if there is no exact match.
         */
        let patternAmbientModules: PatternAmbientModule[];

        let globalObjectType: ObjectType;
        let globalFunctionType: ObjectType;
        let globalArrayType: GenericType;
        let globalReadonlyArrayType: GenericType;
        let globalStringType: ObjectType;
        let globalNumberType: ObjectType;
        let globalBooleanType: ObjectType;
        let globalRegExpType: ObjectType;
        let globalThisType: GenericType;
        let anyArrayType: Type;
        let autoArrayType: Type;
        let anyReadonlyArrayType: Type;
        let deferredGlobalNonNullableTypeAlias: Symbol;

        // The library files are only loaded when the feature is used.
        // This allows users to just specify library files they want to used through --lib
        // and they will not get an error from not having unrelated library files
        let deferredGlobalESSymbolConstructorSymbol: Symbol;
        let deferredGlobalESSymbolType: ObjectType;
        let deferredGlobalTypedPropertyDescriptorType: GenericType;
        let deferredGlobalPromiseType: GenericType;
        let deferredGlobalPromiseConstructorSymbol: Symbol;
        let deferredGlobalPromiseConstructorLikeType: ObjectType;
        let deferredGlobalIterableType: GenericType;
        let deferredGlobalIteratorType: GenericType;
        let deferredGlobalIterableIteratorType: GenericType;
        let deferredGlobalAsyncIterableType: GenericType;
        let deferredGlobalAsyncIteratorType: GenericType;
        let deferredGlobalAsyncIterableIteratorType: GenericType;
        let deferredGlobalTemplateStringsArrayType: ObjectType;

        let deferredNodes: Node[];
        let deferredUnusedIdentifierNodes: Node[];

        let flowLoopStart = 0;
        let flowLoopCount = 0;
        let sharedFlowCount = 0;
        let flowAnalysisDisabled = false;

        const emptyStringType = getLiteralType("");
        const zeroType = getLiteralType(0);

        const resolutionTargets: TypeSystemEntity[] = [];
        const resolutionResults: boolean[] = [];
        const resolutionPropertyNames: TypeSystemPropertyName[] = [];

        let suggestionCount = 0;
        const maximumSuggestionCount = 10;
        const mergedSymbols: Symbol[] = [];
        const symbolLinks: SymbolLinks[] = [];
        const nodeLinks: NodeLinks[] = [];
        const flowLoopCaches: Map<Type>[] = [];
        const flowLoopNodes: FlowNode[] = [];
        const flowLoopKeys: string[] = [];
        const flowLoopTypes: Type[][] = [];
        const sharedFlowNodes: FlowNode[] = [];
        const sharedFlowTypes: FlowType[] = [];
        const potentialThisCollisions: Node[] = [];
        const potentialNewTargetCollisions: Node[] = [];
        const awaitedTypeStack: number[] = [];

        const diagnostics = createDiagnosticCollection();
        // Suggestion diagnostics must have a file. Keyed by source file name.
        const suggestionDiagnostics = createMultiMap<Diagnostic>();
        function addSuggestionDiagnostic(diag: Diagnostic): void {
            suggestionDiagnostics.add(diag.file.fileName, { ...diag, category: DiagnosticCategory.Suggestion });
        }
        function addErrorOrSuggestionDiagnostic(isError: boolean, diag: Diagnostic): void {
            if (isError) {
                diagnostics.add(diag);
            }
            else {
                addSuggestionDiagnostic(diag);
            }
        }

        const enum TypeFacts {
            None = 0,
            TypeofEQString = 1 << 0,      // typeof x === "string"
            TypeofEQNumber = 1 << 1,      // typeof x === "number"
            TypeofEQBoolean = 1 << 2,     // typeof x === "boolean"
            TypeofEQSymbol = 1 << 3,      // typeof x === "symbol"
            TypeofEQObject = 1 << 4,      // typeof x === "object"
            TypeofEQFunction = 1 << 5,    // typeof x === "function"
            TypeofEQHostObject = 1 << 6,  // typeof x === "xxx"
            TypeofNEString = 1 << 7,      // typeof x !== "string"
            TypeofNENumber = 1 << 8,      // typeof x !== "number"
            TypeofNEBoolean = 1 << 9,     // typeof x !== "boolean"
            TypeofNESymbol = 1 << 10,     // typeof x !== "symbol"
            TypeofNEObject = 1 << 11,     // typeof x !== "object"
            TypeofNEFunction = 1 << 12,   // typeof x !== "function"
            TypeofNEHostObject = 1 << 13, // typeof x !== "xxx"
            EQUndefined = 1 << 14,        // x === undefined
            EQNull = 1 << 15,             // x === null
            EQUndefinedOrNull = 1 << 16,  // x === undefined / x === null
            NEUndefined = 1 << 17,        // x !== undefined
            NENull = 1 << 18,             // x !== null
            NEUndefinedOrNull = 1 << 19,  // x != undefined / x != null
            Truthy = 1 << 20,             // x
            Falsy = 1 << 21,              // !x
            All = (1 << 22) - 1,
            // The following members encode facts about particular kinds of types for use in the getTypeFacts function.
            // The presence of a particular fact means that the given test is true for some (and possibly all) values
            // of that kind of type.
            BaseStringStrictFacts = TypeofEQString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
            BaseStringFacts = BaseStringStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            StringStrictFacts = BaseStringStrictFacts | Truthy | Falsy,
            StringFacts = BaseStringFacts | Truthy,
            EmptyStringStrictFacts = BaseStringStrictFacts | Falsy,
            EmptyStringFacts = BaseStringFacts,
            NonEmptyStringStrictFacts = BaseStringStrictFacts | Truthy,
            NonEmptyStringFacts = BaseStringFacts | Truthy,
            BaseNumberStrictFacts = TypeofEQNumber | TypeofNEString | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
            BaseNumberFacts = BaseNumberStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            NumberStrictFacts = BaseNumberStrictFacts | Truthy | Falsy,
            NumberFacts = BaseNumberFacts | Truthy,
            ZeroStrictFacts = BaseNumberStrictFacts | Falsy,
            ZeroFacts = BaseNumberFacts,
            NonZeroStrictFacts = BaseNumberStrictFacts | Truthy,
            NonZeroFacts = BaseNumberFacts | Truthy,
            BaseBooleanStrictFacts = TypeofEQBoolean | TypeofNEString | TypeofNENumber | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull,
            BaseBooleanFacts = BaseBooleanStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            BooleanStrictFacts = BaseBooleanStrictFacts | Truthy | Falsy,
            BooleanFacts = BaseBooleanFacts | Truthy,
            FalseStrictFacts = BaseBooleanStrictFacts | Falsy,
            FalseFacts = BaseBooleanFacts,
            TrueStrictFacts = BaseBooleanStrictFacts | Truthy,
            TrueFacts = BaseBooleanFacts | Truthy,
            SymbolStrictFacts = TypeofEQSymbol | TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
            SymbolFacts = SymbolStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            ObjectStrictFacts = TypeofEQObject | TypeofEQHostObject | TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEFunction | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
            ObjectFacts = ObjectStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            FunctionStrictFacts = TypeofEQFunction | TypeofEQHostObject | TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | NEUndefined | NENull | NEUndefinedOrNull | Truthy,
            FunctionFacts = FunctionStrictFacts | EQUndefined | EQNull | EQUndefinedOrNull | Falsy,
            UndefinedFacts = TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEObject | TypeofNEFunction | TypeofNEHostObject | EQUndefined | EQUndefinedOrNull | NENull | Falsy,
            NullFacts = TypeofEQObject | TypeofNEString | TypeofNENumber | TypeofNEBoolean | TypeofNESymbol | TypeofNEFunction | TypeofNEHostObject | EQNull | EQUndefinedOrNull | NEUndefined | Falsy,
        }

        const typeofEQFacts = createMapFromTemplate({
            string: TypeFacts.TypeofEQString,
            number: TypeFacts.TypeofEQNumber,
            boolean: TypeFacts.TypeofEQBoolean,
            symbol: TypeFacts.TypeofEQSymbol,
            undefined: TypeFacts.EQUndefined,
            object: TypeFacts.TypeofEQObject,
            function: TypeFacts.TypeofEQFunction
        });
        const typeofNEFacts = createMapFromTemplate({
            string: TypeFacts.TypeofNEString,
            number: TypeFacts.TypeofNENumber,
            boolean: TypeFacts.TypeofNEBoolean,
            symbol: TypeFacts.TypeofNESymbol,
            undefined: TypeFacts.NEUndefined,
            object: TypeFacts.TypeofNEObject,
            function: TypeFacts.TypeofNEFunction
        });
        const typeofTypesByName = createMapFromTemplate<Type>({
            string: stringType,
            number: numberType,
            boolean: booleanType,
            symbol: esSymbolType,
            undefined: undefinedType
        });
        const typeofType = createTypeofType();

        let _jsxNamespace: __String;
        let _jsxFactoryEntity: EntityName;

        const subtypeRelation = createMap<RelationComparisonResult>();
        const assignableRelation = createMap<RelationComparisonResult>();
        const definitelyAssignableRelation = createMap<RelationComparisonResult>();
        const comparableRelation = createMap<RelationComparisonResult>();
        const identityRelation = createMap<RelationComparisonResult>();
        const enumRelation = createMap<boolean>();

        type TypeSystemEntity = Symbol | Type | Signature;

        const enum TypeSystemPropertyName {
            Type,
            ResolvedBaseConstructorType,
            DeclaredType,
            ResolvedReturnType,
            ResolvedBaseConstraint,
        }

        const enum CheckMode {
            Normal = 0,                // Normal type checking
            SkipContextSensitive = 1,  // Skip context sensitive function expressions
            Inferential = 2,           // Inferential typing
            Contextual = 3,            // Normal type checking informed by a contextual type, therefore not cacheable
        }

        const enum CallbackCheck {
            None,
            Bivariant,
            Strict,
        }

        const enum MappedTypeModifiers {
            IncludeReadonly = 1 << 0,
            ExcludeReadonly = 1 << 1,
            IncludeOptional = 1 << 2,
            ExcludeOptional = 1 << 3,
        }

        const enum ExpandingFlags {
            None = 0,
            Source = 1,
            Target = 1 << 1,
            Both = Source | Target,
        }

        const enum TypeIncludes {
            Any = 1 << 0,
            Undefined = 1 << 1,
            Null = 1 << 2,
            Never = 1 << 3,
            NonWideningType = 1 << 4,
            String = 1 << 5,
            Number = 1 << 6,
            ESSymbol = 1 << 7,
            LiteralOrUniqueESSymbol = 1 << 8,
            ObjectType = 1 << 9,
            EmptyObject = 1 << 10,
            Union = 1 << 11,
            Wildcard = 1 << 12,
        }

        const enum MembersOrExportsResolutionKind {
            resolvedExports = "resolvedExports",
            resolvedMembers = "resolvedMembers"
        }

        const builtinGlobals = createSymbolTable();
        builtinGlobals.set(undefinedSymbol.escapedName, undefinedSymbol);

        const isNotOverloadAndNotAccessor = and(isNotOverload, isNotAccessor);

        initializeTypeChecker();

        return checker;

        /**
         * @deprecated
         */
        function getSymbolDisplayBuilder(): SymbolDisplayBuilder {
            return {
                buildTypeDisplay(type, writer, enclosingDeclaration?, flags?) {
                    typeToString(type, enclosingDeclaration, flags, emitTextWriterWrapper(writer));
                },
                buildSymbolDisplay(symbol, writer, enclosingDeclaration?, meaning?, flags?) {
                    symbolToString(symbol, enclosingDeclaration, meaning, flags | SymbolFormatFlags.AllowAnyNodeKind, emitTextWriterWrapper(writer));
                },
                buildSignatureDisplay(signature, writer, enclosing?, flags?, kind?) {
                    signatureToString(signature, enclosing, flags, kind, emitTextWriterWrapper(writer));
                },
                buildIndexSignatureDisplay(info, writer, kind, enclosing?, flags?) {
                    const sig = nodeBuilder.indexInfoToIndexSignatureDeclaration(info, kind, enclosing, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors, writer);
                    const printer = createPrinter({ removeComments: true });
                    printer.writeNode(EmitHint.Unspecified, sig, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildParameterDisplay(symbol, writer, enclosing?, flags?) {
                    const node = nodeBuilder.symbolToParameterDeclaration(symbol, enclosing, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors, writer);
                    const printer = createPrinter({ removeComments: true });
                    printer.writeNode(EmitHint.Unspecified, node, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildTypeParameterDisplay(tp, writer, enclosing?, flags?) {
                    const node = nodeBuilder.typeParameterToDeclaration(tp, enclosing, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | NodeBuilderFlags.OmitParameterModifiers, writer);
                    const printer = createPrinter({ removeComments: true });
                    printer.writeNode(EmitHint.Unspecified, node, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildTypePredicateDisplay(predicate, writer, enclosing?, flags?) {
                    typePredicateToString(predicate, enclosing, flags, emitTextWriterWrapper(writer));
                },
                buildTypeParameterDisplayFromSymbol(symbol, writer, enclosing?, flags?) {
                    const nodes = nodeBuilder.symbolToTypeParameterDeclarations(symbol, enclosing, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors, writer);
                    const printer = createPrinter({ removeComments: true });
                    printer.writeList(ListFormat.TypeParameters, nodes, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildDisplayForParametersAndDelimiters(thisParameter, parameters, writer, enclosing?, originalFlags?) {
                    const printer = createPrinter({ removeComments: true });
                    const flags = NodeBuilderFlags.OmitParameterModifiers | NodeBuilderFlags.IgnoreErrors | toNodeBuilderFlags(originalFlags);
                    const thisParameterArray = thisParameter ? [nodeBuilder.symbolToParameterDeclaration(thisParameter, enclosing, flags)] : [];
                    const params = createNodeArray([...thisParameterArray, ...map(parameters, param => nodeBuilder.symbolToParameterDeclaration(param, enclosing, flags))]);
                    printer.writeList(ListFormat.CallExpressionArguments, params, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildDisplayForTypeParametersAndDelimiters(typeParameters, writer, enclosing?, flags?) {
                    const printer = createPrinter({ removeComments: true });
                    const args = createNodeArray(map(typeParameters, p => nodeBuilder.typeParameterToDeclaration(p, enclosing, toNodeBuilderFlags(flags))));
                    printer.writeList(ListFormat.TypeParameters, args, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                },
                buildReturnTypeDisplay(signature, writer, enclosing?, flags?) {
                    writer.writePunctuation(":");
                    writer.writeSpace(" ");
                    const predicate = getTypePredicateOfSignature(signature);
                    if (predicate) {
                        return typePredicateToString(predicate, enclosing, flags, emitTextWriterWrapper(writer));
                    }
                    const node = nodeBuilder.typeToTypeNode(getReturnTypeOfSignature(signature), enclosing, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors, writer);
                    const printer = createPrinter({ removeComments: true });
                    printer.writeNode(EmitHint.Unspecified, node, getSourceFileOfNode(getParseTreeNode(enclosing)), emitTextWriterWrapper(writer));
                }
            };

            function emitTextWriterWrapper(underlying: SymbolWriter): EmitTextWriter {
                return {
                    write: noop,
                    writeTextOfNode: noop,
                    writeLine: noop,
                    increaseIndent() {
                        return underlying.increaseIndent();
                    },
                    decreaseIndent() {
                        return underlying.decreaseIndent();
                    },
                    getText() {
                        return "";
                    },
                    rawWrite: noop,
                    writeLiteral(s) {
                        return underlying.writeStringLiteral(s);
                    },
                    getTextPos() {
                        return 0;
                    },
                    getLine() {
                        return 0;
                    },
                    getColumn() {
                        return 0;
                    },
                    getIndent() {
                        return 0;
                    },
                    isAtStartOfLine() {
                        return false;
                    },
                    clear() {
                        return underlying.clear();
                    },

                    writeKeyword(text) {
                        return underlying.writeKeyword(text);
                    },
                    writeOperator(text) {
                        return underlying.writeOperator(text);
                    },
                    writePunctuation(text) {
                        return underlying.writePunctuation(text);
                    },
                    writeSpace(text) {
                        return underlying.writeSpace(text);
                    },
                    writeStringLiteral(text) {
                        return underlying.writeStringLiteral(text);
                    },
                    writeParameter(text) {
                        return underlying.writeParameter(text);
                    },
                    writeProperty(text) {
                        return underlying.writeProperty(text);
                    },
                    writeSymbol(text, symbol) {
                        return underlying.writeSymbol(text, symbol);
                    },
                    trackSymbol(symbol, enclosing?, meaning?) {
                        return underlying.trackSymbol && underlying.trackSymbol(symbol, enclosing, meaning);
                    },
                    reportInaccessibleThisError() {
                        return underlying.reportInaccessibleThisError && underlying.reportInaccessibleThisError();
                    },
                    reportPrivateInBaseOfClassExpression(name) {
                        return underlying.reportPrivateInBaseOfClassExpression && underlying.reportPrivateInBaseOfClassExpression(name);
                    },
                    reportInaccessibleUniqueSymbolError() {
                        return underlying.reportInaccessibleUniqueSymbolError && underlying.reportInaccessibleUniqueSymbolError();
                    }
                };
            }
        }

        function getJsxNamespace(location: Node | undefined): __String {
            if (location) {
                const file = getSourceFileOfNode(location);
                if (file) {
                    if (file.localJsxNamespace) {
                        return file.localJsxNamespace;
                    }
                    const jsxPragma = file.pragmas.get("jsx");
                    if (jsxPragma) {
                        const chosenpragma = isArray(jsxPragma) ? jsxPragma[0] : jsxPragma;
                        file.localJsxFactory = parseIsolatedEntityName(chosenpragma.arguments.factory, languageVersion);
                        if (file.localJsxFactory) {
                            return file.localJsxNamespace = getFirstIdentifier(file.localJsxFactory).escapedText;
                        }
                    }
                }
            }
            if (!_jsxNamespace) {
                _jsxNamespace = "React" as __String;
                if (compilerOptions.jsxFactory) {
                    _jsxFactoryEntity = parseIsolatedEntityName(compilerOptions.jsxFactory, languageVersion);
                    if (_jsxFactoryEntity) {
                        _jsxNamespace = getFirstIdentifier(_jsxFactoryEntity).escapedText;
                    }
                }
                else if (compilerOptions.reactNamespace) {
                    _jsxNamespace = escapeLeadingUnderscores(compilerOptions.reactNamespace);
                }
            }
            return _jsxNamespace;
        }

        function getEmitResolver(sourceFile: SourceFile, cancellationToken: CancellationToken) {
            // Ensure we have all the type information in place for this file so that all the
            // emitter questions of this resolver will return the right information.
            getDiagnostics(sourceFile, cancellationToken);
            return emitResolver;
        }

        function error(location: Node, message: DiagnosticMessage, arg0?: string | number, arg1?: string | number, arg2?: string | number, arg3?: string | number): void {
            const diagnostic = location
                ? createDiagnosticForNode(location, message, arg0, arg1, arg2, arg3)
                : createCompilerDiagnostic(message, arg0, arg1, arg2, arg3);
            diagnostics.add(diagnostic);
        }

        function createSymbol(flags: SymbolFlags, name: __String, checkFlags?: CheckFlags) {
            symbolCount++;
            const symbol = <TransientSymbol>(new Symbol(flags | SymbolFlags.Transient, name));
            symbol.checkFlags = checkFlags || 0;
            return symbol;
        }

        function isTransientSymbol(symbol: Symbol): symbol is TransientSymbol {
            return (symbol.flags & SymbolFlags.Transient) !== 0;
        }

        function getExcludedSymbolFlags(flags: SymbolFlags): SymbolFlags {
            let result: SymbolFlags = 0;
            if (flags & SymbolFlags.BlockScopedVariable) result |= SymbolFlags.BlockScopedVariableExcludes;
            if (flags & SymbolFlags.FunctionScopedVariable) result |= SymbolFlags.FunctionScopedVariableExcludes;
            if (flags & SymbolFlags.Property) result |= SymbolFlags.PropertyExcludes;
            if (flags & SymbolFlags.EnumMember) result |= SymbolFlags.EnumMemberExcludes;
            if (flags & SymbolFlags.Function) result |= SymbolFlags.FunctionExcludes;
            if (flags & SymbolFlags.Class) result |= SymbolFlags.ClassExcludes;
            if (flags & SymbolFlags.Interface) result |= SymbolFlags.InterfaceExcludes;
            if (flags & SymbolFlags.RegularEnum) result |= SymbolFlags.RegularEnumExcludes;
            if (flags & SymbolFlags.ConstEnum) result |= SymbolFlags.ConstEnumExcludes;
            if (flags & SymbolFlags.ValueModule) result |= SymbolFlags.ValueModuleExcludes;
            if (flags & SymbolFlags.Method) result |= SymbolFlags.MethodExcludes;
            if (flags & SymbolFlags.GetAccessor) result |= SymbolFlags.GetAccessorExcludes;
            if (flags & SymbolFlags.SetAccessor) result |= SymbolFlags.SetAccessorExcludes;
            if (flags & SymbolFlags.TypeParameter) result |= SymbolFlags.TypeParameterExcludes;
            if (flags & SymbolFlags.TypeAlias) result |= SymbolFlags.TypeAliasExcludes;
            if (flags & SymbolFlags.Alias) result |= SymbolFlags.AliasExcludes;
            return result;
        }

        function recordMergedSymbol(target: Symbol, source: Symbol) {
            if (!source.mergeId) {
                source.mergeId = nextMergeId;
                nextMergeId++;
            }
            mergedSymbols[source.mergeId] = target;
        }

        function cloneSymbol(symbol: Symbol): Symbol {
            const result = createSymbol(symbol.flags, symbol.escapedName);
            result.declarations = symbol.declarations ? symbol.declarations.slice() : [];
            result.parent = symbol.parent;
            if (symbol.valueDeclaration) result.valueDeclaration = symbol.valueDeclaration;
            if (symbol.constEnumOnlyModule) result.constEnumOnlyModule = true;
            if (symbol.members) result.members = cloneMap(symbol.members);
            if (symbol.exports) result.exports = cloneMap(symbol.exports);
            recordMergedSymbol(result, symbol);
            return result;
        }

        function mergeSymbol(target: Symbol, source: Symbol) {
            if (!(target.flags & getExcludedSymbolFlags(source.flags)) ||
                (source.flags | target.flags) & SymbolFlags.JSContainer) {
                // Javascript static-property-assignment declarations always merge, even though they are also values
                if (source.flags & SymbolFlags.ValueModule && target.flags & SymbolFlags.ValueModule && target.constEnumOnlyModule && !source.constEnumOnlyModule) {
                    // reset flag when merging instantiated module into value module that has only const enums
                    target.constEnumOnlyModule = false;
                }
                target.flags |= source.flags;
                if (source.valueDeclaration &&
                    (!target.valueDeclaration ||
                        (target.valueDeclaration.kind === SyntaxKind.ModuleDeclaration && source.valueDeclaration.kind !== SyntaxKind.ModuleDeclaration))) {
                    // other kinds of value declarations take precedence over modules
                    target.valueDeclaration = source.valueDeclaration;
                }
                addRange(target.declarations, source.declarations);
                if (source.members) {
                    if (!target.members) target.members = createSymbolTable();
                    mergeSymbolTable(target.members, source.members);
                }
                if (source.exports) {
                    if (!target.exports) target.exports = createSymbolTable();
                    mergeSymbolTable(target.exports, source.exports);
                }
                if ((source.flags | target.flags) & SymbolFlags.JSContainer) {
                    const sourceInitializer = getJSInitializerSymbol(source);
                    const targetInitializer = getJSInitializerSymbol(target);
                    if (sourceInitializer !== source || targetInitializer !== target) {
                        mergeSymbol(targetInitializer, sourceInitializer);
                    }
                }
                recordMergedSymbol(target, source);
            }
            else if (target.flags & SymbolFlags.NamespaceModule) {
                error(getNameOfDeclaration(source.declarations[0]), Diagnostics.Cannot_augment_module_0_with_value_exports_because_it_resolves_to_a_non_module_entity, symbolToString(target));
            }
            else {
                const message = target.flags & SymbolFlags.Enum || source.flags & SymbolFlags.Enum
                    ? Diagnostics.Enum_declarations_can_only_merge_with_namespace_or_other_enum_declarations
                    : target.flags & SymbolFlags.BlockScopedVariable || source.flags & SymbolFlags.BlockScopedVariable
                        ? Diagnostics.Cannot_redeclare_block_scoped_variable_0
                        : Diagnostics.Duplicate_identifier_0;
                forEach(source.declarations, node => {
                    const errorNode = (getJavascriptInitializer(node, /*isPrototypeAssignment*/ false) ? getOuterNameOfJsInitializer(node) : getNameOfDeclaration(node)) || node;
                    error(errorNode, message, symbolToString(source));
                });
                forEach(target.declarations, node => {
                    const errorNode = (getJavascriptInitializer(node, /*isPrototypeAssignment*/ false) ? getOuterNameOfJsInitializer(node) : getNameOfDeclaration(node)) || node;
                    error(errorNode, message, symbolToString(source));
                });
            }
        }

        function combineSymbolTables(first: SymbolTable | undefined, second: SymbolTable | undefined): SymbolTable | undefined {
            if (!first || first.size === 0) return second;
            if (!second || second.size === 0) return first;
            const combined = createSymbolTable();
            mergeSymbolTable(combined, first);
            mergeSymbolTable(combined, second);
            return combined;
        }

        function mergeSymbolTable(target: SymbolTable, source: SymbolTable) {
            source.forEach((sourceSymbol, id) => {
                let targetSymbol = target.get(id);
                if (!targetSymbol) {
                    target.set(id, sourceSymbol);
                }
                else {
                    if (!(targetSymbol.flags & SymbolFlags.Transient)) {
                        targetSymbol = cloneSymbol(targetSymbol);
                        target.set(id, targetSymbol);
                    }
                    mergeSymbol(targetSymbol, sourceSymbol);
                }
            });
        }

        function mergeModuleAugmentation(moduleName: StringLiteral | Identifier): void {
            const moduleAugmentation = <ModuleDeclaration>moduleName.parent;
            if (moduleAugmentation.symbol.declarations[0] !== moduleAugmentation) {
                // this is a combined symbol for multiple augmentations within the same file.
                // its symbol already has accumulated information for all declarations
                // so we need to add it just once - do the work only for first declaration
                Debug.assert(moduleAugmentation.symbol.declarations.length > 1);
                return;
            }

            if (isGlobalScopeAugmentation(moduleAugmentation)) {
                mergeSymbolTable(globals, moduleAugmentation.symbol.exports);
            }
            else {
                // find a module that about to be augmented
                // do not validate names of augmentations that are defined in ambient context
                const moduleNotFoundError = !(moduleName.parent.parent.flags & NodeFlags.Ambient)
                    ? Diagnostics.Invalid_module_name_in_augmentation_module_0_cannot_be_found
                    : undefined;
                let mainModule = resolveExternalModuleNameWorker(moduleName, moduleName, moduleNotFoundError, /*isForAugmentation*/ true);
                if (!mainModule) {
                    return;
                }
                // obtain item referenced by 'export='
                mainModule = resolveExternalModuleSymbol(mainModule);
                if (mainModule.flags & SymbolFlags.Namespace) {
                    // if module symbol has already been merged - it is safe to use it.
                    // otherwise clone it
                    mainModule = mainModule.flags & SymbolFlags.Transient ? mainModule : cloneSymbol(mainModule);
                    mergeSymbol(mainModule, moduleAugmentation.symbol);
                }
                else {
                    // moduleName will be a StringLiteral since this is not `declare global`.
                    error(moduleName, Diagnostics.Cannot_augment_module_0_because_it_resolves_to_a_non_module_entity, (moduleName as StringLiteral).text);
                }
            }
        }

        function addToSymbolTable(target: SymbolTable, source: SymbolTable, message: DiagnosticMessage) {
            source.forEach((sourceSymbol, id) => {
                const targetSymbol = target.get(id);
                if (targetSymbol) {
                    // Error on redeclarations
                    forEach(targetSymbol.declarations, addDeclarationDiagnostic(unescapeLeadingUnderscores(id), message));
                }
                else {
                    target.set(id, sourceSymbol);
                }
            });

            function addDeclarationDiagnostic(id: string, message: DiagnosticMessage) {
                return (declaration: Declaration) => diagnostics.add(createDiagnosticForNode(declaration, message, id));
            }
        }

        function getSymbolLinks(symbol: Symbol): SymbolLinks {
            if (symbol.flags & SymbolFlags.Transient) return <TransientSymbol>symbol;
            const id = getSymbolId(symbol);
            return symbolLinks[id] || (symbolLinks[id] = {});
        }

        function getNodeLinks(node: Node): NodeLinks {
            const nodeId = getNodeId(node);
            return nodeLinks[nodeId] || (nodeLinks[nodeId] = { flags: 0 });
        }

        function isGlobalSourceFile(node: Node) {
            return node.kind === SyntaxKind.SourceFile && !isExternalOrCommonJsModule(<SourceFile>node);
        }

        function getSymbol(symbols: SymbolTable, name: __String, meaning: SymbolFlags): Symbol {
            if (meaning) {
                const symbol = symbols.get(name);
                if (symbol) {
                    Debug.assert((getCheckFlags(symbol) & CheckFlags.Instantiated) === 0, "Should never get an instantiated symbol here.");
                    if (symbol.flags & meaning) {
                        return symbol;
                    }
                    if (symbol.flags & SymbolFlags.Alias) {
                        const target = resolveAlias(symbol);
                        // Unknown symbol means an error occurred in alias resolution, treat it as positive answer to avoid cascading errors
                        if (target === unknownSymbol || target.flags & meaning) {
                            return symbol;
                        }
                    }
                }
            }
            // return undefined if we can't find a symbol.
        }

        /**
         * Get symbols that represent parameter-property-declaration as parameter and as property declaration
         * @param parameter a parameterDeclaration node
         * @param parameterName a name of the parameter to get the symbols for.
         * @return a tuple of two symbols
         */
        function getSymbolsOfParameterPropertyDeclaration(parameter: ParameterDeclaration, parameterName: __String): [Symbol, Symbol] {
            const constructorDeclaration = parameter.parent;
            const classDeclaration = parameter.parent.parent;

            const parameterSymbol = getSymbol(constructorDeclaration.locals, parameterName, SymbolFlags.Value);
            const propertySymbol = getSymbol(getMembersOfSymbol(classDeclaration.symbol), parameterName, SymbolFlags.Value);

            if (parameterSymbol && propertySymbol) {
                return [parameterSymbol, propertySymbol];
            }

            Debug.fail("There should exist two symbols, one as property declaration and one as parameter declaration");
        }

        function isBlockScopedNameDeclaredBeforeUse(declaration: Declaration, usage: Node): boolean {
            const declarationFile = getSourceFileOfNode(declaration);
            const useFile = getSourceFileOfNode(usage);
            if (declarationFile !== useFile) {
                if ((modulekind && (declarationFile.externalModuleIndicator || useFile.externalModuleIndicator)) ||
                    (!compilerOptions.outFile && !compilerOptions.out) ||
                    isInTypeQuery(usage) ||
                    declaration.flags & NodeFlags.Ambient) {
                    // nodes are in different files and order cannot be determined
                    return true;
                }
                // declaration is after usage
                // can be legal if usage is deferred (i.e. inside function or in initializer of instance property)
                if (isUsedInFunctionOrInstanceProperty(usage, declaration)) {
                    return true;
                }
                const sourceFiles = host.getSourceFiles();
                return sourceFiles.indexOf(declarationFile) <= sourceFiles.indexOf(useFile);
            }

            if (declaration.pos <= usage.pos) {
                // declaration is before usage
                if (declaration.kind === SyntaxKind.BindingElement) {
                    // still might be illegal if declaration and usage are both binding elements (eg var [a = b, b = b] = [1, 2])
                    const errorBindingElement = getAncestor(usage, SyntaxKind.BindingElement) as BindingElement;
                    if (errorBindingElement) {
                        return findAncestor(errorBindingElement, isBindingElement) !== findAncestor(declaration, isBindingElement) ||
                            declaration.pos < errorBindingElement.pos;
                    }
                    // or it might be illegal if usage happens before parent variable is declared (eg var [a] = a)
                    return isBlockScopedNameDeclaredBeforeUse(getAncestor(declaration, SyntaxKind.VariableDeclaration) as Declaration, usage);
                }
                else if (declaration.kind === SyntaxKind.VariableDeclaration) {
                    // still might be illegal if usage is in the initializer of the variable declaration (eg var a = a)
                    return !isImmediatelyUsedInInitializerOfBlockScopedVariable(declaration as VariableDeclaration, usage);
                }
                return true;
            }


            // declaration is after usage, but it can still be legal if usage is deferred:
            // 1. inside an export specifier
            // 2. inside a function
            // 3. inside an instance property initializer, a reference to a non-instance property
            // 4. inside a static property initializer, a reference to a static method in the same class
            // 5. inside a TS export= declaration (since we will move the export statement during emit to avoid TDZ)
            // or if usage is in a type context:
            // 1. inside a type query (typeof in type position)
            if (usage.parent.kind === SyntaxKind.ExportSpecifier || (usage.parent.kind === SyntaxKind.ExportAssignment && (usage.parent as ExportAssignment).isExportEquals)) {
                // export specifiers do not use the variable, they only make it available for use
                return true;
            }
            // When resolving symbols for exports, the `usage` location passed in can be the export site directly
            if (usage.kind === SyntaxKind.ExportAssignment && (usage as ExportAssignment).isExportEquals) {
                return true;
            }

            const container = getEnclosingBlockScopeContainer(declaration);
            return isInTypeQuery(usage) || isUsedInFunctionOrInstanceProperty(usage, declaration, container);

            function isImmediatelyUsedInInitializerOfBlockScopedVariable(declaration: VariableDeclaration, usage: Node): boolean {
                const container = getEnclosingBlockScopeContainer(declaration);

                switch (declaration.parent.parent.kind) {
                    case SyntaxKind.VariableStatement:
                    case SyntaxKind.ForStatement:
                    case SyntaxKind.ForOfStatement:
                        // variable statement/for/for-of statement case,
                        // use site should not be inside variable declaration (initializer of declaration or binding element)
                        if (isSameScopeDescendentOf(usage, declaration, container)) {
                            return true;
                        }
                        break;
                }

                // ForIn/ForOf case - use site should not be used in expression part
                return isForInOrOfStatement(declaration.parent.parent) && isSameScopeDescendentOf(usage, declaration.parent.parent.expression, container);
            }

            function isUsedInFunctionOrInstanceProperty(usage: Node, declaration: Node, container?: Node): boolean {
                return !!findAncestor(usage, current => {
                    if (current === container) {
                        return "quit";
                    }
                    if (isFunctionLike(current)) {
                        return true;
                    }

                    const initializerOfProperty = current.parent &&
                        current.parent.kind === SyntaxKind.PropertyDeclaration &&
                        (<PropertyDeclaration>current.parent).initializer === current;

                    if (initializerOfProperty) {
                        if (hasModifier(current.parent, ModifierFlags.Static)) {
                            if (declaration.kind === SyntaxKind.MethodDeclaration) {
                                return true;
                            }
                        }
                        else {
                            const isDeclarationInstanceProperty = declaration.kind === SyntaxKind.PropertyDeclaration && !hasModifier(declaration, ModifierFlags.Static);
                            if (!isDeclarationInstanceProperty || getContainingClass(usage) !== getContainingClass(declaration)) {
                                return true;
                            }
                        }
                    }
                });
            }
        }

        /**
         * Resolve a given name for a given meaning at a given location. An error is reported if the name was not found and
         * the nameNotFoundMessage argument is not undefined. Returns the resolved symbol, or undefined if no symbol with
         * the given name can be found.
         *
         * @param isUse If true, this will count towards --noUnusedLocals / --noUnusedParameters.
         */
        function resolveName(
            location: Node | undefined,
            name: __String,
            meaning: SymbolFlags,
            nameNotFoundMessage: DiagnosticMessage | undefined,
            nameArg: __String | Identifier,
            isUse: boolean,
            excludeGlobals = false,
            suggestedNameNotFoundMessage?: DiagnosticMessage): Symbol {
            return resolveNameHelper(location, name, meaning, nameNotFoundMessage, nameArg, isUse, excludeGlobals, getSymbol, suggestedNameNotFoundMessage);
        }

        function resolveNameHelper(
            location: Node | undefined,
            name: __String,
            meaning: SymbolFlags,
            nameNotFoundMessage: DiagnosticMessage,
            nameArg: __String | Identifier,
            isUse: boolean,
            excludeGlobals: boolean,
            lookup: typeof getSymbol,
            suggestedNameNotFoundMessage?: DiagnosticMessage): Symbol {
            const originalLocation = location; // needed for did-you-mean error reporting, which gathers candidates starting from the original location
            let result: Symbol;
            let lastLocation: Node;
            let lastSelfReferenceLocation: Node;
            let propertyWithInvalidInitializer: Node;
            const errorLocation = location;
            let grandparent: Node;
            let isInExternalModule = false;

            loop: while (location) {
                // Locals of a source file are not in scope (because they get merged into the global symbol table)
                if (location.locals && !isGlobalSourceFile(location)) {
                    if (result = lookup(location.locals, name, meaning)) {
                        let useResult = true;
                        if (isFunctionLike(location) && lastLocation && lastLocation !== (<FunctionLikeDeclaration>location).body) {
                            // symbol lookup restrictions for function-like declarations
                            // - Type parameters of a function are in scope in the entire function declaration, including the parameter
                            //   list and return type. However, local types are only in scope in the function body.
                            // - parameters are only in the scope of function body
                            // This restriction does not apply to JSDoc comment types because they are parented
                            // at a higher level than type parameters would normally be
                            if (meaning & result.flags & SymbolFlags.Type && lastLocation.kind !== SyntaxKind.JSDocComment) {
                                useResult = result.flags & SymbolFlags.TypeParameter
                                    // type parameters are visible in parameter list, return type and type parameter list
                                    ? lastLocation === (<FunctionLikeDeclaration>location).type ||
                                    lastLocation.kind === SyntaxKind.Parameter ||
                                    lastLocation.kind === SyntaxKind.TypeParameter
                                    // local types not visible outside the function body
                                    : false;
                            }
                            if (meaning & SymbolFlags.Value && result.flags & SymbolFlags.FunctionScopedVariable) {
                                // parameters are visible only inside function body, parameter list and return type
                                // technically for parameter list case here we might mix parameters and variables declared in function,
                                // however it is detected separately when checking initializers of parameters
                                // to make sure that they reference no variables declared after them.
                                useResult =
                                    lastLocation.kind === SyntaxKind.Parameter ||
                                    (
                                        lastLocation === (<FunctionLikeDeclaration>location).type &&
                                        result.valueDeclaration.kind === SyntaxKind.Parameter
                                    );
                            }
                        }
                        else if (location.kind === SyntaxKind.ConditionalType) {
                            // A type parameter declared using 'infer T' in a conditional type is visible only in
                            // the true branch of the conditional type.
                            useResult = lastLocation === (<ConditionalTypeNode>location).trueType;
                        }

                        if (useResult) {
                            break loop;
                        }
                        else {
                            result = undefined;
                        }
                    }
                }
                switch (location.kind) {
                    case SyntaxKind.SourceFile:
                        if (!isExternalOrCommonJsModule(<SourceFile>location)) break;
                        isInExternalModule = true;
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                        const moduleExports = getSymbolOfNode(location).exports;
                        if (location.kind === SyntaxKind.SourceFile || isAmbientModule(location)) {

                            // It's an external module. First see if the module has an export default and if the local
                            // name of that export default matches.
                            if (result = moduleExports.get(InternalSymbolName.Default)) {
                                const localSymbol = getLocalSymbolForExportDefault(result);
                                if (localSymbol && (result.flags & meaning) && localSymbol.escapedName === name) {
                                    break loop;
                                }
                                result = undefined;
                            }

                            // Because of module/namespace merging, a module's exports are in scope,
                            // yet we never want to treat an export specifier as putting a member in scope.
                            // Therefore, if the name we find is purely an export specifier, it is not actually considered in scope.
                            // Two things to note about this:
                            //     1. We have to check this without calling getSymbol. The problem with calling getSymbol
                            //        on an export specifier is that it might find the export specifier itself, and try to
                            //        resolve it as an alias. This will cause the checker to consider the export specifier
                            //        a circular alias reference when it might not be.
                            //     2. We check === SymbolFlags.Alias in order to check that the symbol is *purely*
                            //        an alias. If we used &, we'd be throwing out symbols that have non alias aspects,
                            //        which is not the desired behavior.
                            const moduleExport = moduleExports.get(name);
                            if (moduleExport &&
                                moduleExport.flags === SymbolFlags.Alias &&
                                getDeclarationOfKind(moduleExport, SyntaxKind.ExportSpecifier)) {
                                break;
                            }
                        }

                        if (result = lookup(moduleExports, name, meaning & SymbolFlags.ModuleMember)) {
                            break loop;
                        }
                        break;
                    case SyntaxKind.EnumDeclaration:
                        if (result = lookup(getSymbolOfNode(location).exports, name, meaning & SymbolFlags.EnumMember)) {
                            break loop;
                        }
                        break;
                    case SyntaxKind.PropertyDeclaration:
                    case SyntaxKind.PropertySignature:
                        // TypeScript 1.0 spec (April 2014): 8.4.1
                        // Initializer expressions for instance member variables are evaluated in the scope
                        // of the class constructor body but are not permitted to reference parameters or
                        // local variables of the constructor. This effectively means that entities from outer scopes
                        // by the same name as a constructor parameter or local variable are inaccessible
                        // in initializer expressions for instance member variables.
                        if (isClassLike(location.parent) && !hasModifier(location, ModifierFlags.Static)) {
                            const ctor = findConstructorDeclaration(location.parent);
                            if (ctor && ctor.locals) {
                                if (lookup(ctor.locals, name, meaning & SymbolFlags.Value)) {
                                    // Remember the property node, it will be used later to report appropriate error
                                    propertyWithInvalidInitializer = location;
                                }
                            }
                        }
                        break;
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.InterfaceDeclaration:
                        if (result = lookup(getMembersOfSymbol(getSymbolOfNode(location)), name, meaning & SymbolFlags.Type)) {
                            if (!isTypeParameterSymbolDeclaredInContainer(result, location)) {
                                // ignore type parameters not declared in this container
                                result = undefined;
                                break;
                            }
                            if (lastLocation && hasModifier(lastLocation, ModifierFlags.Static)) {
                                // TypeScript 1.0 spec (April 2014): 3.4.1
                                // The scope of a type parameter extends over the entire declaration with which the type
                                // parameter list is associated, with the exception of static member declarations in classes.
                                error(errorLocation, Diagnostics.Static_members_cannot_reference_class_type_parameters);
                                return undefined;
                            }
                            break loop;
                        }
                        if (location.kind === SyntaxKind.ClassExpression && meaning & SymbolFlags.Class) {
                            const className = (<ClassExpression>location).name;
                            if (className && name === className.escapedText) {
                                result = location.symbol;
                                break loop;
                            }
                        }
                        break;
                    case SyntaxKind.ExpressionWithTypeArguments:
                        // The type parameters of a class are not in scope in the base class expression.
                        if (lastLocation === (<ExpressionWithTypeArguments>location).expression && (<HeritageClause>location.parent).token === SyntaxKind.ExtendsKeyword) {
                            const container = location.parent.parent;
                            if (isClassLike(container) && (result = lookup(getSymbolOfNode(container).members, name, meaning & SymbolFlags.Type))) {
                                if (nameNotFoundMessage) {
                                    error(errorLocation, Diagnostics.Base_class_expressions_cannot_reference_class_type_parameters);
                                }
                                return undefined;
                            }
                        }
                        break;
                    // It is not legal to reference a class's own type parameters from a computed property name that
                    // belongs to the class. For example:
                    //
                    //   function foo<T>() { return '' }
                    //   class C<T> { // <-- Class's own type parameter T
                    //       [foo<T>()]() { } // <-- Reference to T from class's own computed property
                    //   }
                    //
                    case SyntaxKind.ComputedPropertyName:
                        grandparent = location.parent.parent;
                        if (isClassLike(grandparent) || grandparent.kind === SyntaxKind.InterfaceDeclaration) {
                            // A reference to this grandparent's type parameters would be an error
                            if (result = lookup(getSymbolOfNode(grandparent).members, name, meaning & SymbolFlags.Type)) {
                                error(errorLocation, Diagnostics.A_computed_property_name_cannot_reference_a_type_parameter_from_its_containing_type);
                                return undefined;
                            }
                        }
                        break;
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.MethodSignature:
                    case SyntaxKind.Constructor:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.ArrowFunction:
                        if (meaning & SymbolFlags.Variable && name === "arguments") {
                            result = argumentsSymbol;
                            break loop;
                        }
                        break;
                    case SyntaxKind.FunctionExpression:
                        if (meaning & SymbolFlags.Variable && name === "arguments") {
                            result = argumentsSymbol;
                            break loop;
                        }

                        if (meaning & SymbolFlags.Function) {
                            const functionName = (<FunctionExpression>location).name;
                            if (functionName && name === functionName.escapedText) {
                                result = location.symbol;
                                break loop;
                            }
                        }
                        break;
                    case SyntaxKind.Decorator:
                        // Decorators are resolved at the class declaration. Resolving at the parameter
                        // or member would result in looking up locals in the method.
                        //
                        //   function y() {}
                        //   class C {
                        //       method(@y x, y) {} // <-- decorator y should be resolved at the class declaration, not the parameter.
                        //   }
                        //
                        if (location.parent && location.parent.kind === SyntaxKind.Parameter) {
                            location = location.parent;
                        }
                        //
                        //   function y() {}
                        //   class C {
                        //       @y method(x, y) {} // <-- decorator y should be resolved at the class declaration, not the method.
                        //   }
                        //
                        if (location.parent && isClassElement(location.parent)) {
                            location = location.parent;
                        }
                        break;
                }
                if (isSelfReferenceLocation(location)) {
                    lastSelfReferenceLocation = location;
                }
                lastLocation = location;
                location = location.parent;
            }

            // We just climbed up parents looking for the name, meaning that we started in a descendant node of `lastLocation`.
            // If `result === lastSelfReferenceLocation.symbol`, that means that we are somewhere inside `lastSelfReferenceLocation` looking up a name, and resolving to `lastLocation` itself.
            // That means that this is a self-reference of `lastLocation`, and shouldn't count this when considering whether `lastLocation` is used.
            if (isUse && result && nameNotFoundMessage && noUnusedIdentifiers && (!lastSelfReferenceLocation || result !== lastSelfReferenceLocation.symbol)) {
                result.isReferenced |= meaning;
            }

            if (!result) {
                if (lastLocation) {
                    Debug.assert(lastLocation.kind === SyntaxKind.SourceFile);
                    if ((lastLocation as SourceFile).commonJsModuleIndicator && name === "exports") {
                        return lastLocation.symbol;
                    }
                }

                if (!excludeGlobals) {
                    result = lookup(globals, name, meaning);
                }
            }

            if (!result) {
                if (nameNotFoundMessage) {
                    if (!errorLocation ||
                        !checkAndReportErrorForMissingPrefix(errorLocation, name, nameArg) &&
                        !checkAndReportErrorForExtendingInterface(errorLocation) &&
                        !checkAndReportErrorForUsingTypeAsNamespace(errorLocation, name, meaning) &&
                        !checkAndReportErrorForUsingTypeAsValue(errorLocation, name, meaning) &&
                        !checkAndReportErrorForUsingNamespaceModuleAsValue(errorLocation, name, meaning)) {
                        let suggestion: string | undefined;
                        if (suggestedNameNotFoundMessage && suggestionCount < maximumSuggestionCount) {
                            suggestion = getSuggestionForNonexistentSymbol(originalLocation, name, meaning);
                            if (suggestion) {
                                error(errorLocation, suggestedNameNotFoundMessage, diagnosticName(nameArg), suggestion);
                            }
                        }
                        if (!suggestion) {
                            error(errorLocation, nameNotFoundMessage, diagnosticName(nameArg));
                        }
                        suggestionCount++;
                    }
                }
                return undefined;
            }

            // Perform extra checks only if error reporting was requested
            if (nameNotFoundMessage) {
                if (propertyWithInvalidInitializer) {
                    // We have a match, but the reference occurred within a property initializer and the identifier also binds
                    // to a local variable in the constructor where the code will be emitted.
                    const propertyName = (<PropertyDeclaration>propertyWithInvalidInitializer).name;
                    error(errorLocation, Diagnostics.Initializer_of_instance_member_variable_0_cannot_reference_identifier_1_declared_in_the_constructor,
                        declarationNameToString(propertyName), diagnosticName(nameArg));
                    return undefined;
                }

                // Only check for block-scoped variable if we have an error location and are looking for the
                // name with variable meaning
                //      For example,
                //          declare module foo {
                //              interface bar {}
                //          }
                //      const foo/*1*/: foo/*2*/.bar;
                // The foo at /*1*/ and /*2*/ will share same symbol with two meanings:
                // block-scoped variable and namespace module. However, only when we
                // try to resolve name in /*1*/ which is used in variable position,
                // we want to check for block-scoped
                if (errorLocation &&
                    (meaning & SymbolFlags.BlockScopedVariable ||
                     ((meaning & SymbolFlags.Class || meaning & SymbolFlags.Enum) && (meaning & SymbolFlags.Value) === SymbolFlags.Value))) {
                    const exportOrLocalSymbol = getExportSymbolOfValueSymbolIfExported(result);
                    if (exportOrLocalSymbol.flags & SymbolFlags.BlockScopedVariable || exportOrLocalSymbol.flags & SymbolFlags.Class || exportOrLocalSymbol.flags & SymbolFlags.Enum) {
                        checkResolvedBlockScopedVariable(exportOrLocalSymbol, errorLocation);
                    }
                }

                // If we're in an external module, we can't reference value symbols created from UMD export declarations
                if (result && isInExternalModule && (meaning & SymbolFlags.Value) === SymbolFlags.Value) {
                    const decls = result.declarations;
                    if (decls && decls.length === 1 && decls[0].kind === SyntaxKind.NamespaceExportDeclaration) {
                        error(errorLocation, Diagnostics._0_refers_to_a_UMD_global_but_the_current_file_is_a_module_Consider_adding_an_import_instead, unescapeLeadingUnderscores(name));
                    }
                }
            }
            return result;
        }

        function isSelfReferenceLocation(node: Node): boolean {
            switch (node.kind) {
                case SyntaxKind.FunctionDeclaration:
                case SyntaxKind.ClassDeclaration:
                case SyntaxKind.InterfaceDeclaration:
                case SyntaxKind.EnumDeclaration:
                case SyntaxKind.TypeAliasDeclaration:
                case SyntaxKind.ModuleDeclaration: // For `namespace N { N; }`
                    return true;
                default:
                    return false;
            }
        }

        function diagnosticName(nameArg: __String | Identifier) {
            return isString(nameArg) ? unescapeLeadingUnderscores(nameArg as __String) : declarationNameToString(nameArg as Identifier);
        }

        function isTypeParameterSymbolDeclaredInContainer(symbol: Symbol, container: Node) {
            for (const decl of symbol.declarations) {
                if (decl.kind === SyntaxKind.TypeParameter && decl.parent === container) {
                    return true;
                }
            }

            return false;
        }

        function checkAndReportErrorForMissingPrefix(errorLocation: Node, name: __String, nameArg: __String | Identifier): boolean {
            if (!isIdentifier(errorLocation) || errorLocation.escapedText !== name || isTypeReferenceIdentifier(errorLocation) || isInTypeQuery(errorLocation)) {
                return false;
            }

            const container = getThisContainer(errorLocation, /*includeArrowFunctions*/ true);
            let location = container;
            while (location) {
                if (isClassLike(location.parent)) {
                    const classSymbol = getSymbolOfNode(location.parent);
                    if (!classSymbol) {
                        break;
                    }

                    // Check to see if a static member exists.
                    const constructorType = getTypeOfSymbol(classSymbol);
                    if (getPropertyOfType(constructorType, name)) {
                        error(errorLocation, Diagnostics.Cannot_find_name_0_Did_you_mean_the_static_member_1_0, diagnosticName(nameArg), symbolToString(classSymbol));
                        return true;
                    }

                    // No static member is present.
                    // Check if we're in an instance method and look for a relevant instance member.
                    if (location === container && !hasModifier(location, ModifierFlags.Static)) {
                        const instanceType = (<InterfaceType>getDeclaredTypeOfSymbol(classSymbol)).thisType;
                        if (getPropertyOfType(instanceType, name)) {
                            error(errorLocation, Diagnostics.Cannot_find_name_0_Did_you_mean_the_instance_member_this_0, diagnosticName(nameArg));
                            return true;
                        }
                    }
                }

                location = location.parent;
            }
            return false;
        }


        function checkAndReportErrorForExtendingInterface(errorLocation: Node): boolean {
            const expression = getEntityNameForExtendingInterface(errorLocation);
            const isError = !!(expression && resolveEntityName(expression, SymbolFlags.Interface, /*ignoreErrors*/ true));
            if (isError) {
                error(errorLocation, Diagnostics.Cannot_extend_an_interface_0_Did_you_mean_implements, getTextOfNode(expression));
            }
            return isError;
        }
        /**
         * Climbs up parents to an ExpressionWithTypeArguments, and returns its expression,
         * but returns undefined if that expression is not an EntityNameExpression.
         */
        function getEntityNameForExtendingInterface(node: Node): EntityNameExpression | undefined {
            switch (node.kind) {
                case SyntaxKind.Identifier:
                case SyntaxKind.PropertyAccessExpression:
                    return node.parent ? getEntityNameForExtendingInterface(node.parent) : undefined;
                case SyntaxKind.ExpressionWithTypeArguments:
                    if (isEntityNameExpression((<ExpressionWithTypeArguments>node).expression)) {
                        return <EntityNameExpression>(<ExpressionWithTypeArguments>node).expression;
                    }
                    // falls through
                default:
                    return undefined;
            }
        }

        function checkAndReportErrorForUsingTypeAsNamespace(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            const namespaceMeaning = SymbolFlags.Namespace | (isInJavaScriptFile(errorLocation) ? SymbolFlags.Value : 0);
            if (meaning === namespaceMeaning) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.Type & ~namespaceMeaning, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                const parent = errorLocation.parent;
                if (symbol) {
                    if (isQualifiedName(parent)) {
                        Debug.assert(parent.left === errorLocation, "Should only be resolving left side of qualified name as a namespace");
                        const propName = parent.right.escapedText;
                        const propType = getPropertyOfType(getDeclaredTypeOfSymbol(symbol), propName);
                        if (propType) {
                            error(
                                parent,
                                Diagnostics.Cannot_access_0_1_because_0_is_a_type_but_not_a_namespace_Did_you_mean_to_retrieve_the_type_of_the_property_1_in_0_with_0_1,
                                unescapeLeadingUnderscores(name),
                                unescapeLeadingUnderscores(propName),
                            );
                            return true;
                        }
                    }
                    error(errorLocation, Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_namespace_here, unescapeLeadingUnderscores(name));
                    return true;
                }
            }

            return false;
        }

        function checkAndReportErrorForUsingTypeAsValue(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            if (meaning & (SymbolFlags.Value & ~SymbolFlags.NamespaceModule)) {
                if (name === "any" || name === "string" || name === "number" || name === "boolean" || name === "never") {
                    error(errorLocation, Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_value_here, unescapeLeadingUnderscores(name));
                    return true;
                }
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.Type & ~SymbolFlags.Value, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol && !(symbol.flags & SymbolFlags.NamespaceModule)) {
                    error(errorLocation, Diagnostics._0_only_refers_to_a_type_but_is_being_used_as_a_value_here, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            return false;
        }

        function checkAndReportErrorForUsingNamespaceModuleAsValue(errorLocation: Node, name: __String, meaning: SymbolFlags): boolean {
            if (meaning & (SymbolFlags.Value & ~SymbolFlags.NamespaceModule & ~SymbolFlags.Type)) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, SymbolFlags.NamespaceModule & ~SymbolFlags.Value, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol) {
                    error(errorLocation, Diagnostics.Cannot_use_namespace_0_as_a_value, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            else if (meaning & (SymbolFlags.Type & ~SymbolFlags.NamespaceModule & ~SymbolFlags.Value)) {
                const symbol = resolveSymbol(resolveName(errorLocation, name, (SymbolFlags.ValueModule | SymbolFlags.NamespaceModule) & ~SymbolFlags.Type, /*nameNotFoundMessage*/undefined, /*nameArg*/ undefined, /*isUse*/ false));
                if (symbol) {
                    error(errorLocation, Diagnostics.Cannot_use_namespace_0_as_a_type, unescapeLeadingUnderscores(name));
                    return true;
                }
            }
            return false;
        }

        function checkResolvedBlockScopedVariable(result: Symbol, errorLocation: Node): void {
            Debug.assert(!!(result.flags & SymbolFlags.BlockScopedVariable || result.flags & SymbolFlags.Class || result.flags & SymbolFlags.Enum));
            // Block-scoped variables cannot be used before their definition
            const declaration = forEach(result.declarations, d => isBlockOrCatchScoped(d) || isClassLike(d) || (d.kind === SyntaxKind.EnumDeclaration) ? d : undefined);

            Debug.assert(declaration !== undefined, "Declaration to checkResolvedBlockScopedVariable is undefined");

            if (!(declaration.flags & NodeFlags.Ambient) && !isBlockScopedNameDeclaredBeforeUse(declaration, errorLocation)) {
                if (result.flags & SymbolFlags.BlockScopedVariable) {
                    error(errorLocation, Diagnostics.Block_scoped_variable_0_used_before_its_declaration, declarationNameToString(getNameOfDeclaration(declaration)));
                }
                else if (result.flags & SymbolFlags.Class) {
                    error(errorLocation, Diagnostics.Class_0_used_before_its_declaration, declarationNameToString(getNameOfDeclaration(declaration)));
                }
                else if (result.flags & SymbolFlags.RegularEnum) {
                    error(errorLocation, Diagnostics.Enum_0_used_before_its_declaration, declarationNameToString(getNameOfDeclaration(declaration)));
                }
            }
        }

        /* Starting from 'initial' node walk up the parent chain until 'stopAt' node is reached.
         * If at any point current node is equal to 'parent' node - return true.
         * Return false if 'stopAt' node is reached or isFunctionLike(current) === true.
         */
        function isSameScopeDescendentOf(initial: Node, parent: Node, stopAt: Node): boolean {
            return parent && !!findAncestor(initial, n => n === stopAt || isFunctionLike(n) ? "quit" : n === parent);
        }

        function getAnyImportSyntax(node: Node): AnyImportSyntax | undefined {
            switch (node.kind) {
                case SyntaxKind.ImportEqualsDeclaration:
                    return node as ImportEqualsDeclaration;
                case SyntaxKind.ImportClause:
                    return (node as ImportClause).parent;
                case SyntaxKind.NamespaceImport:
                    return (node as NamespaceImport).parent.parent;
                case SyntaxKind.ImportSpecifier:
                    return (node as ImportSpecifier).parent.parent.parent;
                default:
                    return undefined;
            }
        }

        function getDeclarationOfAliasSymbol(symbol: Symbol): Declaration | undefined {
            return find<Declaration>(symbol.declarations, isAliasSymbolDeclaration);
        }

        function getTargetOfImportEqualsDeclaration(node: ImportEqualsDeclaration, dontResolveAlias: boolean): Symbol {
            if (node.moduleReference.kind === SyntaxKind.ExternalModuleReference) {
                return resolveExternalModuleSymbol(resolveExternalModuleName(node, getExternalModuleImportEqualsDeclarationExpression(node)));
            }
            return getSymbolOfPartOfRightHandSideOfImportEquals(node.moduleReference, dontResolveAlias);
        }

        function resolveExportByName(moduleSymbol: Symbol, name: __String, dontResolveAlias: boolean) {
            const exportValue = moduleSymbol.exports.get(InternalSymbolName.ExportEquals);
            return exportValue
                ? getPropertyOfType(getTypeOfSymbol(exportValue), name)
                : resolveSymbol(moduleSymbol.exports.get(name), dontResolveAlias);
        }

        function canHaveSyntheticDefault(file: SourceFile | undefined, moduleSymbol: Symbol, dontResolveAlias: boolean) {
            if (!allowSyntheticDefaultImports) {
                return false;
            }
            // Declaration files (and ambient modules)
            if (!file || file.isDeclarationFile) {
                // Definitely cannot have a synthetic default if they have a default member specified
                if (resolveExportByName(moduleSymbol, InternalSymbolName.Default, dontResolveAlias)) {
                    return false;
                }
                // It _might_ still be incorrect to assume there is no __esModule marker on the import at runtime, even if there is no `default` member
                // So we check a bit more,
                if (resolveExportByName(moduleSymbol, escapeLeadingUnderscores("__esModule"), dontResolveAlias)) {
                    // If there is an `__esModule` specified in the declaration (meaning someone explicitly added it or wrote it in their code),
                    // it definitely is a module and does not have a synthetic default
                    return false;
                }
                // There are _many_ declaration files not written with esmodules in mind that still get compiled into a format with __esModule set
                // Meaning there may be no default at runtime - however to be on the permissive side, we allow access to a synthetic default member
                // as there is no marker to indicate if the accompanying JS has `__esModule` or not, or is even native esm
                return true;
            }
            // TypeScript files never have a synthetic default (as they are always emitted with an __esModule marker) _unless_ they contain an export= statement
            if (!isSourceFileJavaScript(file)) {
                return hasExportAssignmentSymbol(moduleSymbol);
            }
            // JS files have a synthetic default if they do not contain ES2015+ module syntax (export = is not valid in js) _and_ do not have an __esModule marker
            return !file.externalModuleIndicator && !resolveExportByName(moduleSymbol, escapeLeadingUnderscores("__esModule"), dontResolveAlias);
        }

        function getTargetOfImportClause(node: ImportClause, dontResolveAlias: boolean): Symbol {
            const moduleSymbol = resolveExternalModuleName(node, node.parent.moduleSpecifier);

            if (moduleSymbol) {
                let exportDefaultSymbol: Symbol;
                if (isShorthandAmbientModuleSymbol(moduleSymbol)) {
                    exportDefaultSymbol = moduleSymbol;
                }
                else {
                    exportDefaultSymbol = resolveExportByName(moduleSymbol, InternalSymbolName.Default, dontResolveAlias);
                }

                const file = find(moduleSymbol.declarations, isSourceFile);
                const hasSyntheticDefault = canHaveSyntheticDefault(file, moduleSymbol, dontResolveAlias);
                if (!exportDefaultSymbol && !hasSyntheticDefault) {
                    error(node.name, Diagnostics.Module_0_has_no_default_export, symbolToString(moduleSymbol));
                }
                else if (!exportDefaultSymbol && hasSyntheticDefault) {
                    // per emit behavior, a synthetic default overrides a "real" .default member if `__esModule` is not present
                    return resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias) || resolveSymbol(moduleSymbol, dontResolveAlias);
                }
                return exportDefaultSymbol;
            }
        }

        function getTargetOfNamespaceImport(node: NamespaceImport, dontResolveAlias: boolean): Symbol {
            const moduleSpecifier = node.parent.parent.moduleSpecifier;
            return resolveESModuleSymbol(resolveExternalModuleName(node, moduleSpecifier), moduleSpecifier, dontResolveAlias);
        }

        // This function creates a synthetic symbol that combines the value side of one symbol with the
        // type/namespace side of another symbol. Consider this example:
        //
        //   declare module graphics {
        //       interface Point {
        //           x: number;
        //           y: number;
        //       }
        //   }
        //   declare var graphics: {
        //       Point: new (x: number, y: number) => graphics.Point;
        //   }
        //   declare module "graphics" {
        //       export = graphics;
        //   }
        //
        // An 'import { Point } from "graphics"' needs to create a symbol that combines the value side 'Point'
        // property with the type/namespace side interface 'Point'.
        function combineValueAndTypeSymbols(valueSymbol: Symbol, typeSymbol: Symbol): Symbol {
            if (valueSymbol === unknownSymbol && typeSymbol === unknownSymbol) {
                return unknownSymbol;
            }
            if (valueSymbol.flags & (SymbolFlags.Type | SymbolFlags.Namespace)) {
                return valueSymbol;
            }
            const result = createSymbol(valueSymbol.flags | typeSymbol.flags, valueSymbol.escapedName);
            result.declarations = concatenate(valueSymbol.declarations, typeSymbol.declarations);
            result.parent = valueSymbol.parent || typeSymbol.parent;
            if (valueSymbol.valueDeclaration) result.valueDeclaration = valueSymbol.valueDeclaration;
            if (typeSymbol.members) result.members = typeSymbol.members;
            if (valueSymbol.exports) result.exports = valueSymbol.exports;
            return result;
        }

        function getExportOfModule(symbol: Symbol, name: __String, dontResolveAlias: boolean): Symbol {
            if (symbol.flags & SymbolFlags.Module) {
                return resolveSymbol(getExportsOfSymbol(symbol).get(name), dontResolveAlias);
            }
        }

        function getPropertyOfVariable(symbol: Symbol, name: __String): Symbol {
            if (symbol.flags & SymbolFlags.Variable) {
                const typeAnnotation = (<VariableDeclaration>symbol.valueDeclaration).type;
                if (typeAnnotation) {
                    return resolveSymbol(getPropertyOfType(getTypeFromTypeNode(typeAnnotation), name));
                }
            }
        }

        function getExternalModuleMember(node: ImportDeclaration | ExportDeclaration, specifier: ImportOrExportSpecifier, dontResolveAlias?: boolean): Symbol {
            const moduleSymbol = resolveExternalModuleName(node, node.moduleSpecifier);
            const targetSymbol = resolveESModuleSymbol(moduleSymbol, node.moduleSpecifier, dontResolveAlias);
            if (targetSymbol) {
                const name = specifier.propertyName || specifier.name;
                if (name.escapedText) {
                    if (isShorthandAmbientModuleSymbol(moduleSymbol)) {
                        return moduleSymbol;
                    }

                    let symbolFromVariable: Symbol;
                    // First check if module was specified with "export=". If so, get the member from the resolved type
                    if (moduleSymbol && moduleSymbol.exports && moduleSymbol.exports.get("export=" as __String)) {
                        symbolFromVariable = getPropertyOfType(getTypeOfSymbol(targetSymbol), name.escapedText);
                    }
                    else {
                        symbolFromVariable = getPropertyOfVariable(targetSymbol, name.escapedText);
                    }
                    // if symbolFromVariable is export - get its final target
                    symbolFromVariable = resolveSymbol(symbolFromVariable, dontResolveAlias);
                    let symbolFromModule = getExportOfModule(targetSymbol, name.escapedText, dontResolveAlias);
                    // If the export member we're looking for is default, and there is no real default but allowSyntheticDefaultImports is on, return the entire module as the default
                    if (!symbolFromModule && allowSyntheticDefaultImports && name.escapedText === InternalSymbolName.Default) {
                        symbolFromModule = resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias) || resolveSymbol(moduleSymbol, dontResolveAlias);
                    }
                    const symbol = symbolFromModule && symbolFromVariable ?
                        combineValueAndTypeSymbols(symbolFromVariable, symbolFromModule) :
                        symbolFromModule || symbolFromVariable;
                    if (!symbol) {
                        const moduleName = getFullyQualifiedName(moduleSymbol);
                        const declarationName = declarationNameToString(name);
                        const suggestion = getSuggestionForNonexistentModule(name, targetSymbol);
                        if (suggestion !== undefined) {
                            error(name, Diagnostics.Module_0_has_no_exported_member_1_Did_you_mean_2, moduleName, declarationName, suggestion);
                        }
                        else {
                            error(name, Diagnostics.Module_0_has_no_exported_member_1, moduleName, declarationName);
                        }
                    }
                    return symbol;
                }
            }
        }

        function getTargetOfImportSpecifier(node: ImportSpecifier, dontResolveAlias: boolean): Symbol {
            return getExternalModuleMember(node.parent.parent.parent, node, dontResolveAlias);
        }

        function getTargetOfNamespaceExportDeclaration(node: NamespaceExportDeclaration, dontResolveAlias: boolean): Symbol {
            return resolveExternalModuleSymbol(node.parent.symbol, dontResolveAlias);
        }

        function getTargetOfExportSpecifier(node: ExportSpecifier, meaning: SymbolFlags, dontResolveAlias?: boolean) {
            return node.parent.parent.moduleSpecifier ?
                getExternalModuleMember(node.parent.parent, node, dontResolveAlias) :
                resolveEntityName(node.propertyName || node.name, meaning, /*ignoreErrors*/ false, dontResolveAlias);
        }

        function getTargetOfExportAssignment(node: ExportAssignment, dontResolveAlias: boolean): Symbol {
            return resolveEntityName(<EntityNameExpression>node.expression, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, /*ignoreErrors*/ false, dontResolveAlias);
        }

        function getTargetOfAliasDeclaration(node: Declaration, dontRecursivelyResolve?: boolean): Symbol {
            switch (node.kind) {
                case SyntaxKind.ImportEqualsDeclaration:
                    return getTargetOfImportEqualsDeclaration(<ImportEqualsDeclaration>node, dontRecursivelyResolve);
                case SyntaxKind.ImportClause:
                    return getTargetOfImportClause(<ImportClause>node, dontRecursivelyResolve);
                case SyntaxKind.NamespaceImport:
                    return getTargetOfNamespaceImport(<NamespaceImport>node, dontRecursivelyResolve);
                case SyntaxKind.ImportSpecifier:
                    return getTargetOfImportSpecifier(<ImportSpecifier>node, dontRecursivelyResolve);
                case SyntaxKind.ExportSpecifier:
                    return getTargetOfExportSpecifier(<ExportSpecifier>node, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, dontRecursivelyResolve);
                case SyntaxKind.ExportAssignment:
                    return getTargetOfExportAssignment(<ExportAssignment>node, dontRecursivelyResolve);
                case SyntaxKind.NamespaceExportDeclaration:
                    return getTargetOfNamespaceExportDeclaration(<NamespaceExportDeclaration>node, dontRecursivelyResolve);
            }
        }

        /**
         * Indicates that a symbol is an alias that does not merge with a local declaration.
         */
        function isNonLocalAlias(symbol: Symbol, excludes = SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace) {
            return symbol && (symbol.flags & (SymbolFlags.Alias | excludes)) === SymbolFlags.Alias;
        }

        function resolveSymbol(symbol: Symbol, dontResolveAlias?: boolean): Symbol {
            const shouldResolve = !dontResolveAlias && isNonLocalAlias(symbol);
            return shouldResolve ? resolveAlias(symbol) : symbol;
        }

        function resolveAlias(symbol: Symbol): Symbol {
            Debug.assert((symbol.flags & SymbolFlags.Alias) !== 0, "Should only get Alias here.");
            const links = getSymbolLinks(symbol);
            if (!links.target) {
                links.target = resolvingSymbol;
                const node = getDeclarationOfAliasSymbol(symbol);
                Debug.assert(!!node);
                const target = getTargetOfAliasDeclaration(node);
                if (links.target === resolvingSymbol) {
                    links.target = target || unknownSymbol;
                }
                else {
                    error(node, Diagnostics.Circular_definition_of_import_alias_0, symbolToString(symbol));
                }
            }
            else if (links.target === resolvingSymbol) {
                links.target = unknownSymbol;
            }
            return links.target;
        }

        function markExportAsReferenced(node: ImportEqualsDeclaration | ExportAssignment | ExportSpecifier) {
            const symbol = getSymbolOfNode(node);
            const target = resolveAlias(symbol);
            if (target) {
                const markAlias = target === unknownSymbol ||
                    ((target.flags & SymbolFlags.Value) && !isConstEnumOrConstEnumOnlyModule(target));

                if (markAlias) {
                    markAliasSymbolAsReferenced(symbol);
                }
            }
        }

        // When an alias symbol is referenced, we need to mark the entity it references as referenced and in turn repeat that until
        // we reach a non-alias or an exported entity (which is always considered referenced). We do this by checking the target of
        // the alias as an expression (which recursively takes us back here if the target references another alias).
        function markAliasSymbolAsReferenced(symbol: Symbol) {
            const links = getSymbolLinks(symbol);
            if (!links.referenced) {
                links.referenced = true;
                const node = getDeclarationOfAliasSymbol(symbol);
                Debug.assert(!!node);
                if (node.kind === SyntaxKind.ExportAssignment) {
                    // export default <symbol>
                    checkExpressionCached((<ExportAssignment>node).expression);
                }
                else if (node.kind === SyntaxKind.ExportSpecifier) {
                    // export { <symbol> } or export { <symbol> as foo }
                    checkExpressionCached((<ExportSpecifier>node).propertyName || (<ExportSpecifier>node).name);
                }
                else if (isInternalModuleImportEqualsDeclaration(node)) {
                    // import foo = <symbol>
                    checkExpressionCached(<Expression>node.moduleReference);
                }
            }
        }

        // This function is only for imports with entity names
        function getSymbolOfPartOfRightHandSideOfImportEquals(entityName: EntityName, dontResolveAlias?: boolean): Symbol {
            // There are three things we might try to look for. In the following examples,
            // the search term is enclosed in |...|:
            //
            //     import a = |b|; // Namespace
            //     import a = |b.c|; // Value, type, namespace
            //     import a = |b.c|.d; // Namespace
            if (entityName.kind === SyntaxKind.Identifier && isRightSideOfQualifiedNameOrPropertyAccess(entityName)) {
                entityName = <QualifiedName>entityName.parent;
            }
            // Check for case 1 and 3 in the above example
            if (entityName.kind === SyntaxKind.Identifier || entityName.parent.kind === SyntaxKind.QualifiedName) {
                return resolveEntityName(entityName, SymbolFlags.Namespace, /*ignoreErrors*/ false, dontResolveAlias);
            }
            else {
                // Case 2 in above example
                // entityName.kind could be a QualifiedName or a Missing identifier
                Debug.assert(entityName.parent.kind === SyntaxKind.ImportEqualsDeclaration);
                return resolveEntityName(entityName, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace, /*ignoreErrors*/ false, dontResolveAlias);
            }
        }

        function getFullyQualifiedName(symbol: Symbol): string {
            return symbol.parent ? getFullyQualifiedName(symbol.parent) + "." + symbolToString(symbol) : symbolToString(symbol);
        }

        /**
         * Resolves a qualified name and any involved aliases.
         */
        function resolveEntityName(name: EntityNameOrEntityNameExpression, meaning: SymbolFlags, ignoreErrors?: boolean, dontResolveAlias?: boolean, location?: Node): Symbol | undefined {
            if (nodeIsMissing(name)) {
                return undefined;
            }

            const namespaceMeaning = SymbolFlags.Namespace | (isInJavaScriptFile(name) ? meaning & SymbolFlags.Value : 0);
            let symbol: Symbol;
            if (name.kind === SyntaxKind.Identifier) {
                const message = meaning === namespaceMeaning ? Diagnostics.Cannot_find_namespace_0 : Diagnostics.Cannot_find_name_0;

                symbol = resolveName(location || name, name.escapedText, meaning, ignoreErrors ? undefined : message, name, /*isUse*/ true);
                if (!symbol) {
                    return undefined;
                }
            }
            else if (name.kind === SyntaxKind.QualifiedName || name.kind === SyntaxKind.PropertyAccessExpression) {
                const left = name.kind === SyntaxKind.QualifiedName ? name.left : name.expression;
                const right = name.kind === SyntaxKind.QualifiedName ? name.right : name.name;
                let namespace = resolveEntityName(left, namespaceMeaning, ignoreErrors, /*dontResolveAlias*/ false, location);
                if (!namespace || nodeIsMissing(right)) {
                    return undefined;
                }
                else if (namespace === unknownSymbol) {
                    return namespace;
                }
                if (isInJavaScriptFile(name)) {
                    const initializer = getDeclaredJavascriptInitializer(namespace.valueDeclaration) || getAssignedJavascriptInitializer(namespace.valueDeclaration);
                    if (initializer) {
                        namespace = getSymbolOfNode(initializer);
                    }
                    if (namespace.valueDeclaration &&
                        isVariableDeclaration(namespace.valueDeclaration) &&
                        isCommonJsRequire(namespace.valueDeclaration.initializer)) {
                        const moduleName = (namespace.valueDeclaration.initializer as CallExpression).arguments[0] as StringLiteral;
                        const moduleSym = resolveExternalModuleName(moduleName, moduleName);
                        if (moduleSym) {
                            const resolvedModuleSymbol = resolveExternalModuleSymbol(moduleSym);
                            if (resolvedModuleSymbol) {
                                namespace = resolvedModuleSymbol;
                            }
                        }
                    }
                }
                symbol = getSymbol(getExportsOfSymbol(namespace), right.escapedText, meaning);
                if (!symbol) {
                    if (!ignoreErrors) {
                        error(right, Diagnostics.Namespace_0_has_no_exported_member_1, getFullyQualifiedName(namespace), declarationNameToString(right));
                    }
                    return undefined;
                }
            }
            else {
                Debug.assertNever(name, "Unknown entity name kind.");
            }
            Debug.assert((getCheckFlags(symbol) & CheckFlags.Instantiated) === 0, "Should never get an instantiated symbol here.");
            return (symbol.flags & meaning) || dontResolveAlias ? symbol : resolveAlias(symbol);
        }

        function resolveExternalModuleName(location: Node, moduleReferenceExpression: Expression): Symbol {
            return resolveExternalModuleNameWorker(location, moduleReferenceExpression, Diagnostics.Cannot_find_module_0);
        }

        function resolveExternalModuleNameWorker(location: Node, moduleReferenceExpression: Expression, moduleNotFoundError: DiagnosticMessage, isForAugmentation = false): Symbol {
            return isStringLiteralLike(moduleReferenceExpression)
                ? resolveExternalModule(location, moduleReferenceExpression.text, moduleNotFoundError, moduleReferenceExpression, isForAugmentation)
                : undefined;
        }

        function resolveExternalModule(location: Node, moduleReference: string, moduleNotFoundError: DiagnosticMessage, errorNode: Node, isForAugmentation = false): Symbol {
            if (moduleReference === undefined) {
                return;
            }

            if (startsWith(moduleReference, "@types/")) {
                const diag = Diagnostics.Cannot_import_type_declaration_files_Consider_importing_0_instead_of_1;
                const withoutAtTypePrefix = removePrefix(moduleReference, "@types/");
                error(errorNode, diag, withoutAtTypePrefix, moduleReference);
            }

            const ambientModule = tryFindAmbientModule(moduleReference, /*withAugmentations*/ true);
            if (ambientModule) {
                return ambientModule;
            }
            const resolvedModule = getResolvedModule(getSourceFileOfNode(location), moduleReference);
            const resolutionDiagnostic = resolvedModule && getResolutionDiagnostic(compilerOptions, resolvedModule);
            const sourceFile = resolvedModule && !resolutionDiagnostic && host.getSourceFile(resolvedModule.resolvedFileName);
            if (sourceFile) {
                if (sourceFile.symbol) {
                    if (resolvedModule.isExternalLibraryImport && !extensionIsTypeScript(resolvedModule.extension)) {
                        addSuggestionDiagnostic(createModuleImplicitlyAnyDiagnostic(errorNode, resolvedModule, moduleReference));
                    }
                    // merged symbol is module declaration symbol combined with all augmentations
                    return getMergedSymbol(sourceFile.symbol);
                }
                if (moduleNotFoundError) {
                    // report errors only if it was requested
                    error(errorNode, Diagnostics.File_0_is_not_a_module, sourceFile.fileName);
                }
                return undefined;
            }

            if (patternAmbientModules) {
                const pattern = findBestPatternMatch(patternAmbientModules, _ => _.pattern, moduleReference);
                if (pattern) {
                    return getMergedSymbol(pattern.symbol);
                }
            }

            // May be an untyped module. If so, ignore resolutionDiagnostic.
            if (resolvedModule && !extensionIsTypeScript(resolvedModule.extension) && resolutionDiagnostic === undefined || resolutionDiagnostic === Diagnostics.Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type) {
                if (isForAugmentation) {
                    const diag = Diagnostics.Invalid_module_name_in_augmentation_Module_0_resolves_to_an_untyped_module_at_1_which_cannot_be_augmented;
                    error(errorNode, diag, moduleReference, resolvedModule.resolvedFileName);
                }
                else {
                    addErrorOrSuggestionDiagnostic(noImplicitAny && !!moduleNotFoundError, createModuleImplicitlyAnyDiagnostic(errorNode, resolvedModule, moduleReference));
                }
                // Failed imports and untyped modules are both treated in an untyped manner; only difference is whether we give a diagnostic first.
                return undefined;
            }

            if (moduleNotFoundError) {
                // report errors only if it was requested
                if (resolutionDiagnostic) {
                    error(errorNode, resolutionDiagnostic, moduleReference, resolvedModule.resolvedFileName);
                }
                else {
                    const tsExtension = tryExtractTypeScriptExtension(moduleReference);
                    if (tsExtension) {
                        const diag = Diagnostics.An_import_path_cannot_end_with_a_0_extension_Consider_importing_1_instead;
                        error(errorNode, diag, tsExtension, removeExtension(moduleReference, tsExtension));
                    }
                    else {
                        error(errorNode, moduleNotFoundError, moduleReference);
                    }
                }
            }
            return undefined;
        }

        function createModuleImplicitlyAnyDiagnostic(errorNode: Node, { packageId, resolvedFileName }: ResolvedModuleFull, moduleReference: string): Diagnostic {
            const errorInfo = packageId && chainDiagnosticMessages(
                /*details*/ undefined,
                Diagnostics.Try_npm_install_types_Slash_0_if_it_exists_or_add_a_new_declaration_d_ts_file_containing_declare_module_0,
                getMangledNameForScopedPackage(packageId.name));
            return createDiagnosticForNodeFromMessageChain(errorNode, chainDiagnosticMessages(
                errorInfo,
                Diagnostics.Could_not_find_a_declaration_file_for_module_0_1_implicitly_has_an_any_type,
                moduleReference,
                resolvedFileName));
        }

        // An external module with an 'export =' declaration resolves to the target of the 'export =' declaration,
        // and an external module with no 'export =' declaration resolves to the module itself.
        function resolveExternalModuleSymbol(moduleSymbol: Symbol, dontResolveAlias?: boolean): Symbol {
            return moduleSymbol && getMergedSymbol(resolveSymbol(moduleSymbol.exports.get(InternalSymbolName.ExportEquals), dontResolveAlias)) || moduleSymbol;
        }

        // An external module with an 'export =' declaration may be referenced as an ES6 module provided the 'export ='
        // references a symbol that is at least declared as a module or a variable. The target of the 'export =' may
        // combine other declarations with the module or variable (e.g. a class/module, function/module, interface/variable).
        function resolveESModuleSymbol(moduleSymbol: Symbol, moduleReferenceExpression: Expression, dontResolveAlias: boolean): Symbol {
            const symbol = resolveExternalModuleSymbol(moduleSymbol, dontResolveAlias);
            if (!dontResolveAlias && symbol) {
                if (!(symbol.flags & (SymbolFlags.Module | SymbolFlags.Variable))) {
                    error(moduleReferenceExpression, Diagnostics.Module_0_resolves_to_a_non_module_entity_and_cannot_be_imported_using_this_construct, symbolToString(moduleSymbol));
                    return symbol;
                }
                if (compilerOptions.esModuleInterop) {
                    const referenceParent = moduleReferenceExpression.parent;
                    if (
                        (isImportDeclaration(referenceParent) && getNamespaceDeclarationNode(referenceParent)) ||
                        isImportCall(referenceParent)
                    ) {
                        const type = getTypeOfSymbol(symbol);
                        let sigs = getSignaturesOfStructuredType(type, SignatureKind.Call);
                        if (!sigs || !sigs.length) {
                            sigs = getSignaturesOfStructuredType(type, SignatureKind.Construct);
                        }
                        if (sigs && sigs.length) {
                            const moduleType = getTypeWithSyntheticDefaultImportType(type, symbol, moduleSymbol);
                            // Create a new symbol which has the module's type less the call and construct signatures
                            const result = createSymbol(symbol.flags, symbol.escapedName);
                            result.declarations = symbol.declarations ? symbol.declarations.slice() : [];
                            result.parent = symbol.parent;
                            result.target = symbol;
                            result.originatingImport = referenceParent;
                            if (symbol.valueDeclaration) result.valueDeclaration = symbol.valueDeclaration;
                            if (symbol.constEnumOnlyModule) result.constEnumOnlyModule = true;
                            if (symbol.members) result.members = cloneMap(symbol.members);
                            if (symbol.exports) result.exports = cloneMap(symbol.exports);
                            const resolvedModuleType = resolveStructuredTypeMembers(moduleType as StructuredType); // Should already be resolved from the signature checks above
                            result.type = createAnonymousType(result, resolvedModuleType.members, emptyArray, emptyArray, resolvedModuleType.stringIndexInfo, resolvedModuleType.numberIndexInfo);
                            return result;
                        }
                    }
                }
            }
            return symbol;
        }

        function hasExportAssignmentSymbol(moduleSymbol: Symbol): boolean {
            return moduleSymbol.exports.get(InternalSymbolName.ExportEquals) !== undefined;
        }

        function getExportsOfModuleAsArray(moduleSymbol: Symbol): Symbol[] {
            return symbolsToArray(getExportsOfModule(moduleSymbol));
        }

        function getExportsAndPropertiesOfModule(moduleSymbol: Symbol): Symbol[] {
            const exports = getExportsOfModuleAsArray(moduleSymbol);
            const exportEquals = resolveExternalModuleSymbol(moduleSymbol);
            if (exportEquals !== moduleSymbol) {
                addRange(exports, getPropertiesOfType(getTypeOfSymbol(exportEquals)));
            }
            return exports;
        }

        function tryGetMemberInModuleExports(memberName: __String, moduleSymbol: Symbol): Symbol | undefined {
            const symbolTable = getExportsOfModule(moduleSymbol);
            if (symbolTable) {
                return symbolTable.get(memberName);
            }
        }

        function tryGetMemberInModuleExportsAndProperties(memberName: __String, moduleSymbol: Symbol): Symbol | undefined {
            const symbol = tryGetMemberInModuleExports(memberName, moduleSymbol);
            if (symbol) {
                return symbol;
            }

            const exportEquals = resolveExternalModuleSymbol(moduleSymbol);
            if (exportEquals === moduleSymbol) {
                return undefined;
            }

            const type = getTypeOfSymbol(exportEquals);
            return type.flags & TypeFlags.Primitive ? undefined : getPropertyOfType(type, memberName);
        }

        function getExportsOfSymbol(symbol: Symbol): SymbolTable {
            return symbol.flags & SymbolFlags.Class ? getResolvedMembersOrExportsOfSymbol(symbol, MembersOrExportsResolutionKind.resolvedExports) :
                symbol.flags & SymbolFlags.Module ? getExportsOfModule(symbol) :
                symbol.exports || emptySymbols;
        }

        function getExportsOfModule(moduleSymbol: Symbol): SymbolTable {
            const links = getSymbolLinks(moduleSymbol);
            return links.resolvedExports || (links.resolvedExports = getExportsOfModuleWorker(moduleSymbol));
        }

        interface ExportCollisionTracker {
            specifierText: string;
            exportsWithDuplicate: ExportDeclaration[];
        }

        type ExportCollisionTrackerTable = UnderscoreEscapedMap<ExportCollisionTracker>;

        /**
         * Extends one symbol table with another while collecting information on name collisions for error message generation into the `lookupTable` argument
         * Not passing `lookupTable` and `exportNode` disables this collection, and just extends the tables
         */
        function extendExportSymbols(target: SymbolTable, source: SymbolTable | undefined, lookupTable?: ExportCollisionTrackerTable, exportNode?: ExportDeclaration) {
            if (!source) return;
            source.forEach((sourceSymbol, id) => {
                if (id === InternalSymbolName.Default) return;

                const targetSymbol = target.get(id);
                if (!targetSymbol) {
                    target.set(id, sourceSymbol);
                    if (lookupTable && exportNode) {
                        lookupTable.set(id, {
                            specifierText: getTextOfNode(exportNode.moduleSpecifier)
                        } as ExportCollisionTracker);
                    }
                }
                else if (lookupTable && exportNode && targetSymbol && resolveSymbol(targetSymbol) !== resolveSymbol(sourceSymbol)) {
                    const collisionTracker = lookupTable.get(id);
                    if (!collisionTracker.exportsWithDuplicate) {
                        collisionTracker.exportsWithDuplicate = [exportNode];
                    }
                    else {
                        collisionTracker.exportsWithDuplicate.push(exportNode);
                    }
                }
            });
        }

        function getExportsOfModuleWorker(moduleSymbol: Symbol): SymbolTable {
            const visitedSymbols: Symbol[] = [];

            // A module defined by an 'export=' consists on one export that needs to be resolved
            moduleSymbol = resolveExternalModuleSymbol(moduleSymbol);

            return visit(moduleSymbol) || emptySymbols;

            // The ES6 spec permits export * declarations in a module to circularly reference the module itself. For example,
            // module 'a' can 'export * from "b"' and 'b' can 'export * from "a"' without error.
            function visit(symbol: Symbol): SymbolTable {
                if (!(symbol && symbol.flags & SymbolFlags.HasExports && pushIfUnique(visitedSymbols, symbol))) {
                    return;
                }
                const symbols = cloneMap(symbol.exports);
                // All export * declarations are collected in an __export symbol by the binder
                const exportStars = symbol.exports.get(InternalSymbolName.ExportStar);
                if (exportStars) {
                    const nestedSymbols = createSymbolTable();
                    const lookupTable = createMap<ExportCollisionTracker>() as ExportCollisionTrackerTable;
                    for (const node of exportStars.declarations) {
                        const resolvedModule = resolveExternalModuleName(node, (node as ExportDeclaration).moduleSpecifier);
                        const exportedSymbols = visit(resolvedModule);
                        extendExportSymbols(
                            nestedSymbols,
                            exportedSymbols,
                            lookupTable,
                            node as ExportDeclaration
                        );
                    }
                    lookupTable.forEach(({ exportsWithDuplicate }, id) => {
                        // It's not an error if the file with multiple `export *`s with duplicate names exports a member with that name itself
                        if (id === "export=" || !(exportsWithDuplicate && exportsWithDuplicate.length) || symbols.has(id)) {
                            return;
                        }
                        for (const node of exportsWithDuplicate) {
                            diagnostics.add(createDiagnosticForNode(
                                node,
                                Diagnostics.Module_0_has_already_exported_a_member_named_1_Consider_explicitly_re_exporting_to_resolve_the_ambiguity,
                                lookupTable.get(id).specifierText,
                                unescapeLeadingUnderscores(id)
                            ));
                        }
                    });
                    extendExportSymbols(symbols, nestedSymbols);
                }
                return symbols;
            }
        }

        function getMergedSymbol(symbol: Symbol): Symbol {
            let merged: Symbol;
            return symbol && symbol.mergeId && (merged = mergedSymbols[symbol.mergeId]) ? merged : symbol;
        }

        function getSymbolOfNode(node: Node): Symbol {
            return getMergedSymbol(node.symbol && getLateBoundSymbol(node.symbol));
        }

        function getParentOfSymbol(symbol: Symbol): Symbol {
            return getMergedSymbol(symbol.parent && getLateBoundSymbol(symbol.parent));
        }

        function getExportSymbolOfValueSymbolIfExported(symbol: Symbol): Symbol {
            return symbol && (symbol.flags & SymbolFlags.ExportValue) !== 0
                ? getMergedSymbol(symbol.exportSymbol)
                : symbol;
        }

        function symbolIsValue(symbol: Symbol): boolean {
            return !!(symbol.flags & SymbolFlags.Value || symbol.flags & SymbolFlags.Alias && resolveAlias(symbol).flags & SymbolFlags.Value);
        }

        function findConstructorDeclaration(node: ClassLikeDeclaration): ConstructorDeclaration {
            const members = node.members;
            for (const member of members) {
                if (member.kind === SyntaxKind.Constructor && nodeIsPresent((<ConstructorDeclaration>member).body)) {
                    return <ConstructorDeclaration>member;
                }
            }
        }

        function createType(flags: TypeFlags): Type {
            const result = new Type(checker, flags);
            typeCount++;
            result.id = typeCount;
            return result;
        }

        function createIntrinsicType(kind: TypeFlags, intrinsicName: string): IntrinsicType {
            const type = <IntrinsicType>createType(kind);
            type.intrinsicName = intrinsicName;
            return type;
        }

        function createBooleanType(trueFalseTypes: Type[]): IntrinsicType & UnionType {
            const type = <IntrinsicType & UnionType>getUnionType(trueFalseTypes);
            type.flags |= TypeFlags.Boolean;
            type.intrinsicName = "boolean";
            return type;
        }

        function createObjectType(objectFlags: ObjectFlags, symbol?: Symbol): ObjectType {
            const type = <ObjectType>createType(TypeFlags.Object);
            type.objectFlags = objectFlags;
            type.symbol = symbol;
            return type;
        }

        function createTypeofType() {
            return getUnionType(arrayFrom(typeofEQFacts.keys(), getLiteralType));
        }

        // A reserved member name starts with two underscores, but the third character cannot be an underscore
        // or the @ symbol. A third underscore indicates an escaped form of an identifer that started
        // with at least two underscores. The @ character indicates that the name is denoted by a well known ES
        // Symbol instance.
        function isReservedMemberName(name: __String) {
            return (name as string).charCodeAt(0) === CharacterCodes._ &&
                (name as string).charCodeAt(1) === CharacterCodes._ &&
                (name as string).charCodeAt(2) !== CharacterCodes._ &&
                (name as string).charCodeAt(2) !== CharacterCodes.at;
        }

        function getNamedMembers(members: SymbolTable): Symbol[] {
            let result: Symbol[];
            members.forEach((symbol, id) => {
                if (!isReservedMemberName(id)) {
                    if (!result) result = [];
                    if (symbolIsValue(symbol)) {
                        result.push(symbol);
                    }
                }
            });
            return result || emptyArray;
        }

        function setStructuredTypeMembers(type: StructuredType, members: SymbolTable, callSignatures: Signature[], constructSignatures: Signature[], stringIndexInfo: IndexInfo, numberIndexInfo: IndexInfo): ResolvedType {
            (<ResolvedType>type).members = members;
            (<ResolvedType>type).properties = getNamedMembers(members);
            (<ResolvedType>type).callSignatures = callSignatures;
            (<ResolvedType>type).constructSignatures = constructSignatures;
            if (stringIndexInfo) (<ResolvedType>type).stringIndexInfo = stringIndexInfo;
            if (numberIndexInfo) (<ResolvedType>type).numberIndexInfo = numberIndexInfo;
            return <ResolvedType>type;
        }

        function createAnonymousType(symbol: Symbol, members: SymbolTable, callSignatures: Signature[], constructSignatures: Signature[], stringIndexInfo: IndexInfo, numberIndexInfo: IndexInfo): ResolvedType {
            return setStructuredTypeMembers(createObjectType(ObjectFlags.Anonymous, symbol),
                members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function forEachSymbolTableInScope<T>(enclosingDeclaration: Node, callback: (symbolTable: SymbolTable) => T): T {
            let result: T;
            for (let location = enclosingDeclaration; location; location = location.parent) {
                // Locals of a source file are not in scope (because they get merged into the global symbol table)
                if (location.locals && !isGlobalSourceFile(location)) {
                    if (result = callback(location.locals)) {
                        return result;
                    }
                }
                switch (location.kind) {
                    case SyntaxKind.SourceFile:
                        if (!isExternalOrCommonJsModule(<SourceFile>location)) {
                            break;
                        }
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                        if (result = callback(getSymbolOfNode(location).exports)) {
                            return result;
                        }
                        break;
                }
            }

            return callback(globals);
        }

        function getQualifiedLeftMeaning(rightMeaning: SymbolFlags) {
            // If we are looking in value space, the parent meaning is value, other wise it is namespace
            return rightMeaning === SymbolFlags.Value ? SymbolFlags.Value : SymbolFlags.Namespace;
        }

        function getAccessibleSymbolChain(symbol: Symbol, enclosingDeclaration: Node | undefined, meaning: SymbolFlags, useOnlyExternalAliasing: boolean, visitedSymbolTablesMap: Map<SymbolTable[]> = createMap()): Symbol[] | undefined {
            if (!(symbol && !isPropertyOrMethodDeclarationSymbol(symbol))) {
                return undefined;
            }

            const id = "" + getSymbolId(symbol);
            let visitedSymbolTables: SymbolTable[];
            if (visitedSymbolTablesMap.has(id)) {
                visitedSymbolTables = visitedSymbolTablesMap.get(id);
            }
            else {
                visitedSymbolTablesMap.set(id, visitedSymbolTables = []);
            }
            return forEachSymbolTableInScope(enclosingDeclaration, getAccessibleSymbolChainFromSymbolTable);

            /**
             * @param {ignoreQualification} boolean Set when a symbol is being looked for through the exports of another symbol (meaning we have a route to qualify it already)
             */
            function getAccessibleSymbolChainFromSymbolTable(symbols: SymbolTable, ignoreQualification?: boolean): Symbol[] | undefined {
                if (!pushIfUnique(visitedSymbolTables, symbols)) {
                    return undefined;
                }

                const result = trySymbolTable(symbols, ignoreQualification);
                visitedSymbolTables.pop();
                return result;
            }

            function canQualifySymbol(symbolFromSymbolTable: Symbol, meaning: SymbolFlags) {
                // If the symbol is equivalent and doesn't need further qualification, this symbol is accessible
                return !needsQualification(symbolFromSymbolTable, enclosingDeclaration, meaning) ||
                    // If symbol needs qualification, make sure that parent is accessible, if it is then this symbol is accessible too
                    !!getAccessibleSymbolChain(symbolFromSymbolTable.parent, enclosingDeclaration, getQualifiedLeftMeaning(meaning), useOnlyExternalAliasing, visitedSymbolTablesMap);
            }

            function isAccessible(symbolFromSymbolTable: Symbol, resolvedAliasSymbol?: Symbol, ignoreQualification?: boolean) {
                return symbol === (resolvedAliasSymbol || symbolFromSymbolTable) &&
                    // if the symbolFromSymbolTable is not external module (it could be if it was determined as ambient external module and would be in globals table)
                    // and if symbolFromSymbolTable or alias resolution matches the symbol,
                    // check the symbol can be qualified, it is only then this symbol is accessible
                    !some(symbolFromSymbolTable.declarations, hasExternalModuleSymbol) &&
                    (ignoreQualification || canQualifySymbol(symbolFromSymbolTable, meaning));
            }

            function trySymbolTable(symbols: SymbolTable, ignoreQualification: boolean | undefined) {
                // If symbol is directly available by its name in the symbol table
                if (isAccessible(symbols.get(symbol.escapedName), /*resolvedAliasSymbol*/ undefined, ignoreQualification)) {
                    return [symbol];
                }

                // Check if symbol is any of the alias
                return forEachEntry(symbols, symbolFromSymbolTable => {
                    if (symbolFromSymbolTable.flags & SymbolFlags.Alias
                        && symbolFromSymbolTable.escapedName !== InternalSymbolName.ExportEquals
                        && symbolFromSymbolTable.escapedName !== InternalSymbolName.Default
                        && !(isUMDExportSymbol(symbolFromSymbolTable) && enclosingDeclaration && isExternalModule(getSourceFileOfNode(enclosingDeclaration)))
                        // If `!useOnlyExternalAliasing`, we can use any type of alias to get the name
                        && (!useOnlyExternalAliasing || some(symbolFromSymbolTable.declarations, isExternalModuleImportEqualsDeclaration))) {

                        const resolvedImportedSymbol = resolveAlias(symbolFromSymbolTable);
                        if (isAccessible(symbolFromSymbolTable, resolvedImportedSymbol, ignoreQualification)) {
                            return [symbolFromSymbolTable];
                        }

                        // Look in the exported members, if we can find accessibleSymbolChain, symbol is accessible using this chain
                        // but only if the symbolFromSymbolTable can be qualified
                        const candidateTable = getExportsOfSymbol(resolvedImportedSymbol);
                        const accessibleSymbolsFromExports = candidateTable && getAccessibleSymbolChainFromSymbolTable(candidateTable, /*ignoreQualification*/ true);
                        if (accessibleSymbolsFromExports && canQualifySymbol(symbolFromSymbolTable, getQualifiedLeftMeaning(meaning))) {
                            return [symbolFromSymbolTable].concat(accessibleSymbolsFromExports);
                        }
                    }
                });
            }
        }

        function needsQualification(symbol: Symbol, enclosingDeclaration: Node | undefined, meaning: SymbolFlags) {
            let qualify = false;
            forEachSymbolTableInScope(enclosingDeclaration, symbolTable => {
                // If symbol of this name is not available in the symbol table we are ok
                let symbolFromSymbolTable = symbolTable.get(symbol.escapedName);
                if (!symbolFromSymbolTable) {
                    // Continue to the next symbol table
                    return false;
                }
                // If the symbol with this name is present it should refer to the symbol
                if (symbolFromSymbolTable === symbol) {
                    // No need to qualify
                    return true;
                }

                // Qualify if the symbol from symbol table has same meaning as expected
                symbolFromSymbolTable = (symbolFromSymbolTable.flags & SymbolFlags.Alias && !getDeclarationOfKind(symbolFromSymbolTable, SyntaxKind.ExportSpecifier)) ? resolveAlias(symbolFromSymbolTable) : symbolFromSymbolTable;
                if (symbolFromSymbolTable.flags & meaning) {
                    qualify = true;
                    return true;
                }

                // Continue to the next symbol table
                return false;
            });

            return qualify;
        }

        function isPropertyOrMethodDeclarationSymbol(symbol: Symbol) {
            if (symbol.declarations && symbol.declarations.length) {
                for (const declaration of symbol.declarations) {
                    switch (declaration.kind) {
                        case SyntaxKind.PropertyDeclaration:
                        case SyntaxKind.MethodDeclaration:
                        case SyntaxKind.GetAccessor:
                        case SyntaxKind.SetAccessor:
                            continue;
                        default:
                            return false;
                    }
                }
                return true;
            }
            return false;
        }

        function isTypeSymbolAccessible(typeSymbol: Symbol, enclosingDeclaration: Node): boolean {
            const access = isSymbolAccessible(typeSymbol, enclosingDeclaration, SymbolFlags.Type, /*shouldComputeAliasesToMakeVisible*/ false);
            return access.accessibility === SymbolAccessibility.Accessible;
        }

        function isValueSymbolAccessible(typeSymbol: Symbol, enclosingDeclaration: Node): boolean {
            const access = isSymbolAccessible(typeSymbol, enclosingDeclaration, SymbolFlags.Value, /*shouldComputeAliasesToMakeVisible*/ false);
            return access.accessibility === SymbolAccessibility.Accessible;
        }

        /**
         * Check if the given symbol in given enclosing declaration is accessible and mark all associated alias to be visible if requested
         *
         * @param symbol a Symbol to check if accessible
         * @param enclosingDeclaration a Node containing reference to the symbol
         * @param meaning a SymbolFlags to check if such meaning of the symbol is accessible
         * @param shouldComputeAliasToMakeVisible a boolean value to indicate whether to return aliases to be mark visible in case the symbol is accessible
         */
        function isSymbolAccessible(symbol: Symbol, enclosingDeclaration: Node, meaning: SymbolFlags, shouldComputeAliasesToMakeVisible: boolean): SymbolAccessibilityResult {
            if (symbol && enclosingDeclaration && !(symbol.flags & SymbolFlags.TypeParameter)) {
                const initialSymbol = symbol;
                let meaningToLook = meaning;
                while (symbol) {
                    // Symbol is accessible if it by itself is accessible
                    const accessibleSymbolChain = getAccessibleSymbolChain(symbol, enclosingDeclaration, meaningToLook, /*useOnlyExternalAliasing*/ false);
                    if (accessibleSymbolChain) {
                        const hasAccessibleDeclarations = hasVisibleDeclarations(accessibleSymbolChain[0], shouldComputeAliasesToMakeVisible);
                        if (!hasAccessibleDeclarations) {
                            return {
                                accessibility: SymbolAccessibility.NotAccessible,
                                errorSymbolName: symbolToString(initialSymbol, enclosingDeclaration, meaning),
                                errorModuleName: symbol !== initialSymbol ? symbolToString(symbol, enclosingDeclaration, SymbolFlags.Namespace) : undefined,
                            };
                        }
                        return hasAccessibleDeclarations;
                    }

                    // If we haven't got the accessible symbol, it doesn't mean the symbol is actually inaccessible.
                    // It could be a qualified symbol and hence verify the path
                    // e.g.:
                    // module m {
                    //     export class c {
                    //     }
                    // }
                    // const x: typeof m.c
                    // In the above example when we start with checking if typeof m.c symbol is accessible,
                    // we are going to see if c can be accessed in scope directly.
                    // But it can't, hence the accessible is going to be undefined, but that doesn't mean m.c is inaccessible
                    // It is accessible if the parent m is accessible because then m.c can be accessed through qualification
                    meaningToLook = getQualifiedLeftMeaning(meaning);
                    symbol = getParentOfSymbol(symbol);
                }

                // This could be a symbol that is not exported in the external module
                // or it could be a symbol from different external module that is not aliased and hence cannot be named
                const symbolExternalModule = forEach(initialSymbol.declarations, getExternalModuleContainer);
                if (symbolExternalModule) {
                    const enclosingExternalModule = getExternalModuleContainer(enclosingDeclaration);
                    if (symbolExternalModule !== enclosingExternalModule) {
                        // name from different external module that is not visible
                        return {
                            accessibility: SymbolAccessibility.CannotBeNamed,
                            errorSymbolName: symbolToString(initialSymbol, enclosingDeclaration, meaning),
                            errorModuleName: symbolToString(symbolExternalModule)
                        };
                    }
                }

                // Just a local name that is not accessible
                return {
                    accessibility: SymbolAccessibility.NotAccessible,
                    errorSymbolName: symbolToString(initialSymbol, enclosingDeclaration, meaning),
                };
            }

            return { accessibility: SymbolAccessibility.Accessible };

            function getExternalModuleContainer(declaration: Node) {
                const node = findAncestor(declaration, hasExternalModuleSymbol);
                return node && getSymbolOfNode(node);
            }
        }

        function hasExternalModuleSymbol(declaration: Node) {
            return isAmbientModule(declaration) || (declaration.kind === SyntaxKind.SourceFile && isExternalOrCommonJsModule(<SourceFile>declaration));
        }

        function hasVisibleDeclarations(symbol: Symbol, shouldComputeAliasToMakeVisible: boolean): SymbolVisibilityResult {
            let aliasesToMakeVisible: AnyImportSyntax[];
            if (forEach(symbol.declarations, declaration => !getIsDeclarationVisible(declaration))) {
                return undefined;
            }
            return { accessibility: SymbolAccessibility.Accessible, aliasesToMakeVisible };

            function getIsDeclarationVisible(declaration: Declaration) {
                if (!isDeclarationVisible(declaration)) {
                    // Mark the unexported alias as visible if its parent is visible
                    // because these kind of aliases can be used to name types in declaration file

                    const anyImportSyntax = getAnyImportSyntax(declaration);
                    if (anyImportSyntax &&
                        !hasModifier(anyImportSyntax, ModifierFlags.Export) && // import clause without export
                        isDeclarationVisible(<Declaration>anyImportSyntax.parent)) {
                        // In function "buildTypeDisplay" where we decide whether to write type-alias or serialize types,
                        // we want to just check if type- alias is accessible or not but we don't care about emitting those alias at that time
                        // since we will do the emitting later in trackSymbol.
                        if (shouldComputeAliasToMakeVisible) {
                            getNodeLinks(declaration).isVisible = true;
                            aliasesToMakeVisible = appendIfUnique(aliasesToMakeVisible, anyImportSyntax);
                        }
                        return true;
                    }

                    // Declaration is not visible
                    return false;
                }

                return true;
            }
        }

        function isEntityNameVisible(entityName: EntityNameOrEntityNameExpression, enclosingDeclaration: Node): SymbolVisibilityResult {
            // get symbol of the first identifier of the entityName
            let meaning: SymbolFlags;
            if (entityName.parent.kind === SyntaxKind.TypeQuery ||
                isExpressionWithTypeArgumentsInClassExtendsClause(entityName.parent) ||
                entityName.parent.kind === SyntaxKind.ComputedPropertyName) {
                // Typeof value
                meaning = SymbolFlags.Value | SymbolFlags.ExportValue;
            }
            else if (entityName.kind === SyntaxKind.QualifiedName || entityName.kind === SyntaxKind.PropertyAccessExpression ||
                entityName.parent.kind === SyntaxKind.ImportEqualsDeclaration) {
                // Left identifier from type reference or TypeAlias
                // Entity name of the import declaration
                meaning = SymbolFlags.Namespace;
            }
            else {
                // Type Reference or TypeAlias entity = Identifier
                meaning = SymbolFlags.Type;
            }

            const firstIdentifier = getFirstIdentifier(entityName);
            const symbol = resolveName(enclosingDeclaration, firstIdentifier.escapedText, meaning, /*nodeNotFoundErrorMessage*/ undefined, /*nameArg*/ undefined, /*isUse*/ false);

            // Verify if the symbol is accessible
            return (symbol && hasVisibleDeclarations(symbol, /*shouldComputeAliasToMakeVisible*/ true)) || {
                accessibility: SymbolAccessibility.NotAccessible,
                errorSymbolName: getTextOfNode(firstIdentifier),
                errorNode: firstIdentifier
            };
        }

        function symbolToString(symbol: Symbol, enclosingDeclaration?: Node, meaning?: SymbolFlags, flags: SymbolFormatFlags = SymbolFormatFlags.AllowAnyNodeKind, writer?: EmitTextWriter): string {
            let nodeFlags = NodeBuilderFlags.IgnoreErrors;
            if (flags & SymbolFormatFlags.UseOnlyExternalAliasing) {
                nodeFlags |= NodeBuilderFlags.UseOnlyExternalAliasing;
            }
            if (flags & SymbolFormatFlags.WriteTypeParametersOrArguments) {
                nodeFlags |= NodeBuilderFlags.WriteTypeParametersInQualifiedName;
            }
            if (flags & SymbolFormatFlags.UseAliasDefinedOutsideCurrentScope) {
                nodeFlags |= NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope;
            }
            const builder = flags & SymbolFormatFlags.AllowAnyNodeKind ? nodeBuilder.symbolToExpression : nodeBuilder.symbolToEntityName;
            return writer ? symbolToStringWorker(writer).getText() : usingSingleLineStringWriter(symbolToStringWorker);

            function symbolToStringWorker(writer: EmitTextWriter) {
                const entity = builder(symbol, meaning, enclosingDeclaration, nodeFlags);
                const printer = createPrinter({ removeComments: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, entity, /*sourceFile*/ sourceFile, writer);
                return writer;
            }
        }

        function signatureToString(signature: Signature, enclosingDeclaration?: Node, flags?: TypeFormatFlags, kind?: SignatureKind, writer?: EmitTextWriter): string {
            return writer ? signatureToStringWorker(writer).getText() : usingSingleLineStringWriter(signatureToStringWorker);

            function signatureToStringWorker(writer: EmitTextWriter) {
                let sigOutput: SyntaxKind;
                if (flags & TypeFormatFlags.WriteArrowStyleSignature) {
                    sigOutput = kind === SignatureKind.Construct ? SyntaxKind.ConstructorType : SyntaxKind.FunctionType;
                }
                else {
                    sigOutput = kind === SignatureKind.Construct ? SyntaxKind.ConstructSignature : SyntaxKind.CallSignature;
                }
                const sig = nodeBuilder.signatureToSignatureDeclaration(signature, sigOutput, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | NodeBuilderFlags.WriteTypeParametersInQualifiedName);
                const printer = createPrinter({ removeComments: true, omitTrailingSemicolon: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, sig, /*sourceFile*/ sourceFile, writer);
                return writer;
            }
        }

        function typeToString(type: Type, enclosingDeclaration?: Node, flags: TypeFormatFlags = TypeFormatFlags.AllowUniqueESSymbolType, writer: EmitTextWriter = createTextWriter("")): string {
            const typeNode = nodeBuilder.typeToTypeNode(type, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors, writer);
            Debug.assert(typeNode !== undefined, "should always get typenode");
            const options = { removeComments: true };
            const printer = createPrinter(options);
            const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
            printer.writeNode(EmitHint.Unspecified, typeNode, /*sourceFile*/ sourceFile, writer);
            const result = writer.getText();

            const maxLength = compilerOptions.noErrorTruncation || flags & TypeFormatFlags.NoTruncation ? undefined : 100;
            if (maxLength && result && result.length >= maxLength) {
                return result.substr(0, maxLength - "...".length) + "...";
            }
            return result;
        }

        function toNodeBuilderFlags(flags?: TypeFormatFlags): NodeBuilderFlags {
            return flags & TypeFormatFlags.NodeBuilderFlagsMask;
        }

        function createNodeBuilder() {
            return {
                typeToTypeNode: (type: Type, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = typeToTypeNodeHelper(type, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                indexInfoToIndexSignatureDeclaration: (indexInfo: IndexInfo, kind: IndexKind, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = indexInfoToIndexSignatureDeclarationHelper(indexInfo, kind, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                signatureToSignatureDeclaration: (signature: Signature, kind: SyntaxKind, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = signatureToSignatureDeclarationHelper(signature, kind, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                symbolToEntityName: (symbol: Symbol, meaning: SymbolFlags, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = symbolToName(symbol, context, meaning, /*expectsIdentifier*/ false);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                symbolToExpression: (symbol: Symbol, meaning: SymbolFlags, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = symbolToExpression(symbol, context, meaning);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                symbolToTypeParameterDeclarations: (symbol: Symbol, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = typeParametersToTypeParameterDeclarations(symbol, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                symbolToParameterDeclaration: (symbol: Symbol, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = symbolToParameterDeclaration(symbol, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
                typeParameterToDeclaration: (parameter: TypeParameter, enclosingDeclaration?: Node, flags?: NodeBuilderFlags, tracker?: SymbolTracker) => {
                    Debug.assert(enclosingDeclaration === undefined || (enclosingDeclaration.flags & NodeFlags.Synthesized) === 0);
                    const context = createNodeBuilderContext(enclosingDeclaration, flags, tracker);
                    const resultingNode = typeParameterToDeclaration(parameter, context);
                    const result = context.encounteredError ? undefined : resultingNode;
                    return result;
                },
            };

            function createNodeBuilderContext(enclosingDeclaration: Node | undefined, flags: NodeBuilderFlags | undefined, tracker: SymbolTracker | undefined): NodeBuilderContext {
                return {
                    enclosingDeclaration,
                    flags,
                    tracker: tracker && tracker.trackSymbol ? tracker : { trackSymbol: noop },
                    encounteredError: false,
                    symbolStack: undefined
                };
            }

            function typeToTypeNodeHelper(type: Type, context: NodeBuilderContext): TypeNode {
                const inTypeAlias = context.flags & NodeBuilderFlags.InTypeAlias;
                context.flags &= ~NodeBuilderFlags.InTypeAlias;

                if (!type) {
                    context.encounteredError = true;
                    return undefined;
                }

                if (type.flags & TypeFlags.Any) {
                    return createKeywordTypeNode(SyntaxKind.AnyKeyword);
                }
                if (type.flags & TypeFlags.String) {
                    return createKeywordTypeNode(SyntaxKind.StringKeyword);
                }
                if (type.flags & TypeFlags.Number) {
                    return createKeywordTypeNode(SyntaxKind.NumberKeyword);
                }
                if (type.flags & TypeFlags.Boolean) {
                    return createKeywordTypeNode(SyntaxKind.BooleanKeyword);
                }
                if (type.flags & TypeFlags.EnumLiteral && !(type.flags & TypeFlags.Union)) {
                    const parentSymbol = getParentOfSymbol(type.symbol);
                    const parentName = symbolToName(parentSymbol, context, SymbolFlags.Type, /*expectsIdentifier*/ false);
                    const enumLiteralName = getDeclaredTypeOfSymbol(parentSymbol) === type ? parentName : createQualifiedName(parentName, symbolName(type.symbol));
                    return createTypeReferenceNode(enumLiteralName, /*typeArguments*/ undefined);
                }
                if (type.flags & TypeFlags.EnumLike) {
                    const name = symbolToName(type.symbol, context, SymbolFlags.Type, /*expectsIdentifier*/ false);
                    return createTypeReferenceNode(name, /*typeArguments*/ undefined);
                }
                if (type.flags & (TypeFlags.StringLiteral)) {
                    return createLiteralTypeNode(setEmitFlags(createLiteral((<StringLiteralType>type).value), EmitFlags.NoAsciiEscaping));
                }
                if (type.flags & (TypeFlags.NumberLiteral)) {
                    return createLiteralTypeNode((createLiteral((<NumberLiteralType>type).value)));
                }
                if (type.flags & TypeFlags.BooleanLiteral) {
                    return (<IntrinsicType>type).intrinsicName === "true" ? createTrue() : createFalse();
                }
                if (type.flags & TypeFlags.UniqueESSymbol) {
                    if (!(context.flags & NodeBuilderFlags.AllowUniqueESSymbolType)) {
                        if (isValueSymbolAccessible(type.symbol, context.enclosingDeclaration)) {
                            return createTypeQueryNode(symbolToName(type.symbol, context, SymbolFlags.Value, /*expectsIdentifier*/ false));
                        }
                        if (context.tracker.reportInaccessibleUniqueSymbolError) {
                            context.tracker.reportInaccessibleUniqueSymbolError();
                        }
                    }
                    return createTypeOperatorNode(SyntaxKind.UniqueKeyword, createKeywordTypeNode(SyntaxKind.SymbolKeyword));
                }
                if (type.flags & TypeFlags.Void) {
                    return createKeywordTypeNode(SyntaxKind.VoidKeyword);
                }
                if (type.flags & TypeFlags.Undefined) {
                    return createKeywordTypeNode(SyntaxKind.UndefinedKeyword);
                }
                if (type.flags & TypeFlags.Null) {
                    return createKeywordTypeNode(SyntaxKind.NullKeyword);
                }
                if (type.flags & TypeFlags.Never) {
                    return createKeywordTypeNode(SyntaxKind.NeverKeyword);
                }
                if (type.flags & TypeFlags.ESSymbol) {
                    return createKeywordTypeNode(SyntaxKind.SymbolKeyword);
                }
                if (type.flags & TypeFlags.NonPrimitive) {
                    return createKeywordTypeNode(SyntaxKind.ObjectKeyword);
                }
                if (type.flags & TypeFlags.TypeParameter && (type as TypeParameter).isThisType) {
                    if (context.flags & NodeBuilderFlags.InObjectTypeLiteral) {
                        if (!context.encounteredError && !(context.flags & NodeBuilderFlags.AllowThisInObjectLiteral)) {
                            context.encounteredError = true;
                        }
                        if (context.tracker.reportInaccessibleThisError) {
                            context.tracker.reportInaccessibleThisError();
                        }
                    }
                    return createThis();
                }

                const objectFlags = getObjectFlags(type);

                if (objectFlags & ObjectFlags.Reference) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    return typeReferenceToTypeNode(<TypeReference>type);
                }
                if (type.flags & TypeFlags.TypeParameter || objectFlags & ObjectFlags.ClassOrInterface) {
                    const name = type.symbol ? symbolToName(type.symbol, context, SymbolFlags.Type, /*expectsIdentifier*/ false) : createIdentifier("?");
                    // Ignore constraint/default when creating a usage (as opposed to declaration) of a type parameter.
                    return createTypeReferenceNode(name, /*typeArguments*/ undefined);
                }
                if (!inTypeAlias && type.aliasSymbol && (context.flags & NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope || isTypeSymbolAccessible(type.aliasSymbol, context.enclosingDeclaration))) {
                    const name = symbolToTypeReferenceName(type.aliasSymbol);
                    const typeArgumentNodes = mapToTypeNodes(type.aliasTypeArguments, context);
                    return createTypeReferenceNode(name, typeArgumentNodes);
                }
                if (type.flags & (TypeFlags.Union | TypeFlags.Intersection)) {
                    const types = type.flags & TypeFlags.Union ? formatUnionTypes((<UnionType>type).types) : (<IntersectionType>type).types;
                    const typeNodes = mapToTypeNodes(types, context);
                    if (typeNodes && typeNodes.length > 0) {
                        const unionOrIntersectionTypeNode = createUnionOrIntersectionTypeNode(type.flags & TypeFlags.Union ? SyntaxKind.UnionType : SyntaxKind.IntersectionType, typeNodes);
                        return unionOrIntersectionTypeNode;
                    }
                    else {
                        if (!context.encounteredError && !(context.flags & NodeBuilderFlags.AllowEmptyUnionOrIntersection)) {
                            context.encounteredError = true;
                        }
                        return undefined;
                    }
                }
                if (objectFlags & (ObjectFlags.Anonymous | ObjectFlags.Mapped)) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    // The type is an object literal type.
                    return createAnonymousTypeNode(<ObjectType>type);
                }
                if (type.flags & TypeFlags.Index) {
                    const indexedType = (<IndexType>type).type;
                    const indexTypeNode = typeToTypeNodeHelper(indexedType, context);
                    return createTypeOperatorNode(indexTypeNode);
                }
                if (type.flags & TypeFlags.IndexedAccess) {
                    const objectTypeNode = typeToTypeNodeHelper((<IndexedAccessType>type).objectType, context);
                    const indexTypeNode = typeToTypeNodeHelper((<IndexedAccessType>type).indexType, context);
                    return createIndexedAccessTypeNode(objectTypeNode, indexTypeNode);
                }
                if (type.flags & TypeFlags.Conditional) {
                    const checkTypeNode = typeToTypeNodeHelper((<ConditionalType>type).checkType, context);
                    const extendsTypeNode = typeToTypeNodeHelper((<ConditionalType>type).extendsType, context);
                    const trueTypeNode = typeToTypeNodeHelper(getTrueTypeFromConditionalType(<ConditionalType>type), context);
                    const falseTypeNode = typeToTypeNodeHelper(getFalseTypeFromConditionalType(<ConditionalType>type), context);
                    return createConditionalTypeNode(checkTypeNode, extendsTypeNode, trueTypeNode, falseTypeNode);
                }
                if (type.flags & TypeFlags.Substitution) {
                    return typeToTypeNodeHelper((<SubstitutionType>type).typeParameter, context);
                }

                Debug.fail("Should be unreachable.");

                function createMappedTypeNodeFromType(type: MappedType) {
                    Debug.assert(!!(type.flags & TypeFlags.Object));
                    const readonlyToken = type.declaration.readonlyToken ? <ReadonlyToken | PlusToken | MinusToken>createToken(type.declaration.readonlyToken.kind) : undefined;
                    const questionToken = type.declaration.questionToken ? <QuestionToken | PlusToken | MinusToken>createToken(type.declaration.questionToken.kind) : undefined;
                    const typeParameterNode = typeParameterToDeclaration(getTypeParameterFromMappedType(type), context, getConstraintTypeFromMappedType(type));
                    const templateTypeNode = typeToTypeNodeHelper(getTemplateTypeFromMappedType(type), context);
                    const mappedTypeNode = createMappedTypeNode(readonlyToken, typeParameterNode, questionToken, templateTypeNode);
                    return setEmitFlags(mappedTypeNode, EmitFlags.SingleLine);
                }

                function createAnonymousTypeNode(type: ObjectType): TypeNode {
                    const symbol = type.symbol;
                    if (symbol) {
                        // Always use 'typeof T' for type of class, enum, and module objects
                        if (symbol.flags & SymbolFlags.Class && !getBaseTypeVariableOfClass(symbol) && !(symbol.valueDeclaration.kind === SyntaxKind.ClassExpression && context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral) ||
                            symbol.flags & (SymbolFlags.Enum | SymbolFlags.ValueModule) ||
                            shouldWriteTypeOfFunctionSymbol()) {
                            return createTypeQueryNodeFromSymbol(symbol, SymbolFlags.Value);
                        }
                        else if (contains(context.symbolStack, symbol)) {
                            // If type is an anonymous type literal in a type alias declaration, use type alias name
                            const typeAlias = getTypeAliasForTypeLiteral(type);
                            if (typeAlias) {
                                // The specified symbol flags need to be reinterpreted as type flags
                                const entityName = symbolToName(typeAlias, context, SymbolFlags.Type, /*expectsIdentifier*/ false);
                                return createTypeReferenceNode(entityName, /*typeArguments*/ undefined);
                            }
                            else {
                                return createKeywordTypeNode(SyntaxKind.AnyKeyword);
                            }
                        }
                        else {
                            // Since instantiations of the same anonymous type have the same symbol, tracking symbols instead
                            // of types allows us to catch circular references to instantiations of the same anonymous type
                            if (!context.symbolStack) {
                                context.symbolStack = [];
                            }

                            const isConstructorObject = getObjectFlags(type) & ObjectFlags.Anonymous && type.symbol && type.symbol.flags & SymbolFlags.Class;
                            if (isConstructorObject) {
                                return createTypeNodeFromObjectType(type);
                            }
                            else {
                                context.symbolStack.push(symbol);
                                const result = createTypeNodeFromObjectType(type);
                                context.symbolStack.pop();
                                return result;
                            }
                        }
                    }
                    else {
                        // Anonymous types without a symbol are never circular.
                        return createTypeNodeFromObjectType(type);
                    }

                    function shouldWriteTypeOfFunctionSymbol() {
                        const isStaticMethodSymbol = !!(symbol.flags & SymbolFlags.Method) &&  // typeof static method
                            some(symbol.declarations, declaration => hasModifier(declaration, ModifierFlags.Static));
                        const isNonLocalFunctionSymbol = !!(symbol.flags & SymbolFlags.Function) &&
                            (symbol.parent || // is exported function symbol
                                forEach(symbol.declarations, declaration =>
                                    declaration.parent.kind === SyntaxKind.SourceFile || declaration.parent.kind === SyntaxKind.ModuleBlock));
                        if (isStaticMethodSymbol || isNonLocalFunctionSymbol) {
                            // typeof is allowed only for static/non local functions
                            return (!!(context.flags & NodeBuilderFlags.UseTypeOfFunction) || contains(context.symbolStack, symbol)) && // it is type of the symbol uses itself recursively
                                (!(context.flags & NodeBuilderFlags.UseStructuralFallback) || isValueSymbolAccessible(symbol, context.enclosingDeclaration)); // And the build is going to succeed without visibility error or there is no structural fallback allowed
                        }
                    }
                }

                function createTypeNodeFromObjectType(type: ObjectType): TypeNode {
                    if (isGenericMappedType(type)) {
                        return createMappedTypeNodeFromType(type);
                    }

                    const resolved = resolveStructuredTypeMembers(type);
                    if (!resolved.properties.length && !resolved.stringIndexInfo && !resolved.numberIndexInfo) {
                        if (!resolved.callSignatures.length && !resolved.constructSignatures.length) {
                            return setEmitFlags(createTypeLiteralNode(/*members*/ undefined), EmitFlags.SingleLine);
                        }

                        if (resolved.callSignatures.length === 1 && !resolved.constructSignatures.length) {
                            const signature = resolved.callSignatures[0];
                            const signatureNode = <FunctionTypeNode>signatureToSignatureDeclarationHelper(signature, SyntaxKind.FunctionType, context);
                            return signatureNode;

                        }

                        if (resolved.constructSignatures.length === 1 && !resolved.callSignatures.length) {
                            const signature = resolved.constructSignatures[0];
                            const signatureNode = <ConstructorTypeNode>signatureToSignatureDeclarationHelper(signature, SyntaxKind.ConstructorType, context);
                            return signatureNode;
                        }
                    }

                    const savedFlags = context.flags;
                    context.flags |= NodeBuilderFlags.InObjectTypeLiteral;
                    const members = createTypeNodesFromResolvedType(resolved);
                    context.flags = savedFlags;
                    const typeLiteralNode = createTypeLiteralNode(members);
                    return setEmitFlags(typeLiteralNode, (context.flags & NodeBuilderFlags.MultilineObjectLiterals) ? 0 : EmitFlags.SingleLine);
                }

                function createTypeQueryNodeFromSymbol(symbol: Symbol, symbolFlags: SymbolFlags) {
                    const entityName = symbolToName(symbol, context, symbolFlags, /*expectsIdentifier*/ false);
                    return createTypeQueryNode(entityName);
                }

                function symbolToTypeReferenceName(symbol: Symbol) {
                    // Unnamed function expressions and arrow functions have reserved names that we don't want to display
                    const entityName = symbol.flags & SymbolFlags.Class || !isReservedMemberName(symbol.escapedName) ? symbolToName(symbol, context, SymbolFlags.Type, /*expectsIdentifier*/ false) : createIdentifier("");
                    return entityName;
                }

                function typeReferenceToTypeNode(type: TypeReference) {
                    const typeArguments: Type[] = type.typeArguments || emptyArray;
                    if (type.target === globalArrayType) {
                        if (context.flags & NodeBuilderFlags.WriteArrayAsGenericType) {
                            const typeArgumentNode = typeToTypeNodeHelper(typeArguments[0], context);
                            return createTypeReferenceNode("Array", [typeArgumentNode]);
                        }

                        const elementType = typeToTypeNodeHelper(typeArguments[0], context);
                        return createArrayTypeNode(elementType);
                    }
                    else if (type.target.objectFlags & ObjectFlags.Tuple) {
                        if (typeArguments.length > 0) {
                            const tupleConstituentNodes = mapToTypeNodes(typeArguments.slice(0, getTypeReferenceArity(type)), context);
                            if (tupleConstituentNodes && tupleConstituentNodes.length > 0) {
                                return createTupleTypeNode(tupleConstituentNodes);
                            }
                        }
                        if (context.encounteredError || (context.flags & NodeBuilderFlags.AllowEmptyTuple)) {
                            return createTupleTypeNode([]);
                        }
                        context.encounteredError = true;
                        return undefined;
                    }
                    else if (context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral &&
                        type.symbol.valueDeclaration &&
                        type.symbol.valueDeclaration.kind === SyntaxKind.ClassExpression) {
                        return createAnonymousTypeNode(type);
                    }
                    else {
                        const outerTypeParameters = type.target.outerTypeParameters;
                        let i = 0;
                        let qualifiedName: QualifiedName | undefined;
                        if (outerTypeParameters) {
                            const length = outerTypeParameters.length;
                            while (i < length) {
                                // Find group of type arguments for type parameters with the same declaring container.
                                const start = i;
                                const parent = getParentSymbolOfTypeParameter(outerTypeParameters[i]);
                                do {
                                    i++;
                                } while (i < length && getParentSymbolOfTypeParameter(outerTypeParameters[i]) === parent);
                                // When type parameters are their own type arguments for the whole group (i.e. we have
                                // the default outer type arguments), we don't show the group.
                                if (!rangeEquals(outerTypeParameters, typeArguments, start, i)) {
                                    const typeArgumentSlice = mapToTypeNodes(typeArguments.slice(start, i), context);
                                    const typeArgumentNodes = typeArgumentSlice && createNodeArray(typeArgumentSlice);
                                    const namePart = symbolToTypeReferenceName(parent);
                                    (namePart.kind === SyntaxKind.Identifier ? namePart : namePart.right).typeArguments = typeArgumentNodes;

                                    if (qualifiedName) {
                                        Debug.assert(!qualifiedName.right);
                                        qualifiedName = addToQualifiedNameMissingRightIdentifier(qualifiedName, namePart);
                                        qualifiedName = createQualifiedName(qualifiedName, /*right*/ undefined);
                                    }
                                    else {
                                        qualifiedName = createQualifiedName(namePart, /*right*/ undefined);
                                    }
                                }
                            }
                        }

                        let entityName: EntityName;
                        const nameIdentifier = symbolToTypeReferenceName(type.symbol);
                        if (qualifiedName) {
                            Debug.assert(!qualifiedName.right);
                            qualifiedName = addToQualifiedNameMissingRightIdentifier(qualifiedName, nameIdentifier);
                            entityName = qualifiedName;
                        }
                        else {
                            entityName = nameIdentifier;
                        }

                        let typeArgumentNodes: ReadonlyArray<TypeNode> | undefined;
                        if (typeArguments.length > 0) {
                            const typeParameterCount = (type.target.typeParameters || emptyArray).length;
                            typeArgumentNodes = mapToTypeNodes(typeArguments.slice(i, typeParameterCount), context);
                        }

                        if (typeArgumentNodes) {
                            const lastIdentifier = entityName.kind === SyntaxKind.Identifier ? entityName : entityName.right;
                            lastIdentifier.typeArguments = undefined;
                        }

                        return createTypeReferenceNode(entityName, typeArgumentNodes);
                    }
                }

                function addToQualifiedNameMissingRightIdentifier(left: QualifiedName, right: Identifier | QualifiedName) {
                    Debug.assert(left.right === undefined);

                    if (right.kind === SyntaxKind.Identifier) {
                        left.right = right;
                        return left;
                    }

                    let rightPart = right;
                    while (rightPart.left.kind !== SyntaxKind.Identifier) {
                        rightPart = rightPart.left;
                    }

                    left.right = rightPart.left;
                    rightPart.left = left;
                    return right;
                }

                function createTypeNodesFromResolvedType(resolvedType: ResolvedType): TypeElement[] {
                    const typeElements: TypeElement[] = [];
                    for (const signature of resolvedType.callSignatures) {
                        typeElements.push(<CallSignatureDeclaration>signatureToSignatureDeclarationHelper(signature, SyntaxKind.CallSignature, context));
                    }
                    for (const signature of resolvedType.constructSignatures) {
                        typeElements.push(<ConstructSignatureDeclaration>signatureToSignatureDeclarationHelper(signature, SyntaxKind.ConstructSignature, context));
                    }
                    if (resolvedType.stringIndexInfo) {
                        const indexInfo = resolvedType.objectFlags & ObjectFlags.ReverseMapped ?
                            createIndexInfo(anyType, resolvedType.stringIndexInfo.isReadonly, resolvedType.stringIndexInfo.declaration) :
                            resolvedType.stringIndexInfo;
                        typeElements.push(indexInfoToIndexSignatureDeclarationHelper(indexInfo, IndexKind.String, context));
                    }
                    if (resolvedType.numberIndexInfo) {
                        typeElements.push(indexInfoToIndexSignatureDeclarationHelper(resolvedType.numberIndexInfo, IndexKind.Number, context));
                    }

                    const properties = resolvedType.properties;
                    if (!properties) {
                        return typeElements;
                    }

                    for (const propertySymbol of properties) {
                        if (context.flags & NodeBuilderFlags.WriteClassExpressionAsTypeLiteral) {
                            if (propertySymbol.flags & SymbolFlags.Prototype) {
                                continue;
                            }
                            if (getDeclarationModifierFlagsFromSymbol(propertySymbol) & (ModifierFlags.Private | ModifierFlags.Protected) && context.tracker.reportPrivateInBaseOfClassExpression) {
                                context.tracker.reportPrivateInBaseOfClassExpression(unescapeLeadingUnderscores(propertySymbol.escapedName));
                            }
                        }
                        const propertyType = getCheckFlags(propertySymbol) & CheckFlags.ReverseMapped && context.flags & NodeBuilderFlags.InReverseMappedType ?
                            anyType : getTypeOfSymbol(propertySymbol);
                        const saveEnclosingDeclaration = context.enclosingDeclaration;
                        context.enclosingDeclaration = undefined;
                        if (getCheckFlags(propertySymbol) & CheckFlags.Late) {
                            const decl = firstOrUndefined(propertySymbol.declarations);
                            const name = hasLateBindableName(decl) && resolveEntityName(decl.name.expression, SymbolFlags.Value);
                            if (name && context.tracker.trackSymbol) {
                                context.tracker.trackSymbol(name, saveEnclosingDeclaration, SymbolFlags.Value);
                            }
                        }
                        const propertyName = symbolToName(propertySymbol, context, SymbolFlags.Value, /*expectsIdentifier*/ true);
                        context.enclosingDeclaration = saveEnclosingDeclaration;
                        const optionalToken = propertySymbol.flags & SymbolFlags.Optional ? createToken(SyntaxKind.QuestionToken) : undefined;
                        if (propertySymbol.flags & (SymbolFlags.Function | SymbolFlags.Method) && !getPropertiesOfObjectType(propertyType).length) {
                            const signatures = getSignaturesOfType(propertyType, SignatureKind.Call);
                            for (const signature of signatures) {
                                const methodDeclaration = <MethodSignature>signatureToSignatureDeclarationHelper(signature, SyntaxKind.MethodSignature, context);
                                methodDeclaration.name = propertyName;
                                methodDeclaration.questionToken = optionalToken;
                                typeElements.push(methodDeclaration);
                            }
                        }
                        else {
                            const savedFlags = context.flags;
                            context.flags |= !!(getCheckFlags(propertySymbol) & CheckFlags.ReverseMapped) ? NodeBuilderFlags.InReverseMappedType : 0;
                            const propertyTypeNode = propertyType ? typeToTypeNodeHelper(propertyType, context) : createKeywordTypeNode(SyntaxKind.AnyKeyword);
                            context.flags = savedFlags;

                            const modifiers = isReadonlySymbol(propertySymbol) ? [createToken(SyntaxKind.ReadonlyKeyword)] : undefined;
                            const propertySignature = createPropertySignature(
                                modifiers,
                                propertyName,
                                optionalToken,
                                propertyTypeNode,
                               /*initializer*/ undefined);
                            typeElements.push(propertySignature);
                        }
                    }
                    return typeElements.length ? typeElements : undefined;
                }
            }

            function mapToTypeNodes(types: Type[], context: NodeBuilderContext): TypeNode[] {
                if (some(types)) {
                    const result = [];
                    for (const type of types) {
                        const typeNode = typeToTypeNodeHelper(type, context);
                        if (typeNode) {
                            result.push(typeNode);
                        }
                    }

                    return result;
                }
            }

            function indexInfoToIndexSignatureDeclarationHelper(indexInfo: IndexInfo, kind: IndexKind, context: NodeBuilderContext): IndexSignatureDeclaration {
                const name = getNameFromIndexInfo(indexInfo) || "x";
                const indexerTypeNode = createKeywordTypeNode(kind === IndexKind.String ? SyntaxKind.StringKeyword : SyntaxKind.NumberKeyword);

                const indexingParameter = createParameter(
                    /*decorators*/ undefined,
                    /*modifiers*/ undefined,
                    /*dotDotDotToken*/ undefined,
                    name,
                    /*questionToken*/ undefined,
                    indexerTypeNode,
                    /*initializer*/ undefined);
                const typeNode = indexInfo.type ? typeToTypeNodeHelper(indexInfo.type, context) : typeToTypeNodeHelper(anyType, context);
                if (!indexInfo.type && !(context.flags & NodeBuilderFlags.AllowEmptyIndexInfoType)) {
                    context.encounteredError = true;
                }
                return createIndexSignature(
                    /*decorators*/ undefined,
                    indexInfo.isReadonly ? [createToken(SyntaxKind.ReadonlyKeyword)] : undefined,
                    [indexingParameter],
                    typeNode);
            }

            function signatureToSignatureDeclarationHelper(signature: Signature, kind: SyntaxKind, context: NodeBuilderContext): SignatureDeclaration {
                let typeParameters: TypeParameterDeclaration[];
                let typeArguments: TypeNode[];
                if (context.flags & NodeBuilderFlags.WriteTypeArgumentsOfSignature && signature.target && signature.mapper && signature.target.typeParameters) {
                    typeArguments = signature.target.typeParameters.map(parameter => typeToTypeNodeHelper(instantiateType(parameter, signature.mapper), context));
                }
                else {
                    typeParameters = signature.typeParameters && signature.typeParameters.map(parameter => typeParameterToDeclaration(parameter, context));
                }
                const parameters = signature.parameters.map(parameter => symbolToParameterDeclaration(parameter, context));
                if (signature.thisParameter) {
                    const thisParameter = symbolToParameterDeclaration(signature.thisParameter, context);
                    parameters.unshift(thisParameter);
                }
                let returnTypeNode: TypeNode;
                const typePredicate = getTypePredicateOfSignature(signature);
                if (typePredicate) {
                    const parameterName = typePredicate.kind === TypePredicateKind.Identifier ?
                        setEmitFlags(createIdentifier(typePredicate.parameterName), EmitFlags.NoAsciiEscaping) :
                        createThisTypeNode();
                    const typeNode = typeToTypeNodeHelper(typePredicate.type, context);
                    returnTypeNode = createTypePredicateNode(parameterName, typeNode);
                }
                else {
                    const returnType = getReturnTypeOfSignature(signature);
                    returnTypeNode = returnType && typeToTypeNodeHelper(returnType, context);
                }
                if (context.flags & NodeBuilderFlags.SuppressAnyReturnType) {
                    if (returnTypeNode && returnTypeNode.kind === SyntaxKind.AnyKeyword) {
                        returnTypeNode = undefined;
                    }
                }
                else if (!returnTypeNode) {
                    returnTypeNode = createKeywordTypeNode(SyntaxKind.AnyKeyword);
                }
                return createSignatureDeclaration(kind, typeParameters, parameters, returnTypeNode, typeArguments);
            }

            function typeParameterToDeclaration(type: TypeParameter, context: NodeBuilderContext, constraint = getConstraintFromTypeParameter(type)): TypeParameterDeclaration {
                const savedContextFlags = context.flags;
                context.flags &= ~NodeBuilderFlags.WriteTypeParametersInQualifiedName; // Avoids potential infinite loop when building for a claimspace with a generic
                const name = symbolToName(type.symbol, context, SymbolFlags.Type, /*expectsIdentifier*/ true);
                const constraintNode = constraint && typeToTypeNodeHelper(constraint, context);
                const defaultParameter = getDefaultFromTypeParameter(type);
                const defaultParameterNode = defaultParameter && typeToTypeNodeHelper(defaultParameter, context);
                context.flags = savedContextFlags;
                return createTypeParameterDeclaration(name, constraintNode, defaultParameterNode);
            }

            function symbolToParameterDeclaration(parameterSymbol: Symbol, context: NodeBuilderContext): ParameterDeclaration {
                const parameterDeclaration = getDeclarationOfKind<ParameterDeclaration>(parameterSymbol, SyntaxKind.Parameter);
                Debug.assert(!!parameterDeclaration || isTransientSymbol(parameterSymbol) && !!parameterSymbol.isRestParameter);

                let parameterType = getTypeOfSymbol(parameterSymbol);
                if (parameterDeclaration && isRequiredInitializedParameter(parameterDeclaration)) {
                    parameterType = getOptionalType(parameterType);
                }
                const parameterTypeNode = typeToTypeNodeHelper(parameterType, context);

                const modifiers = !(context.flags & NodeBuilderFlags.OmitParameterModifiers) && parameterDeclaration && parameterDeclaration.modifiers && parameterDeclaration.modifiers.map(getSynthesizedClone);
                const dotDotDotToken = !parameterDeclaration || isRestParameter(parameterDeclaration) ? createToken(SyntaxKind.DotDotDotToken) : undefined;
                const name = parameterDeclaration
                    ? parameterDeclaration.name ?
                        parameterDeclaration.name.kind === SyntaxKind.Identifier ?
                            setEmitFlags(getSynthesizedClone(parameterDeclaration.name), EmitFlags.NoAsciiEscaping) :
                            cloneBindingName(parameterDeclaration.name) :
                        symbolName(parameterSymbol)
                    : symbolName(parameterSymbol);
                const questionToken = parameterDeclaration && isOptionalParameter(parameterDeclaration) ? createToken(SyntaxKind.QuestionToken) : undefined;
                const parameterNode = createParameter(
                    /*decorators*/ undefined,
                    modifiers,
                    dotDotDotToken,
                    name,
                    questionToken,
                    parameterTypeNode,
                    /*initializer*/ undefined);
                return parameterNode;

                function cloneBindingName(node: BindingName): BindingName {
                    return <BindingName>elideInitializerAndSetEmitFlags(node);
                    function elideInitializerAndSetEmitFlags(node: Node): Node {
                        const visited = visitEachChild(node, elideInitializerAndSetEmitFlags, nullTransformationContext, /*nodesVisitor*/ undefined, elideInitializerAndSetEmitFlags);
                        const clone = nodeIsSynthesized(visited) ? visited : getSynthesizedClone(visited);
                        if (clone.kind === SyntaxKind.BindingElement) {
                            (<BindingElement>clone).initializer = undefined;
                        }
                        return setEmitFlags(clone, EmitFlags.SingleLine | EmitFlags.NoAsciiEscaping);
                    }
                }
            }

            function lookupSymbolChain(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags) {
                context.tracker.trackSymbol(symbol, context.enclosingDeclaration, meaning);
                // Try to get qualified name if the symbol is not a type parameter and there is an enclosing declaration.
                let chain: Symbol[];
                const isTypeParameter = symbol.flags & SymbolFlags.TypeParameter;
                if (!isTypeParameter && (context.enclosingDeclaration || context.flags & NodeBuilderFlags.UseFullyQualifiedType)) {
                    chain = getSymbolChain(symbol, meaning, /*endOfChain*/ true);
                    Debug.assert(chain && chain.length > 0);
                }
                else {
                    chain = [symbol];
                }
                return chain;

                /** @param endOfChain Set to false for recursive calls; non-recursive calls should always output something. */
                function getSymbolChain(symbol: Symbol, meaning: SymbolFlags, endOfChain: boolean): Symbol[] | undefined {
                    let accessibleSymbolChain = getAccessibleSymbolChain(symbol, context.enclosingDeclaration, meaning, !!(context.flags & NodeBuilderFlags.UseOnlyExternalAliasing));
                    let parentSymbol: Symbol;

                    if (!accessibleSymbolChain ||
                        needsQualification(accessibleSymbolChain[0], context.enclosingDeclaration, accessibleSymbolChain.length === 1 ? meaning : getQualifiedLeftMeaning(meaning))) {

                        // Go up and add our parent.
                        const parent = getParentOfSymbol(accessibleSymbolChain ? accessibleSymbolChain[0] : symbol);
                        if (parent) {
                            const parentChain = getSymbolChain(parent, getQualifiedLeftMeaning(meaning), /*endOfChain*/ false);
                            if (parentChain) {
                                parentSymbol = parent;
                                accessibleSymbolChain = parentChain.concat(accessibleSymbolChain || [symbol]);
                            }
                        }
                    }

                    if (accessibleSymbolChain) {
                        return accessibleSymbolChain;
                    }
                    if (
                        // If this is the last part of outputting the symbol, always output. The cases apply only to parent symbols.
                        endOfChain ||
                        // If a parent symbol is an external module, don't write it. (We prefer just `x` vs `"foo/bar".x`.)
                        !(!parentSymbol && forEach(symbol.declarations, hasExternalModuleSymbol)) &&
                        // If a parent symbol is an anonymous type, don't write it.
                        !(symbol.flags & (SymbolFlags.TypeLiteral | SymbolFlags.ObjectLiteral))) {

                        return [symbol];
                    }
                }
            }

            function typeParametersToTypeParameterDeclarations(symbol: Symbol, context: NodeBuilderContext) {
                let typeParameterNodes: NodeArray<TypeParameterDeclaration> | undefined;
                const targetSymbol = getTargetSymbol(symbol);
                if (targetSymbol.flags & (SymbolFlags.Class | SymbolFlags.Interface | SymbolFlags.TypeAlias)) {
                    typeParameterNodes = createNodeArray(map(getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol), tp => typeParameterToDeclaration(tp, context)));
                }
                return typeParameterNodes;
            }

            function lookupTypeParameterNodes(chain: Symbol[], index: number, context: NodeBuilderContext) {
                Debug.assert(chain && 0 <= index && index < chain.length);
                const symbol = chain[index];
                let typeParameterNodes: ReadonlyArray<TypeNode> | ReadonlyArray<TypeParameterDeclaration> | undefined;
                if (context.flags & NodeBuilderFlags.WriteTypeParametersInQualifiedName && index < (chain.length - 1)) {
                    const parentSymbol = symbol;
                    const nextSymbol = chain[index + 1];
                    if (getCheckFlags(nextSymbol) & CheckFlags.Instantiated) {
                        const params = getTypeParametersOfClassOrInterface(
                            parentSymbol.flags & SymbolFlags.Alias ? resolveAlias(parentSymbol) : parentSymbol
                        );
                        typeParameterNodes = mapToTypeNodes(map(params, (nextSymbol as TransientSymbol).mapper), context);
                    }
                    else {
                        typeParameterNodes = typeParametersToTypeParameterDeclarations(symbol, context);
                    }
                }
                return typeParameterNodes;
            }

            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: true): Identifier;
            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: false): EntityName;
            function symbolToName(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags, expectsIdentifier: boolean): EntityName {
                const chain = lookupSymbolChain(symbol, context, meaning);

                if (expectsIdentifier && chain.length !== 1
                    && !context.encounteredError
                    && !(context.flags & NodeBuilderFlags.AllowQualifedNameInPlaceOfIdentifier)) {
                    context.encounteredError = true;
                }
                return createEntityNameFromSymbolChain(chain, chain.length - 1);

                function createEntityNameFromSymbolChain(chain: Symbol[], index: number): EntityName {
                    const typeParameterNodes = lookupTypeParameterNodes(chain, index, context);
                    const symbol = chain[index];

                    if (index === 0) {
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                    }
                    const symbolName = getNameOfSymbolAsWritten(symbol, context);
                    if (index === 0) {
                        context.flags ^= NodeBuilderFlags.InInitialEntityName;
                    }
                    const identifier = setEmitFlags(createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                    identifier.symbol = symbol;

                    return index > 0 ? createQualifiedName(createEntityNameFromSymbolChain(chain, index - 1), identifier) : identifier;
                }
            }

            function symbolToExpression(symbol: Symbol, context: NodeBuilderContext, meaning: SymbolFlags) {
                const chain = lookupSymbolChain(symbol, context, meaning);

                return createExpressionFromSymbolChain(chain, chain.length - 1);

                function createExpressionFromSymbolChain(chain: Symbol[], index: number): Expression {
                    const typeParameterNodes = lookupTypeParameterNodes(chain, index, context);
                    const symbol = chain[index];

                    if (index === 0) {
                        context.flags |= NodeBuilderFlags.InInitialEntityName;
                    }
                    let symbolName = getNameOfSymbolAsWritten(symbol, context);
                    if (index === 0) {
                        context.flags ^= NodeBuilderFlags.InInitialEntityName;
                    }
                    let firstChar = symbolName.charCodeAt(0);
                    const canUsePropertyAccess = isIdentifierStart(firstChar, languageVersion);
                    if (index === 0 || canUsePropertyAccess) {
                        const identifier = setEmitFlags(createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                        identifier.symbol = symbol;

                        return index > 0 ? createPropertyAccess(createExpressionFromSymbolChain(chain, index - 1), identifier) : identifier;
                    }
                    else {
                        if (firstChar === CharacterCodes.openBracket) {
                            symbolName = symbolName.substring(1, symbolName.length - 1);
                            firstChar = symbolName.charCodeAt(0);
                        }
                        let expression: Expression;
                        if (isSingleOrDoubleQuote(firstChar)) {
                            expression = createLiteral(symbolName.substring(1, symbolName.length - 1).replace(/\\./g, s => s.substring(1)));
                            (expression as StringLiteral).singleQuote = firstChar === CharacterCodes.singleQuote;
                        }
                        else if (("" + +symbolName) === symbolName) {
                            expression = createLiteral(+symbolName);
                        }
                        if (!expression) {
                            expression = setEmitFlags(createIdentifier(symbolName, typeParameterNodes), EmitFlags.NoAsciiEscaping);
                            expression.symbol = symbol;
                        }
                        return createElementAccess(createExpressionFromSymbolChain(chain, index - 1), expression);
                    }
                }
            }
        }

        function typePredicateToString(typePredicate: TypePredicate, enclosingDeclaration?: Node, flags?: TypeFormatFlags, writer?: EmitTextWriter): string {
            return writer ? typePredicateToStringWorker(writer).getText() : usingSingleLineStringWriter(typePredicateToStringWorker);

            function typePredicateToStringWorker(writer: EmitTextWriter) {
                const predicate = createTypePredicateNode(
                    typePredicate.kind === TypePredicateKind.Identifier ? createIdentifier(typePredicate.parameterName) : createThisTypeNode(),
                    nodeBuilder.typeToTypeNode(typePredicate.type, enclosingDeclaration, toNodeBuilderFlags(flags) | NodeBuilderFlags.IgnoreErrors | NodeBuilderFlags.WriteTypeParametersInQualifiedName)
                );
                const printer = createPrinter({ removeComments: true });
                const sourceFile = enclosingDeclaration && getSourceFileOfNode(enclosingDeclaration);
                printer.writeNode(EmitHint.Unspecified, predicate, /*sourceFile*/ sourceFile, writer);
                return writer;
            }
        }

        function formatUnionTypes(types: Type[]): Type[] {
            const result: Type[] = [];
            let flags: TypeFlags = 0;
            for (let i = 0; i < types.length; i++) {
                const t = types[i];
                flags |= t.flags;
                if (!(t.flags & TypeFlags.Nullable)) {
                    if (t.flags & (TypeFlags.BooleanLiteral | TypeFlags.EnumLiteral)) {
                        const baseType = t.flags & TypeFlags.BooleanLiteral ? booleanType : getBaseTypeOfEnumLiteralType(<LiteralType>t);
                        if (baseType.flags & TypeFlags.Union) {
                            const count = (<UnionType>baseType).types.length;
                            if (i + count <= types.length && types[i + count - 1] === (<UnionType>baseType).types[count - 1]) {
                                result.push(baseType);
                                i += count - 1;
                                continue;
                            }
                        }
                    }
                    result.push(t);
                }
            }
            if (flags & TypeFlags.Null) result.push(nullType);
            if (flags & TypeFlags.Undefined) result.push(undefinedType);
            return result || types;
        }

        function visibilityToString(flags: ModifierFlags): string | undefined {
            if (flags === ModifierFlags.Private) {
                return "private";
            }
            if (flags === ModifierFlags.Protected) {
                return "protected";
            }
            return "public";
        }

        function getTypeAliasForTypeLiteral(type: Type): Symbol {
            if (type.symbol && type.symbol.flags & SymbolFlags.TypeLiteral) {
                const node = findAncestor(type.symbol.declarations[0].parent, n => n.kind !== SyntaxKind.ParenthesizedType);
                if (node.kind === SyntaxKind.TypeAliasDeclaration) {
                    return getSymbolOfNode(node);
                }
            }
            return undefined;
        }

        function isTopLevelInExternalModuleAugmentation(node: Node): boolean {
            return node && node.parent &&
                node.parent.kind === SyntaxKind.ModuleBlock &&
                isExternalModuleAugmentation(node.parent.parent);
        }

        function literalTypeToString(type: LiteralType) {
            return type.flags & TypeFlags.StringLiteral ? '"' + escapeString((<StringLiteralType>type).value) + '"' : "" + (<NumberLiteralType>type).value;
        }

        interface NodeBuilderContext {
            enclosingDeclaration: Node | undefined;
            flags: NodeBuilderFlags | undefined;
            tracker: SymbolTracker | undefined;

            // State
            encounteredError: boolean;
            symbolStack: Symbol[] | undefined;
        }

        function isDefaultBindingContext(location: Node) {
            return location.kind === SyntaxKind.SourceFile || isAmbientModule(location);
        }

        /**
         * Gets a human-readable name for a symbol.
         * Should *not* be used for the right-hand side of a `.` -- use `symbolName(symbol)` for that instead.
         *
         * Unlike `symbolName(symbol)`, this will include quotes if the name is from a string literal.
         * It will also use a representation of a number as written instead of a decimal form, e.g. `0o11` instead of `9`.
         */
        function getNameOfSymbolAsWritten(symbol: Symbol, context?: NodeBuilderContext): string {
            if (context && symbol.escapedName === InternalSymbolName.Default && !(context.flags & NodeBuilderFlags.UseAliasDefinedOutsideCurrentScope) &&
                // If it's not the first part of an entity name, it must print as `default`
                (!(context.flags & NodeBuilderFlags.InInitialEntityName) ||
                // if the symbol is synthesized, it will only be referenced externally it must print as `default`
                !symbol.declarations ||
                // if not in the same binding context (source file, module declaration), it must print as `default`
                (context.enclosingDeclaration && findAncestor(symbol.declarations[0], isDefaultBindingContext) !== findAncestor(context.enclosingDeclaration, isDefaultBindingContext)))) {
                return "default";
            }
            if (symbol.declarations && symbol.declarations.length) {
                const declaration = symbol.declarations[0];
                const name = getNameOfDeclaration(declaration);
                if (name) {
                    return declarationNameToString(name);
                }
                if (declaration.parent && declaration.parent.kind === SyntaxKind.VariableDeclaration) {
                    return declarationNameToString((<VariableDeclaration>declaration.parent).name);
                }
                if (context && !context.encounteredError && !(context.flags & NodeBuilderFlags.AllowAnonymousIdentifier)) {
                    context.encounteredError = true;
                }
                switch (declaration.kind) {
                    case SyntaxKind.ClassExpression:
                        return "(Anonymous class)";
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.ArrowFunction:
                        return "(Anonymous function)";
                }
            }
            if ((symbol as TransientSymbol).nameType && (symbol as TransientSymbol).nameType.flags & TypeFlags.StringLiteral) {
                const stringValue = ((symbol as TransientSymbol).nameType as StringLiteralType).value;
                if (!isIdentifierText(stringValue, compilerOptions.target)) {
                    return `"${escapeString(stringValue, CharacterCodes.doubleQuote)}"`;
                }
            }
            return symbolName(symbol);
        }

        function isDeclarationVisible(node: Declaration): boolean {
            if (node) {
                const links = getNodeLinks(node);
                if (links.isVisible === undefined) {
                    links.isVisible = !!determineIfDeclarationIsVisible();
                }
                return links.isVisible;
            }

            return false;

            function determineIfDeclarationIsVisible() {
                switch (node.kind) {
                    case SyntaxKind.BindingElement:
                        return isDeclarationVisible(<Declaration>node.parent.parent);
                    case SyntaxKind.VariableDeclaration:
                        if (isBindingPattern((node as VariableDeclaration).name) &&
                            !((node as VariableDeclaration).name as BindingPattern).elements.length) {
                            // If the binding pattern is empty, this variable declaration is not visible
                            return false;
                        }
                        // falls through
                    case SyntaxKind.ModuleDeclaration:
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.InterfaceDeclaration:
                    case SyntaxKind.TypeAliasDeclaration:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.EnumDeclaration:
                    case SyntaxKind.ImportEqualsDeclaration:
                        // external module augmentation is always visible
                        if (isExternalModuleAugmentation(node)) {
                            return true;
                        }
                        const parent = getDeclarationContainer(node);
                        // If the node is not exported or it is not ambient module element (except import declaration)
                        if (!(getCombinedModifierFlags(node) & ModifierFlags.Export) &&
                            !(node.kind !== SyntaxKind.ImportEqualsDeclaration && parent.kind !== SyntaxKind.SourceFile && parent.flags & NodeFlags.Ambient)) {
                            return isGlobalSourceFile(parent);
                        }
                        // Exported members/ambient module elements (exception import declaration) are visible if parent is visible
                        return isDeclarationVisible(<Declaration>parent);

                    case SyntaxKind.PropertyDeclaration:
                    case SyntaxKind.PropertySignature:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.MethodSignature:
                        if (hasModifier(node, ModifierFlags.Private | ModifierFlags.Protected)) {
                            // Private/protected properties/methods are not visible
                            return false;
                        }
                        // Public properties/methods are visible if its parents are visible, so:
                        // falls through

                    case SyntaxKind.Constructor:
                    case SyntaxKind.ConstructSignature:
                    case SyntaxKind.CallSignature:
                    case SyntaxKind.IndexSignature:
                    case SyntaxKind.Parameter:
                    case SyntaxKind.ModuleBlock:
                    case SyntaxKind.FunctionType:
                    case SyntaxKind.ConstructorType:
                    case SyntaxKind.TypeLiteral:
                    case SyntaxKind.TypeReference:
                    case SyntaxKind.ArrayType:
                    case SyntaxKind.TupleType:
                    case SyntaxKind.UnionType:
                    case SyntaxKind.IntersectionType:
                    case SyntaxKind.ParenthesizedType:
                        return isDeclarationVisible(<Declaration>node.parent);

                    // Default binding, import specifier and namespace import is visible
                    // only on demand so by default it is not visible
                    case SyntaxKind.ImportClause:
                    case SyntaxKind.NamespaceImport:
                    case SyntaxKind.ImportSpecifier:
                        return false;

                    // Type parameters are always visible
                    case SyntaxKind.TypeParameter:
                    // Source file and namespace export are always visible
                    case SyntaxKind.SourceFile:
                    case SyntaxKind.NamespaceExportDeclaration:
                        return true;

                    // Export assignments do not create name bindings outside the module
                    case SyntaxKind.ExportAssignment:
                        return false;

                    default:
                        return false;
                }
            }
        }

        function collectLinkedAliases(node: Identifier, setVisibility?: boolean): Node[] | undefined {
            let exportSymbol: Symbol;
            if (node.parent && node.parent.kind === SyntaxKind.ExportAssignment) {
                exportSymbol = resolveName(node, node.escapedText, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias, /*nameNotFoundMessage*/ undefined, node, /*isUse*/ false);
            }
            else if (node.parent.kind === SyntaxKind.ExportSpecifier) {
                exportSymbol = getTargetOfExportSpecifier(<ExportSpecifier>node.parent, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace | SymbolFlags.Alias);
            }
            let result: Node[];
            if (exportSymbol) {
                buildVisibleNodeList(exportSymbol.declarations);
            }
            return result;

            function buildVisibleNodeList(declarations: Declaration[]) {
                forEach(declarations, declaration => {
                    const resultNode = getAnyImportSyntax(declaration) || declaration;
                    if (setVisibility) {
                        getNodeLinks(declaration).isVisible = true;
                    }
                    else {
                        result = result || [];
                        pushIfUnique(result, resultNode);
                    }

                    if (isInternalModuleImportEqualsDeclaration(declaration)) {
                        // Add the referenced top container visible
                        const internalModuleReference = <Identifier | QualifiedName>declaration.moduleReference;
                        const firstIdentifier = getFirstIdentifier(internalModuleReference);
                        const importSymbol = resolveName(declaration, firstIdentifier.escapedText, SymbolFlags.Value | SymbolFlags.Type | SymbolFlags.Namespace,
                            undefined, undefined, /*isUse*/ false);
                        if (importSymbol) {
                            buildVisibleNodeList(importSymbol.declarations);
                        }
                    }
                });
            }
        }

        /**
         * Push an entry on the type resolution stack. If an entry with the given target and the given property name
         * is already on the stack, and no entries in between already have a type, then a circularity has occurred.
         * In this case, the result values of the existing entry and all entries pushed after it are changed to false,
         * and the value false is returned. Otherwise, the new entry is just pushed onto the stack, and true is returned.
         * In order to see if the same query has already been done before, the target object and the propertyName both
         * must match the one passed in.
         *
         * @param target The symbol, type, or signature whose type is being queried
         * @param propertyName The property name that should be used to query the target for its type
         */
        function pushTypeResolution(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): boolean {
            const resolutionCycleStartIndex = findResolutionCycleStartIndex(target, propertyName);
            if (resolutionCycleStartIndex >= 0) {
                // A cycle was found
                const { length } = resolutionTargets;
                for (let i = resolutionCycleStartIndex; i < length; i++) {
                    resolutionResults[i] = false;
                }
                return false;
            }
            resolutionTargets.push(target);
            resolutionResults.push(/*items*/ true);
            resolutionPropertyNames.push(propertyName);
            return true;
        }

        function findResolutionCycleStartIndex(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): number {
            for (let i = resolutionTargets.length - 1; i >= 0; i--) {
                if (hasType(resolutionTargets[i], resolutionPropertyNames[i])) {
                    return -1;
                }
                if (resolutionTargets[i] === target && resolutionPropertyNames[i] === propertyName) {
                    return i;
                }
            }

            return -1;
        }

        function hasType(target: TypeSystemEntity, propertyName: TypeSystemPropertyName): Type | boolean {
            if (propertyName === TypeSystemPropertyName.Type) {
                return getSymbolLinks(<Symbol>target).type;
            }
            if (propertyName === TypeSystemPropertyName.DeclaredType) {
                return getSymbolLinks(<Symbol>target).declaredType;
            }
            if (propertyName === TypeSystemPropertyName.ResolvedBaseConstructorType) {
                return (<InterfaceType>target).resolvedBaseConstructorType;
            }
            if (propertyName === TypeSystemPropertyName.ResolvedReturnType) {
                return (<Signature>target).resolvedReturnType;
            }
            if (propertyName === TypeSystemPropertyName.ResolvedBaseConstraint) {
                const bc = (<TypeParameter | UnionOrIntersectionType>target).resolvedBaseConstraint;
                return bc && bc !== circularConstraintType;
            }

            Debug.fail("Unhandled TypeSystemPropertyName " + propertyName);
        }

        // Pop an entry from the type resolution stack and return its associated result value. The result value will
        // be true if no circularities were detected, or false if a circularity was found.
        function popTypeResolution(): boolean {
            resolutionTargets.pop();
            resolutionPropertyNames.pop();
            return resolutionResults.pop();
        }

        function getDeclarationContainer(node: Node): Node {
            node = findAncestor(getRootDeclaration(node), node => {
                switch (node.kind) {
                    case SyntaxKind.VariableDeclaration:
                    case SyntaxKind.VariableDeclarationList:
                    case SyntaxKind.ImportSpecifier:
                    case SyntaxKind.NamedImports:
                    case SyntaxKind.NamespaceImport:
                    case SyntaxKind.ImportClause:
                        return false;
                    default:
                        return true;
                }
            });
            return node && node.parent;
        }

        function getTypeOfPrototypeProperty(prototype: Symbol): Type {
            // TypeScript 1.0 spec (April 2014): 8.4
            // Every class automatically contains a static property member named 'prototype',
            // the type of which is an instantiation of the class type with type Any supplied as a type argument for each type parameter.
            // It is an error to explicitly declare a static property member with the name 'prototype'.
            const classType = <InterfaceType>getDeclaredTypeOfSymbol(getParentOfSymbol(prototype));
            return classType.typeParameters ? createTypeReference(<GenericType>classType, map(classType.typeParameters, _ => anyType)) : classType;
        }

        // Return the type of the given property in the given type, or undefined if no such property exists
        function getTypeOfPropertyOfType(type: Type, name: __String): Type {
            const prop = getPropertyOfType(type, name);
            return prop ? getTypeOfSymbol(prop) : undefined;
        }

        function isTypeAny(type: Type) {
            return type && (type.flags & TypeFlags.Any) !== 0;
        }

        // Return the type of a binding element parent. We check SymbolLinks first to see if a type has been
        // assigned by contextual typing.
        function getTypeForBindingElementParent(node: BindingElementGrandparent) {
            const symbol = getSymbolOfNode(node);
            return symbol && getSymbolLinks(symbol).type || getTypeForVariableLikeDeclaration(node, /*includeOptionality*/ false);
        }

        function isComputedNonLiteralName(name: PropertyName): boolean {
            return name.kind === SyntaxKind.ComputedPropertyName && !isStringOrNumericLiteral(name.expression);
        }

        function getRestType(source: Type, properties: PropertyName[], symbol: Symbol): Type {
            source = filterType(source, t => !(t.flags & TypeFlags.Nullable));
            if (source.flags & TypeFlags.Never) {
                return emptyObjectType;
            }

            if (source.flags & TypeFlags.Union) {
                return mapType(source, t => getRestType(t, properties, symbol));
            }

            const members = createSymbolTable();
            const names = createUnderscoreEscapedMap<true>();
            for (const name of properties) {
                names.set(getTextOfPropertyName(name), true);
            }
            for (const prop of getPropertiesOfType(source)) {
                const inNamesToRemove = names.has(prop.escapedName);
                const isPrivate = getDeclarationModifierFlagsFromSymbol(prop) & (ModifierFlags.Private | ModifierFlags.Protected);
                const isSetOnlyAccessor = prop.flags & SymbolFlags.SetAccessor && !(prop.flags & SymbolFlags.GetAccessor);
                if (!inNamesToRemove && !isPrivate && !isClassMethod(prop) && !isSetOnlyAccessor) {
                    members.set(prop.escapedName, prop);
                }
            }
            const stringIndexInfo = getIndexInfoOfType(source, IndexKind.String);
            const numberIndexInfo = getIndexInfoOfType(source, IndexKind.Number);
            return createAnonymousType(symbol, members, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);
        }

        /** Return the inferred type for a binding element */
        function getTypeForBindingElement(declaration: BindingElement): Type {
            const pattern = declaration.parent;
            let parentType = getTypeForBindingElementParent(pattern.parent);
            // If parent has the unknown (error) type, then so does this binding element
            if (parentType === unknownType) {
                return unknownType;
            }
            // If no type was specified or inferred for parent,
            // infer from the initializer of the binding element if one is present.
            // Otherwise, go with the undefined type of the parent.
            if (!parentType) {
                return declaration.initializer ? checkDeclarationInitializer(declaration) : parentType;
            }
            if (isTypeAny(parentType)) {
                return parentType;
            }

            let type: Type;
            if (pattern.kind === SyntaxKind.ObjectBindingPattern) {
                if (declaration.dotDotDotToken) {
                    if (!isValidSpreadType(parentType)) {
                        error(declaration, Diagnostics.Rest_types_may_only_be_created_from_object_types);
                        return unknownType;
                    }
                    const literalMembers: PropertyName[] = [];
                    for (const element of pattern.elements) {
                        if (!element.dotDotDotToken) {
                            literalMembers.push(element.propertyName || element.name as Identifier);
                        }
                    }
                    type = getRestType(parentType, literalMembers, declaration.symbol);
                }
                else {
                    // Use explicitly specified property name ({ p: xxx } form), or otherwise the implied name ({ p } form)
                    const name = declaration.propertyName || <Identifier>declaration.name;
                    if (isComputedNonLiteralName(name)) {
                        // computed properties with non-literal names are treated as 'any'
                        return anyType;
                    }

                    // Use type of the specified property, or otherwise, for a numeric name, the type of the numeric index signature,
                    // or otherwise the type of the string index signature.
                    const text = getTextOfPropertyName(name);

                    // Relax null check on ambient destructuring parameters, since the parameters have no implementation and are just documentation
                    if (strictNullChecks && declaration.flags & NodeFlags.Ambient && isParameterDeclaration(declaration)) {
                        parentType = getNonNullableType(parentType);
                    }
                    const propType = getTypeOfPropertyOfType(parentType, text);
                    const declaredType = propType && getConstraintForLocation(propType, declaration.name);
                    type = declaredType && getFlowTypeOfReference(declaration, declaredType) ||
                        isNumericLiteralName(text) && getIndexTypeOfType(parentType, IndexKind.Number) ||
                        getIndexTypeOfType(parentType, IndexKind.String);
                    if (!type) {
                        error(name, Diagnostics.Type_0_has_no_property_1_and_no_string_index_signature, typeToString(parentType), declarationNameToString(name));
                        return unknownType;
                    }
                }
            }
            else {
                // This elementType will be used if the specific property corresponding to this index is not
                // present (aka the tuple element property). This call also checks that the parentType is in
                // fact an iterable or array (depending on target language).
                const elementType = checkIteratedTypeOrElementType(parentType, pattern, /*allowStringInput*/ false, /*allowAsyncIterables*/ false);
                if (declaration.dotDotDotToken) {
                    // Rest element has an array type with the same element type as the parent type
                    type = createArrayType(elementType);
                }
                else {
                    // Use specific property type when parent is a tuple or numeric index type when parent is an array
                    const propName = "" + pattern.elements.indexOf(declaration);
                    type = isTupleLikeType(parentType)
                        ? getTypeOfPropertyOfType(parentType, propName as __String)
                        : elementType;
                    if (!type) {
                        if (isTupleType(parentType)) {
                            error(declaration, Diagnostics.Tuple_type_0_with_length_1_cannot_be_assigned_to_tuple_with_length_2, typeToString(parentType), getTypeReferenceArity(<TypeReference>parentType), pattern.elements.length);
                        }
                        else {
                            error(declaration, Diagnostics.Type_0_has_no_property_1, typeToString(parentType), propName);
                        }
                        return unknownType;
                    }
                }
            }
            // In strict null checking mode, if a default value of a non-undefined type is specified, remove
            // undefined from the final type.
            if (strictNullChecks && declaration.initializer && !(getFalsyFlags(checkExpressionCached(declaration.initializer)) & TypeFlags.Undefined)) {
                type = getTypeWithFacts(type, TypeFacts.NEUndefined);
            }
            return declaration.initializer ?
                getUnionType([type, checkExpressionCached(declaration.initializer)], UnionReduction.Subtype) :
                type;
        }

        function getTypeForDeclarationFromJSDocComment(declaration: Node) {
            const jsdocType = getJSDocType(declaration);
            if (jsdocType) {
                return getTypeFromTypeNode(jsdocType);
            }
            return undefined;
        }

        function isNullOrUndefined(node: Expression) {
            const expr = skipParentheses(node);
            return expr.kind === SyntaxKind.NullKeyword || expr.kind === SyntaxKind.Identifier && getResolvedSymbol(<Identifier>expr) === undefinedSymbol;
        }

        function isEmptyArrayLiteral(node: Expression) {
            const expr = skipParentheses(node);
            return expr.kind === SyntaxKind.ArrayLiteralExpression && (<ArrayLiteralExpression>expr).elements.length === 0;
        }

        function addOptionality(type: Type, optional = true): Type {
            return strictNullChecks && optional ? getOptionalType(type) : type;
        }

        // Return the inferred type for a variable, parameter, or property declaration
        function getTypeForVariableLikeDeclaration(declaration: ParameterDeclaration | PropertyDeclaration | PropertySignature | VariableDeclaration | BindingElement, includeOptionality: boolean): Type {
            // A variable declared in a for..in statement is of type string, or of type keyof T when the
            // right hand expression is of a type parameter type.
            if (isVariableDeclaration(declaration) && declaration.parent.parent.kind === SyntaxKind.ForInStatement) {
                const indexType = getIndexType(checkNonNullExpression(declaration.parent.parent.expression));
                return indexType.flags & (TypeFlags.TypeParameter | TypeFlags.Index) ? indexType : stringType;
            }

            if (isVariableDeclaration(declaration) && declaration.parent.parent.kind === SyntaxKind.ForOfStatement) {
                // checkRightHandSideOfForOf will return undefined if the for-of expression type was
                // missing properties/signatures required to get its iteratedType (like
                // [Symbol.iterator] or next). This may be because we accessed properties from anyType,
                // or it may have led to an error inside getElementTypeOfIterable.
                const forOfStatement = declaration.parent.parent;
                return checkRightHandSideOfForOf(forOfStatement.expression, forOfStatement.awaitModifier) || anyType;
            }

            if (isBindingPattern(declaration.parent)) {
                return getTypeForBindingElement(<BindingElement>declaration);
            }

            const isOptional = !isBindingElement(declaration) && !isVariableDeclaration(declaration) && !!declaration.questionToken && includeOptionality;
            // Use type from type annotation if one is present
            const declaredType = tryGetTypeFromEffectiveTypeNode(declaration);
            if (declaredType) {
                return addOptionality(declaredType, isOptional);
            }

            if ((noImplicitAny || isInJavaScriptFile(declaration)) &&
                declaration.kind === SyntaxKind.VariableDeclaration && !isBindingPattern(declaration.name) &&
                !(getCombinedModifierFlags(declaration) & ModifierFlags.Export) && !(declaration.flags & NodeFlags.Ambient)) {
                // If --noImplicitAny is on or the declaration is in a Javascript file,
                // use control flow tracked 'any' type for non-ambient, non-exported var or let variables with no
                // initializer or a 'null' or 'undefined' initializer.
                if (!(getCombinedNodeFlags(declaration) & NodeFlags.Const) && (!declaration.initializer || isNullOrUndefined(declaration.initializer))) {
                    return autoType;
                }
                // Use control flow tracked 'any[]' type for non-ambient, non-exported variables with an empty array
                // literal initializer.
                if (declaration.initializer && isEmptyArrayLiteral(declaration.initializer)) {
                    return autoArrayType;
                }
            }

            if (declaration.kind === SyntaxKind.Parameter) {
                const func = <FunctionLikeDeclaration>declaration.parent;
                // For a parameter of a set accessor, use the type of the get accessor if one is present
                if (func.kind === SyntaxKind.SetAccessor && !hasNonBindableDynamicName(func)) {
                    const getter = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration.parent), SyntaxKind.GetAccessor);
                    if (getter) {
                        const getterSignature = getSignatureFromDeclaration(getter);
                        const thisParameter = getAccessorThisParameter(func as AccessorDeclaration);
                        if (thisParameter && declaration === thisParameter) {
                            // Use the type from the *getter*
                            Debug.assert(!thisParameter.type);
                            return getTypeOfSymbol(getterSignature.thisParameter);
                        }
                        return getReturnTypeOfSignature(getterSignature);
                    }
                }
                // Use contextual parameter type if one is available
                let type: Type;
                if (declaration.symbol.escapedName === "this") {
                    type = getContextualThisParameterType(func);
                }
                else {
                    type = getContextuallyTypedParameterType(declaration);
                }
                if (type) {
                    return addOptionality(type, isOptional);
                }
            }

            // Use the type of the initializer expression if one is present
            if (declaration.initializer) {
                const type = checkDeclarationInitializer(declaration);
                return addOptionality(type, isOptional);
            }

            if (isJsxAttribute(declaration)) {
                // if JSX attribute doesn't have initializer, by default the attribute will have boolean value of true.
                // I.e <Elem attr /> is sugar for <Elem attr={true} />
                return trueType;
            }

            // If the declaration specifies a binding pattern, use the type implied by the binding pattern
            if (isBindingPattern(declaration.name)) {
                return getTypeFromBindingPattern(declaration.name, /*includePatternInType*/ false, /*reportErrors*/ true);
            }

            // No type specified and nothing can be inferred
            return undefined;
        }

        function getWidenedTypeFromJSSpecialPropertyDeclarations(symbol: Symbol) {
            // function/class/{} assignments are fresh declarations, not property assignments, so only add prototype assignments
            const specialDeclaration = getAssignedJavascriptInitializer(symbol.valueDeclaration);
            if (specialDeclaration) {
                return getWidenedLiteralType(checkExpressionCached(specialDeclaration));
            }
            const types: Type[] = [];
            let definedInConstructor = false;
            let definedInMethod = false;
            let jsDocType: Type;
            for (const declaration of symbol.declarations) {
                const expression = declaration.kind === SyntaxKind.BinaryExpression ? <BinaryExpression>declaration :
                    declaration.kind === SyntaxKind.PropertyAccessExpression ? <BinaryExpression>getAncestor(declaration, SyntaxKind.BinaryExpression) :
                        undefined;

                if (!expression) {
                    return unknownType;
                }

                if (isPropertyAccessExpression(expression.left) && expression.left.expression.kind === SyntaxKind.ThisKeyword) {
                    if (getThisContainer(expression, /*includeArrowFunctions*/ false).kind === SyntaxKind.Constructor) {
                        definedInConstructor = true;
                    }
                    else {
                        definedInMethod = true;
                    }
                }

                // If there is a JSDoc type, use it
                const type = getTypeForDeclarationFromJSDocComment(expression.parent);
                if (type) {
                    const declarationType = getWidenedType(type);
                    if (!jsDocType) {
                        jsDocType = declarationType;
                    }
                    else if (jsDocType !== unknownType && declarationType !== unknownType &&
                             !isTypeIdenticalTo(jsDocType, declarationType) &&
                             !(symbol.flags & SymbolFlags.JSContainer)) {
                        errorNextVariableOrPropertyDeclarationMustHaveSameType(jsDocType, declaration, declarationType);
                    }
                }
                else if (!jsDocType) {
                    // If we don't have an explicit JSDoc type, get the type from the expression.
                    types.push(getWidenedLiteralType(checkExpressionCached(expression.right)));
                }
            }

            const type = jsDocType || getUnionType(types, UnionReduction.Subtype);
            return getWidenedType(addOptionality(type, definedInMethod && !definedInConstructor));
        }

        // Return the type implied by a binding pattern element. This is the type of the initializer of the element if
        // one is present. Otherwise, if the element is itself a binding pattern, it is the type implied by the binding
        // pattern. Otherwise, it is the type any.
        function getTypeFromBindingElement(element: BindingElement, includePatternInType?: boolean, reportErrors?: boolean): Type {
            if (element.initializer) {
                return checkDeclarationInitializer(element);
            }
            if (isBindingPattern(element.name)) {
                return getTypeFromBindingPattern(element.name, includePatternInType, reportErrors);
            }
            if (reportErrors && noImplicitAny && !declarationBelongsToPrivateAmbientMember(element)) {
                reportImplicitAnyError(element, anyType);
            }
            return anyType;
        }

        // Return the type implied by an object binding pattern
        function getTypeFromObjectBindingPattern(pattern: ObjectBindingPattern, includePatternInType: boolean, reportErrors: boolean): Type {
            const members = createSymbolTable();
            let stringIndexInfo: IndexInfo;
            let hasComputedProperties = false;
            forEach(pattern.elements, e => {
                const name = e.propertyName || <Identifier>e.name;
                if (isComputedNonLiteralName(name)) {
                    // do not include computed properties in the implied type
                    hasComputedProperties = true;
                    return;
                }
                if (e.dotDotDotToken) {
                    stringIndexInfo = createIndexInfo(anyType, /*isReadonly*/ false);
                    return;
                }

                const text = getTextOfPropertyName(name);
                const flags = SymbolFlags.Property | (e.initializer ? SymbolFlags.Optional : 0);
                const symbol = createSymbol(flags, text);
                symbol.type = getTypeFromBindingElement(e, includePatternInType, reportErrors);
                symbol.bindingElement = e;
                members.set(symbol.escapedName, symbol);
            });
            const result = createAnonymousType(undefined, members, emptyArray, emptyArray, stringIndexInfo, undefined);
            if (includePatternInType) {
                result.pattern = pattern;
            }
            if (hasComputedProperties) {
                result.objectFlags |= ObjectFlags.ObjectLiteralPatternWithComputedProperties;
            }
            return result;
        }

        // Return the type implied by an array binding pattern
        function getTypeFromArrayBindingPattern(pattern: BindingPattern, includePatternInType: boolean, reportErrors: boolean): Type {
            const elements = pattern.elements;
            const lastElement = lastOrUndefined(elements);
            if (elements.length === 0 || (!isOmittedExpression(lastElement) && lastElement.dotDotDotToken)) {
                return languageVersion >= ScriptTarget.ES2015 ? createIterableType(anyType) : anyArrayType;
            }
            // If the pattern has at least one element, and no rest element, then it should imply a tuple type.
            const elementTypes = map(elements, e => isOmittedExpression(e) ? anyType : getTypeFromBindingElement(e, includePatternInType, reportErrors));
            let result = createTupleType(elementTypes);
            if (includePatternInType) {
                result = cloneTypeReference(result);
                result.pattern = pattern;
            }
            return result;
        }

        // Return the type implied by a binding pattern. This is the type implied purely by the binding pattern itself
        // and without regard to its context (i.e. without regard any type annotation or initializer associated with the
        // declaration in which the binding pattern is contained). For example, the implied type of [x, y] is [any, any]
        // and the implied type of { x, y: z = 1 } is { x: any; y: number; }. The type implied by a binding pattern is
        // used as the contextual type of an initializer associated with the binding pattern. Also, for a destructuring
        // parameter with no type annotation or initializer, the type implied by the binding pattern becomes the type of
        // the parameter.
        function getTypeFromBindingPattern(pattern: BindingPattern, includePatternInType?: boolean, reportErrors?: boolean): Type {
            return pattern.kind === SyntaxKind.ObjectBindingPattern
                ? getTypeFromObjectBindingPattern(pattern, includePatternInType, reportErrors)
                : getTypeFromArrayBindingPattern(pattern, includePatternInType, reportErrors);
        }

        // Return the type associated with a variable, parameter, or property declaration. In the simple case this is the type
        // specified in a type annotation or inferred from an initializer. However, in the case of a destructuring declaration it
        // is a bit more involved. For example:
        //
        //   var [x, s = ""] = [1, "one"];
        //
        // Here, the array literal [1, "one"] is contextually typed by the type [any, string], which is the implied type of the
        // binding pattern [x, s = ""]. Because the contextual type is a tuple type, the resulting type of [1, "one"] is the
        // tuple type [number, string]. Thus, the type inferred for 'x' is number and the type inferred for 's' is string.
        function getWidenedTypeForVariableLikeDeclaration(declaration: ParameterDeclaration | PropertyDeclaration | PropertySignature | VariableDeclaration | BindingElement, reportErrors?: boolean): Type {
            let type = getTypeForVariableLikeDeclaration(declaration, /*includeOptionality*/ true);
            if (type) {
                if (reportErrors) {
                    reportErrorsFromWidening(declaration, type);
                }

                // always widen a 'unique symbol' type if the type was created for a different declaration.
                if (type.flags & TypeFlags.UniqueESSymbol && (isBindingElement(declaration) || !declaration.type) && type.symbol !== getSymbolOfNode(declaration)) {
                    type = esSymbolType;
                }

                return getWidenedType(type);
            }

            // Rest parameters default to type any[], other parameters default to type any
            type = isParameter(declaration) && declaration.dotDotDotToken ? anyArrayType : anyType;

            // Report implicit any errors unless this is a private property within an ambient declaration
            if (reportErrors && noImplicitAny) {
                if (!declarationBelongsToPrivateAmbientMember(declaration)) {
                    reportImplicitAnyError(declaration, type);
                }
            }
            return type;
        }

        function declarationBelongsToPrivateAmbientMember(declaration: VariableLikeDeclaration) {
            const root = getRootDeclaration(declaration);
            const memberDeclaration = root.kind === SyntaxKind.Parameter ? root.parent : root;
            return isPrivateWithinAmbient(memberDeclaration);
        }

        function tryGetTypeFromEffectiveTypeNode(declaration: Declaration) {
            const typeNode = getEffectiveTypeAnnotationNode(declaration);
            if (typeNode) {
                return getTypeFromTypeNode(typeNode);
            }
        }

        function getTypeOfVariableOrParameterOrProperty(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                // Handle prototype property
                if (symbol.flags & SymbolFlags.Prototype) {
                    return links.type = getTypeOfPrototypeProperty(symbol);
                }
                // Handle catch clause variables
                const declaration = symbol.valueDeclaration;
                if (isCatchClauseVariableDeclarationOrBindingElement(declaration)) {
                    return links.type = anyType;
                }
                // Handle export default expressions
                if (declaration.kind === SyntaxKind.ExportAssignment) {
                    return links.type = checkExpression((<ExportAssignment>declaration).expression);
                }
                if (isInJavaScriptFile(declaration) && isJSDocPropertyLikeTag(declaration) && declaration.typeExpression) {
                    return links.type = getTypeFromTypeNode(declaration.typeExpression.type);
                }
                // Handle variable, parameter or property
                if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                    return unknownType;
                }

                let type: Type;
                // Handle certain special assignment kinds, which happen to union across multiple declarations:
                // * module.exports = expr
                // * exports.p = expr
                // * this.p = expr
                // * className.prototype.method = expr
                if (declaration.kind === SyntaxKind.BinaryExpression ||
                    declaration.kind === SyntaxKind.PropertyAccessExpression && declaration.parent.kind === SyntaxKind.BinaryExpression) {
                    type = getWidenedTypeFromJSSpecialPropertyDeclarations(symbol);
                }
                else if (isJSDocPropertyTag(declaration)
                    || isPropertyAccessExpression(declaration)
                    || isIdentifier(declaration)
                    || (isMethodDeclaration(declaration) && !isObjectLiteralMethod(declaration))
                    || isMethodSignature(declaration)) {

                    // Symbol is property of some kind that is merged with something - should use `getTypeOfFuncClassEnumModule` and not `getTypeOfVariableOrParameterOrProperty`
                    if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.Class | SymbolFlags.Enum | SymbolFlags.ValueModule)) {
                        return getTypeOfFuncClassEnumModule(symbol);
                    }
                    type = tryGetTypeFromEffectiveTypeNode(declaration) || anyType;
                }
                else if (isPropertyAssignment(declaration)) {
                    type = tryGetTypeFromEffectiveTypeNode(declaration) || checkPropertyAssignment(declaration);
                }
                else if (isJsxAttribute(declaration)) {
                    type = tryGetTypeFromEffectiveTypeNode(declaration) || checkJsxAttribute(declaration);
                }
                else if (isShorthandPropertyAssignment(declaration)) {
                    type = tryGetTypeFromEffectiveTypeNode(declaration) || checkExpressionForMutableLocation(declaration.name, CheckMode.Normal);
                }
                else if (isObjectLiteralMethod(declaration)) {
                    type = tryGetTypeFromEffectiveTypeNode(declaration) || checkObjectLiteralMethod(declaration, CheckMode.Normal);
                }
                else if (isParameter(declaration)
                    || isPropertyDeclaration(declaration)
                    || isPropertySignature(declaration)
                    || isVariableDeclaration(declaration)
                    || isBindingElement(declaration)) {
                    type = getWidenedTypeForVariableLikeDeclaration(declaration, /*reportErrors*/ true);
                }
                else {
                    Debug.fail("Unhandled declaration kind! " + Debug.showSyntaxKind(declaration));
                }

                if (!popTypeResolution()) {
                    type = reportCircularityError(symbol);
                }
                links.type = type;
            }
            return links.type;
        }

        function getAnnotatedAccessorType(accessor: AccessorDeclaration): Type {
            if (accessor) {
                if (accessor.kind === SyntaxKind.GetAccessor) {
                    const getterTypeAnnotation = getEffectiveReturnTypeNode(accessor);
                    return getterTypeAnnotation && getTypeFromTypeNode(getterTypeAnnotation);
                }
                else {
                    const setterTypeAnnotation = getEffectiveSetAccessorTypeAnnotationNode(accessor);
                    return setterTypeAnnotation && getTypeFromTypeNode(setterTypeAnnotation);
                }
            }
            return undefined;
        }

        function getAnnotatedAccessorThisParameter(accessor: AccessorDeclaration): Symbol | undefined {
            const parameter = getAccessorThisParameter(accessor);
            return parameter && parameter.symbol;
        }

        function getThisTypeOfDeclaration(declaration: SignatureDeclaration): Type | undefined {
            return getThisTypeOfSignature(getSignatureFromDeclaration(declaration));
        }

        function getTypeOfAccessors(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                const getter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.GetAccessor);
                const setter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.SetAccessor);

                if (getter && isInJavaScriptFile(getter)) {
                    const jsDocType = getTypeForDeclarationFromJSDocComment(getter);
                    if (jsDocType) {
                        return links.type = jsDocType;
                    }
                }

                if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                    return unknownType;
                }

                let type: Type;

                // First try to see if the user specified a return type on the get-accessor.
                const getterReturnType = getAnnotatedAccessorType(getter);
                if (getterReturnType) {
                    type = getterReturnType;
                }
                else {
                    // If the user didn't specify a return type, try to use the set-accessor's parameter type.
                    const setterParameterType = getAnnotatedAccessorType(setter);
                    if (setterParameterType) {
                        type = setterParameterType;
                    }
                    else {
                        // If there are no specified types, try to infer it from the body of the get accessor if it exists.
                        if (getter && getter.body) {
                            type = getReturnTypeFromBody(getter);
                        }
                        // Otherwise, fall back to 'any'.
                        else {
                            if (noImplicitAny) {
                                if (setter) {
                                    error(setter, Diagnostics.Property_0_implicitly_has_type_any_because_its_set_accessor_lacks_a_parameter_type_annotation, symbolToString(symbol));
                                }
                                else {
                                    Debug.assert(!!getter, "there must existed getter as we are current checking either setter or getter in this function");
                                    error(getter, Diagnostics.Property_0_implicitly_has_type_any_because_its_get_accessor_lacks_a_return_type_annotation, symbolToString(symbol));
                                }
                            }
                            type = anyType;
                        }
                    }
                }
                if (!popTypeResolution()) {
                    type = anyType;
                    if (noImplicitAny) {
                        const getter = getDeclarationOfKind<AccessorDeclaration>(symbol, SyntaxKind.GetAccessor);
                        error(getter, Diagnostics._0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions, symbolToString(symbol));
                    }
                }
                links.type = type;
            }
            return links.type;
        }

        function getBaseTypeVariableOfClass(symbol: Symbol) {
            const baseConstructorType = getBaseConstructorTypeOfClass(getDeclaredTypeOfClassOrInterface(symbol));
            return baseConstructorType.flags & TypeFlags.TypeVariable ? baseConstructorType : undefined;
        }

        function getTypeOfFuncClassEnumModule(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                if (symbol.flags & SymbolFlags.Module && isShorthandAmbientModuleSymbol(symbol)) {
                    links.type = anyType;
                }
                else {
                    const type = createObjectType(ObjectFlags.Anonymous, symbol);
                    if (symbol.flags & SymbolFlags.Class) {
                        const baseTypeVariable = getBaseTypeVariableOfClass(symbol);
                        links.type = baseTypeVariable ? getIntersectionType([type, baseTypeVariable]) : type;
                    }
                    else {
                        links.type = strictNullChecks && symbol.flags & SymbolFlags.Optional ? getOptionalType(type) : type;
                    }
                }
            }
            return links.type;
        }

        function getTypeOfEnumMember(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                links.type = getDeclaredTypeOfEnumMember(symbol);
            }
            return links.type;
        }

        function getTypeOfAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                const targetSymbol = resolveAlias(symbol);

                // It only makes sense to get the type of a value symbol. If the result of resolving
                // the alias is not a value, then it has no type. To get the type associated with a
                // type symbol, call getDeclaredTypeOfSymbol.
                // This check is important because without it, a call to getTypeOfSymbol could end
                // up recursively calling getTypeOfAlias, causing a stack overflow.
                links.type = targetSymbol.flags & SymbolFlags.Value
                    ? getTypeOfSymbol(targetSymbol)
                    : unknownType;
            }
            return links.type;
        }

        function getTypeOfInstantiatedSymbol(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.type) {
                if (symbolInstantiationDepth === 100) {
                    error(symbol.valueDeclaration, Diagnostics.Generic_type_instantiation_is_excessively_deep_and_possibly_infinite);
                    links.type = unknownType;
                }
                else {
                    if (!pushTypeResolution(symbol, TypeSystemPropertyName.Type)) {
                        return unknownType;
                    }
                    symbolInstantiationDepth++;
                    let type = instantiateType(getTypeOfSymbol(links.target), links.mapper);
                    symbolInstantiationDepth--;
                    if (!popTypeResolution()) {
                        type = reportCircularityError(symbol);
                    }
                    links.type = type;
                }
            }
            return links.type;
        }

        function reportCircularityError(symbol: Symbol) {
            // Check if variable has type annotation that circularly references the variable itself
            if (getEffectiveTypeAnnotationNode(<VariableLikeDeclaration>symbol.valueDeclaration)) {
                error(symbol.valueDeclaration, Diagnostics._0_is_referenced_directly_or_indirectly_in_its_own_type_annotation,
                    symbolToString(symbol));
                return unknownType;
            }
            // Otherwise variable has initializer that circularly references the variable itself
            if (noImplicitAny) {
                error(symbol.valueDeclaration, Diagnostics._0_implicitly_has_type_any_because_it_does_not_have_a_type_annotation_and_is_referenced_directly_or_indirectly_in_its_own_initializer,
                    symbolToString(symbol));
            }
            return anyType;
        }

        function getTypeOfSymbol(symbol: Symbol): Type {
            if (getCheckFlags(symbol) & CheckFlags.Instantiated) {
                return getTypeOfInstantiatedSymbol(symbol);
            }
            if (getCheckFlags(symbol) & CheckFlags.ReverseMapped) {
                return getTypeOfReverseMappedSymbol(symbol as ReverseMappedSymbol);
            }
            if (symbol.flags & (SymbolFlags.Variable | SymbolFlags.Property)) {
                return getTypeOfVariableOrParameterOrProperty(symbol);
            }
            if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.Class | SymbolFlags.Enum | SymbolFlags.ValueModule)) {
                return getTypeOfFuncClassEnumModule(symbol);
            }
            if (symbol.flags & SymbolFlags.EnumMember) {
                return getTypeOfEnumMember(symbol);
            }
            if (symbol.flags & SymbolFlags.Accessor) {
                return getTypeOfAccessors(symbol);
            }
            if (symbol.flags & SymbolFlags.Alias) {
                return getTypeOfAlias(symbol);
            }
            return unknownType;
        }

        function isReferenceToType(type: Type, target: Type) {
            return type !== undefined
                && target !== undefined
                && (getObjectFlags(type) & ObjectFlags.Reference) !== 0
                && (<TypeReference>type).target === target;
        }

        function getTargetType(type: Type): Type {
            return getObjectFlags(type) & ObjectFlags.Reference ? (<TypeReference>type).target : type;
        }

        function hasBaseType(type: Type, checkBase: Type) {
            return check(type);
            function check(type: Type): boolean {
                if (getObjectFlags(type) & (ObjectFlags.ClassOrInterface | ObjectFlags.Reference)) {
                    const target = <InterfaceType>getTargetType(type);
                    return target === checkBase || forEach(getBaseTypes(target), check);
                }
                else if (type.flags & TypeFlags.Intersection) {
                    return forEach((<IntersectionType>type).types, check);
                }
            }
        }

        // Appends the type parameters given by a list of declarations to a set of type parameters and returns the resulting set.
        // The function allocates a new array if the input type parameter set is undefined, but otherwise it modifies the set
        // in-place and returns the same array.
        function appendTypeParameters(typeParameters: TypeParameter[], declarations: ReadonlyArray<TypeParameterDeclaration>): TypeParameter[] {
            for (const declaration of declarations) {
                const tp = getDeclaredTypeOfTypeParameter(getSymbolOfNode(declaration));
                typeParameters = appendIfUnique(typeParameters, tp);
            }
            return typeParameters;
        }

        // Return the outer type parameters of a node or undefined if the node has no outer type parameters.
        function getOuterTypeParameters(node: Node, includeThisTypes?: boolean): TypeParameter[] {
            while (true) {
                node = node.parent;
                if (!node) {
                    return undefined;
                }
                switch (node.kind) {
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                    case SyntaxKind.InterfaceDeclaration:
                    case SyntaxKind.CallSignature:
                    case SyntaxKind.ConstructSignature:
                    case SyntaxKind.MethodSignature:
                    case SyntaxKind.FunctionType:
                    case SyntaxKind.ConstructorType:
                    case SyntaxKind.JSDocFunctionType:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.ArrowFunction:
                    case SyntaxKind.TypeAliasDeclaration:
                    case SyntaxKind.JSDocTemplateTag:
                    case SyntaxKind.MappedType:
                    case SyntaxKind.ConditionalType:
                        const outerTypeParameters = getOuterTypeParameters(node, includeThisTypes);
                        if (node.kind === SyntaxKind.MappedType) {
                            return append(outerTypeParameters, getDeclaredTypeOfTypeParameter(getSymbolOfNode((<MappedTypeNode>node).typeParameter)));
                        }
                        else if (node.kind === SyntaxKind.ConditionalType) {
                            return concatenate(outerTypeParameters, getInferTypeParameters(<ConditionalTypeNode>node));
                        }
                        const outerAndOwnTypeParameters = appendTypeParameters(outerTypeParameters, getEffectiveTypeParameterDeclarations(<DeclarationWithTypeParameters>node) || emptyArray);
                        const thisType = includeThisTypes &&
                            (node.kind === SyntaxKind.ClassDeclaration || node.kind === SyntaxKind.ClassExpression || node.kind === SyntaxKind.InterfaceDeclaration) &&
                            getDeclaredTypeOfClassOrInterface(getSymbolOfNode(node)).thisType;
                        return thisType ? append(outerAndOwnTypeParameters, thisType) : outerAndOwnTypeParameters;
                }
            }
        }

        // The outer type parameters are those defined by enclosing generic classes, methods, or functions.
        function getOuterTypeParametersOfClassOrInterface(symbol: Symbol): TypeParameter[] {
            const declaration = symbol.flags & SymbolFlags.Class ? symbol.valueDeclaration : getDeclarationOfKind(symbol, SyntaxKind.InterfaceDeclaration);
            return getOuterTypeParameters(declaration);
        }

        // The local type parameters are the combined set of type parameters from all declarations of the class,
        // interface, or type alias.
        function getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol: Symbol): TypeParameter[] {
            let result: TypeParameter[];
            for (const node of symbol.declarations) {
                if (node.kind === SyntaxKind.InterfaceDeclaration || node.kind === SyntaxKind.ClassDeclaration ||
                    node.kind === SyntaxKind.ClassExpression || node.kind === SyntaxKind.TypeAliasDeclaration) {
                    const declaration = <InterfaceDeclaration | TypeAliasDeclaration>node;
                    if (declaration.typeParameters) {
                        result = appendTypeParameters(result, declaration.typeParameters);
                    }
                }
            }
            return result;
        }

        // The full set of type parameters for a generic class or interface type consists of its outer type parameters plus
        // its locally declared type parameters.
        function getTypeParametersOfClassOrInterface(symbol: Symbol): TypeParameter[] {
            return concatenate(getOuterTypeParametersOfClassOrInterface(symbol), getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol));
        }

        // A type is a mixin constructor if it has a single construct signature taking no type parameters and a single
        // rest parameter of type any[].
        function isMixinConstructorType(type: Type) {
            const signatures = getSignaturesOfType(type, SignatureKind.Construct);
            if (signatures.length === 1) {
                const s = signatures[0];
                return !s.typeParameters && s.parameters.length === 1 && s.hasRestParameter && getTypeOfParameter(s.parameters[0]) === anyArrayType;
            }
            return false;
        }

        function isConstructorType(type: Type): boolean {
            if (isValidBaseType(type) && getSignaturesOfType(type, SignatureKind.Construct).length > 0) {
                return true;
            }
            if (type.flags & TypeFlags.TypeVariable) {
                const constraint = getBaseConstraintOfType(type);
                return constraint && isValidBaseType(constraint) && isMixinConstructorType(constraint);
            }
            return false;
        }

        function getBaseTypeNodeOfClass(type: InterfaceType): ExpressionWithTypeArguments {
            const decl = <ClassLikeDeclaration>type.symbol.valueDeclaration;
            if (isInJavaScriptFile(decl)) {
                // Prefer an @augments tag because it may have type parameters.
                const tag = getJSDocAugmentsTag(decl);
                if (tag) {
                    return tag.class;
                }
            }

            return getClassExtendsHeritageClauseElement(decl);
        }

        function getConstructorsForTypeArguments(type: Type, typeArgumentNodes: ReadonlyArray<TypeNode>, location: Node): Signature[] {
            const typeArgCount = length(typeArgumentNodes);
            const isJavaScript = isInJavaScriptFile(location);
            return filter(getSignaturesOfType(type, SignatureKind.Construct),
                sig => (isJavaScript || typeArgCount >= getMinTypeArgumentCount(sig.typeParameters)) && typeArgCount <= length(sig.typeParameters));
        }

        function getInstantiatedConstructorsForTypeArguments(type: Type, typeArgumentNodes: ReadonlyArray<TypeNode>, location: Node): Signature[] {
            const signatures = getConstructorsForTypeArguments(type, typeArgumentNodes, location);
            const typeArguments = map(typeArgumentNodes, getTypeFromTypeNode);
            return sameMap(signatures, sig => some(sig.typeParameters) ? getSignatureInstantiation(sig, typeArguments, isInJavaScriptFile(location)) : sig);
        }

        /**
         * The base constructor of a class can resolve to
         * * undefinedType if the class has no extends clause,
         * * unknownType if an error occurred during resolution of the extends expression,
         * * nullType if the extends expression is the null value,
         * * anyType if the extends expression has type any, or
         * * an object type with at least one construct signature.
         */
        function getBaseConstructorTypeOfClass(type: InterfaceType): Type {
            if (!type.resolvedBaseConstructorType) {
                const decl = <ClassLikeDeclaration>type.symbol.valueDeclaration;
                const extended = getClassExtendsHeritageClauseElement(decl);
                const baseTypeNode = getBaseTypeNodeOfClass(type);
                if (!baseTypeNode) {
                    return type.resolvedBaseConstructorType = undefinedType;
                }
                if (!pushTypeResolution(type, TypeSystemPropertyName.ResolvedBaseConstructorType)) {
                    return unknownType;
                }
                const baseConstructorType = checkExpression(baseTypeNode.expression);
                if (extended && baseTypeNode !== extended) {
                    Debug.assert(!extended.typeArguments); // Because this is in a JS file, and baseTypeNode is in an @extends tag
                    checkExpression(extended.expression);
                }
                if (baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection)) {
                    // Resolving the members of a class requires us to resolve the base class of that class.
                    // We force resolution here such that we catch circularities now.
                    resolveStructuredTypeMembers(<ObjectType>baseConstructorType);
                }
                if (!popTypeResolution()) {
                    error(type.symbol.valueDeclaration, Diagnostics._0_is_referenced_directly_or_indirectly_in_its_own_base_expression, symbolToString(type.symbol));
                    return type.resolvedBaseConstructorType = unknownType;
                }
                if (!(baseConstructorType.flags & TypeFlags.Any) && baseConstructorType !== nullWideningType && !isConstructorType(baseConstructorType)) {
                    error(baseTypeNode.expression, Diagnostics.Type_0_is_not_a_constructor_function_type, typeToString(baseConstructorType));
                    return type.resolvedBaseConstructorType = unknownType;
                }
                type.resolvedBaseConstructorType = baseConstructorType;
            }
            return type.resolvedBaseConstructorType;
        }

        function getBaseTypes(type: InterfaceType): BaseType[] {
            if (!type.resolvedBaseTypes) {
                if (type.objectFlags & ObjectFlags.Tuple) {
                    type.resolvedBaseTypes = [createArrayType(getUnionType(type.typeParameters))];
                }
                else if (type.symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                    if (type.symbol.flags & SymbolFlags.Class) {
                        resolveBaseTypesOfClass(type);
                    }
                    if (type.symbol.flags & SymbolFlags.Interface) {
                        resolveBaseTypesOfInterface(type);
                    }
                }
                else {
                    Debug.fail("type must be class or interface");
                }
            }
            return type.resolvedBaseTypes;
        }

        function resolveBaseTypesOfClass(type: InterfaceType) {
            type.resolvedBaseTypes = resolvingEmptyArray;
            const baseConstructorType = getApparentType(getBaseConstructorTypeOfClass(type));
            if (!(baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection | TypeFlags.Any))) {
                return type.resolvedBaseTypes = emptyArray;
            }
            const baseTypeNode = getBaseTypeNodeOfClass(type);
            const typeArgs = typeArgumentsFromTypeReferenceNode(baseTypeNode);
            let baseType: Type;
            const originalBaseType = baseConstructorType && baseConstructorType.symbol ? getDeclaredTypeOfSymbol(baseConstructorType.symbol) : undefined;
            if (baseConstructorType.symbol && baseConstructorType.symbol.flags & SymbolFlags.Class &&
                areAllOuterTypeParametersApplied(originalBaseType)) {
                // When base constructor type is a class with no captured type arguments we know that the constructors all have the same type parameters as the
                // class and all return the instance type of the class. There is no need for further checks and we can apply the
                // type arguments in the same manner as a type reference to get the same error reporting experience.
                baseType = getTypeFromClassOrInterfaceReference(baseTypeNode, baseConstructorType.symbol, typeArgs);
            }
            else if (baseConstructorType.flags & TypeFlags.Any) {
                baseType = baseConstructorType;
            }
            else {
                // The class derives from a "class-like" constructor function, check that we have at least one construct signature
                // with a matching number of type parameters and use the return type of the first instantiated signature. Elsewhere
                // we check that all instantiated signatures return the same type.
                const constructors = getInstantiatedConstructorsForTypeArguments(baseConstructorType, baseTypeNode.typeArguments, baseTypeNode);
                if (!constructors.length) {
                    error(baseTypeNode.expression, Diagnostics.No_base_constructor_has_the_specified_number_of_type_arguments);
                    return type.resolvedBaseTypes = emptyArray;
                }
                baseType = getReturnTypeOfSignature(constructors[0]);
            }

            if (baseType === unknownType) {
                return type.resolvedBaseTypes = emptyArray;
            }
            if (!isValidBaseType(baseType)) {
                error(baseTypeNode.expression, Diagnostics.Base_constructor_return_type_0_is_not_a_class_or_interface_type, typeToString(baseType));
                return type.resolvedBaseTypes = emptyArray;
            }
            if (type === baseType || hasBaseType(baseType, type)) {
                error(type.symbol.valueDeclaration, Diagnostics.Type_0_recursively_references_itself_as_a_base_type,
                    typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType));
                return type.resolvedBaseTypes = emptyArray;
            }
            if (type.resolvedBaseTypes === resolvingEmptyArray) {
                // Circular reference, likely through instantiation of default parameters
                // (otherwise there'd be an error from hasBaseType) - this is fine, but `.members` should be reset
                // as `getIndexedAccessType` via `instantiateType` via `getTypeFromClassOrInterfaceReference` forces a
                // partial instantiation of the members without the base types fully resolved
                (type as Type as ResolvedType).members = undefined;
            }
            return type.resolvedBaseTypes = [baseType];
        }

        function areAllOuterTypeParametersApplied(type: Type): boolean {
            // An unapplied type parameter has its symbol still the same as the matching argument symbol.
            // Since parameters are applied outer-to-inner, only the last outer parameter needs to be checked.
            const outerTypeParameters = (<InterfaceType>type).outerTypeParameters;
            if (outerTypeParameters) {
                const last = outerTypeParameters.length - 1;
                const typeArguments = (<TypeReference>type).typeArguments;
                return outerTypeParameters[last].symbol !== typeArguments[last].symbol;
            }
            return true;
        }

        // A valid base type is `any`, any non-generic object type or intersection of non-generic
        // object types.
        function isValidBaseType(type: Type): type is BaseType {
            return type.flags & (TypeFlags.Object | TypeFlags.NonPrimitive | TypeFlags.Any) && !isGenericMappedType(type) ||
                type.flags & TypeFlags.Intersection && !forEach((<IntersectionType>type).types, t => !isValidBaseType(t));
        }

        function resolveBaseTypesOfInterface(type: InterfaceType): void {
            type.resolvedBaseTypes = type.resolvedBaseTypes || emptyArray;
            for (const declaration of type.symbol.declarations) {
                if (declaration.kind === SyntaxKind.InterfaceDeclaration && getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration)) {
                    for (const node of getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration)) {
                        const baseType = getTypeFromTypeNode(node);
                        if (baseType !== unknownType) {
                            if (isValidBaseType(baseType)) {
                                if (type !== baseType && !hasBaseType(baseType, type)) {
                                    if (type.resolvedBaseTypes === emptyArray) {
                                        type.resolvedBaseTypes = [<ObjectType>baseType];
                                    }
                                    else {
                                        type.resolvedBaseTypes.push(baseType);
                                    }
                                }
                                else {
                                    error(declaration, Diagnostics.Type_0_recursively_references_itself_as_a_base_type, typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType));
                                }
                            }
                            else {
                                error(node, Diagnostics.An_interface_may_only_extend_a_class_or_another_interface);
                            }
                        }
                    }
                }
            }
        }

        /**
         * Returns true if the interface given by the symbol is free of "this" references.
         *
         * Specifically, the result is true if the interface itself contains no references
         * to "this" in its body, if all base types are interfaces,
         * and if none of the base interfaces have a "this" type.
         */
        function isThislessInterface(symbol: Symbol): boolean {
            for (const declaration of symbol.declarations) {
                if (declaration.kind === SyntaxKind.InterfaceDeclaration) {
                    if (declaration.flags & NodeFlags.ContainsThis) {
                        return false;
                    }
                    const baseTypeNodes = getInterfaceBaseTypeNodes(<InterfaceDeclaration>declaration);
                    if (baseTypeNodes) {
                        for (const node of baseTypeNodes) {
                            if (isEntityNameExpression(node.expression)) {
                                const baseSymbol = resolveEntityName(node.expression, SymbolFlags.Type, /*ignoreErrors*/ true);
                                if (!baseSymbol || !(baseSymbol.flags & SymbolFlags.Interface) || getDeclaredTypeOfClassOrInterface(baseSymbol).thisType) {
                                    return false;
                                }
                            }
                        }
                    }
                }
            }
            return true;
        }

        function getDeclaredTypeOfClassOrInterface(symbol: Symbol): InterfaceType {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                const kind = symbol.flags & SymbolFlags.Class ? ObjectFlags.Class : ObjectFlags.Interface;
                const type = links.declaredType = <InterfaceType>createObjectType(kind, symbol);
                const outerTypeParameters = getOuterTypeParametersOfClassOrInterface(symbol);
                const localTypeParameters = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                // A class or interface is generic if it has type parameters or a "this" type. We always give classes a "this" type
                // because it is not feasible to analyze all members to determine if the "this" type escapes the class (in particular,
                // property types inferred from initializers and method return types inferred from return statements are very hard
                // to exhaustively analyze). We give interfaces a "this" type if we can't definitely determine that they are free of
                // "this" references.
                if (outerTypeParameters || localTypeParameters || kind === ObjectFlags.Class || !isThislessInterface(symbol)) {
                    type.objectFlags |= ObjectFlags.Reference;
                    type.typeParameters = concatenate(outerTypeParameters, localTypeParameters);
                    type.outerTypeParameters = outerTypeParameters;
                    type.localTypeParameters = localTypeParameters;
                    (<GenericType>type).instantiations = createMap<TypeReference>();
                    (<GenericType>type).instantiations.set(getTypeListId(type.typeParameters), <GenericType>type);
                    (<GenericType>type).target = <GenericType>type;
                    (<GenericType>type).typeArguments = type.typeParameters;
                    type.thisType = <TypeParameter>createType(TypeFlags.TypeParameter);
                    type.thisType.isThisType = true;
                    type.thisType.symbol = symbol;
                    type.thisType.constraint = type;
                }
            }
            return <InterfaceType>links.declaredType;
        }

        function getDeclaredTypeOfTypeAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                // Note that we use the links object as the target here because the symbol object is used as the unique
                // identity for resolution of the 'type' property in SymbolLinks.
                if (!pushTypeResolution(symbol, TypeSystemPropertyName.DeclaredType)) {
                    return unknownType;
                }

                const declaration = <JSDocTypedefTag | TypeAliasDeclaration>find(symbol.declarations, d =>
                    d.kind === SyntaxKind.JSDocTypedefTag || d.kind === SyntaxKind.TypeAliasDeclaration);
                const typeNode = declaration.kind === SyntaxKind.JSDocTypedefTag ? declaration.typeExpression : declaration.type;
                // If typeNode is missing, we will error in checkJSDocTypedefTag.
                let type = typeNode ? getTypeFromTypeNode(typeNode) : unknownType;

                if (popTypeResolution()) {
                    const typeParameters = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol);
                    if (typeParameters) {
                        // Initialize the instantiation cache for generic type aliases. The declared type corresponds to
                        // an instantiation of the type alias with the type parameters supplied as type arguments.
                        links.typeParameters = typeParameters;
                        links.instantiations = createMap<Type>();
                        links.instantiations.set(getTypeListId(typeParameters), type);
                    }
                }
                else {
                    type = unknownType;
                    error(declaration.name, Diagnostics.Type_alias_0_circularly_references_itself, symbolToString(symbol));
                }
                links.declaredType = type;
            }
            return links.declaredType;
        }

        function isLiteralEnumMember(member: EnumMember) {
            const expr = member.initializer;
            if (!expr) {
                return !(member.flags & NodeFlags.Ambient);
            }
            switch (expr.kind) {
                case SyntaxKind.StringLiteral:
                case SyntaxKind.NumericLiteral:
                    return true;
                case SyntaxKind.PrefixUnaryExpression:
                    return (<PrefixUnaryExpression>expr).operator === SyntaxKind.MinusToken &&
                        (<PrefixUnaryExpression>expr).operand.kind === SyntaxKind.NumericLiteral;
                case SyntaxKind.Identifier:
                    return nodeIsMissing(expr) || !!getSymbolOfNode(member.parent).exports.get((<Identifier>expr).escapedText);
                default:
                    return false;
            }
        }

        function getEnumKind(symbol: Symbol): EnumKind {
            const links = getSymbolLinks(symbol);
            if (links.enumKind !== undefined) {
                return links.enumKind;
            }
            let hasNonLiteralMember = false;
            for (const declaration of symbol.declarations) {
                if (declaration.kind === SyntaxKind.EnumDeclaration) {
                    for (const member of (<EnumDeclaration>declaration).members) {
                        if (member.initializer && member.initializer.kind === SyntaxKind.StringLiteral) {
                            return links.enumKind = EnumKind.Literal;
                        }
                        if (!isLiteralEnumMember(member)) {
                            hasNonLiteralMember = true;
                        }
                    }
                }
            }
            return links.enumKind = hasNonLiteralMember ? EnumKind.Numeric : EnumKind.Literal;
        }

        function getBaseTypeOfEnumLiteralType(type: Type) {
            return type.flags & TypeFlags.EnumLiteral && !(type.flags & TypeFlags.Union) ? getDeclaredTypeOfSymbol(getParentOfSymbol(type.symbol)) : type;
        }

        function getDeclaredTypeOfEnum(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (links.declaredType) {
                return links.declaredType;
            }
            if (getEnumKind(symbol) === EnumKind.Literal) {
                enumCount++;
                const memberTypeList: Type[] = [];
                for (const declaration of symbol.declarations) {
                    if (declaration.kind === SyntaxKind.EnumDeclaration) {
                        for (const member of (<EnumDeclaration>declaration).members) {
                            const memberType = getLiteralType(getEnumMemberValue(member), enumCount, getSymbolOfNode(member));
                            getSymbolLinks(getSymbolOfNode(member)).declaredType = memberType;
                            memberTypeList.push(memberType);
                        }
                    }
                }
                if (memberTypeList.length) {
                    const enumType = getUnionType(memberTypeList, UnionReduction.Literal, symbol, /*aliasTypeArguments*/ undefined);
                    if (enumType.flags & TypeFlags.Union) {
                        enumType.flags |= TypeFlags.EnumLiteral;
                        enumType.symbol = symbol;
                    }
                    return links.declaredType = enumType;
                }
            }
            const enumType = createType(TypeFlags.Enum);
            enumType.symbol = symbol;
            return links.declaredType = enumType;
        }

        function getDeclaredTypeOfEnumMember(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                const enumType = getDeclaredTypeOfEnum(getParentOfSymbol(symbol));
                if (!links.declaredType) {
                    links.declaredType = enumType;
                }
            }
            return links.declaredType;
        }

        function getDeclaredTypeOfTypeParameter(symbol: Symbol): TypeParameter {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                const type = <TypeParameter>createType(TypeFlags.TypeParameter);
                type.symbol = symbol;
                links.declaredType = type;
            }
            return <TypeParameter>links.declaredType;
        }

        function getDeclaredTypeOfAlias(symbol: Symbol): Type {
            const links = getSymbolLinks(symbol);
            if (!links.declaredType) {
                links.declaredType = getDeclaredTypeOfSymbol(resolveAlias(symbol));
            }
            return links.declaredType;
        }

        function getDeclaredTypeOfSymbol(symbol: Symbol): Type {
            return tryGetDeclaredTypeOfSymbol(symbol) || unknownType;
        }

        function tryGetDeclaredTypeOfSymbol(symbol: Symbol): Type | undefined {
            if (symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                return getDeclaredTypeOfClassOrInterface(symbol);
            }
            if (symbol.flags & SymbolFlags.TypeAlias) {
                return getDeclaredTypeOfTypeAlias(symbol);
            }
            if (symbol.flags & SymbolFlags.TypeParameter) {
                return getDeclaredTypeOfTypeParameter(symbol);
            }
            if (symbol.flags & SymbolFlags.Enum) {
                return getDeclaredTypeOfEnum(symbol);
            }
            if (symbol.flags & SymbolFlags.EnumMember) {
                return getDeclaredTypeOfEnumMember(symbol);
            }
            if (symbol.flags & SymbolFlags.Alias) {
                return getDeclaredTypeOfAlias(symbol);
            }
            return undefined;
        }

        /**
         * A type is free of this references if it's the any, string, number, boolean, symbol, or void keyword, a string
         * literal type, an array with an element type that is free of this references, or a type reference that is
         * free of this references.
         */
        function isThislessType(node: TypeNode): boolean {
            switch (node.kind) {
                case SyntaxKind.AnyKeyword:
                case SyntaxKind.StringKeyword:
                case SyntaxKind.NumberKeyword:
                case SyntaxKind.BooleanKeyword:
                case SyntaxKind.SymbolKeyword:
                case SyntaxKind.ObjectKeyword:
                case SyntaxKind.VoidKeyword:
                case SyntaxKind.UndefinedKeyword:
                case SyntaxKind.NullKeyword:
                case SyntaxKind.NeverKeyword:
                case SyntaxKind.LiteralType:
                    return true;
                case SyntaxKind.ArrayType:
                    return isThislessType((<ArrayTypeNode>node).elementType);
                case SyntaxKind.TypeReference:
                    return !(node as TypeReferenceNode).typeArguments || (node as TypeReferenceNode).typeArguments.every(isThislessType);
            }
            return false;
        }

        /** A type parameter is thisless if its contraint is thisless, or if it has no constraint. */
        function isThislessTypeParameter(node: TypeParameterDeclaration) {
            return !node.constraint || isThislessType(node.constraint);
        }

        /**
         * A variable-like declaration is free of this references if it has a type annotation
         * that is thisless, or if it has no type annotation and no initializer (and is thus of type any).
         */
        function isThislessVariableLikeDeclaration(node: VariableLikeDeclaration): boolean {
            const typeNode = getEffectiveTypeAnnotationNode(node);
            return typeNode ? isThislessType(typeNode) : !hasInitializer(node);
        }

        /**
         * A function-like declaration is considered free of `this` references if it has a return type
         * annotation that is free of this references and if each parameter is thisless and if
         * each type parameter (if present) is thisless.
         */
        function isThislessFunctionLikeDeclaration(node: FunctionLikeDeclaration): boolean {
            const returnType = getEffectiveReturnTypeNode(node);
            return (node.kind === SyntaxKind.Constructor || (returnType && isThislessType(returnType))) &&
                node.parameters.every(isThislessVariableLikeDeclaration) &&
                (!node.typeParameters || node.typeParameters.every(isThislessTypeParameter));
        }

        /**
         * Returns true if the class or interface member given by the symbol is free of "this" references. The
         * function may return false for symbols that are actually free of "this" references because it is not
         * feasible to perform a complete analysis in all cases. In particular, property members with types
         * inferred from their initializers and function members with inferred return types are conservatively
         * assumed not to be free of "this" references.
         */
        function isThisless(symbol: Symbol): boolean {
            if (symbol.declarations && symbol.declarations.length === 1) {
                const declaration = symbol.declarations[0];
                if (declaration) {
                    switch (declaration.kind) {
                        case SyntaxKind.PropertyDeclaration:
                        case SyntaxKind.PropertySignature:
                            return isThislessVariableLikeDeclaration(<VariableLikeDeclaration>declaration);
                        case SyntaxKind.MethodDeclaration:
                        case SyntaxKind.MethodSignature:
                        case SyntaxKind.Constructor:
                            return isThislessFunctionLikeDeclaration(<FunctionLikeDeclaration>declaration);
                    }
                }
            }
            return false;
        }

        // The mappingThisOnly flag indicates that the only type parameter being mapped is "this". When the flag is true,
        // we check symbols to see if we can quickly conclude they are free of "this" references, thus needing no instantiation.
        function createInstantiatedSymbolTable(symbols: Symbol[], mapper: TypeMapper, mappingThisOnly: boolean): SymbolTable {
            const result = createSymbolTable();
            for (const symbol of symbols) {
                result.set(symbol.escapedName, mappingThisOnly && isThisless(symbol) ? symbol : instantiateSymbol(symbol, mapper));
            }
            return result;
        }

        function addInheritedMembers(symbols: SymbolTable, baseSymbols: Symbol[]) {
            for (const s of baseSymbols) {
                if (!symbols.has(s.escapedName)) {
                    symbols.set(s.escapedName, s);
                }
            }
        }

        function resolveDeclaredMembers(type: InterfaceType): InterfaceTypeWithDeclaredMembers {
            if (!(<InterfaceTypeWithDeclaredMembers>type).declaredProperties) {
                const symbol = type.symbol;
                const members = getMembersOfSymbol(symbol);
                (<InterfaceTypeWithDeclaredMembers>type).declaredProperties = getNamedMembers(members);
                (<InterfaceTypeWithDeclaredMembers>type).declaredCallSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.Call));
                (<InterfaceTypeWithDeclaredMembers>type).declaredConstructSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.New));
                (<InterfaceTypeWithDeclaredMembers>type).declaredStringIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.String);
                (<InterfaceTypeWithDeclaredMembers>type).declaredNumberIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.Number);
            }
            return <InterfaceTypeWithDeclaredMembers>type;
        }

        /**
         * Indicates whether a type can be used as a late-bound name.
         */
        function isTypeUsableAsLateBoundName(type: Type): type is LiteralType | UniqueESSymbolType {
            return !!(type.flags & TypeFlags.StringOrNumberLiteralOrUnique);
        }

        /**
         * Indicates whether a declaration name is definitely late-bindable.
         * A declaration name is only late-bindable if:
         * - It is a `ComputedPropertyName`.
         * - Its expression is an `Identifier` or either a `PropertyAccessExpression` an
         * `ElementAccessExpression` consisting only of these same three types of nodes.
         * - The type of its expression is a string or numeric literal type, or is a `unique symbol` type.
         */
        function isLateBindableName(node: DeclarationName): node is LateBoundName {
            return isComputedPropertyName(node)
                && isEntityNameExpression(node.expression)
                && isTypeUsableAsLateBoundName(checkComputedPropertyName(node));
        }

        /**
         * Indicates whether a declaration has a late-bindable dynamic name.
         */
        function hasLateBindableName(node: Declaration): node is LateBoundDeclaration {
            const name = getNameOfDeclaration(node);
            return name && isLateBindableName(name);
        }

        /**
         * Indicates whether a declaration has a dynamic name that cannot be late-bound.
         */
        function hasNonBindableDynamicName(node: Declaration) {
            return hasDynamicName(node) && !hasLateBindableName(node);
        }

        /**
         * Indicates whether a declaration name is a dynamic name that cannot be late-bound.
         */
        function isNonBindableDynamicName(node: DeclarationName) {
            return isDynamicName(node) && !isLateBindableName(node);
        }

        /**
         * Gets the symbolic name for a late-bound member from its type.
         */
        function getLateBoundNameFromType(type: LiteralType | UniqueESSymbolType) {
            if (type.flags & TypeFlags.UniqueESSymbol) {
                return `__@${type.symbol.escapedName}@${getSymbolId(type.symbol)}` as __String;
            }
            if (type.flags & TypeFlags.StringOrNumberLiteral) {
                return escapeLeadingUnderscores("" + (<LiteralType>type).value);
            }
        }

        /**
         * Adds a declaration to a late-bound dynamic member. This performs the same function for
         * late-bound members that `addDeclarationToSymbol` in binder.ts performs for early-bound
         * members.
         */
        function addDeclarationToLateBoundSymbol(symbol: Symbol, member: LateBoundDeclaration, symbolFlags: SymbolFlags) {
            Debug.assert(!!(getCheckFlags(symbol) & CheckFlags.Late), "Expected a late-bound symbol.");
            symbol.flags |= symbolFlags;
            getSymbolLinks(member.symbol).lateSymbol = symbol;
            if (!symbol.declarations) {
                symbol.declarations = [member];
            }
            else {
                symbol.declarations.push(member);
            }
            if (symbolFlags & SymbolFlags.Value) {
                const valueDeclaration = symbol.valueDeclaration;
                if (!valueDeclaration || valueDeclaration.kind !== member.kind) {
                    symbol.valueDeclaration = member;
                }
            }
        }

        /**
         * Performs late-binding of a dynamic member. This performs the same function for
         * late-bound members that `declareSymbol` in binder.ts performs for early-bound
         * members.
         *
         * If a symbol is a dynamic name from a computed property, we perform an additional "late"
         * binding phase to attempt to resolve the name for the symbol from the type of the computed
         * property's expression. If the type of the expression is a string-literal, numeric-literal,
         * or unique symbol type, we can use that type as the name of the symbol.
         *
         * For example, given:
         *
         *   const x = Symbol();
         *
         *   interface I {
         *     [x]: number;
         *   }
         *
         * The binder gives the property `[x]: number` a special symbol with the name "__computed".
         * In the late-binding phase we can type-check the expression `x` and see that it has a
         * unique symbol type which we can then use as the name of the member. This allows users
         * to define custom symbols that can be used in the members of an object type.
         *
         * @param parent The containing symbol for the member.
         * @param earlySymbols The early-bound symbols of the parent.
         * @param lateSymbols The late-bound symbols of the parent.
         * @param decl The member to bind.
         */
        function lateBindMember(parent: Symbol, earlySymbols: SymbolTable | undefined, lateSymbols: SymbolTable, decl: LateBoundDeclaration) {
            Debug.assert(!!decl.symbol, "The member is expected to have a symbol.");
            const links = getNodeLinks(decl);
            if (!links.resolvedSymbol) {
                // In the event we attempt to resolve the late-bound name of this member recursively,
                // fall back to the early-bound name of this member.
                links.resolvedSymbol = decl.symbol;
                const type = checkComputedPropertyName(decl.name);
                if (isTypeUsableAsLateBoundName(type)) {
                    const memberName = getLateBoundNameFromType(type);
                    const symbolFlags = decl.symbol.flags;

                    // Get or add a late-bound symbol for the member. This allows us to merge late-bound accessor declarations.
                    let lateSymbol = lateSymbols.get(memberName);
                    if (!lateSymbol) lateSymbols.set(memberName, lateSymbol = createSymbol(SymbolFlags.None, memberName, CheckFlags.Late));

                    // Report an error if a late-bound member has the same name as an early-bound member,
                    // or if we have another early-bound symbol declaration with the same name and
                    // conflicting flags.
                    const earlySymbol = earlySymbols && earlySymbols.get(memberName);
                    if (lateSymbol.flags & getExcludedSymbolFlags(symbolFlags) || earlySymbol) {
                        // If we have an existing early-bound member, combine its declarations so that we can
                        // report an error at each declaration.
                        const declarations = earlySymbol ? concatenate(earlySymbol.declarations, lateSymbol.declarations) : lateSymbol.declarations;
                        const name = declarationNameToString(decl.name);
                        forEach(declarations, declaration => error(getNameOfDeclaration(declaration) || declaration, Diagnostics.Duplicate_declaration_0, name));
                        error(decl.name || decl, Diagnostics.Duplicate_declaration_0, name);
                        lateSymbol = createSymbol(SymbolFlags.None, memberName, CheckFlags.Late);
                    }

                    const symbolLinks = getSymbolLinks(lateSymbol);
                    if (!symbolLinks.nameType) {
                        // Retain link to name type so that it can be reused later
                        symbolLinks.nameType = type;
                    }

                    addDeclarationToLateBoundSymbol(lateSymbol, decl, symbolFlags);
                    if (lateSymbol.parent) {
                        Debug.assert(lateSymbol.parent === parent, "Existing symbol parent should match new one");
                    }
                    else {
                        lateSymbol.parent = parent;
                    }
                    return links.resolvedSymbol = lateSymbol;
                }
            }
            return links.resolvedSymbol;
        }

        function getResolvedMembersOrExportsOfSymbol(symbol: Symbol, resolutionKind: MembersOrExportsResolutionKind) {
            const links = getSymbolLinks(symbol);
            if (!links[resolutionKind]) {
                const isStatic = resolutionKind === MembersOrExportsResolutionKind.resolvedExports;
                const earlySymbols = !isStatic ? symbol.members :
                    symbol.flags & SymbolFlags.Module ? getExportsOfModuleWorker(symbol) :
                    symbol.exports;

                // In the event we recursively resolve the members/exports of the symbol, we
                // set the initial value of resolvedMembers/resolvedExports to the early-bound
                // members/exports of the symbol.
                links[resolutionKind] = earlySymbols || emptySymbols;

                // fill in any as-yet-unresolved late-bound members.
                const lateSymbols = createSymbolTable();
                for (const decl of symbol.declarations) {
                    const members = getMembersOfDeclaration(decl);
                    if (members) {
                        for (const member of members) {
                            if (isStatic === hasStaticModifier(member) && hasLateBindableName(member)) {
                                lateBindMember(symbol, earlySymbols, lateSymbols, member);
                            }
                        }
                    }
                }

                links[resolutionKind] = combineSymbolTables(earlySymbols, lateSymbols) || emptySymbols;
            }

            return links[resolutionKind];
        }

        /**
         * Gets a SymbolTable containing both the early- and late-bound members of a symbol.
         *
         * For a description of late-binding, see `lateBindMember`.
         */
        function getMembersOfSymbol(symbol: Symbol) {
            return symbol.flags & SymbolFlags.LateBindingContainer
                ? getResolvedMembersOrExportsOfSymbol(symbol, MembersOrExportsResolutionKind.resolvedMembers)
                : symbol.members || emptySymbols;
        }

        /**
         * If a symbol is the dynamic name of the member of an object type, get the late-bound
         * symbol of the member.
         *
         * For a description of late-binding, see `lateBindMember`.
         */
        function getLateBoundSymbol(symbol: Symbol): Symbol {
            if (symbol.flags & SymbolFlags.ClassMember && symbol.escapedName === InternalSymbolName.Computed) {
                const links = getSymbolLinks(symbol);
                if (!links.lateSymbol && some(symbol.declarations, hasLateBindableName)) {
                    // force late binding of members/exports. This will set the late-bound symbol
                    if (some(symbol.declarations, hasStaticModifier)) {
                        getExportsOfSymbol(symbol.parent);
                    }
                    else {
                        getMembersOfSymbol(symbol.parent);
                    }
                }
                return links.lateSymbol || (links.lateSymbol = symbol);
            }
            return symbol;
        }

        function getTypeWithThisArgument(type: Type, thisArgument?: Type, needApparentType?: boolean): Type {
            if (getObjectFlags(type) & ObjectFlags.Reference) {
                const target = (<TypeReference>type).target;
                const typeArguments = (<TypeReference>type).typeArguments;
                if (length(target.typeParameters) === length(typeArguments)) {
                    const ref = createTypeReference(target, concatenate(typeArguments, [thisArgument || target.thisType]));
                    return needApparentType ? getApparentType(ref) : ref;
                }
            }
            else if (type.flags & TypeFlags.Intersection) {
                return getIntersectionType(map((<IntersectionType>type).types, t => getTypeWithThisArgument(t, thisArgument, needApparentType)));
            }
            return needApparentType ? getApparentType(type) : type;
        }

        function resolveObjectTypeMembers(type: ObjectType, source: InterfaceTypeWithDeclaredMembers, typeParameters: TypeParameter[], typeArguments: Type[]) {
            let mapper: TypeMapper;
            let members: SymbolTable;
            let callSignatures: Signature[];
            let constructSignatures: Signature[];
            let stringIndexInfo: IndexInfo;
            let numberIndexInfo: IndexInfo;
            if (rangeEquals(typeParameters, typeArguments, 0, typeParameters.length)) {
                mapper = identityMapper;
                members = source.symbol ? getMembersOfSymbol(source.symbol) : createSymbolTable(source.declaredProperties);
                callSignatures = source.declaredCallSignatures;
                constructSignatures = source.declaredConstructSignatures;
                stringIndexInfo = source.declaredStringIndexInfo;
                numberIndexInfo = source.declaredNumberIndexInfo;
            }
            else {
                mapper = createTypeMapper(typeParameters, typeArguments);
                members = createInstantiatedSymbolTable(source.declaredProperties, mapper, /*mappingThisOnly*/ typeParameters.length === 1);
                callSignatures = instantiateSignatures(source.declaredCallSignatures, mapper);
                constructSignatures = instantiateSignatures(source.declaredConstructSignatures, mapper);
                stringIndexInfo = instantiateIndexInfo(source.declaredStringIndexInfo, mapper);
                numberIndexInfo = instantiateIndexInfo(source.declaredNumberIndexInfo, mapper);
            }
            const baseTypes = getBaseTypes(source);
            if (baseTypes.length) {
                if (source.symbol && members === getMembersOfSymbol(source.symbol)) {
                    members = createSymbolTable(source.declaredProperties);
                }
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
                const thisArgument = lastOrUndefined(typeArguments);
                for (const baseType of baseTypes) {
                    const instantiatedBaseType = thisArgument ? getTypeWithThisArgument(instantiateType(baseType, mapper), thisArgument) : baseType;
                    addInheritedMembers(members, getPropertiesOfType(instantiatedBaseType));
                    callSignatures = concatenate(callSignatures, getSignaturesOfType(instantiatedBaseType, SignatureKind.Call));
                    constructSignatures = concatenate(constructSignatures, getSignaturesOfType(instantiatedBaseType, SignatureKind.Construct));
                    if (!stringIndexInfo) {
                        stringIndexInfo = instantiatedBaseType === anyType ?
                            createIndexInfo(anyType, /*isReadonly*/ false) :
                            getIndexInfoOfType(instantiatedBaseType, IndexKind.String);
                    }
                    numberIndexInfo = numberIndexInfo || getIndexInfoOfType(instantiatedBaseType, IndexKind.Number);
                }
            }
            setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function resolveClassOrInterfaceMembers(type: InterfaceType): void {
            resolveObjectTypeMembers(type, resolveDeclaredMembers(type), emptyArray, emptyArray);
        }

        function resolveTypeReferenceMembers(type: TypeReference): void {
            const source = resolveDeclaredMembers(type.target);
            const typeParameters = concatenate(source.typeParameters, [source.thisType]);
            const typeArguments = type.typeArguments && type.typeArguments.length === typeParameters.length ?
                type.typeArguments : concatenate(type.typeArguments, [type]);
            resolveObjectTypeMembers(type, source, typeParameters, typeArguments);
        }

        function createSignature(
            declaration: SignatureDeclaration,
            typeParameters: TypeParameter[],
            thisParameter: Symbol | undefined,
            parameters: Symbol[],
            resolvedReturnType: Type | undefined,
            resolvedTypePredicate: TypePredicate | undefined,
            minArgumentCount: number,
            hasRestParameter: boolean,
            hasLiteralTypes: boolean,
        ): Signature {
            const sig = new Signature(checker);
            sig.declaration = declaration;
            sig.typeParameters = typeParameters;
            sig.parameters = parameters;
            sig.thisParameter = thisParameter;
            sig.resolvedReturnType = resolvedReturnType;
            sig.resolvedTypePredicate = resolvedTypePredicate;
            sig.minArgumentCount = minArgumentCount;
            sig.hasRestParameter = hasRestParameter;
            sig.hasLiteralTypes = hasLiteralTypes;
            sig.target = undefined;
            sig.mapper = undefined;
            return sig;
        }

        function cloneSignature(sig: Signature): Signature {
            return createSignature(sig.declaration, sig.typeParameters, sig.thisParameter, sig.parameters, /*resolvedReturnType*/ undefined,
                /*resolvedTypePredicate*/ undefined, sig.minArgumentCount, sig.hasRestParameter, sig.hasLiteralTypes);
        }

        function getDefaultConstructSignatures(classType: InterfaceType): Signature[] {
            const baseConstructorType = getBaseConstructorTypeOfClass(classType);
            const baseSignatures = getSignaturesOfType(baseConstructorType, SignatureKind.Construct);
            if (baseSignatures.length === 0) {
                return [createSignature(undefined, classType.localTypeParameters, undefined, emptyArray, classType, /*resolvedTypePredicate*/ undefined, 0, /*hasRestParameter*/ false, /*hasLiteralTypes*/ false)];
            }
            const baseTypeNode = getBaseTypeNodeOfClass(classType);
            const isJavaScript = isInJavaScriptFile(baseTypeNode);
            const typeArguments = typeArgumentsFromTypeReferenceNode(baseTypeNode);
            const typeArgCount = length(typeArguments);
            const result: Signature[] = [];
            for (const baseSig of baseSignatures) {
                const minTypeArgumentCount = getMinTypeArgumentCount(baseSig.typeParameters);
                const typeParamCount = length(baseSig.typeParameters);
                if (isJavaScript || typeArgCount >= minTypeArgumentCount && typeArgCount <= typeParamCount) {
                    const sig = typeParamCount ? createSignatureInstantiation(baseSig, fillMissingTypeArguments(typeArguments, baseSig.typeParameters, minTypeArgumentCount, isJavaScript)) : cloneSignature(baseSig);
                    sig.typeParameters = classType.localTypeParameters;
                    sig.resolvedReturnType = classType;
                    result.push(sig);
                }
            }
            return result;
        }

        function findMatchingSignature(signatureList: Signature[], signature: Signature, partialMatch: boolean, ignoreThisTypes: boolean, ignoreReturnTypes: boolean): Signature {
            for (const s of signatureList) {
                if (compareSignaturesIdentical(s, signature, partialMatch, ignoreThisTypes, ignoreReturnTypes, compareTypesIdentical)) {
                    return s;
                }
            }
        }

        function findMatchingSignatures(signatureLists: Signature[][], signature: Signature, listIndex: number): Signature[] {
            if (signature.typeParameters) {
                // We require an exact match for generic signatures, so we only return signatures from the first
                // signature list and only if they have exact matches in the other signature lists.
                if (listIndex > 0) {
                    return undefined;
                }
                for (let i = 1; i < signatureLists.length; i++) {
                    if (!findMatchingSignature(signatureLists[i], signature, /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ false)) {
                        return undefined;
                    }
                }
                return [signature];
            }
            let result: Signature[];
            for (let i = 0; i < signatureLists.length; i++) {
                // Allow matching non-generic signatures to have excess parameters and different return types
                const match = i === listIndex ? signature : findMatchingSignature(signatureLists[i], signature, /*partialMatch*/ true, /*ignoreThisTypes*/ true, /*ignoreReturnTypes*/ true);
                if (!match) {
                    return undefined;
                }
                result = appendIfUnique(result, match);
            }
            return result;
        }

        // The signatures of a union type are those signatures that are present in each of the constituent types.
        // Generic signatures must match exactly, but non-generic signatures are allowed to have extra optional
        // parameters and may differ in return types. When signatures differ in return types, the resulting return
        // type is the union of the constituent return types.
        function getUnionSignatures(types: Type[], kind: SignatureKind): Signature[] {
            const signatureLists = map(types, t => getSignaturesOfType(t, kind));
            let result: Signature[];
            for (let i = 0; i < signatureLists.length; i++) {
                for (const signature of signatureLists[i]) {
                    // Only process signatures with parameter lists that aren't already in the result list
                    if (!result || !findMatchingSignature(result, signature, /*partialMatch*/ false, /*ignoreThisTypes*/ true, /*ignoreReturnTypes*/ true)) {
                        const unionSignatures = findMatchingSignatures(signatureLists, signature, i);
                        if (unionSignatures) {
                            let s = signature;
                            // Union the result types when more than one signature matches
                            if (unionSignatures.length > 1) {
                                let thisParameter = signature.thisParameter;
                                if (forEach(unionSignatures, sig => sig.thisParameter)) {
                                    const thisType = getUnionType(map(unionSignatures, sig => sig.thisParameter ? getTypeOfSymbol(sig.thisParameter) : anyType), UnionReduction.Subtype);
                                    thisParameter = createSymbolWithType(signature.thisParameter, thisType);
                                }
                                s = cloneSignature(signature);
                                s.thisParameter = thisParameter;
                                s.unionSignatures = unionSignatures;
                            }
                            (result || (result = [])).push(s);
                        }
                    }
                }
            }
            return result || emptyArray;
        }

        function getUnionIndexInfo(types: Type[], kind: IndexKind): IndexInfo {
            const indexTypes: Type[] = [];
            let isAnyReadonly = false;
            for (const type of types) {
                const indexInfo = getIndexInfoOfType(type, kind);
                if (!indexInfo) {
                    return undefined;
                }
                indexTypes.push(indexInfo.type);
                isAnyReadonly = isAnyReadonly || indexInfo.isReadonly;
            }
            return createIndexInfo(getUnionType(indexTypes, UnionReduction.Subtype), isAnyReadonly);
        }

        function resolveUnionTypeMembers(type: UnionType) {
            // The members and properties collections are empty for union types. To get all properties of a union
            // type use getPropertiesOfType (only the language service uses this).
            const callSignatures = getUnionSignatures(type.types, SignatureKind.Call);
            const constructSignatures = getUnionSignatures(type.types, SignatureKind.Construct);
            const stringIndexInfo = getUnionIndexInfo(type.types, IndexKind.String);
            const numberIndexInfo = getUnionIndexInfo(type.types, IndexKind.Number);
            setStructuredTypeMembers(type, emptySymbols, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        function intersectTypes(type1: Type, type2: Type): Type {
            return !type1 ? type2 : !type2 ? type1 : getIntersectionType([type1, type2]);
        }

        function intersectIndexInfos(info1: IndexInfo, info2: IndexInfo): IndexInfo {
            return !info1 ? info2 : !info2 ? info1 : createIndexInfo(
                getIntersectionType([info1.type, info2.type]), info1.isReadonly && info2.isReadonly);
        }

        function unionSpreadIndexInfos(info1: IndexInfo, info2: IndexInfo): IndexInfo {
            return info1 && info2 && createIndexInfo(
                getUnionType([info1.type, info2.type]), info1.isReadonly || info2.isReadonly);
        }

        function includeMixinType(type: Type, types: Type[], index: number): Type {
            const mixedTypes: Type[] = [];
            for (let i = 0; i < types.length; i++) {
                if (i === index) {
                    mixedTypes.push(type);
                }
                else if (isMixinConstructorType(types[i])) {
                    mixedTypes.push(getReturnTypeOfSignature(getSignaturesOfType(types[i], SignatureKind.Construct)[0]));
                }
            }
            return getIntersectionType(mixedTypes);
        }

        function resolveIntersectionTypeMembers(type: IntersectionType) {
            // The members and properties collections are empty for intersection types. To get all properties of an
            // intersection type use getPropertiesOfType (only the language service uses this).
            let callSignatures: Signature[] = emptyArray;
            let constructSignatures: Signature[] = emptyArray;
            let stringIndexInfo: IndexInfo;
            let numberIndexInfo: IndexInfo;
            const types = type.types;
            const mixinCount = countWhere(types, isMixinConstructorType);
            for (let i = 0; i < types.length; i++) {
                const t = type.types[i];
                // When an intersection type contains mixin constructor types, the construct signatures from
                // those types are discarded and their return types are mixed into the return types of all
                // other construct signatures in the intersection type. For example, the intersection type
                // '{ new(...args: any[]) => A } & { new(s: string) => B }' has a single construct signature
                // 'new(s: string) => A & B'.
                if (mixinCount === 0 || mixinCount === types.length && i === 0 || !isMixinConstructorType(t)) {
                    let signatures = getSignaturesOfType(t, SignatureKind.Construct);
                    if (signatures.length && mixinCount > 0) {
                        signatures = map(signatures, s => {
                            const clone = cloneSignature(s);
                            clone.resolvedReturnType = includeMixinType(getReturnTypeOfSignature(s), types, i);
                            return clone;
                        });
                    }
                    constructSignatures = concatenate(constructSignatures, signatures);
                }
                callSignatures = concatenate(callSignatures, getSignaturesOfType(t, SignatureKind.Call));
                stringIndexInfo = intersectIndexInfos(stringIndexInfo, getIndexInfoOfType(t, IndexKind.String));
                numberIndexInfo = intersectIndexInfos(numberIndexInfo, getIndexInfoOfType(t, IndexKind.Number));
            }
            setStructuredTypeMembers(type, emptySymbols, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
        }

        /**
         * Converts an AnonymousType to a ResolvedType.
         */
        function resolveAnonymousTypeMembers(type: AnonymousType) {
            const symbol = type.symbol;
            if (type.target) {
                const members = createInstantiatedSymbolTable(getPropertiesOfObjectType(type.target), type.mapper, /*mappingThisOnly*/ false);
                const callSignatures = instantiateSignatures(getSignaturesOfType(type.target, SignatureKind.Call), type.mapper);
                const constructSignatures = instantiateSignatures(getSignaturesOfType(type.target, SignatureKind.Construct), type.mapper);
                const stringIndexInfo = instantiateIndexInfo(getIndexInfoOfType(type.target, IndexKind.String), type.mapper);
                const numberIndexInfo = instantiateIndexInfo(getIndexInfoOfType(type.target, IndexKind.Number), type.mapper);
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
            }
            else if (symbol.flags & SymbolFlags.TypeLiteral) {
                const members = getMembersOfSymbol(symbol);
                const callSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.Call));
                const constructSignatures = getSignaturesOfSymbol(members.get(InternalSymbolName.New));
                const stringIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.String);
                const numberIndexInfo = getIndexInfoOfSymbol(symbol, IndexKind.Number);
                setStructuredTypeMembers(type, members, callSignatures, constructSignatures, stringIndexInfo, numberIndexInfo);
            }
            else {
                // Combinations of function, class, enum and module
                let members = emptySymbols;
                let stringIndexInfo: IndexInfo;
                if (symbol.exports) {
                    members = getExportsOfSymbol(symbol);
                }
                setStructuredTypeMembers(type, members, emptyArray, emptyArray, undefined, undefined);
                if (symbol.flags & SymbolFlags.Class) {
                    const classType = getDeclaredTypeOfClassOrInterface(symbol);
                    const baseConstructorType = getBaseConstructorTypeOfClass(classType);
                    if (baseConstructorType.flags & (TypeFlags.Object | TypeFlags.Intersection | TypeFlags.TypeVariable)) {
                        members = createSymbolTable(getNamedMembers(members));
                        addInheritedMembers(members, getPropertiesOfType(baseConstructorType));
                    }
                    else if (baseConstructorType === anyType) {
                        stringIndexInfo = createIndexInfo(anyType, /*isReadonly*/ false);
                    }
                }
                const numberIndexInfo = symbol.flags & SymbolFlags.Enum ? enumNumberIndexInfo : undefined;
                setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);
                // We resolve the members before computing the signatures because a signature may use
                // typeof with a qualified name expression that circularly references the type we are
                // in the process of resolving (see issue #6072). The temporarily empty signature list
                // will never be observed because a qualified name can't reference signatures.
                if (symbol.flags & (SymbolFlags.Function | SymbolFlags.Method)) {
                    (<ResolvedType>type).callSignatures = getSignaturesOfSymbol(symbol);
                }
                // And likewise for construct signatures for classes
                if (symbol.flags & SymbolFlags.Class) {
                    const classType = getDeclaredTypeOfClassOrInterface(symbol);
                    let constructSignatures = getSignaturesOfSymbol(symbol.members.get(InternalSymbolName.Constructor));
                    if (!constructSignatures.length) {
                        constructSignatures = getDefaultConstructSignatures(classType);
                    }
                    (<ResolvedType>type).constructSignatures = constructSignatures;
                }
            }
        }

        function resolveReverseMappedTypeMembers(type: ReverseMappedType) {
            const indexInfo = getIndexInfoOfType(type.source, IndexKind.String);
            const modifiers = getMappedTypeModifiers(type.mappedType);
            const readonlyMask = modifiers & MappedTypeModifiers.IncludeReadonly ? false : true;
            const optionalMask = modifiers & MappedTypeModifiers.IncludeOptional ? 0 : SymbolFlags.Optional;
            const stringIndexInfo = indexInfo && createIndexInfo(inferReverseMappedType(indexInfo.type, type.mappedType), readonlyMask && indexInfo.isReadonly);
            const members = createSymbolTable();
            for (const prop of getPropertiesOfType(type.source)) {
                const checkFlags = CheckFlags.ReverseMapped | (readonlyMask && isReadonlySymbol(prop) ? CheckFlags.Readonly : 0);
                const inferredProp = createSymbol(SymbolFlags.Property | prop.flags & optionalMask, prop.escapedName, checkFlags) as ReverseMappedSymbol;
                inferredProp.declarations = prop.declarations;
                inferredProp.propertyType = getTypeOfSymbol(prop);
                inferredProp.mappedType = type.mappedType;
                members.set(prop.escapedName, inferredProp);
            }
            setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, undefined);
        }

        /** Resolve the members of a mapped type { [P in K]: T } */
        function resolveMappedTypeMembers(type: MappedType) {
            const members: SymbolTable = createSymbolTable();
            let stringIndexInfo: IndexInfo;
            // Resolve upfront such that recursive references see an empty object type.
            setStructuredTypeMembers(type, emptySymbols, emptyArray, emptyArray, undefined, undefined);
            // In { [P in K]: T }, we refer to P as the type parameter type, K as the constraint type,
            // and T as the template type.
            const typeParameter = getTypeParameterFromMappedType(type);
            const constraintType = getConstraintTypeFromMappedType(type);
            const templateType = getTemplateTypeFromMappedType(<MappedType>type.target || type);
            const modifiersType = getApparentType(getModifiersTypeFromMappedType(type)); // The 'T' in 'keyof T'
            const templateModifiers = getMappedTypeModifiers(type);
            const constraintDeclaration = type.declaration.typeParameter.constraint;
            if (constraintDeclaration.kind === SyntaxKind.TypeOperator &&
                (<TypeOperatorNode>constraintDeclaration).operator === SyntaxKind.KeyOfKeyword) {
                // We have a { [P in keyof T]: X }
                for (const propertySymbol of getPropertiesOfType(modifiersType)) {
                    addMemberForKeyType(getLiteralTypeFromPropertyName(propertySymbol), propertySymbol);
                }
                if (modifiersType.flags & TypeFlags.Any || getIndexInfoOfType(modifiersType, IndexKind.String)) {
                    addMemberForKeyType(stringType);
                }
            }
            else {
                // First, if the constraint type is a type parameter, obtain the base constraint. Then,
                // if the key type is a 'keyof X', obtain 'keyof C' where C is the base constraint of X.
                // Finally, iterate over the constituents of the resulting iteration type.
                const keyType = constraintType.flags & TypeFlags.InstantiableNonPrimitive ? getApparentType(constraintType) : constraintType;
                const iterationType = keyType.flags & TypeFlags.Index ? getIndexType(getApparentType((<IndexType>keyType).type)) : keyType;
                forEachType(iterationType, addMemberForKeyType);
            }
            setStructuredTypeMembers(type, members, emptyArray, emptyArray, stringIndexInfo, undefined);

            function addMemberForKeyType(t: Type, propertySymbolOrIndex?: Symbol | number) {
                let propertySymbol: Symbol;
                // forEachType delegates to forEach, which calls with a numeric second argument
                //  the type system currently doesn't catch this incompatibility, so we annotate
                //  the function ourselves to indicate the runtime behavior and deal with it here
                if (typeof propertySymbolOrIndex === "object") {
                    propertySymbol = propertySymbolOrIndex;
                }
                // Create a mapper from T to the current iteration type constituent. Then, if the
                // mapped type is itself an instantiated type, combine the iteration mapper with the
                // instantiation mapper.
                const templateMapper = combineTypeMappers(type.mapper, createTypeMapper([typeParameter], [t]));
                const propType = instantiateType(templateType, templateMapper);
                // If the current iteration type constituent is a string literal type, create a property.
                // Otherwise, for type string create a string index signature.
                if (t.flags & TypeFlags.StringLiteral) {
                    const propName = getLateBoundNameFromType(t as LiteralType | UniqueESSymbolType);
                    const modifiersProp = getPropertyOfType(modifiersType, propName);
                    const isOptional = !!(templateModifiers & MappedTypeModifiers.IncludeOptional ||
                        !(templateModifiers & MappedTypeModifiers.ExcludeOptional) && modifiersProp && modifiersProp.flags & SymbolFlags.Optional);
                    const isReadonly = !!(templateModifiers & MappedTypeModifiers.IncludeReadonly ||
                        !(templateModifiers & MappedTypeModifiers.ExcludeReadonly) && modifiersProp && isReadonlySymbol(modifiersProp));
                    const prop = createSymbol(SymbolFlags.Property | (isOptional ? SymbolFlags.Optional : 0), propName, isReadonly ? CheckFlags.Readonly : 0);
                    // When creating an optional property in strictNullChecks mode, if 'undefined' isn't assignable to the
                    // type, we include 'undefined' in the type. Similarly, when creating a non-optional property in strictNullChecks
                    // mode, if the underlying property is optional we remove 'undefined' from the type.
                    prop.type = strictNullChecks && isOptional && !isTypeAssignableTo(undefinedType, propType) ? getOptionalType(propType) :
                        strictNullChecks && !isOptional && modifiersProp && modifiersProp.flags & SymbolFlags.Optional ? getTypeWithFacts(propType, TypeFacts.NEUndefined) :
                        propType;
                    if (propertySymbol) {
                        prop.syntheticOrigin = propertySymbol;
                        prop.declarations = propertySymbol.declarations;
                    }
                    prop.nameType = t;
                    members.set(propName, prop);
                }
                else if (t.flags & (TypeFlags.Any | TypeFlags.String)) {
                    stringIndexInfo = createIndexInfo(propType, !!(templateModifiers & MappedTypeModifiers.IncludeReadonly));
                }
            }
        }

        function getTypeParameterFromMappedType(type: MappedType) {
            return type.typeParameter ||
                (type.typeParameter = getDeclaredTypeOfTypeParameter(getSymbolOfNode(type.declaration.typeParameter)));
        }

        function getConstraintTypeFromMappedType(type: MappedType) {
            return type.constraintType ||
                (type.constraintType = instantiateType(getConstraintOfTypeParameter(getTypeParameterFromMappedType(type)), type.mapper || identityMapper) || unknownType);
        }

        function getTemplateTypeFromMappedType(type: MappedType) {
            return type.templateType ||
                (type.templateType = type.declaration.type ?
                    instantiateType(addOptionality(getTypeFromTypeNode(type.declaration.type), !!(getMappedTypeModifiers(type) & MappedTypeModifiers.IncludeOptional)), type.mapper || identityMapper) :
                    unknownType);
        }

        function getModifiersTypeFromMappedType(type: MappedType) {
            if (!type.modifiersType) {
                const constraintDeclaration = type.declaration.typeParameter.constraint;
                if (constraintDeclaration.kind === SyntaxKind.TypeOperator &&
                    (<TypeOperatorNode>constraintDeclaration).operator === SyntaxKind.KeyOfKeyword) {
                    // If the constraint declaration is a 'keyof T' node, the modifiers type is T. We check
                    // AST nodes here because, when T is a non-generic type, the logic below eagerly resolves
                    // 'keyof T' to a literal union type and we can't recover T from that type.
                    type.modifiersType = instantiateType(getTypeFromTypeNode((<TypeOperatorNode>constraintDeclaration).type), type.mapper || identityMapper);
                }
                else {
                    // Otherwise, get the declared constraint type, and if the constraint type is a type parameter,
                    // get the constraint of that type parameter. If the resulting type is an indexed type 'keyof T',
                    // the modifiers type is T. Otherwise, the modifiers type is {}.
                    const declaredType = <MappedType>getTypeFromMappedTypeNode(type.declaration);
                    const constraint = getConstraintTypeFromMappedType(declaredType);
                    const extendedConstraint = constraint && constraint.flags & TypeFlags.TypeParameter ? getConstraintOfTypeParameter(<TypeParameter>constraint) : constraint;
                    type.modifiersType = extendedConstraint && extendedConstraint.flags & TypeFlags.Index ? instantiateType((<IndexType>extendedConstraint).type, type.mapper || identityMapper) : emptyObjectType;
                }
            }
            return type.modifiersType;
        }

        function getMappedTypeModifiers(type: MappedType): MappedTypeModifiers {
            const declaration = type.declaration;
            return (declaration.readonlyToken ? declaration.readonlyToken.kind === SyntaxKind.MinusToken ? MappedTypeModifiers.ExcludeReadonly : MappedTypeModifiers.IncludeReadonly : 0) |
                (declaration.questionToken ? declaration.questionToken.kind === SyntaxKind.MinusToken ? MappedTypeModifiers.ExcludeOptional : MappedTypeModifiers.IncludeOptional : 0);
        }

        function getMappedTypeOptionality(type: MappedType): number {
            const modifiers = getMappedTypeModifiers(type);
            return modifiers & MappedTypeModifiers.ExcludeOptional ? -1 : modifiers & MappedTypeModifiers.IncludeOptional ? 1 : 0;
        }

        function getCombinedMappedTypeOptionality(type: MappedType): number {
            const optionality = getMappedTypeOptionality(type);
            const modifiersType = getModifiersTypeFromMappedType(type);
            return optionality || (isGenericMappedType(modifiersType) ? getMappedTypeOptionality(modifiersType) : 0);
        }

        function isPartialMappedType(type: Type) {
            return !!(getObjectFlags(type) & ObjectFlags.Mapped && getMappedTypeModifiers(<MappedType>type) & MappedTypeModifiers.IncludeOptional);
        }

        function isGenericMappedType(type: Type): type is MappedType {
            return getObjectFlags(type) & ObjectFlags.Mapped && isGenericIndexType(getConstraintTypeFromMappedType(<MappedType>type));
        }

        function resolveStructuredTypeMembers(type: StructuredType): ResolvedType {
            if (!(<ResolvedType>type).members) {
                if (type.flags & TypeFlags.Object) {
                    if ((<ObjectType>type).objectFlags & ObjectFlags.Reference) {
                        resolveTypeReferenceMembers(<TypeReference>type);
                    }
                    else if ((<ObjectType>type).objectFlags & ObjectFlags.ClassOrInterface) {
                        resolveClassOrInterfaceMembers(<InterfaceType>type);
                    }
                    else if ((<ReverseMappedType>type).objectFlags & ObjectFlags.ReverseMapped) {
                        resolveReverseMappedTypeMembers(type as ReverseMappedType);
                    }
                    else if ((<ObjectType>type).objectFlags & ObjectFlags.Anonymous) {
                        resolveAnonymousTypeMembers(<AnonymousType>type);
                    }
                    else if ((<MappedType>type).objectFlags & ObjectFlags.Mapped) {
                        resolveMappedTypeMembers(<MappedType>type);
                    }
                }
                else if (type.flags & TypeFlags.Union) {
                    resolveUnionTypeMembers(<UnionType>type);
                }
                else if (type.flags & TypeFlags.Intersection) {
                    resolveIntersectionTypeMembers(<IntersectionType>type);
                }
            }
            return <ResolvedType>type;
        }

        /** Return properties of an object type or an empty array for other types */
        function getPropertiesOfObjectType(type: Type): Symbol[] {
            if (type.flags & TypeFlags.Object) {
                return resolveStructuredTypeMembers(<ObjectType>type).properties;
            }
            return emptyArray;
        }

        /** If the given type is an object type and that type has a property by the given name,
         * return the symbol for that property. Otherwise return undefined.
         */
        function getPropertyOfObjectType(type: Type, name: __String): Symbol {
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                const symbol = resolved.members.get(name);
                if (symbol && symbolIsValue(symbol)) {
                    return symbol;
                }
            }
        }

        function getPropertiesOfUnionOrIntersectionType(type: UnionOrIntersectionType): Symbol[] {
            if (!type.resolvedProperties) {
                const members = createSymbolTable();
                for (const current of type.types) {
                    for (const prop of getPropertiesOfType(current)) {
                        if (!members.has(prop.escapedName)) {
                            const combinedProp = getPropertyOfUnionOrIntersectionType(type, prop.escapedName);
                            if (combinedProp) {
                                members.set(prop.escapedName, combinedProp);
                            }
                        }
                    }
                    // The properties of a union type are those that are present in all constituent types, so
                    // we only need to check the properties of the first type
                    if (type.flags & TypeFlags.Union) {
                        break;
                    }
                }
                type.resolvedProperties = getNamedMembers(members);
            }
            return type.resolvedProperties;
        }

        function getPropertiesOfType(type: Type): Symbol[] {
            type = getApparentType(type);
            return type.flags & TypeFlags.UnionOrIntersection ?
                getPropertiesOfUnionOrIntersectionType(<UnionType>type) :
                getPropertiesOfObjectType(type);
        }

        function getAllPossiblePropertiesOfTypes(types: Type[]): Symbol[] {
            const unionType = getUnionType(types);
            if (!(unionType.flags & TypeFlags.Union)) {
                return getAugmentedPropertiesOfType(unionType);
            }

            const props = createSymbolTable();
            for (const memberType of types) {
                for (const { escapedName } of getAugmentedPropertiesOfType(memberType)) {
                    if (!props.has(escapedName)) {
                        const prop = createUnionOrIntersectionProperty(unionType as UnionType, escapedName);
                        // May be undefined if the property is private
                        if (prop) props.set(escapedName, prop);
                    }
                }
            }
            return arrayFrom(props.values());
        }

        function getConstraintOfType(type: InstantiableType | UnionOrIntersectionType): Type {
            return type.flags & TypeFlags.TypeParameter ? getConstraintOfTypeParameter(<TypeParameter>type) :
                type.flags & TypeFlags.IndexedAccess ? getConstraintOfIndexedAccess(<IndexedAccessType>type) :
                type.flags & TypeFlags.Conditional ? getConstraintOfConditionalType(<ConditionalType>type) :
                getBaseConstraintOfType(type);
        }

        function getConstraintOfTypeParameter(typeParameter: TypeParameter): Type {
            return hasNonCircularBaseConstraint(typeParameter) ? getConstraintFromTypeParameter(typeParameter) : undefined;
        }

        function getConstraintOfIndexedAccess(type: IndexedAccessType) {
            const transformed = getSimplifiedIndexedAccessType(type);
            if (transformed) {
                return transformed;
            }
            const baseObjectType = getBaseConstraintOfType(type.objectType);
            const baseIndexType = getBaseConstraintOfType(type.indexType);
            if (baseIndexType === stringType && !getIndexInfoOfType(baseObjectType || type.objectType, IndexKind.String)) {
                // getIndexedAccessType returns `any` for X[string] where X doesn't have an index signature.
                // to avoid this, return `undefined`.
                return undefined;
            }
            return baseObjectType || baseIndexType ? getIndexedAccessType(baseObjectType || type.objectType, baseIndexType || type.indexType) : undefined;
        }

        function getDefaultConstraintOfConditionalType(type: ConditionalType) {
            return getUnionType([getTrueTypeFromConditionalType(type), getFalseTypeFromConditionalType(type)]);
        }

        function getConstraintOfDistributiveConditionalType(type: ConditionalType): Type {
            // Check if we have a conditional type of the form 'T extends U ? X : Y', where T is a constrained
            // type parameter. If so, create an instantiation of the conditional type where T is replaced
            // with its constraint. We do this because if the constraint is a union type it will be distributed
            // over the conditional type and possibly reduced. For example, 'T extends undefined ? never : T'
            // removes 'undefined' from T.
            if (type.root.isDistributive) {
                const constraint = getConstraintOfType(type.checkType);
                if (constraint) {
                    const mapper = createTypeMapper([<TypeParameter>type.root.checkType], [constraint]);
                    return getConditionalTypeInstantiation(type, combineTypeMappers(mapper, type.mapper));
                }
            }
            return undefined;
        }

        function getConstraintOfConditionalType(type: ConditionalType) {
            return getConstraintOfDistributiveConditionalType(type) || getDefaultConstraintOfConditionalType(type);
        }

        function getBaseConstraintOfInstantiableNonPrimitiveUnionOrIntersection(type: Type) {
            if (type.flags & (TypeFlags.InstantiableNonPrimitive | TypeFlags.UnionOrIntersection)) {
                const constraint = getResolvedBaseConstraint(<InstantiableType | UnionOrIntersectionType>type);
                if (constraint !== noConstraintType && constraint !== circularConstraintType) {
                    return constraint;
                }
            }
        }

        function getBaseConstraintOfType(type: Type): Type {
            const constraint = getBaseConstraintOfInstantiableNonPrimitiveUnionOrIntersection(type);
            if (!constraint && type.flags & TypeFlags.Index) {
                return stringType;
            }
            return constraint;
        }

        /**
         * This is similar to `getBaseConstraintOfType` except it returns the input type if there's no base constraint, instead of `undefined`
         * It also doesn't map indexes to `string`, as where this is used this would be unneeded (and likely undesirable)
         */
        function getBaseConstraintOrType(type: Type) {
            return getBaseConstraintOfType(type) || type;
        }

        function hasNonCircularBaseConstraint(type: InstantiableType): boolean {
            return getResolvedBaseConstraint(type) !== circularConstraintType;
        }

        /**
         * Return the resolved base constraint of a type variable. The noConstraintType singleton is returned if the
         * type variable has no constraint, and the circularConstraintType singleton is returned if the constraint
         * circularly references the type variable.
         */
        function getResolvedBaseConstraint(type: InstantiableType | UnionOrIntersectionType): Type {
            let circular: boolean;
            if (!type.resolvedBaseConstraint) {
                const constraint = getBaseConstraint(type);
                type.resolvedBaseConstraint = circular ? circularConstraintType : getTypeWithThisArgument(constraint || noConstraintType, type);
            }
            return type.resolvedBaseConstraint;

            function getBaseConstraint(t: Type): Type {
                if (!pushTypeResolution(t, TypeSystemPropertyName.ResolvedBaseConstraint)) {
                    circular = true;
                    return undefined;
                }
                const result = computeBaseConstraint(t);
                if (!popTypeResolution()) {
                    circular = true;
                    return undefined;
                }
                return result;
            }

            function computeBaseConstraint(t: Type): Type {
                if (t.flags & TypeFlags.TypeParameter) {
                    const constraint = getConstraintFromTypeParameter(<TypeParameter>t);
                    return (t as TypeParameter).isThisType || !constraint ?
                        constraint :
                        getBaseConstraint(constraint);
                }
                if (t.flags & TypeFlags.UnionOrIntersection) {
                    const types = (<UnionOrIntersectionType>t).types;
                    const baseTypes: Type[] = [];
                    for (const type of types) {
                        const baseType = getBaseConstraint(type);
                        if (baseType) {
                            baseTypes.push(baseType);
                        }
                    }
                    return t.flags & TypeFlags.Union && baseTypes.length === types.length ? getUnionType(baseTypes) :
                        t.flags & TypeFlags.Intersection && baseTypes.length ? getIntersectionType(baseTypes) :
                            undefined;
                }
                if (t.flags & TypeFlags.Index) {
                    return stringType;
                }
                if (t.flags & TypeFlags.IndexedAccess) {
                    const transformed = getSimplifiedIndexedAccessType(<IndexedAccessType>t);
                    if (transformed) {
                        return getBaseConstraint(transformed);
                    }
                    const baseObjectType = getBaseConstraint((<IndexedAccessType>t).objectType);
                    const baseIndexType = getBaseConstraint((<IndexedAccessType>t).indexType);
                    const baseIndexedAccess = baseObjectType && baseIndexType ? getIndexedAccessType(baseObjectType, baseIndexType) : undefined;
                    return baseIndexedAccess && baseIndexedAccess !== unknownType ? getBaseConstraint(baseIndexedAccess) : undefined;
                }
                if (t.flags & TypeFlags.Conditional) {
                    const constraint = getConstraintOfConditionalType(<ConditionalType>t);
                    return constraint && getBaseConstraint(constraint);
                }
                if (t.flags & TypeFlags.Substitution) {
                    return getBaseConstraint((<SubstitutionType>t).substitute);
                }
                if (isGenericMappedType(t)) {
                    return emptyObjectType;
                }
                return t;
            }
        }

        function getApparentTypeOfIntersectionType(type: IntersectionType) {
            return type.resolvedApparentType || (type.resolvedApparentType = getTypeWithThisArgument(type, type, /*apparentType*/ true));
        }

        function getResolvedTypeParameterDefault(typeParameter: TypeParameter): Type | undefined {
            if (!typeParameter.default) {
                if (typeParameter.target) {
                    const targetDefault = getResolvedTypeParameterDefault(typeParameter.target);
                    typeParameter.default = targetDefault ? instantiateType(targetDefault, typeParameter.mapper) : noConstraintType;
                }
                else {
                    // To block recursion, set the initial value to the resolvingDefaultType.
                    typeParameter.default = resolvingDefaultType;
                    const defaultDeclaration = typeParameter.symbol && forEach(typeParameter.symbol.declarations, decl => isTypeParameterDeclaration(decl) && decl.default);
                    const defaultType = defaultDeclaration ? getTypeFromTypeNode(defaultDeclaration) : noConstraintType;
                    if (typeParameter.default === resolvingDefaultType) {
                        // If we have not been called recursively, set the correct default type.
                        typeParameter.default = defaultType;
                    }
                }
            }
            else if (typeParameter.default === resolvingDefaultType) {
                // If we are called recursively for this type parameter, mark the default as circular.
                typeParameter.default = circularConstraintType;
            }
            return typeParameter.default;
        }

        /**
         * Gets the default type for a type parameter.
         *
         * If the type parameter is the result of an instantiation, this gets the instantiated
         * default type of its target. If the type parameter has no default type or the default is
         * circular, `undefined` is returned.
         */
        function getDefaultFromTypeParameter(typeParameter: TypeParameter): Type | undefined {
            const defaultType = getResolvedTypeParameterDefault(typeParameter);
            return defaultType !== noConstraintType && defaultType !== circularConstraintType ? defaultType : undefined;
        }

        function hasNonCircularTypeParameterDefault(typeParameter: TypeParameter) {
            return getResolvedTypeParameterDefault(typeParameter) !== circularConstraintType;
        }

        /**
         * Indicates whether the declaration of a typeParameter has a default type.
         */
        function hasTypeParameterDefault(typeParameter: TypeParameter): boolean {
            return !!(typeParameter.symbol && forEach(typeParameter.symbol.declarations, decl => isTypeParameterDeclaration(decl) && decl.default));
        }

        /**
         * For a type parameter, return the base constraint of the type parameter. For the string, number,
         * boolean, and symbol primitive types, return the corresponding object types. Otherwise return the
         * type itself. Note that the apparent type of a union type is the union type itself.
         */
        function getApparentType(type: Type): Type {
            const t = type.flags & TypeFlags.Instantiable ? getBaseConstraintOfType(type) || emptyObjectType : type;
            return t.flags & TypeFlags.Intersection ? getApparentTypeOfIntersectionType(<IntersectionType>t) :
                t.flags & TypeFlags.StringLike ? globalStringType :
                t.flags & TypeFlags.NumberLike ? globalNumberType :
                t.flags & TypeFlags.BooleanLike ? globalBooleanType :
                t.flags & TypeFlags.ESSymbolLike ? getGlobalESSymbolType(/*reportErrors*/ languageVersion >= ScriptTarget.ES2015) :
                t.flags & TypeFlags.NonPrimitive ? emptyObjectType :
                t;
        }

        function createUnionOrIntersectionProperty(containingType: UnionOrIntersectionType, name: __String): Symbol | undefined {
            let props: Symbol[];
            const isUnion = containingType.flags & TypeFlags.Union;
            const excludeModifiers = isUnion ? ModifierFlags.NonPublicAccessibilityModifier : 0;
            // Flags we want to propagate to the result if they exist in all source symbols
            let commonFlags = isUnion ? SymbolFlags.None : SymbolFlags.Optional;
            let syntheticFlag = CheckFlags.SyntheticMethod;
            let checkFlags = 0;
            for (const current of containingType.types) {
                const type = getApparentType(current);
                if (type !== unknownType) {
                    const prop = getPropertyOfType(type, name);
                    const modifiers = prop ? getDeclarationModifierFlagsFromSymbol(prop) : 0;
                    if (prop && !(modifiers & excludeModifiers)) {
                        commonFlags &= prop.flags;
                        props = appendIfUnique(props, prop);
                        checkFlags |= (isReadonlySymbol(prop) ? CheckFlags.Readonly : 0) |
                            (!(modifiers & ModifierFlags.NonPublicAccessibilityModifier) ? CheckFlags.ContainsPublic : 0) |
                            (modifiers & ModifierFlags.Protected ? CheckFlags.ContainsProtected : 0) |
                            (modifiers & ModifierFlags.Private ? CheckFlags.ContainsPrivate : 0) |
                            (modifiers & ModifierFlags.Static ? CheckFlags.ContainsStatic : 0);
                        if (!isMethodLike(prop)) {
                            syntheticFlag = CheckFlags.SyntheticProperty;
                        }
                    }
                    else if (isUnion) {
                        checkFlags |= CheckFlags.Partial;
                    }
                }
            }
            if (!props) {
                return undefined;
            }
            if (props.length === 1 && !(checkFlags & CheckFlags.Partial)) {
                return props[0];
            }
            const propTypes: Type[] = [];
            const declarations: Declaration[] = [];
            let commonType: Type;
            for (const prop of props) {
                if (prop.declarations) {
                    addRange(declarations, prop.declarations);
                }
                const type = getTypeOfSymbol(prop);
                if (!commonType) {
                    commonType = type;
                }
                else if (type !== commonType) {
                    checkFlags |= CheckFlags.HasNonUniformType;
                }
                propTypes.push(type);
            }
            const result = createSymbol(SymbolFlags.Property | commonFlags, name, syntheticFlag | checkFlags);
            result.containingType = containingType;
            result.declarations = declarations;
            result.type = isUnion ? getUnionType(propTypes) : getIntersectionType(propTypes);
            return result;
        }

        // Return the symbol for a given property in a union or intersection type, or undefined if the property
        // does not exist in any constituent type. Note that the returned property may only be present in some
        // constituents, in which case the isPartial flag is set when the containing type is union type. We need
        // these partial properties when identifying discriminant properties, but otherwise they are filtered out
        // and do not appear to be present in the union type.
        function getUnionOrIntersectionProperty(type: UnionOrIntersectionType, name: __String): Symbol {
            const properties = type.propertyCache || (type.propertyCache = createSymbolTable());
            let property = properties.get(name);
            if (!property) {
                property = createUnionOrIntersectionProperty(type, name);
                if (property) {
                    properties.set(name, property);
                }
            }
            return property;
        }

        function getPropertyOfUnionOrIntersectionType(type: UnionOrIntersectionType, name: __String): Symbol {
            const property = getUnionOrIntersectionProperty(type, name);
            // We need to filter out partial properties in union types
            return property && !(getCheckFlags(property) & CheckFlags.Partial) ? property : undefined;
        }

        /**
         * Return the symbol for the property with the given name in the given type. Creates synthetic union properties when
         * necessary, maps primitive types and type parameters are to their apparent types, and augments with properties from
         * Object and Function as appropriate.
         *
         * @param type a type to look up property from
         * @param name a name of property to look up in a given type
         */
        function getPropertyOfType(type: Type, name: __String): Symbol | undefined {
            type = getApparentType(type);
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                const symbol = resolved.members.get(name);
                if (symbol && symbolIsValue(symbol)) {
                    return symbol;
                }
                if (resolved === anyFunctionType || resolved.callSignatures.length || resolved.constructSignatures.length) {
                    const symbol = getPropertyOfObjectType(globalFunctionType, name);
                    if (symbol) {
                        return symbol;
                    }
                }
                return getPropertyOfObjectType(globalObjectType, name);
            }
            if (type.flags & TypeFlags.UnionOrIntersection) {
                return getPropertyOfUnionOrIntersectionType(<UnionOrIntersectionType>type, name);
            }
            return undefined;
        }

        function getSignaturesOfStructuredType(type: Type, kind: SignatureKind): Signature[] {
            if (type.flags & TypeFlags.StructuredType) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                return kind === SignatureKind.Call ? resolved.callSignatures : resolved.constructSignatures;
            }
            return emptyArray;
        }

        /**
         * Return the signatures of the given kind in the given type. Creates synthetic union signatures when necessary and
         * maps primitive types and type parameters are to their apparent types.
         */
        function getSignaturesOfType(type: Type, kind: SignatureKind): Signature[] {
            return getSignaturesOfStructuredType(getApparentType(type), kind);
        }

        function getIndexInfoOfStructuredType(type: Type, kind: IndexKind): IndexInfo | undefined {
            if (type.flags & TypeFlags.StructuredType) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                return kind === IndexKind.String ? resolved.stringIndexInfo : resolved.numberIndexInfo;
            }
        }

        function getIndexTypeOfStructuredType(type: Type, kind: IndexKind): Type | undefined {
            const info = getIndexInfoOfStructuredType(type, kind);
            return info && info.type;
        }

        // Return the indexing info of the given kind in the given type. Creates synthetic union index types when necessary and
        // maps primitive types and type parameters are to their apparent types.
        function getIndexInfoOfType(type: Type, kind: IndexKind): IndexInfo {
            return getIndexInfoOfStructuredType(getApparentType(type), kind);
        }

        // Return the index type of the given kind in the given type. Creates synthetic union index types when necessary and
        // maps primitive types and type parameters are to their apparent types.
        function getIndexTypeOfType(type: Type, kind: IndexKind): Type {
            return getIndexTypeOfStructuredType(getApparentType(type), kind);
        }

        function getImplicitIndexTypeOfType(type: Type, kind: IndexKind): Type {
            if (isObjectTypeWithInferableIndex(type)) {
                const propTypes: Type[] = [];
                for (const prop of getPropertiesOfType(type)) {
                    if (kind === IndexKind.String || isNumericLiteralName(prop.escapedName)) {
                        propTypes.push(getTypeOfSymbol(prop));
                    }
                }
                if (propTypes.length) {
                    return getUnionType(propTypes, UnionReduction.Subtype);
                }
            }
            return undefined;
        }

        // Return list of type parameters with duplicates removed (duplicate identifier errors are generated in the actual
        // type checking functions).
        function getTypeParametersFromDeclaration(declaration: DeclarationWithTypeParameters): TypeParameter[] {
            let result: TypeParameter[];
            forEach(getEffectiveTypeParameterDeclarations(declaration), node => {
                const tp = getDeclaredTypeOfTypeParameter(node.symbol);
                result = appendIfUnique(result, tp);
            });
            return result;
        }

        function symbolsToArray(symbols: SymbolTable): Symbol[] {
            const result: Symbol[] = [];
            symbols.forEach((symbol, id) => {
                if (!isReservedMemberName(id)) {
                    result.push(symbol);
                }
            });
            return result;
        }

        function isJSDocOptionalParameter(node: ParameterDeclaration) {
            if (isInJavaScriptFile(node)) {
                if (node.type && node.type.kind === SyntaxKind.JSDocOptionalType) {
                    return true;
                }
                const paramTags = getJSDocParameterTags(node);
                if (paramTags) {
                    for (const paramTag of paramTags) {
                        if (paramTag.isBracketed) {
                            return true;
                        }

                        if (paramTag.typeExpression) {
                            return paramTag.typeExpression.type.kind === SyntaxKind.JSDocOptionalType;
                        }
                    }
                }
            }
        }

        function tryFindAmbientModule(moduleName: string, withAugmentations: boolean) {
            if (isExternalModuleNameRelative(moduleName)) {
                return undefined;
            }
            const symbol = getSymbol(globals, '"' + moduleName + '"' as __String, SymbolFlags.ValueModule);
            // merged symbol is module declaration symbol combined with all augmentations
            return symbol && withAugmentations ? getMergedSymbol(symbol) : symbol;
        }

        function isOptionalParameter(node: ParameterDeclaration) {
            if (hasQuestionToken(node) || isJSDocOptionalParameter(node)) {
                return true;
            }

            if (node.initializer) {
                const signature = getSignatureFromDeclaration(node.parent);
                const parameterIndex = node.parent.parameters.indexOf(node);
                Debug.assert(parameterIndex >= 0);
                return parameterIndex >= signature.minArgumentCount;
            }
            const iife = getImmediatelyInvokedFunctionExpression(node.parent);
            if (iife) {
                return !node.type &&
                    !node.dotDotDotToken &&
                    node.parent.parameters.indexOf(node) >= iife.arguments.length;
            }

            return false;
        }

        function createTypePredicateFromTypePredicateNode(node: TypePredicateNode): IdentifierTypePredicate | ThisTypePredicate {
            const { parameterName } = node;
            const type = getTypeFromTypeNode(node.type);
            if (parameterName.kind === SyntaxKind.Identifier) {
                return createIdentifierTypePredicate(
                    parameterName && parameterName.escapedText as string, // TODO: GH#18217
                    parameterName && getTypePredicateParameterIndex(node.parent.parameters, parameterName),
                    type);
            }
            else {
                return createThisTypePredicate(type);
            }
        }

        function createIdentifierTypePredicate(parameterName: string | undefined, parameterIndex: number | undefined, type: Type): IdentifierTypePredicate {
            return { kind: TypePredicateKind.Identifier, parameterName, parameterIndex, type };
        }

        function createThisTypePredicate(type: Type): ThisTypePredicate {
            return { kind: TypePredicateKind.This, type };
        }

        /**
         * Gets the minimum number of type arguments needed to satisfy all non-optional type
         * parameters.
         */
        function getMinTypeArgumentCount(typeParameters: TypeParameter[] | undefined): number {
            let minTypeArgumentCount = 0;
            if (typeParameters) {
                for (let i = 0; i < typeParameters.length; i++) {
                    if (!hasTypeParameterDefault(typeParameters[i])) {
                        minTypeArgumentCount = i + 1;
                    }
                }
            }
            return minTypeArgumentCount;
        }

        /**
         * Fill in default types for unsupplied type arguments. If `typeArguments` is undefined
         * when a default type is supplied, a new array will be created and returned.
         *
         * @param typeArguments The supplied type arguments.
         * @param typeParameters The requested type parameters.
         * @param minTypeArgumentCount The minimum number of required type arguments.
         */
        function fillMissingTypeArguments(typeArguments: Type[] | undefined, typeParameters: TypeParameter[] | undefined, minTypeArgumentCount: number, isJavaScriptImplicitAny: boolean) {
            const numTypeParameters = length(typeParameters);
            if (numTypeParameters) {
                const numTypeArguments = length(typeArguments);
                if (isJavaScriptImplicitAny || (numTypeArguments >= minTypeArgumentCount && numTypeArguments <= numTypeParameters)) {
                    if (!typeArguments) {
                        typeArguments = [];
                    }

                    // Map an unsatisfied type parameter with a default type.
                    // If a type parameter does not have a default type, or if the default type
                    // is a forward reference, the empty object type is used.
                    for (let i = numTypeArguments; i < numTypeParameters; i++) {
                        typeArguments[i] = getDefaultTypeArgumentType(isJavaScriptImplicitAny);
                    }
                    for (let i = numTypeArguments; i < numTypeParameters; i++) {
                        const mapper = createTypeMapper(typeParameters, typeArguments);
                        let defaultType = getDefaultFromTypeParameter(typeParameters[i]);
                        if (defaultType && isTypeIdenticalTo(defaultType, emptyObjectType) && isJavaScriptImplicitAny) {
                            defaultType = anyType;
                        }
                        typeArguments[i] = defaultType ? instantiateType(defaultType, mapper) : getDefaultTypeArgumentType(isJavaScriptImplicitAny);
                    }
                    typeArguments.length = typeParameters.length;
                }
            }
            return typeArguments;
        }

        function getSignatureFromDeclaration(declaration: SignatureDeclaration): Signature {
            const links = getNodeLinks(declaration);
            if (!links.resolvedSignature) {
                const parameters: Symbol[] = [];
                let hasLiteralTypes = false;
                let minArgumentCount = 0;
                let thisParameter: Symbol;
                let hasThisParameter: boolean;
                const iife = getImmediatelyInvokedFunctionExpression(declaration);
                const isJSConstructSignature = isJSDocConstructSignature(declaration);
                const isUntypedSignatureInJSFile = !iife && !isJSConstructSignature && isInJavaScriptFile(declaration) && !hasJSDocParameterTags(declaration);

                // If this is a JSDoc construct signature, then skip the first parameter in the
                // parameter list.  The first parameter represents the return type of the construct
                // signature.
                for (let i = isJSConstructSignature ? 1 : 0; i < declaration.parameters.length; i++) {
                    const param = declaration.parameters[i];

                    let paramSymbol = param.symbol;
                    // Include parameter symbol instead of property symbol in the signature
                    if (paramSymbol && !!(paramSymbol.flags & SymbolFlags.Property) && !isBindingPattern(param.name)) {
                        const resolvedSymbol = resolveName(param, paramSymbol.escapedName, SymbolFlags.Value, undefined, undefined, /*isUse*/ false);
                        paramSymbol = resolvedSymbol;
                    }
                    if (i === 0 && paramSymbol.escapedName === "this") {
                        hasThisParameter = true;
                        thisParameter = param.symbol;
                    }
                    else {
                        parameters.push(paramSymbol);
                    }

                    if (param.type && param.type.kind === SyntaxKind.LiteralType) {
                        hasLiteralTypes = true;
                    }

                    // Record a new minimum argument count if this is not an optional parameter
                    const isOptionalParameter = param.initializer || param.questionToken || param.dotDotDotToken ||
                        iife && parameters.length > iife.arguments.length && !param.type ||
                        isJSDocOptionalParameter(param) ||
                        isUntypedSignatureInJSFile;
                    if (!isOptionalParameter) {
                        minArgumentCount = parameters.length;
                    }
                }

                // If only one accessor includes a this-type annotation, the other behaves as if it had the same type annotation
                if ((declaration.kind === SyntaxKind.GetAccessor || declaration.kind === SyntaxKind.SetAccessor) &&
                    !hasNonBindableDynamicName(declaration) &&
                    (!hasThisParameter || !thisParameter)) {
                    const otherKind = declaration.kind === SyntaxKind.GetAccessor ? SyntaxKind.SetAccessor : SyntaxKind.GetAccessor;
                    const other = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration), otherKind);
                    if (other) {
                        thisParameter = getAnnotatedAccessorThisParameter(other);
                    }
                }

                const classType = declaration.kind === SyntaxKind.Constructor ?
                    getDeclaredTypeOfClassOrInterface(getMergedSymbol((<ClassDeclaration>declaration.parent).symbol))
                    : undefined;
                const typeParameters = classType ? classType.localTypeParameters : getTypeParametersFromDeclaration(declaration);
                const returnType = getSignatureReturnTypeFromDeclaration(declaration, isJSConstructSignature, classType);
                const hasRestLikeParameter = hasRestParameter(declaration) || isInJavaScriptFile(declaration) && maybeAddJsSyntheticRestParameter(declaration, parameters);
                links.resolvedSignature = createSignature(declaration, typeParameters, thisParameter, parameters, returnType, /*resolvedTypePredicate*/ undefined, minArgumentCount, hasRestLikeParameter, hasLiteralTypes);
            }
            return links.resolvedSignature;
        }

        function maybeAddJsSyntheticRestParameter(declaration: SignatureDeclaration, parameters: Symbol[]): boolean {
            // JS functions get a free rest parameter if:
            // a) The last parameter has `...` preceding its type
            // b) It references `arguments` somewhere
            const lastParam = lastOrUndefined(declaration.parameters);
            const lastParamTags = lastParam && getJSDocParameterTags(lastParam);
            const lastParamVariadicType = firstDefined(lastParamTags, p =>
                p.typeExpression && isJSDocVariadicType(p.typeExpression.type) ? p.typeExpression.type : undefined);
            if (!lastParamVariadicType && !containsArgumentsReference(declaration)) {
                return false;
            }

            const syntheticArgsSymbol = createSymbol(SymbolFlags.Variable, "args" as __String);
            syntheticArgsSymbol.type = lastParamVariadicType ? createArrayType(getTypeFromTypeNode(lastParamVariadicType.type)) : anyArrayType;
            syntheticArgsSymbol.isRestParameter = true;
            if (lastParamVariadicType) {
                // Replace the last parameter with a rest parameter.
                parameters.pop();
            }
            parameters.push(syntheticArgsSymbol);
            return true;
        }

        function getSignatureReturnTypeFromDeclaration(declaration: SignatureDeclaration, isJSConstructSignature: boolean, classType: Type) {
            if (isJSConstructSignature) {
                return getTypeFromTypeNode(declaration.parameters[0].type);
            }
            else if (classType) {
                return classType;
            }

            const typeNode = getEffectiveReturnTypeNode(declaration);
            if (typeNode) {
                return getTypeFromTypeNode(typeNode);
            }

            // TypeScript 1.0 spec (April 2014):
            // If only one accessor includes a type annotation, the other behaves as if it had the same type annotation.
            if (declaration.kind === SyntaxKind.GetAccessor && !hasNonBindableDynamicName(declaration)) {
                const setter = getDeclarationOfKind<AccessorDeclaration>(getSymbolOfNode(declaration), SyntaxKind.SetAccessor);
                return getAnnotatedAccessorType(setter);
            }

            if (nodeIsMissing((<FunctionLikeDeclaration>declaration).body)) {
                return anyType;
            }
        }

        function containsArgumentsReference(declaration: SignatureDeclaration): boolean {
            const links = getNodeLinks(declaration);
            if (links.containsArgumentsReference === undefined) {
                if (links.flags & NodeCheckFlags.CaptureArguments) {
                    links.containsArgumentsReference = true;
                }
                else {
                    links.containsArgumentsReference = traverse((declaration as FunctionLikeDeclaration).body);
                }
            }
            return links.containsArgumentsReference;

            function traverse(node: Node): boolean {
                if (!node) return false;
                switch (node.kind) {
                    case SyntaxKind.Identifier:
                        return (<Identifier>node).escapedText === "arguments" && isExpressionNode(node);

                    case SyntaxKind.PropertyDeclaration:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                        return (<NamedDeclaration>node).name.kind === SyntaxKind.ComputedPropertyName
                            && traverse((<NamedDeclaration>node).name);

                    default:
                        return !nodeStartsNewLexicalEnvironment(node) && !isPartOfTypeNode(node) && forEachChild(node, traverse);
                }
            }
        }

        function getSignaturesOfSymbol(symbol: Symbol): Signature[] {
            if (!symbol) return emptyArray;
            const result: Signature[] = [];
            for (let i = 0; i < symbol.declarations.length; i++) {
                const node = symbol.declarations[i];
                switch (node.kind) {
                    case SyntaxKind.FunctionType:
                    case SyntaxKind.ConstructorType:
                    case SyntaxKind.FunctionDeclaration:
                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.MethodSignature:
                    case SyntaxKind.Constructor:
                    case SyntaxKind.CallSignature:
                    case SyntaxKind.ConstructSignature:
                    case SyntaxKind.IndexSignature:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                    case SyntaxKind.FunctionExpression:
                    case SyntaxKind.ArrowFunction:
                    case SyntaxKind.JSDocFunctionType:
                        // Don't include signature if node is the implementation of an overloaded function. A node is considered
                        // an implementation node if it has a body and the previous node is of the same kind and immediately
                        // precedes the implementation node (i.e. has the same parent and ends where the implementation starts).
                        if (i > 0 && (<FunctionLikeDeclaration>node).body) {
                            const previous = symbol.declarations[i - 1];
                            if (node.parent === previous.parent && node.kind === previous.kind && node.pos === previous.end) {
                                break;
                            }
                        }
                        result.push(getSignatureFromDeclaration(<SignatureDeclaration>node));
                }
            }
            return result;
        }

        function resolveExternalModuleTypeByLiteral(name: StringLiteral) {
            const moduleSym = resolveExternalModuleName(name, name);
            if (moduleSym) {
                const resolvedModuleSymbol = resolveExternalModuleSymbol(moduleSym);
                if (resolvedModuleSymbol) {
                    return getTypeOfSymbol(resolvedModuleSymbol);
                }
            }

            return anyType;
        }

        function getThisTypeOfSignature(signature: Signature): Type | undefined {
            if (signature.thisParameter) {
                return getTypeOfSymbol(signature.thisParameter);
            }
        }

        function signatureHasTypePredicate(signature: Signature): boolean {
            return getTypePredicateOfSignature(signature) !== undefined;
        }

        function getTypePredicateOfSignature(signature: Signature): TypePredicate | undefined {
            if (!signature.resolvedTypePredicate) {
                if (signature.target) {
                    const targetTypePredicate = getTypePredicateOfSignature(signature.target);
                    signature.resolvedTypePredicate = targetTypePredicate ? instantiateTypePredicate(targetTypePredicate, signature.mapper) : noTypePredicate;
                }
                else if (signature.unionSignatures) {
                    signature.resolvedTypePredicate = getUnionTypePredicate(signature.unionSignatures) || noTypePredicate;
                }
                else {
                    const declaration = signature.declaration;
                    signature.resolvedTypePredicate = declaration && declaration.type && declaration.type.kind === SyntaxKind.TypePredicate ?
                        createTypePredicateFromTypePredicateNode(declaration.type as TypePredicateNode) :
                        noTypePredicate;
                }
                Debug.assert(!!signature.resolvedTypePredicate);
            }
            return signature.resolvedTypePredicate === noTypePredicate ? undefined : signature.resolvedTypePredicate;
        }

        function getReturnTypeOfSignature(signature: Signature): Type {
            if (!signature.resolvedReturnType) {
                if (!pushTypeResolution(signature, TypeSystemPropertyName.ResolvedReturnType)) {
                    return unknownType;
                }
                let type: Type;
                if (signature.target) {
                    type = instantiateType(getReturnTypeOfSignature(signature.target), signature.mapper);
                }
                else if (signature.unionSignatures) {
                    type = getUnionType(map(signature.unionSignatures, getReturnTypeOfSignature), UnionReduction.Subtype);
                }
                else {
                    type = getReturnTypeFromBody(<FunctionLikeDeclaration>signature.declaration);
                }
                if (!popTypeResolution()) {
                    type = anyType;
                    if (noImplicitAny) {
                        const declaration = <Declaration>signature.declaration;
                        const name = getNameOfDeclaration(declaration);
                        if (name) {
                            error(name, Diagnostics._0_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions, declarationNameToString(name));
                        }
                        else {
                            error(declaration, Diagnostics.Function_implicitly_has_return_type_any_because_it_does_not_have_a_return_type_annotation_and_is_referenced_directly_or_indirectly_in_one_of_its_return_expressions);
                        }
                    }
                }
                signature.resolvedReturnType = type;
            }
            return signature.resolvedReturnType;
        }

        function isResolvingReturnTypeOfSignature(signature: Signature) {
            return !signature.resolvedReturnType && findResolutionCycleStartIndex(signature, TypeSystemPropertyName.ResolvedReturnType) >= 0;
        }

        function getRestTypeOfSignature(signature: Signature): Type {
            if (signature.hasRestParameter) {
                const type = getTypeOfSymbol(lastOrUndefined(signature.parameters));
                if (getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).target === globalArrayType) {
                    return (<TypeReference>type).typeArguments[0];
                }
            }
            return anyType;
        }

        function getSignatureInstantiation(signature: Signature, typeArguments: Type[], isJavascript: boolean): Signature {
            typeArguments = fillMissingTypeArguments(typeArguments, signature.typeParameters, getMinTypeArgumentCount(signature.typeParameters), isJavascript);
            const instantiations = signature.instantiations || (signature.instantiations = createMap<Signature>());
            const id = getTypeListId(typeArguments);
            let instantiation = instantiations.get(id);
            if (!instantiation) {
                instantiations.set(id, instantiation = createSignatureInstantiation(signature, typeArguments));
            }
            return instantiation;
        }

        function createSignatureInstantiation(signature: Signature, typeArguments: Type[]): Signature {
            return instantiateSignature(signature, createSignatureTypeMapper(signature, typeArguments), /*eraseTypeParameters*/ true);
        }
        function createSignatureTypeMapper(signature: Signature, typeArguments: Type[]): TypeMapper {
            return createTypeMapper(signature.typeParameters, typeArguments);
        }

        function getErasedSignature(signature: Signature): Signature {
            return signature.typeParameters ?
                signature.erasedSignatureCache || (signature.erasedSignatureCache = createErasedSignature(signature)) :
                signature;
        }

        function createErasedSignature(signature: Signature) {
            // Create an instantiation of the signature where all type arguments are the any type.
            return instantiateSignature(signature, createTypeEraser(signature.typeParameters), /*eraseTypeParameters*/ true);
        }

        function getCanonicalSignature(signature: Signature): Signature {
            return signature.typeParameters ?
                signature.canonicalSignatureCache || (signature.canonicalSignatureCache = createCanonicalSignature(signature)) :
                signature;
        }

        function createCanonicalSignature(signature: Signature) {
            // Create an instantiation of the signature where each unconstrained type parameter is replaced with
            // its original. When a generic class or interface is instantiated, each generic method in the class or
            // interface is instantiated with a fresh set of cloned type parameters (which we need to handle scenarios
            // where different generations of the same type parameter are in scope). This leads to a lot of new type
            // identities, and potentially a lot of work comparing those identities, so here we create an instantiation
            // that uses the original type identities for all unconstrained type parameters.
            return getSignatureInstantiation(
                signature,
                map(signature.typeParameters, tp => tp.target && !getConstraintOfTypeParameter(tp.target) ? tp.target : tp),
                isInJavaScriptFile(signature.declaration));
        }

        function getBaseSignature(signature: Signature) {
            const typeParameters = signature.typeParameters;
            if (typeParameters) {
                const typeEraser = createTypeEraser(typeParameters);
                const baseConstraints = map(typeParameters, tp => instantiateType(getBaseConstraintOfType(tp), typeEraser) || emptyObjectType);
                return instantiateSignature(signature, createTypeMapper(typeParameters, baseConstraints), /*eraseTypeParameters*/ true);
            }
            return signature;
        }

        function getOrCreateTypeFromSignature(signature: Signature): ObjectType {
            // There are two ways to declare a construct signature, one is by declaring a class constructor
            // using the constructor keyword, and the other is declaring a bare construct signature in an
            // object type literal or interface (using the new keyword). Each way of declaring a constructor
            // will result in a different declaration kind.
            if (!signature.isolatedSignatureType) {
                const isConstructor = signature.declaration.kind === SyntaxKind.Constructor || signature.declaration.kind === SyntaxKind.ConstructSignature;
                const type = <ResolvedType>createObjectType(ObjectFlags.Anonymous);
                type.members = emptySymbols;
                type.properties = emptyArray;
                type.callSignatures = !isConstructor ? [signature] : emptyArray;
                type.constructSignatures = isConstructor ? [signature] : emptyArray;
                signature.isolatedSignatureType = type;
            }

            return signature.isolatedSignatureType;
        }

        function getIndexSymbol(symbol: Symbol): Symbol {
            return symbol.members.get(InternalSymbolName.Index);
        }

        function getIndexDeclarationOfSymbol(symbol: Symbol, kind: IndexKind): SignatureDeclaration {
            const syntaxKind = kind === IndexKind.Number ? SyntaxKind.NumberKeyword : SyntaxKind.StringKeyword;
            const indexSymbol = getIndexSymbol(symbol);
            if (indexSymbol) {
                for (const decl of indexSymbol.declarations) {
                    const node = <SignatureDeclaration>decl;
                    if (node.parameters.length === 1) {
                        const parameter = node.parameters[0];
                        if (parameter && parameter.type && parameter.type.kind === syntaxKind) {
                            return node;
                        }
                    }
                }
            }

            return undefined;
        }

        function createIndexInfo(type: Type, isReadonly: boolean, declaration?: SignatureDeclaration): IndexInfo {
            return { type, isReadonly, declaration };
        }

        function getIndexInfoOfSymbol(symbol: Symbol, kind: IndexKind): IndexInfo {
            const declaration = getIndexDeclarationOfSymbol(symbol, kind);
            if (declaration) {
                return createIndexInfo(declaration.type ? getTypeFromTypeNode(declaration.type) : anyType,
                    hasModifier(declaration, ModifierFlags.Readonly), declaration);
            }
            return undefined;
        }

        function getConstraintDeclaration(type: TypeParameter) {
            return type.symbol && getDeclarationOfKind<TypeParameterDeclaration>(type.symbol, SyntaxKind.TypeParameter).constraint;
        }

        function getInferredTypeParameterConstraint(typeParameter: TypeParameter) {
            let inferences: Type[];
            if (typeParameter.symbol) {
                for (const declaration of typeParameter.symbol.declarations) {
                    // When an 'infer T' declaration is immediately contained in a type reference node
                    // (such as 'Foo<infer T>'), T's constraint is inferred from the constraint of the
                    // corresponding type parameter in 'Foo'. When multiple 'infer T' declarations are
                    // present, we form an intersection of the inferred constraint types.
                    if (declaration.parent.kind === SyntaxKind.InferType && declaration.parent.parent.kind === SyntaxKind.TypeReference) {
                        const typeReference = <TypeReferenceNode>declaration.parent.parent;
                        const typeParameters = getTypeParametersForTypeReference(typeReference);
                        if (typeParameters) {
                            const index = typeReference.typeArguments.indexOf(<TypeNode>declaration.parent);
                            if (index < typeParameters.length) {
                                const declaredConstraint = getConstraintOfTypeParameter(typeParameters[index]);
                                if (declaredConstraint) {
                                    // Type parameter constraints can reference other type parameters so
                                    // constraints need to be instantiated. If instantiation produces the
                                    // type parameter itself, we discard that inference. For example, in
                                    //   type Foo<T extends string, U extends T> = [T, U];
                                    //   type Bar<T> = T extends Foo<infer X, infer X> ? Foo<X, X> : T;
                                    // the instantiated constraint for U is X, so we discard that inference.
                                    const mapper = createTypeMapper(typeParameters, getEffectiveTypeArguments(typeReference, typeParameters));
                                    const constraint = instantiateType(declaredConstraint, mapper);
                                    if (constraint !== typeParameter) {
                                        inferences = append(inferences, constraint);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return inferences && getIntersectionType(inferences);
        }

        function getConstraintFromTypeParameter(typeParameter: TypeParameter): Type {
            if (!typeParameter.constraint) {
                if (typeParameter.target) {
                    const targetConstraint = getConstraintOfTypeParameter(typeParameter.target);
                    typeParameter.constraint = targetConstraint ? instantiateType(targetConstraint, typeParameter.mapper) : noConstraintType;
                }
                else {
                    const constraintDeclaration = getConstraintDeclaration(typeParameter);
                    typeParameter.constraint = constraintDeclaration ? getTypeFromTypeNode(constraintDeclaration) :
                        getInferredTypeParameterConstraint(typeParameter) || noConstraintType;
                }
            }
            return typeParameter.constraint === noConstraintType ? undefined : typeParameter.constraint;
        }

        function getParentSymbolOfTypeParameter(typeParameter: TypeParameter): Symbol {
            return getSymbolOfNode(getDeclarationOfKind(typeParameter.symbol, SyntaxKind.TypeParameter).parent);
        }

        function getTypeListId(types: Type[]) {
            let result = "";
            if (types) {
                const length = types.length;
                let i = 0;
                while (i < length) {
                    const startId = types[i].id;
                    let count = 1;
                    while (i + count < length && types[i + count].id === startId + count) {
                        count++;
                    }
                    if (result.length) {
                        result += ",";
                    }
                    result += startId;
                    if (count > 1) {
                        result += ":" + count;
                    }
                    i += count;
                }
            }
            return result;
        }

        // This function is used to propagate certain flags when creating new object type references and union types.
        // It is only necessary to do so if a constituent type might be the undefined type, the null type, the type
        // of an object literal or the anyFunctionType. This is because there are operations in the type checker
        // that care about the presence of such types at arbitrary depth in a containing type.
        function getPropagatingFlagsOfTypes(types: Type[], excludeKinds: TypeFlags): TypeFlags {
            let result: TypeFlags = 0;
            for (const type of types) {
                if (!(type.flags & excludeKinds)) {
                    result |= type.flags;
                }
            }
            return result & TypeFlags.PropagatingFlags;
        }

        function createTypeReference(target: GenericType, typeArguments: Type[]): TypeReference {
            const id = getTypeListId(typeArguments);
            let type = target.instantiations.get(id);
            if (!type) {
                type = <TypeReference>createObjectType(ObjectFlags.Reference, target.symbol);
                target.instantiations.set(id, type);
                type.flags |= typeArguments ? getPropagatingFlagsOfTypes(typeArguments, /*excludeKinds*/ 0) : 0;
                type.target = target;
                type.typeArguments = typeArguments;
            }
            return type;
        }

        function cloneTypeReference(source: TypeReference): TypeReference {
            const type = <TypeReference>createType(source.flags);
            type.symbol = source.symbol;
            type.objectFlags = source.objectFlags;
            type.target = source.target;
            type.typeArguments = source.typeArguments;
            return type;
        }

        function getTypeReferenceArity(type: TypeReference): number {
            return length(type.target.typeParameters);
        }

        /**
         * Get type from type-reference that reference to class or interface
         */
        function getTypeFromClassOrInterfaceReference(node: TypeReferenceType, symbol: Symbol, typeArgs: Type[]): Type {
            const type = <InterfaceType>getDeclaredTypeOfSymbol(getMergedSymbol(symbol));
            const typeParameters = type.localTypeParameters;
            if (typeParameters) {
                const numTypeArguments = length(node.typeArguments);
                const minTypeArgumentCount = getMinTypeArgumentCount(typeParameters);
                const isJs = isInJavaScriptFile(node);
                const isJsImplicitAny = !noImplicitAny && isJs;
                if (!isJsImplicitAny && (numTypeArguments < minTypeArgumentCount || numTypeArguments > typeParameters.length)) {
                    const missingAugmentsTag = isJs && node.parent.kind !== SyntaxKind.JSDocAugmentsTag;
                    const diag = minTypeArgumentCount === typeParameters.length
                        ? missingAugmentsTag
                            ? Diagnostics.Expected_0_type_arguments_provide_these_with_an_extends_tag
                            : Diagnostics.Generic_type_0_requires_1_type_argument_s
                        : missingAugmentsTag
                            ? Diagnostics.Expected_0_1_type_arguments_provide_these_with_an_extends_tag
                            : Diagnostics.Generic_type_0_requires_between_1_and_2_type_arguments;
                    const typeStr = typeToString(type, /*enclosingDeclaration*/ undefined, TypeFormatFlags.WriteArrayAsGenericType);
                    error(node, diag, typeStr, minTypeArgumentCount, typeParameters.length);
                    if (!isJs) {
                        // TODO: Adopt same permissive behavior in TS as in JS to reduce follow-on editing experience failures (requires editing fillMissingTypeArguments)
                        return unknownType;
                    }
                }
                // In a type reference, the outer type parameters of the referenced class or interface are automatically
                // supplied as type arguments and the type reference only specifies arguments for the local type parameters
                // of the class or interface.
                const typeArguments = concatenate(type.outerTypeParameters, fillMissingTypeArguments(typeArgs, typeParameters, minTypeArgumentCount, isJs));
                return createTypeReference(<GenericType>type, typeArguments);
            }
            return checkNoTypeArguments(node, symbol) ? type : unknownType;
            }

        function getTypeAliasInstantiation(symbol: Symbol, typeArguments: Type[]): Type {
            const type = getDeclaredTypeOfSymbol(symbol);
            const links = getSymbolLinks(symbol);
            const typeParameters = links.typeParameters;
            const id = getTypeListId(typeArguments);
            let instantiation = links.instantiations.get(id);
            if (!instantiation) {
                links.instantiations.set(id, instantiation = instantiateType(type, createTypeMapper(typeParameters, fillMissingTypeArguments(typeArguments, typeParameters, getMinTypeArgumentCount(typeParameters), isInJavaScriptFile(symbol.valueDeclaration)))));
            }
            return instantiation;
        }

        /**
         * Get type from reference to type alias. When a type alias is generic, the declared type of the type alias may include
         * references to the type parameters of the alias. We replace those with the actual type arguments by instantiating the
         * declared type. Instantiations are cached using the type identities of the type arguments as the key.
         */
        function getTypeFromTypeAliasReference(node: TypeReferenceType, symbol: Symbol, typeArguments: Type[]): Type {
            const type = getDeclaredTypeOfSymbol(symbol);
            const typeParameters = getSymbolLinks(symbol).typeParameters;
            if (typeParameters) {
                const numTypeArguments = length(node.typeArguments);
                const minTypeArgumentCount = getMinTypeArgumentCount(typeParameters);
                if (numTypeArguments < minTypeArgumentCount || numTypeArguments > typeParameters.length) {
                    error(node,
                        minTypeArgumentCount === typeParameters.length
                            ? Diagnostics.Generic_type_0_requires_1_type_argument_s
                            : Diagnostics.Generic_type_0_requires_between_1_and_2_type_arguments,
                        symbolToString(symbol),
                        minTypeArgumentCount,
                        typeParameters.length);
                    return unknownType;
                }
                return getTypeAliasInstantiation(symbol, typeArguments);
            }
            return checkNoTypeArguments(node, symbol) ? type : unknownType;
            }

        function getTypeReferenceName(node: TypeReferenceType): EntityNameOrEntityNameExpression | undefined {
            switch (node.kind) {
                case SyntaxKind.TypeReference:
                    return node.typeName;
                case SyntaxKind.ExpressionWithTypeArguments:
                    // We only support expressions that are simple qualified names. For other
                    // expressions this produces undefined.
                    const expr = node.expression;
                    if (isEntityNameExpression(expr)) {
                        return expr;
                    }
                // fall through;
            }

            return undefined;
        }

        function resolveTypeReferenceName(typeReferenceName: EntityNameExpression | EntityName, meaning: SymbolFlags) {
            if (!typeReferenceName) {
                return unknownSymbol;
            }

            return resolveEntityName(typeReferenceName, meaning) || unknownSymbol;
        }

        function getTypeReferenceType(node: TypeReferenceType, symbol: Symbol) {
            const typeArguments = typeArgumentsFromTypeReferenceNode(node); // Do unconditionally so we mark type arguments as referenced.
            if (symbol === unknownSymbol) {
                return unknownType;
            }

            const type = getTypeReferenceTypeWorker(node, symbol, typeArguments);
            if (type) {
                return type;
            }

            // Get type from reference to named type that cannot be generic (enum or type parameter)
            const res = tryGetDeclaredTypeOfSymbol(symbol);
            if (res) {
                return checkNoTypeArguments(node, symbol) ?
                    res.flags & TypeFlags.TypeParameter ? getConstrainedTypeParameter(<TypeParameter>res, node) : res :
                    unknownType;
            }

            if (!(symbol.flags & SymbolFlags.Value && isJSDocTypeReference(node))) {
                return unknownType;
            }

            // A jsdoc TypeReference may have resolved to a value (as opposed to a type). If
            // the symbol is a constructor function, return the inferred class type; otherwise,
            // the type of this reference is just the type of the value we resolved to.
            const assignedType = getAssignedClassType(symbol);
            const valueType = getTypeOfSymbol(symbol);
            const referenceType = valueType.symbol && !isInferredClassType(valueType) && getTypeReferenceTypeWorker(node, valueType.symbol, typeArguments);
            if (referenceType || assignedType) {
                return referenceType && assignedType ? getIntersectionType([assignedType, referenceType]) : referenceType || assignedType;
            }

            // Resolve the type reference as a Type for the purpose of reporting errors.
            resolveTypeReferenceName(getTypeReferenceName(node), SymbolFlags.Type);
            return valueType;
        }

        function getTypeReferenceTypeWorker(node: TypeReferenceType, symbol: Symbol, typeArguments: Type[]): Type | undefined {
            if (symbol.flags & (SymbolFlags.Class | SymbolFlags.Interface)) {
                return getTypeFromClassOrInterfaceReference(node, symbol, typeArguments);
            }

            if (symbol.flags & SymbolFlags.TypeAlias) {
                return getTypeFromTypeAliasReference(node, symbol, typeArguments);
            }

            if (symbol.flags & SymbolFlags.Function &&
                isJSDocTypeReference(node) &&
                (symbol.members || getJSDocClassTag(symbol.valueDeclaration))) {
                return getInferredClassType(symbol);
            }
        }

        function getSubstitutionType(typeParameter: TypeParameter, substitute: Type) {
            const result = <SubstitutionType>createType(TypeFlags.Substitution);
            result.typeParameter = typeParameter;
            result.substitute = substitute;
            return result;
        }

        function getConstrainedTypeParameter(typeParameter: TypeParameter, node: Node) {
            let constraints: Type[];
            while (isPartOfTypeNode(node)) {
                const parent = node.parent;
                if (parent.kind === SyntaxKind.ConditionalType && node === (<ConditionalTypeNode>parent).trueType) {
                    if (getTypeFromTypeNode((<ConditionalTypeNode>parent).checkType) === typeParameter) {
                        constraints = append(constraints, getTypeFromTypeNode((<ConditionalTypeNode>parent).extendsType));
                    }
                }
                node = parent;
            }
            return constraints ? getSubstitutionType(typeParameter, getIntersectionType(append(constraints, typeParameter))) : typeParameter;
        }

        function isJSDocTypeReference(node: TypeReferenceType): node is TypeReferenceNode {
            return node.flags & NodeFlags.JSDoc && node.kind === SyntaxKind.TypeReference;
        }

        function checkNoTypeArguments(node: TypeReferenceType, symbol?: Symbol) {
            if (node.typeArguments) {
                error(node, Diagnostics.Type_0_is_not_generic, symbol ? symbolToString(symbol) : declarationNameToString((<TypeReferenceNode>node).typeName));
                return false;
            }
            return true;
        }

        function getIntendedTypeFromJSDocTypeReference(node: TypeReferenceNode): Type {
            if (isIdentifier(node.typeName)) {
                const typeArgs = node.typeArguments;
                switch (node.typeName.escapedText) {
                    case "String":
                        checkNoTypeArguments(node);
                        return stringType;
                    case "Number":
                        checkNoTypeArguments(node);
                        return numberType;
                    case "Boolean":
                        checkNoTypeArguments(node);
                        return booleanType;
                    case "Void":
                        checkNoTypeArguments(node);
                        return voidType;
                    case "Undefined":
                        checkNoTypeArguments(node);
                        return undefinedType;
                    case "Null":
                        checkNoTypeArguments(node);
                        return nullType;
                    case "Function":
                    case "function":
                        checkNoTypeArguments(node);
                        return globalFunctionType;
                    case "Array":
                    case "array":
                        return !typeArgs || !typeArgs.length ? anyArrayType : undefined;
                    case "Promise":
                    case "promise":
                        return !typeArgs || !typeArgs.length ? createPromiseType(anyType) : undefined;
                    case "Object":
                        if (typeArgs && typeArgs.length === 2) {
                            if (isJSDocIndexSignature(node)) {
                                const indexed = getTypeFromTypeNode(typeArgs[0]);
                                const target = getTypeFromTypeNode(typeArgs[1]);
                                const index = createIndexInfo(target, /*isReadonly*/ false);
                                return createAnonymousType(undefined, emptySymbols, emptyArray, emptyArray, indexed === stringType && index, indexed === numberType && index);
                            }
                            return anyType;
                        }
                        checkNoTypeArguments(node);
                        return anyType;
                }
            }
        }

        function getTypeFromJSDocNullableTypeNode(node: JSDocNullableType) {
            const type = getTypeFromTypeNode(node.type);
            return strictNullChecks ? getNullableType(type, TypeFlags.Null) : type;
        }

        function getTypeFromTypeReference(node: TypeReferenceType): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                let symbol: Symbol;
                let type: Type;
                let meaning = SymbolFlags.Type;
                if (isJSDocTypeReference(node)) {
                    type = getIntendedTypeFromJSDocTypeReference(node);
                    meaning |= SymbolFlags.Value;
                }
                if (!type) {
                    symbol = resolveTypeReferenceName(getTypeReferenceName(node), meaning);
                    type = getTypeReferenceType(node, symbol);
                }
                // Cache both the resolved symbol and the resolved type. The resolved symbol is needed in when we check the
                // type reference in checkTypeReferenceNode.
                links.resolvedSymbol = symbol;
                links.resolvedType = type;
            }
            return links.resolvedType;
        }

        function typeArgumentsFromTypeReferenceNode(node: TypeReferenceType): Type[] {
            return map(node.typeArguments, getTypeFromTypeNode);
        }

        function getTypeFromTypeQueryNode(node: TypeQueryNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                // TypeScript 1.0 spec (April 2014): 3.6.3
                // The expression is processed as an identifier expression (section 4.3)
                // or property access expression(section 4.10),
                // the widened type(section 3.9) of which becomes the result.
                links.resolvedType = getWidenedType(checkExpression(node.exprName));
            }
            return links.resolvedType;
        }

        function getTypeOfGlobalSymbol(symbol: Symbol, arity: number): ObjectType {

            function getTypeDeclaration(symbol: Symbol): Declaration {
                const declarations = symbol.declarations;
                for (const declaration of declarations) {
                    switch (declaration.kind) {
                        case SyntaxKind.ClassDeclaration:
                        case SyntaxKind.InterfaceDeclaration:
                        case SyntaxKind.EnumDeclaration:
                            return declaration;
                    }
                }
            }

            if (!symbol) {
                return arity ? emptyGenericType : emptyObjectType;
            }
            const type = getDeclaredTypeOfSymbol(symbol);
            if (!(type.flags & TypeFlags.Object)) {
                error(getTypeDeclaration(symbol), Diagnostics.Global_type_0_must_be_a_class_or_interface_type, symbolName(symbol));
                return arity ? emptyGenericType : emptyObjectType;
            }
            if (length((<InterfaceType>type).typeParameters) !== arity) {
                error(getTypeDeclaration(symbol), Diagnostics.Global_type_0_must_have_1_type_parameter_s, symbolName(symbol), arity);
                return arity ? emptyGenericType : emptyObjectType;
            }
            return <ObjectType>type;
        }

        function getGlobalValueSymbol(name: __String, reportErrors: boolean): Symbol {
            return getGlobalSymbol(name, SymbolFlags.Value, reportErrors ? Diagnostics.Cannot_find_global_value_0 : undefined);
        }

        function getGlobalTypeSymbol(name: __String, reportErrors: boolean): Symbol {
            return getGlobalSymbol(name, SymbolFlags.Type, reportErrors ? Diagnostics.Cannot_find_global_type_0 : undefined);
        }

        function getGlobalSymbol(name: __String, meaning: SymbolFlags, diagnostic: DiagnosticMessage): Symbol {
            // Don't track references for global symbols anyway, so value if `isReference` is arbitrary
            return resolveName(undefined, name, meaning, diagnostic, name, /*isUse*/ false);
        }

        function getGlobalType(name: __String, arity: 0, reportErrors: boolean): ObjectType;
        function getGlobalType(name: __String, arity: number, reportErrors: boolean): GenericType;
        function getGlobalType(name: __String, arity: number, reportErrors: boolean): ObjectType {
            const symbol = getGlobalTypeSymbol(name, reportErrors);
            return symbol || reportErrors ? getTypeOfGlobalSymbol(symbol, arity) : undefined;
        }

        function getGlobalTypedPropertyDescriptorType() {
            return deferredGlobalTypedPropertyDescriptorType || (deferredGlobalTypedPropertyDescriptorType = getGlobalType("TypedPropertyDescriptor" as __String, /*arity*/ 1, /*reportErrors*/ true)) || emptyGenericType;
        }

        function getGlobalTemplateStringsArrayType() {
            return deferredGlobalTemplateStringsArrayType || (deferredGlobalTemplateStringsArrayType = getGlobalType("TemplateStringsArray" as __String, /*arity*/ 0, /*reportErrors*/ true)) || emptyObjectType;
        }

        function getGlobalESSymbolConstructorSymbol(reportErrors: boolean) {
            return deferredGlobalESSymbolConstructorSymbol || (deferredGlobalESSymbolConstructorSymbol = getGlobalValueSymbol("Symbol" as __String, reportErrors));
        }

        function getGlobalESSymbolType(reportErrors: boolean) {
            return deferredGlobalESSymbolType || (deferredGlobalESSymbolType = getGlobalType("Symbol" as __String, /*arity*/ 0, reportErrors)) || emptyObjectType;
        }

        function getGlobalPromiseType(reportErrors: boolean) {
            return deferredGlobalPromiseType || (deferredGlobalPromiseType = getGlobalType("Promise" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalPromiseConstructorSymbol(reportErrors: boolean): Symbol | undefined {
            return deferredGlobalPromiseConstructorSymbol || (deferredGlobalPromiseConstructorSymbol = getGlobalValueSymbol("Promise" as __String, reportErrors));
        }

        function getGlobalPromiseConstructorLikeType(reportErrors: boolean) {
            return deferredGlobalPromiseConstructorLikeType || (deferredGlobalPromiseConstructorLikeType = getGlobalType("PromiseConstructorLike" as __String, /*arity*/ 0, reportErrors)) || emptyObjectType;
        }

        function getGlobalAsyncIterableType(reportErrors: boolean) {
            return deferredGlobalAsyncIterableType || (deferredGlobalAsyncIterableType = getGlobalType("AsyncIterable" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalAsyncIteratorType(reportErrors: boolean) {
            return deferredGlobalAsyncIteratorType || (deferredGlobalAsyncIteratorType = getGlobalType("AsyncIterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalAsyncIterableIteratorType(reportErrors: boolean) {
            return deferredGlobalAsyncIterableIteratorType || (deferredGlobalAsyncIterableIteratorType = getGlobalType("AsyncIterableIterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalIterableType(reportErrors: boolean) {
            return deferredGlobalIterableType || (deferredGlobalIterableType = getGlobalType("Iterable" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalIteratorType(reportErrors: boolean) {
            return deferredGlobalIteratorType || (deferredGlobalIteratorType = getGlobalType("Iterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalIterableIteratorType(reportErrors: boolean) {
            return deferredGlobalIterableIteratorType || (deferredGlobalIterableIteratorType = getGlobalType("IterableIterator" as __String, /*arity*/ 1, reportErrors)) || emptyGenericType;
        }

        function getGlobalTypeOrUndefined(name: __String, arity = 0): ObjectType {
            const symbol = getGlobalSymbol(name, SymbolFlags.Type, /*diagnostic*/ undefined);
            return symbol && <GenericType>getTypeOfGlobalSymbol(symbol, arity);
        }

        /**
         * Instantiates a global type that is generic with some element type, and returns that instantiation.
         */
        function createTypeFromGenericGlobalType(genericGlobalType: GenericType, typeArguments: Type[]): ObjectType {
            return genericGlobalType !== emptyGenericType ? createTypeReference(genericGlobalType, typeArguments) : emptyObjectType;
        }

        function createTypedPropertyDescriptorType(propertyType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalTypedPropertyDescriptorType(), [propertyType]);
        }

        function createAsyncIterableType(iteratedType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalAsyncIterableType(/*reportErrors*/ true), [iteratedType]);
        }

        function createAsyncIterableIteratorType(iteratedType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalAsyncIterableIteratorType(/*reportErrors*/ true), [iteratedType]);
        }

        function createIterableType(iteratedType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalIterableType(/*reportErrors*/ true), [iteratedType]);
        }

        function createIterableIteratorType(iteratedType: Type): Type {
            return createTypeFromGenericGlobalType(getGlobalIterableIteratorType(/*reportErrors*/ true), [iteratedType]);
        }

        function createArrayType(elementType: Type): ObjectType {
            return createTypeFromGenericGlobalType(globalArrayType, [elementType]);
        }

        function getTypeFromArrayTypeNode(node: ArrayTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = createArrayType(getTypeFromTypeNode(node.elementType));
            }
            return links.resolvedType;
        }

        // We represent tuple types as type references to synthesized generic interface types created by
        // this function. The types are of the form:
        //
        //   interface Tuple<T0, T1, T2, ...> extends Array<T0 | T1 | T2 | ...> { 0: T0, 1: T1, 2: T2, ... }
        //
        // Note that the generic type created by this function has no symbol associated with it. The same
        // is true for each of the synthesized type parameters.
        function createTupleTypeOfArity(arity: number): GenericType {
            const typeParameters: TypeParameter[] = [];
            const properties: Symbol[] = [];
            for (let i = 0; i < arity; i++) {
                const typeParameter = <TypeParameter>createType(TypeFlags.TypeParameter);
                typeParameters.push(typeParameter);
                const property = createSymbol(SymbolFlags.Property, "" + i as __String);
                property.type = typeParameter;
                properties.push(property);
            }
            const lengthSymbol = createSymbol(SymbolFlags.Property, "length" as __String);
            lengthSymbol.type = getLiteralType(arity);
            properties.push(lengthSymbol);
            const type = <GenericType & InterfaceTypeWithDeclaredMembers>createObjectType(ObjectFlags.Tuple | ObjectFlags.Reference);
            type.typeParameters = typeParameters;
            type.outerTypeParameters = undefined;
            type.localTypeParameters = typeParameters;
            type.instantiations = createMap<TypeReference>();
            type.instantiations.set(getTypeListId(type.typeParameters), <GenericType>type);
            type.target = <GenericType>type;
            type.typeArguments = type.typeParameters;
            type.thisType = <TypeParameter>createType(TypeFlags.TypeParameter);
            type.thisType.isThisType = true;
            type.thisType.constraint = type;
            type.declaredProperties = properties;
            type.declaredCallSignatures = emptyArray;
            type.declaredConstructSignatures = emptyArray;
            type.declaredStringIndexInfo = undefined;
            type.declaredNumberIndexInfo = undefined;
            return type;
        }

        function getTupleTypeOfArity(arity: number): GenericType {
            return tupleTypes[arity] || (tupleTypes[arity] = createTupleTypeOfArity(arity));
        }

        function createTupleType(elementTypes: Type[]) {
            return createTypeReference(getTupleTypeOfArity(elementTypes.length), elementTypes);
        }

        function getTypeFromTupleTypeNode(node: TupleTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = createTupleType(map(node.elementTypes, getTypeFromTypeNode));
            }
            return links.resolvedType;
        }

        function getTypeId(type: Type) {
            return type.id;
        }

        function containsType(types: Type[], type: Type): boolean {
            return binarySearch(types, type, getTypeId, compareValues) >= 0;
        }

        // Return true if the given intersection type contains (a) more than one unit type or (b) an object
        // type and a nullable type (null or undefined).
        function isEmptyIntersectionType(type: IntersectionType) {
            let combined: TypeFlags = 0;
            for (const t of type.types) {
                if (t.flags & TypeFlags.Unit && combined & TypeFlags.Unit) {
                    return true;
                }
                combined |= t.flags;
                if (combined & TypeFlags.Nullable && combined & (TypeFlags.Object | TypeFlags.NonPrimitive)) {
                    return true;
                }
            }
            return false;
        }

        function addTypeToUnion(typeSet: Type[], includes: TypeIncludes, type: Type) {
            const flags = type.flags;
            if (flags & TypeFlags.Union) {
                includes = addTypesToUnion(typeSet, includes, (<UnionType>type).types);
            }
            else if (flags & TypeFlags.Any) {
                includes |= TypeIncludes.Any;
                if (type === wildcardType) includes |= TypeIncludes.Wildcard;
            }
            else if (!strictNullChecks && flags & TypeFlags.Nullable) {
                if (flags & TypeFlags.Undefined) includes |= TypeIncludes.Undefined;
                if (flags & TypeFlags.Null) includes |= TypeIncludes.Null;
                if (!(flags & TypeFlags.ContainsWideningType)) includes |= TypeIncludes.NonWideningType;
            }
            else if (!(flags & TypeFlags.Never || flags & TypeFlags.Intersection && isEmptyIntersectionType(<IntersectionType>type))) {
                // We ignore 'never' types in unions. Likewise, we ignore intersections of unit types as they are
                // another form of 'never' (in that they have an empty value domain). We could in theory turn
                // intersections of unit types into 'never' upon construction, but deferring the reduction makes it
                // easier to reason about their origin.
                if (flags & TypeFlags.String) includes |= TypeIncludes.String;
                if (flags & TypeFlags.Number) includes |= TypeIncludes.Number;
                if (flags & TypeFlags.ESSymbol) includes |= TypeIncludes.ESSymbol;
                if (flags & TypeFlags.StringOrNumberLiteralOrUnique) includes |= TypeIncludes.LiteralOrUniqueESSymbol;
                const len = typeSet.length;
                const index = len && type.id > typeSet[len - 1].id ? ~len : binarySearch(typeSet, type, getTypeId, compareValues);
                if (index < 0) {
                    if (!(flags & TypeFlags.Object && (<ObjectType>type).objectFlags & ObjectFlags.Anonymous &&
                        type.symbol && type.symbol.flags & (SymbolFlags.Function | SymbolFlags.Method) && containsIdenticalType(typeSet, type))) {
                        typeSet.splice(~index, 0, type);
                    }
                }
            }
            return includes;
        }

        // Add the given types to the given type set. Order is preserved, duplicates are removed,
        // and nested types of the given kind are flattened into the set.
        function addTypesToUnion(typeSet: Type[], includes: TypeIncludes, types: Type[]): TypeIncludes {
            for (const type of types) {
                includes = addTypeToUnion(typeSet, includes, type);
            }
            return includes;
        }

        function containsIdenticalType(types: Type[], type: Type) {
            for (const t of types) {
                if (isTypeIdenticalTo(t, type)) {
                    return true;
                }
            }
            return false;
        }

        function isSubtypeOfAny(source: Type, targets: Type[]): boolean {
            for (const target of targets) {
                if (source !== target && isTypeSubtypeOf(source, target) && (
                    !(getObjectFlags(getTargetType(source)) & ObjectFlags.Class) ||
                    !(getObjectFlags(getTargetType(target)) & ObjectFlags.Class) ||
                    isTypeDerivedFrom(source, target))) {
                    return true;
                }
            }
            return false;
        }

        function isSetOfLiteralsFromSameEnum(types: Type[]): boolean {
            const first = types[0];
            if (first.flags & TypeFlags.EnumLiteral) {
                const firstEnum = getParentOfSymbol(first.symbol);
                for (let i = 1; i < types.length; i++) {
                    const other = types[i];
                    if (!(other.flags & TypeFlags.EnumLiteral) || (firstEnum !== getParentOfSymbol(other.symbol))) {
                        return false;
                    }
                }
                return true;
            }

            return false;
        }

        function removeSubtypes(types: Type[]) {
            if (types.length === 0 || isSetOfLiteralsFromSameEnum(types)) {
                return;
            }
            let i = types.length;
            while (i > 0) {
                i--;
                if (isSubtypeOfAny(types[i], types)) {
                    orderedRemoveItemAt(types, i);
                }
            }
        }

        function removeRedundantLiteralTypes(types: Type[], includes: TypeIncludes) {
            let i = types.length;
            while (i > 0) {
                i--;
                const t = types[i];
                const remove =
                    t.flags & TypeFlags.StringLiteral && includes & TypeIncludes.String ||
                    t.flags & TypeFlags.NumberLiteral && includes & TypeIncludes.Number ||
                    t.flags & TypeFlags.UniqueESSymbol && includes & TypeIncludes.ESSymbol ||
                    t.flags & TypeFlags.StringOrNumberLiteral && t.flags & TypeFlags.FreshLiteral && containsType(types, (<LiteralType>t).regularType);
                if (remove) {
                    orderedRemoveItemAt(types, i);
                }
            }
        }

        // We sort and deduplicate the constituent types based on object identity. If the subtypeReduction
        // flag is specified we also reduce the constituent type set to only include types that aren't subtypes
        // of other types. Subtype reduction is expensive for large union types and is possible only when union
        // types are known not to circularly reference themselves (as is the case with union types created by
        // expression constructs such as array literals and the || and ?: operators). Named types can
        // circularly reference themselves and therefore cannot be subtype reduced during their declaration.
        // For example, "type Item = string | (() => Item" is a named type that circularly references itself.
        function getUnionType(types: Type[], unionReduction: UnionReduction = UnionReduction.Literal, aliasSymbol?: Symbol, aliasTypeArguments?: Type[]): Type {
            if (types.length === 0) {
                return neverType;
            }
            if (types.length === 1) {
                return types[0];
            }
            const typeSet: Type[] = [];
            const includes = addTypesToUnion(typeSet, 0, types);
            if (includes & TypeIncludes.Any) {
                return includes & TypeIncludes.Wildcard ? wildcardType : anyType;
            }
            switch (unionReduction) {
                case UnionReduction.Literal:
                    if (includes & TypeIncludes.LiteralOrUniqueESSymbol) {
                        removeRedundantLiteralTypes(typeSet, includes);
                    }
                    break;
                case UnionReduction.Subtype:
                    removeSubtypes(typeSet);
                    break;
            }
            if (typeSet.length === 0) {
                return includes & TypeIncludes.Null ? includes & TypeIncludes.NonWideningType ? nullType : nullWideningType :
                    includes & TypeIncludes.Undefined ? includes & TypeIncludes.NonWideningType ? undefinedType : undefinedWideningType :
                        neverType;
            }
            return getUnionTypeFromSortedList(typeSet, aliasSymbol, aliasTypeArguments);
        }

        function getUnionTypePredicate(signatures: ReadonlyArray<Signature>): TypePredicate {
            let first: TypePredicate | undefined;
            const types: Type[] = [];
            for (const sig of signatures) {
                const pred = getTypePredicateOfSignature(sig);
                if (!pred) {
                    continue;
                }

                if (first) {
                    if (!typePredicateKindsMatch(first, pred)) {
                        // No common type predicate.
                        return undefined;
                    }
                }
                else {
                    first = pred;
                }
                types.push(pred.type);
            }
            if (!first) {
                // No union signatures had a type predicate.
                return undefined;
            }
            const unionType = getUnionType(types);
            return isIdentifierTypePredicate(first)
                ? createIdentifierTypePredicate(first.parameterName, first.parameterIndex, unionType)
                : createThisTypePredicate(unionType);
        }

        function typePredicateKindsMatch(a: TypePredicate, b: TypePredicate): boolean {
            return isIdentifierTypePredicate(a)
                ? isIdentifierTypePredicate(b) && a.parameterIndex === b.parameterIndex
                : !isIdentifierTypePredicate(b);
        }

        // This function assumes the constituent type list is sorted and deduplicated.
        function getUnionTypeFromSortedList(types: Type[], aliasSymbol?: Symbol, aliasTypeArguments?: Type[]): Type {
            if (types.length === 0) {
                return neverType;
            }
            if (types.length === 1) {
                return types[0];
            }
            const id = getTypeListId(types);
            let type = unionTypes.get(id);
            if (!type) {
                const propagatedFlags = getPropagatingFlagsOfTypes(types, /*excludeKinds*/ TypeFlags.Nullable);
                type = <UnionType>createType(TypeFlags.Union | propagatedFlags);
                unionTypes.set(id, type);
                type.types = types;
                /*
                Note: This is the alias symbol (or lack thereof) that we see when we first encounter this union type.
                For aliases of identical unions, eg `type T = A | B; type U = A | B`, the symbol of the first alias encountered is the aliasSymbol.
                (In the language service, the order may depend on the order in which a user takes actions, such as hovering over symbols.)
                It's important that we create equivalent union types only once, so that's an unfortunate side effect.
                */
                type.aliasSymbol = aliasSymbol;
                type.aliasTypeArguments = aliasTypeArguments;
            }
            return type;
        }

        function getTypeFromUnionTypeNode(node: UnionTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getUnionType(map(node.types, getTypeFromTypeNode), UnionReduction.Literal,
                    getAliasSymbolForTypeNode(node), getAliasTypeArgumentsForTypeNode(node));
            }
            return links.resolvedType;
        }

        function addTypeToIntersection(typeSet: Type[], includes: TypeIncludes, type: Type) {
            const flags = type.flags;
            if (flags & TypeFlags.Intersection) {
                includes = addTypesToIntersection(typeSet, includes, (<IntersectionType>type).types);
            }
            else if (flags & TypeFlags.Any) {
                includes |= TypeIncludes.Any;
                if (type === wildcardType) includes |= TypeIncludes.Wildcard;
            }
            else if (flags & TypeFlags.Never) {
                includes |= TypeIncludes.Never;
            }
            else if (getObjectFlags(type) & ObjectFlags.Anonymous && isEmptyObjectType(type)) {
                includes |= TypeIncludes.EmptyObject;
            }
            else if ((strictNullChecks || !(flags & TypeFlags.Nullable)) && !contains(typeSet, type)) {
                if (flags & TypeFlags.Object) {
                    includes |= TypeIncludes.ObjectType;
                }
                if (flags & TypeFlags.Union) {
                    includes |= TypeIncludes.Union;
                }
                if (!(flags & TypeFlags.Object && (<ObjectType>type).objectFlags & ObjectFlags.Anonymous &&
                    type.symbol && type.symbol.flags & (SymbolFlags.Function | SymbolFlags.Method) && containsIdenticalType(typeSet, type))) {
                    typeSet.push(type);
                }
            }
            return includes;
        }

        // Add the given types to the given type set. Order is preserved, freshness is removed from literal
        // types, duplicates are removed, and nested types of the given kind are flattened into the set.
        function addTypesToIntersection(typeSet: Type[], includes: TypeIncludes, types: Type[]) {
            for (const type of types) {
                includes = addTypeToIntersection(typeSet, includes, getRegularTypeOfLiteralType(type));
            }
            return includes;
        }

        // We normalize combinations of intersection and union types based on the distributive property of the '&'
        // operator. Specifically, because X & (A | B) is equivalent to X & A | X & B, we can transform intersection
        // types with union type constituents into equivalent union types with intersection type constituents and
        // effectively ensure that union types are always at the top level in type representations.
        //
        // We do not perform structural deduplication on intersection types. Intersection types are created only by the &
        // type operator and we can't reduce those because we want to support recursive intersection types. For example,
        // a type alias of the form "type List<T> = T & { next: List<T> }" cannot be reduced during its declaration.
        // Also, unlike union types, the order of the constituent types is preserved in order that overload resolution
        // for intersections of types with signatures can be deterministic.
        function getIntersectionType(types: Type[], aliasSymbol?: Symbol, aliasTypeArguments?: Type[]): Type {
            if (types.length === 0) {
                return emptyObjectType;
            }
            const typeSet: Type[] = [];
            const includes = addTypesToIntersection(typeSet, 0, types);
            if (includes & TypeIncludes.Never) {
                return neverType;
            }
            if (includes & TypeIncludes.Any) {
                return includes & TypeIncludes.Wildcard ? wildcardType : anyType;
            }
            if (includes & TypeIncludes.EmptyObject && !(includes & TypeIncludes.ObjectType)) {
                typeSet.push(emptyObjectType);
            }
            if (typeSet.length === 1) {
                return typeSet[0];
            }
            if (includes & TypeIncludes.Union) {
                // We are attempting to construct a type of the form X & (A | B) & Y. Transform this into a type of
                // the form X & A & Y | X & B & Y and recursively reduce until no union type constituents remain.
                const unionIndex = findIndex(typeSet, t => (t.flags & TypeFlags.Union) !== 0);
                const unionType = <UnionType>typeSet[unionIndex];
                return getUnionType(map(unionType.types, t => getIntersectionType(replaceElement(typeSet, unionIndex, t))),
                    UnionReduction.Literal, aliasSymbol, aliasTypeArguments);
            }
            const id = getTypeListId(typeSet);
            let type = intersectionTypes.get(id);
            if (!type) {
                const propagatedFlags = getPropagatingFlagsOfTypes(typeSet, /*excludeKinds*/ TypeFlags.Nullable);
                type = <IntersectionType>createType(TypeFlags.Intersection | propagatedFlags);
                intersectionTypes.set(id, type);
                type.types = typeSet;
                type.aliasSymbol = aliasSymbol; // See comment in `getUnionTypeFromSortedList`.
                type.aliasTypeArguments = aliasTypeArguments;
            }
            return type;
        }

        function getTypeFromIntersectionTypeNode(node: IntersectionTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getIntersectionType(map(node.types, getTypeFromTypeNode),
                    getAliasSymbolForTypeNode(node), getAliasTypeArgumentsForTypeNode(node));
            }
            return links.resolvedType;
        }

        function getIndexTypeForGenericType(type: InstantiableType | UnionOrIntersectionType) {
            if (!type.resolvedIndexType) {
                type.resolvedIndexType = <IndexType>createType(TypeFlags.Index);
                type.resolvedIndexType.type = type;
            }
            return type.resolvedIndexType;
        }

        function getLiteralTypeFromPropertyName(prop: Symbol) {
            const links = getSymbolLinks(getLateBoundSymbol(prop));
            if (!links.nameType) {
                if (links.target && links.target !== unknownSymbol && links.target !== resolvingSymbol) {
                    Debug.assert(links.target.escapedName === prop.escapedName || links.target.escapedName === InternalSymbolName.Computed, "Target symbol and symbol do not have the same name");
                    links.nameType = getLiteralTypeFromPropertyName(links.target);
                }
                else {
                    links.nameType = getDeclarationModifierFlagsFromSymbol(prop) & ModifierFlags.NonPublicAccessibilityModifier || isKnownSymbol(prop) ?
                        neverType :
                        getLiteralType(symbolName(prop));
                }
            }
            return links.nameType;
        }

        function getLiteralTypeFromPropertyNames(type: Type) {
            return getUnionType(map(getPropertiesOfType(type), getLiteralTypeFromPropertyName));
        }

        function getIndexType(type: Type): Type {
            return type.flags & TypeFlags.Intersection ? getUnionType(map((<IntersectionType>type).types, t => getIndexType(t))) :
                maybeTypeOfKind(type, TypeFlags.InstantiableNonPrimitive) ? getIndexTypeForGenericType(<InstantiableType | UnionOrIntersectionType>type) :
                getObjectFlags(type) & ObjectFlags.Mapped ? getConstraintTypeFromMappedType(<MappedType>type) :
                type === wildcardType ? wildcardType :
                type.flags & TypeFlags.Any || getIndexInfoOfType(type, IndexKind.String) ? stringType :
                getLiteralTypeFromPropertyNames(type);
        }

        function getIndexTypeOrString(type: Type): Type {
            const indexType = getIndexType(type);
            return indexType.flags & TypeFlags.Never ? stringType : indexType;
        }

        function getTypeFromTypeOperatorNode(node: TypeOperatorNode) {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                switch (node.operator) {
                    case SyntaxKind.KeyOfKeyword:
                        links.resolvedType = getIndexType(getTypeFromTypeNode(node.type));
                        break;
                    case SyntaxKind.UniqueKeyword:
                        links.resolvedType = node.type.kind === SyntaxKind.SymbolKeyword
                            ? getESSymbolLikeTypeForNode(walkUpParenthesizedTypes(node.parent))
                            : unknownType;
                        break;
                }
            }
            return links.resolvedType;
        }

        function createIndexedAccessType(objectType: Type, indexType: Type) {
            const type = <IndexedAccessType>createType(TypeFlags.IndexedAccess);
            type.objectType = objectType;
            type.indexType = indexType;
            return type;
        }

        function getPropertyTypeForIndexType(objectType: Type, indexType: Type, accessNode: ElementAccessExpression | IndexedAccessTypeNode, cacheSymbol: boolean) {
            const accessExpression = accessNode && accessNode.kind === SyntaxKind.ElementAccessExpression ? accessNode : undefined;
            const propName = isTypeUsableAsLateBoundName(indexType) ? getLateBoundNameFromType(indexType) :
                accessExpression && checkThatExpressionIsProperSymbolReference(accessExpression.argumentExpression, indexType, /*reportError*/ false) ?
                    getPropertyNameForKnownSymbolName(idText((<PropertyAccessExpression>accessExpression.argumentExpression).name)) :
                    undefined;
            if (propName !== undefined) {
                const prop = getPropertyOfType(objectType, propName);
                if (prop) {
                    if (accessExpression) {
                        markPropertyAsReferenced(prop, accessExpression, /*isThisAccess*/ accessExpression.expression.kind === SyntaxKind.ThisKeyword);
                        if (isAssignmentTarget(accessExpression) && (isReferenceToReadonlyEntity(accessExpression, prop) || isReferenceThroughNamespaceImport(accessExpression))) {
                            error(accessExpression.argumentExpression, Diagnostics.Cannot_assign_to_0_because_it_is_a_constant_or_a_read_only_property, symbolToString(prop));
                            return unknownType;
                        }
                        if (cacheSymbol) {
                            getNodeLinks(accessNode).resolvedSymbol = prop;
                        }
                    }
                    return getTypeOfSymbol(prop);
                }
            }
            if (!(indexType.flags & TypeFlags.Nullable) && isTypeAssignableToKind(indexType, TypeFlags.StringLike | TypeFlags.NumberLike | TypeFlags.ESSymbolLike)) {
                if (isTypeAny(objectType)) {
                    return objectType;
                }
                const indexInfo = isTypeAssignableToKind(indexType, TypeFlags.NumberLike) && getIndexInfoOfType(objectType, IndexKind.Number) ||
                    getIndexInfoOfType(objectType, IndexKind.String) ||
                    undefined;
                if (indexInfo) {
                    if (accessExpression && indexInfo.isReadonly && (isAssignmentTarget(accessExpression) || isDeleteTarget(accessExpression))) {
                        error(accessExpression, Diagnostics.Index_signature_in_type_0_only_permits_reading, typeToString(objectType));
                    }
                    return indexInfo.type;
                }
                if (accessExpression && !isConstEnumObjectType(objectType)) {
                    if (noImplicitAny && !compilerOptions.suppressImplicitAnyIndexErrors) {
                        if (getIndexTypeOfType(objectType, IndexKind.Number)) {
                            error(accessExpression.argumentExpression, Diagnostics.Element_implicitly_has_an_any_type_because_index_expression_is_not_of_type_number);
                        }
                        else {
                            error(accessExpression, Diagnostics.Element_implicitly_has_an_any_type_because_type_0_has_no_index_signature, typeToString(objectType));
                        }
                    }
                    return anyType;
                }
            }
            if (accessNode) {
                const indexNode = accessNode.kind === SyntaxKind.ElementAccessExpression ? accessNode.argumentExpression : accessNode.indexType;
                if (indexType.flags & (TypeFlags.StringLiteral | TypeFlags.NumberLiteral)) {
                    error(indexNode, Diagnostics.Property_0_does_not_exist_on_type_1, "" + (<LiteralType>indexType).value, typeToString(objectType));
                }
                else if (indexType.flags & (TypeFlags.String | TypeFlags.Number)) {
                    error(indexNode, Diagnostics.Type_0_has_no_matching_index_signature_for_type_1, typeToString(objectType), typeToString(indexType));
                }
                else {
                    error(indexNode, Diagnostics.Type_0_cannot_be_used_as_an_index_type, typeToString(indexType));
                }
                return unknownType;
            }
            return anyType;
        }

        function isGenericObjectType(type: Type): boolean {
            return maybeTypeOfKind(type, TypeFlags.InstantiableNonPrimitive | TypeFlags.GenericMappedType);
        }

        function isGenericIndexType(type: Type): boolean {
            return maybeTypeOfKind(type, TypeFlags.InstantiableNonPrimitive | TypeFlags.Index);
        }

        // Return true if the given type is a non-generic object type with a string index signature and no
        // other members.
        function isStringIndexOnlyType(type: Type) {
            if (type.flags & TypeFlags.Object && !isGenericMappedType(type)) {
                const t = resolveStructuredTypeMembers(<ObjectType>type);
                return t.properties.length === 0 &&
                    t.callSignatures.length === 0 && t.constructSignatures.length === 0 &&
                    t.stringIndexInfo && !t.numberIndexInfo;
            }
            return false;
        }

        function isMappedTypeToNever(type: Type) {
            return getObjectFlags(type) & ObjectFlags.Mapped && getTemplateTypeFromMappedType(type as MappedType) === neverType;
        }

        // Transform an indexed access to a simpler form, if possible. Return the simpler form, or return
        // undefined if no transformation is possible.
        function getSimplifiedIndexedAccessType(type: IndexedAccessType): Type {
            const objectType = type.objectType;
            if (objectType.flags & TypeFlags.Intersection && isGenericObjectType(objectType)) {
                // Given an indexed access type T[K], if T is an intersection containing one or more generic types and one or
                // more object types with only a string index signature, e.g. '(U & V & { [x: string]: D })[K]', return a
                // transformed type of the form '(U & V)[K] | D'. This allows us to properly reason about higher order indexed
                // access types with default property values as expressed by D.
                if (some((<IntersectionType>objectType).types, isStringIndexOnlyType)) {
                    const regularTypes: Type[] = [];
                    const stringIndexTypes: Type[] = [];
                    for (const t of (<IntersectionType>objectType).types) {
                        if (isStringIndexOnlyType(t)) {
                            stringIndexTypes.push(getIndexTypeOfType(t, IndexKind.String));
                        }
                        else {
                            regularTypes.push(t);
                        }
                    }
                    return getUnionType([
                        getIndexedAccessType(getIntersectionType(regularTypes), type.indexType),
                        getIntersectionType(stringIndexTypes)
                    ]);
                }
                // Given an indexed access type T[K], if T is an intersection containing one or more generic types and one or
                // more mapped types with a template type `never`, '(U & V & { [P in T]: never })[K]', return a
                // transformed type that removes the never-mapped type: '(U & V)[K]'. This mirrors what would happen
                // eventually anyway, but it easier to reason about.
                if (some((<IntersectionType>objectType).types, isMappedTypeToNever)) {
                    const nonNeverTypes = filter((<IntersectionType>objectType).types, t => !isMappedTypeToNever(t));
                    return getIndexedAccessType(getIntersectionType(nonNeverTypes), type.indexType);
                }
            }

            // If the object type is a mapped type { [P in K]: E }, where K is generic, instantiate E using a mapper
            // that substitutes the index type for P. For example, for an index access { [P in K]: Box<T[P]> }[X], we
            // construct the type Box<T[X]>.
            if (isGenericMappedType(objectType)) {
                return substituteIndexedMappedType(objectType, type);
            }
            if (objectType.flags & TypeFlags.TypeParameter) {
                const constraint = getConstraintFromTypeParameter(objectType as TypeParameter);
                if (constraint && isGenericMappedType(constraint)) {
                    return substituteIndexedMappedType(constraint, type);
                }
            }
            return undefined;
        }

        function substituteIndexedMappedType(objectType: MappedType, type: IndexedAccessType) {
            const mapper = createTypeMapper([getTypeParameterFromMappedType(objectType)], [type.indexType]);
            const templateMapper = combineTypeMappers(objectType.mapper, mapper);
            return instantiateType(getTemplateTypeFromMappedType(objectType), templateMapper);
        }

        function getIndexedAccessType(objectType: Type, indexType: Type, accessNode?: ElementAccessExpression | IndexedAccessTypeNode): Type {
            if (objectType === wildcardType || indexType === wildcardType) {
                return wildcardType;
            }
            // If the index type is generic, or if the object type is generic and doesn't originate in an expression,
            // we are performing a higher-order index access where we cannot meaningfully access the properties of the
            // object type. Note that for a generic T and a non-generic K, we eagerly resolve T[K] if it originates in
            // an expression. This is to preserve backwards compatibility. For example, an element access 'this["foo"]'
            // has always been resolved eagerly using the constraint type of 'this' at the given location.
            if (isGenericIndexType(indexType) || !(accessNode && accessNode.kind === SyntaxKind.ElementAccessExpression) && isGenericObjectType(objectType)) {
                if (objectType.flags & TypeFlags.Any) {
                    return objectType;
                }
                // Defer the operation by creating an indexed access type.
                const id = objectType.id + "," + indexType.id;
                let type = indexedAccessTypes.get(id);
                if (!type) {
                    indexedAccessTypes.set(id, type = createIndexedAccessType(objectType, indexType));
                }
                return type;
            }
            // In the following we resolve T[K] to the type of the property in T selected by K.
            // We treat boolean as different from other unions to improve errors;
            // skipping straight to getPropertyTypeForIndexType gives errors with 'boolean' instead of 'true'.
            const apparentObjectType = getApparentType(objectType);
            if (indexType.flags & TypeFlags.Union && !(indexType.flags & TypeFlags.Boolean)) {
                const propTypes: Type[] = [];
                for (const t of (<UnionType>indexType).types) {
                    const propType = getPropertyTypeForIndexType(apparentObjectType, t, accessNode, /*cacheSymbol*/ false);
                    if (propType === unknownType) {
                        return unknownType;
                    }
                    propTypes.push(propType);
                }
                return getUnionType(propTypes);
            }
            return getPropertyTypeForIndexType(apparentObjectType, indexType, accessNode, /*cacheSymbol*/ true);
        }

        function getTypeFromIndexedAccessTypeNode(node: IndexedAccessTypeNode) {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getIndexedAccessType(getTypeFromTypeNode(node.objectType), getTypeFromTypeNode(node.indexType), node);
            }
            return links.resolvedType;
        }

        function getTypeFromMappedTypeNode(node: MappedTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const type = <MappedType>createObjectType(ObjectFlags.Mapped, node.symbol);
                type.declaration = node;
                type.aliasSymbol = getAliasSymbolForTypeNode(node);
                type.aliasTypeArguments = getAliasTypeArgumentsForTypeNode(node);
                links.resolvedType = type;
                // Eagerly resolve the constraint type which forces an error if the constraint type circularly
                // references itself through one or more type aliases.
                getConstraintTypeFromMappedType(type);
            }
            return links.resolvedType;
        }

        function getActualTypeParameter(type: Type) {
            return type.flags & TypeFlags.Substitution ? (<SubstitutionType>type).typeParameter : type;
        }

        function getConditionalType(root: ConditionalRoot, mapper: TypeMapper): Type {
            const checkType = instantiateType(root.checkType, mapper);
            const extendsType = instantiateType(root.extendsType, mapper);
            if (checkType === wildcardType || extendsType === wildcardType) {
                return wildcardType;
            }
            // If this is a distributive conditional type and the check type is generic we need to defer
            // resolution of the conditional type such that a later instantiation will properly distribute
            // over union types.
            if (!root.isDistributive || !maybeTypeOfKind(checkType, TypeFlags.Instantiable)) {
                let combinedMapper: TypeMapper;
                if (root.inferTypeParameters) {
                    const context = createInferenceContext(root.inferTypeParameters, /*signature*/ undefined, InferenceFlags.None);
                    // We don't want inferences from constraints as they may cause us to eagerly resolve the
                    // conditional type instead of deferring resolution. Also, we always want strict function
                    // types rules (i.e. proper contravariance) for inferences.
                    inferTypes(context.inferences, checkType, extendsType, InferencePriority.NoConstraints | InferencePriority.AlwaysStrict);
                    combinedMapper = combineTypeMappers(mapper, context);
                }
                // Return union of trueType and falseType for 'any' since it matches anything
                if (checkType.flags & TypeFlags.Any) {
                    return getUnionType([instantiateType(root.trueType, combinedMapper || mapper), instantiateType(root.falseType, mapper)]);
                }
                // Instantiate the extends type including inferences for 'infer T' type parameters
                const inferredExtendsType = combinedMapper ? instantiateType(root.extendsType, combinedMapper) : extendsType;
                // Return falseType for a definitely false extends check. We check an instantations of the two
                // types with type parameters mapped to the wildcard type, the most permissive instantiations
                // possible (the wildcard type is assignable to and from all types). If those are not related,
                // then no instatiations will be and we can just return the false branch type.
                if (!isTypeAssignableTo(getWildcardInstantiation(checkType), getWildcardInstantiation(inferredExtendsType))) {
                    return instantiateType(root.falseType, mapper);
                }
                // Return trueType for a definitely true extends check. The definitely assignable relation excludes
                // type variable constraints from consideration. Without the definitely assignable relation, the type
                //   type Foo<T extends { x: any }> = T extends { x: string } ? string : number
                // would immediately resolve to 'string' instead of being deferred.
                if (checkTypeRelatedTo(checkType, inferredExtendsType, definitelyAssignableRelation, /*errorNode*/ undefined)) {
                    return instantiateType(root.trueType, combinedMapper || mapper);
                }
            }
            // Return a deferred type for a check that is neither definitely true nor definitely false
            const erasedCheckType = getActualTypeParameter(checkType);
            const result = <ConditionalType>createType(TypeFlags.Conditional);
            result.root = root;
            result.checkType = erasedCheckType;
            result.extendsType = extendsType;
            result.mapper = mapper;
            result.aliasSymbol = root.aliasSymbol;
            result.aliasTypeArguments = instantiateTypes(root.aliasTypeArguments, mapper);
            return result;
        }

        function getTrueTypeFromConditionalType(type: ConditionalType) {
            return type.resolvedTrueType || (type.resolvedTrueType = instantiateType(type.root.trueType, type.mapper));
        }

        function getFalseTypeFromConditionalType(type: ConditionalType) {
            return type.resolvedFalseType || (type.resolvedFalseType = instantiateType(type.root.falseType, type.mapper));
        }

        function getInferTypeParameters(node: ConditionalTypeNode): TypeParameter[] {
            let result: TypeParameter[];
            if (node.locals) {
                node.locals.forEach(symbol => {
                    if (symbol.flags & SymbolFlags.TypeParameter) {
                        result = append(result, getDeclaredTypeOfSymbol(symbol));
                    }
                });
            }
            return result;
        }

        function getTypeFromConditionalTypeNode(node: ConditionalTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                const checkType = getTypeFromTypeNode(node.checkType);
                const aliasTypeArguments = getAliasTypeArgumentsForTypeNode(node);
                const allOuterTypeParameters = getOuterTypeParameters(node, /*includeThisTypes*/ true);
                const outerTypeParameters = aliasTypeArguments ? allOuterTypeParameters : filter(allOuterTypeParameters, tp => isTypeParameterPossiblyReferenced(tp, node));
                const root: ConditionalRoot = {
                    node,
                    checkType,
                    extendsType: getTypeFromTypeNode(node.extendsType),
                    trueType: getTypeFromTypeNode(node.trueType),
                    falseType: getTypeFromTypeNode(node.falseType),
                    isDistributive: !!(checkType.flags & TypeFlags.TypeParameter),
                    inferTypeParameters: getInferTypeParameters(node),
                    outerTypeParameters,
                    instantiations: undefined,
                    aliasSymbol: getAliasSymbolForTypeNode(node),
                    aliasTypeArguments
                };
                links.resolvedType = getConditionalType(root, /*mapper*/ undefined);
                if (outerTypeParameters) {
                    root.instantiations = createMap<Type>();
                    root.instantiations.set(getTypeListId(outerTypeParameters), links.resolvedType);
                }
            }
            return links.resolvedType;
        }

        function getTypeFromInferTypeNode(node: InferTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getDeclaredTypeOfTypeParameter(getSymbolOfNode(node.typeParameter));
            }
            return links.resolvedType;
        }

        function getTypeFromTypeLiteralOrFunctionOrConstructorTypeNode(node: TypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                // Deferred resolution of members is handled by resolveObjectTypeMembers
                const aliasSymbol = getAliasSymbolForTypeNode(node);
                if (getMembersOfSymbol(node.symbol).size === 0 && !aliasSymbol) {
                    links.resolvedType = emptyTypeLiteralType;
                }
                else {
                    let type = createObjectType(ObjectFlags.Anonymous, node.symbol);
                    type.aliasSymbol = aliasSymbol;
                    type.aliasTypeArguments = getAliasTypeArgumentsForTypeNode(node);
                    if (isJSDocTypeLiteral(node) && node.isArrayType) {
                        type = createArrayType(type);
                    }
                    links.resolvedType = type;
                }
            }
            return links.resolvedType;
        }

        function getAliasSymbolForTypeNode(node: TypeNode) {
            return node.parent.kind === SyntaxKind.TypeAliasDeclaration ? getSymbolOfNode(node.parent) : undefined;
        }

        function getAliasTypeArgumentsForTypeNode(node: TypeNode) {
            const symbol = getAliasSymbolForTypeNode(node);
            return symbol ? getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(symbol) : undefined;
        }

        /**
         * Since the source of spread types are object literals, which are not binary,
         * this function should be called in a left folding style, with left = previous result of getSpreadType
         * and right = the new element to be spread.
         */
        function getSpreadType(left: Type, right: Type, symbol: Symbol, typeFlags: TypeFlags, objectFlags: ObjectFlags): Type {
            if (left.flags & TypeFlags.Any || right.flags & TypeFlags.Any) {
                return anyType;
            }
            if (left.flags & TypeFlags.Never) {
                return right;
            }
            if (right.flags & TypeFlags.Never) {
                return left;
            }
            if (left.flags & TypeFlags.Union) {
                return mapType(left, t => getSpreadType(t, right, symbol, typeFlags, objectFlags));
            }
            if (right.flags & TypeFlags.Union) {
                return mapType(right, t => getSpreadType(left, t, symbol, typeFlags, objectFlags));
            }
            if (right.flags & (TypeFlags.BooleanLike | TypeFlags.NumberLike | TypeFlags.StringLike | TypeFlags.EnumLike | TypeFlags.NonPrimitive)) {
                return left;
            }

            const members = createSymbolTable();
            const skippedPrivateMembers = createUnderscoreEscapedMap<boolean>();
            let stringIndexInfo: IndexInfo;
            let numberIndexInfo: IndexInfo;
            if (left === emptyObjectType) {
                // for the first spread element, left === emptyObjectType, so take the right's string indexer
                stringIndexInfo = getIndexInfoOfType(right, IndexKind.String);
                numberIndexInfo = getIndexInfoOfType(right, IndexKind.Number);
            }
            else {
                stringIndexInfo = unionSpreadIndexInfos(getIndexInfoOfType(left, IndexKind.String), getIndexInfoOfType(right, IndexKind.String));
                numberIndexInfo = unionSpreadIndexInfos(getIndexInfoOfType(left, IndexKind.Number), getIndexInfoOfType(right, IndexKind.Number));
            }

            for (const rightProp of getPropertiesOfType(right)) {
                // we approximate own properties as non-methods plus methods that are inside the object literal
                const isSetterWithoutGetter = rightProp.flags & SymbolFlags.SetAccessor && !(rightProp.flags & SymbolFlags.GetAccessor);
                if (getDeclarationModifierFlagsFromSymbol(rightProp) & (ModifierFlags.Private | ModifierFlags.Protected)) {
                    skippedPrivateMembers.set(rightProp.escapedName, true);
                }
                else if (!isClassMethod(rightProp) && !isSetterWithoutGetter) {
                    members.set(rightProp.escapedName, getNonReadonlySymbol(rightProp));
                }
            }

            for (const leftProp of getPropertiesOfType(left)) {
                if (leftProp.flags & SymbolFlags.SetAccessor && !(leftProp.flags & SymbolFlags.GetAccessor)
                    || skippedPrivateMembers.has(leftProp.escapedName)
                    || isClassMethod(leftProp)) {
                    continue;
                }
                if (members.has(leftProp.escapedName)) {
                    const rightProp = members.get(leftProp.escapedName);
                    const rightType = getTypeOfSymbol(rightProp);
                    if (rightProp.flags & SymbolFlags.Optional) {
                        const declarations: Declaration[] = concatenate(leftProp.declarations, rightProp.declarations);
                        const flags = SymbolFlags.Property | (leftProp.flags & SymbolFlags.Optional);
                        const result = createSymbol(flags, leftProp.escapedName);
                        result.type = getUnionType([getTypeOfSymbol(leftProp), getTypeWithFacts(rightType, TypeFacts.NEUndefined)]);
                        result.leftSpread = leftProp;
                        result.rightSpread = rightProp;
                        result.declarations = declarations;
                        members.set(leftProp.escapedName, result);
                    }
                }
                else {
                    members.set(leftProp.escapedName, getNonReadonlySymbol(leftProp));
                }
            }

            const spread = createAnonymousType(
                symbol,
                members,
                emptyArray,
                emptyArray,
                getNonReadonlyIndexSignature(stringIndexInfo),
                getNonReadonlyIndexSignature(numberIndexInfo));
            spread.flags |= typeFlags | TypeFlags.ContainsObjectLiteral;
            (spread as ObjectType).objectFlags |= objectFlags | (ObjectFlags.ObjectLiteral | ObjectFlags.ContainsSpread);
            return spread;
        }

        function getNonReadonlySymbol(prop: Symbol) {
            if (!isReadonlySymbol(prop)) {
                return prop;
            }
            const flags = SymbolFlags.Property | (prop.flags & SymbolFlags.Optional);
            const result = createSymbol(flags, prop.escapedName);
            result.type = getTypeOfSymbol(prop);
            result.declarations = prop.declarations;
            result.syntheticOrigin = prop;
            return result;
        }

        function getNonReadonlyIndexSignature(index: IndexInfo) {
            if (index && index.isReadonly) {
                return createIndexInfo(index.type, /*isReadonly*/ false, index.declaration);
            }
            return index;
        }

        function isClassMethod(prop: Symbol) {
            return prop.flags & SymbolFlags.Method && find(prop.declarations, decl => isClassLike(decl.parent));
        }

        function createLiteralType(flags: TypeFlags, value: string | number, symbol: Symbol) {
            const type = <LiteralType>createType(flags);
            type.symbol = symbol;
            type.value = value;
            return type;
        }

        function getFreshTypeOfLiteralType(type: Type) {
            if (type.flags & TypeFlags.StringOrNumberLiteral && !(type.flags & TypeFlags.FreshLiteral)) {
                if (!(<LiteralType>type).freshType) {
                    const freshType = createLiteralType(type.flags | TypeFlags.FreshLiteral, (<LiteralType>type).value, (<LiteralType>type).symbol);
                    freshType.regularType = <LiteralType>type;
                    (<LiteralType>type).freshType = freshType;
                }
                return (<LiteralType>type).freshType;
            }
            return type;
        }

        function getRegularTypeOfLiteralType(type: Type) {
            return type.flags & TypeFlags.StringOrNumberLiteral && type.flags & TypeFlags.FreshLiteral ? (<LiteralType>type).regularType : type;
        }

        function getLiteralType(value: string | number, enumId?: number, symbol?: Symbol) {
            // We store all literal types in a single map with keys of the form '#NNN' and '@SSS',
            // where NNN is the text representation of a numeric literal and SSS are the characters
            // of a string literal. For literal enum members we use 'EEE#NNN' and 'EEE@SSS', where
            // EEE is a unique id for the containing enum type.
            const qualifier = typeof value === "number" ? "#" : "@";
            const key = enumId ? enumId + qualifier + value : qualifier + value;
            let type = literalTypes.get(key);
            if (!type) {
                const flags = (typeof value === "number" ? TypeFlags.NumberLiteral : TypeFlags.StringLiteral) | (enumId ? TypeFlags.EnumLiteral : 0);
                literalTypes.set(key, type = createLiteralType(flags, value, symbol));
            }
            return type;
        }

        function getTypeFromLiteralTypeNode(node: LiteralTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getRegularTypeOfLiteralType(checkExpression(node.literal));
            }
            return links.resolvedType;
        }

        function createUniqueESSymbolType(symbol: Symbol) {
            const type = <UniqueESSymbolType>createType(TypeFlags.UniqueESSymbol);
            type.symbol = symbol;
            return type;
        }

        function getESSymbolLikeTypeForNode(node: Node) {
            if (isValidESSymbolDeclaration(node)) {
                const symbol = getSymbolOfNode(node);
                const links = getSymbolLinks(symbol);
                return links.uniqueESSymbolType || (links.uniqueESSymbolType = createUniqueESSymbolType(symbol));
            }
            return esSymbolType;
        }

        function getThisType(node: Node): Type {
            const container = getThisContainer(node, /*includeArrowFunctions*/ false);
            const parent = container && container.parent;
            if (parent && (isClassLike(parent) || parent.kind === SyntaxKind.InterfaceDeclaration)) {
                if (!hasModifier(container, ModifierFlags.Static) &&
                    (container.kind !== SyntaxKind.Constructor || isNodeDescendantOf(node, (<ConstructorDeclaration>container).body))) {
                    return getDeclaredTypeOfClassOrInterface(getSymbolOfNode(parent)).thisType;
                }
            }
            error(node, Diagnostics.A_this_type_is_available_only_in_a_non_static_member_of_a_class_or_interface);
            return unknownType;
        }

        function getTypeFromThisTypeNode(node: ThisExpression | ThisTypeNode): Type {
            const links = getNodeLinks(node);
            if (!links.resolvedType) {
                links.resolvedType = getThisType(node);
            }
            return links.resolvedType;
        }

        function getTypeFromTypeNode(node: TypeNode): Type {
            switch (node.kind) {
                case SyntaxKind.AnyKeyword:
                case SyntaxKind.JSDocAllType:
                case SyntaxKind.JSDocUnknownType:
                    return anyType;
                case SyntaxKind.StringKeyword:
                    return stringType;
                case SyntaxKind.NumberKeyword:
                    return numberType;
                case SyntaxKind.BooleanKeyword:
                    return booleanType;
                case SyntaxKind.SymbolKeyword:
                    return esSymbolType;
                case SyntaxKind.VoidKeyword:
                    return voidType;
                case SyntaxKind.UndefinedKeyword:
                    return undefinedType;
                case SyntaxKind.NullKeyword:
                    return nullType;
                case SyntaxKind.NeverKeyword:
                    return neverType;
                case SyntaxKind.ObjectKeyword:
                    return node.flags & NodeFlags.JavaScriptFile ? anyType : nonPrimitiveType;
                case SyntaxKind.ThisType:
                case SyntaxKind.ThisKeyword:
                    return getTypeFromThisTypeNode(node as ThisExpression | ThisTypeNode);
                case SyntaxKind.LiteralType:
                    return getTypeFromLiteralTypeNode(<LiteralTypeNode>node);
                case SyntaxKind.TypeReference:
                    return getTypeFromTypeReference(<TypeReferenceNode>node);
                case SyntaxKind.TypePredicate:
                    return booleanType;
                case SyntaxKind.ExpressionWithTypeArguments:
                    return getTypeFromTypeReference(<ExpressionWithTypeArguments>node);
                case SyntaxKind.TypeQuery:
                    return getTypeFromTypeQueryNode(<TypeQueryNode>node);
                case SyntaxKind.ArrayType:
                    return getTypeFromArrayTypeNode(<ArrayTypeNode>node);
                case SyntaxKind.TupleType:
                    return getTypeFromTupleTypeNode(<TupleTypeNode>node);
                case SyntaxKind.UnionType:
                    return getTypeFromUnionTypeNode(<UnionTypeNode>node);
                case SyntaxKind.IntersectionType:
                    return getTypeFromIntersectionTypeNode(<IntersectionTypeNode>node);
                case SyntaxKind.JSDocNullableType:
                    return getTypeFromJSDocNullableTypeNode(<JSDocNullableType>node);
                case SyntaxKind.ParenthesizedType:
                case SyntaxKind.JSDocNonNullableType:
                case SyntaxKind.JSDocOptionalType:
                case SyntaxKind.JSDocTypeExpression:
                    return getTypeFromTypeNode((<ParenthesizedTypeNode | JSDocTypeReferencingNode | JSDocTypeExpression>node).type);
                case SyntaxKind.JSDocVariadicType:
                    return getTypeFromJSDocVariadicType(node as JSDocVariadicType);
                case SyntaxKind.FunctionType:
                case SyntaxKind.ConstructorType:
                case SyntaxKind.TypeLiteral:
                case SyntaxKind.JSDocTypeLiteral:
                case SyntaxKind.JSDocFunctionType:
                    return getTypeFromTypeLiteralOrFunctionOrConstructorTypeNode(node);
                case SyntaxKind.TypeOperator:
                    return getTypeFromTypeOperatorNode(<TypeOperatorNode>node);
                case SyntaxKind.IndexedAccessType:
                    return getTypeFromIndexedAccessTypeNode(<IndexedAccessTypeNode>node);
                case SyntaxKind.MappedType:
                    return getTypeFromMappedTypeNode(<MappedTypeNode>node);
                case SyntaxKind.ConditionalType:
                    return getTypeFromConditionalTypeNode(<ConditionalTypeNode>node);
                case SyntaxKind.InferType:
                    return getTypeFromInferTypeNode(<InferTypeNode>node);
                // This function assumes that an identifier or qualified name is a type expression
                // Callers should first ensure this by calling isTypeNode
                case SyntaxKind.Identifier:
                case SyntaxKind.QualifiedName:
                    const symbol = getSymbolAtLocation(node);
                    return symbol && getDeclaredTypeOfSymbol(symbol);
                default:
                    return unknownType;
            }
        }

        function instantiateList<T>(items: T[], mapper: TypeMapper, instantiator: (item: T, mapper: TypeMapper) => T): T[] {
            if (items && items.length) {
                for (let i = 0; i < items.length; i++) {
                    const item = items[i];
                    const mapped = instantiator(item, mapper);
                    if (item !== mapped) {
                        const result = i === 0 ? [] : items.slice(0, i);
                        result.push(mapped);
                        for (i++; i < items.length; i++) {
                            result.push(instantiator(items[i], mapper));
                        }
                        return result;
                    }
                }
            }
            return items;
        }

        function instantiateTypes(types: Type[], mapper: TypeMapper) {
            return instantiateList(types, mapper, instantiateType);
        }

        function instantiateSignatures(signatures: Signature[], mapper: TypeMapper) {
            return instantiateList(signatures, mapper, instantiateSignature);
        }

        function makeUnaryTypeMapper(source: Type, target: Type) {
            return (t: Type) => t === source ? target : t;
        }

        function makeBinaryTypeMapper(source1: Type, target1: Type, source2: Type, target2: Type) {
            return (t: Type) => t === source1 ? target1 : t === source2 ? target2 : t;
        }

        function makeArrayTypeMapper(sources: Type[], targets: Type[]) {
            return (t: Type) => {
                for (let i = 0; i < sources.length; i++) {
                    if (t === sources[i]) {
                        return targets ? targets[i] : anyType;
                    }
                }
                return t;
            };
        }

        function createTypeMapper(sources: TypeParameter[], targets: Type[]): TypeMapper {
            Debug.assert(targets === undefined || sources.length === targets.length);
            return sources.length === 1 ? makeUnaryTypeMapper(sources[0], targets ? targets[0] : anyType) :
                sources.length === 2 ? makeBinaryTypeMapper(sources[0], targets ? targets[0] : anyType, sources[1], targets ? targets[1] : anyType) :
                makeArrayTypeMapper(sources, targets);
        }

        function createTypeEraser(sources: TypeParameter[]): TypeMapper {
            return createTypeMapper(sources, /*targets*/ undefined);
        }

        /**
         * Maps forward-references to later types parameters to the empty object type.
         * This is used during inference when instantiating type parameter defaults.
         */
        function createBackreferenceMapper(typeParameters: TypeParameter[], index: number): TypeMapper {
            return t => typeParameters.indexOf(t) >= index ? emptyObjectType : t;
        }

        function isInferenceContext(mapper: TypeMapper): mapper is InferenceContext {
            return !!(<InferenceContext>mapper).typeParameters;
        }

        function cloneTypeMapper(mapper: TypeMapper): TypeMapper {
            return mapper && isInferenceContext(mapper) ?
                createInferenceContext(mapper.typeParameters, mapper.signature, mapper.flags | InferenceFlags.NoDefault, mapper.compareTypes, mapper.inferences) :
                mapper;
        }

        function combineTypeMappers(mapper1: TypeMapper, mapper2: TypeMapper): TypeMapper {
            if (!mapper1) return mapper2;
            if (!mapper2) return mapper1;
            return t => instantiateType(mapper1(t), mapper2);
        }

        function createReplacementMapper(source: Type, target: Type, baseMapper: TypeMapper): TypeMapper {
            return t => t === source ? target : baseMapper(t);
        }

        function wildcardMapper(type: Type) {
            return type.flags & TypeFlags.TypeParameter ? wildcardType : type;
        }

        function cloneTypeParameter(typeParameter: TypeParameter): TypeParameter {
            const result = <TypeParameter>createType(TypeFlags.TypeParameter);
            result.symbol = typeParameter.symbol;
            result.target = typeParameter;
            return result;
        }

        function instantiateTypePredicate(predicate: TypePredicate, mapper: TypeMapper): ThisTypePredicate | IdentifierTypePredicate {
            if (isIdentifierTypePredicate(predicate)) {
                return {
                    kind: TypePredicateKind.Identifier,
                    parameterName: predicate.parameterName,
                    parameterIndex: predicate.parameterIndex,
                    type: instantiateType(predicate.type, mapper)
                };
            }
            else {
                return {
                    kind: TypePredicateKind.This,
                    type: instantiateType(predicate.type, mapper)
                };
            }
        }

        function instantiateSignature(signature: Signature, mapper: TypeMapper, eraseTypeParameters?: boolean): Signature {
            let freshTypeParameters: TypeParameter[];
            if (signature.typeParameters && !eraseTypeParameters) {
                // First create a fresh set of type parameters, then include a mapping from the old to the
                // new type parameters in the mapper function. Finally store this mapper in the new type
                // parameters such that we can use it when instantiating constraints.
                freshTypeParameters = map(signature.typeParameters, cloneTypeParameter);
                mapper = combineTypeMappers(createTypeMapper(signature.typeParameters, freshTypeParameters), mapper);
                for (const tp of freshTypeParameters) {
                    tp.mapper = mapper;
                }
            }
            // Don't compute resolvedReturnType and resolvedTypePredicate now,
            // because using `mapper` now could trigger inferences to become fixed. (See `createInferenceContext`.)
            // See GH#17600.
            const result = createSignature(signature.declaration, freshTypeParameters,
                signature.thisParameter && instantiateSymbol(signature.thisParameter, mapper),
                instantiateList(signature.parameters, mapper, instantiateSymbol),
                /*resolvedReturnType*/ undefined,
                /*resolvedTypePredicate*/ undefined,
                signature.minArgumentCount,
                signature.hasRestParameter,
                signature.hasLiteralTypes);
            result.target = signature;
            result.mapper = mapper;
            return result;
        }

        function instantiateSymbol(symbol: Symbol, mapper: TypeMapper): Symbol {
            const links = getSymbolLinks(symbol);
            if (links.type && !maybeTypeOfKind(links.type, TypeFlags.Object | TypeFlags.Instantiable)) {
                // If the type of the symbol is already resolved, and if that type could not possibly
                // be affected by instantiation, simply return the symbol itself.
                return symbol;
            }
            if (getCheckFlags(symbol) & CheckFlags.Instantiated) {
                // If symbol being instantiated is itself a instantiation, fetch the original target and combine the
                // type mappers. This ensures that original type identities are properly preserved and that aliases
                // always reference a non-aliases.
                symbol = links.target;
                mapper = combineTypeMappers(links.mapper, mapper);
            }
            // Keep the flags from the symbol we're instantiating.  Mark that is instantiated, and
            // also transient so that we can just store data on it directly.
            const result = createSymbol(symbol.flags, symbol.escapedName, CheckFlags.Instantiated);
            result.declarations = symbol.declarations;
            result.parent = symbol.parent;
            result.target = symbol;
            result.mapper = mapper;
            if (symbol.valueDeclaration) {
                result.valueDeclaration = symbol.valueDeclaration;
            }
            if ((symbol as TransientSymbol).isRestParameter) {
                result.isRestParameter = (symbol as TransientSymbol).isRestParameter;
            }
            return result;
        }

        function getAnonymousTypeInstantiation(type: AnonymousType, mapper: TypeMapper) {
            const target = type.objectFlags & ObjectFlags.Instantiated ? type.target : type;
            const symbol = target.symbol;
            const links = getSymbolLinks(symbol);
            let typeParameters = links.outerTypeParameters;
            if (!typeParameters) {
                // The first time an anonymous type is instantiated we compute and store a list of the type
                // parameters that are in scope (and therefore potentially referenced). For type literals that
                // aren't the right hand side of a generic type alias declaration we optimize by reducing the
                // set of type parameters to those that are possibly referenced in the literal.
                const declaration = symbol.declarations[0];
                const outerTypeParameters = getOuterTypeParameters(declaration, /*includeThisTypes*/ true) || emptyArray;
                typeParameters = symbol.flags & SymbolFlags.TypeLiteral && !target.aliasTypeArguments ?
                    filter(outerTypeParameters, tp => isTypeParameterPossiblyReferenced(tp, declaration)) :
                    outerTypeParameters;
                links.outerTypeParameters = typeParameters;
                if (typeParameters.length) {
                    links.instantiations = createMap<Type>();
                    links.instantiations.set(getTypeListId(typeParameters), target);
                }
            }
            if (typeParameters.length) {
                // We are instantiating an anonymous type that has one or more type parameters in scope. Apply the
                // mapper to the type parameters to produce the effective list of type arguments, and compute the
                // instantiation cache key from the type IDs of the type arguments.
                const combinedMapper = type.objectFlags & ObjectFlags.Instantiated ? combineTypeMappers(type.mapper, mapper) : mapper;
                const typeArguments = map(typeParameters, combinedMapper);
                const id = getTypeListId(typeArguments);
                let result = links.instantiations.get(id);
                if (!result) {
                    const newMapper = createTypeMapper(typeParameters, typeArguments);
                    result = target.objectFlags & ObjectFlags.Mapped ? instantiateMappedType(<MappedType>target, newMapper) : instantiateAnonymousType(target, newMapper);
                    links.instantiations.set(id, result);
                }
                return result;
            }
            return type;
        }

        function isTypeParameterPossiblyReferenced(tp: TypeParameter, node: Node) {
            // If the type parameter doesn't have exactly one declaration, if there are invening statement blocks
            // between the node and the type parameter declaration, if the node contains actual references to the
            // type parameter, or if the node contains type queries, we consider the type parameter possibly referenced.
            if (tp.symbol && tp.symbol.declarations && tp.symbol.declarations.length === 1) {
                const container = tp.symbol.declarations[0].parent;
                if (findAncestor(node, n => n.kind === SyntaxKind.Block ? "quit" : n === container)) {
                    return forEachChild(node, containsReference);
                }
            }
            return true;
            function containsReference(node: Node): boolean {
                switch (node.kind) {
                    case SyntaxKind.ThisType:
                        return tp.isThisType;
                    case SyntaxKind.Identifier:
                        return !tp.isThisType && isPartOfTypeNode(node) && getTypeFromTypeNode(<TypeNode>node) === tp;
                    case SyntaxKind.TypeQuery:
                        return true;
                }
                return forEachChild(node, containsReference);
            }
        }

        function instantiateMappedType(type: MappedType, mapper: TypeMapper): Type {
            // Check if we have a homomorphic mapped type, i.e. a type of the form { [P in keyof T]: X } for some
            // type variable T. If so, the mapped type is distributive over a union type and when T is instantiated
            // to a union type A | B, we produce { [P in keyof A]: X } | { [P in keyof B]: X }. Furthermore, for
            // homomorphic mapped types we leave primitive types alone. For example, when T is instantiated to a
            // union type A | undefined, we produce { [P in keyof A]: X } | undefined.
            const constraintType = getConstraintTypeFromMappedType(type);
            if (constraintType.flags & TypeFlags.Index) {
                const typeVariable = (<IndexType>constraintType).type;
                if (typeVariable.flags & TypeFlags.TypeParameter) {
                    const mappedTypeVariable = instantiateType(typeVariable, mapper);
                    if (typeVariable !== mappedTypeVariable) {
                        return mapType(mappedTypeVariable, t => {
                            if (isMappableType(t)) {
                                return instantiateAnonymousType(type, createReplacementMapper(typeVariable, t, mapper));
                            }
                            return t;
                        });
                    }
                }
            }
            return instantiateAnonymousType(type, mapper);
        }

        function isMappableType(type: Type) {
            return type.flags & (TypeFlags.Any | TypeFlags.InstantiableNonPrimitive | TypeFlags.Object | TypeFlags.Intersection);
        }

        function instantiateAnonymousType(type: AnonymousType, mapper: TypeMapper): AnonymousType {
            const result = <AnonymousType>createObjectType(type.objectFlags | ObjectFlags.Instantiated, type.symbol);
            if (type.objectFlags & ObjectFlags.Mapped) {
                (<MappedType>result).declaration = (<MappedType>type).declaration;
            }
            result.target = type;
            result.mapper = mapper;
            result.aliasSymbol = type.aliasSymbol;
            result.aliasTypeArguments = instantiateTypes(type.aliasTypeArguments, mapper);
            return result;
        }

        function getConditionalTypeInstantiation(type: ConditionalType, mapper: TypeMapper): Type {
            const root = type.root;
            if (root.outerTypeParameters) {
                // We are instantiating a conditional type that has one or more type parameters in scope. Apply the
                // mapper to the type parameters to produce the effective list of type arguments, and compute the
                // instantiation cache key from the type IDs of the type arguments.
                const typeArguments = map(root.outerTypeParameters, mapper);
                const id = getTypeListId(typeArguments);
                let result = root.instantiations.get(id);
                if (!result) {
                    const newMapper = createTypeMapper(root.outerTypeParameters, typeArguments);
                    result = instantiateConditionalType(root, newMapper);
                    root.instantiations.set(id, result);
                }
                return result;
            }
            return type;
        }

        function instantiateConditionalType(root: ConditionalRoot, mapper: TypeMapper): Type {
            // Check if we have a conditional type where the check type is a naked type parameter. If so,
            // the conditional type is distributive over union types and when T is instantiated to a union
            // type A | B, we produce (A extends U ? X : Y) | (B extends U ? X : Y).
            if (root.isDistributive) {
                const checkType = <TypeParameter>root.checkType;
                const instantiatedType = mapper(checkType);
                if (checkType !== instantiatedType && instantiatedType.flags & (TypeFlags.Union | TypeFlags.Never)) {
                    return mapType(instantiatedType, t => getConditionalType(root, createReplacementMapper(checkType, t, mapper)));
                }
            }
            return getConditionalType(root, mapper);
        }

        function instantiateType(type: Type, mapper: TypeMapper): Type {
            if (type && mapper && mapper !== identityMapper) {
                if (type.flags & TypeFlags.TypeParameter) {
                    return mapper(<TypeParameter>type);
                }
                if (type.flags & TypeFlags.Object) {
                    if ((<ObjectType>type).objectFlags & ObjectFlags.Anonymous) {
                        // If the anonymous type originates in a declaration of a function, method, class, or
                        // interface, in an object type literal, or in an object literal expression, we may need
                        // to instantiate the type because it might reference a type parameter.
                        return type.symbol && type.symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.Class | SymbolFlags.TypeLiteral | SymbolFlags.ObjectLiteral) && type.symbol.declarations ?
                            getAnonymousTypeInstantiation(<AnonymousType>type, mapper) : type;
                    }
                    if ((<ObjectType>type).objectFlags & ObjectFlags.Mapped) {
                        return getAnonymousTypeInstantiation(<MappedType>type, mapper);
                    }
                    if ((<ObjectType>type).objectFlags & ObjectFlags.Reference) {
                        const typeArguments = (<TypeReference>type).typeArguments;
                        const newTypeArguments = instantiateTypes(typeArguments, mapper);
                        return newTypeArguments !== typeArguments ? createTypeReference((<TypeReference>type).target, newTypeArguments) : type;
                    }
                }
                if (type.flags & TypeFlags.Union && !(type.flags & TypeFlags.Primitive)) {
                    const types = (<UnionType>type).types;
                    const newTypes = instantiateTypes(types, mapper);
                    return newTypes !== types ? getUnionType(newTypes, UnionReduction.Literal, type.aliasSymbol, instantiateTypes(type.aliasTypeArguments, mapper)) : type;
                }
                if (type.flags & TypeFlags.Intersection) {
                    const types = (<IntersectionType>type).types;
                    const newTypes = instantiateTypes(types, mapper);
                    return newTypes !== types ? getIntersectionType(newTypes, type.aliasSymbol, instantiateTypes(type.aliasTypeArguments, mapper)) : type;
                }
                if (type.flags & TypeFlags.Index) {
                    return getIndexType(instantiateType((<IndexType>type).type, mapper));
                }
                if (type.flags & TypeFlags.IndexedAccess) {
                    return getIndexedAccessType(instantiateType((<IndexedAccessType>type).objectType, mapper), instantiateType((<IndexedAccessType>type).indexType, mapper));
                }
                if (type.flags & TypeFlags.Conditional) {
                    return getConditionalTypeInstantiation(<ConditionalType>type, combineTypeMappers((<ConditionalType>type).mapper, mapper));
                }
                if (type.flags & TypeFlags.Substitution) {
                    return mapper((<SubstitutionType>type).typeParameter);
                }
            }
            return type;
        }

        function getWildcardInstantiation(type: Type) {
            return type.flags & (TypeFlags.Primitive | TypeFlags.Any | TypeFlags.Never) ? type :
                type.wildcardInstantiation || (type.wildcardInstantiation = instantiateType(type, wildcardMapper));
        }

        function instantiateIndexInfo(info: IndexInfo, mapper: TypeMapper): IndexInfo {
            return info && createIndexInfo(instantiateType(info.type, mapper), info.isReadonly, info.declaration);
        }

        // Returns true if the given expression contains (at any level of nesting) a function or arrow expression
        // that is subject to contextual typing.
        function isContextSensitive(node: Expression | MethodDeclaration | ObjectLiteralElementLike | JsxAttributeLike): boolean {
            Debug.assert(node.kind !== SyntaxKind.MethodDeclaration || isObjectLiteralMethod(node));
            switch (node.kind) {
                case SyntaxKind.FunctionExpression:
                case SyntaxKind.ArrowFunction:
                case SyntaxKind.MethodDeclaration:
                    return isContextSensitiveFunctionLikeDeclaration(<FunctionExpression | ArrowFunction | MethodDeclaration>node);
                case SyntaxKind.ObjectLiteralExpression:
                    return forEach((<ObjectLiteralExpression>node).properties, isContextSensitive);
                case SyntaxKind.ArrayLiteralExpression:
                    return forEach((<ArrayLiteralExpression>node).elements, isContextSensitive);
                case SyntaxKind.ConditionalExpression:
                    return isContextSensitive((<ConditionalExpression>node).whenTrue) ||
                        isContextSensitive((<ConditionalExpression>node).whenFalse);
                case SyntaxKind.BinaryExpression:
                    return (<BinaryExpression>node).operatorToken.kind === SyntaxKind.BarBarToken &&
                        (isContextSensitive((<BinaryExpression>node).left) || isContextSensitive((<BinaryExpression>node).right));
                case SyntaxKind.PropertyAssignment:
                    return isContextSensitive((<PropertyAssignment>node).initializer);
                case SyntaxKind.ParenthesizedExpression:
                    return isContextSensitive((<ParenthesizedExpression>node).expression);
                case SyntaxKind.JsxAttributes:
                    return forEach((<JsxAttributes>node).properties, isContextSensitive);
                case SyntaxKind.JsxAttribute:
                    // If there is no initializer, JSX attribute has a boolean value of true which is not context sensitive.
                    return (<JsxAttribute>node).initializer && isContextSensitive((<JsxAttribute>node).initializer);
                case SyntaxKind.JsxExpression:
                    // It is possible to that node.expression is undefined (e.g <div x={} />)
                    return (<JsxExpression>node).expression && isContextSensitive((<JsxExpression>node).expression);
            }

            return false;
        }

        function isContextSensitiveFunctionLikeDeclaration(node: FunctionLikeDeclaration) {
            // Functions with type parameters are not context sensitive.
            if (node.typeParameters) {
                return false;
            }
            // Functions with any parameters that lack type annotations are context sensitive.
            if (forEach(node.parameters, p => !getEffectiveTypeAnnotationNode(p))) {
                return true;
            }
            if (node.kind !== SyntaxKind.ArrowFunction) {
                // If the first parameter is not an explicit 'this' parameter, then the function has
                // an implicit 'this' parameter which is subject to contextual typing.
                const parameter = firstOrUndefined(node.parameters);
                if (!(parameter && parameterIsThisKeyword(parameter))) {
                    return true;
                }
            }

            // TODO(anhans): A block should be context-sensitive if it has a context-sensitive return value.
            return node.body.kind === SyntaxKind.Block ? false : isContextSensitive(node.body);
        }

        function isContextSensitiveFunctionOrObjectLiteralMethod(func: Node): func is FunctionExpression | ArrowFunction | MethodDeclaration {
            return (isFunctionExpressionOrArrowFunction(func) || isObjectLiteralMethod(func)) && isContextSensitiveFunctionLikeDeclaration(func);
        }

        function getTypeWithoutSignatures(type: Type): Type {
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                if (resolved.constructSignatures.length) {
                    const result = <ResolvedType>createObjectType(ObjectFlags.Anonymous, type.symbol);
                    result.members = resolved.members;
                    result.properties = resolved.properties;
                    result.callSignatures = emptyArray;
                    result.constructSignatures = emptyArray;
                    return result;
                }
            }
            else if (type.flags & TypeFlags.Intersection) {
                return getIntersectionType(map((<IntersectionType>type).types, getTypeWithoutSignatures));
            }
            return type;
        }

        // TYPE CHECKING

        function isTypeIdenticalTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, identityRelation);
        }

        function compareTypesIdentical(source: Type, target: Type): Ternary {
            return isTypeRelatedTo(source, target, identityRelation) ? Ternary.True : Ternary.False;
        }

        function compareTypesAssignable(source: Type, target: Type): Ternary {
            return isTypeRelatedTo(source, target, assignableRelation) ? Ternary.True : Ternary.False;
        }

        function isTypeSubtypeOf(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, subtypeRelation);
        }

        function isTypeAssignableTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, assignableRelation);
        }

        // An object type S is considered to be derived from an object type T if
        // S is a union type and every constituent of S is derived from T,
        // T is a union type and S is derived from at least one constituent of T, or
        // S is a type variable with a base constraint that is derived from T,
        // T is one of the global types Object and Function and S is a subtype of T, or
        // T occurs directly or indirectly in an 'extends' clause of S.
        // Note that this check ignores type parameters and only considers the
        // inheritance hierarchy.
        function isTypeDerivedFrom(source: Type, target: Type): boolean {
            return source.flags & TypeFlags.Union ? every((<UnionType>source).types, t => isTypeDerivedFrom(t, target)) :
                target.flags & TypeFlags.Union ? some((<UnionType>target).types, t => isTypeDerivedFrom(source, t)) :
                source.flags & TypeFlags.InstantiableNonPrimitive ? isTypeDerivedFrom(getBaseConstraintOfType(source) || emptyObjectType, target) :
                target === globalObjectType || target === globalFunctionType ? isTypeSubtypeOf(source, target) :
                hasBaseType(source, getTargetType(target));
            }

        /**
         * This is *not* a bi-directional relationship.
         * If one needs to check both directions for comparability, use a second call to this function or 'checkTypeComparableTo'.
         *
         * A type S is comparable to a type T if some (but not necessarily all) of the possible values of S are also possible values of T.
         * It is used to check following cases:
         *   - the types of the left and right sides of equality/inequality operators (`===`, `!==`, `==`, `!=`).
         *   - the types of `case` clause expressions and their respective `switch` expressions.
         *   - the type of an expression in a type assertion with the type being asserted.
         */
        function isTypeComparableTo(source: Type, target: Type): boolean {
            return isTypeRelatedTo(source, target, comparableRelation);
        }

        function areTypesComparable(type1: Type, type2: Type): boolean {
            return isTypeComparableTo(type1, type2) || isTypeComparableTo(type2, type1);
        }

        function checkTypeAssignableTo(source: Type, target: Type, errorNode: Node, headMessage?: DiagnosticMessage, containingMessageChain?: () => DiagnosticMessageChain | undefined): boolean {
            return checkTypeRelatedTo(source, target, assignableRelation, errorNode, headMessage, containingMessageChain);
        }

        /**
         * This is *not* a bi-directional relationship.
         * If one needs to check both directions for comparability, use a second call to this function or 'isTypeComparableTo'.
         */
        function checkTypeComparableTo(source: Type, target: Type, errorNode: Node, headMessage?: DiagnosticMessage, containingMessageChain?: () => DiagnosticMessageChain | undefined): boolean {
            return checkTypeRelatedTo(source, target, comparableRelation, errorNode, headMessage, containingMessageChain);
        }

        function isSignatureAssignableTo(source: Signature,
            target: Signature,
            ignoreReturnTypes: boolean): boolean {
            return compareSignaturesRelated(source, target, CallbackCheck.None, ignoreReturnTypes, /*reportErrors*/ false,
                /*errorReporter*/ undefined, compareTypesAssignable) !== Ternary.False;
        }

        type ErrorReporter = (message: DiagnosticMessage, arg0?: string, arg1?: string) => void;

        /**
         * See signatureRelatedTo, compareSignaturesIdentical
         */
        function compareSignaturesRelated(source: Signature,
            target: Signature,
            callbackCheck: CallbackCheck,
            ignoreReturnTypes: boolean,
            reportErrors: boolean,
            errorReporter: ErrorReporter,
            compareTypes: TypeComparer): Ternary {
            // TODO (drosen): De-duplicate code between related functions.
            if (source === target) {
                return Ternary.True;
            }
            if (!target.hasRestParameter && source.minArgumentCount > target.parameters.length) {
                return Ternary.False;
            }

            if (source.typeParameters && source.typeParameters !== target.typeParameters) {
                target = getCanonicalSignature(target);
                source = instantiateSignatureInContextOf(source, target, /*contextualMapper*/ undefined, compareTypes);
            }

            const kind = target.declaration ? target.declaration.kind : SyntaxKind.Unknown;
            const strictVariance = !callbackCheck && strictFunctionTypes && kind !== SyntaxKind.MethodDeclaration &&
                kind !== SyntaxKind.MethodSignature && kind !== SyntaxKind.Constructor;
            let result = Ternary.True;

            const sourceThisType = getThisTypeOfSignature(source);
            if (sourceThisType && sourceThisType !== voidType) {
                const targetThisType = getThisTypeOfSignature(target);
                if (targetThisType) {
                    // void sources are assignable to anything.
                    const related = !strictVariance && compareTypes(sourceThisType, targetThisType, /*reportErrors*/ false)
                        || compareTypes(targetThisType, sourceThisType, reportErrors);
                    if (!related) {
                        if (reportErrors) {
                            errorReporter(Diagnostics.The_this_types_of_each_signature_are_incompatible);
                        }
                        return Ternary.False;
                    }
                    result &= related;
                }
            }

            const sourceMax = getNumNonRestParameters(source);
            const targetMax = getNumNonRestParameters(target);
            const checkCount = getNumParametersToCheckForSignatureRelatability(source, sourceMax, target, targetMax);
            const sourceParams = source.parameters;
            const targetParams = target.parameters;
            for (let i = 0; i < checkCount; i++) {
                const sourceType = i < sourceMax ? getTypeOfParameter(sourceParams[i]) : getRestTypeOfSignature(source);
                const targetType = i < targetMax ? getTypeOfParameter(targetParams[i]) : getRestTypeOfSignature(target);
                // In order to ensure that any generic type Foo<T> is at least co-variant with respect to T no matter
                // how Foo uses T, we need to relate parameters bi-variantly (given that parameters are input positions,
                // they naturally relate only contra-variantly). However, if the source and target parameters both have
                // function types with a single call signature, we know we are relating two callback parameters. In
                // that case it is sufficient to only relate the parameters of the signatures co-variantly because,
                // similar to return values, callback parameters are output positions. This means that a Promise<T>,
                // where T is used only in callback parameter positions, will be co-variant (as opposed to bi-variant)
                // with respect to T.
                const sourceSig = callbackCheck ? undefined : getSingleCallSignature(getNonNullableType(sourceType));
                const targetSig = callbackCheck ? undefined : getSingleCallSignature(getNonNullableType(targetType));
                const callbacks = sourceSig && targetSig && !signatureHasTypePredicate(sourceSig) && !signatureHasTypePredicate(targetSig) &&
                    (getFalsyFlags(sourceType) & TypeFlags.Nullable) === (getFalsyFlags(targetType) & TypeFlags.Nullable);
                const related = callbacks ?
                    compareSignaturesRelated(targetSig, sourceSig, strictVariance ? CallbackCheck.Strict : CallbackCheck.Bivariant, /*ignoreReturnTypes*/ false, reportErrors, errorReporter, compareTypes) :
                    !callbackCheck && !strictVariance && compareTypes(sourceType, targetType, /*reportErrors*/ false) || compareTypes(targetType, sourceType, reportErrors);
                if (!related) {
                    if (reportErrors) {
                        errorReporter(Diagnostics.Types_of_parameters_0_and_1_are_incompatible,
                            symbolName(sourceParams[i < sourceMax ? i : sourceMax]),
                            symbolName(targetParams[i < targetMax ? i : targetMax]));
                    }
                    return Ternary.False;
                }
                result &= related;
            }

            if (!ignoreReturnTypes) {
                const targetReturnType = getReturnTypeOfSignature(target);
                if (targetReturnType === voidType) {
                    return result;
                }
                const sourceReturnType = getReturnTypeOfSignature(source);

                // The following block preserves behavior forbidding boolean returning functions from being assignable to type guard returning functions
                const targetTypePredicate = getTypePredicateOfSignature(target);
                if (targetTypePredicate) {
                    const sourceTypePredicate = getTypePredicateOfSignature(source);
                    if (sourceTypePredicate) {
                        result &= compareTypePredicateRelatedTo(sourceTypePredicate, targetTypePredicate, source.declaration, target.declaration, reportErrors, errorReporter, compareTypes);
                    }
                    else if (isIdentifierTypePredicate(targetTypePredicate)) {
                        if (reportErrors) {
                            errorReporter(Diagnostics.Signature_0_must_be_a_type_predicate, signatureToString(source));
                        }
                        return Ternary.False;
                    }
                }
                else {
                    // When relating callback signatures, we still need to relate return types bi-variantly as otherwise
                    // the containing type wouldn't be co-variant. For example, interface Foo<T> { add(cb: () => T): void }
                    // wouldn't be co-variant for T without this rule.
                    result &= callbackCheck === CallbackCheck.Bivariant && compareTypes(targetReturnType, sourceReturnType, /*reportErrors*/ false) ||
                        compareTypes(sourceReturnType, targetReturnType, reportErrors);
                }

            }

            return result;
        }

        function compareTypePredicateRelatedTo(
            source: TypePredicate,
            target: TypePredicate,
            sourceDeclaration: SignatureDeclaration,
            targetDeclaration: SignatureDeclaration,
            reportErrors: boolean,
            errorReporter: ErrorReporter,
            compareTypes: (s: Type, t: Type, reportErrors?: boolean) => Ternary): Ternary {
            if (source.kind !== target.kind) {
                if (reportErrors) {
                    errorReporter(Diagnostics.A_this_based_type_guard_is_not_compatible_with_a_parameter_based_type_guard);
                    errorReporter(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
                }
                return Ternary.False;
            }

            if (source.kind === TypePredicateKind.Identifier) {
                const targetPredicate = target as IdentifierTypePredicate;
                const sourceIndex = source.parameterIndex - (getThisParameter(sourceDeclaration) ? 1 : 0);
                const targetIndex = targetPredicate.parameterIndex - (getThisParameter(targetDeclaration) ? 1 : 0);
                if (sourceIndex !== targetIndex) {
                    if (reportErrors) {
                        errorReporter(Diagnostics.Parameter_0_is_not_in_the_same_position_as_parameter_1, source.parameterName, targetPredicate.parameterName);
                        errorReporter(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
                    }
                    return Ternary.False;
                }
            }

            const related = compareTypes(source.type, target.type, reportErrors);
            if (related === Ternary.False && reportErrors) {
                errorReporter(Diagnostics.Type_predicate_0_is_not_assignable_to_1, typePredicateToString(source), typePredicateToString(target));
            }
            return related;
        }

        function isImplementationCompatibleWithOverload(implementation: Signature, overload: Signature): boolean {
            const erasedSource = getErasedSignature(implementation);
            const erasedTarget = getErasedSignature(overload);

            // First see if the return types are compatible in either direction.
            const sourceReturnType = getReturnTypeOfSignature(erasedSource);
            const targetReturnType = getReturnTypeOfSignature(erasedTarget);
            if (targetReturnType === voidType
                || isTypeRelatedTo(targetReturnType, sourceReturnType, assignableRelation)
                || isTypeRelatedTo(sourceReturnType, targetReturnType, assignableRelation)) {

                return isSignatureAssignableTo(erasedSource, erasedTarget, /*ignoreReturnTypes*/ true);
            }

            return false;
        }

        function getNumNonRestParameters(signature: Signature) {
            const numParams = signature.parameters.length;
            return signature.hasRestParameter ?
                numParams - 1 :
                numParams;
        }

        function getNumParametersToCheckForSignatureRelatability(source: Signature, sourceNonRestParamCount: number, target: Signature, targetNonRestParamCount: number) {
            if (source.hasRestParameter === target.hasRestParameter) {
                if (source.hasRestParameter) {
                    // If both have rest parameters, get the max and add 1 to
                    // compensate for the rest parameter.
                    return Math.max(sourceNonRestParamCount, targetNonRestParamCount) + 1;
                }
                else {
                    return Math.min(sourceNonRestParamCount, targetNonRestParamCount);
                }
            }
            else {
                // Return the count for whichever signature doesn't have rest parameters.
                return source.hasRestParameter ?
                    targetNonRestParamCount :
                    sourceNonRestParamCount;
            }
        }

        function isEmptyResolvedType(t: ResolvedType) {
            return t.properties.length === 0 &&
                t.callSignatures.length === 0 &&
                t.constructSignatures.length === 0 &&
                !t.stringIndexInfo &&
                !t.numberIndexInfo;
        }

        function isEmptyObjectType(type: Type): boolean {
            return type.flags & TypeFlags.Object ? isEmptyResolvedType(resolveStructuredTypeMembers(<ObjectType>type)) :
                type.flags & TypeFlags.NonPrimitive ? true :
                type.flags & TypeFlags.Union ? forEach((<UnionType>type).types, isEmptyObjectType) :
                type.flags & TypeFlags.Intersection ? !forEach((<UnionType>type).types, t => !isEmptyObjectType(t)) :
                false;
        }

        function isEnumTypeRelatedTo(sourceSymbol: Symbol, targetSymbol: Symbol, errorReporter?: ErrorReporter) {
            if (sourceSymbol === targetSymbol) {
                return true;
            }
            const id = getSymbolId(sourceSymbol) + "," + getSymbolId(targetSymbol);
            const relation = enumRelation.get(id);
            if (relation !== undefined) {
                return relation;
            }
            if (sourceSymbol.escapedName !== targetSymbol.escapedName || !(sourceSymbol.flags & SymbolFlags.RegularEnum) || !(targetSymbol.flags & SymbolFlags.RegularEnum)) {
                enumRelation.set(id, false);
                return false;
            }
            const targetEnumType = getTypeOfSymbol(targetSymbol);
            for (const property of getPropertiesOfType(getTypeOfSymbol(sourceSymbol))) {
                if (property.flags & SymbolFlags.EnumMember) {
                    const targetProperty = getPropertyOfType(targetEnumType, property.escapedName);
                    if (!targetProperty || !(targetProperty.flags & SymbolFlags.EnumMember)) {
                        if (errorReporter) {
                            errorReporter(Diagnostics.Property_0_is_missing_in_type_1, symbolName(property),
                                typeToString(getDeclaredTypeOfSymbol(targetSymbol), /*enclosingDeclaration*/ undefined, TypeFormatFlags.UseFullyQualifiedType));
                        }
                        enumRelation.set(id, false);
                        return false;
                    }
                }
            }
            enumRelation.set(id, true);
            return true;
        }

        function isSimpleTypeRelatedTo(source: Type, target: Type, relation: Map<RelationComparisonResult>, errorReporter?: ErrorReporter) {
            const s = source.flags;
            const t = target.flags;
            if (t & TypeFlags.Any || s & TypeFlags.Never || source === wildcardType) return true;
            if (t & TypeFlags.Never) return false;
            if (s & TypeFlags.StringLike && t & TypeFlags.String) return true;
            if (s & TypeFlags.StringLiteral && s & TypeFlags.EnumLiteral &&
                t & TypeFlags.StringLiteral && !(t & TypeFlags.EnumLiteral) &&
                (<LiteralType>source).value === (<LiteralType>target).value) return true;
            if (s & TypeFlags.NumberLike && t & TypeFlags.Number) return true;
            if (s & TypeFlags.NumberLiteral && s & TypeFlags.EnumLiteral &&
                t & TypeFlags.NumberLiteral && !(t & TypeFlags.EnumLiteral) &&
                (<LiteralType>source).value === (<LiteralType>target).value) return true;
            if (s & TypeFlags.BooleanLike && t & TypeFlags.Boolean) return true;
            if (s & TypeFlags.ESSymbolLike && t & TypeFlags.ESSymbol) return true;
            if (s & TypeFlags.Enum && t & TypeFlags.Enum && isEnumTypeRelatedTo(source.symbol, target.symbol, errorReporter)) return true;
            if (s & TypeFlags.EnumLiteral && t & TypeFlags.EnumLiteral) {
                if (s & TypeFlags.Union && t & TypeFlags.Union && isEnumTypeRelatedTo(source.symbol, target.symbol, errorReporter)) return true;
                if (s & TypeFlags.Literal && t & TypeFlags.Literal &&
                    (<LiteralType>source).value === (<LiteralType>target).value &&
                    isEnumTypeRelatedTo(getParentOfSymbol(source.symbol), getParentOfSymbol(target.symbol), errorReporter)) return true;
            }
            if (s & TypeFlags.Undefined && (!strictNullChecks || t & (TypeFlags.Undefined | TypeFlags.Void))) return true;
            if (s & TypeFlags.Null && (!strictNullChecks || t & TypeFlags.Null)) return true;
            if (s & TypeFlags.Object && t & TypeFlags.NonPrimitive) return true;
            if (s & TypeFlags.UniqueESSymbol || t & TypeFlags.UniqueESSymbol) return false;
            if (relation === assignableRelation || relation === definitelyAssignableRelation || relation === comparableRelation) {
                if (s & TypeFlags.Any) return true;
                // Type number or any numeric literal type is assignable to any numeric enum type or any
                // numeric enum literal type. This rule exists for backwards compatibility reasons because
                // bit-flag enum types sometimes look like literal enum types with numeric literal values.
                if (s & (TypeFlags.Number | TypeFlags.NumberLiteral) && !(s & TypeFlags.EnumLiteral) && (
                    t & TypeFlags.Enum || t & TypeFlags.NumberLiteral && t & TypeFlags.EnumLiteral)) return true;
            }
            return false;
        }

        function isTypeRelatedTo(source: Type, target: Type, relation: Map<RelationComparisonResult>) {
            if (source.flags & TypeFlags.StringOrNumberLiteral && source.flags & TypeFlags.FreshLiteral) {
                source = (<LiteralType>source).regularType;
            }
            if (target.flags & TypeFlags.StringOrNumberLiteral && target.flags & TypeFlags.FreshLiteral) {
                target = (<LiteralType>target).regularType;
            }
            if (source === target ||
                relation === comparableRelation && !(target.flags & TypeFlags.Never) && isSimpleTypeRelatedTo(target, source, relation) ||
                relation !== identityRelation && isSimpleTypeRelatedTo(source, target, relation)) {
                return true;
            }
            if (source.flags & TypeFlags.Object && target.flags & TypeFlags.Object) {
                const related = relation.get(getRelationKey(source, target, relation));
                if (related !== undefined) {
                    return related === RelationComparisonResult.Succeeded;
                }
            }
            if (source.flags & TypeFlags.StructuredOrInstantiable || target.flags & TypeFlags.StructuredOrInstantiable) {
                return checkTypeRelatedTo(source, target, relation, /*errorNode*/ undefined);
            }
            return false;
        }

        function isIgnoredJsxProperty(source: Type, sourceProp: Symbol, targetMemberType: Type | undefined) {
            return getObjectFlags(source) & ObjectFlags.JsxAttributes && !(isUnhyphenatedJsxName(sourceProp.escapedName) || targetMemberType);
        }

        /**
         * Checks if 'source' is related to 'target' (e.g.: is a assignable to).
         * @param source The left-hand-side of the relation.
         * @param target The right-hand-side of the relation.
         * @param relation The relation considered. One of 'identityRelation', 'subtypeRelation', 'assignableRelation', or 'comparableRelation'.
         * Used as both to determine which checks are performed and as a cache of previously computed results.
         * @param errorNode The suggested node upon which all errors will be reported, if defined. This may or may not be the actual node used.
         * @param headMessage If the error chain should be prepended by a head message, then headMessage will be used.
         * @param containingMessageChain A chain of errors to prepend any new errors found.
         */
        function checkTypeRelatedTo(
            source: Type,
            target: Type,
            relation: Map<RelationComparisonResult>,
            errorNode: Node,
            headMessage?: DiagnosticMessage,
            containingMessageChain?: () => DiagnosticMessageChain | undefined): boolean {

            let errorInfo: DiagnosticMessageChain;
            let maybeKeys: string[];
            let sourceStack: Type[];
            let targetStack: Type[];
            let maybeCount = 0;
            let depth = 0;
            let expandingFlags = ExpandingFlags.None;
            let overflow = false;
            let isIntersectionConstituent = false;

            Debug.assert(relation !== identityRelation || !errorNode, "no error reporting in identity checking");

            const result = isRelatedTo(source, target, /*reportErrors*/ !!errorNode, headMessage);
            if (overflow) {
                error(errorNode, Diagnostics.Excessive_stack_depth_comparing_types_0_and_1, typeToString(source), typeToString(target));
            }
            else if (errorInfo) {
                if (containingMessageChain) {
                    const chain = containingMessageChain();
                    if (chain) {
                        errorInfo = concatenateDiagnosticMessageChains(chain, errorInfo);
                    }
                }

                diagnostics.add(createDiagnosticForNodeFromMessageChain(errorNode, errorInfo));
            }
            // Check if we should issue an extra diagnostic to produce a quickfix for a slightly incorrect import statement
            if (headMessage && errorNode && !result && source.symbol) {
                const links = getSymbolLinks(source.symbol);
                if (links.originatingImport && !isImportCall(links.originatingImport)) {
                    const helpfulRetry = checkTypeRelatedTo(getTypeOfSymbol(links.target), target, relation, /*errorNode*/ undefined);
                    if (helpfulRetry) {
                        // Likely an incorrect import. Issue a helpful diagnostic to produce a quickfix to change the import
                        diagnostics.add(createDiagnosticForNode(links.originatingImport, Diagnostics.A_namespace_style_import_cannot_be_called_or_constructed_and_will_cause_a_failure_at_runtime));
                    }
                }
            }
            return result !== Ternary.False;

            function reportError(message: DiagnosticMessage, arg0?: string, arg1?: string, arg2?: string): void {
                Debug.assert(!!errorNode);
                errorInfo = chainDiagnosticMessages(errorInfo, message, arg0, arg1, arg2);
            }

            function reportRelationError(message: DiagnosticMessage, source: Type, target: Type) {
                let sourceType = typeToString(source);
                let targetType = typeToString(target);
                if (sourceType === targetType) {
                    sourceType = typeToString(source, /*enclosingDeclaration*/ undefined, TypeFormatFlags.UseFullyQualifiedType);
                    targetType = typeToString(target, /*enclosingDeclaration*/ undefined, TypeFormatFlags.UseFullyQualifiedType);
                }

                if (!message) {
                    if (relation === comparableRelation) {
                        message = Diagnostics.Type_0_is_not_comparable_to_type_1;
                    }
                    else if (sourceType === targetType) {
                        message = Diagnostics.Type_0_is_not_assignable_to_type_1_Two_different_types_with_this_name_exist_but_they_are_unrelated;
                    }
                    else {
                        message = Diagnostics.Type_0_is_not_assignable_to_type_1;
                    }
                }

                reportError(message, sourceType, targetType);
            }

            function tryElaborateErrorsForPrimitivesAndObjects(source: Type, target: Type) {
                const sourceType = typeToString(source);
                const targetType = typeToString(target);

                if ((globalStringType === source && stringType === target) ||
                    (globalNumberType === source && numberType === target) ||
                    (globalBooleanType === source && booleanType === target) ||
                    (getGlobalESSymbolType(/*reportErrors*/ false) === source && esSymbolType === target)) {
                    reportError(Diagnostics._0_is_a_primitive_but_1_is_a_wrapper_object_Prefer_using_0_when_possible, targetType, sourceType);
                }
            }

            function isUnionOrIntersectionTypeWithoutNullableConstituents(type: Type): boolean {
                if (!(type.flags & TypeFlags.UnionOrIntersection)) {
                    return false;
                }
                // at this point we know that this is union or intersection type possibly with nullable constituents.
                // check if we still will have compound type if we ignore nullable components.
                let seenNonNullable = false;
                for (const t of (<UnionOrIntersectionType>type).types) {
                    if (t.flags & TypeFlags.Nullable) {
                        continue;
                    }
                    if (seenNonNullable) {
                        return true;
                    }
                    seenNonNullable = true;
                }
                return false;
            }

            /**
             * Compare two types and return
             * * Ternary.True if they are related with no assumptions,
             * * Ternary.Maybe if they are related with assumptions of other relationships, or
             * * Ternary.False if they are not related.
             */
            function isRelatedTo(source: Type, target: Type, reportErrors?: boolean, headMessage?: DiagnosticMessage): Ternary {
                if (source.flags & TypeFlags.StringOrNumberLiteral && source.flags & TypeFlags.FreshLiteral) {
                    source = (<LiteralType>source).regularType;
                }
                if (target.flags & TypeFlags.StringOrNumberLiteral && target.flags & TypeFlags.FreshLiteral) {
                    target = (<LiteralType>target).regularType;
                }
                if (source.flags & TypeFlags.Substitution) {
                    source = relation === definitelyAssignableRelation ? (<SubstitutionType>source).typeParameter : (<SubstitutionType>source).substitute;
                }
                if (target.flags & TypeFlags.Substitution) {
                    target = (<SubstitutionType>target).typeParameter;
                }

                // both types are the same - covers 'they are the same primitive type or both are Any' or the same type parameter cases
                if (source === target) return Ternary.True;

                if (relation === identityRelation) {
                    return isIdenticalTo(source, target);
                }

                if (relation === comparableRelation && !(target.flags & TypeFlags.Never) && isSimpleTypeRelatedTo(target, source, relation) ||
                    isSimpleTypeRelatedTo(source, target, relation, reportErrors ? reportError : undefined)) return Ternary.True;

                if (isObjectLiteralType(source) && source.flags & TypeFlags.FreshLiteral) {
                    const discriminantType = target.flags & TypeFlags.Union ? findMatchingDiscriminantType(source, target as UnionType) : undefined;
                    if (hasExcessProperties(<FreshObjectLiteralType>source, target, discriminantType, reportErrors)) {
                        if (reportErrors) {
                            reportRelationError(headMessage, source, target);
                        }
                        return Ternary.False;
                    }
                    // Above we check for excess properties with respect to the entire target type. When union
                    // and intersection types are further deconstructed on the target side, we don't want to
                    // make the check again (as it might fail for a partial target type). Therefore we obtain
                    // the regular source type and proceed with that.
                    if (isUnionOrIntersectionTypeWithoutNullableConstituents(target) && !discriminantType) {
                        source = getRegularTypeOfObjectLiteral(source);
                    }
                }

                if (relation !== comparableRelation &&
                    !(source.flags & TypeFlags.UnionOrIntersection) &&
                    !(target.flags & TypeFlags.Union) &&
                    !isIntersectionConstituent &&
                    source !== globalObjectType &&
                    (getPropertiesOfType(source).length > 0 || typeHasCallOrConstructSignatures(source)) &&
                    isWeakType(target) &&
                    !hasCommonProperties(source, target)) {
                    if (reportErrors) {
                        const calls = getSignaturesOfType(source, SignatureKind.Call);
                        const constructs = getSignaturesOfType(source, SignatureKind.Construct);
                        if (calls.length > 0 && isRelatedTo(getReturnTypeOfSignature(calls[0]), target, /*reportErrors*/ false) ||
                            constructs.length > 0 && isRelatedTo(getReturnTypeOfSignature(constructs[0]), target, /*reportErrors*/ false)) {
                            reportError(Diagnostics.Value_of_type_0_has_no_properties_in_common_with_type_1_Did_you_mean_to_call_it, typeToString(source), typeToString(target));
                        }
                        else {
                            reportError(Diagnostics.Type_0_has_no_properties_in_common_with_type_1, typeToString(source), typeToString(target));
                        }
                    }
                    return Ternary.False;
                }

                let result = Ternary.False;
                const saveErrorInfo = errorInfo;
                const saveIsIntersectionConstituent = isIntersectionConstituent;
                isIntersectionConstituent = false;

                // Note that these checks are specifically ordered to produce correct results. In particular,
                // we need to deconstruct unions before intersections (because unions are always at the top),
                // and we need to handle "each" relations before "some" relations for the same kind of type.
                if (source.flags & TypeFlags.Union) {
                    result = relation === comparableRelation ?
                        someTypeRelatedToType(source as UnionType, target, reportErrors && !(source.flags & TypeFlags.Primitive)) :
                        eachTypeRelatedToType(source as UnionType, target, reportErrors && !(source.flags & TypeFlags.Primitive));
                }
                else {
                    if (target.flags & TypeFlags.Union) {
                        result = typeRelatedToSomeType(source, <UnionType>target, reportErrors && !(source.flags & TypeFlags.Primitive) && !(target.flags & TypeFlags.Primitive));
                    }
                    else if (target.flags & TypeFlags.Intersection) {
                        isIntersectionConstituent = true;
                        result = typeRelatedToEachType(source, target as IntersectionType, reportErrors);
                    }
                    else if (source.flags & TypeFlags.Intersection) {
                        // Check to see if any constituents of the intersection are immediately related to the target.
                        //
                        // Don't report errors though. Checking whether a constituent is related to the source is not actually
                        // useful and leads to some confusing error messages. Instead it is better to let the below checks
                        // take care of this, or to not elaborate at all. For instance,
                        //
                        //    - For an object type (such as 'C = A & B'), users are usually more interested in structural errors.
                        //
                        //    - For a union type (such as '(A | B) = (C & D)'), it's better to hold onto the whole intersection
                        //          than to report that 'D' is not assignable to 'A' or 'B'.
                        //
                        //    - For a primitive type or type parameter (such as 'number = A & B') there is no point in
                        //          breaking the intersection apart.
                        result = someTypeRelatedToType(<IntersectionType>source, target, /*reportErrors*/ false);
                    }
                    if (!result && (source.flags & TypeFlags.StructuredOrInstantiable || target.flags & TypeFlags.StructuredOrInstantiable)) {
                        if (result = recursiveTypeRelatedTo(source, target, reportErrors)) {
                            errorInfo = saveErrorInfo;
                        }
                    }
                }

                isIntersectionConstituent = saveIsIntersectionConstituent;

                if (!result && reportErrors) {
                    if (source.flags & TypeFlags.Object && target.flags & TypeFlags.Primitive) {
                        tryElaborateErrorsForPrimitivesAndObjects(source, target);
                    }
                    else if (source.symbol && source.flags & TypeFlags.Object && globalObjectType === source) {
                        reportError(Diagnostics.The_Object_type_is_assignable_to_very_few_other_types_Did_you_mean_to_use_the_any_type_instead);
                    }
                    reportRelationError(headMessage, source, target);
                }
                return result;
            }

            function isIdenticalTo(source: Type, target: Type): Ternary {
                let result: Ternary;
                const flags = source.flags & target.flags;
                if (flags & TypeFlags.Object) {
                    return recursiveTypeRelatedTo(source, target, /*reportErrors*/ false);
                }
                if (flags & (TypeFlags.Union | TypeFlags.Intersection)) {
                    if (result = eachTypeRelatedToSomeType(<UnionOrIntersectionType>source, <UnionOrIntersectionType>target)) {
                        if (result &= eachTypeRelatedToSomeType(<UnionOrIntersectionType>target, <UnionOrIntersectionType>source)) {
                            return result;
                        }
                    }
                }
                if (flags & TypeFlags.Index) {
                    return isRelatedTo((<IndexType>source).type, (<IndexType>target).type, /*reportErrors*/ false);
                }
                if (flags & TypeFlags.IndexedAccess) {
                    if (result = isRelatedTo((<IndexedAccessType>source).objectType, (<IndexedAccessType>target).objectType, /*reportErrors*/ false)) {
                        if (result &= isRelatedTo((<IndexedAccessType>source).indexType, (<IndexedAccessType>target).indexType, /*reportErrors*/ false)) {
                            return result;
                        }
                    }
                }
                if (flags & TypeFlags.Conditional) {
                    if ((<ConditionalType>source).root.isDistributive === (<ConditionalType>target).root.isDistributive) {
                        if (result = isRelatedTo((<ConditionalType>source).checkType, (<ConditionalType>target).checkType, /*reportErrors*/ false)) {
                            if (result &= isRelatedTo((<ConditionalType>source).extendsType, (<ConditionalType>target).extendsType, /*reportErrors*/ false)) {
                                if (result &= isRelatedTo(getTrueTypeFromConditionalType(<ConditionalType>source), getTrueTypeFromConditionalType(<ConditionalType>target), /*reportErrors*/ false)) {
                                    if (result &= isRelatedTo(getFalseTypeFromConditionalType(<ConditionalType>source), getFalseTypeFromConditionalType(<ConditionalType>target), /*reportErrors*/ false)) {
                                        return result;
                                    }
                                }
                            }
                        }
                    }
                }
                if (flags & TypeFlags.Substitution) {
                    return isRelatedTo((<SubstitutionType>source).substitute, (<SubstitutionType>target).substitute, /*reportErrors*/ false);
                }
                return Ternary.False;
            }

            function hasExcessProperties(source: FreshObjectLiteralType, target: Type, discriminant: Type | undefined, reportErrors: boolean): boolean {
                if (maybeTypeOfKind(target, TypeFlags.Object) && !(getObjectFlags(target) & ObjectFlags.ObjectLiteralPatternWithComputedProperties)) {
                    const isComparingJsxAttributes = !!(getObjectFlags(source) & ObjectFlags.JsxAttributes);
                    if ((relation === assignableRelation || relation === definitelyAssignableRelation || relation === comparableRelation) &&
                        (isTypeSubsetOf(globalObjectType, target) || (!isComparingJsxAttributes && isEmptyObjectType(target)))) {
                        return false;
                    }
                    if (discriminant) {
                        // check excess properties against discriminant type only, not the entire union
                        return hasExcessProperties(source, discriminant, /*discriminant*/ undefined, reportErrors);
                    }
                    for (const prop of getPropertiesOfObjectType(source)) {
                        if (!isKnownProperty(target, prop.escapedName, isComparingJsxAttributes)) {
                            if (reportErrors) {
                                // We know *exactly* where things went wrong when comparing the types.
                                // Use this property as the error node as this will be more helpful in
                                // reasoning about what went wrong.
                                Debug.assert(!!errorNode);
                                if (isJsxAttributes(errorNode) || isJsxOpeningLikeElement(errorNode)) {
                                    // JsxAttributes has an object-literal flag and undergo same type-assignablity check as normal object-literal.
                                    // However, using an object-literal error message will be very confusing to the users so we give different a message.
                                    reportError(Diagnostics.Property_0_does_not_exist_on_type_1, symbolToString(prop), typeToString(target));
                                }
                                else {
                                    // use the property's value declaration if the property is assigned inside the literal itself
                                    const objectLiteralDeclaration = source.symbol && firstOrUndefined(source.symbol.declarations);
                                    let suggestion;
                                    if (prop.valueDeclaration && findAncestor(prop.valueDeclaration, d => d === objectLiteralDeclaration)) {
                                        const propDeclaration = prop.valueDeclaration as ObjectLiteralElementLike;
                                        Debug.assertNode(propDeclaration, isObjectLiteralElementLike);

                                        errorNode = propDeclaration;

                                        if (isIdentifier(propDeclaration.name)) {
                                            suggestion = getSuggestionForNonexistentProperty(propDeclaration.name, target);
                                        }
                                    }

                                    if (suggestion !== undefined) {
                                        reportError(Diagnostics.Object_literal_may_only_specify_known_properties_but_0_does_not_exist_in_type_1_Did_you_mean_to_write_2,
                                            symbolToString(prop), typeToString(target), suggestion);
                                    }
                                    else {
                                        reportError(Diagnostics.Object_literal_may_only_specify_known_properties_and_0_does_not_exist_in_type_1,
                                            symbolToString(prop), typeToString(target));
                                    }
                                }
                            }
                            return true;
                        }
                    }
                }
                return false;
            }

            function eachTypeRelatedToSomeType(source: UnionOrIntersectionType, target: UnionOrIntersectionType): Ternary {
                let result = Ternary.True;
                const sourceTypes = source.types;
                for (const sourceType of sourceTypes) {
                    const related = typeRelatedToSomeType(sourceType, target, /*reportErrors*/ false);
                    if (!related) {
                        return Ternary.False;
                    }
                    result &= related;
                }
                return result;
            }

            function typeRelatedToSomeType(source: Type, target: UnionOrIntersectionType, reportErrors: boolean): Ternary {
                const targetTypes = target.types;
                if (target.flags & TypeFlags.Union && containsType(targetTypes, source)) {
                    return Ternary.True;
                }
                for (const type of targetTypes) {
                    const related = isRelatedTo(source, type, /*reportErrors*/ false);
                    if (related) {
                        return related;
                    }
                }
                if (reportErrors) {
                    const discriminantType = findMatchingDiscriminantType(source, target);
                    isRelatedTo(source, discriminantType || targetTypes[targetTypes.length - 1], /*reportErrors*/ true);
                }
                return Ternary.False;
            }

            // Keep this up-to-date with the same logic within `getApparentTypeOfContextualType`, since they should behave similarly
            function findMatchingDiscriminantType(source: Type, target: UnionOrIntersectionType) {
                let match: Type;
                const sourceProperties = getPropertiesOfObjectType(source);
                if (sourceProperties) {
                    const sourcePropertiesFiltered = findDiscriminantProperties(sourceProperties, target);
                    if (sourcePropertiesFiltered) {
                        for (const sourceProperty of sourcePropertiesFiltered) {
                            const sourceType = getTypeOfSymbol(sourceProperty);
                            for (const type of target.types) {
                                const targetType = getTypeOfPropertyOfType(type, sourceProperty.escapedName);
                                if (targetType && isRelatedTo(sourceType, targetType)) {
                                    if (type === match) continue; // Finding multiple fields which discriminate to the same type is fine
                                    if (match) {
                                        return undefined;
                                    }
                                    match = type;
                                }
                            }
                        }
                    }
                }
                return match;
            }

            function typeRelatedToEachType(source: Type, target: IntersectionType, reportErrors: boolean): Ternary {
                let result = Ternary.True;
                const targetTypes = target.types;
                for (const targetType of targetTypes) {
                    const related = isRelatedTo(source, targetType, reportErrors);
                    if (!related) {
                        return Ternary.False;
                    }
                    result &= related;
                }
                return result;
            }

            function someTypeRelatedToType(source: UnionOrIntersectionType, target: Type, reportErrors: boolean): Ternary {
                const sourceTypes = source.types;
                if (source.flags & TypeFlags.Union && containsType(sourceTypes, target)) {
                    return Ternary.True;
                }
                const len = sourceTypes.length;
                for (let i = 0; i < len; i++) {
                    const related = isRelatedTo(sourceTypes[i], target, reportErrors && i === len - 1);
                    if (related) {
                        return related;
                    }
                }
                return Ternary.False;
            }

            function eachTypeRelatedToType(source: UnionOrIntersectionType, target: Type, reportErrors: boolean): Ternary {
                let result = Ternary.True;
                const sourceTypes = source.types;
                for (const sourceType of sourceTypes) {
                    const related = isRelatedTo(sourceType, target, reportErrors);
                    if (!related) {
                        return Ternary.False;
                    }
                    result &= related;
                }
                return result;
            }

            function typeArgumentsRelatedTo(source: TypeReference, target: TypeReference, variances: Variance[], reportErrors: boolean): Ternary {
                const sources = source.typeArguments || emptyArray;
                const targets = target.typeArguments || emptyArray;
                if (sources.length !== targets.length && relation === identityRelation) {
                    return Ternary.False;
                }
                const length = sources.length <= targets.length ? sources.length : targets.length;
                let result = Ternary.True;
                for (let i = 0; i < length; i++) {
                    // When variance information isn't available we default to covariance. This happens
                    // in the process of computing variance information for recursive types and when
                    // comparing 'this' type arguments.
                    const variance = i < variances.length ? variances[i] : Variance.Covariant;
                    // We ignore arguments for independent type parameters (because they're never witnessed).
                    if (variance !== Variance.Independent) {
                        const s = sources[i];
                        const t = targets[i];
                        let related = Ternary.True;
                        if (variance === Variance.Covariant) {
                            related = isRelatedTo(s, t, reportErrors);
                        }
                        else if (variance === Variance.Contravariant) {
                            related = isRelatedTo(t, s, reportErrors);
                        }
                        else if (variance === Variance.Bivariant) {
                            // In the bivariant case we first compare contravariantly without reporting
                            // errors. Then, if that doesn't succeed, we compare covariantly with error
                            // reporting. Thus, error elaboration will be based on the the covariant check,
                            // which is generally easier to reason about.
                            related = isRelatedTo(t, s, /*reportErrors*/ false);
                            if (!related) {
                                related = isRelatedTo(s, t, reportErrors);
                            }
                        }
                        else {
                            // In the invariant case we first compare covariantly, and only when that
                            // succeeds do we proceed to compare contravariantly. Thus, error elaboration
                            // will typically be based on the covariant check.
                            related = isRelatedTo(s, t, reportErrors);
                            if (related) {
                                related &= isRelatedTo(t, s, reportErrors);
                            }
                        }
                        if (!related) {
                            return Ternary.False;
                        }
                        result &= related;
                    }
                }
                return result;
            }

            // Determine if possibly recursive types are related. First, check if the result is already available in the global cache.
            // Second, check if we have already started a comparison of the given two types in which case we assume the result to be true.
            // Third, check if both types are part of deeply nested chains of generic type instantiations and if so assume the types are
            // equal and infinitely expanding. Fourth, if we have reached a depth of 100 nested comparisons, assume we have runaway recursion
            // and issue an error. Otherwise, actually compare the structure of the two types.
            function recursiveTypeRelatedTo(source: Type, target: Type, reportErrors: boolean): Ternary {
                if (overflow) {
                    return Ternary.False;
                }
                const id = getRelationKey(source, target, relation);
                const related = relation.get(id);
                if (related !== undefined) {
                    if (reportErrors && related === RelationComparisonResult.Failed) {
                        // We are elaborating errors and the cached result is an unreported failure. Record the result as a reported
                        // failure and continue computing the relation such that errors get reported.
                        relation.set(id, RelationComparisonResult.FailedAndReported);
                    }
                    else {
                        return related === RelationComparisonResult.Succeeded ? Ternary.True : Ternary.False;
                    }
                }
                if (!maybeKeys) {
                    maybeKeys = [];
                    sourceStack = [];
                    targetStack = [];
                }
                else {
                    for (let i = 0; i < maybeCount; i++) {
                        // If source and target are already being compared, consider them related with assumptions
                        if (id === maybeKeys[i]) {
                            return Ternary.Maybe;
                        }
                    }
                    if (depth === 100) {
                        overflow = true;
                        return Ternary.False;
                    }
                }
                const maybeStart = maybeCount;
                maybeKeys[maybeCount] = id;
                maybeCount++;
                sourceStack[depth] = source;
                targetStack[depth] = target;
                depth++;
                const saveExpandingFlags = expandingFlags;
                if (!(expandingFlags & ExpandingFlags.Source) && isDeeplyNestedType(source, sourceStack, depth)) expandingFlags |= ExpandingFlags.Source;
                if (!(expandingFlags & ExpandingFlags.Target) && isDeeplyNestedType(target, targetStack, depth)) expandingFlags |= ExpandingFlags.Target;
                const result = expandingFlags !== ExpandingFlags.Both ? structuredTypeRelatedTo(source, target, reportErrors) : Ternary.Maybe;
                expandingFlags = saveExpandingFlags;
                depth--;
                if (result) {
                    if (result === Ternary.True || depth === 0) {
                        // If result is definitely true, record all maybe keys as having succeeded
                        for (let i = maybeStart; i < maybeCount; i++) {
                            relation.set(maybeKeys[i], RelationComparisonResult.Succeeded);
                        }
                        maybeCount = maybeStart;
                    }
                }
                else {
                    // A false result goes straight into global cache (when something is false under
                    // assumptions it will also be false without assumptions)
                    relation.set(id, reportErrors ? RelationComparisonResult.FailedAndReported : RelationComparisonResult.Failed);
                    maybeCount = maybeStart;
                }
                return result;
            }

            function getConstraintForRelation(type: Type) {
                return relation === definitelyAssignableRelation ? undefined : getConstraintOfType(type);
            }

            function structuredTypeRelatedTo(source: Type, target: Type, reportErrors: boolean): Ternary {
                let result: Ternary;
                let originalErrorInfo: DiagnosticMessageChain;
                const saveErrorInfo = errorInfo;
                if (target.flags & TypeFlags.TypeParameter) {
                    // A source type { [P in keyof T]: X } is related to a target type T if X is related to T[P].
                    if (getObjectFlags(source) & ObjectFlags.Mapped && getConstraintTypeFromMappedType(<MappedType>source) === getIndexType(target)) {
                        if (!(getMappedTypeModifiers(<MappedType>source) & MappedTypeModifiers.IncludeOptional)) {
                            const templateType = getTemplateTypeFromMappedType(<MappedType>source);
                            const indexedAccessType = getIndexedAccessType(target, getTypeParameterFromMappedType(<MappedType>source));
                            if (result = isRelatedTo(templateType, indexedAccessType, reportErrors)) {
                                return result;
                            }
                        }
                    }
                }
                else if (target.flags & TypeFlags.Index) {
                    // A keyof S is related to a keyof T if T is related to S.
                    if (source.flags & TypeFlags.Index) {
                        if (result = isRelatedTo((<IndexType>target).type, (<IndexType>source).type, /*reportErrors*/ false)) {
                            return result;
                        }
                    }
                    // A type S is assignable to keyof T if S is assignable to keyof C, where C is the
                    // constraint of T.
                    const constraint = getConstraintForRelation((<IndexType>target).type);
                    if (constraint) {
                        if (result = isRelatedTo(source, getIndexType(constraint), reportErrors)) {
                            return result;
                        }
                    }
                }
                else if (target.flags & TypeFlags.IndexedAccess) {
                    // A type S is related to a type T[K] if S is related to A[K], where K is string-like and
                    // A is the apparent type of T.
                    const constraint = getConstraintForRelation(<IndexedAccessType>target);
                    if (constraint) {
                        if (result = isRelatedTo(source, constraint, reportErrors)) {
                            errorInfo = saveErrorInfo;
                            return result;
                        }
                    }
                }
                else if (isGenericMappedType(target)) {
                    // A source type T is related to a target type { [P in X]: T[P] }
                    const template = getTemplateTypeFromMappedType(target);
                    const modifiers = getMappedTypeModifiers(target);
                    if (!(modifiers & MappedTypeModifiers.ExcludeOptional)) {
                    if (template.flags & TypeFlags.IndexedAccess && (<IndexedAccessType>template).objectType === source &&
                        (<IndexedAccessType>template).indexType === getTypeParameterFromMappedType(target)) {
                        return Ternary.True;
                    }
                    // A source type T is related to a target type { [P in keyof T]: X } if T[P] is related to X.
                    if (!isGenericMappedType(source) && getConstraintTypeFromMappedType(target) === getIndexType(source)) {
                        const indexedAccessType = getIndexedAccessType(source, getTypeParameterFromMappedType(target));
                        const templateType = getTemplateTypeFromMappedType(target);
                        if (result = isRelatedTo(indexedAccessType, templateType, reportErrors)) {
                            errorInfo = saveErrorInfo;
                            return result;
                        }
                    }
                }
                }

                if (source.flags & TypeFlags.TypeParameter) {
                    let constraint = getConstraintForRelation(<TypeParameter>source);
                    // A type parameter with no constraint is not related to the non-primitive object type.
                    if (constraint || !(target.flags & TypeFlags.NonPrimitive)) {
                        if (!constraint || constraint.flags & TypeFlags.Any) {
                            constraint = emptyObjectType;
                        }
                        // Report constraint errors only if the constraint is not the empty object type
                        const reportConstraintErrors = reportErrors && constraint !== emptyObjectType;
                        if (result = isRelatedTo(constraint, target, reportConstraintErrors)) {
                            errorInfo = saveErrorInfo;
                            return result;
                        }
                    }
                }
                else if (source.flags & TypeFlags.IndexedAccess) {
                    // A type S[K] is related to a type T if A[K] is related to T, where K is string-like and
                    // A is the apparent type of S.
                    const constraint = getConstraintForRelation(<IndexedAccessType>source);
                    if (constraint) {
                        if (result = isRelatedTo(constraint, target, reportErrors)) {
                            errorInfo = saveErrorInfo;
                            return result;
                        }
                    }
                    else if (target.flags & TypeFlags.IndexedAccess) {
                        if (result = isRelatedTo((<IndexedAccessType>source).objectType, (<IndexedAccessType>target).objectType, reportErrors)) {
                            result &= isRelatedTo((<IndexedAccessType>source).indexType, (<IndexedAccessType>target).indexType, reportErrors);
                        }
                        if (result) {
                            errorInfo = saveErrorInfo;
                            return result;
                        }
                    }
                }
                else if (source.flags & TypeFlags.Conditional) {
                    if (target.flags & TypeFlags.Conditional) {
                        if (isTypeIdenticalTo((<ConditionalType>source).checkType, (<ConditionalType>target).checkType) &&
                            isTypeIdenticalTo((<ConditionalType>source).extendsType, (<ConditionalType>target).extendsType)) {
                            if (result = isRelatedTo(getTrueTypeFromConditionalType(<ConditionalType>source), getTrueTypeFromConditionalType(<ConditionalType>target), reportErrors)) {
                                result &= isRelatedTo(getFalseTypeFromConditionalType(<ConditionalType>source), getFalseTypeFromConditionalType(<ConditionalType>target), reportErrors);
                            }
                            if (result) {
                                errorInfo = saveErrorInfo;
                                return result;
                            }
                        }
                    }
                    else if (relation !== definitelyAssignableRelation) {
                        const distributiveConstraint = getConstraintOfDistributiveConditionalType(<ConditionalType>source);
                        if (distributiveConstraint) {
                            if (result = isRelatedTo(distributiveConstraint, target, reportErrors)) {
                                errorInfo = saveErrorInfo;
                                return result;
                            }
                        }
                        const defaultConstraint = getDefaultConstraintOfConditionalType(<ConditionalType>source);
                        if (defaultConstraint) {
                            if (result = isRelatedTo(defaultConstraint, target, reportErrors)) {
                                errorInfo = saveErrorInfo;
                                return result;
                            }
                        }
                    }
                }
                else {
                    if (getObjectFlags(source) & ObjectFlags.Reference && getObjectFlags(target) & ObjectFlags.Reference && (<TypeReference>source).target === (<TypeReference>target).target &&
                        !(getObjectFlags(source) & ObjectFlags.MarkerType || getObjectFlags(target) & ObjectFlags.MarkerType)) {
                        // We have type references to the same generic type, and the type references are not marker
                        // type references (which are intended by be compared structurally). Obtain the variance
                        // information for the type parameters and relate the type arguments accordingly.
                        const variances = getVariances((<TypeReference>source).target);
                        if (result = typeArgumentsRelatedTo(<TypeReference>source, <TypeReference>target, variances, reportErrors)) {
                            return result;
                        }
                        // The type arguments did not relate appropriately, but it may be because we have no variance
                        // information (in which case typeArgumentsRelatedTo defaulted to covariance for all type
                        // arguments). It might also be the case that the target type has a 'void' type argument for
                        // a covariant type parameter that is only used in return positions within the generic type
                        // (in which case any type argument is permitted on the source side). In those cases we proceed
                        // with a structural comparison. Otherwise, we know for certain the instantiations aren't
                        // related and we can return here.
                        if (variances !== emptyArray && !hasCovariantVoidArgument(<TypeReference>target, variances)) {
                            // In some cases generic types that are covariant in regular type checking mode become
                            // invariant in --strictFunctionTypes mode because one or more type parameters are used in
                            // both co- and contravariant positions. In order to make it easier to diagnose *why* such
                            // types are invariant, if any of the type parameters are invariant we reset the reported
                            // errors and instead force a structural comparison (which will include elaborations that
                            // reveal the reason).
                            if (!(reportErrors && some(variances, v => v === Variance.Invariant))) {
                                return Ternary.False;
                            }
                            // We remember the original error information so we can restore it in case the structural
                            // comparison unexpectedly succeeds. This can happen when the structural comparison result
                            // is a Ternary.Maybe for example caused by the recursion depth limiter.
                            originalErrorInfo = errorInfo;
                            errorInfo = saveErrorInfo;
                        }
                    }
                    // Even if relationship doesn't hold for unions, intersections, or generic type references,
                    // it may hold in a structural comparison.
                    const sourceIsPrimitive = !!(source.flags & TypeFlags.Primitive);
                    if (relation !== identityRelation) {
                        source = getApparentType(source);
                    }
                    // In a check of the form X = A & B, we will have previously checked if A relates to X or B relates
                    // to X. Failing both of those we want to check if the aggregation of A and B's members structurally
                    // relates to X. Thus, we include intersection types on the source side here.
                    if (source.flags & (TypeFlags.Object | TypeFlags.Intersection) && target.flags & TypeFlags.Object) {
                        // Report structural errors only if we haven't reported any errors yet
                        const reportStructuralErrors = reportErrors && errorInfo === saveErrorInfo && !sourceIsPrimitive;
                        // An empty object type is related to any mapped type that includes a '?' modifier.
                        if (isPartialMappedType(target) && !isGenericMappedType(source) && isEmptyObjectType(source)) {
                            result = Ternary.True;
                        }
                        else if (isGenericMappedType(target)) {
                            result = isGenericMappedType(source) ? mappedTypeRelatedTo(source, target, reportStructuralErrors) : Ternary.False;
                        }
                        else {
                            result = propertiesRelatedTo(source, target, reportStructuralErrors);
                            if (result) {
                                result &= signaturesRelatedTo(source, target, SignatureKind.Call, reportStructuralErrors);
                                if (result) {
                                    result &= signaturesRelatedTo(source, target, SignatureKind.Construct, reportStructuralErrors);
                                    if (result) {
                                        result &= indexTypesRelatedTo(source, target, IndexKind.String, sourceIsPrimitive, reportStructuralErrors);
                                        if (result) {
                                            result &= indexTypesRelatedTo(source, target, IndexKind.Number, sourceIsPrimitive, reportStructuralErrors);
                                        }
                                    }
                                }
                            }
                        }
                        if (result) {
                            if (!originalErrorInfo) {
                                errorInfo = saveErrorInfo;
                                return result;
                            }
                            errorInfo = originalErrorInfo;
                        }
                    }
                }
                return Ternary.False;
            }

            // A type [P in S]: X is related to a type [Q in T]: Y if T is related to S and X' is
            // related to Y, where X' is an instantiation of X in which P is replaced with Q. Notice
            // that S and T are contra-variant whereas X and Y are co-variant.
            function mappedTypeRelatedTo(source: MappedType, target: MappedType, reportErrors: boolean): Ternary {
                const modifiersRelated = relation === comparableRelation || (
                    relation === identityRelation ? getMappedTypeModifiers(source) === getMappedTypeModifiers(target) :
                        getCombinedMappedTypeOptionality(source) <= getCombinedMappedTypeOptionality(target));
                if (modifiersRelated) {
                    let result: Ternary;
                    if (result = isRelatedTo(getConstraintTypeFromMappedType(target), getConstraintTypeFromMappedType(source), reportErrors)) {
                        const mapper = createTypeMapper([getTypeParameterFromMappedType(source)], [getTypeParameterFromMappedType(target)]);
                        return result & isRelatedTo(instantiateType(getTemplateTypeFromMappedType(source), mapper), getTemplateTypeFromMappedType(target), reportErrors);
                    }
                }
                return Ternary.False;
            }

            function propertiesRelatedTo(source: Type, target: Type, reportErrors: boolean): Ternary {
                if (relation === identityRelation) {
                    return propertiesIdenticalTo(source, target);
                }
                const requireOptionalProperties = relation === subtypeRelation && !isObjectLiteralType(source) && !isEmptyArrayLiteralType(source);
                const unmatchedProperty = getUnmatchedProperty(source, target, requireOptionalProperties);
                if (unmatchedProperty) {
                    if (reportErrors) {
                        reportError(Diagnostics.Property_0_is_missing_in_type_1, symbolToString(unmatchedProperty), typeToString(source));
                    }
                    return Ternary.False;
                }
                if (isObjectLiteralType(target)) {
                    for (const sourceProp of getPropertiesOfType(source)) {
                        if (!getPropertyOfObjectType(target, sourceProp.escapedName)) {
                            const sourceType = getTypeOfSymbol(sourceProp);
                            if (!(sourceType === undefinedType || sourceType === undefinedWideningType)) {
                                if (reportErrors) {
                                    reportError(Diagnostics.Property_0_does_not_exist_on_type_1, symbolToString(sourceProp), typeToString(target));
                                }
                                return Ternary.False;
                            }
                        }
                    }
                }
                let result = Ternary.True;
                const properties = getPropertiesOfObjectType(target);
                for (const targetProp of properties) {
                    if (!(targetProp.flags & SymbolFlags.Prototype)) {
                        const sourceProp = getPropertyOfType(source, targetProp.escapedName);
                        if (sourceProp && sourceProp !== targetProp) {
                            if (isIgnoredJsxProperty(source, sourceProp, getTypeOfSymbol(targetProp))) {
                                continue;
                            }
                            const sourcePropFlags = getDeclarationModifierFlagsFromSymbol(sourceProp);
                            const targetPropFlags = getDeclarationModifierFlagsFromSymbol(targetProp);
                            if (sourcePropFlags & ModifierFlags.Private || targetPropFlags & ModifierFlags.Private) {
                                if (getCheckFlags(sourceProp) & CheckFlags.ContainsPrivate) {
                                    if (reportErrors) {
                                        reportError(Diagnostics.Property_0_has_conflicting_declarations_and_is_inaccessible_in_type_1, symbolToString(sourceProp), typeToString(source));
                                    }
                                    return Ternary.False;
                                }
                                if (sourceProp.valueDeclaration !== targetProp.valueDeclaration) {
                                    if (reportErrors) {
                                        if (sourcePropFlags & ModifierFlags.Private && targetPropFlags & ModifierFlags.Private) {
                                            reportError(Diagnostics.Types_have_separate_declarations_of_a_private_property_0, symbolToString(targetProp));
                                        }
                                        else {
                                            reportError(Diagnostics.Property_0_is_private_in_type_1_but_not_in_type_2, symbolToString(targetProp),
                                                typeToString(sourcePropFlags & ModifierFlags.Private ? source : target),
                                                typeToString(sourcePropFlags & ModifierFlags.Private ? target : source));
                                        }
                                    }
                                    return Ternary.False;
                                }
                            }
                            else if (targetPropFlags & ModifierFlags.Protected) {
                                if (!isValidOverrideOf(sourceProp, targetProp)) {
                                    if (reportErrors) {
                                        reportError(Diagnostics.Property_0_is_protected_but_type_1_is_not_a_class_derived_from_2, symbolToString(targetProp),
                                            typeToString(getDeclaringClass(sourceProp) || source), typeToString(getDeclaringClass(targetProp) || target));
                                    }
                                    return Ternary.False;
                                }
                            }
                            else if (sourcePropFlags & ModifierFlags.Protected) {
                                if (reportErrors) {
                                    reportError(Diagnostics.Property_0_is_protected_in_type_1_but_public_in_type_2,
                                        symbolToString(targetProp), typeToString(source), typeToString(target));
                                }
                                return Ternary.False;
                            }
                            const related = isRelatedTo(getTypeOfSymbol(sourceProp), getTypeOfSymbol(targetProp), reportErrors);
                            if (!related) {
                                if (reportErrors) {
                                    reportError(Diagnostics.Types_of_property_0_are_incompatible, symbolToString(targetProp));
                                }
                                return Ternary.False;
                            }
                            result &= related;
                            // When checking for comparability, be more lenient with optional properties.
                            if (relation !== comparableRelation && sourceProp.flags & SymbolFlags.Optional && !(targetProp.flags & SymbolFlags.Optional)) {
                                // TypeScript 1.0 spec (April 2014): 3.8.3
                                // S is a subtype of a type T, and T is a supertype of S if ...
                                // S' and T are object types and, for each member M in T..
                                // M is a property and S' contains a property N where
                                // if M is a required property, N is also a required property
                                // (M - property in T)
                                // (N - property in S)
                                if (reportErrors) {
                                    reportError(Diagnostics.Property_0_is_optional_in_type_1_but_required_in_type_2,
                                        symbolToString(targetProp), typeToString(source), typeToString(target));
                                }
                                return Ternary.False;
                            }
                        }
                    }
                }
                return result;
            }

            /**
             * A type is 'weak' if it is an object type with at least one optional property
             * and no required properties, call/construct signatures or index signatures
             */
            function isWeakType(type: Type): boolean {
                if (type.flags & TypeFlags.Object) {
                    const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                    return resolved.callSignatures.length === 0 && resolved.constructSignatures.length === 0 &&
                        !resolved.stringIndexInfo && !resolved.numberIndexInfo &&
                        resolved.properties.length > 0 &&
                        every(resolved.properties, p => !!(p.flags & SymbolFlags.Optional));
                }
                if (type.flags & TypeFlags.Intersection) {
                    return every((<IntersectionType>type).types, isWeakType);
                }
                return false;
            }

            function hasCommonProperties(source: Type, target: Type) {
                const isComparingJsxAttributes = !!(getObjectFlags(source) & ObjectFlags.JsxAttributes);
                for (const prop of getPropertiesOfType(source)) {
                    if (isKnownProperty(target, prop.escapedName, isComparingJsxAttributes)) {
                        return true;
                    }
                }
                return false;
            }

            function propertiesIdenticalTo(source: Type, target: Type): Ternary {
                if (!(source.flags & TypeFlags.Object && target.flags & TypeFlags.Object)) {
                    return Ternary.False;
                }
                const sourceProperties = getPropertiesOfObjectType(source);
                const targetProperties = getPropertiesOfObjectType(target);
                if (sourceProperties.length !== targetProperties.length) {
                    return Ternary.False;
                }
                let result = Ternary.True;
                for (const sourceProp of sourceProperties) {
                    const targetProp = getPropertyOfObjectType(target, sourceProp.escapedName);
                    if (!targetProp) {
                        return Ternary.False;
                    }
                    const related = compareProperties(sourceProp, targetProp, isRelatedTo);
                    if (!related) {
                        return Ternary.False;
                    }
                    result &= related;
                }
                return result;
            }

            function signaturesRelatedTo(source: Type, target: Type, kind: SignatureKind, reportErrors: boolean): Ternary {
                if (relation === identityRelation) {
                    return signaturesIdenticalTo(source, target, kind);
                }
                if (target === anyFunctionType || source === anyFunctionType) {
                    return Ternary.True;
                }

                const sourceSignatures = getSignaturesOfType(source, kind);
                const targetSignatures = getSignaturesOfType(target, kind);
                if (kind === SignatureKind.Construct && sourceSignatures.length && targetSignatures.length) {
                    if (isAbstractConstructorType(source) && !isAbstractConstructorType(target)) {
                        // An abstract constructor type is not assignable to a non-abstract constructor type
                        // as it would otherwise be possible to new an abstract class. Note that the assignability
                        // check we perform for an extends clause excludes construct signatures from the target,
                        // so this check never proceeds.
                        if (reportErrors) {
                            reportError(Diagnostics.Cannot_assign_an_abstract_constructor_type_to_a_non_abstract_constructor_type);
                        }
                        return Ternary.False;
                    }
                    if (!constructorVisibilitiesAreCompatible(sourceSignatures[0], targetSignatures[0], reportErrors)) {
                        return Ternary.False;
                    }
                }

                let result = Ternary.True;
                const saveErrorInfo = errorInfo;

                if (getObjectFlags(source) & ObjectFlags.Instantiated && getObjectFlags(target) & ObjectFlags.Instantiated && source.symbol === target.symbol) {
                    // We have instantiations of the same anonymous type (which typically will be the type of a
                    // method). Simply do a pairwise comparison of the signatures in the two signature lists instead
                    // of the much more expensive N * M comparison matrix we explore below. We erase type parameters
                    // as they are known to always be the same.
                    for (let i = 0; i < targetSignatures.length; i++) {
                        const related = signatureRelatedTo(sourceSignatures[i], targetSignatures[i], /*erase*/ true, reportErrors);
                        if (!related) {
                            return Ternary.False;
                        }
                        result &= related;
                    }
                }
                else if (sourceSignatures.length === 1 && targetSignatures.length === 1) {
                    // For simple functions (functions with a single signature) we only erase type parameters for
                    // the comparable relation. Otherwise, if the source signature is generic, we instantiate it
                    // in the context of the target signature before checking the relationship. Ideally we'd do
                    // this regardless of the number of signatures, but the potential costs are prohibitive due
                    // to the quadratic nature of the logic below.
                    const eraseGenerics = relation === comparableRelation || compilerOptions.noStrictGenericChecks;
                    result = signatureRelatedTo(sourceSignatures[0], targetSignatures[0], eraseGenerics, reportErrors);
                }
                else {
                    outer: for (const t of targetSignatures) {
                        // Only elaborate errors from the first failure
                        let shouldElaborateErrors = reportErrors;
                        for (const s of sourceSignatures) {
                            const related = signatureRelatedTo(s, t, /*erase*/ true, shouldElaborateErrors);
                            if (related) {
                                result &= related;
                                errorInfo = saveErrorInfo;
                                continue outer;
                            }
                            shouldElaborateErrors = false;
                        }

                        if (shouldElaborateErrors) {
                            reportError(Diagnostics.Type_0_provides_no_match_for_the_signature_1,
                                typeToString(source),
                                signatureToString(t, /*enclosingDeclaration*/ undefined, /*flags*/ undefined, kind));
                        }
                        return Ternary.False;
                    }
                }
                return result;
            }

            /**
             * See signatureAssignableTo, compareSignaturesIdentical
             */
            function signatureRelatedTo(source: Signature, target: Signature, erase: boolean, reportErrors: boolean): Ternary {
                return compareSignaturesRelated(erase ? getErasedSignature(source) : source, erase ? getErasedSignature(target) : target,
                    CallbackCheck.None, /*ignoreReturnTypes*/ false, reportErrors, reportError, isRelatedTo);
            }

            function signaturesIdenticalTo(source: Type, target: Type, kind: SignatureKind): Ternary {
                const sourceSignatures = getSignaturesOfType(source, kind);
                const targetSignatures = getSignaturesOfType(target, kind);
                if (sourceSignatures.length !== targetSignatures.length) {
                    return Ternary.False;
                }
                let result = Ternary.True;
                for (let i = 0; i < sourceSignatures.length; i++) {
                    const related = compareSignaturesIdentical(sourceSignatures[i], targetSignatures[i], /*partialMatch*/ false, /*ignoreThisTypes*/ false, /*ignoreReturnTypes*/ false, isRelatedTo);
                    if (!related) {
                        return Ternary.False;
                    }
                    result &= related;
                }
                return result;
            }

            function eachPropertyRelatedTo(source: Type, target: Type, kind: IndexKind, reportErrors: boolean): Ternary {
                let result = Ternary.True;
                for (const prop of getPropertiesOfObjectType(source)) {
                    if (isIgnoredJsxProperty(source, prop, /*targetMemberType*/ undefined)) {
                        continue;
                    }
                    // Skip over symbol-named members
                    const nameType = getLiteralTypeFromPropertyName(prop);
                    if (nameType !== undefined && !(isRelatedTo(nameType, stringType) || isRelatedTo(nameType, numberType))) {
                        continue;
                    }
                    if (kind === IndexKind.String || isNumericLiteralName(prop.escapedName)) {
                        const related = isRelatedTo(getTypeOfSymbol(prop), target, reportErrors);
                        if (!related) {
                            if (reportErrors) {
                                reportError(Diagnostics.Property_0_is_incompatible_with_index_signature, symbolToString(prop));
                            }
                            return Ternary.False;
                        }
                        result &= related;
                    }
                }
                return result;
            }

            function indexInfoRelatedTo(sourceInfo: IndexInfo, targetInfo: IndexInfo, reportErrors: boolean) {
                const related = isRelatedTo(sourceInfo.type, targetInfo.type, reportErrors);
                if (!related && reportErrors) {
                    reportError(Diagnostics.Index_signatures_are_incompatible);
                }
                return related;
            }

            function indexTypesRelatedTo(source: Type, target: Type, kind: IndexKind, sourceIsPrimitive: boolean, reportErrors: boolean) {
                if (relation === identityRelation) {
                    return indexTypesIdenticalTo(source, target, kind);
                }
                const targetInfo = getIndexInfoOfType(target, kind);
                if (!targetInfo || targetInfo.type.flags & TypeFlags.Any && !sourceIsPrimitive) {
                    // Index signature of type any permits assignment from everything but primitives
                    return Ternary.True;
                }
                const sourceInfo = getIndexInfoOfType(source, kind) ||
                    kind === IndexKind.Number && getIndexInfoOfType(source, IndexKind.String);
                if (sourceInfo) {
                    return indexInfoRelatedTo(sourceInfo, targetInfo, reportErrors);
                }
                if (isGenericMappedType(source)) {
                    // A generic mapped type { [P in K]: T } is related to an index signature { [x: string]: U }
                    // if T is related to U.
                    return kind === IndexKind.String && isRelatedTo(getTemplateTypeFromMappedType(source), targetInfo.type, reportErrors);
                }
                if (isObjectTypeWithInferableIndex(source)) {
                    let related = Ternary.True;
                    if (kind === IndexKind.String) {
                        const sourceNumberInfo = getIndexInfoOfType(source, IndexKind.Number);
                        if (sourceNumberInfo) {
                            related = indexInfoRelatedTo(sourceNumberInfo, targetInfo, reportErrors);
                        }
                    }
                    if (related) {
                        related &= eachPropertyRelatedTo(source, targetInfo.type, kind, reportErrors);
                    }
                    return related;
                }
                if (reportErrors) {
                    reportError(Diagnostics.Index_signature_is_missing_in_type_0, typeToString(source));
                }
                return Ternary.False;
            }

            function indexTypesIdenticalTo(source: Type, target: Type, indexKind: IndexKind): Ternary {
                const targetInfo = getIndexInfoOfType(target, indexKind);
                const sourceInfo = getIndexInfoOfType(source, indexKind);
                if (!sourceInfo && !targetInfo) {
                    return Ternary.True;
                }
                if (sourceInfo && targetInfo && sourceInfo.isReadonly === targetInfo.isReadonly) {
                    return isRelatedTo(sourceInfo.type, targetInfo.type);
                }
                return Ternary.False;
            }

            function constructorVisibilitiesAreCompatible(sourceSignature: Signature, targetSignature: Signature, reportErrors: boolean) {
                if (!sourceSignature.declaration || !targetSignature.declaration) {
                    return true;
                }

                const sourceAccessibility = getSelectedModifierFlags(sourceSignature.declaration, ModifierFlags.NonPublicAccessibilityModifier);
                const targetAccessibility = getSelectedModifierFlags(targetSignature.declaration, ModifierFlags.NonPublicAccessibilityModifier);

                // A public, protected and private signature is assignable to a private signature.
                if (targetAccessibility === ModifierFlags.Private) {
                    return true;
                }

                // A public and protected signature is assignable to a protected signature.
                if (targetAccessibility === ModifierFlags.Protected && sourceAccessibility !== ModifierFlags.Private) {
                    return true;
                }

                // Only a public signature is assignable to public signature.
                if (targetAccessibility !== ModifierFlags.Protected && !sourceAccessibility) {
                    return true;
                }

                if (reportErrors) {
                    reportError(Diagnostics.Cannot_assign_a_0_constructor_type_to_a_1_constructor_type, visibilityToString(sourceAccessibility), visibilityToString(targetAccessibility));
                }

                return false;
            }
        }

        // Return a type reference where the source type parameter is replaced with the target marker
        // type, and flag the result as a marker type reference.
        function getMarkerTypeReference(type: GenericType, source: TypeParameter, target: Type) {
            const result = createTypeReference(type, map(type.typeParameters, t => t === source ? target : t));
            result.objectFlags |= ObjectFlags.MarkerType;
            return result;
        }

        // Return an array containing the variance of each type parameter. The variance is effectively
        // a digest of the type comparisons that occur for each type argument when instantiations of the
        // generic type are structurally compared. We infer the variance information by comparing
        // instantiations of the generic type for type arguments with known relations. The function
        // returns the emptyArray singleton if we're not in strictFunctionTypes mode or if the function
        // has been invoked recursively for the given generic type.
        function getVariances(type: GenericType): Variance[] {
            if (!strictFunctionTypes) {
                return emptyArray;
            }
            const typeParameters = type.typeParameters || emptyArray;
            let variances = type.variances;
            if (!variances) {
                if (type === globalArrayType || type === globalReadonlyArrayType) {
                    // Arrays are known to be covariant, no need to spend time computing this
                    variances = [Variance.Covariant];
                }
                else {
                    // The emptyArray singleton is used to signal a recursive invocation.
                    type.variances = emptyArray;
                    variances = [];
                    for (const tp of typeParameters) {
                        // We first compare instantiations where the type parameter is replaced with
                        // marker types that have a known subtype relationship. From this we can infer
                        // invariance, covariance, contravariance or bivariance.
                        const typeWithSuper = getMarkerTypeReference(type, tp, markerSuperType);
                        const typeWithSub = getMarkerTypeReference(type, tp, markerSubType);
                        let variance = (isTypeAssignableTo(typeWithSub, typeWithSuper) ? Variance.Covariant : 0) |
                            (isTypeAssignableTo(typeWithSuper, typeWithSub) ? Variance.Contravariant : 0);
                        // If the instantiations appear to be related bivariantly it may be because the
                        // type parameter is independent (i.e. it isn't witnessed anywhere in the generic
                        // type). To determine this we compare instantiations where the type parameter is
                        // replaced with marker types that are known to be unrelated.
                        if (variance === Variance.Bivariant && isTypeAssignableTo(getMarkerTypeReference(type, tp, markerOtherType), typeWithSuper)) {
                            variance = Variance.Independent;
                        }
                        variances.push(variance);
                    }
                }
                type.variances = variances;
            }
            return variances;
        }

        // Return true if the given type reference has a 'void' type argument for a covariant type parameter.
        // See comment at call in recursiveTypeRelatedTo for when this case matters.
        function hasCovariantVoidArgument(type: TypeReference, variances: Variance[]): boolean {
            for (let i = 0; i < variances.length; i++) {
                if (variances[i] === Variance.Covariant && type.typeArguments[i].flags & TypeFlags.Void) {
                    return true;
                }
            }
            return false;
        }

        function isUnconstrainedTypeParameter(type: Type) {
            return type.flags & TypeFlags.TypeParameter && !getConstraintFromTypeParameter(<TypeParameter>type);
        }

        function isTypeReferenceWithGenericArguments(type: Type): boolean {
            return getObjectFlags(type) & ObjectFlags.Reference && some((<TypeReference>type).typeArguments, t => isUnconstrainedTypeParameter(t) || isTypeReferenceWithGenericArguments(t));
        }

        /**
         * getTypeReferenceId(A<T, number, U>) returns "111=0-12=1"
         *   where A.id=111 and number.id=12
         */
        function getTypeReferenceId(type: TypeReference, typeParameters: Type[], depth = 0) {
            let result = "" + type.target.id;
            for (const t of type.typeArguments) {
                if (isUnconstrainedTypeParameter(t)) {
                    let index = typeParameters.indexOf(t);
                    if (index < 0) {
                        index = typeParameters.length;
                        typeParameters.push(t);
                    }
                    result += "=" + index;
                }
                else if (depth < 4 && isTypeReferenceWithGenericArguments(t)) {
                    result += "<" + getTypeReferenceId(t as TypeReference, typeParameters, depth + 1) + ">";
                }
                else {
                    result += "-" + t.id;
                }
            }
            return result;
        }

        /**
         * To improve caching, the relation key for two generic types uses the target's id plus ids of the type parameters.
         * For other cases, the types ids are used.
         */
        function getRelationKey(source: Type, target: Type, relation: Map<RelationComparisonResult>) {
            if (relation === identityRelation && source.id > target.id) {
                const temp = source;
                source = target;
                target = temp;
            }
            if (isTypeReferenceWithGenericArguments(source) && isTypeReferenceWithGenericArguments(target)) {
                const typeParameters: Type[] = [];
                return getTypeReferenceId(<TypeReference>source, typeParameters) + "," + getTypeReferenceId(<TypeReference>target, typeParameters);
            }
            return source.id + "," + target.id;
        }

        // Invoke the callback for each underlying property symbol of the given symbol and return the first
        // value that isn't undefined.
        function forEachProperty<T>(prop: Symbol, callback: (p: Symbol) => T): T {
            if (getCheckFlags(prop) & CheckFlags.Synthetic) {
                for (const t of (<TransientSymbol>prop).containingType.types) {
                    const p = getPropertyOfType(t, prop.escapedName);
                    const result = p && forEachProperty(p, callback);
                    if (result) {
                        return result;
                    }
                }
                return undefined;
            }
            return callback(prop);
        }

        // Return the declaring class type of a property or undefined if property not declared in class
        function getDeclaringClass(prop: Symbol) {
            return prop.parent && prop.parent.flags & SymbolFlags.Class ? getDeclaredTypeOfSymbol(getParentOfSymbol(prop)) : undefined;
        }

        // Return true if some underlying source property is declared in a class that derives
        // from the given base class.
        function isPropertyInClassDerivedFrom(prop: Symbol, baseClass: Type) {
            return forEachProperty(prop, sp => {
                const sourceClass = getDeclaringClass(sp);
                return sourceClass ? hasBaseType(sourceClass, baseClass) : false;
            });
        }

        // Return true if source property is a valid override of protected parts of target property.
        function isValidOverrideOf(sourceProp: Symbol, targetProp: Symbol) {
            return !forEachProperty(targetProp, tp => getDeclarationModifierFlagsFromSymbol(tp) & ModifierFlags.Protected ?
                !isPropertyInClassDerivedFrom(sourceProp, getDeclaringClass(tp)) : false);
        }

        // Return true if the given class derives from each of the declaring classes of the protected
        // constituents of the given property.
        function isClassDerivedFromDeclaringClasses(checkClass: Type, prop: Symbol) {
            return forEachProperty(prop, p => getDeclarationModifierFlagsFromSymbol(p) & ModifierFlags.Protected ?
                !hasBaseType(checkClass, getDeclaringClass(p)) : false) ? undefined : checkClass;
        }

        // Return true if the given type is deeply nested. We consider this to be the case when structural type comparisons
        // for 5 or more occurrences or instantiations of the type have been recorded on the given stack. It is possible,
        // though highly unlikely, for this test to be true in a situation where a chain of instantiations is not infinitely
        // expanding. Effectively, we will generate a false positive when two types are structurally equal to at least 5
        // levels, but unequal at some level beyond that.
        function isDeeplyNestedType(type: Type, stack: Type[], depth: number): boolean {
            // We track all object types that have an associated symbol (representing the origin of the type)
            if (depth >= 5 && type.flags & TypeFlags.Object) {
                const symbol = type.symbol;
                if (symbol) {
                    let count = 0;
                    for (let i = 0; i < depth; i++) {
                        const t = stack[i];
                        if (t.flags & TypeFlags.Object && t.symbol === symbol) {
                            count++;
                            if (count >= 5) return true;
                        }
                    }
                }
            }
            return false;
        }

        function isPropertyIdenticalTo(sourceProp: Symbol, targetProp: Symbol): boolean {
            return compareProperties(sourceProp, targetProp, compareTypesIdentical) !== Ternary.False;
        }

        function compareProperties(sourceProp: Symbol, targetProp: Symbol, compareTypes: (source: Type, target: Type) => Ternary): Ternary {
            // Two members are considered identical when
            // - they are public properties with identical names, optionality, and types,
            // - they are private or protected properties originating in the same declaration and having identical types
            if (sourceProp === targetProp) {
                return Ternary.True;
            }
            const sourcePropAccessibility = getDeclarationModifierFlagsFromSymbol(sourceProp) & ModifierFlags.NonPublicAccessibilityModifier;
            const targetPropAccessibility = getDeclarationModifierFlagsFromSymbol(targetProp) & ModifierFlags.NonPublicAccessibilityModifier;
            if (sourcePropAccessibility !== targetPropAccessibility) {
                return Ternary.False;
            }
            if (sourcePropAccessibility) {
                if (getTargetSymbol(sourceProp) !== getTargetSymbol(targetProp)) {
                    return Ternary.False;
                }
            }
            else {
                if ((sourceProp.flags & SymbolFlags.Optional) !== (targetProp.flags & SymbolFlags.Optional)) {
                    return Ternary.False;
                }
            }
            if (isReadonlySymbol(sourceProp) !== isReadonlySymbol(targetProp)) {
                return Ternary.False;
            }
            return compareTypes(getTypeOfSymbol(sourceProp), getTypeOfSymbol(targetProp));
        }

        function isMatchingSignature(source: Signature, target: Signature, partialMatch: boolean) {
            // A source signature matches a target signature if the two signatures have the same number of required,
            // optional, and rest parameters.
            if (source.parameters.length === target.parameters.length &&
                source.minArgumentCount === target.minArgumentCount &&
                source.hasRestParameter === target.hasRestParameter) {
                return true;
            }
            // A source signature partially matches a target signature if the target signature has no fewer required
            // parameters and no more overall parameters than the source signature (where a signature with a rest
            // parameter is always considered to have more overall parameters than one without).
            const sourceRestCount = source.hasRestParameter ? 1 : 0;
            const targetRestCount = target.hasRestParameter ? 1 : 0;
            if (partialMatch && source.minArgumentCount <= target.minArgumentCount && (
                sourceRestCount > targetRestCount ||
                sourceRestCount === targetRestCount && source.parameters.length >= target.parameters.length)) {
                return true;
            }
            return false;
        }

        /**
         * See signatureRelatedTo, compareSignaturesIdentical
         */
        function compareSignaturesIdentical(source: Signature, target: Signature, partialMatch: boolean, ignoreThisTypes: boolean, ignoreReturnTypes: boolean, compareTypes: (s: Type, t: Type) => Ternary): Ternary {
            // TODO (drosen): De-duplicate code between related functions.
            if (source === target) {
                return Ternary.True;
            }
            if (!(isMatchingSignature(source, target, partialMatch))) {
                return Ternary.False;
            }
            // Check that the two signatures have the same number of type parameters. We might consider
            // also checking that any type parameter constraints match, but that would require instantiating
            // the constraints with a common set of type arguments to get relatable entities in places where
            // type parameters occur in the constraints. The complexity of doing that doesn't seem worthwhile,
            // particularly as we're comparing erased versions of the signatures below.
            if (length(source.typeParameters) !== length(target.typeParameters)) {
                return Ternary.False;
            }
            // Spec 1.0 Section 3.8.3 & 3.8.4:
            // M and N (the signatures) are instantiated using type Any as the type argument for all type parameters declared by M and N
            source = getErasedSignature(source);
            target = getErasedSignature(target);
            let result = Ternary.True;

            if (!ignoreThisTypes) {
                const sourceThisType = getThisTypeOfSignature(source);
                if (sourceThisType) {
                    const targetThisType = getThisTypeOfSignature(target);
                    if (targetThisType) {
                        const related = compareTypes(sourceThisType, targetThisType);
                        if (!related) {
                            return Ternary.False;
                        }
                        result &= related;
                    }
                }
            }

            const targetLen = target.parameters.length;
            for (let i = 0; i < targetLen; i++) {
                const s = isRestParameterIndex(source, i) ? getRestTypeOfSignature(source) : getTypeOfParameter(source.parameters[i]);
                const t = isRestParameterIndex(target, i) ? getRestTypeOfSignature(target) : getTypeOfParameter(target.parameters[i]);
                const related = compareTypes(s, t);
                if (!related) {
                    return Ternary.False;
                }
                result &= related;
            }
            if (!ignoreReturnTypes) {
                const sourceTypePredicate = getTypePredicateOfSignature(source);
                const targetTypePredicate = getTypePredicateOfSignature(target);
                result &= sourceTypePredicate !== undefined || targetTypePredicate !== undefined
                    ? compareTypePredicatesIdentical(sourceTypePredicate, targetTypePredicate, compareTypes)
                    // If they're both type predicates their return types will both be `boolean`, so no need to compare those.
                    : compareTypes(getReturnTypeOfSignature(source), getReturnTypeOfSignature(target));
            }
            return result;
        }

        function compareTypePredicatesIdentical(source: TypePredicate | undefined, target: TypePredicate | undefined, compareTypes: (s: Type, t: Type) => Ternary): Ternary {
            return source === undefined || target === undefined || !typePredicateKindsMatch(source, target) ? Ternary.False : compareTypes(source.type, target.type);
        }

        function isRestParameterIndex(signature: Signature, parameterIndex: number) {
            return signature.hasRestParameter && parameterIndex >= signature.parameters.length - 1;
        }

        function literalTypesWithSameBaseType(types: Type[]): boolean {
            let commonBaseType: Type;
            for (const t of types) {
                const baseType = getBaseTypeOfLiteralType(t);
                if (!commonBaseType) {
                    commonBaseType = baseType;
                }
                if (baseType === t || baseType !== commonBaseType) {
                    return false;
                }
            }
            return true;
        }

        // When the candidate types are all literal types with the same base type, return a union
        // of those literal types. Otherwise, return the leftmost type for which no type to the
        // right is a supertype.
        function getSupertypeOrUnion(types: Type[]): Type {
            return literalTypesWithSameBaseType(types) ?
                getUnionType(types) :
                reduceLeft(types, (s, t) => isTypeSubtypeOf(s, t) ? t : s);
        }

        function getCommonSupertype(types: Type[]): Type {
            if (!strictNullChecks) {
                return getSupertypeOrUnion(types);
            }
            const primaryTypes = filter(types, t => !(t.flags & TypeFlags.Nullable));
            return primaryTypes.length ?
                getNullableType(getSupertypeOrUnion(primaryTypes), getFalsyFlagsOfTypes(types) & TypeFlags.Nullable) :
                getUnionType(types, UnionReduction.Subtype);
        }

        // Return the leftmost type for which no type to the right is a subtype.
        function getCommonSubtype(types: Type[]) {
            return reduceLeft(types, (s, t) => isTypeSubtypeOf(t, s) ? t : s);
        }

        function isArrayType(type: Type): boolean {
            return getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).target === globalArrayType;
        }

        function isArrayLikeType(type: Type): boolean {
            // A type is array-like if it is a reference to the global Array or global ReadonlyArray type,
            // or if it is not the undefined or null type and if it is assignable to ReadonlyArray<any>
            return getObjectFlags(type) & ObjectFlags.Reference && ((<TypeReference>type).target === globalArrayType || (<TypeReference>type).target === globalReadonlyArrayType) ||
                !(type.flags & TypeFlags.Nullable) && isTypeAssignableTo(type, anyReadonlyArrayType);
        }

        function isEmptyArrayLiteralType(type: Type): boolean {
            const elementType = isArrayType(type) ? (<TypeReference>type).typeArguments[0] : undefined;
            return elementType === undefinedWideningType || elementType === implicitNeverType;
        }

        function isTupleLikeType(type: Type): boolean {
            return !!getPropertyOfType(type, "0" as __String);
        }

        function isUnitType(type: Type): boolean {
            return !!(type.flags & TypeFlags.Unit);
        }

        function isLiteralType(type: Type): boolean {
            return type.flags & TypeFlags.Boolean ? true :
                type.flags & TypeFlags.Union ? type.flags & TypeFlags.EnumLiteral ? true : !forEach((<UnionType>type).types, t => !isUnitType(t)) :
                isUnitType(type);
        }

        function getBaseTypeOfLiteralType(type: Type): Type {
            return type.flags & TypeFlags.EnumLiteral ? getBaseTypeOfEnumLiteralType(<LiteralType>type) :
                type.flags & TypeFlags.StringLiteral ? stringType :
                type.flags & TypeFlags.NumberLiteral ? numberType :
                type.flags & TypeFlags.BooleanLiteral ? booleanType :
                type.flags & TypeFlags.Union ? getUnionType(sameMap((<UnionType>type).types, getBaseTypeOfLiteralType)) :
                type;
        }

        function getWidenedLiteralType(type: Type): Type {
            return type.flags & TypeFlags.EnumLiteral ? getBaseTypeOfEnumLiteralType(<LiteralType>type) :
                type.flags & TypeFlags.StringLiteral && type.flags & TypeFlags.FreshLiteral ? stringType :
                type.flags & TypeFlags.NumberLiteral && type.flags & TypeFlags.FreshLiteral ? numberType :
                type.flags & TypeFlags.BooleanLiteral ? booleanType :
                type.flags & TypeFlags.Union ? getUnionType(sameMap((<UnionType>type).types, getWidenedLiteralType)) :
                type;
        }

        function getWidenedUniqueESSymbolType(type: Type): Type {
            return type.flags & TypeFlags.UniqueESSymbol ? esSymbolType :
                type.flags & TypeFlags.Union ? getUnionType(sameMap((<UnionType>type).types, getWidenedUniqueESSymbolType)) :
                type;
        }

        function getWidenedLiteralLikeTypeForContextualType(type: Type, contextualType: Type) {
            if (!isLiteralOfContextualType(type, contextualType)) {
                type = getWidenedUniqueESSymbolType(getWidenedLiteralType(type));
            }
            return type;
        }

        /**
         * Check if a Type was written as a tuple type literal.
         * Prefer using isTupleLikeType() unless the use of `elementTypes` is required.
         */
        function isTupleType(type: Type): boolean {
            return !!(getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).target.objectFlags & ObjectFlags.Tuple);
        }

        function getFalsyFlagsOfTypes(types: Type[]): TypeFlags {
            let result: TypeFlags = 0;
            for (const t of types) {
                result |= getFalsyFlags(t);
            }
            return result;
        }

        // Returns the String, Number, Boolean, StringLiteral, NumberLiteral, BooleanLiteral, Void, Undefined, or Null
        // flags for the string, number, boolean, "", 0, false, void, undefined, or null types respectively. Returns
        // no flags for all other types (including non-falsy literal types).
        function getFalsyFlags(type: Type): TypeFlags {
            return type.flags & TypeFlags.Union ? getFalsyFlagsOfTypes((<UnionType>type).types) :
                type.flags & TypeFlags.StringLiteral ? (<LiteralType>type).value === "" ? TypeFlags.StringLiteral : 0 :
                type.flags & TypeFlags.NumberLiteral ? (<LiteralType>type).value === 0 ? TypeFlags.NumberLiteral : 0 :
                type.flags & TypeFlags.BooleanLiteral ? type === falseType ? TypeFlags.BooleanLiteral : 0 :
                type.flags & TypeFlags.PossiblyFalsy;
        }

        function removeDefinitelyFalsyTypes(type: Type): Type {
            return getFalsyFlags(type) & TypeFlags.DefinitelyFalsy ?
                filterType(type, t => !(getFalsyFlags(t) & TypeFlags.DefinitelyFalsy)) :
                type;
        }

        function extractDefinitelyFalsyTypes(type: Type): Type {
            return mapType(type, getDefinitelyFalsyPartOfType);
        }

        function getDefinitelyFalsyPartOfType(type: Type): Type {
            return type.flags & TypeFlags.String ? emptyStringType :
                type.flags & TypeFlags.Number ? zeroType :
                type.flags & TypeFlags.Boolean || type === falseType ? falseType :
                type.flags & (TypeFlags.Void | TypeFlags.Undefined | TypeFlags.Null) ||
                type.flags & TypeFlags.StringLiteral && (<LiteralType>type).value === "" ||
                type.flags & TypeFlags.NumberLiteral && (<LiteralType>type).value === 0 ? type :
                neverType;
        }

        /**
         * Add undefined or null or both to a type if they are missing.
         * @param type - type to add undefined and/or null to if not present
         * @param flags - Either TypeFlags.Undefined or TypeFlags.Null, or both
         */
        function getNullableType(type: Type, flags: TypeFlags): Type {
            const missing = (flags & ~type.flags) & (TypeFlags.Undefined | TypeFlags.Null);
            return missing === 0 ? type :
                missing === TypeFlags.Undefined ? getUnionType([type, undefinedType]) :
                missing === TypeFlags.Null ? getUnionType([type, nullType]) :
                getUnionType([type, undefinedType, nullType]);
        }

        function getOptionalType(type: Type): Type {
            Debug.assert(strictNullChecks);
            return type.flags & TypeFlags.Undefined ? type : getUnionType([type, undefinedType]);
        }

        function getGlobalNonNullableTypeInstantiation(type: Type) {
            if (!deferredGlobalNonNullableTypeAlias) {
                deferredGlobalNonNullableTypeAlias = getGlobalSymbol("NonNullable" as __String, SymbolFlags.TypeAlias, /*diagnostic*/ undefined) || unknownSymbol;
            }
            // Use NonNullable global type alias if available to improve quick info/declaration emit
            if (deferredGlobalNonNullableTypeAlias !== unknownSymbol) {
                return getTypeAliasInstantiation(deferredGlobalNonNullableTypeAlias, [type]);
            }
            return getTypeWithFacts(type, TypeFacts.NEUndefinedOrNull); // Type alias unavailable, fall back to non-higherorder behavior
        }

        function getNonNullableType(type: Type): Type {
            return strictNullChecks ? getGlobalNonNullableTypeInstantiation(type) : type;
        }

        /**
         * Return true if type was inferred from an object literal, written as an object type literal, or is the shape of a module
         * with no call or construct signatures.
         */
        function isObjectTypeWithInferableIndex(type: Type) {
            return type.symbol && (type.symbol.flags & (SymbolFlags.ObjectLiteral | SymbolFlags.TypeLiteral | SymbolFlags.ValueModule)) !== 0 &&
                !typeHasCallOrConstructSignatures(type);
        }

        function createSymbolWithType(source: Symbol, type: Type) {
            const symbol = createSymbol(source.flags, source.escapedName);
            symbol.declarations = source.declarations;
            symbol.parent = source.parent;
            symbol.type = type;
            symbol.target = source;
            if (source.valueDeclaration) {
                symbol.valueDeclaration = source.valueDeclaration;
            }
            return symbol;
        }

        function transformTypeOfMembers(type: Type, f: (propertyType: Type) => Type) {
            const members = createSymbolTable();
            for (const property of getPropertiesOfObjectType(type)) {
                const original = getTypeOfSymbol(property);
                const updated = f(original);
                members.set(property.escapedName, updated === original ? property : createSymbolWithType(property, updated));
            }
            return members;
        }

        /**
         * If the the provided object literal is subject to the excess properties check,
         * create a new that is exempt. Recursively mark object literal members as exempt.
         * Leave signatures alone since they are not subject to the check.
         */
        function getRegularTypeOfObjectLiteral(type: Type): Type {
            if (!(isObjectLiteralType(type) && type.flags & TypeFlags.FreshLiteral)) {
                return type;
            }
            const regularType = (<FreshObjectLiteralType>type).regularType;
            if (regularType) {
                return regularType;
            }

            const resolved = <ResolvedType>type;
            const members = transformTypeOfMembers(type, getRegularTypeOfObjectLiteral);
            const regularNew = createAnonymousType(resolved.symbol,
                members,
                resolved.callSignatures,
                resolved.constructSignatures,
                resolved.stringIndexInfo,
                resolved.numberIndexInfo);
            regularNew.flags = resolved.flags & ~TypeFlags.FreshLiteral;
            regularNew.objectFlags |= ObjectFlags.ObjectLiteral;
            (<FreshObjectLiteralType>type).regularType = regularNew;
            return regularNew;
        }

        function createWideningContext(parent: WideningContext, propertyName: __String, siblings: Type[]): WideningContext {
            return { parent, propertyName, siblings, resolvedPropertyNames: undefined };
        }

        function getSiblingsOfContext(context: WideningContext): Type[] {
            if (!context.siblings) {
                const siblings: Type[] = [];
                for (const type of getSiblingsOfContext(context.parent)) {
                    if (isObjectLiteralType(type)) {
                        const prop = getPropertyOfObjectType(type, context.propertyName);
                        if (prop) {
                            forEachType(getTypeOfSymbol(prop), t => {
                                siblings.push(t);
                            });
                        }
                    }
                }
                context.siblings = siblings;
            }
            return context.siblings;
        }

        function getPropertyNamesOfContext(context: WideningContext): __String[] {
            if (!context.resolvedPropertyNames) {
                const names = createMap<boolean>() as UnderscoreEscapedMap<boolean>;
                for (const t of getSiblingsOfContext(context)) {
                    if (isObjectLiteralType(t) && !(getObjectFlags(t) & ObjectFlags.ContainsSpread)) {
                        for (const prop of getPropertiesOfType(t)) {
                            names.set(prop.escapedName, true);
                        }
                    }
                }
                context.resolvedPropertyNames = arrayFrom(names.keys());
            }
            return context.resolvedPropertyNames;
        }

        function getWidenedProperty(prop: Symbol, context: WideningContext): Symbol {
            const original = getTypeOfSymbol(prop);
            const propContext = context && createWideningContext(context, prop.escapedName, /*siblings*/ undefined);
            const widened = getWidenedTypeWithContext(original, propContext);
            return widened === original ? prop : createSymbolWithType(prop, widened);
        }

        function getUndefinedProperty(name: __String) {
            const cached = undefinedProperties.get(name);
            if (cached) {
                return cached;
            }
            const result = createSymbol(SymbolFlags.Property | SymbolFlags.Optional, name);
            result.type = undefinedType;
            const associatedKeyType = getLiteralType(unescapeLeadingUnderscores(name));
            if (associatedKeyType.flags & TypeFlags.StringLiteral) {
                result.nameType = associatedKeyType;
            }
            undefinedProperties.set(name, result);
            return result;
        }

        function getWidenedTypeOfObjectLiteral(type: Type, context: WideningContext): Type {
            const members = createSymbolTable();
            for (const prop of getPropertiesOfObjectType(type)) {
                // Since get accessors already widen their return value there is no need to
                // widen accessor based properties here.
                members.set(prop.escapedName, prop.flags & SymbolFlags.Property ? getWidenedProperty(prop, context) : prop);
            }
            if (context) {
                for (const name of getPropertyNamesOfContext(context)) {
                    if (!members.has(name)) {
                        members.set(name, getUndefinedProperty(name));
                    }
                }
            }
            const stringIndexInfo = getIndexInfoOfType(type, IndexKind.String);
            const numberIndexInfo = getIndexInfoOfType(type, IndexKind.Number);
            return createAnonymousType(type.symbol, members, emptyArray, emptyArray,
                stringIndexInfo && createIndexInfo(getWidenedType(stringIndexInfo.type), stringIndexInfo.isReadonly),
                numberIndexInfo && createIndexInfo(getWidenedType(numberIndexInfo.type), numberIndexInfo.isReadonly));
        }

        function getWidenedType(type: Type) {
            return getWidenedTypeWithContext(type, /*context*/ undefined);
        }

        function getWidenedTypeWithContext(type: Type, context: WideningContext): Type {
            if (type.flags & TypeFlags.RequiresWidening) {
                if (type.flags & TypeFlags.Nullable) {
                    return anyType;
                }
                if (isObjectLiteralType(type)) {
                    return getWidenedTypeOfObjectLiteral(type, context);
                }
                if (type.flags & TypeFlags.Union) {
                    const unionContext = context || createWideningContext(/*parent*/ undefined, /*propertyName*/ undefined, (<UnionType>type).types);
                    const widenedTypes = sameMap((<UnionType>type).types, t => t.flags & TypeFlags.Nullable ? t : getWidenedTypeWithContext(t, unionContext));
                    // Widening an empty object literal transitions from a highly restrictive type to
                    // a highly inclusive one. For that reason we perform subtype reduction here if the
                    // union includes empty object types (e.g. reducing {} | string to just {}).
                    return getUnionType(widenedTypes, some(widenedTypes, isEmptyObjectType) ? UnionReduction.Subtype : UnionReduction.Literal);
                }
                if (isArrayType(type) || isTupleType(type)) {
                    return createTypeReference((<TypeReference>type).target, sameMap((<TypeReference>type).typeArguments, getWidenedType));
                }
            }
            return type;
        }

        /**
         * Reports implicit any errors that occur as a result of widening 'null' and 'undefined'
         * to 'any'. A call to reportWideningErrorsInType is normally accompanied by a call to
         * getWidenedType. But in some cases getWidenedType is called without reporting errors
         * (type argument inference is an example).
         *
         * The return value indicates whether an error was in fact reported. The particular circumstances
         * are on a best effort basis. Currently, if the null or undefined that causes widening is inside
         * an object literal property (arbitrarily deeply), this function reports an error. If no error is
         * reported, reportImplicitAnyError is a suitable fallback to report a general error.
         */
        function reportWideningErrorsInType(type: Type): boolean {
            let errorReported = false;
            if (type.flags & TypeFlags.ContainsWideningType) {
                if (type.flags & TypeFlags.Union) {
                    if (some((<UnionType>type).types, isEmptyObjectType)) {
                        errorReported = true;
                    }
                    else {
                        for (const t of (<UnionType>type).types) {
                            if (reportWideningErrorsInType(t)) {
                                errorReported = true;
                            }
                        }
                    }
                }
                if (isArrayType(type) || isTupleType(type)) {
                    for (const t of (<TypeReference>type).typeArguments) {
                        if (reportWideningErrorsInType(t)) {
                            errorReported = true;
                        }
                    }
                }
                if (isObjectLiteralType(type)) {
                    for (const p of getPropertiesOfObjectType(type)) {
                        const t = getTypeOfSymbol(p);
                        if (t.flags & TypeFlags.ContainsWideningType) {
                            if (!reportWideningErrorsInType(t)) {
                                error(p.valueDeclaration, Diagnostics.Object_literal_s_property_0_implicitly_has_an_1_type, symbolToString(p), typeToString(getWidenedType(t)));
                            }
                            errorReported = true;
                        }
                    }
                }
            }
            return errorReported;
        }

        function reportImplicitAnyError(declaration: Declaration, type: Type) {
            const typeAsString = typeToString(getWidenedType(type));
            let diagnostic: DiagnosticMessage;
            switch (declaration.kind) {
                case SyntaxKind.PropertyDeclaration:
                case SyntaxKind.PropertySignature:
                    diagnostic = Diagnostics.Member_0_implicitly_has_an_1_type;
                    break;
                case SyntaxKind.Parameter:
                    diagnostic = (<ParameterDeclaration>declaration).dotDotDotToken ?
                        Diagnostics.Rest_parameter_0_implicitly_has_an_any_type :
                        Diagnostics.Parameter_0_implicitly_has_an_1_type;
                    break;
                case SyntaxKind.BindingElement:
                    diagnostic = Diagnostics.Binding_element_0_implicitly_has_an_1_type;
                    break;
                case SyntaxKind.FunctionDeclaration:
                case SyntaxKind.MethodDeclaration:
                case SyntaxKind.MethodSignature:
                case SyntaxKind.GetAccessor:
                case SyntaxKind.SetAccessor:
                case SyntaxKind.FunctionExpression:
                case SyntaxKind.ArrowFunction:
                    if (!(declaration as NamedDeclaration).name) {
                        error(declaration, Diagnostics.Function_expression_which_lacks_return_type_annotation_implicitly_has_an_0_return_type, typeAsString);
                        return;
                    }
                    diagnostic = Diagnostics._0_which_lacks_return_type_annotation_implicitly_has_an_1_return_type;
                    break;
                case SyntaxKind.MappedType:
                    error(declaration, Diagnostics.Mapped_object_type_implicitly_has_an_any_template_type);
                    return;
                default:
                    diagnostic = Diagnostics.Variable_0_implicitly_has_an_1_type;
            }
            error(declaration, diagnostic, declarationNameToString(getNameOfDeclaration(declaration)), typeAsString);
        }

        function reportErrorsFromWidening(declaration: Declaration, type: Type) {
            if (produceDiagnostics && noImplicitAny && type.flags & TypeFlags.ContainsWideningType) {
                // Report implicit any error within type if possible, otherwise report error on declaration
                if (!reportWideningErrorsInType(type)) {
                    reportImplicitAnyError(declaration, type);
                }
            }
        }

        function forEachMatchingParameterType(source: Signature, target: Signature, callback: (s: Type, t: Type) => void) {
            const sourceMax = source.parameters.length;
            const targetMax = target.parameters.length;
            let count: number;
            if (source.hasRestParameter && target.hasRestParameter) {
                count = Math.max(sourceMax, targetMax);
            }
            else if (source.hasRestParameter) {
                count = targetMax;
            }
            else if (target.hasRestParameter) {
                count = sourceMax;
            }
            else {
                count = Math.min(sourceMax, targetMax);
            }
            for (let i = 0; i < count; i++) {
                callback(getTypeAtPosition(source, i), getTypeAtPosition(target, i));
            }
        }

        function createInferenceContext(typeParameters: TypeParameter[], signature: Signature | undefined, flags: InferenceFlags, compareTypes?: TypeComparer, baseInferences?: InferenceInfo[]): InferenceContext {
            const inferences = baseInferences ? baseInferences.map(cloneInferenceInfo) : typeParameters.map(createInferenceInfo);
            const context = mapper as InferenceContext;
            context.typeParameters = typeParameters;
            context.signature = signature;
            context.inferences = inferences;
            context.flags = flags;
            context.compareTypes = compareTypes || compareTypesAssignable;
            return context;

            function mapper(t: Type): Type {
                for (let i = 0; i < inferences.length; i++) {
                    if (t === inferences[i].typeParameter) {
                        inferences[i].isFixed = true;
                        return getInferredType(context, i);
                    }
                }
                return t;
            }
        }

        function createInferenceInfo(typeParameter: TypeParameter): InferenceInfo {
            return {
                typeParameter,
                candidates: undefined,
                contraCandidates: undefined,
                inferredType: undefined,
                priority: undefined,
                topLevel: true,
                isFixed: false
            };
        }

        function cloneInferenceInfo(inference: InferenceInfo): InferenceInfo {
            return {
                typeParameter: inference.typeParameter,
                candidates: inference.candidates && inference.candidates.slice(),
                contraCandidates: inference.contraCandidates && inference.contraCandidates.slice(),
                inferredType: inference.inferredType,
                priority: inference.priority,
                topLevel: inference.topLevel,
                isFixed: inference.isFixed
            };
        }

        // Return true if the given type could possibly reference a type parameter for which
        // we perform type inference (i.e. a type parameter of a generic function). We cache
        // results for union and intersection types for performance reasons.
        function couldContainTypeVariables(type: Type): boolean {
            const objectFlags = getObjectFlags(type);
            return !!(type.flags & TypeFlags.Instantiable ||
                objectFlags & ObjectFlags.Reference && forEach((<TypeReference>type).typeArguments, couldContainTypeVariables) ||
                objectFlags & ObjectFlags.Anonymous && type.symbol && type.symbol.flags & (SymbolFlags.Function | SymbolFlags.Method | SymbolFlags.TypeLiteral | SymbolFlags.Class) ||
                objectFlags & ObjectFlags.Mapped ||
                type.flags & TypeFlags.UnionOrIntersection && couldUnionOrIntersectionContainTypeVariables(<UnionOrIntersectionType>type));
        }

        function couldUnionOrIntersectionContainTypeVariables(type: UnionOrIntersectionType): boolean {
            if (type.couldContainTypeVariables === undefined) {
                type.couldContainTypeVariables = forEach(type.types, couldContainTypeVariables);
            }
            return type.couldContainTypeVariables;
        }

        function isTypeParameterAtTopLevel(type: Type, typeParameter: TypeParameter): boolean {
            return type === typeParameter || type.flags & TypeFlags.UnionOrIntersection && forEach((<UnionOrIntersectionType>type).types, t => isTypeParameterAtTopLevel(t, typeParameter));
        }

        /** Create an object with properties named in the string literal type. Every property has type `any` */
        function createEmptyObjectTypeFromStringLiteral(type: Type) {
            const members = createSymbolTable();
            forEachType(type, t => {
                if (!(t.flags & TypeFlags.StringLiteral)) {
                    return;
                }
                const name = escapeLeadingUnderscores((t as StringLiteralType).value);
                const literalProp = createSymbol(SymbolFlags.Property, name);
                literalProp.type = anyType;
                if (t.symbol) {
                    literalProp.declarations = t.symbol.declarations;
                    literalProp.valueDeclaration = t.symbol.valueDeclaration;
                }
                members.set(name, literalProp);
            });
            const indexInfo = type.flags & TypeFlags.String ? createIndexInfo(emptyObjectType, /*isReadonly*/ false) : undefined;
            return createAnonymousType(undefined, members, emptyArray, emptyArray, indexInfo, undefined);
        }

        /**
         * Infer a suitable input type for a homomorphic mapped type { [P in keyof T]: X }. We construct
         * an object type with the same set of properties as the source type, where the type of each
         * property is computed by inferring from the source property type to X for the type
         * variable T[P] (i.e. we treat the type T[P] as the type variable we're inferring for).
         */
        function inferTypeForHomomorphicMappedType(source: Type, target: MappedType): Type {
            const key = source.id + "," + target.id;
            if (reverseMappedCache.has(key)) {
                return reverseMappedCache.get(key);
            }
            reverseMappedCache.set(key, undefined);
            const type = createReverseMappedType(source, target);
            reverseMappedCache.set(key, type);
            return type;
        }

        function createReverseMappedType(source: Type, target: MappedType) {
            const properties = getPropertiesOfType(source);
            if (properties.length === 0 && !getIndexInfoOfType(source, IndexKind.String)) {
                return undefined;
            }
            // If any property contains context sensitive functions that have been skipped, the source type
            // is incomplete and we can't infer a meaningful input type.
            for (const prop of properties) {
                if (getTypeOfSymbol(prop).flags & TypeFlags.ContainsAnyFunctionType) {
                    return undefined;
                }
            }
            const reversed = createObjectType(ObjectFlags.ReverseMapped | ObjectFlags.Anonymous, /*symbol*/ undefined) as ReverseMappedType;
            reversed.source = source;
            reversed.mappedType = target;
            return reversed;
        }

        function getTypeOfReverseMappedSymbol(symbol: ReverseMappedSymbol) {
            return inferReverseMappedType(symbol.propertyType, symbol.mappedType);
        }

        function inferReverseMappedType(sourceType: Type, target: MappedType): Type {
            const typeParameter = <TypeParameter>getIndexedAccessType((<IndexType>getConstraintTypeFromMappedType(target)).type, getTypeParameterFromMappedType(target));
            const templateType = getTemplateTypeFromMappedType(target);
            const inference = createInferenceInfo(typeParameter);
            inferTypes([inference], sourceType, templateType);
            return getTypeFromInference(inference);
        }

        function getUnmatchedProperty(source: Type, target: Type, requireOptionalProperties: boolean) {
            const properties = target.flags & TypeFlags.Intersection ? getPropertiesOfUnionOrIntersectionType(<IntersectionType>target) : getPropertiesOfObjectType(target);
            for (const targetProp of properties) {
                if (requireOptionalProperties || !(targetProp.flags & SymbolFlags.Optional)) {
                    const sourceProp = getPropertyOfType(source, targetProp.escapedName);
                    if (!sourceProp) {
                        return targetProp;
                    }
                }
            }
            return undefined;
        }

        function getTypeFromInference(inference: InferenceInfo) {
            return inference.candidates ? getUnionType(inference.candidates, UnionReduction.Subtype) :
                inference.contraCandidates ? getIntersectionType(inference.contraCandidates) :
                emptyObjectType;
        }

        function inferTypes(inferences: InferenceInfo[], originalSource: Type, originalTarget: Type, priority: InferencePriority = 0) {
            let symbolStack: Symbol[];
            let visited: Map<boolean>;
            let contravariant = false;
            let propagationType: Type;
            inferFromTypes(originalSource, originalTarget);

            function inferFromTypes(source: Type, target: Type) {
                if (!couldContainTypeVariables(target)) {
                    return;
                }
                if (source.flags & TypeFlags.Any) {
                    // We are inferring from an 'any' type. We want to infer this type for every type parameter
                    // referenced in the target type, so we record it as the propagation type and infer from the
                    // target to itself. Then, as we find candidates we substitute the propagation type.
                    const savePropagationType = propagationType;
                    propagationType = source;
                    inferFromTypes(target, target);
                    propagationType = savePropagationType;
                    return;
                }
                if (source.aliasSymbol && source.aliasTypeArguments && source.aliasSymbol === target.aliasSymbol) {
                    // Source and target are types originating in the same generic type alias declaration.
                    // Simply infer from source type arguments to target type arguments.
                    const sourceTypes = source.aliasTypeArguments;
                    const targetTypes = target.aliasTypeArguments;
                    for (let i = 0; i < sourceTypes.length; i++) {
                        inferFromTypes(sourceTypes[i], targetTypes[i]);
                    }
                    return;
                }
                if (source.flags & TypeFlags.Union && target.flags & TypeFlags.Union && !(source.flags & TypeFlags.EnumLiteral && target.flags & TypeFlags.EnumLiteral) ||
                    source.flags & TypeFlags.Intersection && target.flags & TypeFlags.Intersection) {
                    // Source and target are both unions or both intersections. If source and target
                    // are the same type, just relate each constituent type to itself.
                    if (source === target) {
                        for (const t of (<UnionOrIntersectionType>source).types) {
                            inferFromTypes(t, t);
                        }
                        return;
                    }
                    // Find each source constituent type that has an identically matching target constituent
                    // type, and for each such type infer from the type to itself. When inferring from a
                    // type to itself we effectively find all type parameter occurrences within that type
                    // and infer themselves as their type arguments. We have special handling for numeric
                    // and string literals because the number and string types are not represented as unions
                    // of all their possible values.
                    let matchingTypes: Type[];
                    for (const t of (<UnionOrIntersectionType>source).types) {
                        if (typeIdenticalToSomeType(t, (<UnionOrIntersectionType>target).types)) {
                            (matchingTypes || (matchingTypes = [])).push(t);
                            inferFromTypes(t, t);
                        }
                        else if (t.flags & (TypeFlags.NumberLiteral | TypeFlags.StringLiteral)) {
                            const b = getBaseTypeOfLiteralType(t);
                            if (typeIdenticalToSomeType(b, (<UnionOrIntersectionType>target).types)) {
                                (matchingTypes || (matchingTypes = [])).push(t, b);
                            }
                        }
                    }
                    // Next, to improve the quality of inferences, reduce the source and target types by
                    // removing the identically matched constituents. For example, when inferring from
                    // 'string | string[]' to 'string | T' we reduce the types to 'string[]' and 'T'.
                    if (matchingTypes) {
                        source = removeTypesFromUnionOrIntersection(<UnionOrIntersectionType>source, matchingTypes);
                        target = removeTypesFromUnionOrIntersection(<UnionOrIntersectionType>target, matchingTypes);
                    }
                }
                if (target.flags & TypeFlags.TypeVariable) {
                    // If target is a type parameter, make an inference, unless the source type contains
                    // the anyFunctionType (the wildcard type that's used to avoid contextually typing functions).
                    // Because the anyFunctionType is internal, it should not be exposed to the user by adding
                    // it as an inference candidate. Hopefully, a better candidate will come along that does
                    // not contain anyFunctionType when we come back to this argument for its second round
                    // of inference. Also, we exclude inferences for silentNeverType (which is used as a wildcard
                    // when constructing types from type parameters that had no inference candidates).
                    if (source.flags & TypeFlags.ContainsAnyFunctionType || source === silentNeverType) {
                        return;
                    }
                    const inference = getInferenceInfoForType(target);
                    if (inference) {
                        if (!inference.isFixed) {
                            if (inference.priority === undefined || priority < inference.priority) {
                                inference.candidates = undefined;
                                inference.contraCandidates = undefined;
                                inference.priority = priority;
                            }
                            if (priority === inference.priority) {
                                const candidate = propagationType || source;
                                if (contravariant) {
                                    inference.contraCandidates = append(inference.contraCandidates, candidate);
                                }
                                else {
                                    inference.candidates = append(inference.candidates, candidate);
                                }
                            }
                            if (!(priority & InferencePriority.ReturnType) && target.flags & TypeFlags.TypeParameter && !isTypeParameterAtTopLevel(originalTarget, <TypeParameter>target)) {
                                inference.topLevel = false;
                            }
                        }
                        return;
                    }
                }
                if (getObjectFlags(source) & ObjectFlags.Reference && getObjectFlags(target) & ObjectFlags.Reference && (<TypeReference>source).target === (<TypeReference>target).target) {
                    // If source and target are references to the same generic type, infer from type arguments
                    const sourceTypes = (<TypeReference>source).typeArguments || emptyArray;
                    const targetTypes = (<TypeReference>target).typeArguments || emptyArray;
                    const count = sourceTypes.length < targetTypes.length ? sourceTypes.length : targetTypes.length;
                    const variances = getVariances((<TypeReference>source).target);
                    for (let i = 0; i < count; i++) {
                        if (i < variances.length && variances[i] === Variance.Contravariant) {
                            inferFromContravariantTypes(sourceTypes[i], targetTypes[i]);
                        }
                        else {
                            inferFromTypes(sourceTypes[i], targetTypes[i]);
                        }
                    }
                }
                else if (source.flags & TypeFlags.Index && target.flags & TypeFlags.Index) {
                    contravariant = !contravariant;
                    inferFromTypes((<IndexType>source).type, (<IndexType>target).type);
                    contravariant = !contravariant;
                }
                else if ((isLiteralType(source) || source.flags & TypeFlags.String) && target.flags & TypeFlags.Index) {
                    const empty = createEmptyObjectTypeFromStringLiteral(source);
                    contravariant = !contravariant;
                    inferFromTypes(empty, (target as IndexType).type);
                    contravariant = !contravariant;
                }
                else if (source.flags & TypeFlags.IndexedAccess && target.flags & TypeFlags.IndexedAccess) {
                    inferFromTypes((<IndexedAccessType>source).objectType, (<IndexedAccessType>target).objectType);
                    inferFromTypes((<IndexedAccessType>source).indexType, (<IndexedAccessType>target).indexType);
                }
                else if (source.flags & TypeFlags.Conditional && target.flags & TypeFlags.Conditional) {
                    inferFromTypes((<ConditionalType>source).checkType, (<ConditionalType>target).checkType);
                    inferFromTypes((<ConditionalType>source).extendsType, (<ConditionalType>target).extendsType);
                    inferFromTypes(getTrueTypeFromConditionalType(<ConditionalType>source), getTrueTypeFromConditionalType(<ConditionalType>target));
                    inferFromTypes(getFalseTypeFromConditionalType(<ConditionalType>source), getFalseTypeFromConditionalType(<ConditionalType>target));
                }
                else if (target.flags & TypeFlags.UnionOrIntersection) {
                    const targetTypes = (<UnionOrIntersectionType>target).types;
                    let typeVariableCount = 0;
                    let typeVariable: TypeParameter | IndexedAccessType;
                    // First infer to each type in union or intersection that isn't a type variable
                    for (const t of targetTypes) {
                        if (getInferenceInfoForType(t)) {
                            typeVariable = <InstantiableType>t;
                            typeVariableCount++;
                        }
                        else {
                            inferFromTypes(source, t);
                        }
                    }
                    // Next, if target containings a single naked type variable, make a secondary inference to that type
                    // variable. This gives meaningful results for union types in co-variant positions and intersection
                    // types in contra-variant positions (such as callback parameters).
                    if (typeVariableCount === 1) {
                        const savePriority = priority;
                        priority |= InferencePriority.NakedTypeVariable;
                        inferFromTypes(source, typeVariable);
                        priority = savePriority;
                    }
                }
                else if (source.flags & TypeFlags.Union) {
                    // Source is a union or intersection type, infer from each constituent type
                    const sourceTypes = (<UnionOrIntersectionType>source).types;
                    for (const sourceType of sourceTypes) {
                        inferFromTypes(sourceType, target);
                    }
                }
                else {
                    if (!(priority & InferencePriority.NoConstraints && source.flags & (TypeFlags.Intersection | TypeFlags.Instantiable))) {
                        source = getApparentType(source);
                    }
                    if (source.flags & (TypeFlags.Object | TypeFlags.Intersection)) {
                        const key = source.id + "," + target.id;
                        if (visited && visited.get(key)) {
                            return;
                        }
                        (visited || (visited = createMap<boolean>())).set(key, true);
                        // If we are already processing another target type with the same associated symbol (such as
                        // an instantiation of the same generic type), we do not explore this target as it would yield
                        // no further inferences. We exclude the static side of classes from this check since it shares
                        // its symbol with the instance side which would lead to false positives.
                        const isNonConstructorObject = target.flags & TypeFlags.Object &&
                            !(getObjectFlags(target) & ObjectFlags.Anonymous && target.symbol && target.symbol.flags & SymbolFlags.Class);
                        const symbol = isNonConstructorObject ? target.symbol : undefined;
                        if (symbol) {
                            if (contains(symbolStack, symbol)) {
                                return;
                            }
                            (symbolStack || (symbolStack = [])).push(symbol);
                            inferFromObjectTypes(source, target);
                            symbolStack.pop();
                        }
                        else {
                            inferFromObjectTypes(source, target);
                        }
                    }
                }
            }

            function inferFromContravariantTypes(source: Type, target: Type) {
                if (strictFunctionTypes || priority & InferencePriority.AlwaysStrict) {
                    contravariant = !contravariant;
                    inferFromTypes(source, target);
                    contravariant = !contravariant;
                }
                else {
                    inferFromTypes(source, target);
                }
            }

            function getInferenceInfoForType(type: Type) {
                if (type.flags & TypeFlags.TypeVariable) {
                    for (const inference of inferences) {
                        if (type === inference.typeParameter) {
                            return inference;
                        }
                    }
                }
                return undefined;
            }

            function inferFromObjectTypes(source: Type, target: Type) {
                if (isGenericMappedType(source) && isGenericMappedType(target)) {
                    // The source and target types are generic types { [P in S]: X } and { [P in T]: Y }, so we infer
                    // from S to T and from X to Y.
                    inferFromTypes(getConstraintTypeFromMappedType(source), getConstraintTypeFromMappedType(target));
                    inferFromTypes(getTemplateTypeFromMappedType(source), getTemplateTypeFromMappedType(target));
                }
                if (getObjectFlags(target) & ObjectFlags.Mapped) {
                    const constraintType = getConstraintTypeFromMappedType(<MappedType>target);
                    if (constraintType.flags & TypeFlags.Index) {
                        // We're inferring from some source type S to a homomorphic mapped type { [P in keyof T]: X },
                        // where T is a type variable. Use inferTypeForHomomorphicMappedType to infer a suitable source
                        // type and then make a secondary inference from that type to T. We make a secondary inference
                        // such that direct inferences to T get priority over inferences to Partial<T>, for example.
                        const inference = getInferenceInfoForType((<IndexType>constraintType).type);
                        if (inference && !inference.isFixed) {
                            const inferredType = inferTypeForHomomorphicMappedType(source, <MappedType>target);
                            if (inferredType) {
                                const savePriority = priority;
                                priority |= InferencePriority.HomomorphicMappedType;
                                inferFromTypes(inferredType, inference.typeParameter);
                                priority = savePriority;
                            }
                        }
                        return;
                    }
                    if (constraintType.flags & TypeFlags.TypeParameter) {
                        // We're inferring from some source type S to a mapped type { [P in T]: X }, where T is a type
                        // parameter. Infer from 'keyof S' to T and infer from a union of each property type in S to X.
                        const savePriority = priority;
                        priority |= InferencePriority.MappedTypeConstraint;
                        inferFromTypes(getIndexType(source), constraintType);
                        priority = savePriority;
                        inferFromTypes(getUnionType(map(getPropertiesOfType(source), getTypeOfSymbol)), getTemplateTypeFromMappedType(<MappedType>target));
                        return;
                    }
                }
                // Infer from the members of source and target only if the two types are possibly related. We check
                // in both directions because we may be inferring for a co-variant or a contra-variant position.
                if (!getUnmatchedProperty(source, target, /*requireOptionalProperties*/ false) || !getUnmatchedProperty(target, source, /*requireOptionalProperties*/ false)) {
                    inferFromProperties(source, target);
                    inferFromSignatures(source, target, SignatureKind.Call);
                    inferFromSignatures(source, target, SignatureKind.Construct);
                    inferFromIndexTypes(source, target);
                }
            }

            function inferFromProperties(source: Type, target: Type) {
                const properties = getPropertiesOfObjectType(target);
                for (const targetProp of properties) {
                    const sourceProp = getPropertyOfType(source, targetProp.escapedName);
                    if (sourceProp) {
                        inferFromTypes(getTypeOfSymbol(sourceProp), getTypeOfSymbol(targetProp));
                    }
                }
            }

            function inferFromSignatures(source: Type, target: Type, kind: SignatureKind) {
                const sourceSignatures = getSignaturesOfType(source, kind);
                const targetSignatures = getSignaturesOfType(target, kind);
                const sourceLen = sourceSignatures.length;
                const targetLen = targetSignatures.length;
                const len = sourceLen < targetLen ? sourceLen : targetLen;
                for (let i = 0; i < len; i++) {
                    inferFromSignature(getBaseSignature(sourceSignatures[sourceLen - len + i]), getBaseSignature(targetSignatures[targetLen - len + i]));
                }
            }

            function inferFromSignature(source: Signature, target: Signature) {
                forEachMatchingParameterType(source, target, inferFromContravariantTypes);

                const sourceTypePredicate = getTypePredicateOfSignature(source);
                const targetTypePredicate = getTypePredicateOfSignature(target);
                if (sourceTypePredicate && targetTypePredicate && sourceTypePredicate.kind === targetTypePredicate.kind) {
                    inferFromTypes(sourceTypePredicate.type, targetTypePredicate.type);
                }
                else {
                    inferFromTypes(getReturnTypeOfSignature(source), getReturnTypeOfSignature(target));
                }
            }

            function inferFromIndexTypes(source: Type, target: Type) {
                const targetStringIndexType = getIndexTypeOfType(target, IndexKind.String);
                if (targetStringIndexType) {
                    const sourceIndexType = getIndexTypeOfType(source, IndexKind.String) ||
                        getImplicitIndexTypeOfType(source, IndexKind.String);
                    if (sourceIndexType) {
                        inferFromTypes(sourceIndexType, targetStringIndexType);
                    }
                }
                const targetNumberIndexType = getIndexTypeOfType(target, IndexKind.Number);
                if (targetNumberIndexType) {
                    const sourceIndexType = getIndexTypeOfType(source, IndexKind.Number) ||
                        getIndexTypeOfType(source, IndexKind.String) ||
                        getImplicitIndexTypeOfType(source, IndexKind.Number);
                    if (sourceIndexType) {
                        inferFromTypes(sourceIndexType, targetNumberIndexType);
                    }
                }
            }
        }

        function typeIdenticalToSomeType(type: Type, types: Type[]): boolean {
            for (const t of types) {
                if (isTypeIdenticalTo(t, type)) {
                    return true;
                }
            }
            return false;
        }

        /**
         * Return a new union or intersection type computed by removing a given set of types
         * from a given union or intersection type.
         */
        function removeTypesFromUnionOrIntersection(type: UnionOrIntersectionType, typesToRemove: Type[]) {
            const reducedTypes: Type[] = [];
            for (const t of type.types) {
                if (!typeIdenticalToSomeType(t, typesToRemove)) {
                    reducedTypes.push(t);
                }
            }
            return type.flags & TypeFlags.Union ? getUnionType(reducedTypes) : getIntersectionType(reducedTypes);
        }

        function hasPrimitiveConstraint(type: TypeParameter): boolean {
            const constraint = getConstraintOfTypeParameter(type);
            return constraint && maybeTypeOfKind(constraint, TypeFlags.Primitive | TypeFlags.Index);
        }

        function isObjectLiteralType(type: Type) {
            return !!(getObjectFlags(type) & ObjectFlags.ObjectLiteral);
        }

        function widenObjectLiteralCandidates(candidates: Type[]): Type[] {
            if (candidates.length > 1) {
                const objectLiterals = filter(candidates, isObjectLiteralType);
                if (objectLiterals.length) {
                    const objectLiteralsType = getWidenedType(getUnionType(objectLiterals, UnionReduction.Subtype));
                    return concatenate(filter(candidates, t => !isObjectLiteralType(t)), [objectLiteralsType]);
                }
            }
            return candidates;
        }

        function getInferredType(context: InferenceContext, index: number): Type {
            const inference = context.inferences[index];
            let inferredType = inference.inferredType;
            if (!inferredType) {
                const signature = context.signature;
                if (signature) {
                    if (inference.candidates) {
                        // Extract all object literal types and replace them with a single widened and normalized type.
                        const candidates = widenObjectLiteralCandidates(inference.candidates);
                        // We widen inferred literal types if
                        // all inferences were made to top-level ocurrences of the type parameter, and
                        // the type parameter has no constraint or its constraint includes no primitive or literal types, and
                        // the type parameter was fixed during inference or does not occur at top-level in the return type.
                        const widenLiteralTypes = inference.topLevel &&
                            !hasPrimitiveConstraint(inference.typeParameter) &&
                            (inference.isFixed || !isTypeParameterAtTopLevel(getReturnTypeOfSignature(signature), inference.typeParameter));
                        const baseCandidates = widenLiteralTypes ? sameMap(candidates, getWidenedLiteralType) : candidates;
                        // If all inferences were made from contravariant positions, infer a common subtype. Otherwise, if
                        // union types were requested or if all inferences were made from the return type position, infer a
                        // union type. Otherwise, infer a common supertype.
                        const unwidenedType = context.flags & InferenceFlags.InferUnionTypes || inference.priority & InferencePriority.PriorityImpliesUnion ?
                            getUnionType(baseCandidates, UnionReduction.Subtype) :
                            getCommonSupertype(baseCandidates);
                        inferredType = getWidenedType(unwidenedType);
                        // If we have inferred 'never' but have contravariant candidates. To get a more specific type we
                        // infer from the contravariant candidates instead.
                        if (inferredType.flags & TypeFlags.Never && inference.contraCandidates) {
                            inferredType = getCommonSubtype(inference.contraCandidates);
                        }
                    }
                    else if (inference.contraCandidates) {
                        // We only have contravariant inferences, infer the best common subtype of those
                        inferredType = getCommonSubtype(inference.contraCandidates);
                    }
                    else if (context.flags & InferenceFlags.NoDefault) {
                        // We use silentNeverType as the wildcard that signals no inferences.
                        inferredType = silentNeverType;
                    }
                    else {
                        // Infer either the default or the empty object type when no inferences were
                        // made. It is important to remember that in this case, inference still
                        // succeeds, meaning there is no error for not having inference candidates. An
                        // inference error only occurs when there are *conflicting* candidates, i.e.
                        // candidates with no common supertype.
                        const defaultType = getDefaultFromTypeParameter(inference.typeParameter);
                        if (defaultType) {
                            // Instantiate the default type. Any forward reference to a type
                            // parameter should be instantiated to the empty object type.
                            inferredType = instantiateType(defaultType,
                                combineTypeMappers(
                                    createBackreferenceMapper(context.signature.typeParameters, index),
                                    context));
                        }
                        else {
                            inferredType = getDefaultTypeArgumentType(!!(context.flags & InferenceFlags.AnyDefault));
                        }
                    }
                }
                else {
                    inferredType = getTypeFromInference(inference);
                }

                inferredType = getWidenedUniqueESSymbolType(inferredType);
                inference.inferredType = inferredType;

                const constraint = getConstraintOfTypeParameter(inference.typeParameter);
                if (constraint) {
                    const instantiatedConstraint = instantiateType(constraint, context);
                    if (!context.compareTypes(inferredType, getTypeWithThisArgument(instantiatedConstraint, inferredType))) {
                        inference.inferredType = inferredType = getWidenedUniqueESSymbolType(instantiatedConstraint);
                    }
                }
            }

            return inferredType;
        }

        function getDefaultTypeArgumentType(isInJavaScriptFile: boolean): Type {
            return isInJavaScriptFile ? anyType : emptyObjectType;
        }

        function getInferredTypes(context: InferenceContext): Type[] {
            const result: Type[] = [];
            for (let i = 0; i < context.inferences.length; i++) {
                result.push(getInferredType(context, i));
            }
            return result;
        }

        // EXPRESSION TYPE CHECKING

        function getResolvedSymbol(node: Identifier): Symbol {
            const links = getNodeLinks(node);
            if (!links.resolvedSymbol) {
                links.resolvedSymbol = !nodeIsMissing(node) &&
                    resolveName(
                        node,
                        node.escapedText,
                        SymbolFlags.Value | SymbolFlags.ExportValue,
                        Diagnostics.Cannot_find_name_0,
                        node,
                        !isWriteOnlyAccess(node),
                        /*excludeGlobals*/ false,
                        Diagnostics.Cannot_find_name_0_Did_you_mean_1) || unknownSymbol;
            }
            return links.resolvedSymbol;
        }

        function isInTypeQuery(node: Node): boolean {
            // TypeScript 1.0 spec (April 2014): 3.6.3
            // A type query consists of the keyword typeof followed by an expression.
            // The expression is restricted to a single identifier or a sequence of identifiers separated by periods
            return !!findAncestor(
                node,
                n => n.kind === SyntaxKind.TypeQuery ? true : n.kind === SyntaxKind.Identifier || n.kind === SyntaxKind.QualifiedName ? false : "quit");
        }

        // Return the flow cache key for a "dotted name" (i.e. a sequence of identifiers
        // separated by dots). The key consists of the id of the symbol referenced by the
        // leftmost identifier followed by zero or more property names separated by dots.
        // The result is undefined if the reference isn't a dotted name. We prefix nodes
        // occurring in an apparent type position with '@' because the control flow type
        // of such nodes may be based on the apparent type instead of the declared type.
        function getFlowCacheKey(node: Node): string | undefined {
            if (node.kind === SyntaxKind.Identifier) {
                const symbol = getResolvedSymbol(<Identifier>node);
                return symbol !== unknownSymbol ? (isConstraintPosition(node) ? "@" : "") + getSymbolId(symbol) : undefined;
            }
            if (node.kind === SyntaxKind.ThisKeyword) {
                return "0";
            }
            if (node.kind === SyntaxKind.PropertyAccessExpression) {
                const key = getFlowCacheKey((<PropertyAccessExpression>node).expression);
                return key && key + "." + idText((<PropertyAccessExpression>node).name);
            }
            if (node.kind === SyntaxKind.BindingElement) {
                const container = (node as BindingElement).parent.parent;
                const key = container.kind === SyntaxKind.BindingElement ? getFlowCacheKey(container) : (container.initializer && getFlowCacheKey(container.initializer));
                const text = getBindingElementNameText(node as BindingElement);
                const result = key && text && (key + "." + text);
                return result;
            }
            return undefined;
        }

        function getBindingElementNameText(element: BindingElement): string | undefined {
            if (element.parent.kind === SyntaxKind.ObjectBindingPattern) {
                const name = element.propertyName || element.name;
                switch (name.kind) {
                    case SyntaxKind.Identifier:
                        return idText(name);
                    case SyntaxKind.ComputedPropertyName:
                        return isStringOrNumericLiteral(name.expression) ? name.expression.text : undefined;
                    case SyntaxKind.StringLiteral:
                    case SyntaxKind.NumericLiteral:
                        return name.text;
                    default:
                        // Per types, array and object binding patterns remain, however they should never be present if propertyName is not defined
                        Debug.fail("Unexpected name kind for binding element name");
                }
            }
            else {
                 return "" + element.parent.elements.indexOf(element);
            }
        }

        function isMatchingReference(source: Node, target: Node): boolean {
            switch (source.kind) {
                case SyntaxKind.Identifier:
                    return target.kind === SyntaxKind.Identifier && getResolvedSymbol(<Identifier>source) === getResolvedSymbol(<Identifier>target) ||
                        (target.kind === SyntaxKind.VariableDeclaration || target.kind === SyntaxKind.BindingElement) &&
                        getExportSymbolOfValueSymbolIfExported(getResolvedSymbol(<Identifier>source)) === getSymbolOfNode(target);
                case SyntaxKind.ThisKeyword:
                    return target.kind === SyntaxKind.ThisKeyword;
                case SyntaxKind.SuperKeyword:
                    return target.kind === SyntaxKind.SuperKeyword;
                case SyntaxKind.PropertyAccessExpression:
                    return target.kind === SyntaxKind.PropertyAccessExpression &&
                        (<PropertyAccessExpression>source).name.escapedText === (<PropertyAccessExpression>target).name.escapedText &&
                        isMatchingReference((<PropertyAccessExpression>source).expression, (<PropertyAccessExpression>target).expression);
                case SyntaxKind.BindingElement:
                    if (target.kind !== SyntaxKind.PropertyAccessExpression) return false;
                    const t = target as PropertyAccessExpression;
                    if (t.name.escapedText !== getBindingElementNameText(source as BindingElement)) return false;
                    if (source.parent.parent.kind === SyntaxKind.BindingElement && isMatchingReference(source.parent.parent, t.expression)) {
                            return true;
                    }
                    if (source.parent.parent.kind === SyntaxKind.VariableDeclaration) {
                        const maybeId = (source.parent.parent as VariableDeclaration).initializer;
                        return maybeId && isMatchingReference(maybeId, t.expression);
                    }
            }
            return false;
        }

        function containsMatchingReference(source: Node, target: Node) {
            while (source.kind === SyntaxKind.PropertyAccessExpression) {
                source = (<PropertyAccessExpression>source).expression;
                if (isMatchingReference(source, target)) {
                    return true;
                }
            }
            return false;
        }

        // Return true if target is a property access xxx.yyy, source is a property access xxx.zzz, the declared
        // type of xxx is a union type, and yyy is a property that is possibly a discriminant. We consider a property
        // a possible discriminant if its type differs in the constituents of containing union type, and if every
        // choice is a unit type or a union of unit types.
        function containsMatchingReferenceDiscriminant(source: Node, target: Node) {
            return target.kind === SyntaxKind.PropertyAccessExpression &&
                containsMatchingReference(source, (<PropertyAccessExpression>target).expression) &&
                isDiscriminantProperty(getDeclaredTypeOfReference((<PropertyAccessExpression>target).expression), (<PropertyAccessExpression>target).name.escapedText);
        }

        function getDeclaredTypeOfReference(expr: Node): Type {
            if (expr.kind === SyntaxKind.Identifier) {
                return getTypeOfSymbol(getResolvedSymbol(<Identifier>expr));
            }
            if (expr.kind === SyntaxKind.PropertyAccessExpression) {
                const type = getDeclaredTypeOfReference((<PropertyAccessExpression>expr).expression);
                return type && getTypeOfPropertyOfType(type, (<PropertyAccessExpression>expr).name.escapedText);
            }
            return undefined;
        }

        function isDiscriminantProperty(type: Type, name: __String) {
            if (type && type.flags & TypeFlags.Union) {
                const prop = getUnionOrIntersectionProperty(<UnionType>type, name);
                if (prop && getCheckFlags(prop) & CheckFlags.SyntheticProperty) {
                    if ((<TransientSymbol>prop).isDiscriminantProperty === undefined) {
                        (<TransientSymbol>prop).isDiscriminantProperty = (<TransientSymbol>prop).checkFlags & CheckFlags.HasNonUniformType && isLiteralType(getTypeOfSymbol(prop));
                    }
                    return (<TransientSymbol>prop).isDiscriminantProperty;
                }
            }
            return false;
        }

        function findDiscriminantProperties(sourceProperties: Symbol[], target: Type): Symbol[] | undefined {
            let result: Symbol[];
            for (const sourceProperty of sourceProperties) {
                if (isDiscriminantProperty(target, sourceProperty.escapedName)) {
                    if (result) {
                        result.push(sourceProperty);
                        continue;
                    }
                    result = [sourceProperty];
                }
            }
            return result;
        }

        function isOrContainsMatchingReference(source: Node, target: Node) {
            return isMatchingReference(source, target) || containsMatchingReference(source, target);
        }

        function hasMatchingArgument(callExpression: CallExpression, reference: Node) {
            if (callExpression.arguments) {
                for (const argument of callExpression.arguments) {
                    if (isOrContainsMatchingReference(reference, argument)) {
                        return true;
                    }
                }
            }
            if (callExpression.expression.kind === SyntaxKind.PropertyAccessExpression &&
                isOrContainsMatchingReference(reference, (<PropertyAccessExpression>callExpression.expression).expression)) {
                return true;
            }
            return false;
        }

        function getFlowNodeId(flow: FlowNode): number {
            if (!flow.id) {
                flow.id = nextFlowId;
                nextFlowId++;
            }
            return flow.id;
        }

        function typeMaybeAssignableTo(source: Type, target: Type) {
            if (!(source.flags & TypeFlags.Union)) {
                return isTypeAssignableTo(source, target);
            }
            for (const t of (<UnionType>source).types) {
                if (isTypeAssignableTo(t, target)) {
                    return true;
                }
            }
            return false;
        }

        // Remove those constituent types of declaredType to which no constituent type of assignedType is assignable.
        // For example, when a variable of type number | string | boolean is assigned a value of type number | boolean,
        // we remove type string.
        function getAssignmentReducedType(declaredType: UnionType, assignedType: Type) {
            if (declaredType !== assignedType) {
                if (assignedType.flags & TypeFlags.Never) {
                    return assignedType;
                }
                const reducedType = filterType(declaredType, t => typeMaybeAssignableTo(assignedType, t));
                if (!(reducedType.flags & TypeFlags.Never)) {
                    return reducedType;
                }
            }
            return declaredType;
        }

        function getTypeFactsOfTypes(types: Type[]): TypeFacts {
            let result: TypeFacts = TypeFacts.None;
            for (const t of types) {
                result |= getTypeFacts(t);
            }
            return result;
        }

        function isFunctionObjectType(type: ObjectType): boolean {
            // We do a quick check for a "bind" property before performing the more expensive subtype
            // check. This gives us a quicker out in the common case where an object type is not a function.
            const resolved = resolveStructuredTypeMembers(type);
            return !!(resolved.callSignatures.length || resolved.constructSignatures.length ||
                resolved.members.get("bind" as __String) && isTypeSubtypeOf(type, globalFunctionType));
        }

        function getTypeFacts(type: Type): TypeFacts {
            const flags = type.flags;
            if (flags & TypeFlags.String) {
                return strictNullChecks ? TypeFacts.StringStrictFacts : TypeFacts.StringFacts;
            }
            if (flags & TypeFlags.StringLiteral) {
                const isEmpty = (<LiteralType>type).value === "";
                return strictNullChecks ?
                    isEmpty ? TypeFacts.EmptyStringStrictFacts : TypeFacts.NonEmptyStringStrictFacts :
                    isEmpty ? TypeFacts.EmptyStringFacts : TypeFacts.NonEmptyStringFacts;
            }
            if (flags & (TypeFlags.Number | TypeFlags.Enum)) {
                return strictNullChecks ? TypeFacts.NumberStrictFacts : TypeFacts.NumberFacts;
            }
            if (flags & TypeFlags.NumberLiteral) {
                const isZero = (<LiteralType>type).value === 0;
                return strictNullChecks ?
                    isZero ? TypeFacts.ZeroStrictFacts : TypeFacts.NonZeroStrictFacts :
                    isZero ? TypeFacts.ZeroFacts : TypeFacts.NonZeroFacts;
            }
            if (flags & TypeFlags.Boolean) {
                return strictNullChecks ? TypeFacts.BooleanStrictFacts : TypeFacts.BooleanFacts;
            }
            if (flags & TypeFlags.BooleanLike) {
                return strictNullChecks ?
                    type === falseType ? TypeFacts.FalseStrictFacts : TypeFacts.TrueStrictFacts :
                    type === falseType ? TypeFacts.FalseFacts : TypeFacts.TrueFacts;
            }
            if (flags & TypeFlags.Object) {
                return isFunctionObjectType(<ObjectType>type) ?
                    strictNullChecks ? TypeFacts.FunctionStrictFacts : TypeFacts.FunctionFacts :
                    strictNullChecks ? TypeFacts.ObjectStrictFacts : TypeFacts.ObjectFacts;
            }
            if (flags & (TypeFlags.Void | TypeFlags.Undefined)) {
                return TypeFacts.UndefinedFacts;
            }
            if (flags & TypeFlags.Null) {
                return TypeFacts.NullFacts;
            }
            if (flags & TypeFlags.ESSymbolLike) {
                return strictNullChecks ? TypeFacts.SymbolStrictFacts : TypeFacts.SymbolFacts;
            }
            if (flags & TypeFlags.NonPrimitive) {
                return strictNullChecks ? TypeFacts.ObjectStrictFacts : TypeFacts.ObjectFacts;
            }
            if (flags & TypeFlags.Instantiable) {
                return getTypeFacts(getBaseConstraintOfType(type) || emptyObjectType);
            }
            if (flags & TypeFlags.UnionOrIntersection) {
                return getTypeFactsOfTypes((<UnionOrIntersectionType>type).types);
            }
            return TypeFacts.All;
        }

        function getTypeWithFacts(type: Type, include: TypeFacts) {
            return filterType(type, t => (getTypeFacts(t) & include) !== 0);
        }

        function getTypeWithDefault(type: Type, defaultExpression: Expression) {
            if (defaultExpression) {
                const defaultType = getTypeOfExpression(defaultExpression);
                return getUnionType([getTypeWithFacts(type, TypeFacts.NEUndefined), defaultType]);
            }
            return type;
        }

        function getTypeOfDestructuredProperty(type: Type, name: PropertyName) {
            const text = getTextOfPropertyName(name);
            return getTypeOfPropertyOfType(type, text) ||
                isNumericLiteralName(text) && getIndexTypeOfType(type, IndexKind.Number) ||
                getIndexTypeOfType(type, IndexKind.String) ||
                unknownType;
        }

        function getTypeOfDestructuredArrayElement(type: Type, index: number) {
            return isTupleLikeType(type) && getTypeOfPropertyOfType(type, "" + index as __String) ||
                checkIteratedTypeOrElementType(type, /*errorNode*/ undefined, /*allowStringInput*/ false, /*allowAsyncIterables*/ false) ||
                unknownType;
        }

        function getTypeOfDestructuredSpreadExpression(type: Type) {
            return createArrayType(checkIteratedTypeOrElementType(type, /*errorNode*/ undefined, /*allowStringInput*/ false, /*allowAsyncIterables*/ false) || unknownType);
        }

        function getAssignedTypeOfBinaryExpression(node: BinaryExpression): Type {
            const isDestructuringDefaultAssignment =
                node.parent.kind === SyntaxKind.ArrayLiteralExpression && isDestructuringAssignmentTarget(node.parent) ||
                node.parent.kind === SyntaxKind.PropertyAssignment && isDestructuringAssignmentTarget(node.parent.parent);
            return isDestructuringDefaultAssignment ?
                getTypeWithDefault(getAssignedType(node), node.right) :
                getTypeOfExpression(node.right);
        }

        function isDestructuringAssignmentTarget(parent: Node) {
            return parent.parent.kind === SyntaxKind.BinaryExpression && (parent.parent as BinaryExpression).left === parent ||
                parent.parent.kind === SyntaxKind.ForOfStatement && (parent.parent as ForOfStatement).initializer === parent;
        }

        function getAssignedTypeOfArrayLiteralElement(node: ArrayLiteralExpression, element: Expression): Type {
            return getTypeOfDestructuredArrayElement(getAssignedType(node), node.elements.indexOf(element));
        }

        function getAssignedTypeOfSpreadExpression(node: SpreadElement): Type {
            return getTypeOfDestructuredSpreadExpression(getAssignedType(<ArrayLiteralExpression>node.parent));
        }

        function getAssignedTypeOfPropertyAssignment(node: PropertyAssignment | ShorthandPropertyAssignment): Type {
            return getTypeOfDestructuredProperty(getAssignedType(node.parent), node.name);
        }

        function getAssignedTypeOfShorthandPropertyAssignment(node: ShorthandPropertyAssignment): Type {
            return getTypeWithDefault(getAssignedTypeOfPropertyAssignment(node), node.objectAssignmentInitializer);
        }

        function getAssignedType(node: Expression): Type {
            const parent = node.parent;
            switch (parent.kind) {
                case SyntaxKind.ForInStatement:
                    return stringType;
                case SyntaxKind.ForOfStatement:
                    return checkRightHandSideOfForOf((<ForOfStatement>parent).expression, (<ForOfStatement>parent).awaitModifier) || unknownType;
                case SyntaxKind.BinaryExpression:
                    return getAssignedTypeOfBinaryExpression(<BinaryExpression>parent);
                case SyntaxKind.DeleteExpression:
                    return undefinedType;
                case SyntaxKind.ArrayLiteralExpression:
                    return getAssignedTypeOfArrayLiteralElement(<ArrayLiteralExpression>parent, node);
                case SyntaxKind.SpreadElement:
                    return getAssignedTypeOfSpreadExpression(<SpreadElement>parent);
                case SyntaxKind.PropertyAssignment:
                    return getAssignedTypeOfPropertyAssignment(<PropertyAssignment>parent);
                case SyntaxKind.ShorthandPropertyAssignment:
                    return getAssignedTypeOfShorthandPropertyAssignment(<ShorthandPropertyAssignment>parent);
            }
            return unknownType;
        }

        function getInitialTypeOfBindingElement(node: BindingElement): Type {
            const pattern = node.parent;
            const parentType = getInitialType(<VariableDeclaration | BindingElement>pattern.parent);
            const type = pattern.kind === SyntaxKind.ObjectBindingPattern ?
                getTypeOfDestructuredProperty(parentType, node.propertyName || <Identifier>node.name) :
                !node.dotDotDotToken ?
                    getTypeOfDestructuredArrayElement(parentType, pattern.elements.indexOf(node)) :
                    getTypeOfDestructuredSpreadExpression(parentType);
            return getTypeWithDefault(type, node.initializer);
        }

        function getTypeOfInitializer(node: Expression) {
            // Return the cached type if one is available. If the type of the variable was inferred
            // from its initializer, we'll already have cached the type. Otherwise we compute it now
            // without caching such that transient types are reflected.
            const links = getNodeLinks(node);
            return links.resolvedType || getTypeOfExpression(node);
        }

        function getInitialTypeOfVariableDeclaration(node: VariableDeclaration) {
            if (node.initializer) {
                return getTypeOfInitializer(node.initializer);
            }
            if (node.parent.parent.kind === SyntaxKind.ForInStatement) {
                return stringType;
            }
            if (node.parent.parent.kind === SyntaxKind.ForOfStatement) {
                return checkRightHandSideOfForOf(node.parent.parent.expression, node.parent.parent.awaitModifier) || unknownType;
            }
            return unknownType;
        }

        function getInitialType(node: VariableDeclaration | BindingElement) {
            return node.kind === SyntaxKind.VariableDeclaration ?
                getInitialTypeOfVariableDeclaration(node) :
                getInitialTypeOfBindingElement(node);
        }

        function getInitialOrAssignedType(node: VariableDeclaration | BindingElement | Expression) {
            return node.kind === SyntaxKind.VariableDeclaration || node.kind === SyntaxKind.BindingElement ?
                getInitialType(<VariableDeclaration | BindingElement>node) :
                getAssignedType(node);
        }

        function isEmptyArrayAssignment(node: VariableDeclaration | BindingElement | Expression) {
            return node.kind === SyntaxKind.VariableDeclaration && (<VariableDeclaration>node).initializer &&
                isEmptyArrayLiteral((<VariableDeclaration>node).initializer) ||
                node.kind !== SyntaxKind.BindingElement && node.parent.kind === SyntaxKind.BinaryExpression &&
                isEmptyArrayLiteral((<BinaryExpression>node.parent).right);
        }

        function getReferenceCandidate(node: Expression): Expression {
            switch (node.kind) {
                case SyntaxKind.ParenthesizedExpression:
                    return getReferenceCandidate((<ParenthesizedExpression>node).expression);
                case SyntaxKind.BinaryExpression:
                    switch ((<BinaryExpression>node).operatorToken.kind) {
                        case SyntaxKind.EqualsToken:
                            return getReferenceCandidate((<BinaryExpression>node).left);
                        case SyntaxKind.CommaToken:
                            return getReferenceCandidate((<BinaryExpression>node).right);
                    }
            }
            return node;
        }

        function getReferenceRoot(node: Node): Node {
            const parent = node.parent;
            return parent.kind === SyntaxKind.ParenthesizedExpression ||
                parent.kind === SyntaxKind.BinaryExpression && (<BinaryExpression>parent).operatorToken.kind === SyntaxKind.EqualsToken && (<BinaryExpression>parent).left === node ||
                parent.kind === SyntaxKind.BinaryExpression && (<BinaryExpression>parent).operatorToken.kind === SyntaxKind.CommaToken && (<BinaryExpression>parent).right === node ?
                getReferenceRoot(parent) : node;
        }

        function getTypeOfSwitchClause(clause: CaseClause | DefaultClause) {
            if (clause.kind === SyntaxKind.CaseClause) {
                const caseType = getRegularTypeOfLiteralType(getTypeOfExpression(clause.expression));
                return isUnitType(caseType) ? caseType : undefined;
            }
            return neverType;
        }

        function getSwitchClauseTypes(switchStatement: SwitchStatement): Type[] {
            const links = getNodeLinks(switchStatement);
            if (!links.switchTypes) {
                // If all case clauses specify expressions that have unit types, we return an array
                // of those unit types. Otherwise we return an empty array.
                links.switchTypes = [];
                for (const clause of switchStatement.caseBlock.clauses) {
                    const type = getTypeOfSwitchClause(clause);
                    if (type === undefined) {
                        return links.switchTypes = emptyArray;
                    }
                    links.switchTypes.push(type);
                }
            }
            return links.switchTypes;
        }

        function eachTypeContainedIn(source: Type, types: Type[]) {
            return source.flags & TypeFlags.Union ? !forEach((<UnionType>source).types, t => !contains(types, t)) : contains(types, source);
        }

        function isTypeSubsetOf(source: Type, target: Type) {
            return source === target || target.flags & TypeFlags.Union && isTypeSubsetOfUnion(source, <UnionType>target);
        }

        function isTypeSubsetOfUnion(source: Type, target: UnionType) {
            if (source.flags & TypeFlags.Union) {
                for (const t of (<UnionType>source).types) {
                    if (!containsType(target.types, t)) {
                        return false;
                    }
                }
                return true;
            }
            if (source.flags & TypeFlags.EnumLiteral && getBaseTypeOfEnumLiteralType(<LiteralType>source) === target) {
                return true;
            }
            return containsType(target.types, source);
        }

        function forEachType<T>(type: Type, f: (t: Type) => T): T {
            return type.flags & TypeFlags.Union ? forEach((<UnionType>type).types, f) : f(type);
        }

        function filterType(type: Type, f: (t: Type) => boolean): Type {
            if (type.flags & TypeFlags.Union) {
                const types = (<UnionType>type).types;
                const filtered = filter(types, f);
                return filtered === types ? type : getUnionTypeFromSortedList(filtered);
            }
            return f(type) ? type : neverType;
        }

        // Apply a mapping function to a type and return the resulting type. If the source type
        // is a union type, the mapping function is applied to each constituent type and a union
        // of the resulting types is returned.
        function mapType(type: Type, mapper: (t: Type) => Type, noReductions?: boolean): Type {
            if (type.flags & TypeFlags.Never) {
                return type;
            }
            if (!(type.flags & TypeFlags.Union)) {
                return mapper(type);
            }
            const types = (<UnionType>type).types;
            let mappedType: Type;
            let mappedTypes: Type[];
            for (const current of types) {
                const t = mapper(current);
                if (t) {
                    if (!mappedType) {
                        mappedType = t;
                    }
                    else if (!mappedTypes) {
                        mappedTypes = [mappedType, t];
                    }
                    else {
                        mappedTypes.push(t);
                    }
                }
            }
            return mappedTypes ? getUnionType(mappedTypes, noReductions ? UnionReduction.None : UnionReduction.Literal) : mappedType;
        }

        function extractTypesOfKind(type: Type, kind: TypeFlags) {
            return filterType(type, t => (t.flags & kind) !== 0);
        }

        // Return a new type in which occurrences of the string and number primitive types in
        // typeWithPrimitives have been replaced with occurrences of string literals and numeric
        // literals in typeWithLiterals, respectively.
        function replacePrimitivesWithLiterals(typeWithPrimitives: Type, typeWithLiterals: Type) {
            if (isTypeSubsetOf(stringType, typeWithPrimitives) && maybeTypeOfKind(typeWithLiterals, TypeFlags.StringLiteral) ||
                isTypeSubsetOf(numberType, typeWithPrimitives) && maybeTypeOfKind(typeWithLiterals, TypeFlags.NumberLiteral)) {
                return mapType(typeWithPrimitives, t =>
                    t.flags & TypeFlags.String ? extractTypesOfKind(typeWithLiterals, TypeFlags.String | TypeFlags.StringLiteral) :
                        t.flags & TypeFlags.Number ? extractTypesOfKind(typeWithLiterals, TypeFlags.Number | TypeFlags.NumberLiteral) :
                            t);
            }
            return typeWithPrimitives;
        }

        function isIncomplete(flowType: FlowType) {
            return flowType.flags === 0;
        }

        function getTypeFromFlowType(flowType: FlowType) {
            return flowType.flags === 0 ? (<IncompleteType>flowType).type : <Type>flowType;
        }

        function createFlowType(type: Type, incomplete: boolean): FlowType {
            return incomplete ? { flags: 0, type } : type;
        }

        // An evolving array type tracks the element types that have so far been seen in an
        // 'x.push(value)' or 'x[n] = value' operation along the control flow graph. Evolving
        // array types are ultimately converted into manifest array types (using getFinalArrayType)
        // and never escape the getFlowTypeOfReference function.
        function createEvolvingArrayType(elementType: Type): EvolvingArrayType {
            const result = <EvolvingArrayType>createObjectType(ObjectFlags.EvolvingArray);
            result.elementType = elementType;
            return result;
        }

        function getEvolvingArrayType(elementType: Type): EvolvingArrayType {
            return evolvingArrayTypes[elementType.id] || (evolvingArrayTypes[elementType.id] = createEvolvingArrayType(elementType));
        }

        // When adding evolving array element types we do not perform subtype reduction. Instead,
        // we defer subtype reduction until the evolving array type is finalized into a manifest
        // array type.
        function addEvolvingArrayElementType(evolvingArrayType: EvolvingArrayType, node: Expression): EvolvingArrayType {
            const elementType = getBaseTypeOfLiteralType(getContextFreeTypeOfExpression(node));
            return isTypeSubsetOf(elementType, evolvingArrayType.elementType) ? evolvingArrayType : getEvolvingArrayType(getUnionType([evolvingArrayType.elementType, elementType]));
        }

        function createFinalArrayType(elementType: Type) {
            return elementType.flags & TypeFlags.Never ?
                autoArrayType :
                createArrayType(elementType.flags & TypeFlags.Union ?
                    getUnionType((<UnionType>elementType).types, UnionReduction.Subtype) :
                    elementType);
        }

        // We perform subtype reduction upon obtaining the final array type from an evolving array type.
        function getFinalArrayType(evolvingArrayType: EvolvingArrayType): Type {
            return evolvingArrayType.finalArrayType || (evolvingArrayType.finalArrayType = createFinalArrayType(evolvingArrayType.elementType));
        }

        function finalizeEvolvingArrayType(type: Type): Type {
            return getObjectFlags(type) & ObjectFlags.EvolvingArray ? getFinalArrayType(<EvolvingArrayType>type) : type;
        }

        function getElementTypeOfEvolvingArrayType(type: Type) {
            return getObjectFlags(type) & ObjectFlags.EvolvingArray ? (<EvolvingArrayType>type).elementType : neverType;
        }

        function isEvolvingArrayTypeList(types: Type[]) {
            let hasEvolvingArrayType = false;
            for (const t of types) {
                if (!(t.flags & TypeFlags.Never)) {
                    if (!(getObjectFlags(t) & ObjectFlags.EvolvingArray)) {
                        return false;
                    }
                    hasEvolvingArrayType = true;
                }
            }
            return hasEvolvingArrayType;
        }

        // At flow control branch or loop junctions, if the type along every antecedent code path
        // is an evolving array type, we construct a combined evolving array type. Otherwise we
        // finalize all evolving array types.
        function getUnionOrEvolvingArrayType(types: Type[], subtypeReduction: UnionReduction) {
            return isEvolvingArrayTypeList(types) ?
                getEvolvingArrayType(getUnionType(map(types, getElementTypeOfEvolvingArrayType))) :
                getUnionType(sameMap(types, finalizeEvolvingArrayType), subtypeReduction);
        }

        // Return true if the given node is 'x' in an 'x.length', x.push(value)', 'x.unshift(value)' or
        // 'x[n] = value' operation, where 'n' is an expression of type any, undefined, or a number-like type.
        function isEvolvingArrayOperationTarget(node: Node) {
            const root = getReferenceRoot(node);
            const parent = root.parent;
            const isLengthPushOrUnshift = parent.kind === SyntaxKind.PropertyAccessExpression && (
                (<PropertyAccessExpression>parent).name.escapedText === "length" ||
                parent.parent.kind === SyntaxKind.CallExpression && isPushOrUnshiftIdentifier((<PropertyAccessExpression>parent).name));
            const isElementAssignment = parent.kind === SyntaxKind.ElementAccessExpression &&
                (<ElementAccessExpression>parent).expression === root &&
                parent.parent.kind === SyntaxKind.BinaryExpression &&
                (<BinaryExpression>parent.parent).operatorToken.kind === SyntaxKind.EqualsToken &&
                (<BinaryExpression>parent.parent).left === parent &&
                !isAssignmentTarget(parent.parent) &&
                isTypeAssignableToKind(getTypeOfExpression((<ElementAccessExpression>parent).argumentExpression), TypeFlags.NumberLike);
            return isLengthPushOrUnshift || isElementAssignment;
        }

        function maybeTypePredicateCall(node: CallExpression) {
            const links = getNodeLinks(node);
            if (links.maybeTypePredicate === undefined) {
                links.maybeTypePredicate = getMaybeTypePredicate(node);
            }
            return links.maybeTypePredicate;
        }

        function getMaybeTypePredicate(node: CallExpression) {
            if (node.expression.kind !== SyntaxKind.SuperKeyword) {
                const funcType = checkNonNullExpression(node.expression);
                if (funcType !== silentNeverType) {
                    const apparentType = getApparentType(funcType);
                    return apparentType !== unknownType && some(getSignaturesOfType(apparentType, SignatureKind.Call), signatureHasTypePredicate);
                }
            }
            return false;
        }

        function reportFlowControlError(node: Node) {
            const block = <Block | ModuleBlock | SourceFile>findAncestor(node, isFunctionOrModuleBlock);
            const sourceFile = getSourceFileOfNode(node);
            const span = getSpanOfTokenAtPosition(sourceFile, block.statements.pos);
            diagnostics.add(createFileDiagnostic(sourceFile, span.start, span.length, Diagnostics.The_containing_function_or_module_body_is_too_large_for_control_flow_analysis));
        }

        function getFlowTypeOfReference(reference: Node, declaredType: Type, initialType = declaredType, flowContainer?: Node, couldBeUninitialized?: boolean) {
            let key: string;
            let flowDepth = 0;
            if (flowAnalysisDisabled) {
                return unknownType;
            }
            if (!reference.flowNode || !couldBeUninitialized && !(declaredType.flags & TypeFlags.Narrowable)) {
                return declaredType;
            }
            const sharedFlowStart = sharedFlowCount;
            const evolvedType = getTypeFromFlowType(getTypeAtFlowNode(reference.flowNode));
            sharedFlowCount = sharedFlowStart;
            // When the reference is 'x' in an 'x.length', 'x.push(value)', 'x.unshift(value)' or x[n] = value' operation,
            // we give type 'any[]' to 'x' instead of using the type determined by control flow analysis such that operations
            // on empty arrays are possible without implicit any errors and new element types can be inferred without
            // type mismatch errors.
            const resultType = getObjectFlags(evolvedType) & ObjectFlags.EvolvingArray && isEvolvingArrayOperationTarget(reference) ? anyArrayType : finalizeEvolvingArrayType(evolvedType);
            if (reference.parent && reference.parent.kind === SyntaxKind.NonNullExpression && getTypeWithFacts(resultType, TypeFacts.NEUndefinedOrNull).flags & TypeFlags.Never) {
                return declaredType;
            }
            return resultType;

            function getTypeAtFlowNode(flow: FlowNode): FlowType {
                if (flowDepth === 2500) {
                    // We have made 2500 recursive invocations. To avoid overflowing the call stack we report an error
                    // and disable further control flow analysis in the containing function or module body.
                    flowAnalysisDisabled = true;
                    reportFlowControlError(reference);
                    return unknownType;
                }
                flowDepth++;
                while (true) {
                    const flags = flow.flags;
                    if (flags & FlowFlags.Shared) {
                        // We cache results of flow type resolution for shared nodes that were previously visited in
                        // the same getFlowTypeOfReference invocation. A node is considered shared when it is the
                        // antecedent of more than one node.
                        for (let i = sharedFlowStart; i < sharedFlowCount; i++) {
                            if (sharedFlowNodes[i] === flow) {
                                flowDepth--;
                                return sharedFlowTypes[i];
                            }
                        }
                    }
                    let type: FlowType;
                    if (flags & FlowFlags.AfterFinally) {
                        // block flow edge: finally -> pre-try (for larger explanation check comment in binder.ts - bindTryStatement
                        (<AfterFinallyFlow>flow).locked = true;
                        type = getTypeAtFlowNode((<AfterFinallyFlow>flow).antecedent);
                        (<AfterFinallyFlow>flow).locked = false;
                    }
                    else if (flags & FlowFlags.PreFinally) {
                        // locked pre-finally flows are filtered out in getTypeAtFlowBranchLabel
                        // so here just redirect to antecedent
                        flow = (<PreFinallyFlow>flow).antecedent;
                        continue;
                    }
                    else if (flags & FlowFlags.Assignment) {
                        type = getTypeAtFlowAssignment(<FlowAssignment>flow);
                        if (!type) {
                            flow = (<FlowAssignment>flow).antecedent;
                            continue;
                        }
                    }
                    else if (flags & FlowFlags.Condition) {
                        type = getTypeAtFlowCondition(<FlowCondition>flow);
                    }
                    else if (flags & FlowFlags.SwitchClause) {
                        type = getTypeAtSwitchClause(<FlowSwitchClause>flow);
                    }
                    else if (flags & FlowFlags.Label) {
                        if ((<FlowLabel>flow).antecedents.length === 1) {
                            flow = (<FlowLabel>flow).antecedents[0];
                            continue;
                        }
                        type = flags & FlowFlags.BranchLabel ?
                            getTypeAtFlowBranchLabel(<FlowLabel>flow) :
                            getTypeAtFlowLoopLabel(<FlowLabel>flow);
                    }
                    else if (flags & FlowFlags.ArrayMutation) {
                        type = getTypeAtFlowArrayMutation(<FlowArrayMutation>flow);
                        if (!type) {
                            flow = (<FlowArrayMutation>flow).antecedent;
                            continue;
                        }
                    }
                    else if (flags & FlowFlags.Start) {
                        // Check if we should continue with the control flow of the containing function.
                        const container = (<FlowStart>flow).container;
                        if (container && container !== flowContainer && reference.kind !== SyntaxKind.PropertyAccessExpression && reference.kind !== SyntaxKind.ThisKeyword) {
                            flow = container.flowNode;
                            continue;
                        }
                        // At the top of the flow we have the initial type.
                        type = initialType;
                    }
                    else {
                        // Unreachable code errors are reported in the binding phase. Here we
                        // simply return the non-auto declared type to reduce follow-on errors.
                        type = convertAutoToAny(declaredType);
                    }
                    if (flags & FlowFlags.Shared) {
                        // Record visited node and the associated type in the cache.
                        sharedFlowNodes[sharedFlowCount] = flow;
                        sharedFlowTypes[sharedFlowCount] = type;
                        sharedFlowCount++;
                    }
                    flowDepth--;
                    return type;
                }
            }

            function getTypeAtFlowAssignment(flow: FlowAssignment) {
                const node = flow.node;
                // Assignments only narrow the computed type if the declared type is a union type. Thus, we
                // only need to evaluate the assigned type if the declared type is a union type.
                if (isMatchingReference(reference, node)) {
                    if (getAssignmentTargetKind(node) === AssignmentKind.Compound) {
                        const flowType = getTypeAtFlowNode(flow.antecedent);
                        return createFlowType(getBaseTypeOfLiteralType(getTypeFromFlowType(flowType)), isIncomplete(flowType));
                    }
                    if (declaredType === autoType || declaredType === autoArrayType) {
                        if (isEmptyArrayAssignment(node)) {
                            return getEvolvingArrayType(neverType);
                        }
                        const assignedType = getBaseTypeOfLiteralType(getInitialOrAssignedType(node));
                        return isTypeAssignableTo(assignedType, declaredType) ? assignedType : anyArrayType;
                    }
                    if (declaredType.flags & TypeFlags.Union) {
                        return getAssignmentReducedType(<UnionType>declaredType, getInitialOrAssignedType(node));
                    }
                    return declaredType;
                }
                // We didn't have a direct match. However, if the reference is a dotted name, this
                // may be an assignment to a left hand part of the reference. For example, for a
                // reference 'x.y.z', we may be at an assignment to 'x.y' or 'x'. In that case,
                // return the declared type.
                if (containsMatchingReference(reference, node)) {
                    return declaredType;
                }
                // Assignment doesn't affect reference
                return undefined;
            }

            function getTypeAtFlowArrayMutation(flow: FlowArrayMutation): FlowType {
                if (declaredType === autoType || declaredType === autoArrayType) {
                    const node = flow.node;
                    const expr = node.kind === SyntaxKind.CallExpression ?
                        (<PropertyAccessExpression>node.expression).expression :
                        (<ElementAccessExpression>node.left).expression;
                    if (isMatchingReference(reference, getReferenceCandidate(expr))) {
                        const flowType = getTypeAtFlowNode(flow.antecedent);
                        const type = getTypeFromFlowType(flowType);
                        if (getObjectFlags(type) & ObjectFlags.EvolvingArray) {
                            let evolvedType = <EvolvingArrayType>type;
                            if (node.kind === SyntaxKind.CallExpression) {
                                for (const arg of node.arguments) {
                                    evolvedType = addEvolvingArrayElementType(evolvedType, arg);
                                }
                            }
                            else {
                                const indexType = getTypeOfExpression((<ElementAccessExpression>node.left).argumentExpression);
                                if (isTypeAssignableToKind(indexType, TypeFlags.NumberLike)) {
                                    evolvedType = addEvolvingArrayElementType(evolvedType, node.right);
                                }
                            }
                            return evolvedType === type ? flowType : createFlowType(evolvedType, isIncomplete(flowType));
                        }
                        return flowType;
                    }
                }
                return undefined;
            }

            function getTypeAtFlowCondition(flow: FlowCondition): FlowType {
                const flowType = getTypeAtFlowNode(flow.antecedent);
                const type = getTypeFromFlowType(flowType);
                if (type.flags & TypeFlags.Never) {
                    return flowType;
                }
                // If we have an antecedent type (meaning we're reachable in some way), we first
                // attempt to narrow the antecedent type. If that produces the never type, and if
                // the antecedent type is incomplete (i.e. a transient type in a loop), then we
                // take the type guard as an indication that control *could* reach here once we
                // have the complete type. We proceed by switching to the silent never type which
                // doesn't report errors when operators are applied to it. Note that this is the
                // *only* place a silent never type is ever generated.
                const assumeTrue = (flow.flags & FlowFlags.TrueCondition) !== 0;
                const nonEvolvingType = finalizeEvolvingArrayType(type);
                const narrowedType = narrowType(nonEvolvingType, flow.expression, assumeTrue);
                if (narrowedType === nonEvolvingType) {
                    return flowType;
                }
                const incomplete = isIncomplete(flowType);
                const resultType = incomplete && narrowedType.flags & TypeFlags.Never ? silentNeverType : narrowedType;
                return createFlowType(resultType, incomplete);
            }

            function getTypeAtSwitchClause(flow: FlowSwitchClause): FlowType {
                const flowType = getTypeAtFlowNode(flow.antecedent);
                let type = getTypeFromFlowType(flowType);
                const expr = flow.switchStatement.expression;
                if (isMatchingReference(reference, expr)) {
                    type = narrowTypeBySwitchOnDiscriminant(type, flow.switchStatement, flow.clauseStart, flow.clauseEnd);
                }
                else if (isMatchingReferenceDiscriminant(expr, type)) {
                    type = narrowTypeByDiscriminant(type, <PropertyAccessExpression>expr, t => narrowTypeBySwitchOnDiscriminant(t, flow.switchStatement, flow.clauseStart, flow.clauseEnd));
                }
                return createFlowType(type, isIncomplete(flowType));
            }

            function getTypeAtFlowBranchLabel(flow: FlowLabel): FlowType {
                const antecedentTypes: Type[] = [];
                let subtypeReduction = false;
                let seenIncomplete = false;
                for (const antecedent of flow.antecedents) {
                    if (antecedent.flags & FlowFlags.PreFinally && (<PreFinallyFlow>antecedent).lock.locked) {
                        // if flow correspond to branch from pre-try to finally and this branch is locked - this means that
                        // we initially have started following the flow outside the finally block.
                        // in this case we should ignore this branch.
                        continue;
                    }
                    const flowType = getTypeAtFlowNode(antecedent);
                    const type = getTypeFromFlowType(flowType);
                    // If the type at a particular antecedent path is the declared type and the
                    // reference is known to always be assigned (i.e. when declared and initial types
                    // are the same), there is no reason to process more antecedents since the only
                    // possible outcome is subtypes that will be removed in the final union type anyway.
                    if (type === declaredType && declaredType === initialType) {
                        return type;
                    }
                    pushIfUnique(antecedentTypes, type);
                    // If an antecedent type is not a subset of the declared type, we need to perform
                    // subtype reduction. This happens when a "foreign" type is injected into the control
                    // flow using the instanceof operator or a user defined type predicate.
                    if (!isTypeSubsetOf(type, declaredType)) {
                        subtypeReduction = true;
                    }
                    if (isIncomplete(flowType)) {
                        seenIncomplete = true;
                    }
                }
                return createFlowType(getUnionOrEvolvingArrayType(antecedentTypes, subtypeReduction ? UnionReduction.Subtype : UnionReduction.Literal), seenIncomplete);
            }

            function getTypeAtFlowLoopLabel(flow: FlowLabel): FlowType {
                // If we have previously computed the control flow type for the reference at
                // this flow loop junction, return the cached type.
                const id = getFlowNodeId(flow);
                const cache = flowLoopCaches[id] || (flowLoopCaches[id] = createMap<Type>());
                if (!key) {
                    key = getFlowCacheKey(reference);
                    // No cache key is generated when binding patterns are in unnarrowable situations
                    if (!key) {
                        return declaredType;
                    }
                }
                const cached = cache.get(key);
                if (cached) {
                    return cached;
                }
                // If this flow loop junction and reference are already being processed, return
                // the union of the types computed for each branch so far, marked as incomplete.
                // It is possible to see an empty array in cases where loops are nested and the
                // back edge of the outer loop reaches an inner loop that is already being analyzed.
                // In such cases we restart the analysis of the inner loop, which will then see
                // a non-empty in-process array for the outer loop and eventually terminate because
                // the first antecedent of a loop junction is always the non-looping control flow
                // path that leads to the top.
                for (let i = flowLoopStart; i < flowLoopCount; i++) {
                    if (flowLoopNodes[i] === flow && flowLoopKeys[i] === key && flowLoopTypes[i].length) {
                        return createFlowType(getUnionOrEvolvingArrayType(flowLoopTypes[i], UnionReduction.Literal), /*incomplete*/ true);
                    }
                }
                // Add the flow loop junction and reference to the in-process stack and analyze
                // each antecedent code path.
                const antecedentTypes: Type[] = [];
                let subtypeReduction = false;
                let firstAntecedentType: FlowType;
                flowLoopNodes[flowLoopCount] = flow;
                flowLoopKeys[flowLoopCount] = key;
                flowLoopTypes[flowLoopCount] = antecedentTypes;
                for (const antecedent of flow.antecedents) {
                    flowLoopCount++;
                    const flowType = getTypeAtFlowNode(antecedent);
                    flowLoopCount--;
                    if (!firstAntecedentType) {
                        firstAntecedentType = flowType;
                    }
                    const type = getTypeFromFlowType(flowType);
                    // If we see a value appear in the cache it is a sign that control flow analysis
                    // was restarted and completed by checkExpressionCached. We can simply pick up
                    // the resulting type and bail out.
                    const cached = cache.get(key);
                    if (cached) {
                        return cached;
                    }
                    pushIfUnique(antecedentTypes, type);
                    // If an antecedent type is not a subset of the declared type, we need to perform
                    // subtype reduction. This happens when a "foreign" type is injected into the control
                    // flow using the instanceof operator or a user defined type predicate.
                    if (!isTypeSubsetOf(type, declaredType)) {
                        subtypeReduction = true;
                    }
                    // If the type at a particular antecedent path is the declared type there is no
                    // reason to process more antecedents since the only possible outcome is subtypes
                    // that will be removed in the final union type anyway.
                    if (type === declaredType) {
                        break;
                    }
                }
                // The result is incomplete if the first antecedent (the non-looping control flow path)
                // is incomplete.
                const result = getUnionOrEvolvingArrayType(antecedentTypes, subtypeReduction ? UnionReduction.Subtype : UnionReduction.Literal);
                if (isIncomplete(firstAntecedentType)) {
                    return createFlowType(result, /*incomplete*/ true);
                }
                cache.set(key, result);
                return result;
            }

            function isMatchingReferenceDiscriminant(expr: Expression, computedType: Type) {
                return expr.kind === SyntaxKind.PropertyAccessExpression &&
                    computedType.flags & TypeFlags.Union &&
                    isMatchingReference(reference, (<PropertyAccessExpression>expr).expression) &&
                    isDiscriminantProperty(computedType, (<PropertyAccessExpression>expr).name.escapedText);
            }

            function narrowTypeByDiscriminant(type: Type, propAccess: PropertyAccessExpression, narrowType: (t: Type) => Type): Type {
                const propName = propAccess.name.escapedText;
                const propType = getTypeOfPropertyOfType(type, propName);
                const narrowedPropType = propType && narrowType(propType);
                return propType === narrowedPropType ? type : filterType(type, t => isTypeComparableTo(getTypeOfPropertyOfType(t, propName), narrowedPropType));
            }

            function narrowTypeByTruthiness(type: Type, expr: Expression, assumeTrue: boolean): Type {
                if (isMatchingReference(reference, expr)) {
                    return getTypeWithFacts(type, assumeTrue ? TypeFacts.Truthy : TypeFacts.Falsy);
                }
                if (isMatchingReferenceDiscriminant(expr, declaredType)) {
                    return narrowTypeByDiscriminant(type, <PropertyAccessExpression>expr, t => getTypeWithFacts(t, assumeTrue ? TypeFacts.Truthy : TypeFacts.Falsy));
                }
                if (containsMatchingReferenceDiscriminant(reference, expr)) {
                    return declaredType;
                }
                return type;
            }

            function isTypePresencePossible(type: Type, propName: __String, assumeTrue: boolean) {
                if (getIndexInfoOfType(type, IndexKind.String)) {
                    return true;
                }
                const prop = getPropertyOfType(type, propName);
                if (prop) {
                    return prop.flags & SymbolFlags.Optional ? true : assumeTrue;
                }
                return !assumeTrue;
            }

            function narrowByInKeyword(type: Type, literal: LiteralExpression, assumeTrue: boolean) {
                if ((type.flags & (TypeFlags.Union | TypeFlags.Object)) || (type.flags & TypeFlags.TypeParameter && (type as TypeParameter).isThisType)) {
                    const propName = escapeLeadingUnderscores(literal.text);
                    return filterType(type, t => isTypePresencePossible(t, propName, assumeTrue));
                }
                return type;
            }

            function narrowTypeByBinaryExpression(type: Type, expr: BinaryExpression, assumeTrue: boolean): Type {
                switch (expr.operatorToken.kind) {
                    case SyntaxKind.EqualsToken:
                        return narrowTypeByTruthiness(type, expr.left, assumeTrue);
                    case SyntaxKind.EqualsEqualsToken:
                    case SyntaxKind.ExclamationEqualsToken:
                    case SyntaxKind.EqualsEqualsEqualsToken:
                    case SyntaxKind.ExclamationEqualsEqualsToken:
                        const operator = expr.operatorToken.kind;
                        const left = getReferenceCandidate(expr.left);
                        const right = getReferenceCandidate(expr.right);
                        if (left.kind === SyntaxKind.TypeOfExpression && isStringLiteralLike(right)) {
                            return narrowTypeByTypeof(type, <TypeOfExpression>left, operator, right, assumeTrue);
                        }
                        if (right.kind === SyntaxKind.TypeOfExpression && isStringLiteralLike(left)) {
                            return narrowTypeByTypeof(type, <TypeOfExpression>right, operator, left, assumeTrue);
                        }
                        if (isMatchingReference(reference, left)) {
                            return narrowTypeByEquality(type, operator, right, assumeTrue);
                        }
                        if (isMatchingReference(reference, right)) {
                            return narrowTypeByEquality(type, operator, left, assumeTrue);
                        }
                        if (isMatchingReferenceDiscriminant(left, declaredType)) {
                            return narrowTypeByDiscriminant(type, <PropertyAccessExpression>left, t => narrowTypeByEquality(t, operator, right, assumeTrue));
                        }
                        if (isMatchingReferenceDiscriminant(right, declaredType)) {
                            return narrowTypeByDiscriminant(type, <PropertyAccessExpression>right, t => narrowTypeByEquality(t, operator, left, assumeTrue));
                        }
                        if (containsMatchingReferenceDiscriminant(reference, left) || containsMatchingReferenceDiscriminant(reference, right)) {
                            return declaredType;
                        }
                        break;
                    case SyntaxKind.InstanceOfKeyword:
                        return narrowTypeByInstanceof(type, expr, assumeTrue);
                    case SyntaxKind.InKeyword:
                        const target = getReferenceCandidate(expr.right);
                        if (isStringLiteralLike(expr.left) && isMatchingReference(reference, target)) {
                            return narrowByInKeyword(type, expr.left, assumeTrue);
                        }
                        break;
                    case SyntaxKind.CommaToken:
                        return narrowType(type, expr.right, assumeTrue);
                }
                return type;
            }

            function narrowTypeByEquality(type: Type, operator: SyntaxKind, value: Expression, assumeTrue: boolean): Type {
                if (type.flags & TypeFlags.Any) {
                    return type;
                }
                if (operator === SyntaxKind.ExclamationEqualsToken || operator === SyntaxKind.ExclamationEqualsEqualsToken) {
                    assumeTrue = !assumeTrue;
                }
                const valueType = getTypeOfExpression(value);
                if (valueType.flags & TypeFlags.Nullable) {
                    if (!strictNullChecks) {
                        return type;
                    }
                    const doubleEquals = operator === SyntaxKind.EqualsEqualsToken || operator === SyntaxKind.ExclamationEqualsToken;
                    const facts = doubleEquals ?
                        assumeTrue ? TypeFacts.EQUndefinedOrNull : TypeFacts.NEUndefinedOrNull :
                        value.kind === SyntaxKind.NullKeyword ?
                            assumeTrue ? TypeFacts.EQNull : TypeFacts.NENull :
                            assumeTrue ? TypeFacts.EQUndefined : TypeFacts.NEUndefined;
                    return getTypeWithFacts(type, facts);
                }
                if (type.flags & TypeFlags.NotUnionOrUnit) {
                    return type;
                }
                if (assumeTrue) {
                    const narrowedType = filterType(type, t => areTypesComparable(t, valueType));
                    return narrowedType.flags & TypeFlags.Never ? type : replacePrimitivesWithLiterals(narrowedType, valueType);
                }
                if (isUnitType(valueType)) {
                    const regularType = getRegularTypeOfLiteralType(valueType);
                    return filterType(type, t => getRegularTypeOfLiteralType(t) !== regularType);
                }
                return type;
            }

            function narrowTypeByTypeof(type: Type, typeOfExpr: TypeOfExpression, operator: SyntaxKind, literal: LiteralExpression, assumeTrue: boolean): Type {
                // We have '==', '!=', '====', or !==' operator with 'typeof xxx' and string literal operands
                const target = getReferenceCandidate(typeOfExpr.expression);
                if (!isMatchingReference(reference, target)) {
                    // For a reference of the form 'x.y', a 'typeof x === ...' type guard resets the
                    // narrowed type of 'y' to its declared type.
                    if (containsMatchingReference(reference, target)) {
                        return declaredType;
                    }
                    return type;
                }
                if (operator === SyntaxKind.ExclamationEqualsToken || operator === SyntaxKind.ExclamationEqualsEqualsToken) {
                    assumeTrue = !assumeTrue;
                }
                if (assumeTrue && !(type.flags & TypeFlags.Union)) {
                    // We narrow a non-union type to an exact primitive type if the non-union type
                    // is a supertype of that primitive type. For example, type 'any' can be narrowed
                    // to one of the primitive types.
                    const targetType = typeofTypesByName.get(literal.text);
                    if (targetType) {
                        if (isTypeSubtypeOf(targetType, type)) {
                            return targetType;
                        }
                        if (type.flags & TypeFlags.Instantiable) {
                            const constraint = getBaseConstraintOfType(type) || anyType;
                            if (isTypeSubtypeOf(targetType, constraint)) {
                                return getIntersectionType([type, targetType]);
                            }
                        }
                    }
                }
                const facts = assumeTrue ?
                    typeofEQFacts.get(literal.text) || TypeFacts.TypeofEQHostObject :
                    typeofNEFacts.get(literal.text) || TypeFacts.TypeofNEHostObject;
                return getTypeWithFacts(type, facts);
            }

            function narrowTypeBySwitchOnDiscriminant(type: Type, switchStatement: SwitchStatement, clauseStart: number, clauseEnd: number) {
                // We only narrow if all case expressions specify values with unit types
                const switchTypes = getSwitchClauseTypes(switchStatement);
                if (!switchTypes.length) {
                    return type;
                }
                const clauseTypes = switchTypes.slice(clauseStart, clauseEnd);
                const hasDefaultClause = clauseStart === clauseEnd || contains(clauseTypes, neverType);
                const discriminantType = getUnionType(clauseTypes);
                const caseType =
                    discriminantType.flags & TypeFlags.Never ? neverType :
                        replacePrimitivesWithLiterals(filterType(type, t => areTypesComparable(discriminantType, t)), discriminantType);
                if (!hasDefaultClause) {
                    return caseType;
                }
                const defaultType = filterType(type, t => !(isUnitType(t) && contains(switchTypes, getRegularTypeOfLiteralType(t))));
                return caseType.flags & TypeFlags.Never ? defaultType : getUnionType([caseType, defaultType]);
            }

            function narrowTypeByInstanceof(type: Type, expr: BinaryExpression, assumeTrue: boolean): Type {
                const left = getReferenceCandidate(expr.left);
                if (!isMatchingReference(reference, left)) {
                    // For a reference of the form 'x.y', an 'x instanceof T' type guard resets the
                    // narrowed type of 'y' to its declared type.
                    if (containsMatchingReference(reference, left)) {
                        return declaredType;
                    }
                    return type;
                }

                // Check that right operand is a function type with a prototype property
                const rightType = getTypeOfExpression(expr.right);
                if (!isTypeSubtypeOf(rightType, globalFunctionType)) {
                    return type;
                }

                let targetType: Type;
                const prototypeProperty = getPropertyOfType(rightType, "prototype" as __String);
                if (prototypeProperty) {
                    // Target type is type of the prototype property
                    const prototypePropertyType = getTypeOfSymbol(prototypeProperty);
                    if (!isTypeAny(prototypePropertyType)) {
                        targetType = prototypePropertyType;
                    }
                }

                // Don't narrow from 'any' if the target type is exactly 'Object' or 'Function'
                if (isTypeAny(type) && (targetType === globalObjectType || targetType === globalFunctionType)) {
                    return type;
                }

                if (!targetType) {
                    // Target type is type of construct signature
                    let constructSignatures: Signature[];
                    if (getObjectFlags(rightType) & ObjectFlags.Interface) {
                        constructSignatures = resolveDeclaredMembers(<InterfaceType>rightType).declaredConstructSignatures;
                    }
                    else if (getObjectFlags(rightType) & ObjectFlags.Anonymous) {
                        constructSignatures = getSignaturesOfType(rightType, SignatureKind.Construct);
                    }
                    if (constructSignatures && constructSignatures.length) {
                        targetType = getUnionType(map(constructSignatures, signature => getReturnTypeOfSignature(getErasedSignature(signature))));
                    }
                }

                if (targetType) {
                    return getNarrowedType(type, targetType, assumeTrue, isTypeDerivedFrom);
                }

                return type;
            }

            function getNarrowedType(type: Type, candidate: Type, assumeTrue: boolean, isRelated: (source: Type, target: Type) => boolean) {
                if (!assumeTrue) {
                    return filterType(type, t => !isRelated(t, candidate));
                }
                // If the current type is a union type, remove all constituents that couldn't be instances of
                // the candidate type. If one or more constituents remain, return a union of those.
                if (type.flags & TypeFlags.Union) {
                    const assignableType = filterType(type, t => isRelated(t, candidate));
                    if (!(assignableType.flags & TypeFlags.Never)) {
                        return assignableType;
                    }
                }
                // If the candidate type is a subtype of the target type, narrow to the candidate type.
                // Otherwise, if the target type is assignable to the candidate type, keep the target type.
                // Otherwise, if the candidate type is assignable to the target type, narrow to the candidate
                // type. Otherwise, the types are completely unrelated, so narrow to an intersection of the
                // two types.
                return isTypeSubtypeOf(candidate, type) ? candidate :
                    isTypeAssignableTo(type, candidate) ? type :
                        isTypeAssignableTo(candidate, type) ? candidate :
                            getIntersectionType([type, candidate]);
            }

            function narrowTypeByTypePredicate(type: Type, callExpression: CallExpression, assumeTrue: boolean): Type {
                if (!hasMatchingArgument(callExpression, reference) || !maybeTypePredicateCall(callExpression)) {
                    return type;
                }
                const signature = getResolvedSignature(callExpression);
                const predicate = getTypePredicateOfSignature(signature);
                if (!predicate) {
                    return type;
                }

                // Don't narrow from 'any' if the predicate type is exactly 'Object' or 'Function'
                if (isTypeAny(type) && (predicate.type === globalObjectType || predicate.type === globalFunctionType)) {
                    return type;
                }

                if (isIdentifierTypePredicate(predicate)) {
                    const predicateArgument = callExpression.arguments[predicate.parameterIndex - (signature.thisParameter ? 1 : 0)];
                    if (predicateArgument) {
                        if (isMatchingReference(reference, predicateArgument)) {
                            return getNarrowedType(type, predicate.type, assumeTrue, isTypeSubtypeOf);
                        }
                        if (containsMatchingReference(reference, predicateArgument)) {
                            return declaredType;
                        }
                    }
                }
                else {
                    const invokedExpression = skipParentheses(callExpression.expression);
                    if (invokedExpression.kind === SyntaxKind.ElementAccessExpression || invokedExpression.kind === SyntaxKind.PropertyAccessExpression) {
                        const accessExpression = invokedExpression as ElementAccessExpression | PropertyAccessExpression;
                        const possibleReference = skipParentheses(accessExpression.expression);
                        if (isMatchingReference(reference, possibleReference)) {
                            return getNarrowedType(type, predicate.type, assumeTrue, isTypeSubtypeOf);
                        }
                        if (containsMatchingReference(reference, possibleReference)) {
                            return declaredType;
                        }
                    }
                }
                return type;
            }

            // Narrow the given type based on the given expression having the assumed boolean value. The returned type
            // will be a subtype or the same type as the argument.
            function narrowType(type: Type, expr: Expression, assumeTrue: boolean): Type {
                switch (expr.kind) {
                    case SyntaxKind.Identifier:
                    case SyntaxKind.ThisKeyword:
                    case SyntaxKind.SuperKeyword:
                    case SyntaxKind.PropertyAccessExpression:
                        return narrowTypeByTruthiness(type, expr, assumeTrue);
                    case SyntaxKind.CallExpression:
                        return narrowTypeByTypePredicate(type, <CallExpression>expr, assumeTrue);
                    case SyntaxKind.ParenthesizedExpression:
                        return narrowType(type, (<ParenthesizedExpression>expr).expression, assumeTrue);
                    case SyntaxKind.BinaryExpression:
                        return narrowTypeByBinaryExpression(type, <BinaryExpression>expr, assumeTrue);
                    case SyntaxKind.PrefixUnaryExpression:
                        if ((<PrefixUnaryExpression>expr).operator === SyntaxKind.ExclamationToken) {
                            return narrowType(type, (<PrefixUnaryExpression>expr).operand, !assumeTrue);
                        }
                        break;
                }
                return type;
            }
        }

        function getTypeOfSymbolAtLocation(symbol: Symbol, location: Node) {
            symbol = symbol.exportSymbol || symbol;

            // If we have an identifier or a property access at the given location, if the location is
            // an dotted name expression, and if the location is not an assignment target, obtain the type
            // of the expression (which will reflect control flow analysis). If the expression indeed
            // resolved to the given symbol, return the narrowed type.
            if (location.kind === SyntaxKind.Identifier) {
                if (isRightSideOfQualifiedNameOrPropertyAccess(location)) {
                    location = location.parent;
                }
                if (isExpressionNode(location) && !isAssignmentTarget(location)) {
                    const type = getTypeOfExpression(<Expression>location);
                    if (getExportSymbolOfValueSymbolIfExported(getNodeLinks(location).resolvedSymbol) === symbol) {
                        return type;
                    }
                }
            }
            // The location isn't a reference to the given symbol, meaning we're being asked
            // a hypothetical question of what type the symbol would have if there was a reference
            // to it at the given location. Since we have no control flow information for the
            // hypothetical reference (control flow information is created and attached by the
            // binder), we simply return the declared type of the symbol.
            return getTypeOfSymbol(symbol);
        }

        function getControlFlowContainer(node: Node): Node {
            return findAncestor(node.parent, node =>
                isFunctionLike(node) && !getImmediatelyInvokedFunctionExpression(node) ||
                node.kind === SyntaxKind.ModuleBlock ||
                node.kind === SyntaxKind.SourceFile ||
                node.kind === SyntaxKind.PropertyDeclaration);
        }

        // Check if a parameter is assigned anywhere within its declaring function.
        function isParameterAssigned(symbol: Symbol) {
            const func = <FunctionLikeDeclaration>getRootDeclaration(symbol.valueDeclaration).parent;
            const links = getNodeLinks(func);
            if (!(links.flags & NodeCheckFlags.AssignmentsMarked)) {
                links.flags |= NodeCheckFlags.AssignmentsMarked;
                if (!hasParentWithAssignmentsMarked(func)) {
                    markParameterAssignments(func);
                }
            }
            return symbol.isAssigned || false;
        }

        function hasParentWithAssignmentsMarked(node: Node) {
            return !!findAncestor(node.parent, node => isFunctionLike(node) && !!(getNodeLinks(node).flags & NodeCheckFlags.AssignmentsMarked));
        }

        function markParameterAssignments(node: Node) {
            if (node.kind === SyntaxKind.Identifier) {
                if (isAssignmentTarget(node)) {
                    const symbol = getResolvedSymbol(<Identifier>node);
                    if (symbol.valueDeclaration && getRootDeclaration(symbol.valueDeclaration).kind === SyntaxKind.Parameter) {
                        symbol.isAssigned = true;
                    }
                }
            }
            else {
                forEachChild(node, markParameterAssignments);
            }
        }

        function isConstVariable(symbol: Symbol) {
            return symbol.flags & SymbolFlags.Variable && (getDeclarationNodeFlagsFromSymbol(symbol) & NodeFlags.Const) !== 0 && getTypeOfSymbol(symbol) !== autoArrayType;
        }

        /** remove undefined from the annotated type of a parameter when there is an initializer (that doesn't include undefined) */
        function removeOptionalityFromDeclaredType(declaredType: Type, declaration: VariableLikeDeclaration): Type {
            const annotationIncludesUndefined = strictNullChecks &&
                declaration.kind === SyntaxKind.Parameter &&
                declaration.initializer &&
                getFalsyFlags(declaredType) & TypeFlags.Undefined &&
                !(getFalsyFlags(checkExpression(declaration.initializer)) & TypeFlags.Undefined);
            return annotationIncludesUndefined ? getTypeWithFacts(declaredType, TypeFacts.NEUndefined) : declaredType;
        }

        function isConstraintPosition(node: Node) {
            const parent = node.parent;
            return parent.kind === SyntaxKind.PropertyAccessExpression ||
                parent.kind === SyntaxKind.CallExpression && (<CallExpression>parent).expression === node ||
                parent.kind === SyntaxKind.ElementAccessExpression && (<ElementAccessExpression>parent).expression === node ||
                parent.kind === SyntaxKind.BindingElement && (<BindingElement>parent).name === node && !!(<BindingElement>parent).initializer;
        }

        function typeHasNullableConstraint(type: Type) {
            return type.flags & TypeFlags.InstantiableNonPrimitive && maybeTypeOfKind(getBaseConstraintOfType(type) || emptyObjectType, TypeFlags.Nullable);
        }

        function getConstraintForLocation(type: Type, node: Node) {
            // When a node is the left hand expression of a property access, element access, or call expression,
            // and the type of the node includes type variables with constraints that are nullable, we fetch the
            // apparent type of the node *before* performing control flow analysis such that narrowings apply to
            // the constraint type.
            if (isConstraintPosition(node) && forEachType(type, typeHasNullableConstraint)) {
                return mapType(getWidenedType(type), getBaseConstraintOrType);
            }
            return type;
        }

        function markAliasReferenced(symbol: Symbol, location: Node) {
            if (isNonLocalAlias(symbol, /*excludes*/ SymbolFlags.Value) && !isInTypeQuery(location) && !isConstEnumOrConstEnumOnlyModule(resolveAlias(symbol))) {
                markAliasSymbolAsReferenced(symbol);
            }
        }

        function checkIdentifier(node: Identifier): Type {
            const symbol = getResolvedSymbol(node);
            if (symbol === unknownSymbol) {
                return unknownType;
            }

            // As noted in ECMAScript 6 language spec, arrow functions never have an arguments objects.
            // Although in down-level emit of arrow function, we emit it using function expression which means that
            // arguments objects will be bound to the inner object; emitting arrow function natively in ES6, arguments objects
            // will be bound to non-arrow function that contain this arrow function. This results in inconsistent behavior.
            // To avoid that we will give an error to users if they use arguments objects in arrow function so that they
            // can explicitly bound arguments objects
            if (symbol === argumentsSymbol) {
                const container = getContainingFunction(node);
                if (languageVersion < ScriptTarget.ES2015) {
                    if (container.kind === SyntaxKind.ArrowFunction) {
                        error(node, Diagnostics.The_arguments_object_cannot_be_referenced_in_an_arrow_function_in_ES3_and_ES5_Consider_using_a_standard_function_expression);
                    }
                    else if (hasModifier(container, ModifierFlags.Async)) {
                        error(node, Diagnostics.The_arguments_object_cannot_be_referenced_in_an_async_function_or_method_in_ES3_and_ES5_Consider_using_a_standard_function_or_method);
                    }
                }

                getNodeLinks(container).flags |= NodeCheckFlags.CaptureArguments;
                return getTypeOfSymbol(symbol);
            }

            // We should only mark aliases as referenced if there isn't a local value declaration
            // for the symbol. Also, don't mark any property access expression LHS - checkPropertyAccessExpression will handle that
            if (!(node.parent && isPropertyAccessExpression(node.parent) && node.parent.expression === node)) {
                markAliasReferenced(symbol, node);
            }

            const localOrExportSymbol = getExportSymbolOfValueSymbolIfExported(symbol);
            let declaration = localOrExportSymbol.valueDeclaration;

            if (localOrExportSymbol.flags & SymbolFlags.Class) {
                // Due to the emit for class decorators, any reference to the class from inside of the class body
                // must instead be rewritten to point to a temporary variable to avoid issues with the double-bind
                // behavior of class names in ES6.
                if (declaration.kind === SyntaxKind.ClassDeclaration
                    && nodeIsDecorated(declaration as ClassDeclaration)) {
                    let container = getContainingClass(node);
                    while (container !== undefined) {
                        if (container === declaration && container.name !== node) {
                            getNodeLinks(declaration).flags |= NodeCheckFlags.ClassWithConstructorReference;
                            getNodeLinks(node).flags |= NodeCheckFlags.ConstructorReferenceInClass;
                            break;
                        }

                        container = getContainingClass(container);
                    }
                }
                else if (declaration.kind === SyntaxKind.ClassExpression) {
                    // When we emit a class expression with static members that contain a reference
                    // to the constructor in the initializer, we will need to substitute that
                    // binding with an alias as the class name is not in scope.
                    let container = getThisContainer(node, /*includeArrowFunctions*/ false);
                    while (container !== undefined) {
                        if (container.parent === declaration) {
                            if (container.kind === SyntaxKind.PropertyDeclaration && hasModifier(container, ModifierFlags.Static)) {
                                getNodeLinks(declaration).flags |= NodeCheckFlags.ClassWithConstructorReference;
                                getNodeLinks(node).flags |= NodeCheckFlags.ConstructorReferenceInClass;
                            }
                            break;
                        }

                        container = getThisContainer(container, /*includeArrowFunctions*/ false);
                    }
                }
            }

            checkCollisionWithCapturedSuperVariable(node, node);
            checkCollisionWithCapturedThisVariable(node, node);
            checkCollisionWithCapturedNewTargetVariable(node, node);
            checkNestedBlockScopedBinding(node, symbol);

            const type = getConstraintForLocation(getTypeOfSymbol(localOrExportSymbol), node);
            const assignmentKind = getAssignmentTargetKind(node);

            if (assignmentKind) {
                if (!(localOrExportSymbol.flags & SymbolFlags.Variable)) {
                    error(node, Diagnostics.Cannot_assign_to_0_because_it_is_not_a_variable, symbolToString(symbol));
                    return unknownType;
                }
                if (isReadonlySymbol(localOrExportSymbol)) {
                    error(node, Diagnostics.Cannot_assign_to_0_because_it_is_a_constant_or_a_read_only_property, symbolToString(symbol));
                    return unknownType;
                }
            }

            const isAlias = localOrExportSymbol.flags & SymbolFlags.Alias;

            // We only narrow variables and parameters occurring in a non-assignment position. For all other
            // entities we simply return the declared type.
            if (localOrExportSymbol.flags & SymbolFlags.Variable) {
                if (assignmentKind === AssignmentKind.Definite) {
                    return type;
                }
            }
            else if (isAlias) {
                declaration = find<Declaration>(symbol.declarations, isSomeImportDeclaration);
            }
            else {
                return type;
            }

            if (!declaration) {
                return type;
            }

            // The declaration container is the innermost function that encloses the declaration of the variable
            // or parameter. The flow container is the innermost function starting with which we analyze the control
            // flow graph to determine the control flow based type.
            const isParameter = getRootDeclaration(declaration).kind === SyntaxKind.Parameter;
            const declarationContainer = getControlFlowContainer(declaration);
            let flowContainer = getControlFlowContainer(node);
            const isOuterVariable = flowContainer !== declarationContainer;
            const isSpreadDestructuringAsignmentTarget = node.parent && node.parent.parent && isSpreadAssignment(node.parent) && isDestructuringAssignmentTarget(node.parent.parent);
            // When the control flow originates in a function expression or arrow function and we are referencing
            // a const variable or parameter from an outer function, we extend the origin of the control flow
            // analysis to include the immediately enclosing function.
            while (flowContainer !== declarationContainer && (flowContainer.kind === SyntaxKind.FunctionExpression ||
                flowContainer.kind === SyntaxKind.ArrowFunction || isObjectLiteralOrClassExpressionMethod(flowContainer)) &&
                (isConstVariable(localOrExportSymbol) || isParameter && !isParameterAssigned(localOrExportSymbol))) {
                flowContainer = getControlFlowContainer(flowContainer);
            }
            // We only look for uninitialized variables in strict null checking mode, and only when we can analyze
            // the entire control flow graph from the variable's declaration (i.e. when the flow container and
            // declaration container are the same).
            const assumeInitialized = isParameter || isAlias || isOuterVariable || isSpreadDestructuringAsignmentTarget ||
                type !== autoType && type !== autoArrayType && (!strictNullChecks || (type.flags & TypeFlags.Any) !== 0 ||
                isInTypeQuery(node) || node.parent.kind === SyntaxKind.ExportSpecifier) ||
                node.parent.kind === SyntaxKind.NonNullExpression ||
                declaration.kind === SyntaxKind.VariableDeclaration && (<VariableDeclaration>declaration).exclamationToken ||
                declaration.flags & NodeFlags.Ambient;
            const initialType = assumeInitialized ? (isParameter ? removeOptionalityFromDeclaredType(type, declaration as VariableLikeDeclaration) : type) :
                type === autoType || type === autoArrayType ? undefinedType :
                    getOptionalType(type);
            const flowType = getFlowTypeOfReference(node, type, initialType, flowContainer, !assumeInitialized);
            // A variable is considered uninitialized when it is possible to analyze the entire control flow graph
            // from declaration to use, and when the variable's declared type doesn't include undefined but the
            // control flow based type does include undefined.
            if (type === autoType || type === autoArrayType) {
                if (flowType === autoType || flowType === autoArrayType) {
                    if (noImplicitAny) {
                        error(getNameOfDeclaration(declaration), Diagnostics.Variable_0_implicitly_has_type_1_in_some_locations_where_its_type_cannot_be_determined, symbolToString(symbol), typeToString(flowType));
                        error(node, Diagnostics.Variable_0_implicitly_has_an_1_type, symbolToString(symbol), typeToString(flowType));
                    }
                    return convertAutoToAny(flowType);
                }
            }
            else if (!assumeInitialized && !(getFalsyFlags(type) & TypeFlags.Undefined) && getFalsyFlags(flowType) & TypeFlags.Undefined) {
                error(node, Diagnostics.Variable_0_is_used_before_being_assigned, symbolToString(symbol));
                // Return the declared type to reduce follow-on errors
                return type;
            }
            return assignmentKind ? getBaseTypeOfLiteralType(flowType) : flowType;
        }

        function isInsideFunction(node: Node, threshold: Node): boolean {
            return !!findAncestor(node, n => n === threshold ? "quit" : isFunctionLike(n));
        }

        function checkNestedBlockScopedBinding(node: Identifier, symbol: Symbol): void {
            if (languageVersion >= ScriptTarget.ES2015 ||
                (symbol.flags & (SymbolFlags.BlockScopedVariable | SymbolFlags.Class)) === 0 ||
                symbol.valueDeclaration.parent.kind === SyntaxKind.CatchClause) {
                return;
            }

            // 1. walk from the use site up to the declaration and check
            // if there is anything function like between declaration and use-site (is binding/class is captured in function).
            // 2. walk from the declaration up to the boundary of lexical environment and check
            // if there is an iteration statement in between declaration and boundary (is binding/class declared inside iteration statement)

            const container = getEnclosingBlockScopeContainer(symbol.valueDeclaration);
            const usedInFunction = isInsideFunction(node.parent, container);
            let current = container;

            let containedInIterationStatement = false;
            while (current && !nodeStartsNewLexicalEnvironment(current)) {
                if (isIterationStatement(current, /*lookInLabeledStatements*/ false)) {
                    containedInIterationStatement = true;
                    break;
                }
                current = current.parent;
            }

            if (containedInIterationStatement) {
                if (usedInFunction) {
                    // mark iteration statement as containing block-scoped binding captured in some function
                    getNodeLinks(current).flags |= NodeCheckFlags.LoopWithCapturedBlockScopedBinding;
                }

                // mark variables that are declared in loop initializer and reassigned inside the body of ForStatement.
                // if body of ForStatement will be converted to function then we'll need a extra machinery to propagate reassigned values back.
                if (container.kind === SyntaxKind.ForStatement &&
                    getAncestor(symbol.valueDeclaration, SyntaxKind.VariableDeclarationList).parent === container &&
                    isAssignedInBodyOfForStatement(node, <ForStatement>container)) {
                    getNodeLinks(symbol.valueDeclaration).flags |= NodeCheckFlags.NeedsLoopOutParameter;
                }

                // set 'declared inside loop' bit on the block-scoped binding
                getNodeLinks(symbol.valueDeclaration).flags |= NodeCheckFlags.BlockScopedBindingInLoop;
            }

            if (usedInFunction) {
                getNodeLinks(symbol.valueDeclaration).flags |= NodeCheckFlags.CapturedBlockScopedBinding;
            }
        }

        function isAssignedInBodyOfForStatement(node: Identifier, container: ForStatement): boolean {
            // skip parenthesized nodes
            let current: Node = node;
            while (current.parent.kind === SyntaxKind.ParenthesizedExpression) {
                current = current.parent;
            }

            // check if node is used as LHS in some assignment expression
            let isAssigned = false;
            if (isAssignmentTarget(current)) {
                isAssigned = true;
            }
            else if ((current.parent.kind === SyntaxKind.PrefixUnaryExpression || current.parent.kind === SyntaxKind.PostfixUnaryExpression)) {
                const expr = <PrefixUnaryExpression | PostfixUnaryExpression>current.parent;
                isAssigned = expr.operator === SyntaxKind.PlusPlusToken || expr.operator === SyntaxKind.MinusMinusToken;
            }

            if (!isAssigned) {
                return false;
            }

            // at this point we know that node is the target of assignment
            // now check that modification happens inside the statement part of the ForStatement
            return !!findAncestor(current, n => n === container ? "quit" : n === container.statement);
        }

        function captureLexicalThis(node: Node, container: Node): void {
            getNodeLinks(node).flags |= NodeCheckFlags.LexicalThis;
            if (container.kind === SyntaxKind.PropertyDeclaration || container.kind === SyntaxKind.Constructor) {
                const classNode = container.parent;
                getNodeLinks(classNode).flags |= NodeCheckFlags.CaptureThis;
            }
            else {
                getNodeLinks(container).flags |= NodeCheckFlags.CaptureThis;
            }
        }

        function findFirstSuperCall(n: Node): SuperCall | undefined {
            if (isSuperCall(n)) {
                return n;
            }
            else if (isFunctionLike(n)) {
                return undefined;
            }
            return forEachChild(n, findFirstSuperCall);
        }

        /**
         * Return a cached result if super-statement is already found.
         * Otherwise, find a super statement in a given constructor function and cache the result in the node-links of the constructor
         *
         * @param constructor constructor-function to look for super statement
         */
        function getSuperCallInConstructor(constructor: ConstructorDeclaration): SuperCall | undefined {
            const links = getNodeLinks(constructor);

            // Only trying to find super-call if we haven't yet tried to find one.  Once we try, we will record the result
            if (links.hasSuperCall === undefined) {
                links.superCall = findFirstSuperCall(constructor.body);
                links.hasSuperCall = links.superCall ? true : false;
            }
            return links.superCall;
        }

        /**
         * Check if the given class-declaration extends null then return true.
         * Otherwise, return false
         * @param classDecl a class declaration to check if it extends null
         */
        function classDeclarationExtendsNull(classDecl: ClassDeclaration): boolean {
            const classSymbol = getSymbolOfNode(classDecl);
            const classInstanceType = <InterfaceType>getDeclaredTypeOfSymbol(classSymbol);
            const baseConstructorType = getBaseConstructorTypeOfClass(classInstanceType);

            return baseConstructorType === nullWideningType;
        }

        function checkThisBeforeSuper(node: Node, container: Node, diagnosticMessage: DiagnosticMessage) {
            const containingClassDecl = <ClassDeclaration>container.parent;
            const baseTypeNode = getClassExtendsHeritageClauseElement(containingClassDecl);

            // If a containing class does not have extends clause or the class extends null
            // skip checking whether super statement is called before "this" accessing.
            if (baseTypeNode && !classDeclarationExtendsNull(containingClassDecl)) {
                const superCall = getSuperCallInConstructor(<ConstructorDeclaration>container);

                // We should give an error in the following cases:
                //      - No super-call
                //      - "this" is accessing before super-call.
                //          i.e super(this)
                //              this.x; super();
                // We want to make sure that super-call is done before accessing "this" so that
                // "this" is not accessed as a parameter of the super-call.
                if (!superCall || superCall.end > node.pos) {
                    // In ES6, super inside constructor of class-declaration has to precede "this" accessing
                    error(node, diagnosticMessage);
                }
            }
        }

        function checkThisExpression(node: Node): Type {
            // Stop at the first arrow function so that we can
            // tell whether 'this' needs to be captured.
            let container = getThisContainer(node, /* includeArrowFunctions */ true);
            let needToCaptureLexicalThis = false;

            if (container.kind === SyntaxKind.Constructor) {
                checkThisBeforeSuper(node, container, Diagnostics.super_must_be_called_before_accessing_this_in_the_constructor_of_a_derived_class);
            }

            // Now skip arrow functions to get the "real" owner of 'this'.
            if (container.kind === SyntaxKind.ArrowFunction) {
                container = getThisContainer(container, /* includeArrowFunctions */ false);

                // When targeting es6, arrow function lexically bind "this" so we do not need to do the work of binding "this" in emitted code
                needToCaptureLexicalThis = (languageVersion < ScriptTarget.ES2015);
            }

            switch (container.kind) {
                case SyntaxKind.ModuleDeclaration:
                    error(node, Diagnostics.this_cannot_be_referenced_in_a_module_or_namespace_body);
                    // do not return here so in case if lexical this is captured - it will be reflected in flags on NodeLinks
                    break;
                case SyntaxKind.EnumDeclaration:
                    error(node, Diagnostics.this_cannot_be_referenced_in_current_location);
                    // do not return here so in case if lexical this is captured - it will be reflected in flags on NodeLinks
                    break;
                case SyntaxKind.Constructor:
                    if (isInConstructorArgumentInitializer(node, container)) {
                        error(node, Diagnostics.this_cannot_be_referenced_in_constructor_arguments);
                        // do not return here so in case if lexical this is captured - it will be reflected in flags on NodeLinks
                    }
                    break;
                case SyntaxKind.PropertyDeclaration:
                case SyntaxKind.PropertySignature:
                    if (hasModifier(container, ModifierFlags.Static)) {
                        error(node, Diagnostics.this_cannot_be_referenced_in_a_static_property_initializer);
                        // do not return here so in case if lexical this is captured - it will be reflected in flags on NodeLinks
                    }
                    break;
                case SyntaxKind.ComputedPropertyName:
                    error(node, Diagnostics.this_cannot_be_referenced_in_a_computed_property_name);
                    break;
            }

            if (needToCaptureLexicalThis) {
                captureLexicalThis(node, container);
            }

            const type = tryGetThisTypeAt(node, container);
            if (!type && noImplicitThis) {
                // With noImplicitThis, functions may not reference 'this' if it has type 'any'
                error(node, Diagnostics.this_implicitly_has_type_any_because_it_does_not_have_a_type_annotation);
            }
            return type || anyType;
        }

        function tryGetThisTypeAt(node: Node, container = getThisContainer(node, /*includeArrowFunctions*/ false)): Type | undefined {
            if (isFunctionLike(container) &&
                (!isInParameterInitializerBeforeContainingFunction(node) || getThisParameter(container))) {
                // Note: a parameter initializer should refer to class-this unless function-this is explicitly annotated.

                // If this is a function in a JS file, it might be a class method. Check if it's the RHS
                // of a x.prototype.y = function [name]() { .... }
                if (container.kind === SyntaxKind.FunctionExpression &&
                    container.parent.kind === SyntaxKind.BinaryExpression &&
                    getSpecialPropertyAssignmentKind(container.parent as BinaryExpression) === SpecialPropertyAssignmentKind.PrototypeProperty) {
                    // Get the 'x' of 'x.prototype.y = f' (here, 'f' is 'container')
                    const className = (((container.parent as BinaryExpression)   // x.prototype.y = f
                        .left as PropertyAccessExpression)       // x.prototype.y
                        .expression as PropertyAccessExpression) // x.prototype
                        .expression;                             // x
                    const classSymbol = checkExpression(className).symbol;
                    if (classSymbol && classSymbol.members && (classSymbol.flags & SymbolFlags.Function)) {
                        return getFlowTypeOfReference(node, getInferredClassType(classSymbol));
                    }
                }

                const thisType = getThisTypeOfDeclaration(container) || getContextualThisParameterType(container);
                if (thisType) {
                    return getFlowTypeOfReference(node, thisType);
                }
            }

            if (isClassLike(container.parent)) {
                const symbol = getSymbolOfNode(container.parent);
                const type = hasModifier(container, ModifierFlags.Static) ? getTypeOfSymbol(symbol) : (<InterfaceType>getDeclaredTypeOfSymbol(symbol)).thisType;
                return getFlowTypeOfReference(node, type);
            }

            if (isInJavaScriptFile(node)) {
                const type = getTypeForThisExpressionFromJSDoc(container);
                if (type && type !== unknownType) {
                    return getFlowTypeOfReference(node, type);
                }
            }
        }

        function getTypeForThisExpressionFromJSDoc(node: Node) {
            const jsdocType = getJSDocType(node);
            if (jsdocType && jsdocType.kind === SyntaxKind.JSDocFunctionType) {
                const jsDocFunctionType = <JSDocFunctionType>jsdocType;
                if (jsDocFunctionType.parameters.length > 0 &&
                    jsDocFunctionType.parameters[0].name &&
                    (jsDocFunctionType.parameters[0].name as Identifier).escapedText === "this") {
                    return getTypeFromTypeNode(jsDocFunctionType.parameters[0].type);
                }
            }
        }

        function isInConstructorArgumentInitializer(node: Node, constructorDecl: Node): boolean {
            return !!findAncestor(node, n => n === constructorDecl ? "quit" : n.kind === SyntaxKind.Parameter);
        }

        function checkSuperExpression(node: Node): Type {
            const isCallExpression = node.parent.kind === SyntaxKind.CallExpression && (<CallExpression>node.parent).expression === node;

            let container = getSuperContainer(node, /*stopOnFunctions*/ true);
            let needToCaptureLexicalThis = false;

            // adjust the container reference in case if super is used inside arrow functions with arbitrarily deep nesting
            if (!isCallExpression) {
                while (container && container.kind === SyntaxKind.ArrowFunction) {
                    container = getSuperContainer(container, /*stopOnFunctions*/ true);
                    needToCaptureLexicalThis = languageVersion < ScriptTarget.ES2015;
                }
            }

            const canUseSuperExpression = isLegalUsageOfSuperExpression(container);
            let nodeCheckFlag: NodeCheckFlags = 0;

            if (!canUseSuperExpression) {
                // issue more specific error if super is used in computed property name
                // class A { foo() { return "1" }}
                // class B {
                //     [super.foo()]() {}
                // }
                const current = findAncestor(node, n => n === container ? "quit" : n.kind === SyntaxKind.ComputedPropertyName);
                if (current && current.kind === SyntaxKind.ComputedPropertyName) {
                    error(node, Diagnostics.super_cannot_be_referenced_in_a_computed_property_name);
                }
                else if (isCallExpression) {
                    error(node, Diagnostics.Super_calls_are_not_permitted_outside_constructors_or_in_nested_functions_inside_constructors);
                }
                else if (!container || !container.parent || !(isClassLike(container.parent) || container.parent.kind === SyntaxKind.ObjectLiteralExpression)) {
                    error(node, Diagnostics.super_can_only_be_referenced_in_members_of_derived_classes_or_object_literal_expressions);
                }
                else {
                    error(node, Diagnostics.super_property_access_is_permitted_only_in_a_constructor_member_function_or_member_accessor_of_a_derived_class);
                }
                return unknownType;
            }

            if (!isCallExpression && container.kind === SyntaxKind.Constructor) {
                checkThisBeforeSuper(node, container, Diagnostics.super_must_be_called_before_accessing_a_property_of_super_in_the_constructor_of_a_derived_class);
            }

            if (hasModifier(container, ModifierFlags.Static) || isCallExpression) {
                nodeCheckFlag = NodeCheckFlags.SuperStatic;
            }
            else {
                nodeCheckFlag = NodeCheckFlags.SuperInstance;
            }

            getNodeLinks(node).flags |= nodeCheckFlag;

            // Due to how we emit async functions, we need to specialize the emit for an async method that contains a `super` reference.
            // This is due to the fact that we emit the body of an async function inside of a generator function. As generator
            // functions cannot reference `super`, we emit a helper inside of the method body, but outside of the generator. This helper
            // uses an arrow function, which is permitted to reference `super`.
            //
            // There are two primary ways we can access `super` from within an async method. The first is getting the value of a property
            // or indexed access on super, either as part of a right-hand-side expression or call expression. The second is when setting the value
            // of a property or indexed access, either as part of an assignment expression or destructuring assignment.
            //
            // The simplest case is reading a value, in which case we will emit something like the following:
            //
            //  // ts
            //  ...
            //  async asyncMethod() {
            //    let x = await super.asyncMethod();
            //    return x;
            //  }
            //  ...
            //
            //  // js
            //  ...
            //  asyncMethod() {
            //      const _super = name => super[name];
            //      return __awaiter(this, arguments, Promise, function *() {
            //          let x = yield _super("asyncMethod").call(this);
            //          return x;
            //      });
            //  }
            //  ...
            //
            // The more complex case is when we wish to assign a value, especially as part of a destructuring assignment. As both cases
            // are legal in ES6, but also likely less frequent, we emit the same more complex helper for both scenarios:
            //
            //  // ts
            //  ...
            //  async asyncMethod(ar: Promise<any[]>) {
            //      [super.a, super.b] = await ar;
            //  }
            //  ...
            //
            //  // js
            //  ...
            //  asyncMethod(ar) {
            //      const _super = (function (geti, seti) {
            //          const cache = Object.create(null);
            //          return name => cache[name] || (cache[name] = { get value() { return geti(name); }, set value(v) { seti(name, v); } });
            //      })(name => super[name], (name, value) => super[name] = value);
            //      return __awaiter(this, arguments, Promise, function *() {
            //          [_super("a").value, _super("b").value] = yield ar;
            //      });
            //  }
            //  ...
            //
            // This helper creates an object with a "value" property that wraps the `super` property or indexed access for both get and set.
            // This is required for destructuring assignments, as a call expression cannot be used as the target of a destructuring assignment
            // while a property access can.
            if (container.kind === SyntaxKind.MethodDeclaration && hasModifier(container, ModifierFlags.Async)) {
                if (isSuperProperty(node.parent) && isAssignmentTarget(node.parent)) {
                    getNodeLinks(container).flags |= NodeCheckFlags.AsyncMethodWithSuperBinding;
                }
                else {
                    getNodeLinks(container).flags |= NodeCheckFlags.AsyncMethodWithSuper;
                }
            }

            if (needToCaptureLexicalThis) {
                // call expressions are allowed only in constructors so they should always capture correct 'this'
                // super property access expressions can also appear in arrow functions -
                // in this case they should also use correct lexical this
                captureLexicalThis(node.parent, container);
            }

            if (container.parent.kind === SyntaxKind.ObjectLiteralExpression) {
                if (languageVersion < ScriptTarget.ES2015) {
                    error(node, Diagnostics.super_is_only_allowed_in_members_of_object_literal_expressions_when_option_target_is_ES2015_or_higher);
                    return unknownType;
                }
                else {
                    // for object literal assume that type of 'super' is 'any'
                    return anyType;
                }
            }

            // at this point the only legal case for parent is ClassLikeDeclaration
            const classLikeDeclaration = <ClassLikeDeclaration>container.parent;
            if (!getClassExtendsHeritageClauseElement(classLikeDeclaration)) {
                error(node, Diagnostics.super_can_only_be_referenced_in_a_derived_class);
                return unknownType;
            }

            const classType = <InterfaceType>getDeclaredTypeOfSymbol(getSymbolOfNode(classLikeDeclaration));
            const baseClassType = classType && getBaseTypes(classType)[0];
            if (!baseClassType) {
                return unknownType;
            }

            if (container.kind === SyntaxKind.Constructor && isInConstructorArgumentInitializer(node, container)) {
                // issue custom error message for super property access in constructor arguments (to be aligned with old compiler)
                error(node, Diagnostics.super_cannot_be_referenced_in_constructor_arguments);
                return unknownType;
            }

            return nodeCheckFlag === NodeCheckFlags.SuperStatic
                ? getBaseConstructorTypeOfClass(classType)
                : getTypeWithThisArgument(baseClassType, classType.thisType);

            function isLegalUsageOfSuperExpression(container: Node): boolean {
                if (!container) {
                    return false;
                }

                if (isCallExpression) {
                    // TS 1.0 SPEC (April 2014): 4.8.1
                    // Super calls are only permitted in constructors of derived classes
                    return container.kind === SyntaxKind.Constructor;
                }
                else {
                    // TS 1.0 SPEC (April 2014)
                    // 'super' property access is allowed
                    // - In a constructor, instance member function, instance member accessor, or instance member variable initializer where this references a derived class instance
                    // - In a static member function or static member accessor

                    // topmost container must be something that is directly nested in the class declaration\object literal expression
                    if (isClassLike(container.parent) || container.parent.kind === SyntaxKind.ObjectLiteralExpression) {
                        if (hasModifier(container, ModifierFlags.Static)) {
                            return container.kind === SyntaxKind.MethodDeclaration ||
                                container.kind === SyntaxKind.MethodSignature ||
                                container.kind === SyntaxKind.GetAccessor ||
                                container.kind === SyntaxKind.SetAccessor;
                        }
                        else {
                            return container.kind === SyntaxKind.MethodDeclaration ||
                                container.kind === SyntaxKind.MethodSignature ||
                                container.kind === SyntaxKind.GetAccessor ||
                                container.kind === SyntaxKind.SetAccessor ||
                                container.kind === SyntaxKind.PropertyDeclaration ||
                                container.kind === SyntaxKind.PropertySignature ||
                                container.kind === SyntaxKind.Constructor;
                        }
                    }
                }

                return false;
            }
        }

        function getContainingObjectLiteral(func: SignatureDeclaration): ObjectLiteralExpression | undefined {
            return (func.kind === SyntaxKind.MethodDeclaration ||
                func.kind === SyntaxKind.GetAccessor ||
                func.kind === SyntaxKind.SetAccessor) && func.parent.kind === SyntaxKind.ObjectLiteralExpression ? func.parent :
                func.kind === SyntaxKind.FunctionExpression && func.parent.kind === SyntaxKind.PropertyAssignment ? <ObjectLiteralExpression>func.parent.parent :
                    undefined;
        }

        function getThisTypeArgument(type: Type): Type {
            return getObjectFlags(type) & ObjectFlags.Reference && (<TypeReference>type).target === globalThisType ? (<TypeReference>type).typeArguments[0] : undefined;
        }

        function getThisTypeFromContextualType(type: Type): Type {
            return mapType(type, t => {
                return t.flags & TypeFlags.Intersection ? forEach((<IntersectionType>t).types, getThisTypeArgument) : getThisTypeArgument(t);
            });
        }

        function getContextualThisParameterType(func: SignatureDeclaration): Type {
            if (func.kind === SyntaxKind.ArrowFunction) {
                return undefined;
            }
            if (isContextSensitiveFunctionOrObjectLiteralMethod(func)) {
                const contextualSignature = getContextualSignature(func);
                if (contextualSignature) {
                    const thisParameter = contextualSignature.thisParameter;
                    if (thisParameter) {
                        return getTypeOfSymbol(thisParameter);
                    }
                }
            }
            const inJs = isInJavaScriptFile(func);
            if (noImplicitThis || inJs) {
                const containingLiteral = getContainingObjectLiteral(func);
                if (containingLiteral) {
                    // We have an object literal method. Check if the containing object literal has a contextual type
                    // that includes a ThisType<T>. If so, T is the contextual type for 'this'. We continue looking in
                    // any directly enclosing object literals.
                    const contextualType = getApparentTypeOfContextualType(containingLiteral);
                    let literal = containingLiteral;
                    let type = contextualType;
                    while (type) {
                        const thisType = getThisTypeFromContextualType(type);
                        if (thisType) {
                            return instantiateType(thisType, getContextualMapper(containingLiteral));
                        }
                        if (literal.parent.kind !== SyntaxKind.PropertyAssignment) {
                            break;
                        }
                        literal = <ObjectLiteralExpression>literal.parent.parent;
                        type = getApparentTypeOfContextualType(literal);
                    }
                    // There was no contextual ThisType<T> for the containing object literal, so the contextual type
                    // for 'this' is the non-null form of the contextual type for the containing object literal or
                    // the type of the object literal itself.
                    return contextualType ? getNonNullableType(contextualType) : checkExpressionCached(containingLiteral);
                }
                // In an assignment of the form 'obj.xxx = function(...)' or 'obj[xxx] = function(...)', the
                // contextual type for 'this' is 'obj'.
                const { parent } = func;
                if (parent.kind === SyntaxKind.BinaryExpression && (<BinaryExpression>parent).operatorToken.kind === SyntaxKind.EqualsToken) {
                    const target = (<BinaryExpression>parent).left;
                    if (target.kind === SyntaxKind.PropertyAccessExpression || target.kind === SyntaxKind.ElementAccessExpression) {
                        const { expression } = target as PropertyAccessExpression | ElementAccessExpression;
                        // Don't contextually type `this` as `exports` in `exports.Point = function(x, y) { this.x = x; this.y = y; }`
                        if (inJs && isIdentifier(expression)) {
                            const sourceFile = getSourceFileOfNode(parent);
                            if (sourceFile.commonJsModuleIndicator && getResolvedSymbol(expression) === sourceFile.symbol) {
                                return undefined;
                            }
                        }

                        return checkExpressionCached(expression);
                    }
                }
            }
            return undefined;
        }

        // Return contextual type of parameter or undefined if no contextual type is available
        function getContextuallyTypedParameterType(parameter: ParameterDeclaration): Type | undefined {
            const func = parameter.parent;
            if (isContextSensitiveFunctionOrObjectLiteralMethod(func)) {
                const iife = getImmediatelyInvokedFunctionExpression(func);
                if (iife && iife.arguments) {
                    const indexOfParameter = func.parameters.indexOf(parameter);
                    if (parameter.dotDotDotToken) {
                        const restTypes: Type[] = [];
                        for (let i = indexOfParameter; i < iife.arguments.length; i++) {
                            restTypes.push(getWidenedLiteralType(checkExpression(iife.arguments[i])));
                        }
                        return restTypes.length ? createArrayType(getUnionType(restTypes)) : undefined;
                    }
                    const links = getNodeLinks(iife);
                    const cached = links.resolvedSignature;
                    links.resolvedSignature = anySignature;
                    const type = indexOfParameter < iife.arguments.length ?
                        getWidenedLiteralType(checkExpression(iife.arguments[indexOfParameter])) :
                        parameter.initializer ? undefined : undefinedWideningType;
                    links.resolvedSignature = cached;
                    return type;
                }
                const contextualSignature = getContextualSignature(func);
                if (contextualSignature) {
                    const funcHasRestParameters = hasRestParameter(func);
                    const len = func.parameters.length - (funcHasRestParameters ? 1 : 0);
                    let indexOfParameter = func.parameters.indexOf(parameter);
                    if (getThisParameter(func) !== undefined && !contextualSignature.thisParameter) {
                        Debug.assert(indexOfParameter !== 0); // Otherwise we should not have called `getContextuallyTypedParameterType`.
                        indexOfParameter -= 1;
                    }

                    if (indexOfParameter < len) {
                        return getTypeAtPosition(contextualSignature, indexOfParameter);
                    }

                    // If last parameter is contextually rest parameter get its type
                    if (funcHasRestParameters &&
                        indexOfParameter === (func.parameters.length - 1) &&
                        isRestParameterIndex(contextualSignature, func.parameters.length - 1)) {
                        return getTypeOfSymbol(lastOrUndefined(contextualSignature.parameters));
                    }
                }
            }
            return undefined;
        }

        // In a variable, parameter or property declaration with a type annotation,
        //   the contextual type of an initializer expression is the type of the variable, parameter or property.
        // Otherwise, in a parameter declaration of a contextually typed function expression,
        //   the contextual type of an initializer expression is the contextual type of the parameter.
        // Otherwise, in a variable or parameter declaration with a binding pattern name,
        //   the contextual type of an initializer expression is the type implied by the binding pattern.
        // Otherwise, in a binding pattern inside a variable or parameter declaration,
        //   the contextual type of an initializer expression is the type annotation of the containing declaration, if present.
        function getContextualTypeForInitializerExpression(node: Expression): Type {
            const declaration = <VariableLikeDeclaration>node.parent;
            if (hasInitializer(declaration) && node === declaration.initializer) {
                const typeNode = getEffectiveTypeAnnotationNode(declaration);
                if (typeNode) {
                    return getTypeFromTypeNode(typeNode);
                }
                if (declaration.kind === SyntaxKind.Parameter) {
                    const type = getContextuallyTypedParameterType(declaration);
                    if (type) {
                        return type;
                    }
                }
                if (isBindingPattern(declaration.name)) {
                    return getTypeFromBindingPattern(declaration.name, /*includePatternInType*/ true, /*reportErrors*/ false);
                }
                if (isBindingPattern(declaration.parent)) {
                    const parentDeclaration = declaration.parent.parent;
                    const name = (declaration as BindingElement).propertyName || declaration.name;
                    if (parentDeclaration.kind !== SyntaxKind.BindingElement) {
                        const parentTypeNode = getEffectiveTypeAnnotationNode(parentDeclaration);
                        if (parentTypeNode && !isBindingPattern(name)) {
                            const text = getTextOfPropertyName(name);
                            if (text) {
                                return getTypeOfPropertyOfType(getTypeFromTypeNode(parentTypeNode), text);
                            }
                        }
                    }
                }
            }
            return undefined;
        }

        function getContextualTypeForReturnExpression(node: Expression): Type {
            const func = getContainingFunction(node);
            if (func) {
                const functionFlags = getFunctionFlags(func);
                if (functionFlags & FunctionFlags.Generator) { // AsyncGenerator function or Generator function
                    return undefined;
                }

                const contextualReturnType = getContextualReturnType(func);
                return functionFlags & FunctionFlags.Async
                    ? contextualReturnType && getAwaitedTypeOfPromise(contextualReturnType) // Async function
                    : contextualReturnType; // Regular function
            }
            return undefined;
        }

        function getContextualTypeForYieldOperand(node: YieldExpression): Type {
            const func = getContainingFunction(node);
            if (func) {
                const functionFlags = getFunctionFlags(func);
                const contextualReturnType = getContextualReturnType(func);
                if (contextualReturnType) {
                    return node.asteriskToken
                        ? contextualReturnType
                        : getIteratedTypeOfGenerator(contextualReturnType, (functionFlags & FunctionFlags.Async) !== 0);
                }
            }

            return undefined;
        }

        function isInParameterInitializerBeforeContainingFunction(node: Node) {
            let inBindingInitializer = false;
            while (node.parent && !isFunctionLike(node.parent)) {
                if (isParameter(node.parent) && (inBindingInitializer || node.parent.initializer === node)) {
                    return true;
                }
                if (isBindingElement(node.parent) && node.parent.initializer === node) {
                    inBindingInitializer = true;
                }

                node = node.parent;
            }

            return false;
        }

        function getContextualReturnType(functionDecl: SignatureDeclaration): Type {
            // If the containing function has a return type annotation, is a constructor, or is a get accessor whose
            // corresponding set accessor has a type annotation, return statements in the function are contextually typed
            if (functionDecl.kind === SyntaxKind.Constructor ||
                getEffectiveReturnTypeNode(functionDecl) ||
                isGetAccessorWithAnnotatedSetAccessor(functionDecl)) {
                return getReturnTypeOfSignature(getSignatureFromDeclaration(functionDecl));
            }

            // Otherwise, if the containing function is contextually typed by a function type with exactly one call signature
            // and that call signature is non-generic, return statements are contextually typed by the return type of the signature
            const signature = getContextualSignatureForFunctionLikeDeclaration(<FunctionExpression>functionDecl);
            if (signature && !isResolvingReturnTypeOfSignature(signature)) {
                return getReturnTypeOfSignature(signature);
            }

            return undefined;
        }

        // In a typed function call, an argument or substitution expression is contextually typed by the type of the corresponding parameter.
        function getContextualTypeForArgument(callTarget: CallLikeExpression, arg: Expression): Type {
            const args = getEffectiveCallArguments(callTarget);
            const argIndex = args.indexOf(arg); // -1 for e.g. the expression of a CallExpression, or the tag of a TaggedTemplateExpression
            return argIndex === -1 ? undefined : getContextualTypeForArgumentAtIndex(callTarget, argIndex);
        }

        function getContextualTypeForArgumentAtIndex(callTarget: CallLikeExpression, argIndex: number): Type {
            // If we're already in the process of resolving the given signature, don't resolve again as
            // that could cause infinite recursion. Instead, return anySignature.
            const signature = getNodeLinks(callTarget).resolvedSignature === resolvingSignature ? resolvingSignature : getResolvedSignature(callTarget);
            return getTypeAtPosition(signature, argIndex);
        }

        function getContextualTypeForSubstitutionExpression(template: TemplateExpression, substitutionExpression: Expression) {
            if (template.parent.kind === SyntaxKind.TaggedTemplateExpression) {
                return getContextualTypeForArgument(<TaggedTemplateExpression>template.parent, substitutionExpression);
            }

            return undefined;
        }

        function getContextualTypeForBinaryOperand(node: Expression): Type | undefined {
            const binaryExpression = <BinaryExpression>node.parent;
            const { left, operatorToken, right } = binaryExpression;
            switch (operatorToken.kind) {
                case SyntaxKind.EqualsToken:
                    return node === right && isContextSensitiveAssignment(binaryExpression) ? getTypeOfExpression(left) : undefined;
                case SyntaxKind.BarBarToken:
                    // When an || expression has a contextual type, the operands are contextually typed by that type. When an ||
                    // expression has no contextual type, the right operand is contextually typed by the type of the left operand,
                    // except for the special case of Javascript declarations of the form `namespace.prop = namespace.prop || {}`
                    const type = getContextualType(binaryExpression);
                    return !type && node === right && !getDeclaredJavascriptInitializer(binaryExpression.parent) && !getAssignedJavascriptInitializer(binaryExpression) ?
                        getTypeOfExpression(left, /*cache*/ true) : type;
                case SyntaxKind.AmpersandAmpersandToken:
                case SyntaxKind.CommaToken:
                    return node === right ? getContextualType(binaryExpression) : undefined;
                default:
                    return undefined;
            }
        }
        // In an assignment expression, the right operand is contextually typed by the type of the left operand.
        // Don't do this for special property assignments to avoid circularity.
        function isContextSensitiveAssignment(binaryExpression: BinaryExpression): boolean {
            const kind = getSpecialPropertyAssignmentKind(binaryExpression);
            switch (kind) {
                case SpecialPropertyAssignmentKind.None:
                    return true;
                case SpecialPropertyAssignmentKind.Property:
                    // If `binaryExpression.left` was assigned a symbol, then this is a new declaration; otherwise it is an assignment to an existing declaration.
                    // See `bindStaticPropertyAssignment` in `binder.ts`.
                    return !binaryExpression.left.symbol;
                case SpecialPropertyAssignmentKind.ExportsProperty:
                case SpecialPropertyAssignmentKind.ModuleExports:
                case SpecialPropertyAssignmentKind.PrototypeProperty:
                case SpecialPropertyAssignmentKind.ThisProperty:
                case SpecialPropertyAssignmentKind.Prototype:
                    return false;
                default:
                    Debug.assertNever(kind);
            }
        }

        function getTypeOfPropertyOfContextualType(type: Type, name: __String) {
            return mapType(type, t => {
                const prop = t.flags & TypeFlags.StructuredType ? getPropertyOfType(t, name) : undefined;
                return prop ? getTypeOfSymbol(prop) : undefined;
            }, /*noReductions*/ true);
        }

        function getIndexTypeOfContextualType(type: Type, kind: IndexKind) {
            return mapType(type, t => getIndexTypeOfStructuredType(t, kind), /*noReductions*/ true);
        }

        // Return true if the given contextual type is a tuple-like type
        function contextualTypeIsTupleLikeType(type: Type): boolean {
            return !!(type.flags & TypeFlags.Union ? forEach((<UnionType>type).types, isTupleLikeType) : isTupleLikeType(type));
        }

        // In an object literal contextually typed by a type T, the contextual type of a property assignment is the type of
        // the matching property in T, if one exists. Otherwise, it is the type of the numeric index signature in T, if one
        // exists. Otherwise, it is the type of the string index signature in T, if one exists.
        function getContextualTypeForObjectLiteralMethod(node: MethodDeclaration): Type {
            Debug.assert(isObjectLiteralMethod(node));
            if (node.flags & NodeFlags.InWithStatement) {
                // We cannot answer semantic questions within a with block, do not proceed any further
                return undefined;
            }

            return getContextualTypeForObjectLiteralElement(node);
        }

        function getContextualTypeForObjectLiteralElement(element: ObjectLiteralElementLike) {
            const objectLiteral = <ObjectLiteralExpression>element.parent;
            const type = getApparentTypeOfContextualType(objectLiteral);
            if (type) {
                if (!hasNonBindableDynamicName(element)) {
                    // For a (non-symbol) computed property, there is no reason to look up the name
                    // in the type. It will just be "__computed", which does not appear in any
                    // SymbolTable.
                    const symbolName = getSymbolOfNode(element).escapedName;
                    const propertyType = getTypeOfPropertyOfContextualType(type, symbolName);
                    if (propertyType) {
                        return propertyType;
                    }
                }

                return isNumericName(element.name) && getIndexTypeOfContextualType(type, IndexKind.Number) ||
                    getIndexTypeOfContextualType(type, IndexKind.String);
            }

            return undefined;
        }

        // In an array literal contextually typed by a type T, the contextual type of an element expression at index N is
        // the type of the property with the numeric name N in T, if one exists. Otherwise, if T has a numeric index signature,
        // it is the type of the numeric index signature in T. Otherwise, in ES6 and higher, the contextual type is the iterated
        // type of T.
        function getContextualTypeForElementExpression(arrayContextualType: Type | undefined, index: number): Type | undefined {
            return arrayContextualType && (
                getTypeOfPropertyOfContextualType(arrayContextualType, "" + index as __String)
                || getIndexTypeOfContextualType(arrayContextualType, IndexKind.Number)
                || getIteratedTypeOrElementType(arrayContextualType, /*errorNode*/ undefined, /*allowStringInput*/ false, /*allowAsyncIterables*/ false, /*checkAssignability*/ false));
        }

        // In a contextually typed conditional expression, the true/false expressions are contextually typed by the same type.
        function getContextualTypeForConditionalOperand(node: Expression): Type {
            const conditional = <ConditionalExpression>node.parent;
            return node === conditional.whenTrue || node === conditional.whenFalse ? getContextualType(conditional) : undefined;
        }

        function getContextualTypeForChildJsxExpression(node: JsxElement) {
            const attributesType = getApparentTypeOfContextualType(node.openingElement.tagName);
            // JSX expression is in children of JSX Element, we will look for an "children" atttribute (we get the name from JSX.ElementAttributesProperty)
            const jsxChildrenPropertyName = getJsxElementChildrenPropertyName(getJsxNamespaceAt(node));
            return attributesType && !isTypeAny(attributesType) && jsxChildrenPropertyName && jsxChildrenPropertyName !== "" ? getTypeOfPropertyOfContextualType(attributesType, jsxChildrenPropertyName) : undefined;
        }

        function getContextualTypeForJsxExpression(node: JsxExpression): Type {
            const exprParent = node.parent;
            return isJsxAttributeLike(exprParent)
                ? getContextualType(node)
                : isJsxElement(exprParent)
                    ? getContextualTypeForChildJsxExpression(exprParent)
                    : undefined;
        }

        function getContextualTypeForJsxAttribute(attribute: JsxAttribute | JsxSpreadAttribute): Type | undefined {
            // When we trying to resolve JsxOpeningLikeElement as a stateless function element, we will already give its attributes a contextual type
            // which is a type of the parameter of the signature we are trying out.
            // If there is no contextual type (e.g. we are trying to resolve stateful component), get attributes type from resolving element's tagName
            if (isJsxAttribute(attribute)) {
                const attributesType = getApparentTypeOfContextualType(attribute.parent);
                if (!attributesType || isTypeAny(attributesType)) {
                    return undefined;
                }
                return getTypeOfPropertyOfContextualType(attributesType, attribute.name.escapedText);
            }
            else {
                return getContextualType(attribute.parent);
            }
        }

        // Return the contextual type for a given expression node. During overload resolution, a contextual type may temporarily
        // be "pushed" onto a node using the contextualType property.
        function getApparentTypeOfContextualType(node: Expression): Type {
            let contextualType = getContextualType(node);
            contextualType = contextualType && mapType(contextualType, getApparentType);
            if (!(contextualType && contextualType.flags & TypeFlags.Union && isObjectLiteralExpression(node))) {
                return contextualType;
            }
            // Keep the below up-to-date with the work done within `isRelatedTo` by `findMatchingDiscriminantType`
            let match: Type | undefined;
            propLoop: for (const prop of node.properties) {
                if (!prop.symbol) continue;
                if (prop.kind !== SyntaxKind.PropertyAssignment) continue;
                if (isDiscriminantProperty(contextualType, prop.symbol.escapedName)) {
                    const discriminatingType = getTypeOfNode(prop.initializer);
                    for (const type of (contextualType as UnionType).types) {
                        const targetType = getTypeOfPropertyOfType(type, prop.symbol.escapedName);
                        if (targetType && checkTypeAssignableTo(discriminatingType, targetType, /*errorNode*/ undefined)) {
                            if (match) {
                                if (type === match) continue; // Finding multiple fields which discriminate to the same type is fine
                                match = undefined;
                                break propLoop;
                            }
                            match = type;
                        }
                    }
                }
            }
            return match || contextualType;
        }

        /**
         * Woah! Do you really want to use this function?
         *
         * Unless you're trying to get the *non-apparent* type for a
         * value-literal type or you're authoring relevant portions of this algorithm,
         * you probably meant to use 'getApparentTypeOfContextualType'.
         * Otherwise this may not be very useful.
         *
         * In cases where you *are* working on this function, you should understand
         * when it is appropriate to use 'getContextualType' and 'getApparentTypeOfContextualType'.
         *
         *   - Use 'getContextualType' when you are simply going to propagate the result to the expression.
         *   - Use 'getApparentTypeOfContextualType' when you're going to need the members of the type.
         *
         * @param node the expression whose contextual type will be returned.
         * @returns the contextual type of an expression.
         */
        function getContextualType(node: Expression): Type | undefined {
            if (node.flags & NodeFlags.InWithStatement) {
                // We cannot answer semantic questions within a with block, do not proceed any further
                return undefined;
            }
            if (node.contextualType) {
                return node.contextualType;
            }
            const parent = node.parent;
            switch (parent.kind) {
                case SyntaxKind.VariableDeclaration:
                case SyntaxKind.Parameter:
                case SyntaxKind.PropertyDeclaration:
                case SyntaxKind.PropertySignature:
                case SyntaxKind.BindingElement:
                    return getContextualTypeForInitializerExpression(node);
                case SyntaxKind.ArrowFunction:
                case SyntaxKind.ReturnStatement:
                    return getContextualTypeForReturnExpression(node);
                case SyntaxKind.YieldExpression:
                    return getContextualTypeForYieldOperand(<YieldExpression>parent);
                    case SyntaxKind.CallExpression:
                case SyntaxKind.NewExpression:
                    return getContextualTypeForArgument(<CallExpression | NewExpression>parent, node);
                case SyntaxKind.TypeAssertionExpression:
                case SyntaxKind.AsExpression:
                    return getTypeFromTypeNode((<AssertionExpression>parent).type);
                case SyntaxKind.BinaryExpression:
                    return getContextualTypeForBinaryOperand(node);
                case SyntaxKind.PropertyAssignment:
                case SyntaxKind.ShorthandPropertyAssignment:
                    return getContextualTypeForObjectLiteralElement(<ObjectLiteralElementLike>parent);
                case SyntaxKind.SpreadAssignment:
                    return getApparentTypeOfContextualType(parent.parent as ObjectLiteralExpression);
                case SyntaxKind.ArrayLiteralExpression: {
                    const arrayLiteral = <ArrayLiteralExpression>parent;
                    const type = getApparentTypeOfContextualType(arrayLiteral);
                    return getContextualTypeForElementExpression(type, indexOfNode(arrayLiteral.elements, node));
                }
                case SyntaxKind.ConditionalExpression:
                    return getContextualTypeForConditionalOperand(node);
                case SyntaxKind.TemplateSpan:
                    Debug.assert(parent.parent.kind === SyntaxKind.TemplateExpression);
                    return getContextualTypeForSubstitutionExpression(<TemplateExpression>parent.parent, node);
                case SyntaxKind.ParenthesizedExpression: {
                    // Like in `checkParenthesizedExpression`, an `/** @type {xyz} */` comment before a parenthesized expression acts as a type cast.
                    const tag = isInJavaScriptFile(parent) ? getJSDocTypeTag(parent) : undefined;
                    return tag ? getTypeFromTypeNode(tag.typeExpression.type) : getContextualType(<ParenthesizedExpression>parent);
                }
                case SyntaxKind.JsxExpression:
                    return getContextualTypeForJsxExpression(<JsxExpression>parent);
                case SyntaxKind.JsxAttribute:
                case SyntaxKind.JsxSpreadAttribute:
                    return getContextualTypeForJsxAttribute(<JsxAttribute | JsxSpreadAttribute>parent);
                case SyntaxKind.JsxOpeningElement:
                case SyntaxKind.JsxSelfClosingElement:
                    return getContextualJsxElementAttributesType(<JsxOpeningLikeElement>parent);
            }
            return undefined;
        }

        function getContextualMapper(node: Node) {
            node = findAncestor(node, n => !!n.contextualMapper);
            return node ? node.contextualMapper : identityMapper;
        }

        function getContextualJsxElementAttributesType(node: JsxOpeningLikeElement) {
            if (isJsxIntrinsicIdentifier(node.tagName)) {
                return getIntrinsicAttributesTypeFromJsxOpeningLikeElement(node);
            }
            const valueType = checkExpression(node.tagName);
            if (isTypeAny(valueType)) {
                // Short-circuit if the class tag is using an element type 'any'
                return anyType;
            }

            const isJs = isInJavaScriptFile(node);
            return mapType(valueType, t => getJsxSignaturesParameterTypes(t, isJs, node));
        }

        function getJsxSignaturesParameterTypes(valueType: Type, isJs: boolean, context: Node) {
            // If the elemType is a string type, we have to return anyType to prevent an error downstream as we will try to find construct or call signature of the type
            if (valueType.flags & TypeFlags.String) {
                return anyType;
            }
            else if (valueType.flags & TypeFlags.StringLiteral) {
                // If the elemType is a stringLiteral type, we can then provide a check to make sure that the string literal type is one of the Jsx intrinsic element type
                // For example:
                //      var CustomTag: "h1" = "h1";
                //      <CustomTag> Hello World </CustomTag>
                const intrinsicElementsType = getJsxType(JsxNames.IntrinsicElements, context);
                if (intrinsicElementsType !== unknownType) {
                    const stringLiteralTypeName = (<StringLiteralType>valueType).value;
                    const intrinsicProp = getPropertyOfType(intrinsicElementsType, escapeLeadingUnderscores(stringLiteralTypeName));
                    if (intrinsicProp) {
                        return getTypeOfSymbol(intrinsicProp);
                    }
                    const indexSignatureType = getIndexTypeOfType(intrinsicElementsType, IndexKind.String);
                    if (indexSignatureType) {
                        return indexSignatureType;
                    }
                }
                return anyType;
            }

            // Resolve the signatures, preferring constructor
            let signatures = getSignaturesOfType(valueType, SignatureKind.Construct);
            let ctor = true;
            if (signatures.length === 0) {
                // No construct signatures, try call signatures
                signatures = getSignaturesOfType(valueType, SignatureKind.Call);
                ctor = false;
                if (signatures.length === 0) {
                    // We found no signatures at all, which is an error
                    return unknownType;
                }
            }

            return getUnionType(map(signatures, ctor ? t => getJsxPropsTypeFromConstructSignature(t, isJs, context) : t => getJsxPropsTypeFromCallSignature(t, context)), UnionReduction.None);
        }

        function getJsxPropsTypeFromCallSignature(sig: Signature, context: Node) {
            let propsType = getTypeOfFirstParameterOfSignature(sig);
            const intrinsicAttribs = getJsxType(JsxNames.IntrinsicAttributes, context);
            if (intrinsicAttribs !== unknownType) {
                propsType = intersectTypes(intrinsicAttribs, propsType);
            }
            return propsType;
        }

        function getJsxPropsTypeFromClassType(hostClassType: Type, isJs: boolean, context: Node) {
            if (isTypeAny(hostClassType)) {
                return hostClassType;
            }

            const propsName = getJsxElementPropertiesName(getJsxNamespaceAt(context));
            if (propsName === undefined) {
                // There is no type ElementAttributesProperty, return 'any'
                return anyType;
            }
            else if (propsName === "") {
                // If there is no e.g. 'props' member in ElementAttributesProperty, use the element class type instead
                return hostClassType;
            }
            else {
                const attributesType = getTypeOfPropertyOfType(hostClassType, propsName);

                if (!attributesType) {
                    // There is no property named 'props' on this instance type
                    return emptyObjectType;
                }
                else if (isTypeAny(attributesType)) {
                    // Props is of type 'any' or unknown
                    return attributesType;
                }
                else {
                    // Normal case -- add in IntrinsicClassElements<T> and IntrinsicElements
                    let apparentAttributesType = attributesType;
                    const intrinsicClassAttribs = getJsxType(JsxNames.IntrinsicClassAttributes, context);
                    if (intrinsicClassAttribs !== unknownType) {
                        const typeParams = getLocalTypeParametersOfClassOrInterfaceOrTypeAlias(intrinsicClassAttribs.symbol);
                        apparentAttributesType = intersectTypes(
                            typeParams
                                ? createTypeReference(<GenericType>intrinsicClassAttribs, fillMissingTypeArguments([hostClassType], typeParams, getMinTypeArgumentCount(typeParams), isJs))
                                : intrinsicClassAttribs,
                            apparentAttributesType
                        );
                    }

                    const intrinsicAttribs = getJsxType(JsxNames.IntrinsicAttributes, context);
                    if (intrinsicAttribs !== unknownType) {
                        apparentAttributesType = intersectTypes(intrinsicAttribs, apparentAttributesType);
                    }

                    return apparentAttributesType;
                }
            }
        }

        function getJsxPropsTypeFromConstructSignature(sig: Signature, isJs: boolean, context: Node) {
            const hostClassType = getReturnTypeOfSignature(sig);
            if (hostClassType) {
                return getJsxPropsTypeFromClassType(hostClassType, isJs, context);
            }
            return getJsxPropsTypeFromCallSignature(sig, context);
        }



        // If the given type is an object or union type with a single signature, and if that signature has at
        // least as many parameters as the given function, return the signature. Otherwise return undefined.
        function getContextualCallSignature(type: Type, node: FunctionExpression | ArrowFunction | MethodDeclaration): Signature {
            const signatures = getSignaturesOfType(type, SignatureKind.Call);
            if (signatures.length === 1) {
                const signature = signatures[0];
                if (!isAritySmaller(signature, node)) {
                    return signature;
                }
            }
        }

        /** If the contextual signature has fewer parameters than the function expression, do not use it */
        function isAritySmaller(signature: Signature, target: FunctionExpression | ArrowFunction | MethodDeclaration) {
            let targetParameterCount = 0;
            for (; targetParameterCount < target.parameters.length; targetParameterCount++) {
                const param = target.parameters[targetParameterCount];
                if (param.initializer || param.questionToken || param.dotDotDotToken || isJSDocOptionalParameter(param)) {
                    break;
                }
            }
            if (target.parameters.length && parameterIsThisKeyword(target.parameters[0])) {
                targetParameterCount--;
            }
            const sourceLength = signature.hasRestParameter ? Number.MAX_VALUE : signature.parameters.length;
            return sourceLength < targetParameterCount;
        }

        function isFunctionExpressionOrArrowFunction(node: Node): node is FunctionExpression | ArrowFunction {
            return node.kind === SyntaxKind.FunctionExpression || node.kind === SyntaxKind.ArrowFunction;
        }

        function getContextualSignatureForFunctionLikeDeclaration(node: FunctionLikeDeclaration): Signature {
            // Only function expressions, arrow functions, and object literal methods are contextually typed.
            return isFunctionExpressionOrArrowFunction(node) || isObjectLiteralMethod(node)
                ? getContextualSignature(<FunctionExpression>node)
                : undefined;
        }

        function getContextualTypeForFunctionLikeDeclaration(node: FunctionExpression | ArrowFunction | MethodDeclaration) {
            return isObjectLiteralMethod(node) ?
                getContextualTypeForObjectLiteralMethod(node) :
                getApparentTypeOfContextualType(node);
        }

        // Return the contextual signature for a given expression node. A contextual type provides a
        // contextual signature if it has a single call signature and if that call signature is non-generic.
        // If the contextual type is a union type, get the signature from each type possible and if they are
        // all identical ignoring their return type, the result is same signature but with return type as
        // union type of return types from these signatures
        function getContextualSignature(node: FunctionExpression | ArrowFunction | MethodDeclaration): Signature {
            Debug.assert(node.kind !== SyntaxKind.MethodDeclaration || isObjectLiteralMethod(node));
            const type = getContextualTypeForFunctionLikeDeclaration(node);
            if (!type) {
                return undefined;
            }
            if (!(type.flags & TypeFlags.Union)) {
                return getContextualCallSignature(type, node);
            }
            let signatureList: Signature[];
            const types = (<UnionType>type).types;
            for (const current of types) {
                const signature = getContextualCallSignature(current, node);
                if (signature) {
                    if (!signatureList) {
                        // This signature will contribute to contextual union signature
                        signatureList = [signature];
                    }
                    else if (!compareSignaturesIdentical(signatureList[0], signature, /*partialMatch*/ false, /*ignoreThisTypes*/ true, /*ignoreReturnTypes*/ true, compareTypesIdentical)) {
                        // Signatures aren't identical, do not use
                        return undefined;
                    }
                    else {
                        // Use this signature for contextual union signature
                        signatureList.push(signature);
                    }
                }
            }

            // Result is union of signatures collected (return type is union of return types of this signature set)
            let result: Signature;
            if (signatureList) {
                result = cloneSignature(signatureList[0]);
                result.unionSignatures = signatureList;
            }
            return result;
        }

        function checkSpreadExpression(node: SpreadElement, checkMode?: CheckMode): Type {
            if (languageVersion < ScriptTarget.ES2015 && compilerOptions.downlevelIteration) {
                checkExternalEmitHelpers(node, ExternalEmitHelpers.SpreadIncludes);
            }

            const arrayOrIterableType = checkExpression(node.expression, checkMode);
            return checkIteratedTypeOrElementType(arrayOrIterableType, node.expression, /*allowStringInput*/ false, /*allowAsyncIterables*/ false);
        }

        function hasDefaultValue(node: BindingElement | Expression): boolean {
            return (node.kind === SyntaxKind.BindingElement && !!(<BindingElement>node).initializer) ||
                (node.kind === SyntaxKind.BinaryExpression && (<BinaryExpression>node).operatorToken.kind === SyntaxKind.EqualsToken);
        }

        function checkArrayLiteral(node: ArrayLiteralExpression, checkMode: CheckMode | undefined): Type {
            const elements = node.elements;
            let hasSpreadElement = false;
            const elementTypes: Type[] = [];
            const inDestructuringPattern = isAssignmentTarget(node);
            const contextualType = getApparentTypeOfContextualType(node);
            for (let index = 0; index < elements.length; index++) {
                const e = elements[index];
                if (inDestructuringPattern && e.kind === SyntaxKind.SpreadElement) {
                    // Given the following situation:
                    //    var c: {};
                    //    [...c] = ["", 0];
                    //
                    // c is represented in the tree as a spread element in an array literal.
                    // But c really functions as a rest element, and its purpose is to provide
                    // a contextual type for the right hand side of the assignment. Therefore,
                    // instead of calling checkExpression on "...c", which will give an error
                    // if c is not iterable/array-like, we need to act as if we are trying to
                    // get the contextual element type from it. So we do something similar to
                    // getContextualTypeForElementExpression, which will crucially not error
                    // if there is no index type / iterated type.
                    const restArrayType = checkExpression((<SpreadElement>e).expression, checkMode);
                    const restElementType = getIndexTypeOfType(restArrayType, IndexKind.Number) ||
                        getIteratedTypeOrElementType(restArrayType, /*errorNode*/ undefined, /*allowStringInput*/ false, /*allowAsyncIterables*/ false, /*checkAssignability*/ false);
                    if (restElementType) {
                        elementTypes.push(restElementType);
                    }
                }
                else {
                    const elementContextualType = getContextualTypeForElementExpression(contextualType, index);
                    const type = checkExpressionForMutableLocation(e, checkMode, elementContextualType);
                    elementTypes.push(type);
                }
                hasSpreadElement = hasSpreadElement || e.kind === SyntaxKind.SpreadElement;
            }
            if (!hasSpreadElement) {
                // If array literal is actually a destructuring pattern, mark it as an implied type. We do this such
                // that we get the same behavior for "var [x, y] = []" and "[x, y] = []".
                if (inDestructuringPattern && elementTypes.length) {
                    const type = cloneTypeReference(createTupleType(elementTypes));
                    type.pattern = node;
                    return type;
                }
                if (contextualType && contextualTypeIsTupleLikeType(contextualType)) {
                    const pattern = contextualType.pattern;
                    // If array literal is contextually typed by a binding pattern or an assignment pattern, pad the resulting
                    // tuple type with the corresponding binding or assignment element types to make the lengths equal.
                    if (pattern && (pattern.kind === SyntaxKind.ArrayBindingPattern || pattern.kind === SyntaxKind.ArrayLiteralExpression)) {
                        const patternElements = (<BindingPattern | ArrayLiteralExpression>pattern).elements;
                        for (let i = elementTypes.length; i < patternElements.length; i++) {
                            const patternElement = patternElements[i];
                            if (hasDefaultValue(patternElement)) {
                                elementTypes.push((<TypeReference>contextualType).typeArguments[i]);
                            }
                            else {
                                if (patternElement.kind !== SyntaxKind.OmittedExpression) {
                                    error(patternElement, Diagnostics.Initializer_provides_no_value_for_this_binding_element_and_the_binding_element_has_no_default_value);
                                }
                                elementTypes.push(strictNullChecks ? implicitNeverType : undefinedWideningType);
                            }
                        }
                    }
                    if (elementTypes.length) {
                        return createTupleType(elementTypes);
                    }
                }
            }
            return createArrayType(elementTypes.length ?
                getUnionType(elementTypes, UnionReduction.Subtype) :
                strictNullChecks ? implicitNeverType : undefinedWideningType);
        }

        function isNumericName(name: DeclarationName): boolean {
            switch (name.kind) {
                case SyntaxKind.ComputedPropertyName:
                    return isNumericComputedName(name);
                case SyntaxKind.Identifier:
                    return isNumericLiteralName(name.escapedText);
                case SyntaxKind.NumericLiteral:
                case SyntaxKind.StringLiteral:
                    return isNumericLiteralName(name.text);
                default:
                    return false;
            }
        }

        function isNumericComputedName(name: ComputedPropertyName): boolean {
            // It seems odd to consider an expression of type Any to result in a numeric name,
            // but this behavior is consistent with checkIndexedAccess
            return isTypeAssignableToKind(checkComputedPropertyName(name), TypeFlags.NumberLike);
        }

        function isInfinityOrNaNString(name: string | __String): boolean {
            return name === "Infinity" || name === "-Infinity" || name === "NaN";
        }

        function isNumericLiteralName(name: string | __String) {
            // The intent of numeric names is that
            //     - they are names with text in a numeric form, and that
            //     - setting properties/indexing with them is always equivalent to doing so with the numeric literal 'numLit',
            //         acquired by applying the abstract 'ToNumber' operation on the name's text.
            //
            // The subtlety is in the latter portion, as we cannot reliably say that anything that looks like a numeric literal is a numeric name.
            // In fact, it is the case that the text of the name must be equal to 'ToString(numLit)' for this to hold.
            //
            // Consider the property name '"0xF00D"'. When one indexes with '0xF00D', they are actually indexing with the value of 'ToString(0xF00D)'
            // according to the ECMAScript specification, so it is actually as if the user indexed with the string '"61453"'.
            // Thus, the text of all numeric literals equivalent to '61543' such as '0xF00D', '0xf00D', '0170015', etc. are not valid numeric names
            // because their 'ToString' representation is not equal to their original text.
            // This is motivated by ECMA-262 sections 9.3.1, 9.8.1, 11.1.5, and 11.2.1.
            //
            // Here, we test whether 'ToString(ToNumber(name))' is exactly equal to 'name'.
            // The '+' prefix operator is equivalent here to applying the abstract ToNumber operation.
            // Applying the 'toString()' method on a number gives us the abstract ToString operation on a number.
            //
            // Note that this accepts the values 'Infinity', '-Infinity', and 'NaN', and that this is intentional.
            // This is desired behavior, because when indexing with them as numeric entities, you are indexing
            // with the strings '"Infinity"', '"-Infinity"', and '"NaN"' respectively.
            return (+name).toString() === name;
        }

        function checkComputedPropertyName(node: ComputedPropertyName): Type {
            const links = getNodeLinks(node.expression);
            if (!links.resolvedType) {
                links.resolvedType = checkExpression(node.expression);
                // This will allow types number, string, symbol or any. It will also allow enums, the unknown
                // type, and any union of these types (like string | number).
                if (links.resolvedType.flags & TypeFlags.Nullable ||
                    !isTypeAssignableToKind(links.resolvedType, TypeFlags.StringLike | TypeFlags.NumberLike | TypeFlags.ESSymbolLike) &&
                    !isTypeAssignableTo(links.resolvedType, getUnionType([stringType, numberType, esSymbolType]))) {
                    error(node, Diagnostics.A_computed_property_name_must_be_of_type_string_number_symbol_or_any);
                }
                else {
                    checkThatExpressionIsProperSymbolReference(node.expression, links.resolvedType, /*reportError*/ true);
                }
            }

            return links.resolvedType;
        }

        function getObjectLiteralIndexInfo(propertyNodes: NodeArray<ObjectLiteralElementLike>, offset: number, properties: Symbol[], kind: IndexKind): IndexInfo {
            const propTypes: Type[] = [];
            for (let i = 0; i < properties.length; i++) {
                if (kind === IndexKind.String || isNumericName(propertyNodes[i + offset].name)) {
                    propTypes.push(getTypeOfSymbol(properties[i]));
                }
            }
            const unionType = propTypes.length ? getUnionType(propTypes, UnionReduction.Subtype) : undefinedType;
            return createIndexInfo(unionType, /*isReadonly*/ false);
        }

        function checkObjectLiteral(node: ObjectLiteralExpression, checkMode?: CheckMode): Type {
            const inDestructuringPattern = isAssignmentTarget(node);
            // Grammar checking
            checkGrammarObjectLiteralExpression(node, inDestructuringPattern);

            let propertiesTable: SymbolTable;
            let propertiesArray: Symbol[] = [];
            let spread: Type = emptyObjectType;
            let propagatedFlags: TypeFlags = TypeFlags.FreshLiteral;

            const contextualType = getApparentTypeOfContextualType(node);
            const contextualTypeHasPattern = contextualType && contextualType.pattern &&
                (contextualType.pattern.kind === SyntaxKind.ObjectBindingPattern || contextualType.pattern.kind === SyntaxKind.ObjectLiteralExpression);
            const isInJSFile = isInJavaScriptFile(node);
            const isJSObjectLiteral = !contextualType && isInJSFile;
            let typeFlags: TypeFlags = 0;
            let patternWithComputedProperties = false;
            let hasComputedStringProperty = false;
            let hasComputedNumberProperty = false;
            if (isInJSFile && node.properties.length === 0) {
                // an empty JS object literal that nonetheless has members is a JS namespace
                const symbol = getSymbolOfNode(node);
                if (symbol.exports) {
                    propertiesTable = symbol.exports;
                    symbol.exports.forEach(symbol => propertiesArray.push(getMergedSymbol(symbol)));
                    return createObjectLiteralType();
                }
            }
            propertiesTable = createSymbolTable();

            let offset = 0;
            for (let i = 0; i < node.properties.length; i++) {
                const memberDecl = node.properties[i];
                let member = getSymbolOfNode(memberDecl);
                let literalName: __String | undefined;
                if (memberDecl.kind === SyntaxKind.PropertyAssignment ||
                    memberDecl.kind === SyntaxKind.ShorthandPropertyAssignment ||
                    isObjectLiteralMethod(memberDecl)) {
                    let jsdocType: Type;
                    if (isInJSFile) {
                        jsdocType = getTypeForDeclarationFromJSDocComment(memberDecl);
                    }

                    let type: Type;
                    if (memberDecl.kind === SyntaxKind.PropertyAssignment) {
                        if (memberDecl.name.kind === SyntaxKind.ComputedPropertyName) {
                            const t = checkComputedPropertyName(memberDecl.name);
                            if (t.flags & TypeFlags.Literal) {
                                literalName = escapeLeadingUnderscores("" + (t as LiteralType).value);
                            }
                        }
                        type = checkPropertyAssignment(memberDecl, checkMode);
                    }
                    else if (memberDecl.kind === SyntaxKind.MethodDeclaration) {
                        type = checkObjectLiteralMethod(memberDecl, checkMode);
                    }
                    else {
                        Debug.assert(memberDecl.kind === SyntaxKind.ShorthandPropertyAssignment);
                        type = checkExpressionForMutableLocation(memberDecl.name, checkMode);
                    }

                    if (jsdocType) {
                        checkTypeAssignableTo(type, jsdocType, memberDecl);
                        type = jsdocType;
                    }

                    typeFlags |= type.flags;

                    const nameType = hasLateBindableName(memberDecl) ? checkComputedPropertyName(memberDecl.name) : undefined;
                    const hasLateBoundName = nameType && isTypeUsableAsLateBoundName(nameType);
                    const prop = hasLateBoundName
                        ? createSymbol(SymbolFlags.Property | member.flags, getLateBoundNameFromType(nameType as LiteralType | UniqueESSymbolType), CheckFlags.Late)
                        : createSymbol(SymbolFlags.Property | member.flags, literalName || member.escapedName);

                    if (hasLateBoundName) {
                        prop.nameType = nameType;
                    }

                    if (inDestructuringPattern) {
                        // If object literal is an assignment pattern and if the assignment pattern specifies a default value
                        // for the property, make the property optional.
                        const isOptional =
                            (memberDecl.kind === SyntaxKind.PropertyAssignment && hasDefaultValue(memberDecl.initializer)) ||
                            (memberDecl.kind === SyntaxKind.ShorthandPropertyAssignment && memberDecl.objectAssignmentInitializer);
                        if (isOptional) {
                            prop.flags |= SymbolFlags.Optional;
                        }
                        if (!literalName && hasDynamicName(memberDecl)) {
                            patternWithComputedProperties = true;
                        }
                    }
                    else if (contextualTypeHasPattern && !(getObjectFlags(contextualType) & ObjectFlags.ObjectLiteralPatternWithComputedProperties)) {
                        // If object literal is contextually typed by the implied type of a binding pattern, and if the
                        // binding pattern specifies a default value for the property, make the property optional.
                        const impliedProp = getPropertyOfType(contextualType, member.escapedName);
                        if (impliedProp) {
                            prop.flags |= impliedProp.flags & SymbolFlags.Optional;
                        }

                        else if (!compilerOptions.suppressExcessPropertyErrors && !getIndexInfoOfType(contextualType, IndexKind.String)) {
                            error(memberDecl.name, Diagnostics.Object_literal_may_only_specify_known_properties_and_0_does_not_exist_in_type_1,
                                symbolToString(member), typeToString(contextualType));
                        }
                    }
                    prop.declarations = member.declarations;
                    prop.parent = member.parent;
                    if (member.valueDeclaration) {
                        prop.valueDeclaration = member.valueDeclaration;
                    }

                    prop.type = type;
                    prop.target = member;
                    member = prop;
                }
                else if (memberDecl.kind === SyntaxKind.SpreadAssignment) {
                    if (languageVersion < ScriptTarget.ES2015) {
                        checkExternalEmitHelpers(memberDecl, ExternalEmitHelpers.Assign);
                    }
                    if (propertiesArray.length > 0) {
                        spread = getSpreadType(spread, createObjectLiteralType(), node.symbol, propagatedFlags, /*objectFlags*/ 0);
                        propertiesArray = [];
                        propertiesTable = createSymbolTable();
                        hasComputedStringProperty = false;
                        hasComputedNumberProperty = false;
                        typeFlags = 0;
                    }
                    const type = checkExpression(memberDecl.expression);
                    if (!isValidSpreadType(type)) {
                        error(memberDecl, Diagnostics.Spread_types_may_only_be_created_from_object_types);
                        return unknownType;
                    }
                    spread = getSpreadType(spread, type, node.symbol, propagatedFlags, /*objectFlags*/ 0);
                    offset = i + 1;
                    continue;
                }
                else {
                    // TypeScript 1.0 spec (April 2014)
                    // A get accessor declaration is processed in the same manner as
                    // an ordinary function declaration(section 6.1) with no parameters.
                    // A set accessor declaration is processed in the same manner
                    // as an ordinary function declaration with a single parameter and a Void return type.
                    Debug.assert(memberDecl.kind === SyntaxKind.GetAccessor || memberDecl.kind === SyntaxKind.SetAccessor);
                    checkNodeDeferred(memberDecl);
                }

                if (!literalName && hasNonBindableDynamicName(memberDecl)) {
                    if (isNumericName(memberDecl.name)) {
                        hasComputedNumberProperty = true;
                    }
                    else {
                        hasComputedStringProperty = true;
                    }
                }
                else {
                    propertiesTable.set(member.escapedName, member);
                }
                propertiesArray.push(member);
            }

            // If object literal is contextually typed by the implied type of a binding pattern, augment the result
            // type with those properties for which the binding pattern specifies a default value.
            if (contextualTypeHasPattern) {
                for (const prop of getPropertiesOfType(contextualType)) {
                    if (!propertiesTable.get(prop.escapedName) && !(spread && getPropertyOfType(spread, prop.escapedName))) {
                        if (!(prop.flags & SymbolFlags.Optional)) {
                            error(prop.valueDeclaration || (<TransientSymbol>prop).bindingElement,
                                Diagnostics.Initializer_provides_no_value_for_this_binding_element_and_the_binding_element_has_no_default_value);
                        }
                        propertiesTable.set(prop.escapedName, prop);
                        propertiesArray.push(prop);
                    }
                }
            }

            if (spread !== emptyObjectType) {
                if (propertiesArray.length > 0) {
                    spread = getSpreadType(spread, createObjectLiteralType(), node.symbol, propagatedFlags, /*objectFlags*/ 0);
                }
                return spread;
            }

            return createObjectLiteralType();

            function createObjectLiteralType() {
                const stringIndexInfo = isJSObjectLiteral ? jsObjectLiteralIndexInfo : hasComputedStringProperty ? getObjectLiteralIndexInfo(node.properties, offset, propertiesArray, IndexKind.String) : undefined;
                const numberIndexInfo = hasComputedNumberProperty && !isJSObjectLiteral ? getObjectLiteralIndexInfo(node.properties, offset, propertiesArray, IndexKind.Number) : undefined;
                const result = createAnonymousType(node.symbol, propertiesTable, emptyArray, emptyArray, stringIndexInfo, numberIndexInfo);
                const freshObjectLiteralFlag = compilerOptions.suppressExcessPropertyErrors ? 0 : TypeFlags.FreshLiteral;
                result.flags |= TypeFlags.ContainsObjectLiteral | freshObjectLiteralFlag | (typeFlags & TypeFlags.PropagatingFlags);
                result.objectFlags |= ObjectFlags.ObjectLiteral;
                if (patternWithComputedProperties) {
                    result.objectFlags |= ObjectFlags.ObjectLiteralPatternWithComputedProperties;
                }
                if (inDestructuringPattern) {
                    result.pattern = node;
                }
                if (!(result.flags & TypeFlags.Nullable)) {
                    propagatedFlags |= (result.flags & TypeFlags.PropagatingFlags);
                }
                return result;
            }
        }

        function isValidSpreadType(type: Type): boolean {
            return !!(type.flags & (TypeFlags.Any | TypeFlags.NonPrimitive) ||
                getFalsyFlags(type) & TypeFlags.DefinitelyFalsy && isValidSpreadType(removeDefinitelyFalsyTypes(type)) ||
                type.flags & TypeFlags.Object && !isGenericMappedType(type) ||
                type.flags & TypeFlags.UnionOrIntersection && !forEach((<UnionOrIntersectionType>type).types, t => !isValidSpreadType(t)));
        }

        function checkJsxSelfClosingElement(node: JsxSelfClosingElement, checkMode: CheckMode): Type {
            checkJsxOpeningLikeElementOrOpeningFragment(node, checkMode);
            return getJsxElementTypeAt(node) || anyType;
        }

        function checkJsxElement(node: JsxElement, checkMode: CheckMode): Type {
            // Check attributes
            checkJsxOpeningLikeElementOrOpeningFragment(node.openingElement, checkMode);

            // Perform resolution on the closing tag so that rename/go to definition/etc work
            if (isJsxIntrinsicIdentifier(node.closingElement.tagName)) {
                getIntrinsicTagSymbol(node.closingElement);
            }
            else {
                checkExpression(node.closingElement.tagName);
            }

            return getJsxElementTypeAt(node) || anyType;
        }

        function checkJsxFragment(node: JsxFragment, checkMode: CheckMode): Type {
            checkJsxOpeningLikeElementOrOpeningFragment(node.openingFragment, checkMode);

            if (compilerOptions.jsx === JsxEmit.React && (compilerOptions.jsxFactory || getSourceFileOfNode(node).pragmas.has("jsx"))) {
                error(node, compilerOptions.jsxFactory
                    ? Diagnostics.JSX_fragment_is_not_supported_when_using_jsxFactory
                    : Diagnostics.JSX_fragment_is_not_supported_when_using_an_inline_JSX_factory_pragma);
            }

            return getJsxElementTypeAt(node) || anyType;
        }

        /**
         * Returns true iff the JSX element name would be a valid JS identifier, ignoring restrictions about keywords not being identifiers
         */
        function isUnhyphenatedJsxName(name: string | __String) {
            // - is the only character supported in JSX attribute names that isn't valid in JavaScript identifiers
            return !stringContains(name as string, "-");
        }

        /**
         * Returns true iff React would emit this tag name as a string rather than an identifier or qualified name
         */
        function isJsxIntrinsicIdentifier(tagName: JsxTagNameExpression) {
            // TODO (yuisu): comment
            switch (tagName.kind) {
                case SyntaxKind.PropertyAccessExpression:
                case SyntaxKind.ThisKeyword:
                    return false;
                case SyntaxKind.Identifier:
                    return isIntrinsicJsxName((<Identifier>tagName).escapedText);
                default:
                    Debug.fail();
            }
        }

        function checkJsxAttribute(node: JsxAttribute, checkMode?: CheckMode) {
            return node.initializer
                ? checkExpressionForMutableLocation(node.initializer, checkMode)
                : trueType;  // <Elem attr /> is sugar for <Elem attr={true} />
        }

        /**
         * Get attributes type of the JSX opening-like element. The result is from resolving "attributes" property of the opening-like element.
         *
         * @param openingLikeElement a JSX opening-like element
         * @param filter a function to remove attributes that will not participate in checking whether attributes are assignable
         * @return an anonymous type (similar to the one returned by checkObjectLiteral) in which its properties are attributes property.
         * @remarks Because this function calls getSpreadType, it needs to use the same checks as checkObjectLiteral,
         * which also calls getSpreadType.
         */
        function createJsxAttributesTypeFromAttributesProperty(openingLikeElement: JsxOpeningLikeElement, checkMode: CheckMode) {
            const attributes = openingLikeElement.attributes;
            let attributesTable = createSymbolTable();
            let spread: Type = emptyObjectType;
            let hasSpreadAnyType = false;
            let typeToIntersect: Type;
            let explicitlySpecifyChildrenAttribute = false;
            const jsxChildrenPropertyName = getJsxElementChildrenPropertyName(getJsxNamespaceAt(openingLikeElement));

            for (const attributeDecl of attributes.properties) {
                const member = attributeDecl.symbol;
                if (isJsxAttribute(attributeDecl)) {
                    const exprType = checkJsxAttribute(attributeDecl, checkMode);

                    const attributeSymbol = createSymbol(SymbolFlags.Property | SymbolFlags.Transient | member.flags, member.escapedName);
                    attributeSymbol.declarations = member.declarations;
                    attributeSymbol.parent = member.parent;
                    if (member.valueDeclaration) {
                        attributeSymbol.valueDeclaration = member.valueDeclaration;
                    }
                    attributeSymbol.type = exprType;
                    attributeSymbol.target = member;
                    attributesTable.set(attributeSymbol.escapedName, attributeSymbol);
                    if (attributeDecl.name.escapedText === jsxChildrenPropertyName) {
                        explicitlySpecifyChildrenAttribute = true;
                    }
                }
                else {
                    Debug.assert(attributeDecl.kind === SyntaxKind.JsxSpreadAttribute);
                    if (attributesTable.size > 0) {
                        spread = getSpreadType(spread, createJsxAttributesType(), attributes.symbol, /*typeFlags*/ 0, ObjectFlags.JsxAttributes);
                        attributesTable = createSymbolTable();
                    }
                    const exprType = checkExpressionCached(attributeDecl.expression, checkMode);
                    if (isTypeAny(exprType)) {
                        hasSpreadAnyType = true;
                    }
                    if (isValidSpreadType(exprType)) {
                        spread = getSpreadType(spread, exprType, openingLikeElement.symbol, /*typeFlags*/ 0, ObjectFlags.JsxAttributes);
                    }
                    else {
                        typeToIntersect = typeToIntersect ? getIntersectionType([typeToIntersect, exprType]) : exprType;
                    }
                }
            }

            if (!hasSpreadAnyType) {
                if (attributesTable.size > 0) {
                    spread = getSpreadType(spread, createJsxAttributesType(), attributes.symbol, /*typeFlags*/ 0, ObjectFlags.JsxAttributes);
                }
            }

            // Handle children attribute
            const parent = openingLikeElement.parent.kind === SyntaxKind.JsxElement ? openingLikeElement.parent as JsxElement : undefined;
            // We have to check that openingElement of the parent is the one we are visiting as this may not be true for selfClosingElement
            if (parent && parent.openingElement === openingLikeElement && parent.children.length > 0) {
                const childrenTypes: Type[] = checkJsxChildren(parent, checkMode);

                if (!hasSpreadAnyType && jsxChildrenPropertyName && jsxChildrenPropertyName !== "") {
                    // Error if there is a attribute named "children" explicitly specified and children element.
                    // This is because children element will overwrite the value from attributes.
                    // Note: we will not warn "children" attribute overwritten if "children" attribute is specified in object spread.
                    if (explicitlySpecifyChildrenAttribute) {
                        error(attributes, Diagnostics._0_are_specified_twice_The_attribute_named_0_will_be_overwritten, unescapeLeadingUnderscores(jsxChildrenPropertyName));
                    }

                    // If there are children in the body of JSX element, create dummy attribute "children" with the union of children types so that it will pass the attribute checking process
                    const childrenPropSymbol = createSymbol(SymbolFlags.Property | SymbolFlags.Transient, jsxChildrenPropertyName);
                    childrenPropSymbol.type = childrenTypes.length === 1 ?
                        childrenTypes[0] :
                        createArrayType(getUnionType(childrenTypes));
                    const childPropMap = createSymbolTable();
                    childPropMap.set(jsxChildrenPropertyName, childrenPropSymbol);
                    spread = getSpreadType(spread, createAnonymousType(attributes.symbol, childPropMap, emptyArray, emptyArray, /*stringIndexInfo*/ undefined, /*numberIndexInfo*/ undefined),
                        attributes.symbol, /*typeFlags*/ 0, ObjectFlags.JsxAttributes);

                }
            }

            if (hasSpreadAnyType) {
                return anyType;
            }
            return typeToIntersect && spread !== emptyObjectType ? getIntersectionType([typeToIntersect, spread]) : (typeToIntersect || spread);

            /**
             * Create anonymous type from given attributes symbol table.
             * @param symbol a symbol of JsxAttributes containing attributes corresponding to attributesTable
             * @param attributesTable a symbol table of attributes property
             */
            function createJsxAttributesType() {
                const result = createAnonymousType(attributes.symbol, attributesTable, emptyArray, emptyArray, /*stringIndexInfo*/ undefined, /*numberIndexInfo*/ undefined);
                result.flags |= TypeFlags.ContainsObjectLiteral;
                result.objectFlags |= ObjectFlags.ObjectLiteral | ObjectFlags.JsxAttributes;
                return result;
            }
        }

        function checkJsxChildren(node: JsxElement | JsxFragment, checkMode?: CheckMode) {
            const childrenTypes: Type[] = [];
            for (const child of node.children) {
                // In React, JSX text that contains only whitespaces will be ignored so we don't want to type-check that
                // because then type of children property will have constituent of string type.
                if (child.kind === SyntaxKind.JsxText) {
                    if (!child.containsOnlyWhiteSpaces) {
                        childrenTypes.push(stringType);
                    }
                }
                else {
                    childrenTypes.push(checkExpressionForMutableLocation(child, checkMode));
                }
            }
            return childrenTypes;
        }

        /**
         * Check attributes property of opening-like element. This function is called during chooseOverload to get call signature of a JSX opening-like element.
         * (See "checkApplicableSignatureForJsxOpeningLikeElement" for how the function is used)
         * @param node a JSXAttributes to be resolved of its type
         */
        function checkJsxAttributes(node: JsxAttributes, checkMode: CheckMode) {
            return createJsxAttributesTypeFromAttributesProperty(node.parent, checkMode);
        }

        function getJsxType(name: __String, location: Node) {
            const namespace = getJsxNamespaceAt(location);
            const exports = namespace && getExportsOfSymbol(namespace);
            const typeSymbol = exports && getSymbol(exports, name, SymbolFlags.Type);
            return typeSymbol ? getDeclaredTypeOfSymbol(typeSymbol) : unknownType;
        }

        /**
         * Looks up an intrinsic tag name and returns a symbol that either points to an intrinsic
         * property (in which case nodeLinks.jsxFlags will be IntrinsicNamedElement) or an intrinsic
         * string index signature (in which case nodeLinks.jsxFlags will be IntrinsicIndexedElement).
         * May also return unknownSymbol if both of these lookups fail.
         */
        function getIntrinsicTagSymbol(node: JsxOpeningLikeElement | JsxClosingElement): Symbol {
            const links = getNodeLinks(node);
            if (!links.resolvedSymbol) {
                const intrinsicElementsType = getJsxType(JsxNames.IntrinsicElements, node);
                if (intrinsicElementsType !== unknownType) {
                    // Property case
                    if (!isIdentifier(node.tagName)) return Debug.fail();
                    const intrinsicProp = getPropertyOfType(intrinsicElementsType, node.tagName.escapedText);
                    if (intrinsicProp) {
                        links.jsxFlags |= JsxFlags.IntrinsicNamedElement;
                        return links.resolvedSymbol = intrinsicProp;
                    }

                    // Intrinsic string indexer case
                    const indexSignatureType = getIndexTypeOfType(intrinsicElementsType, IndexKind.String);
                    if (indexSignatureType) {
                        links.jsxFlags |= JsxFlags.IntrinsicIndexedElement;
                        return links.resolvedSymbol = intrinsicElementsType.symbol;
                    }

                    // Wasn't found
                    error(node, Diagnostics.Property_0_does_not_exist_on_type_1, idText(node.tagName), "JSX." + JsxNames.IntrinsicElements);
                    return links.resolvedSymbol = unknownSymbol;
                }
                else {
                    if (noImplicitAny) {
                        error(node, Diagnostics.JSX_element_implicitly_has_type_any_because_no_interface_JSX_0_exists, unescapeLeadingUnderscores(JsxNames.IntrinsicElements));
                    }
                    return links.resolvedSymbol = unknownSymbol;
                }
            }
            return links.resolvedSymbol;
        }

        /**
         * Given a JSX element that is a class element, finds the Element Instance Type. If the
         * element is not a class element, or the class element type cannot be determined, returns 'undefined'.
         * For example, in the element <MyClass>, the element instance type is `MyClass` (not `typeof MyClass`).
         */
        function getJsxElementInstanceType(node: JsxOpeningLikeElement, valueType: Type) {
            Debug.assert(!(valueType.flags & TypeFlags.Union));
            if (isTypeAny(valueType)) {
                // Short-circuit if the class tag is using an element type 'any'
                return anyType;
            }

            // Resolve the signatures, preferring constructor
            let signatures = getSignaturesOfType(valueType, SignatureKind.Construct);
            if (signatures.length === 0) {
                // No construct signatures, try call signatures
                signatures = getSignaturesOfType(valueType, SignatureKind.Call);
                if (signatures.length === 0) {
                    // We found no signatures at all, which is an error
                    error(node.tagName, Diagnostics.JSX_element_type_0_does_not_have_any_construct_or_call_signatures, getTextOfNode(node.tagName));
                    return unknownType;
                }
            }

            // Instantiate in context of source type
            const instantiatedSignatures = [];
            for (const signature of signatures) {
                if (signature.typeParameters) {
                    const isJavascript = isInJavaScriptFile(node);
                    const inferenceContext = createInferenceContext(signature.typeParameters, signature, /*flags*/ isJavascript ? InferenceFlags.AnyDefault : InferenceFlags.None);
                    const typeArguments = inferJsxTypeArguments(signature, node, inferenceContext);
                    instantiatedSignatures.push(getSignatureInstantiation(signature, typeArguments, isJavascript));
                }
                else {
                    instantiatedSignatures.push(signature);
                }
            }

            return getUnionType(map(instantiatedSignatures, getReturnTypeOfSignature), UnionReduction.Subtype);
        }

        function getJsxNamespaceAt(location: Node) {
            const namespaceName = getJsxNamespace(location);
            const resolvedNamespace = resolveName(location, namespaceName, SymbolFlags.Namespace, /*diagnosticMessage*/ undefined, namespaceName, /*isUse*/ false);
            if (resolvedNamespace) {
                const candidate = getSymbol(getExportsOfSymbol(resolveSymbol(resolvedNamespace)), JsxNames.JSX, SymbolFlags.Namespace);
                if (candidate) {
                    return candidate;
                }
            }
            // JSX global fallback
            return getGlobalSymbol(JsxNames.JSX, SymbolFlags.Namespace, /*diagnosticMessage*/ undefined);
        }

        /**
         * Look into JSX namespace and then look for container with matching name as nameOfAttribPropContainer.
         * Get a single property from that container if existed. Report an error if there are more than one property.
         *
         * @param nameOfAttribPropContainer a string of value JsxNames.ElementAttributesPropertyNameContainer or JsxNames.ElementChildrenAttributeNameContainer
         *          if other string is given or the container doesn't exist, return undefined.
         */
        function getNameFromJsxElementAttributesContainer(nameOfAttribPropContainer: __String, jsxNamespace: Symbol): __String {
            // JSX.ElementAttributesProperty | JSX.ElementChildrenAttribute [symbol]
            const jsxElementAttribPropInterfaceSym = jsxNamespace && getSymbol(jsxNamespace.exports, nameOfAttribPropContainer, SymbolFlags.Type);
            // JSX.ElementAttributesProperty | JSX.ElementChildrenAttribute [type]
            const jsxElementAttribPropInterfaceType = jsxElementAttribPropInterfaceSym && getDeclaredTypeOfSymbol(jsxElementAttribPropInterfaceSym);
            // The properties of JSX.ElementAttributesProperty | JSX.ElementChildrenAttribute
            const propertiesOfJsxElementAttribPropInterface = jsxElementAttribPropInterfaceType && getPropertiesOfType(jsxElementAttribPropInterfaceType);
            if (propertiesOfJsxElementAttribPropInterface) {
                // Element Attributes has zero properties, so the element attributes type will be the class instance type
                if (propertiesOfJsxElementAttribPropInterface.length === 0) {
                    return "" as __String;
                }
                // Element Attributes has one property, so the element attributes type will be the type of the corresponding
                // property of the class instance type
                else if (propertiesOfJsxElementAttribPropInterface.length === 1) {
                    return propertiesOfJsxElementAttribPropInterface[0].escapedName;
                }
                else if (propertiesOfJsxElementAttribPropInterface.length > 1) {
                    // More than one property on ElementAttributesProperty is an error
                    error(jsxElementAttribPropInterfaceSym.declarations[0], Diagnostics.The_global_type_JSX_0_may_not_have_more_than_one_property, unescapeLeadingUnderscores(nameOfAttribPropContainer));
                }
            }
            return undefined;
        }

        /// e.g. "props" for React.d.ts,
        /// or 'undefined' if ElementAttributesProperty doesn't exist (which means all
        ///     non-intrinsic elements' attributes type is 'any'),
        /// or '' if it has 0 properties (which means every
        ///     non-intrinsic elements' attributes type is the element instance type)
        function getJsxElementPropertiesName(jsxNamespace: Symbol) {
            return getNameFromJsxElementAttributesContainer(JsxNames.ElementAttributesPropertyNameContainer, jsxNamespace);
        }

        function getJsxElementChildrenPropertyName(jsxNamespace: Symbol): __String {
            return getNameFromJsxElementAttributesContainer(JsxNames.ElementChildrenAttributeNameContainer, jsxNamespace);
        }

        function getApparentTypeOfJsxPropsType(propsType: Type): Type {
            if (!propsType) {
                return undefined;
            }
            if (propsType.flags & TypeFlags.Intersection) {
                const propsApparentType: Type[] = [];
                for (const t of (<UnionOrIntersectionType>propsType).types) {
                    propsApparentType.push(getApparentType(t));
                }
                return getIntersectionType(propsApparentType);
            }
            return getApparentType(propsType);
        }

        /**
         * Get JSX attributes type by trying to resolve openingLikeElement as a stateless function component.
         * Return only attributes type of successfully resolved call signature.
         * This function assumes that the caller handled other possible element type of the JSX element (e.g. stateful component)
         * Unlike tryGetAllJsxStatelessFunctionAttributesType, this function is a default behavior of type-checkers.
         * @param openingLikeElement a JSX opening-like element to find attributes type
         * @param elementType a type of the opening-like element. This elementType can't be an union type
         * @param elemInstanceType an element instance type (the result of newing or invoking this tag)
         * @param elementClassType a JSX-ElementClass type. This is a result of looking up ElementClass interface in the JSX global
         */
        function defaultTryGetJsxStatelessFunctionAttributesType(openingLikeElement: JsxOpeningLikeElement, elementType: Type, elemInstanceType: Type, elementClassType?: Type): Type {
            Debug.assert(!(elementType.flags & TypeFlags.Union));
            if (!elementClassType || !isTypeAssignableTo(elemInstanceType, elementClassType)) {
                const jsxStatelessElementType = getJsxStatelessElementTypeAt(openingLikeElement);
                if (jsxStatelessElementType) {
                    // We don't call getResolvedSignature here because we have already resolve the type of JSX Element.
                    const callSignature = getResolvedJsxStatelessFunctionSignature(openingLikeElement, elementType, /*candidatesOutArray*/ undefined);
                    if (callSignature !== unknownSignature) {
                        const callReturnType = callSignature && getReturnTypeOfSignature(callSignature);
                        let paramType = callReturnType && (callSignature.parameters.length === 0 ? emptyObjectType : getTypeOfSymbol(callSignature.parameters[0]));
                        paramType = getApparentTypeOfJsxPropsType(paramType);
                        if (callReturnType && isTypeAssignableTo(callReturnType, jsxStatelessElementType)) {
                            // Intersect in JSX.IntrinsicAttributes if it exists
                            const intrinsicAttributes = getJsxType(JsxNames.IntrinsicAttributes, openingLikeElement);
                            if (intrinsicAttributes !== unknownType) {
                                paramType = intersectTypes(intrinsicAttributes, paramType);
                            }
                            return paramType;
                        }
                    }
                }
            }
            return undefined;
        }

        /**
         * Get JSX attributes type by trying to resolve openingLikeElement as a stateless function component.
         * Return all attributes type of resolved call signature including candidate signatures.
         * This function assumes that the caller handled other possible element type of the JSX element.
         * This function is a behavior used by language service when looking up completion in JSX element.
         * @param openingLikeElement a JSX opening-like element to find attributes type
         * @param elementType a type of the opening-like element. This elementType can't be an union type
         * @param elemInstanceType an element instance type (the result of newing or invoking this tag)
         * @param elementClassType a JSX-ElementClass type. This is a result of looking up ElementClass interface in the JSX global
         */
        function tryGetAllJsxStatelessFunctionAttributesType(openingLikeElement: JsxOpeningLikeElement, elementType: Type, elemInstanceType: Type, elementClassType?: Type): Type {
            Debug.assert(!(elementType.flags & TypeFlags.Union));
            if (!elementClassType || !isTypeAssignableTo(elemInstanceType, elementClassType)) {
                // Is this is a stateless function component? See if its single signature's return type is assignable to the JSX Element Type
                const jsxStatelessElementType = getJsxStatelessElementTypeAt(openingLikeElement);
                if (jsxStatelessElementType) {
                    // We don't call getResolvedSignature because here we have already resolve the type of JSX Element.
                    const candidatesOutArray: Signature[] = [];
                    getResolvedJsxStatelessFunctionSignature(openingLikeElement, elementType, candidatesOutArray);
                    let result: Type;
                    let allMatchingAttributesType: Type;
                    for (const candidate of candidatesOutArray) {
                        const callReturnType = getReturnTypeOfSignature(candidate);
                        let paramType = callReturnType && (candidate.parameters.length === 0 ? emptyObjectType : getTypeOfSymbol(candidate.parameters[0]));
                        paramType = getApparentTypeOfJsxPropsType(paramType);
                        if (callReturnType && isTypeAssignableTo(callReturnType, jsxStatelessElementType)) {
                            let shouldBeCandidate = true;
                            for (const attribute of openingLikeElement.attributes.properties) {
                                if (isJsxAttribute(attribute) &&
                                    isUnhyphenatedJsxName(attribute.name.escapedText) &&
                                    !getPropertyOfType(paramType, attribute.name.escapedText)) {
                                    shouldBeCandidate = false;
                                    break;
                                }
                            }
                            if (shouldBeCandidate) {
                                result = intersectTypes(result, paramType);
                            }
                            allMatchingAttributesType = intersectTypes(allMatchingAttributesType, paramType);
                        }
                    }

                    // If we can't find any matching, just return everything.
                    if (!result) {
                        result = allMatchingAttributesType;
                    }
                    // Intersect in JSX.IntrinsicAttributes if it exists
                    const intrinsicAttributes = getJsxType(JsxNames.IntrinsicAttributes, openingLikeElement);
                    if (intrinsicAttributes !== unknownType) {
                        result = intersectTypes(intrinsicAttributes, result);
                    }
                    return result;
                }
            }
            return undefined;
        }

        /**
         * Resolve attributes type of the given opening-like element. The attributes type is a type of attributes associated with the given elementType.
         * For instance:
         *      declare function Foo(attr: { p1: string}): JSX.Element;
         *      <Foo p1={10} />;  // This function will try resolve "Foo" and return an attributes type of "Foo" which is "{ p1: string }"
         *
         * The function is intended to initially be called from getAttributesTypeFromJsxOpeningLikeElement which already handle JSX-intrinsic-element..
         * This function will try to resolve custom JSX attributes type in following order: string literal, stateless function, and stateful component
         *
         * @param openingLikeElement a non-intrinsic JSXOPeningLikeElement
         * @param shouldIncludeAllStatelessAttributesType a boolean indicating whether to include all attributes types from all stateless function signature
         * @param sourceAttributesType Is the attributes type the user passed, and is used to create inferences in the target type if present
         * @param elementType an instance type of the given opening-like element. If undefined, the function will check type openinglikeElement's tagname.
         * @param elementClassType a JSX-ElementClass type. This is a result of looking up ElementClass interface in the JSX global (imported from react.d.ts)
         * @return attributes type if able to resolve the type of node
         *         anyType if there is no type ElementAttributesProperty or there is an error
         *         emptyObjectType if there is no "prop" in the element instance type
         */
        function resolveCustomJsxElementAttributesType(openingLikeElement: JsxOpeningLikeElement,
            shouldIncludeAllStatelessAttributesType: boolean,
            elementType: Type,
            elementClassType?: Type): Type {

            if (elementType.flags & TypeFlags.Union) {
                const types = (elementType as UnionType).types;
                return getUnionType(types.map(type => {
                    return resolveCustomJsxElementAttributesType(openingLikeElement, shouldIncludeAllStatelessAttributesType, type, elementClassType);
                }), UnionReduction.Subtype);
            }

            // If the elemType is a string type, we have to return anyType to prevent an error downstream as we will try to find construct or call signature of the type
            if (elementType.flags & TypeFlags.String) {
                return anyType;
            }
            else if (elementType.flags & TypeFlags.StringLiteral) {
                // If the elemType is a stringLiteral type, we can then provide a check to make sure that the string literal type is one of the Jsx intrinsic element type
                // For example:
                //      var CustomTag: "h1" = "h1";
                //      <CustomTag> Hello World </CustomTag>
                const intrinsicElementsType = getJsxType(JsxNames.IntrinsicElements, openingLikeElement);
                if (intrinsicElementsType !== unknownType) {
                    const stringLiteralTypeName = (<StringLiteralType>elementType).value;
                    const intrinsicProp = getPropertyOfType(intrinsicElementsType, escapeLeadingUnderscores(stringLiteralTypeName));
                    if (intrinsicProp) {
                        return getTypeOfSymbol(intrinsicProp);
                    }
                    const indexSignatureType = getIndexTypeOfType(intrinsicElementsType, IndexKind.String);
                    if (indexSignatureType) {
                        return indexSignatureType;
                    }
                    error(openingLikeElement, Diagnostics.Property_0_does_not_exist_on_type_1, stringLiteralTypeName, "JSX." + JsxNames.IntrinsicElements);
                }
                // If we need to report an error, we already done so here. So just return any to prevent any more error downstream
                return anyType;
            }

            // Get the element instance type (the result of newing or invoking this tag)
            const elemInstanceType = getJsxElementInstanceType(openingLikeElement, elementType);

            // If we should include all stateless attributes type, then get all attributes type from all stateless function signature.
            // Otherwise get only attributes type from the signature picked by choose-overload logic.
            const statelessAttributesType = shouldIncludeAllStatelessAttributesType ?
                tryGetAllJsxStatelessFunctionAttributesType(openingLikeElement, elementType, elemInstanceType, elementClassType) :
                defaultTryGetJsxStatelessFunctionAttributesType(openingLikeElement, elementType, elemInstanceType, elementClassType);

            if (statelessAttributesType) {
                return statelessAttributesType;
            }

            // Issue an error if this return type isn't assignable to JSX.ElementClass
            if (elementClassType) {
                checkTypeRelatedTo(elemInstanceType, elementClassType, assignableRelation, openingLikeElement, Diagnostics.JSX_element_type_0_is_not_a_constructor_function_for_JSX_elements);
            }

            return getJsxPropsTypeFromClassType(elemInstanceType, isInJavaScriptFile(openingLikeElement), openingLikeElement);
        }

        /**
         * Get attributes type of the given intrinsic opening-like Jsx element by resolving the tag name.
         * The function is intended to be called from a function which has checked that the opening element is an intrinsic element.
         * @param node an intrinsic JSX opening-like element
         */
        function getIntrinsicAttributesTypeFromJsxOpeningLikeElement(node: JsxOpeningLikeElement): Type {
            Debug.assert(isJsxIntrinsicIdentifier(node.tagName));
            const links = getNodeLinks(node);
            if (!links.resolvedJsxElementAttributesType) {
                const symbol = getIntrinsicTagSymbol(node);
                if (links.jsxFlags & JsxFlags.IntrinsicNamedElement) {
                    return links.resolvedJsxElementAttributesType = getTypeOfSymbol(symbol);
                }
                else if (links.jsxFlags & JsxFlags.IntrinsicIndexedElement) {
                    return links.resolvedJsxElementAttributesType = getIndexInfoOfSymbol(symbol, IndexKind.String).type;
                }
                else {
                    return links.resolvedJsxElementAttributesType = unknownType;
                }
            }
            return links.resolvedJsxElementAttributesType;
        }

        /**
         * Get attributes type of the given custom opening-like JSX element.
         * This function is intended to be called from a caller that handles intrinsic JSX element already.
         * @param node a custom JSX opening-like element
         * @param shouldIncludeAllStatelessAttributesType a boolean value used by language service to get all possible attributes type from an overload stateless function component
         */
        function getCustomJsxElementAttributesType(node: JsxOpeningLikeElement, shouldIncludeAllStatelessAttributesType: boolean): Type {
            return resolveCustomJsxElementAttributesType(node, shouldIncludeAllStatelessAttributesType, checkExpression(node.tagName), getJsxElementClassTypeAt(node));
        }

        /**
         * Get all possible attributes type, especially from an overload stateless function component, of the given JSX opening-like element.
         * This function is called by language service (see: completions-tryGetGlobalSymbols).
         * @param node a JSX opening-like element to get attributes type for
         */
        function getAllAttributesTypeFromJsxOpeningLikeElement(node: JsxOpeningLikeElement): Type {
            if (isJsxIntrinsicIdentifier(node.tagName)) {
                return getIntrinsicAttributesTypeFromJsxOpeningLikeElement(node);
            }
            else {
                // Because in language service, the given JSX opening-like element may be incomplete and therefore,
                // we can't resolve to exact signature if the element is a stateless function component so the best thing to do is return all attributes type from all overloads.
                return getCustomJsxElementAttributesType(node, /*shouldIncludeAllStatelessAttributesType*/ true);
            }
        }

        /**
         * Get the attributes type, which indicates the attributes that are valid on the given JSXOpeningLikeElement.
         * @param node a JSXOpeningLikeElement node
         * @return an attributes type of the given node
         */
        function getAttributesTypeFromJsxOpeningLikeElement(node: JsxOpeningLikeElement): Type {
            if (isJsxIntrinsicIdentifier(node.tagName)) {
                return getIntrinsicAttributesTypeFromJsxOpeningLikeElement(node);
            }
            else {
                return getCustomJsxElementAttributesType(node, /*shouldIncludeAllStatelessAttributesType*/ false);
            }
        }

        /**
         * Given a JSX attribute, returns the symbol for the corresponds property
         * of the element attributes type. Will return unknownSymbol for attributes
         * that have no matching element attributes type property.
         */
        function getJsxAttributePropertySymbol(attrib: JsxAttribute): Symbol {
            const attributesType = getAttributesTypeFromJsxOpeningLikeElement(attrib.parent.parent as JsxOpeningElement);
            const prop = getPropertyOfType(attributesType, attrib.name.escapedText);
            return prop || unknownSymbol;
        }

        function getJsxElementClassTypeAt(location: Node): Type {
            const type = getJsxType(JsxNames.ElementClass, location);
            if (type === unknownType) return undefined;
            return type;
        }

        function getJsxElementTypeAt(location: Node): Type {
            return getJsxType(JsxNames.Element, location);
        }

        function getJsxStatelessElementTypeAt(location: Node): Type {
            const jsxElementType = getJsxElementTypeAt(location);
            if (jsxElementType) {
                return getUnionType([jsxElementType, nullType]);
            }
        }

        /**
         * Returns all the properties of the Jsx.IntrinsicElements interface
         */
        function getJsxIntrinsicTagNamesAt(location: Node): Symbol[] {
            const intrinsics = getJsxType(JsxNames.IntrinsicElements, location);
            return intrinsics ? getPropertiesOfType(intrinsics) : emptyArray;
        }

        function checkJsxPreconditions(errorNode: Node) {
            // Preconditions for using JSX
            if ((compilerOptions.jsx || JsxEmit.None) === JsxEmit.None) {
                error(errorNode, Diagnostics.Cannot_use_JSX_unless_the_jsx_flag_is_provided);
            }

            if (getJsxElementTypeAt(errorNode) === undefined) {
                if (noImplicitAny) {
                    error(errorNode, Diagnostics.JSX_element_implicitly_has_type_any_because_the_global_type_JSX_Element_does_not_exist);
                }
            }
        }

        function checkJsxOpeningLikeElementOrOpeningFragment(node: JsxOpeningLikeElement | JsxOpeningFragment, checkMode: CheckMode) {
            const isNodeOpeningLikeElement = isJsxOpeningLikeElement(node);

            if (isNodeOpeningLikeElement) {
                checkGrammarJsxElement(<JsxOpeningLikeElement>node);
            }
            checkJsxPreconditions(node);
            // The reactNamespace/jsxFactory's root symbol should be marked as 'used' so we don't incorrectly elide its import.
            // And if there is no reactNamespace/jsxFactory's symbol in scope when targeting React emit, we should issue an error.
            const reactRefErr = diagnostics && compilerOptions.jsx === JsxEmit.React ? Diagnostics.Cannot_find_name_0 : undefined;
            const reactNamespace = getJsxNamespace(node);
            const reactLocation = isNodeOpeningLikeElement ? (<JsxOpeningLikeElement>node).tagName : node;
            const reactSym = resolveName(reactLocation, reactNamespace, SymbolFlags.Value, reactRefErr, reactNamespace, /*isUse*/ true);
            if (reactSym) {
                // Mark local symbol as referenced here because it might not have been marked
                // if jsx emit was not react as there wont be error being emitted
                reactSym.isReferenced = SymbolFlags.All;

                // If react symbol is alias, mark it as refereced
                if (reactSym.flags & SymbolFlags.Alias && !isConstEnumOrConstEnumOnlyModule(resolveAlias(reactSym))) {
                    markAliasSymbolAsReferenced(reactSym);
                }
            }

            if (isNodeOpeningLikeElement) {
                checkJsxAttributesAssignableToTagNameAttributes(<JsxOpeningLikeElement>node, checkMode);
            }
            else {
                checkJsxChildren((node as JsxOpeningFragment).parent);
            }
        }

        /**
         * Check if a property with the given name is known anywhere in the given type. In an object type, a property
         * is considered known if
         * 1. the object type is empty and the check is for assignability, or
         * 2. if the object type has index signatures, or
         * 3. if the property is actually declared in the object type
         *    (this means that 'toString', for example, is not usually a known property).
         * 4. In a union or intersection type,
         *    a property is considered known if it is known in any constituent type.
         * @param targetType a type to search a given name in
         * @param name a property name to search
         * @param isComparingJsxAttributes a boolean flag indicating whether we are searching in JsxAttributesType
         */
        function isKnownProperty(targetType: Type, name: __String, isComparingJsxAttributes: boolean): boolean {
            if (targetType.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>targetType);
                if (resolved.stringIndexInfo ||
                    resolved.numberIndexInfo && isNumericLiteralName(name) ||
                    getPropertyOfObjectType(targetType, name) ||
                    isComparingJsxAttributes && !isUnhyphenatedJsxName(name)) {
                    // For JSXAttributes, if the attribute has a hyphenated name, consider that the attribute to be known.
                    return true;
                }
            }
            else if (targetType.flags & TypeFlags.UnionOrIntersection) {
                for (const t of (<UnionOrIntersectionType>targetType).types) {
                    if (isKnownProperty(t, name, isComparingJsxAttributes)) {
                        return true;
                    }
                }
            }
            return false;
        }

         /**
          * Check whether the given attributes of JSX opening-like element is assignable to the tagName attributes.
          *      Get the attributes type of the opening-like element through resolving the tagName, "target attributes"
          *      Check assignablity between given attributes property, "source attributes", and the "target attributes"
          * @param openingLikeElement an opening-like JSX element to check its JSXAttributes
          */
        function checkJsxAttributesAssignableToTagNameAttributes(openingLikeElement: JsxOpeningLikeElement, checkMode: CheckMode) {
            // The function involves following steps:
            //      1. Figure out expected attributes type by resolving tagName of the JSX opening-like element, targetAttributesType.
            //         During these steps, we will try to resolve the tagName as intrinsic name, stateless function, stateful component (in the order)
            //      2. Solved JSX attributes type given by users, sourceAttributesType, which is by resolving "attributes" property of the JSX opening-like element.
            //      3. Check if the two are assignable to each other


            // targetAttributesType is a type of an attribute from resolving tagName of an opening-like JSX element.
            const targetAttributesType = isJsxIntrinsicIdentifier(openingLikeElement.tagName) ?
                getIntrinsicAttributesTypeFromJsxOpeningLikeElement(openingLikeElement) :
                getCustomJsxElementAttributesType(openingLikeElement, /*shouldIncludeAllStatelessAttributesType*/ false);

            // sourceAttributesType is a type of an attributes properties.
            // i.e <div attr1={10} attr2="string" />
            //     attr1 and attr2 are treated as JSXAttributes attached in the JsxOpeningLikeElement as "attributes".
            const sourceAttributesType = createJsxAttributesTypeFromAttributesProperty(openingLikeElement, checkMode);

            // If the targetAttributesType is an emptyObjectType, indicating that there is no property named 'props' on this instance type.
            // but there exists a sourceAttributesType, we need to explicitly give an error as normal assignability check allow excess properties and will pass.
            if (targetAttributesType === emptyObjectType && (isTypeAny(sourceAttributesType) || getPropertiesOfType(<ResolvedType>sourceAttributesType).length > 0)) {
                error(openingLikeElement, Diagnostics.JSX_element_class_does_not_support_attributes_because_it_does_not_have_a_0_property, unescapeLeadingUnderscores(getJsxElementPropertiesName(getJsxNamespaceAt(openingLikeElement))));
            }
            else {
                // Check if sourceAttributesType assignable to targetAttributesType though this check will allow excess properties
                const isSourceAttributeTypeAssignableToTarget = checkTypeAssignableTo(sourceAttributesType, targetAttributesType, openingLikeElement.attributes.properties.length > 0 ? openingLikeElement.attributes : openingLikeElement);
                // After we check for assignability, we will do another pass to check that all explicitly specified attributes have correct name corresponding in targetAttributeType.
                // This will allow excess properties in spread type as it is very common pattern to spread outer attributes into React component in its render method.
                if (isSourceAttributeTypeAssignableToTarget && !isTypeAny(sourceAttributesType) && !isTypeAny(targetAttributesType)) {
                    for (const attribute of openingLikeElement.attributes.properties) {
                        if (!isJsxAttribute(attribute)) {
                            continue;
                        }
                        const attrName = attribute.name;
                        const isNotIgnoredJsxProperty = (isUnhyphenatedJsxName(idText(attrName)) || !!(getPropertyOfType(targetAttributesType, attrName.escapedText)));
                        if (isNotIgnoredJsxProperty && !isKnownProperty(targetAttributesType, attrName.escapedText, /*isComparingJsxAttributes*/ true)) {
                            error(attribute, Diagnostics.Property_0_does_not_exist_on_type_1, idText(attrName), typeToString(targetAttributesType));
                            // We break here so that errors won't be cascading
                            break;
                        }
                    }
                }
            }
        }

        function checkJsxExpression(node: JsxExpression, checkMode?: CheckMode) {
            if (node.expression) {
                const type = checkExpression(node.expression, checkMode);
                if (node.dotDotDotToken && type !== anyType && !isArrayType(type)) {
                    error(node, Diagnostics.JSX_spread_child_must_be_an_array_type);
                }
                return type;
            }
            else {
                return unknownType;
            }
        }

        // If a symbol is a synthesized symbol with no value declaration, we assume it is a property. Example of this are the synthesized
        // '.prototype' property as well as synthesized tuple index properties.
        function getDeclarationKindFromSymbol(s: Symbol) {
            return s.valueDeclaration ? s.valueDeclaration.kind : SyntaxKind.PropertyDeclaration;
        }

        function getDeclarationNodeFlagsFromSymbol(s: Symbol): NodeFlags {
            return s.valueDeclaration ? getCombinedNodeFlags(s.valueDeclaration) : 0;
        }

        function isMethodLike(symbol: Symbol) {
            return !!(symbol.flags & SymbolFlags.Method || getCheckFlags(symbol) & CheckFlags.SyntheticMethod);
        }

        /**
         * Check whether the requested property access is valid.
         * Returns true if node is a valid property access, and false otherwise.
         * @param node The node to be checked.
         * @param left The left hand side of the property access (e.g.: the super in `super.foo`).
         * @param type The type of left.
         * @param prop The symbol for the right hand side of the property access.
         */
        function checkPropertyAccessibility(node: PropertyAccessExpression | QualifiedName | VariableLikeDeclaration, left: Expression | QualifiedName, type: Type, prop: Symbol): boolean {
            const flags = getDeclarationModifierFlagsFromSymbol(prop);
            const errorNode = node.kind === SyntaxKind.PropertyAccessExpression || node.kind === SyntaxKind.VariableDeclaration ?
                node.name :
                (<QualifiedName>node).right;

            if (getCheckFlags(prop) & CheckFlags.ContainsPrivate) {
                // Synthetic property with private constituent property
                error(errorNode, Diagnostics.Property_0_has_conflicting_declarations_and_is_inaccessible_in_type_1, symbolToString(prop), typeToString(type));
                return false;
            }

            if (left.kind === SyntaxKind.SuperKeyword) {
                // TS 1.0 spec (April 2014): 4.8.2
                // - In a constructor, instance member function, instance member accessor, or
                //   instance member variable initializer where this references a derived class instance,
                //   a super property access is permitted and must specify a public instance member function of the base class.
                // - In a static member function or static member accessor
                //   where this references the constructor function object of a derived class,
                //   a super property access is permitted and must specify a public static member function of the base class.
                if (languageVersion < ScriptTarget.ES2015) {
                    if (symbolHasNonMethodDeclaration(prop)) {
                        error(errorNode, Diagnostics.Only_public_and_protected_methods_of_the_base_class_are_accessible_via_the_super_keyword);
                        return false;
                    }
                }
                if (flags & ModifierFlags.Abstract) {
                    // A method cannot be accessed in a super property access if the method is abstract.
                    // This error could mask a private property access error. But, a member
                    // cannot simultaneously be private and abstract, so this will trigger an
                    // additional error elsewhere.
                    error(errorNode, Diagnostics.Abstract_method_0_in_class_1_cannot_be_accessed_via_super_expression, symbolToString(prop), typeToString(getDeclaringClass(prop)));
                    return false;
                }
            }

            // Referencing abstract properties within their own constructors is not allowed
            if ((flags & ModifierFlags.Abstract) && isThisProperty(node) && symbolHasNonMethodDeclaration(prop)) {
                const declaringClassDeclaration = getClassLikeDeclarationOfSymbol(getParentOfSymbol(prop));
                if (declaringClassDeclaration && isNodeWithinConstructorOfClass(node, declaringClassDeclaration)) {
                    error(errorNode, Diagnostics.Abstract_property_0_in_class_1_cannot_be_accessed_in_the_constructor, symbolToString(prop), getTextOfIdentifierOrLiteral(declaringClassDeclaration.name));
                    return false;
                }
            }

            // Public properties are otherwise accessible.
            if (!(flags & ModifierFlags.NonPublicAccessibilityModifier)) {
                return true;
            }

            // Property is known to be private or protected at this point

            // Private property is accessible if the property is within the declaring class
            if (flags & ModifierFlags.Private) {
                const declaringClassDeclaration = getClassLikeDeclarationOfSymbol(getParentOfSymbol(prop));
                if (!isNodeWithinClass(node, declaringClassDeclaration)) {
                    error(errorNode, Diagnostics.Property_0_is_private_and_only_accessible_within_class_1, symbolToString(prop), typeToString(getDeclaringClass(prop)));
                    return false;
                }
                return true;
            }

            // Property is known to be protected at this point

            // All protected properties of a supertype are accessible in a super access
            if (left.kind === SyntaxKind.SuperKeyword) {
                return true;
            }

            // Find the first enclosing class that has the declaring classes of the protected constituents
            // of the property as base classes
            const enclosingClass = forEachEnclosingClass(node, enclosingDeclaration => {
                const enclosingClass = <InterfaceType>getDeclaredTypeOfSymbol(getSymbolOfNode(enclosingDeclaration));
                return isClassDerivedFromDeclaringClasses(enclosingClass, prop) ? enclosingClass : undefined;
            });
            // A protected property is accessible if the property is within the declaring class or classes derived from it
            if (!enclosingClass) {
                error(errorNode, Diagnostics.Property_0_is_protected_and_only_accessible_within_class_1_and_its_subclasses, symbolToString(prop), typeToString(getDeclaringClass(prop) || type));
                return false;
            }
            // No further restrictions for static properties
            if (flags & ModifierFlags.Static) {
                return true;
            }
             if (type.flags & TypeFlags.TypeParameter) {
                // get the original type -- represented as the type constraint of the 'this' type
                type = (type as TypeParameter).isThisType ? getConstraintOfTypeParameter(<TypeParameter>type) : getBaseConstraintOfType(<TypeParameter>type);
            }
            if (!type || !hasBaseType(type, enclosingClass)) {
                error(errorNode, Diagnostics.Property_0_is_protected_and_only_accessible_through_an_instance_of_class_1, symbolToString(prop), typeToString(enclosingClass));
                return false;
            }
            return true;
        }

        function symbolHasNonMethodDeclaration(symbol: Symbol) {
            return forEachProperty(symbol, prop => {
                const propKind = getDeclarationKindFromSymbol(prop);
                return propKind !== SyntaxKind.MethodDeclaration && propKind !== SyntaxKind.MethodSignature;
            });
        }

        function checkNonNullExpression(
            node: Expression | QualifiedName,
            nullDiagnostic?: DiagnosticMessage,
            undefinedDiagnostic?: DiagnosticMessage,
            nullOrUndefinedDiagnostic?: DiagnosticMessage,
        ) {
            return checkNonNullType(
                checkExpression(node),
                node,
                nullDiagnostic,
                undefinedDiagnostic,
                nullOrUndefinedDiagnostic
            );
        }

        function checkNonNullType(
            type: Type,
            node: Node,
            nullDiagnostic?: DiagnosticMessage,
            undefinedDiagnostic?: DiagnosticMessage,
            nullOrUndefinedDiagnostic?: DiagnosticMessage
        ): Type {
            const kind = (strictNullChecks ? getFalsyFlags(type) : type.flags) & TypeFlags.Nullable;
            if (kind) {
                error(node, kind & TypeFlags.Undefined ? kind & TypeFlags.Null ?
                    (nullOrUndefinedDiagnostic || Diagnostics.Object_is_possibly_null_or_undefined) :
                    (undefinedDiagnostic || Diagnostics.Object_is_possibly_undefined) :
                    (nullDiagnostic || Diagnostics.Object_is_possibly_null)
                );
                const t = getNonNullableType(type);
                return t.flags & (TypeFlags.Nullable | TypeFlags.Never) ? unknownType : t;
            }
            return type;
        }

        function checkPropertyAccessExpression(node: PropertyAccessExpression) {
            return checkPropertyAccessExpressionOrQualifiedName(node, node.expression, node.name);
        }

        function checkQualifiedName(node: QualifiedName) {
            return checkPropertyAccessExpressionOrQualifiedName(node, node.left, node.right);
        }

        function checkPropertyAccessExpressionOrQualifiedName(node: PropertyAccessExpression | QualifiedName, left: Expression | QualifiedName, right: Identifier) {
            let propType: Type;
            const leftType = checkNonNullExpression(left);
            const parentSymbol = getNodeLinks(left).resolvedSymbol;
            const apparentType = getApparentType(getWidenedType(leftType));
            if (isTypeAny(apparentType) || apparentType === silentNeverType) {
                if (isIdentifier(left) && parentSymbol) {
                    markAliasReferenced(parentSymbol, node);
                }
                return apparentType;
            }
            const assignmentKind = getAssignmentTargetKind(node);
            const prop = getPropertyOfType(apparentType, right.escapedText);
            if (isIdentifier(left) && parentSymbol && !(prop && isConstEnumOrConstEnumOnlyModule(prop))) {
                markAliasReferenced(parentSymbol, node);
            }
            if (!prop) {
                const indexInfo = getIndexInfoOfType(apparentType, IndexKind.String);
                if (!(indexInfo && indexInfo.type)) {
                    if (right.escapedText && !checkAndReportErrorForExtendingInterface(node)) {
                        reportNonexistentProperty(right, leftType.flags & TypeFlags.TypeParameter && (leftType as TypeParameter).isThisType ? apparentType : leftType);
                    }
                    return unknownType;
                }
                if (indexInfo.isReadonly && (isAssignmentTarget(node) || isDeleteTarget(node))) {
                    error(node, Diagnostics.Index_signature_in_type_0_only_permits_reading, typeToString(apparentType));
                }
                propType = indexInfo.type;
            }
            else {
                checkPropertyNotUsedBeforeDeclaration(prop, node, right);
                markPropertyAsReferenced(prop, node, left.kind === SyntaxKind.ThisKeyword);
                getNodeLinks(node).resolvedSymbol = prop;
                checkPropertyAccessibility(node, left, apparentType, prop);
                if (assignmentKind) {
                    if (isReferenceToReadonlyEntity(<Expression>node, prop) || isReferenceThroughNamespaceImport(<Expression>node)) {
                        error(right, Diagnostics.Cannot_assign_to_0_because_it_is_a_constant_or_a_read_only_property, idText(right));
                        return unknownType;
                    }
                }
                propType = getConstraintForLocation(getTypeOfSymbol(prop), node);
            }
            // Only compute control flow type if this is a property access expression that isn't an
            // assignment target, and the referenced property was declared as a variable, property,
            // accessor, or optional method.
            if (node.kind !== SyntaxKind.PropertyAccessExpression ||
                assignmentKind === AssignmentKind.Definite ||
                prop && !(prop.flags & (SymbolFlags.Variable | SymbolFlags.Property | SymbolFlags.Accessor)) && !(prop.flags & SymbolFlags.Method && propType.flags & TypeFlags.Union)) {
                return propType;
            }
            // If strict null checks and strict property initialization checks are enabled, if we have
            // a this.xxx property access, if the property is an instance property without an initializer,
            // and if we are in a constructor of the same class as the property declaration, assume that
            // the property is uninitialized at the top of the control flow.
            let assumeUninitialized = false;
            if (strictNullChecks && strictPropertyInitialization && left.kind === SyntaxKind.ThisKeyword) {
                const declaration = prop && prop.valueDeclaration;
                if (declaration && isInstancePropertyWithoutInitializer(declaration)) {
                    const flowContainer = getControlFlowContainer(node);
                    if (flowContainer.kind === SyntaxKind.Constructor && flowContainer.parent === declaration.parent) {
                        assumeUninitialized = true;
                    }
                }
            }
            const flowType = getFlowTypeOfReference(node, propType, assumeUninitialized ? getOptionalType(propType) : propType);
            if (assumeUninitialized && !(getFalsyFlags(propType) & TypeFlags.Undefined) && getFalsyFlags(flowType) & TypeFlags.Undefined) {
                error(right, Diagnostics.Property_0_is_used_before_being_assigned, symbolToString(prop));
                // Return the declared type to reduce follow-on errors
                return propType;
            }
            return assignmentKind ? getBaseTypeOfLiteralType(flowType) : flowType;
        }

        function checkPropertyNotUsedBeforeDeclaration(prop: Symbol, node: PropertyAccessExpression | QualifiedName, right: Identifier): void {
            const { valueDeclaration } = prop;
            if (!valueDeclaration) {
                return;
            }

            if (isInPropertyInitializer(node) &&
                !isBlockScopedNameDeclaredBeforeUse(valueDeclaration, right)
                && !isPropertyDeclaredInAncestorClass(prop)) {
                error(right, Diagnostics.Block_scoped_variable_0_used_before_its_declaration, idText(right));
            }
            else if (valueDeclaration.kind === SyntaxKind.ClassDeclaration &&
                node.parent.kind !== SyntaxKind.TypeReference &&
                !(valueDeclaration.flags & NodeFlags.Ambient) &&
                !isBlockScopedNameDeclaredBeforeUse(valueDeclaration, right)) {
                error(right, Diagnostics.Class_0_used_before_its_declaration, idText(right));
            }
        }

        function isInPropertyInitializer(node: Node): boolean {
            return !!findAncestor(node, node => {
                switch (node.kind) {
                    case SyntaxKind.PropertyDeclaration:
                        return true;
                    case SyntaxKind.PropertyAssignment:
                        // We might be in `a = { b: this.b }`, so keep looking. See `tests/cases/compiler/useBeforeDeclaration_propertyAssignment.ts`.
                        return false;
                    default:
                        return isExpressionNode(node) ? false : "quit";
                }
            });
        }

        /**
         * It's possible that "prop.valueDeclaration" is a local declaration, but the property was also declared in a superclass.
         * In that case we won't consider it used before its declaration, because it gets its value from the superclass' declaration.
         */
        function isPropertyDeclaredInAncestorClass(prop: Symbol): boolean {
            if (!(prop.parent.flags & SymbolFlags.Class)) {
                return false;
            }
            let classType = getTypeOfSymbol(prop.parent) as InterfaceType;
            while (true) {
                classType = getSuperClass(classType);
                if (!classType) {
                    return false;
                }
                const superProperty = getPropertyOfObjectType(classType, prop.escapedName);
                if (superProperty && superProperty.valueDeclaration) {
                    return true;
                }
            }
        }

        function getSuperClass(classType: InterfaceType): InterfaceType | undefined {
            const x = getBaseTypes(classType);
            if (x.length === 0) {
                return undefined;
            }
            Debug.assert(x.length === 1);
            return x[0] as InterfaceType;
        }

        function reportNonexistentProperty(propNode: Identifier, containingType: Type) {
            let errorInfo: DiagnosticMessageChain;
            if (containingType.flags & TypeFlags.Union && !(containingType.flags & TypeFlags.Primitive)) {
                for (const subtype of (containingType as UnionType).types) {
                    if (!getPropertyOfType(subtype, propNode.escapedText)) {
                        errorInfo = chainDiagnosticMessages(errorInfo, Diagnostics.Property_0_does_not_exist_on_type_1, declarationNameToString(propNode), typeToString(subtype));
                        break;
                    }
                }
            }
            const suggestion = getSuggestionForNonexistentProperty(propNode, containingType);
            if (suggestion !== undefined) {
                errorInfo = chainDiagnosticMessages(errorInfo, Diagnostics.Property_0_does_not_exist_on_type_1_Did_you_mean_2, declarationNameToString(propNode), typeToString(containingType), suggestion);
            }
            else {
                errorInfo = chainDiagnosticMessages(errorInfo, Diagnostics.Property_0_does_not_exist_on_type_1, declarationNameToString(propNode), typeToString(containingType));
            }
            diagnostics.add(createDiagnosticForNodeFromMessageChain(propNode, errorInfo));
        }

        function getSuggestionForNonexistentProperty(node: Identifier, containingType: Type): string | undefined {
            const suggestion = getSpellingSuggestionForName(idText(node), getPropertiesOfType(containingType), SymbolFlags.Value);
            return suggestion && symbolName(suggestion);
        }

        function getSuggestionForNonexistentSymbol(location: Node, outerName: __String, meaning: SymbolFlags): string {
            Debug.assert(outerName !== undefined, "outername should always be defined");
            const result = resolveNameHelper(location, outerName, meaning, /*nameNotFoundMessage*/ undefined, outerName, /*isUse*/ false, /*excludeGlobals*/ false, (symbols, name, meaning) => {
                Debug.assertEqual(outerName, name, "name should equal outerName");
                const symbol = getSymbol(symbols, name, meaning);
                // Sometimes the symbol is found when location is a return type of a function: `typeof x` and `x` is declared in the body of the function
                // So the table *contains* `x` but `x` isn't actually in scope.
                // However, resolveNameHelper will continue and call this callback again, so we'll eventually get a correct suggestion.
                return symbol || getSpellingSuggestionForName(unescapeLeadingUnderscores(name), arrayFrom(symbols.values()), meaning);
            });
            return result && symbolName(result);
        }

        function getSuggestionForNonexistentModule(name: Identifier, targetModule: Symbol): string | undefined {
            const suggestion = targetModule.exports && getSpellingSuggestionForName(idText(name), getExportsOfModuleAsArray(targetModule), SymbolFlags.ModuleMember);
            return suggestion && symbolName(suggestion);
        }

        /**
         * Given a name and a list of symbols whose names are *not* equal to the name, return a spelling suggestion if there is one that is close enough.
         * Names less than length 3 only check for case-insensitive equality, not levenshtein distance.
         *
         * If there is a candidate that's the same except for case, return that.
         * If there is a candidate that's within one edit of the name, return that.
         * Otherwise, return the candidate with the smallest Levenshtein distance,
         *    except for candidates:
         *      * With no name
         *      * Whose meaning doesn't match the `meaning` parameter.
         *      * Whose length differs from the target name by more than 0.34 of the length of the name.
         *      * Whose levenshtein distance is more than 0.4 of the length of the name
         *        (0.4 allows 1 substitution/transposition for every 5 characters,
         *         and 1 insertion/deletion at 3 characters)
         */
        function getSpellingSuggestionForName(name: string, symbols: Symbol[], meaning: SymbolFlags): Symbol | undefined {
            const maximumLengthDifference = Math.min(2, Math.floor(name.length * 0.34));
            let bestDistance = Math.floor(name.length * 0.4) + 1; // If the best result isn't better than this, don't bother.
            let bestCandidate: Symbol | undefined;
            let justCheckExactMatches = false;
            const nameLowerCase = name.toLowerCase();
            for (const candidate of symbols) {
                const candidateName = symbolName(candidate);
                if (!(candidate.flags & meaning && Math.abs(candidateName.length - nameLowerCase.length) <= maximumLengthDifference)) {
                    continue;
                }
                const candidateNameLowerCase = candidateName.toLowerCase();
                if (candidateNameLowerCase === nameLowerCase) {
                    return candidate;
                }
                if (justCheckExactMatches) {
                    continue;
                }
                if (candidateName.length < 3) {
                    // Don't bother, user would have noticed a 2-character name having an extra character
                    continue;
                }
                // Only care about a result better than the best so far.
                const distance = levenshteinWithMax(nameLowerCase, candidateNameLowerCase, bestDistance - 1);
                if (distance === undefined) {
                    continue;
                }
                if (distance < 3) {
                    justCheckExactMatches = true;
                    bestCandidate = candidate;
                }
                else {
                    Debug.assert(distance < bestDistance); // Else `levenshteinWithMax` should return undefined
                    bestDistance = distance;
                    bestCandidate = candidate;
                }
            }
            return bestCandidate;
        }

        function levenshteinWithMax(s1: string, s2: string, max: number): number | undefined {
            let previous = new Array(s2.length + 1);
            let current = new Array(s2.length + 1);
            /** Represents any value > max. We don't care about the particular value. */
            const big = max + 1;

            for (let i = 0; i <= s2.length; i++) {
                previous[i] = i;
            }

            for (let i = 1; i <= s1.length; i++) {
                const c1 = s1.charCodeAt(i - 1);
                const minJ = i > max ? i - max : 1;
                const maxJ = s2.length > max + i ? max + i : s2.length;
                current[0] = i;
                /** Smallest value of the matrix in the ith column. */
                let colMin = i;
                for (let j = 1; j < minJ; j++) {
                    current[j] = big;
                }
                for (let j = minJ; j <= maxJ; j++) {
                    const dist = c1 === s2.charCodeAt(j - 1)
                        ? previous[j - 1]
                        : Math.min(/*delete*/ previous[j] + 1, /*insert*/ current[j - 1] + 1, /*substitute*/ previous[j - 1] + 2);
                    current[j] = dist;
                    colMin = Math.min(colMin, dist);
                }
                for (let j = maxJ + 1; j <= s2.length; j++) {
                    current[j] = big;
                }
                if (colMin > max) {
                    // Give up -- everything in this column is > max and it can't get better in future columns.
                    return undefined;
                }

                const temp = previous;
                previous = current;
                current = temp;
            }

            const res = previous[s2.length];
            return res > max ? undefined : res;
        }

        function markPropertyAsReferenced(prop: Symbol, nodeForCheckWriteOnly: Node | undefined, isThisAccess: boolean) {
            if (!prop || !noUnusedIdentifiers || !(prop.flags & SymbolFlags.ClassMember) || !prop.valueDeclaration || !hasModifier(prop.valueDeclaration, ModifierFlags.Private)) {
                return;
            }
            if (nodeForCheckWriteOnly && isWriteOnlyAccess(nodeForCheckWriteOnly) && !(prop.flags & SymbolFlags.SetAccessor && !(prop.flags & SymbolFlags.GetAccessor))) {
                return;
            }

            if (isThisAccess) {
                // Find any FunctionLikeDeclaration because those create a new 'this' binding. But this should only matter for methods (or getters/setters).
                const containingMethod = findAncestor(nodeForCheckWriteOnly, isFunctionLikeDeclaration);
                if (containingMethod && containingMethod.symbol === prop) {
                    return;
                }
            }

            (getCheckFlags(prop) & CheckFlags.Instantiated ? getSymbolLinks(prop).target : prop).isReferenced = SymbolFlags.All;
        }

        function isValidPropertyAccess(node: PropertyAccessExpression | QualifiedName, propertyName: __String): boolean {
            const left = node.kind === SyntaxKind.PropertyAccessExpression ? node.expression : node.left;
            return isValidPropertyAccessWithType(node, left, propertyName, getWidenedType(checkExpression(left)));
        }

        function isValidPropertyAccessForCompletions(node: PropertyAccessExpression, type: Type, property: Symbol): boolean {
            return isValidPropertyAccessWithType(node, node.expression, property.escapedName, type)
                && (!(property.flags & SymbolFlags.Method) || isValidMethodAccess(property, type));
        }
        function isValidMethodAccess(method: Symbol, actualThisType: Type): boolean {
            const propType = getTypeOfFuncClassEnumModule(method);
            const signatures = getSignaturesOfType(getNonNullableType(propType), SignatureKind.Call);
            Debug.assert(signatures.length !== 0);
            return signatures.some(sig => {
                const signatureThisType = getThisTypeOfSignature(sig);
                return !signatureThisType || isTypeAssignableTo(actualThisType, getInstantiatedSignatureThisType(sig, signatureThisType, actualThisType));
            });
        }
        function getInstantiatedSignatureThisType(sig: Signature, signatureThisType: Type, actualThisType: Type): Type {
            if (!sig.typeParameters) {
                return signatureThisType;
            }
            const context = createInferenceContext(sig.typeParameters, sig, InferenceFlags.None);
            inferTypes(context.inferences, actualThisType, signatureThisType);
            return instantiateType(signatureThisType, createSignatureTypeMapper(sig, getInferredTypes(context)));
        }

        function isValidPropertyAccessWithType(
            node: PropertyAccessExpression | QualifiedName,
            left: LeftHandSideExpression | QualifiedName,
            propertyName: __String,
            type: Type): boolean {

            if (type === unknownType || isTypeAny(type)) {
                return true;
            }
            const prop = getPropertyOfType(type, propertyName);
            return prop ? checkPropertyAccessibility(node, left, type, prop)
                // In js files properties of unions are allowed in completion
                : isInJavaScriptFile(node) && (type.flags & TypeFlags.Union) && (<UnionType>type).types.some(elementType => isValidPropertyAccessWithType(node, left, propertyName, elementType));
        }

        /**
         * Return the symbol of the for-in variable declared or referenced by the given for-in statement.
         */
        function getForInVariableSymbol(node: ForInStatement): Symbol {
            const initializer = node.initializer;
            if (initializer.kind === SyntaxKind.VariableDeclarationList) {
                const variable = (<VariableDeclarationList>initializer).declarations[0];
                if (variable && !isBindingPattern(variable.name)) {
                    return getSymbolOfNode(variable);
                }
            }
            else if (initializer.kind === SyntaxKind.Identifier) {
                return getResolvedSymbol(<Identifier>initializer);
            }
            return undefined;
        }

        /**
         * Return true if the given type is considered to have numeric property names.
         */
        function hasNumericPropertyNames(type: Type) {
            return getIndexTypeOfType(type, IndexKind.Number) && !getIndexTypeOfType(type, IndexKind.String);
        }

        /**
         * Return true if given node is an expression consisting of an identifier (possibly parenthesized)
         * that references a for-in variable for an object with numeric property names.
         */
        function isForInVariableForNumericPropertyNames(expr: Expression) {
            const e = skipParentheses(expr);
            if (e.kind === SyntaxKind.Identifier) {
                const symbol = getResolvedSymbol(<Identifier>e);
                if (symbol.flags & SymbolFlags.Variable) {
                    let child: Node = expr;
                    let node = expr.parent;
                    while (node) {
                        if (node.kind === SyntaxKind.ForInStatement &&
                            child === (<ForInStatement>node).statement &&
                            getForInVariableSymbol(<ForInStatement>node) === symbol &&
                            hasNumericPropertyNames(getTypeOfExpression((<ForInStatement>node).expression))) {
                            return true;
                        }
                        child = node;
                        node = node.parent;
                    }
                }
            }
            return false;
        }

        function checkIndexedAccess(node: ElementAccessExpression): Type {
            const objectType = checkNonNullExpression(node.expression);

            const indexExpression = node.argumentExpression;
            if (!indexExpression) {
                const sourceFile = getSourceFileOfNode(node);
                if (node.parent.kind === SyntaxKind.NewExpression && (<NewExpression>node.parent).expression === node) {
                    const start = skipTrivia(sourceFile.text, node.expression.end);
                    const end = node.end;
                    grammarErrorAtPos(sourceFile, start, end - start, Diagnostics.new_T_cannot_be_used_to_create_an_array_Use_new_Array_T_instead);
                }
                else {
                    const start = node.end - "]".length;
                    const end = node.end;
                    grammarErrorAtPos(sourceFile, start, end - start, Diagnostics.Expression_expected);
                }
                return unknownType;
            }

            const indexType = isForInVariableForNumericPropertyNames(indexExpression) ? numberType : checkExpression(indexExpression);

            if (objectType === unknownType || objectType === silentNeverType) {
                return objectType;
            }

            if (isConstEnumObjectType(objectType) && indexExpression.kind !== SyntaxKind.StringLiteral) {
                error(indexExpression, Diagnostics.A_const_enum_member_can_only_be_accessed_using_a_string_literal);
                return unknownType;
            }

            return checkIndexedAccessIndexType(getIndexedAccessType(objectType, indexType, node), node);
        }

        function checkThatExpressionIsProperSymbolReference(expression: Expression, expressionType: Type, reportError: boolean): boolean {
            if (expressionType === unknownType) {
                // There is already an error, so no need to report one.
                return false;
            }

            if (!isWellKnownSymbolSyntactically(expression)) {
                return false;
            }

            // Make sure the property type is the primitive symbol type
            if ((expressionType.flags & TypeFlags.ESSymbolLike) === 0) {
                if (reportError) {
                    error(expression, Diagnostics.A_computed_property_name_of_the_form_0_must_be_of_type_symbol, getTextOfNode(expression));
                }
                return false;
            }

            // The name is Symbol.<someName>, so make sure Symbol actually resolves to the
            // global Symbol object
            const leftHandSide = <Identifier>(<PropertyAccessExpression>expression).expression;
            const leftHandSideSymbol = getResolvedSymbol(leftHandSide);
            if (!leftHandSideSymbol) {
                return false;
            }

            const globalESSymbol = getGlobalESSymbolConstructorSymbol(/*reportErrors*/ true);
            if (!globalESSymbol) {
                // Already errored when we tried to look up the symbol
                return false;
            }

            if (leftHandSideSymbol !== globalESSymbol) {
                if (reportError) {
                    error(leftHandSide, Diagnostics.Symbol_reference_does_not_refer_to_the_global_Symbol_constructor_object);
                }
                return false;
            }

            return true;
        }

        function callLikeExpressionMayHaveTypeArguments(node: CallLikeExpression): node is CallExpression | NewExpression {
            // TODO: Also include tagged templates (https://github.com/Microsoft/TypeScript/issues/11947)
            return isCallOrNewExpression(node);
        }

        function resolveUntypedCall(node: CallLikeExpression): Signature {
            if (callLikeExpressionMayHaveTypeArguments(node)) {
                // Check type arguments even though we will give an error that untyped calls may not accept type arguments.
                // This gets us diagnostics for the type arguments and marks them as referenced.
                forEach(node.typeArguments, checkSourceElement);
            }

            if (node.kind === SyntaxKind.TaggedTemplateExpression) {
                checkExpression(node.template);
            }
            else if (node.kind !== SyntaxKind.Decorator) {
                forEach((<CallExpression>node).arguments, argument => {
                    checkExpression(argument);
                });
            }
            return anySignature;
        }

        function resolveErrorCall(node: CallLikeExpression): Signature {
            resolveUntypedCall(node);
            return unknownSignature;
        }

        // Re-order candidate signatures into the result array. Assumes the result array to be empty.
        // The candidate list orders groups in reverse, but within a group signatures are kept in declaration order
        // A nit here is that we reorder only signatures that belong to the same symbol,
        // so order how inherited signatures are processed is still preserved.
        // interface A { (x: string): void }
        // interface B extends A { (x: 'foo'): string }
        // const b: B;
        // b('foo') // <- here overloads should be processed as [(x:'foo'): string, (x: string): void]
        function reorderCandidates(signatures: Signature[], result: Signature[]): void {
            let lastParent: Node;
            let lastSymbol: Symbol;
            let cutoffIndex = 0;
            let index: number;
            let specializedIndex = -1;
            let spliceIndex: number;
            Debug.assert(!result.length);
            for (const signature of signatures) {
                const symbol = signature.declaration && getSymbolOfNode(signature.declaration);
                const parent = signature.declaration && signature.declaration.parent;
                if (!lastSymbol || symbol === lastSymbol) {
                    if (lastParent && parent === lastParent) {
                        index++;
                    }
                    else {
                        lastParent = parent;
                        index = cutoffIndex;
                    }
                }
                else {
                    // current declaration belongs to a different symbol
                    // set cutoffIndex so re-orderings in the future won't change result set from 0 to cutoffIndex
                    index = cutoffIndex = result.length;
                    lastParent = parent;
                }
                lastSymbol = symbol;

                // specialized signatures always need to be placed before non-specialized signatures regardless
                // of the cutoff position; see GH#1133
                if (signature.hasLiteralTypes) {
                    specializedIndex++;
                    spliceIndex = specializedIndex;
                    // The cutoff index always needs to be greater than or equal to the specialized signature index
                    // in order to prevent non-specialized signatures from being added before a specialized
                    // signature.
                    cutoffIndex++;
                }
                else {
                    spliceIndex = index;
                }

                result.splice(spliceIndex, 0, signature);
            }
        }

        function getSpreadArgumentIndex(args: ReadonlyArray<Expression>): number {
            for (let i = 0; i < args.length; i++) {
                const arg = args[i];
                if (arg && arg.kind === SyntaxKind.SpreadElement) {
                    return i;
                }
            }
            return -1;
        }

        function hasCorrectArity(node: CallLikeExpression, args: ReadonlyArray<Expression>, signature: Signature, signatureHelpTrailingComma = false) {
            let argCount: number;            // Apparent number of arguments we will have in this call
            let typeArguments: NodeArray<TypeNode>;  // Type arguments (undefined if none)
            let callIsIncomplete: boolean;           // In incomplete call we want to be lenient when we have too few arguments
            let spreadArgIndex = -1;

            if (isJsxOpeningLikeElement(node)) {
                // The arity check will be done in "checkApplicableSignatureForJsxOpeningLikeElement".
                return true;
            }

            if (node.kind === SyntaxKind.TaggedTemplateExpression) {
                // Even if the call is incomplete, we'll have a missing expression as our last argument,
                // so we can say the count is just the arg list length
                argCount = args.length;
                typeArguments = undefined;

                if (node.template.kind === SyntaxKind.TemplateExpression) {
                    // If a tagged template expression lacks a tail literal, the call is incomplete.
                    // Specifically, a template only can end in a TemplateTail or a Missing literal.
                    const lastSpan = lastOrUndefined(node.template.templateSpans);
                    Debug.assert(lastSpan !== undefined); // we should always have at least one span.
                    callIsIncomplete = nodeIsMissing(lastSpan.literal) || !!lastSpan.literal.isUnterminated;
                }
                else {
                    // If the template didn't end in a backtick, or its beginning occurred right prior to EOF,
                    // then this might actually turn out to be a TemplateHead in the future;
                    // so we consider the call to be incomplete.
                    const templateLiteral = <LiteralExpression>node.template;
                    Debug.assert(templateLiteral.kind === SyntaxKind.NoSubstitutionTemplateLiteral);
                    callIsIncomplete = !!templateLiteral.isUnterminated;
                }
            }
            else if (node.kind === SyntaxKind.Decorator) {
                typeArguments = undefined;
                argCount = getEffectiveArgumentCount(node, /*args*/ undefined, signature);
            }
            else {
                if (!node.arguments) {
                    // This only happens when we have something of the form: 'new C'
                    Debug.assert(node.kind === SyntaxKind.NewExpression);

                    return signature.minArgumentCount === 0;
                }

                argCount = signatureHelpTrailingComma ? args.length + 1 : args.length;

                // If we are missing the close parenthesis, the call is incomplete.
                callIsIncomplete = node.arguments.end === node.end;

                typeArguments = node.typeArguments;
                spreadArgIndex = getSpreadArgumentIndex(args);
            }

            // If the user supplied type arguments, but the number of type arguments does not match
            // the declared number of type parameters, the call has an incorrect arity.
            const numTypeParameters = length(signature.typeParameters);
            const minTypeArgumentCount = getMinTypeArgumentCount(signature.typeParameters);
            const hasRightNumberOfTypeArgs = !typeArguments ||
                (typeArguments.length >= minTypeArgumentCount && typeArguments.length <= numTypeParameters);
            if (!hasRightNumberOfTypeArgs) {
                return false;
            }

            // If a spread argument is present, check that it corresponds to a rest parameter or at least that it's in the valid range.
            if (spreadArgIndex >= 0) {
                return isRestParameterIndex(signature, spreadArgIndex) ||
                    signature.minArgumentCount <= spreadArgIndex && spreadArgIndex < signature.parameters.length;
            }

            // Too many arguments implies incorrect arity.
            if (!signature.hasRestParameter && argCount > signature.parameters.length) {
                return false;
            }

            // If the call is incomplete, we should skip the lower bound check.
            const hasEnoughArguments = argCount >= signature.minArgumentCount;
            return callIsIncomplete || hasEnoughArguments;
        }

        // If type has a single call signature and no other members, return that signature. Otherwise, return undefined.
        function getSingleCallSignature(type: Type): Signature {
            if (type.flags & TypeFlags.Object) {
                const resolved = resolveStructuredTypeMembers(<ObjectType>type);
                if (resolved.callSignatures.length === 1 && resolved.constructSignatures.length === 0 &&
                    resolved.properties.length === 0 && !resolved.stringIndexInfo && !resolved.numberIndexInfo) {
                    return resolved.callSignatures[0];
                }
            }
            return undefined;
        }

        // Instantiate a generic signature in the context of a non-generic signature (section 3.8.5 in TypeScript spec)
        function instantiateSignatureInContextOf(signature: Signature, contextualSignature: Signature, contextualMapper?: TypeMapper, compareTypes?: TypeComparer): Signature {
            const context = createInferenceContext(signature.typeParameters, signature, InferenceFlags.InferUnionTypes, compareTypes);
            forEachMatchingParameterType(contextualSignature, signature, (source, target) => {
                // Type parameters from outer context referenced by source type are fixed by instantiation of the source type
                inferTypes(context.inferences, instantiateType(source, contextualMapper || identityMapper), target);
            });
            if (!contextualMapper) {
                inferTypes(context.inferences, getReturnTypeOfSignature(contextualSignature), getReturnTypeOfSignature(signature), InferencePriority.ReturnType);
            }
            return getSignatureInstantiation(signature, getInferredTypes(context), isInJavaScriptFile(contextualSignature.declaration));
        }

        function inferJsxTypeArguments(signature: Signature, node: JsxOpeningLikeElement, context: InferenceContext): Type[] {
            // Skip context sensitive pass
            const skipContextParamType = getTypeAtPosition(signature, 0);
            const checkAttrTypeSkipContextSensitive = checkExpressionWithContextualType(node.attributes, skipContextParamType, identityMapper);
            inferTypes(context.inferences, checkAttrTypeSkipContextSensitive, skipContextParamType);

            // Standard pass
            const paramType = getTypeAtPosition(signature, 0);
            const checkAttrType = checkExpressionWithContextualType(node.attributes, paramType, context);
            inferTypes(context.inferences, checkAttrType, paramType);

            return getInferredTypes(context);
        }

        function inferTypeArguments(node: CallLikeExpression, signature: Signature, args: ReadonlyArray<Expression>, excludeArgument: boolean[], context: InferenceContext): Type[] {
            // Clear out all the inference results from the last time inferTypeArguments was called on this context
            for (const inference of context.inferences) {
                // As an optimization, we don't have to clear (and later recompute) inferred types
                // for type parameters that have already been fixed on the previous call to inferTypeArguments.
                // It would be just as correct to reset all of them. But then we'd be repeating the same work
                // for the type parameters that were fixed, namely the work done by getInferredType.
                if (!inference.isFixed) {
                    inference.inferredType = undefined;
                }
            }

            // If a contextual type is available, infer from that type to the return type of the call expression. For
            // example, given a 'function wrap<T, U>(cb: (x: T) => U): (x: T) => U' and a call expression
            // 'let f: (x: string) => number = wrap(s => s.length)', we infer from the declared type of 'f' to the
            // return type of 'wrap'.
            if (node.kind !== SyntaxKind.Decorator) {
                const contextualType = getContextualType(node);
                if (contextualType) {
                    // We clone the contextual mapper to avoid disturbing a resolution in progress for an
                    // outer call expression. Effectively we just want a snapshot of whatever has been
                    // inferred for any outer call expression so far.
                    const instantiatedType = instantiateType(contextualType, cloneTypeMapper(getContextualMapper(node)));
                    // If the contextual type is a generic function type with a single call signature, we
                    // instantiate the type with its own type parameters and type arguments. This ensures that
                    // the type parameters are not erased to type any during type inference such that they can
                    // be inferred as actual types from the contextual type. For example:
                    //   declare function arrayMap<T, U>(f: (x: T) => U): (a: T[]) => U[];
                    //   const boxElements: <A>(a: A[]) => { value: A }[] = arrayMap(value => ({ value }));
                    // Above, the type of the 'value' parameter is inferred to be 'A'.
                    const contextualSignature = getSingleCallSignature(instantiatedType);
                    const inferenceSourceType = contextualSignature && contextualSignature.typeParameters ?
                        getOrCreateTypeFromSignature(getSignatureInstantiation(contextualSignature, contextualSignature.typeParameters, isInJavaScriptFile(node))) :
                        instantiatedType;
                    const inferenceTargetType = getReturnTypeOfSignature(signature);
                     // Inferences made from return types have lower priority than all other inferences.
                    inferTypes(context.inferences, inferenceSourceType, inferenceTargetType, InferencePriority.ReturnType);
                }
            }

            const thisType = getThisTypeOfSignature(signature);
            if (thisType) {
                const thisArgumentNode = getThisArgumentOfCall(node);
                const thisArgumentType = thisArgumentNode ? checkExpression(thisArgumentNode) : voidType;
                inferTypes(context.inferences, thisArgumentType, thisType);
            }

            // We perform two passes over the arguments. In the first pass we infer from all arguments, but use
            // wildcards for all context sensitive function expressions.
            const argCount = getEffectiveArgumentCount(node, args, signature);
            for (let i = 0; i < argCount; i++) {
                const arg = getEffectiveArgument(node, args, i);
                // If the effective argument is 'undefined', then it is an argument that is present but is synthetic.
                if (arg === undefined || arg.kind !== SyntaxKind.OmittedExpression) {
                    const paramType = getTypeAtPosition(signature, i);
                    let argType = getEffectiveArgumentType(node, i);

                    // If the effective argument type is 'undefined', there is no synthetic type
                    // for the argument. In that case, we should check the argument.
                    if (argType === undefined) {
                        // For context sensitive arguments we pass the identityMapper, which is a signal to treat all
                        // context sensitive function expressions as wildcards
                        const mapper = excludeArgument && excludeArgument[i] !== undefined ? identityMapper : context;
                        argType = checkExpressionWithContextualType(arg, paramType, mapper);
                    }

                    inferTypes(context.inferences, argType, paramType);
                }
            }

            // In the second pass we visit only context sensitive arguments, and only those that aren't excluded, this
            // time treating function expressions normally (which may cause previously inferred type arguments to be fixed
            // as we construct types for contextually typed parameters)
            // Decorators will not have `excludeArgument`, as their arguments cannot be contextually typed.
            // Tagged template expressions will always have `undefined` for `excludeArgument[0]`.
            if (excludeArgument) {
                for (let i = 0; i < argCount; i++) {
                    // No need to check for omitted args and template expressions, their exclusion value is always undefined
                    if (excludeArgument[i] === false) {
                        const arg = args[i];
                        const paramType = getTypeAtPosition(signature, i);
                        inferTypes(context.inferences, checkExpressionWithContextualType(arg, paramType, context), paramType);
                    }
                }
            }
            return getInferredTypes(context);
        }

        function checkTypeArguments(signature: Signature, typeArgumentNodes: ReadonlyArray<TypeNode>, reportErrors: boolean, headMessage?: DiagnosticMessage): Type[] | false {
            const isJavascript = isInJavaScriptFile(signature.declaration);
            const typeParameters = signature.typeParameters;
            const typeArgumentTypes = fillMissingTypeArguments(map(typeArgumentNodes, getTypeFromTypeNode), typeParameters, getMinTypeArgumentCount(typeParameters), isJavascript);
            let mapper: TypeMapper;
            for (let i = 0; i < typeArgumentNodes.length; i++) {
                Debug.assert(typeParameters[i] !== undefined, "Should not call checkTypeArguments with too many type arguments");
                const constraint = getConstraintOfTypeParameter(typeParameters[i]);
                if (!constraint) continue;

                const errorInfo = reportErrors && headMessage && (() => chainDiagnosticMessages(/*details*/ undefined, Diagnostics.Type_0_does_not_satisfy_the_constraint_1));
                const typeArgumentHeadMessage = headMessage || Diagnostics.Type_0_does_not_satisfy_the_constraint_1;
                if (!mapper) {
                    mapper = createTypeMapper(typeParameters, typeArgumentTypes);
                }
                const typeArgument = typeArgumentTypes[i];
                if (!checkTypeAssignableTo(
                    typeArgument,
                    getTypeWithThisArgument(instantiateType(constraint, mapper), typeArgument),
                    reportErrors ? typeArgumentNodes[i] : undefined,
                    typeArgumentHeadMessage,
                    errorInfo)) {
                    return false;
                }
            }
            return typeArgumentTypes;
        }

        /**
         * Check if the given signature can possibly be a signature called by the JSX opening-like element.
         * @param node a JSX opening-like element we are trying to figure its call signature
         * @param signature a candidate signature we are trying whether it is a call signature
         * @param relation a relationship to check parameter and argument type
         * @param excludeArgument
         */
        function checkApplicableSignatureForJsxOpeningLikeElement(node: JsxOpeningLikeElement, signature: Signature, relation: Map<RelationComparisonResult>) {
            // JSX opening-like element has correct arity for stateless-function component if the one of the following condition is true:
            //      1. callIsIncomplete
            //      2. attributes property has same number of properties as the parameter object type.
            //         We can figure that out by resolving attributes property and check number of properties in the resolved type
            // If the call has correct arity, we will then check if the argument type and parameter type is assignable

            const callIsIncomplete = node.attributes.end === node.end;  // If we are missing the close "/>", the call is incomplete
            if (callIsIncomplete) {
                return true;
            }

            const headMessage = Diagnostics.Argument_of_type_0_is_not_assignable_to_parameter_of_type_1;
            // Stateless function components can have maximum of three arguments: "props", "context", and "updater".
            // However "context" and "updater" are implicit and can't be specify by users. Only the first parameter, props,
            // can be specified by users through attributes property.
            const paramType = getTypeAtPosition(signature, 0);
            const attributesType = checkExpressionWithContextualType(node.attributes, paramType, /*contextualMapper*/ undefined);
            const argProperties = getPropertiesOfType(attributesType);
            for (const arg of argProperties) {
                if (!getPropertyOfType(paramType, arg.escapedName) && isUnhyphenatedJsxName(arg.escapedName)) {
                    return false;
                }
            }
            return checkTypeRelatedTo(attributesType, paramType, relation, /*errorNode*/ undefined, headMessage);
        }

        function checkApplicableSignature(
            node: CallLikeExpression,
            args: ReadonlyArray<Expression>,
            signature: Signature,
            relation: Map<RelationComparisonResult>,
            excludeArgument: boolean[],
            reportErrors: boolean) {
            if (isJsxOpeningLikeElement(node)) {
                return checkApplicableSignatureForJsxOpeningLikeElement(node, signature, relation);
            }
            const thisType = getThisTypeOfSignature(signature);
            if (thisType && thisType !== voidType && node.kind !== SyntaxKind.NewExpression) {
                // If the called expression is not of the form `x.f` or `x["f"]`, then sourceType = voidType
                // If the signature's 'this' type is voidType, then the check is skipped -- anything is compatible.
                // If the expression is a new expression, then the check is skipped.
                const thisArgumentNode = getThisArgumentOfCall(node);
                const thisArgumentType = thisArgumentNode ? checkExpression(thisArgumentNode) : voidType;
                const errorNode = reportErrors ? (thisArgumentNode || node) : undefined;
                const headMessage = Diagnostics.The_this_context_of_type_0_is_not_assignable_to_method_s_this_of_type_1;
                if (!checkTypeRelatedTo(thisArgumentType, getThisTypeOfSignature(signature), relation, errorNode, headMessage)) {
                    return false;
                }
            }
            const headMessage = Diagnostics.Argument_of_type_0_is_not_assignable_to_parameter_of_type_1;
            const argCount = getEffectiveArgumentCount(node, args, signature);
            for (let i = 0; i < argCount; i++) {
                const arg = getEffectiveArgument(node, args, i);
                // If the effective argument is 'undefined', then it is an argument that is present but is synthetic.
                if (arg === undefined || arg.kind !== SyntaxKind.OmittedExpression) {
                    // Check spread elements against rest type (from arity check we know spread argument corresponds to a rest parameter)
                    const paramType = getTypeAtPosition(signature, i);
                    // If the effective argument type is undefined, there is no synthetic type for the argument.
                    // In that case, we should check the argument.
                    const argType = getEffectiveArgumentType(node, i) ||
                        checkExpressionWithContextualType(arg, paramType, excludeArgument && excludeArgument[i] ? identityMapper : undefined);
                    // If one or more arguments are still excluded (as indicated by a non-null excludeArgument parameter),
                    // we obtain the regular type of any object literal arguments because we may not have inferred complete
                    // parameter types yet and therefore excess property checks may yield false positives (see #17041).
                    const checkArgType = excludeArgument ? getRegularTypeOfObjectLiteral(argType) : argType;
                    // Use argument expression as error location when reporting errors
                    const errorNode = reportErrors ? getEffectiveArgumentErrorNode(node, i, arg) : undefined;
                    if (!checkTypeRelatedTo(checkArgType, paramType, relation, errorNode, headMessage)) {
                        return false;
                    }
                }
            }

            return true;
        }

        /**
         * Returns the this argument in calls like x.f(...) and x[f](...). Undefined otherwise.
         */
        function getThisArgumentOfCall(node: CallLikeExpression): LeftHandSideExpression {
            if (node.kind === SyntaxKind.CallExpression) {
                const callee = node.expression;
                if (callee.kind === SyntaxKind.PropertyAccessExpression) {
                    return (callee as PropertyAccessExpression).expression;
                }
                else if (callee.kind === SyntaxKind.ElementAccessExpression) {
                    return (callee as ElementAccessExpression).expression;
                }
            }
        }

        /**
         * Returns the effective arguments for an expression that works like a function invocation.
         *
         * If 'node' is a CallExpression or a NewExpression, then its argument list is returned.
         * If 'node' is a TaggedTemplateExpression, a new argument list is constructed from the substitution
         *    expressions, where the first element of the list is `undefined`.
         * If 'node' is a Decorator, the argument list will be `undefined`, and its arguments and types
         *    will be supplied from calls to `getEffectiveArgumentCount` and `getEffectiveArgumentType`.
         */
        function getEffectiveCallArguments(node: CallLikeExpression): ReadonlyArray<Expression> {
            if (node.kind === SyntaxKind.TaggedTemplateExpression) {
                const template = node.template;
                const args: Expression[] = [undefined];
                if (template.kind === SyntaxKind.TemplateExpression) {
                    forEach(template.templateSpans, span => {
                        args.push(span.expression);
                    });
                }
                return args;
            }
            else if (node.kind === SyntaxKind.Decorator) {
                // For a decorator, we return undefined as we will determine
                // the number and types of arguments for a decorator using
                // `getEffectiveArgumentCount` and `getEffectiveArgumentType` below.
                return undefined;
            }
            else if (isJsxOpeningLikeElement(node)) {
                return node.attributes.properties.length > 0 ? [node.attributes] : emptyArray;
            }
            else {
                return node.arguments || emptyArray;
            }
        }


        /**
         * Returns the effective argument count for a node that works like a function invocation.
         * If 'node' is a Decorator, the number of arguments is derived from the decoration
         *    target and the signature:
         *    If 'node.target' is a class declaration or class expression, the effective argument
         *       count is 1.
         *    If 'node.target' is a parameter declaration, the effective argument count is 3.
         *    If 'node.target' is a property declaration, the effective argument count is 2.
         *    If 'node.target' is a method or accessor declaration, the effective argument count
         *       is 3, although it can be 2 if the signature only accepts two arguments, allowing
         *       us to match a property decorator.
         * Otherwise, the argument count is the length of the 'args' array.
         */
        function getEffectiveArgumentCount(node: CallLikeExpression, args: ReadonlyArray<Expression>, signature: Signature) {
            if (node.kind === SyntaxKind.Decorator) {
                switch (node.parent.kind) {
                    case SyntaxKind.ClassDeclaration:
                    case SyntaxKind.ClassExpression:
                        // A class decorator will have one argument (see `ClassDecorator` in core.d.ts)
                        return 1;

                    case SyntaxKind.PropertyDeclaration:
                        // A property declaration decorator will have two arguments (see
                        // `PropertyDecorator` in core.d.ts)
                        return 2;

                    case SyntaxKind.MethodDeclaration:
                    case SyntaxKind.GetAccessor:
                    case SyntaxKind.SetAccessor:
                        // A method or accessor declaration decorator will have two or three arguments (see
                        // `PropertyDecorator` and `MethodDecorator` in core.d.ts)

                        // If we are emitting decorators for ES3, we will only pass two arguments.
                        if (languageVersion === ScriptTarget.ES3) {
                            return 2;
                        }

                        // If the method decorator signature only accepts a target and a key, we will only
                        // type check those arguments.
                        return signature.parameters.length >= 3 ? 3 : 2;

                    case SyntaxKind.Parameter:
                        // A parameter declaration decorator will have three arguments (see
                        // `ParameterDecorator` in core.d.ts)

                        return 3;
                }
            }
            else {
                return args.length;
            }
        }

        /**
         * Returns the effective type of the first argument to a decorator.
         * If 'node' is a class declaration or class expression, the effective argument type
         *    is the type of the static side of the class.
         * If 'node' is a parameter declaration, the effective argument type is either the type
         *    of the static or instance side of the class for the parameter's parent method,
         *    depending on whether the method is declared static.
         *    For a constructor, the type is always the type of the static side of the class.
         * If 'node' is a property, method, or accessor declaration, the effective argument
         *    type is the type of the static or instance side of the parent class for class
         *    element, depending on whether the element is declared static.
         */
        function getEffectiveDecoratorFirstArgumentType(node: Node): Type {
            // The first argument to a decorator is its `target`.
            if (node.kind === SyntaxKind.ClassDeclaration) {
                // For a class decorator, the `target` is the type of the class (e.g. the
                // "static" or "constructor" side of the class)
                const classSymbol = getSymbolOfNode(node);
                return getTypeOfSymbol(classSymbol);
            }

            if (node.kind === SyntaxKind.Parameter) {
                // For a parameter decorator, the `target` is the parent type of the
                // parameter's containing method.
                node = node.parent;
                if (node.kind === SyntaxKind.Constructor) {
                    const classSymbol = getSymbolOfNode(node);
                    return getTypeOfSymbol(classSymbol);
                }
            }

            if (node.kind === SyntaxKind.PropertyDeclaration ||
                node.kind === SyntaxKind.MethodDeclaration ||
                node.kind === SyntaxKind.GetAccessor ||
                node.kind === SyntaxKind.SetAccessor) {
                // For a property or method decorator, the `target` is the
                // "static"-side type of the parent of the member if the member is
                // declared "static"; otherwise, it is the "instance"-side type of the
                // parent of the member.
                return getParentTypeOfClassElement(<ClassElement>node);
            }

            Debug.fail("Unsupported decorator target.");
            return unknownType;
        }

        /**
         * Returns the effective type for the second argument to a decorator.
         * If 'node' is a parameter, its effective argument type is one of the following:
         *    If 'node.parent' is a constructor, the effective argument type is 'any', as we
         *       will emit `undefined`.
         *    If 'node.parent' is a member with an identifier, numeric, or string literal name,
         *       the effective argument type will be a string literal type for the member name.
         *    If 'node.parent' is a computed property name, the effective argument type will
         *       either be a symbol type or the string type.
         * If 'node' is a member with an identifier, numeric, or string literal name, the
         *    effective argument type will be a string literal type for the member name.
         * If 'node' is a computed property name, the effective argument type will either
         *    be a symbol type or the string type.
         * A class decorator does not have a second argument type.
         */
        function getEffectiveDecoratorSecondArgumentType(node: Node) {
            // The second argument to a decorator is its `propertyKey`
            if (node.kind === SyntaxKind.ClassDeclaration) {
                Debug.fail("Class decorators should not have a second synthetic argument.");
                return unknownType;
            }

            if (node.kind === SyntaxKind.Parameter) {
                node = node.parent;
                if (node.kind === SyntaxKind.Constructor) {
                    // For a constructor parameter decorator, the `propertyKey` will be `undefined`.
                    return anyType;
                }

                // For a non-constructor parameter decorator, the `propertyKey` will be either
                // a string or a symbol, based on the name of the parameter's containing method.
            }

            if (node.kind === SyntaxKind.PropertyDeclaration ||
                node.kind === SyntaxKind.MethodDeclaration ||
                node.kind === SyntaxKind.GetAccessor ||
                node.kind === SyntaxKind.SetAccessor) {
                // The `propertyKey` for a property or method decorator will be a
                // string literal type if the member name is an identifier, number, or string;
                // otherwise, if the member name is a computed property name it will
                // be either string or symbol.
                const element = <ClassElement>node;
                switch (element.name.kind) {
                    case SyntaxKind.Identifier:
                        return getLiteralType(idText(element.name));
                    case SyntaxKind.NumericLiteral:
                    case SyntaxKind.StringLiteral:
                        return getLiteralType(element.name.text);

                    case SyntaxKind.ComputedPropertyName:
                        const nameType = checkComputedPropertyName(element.name);
                        if (isTypeAssignableToKind(nameType, TypeFlags.ESSymbolLike)) {
                            return nameType;
                        }
                        else {
                            return stringType;
                        }

                    default:
                        Debug.fail("Unsupported property name.");
                        return unknownType;
                }
            }

            Debug.fail("Unsupported decorator target.");
            return unknownType;
        }

        /**
         * Returns the effective argument type for the third argument to a decorator.
         * If 'node' is a parameter, the effective argument type is the number type.
         * If 'node' is a method or accessor, the effective argument type is a
         *    `TypedPropertyDescriptor<T>` instantiated with the type of the member.
         * Class and property decorators do not have a third effective argument.
         */
        function getEffectiveDecoratorThirdArgumentType(node: Node) {
            // The third argument to a decorator is either its `descriptor` for a method decorator
            // or its `parameterIndex` for a parameter decorator
            if (node.kind === SyntaxKind.ClassDeclaration) {
                Debug.fail("Class decorators should not have a third synthetic argument.");
                return unknownType;
            }

            if (node.kind === SyntaxKind.Parameter) {
                // The `parameterIndex` for a parameter decorator is always a number
                return numberType;
            }

            if (node.kind === SyntaxKind.PropertyDeclaration) {
                Debug.fail("Property decorators should not have a third synthetic argument.");
                return unknownType;
            }

            if (node.kind === SyntaxKind.MethodDeclaration ||
                node.kind === SyntaxKind.GetAccessor ||
                node.kind === SyntaxKind.SetAccessor) {
                // The `descriptor` for a method decorator will be a `TypedPropertyDescriptor<T>`
                // for the type of the member.
                const propertyType = getTypeOfNode(node);
                return createTypedPropertyDescriptorType(propertyType);
            }

            Debug.fail("Unsupported decorator target.");
            return unknownType;
        }

        /**
         * Returns the effective argument type for the provided argument to a decorator.
         */
        function getEffectiveDecoratorArgumentType(node: Decorator, argIndex: number): Type {
            if (argIndex === 0) {
                return getEffectiveDecoratorFirstArgumentType(node.parent);
            }
            else if (argIndex === 1) {
                return getEffectiveDecoratorSecondArgumentType(node.parent);
            }
            else if (argIndex === 2) {
                return getEffectiveDecoratorThirdArgumentType(node.parent);
            }

            Debug.fail("Decorators should not have a fourth synthetic argument.");
            return unknownType;
        }

        /**
         * Gets the effective argument type for an argument in a call expression.
         */
        function getEffectiveArgumentType(node: CallLikeExpression, argIndex: number): Type {
            // Decorators provide special arguments, a tagged template expression provides
            // a special first argument, and string literals get string literal types
            // unless we're reporting errors
            if (node.kind === SyntaxKind.Decorator) {
                return getEffectiveDecoratorArgumentType(node, argIndex);
            }
            else if (argIndex === 0 && node.kind === SyntaxKind.TaggedTemplateExpression) {
                return getGlobalTemplateStringsArrayType();
            }

            // This is not a synthetic argument, so we return 'undefined'
            // to signal that the caller needs to check the argument.
            return undefined;
        }

        /**
         * Gets the effective argument expression for an argument in a call expression.
         */
        function getEffectiveArgument(node: CallLikeExpression, args: ReadonlyArray<Expression>, argIndex: number) {
            // For a decorator or the first argument of a tagged template expression we return undefined.
            if (node.kind === SyntaxKind.Decorator ||
                (argIndex === 0 && node.kind === SyntaxKind.TaggedTemplateExpression)) {
                return undefined;
            }

            return args[argIndex];
        }

        /**
         * Gets the error node to use when reporting errors for an effective argument.
         */
        function getEffectiveArgumentErrorNode(node: CallLikeExpression, argIndex: number, arg: Expression) {
            if (node.kind === SyntaxKind.Decorator) {
                // For a decorator, we use the expression of the decorator for error reporting.
                return node.expression;
            }
            else if (argIndex === 0 && node.kind === SyntaxKind.TaggedTemplateExpression) {
                // For a the first argument of a tagged template expression, we use the template of the tag for error reporting.
                return node.template;
            }
            else {
                return arg;
            }
        }

        function resolveCall(node: CallLikeExpression, signatures: Signature[], candidatesOutArray: Signature[], fallbackError?: DiagnosticMessage): Signature {
            const isTaggedTemplate = node.kind === SyntaxKind.TaggedTemplateExpression;
            const isDecorator = node.kind === SyntaxKind.Decorator;
            const isJsxOpeningOrSelfClosingElement = isJsxOpeningLikeElement(node);

            let typeArguments: NodeArray<TypeNode>;

            if (!isTaggedTemplate && !isDecorator && !isJsxOpeningOrSelfClosingElement) {
                typeArguments = (<CallExpression>node).typeArguments;

                // We already perform checking on the type arguments on the class declaration itself.
                if ((<CallExpression>node).expression.kind !== SyntaxKind.SuperKeyword) {
                    forEach(typeArguments, checkSourceElement);
                }
            }

            const candidates = candidatesOutArray || [];
            // reorderCandidates fills up the candidates array directly
            reorderCandidates(signatures, candidates);
            if (!candidates.length) {
                diagnostics.add(createDiagnosticForNode(node, Diagnostics.Call_target_does_not_contain_any_signatures));
                return resolveErrorCall(node);
            }

            const args = getEffectiveCallArguments(node);

            // The following applies to any value of 'excludeArgument[i]':
            //    - true:      the argument at 'i' is susceptible to a one-time permanent contextual typing.
            //    - undefined: the argument at 'i' is *not* susceptible to permanent contextual typing.
            //    - false:     the argument at 'i' *was* and *has been* permanently contextually typed.
            //
            // The idea is that we will perform type argument inference & assignability checking once
            // without using the susceptible parameters that are functions, and once more for each of those
            // parameters, contextually typing each as we go along.
            //
            // For a tagged template, then the first argument be 'undefined' if necessary
            // because it represents a TemplateStringsArray.
            //
            // For a decorator, no arguments are susceptible to contextual typing due to the fact
            // decorators are applied to a declaration by the emitter, and not to an expression.
            const isSingleNonGenericCandidate = candidates.length === 1 && !candidates[0].typeParameters;
            let excludeArgument: boolean[];
            let excludeCount = 0;
            if (!isDecorator && !isSingleNonGenericCandidate) {
                // We do not need to call `getEffectiveArgumentCount` here as it only
                // applies when calculating the number of arguments for a decorator.
                for (let i = isTaggedTemplate ? 1 : 0; i < args.length; i++) {
                    if (isContextSensitive(args[i])) {
                        if (!excludeArgument) {
                            excludeArgument = new Array(args.length);
                        }
                        excludeArgument[i] = true;
                        excludeCount++;
                    }
                }
            }

            // The following variables are captured and modified by calls to chooseOverload.
            // If overload resolution or type argument inference fails, we want to report the
            // best error possible. The best error is one which says that an argument was not
            // assignable to a parameter. This implies that everything else about the overload
            // was fine. So if there is any overload that is only incorrect because of an
            // argument, we will report an error on that one.
            //
            //     function foo(s: string): void;
            //     function foo(n: number): void; // Report argument error on this overload
            //     function foo(): void;
            //     foo(true);
            //
            // If none of the overloads even made it that far, there are two possibilities.
            // There was a problem with type arguments for some overload, in which case
            // report an error on that. Or none of the overloads even had correct arity,
            // in which case give an arity error.
            //
            //     function foo<T extends string>(x: T): void; // Report type argument error
            //     function foo(): void;
            //     foo<number>(0);
            //
            let candidateForArgumentError: Signature;
            let candidateForTypeArgumentError: Signature;
            let result: Signature;

            // If we are in signature help, a trailing comma indicates that we intend to provide another argument,
            // so we will only accept overloads with arity at least 1 higher than the current number of provided arguments.
            const signatureHelpTrailingComma =
                candidatesOutArray && node.kind === SyntaxKind.CallExpression && node.arguments.hasTrailingComma;

            // Section 4.12.1:
            // if the candidate list contains one or more signatures for which the type of each argument
            // expression is a subtype of each corresponding parameter type, the return type of the first
            // of those signatures becomes the return type of the function call.
            // Otherwise, the return type of the first signature in the candidate list becomes the return
            // type of the function call.
            //
            // Whether the call is an error is determined by assignability of the arguments. The subtype pass
            // is just important for choosing the best signature. So in the case where there is only one
            // signature, the subtype pass is useless. So skipping it is an optimization.
            if (candidates.length > 1) {
                result = chooseOverload(candidates, subtypeRelation, signatureHelpTrailingComma);
            }
            if (!result) {
                result = chooseOverload(candidates, assignableRelation, signatureHelpTrailingComma);
            }
            if