import arc, logging, os, pipes, time
from nagutils import UNKNOWN

log = logging.getLogger(__name__)

class ProxyInitError(EnvironmentError):
    def __init__(self, msg, details = None):
	EnvironmentError.__init__(self, msg)
	self.details = details

def require_voms_proxy(voms, key_path, cert_path, proxy_path,
		       min_proxy_lifetime = 7 * 3600):
    # To the benefit of any commands we'll run.
    os.putenv('X509_USER_PROXY', proxy_path)

    # Check for an existing proxy with sufficient time left.
    uc = arc.UserConfig()
    uc.ProxyPath(proxy_path)
    cred = arc.Credential(uc)
    t_now = time.time()
    t_E = cred.GetEndTime().GetTime()
    if t_E - t_now >= min_proxy_lifetime:
	log.debug('Proxy is still valid for %d h.'%((t_E - t_now)/3600))
	return

    # Make sure the directory exists.
    proxy_dir = os.path.dirname(proxy_path)
    if not os.path.exists(proxy_dir):
	try:
	    log.debug('Creading directory %s.'%proxy_dir)
	    os.makedirs(proxy_dir)
	except OSError, xc:
	    raise ProxyInitError('Cannot create parent directory %s for '
				 'storing X509 proxy: %s'%(proxy_dir, xc))

    # FIXME: Can this be done with the Python API?
    log.debug('Renewing proxy (%s--%s).'
	      % (cred.GetStartTime(), cred.GetEndTime()))
    if voms:
	cmd = 'arcproxy -S %s:all -K %s -C %s -P %s' \
		% (pipes.quote(voms or 'ops'),
		   pipes.quote(key_path),
		   pipes.quote(cert_path),
		   pipes.quote(proxy_path))
    else:
	cmd = 'arcproxy -K %s -C %s -P %s' \
		% (pipes.quote(key_path),
		   pipes.quote(cert_path),
		   pipes.quote(proxy_path))
    fh = os.popen(cmd, 'r')
    output = fh.read()
    err = fh.close()
    if err:
	raise ProxyInitError('Failed to initialize proxy: '
			     'arcproxy exited with %d.'%err,
			     details = 'Output from %s:\n%s'%(cmd, output))

class NagiosPluginVomsMixin(object):
    def __init__(self):
	ap = self.argparser.add_argument_group('VOMS Proxy Options')
	ap.add_argument('--user-proxy', dest = 'user_proxy',
		help = 'Path to a possibly pre-initialized VOMS proxy.')
	ap.add_argument('--user-cert', dest = 'user_cert',
		help = 'Certificate to use for obtaining VOMS proxy.')
	ap.add_argument('--user-key', dest = 'user_key',
		help = 'Certificate key to use for obtaining VOMS proxy.')
	ap.add_argument('--voms', dest = 'voms',
		help = 'VOMS server for Proxy initialization.')
	ap.add_argument('--min-proxy-lifetime', dest = 'min_proxy_lifetime',
		# The default should be strictly larger than the time before
		# jobs are cancelled.  We need at least the time between
		# monitor runs plus grace time.
		default = 7 * 3600,
		help = 'The minimum lifetime in seconds of the proxy '
		       'certificate before it is renewed.')

    @property
    def voms(self):
	if self.opts.voms:
	    return self.opts.voms
	if self.config.has_option('gridproxy', 'default_voms'):
	    return self.config.get('gridproxy', 'default_voms')
	return None

    def voms_suffixed(self, name):
	if self.voms:
	    return name + '-' + self.voms
	else:
	    return name

    def require_voms_proxy(self):
	key_path = self.opts.user_key
	cert_path = self.opts.user_cert
	proxy_path = self.opts.user_proxy
	voms = self.voms
	for section in ['gridproxy.%s'%voms, 'gridproxy']:
	    if not key_path and self.config.has_option(section, 'user_key'):
		key_path = self.config.get(section, 'user_key')
	    if not cert_path and self.config.has_option(section, 'user_cert'):
		cert_path = self.config.get(section, 'user_cert')
	    if not proxy_path and self.config.has_option(section, 'user_proxy'):
		proxy_path = self.config.get(section, 'user_proxy')

	# If no user key or certificate is given, use an externally
	# initialized proxy.
	if not key_path and not cert_path:
	    if proxy_path:
		os.putenv('X509_USER_PROXY', proxy_path)
	    return os.getenv('X509_USER_PROXY')

	# Otherwise, initialize a proxy.
	if not proxy_path:
	    proxy_path = os.path.join(self.tmpdir(), 'proxy-%s.pem'%voms)
	try:
	    require_voms_proxy(
		    voms = voms,
		    key_path = key_path or '/etc/grid-monitoring/userkey.pem',
		    cert_path = cert_path or '/etc/grid-monitoring/usercert.pem',
		    proxy_path = proxy_path,
		    min_proxy_lifetime = self.opts.min_proxy_lifetime)
	except ProxyInitError, xc:
	    if xc.details:
		self.log.error(xc.details)
	    self.nagios_exit(UNKNOWN, str(xc))
	return proxy_path
