﻿// Copyright (c) 2010-2013 AlphaSierraPapa for the SharpDevelop Team
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this
// software and associated documentation files (the "Software"), to deal in the Software
// without restriction, including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
// to whom the Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all copies or
// substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

using System;
using ICSharpCode.NRefactory.PatternMatching;

namespace ICSharpCode.NRefactory.CSharp.Refactoring
{
	/// <summary>
	/// Helper class for constructing pattern ASTs.
	/// </summary>
	public class PatternHelper
	{
		/// <summary>
		/// Produces a choice pattern for <c>expr1 op expr2</c> or <c>expr2 op expr1</c>.
		/// </summary>
		public static Expression CommutativeOperator(Expression expr1, BinaryOperatorType op, Expression expr2)
		{
			return new Choice {
				new BinaryOperatorExpression(expr1, op, expr2),
				new BinaryOperatorExpression(expr2.Clone(), op, expr1.Clone())
			};
		}
		
		/// <summary>
		/// Optionally allows parentheses around the given expression.
		/// </summary>
		public static Expression OptionalParentheses(Expression expr)
		{
			return new OptionalParenthesesPattern(expr);
		}
		
		sealed class OptionalParenthesesPattern : Pattern
		{
			readonly INode child;
			
			public OptionalParenthesesPattern(INode child)
			{
				this.child = child;
			}
			
			public override bool DoMatch(INode other, Match match)
			{
				INode unpacked = ParenthesizedExpression.UnpackParenthesizedExpression(other as Expression);
				return child.DoMatch(unpacked, match);
			}
		}
	}
}
