# -*- coding: utf-8 -*-

"""
Copyright(C) 2007-2008 INL
Written by Romain Bignon <romain AT inl.fr>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, version 3 of the License.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

$Id: pages.py 12119 2008-01-10 15:24:21Z gryzor $
"""

try:
    from twisted.web import http
except ImportError:
    from twisted.protocols import http

from nevow import rend, tags, static, loaders
from graph import GraphImages
from nevow import inevow, guard
from tools import getBacktrace
import os
from nevow.i18n import _, render as i18nrender
from sessions import getSession, getUserSession, authentification

mypath = os.path.dirname(__file__) + os.sep
NULOG_USER_KEY = 'Nulog_User'

class BasePage(rend.Page):

    render_i18n = i18nrender()

    children = dict(css = static.File(mypath + "css/"),
                    img = static.File(mypath + "img/"),
                    js  = static.File(mypath + 'js/'),
                    doc = static.File(mypath + 'doc/'),
                    graphs = GraphImages())

    addSlash = True

    def __init__(self, conf, langconfig, AvatarId=None, content=None):

        rend.Page.__init__(self)
        self.langconfig = langconfig
        self.conf = conf
        self.content = content
        self.AvatarId = AvatarId

        # Remember the directory to find lang files.
        # WARNING: call this function AFTER rend.Page.__init__()
        self.remember(langconfig, inevow.II18NConfig)

    def renderHTTP(self, ctx):

        request = inevow.IRequest(ctx)

        if request.getHeader(NULOG_USER_KEY):
            self.AvatarId = request.getHeader(NULOG_USER_KEY)

        if self.AvatarId and not getSession(ctx).isLogged():
            # If it gives an username and in my session there isn't
            # any username, it is because this is the first login.
            # So we get the User Session stored in files.
            getUserSession(ctx, self.AvatarId)

        # We're overriding renderHTTP to look for a 'lang' query parameter
        # without cluttering up the messages renderer, below.
        # If 'lang' is present then we "force" the translation language. This
        # simulates how user preferences from the session might be used to
        # override the browser's language settings.
        lang = ctx.arg('lang')
        session = inevow.ISession(ctx)

        if lang is not None:
            ctx.remember([lang], inevow.ILanguages)
            session.setComponent(inevow.ILanguages, lang)

            # We remove 'lang' from args
            ctx.locate(inevow.IRequest).args.pop('lang')
        else:
            lang = session.getComponent(inevow.ILanguages)
            if lang:
                ctx.remember([lang], inevow.ILanguages)

        # Let the base class handle it, really.
        return rend.Page.renderHTTP(self, ctx)

class The404Page(BasePage):

    docFactory = loaders.xmlfile(mypath + 'xml/404.xml')

    addSlash = False

    def render_title(self, ctx, data):

         return ctx.tag.clear()[tags.a(href=self.conf.get('Links', 'url'))[self.conf.get('Links', 'maintitle')]]

    def render_head(self, ctx, data):

        ctx.fillSlots('PATH', self.conf.get('Links', 'url'))
        return ctx.tag

    def renderHTTP_notFound(self, ctx):
        self.is404 = True
        self.addSlash = False
        return self.renderHTTP(ctx)

    def render_content(self, ctx, data):
        referer = inevow.IRequest(ctx).getHeader('referer')
        render = [tags.h2[_('File not found')]]
        if referer:
            render += [ctx.tag[tags.p[tags.a(href=referer)[_('Go back')]]]]

        return render


class LoginPage(BasePage):

    docFactory = loaders.xmlfile(mypath + 'xml/nulog.xml')

    addSlash = False

    def render_title(self, ctx, data):

        return ctx.tag.clear()[tags.a(href=self.conf.get('Links', 'url'))[self.conf.get('Links', 'maintitle')]]

    def render_leftmenu(self, ctx, data):
        return ''
    def render_search(self, ctx, data):
        return ''

    def render_langs(self, ctx, data):
        return ''

    def render_content(self, ctx, data):
        data = []
        args = ctx.locate(inevow.IRequest).args
        args = dict((i, args[i][0]) for i in args)

        data += [tags.h2[_('Login')]]

        if args.has_key('login-failure'):
            data += [tags.p(_class='warning')[args['login-failure']]]
        data += [tags.form(action=self.conf.get('Links', 'url') + guard.LOGIN_AVATAR, method='post')[
                        tags.table[
                            tags.tr[
                                tags.td[_("Username:")],
                                tags.td[ tags.input(type='text',name='username') ],
                            ],
                            tags.tr[
                                tags.td[_("Password:")],
                                tags.td[ tags.input(type='password',name='password') ],
                            ]
                        ],
                        tags.input(type='submit'),
                        tags.p,
                    ]
                ]

        return ctx.tag.clear()[tags.div(_class='login')[data]]
