/* This file is part of the Palabos library.
 *
 * Copyright (C) 2011-2015 FlowKit Sarl
 * Route d'Oron 2
 * 1010 Lausanne, Switzerland
 * E-mail contact: contact@flowkit.com
 *
 * The most recent release of Palabos can be downloaded at 
 * <http://www.palabos.org/>
 *
 * The library Palabos is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* Orestis Malaspinas contributed this code.
 */

/** \file
 * A helper for initialising 2D boundaries -- generic implementation.
 */
#ifndef ZOU_HE_BOUNDARY_2D_HH
#define ZOU_HE_BOUNDARY_2D_HH

#include "boundaryCondition/zouHeBoundary2D.h"
#include "boundaryCondition/zouHeDynamics.h"
#include "boundaryCondition/zouHeDynamics.hh"
#include "boundaryCondition/regularizedBoundaryDynamics2D.h"
#include "boundaryCondition/boundaryInstantiator2D.h"
#include "boundaryCondition/wrappedLocalBoundaryProcessor2D.h"

namespace plb {

template<typename T, template<typename U> class Descriptor>
class ZouHeBoundaryManager2D {
public:
    template<int direction, int orientation> static BoundaryCompositeDynamics<T,Descriptor>*
        getVelocityBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int direction, int orientation> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getVelocityBoundaryFunctional();

    template<int direction, int orientation> static BoundaryCompositeDynamics<T,Descriptor>*
        getPressureBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int direction, int orientation> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getPressureBoundaryFunctional();

    template<int xNormal, int yNormal> static BoundaryCompositeDynamics<T,Descriptor>*
        getExternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int xNormal, int yNormal> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getExternalVelocityCornerFunctional();

    template<int xNormal, int yNormal> static BoundaryCompositeDynamics<T,Descriptor>*
        getInternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int xNormal, int yNormal> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getInternalVelocityCornerFunctional();
};

template<typename T, template<typename U> class Descriptor>
class WrappedZouHeBoundaryManager2D {
public:
    template<int direction, int orientation> static BoundaryCompositeDynamics<T,Descriptor>*
        getVelocityBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int direction, int orientation> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getVelocityBoundaryFunctional();

    template<int direction, int orientation> static BoundaryCompositeDynamics<T,Descriptor>*
        getPressureBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int direction, int orientation> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getPressureBoundaryFunctional();

    template<int xNormal, int yNormal> static BoundaryCompositeDynamics<T,Descriptor>*
        getExternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int xNormal, int yNormal> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getExternalVelocityCornerFunctional();

    template<int xNormal, int yNormal> static BoundaryCompositeDynamics<T,Descriptor>*
        getInternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics);
    template<int xNormal, int yNormal> static BoxProcessingFunctional2D_L<T,Descriptor>*
        getInternalVelocityCornerFunctional();
};

////////// ZouHeBoundaryManager2D /////////////////////////////////////////

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoundaryCompositeDynamics<T,Descriptor>* ZouHeBoundaryManager2D<T,Descriptor>::
    getVelocityBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    return new ZouHeVelocityDynamics<T,Descriptor, direction, orientation>(baseDynamics);
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoxProcessingFunctional2D_L<T,Descriptor>*
    ZouHeBoundaryManager2D<T,Descriptor>::
        getVelocityBoundaryFunctional()
{
    return 0;
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoundaryCompositeDynamics<T,Descriptor>* ZouHeBoundaryManager2D<T,Descriptor>::
    getPressureBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    return new ZouHePressureDynamics<T,Descriptor, direction, orientation>(baseDynamics);
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoxProcessingFunctional2D_L<T,Descriptor>*
    ZouHeBoundaryManager2D<T,Descriptor>::
        getPressureBoundaryFunctional()
{
    return 0;
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoundaryCompositeDynamics<T,Descriptor>* ZouHeBoundaryManager2D<T,Descriptor>::
    getExternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    return new StoreVelocityDynamics<T,Descriptor>(baseDynamics);
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoxProcessingFunctional2D_L<T,Descriptor>*
    ZouHeBoundaryManager2D<T,Descriptor>::
        getExternalVelocityCornerFunctional()
{
    return new OuterVelocityCornerFunctional2D<T,Descriptor, xNormal,yNormal>();
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoundaryCompositeDynamics<T,Descriptor>* ZouHeBoundaryManager2D<T,Descriptor>::
    getInternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    return new RegularizedVelocityInnerCornerDynamics2D<T,Descriptor, xNormal, yNormal>(baseDynamics);
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoxProcessingFunctional2D_L<T,Descriptor>*
    ZouHeBoundaryManager2D<T,Descriptor>::getInternalVelocityCornerFunctional()
{
    return 0;
}


////////// WrappedZouHeBoundaryManager2D /////////////////////////////////////////

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoundaryCompositeDynamics<T,Descriptor>* WrappedZouHeBoundaryManager2D<T,Descriptor>::
    getVelocityBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    bool automaticPrepareCollision = false;
    return new ZouHeVelocityDynamics
                   <T,Descriptor, direction, orientation>(baseDynamics, automaticPrepareCollision);
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoxProcessingFunctional2D_L<T,Descriptor>*
    WrappedZouHeBoundaryManager2D<T,Descriptor>::
        getVelocityBoundaryFunctional()
{
    return new WrappedLocalBoundaryFunctional2D<T,Descriptor>();
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoundaryCompositeDynamics<T,Descriptor>* WrappedZouHeBoundaryManager2D<T,Descriptor>::
    getPressureBoundaryDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    bool automaticPrepareCollision = false;
    return new ZouHePressureDynamics
                   <T,Descriptor, direction, orientation>(baseDynamics,automaticPrepareCollision);
}

template<typename T, template<typename U> class Descriptor>
template<int direction, int orientation>
BoxProcessingFunctional2D_L<T,Descriptor>*
    WrappedZouHeBoundaryManager2D<T,Descriptor>::
        getPressureBoundaryFunctional()
{
    return new WrappedLocalBoundaryFunctional2D<T,Descriptor>();
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoundaryCompositeDynamics<T,Descriptor>* WrappedZouHeBoundaryManager2D<T,Descriptor>::
    getExternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    return new StoreVelocityDynamics<T,Descriptor>(baseDynamics);
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoxProcessingFunctional2D_L<T,Descriptor>*
    WrappedZouHeBoundaryManager2D<T,Descriptor>::
        getExternalVelocityCornerFunctional()
{
    return new OuterVelocityCornerFunctional2D<T,Descriptor, xNormal,yNormal>();
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoundaryCompositeDynamics<T,Descriptor>* WrappedZouHeBoundaryManager2D<T,Descriptor>::
    getInternalVelocityCornerDynamics(Dynamics<T,Descriptor>* baseDynamics)
{
    bool automaticPrepareCollision = false;
    return new RegularizedVelocityInnerCornerDynamics2D
                   <T,Descriptor, xNormal, yNormal>(baseDynamics, automaticPrepareCollision);
}

template<typename T, template<typename U> class Descriptor>
template<int xNormal, int yNormal>
BoxProcessingFunctional2D_L<T,Descriptor>*
    WrappedZouHeBoundaryManager2D<T,Descriptor>::getInternalVelocityCornerFunctional()
{
    return new WrappedLocalBoundaryFunctional2D<T,Descriptor>();
}


////////// Factory function //////////////////////////////////////////////////

template<typename T, template<typename U> class Descriptor>
OnLatticeBoundaryCondition2D<T,Descriptor>* createZouHeBoundaryCondition2D() {
    return new BoundaryConditionInstantiator2D <
                   T, Descriptor,
                   WrappedZouHeBoundaryManager2D<T,Descriptor> > ();
}

template<typename T, template<typename U> class Descriptor>
OnLatticeBoundaryCondition2D<T,Descriptor>* createDynamicsBasedZouHeBoundaryCondition2D() {
    return new BoundaryConditionInstantiator2D <
                   T, Descriptor,
                   ZouHeBoundaryManager2D<T,Descriptor> > ();
}

}  // namespace plb

#endif  // ZOU_HE_BOUNDARY_2D_HH
