/*
 * store_cat.cc
 * Copyright (C) Martin Singer 2009-2011 <m_power3@users.sourceforge.net>
 * 
 * pdfchain is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * pdfchain is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "store_cat.h"


/*** List Store : Cat *********************************************************/

// Constructor
cLStore_Cat::cLStore_Cat( Gtk::Window& ref_window )
:
	mFCDialog( ref_window , _("Concatenate - Add PDF files") , Gtk::FILE_CHOOSER_ACTION_OPEN , true , Pdfchain::Id::FILE_TYPE_PDF ),	// parent_window , title , action , select_multiple , file_type
	pTMCRecord( NULL ),
	mMDialog_Password( ref_window )
{
	pTMCRecord = new cTMCRecord_Cat;
	set_column_types( *pTMCRecord );
}


// Destructor
cLStore_Cat::~cLStore_Cat()
{
#ifdef PDFCHAIN_TEST
	std::cout << std::endl << "cLStore_Cat::~cLStore_Cat()";	//TEST
#endif

	if( pTMCRecord != NULL ) { delete pTMCRecord;	pTMCRecord = NULL; }
}


// Method (public) : add new rows (files)
void
cLStore_Cat::addRows ( Glib::RefPtr<Gtk::TreeSelection> ref_tree_selections )
{

	switch ( mFCDialog.run() ) {

		case Gtk::RESPONSE_OK: {
			mFCDialog.hide();

			std::string							str_sourcepath		= "";
			std::vector<std::string>			vect_sourcepaths	= mFCDialog.get_filenames();
			std::vector<std::string>::iterator	iter_sourcepaths	= vect_sourcepaths.begin();
			Gtk::TreeModel::iterator			iter_selections;
			Gtk::TreeModel::iterator			iter_org;
			Gtk::TreeModel::iterator			iter_new;
				
			// Get begin of TreeView selection and get TreeModel iterator of selection
			if ( 0 != ref_tree_selections->count_selected_rows() ) {
	
				std::vector<Gtk::TreeModel::Path>	list_selections = ref_tree_selections->get_selected_rows();
				Gtk::TreeModel::Path				path_selections = *( list_selections.begin() );

				iter_selections = get_iter( path_selections );
			}

			// Add sourcefiles at ListStore
			while ( vect_sourcepaths.end() != iter_sourcepaths ) {
				
				str_sourcepath = *iter_sourcepaths;

				if ( iter_selections ) {
					iter_new = insert_after( iter_selections );
					iter_selections = iter_new;
				}
				else {
					iter_new = append();
				}

				(*iter_new)[ pTMCRecord->mTMColumn_Add					] = TRUE;
				(*iter_new)[ pTMCRecord->mTMColumn_Pages				] = mFCDialog.getStr_Pages();
				(*iter_new)[ pTMCRecord->mTMColumn_EvenOdd				] = mFCDialog.getStr_EvenOdd();
				(*iter_new)[ pTMCRecord->mTMColumn_Rotation				] = mFCDialog.getStr_Rotation();
				(*iter_new)[ pTMCRecord->mTMColumn_SourceFile			] = Glib::filename_to_utf8( Pdfchain::extract_filename_from_path( str_sourcepath ) );
				(*iter_new)[ pTMCRecord->mTMColumn_SourcePath			] = str_sourcepath;
				(*iter_new)[ pTMCRecord->mTMColumn_Tooltip				] = _("<b>Path: </b>") + Glib::Markup::escape_text( Glib::filename_to_utf8( str_sourcepath ) );
				(*iter_new)[ pTMCRecord->mTMColumn_Password				] = mFCDialog.getStr_Password();
				(*iter_new)[ pTMCRecord->mTMColumn_PageNumbers			] = 0;
				(*iter_new)[ pTMCRecord->mTMColumn_Cmd_Handle			] = Pdfchain::Cat::HANDLE_DEFAULT;
				(*iter_new)[ pTMCRecord->mTMColumn_Cmd_EvenOdd			] = mFCDialog.getCmd_EvenOdd();
				(*iter_new)[ pTMCRecord->mTMColumn_Cmd_Rotation			] = mFCDialog.getCmd_Rotation();
				(*iter_new)[ pTMCRecord->mTMColumn_Color_Pages			] = Pdfchain::Color::RGBA_VALID;
				(*iter_new)[ pTMCRecord->mTMColumn_Color_PageNumbers	] = Pdfchain::Color::RGBA_VALID;

				// Check for same file alredy exists at list ...
				mMDialog_Password.init( iter_new->get_value( pTMCRecord->mTMColumn_SourcePath) , iter_new->get_value( pTMCRecord->mTMColumn_Password ) );

				for ( iter_org = children().begin() ; children().end() != iter_org ; ++iter_org ) {
					if ( iter_org->get_value( pTMCRecord->mTMColumn_SourcePath ) == str_sourcepath ) {

						// If two entries with same sourcepaths have different passwords, ask for the right one ...
						if ( (*iter_org)[pTMCRecord->mTMColumn_Password] != (*iter_new)[pTMCRecord->mTMColumn_Password] ) {
							(*iter_org)[pTMCRecord->mTMColumn_Password] = mMDialog_Password.getCorrectPassword( (*iter_org)[pTMCRecord->mTMColumn_Password] );
							(*iter_new)[pTMCRecord->mTMColumn_Password] = mMDialog_Password.getCorrectPassword( (*iter_org)[pTMCRecord->mTMColumn_Password] );
						}

						// Entries with same sourcepaths have same page numbers
						(*iter_new)[pTMCRecord->mTMColumn_PageNumbers] = iter_org->get_value( pTMCRecord->mTMColumn_PageNumbers );
					}
				}

				// Validate page numbners
				if ( 0 == (*iter_new)[pTMCRecord->mTMColumn_PageNumbers] ) {
					iter_new->set_value( pTMCRecord->mTMColumn_PageNumbers , Pdfchain::count_page_numbers( iter_new->get_value( pTMCRecord->mTMColumn_SourcePath ) ) );

					if( 0 == (*iter_new)[pTMCRecord->mTMColumn_PageNumbers] ) {
						iter_new->set_value( pTMCRecord->mTMColumn_Color_PageNumbers , Pdfchain::Color::RGBA_WARNING );
						(*iter_new)[pTMCRecord->mTMColumn_Add] = FALSE;
					}
				}

				validatePages( iter_new );	// Validate the Entry Pages
				iter_sourcepaths++;
			}
			break;
		}
		default:
			mFCDialog.hide();
			break;
	}

	indicateHandles();
	return;
}


// Method (public) : remove selected rows
void
cLStore_Cat::removeRows ( Glib::RefPtr<Gtk::TreeSelection> ref_tree_selections )
{
	std::vector<Gtk::TreeModel::Path>	list_selections = ref_tree_selections->get_selected_rows();
	Gtk::TreeModel::iterator			iter_next;

	if ( list_selections.size() ) {

		// Define row to select as soon as selected rows are removed
		iter_next = get_iter( list_selections.back() );
		if ( ++iter_next == children().end() ) {
			if ( children().begin() != get_iter( list_selections.front() ) ) {
				iter_next = get_iter( list_selections.front() );
				iter_next--;
			}
		}

		// Remove selected rows
		while ( list_selections.size() ) {
			erase( get_iter( list_selections.back() ) );
			list_selections.erase( list_selections.end() );
		}

		if ( iter_next ) ref_tree_selections->select( iter_next );
	}

	indicateHandles();
	return;
}


// Method (public) : copy selected Rows
void
cLStore_Cat::copyRows ( Glib::RefPtr<Gtk::TreeSelection> ref_tree_selections )
{
	std::vector<Gtk::TreeModel::Path> list_selections = ref_tree_selections->get_selected_rows();

	if ( list_selections.size() ) {
		Gtk::TreeModel::iterator iter_pos = get_iter( list_selections.back() );
		Gtk::TreeModel::iterator iter_org;
		Gtk::TreeModel::iterator iter_cpy;

		while ( list_selections.size() ) {
			iter_org = get_iter( list_selections.back() );
			iter_cpy = insert_after( iter_pos );

			copyRow( iter_cpy , iter_org );
			list_selections.erase( list_selections.end() );
		}
	}
	
	return;
}


// Method (public) : move selected rows up
void
cLStore_Cat::moveRowsUp ( Glib::RefPtr<Gtk::TreeSelection> ref_tree_selections )
{
	std::vector<Gtk::TreeModel::Path> list_selections = ref_tree_selections->get_selected_rows();

	if ( list_selections.size() ) {
		Gtk::TreeModel::iterator iter_selections;
		Gtk::TreeModel::iterator iter_upper;

		// Move every row only if first selected row is not the first row at the ListStore
		if ( children().begin() !=  get_iter( list_selections.front() ) ) {

			while ( list_selections.size() ) {
				iter_upper = iter_selections = get_iter( list_selections.front() );
				
				iter_swap( iter_selections , --iter_upper );
				list_selections.erase( list_selections.begin() );
			}
		}
	}

	return;
}


// Method (public) : move selected rows down
void
cLStore_Cat::moveRowsDown ( Glib::RefPtr<Gtk::TreeSelection> ref_tree_selections )
{
	std::vector<Gtk::TreeModel::Path> list_selections = ref_tree_selections->get_selected_rows();

	if ( list_selections.size() ) {
		Gtk::TreeModel::iterator iter_selections;
		Gtk::TreeModel::iterator iter_lower;

		// Move every row only if last selected row is not the last row at the ListStore
		if ( --children().end() != get_iter( list_selections.back() ) ) {
			
			while ( list_selections.size() ) {
				iter_lower = iter_selections = get_iter( list_selections.back() );

				iter_swap( iter_selections , ++iter_lower );
				list_selections.erase( list_selections.end() );
			}
		}
	}
	
	return;
}


// Method (protected) : indicate handles
void
cLStore_Cat::indicateHandles()
{
	guchar							handle		= Pdfchain::Cat::HANDLE_FIRST;
	Glib::ustring					str_path_A	= "";
	Glib::ustring					str_path_B	= "";
	Gtk::TreeModel::iterator		iter_A;
	Gtk::TreeModel::iterator		iter_B;

	// Clear handles (set default handle for every file)
	for ( iter_A = children().begin() ; children().end() != iter_A ; ++iter_A )
		(*iter_A)[pTMCRecord->mTMColumn_Cmd_Handle] = Pdfchain::Cat::HANDLE_DEFAULT;

	// Set Handles ...
	for ( iter_B = children().begin() , handle = Pdfchain::Cat::HANDLE_FIRST ; children().end() != iter_B ; ++iter_B ) {

		// ... only if CheckBox is enabled
		if ( TRUE == (*iter_B)[pTMCRecord->mTMColumn_Add] ) {
			str_path_B = (*iter_B)[pTMCRecord->mTMColumn_SourcePath];

			// Set new handle
			if ( Pdfchain::Cat::HANDLE_DEFAULT == (*iter_B)[pTMCRecord->mTMColumn_Cmd_Handle] )
				(*iter_B)[pTMCRecord->mTMColumn_Cmd_Handle] = handle++;

			// Set reiterarating Handle ...
			for ( iter_A = iter_B ; children().end() != iter_A ; ++iter_A ) {

				// ... only if CheckBox is enabled
				if ( TRUE == (*iter_A)[pTMCRecord->mTMColumn_Add] ) {
					str_path_A = (*iter_A)[pTMCRecord->mTMColumn_SourcePath];

					if( ! str_path_A.compare( str_path_B ) ) // Glib::ustring::compare() returns 0 if strings compare
						(*iter_A)[pTMCRecord->mTMColumn_Cmd_Handle] = iter_B->get_value( pTMCRecord->mTMColumn_Cmd_Handle );
				}
			}
		}
	}

	// Check number of handles
	if ( Pdfchain::Cat::HANDLE_FINAL < --handle ) {
		std::cerr << std::endl << "INFO: Only 26 different PDF files able to processed!";

		// If handle > 'Z' disable file
		for ( iter_A = children().begin() ; children().end() != iter_A ; ++iter_A )
			if ( Pdfchain::Cat::HANDLE_FINAL < (*iter_A)[pTMCRecord->mTMColumn_Cmd_Handle] )
				(*iter_A)[pTMCRecord->mTMColumn_Add] = FALSE;

		// Set handle to 'Z'
		handle = Pdfchain::Cat::HANDLE_FINAL;
	}

	// Count active Files
	vFileNumber = handle - Pdfchain::Cat::HANDLE_FIRST + 1;
	//std::cout << std::endl << "vFileNumber: " << vFileNumber;	//TEST

	// Emit signal PageNumber changed for ComboBox "OutputID"
	mSignal_FileNumber_changed.emit( vFileNumber );
	
	return;
}


// Method (protected) : validate pages (the entry)
gboolean
cLStore_Cat::validatePages( const Gtk::TreeModel::iterator& iter )
{
	Glib::ustring	str_pages		= iter->get_value( pTMCRecord->mTMColumn_Pages );
	guint			page_numbers	= iter->get_value( pTMCRecord->mTMColumn_PageNumbers );

	guint						number		= 0;
	Glib::ustring::size_type	pos_A		= 0;
	Glib::ustring::size_type	pos_B		= 0;
	Glib::ustring::size_type	pos			= 0;
	Glib::ustring				str_sub		= "";

	// Extend string at begin and end with ' '
	str_pages = " " + str_pages + " ";

	// Check for invalid sub string " -"
	if ( str_pages.find( " -" ) != Glib::ustring::npos ) {
		iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
		return FALSE;
	}

	// Check for invalid sub string "- "
	if ( str_pages.find( "- " ) != Glib::ustring::npos ) {
		iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
		return FALSE;
	}

	// Search for ' ' separator to ...
	for ( pos_A = 0 ; Glib::ustring::npos != ( pos_B = str_pages.find( ' ' , pos_A ) ) ; pos_A = pos_B + 1 ) {

		// ... split into sub strings ...
		str_sub = str_pages.substr( pos_A , pos_B - pos_A );

		// ... and validate sub string if it is longer than zero
		if ( 0 != str_sub.length() ) {

			// Count '-' at sub string
			for ( number = 0 , pos = 0 ; Glib::ustring::npos != ( pos = str_sub.find( '-' , pos ) ) ; number++ , pos++ );

			// Max one '-' is per section allowed
			if ( 1 < number ) {
				iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
				return FALSE;
			}
		}
	}

	// Check and erase sub string "end"
	while ( Glib::ustring::npos != ( pos = str_pages.find( "end" ) ) ) {

		// Check character bevore "end" (must be ' ' or '-')
		if ( 0 < pos ) {
			str_sub = str_pages.substr( pos-1 , 1 );

			if ( " " != str_sub && "-" != str_sub ) {
				iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
				return FALSE;
			}
		}

		// Check chracter after "end" (must be ' ' or '-')
		if ( Glib::ustring::npos != pos+3 ) {
			str_sub = str_pages.substr( pos+3 , 1 );

			if ( " " != str_sub && "-" != str_sub ) {
				iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
				return FALSE;
			}
		}

		// Erase sub string "end"
		str_pages.erase( pos , 3 );
	}

	// Check for valid signs: "01234567890 -"
	for ( pos = 0 ; str_pages.length() > pos ; pos++ ){

		if ( ! ( Glib::ustring::npos != Pdfchain::Cat::ALLOWED_SIGNS.find( str_pages.substr( pos , 1 ) ) ) ) {
			iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
			return FALSE;
		}
	}

	// Replace sub string "-" with " "
	while ( Glib::ustring::npos != ( pos = str_pages.find( '-' ) ) )
		str_pages.replace( pos , 1 , 1 , ' ' );

	// Cleanup with doubled " "
	while ( Glib::ustring::npos != ( pos = str_pages.find( "  " ) ) )
		str_pages.replace( pos , 2 , " " );

	// Check page numbers (begin at pos=1 , pos=0 is ' ')
	for ( pos = 1 , number = 0 ; str_pages.length() > pos ; pos++ ) {

		// If digit --> count
		if ( Glib::ustring::npos != Pdfchain::Cat::ALLOWED_DIGITS.find( str_pages.substr( pos , 1 ) ) ) {
			number *= 10;	// FIXME: variable could overflow
			number += ( str_pages.c_str() )[pos] - '0';
		}

		// If separator ' ' ---> check
		else {

			// Check for invalid page number '0'
			if( 0 == number ) {
				iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
				return FALSE;
			}

			// Check for invalid big page number (If counted page_number is 0, file is not countable, but accepted by user)
			if( ( 0 < page_numbers ) && ( number > page_numbers ) ) {
				iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_INVALID );
				return FALSE;
			}

			number = 0;
		}
	}

	// Reset color to valid
	iter->set_value( pTMCRecord->mTMColumn_Color_Pages , Pdfchain::Color::RGBA_VALID );
	return TRUE;
}


// Method (protected) : copy a row
void
cLStore_Cat::copyRow( const Gtk::TreeModel::iterator& iter_copy , const Gtk::TreeModel::iterator& iter_org )
{
	iter_copy->set_value( pTMCRecord->mTMColumn_Add					, iter_org->get_value( pTMCRecord->mTMColumn_Add				) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Pages				, iter_org->get_value( pTMCRecord->mTMColumn_Pages				) );
	iter_copy->set_value( pTMCRecord->mTMColumn_EvenOdd				, iter_org->get_value( pTMCRecord->mTMColumn_EvenOdd			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Rotation			, iter_org->get_value( pTMCRecord->mTMColumn_Rotation			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_SourceFile			, iter_org->get_value( pTMCRecord->mTMColumn_SourceFile			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_SourcePath			, iter_org->get_value( pTMCRecord->mTMColumn_SourcePath			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Tooltip				, iter_org->get_value( pTMCRecord->mTMColumn_Tooltip			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Password			, iter_org->get_value( pTMCRecord->mTMColumn_Password			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_PageNumbers			, iter_org->get_value( pTMCRecord->mTMColumn_PageNumbers		) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Cmd_Handle			, iter_org->get_value( pTMCRecord->mTMColumn_Cmd_Handle			) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Cmd_EvenOdd			, iter_org->get_value( pTMCRecord->mTMColumn_Cmd_EvenOdd		) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Cmd_Rotation		, iter_org->get_value( pTMCRecord->mTMColumn_Cmd_Rotation		) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Color_Pages			, iter_org->get_value( pTMCRecord->mTMColumn_Color_Pages		) );
	iter_copy->set_value( pTMCRecord->mTMColumn_Color_PageNumbers	, iter_org->get_value( pTMCRecord->mTMColumn_Color_PageNumbers	) );
	
	return;
}


// Method (public) : toggle Add
void
cLStore_Cat::toggleAdd( const Gtk::TreePath& treepath )
{
	Gtk::TreeModel::iterator iter = get_iter( treepath );
	(*iter)[pTMCRecord->mTMColumn_Add] = ! (*iter)[pTMCRecord->mTMColumn_Add];
	indicateHandles();
	return;
}


// Method (public) : edit Pages
void
cLStore_Cat::editPages( const Gtk::TreePath& treepath , const Glib::ustring& str_pages )
{
	Gtk::TreeModel::iterator iter = get_iter( treepath );
	iter->set_value( pTMCRecord->mTMColumn_Pages , str_pages );
	validatePages( iter );
	return;
}


// Method (public) : change EvenOdd
void
cLStore_Cat::changeEvenOdd( const Gtk::TreePath& treepath , const Glib::ustring& str_label , const std::string& str_command )
{
	Gtk::TreeModel::iterator iter = get_iter( treepath );
	iter->set_value( pTMCRecord->mTMColumn_EvenOdd , str_label );
	iter->set_value( pTMCRecord->mTMColumn_Cmd_EvenOdd , str_command );
	return;
}


// Method (public) : change Rotation
void
cLStore_Cat::changeRotation( const Gtk::TreePath& treepath , const Glib::ustring& str_label , const std::string& str_command )
{
	Gtk::TreeModel::iterator iter = get_iter( treepath );
	iter->set_value( pTMCRecord->mTMColumn_Rotation , str_label );
	iter->set_value( pTMCRecord->mTMColumn_Cmd_Rotation , str_command );
	return;
}


// Method (public) : edit Password
void
cLStore_Cat::editPassword( const Gtk::TreePath& treepath , const Glib::ustring& str_password )
{
	Gtk::TreeModel::iterator iter = get_iter( treepath );
	Gtk::TreeModel::iterator iter_test;
	std::string str_sourcepath = iter->get_value( pTMCRecord->mTMColumn_SourcePath );

	// Entries with same sourcepaths have the same password
	for ( iter_test = children().begin() ; children().end() != iter_test ; ++iter_test )
		if ( iter_test->get_value( pTMCRecord->mTMColumn_SourcePath ) == str_sourcepath )
			(*iter_test)[pTMCRecord->mTMColumn_Password] = str_password;
	
	return;
}


// Method (public) : create command
std::string
cLStore_Cat::createCommand( bool shuffle )
{
	guchar						handle		= Pdfchain::Cat::HANDLE_DEFAULT;
	std::string					str_command	= "";
	std::string					str_pages	= "";
	std::string::size_type		pos_A		= 0;
	std::string::size_type		pos_B		= 0;
	Gtk::TreeModel::iterator	iter;

	// Indicate each file at list with a handle (same files have same handle)
	//indicateHandles();	// not nessesary - handles are re-indexed with every change

	// Assign a file on each handle
	for ( iter = children().begin() , handle = Pdfchain::Cat::HANDLE_FIRST ;
	      children().end() != iter && Pdfchain::Cat::HANDLE_FINAL >= handle ;
	      ++iter ) {

		if ( handle == (*iter)[pTMCRecord->mTMColumn_Cmd_Handle] ) {
			str_command.append( " " );
			str_command.append( 1 , static_cast<char>( handle++ ) );
			str_command.append( "=" );
			str_command.append( Pdfchain::quote_path( iter->get_value( pTMCRecord->mTMColumn_SourcePath ) ) );
		}
	}

	// If no handle is valid, no file is valid --> No valid command can be generated
	if ( "" == str_command )	return "";

	// Add "input_pw" only if at least one file has a input password
	for ( iter = children().begin() ; children().end() != iter ; ++iter ) {
		if ( "" != (*iter)[pTMCRecord->mTMColumn_Password] ) {
			str_command += " " + Pdfchain::Cmd::PASSWORD_INPUT;
			break;
		}
	}
	
	// Assign the input passowrd on each handle, if there is one
	for ( iter = children().begin() , handle = Pdfchain::Cat::HANDLE_FIRST ;
	      children().end() != iter && Pdfchain::Cat::HANDLE_FINAL >= handle ;
	      ++iter ) {

		if ( handle == (*iter)[pTMCRecord->mTMColumn_Cmd_Handle] ) {
			if ( "" != (*iter)[pTMCRecord->mTMColumn_Password] ) {
				str_command.append( " " );
				str_command.append( 1 , static_cast<char>( handle ) );
				str_command.append( "=" );
				str_command.append( Glib::filename_from_utf8( iter->get_value( pTMCRecord->mTMColumn_Password ) ) );
			}
			
			handle++;
		}
	}
	
	// Select Cat or Shuffle
	switch ( shuffle ) {
		case true:	str_command += " " + Pdfchain::Cmd::SHUFFLE;	break;
		default:	str_command += " " + Pdfchain::Cmd::CAT;		break;
	}
	
	// Assign Pages, EvenOdd, Rotation on each handle
	for ( iter = children().begin() ; children().end() != iter ; ++iter ) {
		if ( Pdfchain::Cat::HANDLE_FIRST <= (*iter)[pTMCRecord->mTMColumn_Cmd_Handle] &&
		     Pdfchain::Cat::HANDLE_FINAL >= (*iter)[pTMCRecord->mTMColumn_Cmd_Handle] ) {

			// Create command
			switch ( validatePages( iter ) ) {

				case FALSE:
					return "";
					break;
						
				case TRUE: {
					str_pages = Glib::filename_from_utf8( iter->get_value( pTMCRecord->mTMColumn_Pages ) );
						
					pos_A = pos_B = 0;

					while ( str_pages.length() > pos_A && str_pages.length() > pos_B ) {

						// Find begin and end of an argument (arguments are separated by ' ')
						for ( pos_A = pos_B ; str_pages.length() > pos_A && 0 == str_pages.compare( pos_A , 1 , " " ) ; pos_A++ );	// Skips all ' '
						for ( pos_B = pos_A ; str_pages.length() > pos_B && 0 != str_pages.compare( pos_B , 1 , " " ) ; pos_B++ );	// Skips all characters, without ' '

						// Append argument
						str_command.append( " " );
						str_command.append( 1 , static_cast<char>( iter->get_value( pTMCRecord->mTMColumn_Cmd_Handle ) ) );
						str_command.append( str_pages.substr( pos_A , pos_B - pos_A ) );

						// EvenOdd
						if ( std::string::npos != (str_pages.substr( pos_A , pos_B - pos_A )).find( "-" ) )
							str_command.append( iter->get_value( pTMCRecord->mTMColumn_Cmd_EvenOdd ) );

						// Rotation
						str_command.append( iter->get_value( pTMCRecord->mTMColumn_Cmd_Rotation ) );
					}
				}
				break;
			}
		}
	}

	return str_command;
}