"""
Contains the code to create and map objects to the Stampede DB schema
via a SQLAlchemy interface.
"""
__rcsid__ = "$Id$"
__author__ = "Monte Goode"
__author__ = "Karan Vahi"

from Pegasus.netlogger.analysis.schema._base import SABase

try:
    from sqlalchemy import *
    from sqlalchemy import orm, func, exc
    from sqlalchemy.orm import relation, backref
except ImportError, e:
    print '** SQLAlchemy library needs to be installed: http://www.sqlalchemy.org/ **\n'
    raise ImportError, e

import time
import warnings

CURRENT_SCHEMA_VERSION = 4.0
        
# Empty classes that will be populated and mapped
# to tables via the SQLAlch mapper.
class DashboardWorkflow(SABase):
    pass
    
class DashboardWorkflowstate(SABase):
    pass


    
def initializeToDashboardDB(db, metadata, kw={}):
    """
    Function to create the Workflow schema that just tracks the root
    level workflows, if it does not exist,
    if it does exist, then connect and set up object mappings.
    
    @type   db: SQLAlch db/engine object.
    @param  db: Engine object to initialize.
    @type   metadata: SQLAlch metadata object.
    @param  metadata: Associated metadata object to initialize.
    @type   kw: dict
    @param  kw: Keywords to pass to Table() functions
    """
    KeyInt = Integer
    # MySQL likes using BIGINT for PKs but some other
    # DB don't like it so swap as needed.
    if db.name == 'mysql':
        KeyInt = BigInteger
        kw['mysql_charset'] = 'latin1'
        
    if db.name == 'sqlite':
        warnings.filterwarnings('ignore', '.*does \*not\* support Decimal*.')
    
    
    # pg_workflow definition
    # ==> Information comes from braindump.txt file
    
    # wf_uuid = autogenerated by database                   wfuuid, submitted, directory, database connection
    # dax_label = label
    # timestamp = pegasus_wf_time
    # submit_hostname = (currently missing)
    # submit_dir = run
    #

    pg_workflow = Table('workflow', metadata,
                        Column('wf_id', KeyInt, primary_key=True, nullable=False),
                        Column('wf_uuid', VARCHAR(255), nullable=False),
                        Column('dax_label', VARCHAR(255), nullable=True),
                        Column('dax_version', VARCHAR(255), nullable=True),
                        Column('dax_file', VARCHAR(255), nullable=True),
                        Column('dag_file_name', VARCHAR(255), nullable=True),
                        Column('timestamp', NUMERIC(precision=16,scale=6), nullable=True),
                        Column('submit_hostname', VARCHAR(255), nullable=True),
                        Column('submit_dir', TEXT, nullable=True),
                        Column('planner_arguments', TEXT, nullable=True),
                        Column('user', VARCHAR(255), nullable=True),
                        Column('grid_dn', VARCHAR(255), nullable=True),
                        Column('planner_version', VARCHAR(255), nullable=True),
                        Column('db_url', TEXT, nullable=True),
                        **kw
    )
    
    Index('wf_id_KEY', pg_workflow.c.wf_id, unique=True)
    Index('wf_uuid_UNIQUE', pg_workflow.c.wf_uuid, unique=True)

    try:
        orm.mapper(DashboardWorkflow, pg_workflow )
    except exc.ArgumentError:
        pass


    pg_workflowstate = Table('workflowstate', metadata,
    # All three columns are marked as primary key to produce the desired
    # effect - ie: it is the combo of the three columns that make a row
    # unique.
                             Column('wf_id', KeyInt, ForeignKey('workflow.wf_id', ondelete='CASCADE'), 
                                    nullable=False, primary_key=True),
                             Column('state', Enum('WORKFLOW_STARTED', 'WORKFLOW_TERMINATED'), 
                                nullable=False, primary_key=True),
                             Column('timestamp', NUMERIC(precision=16,scale=6), nullable=False, primary_key=True,
                                    default=time.time()),
                             Column('restart_count', INT, nullable=False),
                             Column('status', INT, nullable=True),
                             **kw
    )
    
    Index('UNIQUE_WORKFLOWSTATE', pg_workflowstate.c.wf_id, pg_workflowstate.c.state,
        pg_workflowstate.c.timestamp, unique=True)
    
    try:
        orm.mapper(DashboardWorkflowstate, pg_workflowstate)
    except exc.ArgumentError:
        pass
    

    
    metadata.create_all(db)
    pass
    

def main():
    """
    Example of how to creat SQLAlch object and initialize/create
    to Stampede DB schema.
    """
    db = create_engine('sqlite:////tmp/pegasusTest.db', echo=True)
    metadata = MetaData()
    initializeToDashboardDB(db, metadata)
    metadata.bind = db

    sm = orm.sessionmaker(bind=db, autoflush=True, autocommit=False,
        expire_on_commit=True)
    session = orm.scoped_session(sm)
    
    pass
    
if __name__ == '__main__':
    main()
