/*
    PPGNUstepCocoaGlue.m

    Copyright 2014-2017 Josh Freeman
    http://www.twilightedge.com

    This file is part of PikoPixel for GNUstep.
    PikoPixel is a graphical application for drawing & editing pixel-art images.

    PikoPixel is free software: you can redistribute it and/or modify it under
    the terms of the GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the License, or (at your
    option) any later version approved for PikoPixel by its copyright holder (or
    an authorized proxy).

    PikoPixel is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
    details.

    You should have received a copy of the GNU Affero General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef GNUSTEP

#import <Cocoa/Cocoa.h>
#import "NSObject_PPUtilities.h"
#import "PPAppBootUtilities.h"
#import "NSBitmapImageRep_PPUtilities.h"
#import "PPGeometry.h"


@implementation NSObject (PPGNUstepCocoaGlue)

+ (void) ppGSCocoaGlue_InstallPatches
{
    macroSwizzleInstanceMethod(NSView, cacheDisplayInRect:toBitmapImageRep:,
                                ppGSPatch_CacheDisplayInRect:toBitmapImageRep:);


    macroSwizzleInstanceMethod(NSColorWell, activate:, ppGSPatch_Activate:);

    macroSwizzleInstanceMethod(NSColorWell, deactivate, ppGSPatch_Deactivate);

    macroSwizzleInstanceMethod(NSColorWell, performClick:, ppGSPatch_PerformClick:);

    macroSwizzleInstanceMethod(NSColorWell, sendAction:to:, ppGSPatch_SendAction:to:);


    macroSwizzleClassMethod(NSColor, colorWithPatternImage:, ppGSPatch_ColorWithPatternImage:);
}

+ (void) load
{
    macroPerformNSObjectSelectorAfterAppLoads(ppGSCocoaGlue_InstallPatches);
}

@end

@implementation NSView (PPGNUstepCocoaGlue)

// ppGSCocoaGlue_ContentImage: Utility method which returns an NSImage of a view's content

- (NSImage *) ppGSCocoaGlue_ContentImage
{
    NSRect contentBounds;
    NSImage *contentImage;

    contentBounds = [self bounds];
    contentImage = [[[NSImage alloc] initWithSize: contentBounds.size] autorelease];

    [contentImage lockFocus];

    [self drawRect: contentBounds];

    [contentImage unlockFocus];

    return contentImage;
}

// PATCH: -[NSView cacheDisplayInRect:toBitmapImageRep:]
// GNUstep's implmentation only reads the top-level view's pixels, ignoring subviews -
// override adds support for subviews (only first level, though) in order to work properly with
// PPCompositeThumbnail

- (void) ppGSPatch_CacheDisplayInRect: (NSRect) rect
            toBitmapImageRep: (NSBitmapImageRep *) bitmapImageRep
{
    NSImage *contentImage;
    NSEnumerator *subviewsEnumerator;
    NSView *subview;

    [bitmapImageRep ppSetAsCurrentGraphicsContext];

    contentImage = [self ppGSCocoaGlue_ContentImage];

    [contentImage drawInRect: [self bounds]
                    fromRect: [self bounds]
                    operation: NSCompositeCopy
                    fraction: 1.0];

    subviewsEnumerator = [[self subviews] objectEnumerator];

    while (subview = [subviewsEnumerator nextObject])
    {
        contentImage = [subview ppGSCocoaGlue_ContentImage];

        [contentImage drawInRect: [subview frame]
                        fromRect: [subview bounds]
                        operation: NSCompositeSourceOver
                        fraction: 1.0];
    }

    [bitmapImageRep ppRestoreGraphicsContext];
}

@end

@implementation NSColorWell (PPGNUstepCocoaGlue)

static bool gIsActivatingColorWell = NO;

- (void) ppGSPatch_Activate: (BOOL) exclusive
{
    gIsActivatingColorWell = YES;

    [self ppGSPatch_Activate: exclusive];

    gIsActivatingColorWell = NO;
}

- (void) ppGSPatch_Deactivate
{
    if (!gIsActivatingColorWell && [self isActive])
    {
        [[NSColorPanel sharedColorPanel] orderOut: self];
    }

    [self ppGSPatch_Deactivate];
}

- (void) ppGSPatch_PerformClick: (id) sender
{
    if ([self isActive])
    {
        [self deactivate];
    }
    else
    {
        [self activate: YES];
    }
}

- (BOOL) ppGSPatch_SendAction: (SEL) theAction to: (id) theTarget
{
    BOOL didSendAction = [self ppGSPatch_SendAction: theAction to: theTarget];

    //    GNUstep's NSColorPanel currently sets up the bottom row of NSColorWell swatches with
    // the wrong target (should be the NSColorPanel itself, instead it's the main NSColorWell,
    // though NSColorWell instances don't implment the action method, _bottomWellAction:;
    //    The sendAction:to: method does find the appropriate target and redirect the action
    // when the color panel is displayed normally, but not when it's displayed as a popup, so
    // if sendAction:to: fails and the action is _bottomWellAction:, manually forward the
    // message to the colorwell's window (the color panel).

    if (!didSendAction && sel_isEqual(theAction, @selector(_bottomWellAction:)))
    {
        didSendAction = [self ppGSPatch_SendAction: theAction to: [self window]];
    }

    return didSendAction;
}

@end

@implementation NSColor (PPGNUstepCocoaGlue)

#define kMinPatternImageDimension   32

+ (NSColor *) ppGSPatch_ColorWithPatternImage: (NSImage *) image
{
    NSColor *patternColor;
    NSSize imageSize, patternSize;

    patternColor = [self ppGSPatch_ColorWithPatternImage: image];

    if (!patternColor)
        goto ERROR;

    imageSize = [image size];

    if (PPGeometry_IsZeroSize(imageSize))
    {
        goto ERROR;
    }

    patternSize =
            NSMakeSize(imageSize.width * ceilf(kMinPatternImageDimension / imageSize.width),
                        imageSize.height * ceilf(kMinPatternImageDimension / imageSize.height));

    if (!NSEqualSizes(patternSize, imageSize))
    {
        NSImage *patternImage;
        NSColor *resizedPatternColor;

        patternImage = [[[NSImage alloc] initWithSize: patternSize] autorelease];

        if (!patternImage)
            goto ERROR;

        [patternImage lockFocus];

        [patternColor set];
        NSRectFill(PPGeometry_OriginRectOfSize(patternSize));

        [patternImage unlockFocus];

        resizedPatternColor = [self ppGSPatch_ColorWithPatternImage: patternImage];

        if (!resizedPatternColor)
            goto ERROR;

        patternColor = resizedPatternColor;
    }

    return patternColor;

ERROR:
    return patternColor;
}

@end

#endif  // GNUSTEP

