/*
    PPGNUstepGlue_MenuKeyEquivalents.m

    Copyright 2014-2017 Josh Freeman
    http://www.twilightedge.com

    This file is part of PikoPixel for GNUstep.
    PikoPixel is a graphical application for drawing & editing pixel-art images.

    PikoPixel is free software: you can redistribute it and/or modify it under
    the terms of the GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the License, or (at your
    option) any later version approved for PikoPixel by its copyright holder (or
    an authorized proxy).

    PikoPixel is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
    details.

    You should have received a copy of the GNU Affero General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef GNUSTEP

#import <Cocoa/Cocoa.h>
#import "NSObject_PPUtilities.h"
#import "PPAppBootUtilities.h"


#define kModifierNamesArraySize     16

#define macroModifierNamesIndexForModifierMask(mask)        ((mask >> 17) & 0x0F)
#define macroModifierMaskForModifierNamesIndex(index)       (index << 17)


static NSString *gModifierNamesArray[kModifierNamesArraySize];


static void SetupModifierNamesArray(void);

static NSDictionary *KeyToDisplayKeyDict(void);


@implementation NSObject (PPGNUstepGlue_MenuKeyEquivalents)

+ (void) ppGSGlue_MenuKeyEquivalents_InstallPatches
{
    macroSwizzleInstanceMethod(NSMenu, performKeyEquivalent:, ppGSPatch_PerformKeyEquivalent:);

    macroSwizzleInstanceMethod(NSMenuItemCell, _keyEquivalentString,
                                ppGSPatch_KeyEquivalentString);
}

+ (void) ppGSGlue_MenuKeyEquivalents_Install
{
    SetupModifierNamesArray();

    [self ppGSGlue_MenuKeyEquivalents_InstallPatches];
}

+ (void) load
{
    macroPerformNSObjectSelectorAfterAppLoads(ppGSGlue_MenuKeyEquivalents_Install);
}

@end

@implementation NSMenu (PPGNUstepGlue_MenuKeyEquivalents)

- (BOOL) ppGSPatch_PerformKeyEquivalent: (NSEvent *) theEvent
{
    static bool didKeySubstitutionCheck = NO;
    BOOL returnValue;

    if (didKeySubstitutionCheck)
    {
        returnValue = [self ppGSPatch_PerformKeyEquivalent: theEvent];
    }
    else
    {
        NSString *eventChars = [theEvent charactersIgnoringModifiers];

        // Delete key won't trigger the "Delete" menu item because the item's key equivalent is
        // the backspace, so substitute the delete key event with a backspace key event

        if ([eventChars length]
            && ([eventChars characterAtIndex: 0] == NSDeleteCharacter))
        {
            NSEvent *backspaceKeyEvent = [NSEvent keyEventWithType: [theEvent type]
                                                    location: [theEvent locationInWindow]
                                                    modifierFlags: [theEvent modifierFlags]
                                                    timestamp: [theEvent timestamp]
                                                    windowNumber: [theEvent windowNumber]
                                                    context: [theEvent context]
                                                    characters: @"\b"
                                                    charactersIgnoringModifiers: @"\b"
                                                    isARepeat: [theEvent isARepeat]
                                                    keyCode: [theEvent keyCode]];

            if (backspaceKeyEvent)
            {
                theEvent = backspaceKeyEvent;
            }
        }

        didKeySubstitutionCheck = YES;

        returnValue = [self ppGSPatch_PerformKeyEquivalent: theEvent];

        didKeySubstitutionCheck = NO;
    }

    return returnValue;
}

@end

@implementation NSMenuItemCell (PPGNUstepGlue_MenuKeyEquivalents)

- (NSString *) ppGSPatch_KeyEquivalentString
{
    static NSDictionary *keyToDisplayKeyDict = nil;
    static NSCharacterSet *uppercaseLetterCharacterSet = nil;
    NSString *key, *displayKey;
    NSUInteger modifierKeyMask;

    key = [_menuItem keyEquivalent];

    if (!key || ![key length])
    {
        return nil;
    }

    modifierKeyMask = [_menuItem keyEquivalentModifierMask];

    if (!keyToDisplayKeyDict)
    {
        keyToDisplayKeyDict = [KeyToDisplayKeyDict() retain];
    }

    if (!uppercaseLetterCharacterSet)
    {
        uppercaseLetterCharacterSet = [[NSCharacterSet uppercaseLetterCharacterSet] retain];
    }

    displayKey = [keyToDisplayKeyDict objectForKey: key];

    if (displayKey)
    {
        key = displayKey;
    }
    else if ([key rangeOfCharacterFromSet: uppercaseLetterCharacterSet].length)
    {
        modifierKeyMask |= NSShiftKeyMask;
    }

    return [gModifierNamesArray[macroModifierNamesIndexForModifierMask(modifierKeyMask)]
                stringByAppendingString: key];
}

@end

static void SetupModifierNamesArray(void)
{
    NSUInteger i, modifierKeyMask;

    for (i=0; i<kModifierNamesArraySize; i++)
    {
        modifierKeyMask = macroModifierMaskForModifierNamesIndex(i);

        gModifierNamesArray[i] =
            [[NSString stringWithFormat: @"  %@%@%@%@",
                                        (modifierKeyMask & NSControlKeyMask) ? @"Ctrl+" : @"",
                                        (modifierKeyMask & NSAlternateKeyMask) ? @"Alt+" : @"",
                                        (modifierKeyMask & NSCommandKeyMask) ? @"Super+" : @"",
                                        (modifierKeyMask & NSShiftKeyMask) ? @"Shift+" : @""]
                    retain];
    }
}

static NSDictionary *KeyToDisplayKeyDict(void)
{
    NSMutableDictionary *keyToDisplayKeyDict;
    unichar lowercaseChar, uppercaseChar;

    keyToDisplayKeyDict = [NSMutableDictionary dictionaryWithObjectsAndKeys:

                                                    // Tab
                                                        @"\u21E5",
                                                    @"\t",

                                                    // Return
                                                        @"\u21A9",
                                                    @"\r",

                                                    // ESC
                                                        @"\u238B",
                                                    @"\e",

                                                    // Space
                                                        @"\u23B5",
                                                    @" ",

                                                    // Backspace
                                                        @"\u232B",
                                                    @"\b",

                                                    // Left arrow
                                                        @"\u2190",
                                                    @"\uF702",

                                                    // Up arrow
                                                        @"\u2191",
                                                    @"\uF700",

                                                    // Right arrow
                                                        @"\u2192",
                                                    @"\uF703",

                                                    // Down arrow
                                                        @"\u2193",
                                                    @"\uF701",

                                                        nil];

    // Lowercase to uppercase alphabet chars

    for (lowercaseChar = 'a'; lowercaseChar <= 'z'; lowercaseChar++)
    {
        uppercaseChar = 'A' + lowercaseChar - 'a';

        [keyToDisplayKeyDict
                    setObject: [NSString stringWithCharacters: &uppercaseChar length: 1]
                    forKey: [NSString stringWithCharacters: &lowercaseChar length: 1]];
    }

    return [NSDictionary dictionaryWithDictionary: keyToDisplayKeyDict];
}

#endif  // GNUSTEP

