/*
    PPGNUstepGlue_Miscellaneous.m

    Copyright 2014-2017 Josh Freeman
    http://www.twilightedge.com

    This file is part of PikoPixel for GNUstep.
    PikoPixel is a graphical application for drawing & editing pixel-art images.

    PikoPixel is free software: you can redistribute it and/or modify it under
    the terms of the GNU Affero General Public License as published by the
    Free Software Foundation, either version 3 of the License, or (at your
    option) any later version approved for PikoPixel by its copyright holder (or
    an authorized proxy).

    PikoPixel is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
    details.

    You should have received a copy of the GNU Affero General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#ifdef GNUSTEP

#import <Cocoa/Cocoa.h>
#import "NSObject_PPUtilities.h"
#import "PPAppBootUtilities.h"
#import "PPApplication.h"
#import "NSFileManager_PPUtilities.h"
#import "PPDocumentEditPatternPresetsSheetController.h"
#import "PPPopupPanel.h"
#import "PPToolsPanelController.h"
#import "PPCanvasView.h"
#import "PPToolButtonMatrix.h"
#import "NSPasteboard_PPUtilities.h"
#import "NSDocumentController_PPUtilities.h"
#import "PPUserFolderPaths.h"


@implementation NSObject (PPGNUstepGlue_Miscellaneous)

+ (void) ppGSGlue_Miscellaneous_InstallPatches
{
    macroSwizzleInstanceMethod(PPApplication, validateMenuItem:, ppGSPatch_ValidateMenuItem:);


    macroSwizzleInstanceMethod(NSFileManager, ppVerifySupportFileDirectory,
                                ppGSPatch_VerifySupportFileDirectory);


    macroSwizzleInstanceMethod(PPDocumentEditPatternPresetsSheetController,
                                selectPatternsTableRow:,
                                ppGSPatch_SelectPatternsTableRow:);


    macroSwizzleInstanceMethod(PPPopupPanel, canBecomeKeyWindow, ppGSPatch_CanBecomeKeyWindow);


    macroSwizzleInstanceMethod(PPToolsPanelController, defaultPinnedWindowFrame,
                                ppGSPatch_DefaultPinnedWindowFrame);


    macroSwizzleInstanceMethod(PPCanvasView, drawRect:, ppGSPatch_DrawRect:);


    macroSwizzleClassMethod(PPToolButtonMatrix, cellClass, ppGSPatch_CellClass);
}

+ (void) load
{
    macroPerformNSObjectSelectorAfterAppLoads(ppGSGlue_Miscellaneous_InstallPatches);
}

@end

@implementation PPApplication (PPGNUstepGlue_Miscellaneous)

- (BOOL) ppGSPatch_ValidateMenuItem: (id <NSMenuItem>) menuItem
{
    SEL menuItemAction = [menuItem action];

    if (sel_isEqual(menuItemAction, @selector(newDocumentFromPasteboard:)))
    {
        return [NSPasteboard ppPasteboardHasBitmap] ? YES : NO;
    }

    // printing is currently disabled, so both printing-related menu items (print & page setup)
    // use runPageLayout: as their action for convenience when invalidating them

    if (sel_isEqual(menuItemAction, @selector(runPageLayout:)))
    {
        return NO;
    }


    if (sel_isEqual(menuItemAction, @selector(activateNextDocumentWindow:))
        || sel_isEqual(menuItemAction, @selector(activatePreviousDocumentWindow:)))
    {
        return [[NSDocumentController sharedDocumentController] ppHasMultipleDocuments];
    }

    return YES;
}

@end

@implementation NSFileManager (PPGNUstepGlue_Miscellaneous)

- (bool) ppGSPatch_VerifySupportFileDirectory
{
    NSString *supportFolderPath = PPUserFolderPaths_ApplicationSupport();
    BOOL isDirectory = NO;

    if (![supportFolderPath length])
    {
        goto ERROR;
    }

    if ([self fileExistsAtPath: supportFolderPath isDirectory: &isDirectory])
    {
        if (!isDirectory)
            goto ERROR;

        return YES;
    }

    return [self createDirectoryAtPath: supportFolderPath
                    withIntermediateDirectories: YES
                    attributes: nil
                    error: NULL];

ERROR:
    return NO;
}

@end

@implementation PPDocumentEditPatternPresetsSheetController (PPGNUstepGlue_Miscellaneous)

- (void) ppGSPatch_SelectPatternsTableRow: (unsigned) row
{
    if (row < [_editablePatterns count])
    {
        [_editablePatternsTable selectRowIndexes: [NSIndexSet indexSetWithIndex: row]
                                byExtendingSelection: NO];
    }
    else
    {
        [_editablePatternsTable deselectAll: self];
    }

    [self performSelector: @selector(updatePatternViewForCurrentPatternsTableSelection)];
}

@end

@implementation PPPopupPanel (PPGNUstepGlue_Miscellaneous)

// PATCH: -[PPPopupPanel canBecomeKeyWindow]
// GNUstep's implementation of -[NSWindow canBecomeKeyWindow] doesn't automatically return NO
// when the window has no title bar or resize bar (as on OS X), so need PPPopupPanel override
// to prevent popup panels from becoming key

- (BOOL) ppGSPatch_CanBecomeKeyWindow
{
    return NO;
}

@end

@implementation PPToolsPanelController (PPGNUstepGlue_Miscellaneous)

// PATCH: -[PPToolsPanelController defaultPinnedWindowFrame]
// Prevents the tools panel's default position from overlapping the main menu

#define kMinDistanceBetweenToolsPanelAndMainMenu    20

- (NSRect) ppGSPatch_DefaultPinnedWindowFrame
{
    NSRect panelFrame, mainMenuFrame, mainMenuFrameMargin;

    panelFrame = [self ppGSPatch_DefaultPinnedWindowFrame];
    mainMenuFrame = [[[NSApp mainMenu] window] frame];
    mainMenuFrameMargin = NSInsetRect(mainMenuFrame, -kMinDistanceBetweenToolsPanelAndMainMenu,
                                        -kMinDistanceBetweenToolsPanelAndMainMenu);

    if (!NSIsEmptyRect(NSIntersectionRect(panelFrame, mainMenuFrameMargin)))
    {
        panelFrame.origin.y = NSMinY(mainMenuFrameMargin) - panelFrame.size.height;
    }

    return panelFrame;
}

@end

@implementation PPCanvasView (PPGNUstepGlue_Miscellaneous)

// PATCH: -[PPCanvasView drawRect:]
// Fix for PPCanvasView zoomout artifacts that appear when there's only one scrollbar visible
// (but not both) - workaround is to draw over the artifacts by manually filling the area
// outside the visible canvas with the enclosing scrollview's background color (normally
// the canvasview doesn't draw this area and the underlying scrollview's background shows
// through automatically)

- (void) ppGSPatch_DrawRect: (NSRect) rect
{
    // redraw the background surrounding the visible canvas if:
    // - drawing is allowed (zoomed-images draw-mode is 0 (normal))
    // - and the dirty-rect covers some area outside the visible canvas
    // - and at least one scrollbar is visible (should only be one at this point - if they were
    // both visible, the dirty-rect would not be outside the visible canvas)
    if ((_zoomedImagesDrawMode == 0)
        && (!NSContainsRect(_offsetZoomedVisibleCanvasBounds, rect))
        && ((_offsetZoomedCanvasFrame.origin.x == 0.0)
            || (_offsetZoomedCanvasFrame.origin.y == 0.0)))
    {
        static NSColor *backgroundColor = nil;
        NSRect rect1 = NSZeroRect, rect2 = NSZeroRect;

        if (!backgroundColor)
        {
            backgroundColor = [[[self enclosingScrollView] backgroundColor] retain];
        }

        if (_offsetZoomedCanvasFrame.origin.x == 0.0)
        {
            // horizontal scrollbar - background-fill areas are above & below visible canvas

            CGFloat canvasMaxY = NSMaxY(_offsetZoomedCanvasFrame);

            rect1 = NSMakeRect(rect.origin.x,
                                canvasMaxY,
                                rect.size.width,
                                NSMaxY(rect) - canvasMaxY);

            rect2 = NSMakeRect(rect.origin.x,
                                rect.origin.y,
                                rect.size.width,
                                _offsetZoomedCanvasFrame.origin.y - rect.origin.y);
        }
        else if (_offsetZoomedCanvasFrame.origin.y == 0.0)
        {
            // vertical scrollbar - background-fill areas are on left & right of visible canvas

            CGFloat canvasMaxX = NSMaxX(_offsetZoomedCanvasFrame);

            rect1 = NSMakeRect(canvasMaxX,
                                rect.origin.y,
                                NSMaxX(rect) - canvasMaxX,
                                rect.size.height);

            rect2 = NSMakeRect(rect.origin.x,
                                rect.origin.y,
                                _offsetZoomedCanvasFrame.origin.x - rect.origin.x,
                                rect.size.height);
        }

        [backgroundColor set];
        NSRectFill(rect1);
        NSRectFill(rect2);
    }

    [self ppGSPatch_DrawRect: rect];
}

@end

@implementation PPToolButtonMatrix (PPGNUstepGlue_Miscellaneous)

// PATCH: +[PPToolButtonMatrix cellClass]
// GNUstep's nib loader has an issue when loading instances of NSMatrix subclasses (but not
// instances of NSMatrix itself): All of the NSMatrix-subclass' cells become NSActionCells,
// regardless of what their actual class is in the nib file. This is because the nib loader
// mistakenly forces the loaded cells (which originally have the correct class) to be
// reallocated by the class object returned by +cellClass (+[NSMatrix cellClass] returns
// NSActionCell). The workaround is to patch +[PPToolButtonMatrix cellClass] to return the
// correct class for allocating PPToolButtonMatrix's cells: NSButtonCell.

+ (Class) ppGSPatch_CellClass
{
    static Class buttonCellClass = NULL;

    if (!buttonCellClass)
    {
        buttonCellClass = [[NSButtonCell class] retain];
    }

    return buttonCellClass;
}

@end

#endif  // GNUSTEP

