/*	Management

PIRL CVS ID: Management.java,v 1.17 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor;

import	PIRL.Configuration.Configuration;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.Database.Database_Exception;
import	PIRL.Messenger.Message;

import	java.io.Writer;
import	java.io.IOException;
import	java.util.Vector;


/**	<i>Management</i> is the Conductor management interface.
<p>
	The Conductor Management interface provides the ability to set and
	get various processing state and control information from a
	Conductor. It also provides registration control for processing
	state change event notifications and log stream writer attachment.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.17
	@see	Conductor
*/
public interface Management
{
/**	Get the Conductor {@link Conductor#Identity() identity}.
<p>
	@return	A Message containing the Conductor identification parameters.
	@see	Conductor#Identity()
*/
public Message Identity ();

/**	Get the Conductor Configuration.
<p>
	The Configuration that is returned is a copy of the current state of
	the Conductor Configuration, which is dynamically changed during
	processing. All parameters named "PASSWORD" (case insensitive) will
	have their values reset to {@link Processing_Changes#MASKED_PASSWORD}.
<p>
	@return	A copy of the Conductor Configuration.
	@see	Conductor#Preconfigure(Configuration)
	@see	Conductor#Postconfigure(Configuration)
*/
public Configuration Configuration ();

/**	Start Conductor source record processing.
<p>
	If the Conductor is in a negative {@link #Processing_State()
	processing state} ({@link Conductor#WAITING waiting } or {@link
	Conductor#HALTED halted}) it will enter the {@link Conductor#RUNNING
	running} state, reconfigure itself from its Configuration source, and
	source record processing will commence. If the Conductor is in the
	{@link Conductor#RUN_TO_WAIT run-to-wait} state it will clear this to
	the running state; for any other positive state (running or {@link
	Conductor#POLLING polling}) there will be no change.
<p>
	@see	Conductor#Start()
*/
public void Start ();

/**	Request that Conductor stop source record processing.
<p>
	If the Conductor is in a positive {@link #Processing_State()
	processing state} it will enter the {@link Conductor#RUN_TO_WAIT
	run-to-wait} state, in which it will enter the {@link
	Conductor#WAITING waiting} state when the current source record
	completes processing. If the Conductor is in the {@link
	Conductor#POLLING polling} state it will immediately stop polling for
	new source records. There will be no effect for any negative state.
<p>
	@see	Conductor#Stop()
*/
public void Stop ();

/**	Immediately stop Conductor processing and exit.
<p>
	Any open log file is closed. The database server is disconnected. An
	{@link Processing_Changes#Exiting(boolean) exiting} processing event
	is sent to all {@link #Add_Processing_Listener(Processing_Listener)
	processing listeners}. The application exits with a {@link
	Conductor#EXIT_SUCCESS success status}.
<p>
	<b>N.B.</b>: If source processing is {@link #Processing_State() running}
	it is aborted.
<p>
	@see	Conductor#Quit()
*/
public void Quit ();

/**	Get the current Conductor processing state.
<p>
	The possible processing state values are:
<p>
<dl>
<dt>{@link Conductor#RUNNING}
<dd>Source records are being processing.

<dt>{@link Conductor#POLLING}
<dd>No unprocessed source records are currently available for processing;
	the Conductor is {@link #Poll_Interval(int) polling} for source
	records to process.

<dt>{@link Conductor#RUN_TO_WAIT}
<dd>When processing of the current source record completes Conductor will
	go into the waiting state.

<dt>{@link Conductor#WAITING}
<dd>The Conductor is waiting to be told to being processing.

<dt>{@link Conductor#HALTED}
<dd>A problem condition caused the Conductor to halt processing. The
	problem may be the result of the maximum number of {@link
	#Stop_on_Failure(int) sequential failures} of source record
	processing having occured, a database access failure, or some other
	{@link #Processing_Exception system error}.
</dl>
<p>
	The WAITING and HALTED state codes are negative; all others are positive.
<p>
	@return	A Conductor processing state code.
*/
public int Processing_State ();

/**	Get the current cache of source records.
<p>
	The table is expected to be delivered with the record fields in
	{@link Conductor#SOURCES_FIELD_NAMES} order, and with the records
	sorted by increasing {@link Conductor#SEQUENCE_FIELD} order.
<p>
	<b>N.B.</b>: Only the contents of the Conductor source records cache
	is delivered. The contents of the database sources table may be
	much, much larger.
<p>
	@return	A table Vector of record Vectors of field Strings. The first
		record contains the field names.
*/
public Vector<Vector<String>> Sources ();

/**	Get the procedures table.
<p>
	The table is expected to be delivered with the record fields in
	{@link Conductor#PROCEDURES_FIELD_NAMES} order, and with the records in
	processing order.
<p>
	<b>N.B.</b>: The entire contents of the database procedures table is
	delivered.
<p>
	@return	A table Vector of record Vectors of field Strings. The first
		record contains the field names.
*/
public Vector<Vector<String>> Procedures ();

/**	Get the current Conductor processing state variables.
<p>
	All Processing_Changes values are set except the except the flag
	variables - {@link Processing_Changes#Sources_Refreshed(boolean)}, 
	{@link Processing_Changes#Procedures_Changed(boolean)} and {@link
	Processing_Changes#Exiting(boolean)} - which will always be false.
<p>
	@return Processing_Changes object.
	@see	Processing_Changes
*/
public Processing_Changes State ();

/**	Register a processing state change listener.
<p>
	The Conductor sends its {@link Processing_Event processing event}
	notifications to all registered listeners.
<p>
	@param	listener	A Processing_Listener.
	@return	This Management object.
	@see	Processing_Listener
*/
public Management Add_Processing_Listener
	(
	Processing_Listener	listener
	);

/**	Unregister a processing state change listener.
<p>
	@param	listener	The Processing_Listener to be removed from the
		Management list of registered listeners.
	@return	true	If the listener was registered and is now removed;
		false if it was not registered.
	@see	#Add_Processing_Listener(Processing_Listener)
*/
public boolean Remove_Processing_Listener
	(
	Processing_Listener	listener
	);

/**	Register a Writer to receive processing log stream output.
<p>
	The Conductor writes its processing log reports, including the
	output from all pipeline procedures it runs, to all registered
	log Writers.
<p>
	@param	writer	A Writer object.
	@return	This Management object.
	@see	#Enable_Log_Writer(Writer, boolean)
	@see	#Remove_Log_Writer(Writer)
*/
public Management Add_Log_Writer
	(
	Writer	writer
	);

/**	Unregister a log Writer.
<p>
	@param	writer	A Writer object.
	@return	true	If the writer was registered and is now removed;
		false if it was not registered.
	@see	#Add_Log_Writer(Writer)
*/
public boolean Remove_Log_Writer
	(
	Writer	writer
	);

/**	Enable or disable output to a {@link #Add_Log_Writer(Writer)
	registered log stream Writer}.
<p>
	@param	writer	A Writer that has been registered to receive
		Conductor log stream output. If the writer is not {@link
		#Add_Log_Writer(Writer) registered} to receive the Conductor log
		stream nothing is done.
	@param	enable	If false, Conductor log stream output to the Writer is
		suspended without having to unregister the Writer. If true, a
		Writer that has had its log stream output suspended will begin
		receiving it again.
	@return	This Management object.
*/
public Management Enable_Log_Writer
	(
	Writer	writer,
	boolean	enable
	);

/**	Set the interval at which the Conductor will poll for unprocessed
	source records.
<p>
	When the Conductor is in the polling {@link #Processing_State()
	state} it will periodically check the database sources table for
	source records that are marked as unprocessed. Unprocessed source
	records are used to refresh the Conductor cache of source records to
	be processed. When all source records in the cache have been
	processed, the Conductor will enter the polling state if the poll
	interval is positive; it will enter the waiting state if the poll
	interval is not positive (as if the {@link #Stop() stop request} had
	been issued), or will exit if it is not {@link
	#Connected_to_Stage_Manager() connected to a Stage_Manager} at the
	time.
<p>
	@param	seconds	The interval, in seconds, at which the Conductor will
		poll for unprocessed source records. If less than zero, zero will
		be set, but the value will be reset to the Conductor
		Configuration {@link Conductor#POLL_INTERVAL_PARAMETER} value or
		the {@link Conductor#DEFAULT_POLL_INTERVAL} value if the
		Configuration does contain the parameter, the next time the
		Conductor is {@link #Start() started} after having stopped
		processing for any reason.
	@return	This Management object.
	@see	Conductor#Poll_Interval(int)
*/
public Management Poll_Interval
	(
	int		seconds
	);

/**	Get the interval at which the Conductor will poll for unprocessed
	source records.
<p>
	This value may also be obtained from the {@link #Configuration()
	Configuration} {@link Conductor#POLL_INTERVAL_PARAMETER}.
<p>
	@return	The interval, in seconds, at which the Conductor will poll
		for unprocessed source records. If zero, polling has been
		disabled.
	@see	#Poll_Interval(int)
*/
public int Poll_Interval ();

/**	Set the Conductor default Reference_Resolver value.
<p>
	Normally when the Conductor Reference_Resolver is unable to resolve a
	reference it will throw an {@link #Processing_Exception() exception}
	and enter the halted {@link #Processing_State() processing state}, or
	exit if it is not {@link #Connected_to_Stage_Manager() connected to a
	Stage_Manager} at the time. If, however, the {@link
	Reference_Resolver#Default_Value(String) Reference_Resolver default
	value} is set to a non-null String that value will be used for the
	unresolved_reference instead of throwing an exception.
<p>
	@param	value	The default Reference_Resolver String value. If this
		starts with {@link Conductor#UNRESOLVED_REFERENCE_THROWS} (case
		insensitive) null will be used.
	@return	This Management object.
	@see	Conductor#Resolver_Default_Value(String)
	@see	Reference_Resolver
*/
public Management Resolver_Default_Value
	(
	String	value
	);

/**	Get the Conductor default Reference_Resolver value.
<p>
	This value may also be obtained from the {@link #Configuration()
	Configuration} {@link Conductor#UNRESOLVED_REFERENCE_PARAMETER}.
	Note, however, that the value will be {@link
	Conductor#UNRESOLVED_REFERENCE_THROWS} when the actual value is
	null.
<p>
	@return	The default Reference_Resolver String value.
	@see	#Resolver_Default_Value(String)
*/
public String Resolver_Default_Value ();

/**	Set the number of Conductor sequential source processing failures
	at which to stop processing.
<p>
	The Conductor keeps a {@link #Sequential_Failures() count of
	sequential source processing failures}. When this count reaches the
	stop-on-failure limit the Conductor will enter the halted {@link
	#Processing_State() processing state} or exit if it is not {@link
	#Connected_to_Stage_Manager() connected to a Stage_Manager} at the
	time.
<p>
	<b>N.B.</b>: Reaching the stop-on-failure limit does not cause the
	sequential failures count to be {@link #Reset_Sequential_Failures()
	reset}.
<p>
	@param	failure_count	The number of Conductor sequential source
		processing failures at which to stop processing. If zero
		sequential processing failures will never cause processing to
		stop. If negative the current value will not be changed, but the
		value will be reset to the Conductor Configuration {@link
		Conductor#STOP_ON_FAILURE_PARAMETER} value or the {@link
		Conductor#DEFAULT_STOP_ON_FAILURE} value if the Configuration
		does contain the parameter, the next time the Conductor is
		{@link #Start() started} after having stopped processing for
		any reason.
	@return	This Management object.
	@see	#Sequential_Failures()
	@see	Conductor#Stop_on_Failure(int)
*/
public Management Stop_on_Failure
	(
	int		failure_count
	);

/**	Get the number of Conductor sequential source processing failures
	at which to stop processing.
<p>
	This value may also be obtained from the {@link #Configuration()
	Configuration} {@link Conductor#STOP_ON_FAILURE_PARAMETER}
<p>
	@return	The number of Conductor sequential source processing failures
		at which to stop processing. If zero sequential processing
		failures will never cause processing to stop.
	@see	#Stop_on_Failure(int)
*/
public int Stop_on_Failure ();

/**	Get the count of sequential source processing failures that the
	Conductor has accumulated.
<p>
	Other source processing success and failure counts may be obtained
	from the {@link #Configuration() Configuration}:
<p>
<dl>
<dt>{@link Conductor#SOURCE_SUCCESS_COUNT}
<dd>The total number of source records that have successfully completed
	all processing since the Conductor first began processing.

<dt>{@link Conductor#SOURCE_FAILURE_COUNT}
<dd>The number of source records that have resulted in a failed procedure
	condition during processing.

<dt>{@link Conductor#TOTAL_FAILURE_COUNT}
<dd>The total number of source records that failed to be processed for
	any reason since the Conductor first began processing.
</dl>
<p>
	@return	The count of sequential source processing failures that the
		Conductor has accumulated.
	@see	#Stop_on_Failure(int)
	@see	#Reset_Sequential_Failures()
*/
public int Sequential_Failures ();

/**	Reset the count of sequential source processing failures that the
	Conductor has accumulated.
<p>
	If the Conductor is in the halted {@link #Processing_State()
	processing state} it is reset to the waiting state.
<p>
	@return	This Management object.
	@see	#Stop_on_Failure(int)
	@see	Conductor#Reset_Sequential_Failures()
*/
public Management Reset_Sequential_Failures ();

/**	Get the exception that caused Conductor processing to halt.
<p>
	<b>N.B.</b>: When Conductor processing is {@link #Start() started}
	the previous processing exception is cleared.
<p>
	@return	The Exception that caused processing to halt. This will be
		null if processing did not halt as the result of an exception,
		or the current processing state is not halted.
*/
public Exception Processing_Exception ();

/**	Test if the Conductor is connected to a Stage_Manager.
<p>
	@return	true if the Conductor is connected to a Stage_Manager via an
		open Local_Theater; false otherwise.
	@see	PIRL.Conductor.Maestro.Local_Theater
*/
public boolean Connected_to_Stage_Manager ();

}
