/*	JP2_Info

HiRISE CVS ID: JP2_Info.java,v 1.39 2012/11/19 00:13:25 castalia Exp

Copyright (C) 2006-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Image_Tools;

import java.io.File;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.EOFException;
import javax.imageio.ImageIO;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.MemoryCacheImageInputStream;
import java.nio.ByteOrder;
import java.util.Hashtable;
import java.net.UnknownHostException;

import PIRL.PVL.*;
import PIRL.Strings.String_Buffer;

//	Main dependencies.
import PIRL.Viewers.Parameter_View;
import java.util.Vector;
import java.util.Iterator;


/**	<i>JP2_Info</i> provides a set of PVL Parameters that describe the
	contents of a JP2 file.
<p>
	The JP2 file format is defined in Part 1 of the JPEG2000 International
	Standard (ISO/IEC 15444-1:2004(E); "Information technology - JPEG 2000
	image coding system: Core coding system"). This file format is composed
	entirely of a contiguous sequence of boxes. Each box has a type and
	length. Boxes may be nested within other boxes.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.39
	@see		PIRL.Image_Tools.JPEG2000_Info
	@see		PIRL.Image_Tools.JPEG2000_Codestream_Info
*/
public class JP2_Info
	extends JPEG2000_Info
{
public static final String
	ID = "PIRL.Image_Tools.JP2_Info (1.39 2012/11/19 00:13:25)";

//!	Box Type codes.
public static final int
	SIGNATURE_TYPE							= 0x6A502020,
	FILE_TYPE								= 0x66747970,
	HEADER_TYPE								= 0x6A703268,
		IMAGE_HEADER_TYPE					= 0x69686472,
		BITS_PER_COMPONENT_TYPE				= 0x62706363,
		COLOUR_SPECIFICATION_TYPE			= 0x636F6C72,
		PALETTE_TYPE						= 0x70636C72,
		COMPONENT_MAPPING_TYPE				= 0x636D6170,
		CHANNEL_DEFINITION_TYPE				= 0x63646566,
		RESOLUTION_TYPE						= 0x72657320,
			CAPTURE_RESOLUTION_TYPE			= 0x72657363,
			DEFAULT_DISPLAY_RESOLUTION_TYPE	= 0x72657364,
	CONTIGUOUS_CODESTREAM_TYPE				= 0x6A703263,
	INTELLECTUAL_PROPERTY_TYPE				= 0x6A703269,
	XML_TYPE								= 0x786D6C20,
	UUID_TYPE								= 0x75756964,
	UUID_INFO_TYPE							= 0x75696E66,
		UUID_LIST_TYPE						= 0x756C7374,
		URL_TYPE							= 0x75726C20,
	ASSOCIATION_TYPE						= 0x61736F63,
	LABEL_TYPE								= 0X6C626C20,
	PLACEHOLDER_TYPE						= 0x70686C64;

//!	Box names.
public static final String
	SIGNATURE_NAME							= "Signature",
	FILE_NAME								= "File_Type",
	HEADER_NAME								= "JP2_Header",
		IMAGE_HEADER_NAME					= "Image_Header",
		BITS_PER_COMPONENT_NAME				= "Bits_Per_Component",
		COLOUR_SPECIFICATION_NAME			= "Colour_Specification",
		PALETTE_NAME						= "Palette",
		COMPONENT_MAPPING_NAME				= "Component_Mapping",
		CHANNEL_DEFINITION_NAME				= "Channel_Definition",
		RESOLUTION_NAME						= "Resolution",
			CAPTURE_RESOLUTION_NAME			= "Capture_Resolution",
			DEFAULT_DISPLAY_RESOLUTION_NAME	= "Default_Display_Resolution",
	CONTIGUOUS_CODESTREAM_NAME				= "Contiguous_Codestream",
	INTELLECTUAL_PROPERTY_NAME				= "Intellectual_Property",
	XML_NAME								= "XML",
	UUID_NAME								= "UUID",
	UUID_INFO_NAME							= "UUID_Info",
		UUID_LIST_NAME						= "UUID_List",
		URL_NAME							= "URL",
	ASSOCIATION_NAME						= "Association",
	LABEL_NAME								= "Label",
	PLACEHOLDER_NAME						= "Placeholder";

private static Hashtable
	Box_Names								= new Hashtable ();
static
	{
	Box_Names.put (new Integer (SIGNATURE_TYPE),
								SIGNATURE_NAME);
	Box_Names.put (new Integer (FILE_TYPE),
								FILE_NAME);
	Box_Names.put (new Integer (HEADER_TYPE),
								HEADER_NAME);
	Box_Names.put (new Integer (IMAGE_HEADER_TYPE),
								IMAGE_HEADER_NAME);
	Box_Names.put (new Integer (BITS_PER_COMPONENT_TYPE),
								BITS_PER_COMPONENT_NAME);
	Box_Names.put (new Integer (COLOUR_SPECIFICATION_TYPE),
								COLOUR_SPECIFICATION_NAME);
	Box_Names.put (new Integer (PALETTE_TYPE),
								PALETTE_NAME);
	Box_Names.put (new Integer (COMPONENT_MAPPING_TYPE),
								COMPONENT_MAPPING_NAME);
	Box_Names.put (new Integer (CHANNEL_DEFINITION_TYPE),
								CHANNEL_DEFINITION_NAME);
	Box_Names.put (new Integer (RESOLUTION_TYPE),
								RESOLUTION_NAME);
	Box_Names.put (new Integer (CAPTURE_RESOLUTION_TYPE),
								CAPTURE_RESOLUTION_NAME);
	Box_Names.put (new Integer (DEFAULT_DISPLAY_RESOLUTION_TYPE),
								DEFAULT_DISPLAY_RESOLUTION_NAME);
	Box_Names.put (new Integer (CONTIGUOUS_CODESTREAM_TYPE),
								CONTIGUOUS_CODESTREAM_NAME);
	Box_Names.put (new Integer (INTELLECTUAL_PROPERTY_TYPE),
								INTELLECTUAL_PROPERTY_NAME);
	Box_Names.put (new Integer (XML_TYPE),
								XML_NAME);
	Box_Names.put (new Integer (UUID_TYPE),
								UUID_NAME);
	Box_Names.put (new Integer (UUID_INFO_TYPE),
								UUID_INFO_NAME);
	Box_Names.put (new Integer (UUID_LIST_TYPE),
								UUID_LIST_NAME);
	Box_Names.put (new Integer (URL_TYPE),
								URL_NAME);
	Box_Names.put (new Integer (ASSOCIATION_TYPE),
								ASSOCIATION_NAME);
	Box_Names.put (new Integer (LABEL_TYPE),
								LABEL_NAME);
	Box_Names.put (new Integer (PLACEHOLDER_TYPE),
								PLACEHOLDER_NAME);
	}

//!	Box characterization parameter names.
public static final String
	NAME_PARAMETER							= "Name",
	TYPE_PARAMETER							= "Type";

//!	Signature box parameter.
public static final String
	SIGNATURE_PARAMETER						= "Signature";
//!	Signature value.
public static final int
	SIGNATURE								= 0x0D0A870A;

//!	File Type box parameters.
public static final String
	BRAND_PARAMETER							= "Brand",
	MINOR_VERSION_PARAMETER					= "Minor_Version",
	COMPATIBILITY_LIST_PARAMETER			= "Compatibility";
//!	JP2 compatibility value.
public static final String
	JP2_COMPATIBLE							= "jp2 ";

//!	Image Header box parameters.
public static final String
	HEIGHT_PARAMETER						= "Height",
	WIDTH_PARAMETER							= "Width",
	COMPRESSION_TYPE_PARAMETER				= "Compression_Type",
	COLORSPACE_UNKNOWN_PARAMETER			= "Colorspace_Unknown",
	INTELLECTUAL_PROPERTY_PARAMETER			= "Intellectual_Property_Rights";

//!	Colour Specification box parameters.
public static final String
	SPECIFICATION_METHOD_PARAMETER			= "Specification_Method",
	PRECEDENCE_PARAMETER					= "Precedence",
	COLOURSPACE_APPROXIMATION_PARAMETER		= "Colourspace_Approximation",
	ENUMERATED_COLOURSPACE_PARAMETER		= "Enumerated_Colourspace";
//!	Specification method values.
public static final int
	ENUMERATED_COLOURSPACE					= 1,
	RESTRICTED_ICC_PROFILE					= 2;

//!	Palette box parameters.
public static final String
	ENTRIES_PARAMETER						= "Entries",
	COLUMNS_PARAMETER						= "Columns",
	VALUES_PARAMETER						= "Values";

//!	Component Mapping and Channel Definition parameter.
public static final String
	MAP_PARAMETER							= "Map";
//!	Component map indices.
public static final int
	COMPONENT_INDEX							= 0,
	MAP_TYPE_INDEX							= 1,
	PALETTE_INDEX							= 2;
//!	Map type values.
public static final int
	DIRECT_USE								= 0,
	PALETTE_MAPPING							= 1;

//!	Channel Definition indices.
public static final int
	CHANNEL_INDEX							= 0,
	CHANNEL_TYPE_INDEX						= 1,
	CHANNEL_ASSOCIATION_INDEX				= 2;
//!	Channel type values.
public static final int
	COLOUR_IMAGE_DATA						= 0,
	OPACITY									= 1,
	PREMULTIPLIED_OPACITY					= 2;
//!	Channel association values.
public static final int
	IMAGE_ASSOCIATION						= 0,
	NO_ASSOCIATION							= 0xFFFF;

//!	Capture and Default Resolution parameters.
public static final String
	VERTICAL_NUMERATOR_PARAMETER			= "Vertical_Numerator",
	VERTICAL_DENOMINATOR_PARAMETER			= "Vertical_Denominator",
	VERTICAL_EXPONENT_PARAMETER				= "Vertical_Exponent",
	HORIZONTAL_NUMERATOR_PARAMETER			= "Horizontal_Numerator",
	HORIZONTAL_DENOMINATOR_PARAMETER		= "Horizontal_Denominator",
	HORIZONTAL_EXPONENT_PARAMETER			= "Horizontal_Exponent";

//!	Codestream parameters.
public static final String
	CODESTREAM_PARAMETER					= "Codestream";
private JPEG2000_Codestream_Info
	Codestream_Info							=  null;

//!	UUID parameter.
public static final String
	UUID_PARAMETER							= "UUID";
//! UUID value count.
public static final int
	NUMBER_OF_UUID_VALUES					= 16;

//!	URL parameters.
public static final String
	VERSION_PARAMETER						= "Version",
	FLAGS_PARAMETER							= "Flags",
	URL_PARAMETER							= "URL";

//!	Text parameters.
public static final String
	TEXT_PARAMETER							= "Text";

//!	JPIP Placeholder parameters.
public static final String
	ORIGINAL_BOX_PARAMETER					= "Original_Box",
	BIN_ID_PARAMETER						= "Bin_ID",
	EQUIVALENT_BOX_PARAMETER				= "Equivalent_Box",
	CODESTREAM_ID_PARAMETER					= "Codestream_ID",
	TOTAL_CODESTREAMS_PARAMETER				= "Total_Codestreams",
	EXTENDED_BOX_LIST_PARAMETER				= "Extended_Box_List";
//!	Placeholder box flags value bit masks.
public static final int
	PLACEHOLDER_FLAGS_ORIGINAL_MASK				= (1 << 0),
	PLACEHOLDER_FLAGS_EQUIVALENT_MASK			= (1 << 1),
	PLACEHOLDER_FLAGS_CODESTREAM_MASK			= (1 << 2),
	PLACEHOLDER_FLAGS_MULTIPLE_CODESTREAM_MASK	= (1 << 3);

//------------------------------------------------------------------------------
//!	JP2 file validity flags.
protected int
	JP2_Validity							= 0;
//!	JP2 file validity flags.
public static final int
	SIGNATURE_FOUND							= 1 << 0,
	FILE_TYPE_FOUND							= 1 << 1,
	JP2_HEADER_FOUND						= 1 << 2,
	IMAGE_HEADER_FOUND						= 1 << 3,
	COLOUR_SPECIFICATION_FOUND				= 1 << 4,
	CONTIGUOUS_CODESTREAM_FOUND				= 1 << 5,
	JP2_COMPLETE							= SIGNATURE_FOUND |
											  FILE_TYPE_FOUND |
											  JP2_HEADER_FOUND |
											  IMAGE_HEADER_FOUND |
											  COLOUR_SPECIFICATION_FOUND |
											  CONTIGUOUS_CODESTREAM_FOUND;

//	Commonly accessed JP2 info:
private long
	Image_Width								= -1,
	Image_Height							= -1;
private int
	Total_Components						= -1;
private int[]
	Bits_per_Component						= new int[0];


//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_PARAMETERS		= 1 << 2,
	DEBUG_HELPERS			= 1 << 3,
	DEBUG_MAIN				= 1 << 4,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct JP2_Info for a named source.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	source	The source from which to obtain the JP2_Info.
	@throws	UnknownHostException If the source is a URL but the
		hostname is specifies can not be resolved.
	@throws IllegalArgumentException If the source is a URL but does
		not specify the http protocol or a source pahtname.
	@throws	IOException	If there was a problem reading the source.
*/
public JP2_Info
	(
	String	source
	)
	throws IOException, UnknownHostException, IllegalArgumentException
{this (source, Skip_Tiles_Default);}

/**	Construct JP2_Info for a named source.
<p>
	@param	source	The source from which to obtain the JP2_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	UnknownHostException If the source is a URL but the
		hostname is specifies can not be resolved.
	@throws IllegalArgumentException If the source is a URL but does
		not specify the http protocol or a source pathname.
	@throws	IOException	If there was a problem reading the source.
*/
public JP2_Info
	(
	String	source,
	boolean	skip_tiles
	)
	throws IOException, UnknownHostException, IllegalArgumentException
{
super (source);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JP2_Info: " + source);
Read (Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JP2_Info");
}

/**	Construct JP2_Info for a File.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	file	The File from which to obtain the JP2_Info.
	@throws	IOException	If there was a problem reading the file.
*/
public JP2_Info
	(
	File	file
	)
	throws IOException
{this (file, Skip_Tiles_Default);}

/**	Construct JP2_Info for a File.
<p>
	@param	file	The File from which to obtain the JP2_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If there was a problem reading the file.
*/
public JP2_Info
	(
	File	file,
	boolean	skip_tiles
	)
	throws IOException
{
super (file);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JP2_Info: " + file);
Read (Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JP2_Info");
}

/**	Construct JP2_Info for an ImageInputStream.
<p>
	The inclusion of tile segments in the Parameters is controlled by the
	{@link JPEG2000_Info#Skip_Tiles_Default} flag.
<p>
	@param	image_input_stream	The ImageInputStream from which to obtain
		the JP2_Info.
	@throws	IOException	If there was a problem reading the file.
*/
public JP2_Info
	(
	ImageInputStream	image_input_stream
	)
	throws IOException
{this (image_input_stream, Skip_Tiles_Default);}

/**	Construct JP2_Info for an ImageInputStream.
<p>
	@param	image_input_stream	The ImageInputStream from which to obtain
		the JP2_Info.
	@param	skip_tiles	true if tile segments are not to be included
		in the Parameters; false otherwise.
	@throws	IOException	If there was a problem reading the file.
*/
public JP2_Info
	(
	ImageInputStream	image_input_stream,
	boolean				skip_tiles
	)
	throws IOException
{
super (image_input_stream);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> JP2_Info: " + image_input_stream);
Read (Skip_Tiles = skip_tiles);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< JP2_Info");
}

/**	Construct an empty JP2_Info Object.
<p>
	Use one of the Source methods to assign a source file. Then use a Read
	method to obtain the JP2_Info Parameter set for the file.
<p>
	@see	JPEG2000_Info#Source(String)
	@see	#Read(boolean)
*/
public JP2_Info ()
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< JP2_Info");
}

/*==============================================================================
	Accessors
*/
/**	Read all available information from the Source file.
<p>
	If no Source file has been assigned, nothing is done.
<p>
	The Source is read from the beginning. The Source stream will be
	repositioned back to its current position before the method returns.
<p>
	Any information Parameters currently present are removed. Then the
	file is scanned for JP2 information boxes. Each box found is used to
	define the contents of a Parameter Group added to the base
	JPEG2000_Info Aggregate. The name of each Parameter Group is the JP2
	box type composed of four characters (unprintable characters are
	represented by escape sequences). The Group always contains the
	following parameters:
<p>
<dl>
<dt>Name
	<dd>The box type name in descriptive token form.
<dt>Type
	<dd>The box type code in numeric hexadecimal form.
<dt>^Position
	<dd>The position of the box in the Source as a byte offset (from 0).
	<b>N.B.</b>: This parameter is not present if {@link
	JPEG2000_Info#Use_Data_Position(boolean) data position inclusion} is
	disabled.
<dt>Length
	<dd>The length of the box in bytes.
</dl>
<p>
	The remaining Parameters in each Group depend on the box type.
<p>
	When JP2 information boxes are nested, the corresponding Parameter
	Groups are also nested. JPEG2000 codestream (Contiguous_Codestream,
	jp2c) boxes contain a Codestream Parameter Group that is defined by a
	JPEG2000_Codestream_Info object.
<p>
	<b>N.B.</b>: A {@link #WARNING_PARAMETER} will be placed at the end
	of the Parameter set and further Source file scanning halted if the
	previous JP2 box is invalid. The Value of the warning Parameter will
	be a String describing the problem. Possible sources of warning
	parameters are: The beginning of the Source file does not contain a
	valid Signature box or the second box of the Source file is not a
	valid File Type box.
<p>
	@param	skip_tiles	true if tile segments are not to be included
		in the Codestream Parameters; false otherwise.
	@return	This JP2_Info.
	@throws	IOException	If there was a problem reading the Source file.
	@see	JPEG2000_Codestream_Info
*/
public JP2_Info Read
	(
	boolean	skip_tiles
	)
	throws IOException
{
if (Image_Input_Stream == null)
	return this;
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> JP2_Info.Read");
Image_Input_Stream.mark ();
Stream_Position (0);

ByteOrder
	byte_order = Image_Input_Stream.getByteOrder ();
if (byte_order != ByteOrder.BIG_ENDIAN)
	Image_Input_Stream.setByteOrder (ByteOrder.BIG_ENDIAN);

//	Remove any and all current parameters.
Remove_All ();

//	Reset all image characterization values.
Reset_Image_Values ();

//	Add the source file length.
long
	length = -1;
try {length = Image_Input_Stream.length ();}
catch (IOException exception) {}
if (length >= 0)
	{
	try
		{
		Add (new Parameter (DATA_LENGTH_PARAMETER)
			.Value (new Value (length).Units ("bytes"))
			.Comments ("\nTotal source file length."));
		}
	catch (PVL_Exception exception) {}
	}

boolean
	tile_skipping = Skip_Tiles;
Skip_Tiles = skip_tiles;
try {Add_Boxes (this, 0);}
catch (IOException exception)
	{
	throw IO_Error
		("Unable to Read JP2 Box information from the Source"
			+ " at position " + Stream_Position,
		exception);
	}
catch (IllegalStateException exception)
	{
	try {Add (new Parameter (WARNING_PARAMETER)
			.Value (exception.getMessage ()));}
	catch (PVL_Exception except) {}
	}
finally
	{
	Image_Input_Stream.setByteOrder (byte_order);
	Skip_Tiles = tile_skipping;
	try {Image_Input_Stream.reset ();}
	catch (IOException exception)
		{
		throw IO_Error ("Could not reset the original Source position.",
			exception);
		}
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< JP2_Info.Read");
return this;
}

/**	Test if the Source is a file in JP2 format.
<p>
	If the {@link #JP2_Validity() JP2 Validity} value has already been
	determined by {@link #Read(boolean) read}ing the Source file, then if
	both the {@link #SIGNATURE_FOUND} and {@link #FILE_TYPE_FOUND}
	validity flags have been set the file is in JP2 format. Otherwise an
	attempt is made to read the first two JP2 boxes from the beginning of
	the file which will provide the minimally necessary validity flags.
	In this case the Source is repositioned back to its previous position
	after the attempt to read the boxes. If no Source as been assigned
	then the return value is false.
<p>
	@return	true if the beginning of the Source file contains a valid
		signature box followed immediately by a valid file type box; false
		otherwise.
*/
public boolean Is_JP2 ()
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> JP2_Info.Is_JP2");
boolean
	is_JP2 = false;
if (Image_Input_Stream != null)
	{
	if (JP2_Validity == 0)
		{
		Image_Input_Stream.mark ();
		ByteOrder
			byte_order = Image_Input_Stream.getByteOrder ();
		if (byte_order != ByteOrder.BIG_ENDIAN)
			Image_Input_Stream.setByteOrder (ByteOrder.BIG_ENDIAN);
		try
			{
			Image_Input_Stream.seek (0);
			Stream_Position = 0;
			//	Signature.
			Box_Parameter ();
			//	File Type.
			Box_Parameter ();
			}
		catch (Exception exception) {}
		finally
			{
			Image_Input_Stream.setByteOrder (byte_order);
			try {Image_Input_Stream.reset ();}
			catch (IOException exception) {}
			}
		}
	is_JP2 = (JP2_Validity & (SIGNATURE_FOUND | FILE_TYPE_FOUND)) != 0;
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< JP2_Info.Is_JP2 : " + is_JP2);
return is_JP2;
}

/**	Test if the Source file contains a complete JP2 format.
<p>
	The Source must be {@link #Read(boolean) read} for this test to be valid.
<p>
	@return	true if the {@link #JP2_Validity() JP2 Validity} value has
		all the {@link #JP2_COMPLETE} flags set; false otherwise.
*/
public boolean Is_Complete_JP2 ()
{return (JP2_Validity & JP2_COMPLETE) == JP2_COMPLETE;}

/**	Get the current JP2 validity flags.
<p>
	The bit flags in this value are updated as the Source file is
	{@link #Read(boolean) read}:
<p>
<dl>
<dt>{@link #SIGNATURE_FOUND}
	<dd>Set if the beginning of the file contains a Signature box.
<dt>{@link #FILE_TYPE_FOUND}
	<dd>Set if the second box of the file is a File Type box.
<dt>{@link #JP2_HEADER_FOUND}
	<dd>Set if the required JP2 Header box has been found.
<dt>{@link #IMAGE_HEADER_FOUND}
	<dd>Set if the required Image Header box has been found.
<dt>{@link #COLOUR_SPECIFICATION_FOUND}
	<dd>Set if the required Colour Specification box has been found.
<dt>{@link #CONTIGUOUS_CODESTREAM_FOUND}
	<dd>Set if a JPEG2000 Contiguous Codestream box has been found.
</dl>
<p>
	@return	The JP2 validity flags value.
*/
public int JP2_Validity ()
{return JP2_Validity;}

/**	Get the number of bits per pixel for each component in the image.
<p>
	@return	An int array containing the number of bits per pixel for each
		image component, as determined from the Image Header box. A
		negative bits per pixel value signals that the pixel values are
		signed; the number of bits per pixel is abs (value). The array
		will be empty if the values are not yet known.
*/
public int[] Bits_per_Component ()
{return Bits_per_Component;}

//!	Reset all image characterization class data members.
private void Reset_Image_Values ()
{
JP2_Validity			= 0;
Image_Width				=
Image_Height			= -1;
Total_Components		= -1;
Bits_per_Component		= new int[0];
}

/*------------------------------------------------------------------------------
	Codestream values
*/
/**	Get the {@link JPEG2000_Codestream_Info#Codestream_Validity()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Codestream_Validity value, or 0
		if the value is not yet known.
*/
public int Codestream_Validity ()
{return (Codestream_Info == null) ? 0 : Codestream_Info.Codestream_Validity ();}

/**	Get the number of components (bands) in the image.
<p>
	If the value has not been obtained from the Image Header box the
	value will be obtained from a {@link
	JPEG2000_Codestream_Info#Total_Components() JPEG2000 codestream}
	(Contiguous_Codestream, jp2c) box if available.
<p>
	@return	The number of image components, as determined from the Image
		Header box. This will be -1 if the value is not yet known.
*/
public int Total_Components ()
{
if (Total_Components < 0 &&
	Codestream_Info != null)
	return Codestream_Info.Total_Components ();
return Total_Components;
}

/**	Get the width of the image.
<p>
	The image width is the distance on the reference grid from the
	{@link JPEG2000_Codestream_Info#Horizontal_Image_Offset() horizontal
	image offset} to the {@link
	JPEG2000_Codestream_Info#Reference_Grid_Width() reference grid
	width}. If the value has not been obtained from the Image Header box
	the value will be obtained from a {@link
	JPEG2000_Codestream_Info#Image_Width() JPEG2000 codestream}
	(Contiguous_Codestream, jp2c) box if available.
<p>
	<b>N.B.</b>: The image width is measured in terms of reference grid
	columns. However, an individual image pixel may occupy more than one
	reference grid column. The {@link
	JPEG2000_Codestream_Info#Pixel_Width() pixel width} may be
	different for each image component.
<p>
	@return	The reference grid width of the image. This will be -1 if the
		value is not yet known.
*/
public long Image_Width ()
{
if (Image_Width < 0 &&
	Codestream_Info != null)
	return Codestream_Info.Image_Width ();
return Image_Width;
}

/**	Get the height of the image.
<p>
	The image height is the distance on the reference grid from the
	{@link JPEG2000_Codestream_Info#Vertical_Image_Offset() vertical
	image offset} to the {@link
	JPEG2000_Codestream_Info#Reference_Grid_Height() reference grid
	height}. If the value has not been obtained from the Image Header
	box the value will be obtained from a {@link
	JPEG2000_Codestream_Info#Image_Height() JPEG2000 codestream}
	(Contiguous_Codestream, jp2c) box if available.
<p>
	<b>N.B.</b>: The image height is measured in terms of reference grid
	rows. However, an individual image pixel may occupy more than one
	reference grid row. The {@link
	JPEG2000_Codestream_Info#Pixel_Height() pixel height} may be
	different for each image component.
<p> 
	@return	The reference grid height of the image. This will be -1 if the
		value is not yet known.
*/
public long Image_Height ()
{
if (Image_Height < 0 &&
	Codestream_Info != null)
	return Codestream_Info.Image_Height ();
return Image_Height;
}

/**	Get the {@link JPEG2000_Codestream_Info#Reference_Grid_Width()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Reference_Grid_Width value, or -1 if the
		value is not yet known.
*/
public long Reference_Grid_Width ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Reference_Grid_Width ();}

/**	Get the {@link JPEG2000_Codestream_Info#Reference_Grid_Height()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Reference_Grid_Height value, or -1 if the
		value is not yet known.
*/
public long Reference_Grid_Height ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Reference_Grid_Height ();}

/**	Get the {@link JPEG2000_Codestream_Info#Horizontal_Image_Offset()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Horizontal_Image_Offset value, or -1 if the
		value is not yet known.
*/
public long Horizontal_Image_Offset ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Horizontal_Image_Offset ();}

/**	Get the {@link JPEG2000_Codestream_Info#Vertical_Image_Offset()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Vertical_Image_Offset value, or -1 if the
		value is not yet known.
*/
public long Vertical_Image_Offset ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Vertical_Image_Offset ();}

/**	Get the {@link JPEG2000_Codestream_Info#Pixel_Width()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Pixel_Width value, or an empty
		int array if the value is not yet known.
*/
public int[] Pixel_Width ()
{return (Codestream_Info == null) ? new int[0] : Codestream_Info.Pixel_Width ();}

/**	Get the {@link JPEG2000_Codestream_Info#Pixel_Height()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Pixel_Height value, or an empty
		int array if the value is not yet known.
*/
public int[] Pixel_Height ()
{return (Codestream_Info == null) ? new int[0] : Codestream_Info.Pixel_Height ();}

/**	Get the {@link JPEG2000_Codestream_Info#Tile_Width()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Tile_Width value, or -1 if the
		value is not yet known.
*/
public long Tile_Width ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Tile_Width ();}

/**	Get the {@link JPEG2000_Codestream_Info#Tile_Height()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Tile_Height value, or -1 if the
		value is not yet known.
*/
public long Tile_Height ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Tile_Height ();}

/**	Get the {@link JPEG2000_Codestream_Info#Horizontal_Tile_Offset()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Horizontal_Tile_Offset value,
		or -1 if the value is not yet known.
*/
public long Horizontal_Tile_Offset ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Horizontal_Tile_Offset ();}

/**	Get the {@link JPEG2000_Codestream_Info#Vertical_Tile_Offset()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Vertical_Tile_Offset value,
		or -1 if the value is not yet known.
*/
public long Vertical_Tile_Offset ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Vertical_Tile_Offset ();}

/**	Get the {@link JPEG2000_Codestream_Info#Total_Tiles()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Total_Tiles value,
		or -1 if the value is not yet known.
*/
public int Total_Tiles ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Total_Tiles ();}

/**	Test if a {@link JPEG2000_Codestream_Info#Zero_Length_Tile_Part() zero
	length tile-part} was found in the codestream.
<p>
	@return	true if a zero length tile-part was found; false otherwise.
*/
public boolean Zero_Length_Tile_Part ()
{return  (Codestream_Info == null) ? false : Codestream_Info.Zero_Length_Tile_Part ();}

/**	Get the {@link JPEG2000_Codestream_Info#Total_Quality_Layers()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Total_Quality_Layers value, or -1 if
		the value is not yet known.
*/
public int Total_Quality_Layers ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Total_Quality_Layers ();}

/**	Get the {@link JPEG2000_Codestream_Info#Progression_Order()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Progression_Order value, or -1
		if the value is not yet known.
*/
public int Progression_Order ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Progression_Order ();}

/**	Get the {@link JPEG2000_Codestream_Info#Total_Resolution_Levels()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Total_Resolution_Levels value, or -1 if
		the value is not yet known.
*/
public int Total_Resolution_Levels ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Total_Resolution_Levels ();}

/**	Get the {@link JPEG2000_Codestream_Info#Transform()} value.
<p>
	@return	The JPEG2000_Codestream_Info.Transform value, or -1 if the
		value is not yet known.
*/
public int Transform ()
{return (Codestream_Info == null) ? -1 : Codestream_Info.Transform ();}

/*==============================================================================
	Box Parameters
*/
/**	Assemble a Parameter Aggregate describing a JP2 box from a byte array.
<p>
	A single JP2 box is parsed from the byte data. If the data contains
	additional JP2 box contents they are ignored. {@link
	JPEG2000_Info#Use_Data_Position(boolean) Including data position}
	parameters is disabled as is {@link
	JPEG2000_Info#Throw_Illegal_State(boolean) illegal box organizaion
	detection}.
<p>
	@param	data	A byte array.
	@param	offset	The offset where the JP2 box is expected to start.
	@param	length	The maximum amount of byte data to be read. However,
		if the data array ends before length bytes after the starting
		offset, then only the available bytes will be read.
	@return	A Parameter Aggregate containing Parameters describing the
		JP2 box contents. This will be null if the contents of a recognized
		box were not available within the data length.
*/
public static Parameter Box
	(
	byte[]	data,
	int		offset,
	int		length
	)
{
if (data == null)
	return null;
JP2_Info
	info = new JP2_Info ();
info.Source (new MemoryCacheImageInputStream (new ByteArrayInputStream
	(data, offset, length)));
info.Source ().setByteOrder (ByteOrder.BIG_ENDIAN);
info.Use_Data_Position (false);
info.Throw_Illegal_State (false);
Parameter
	box = null;
try {box = info.Box_Parameter ();}
catch (Exception exception) {}
return box;
}

/**	Assemble a Parameter Aggregate from a byte array of JP2 box data.
<p>
	@param	data	A byte array.
	@return	A Parameter Aggregate containing Parameters describing the
		JP2 box contents. This will be null if the contents of a recognized
		box were not available within the data array.
	@see	#Box(byte[], int, int)
*/
public static Parameter Box
	(
	byte[]	data
	)
{return Box (data, 0, data.length);}


private void Add_Boxes
	(
	Parameter	container,
	long		end_position
	)
	throws IOException, IllegalStateException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JP2_Info.Add_Boxes: To " + container.Name () + '\n'
		+"     From stream position " + Stream_Position + '\n'
		+"    Until stream position " + end_position);
Parameter
	box = null;
if (end_position == 0)
	end_position = Long.MAX_VALUE;
try
	{
	while (Stream_Position < end_position &&
			(box = Box_Parameter ()) != null)
		{
		try {container.Add (box);}
		catch (PVL_Exception exception) {}
		}
	}
catch (EOFException exception)
	{
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    JP2_Info.Add_Boxes: EOF at Stream_Postion "
				+ Stream_Position);
	}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JP2_Info.Add_Boxes");
}


private Parameter Box_Parameter ()
	throws IOException, IllegalStateException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JP2_Info.Box_Parameter");
long
	box_position = Stream_Position,
	box_length = Read_Unsigned_int (LENGTH_PARAMETER);
int
	type_code = Read_int (TYPE_PARAMETER);
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("    JP2_Info.Box_Parameter: \"" + int_to_String (type_code)
			+ "\" (0x" + Integer.toHexString (type_code)
			+ ") - " + Box_Name (type_code) + '\n'
		+"    Position = " + box_position + '\n'
		+"      Length = " + box_length);

if (box_length == 0)
	{
	try
		{
		long
			stream_length = Image_Input_Stream.length ();
		box_length = stream_length - box_position;
		if ((DEBUG & DEBUG_PARAMETERS) != 0)
			System.out.println
				("    Stream length = " + stream_length + '\n'
				+"    Remaining length = " + box_length);
		}
	catch (IOException exception) {}
	}
else if (box_length == 1)
	{
	box_length = Read_long ("box long length");
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    Long length = " + box_length);
	}

Parameter
	box = new Parameter (int_to_String (type_code));
try
	{
	box
		.Add (new Parameter (NAME_PARAMETER)
			.Value (Box_Name (type_code)))
		.Add (new Parameter (TYPE_PARAMETER)
			.Value (new Value (type_code).Base (16)));
	if (Use_Data_Position)
		box
			.Add (new Parameter (POSITION_PARAMETER)
				.Value (new Value (box_position).Units ("byte offset")));
	box
		.Add (new Parameter (LENGTH_PARAMETER)
			.Value (new Value (box_length).Units ("bytes")));
	}
catch (PVL_Exception exception) {}

if (JP2_Validity == 0)
	{
	//	The first box must be the Signature.
	if (type_code != SIGNATURE_TYPE ||
		box_length != 12)
		{
		if (Throw_Illegal_State)
			throw new IllegalStateException (ID + '\n'
				+ "Not a valid JP2 file: No initial signature box.");
		}
	else
		JP2_Validity = SIGNATURE_FOUND;
	}
else if (JP2_Validity == SIGNATURE_FOUND)
	{
	//	The second box must be the File Type.
	if (type_code != FILE_TYPE)
		{
		if (Throw_Illegal_State)
			throw new IllegalStateException (ID + '\n'
				+ "Not a valid JP2 file: Missing File Type box.");
		}
	else
		JP2_Validity |= FILE_TYPE_FOUND;
	}

long
	end_position = box_position + box_length;
switch (type_code)
	{
	//	Super boxes that contain additional boxes.
	case HEADER_TYPE:
		JP2_Validity |= JP2_HEADER_FOUND;
	case RESOLUTION_TYPE:
	case UUID_INFO_TYPE:
	case ASSOCIATION_TYPE:
		Add_Boxes (box, end_position);
		break;

	//	Basic boxes.
	case SIGNATURE_TYPE:
		Signature_Parameters (box);
		break;
	case FILE_TYPE:
		File_Type_Parameters (box, end_position);
		break;
	case IMAGE_HEADER_TYPE:
		JP2_Validity |= IMAGE_HEADER_FOUND;
		Image_Header_Parameters (box);
		break;
	case BITS_PER_COMPONENT_TYPE:
		Bits_per_Component_Parameters (box, end_position);
		break;
	case COLOUR_SPECIFICATION_TYPE:
		JP2_Validity |= COLOUR_SPECIFICATION_FOUND;
		Colour_Specification_Parameters (box, end_position);
		break;
	case PALETTE_TYPE:
		Palette_Parameters (box, end_position);
		break;
	case COMPONENT_MAPPING_TYPE:
		Component_Mapping_Parameters (box, end_position);
		break;
	case CHANNEL_DEFINITION_TYPE:
		Channel_Definition_Parameters (box);
		break;
	case CAPTURE_RESOLUTION_TYPE:
	case DEFAULT_DISPLAY_RESOLUTION_TYPE:
		Resolution_Parameters (box);
		break;
	case UUID_TYPE:
		UUID_Parameters (box, end_position);
		break;
	case UUID_LIST_TYPE:
		UUID_List_Parameters (box);
		break;
	case URL_TYPE:
		URL_Parameters (box, end_position);
		break;
	case LABEL_TYPE:
	case XML_TYPE:
		Text_Parameter (box, end_position);
		break;

	//	JPEG2000 codestream box.
	case CONTIGUOUS_CODESTREAM_TYPE:
		JP2_Validity |= CONTIGUOUS_CODESTREAM_FOUND;
		Contiguous_Codestream_Parameters (box, end_position);
		break;

	//	JPIP placeholder box.
	case PLACEHOLDER_TYPE:
		Placeholder_Parameters (box, end_position);
		break;

	//	Opaque boxes.
	case INTELLECTUAL_PROPERTY_TYPE:
	default:
		if (Use_Data_Position)
			Data_Position_Parameters (box, end_position);
		else
			Data_Offset_Parameters (box, 0, end_position - Stream_Position);
	}

if (Stream_Position < end_position)
	Stream_Position  (end_position);
else
if (Throw_Illegal_State &&
	Stream_Position > end_position)
	throw new IllegalStateException (ID + '\n'
		+ "Invalid " + box.Name () + " contents -\n"
		+ "  expected length " + box_length
			+ " but used length " + (Stream_Position - box_position));
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(((box == null) ? "" : box.Description ()) + '\n'
		+"<<< JP2_Info.Box_Parameter: " + box);
return box;
}


private void Signature_Parameters
	(
	Parameter	box
	)
	throws IOException, IllegalStateException
{
try {
int
	signature = Read_int (SIGNATURE_PARAMETER);
box.Add (new Parameter (SIGNATURE_PARAMETER)
	.Value (new Value (signature).Base (16)));
if (Throw_Illegal_State &&
	signature != SIGNATURE)
	throw new IllegalStateException (ID + '\n'
		+ "Invalid Signature: Expected " + SIGNATURE + ", found " + signature);
}
catch (PVL_Exception exception) {}
}


private void File_Type_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException, IllegalStateException
{
try {
boolean
	JP2_compatible = false;
String
	compatibility;
int
	datum = Read_int (BRAND_PARAMETER);
compatibility = int_to_String (datum);
if (compatibility.equals (JP2_COMPATIBLE))
	JP2_compatible = true;
box.Add (new Parameter (BRAND_PARAMETER)
		.Value (compatibility));
box.Add (new Parameter (MINOR_VERSION_PARAMETER)
	.Value (Read_Unsigned_int (MINOR_VERSION_PARAMETER)));

Value
	list = new Value ().Type (Value.SET);
while (Stream_Position < end_position)
	{
	datum = Read_int (COMPATIBILITY_LIST_PARAMETER);
	compatibility = int_to_String (datum);
	if (compatibility.equals (JP2_COMPATIBLE))
		JP2_compatible = true;
	list.Add (new Value (compatibility));
	}
box.Add (new Parameter (COMPATIBILITY_LIST_PARAMETER)
	.Value (list));

if (Throw_Illegal_State &&
	! JP2_compatible)
	throw new IllegalStateException (ID + '\n'
		+ "File Type Compatibility \"" + JP2_COMPATIBLE + "\" not found.");
}
catch (PVL_Exception exception) {}
}


private void Image_Header_Parameters
	(
	Parameter	box
	)
	throws IOException
{
try {
Parameter
	value_bits;
box
	.Add (new Parameter (HEIGHT_PARAMETER)
		.Value (new Value (Image_Height =
			Read_Unsigned_int (HEIGHT_PARAMETER)).Units ("rows")))
	.Add (new Parameter (WIDTH_PARAMETER)
		.Value (new Value (Image_Width =
			Read_Unsigned_int (WIDTH_PARAMETER)).Units ("columns")))
	.Add (new Parameter (TOTAL_COMPONENTS_PARAMETER)
		.Value (Total_Components =
			Read_Unsigned_short (TOTAL_COMPONENTS_PARAMETER)))
	.Add (value_bits = Value_Bits_Parameter (box.Name (), Stream_Position + 1))
	.Add (new Parameter (COMPRESSION_TYPE_PARAMETER)
		.Value (Read_Unsigned_byte (COMPRESSION_TYPE_PARAMETER)))
	.Add (new Parameter (COLORSPACE_UNKNOWN_PARAMETER)
		.Value ((Read_Unsigned_byte (COLORSPACE_UNKNOWN_PARAMETER) != 0) ?
			"true" : "false"))
	.Add (new Parameter (INTELLECTUAL_PROPERTY_PARAMETER)
		.Value ((Read_Unsigned_byte (INTELLECTUAL_PROPERTY_PARAMETER) != 0) ?
			"true" : "false"));

Bits_per_Component = int_Array (value_bits);
}
catch (PVL_Exception exception) {}
}


private void Bits_per_Component_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
Parameter
	value_bits;
box.Add (value_bits = Value_Bits_Parameter (box.Name (), end_position));
//	The values from this box overrides the values from the Image Header box.
Bits_per_Component = int_Array (value_bits);
}
catch (PVL_Exception exception) {}
}


private void Colour_Specification_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
long
	start_position = Stream_Position;
int
	method = Read_Unsigned_byte (SPECIFICATION_METHOD_PARAMETER);
box
	.Add (new Parameter (SPECIFICATION_METHOD_PARAMETER)
		.Value (method))
	.Add (new Parameter (PRECEDENCE_PARAMETER)
		.Value (Read_Unsigned_byte (PRECEDENCE_PARAMETER)))
	.Add (new Parameter (COLOURSPACE_APPROXIMATION_PARAMETER)
		.Value (Read_Unsigned_byte (COLOURSPACE_APPROXIMATION_PARAMETER)));
if (method == ENUMERATED_COLOURSPACE)
	box.Add (new Parameter (ENUMERATED_COLOURSPACE_PARAMETER)
		 .Value (Read_Unsigned_int (ENUMERATED_COLOURSPACE_PARAMETER)));
else
	{
	if (Use_Data_Position)
		Data_Position_Parameters (box, end_position);
	else
		Data_Offset_Parameters (box,
			Stream_Position - start_position, end_position - start_position);
	}
}
catch (PVL_Exception exception) {}
}


private void Palette_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> JP2_Info.Palette_Parameters:\n"
		+"     From stream position " + Stream_Position + '\n'
		+"    Until stream position " + end_position);
try {
int
	entries = Read_Unsigned_short (box.Name () + ' ' + ENTRIES_PARAMETER),
	columns = Read_Unsigned_byte (COLUMNS_PARAMETER);
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("    entries = " + entries + '\n'
		+"    columns = " + columns);
box.Add (new Parameter (ENTRIES_PARAMETER)
	.Value (entries));
box.Add (new Parameter (COLUMNS_PARAMETER)
	.Value (columns));
Parameter
	value_bits =
		Value_Bits_Parameter (box.Name (), Stream_Position + columns);
box.Add (value_bits);
int[]
	bits_per_column = int_Array (value_bits);
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("    bits_per_column = " + bits_per_column);
Value
	palette = new Value ().Type (Value.SEQUENCE),
	palette_row;
while (entries-- != 0)
	{
	palette_row = new Value ().Type (Value.SEQUENCE);
	for (int column = 0;
			 column < columns;
		   ++column)
		palette_row.Add (new Value
			(Read_Value (box.Name () + ' ' + VALUES_PARAMETER,
				bits_per_column[column])));
	palette.Add (palette_row);
	}
box.Add (new Parameter (VALUES_PARAMETER)
	.Value (palette));
}
catch (PVL_Exception exception) {}
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< JP2_Info.Palette_Parameters");
}


private void Component_Mapping_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
String
	description = box.Name () + ' ' + MAP_PARAMETER;
Value
	map = new Value ().Type (Value.SEQUENCE),
	set;
while (Stream_Position < end_position)
	{
	set = new Value ().Type (Value.SEQUENCE)
		.Add (new Value (Read_Unsigned_short (description +
			" component index")))
		.Add (new Value (Read_Unsigned_byte (description +
			" map type")))
		.Add (new Value (Read_Unsigned_byte (description +
			" palette index")));
	map.Add (set);
	}
box.Add (new Parameter (MAP_PARAMETER)
	.Comments ("\nMap entries:\n" +
		"  Component index,\n" +
		"  Map type,\n" +
		"  Palette index")
	.Value (map));
}
catch (PVL_Exception exception) {}
}


private void Channel_Definition_Parameters
	(
	Parameter	box
	)
	throws IOException
{
try {
int
	entries = Read_Unsigned_short (box.Name () + ' ' + ENTRIES_PARAMETER);
box.Add (new Parameter (ENTRIES_PARAMETER)
	.Value (entries));
String
	description = box.Name () + ' ' + MAP_PARAMETER;
Value
	map = new Value ().Type (Value.SEQUENCE),
	set;
while (entries-- != 0)
	{
	set = new Value ().Type (Value.SEQUENCE)
		.Add (new Value (Read_Unsigned_short (description +
			" channel index")))
		.Add (new Value (Read_Unsigned_short (description +
			" channel type")))
		.Add (new Value (Read_Unsigned_short (description +
			" channel association")));
	map.Add (set);
	}
box.Add (new Parameter (MAP_PARAMETER)
	.Comments ("\nChannel definition entries:\n" +
		"  Channel index,\n" +
		"  Channel type,\n" +
		"  Channel association")
	.Value (map));
}
catch (PVL_Exception exception) {}
}


private void Resolution_Parameters
	(
	Parameter	box
	)
	throws IOException
{
try {
box
	.Add (new Parameter (VERTICAL_NUMERATOR_PARAMETER)
		.Value (Read_Unsigned_short (VERTICAL_NUMERATOR_PARAMETER)))
	.Add (new Parameter (VERTICAL_DENOMINATOR_PARAMETER)
		.Value (Read_Unsigned_short (VERTICAL_DENOMINATOR_PARAMETER)));
int
	numerator = Read_Unsigned_short (HORIZONTAL_NUMERATOR_PARAMETER),
	denominator = Read_Unsigned_short (HORIZONTAL_DENOMINATOR_PARAMETER);
box
	.Add (new Parameter (VERTICAL_EXPONENT_PARAMETER)
		.Value (Read_byte (VERTICAL_EXPONENT_PARAMETER)))
	.Add (new Parameter (HORIZONTAL_NUMERATOR_PARAMETER)
		.Value (numerator))
	.Add (new Parameter (HORIZONTAL_DENOMINATOR_PARAMETER)
		.Value (denominator))
	.Add (new Parameter (HORIZONTAL_EXPONENT_PARAMETER)
		.Value (Read_byte (HORIZONTAL_EXPONENT_PARAMETER)));
}
catch (PVL_Exception exception) {}
}


private void UUID_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
long
	start_position = Stream_Position;
box.Add (UUID_Parameter (box.Name ()));
if (Use_Data_Position)
	Data_Position_Parameters (box, end_position);
else
	Data_Offset_Parameters (box,
		Stream_Position - start_position, end_position - start_position);
}
catch (PVL_Exception exception) {}
}


private void UUID_List_Parameters
	(
	Parameter	box
	)
	throws IOException
{
try {
int
	entries = Read_Unsigned_short (box.Name () + ' ' + ENTRIES_PARAMETER);
box.Add (new Parameter (ENTRIES_PARAMETER)
	.Value (entries));
while (entries-- != 0)
	box.Add (UUID_Parameter (box.Name ()));
}
catch (PVL_Exception exception) {}
}


private Parameter UUID_Parameter
	(
	String	description
	)
	throws IOException
{
String
	describe = description + ' ' + UUID_PARAMETER;
Parameter
	parameter = new Parameter (UUID_PARAMETER);
try {
Value
	uuid = new Value ().Type (Value.SEQUENCE);
for (int index = 0;
		 index < NUMBER_OF_UUID_VALUES;
	   ++index)
	uuid.Add (new Value (Read_Unsigned_byte (describe)).Base (16));
parameter.Value (uuid);
}
catch (PVL_Exception exception) {}
return parameter;
}


private void URL_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
box
	.Add (new Parameter (VERSION_PARAMETER)
		.Value (Read_Unsigned_byte (box.Name () + ' ' + VERSION_PARAMETER)))
	.Add (new Parameter (FLAGS_PARAMETER)
		.Value (new Value
			(Read_Value (box.Name () + ' ' + FLAGS_PARAMETER, 24))
		.Base (16)))
	.Add (new Parameter (URL_PARAMETER)
		.Value (Read_String (box.Name (), end_position)));
}
catch (PVL_Exception exception) {}
}


private void Text_Parameter
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
box
	.Add (new Parameter (TEXT_PARAMETER)
		.Value (new Value
			(Read_String (box.Name (), end_position))
		.Type (Value.TEXT)));
}
catch (PVL_Exception exception) {}
}


private void Contiguous_Codestream_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
if (Use_Data_Position)
	Data_Position_Parameters (box, end_position);
else
	Data_Offset_Parameters (box,
		0, end_position - Stream_Position);
try {
JPEG2000_Codestream_Info
	codestream_info = new JPEG2000_Codestream_Info
		(Image_Input_Stream, end_position, Skip_Tiles);
codestream_info.Name (CODESTREAM_PARAMETER);
box.Add (codestream_info);
if (Codestream_Info == null)
	Codestream_Info = codestream_info;
}
catch (PVL_Exception exception) {}
}


private void Placeholder_Parameters
	(
	Parameter	box,
	long		end_position
	)
	throws IOException
{
try {
long
	start_position = Stream_Position;
int
	flags = Read_int (box.Name () + ' ' + FLAGS_PARAMETER);
box
	.Add (new Parameter (FLAGS_PARAMETER)
		.Value (new Value (flags).Base (16))
		.Comments ("\n"
			+ (((flags & PLACEHOLDER_FLAGS_ORIGINAL_MASK) == 0) ?
				"No o" : "O") + "riginal box provided.\n"
			+ (((flags & PLACEHOLDER_FLAGS_EQUIVALENT_MASK) == 0) ?
				"No e" : "E") + "quivalent box provided.\n"
			+ (((flags & PLACEHOLDER_FLAGS_CODESTREAM_MASK) == 0) ?
				"No c" : "C") + "odestream provided."));
Parameter
	group;
if ((flags & PLACEHOLDER_FLAGS_ORIGINAL_MASK) != 0)
	{
	group = new Parameter (ORIGINAL_BOX_PARAMETER);
	group.Add (new Parameter (BIN_ID_PARAMETER)
		.Value (Read_long (box.Name ()
			+ ' ' + ORIGINAL_BOX_PARAMETER + ' ' + BIN_ID_PARAMETER)));
	Box_Header_Parameters (group);
	box.Add (group);
	}
if ((flags & PLACEHOLDER_FLAGS_EQUIVALENT_MASK) != 0)
	{
	group = new Parameter (EQUIVALENT_BOX_PARAMETER);
	group.Add (new Parameter (BIN_ID_PARAMETER)
		.Value (Read_long (box.Name ()
			+ ' ' + EQUIVALENT_BOX_PARAMETER + ' ' + BIN_ID_PARAMETER)));
	Box_Header_Parameters (group);
	box.Add (group);
	}
if ((flags & PLACEHOLDER_FLAGS_CODESTREAM_MASK) != 0)
	{
	box.Add (new Parameter (CODESTREAM_ID_PARAMETER)
		.Value (Read_long (box.Name () + ' ' + CODESTREAM_ID_PARAMETER)));
	long
		total = Read_Unsigned_int (box.Name () + ' ' +
			TOTAL_CODESTREAMS_PARAMETER);
	if ((flags & PLACEHOLDER_FLAGS_MULTIPLE_CODESTREAM_MASK) == 0)
		total = 1;
	box.Add (new Parameter (TOTAL_CODESTREAMS_PARAMETER)
		.Value (total));
	}
if (Stream_Position != end_position)
	{
	group = new Parameter (EXTENDED_BOX_LIST_PARAMETER);
	if (Use_Data_Position)
		Data_Position_Parameters (group, end_position);
	else
		Data_Offset_Parameters (group,
			Stream_Position - start_position, end_position - start_position);
	box.Add (group);
	}
}
catch (PVL_Exception exception) {}
}


private void Box_Header_Parameters
	(
	Parameter	box
	)
	throws IOException
{
try {
long
	box_length = Read_Unsigned_int (box.Name () + ' ' + LENGTH_PARAMETER);
int
	type_code  = Read_int (box.Name () + ' ' + TYPE_PARAMETER);
if (box_length == 1)
	box_length = Read_long (box.Name () + " box long length");
box
	.Add (new Parameter (NAME_PARAMETER)
		.Value (Box_Name (type_code)))
	.Add (new Parameter (TYPE_PARAMETER)
		.Value (new Value (type_code).Base (16)))
	.Add (new Parameter (LENGTH_PARAMETER)
		.Value (new Value (box_length).Units ("bytes")));
	}
catch (PVL_Exception exception) {}
}


/*==============================================================================
	Helpers
*/
/**	Get the descriptive box name for a box type.
<p>
	@param	type_code	The box type code value. This may be obtained
		from the {@link #TYPE_PARAMETER} present in each box Parameter
		Group.
	@return	The box name String as used in the {@link #NAME_PARAMETER}
		of each box Parameter Group.
*/
public static String Box_Name
	(
	int		type_code
	)
{
String
	string = (String)Box_Names.get (new Integer (type_code));
if (string == null)
	string = UNKNOWN;
return string;
}

/**	Convert an int value to a String.
<p>
	Each byte, starting with the most significant byte, of the value
	is appended as a character to a String. Non-printable characters
	are expanded to escape sequences.
<p>
	@param	value	The int value to be converted.
	@return	The String repesentation of the value.
*/
public static String int_to_String
	(
	int		value
	)
{
String_Buffer
	string = new String_Buffer ();
int
	shift = 24,
	mask = 0x7F000000;
for (shift = 24;
	 shift >= 0;
	 shift -= 8,
	 	mask >>= 8)
	string.append ((char)((value & mask) >> shift));
//	Escape any non-printable characters.
string.special_to_escape ();
return string.toString ();
}

/**	Generate an IOException containing a message.
<p>
	If the provided message does not already contain the class {@link
	#ID} the ID and a NL character is prepended to the message (or the ID
	becomes the message if it is null). If a non-null exception is
	provided, the message from the exception is appended, after a NL
	character, to the message used in constructing the IOException.
<p>
	If the exception argument is an instance of EOFException the returned
	IOException will also be an EOFException. If the exception argument
	is non-null and contains a cause, an attempt is made to apply the
	cause to the new IOException (failure to apply the cause is ignored).
<p>
	@param	message	The message String to be included in the IOException.
	@return	An IOException.
	@see	#IOException(String, Exception)
*/
private IOException IO_Error
	(
	String		message,
	Exception	exception
	)
{
if (message == null)
	message = ID;
else if (message.indexOf (ID) < 0)
	message = ID + '\n' + message;
if (exception != null)
	message += "\n" + exception.getMessage ();
IOException
	except;
if (exception instanceof EOFException)
	except = new EOFException (message);
else
	except = new IOException (message);
if (exception != null)
	{
	Throwable
		cause = exception.getCause ();
	if (cause != null)
		{
		try {except.initCause (cause);}
		catch (IllegalStateException e) {}
		}
	}
return except;
}

/**	Generate an IOException containing a message.
<p>
	@param	message	The message String to be included in the IOException.
	@return	An IOException.
	@see	#IOException(String, Exception)
*/
private IOException IO_Error
	(
	String		message
	)
{return IO_Error (message, null);}

/*==============================================================================
	Main
*/
//!	Exit status values.
public static final int
	SUCCESS			= 0,
	SYNTAX_ERROR	= 1,
	IO_EXCEPTION	= 2,
	NO_INFO			= 3,
	PVL_ERROR		= 4,
	INVALID_FILE	= 5;

/**	Report the JP2_Info found in a file.
<p>
	@param	arguments	The arguments String array.
	@see	#Usage()
*/
public static void main
	(
	String[]	arguments
	)
{
if ((DEBUG & DEBUG_MAIN) != 0)
	System.out.println
		("*** " + ID);
boolean
	skip_tiles = true,
	EOC_search = false,
	GUI = false;
String
	source = null;
Vector
	names = new Vector (),
	patterns = new Vector ();
int
	exit_status = SUCCESS;

for (int argument = 0;
		 argument < arguments.length;
	   ++argument)
	{
	if (arguments[argument].startsWith ("-") &&
		arguments[argument].length () > 1)
		{
		switch (arguments[argument].charAt (1))
			{
			case 'N':
			case 'n':
				int
					index = arguments[argument].indexOf ('_');
				if (index < 0)
					{
					System.out.println
						("Unknown option: " + arguments[argument]);
					Usage ();
					}
				switch (arguments[argument].charAt (++index))
					{
					case 'T':
					case 't':
						skip_tiles = true;
						break;
					case 'E':
					case 'e':
						EOC_search = false;
						break;
					default:
						System.out.println
							("Unknown option: " + arguments[argument]);
					}
				break;
			case 'T':
			case 't':
				skip_tiles = false;
				break;
			case 'E':
			case 'e':
				EOC_search = true;
				break;
			case 'F':
			case 'f':
				if (++argument == arguments.length)
					Usage ();
				names.add (arguments[argument]);
				break;
			case 'P':
			case 'p':
				if (++argument == arguments.length)
					Usage ();
				patterns.add (arguments[argument]);
				break;
			case 'G':
			case 'g':
				GUI = true;
				break;
			default:
				System.out.println
					("Unknown option: " + arguments[argument]);
			case 'H':
			case 'h':
				Usage ();
			}
		}
	else if (source == null)
		source = arguments[argument];
	else
		{
		System.err.println ("Only one source, please.");
		Usage ();
		}
	}
if (source == null)
	{
	System.err.println ("A source must be specified.");
	Usage ();
	}

//	Get the Info.
JP2_Info
	info = null;
JPEG2000_Codestream_Info.EOC_Search = EOC_search;
try {info = new JP2_Info (source, skip_tiles);}
catch (Exception exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (IO_EXCEPTION);
	}

//	Report the Info.
if (info == null)
	{
	System.out.println ("No JP2 Info");
	System.exit (NO_INFO);
	}

int
	validity;
if (! info.Is_Complete_JP2 ())
	{
	System.out.println ("\n>>> WARNING <<< Incomplete JP2 file contents!\n");
	validity = info.JP2_Validity ();
	if ((validity & SIGNATURE_FOUND) == 0)
		System.out.println ("    Signature box not found.");
	if ((validity & FILE_TYPE_FOUND) == 0)
		System.out.println ("    File Type box not found.");
	if ((validity & JP2_HEADER_FOUND) == 0)
		System.out.println ("    JP2 Header box not found.");
	if ((validity & IMAGE_HEADER_FOUND) == 0)
		System.out.println ("    Image Header box not found.");
	if ((validity & COLOUR_SPECIFICATION_FOUND) == 0)
		System.out.println ("    Colour Specification box not found.");
	if ((validity & CONTIGUOUS_CODESTREAM_FOUND) == 0)
		System.out.println ("    JPEG2000 Contiguous Codestream box not found.");
	System.out.println ();
	exit_status = INVALID_FILE;
	}

validity = info.Codestream_Validity ();
boolean
	EOC_missing =
		((! info.Skip_Tiles ()) &&
		((validity & JPEG2000_Codestream_Info.EOC_FOUND) == 0)) ||
		(info.Zero_Length_Tile_Part () &&
		 JPEG2000_Codestream_Info.EOC_Search);
if ((validity != 0 &&
	(validity != JPEG2000_Codestream_Info.CODESTREAM_COMPLETE)) ||
	EOC_missing)
	{
	if (info.Is_Complete_JP2 ())
		System.out.println ();
	System.out.println (">>> WARNING <<< Incomplete JPEG2000 codestream.\n");
	if ((validity & JPEG2000_Codestream_Info.SOC_FOUND) == 0)
		System.out.println ("    Start of Codestream marker not found.");
	if ((validity & JPEG2000_Codestream_Info.SIZ_FOUND) == 0)
		System.out.println ("    Size marker not found.");
	if ((validity & JPEG2000_Codestream_Info.COD_FOUND) == 0)
		System.out.println ("    Coding Style Default marker not found.");
	if ((validity & JPEG2000_Codestream_Info.QCD_FOUND) == 0)
		System.out.println ("    Quantization Default marker not found.");
	if (EOC_missing)
		System.out.println ("    End of Codestream marker not found.");
	System.out.println ();
	exit_status = INVALID_FILE;
	}

if (! names.isEmpty () ||
	! patterns.isEmpty ())
	{
	Parameter
		select = new Parameter (),
		parameter;
	Selector
		criteria;
	String
		name;
	int
		root_name_length = info.Name ().length () + 1;
	Iterator
		list;
	if (! names.isEmpty ())
		{
		criteria = new Selection ().Name (true).Specific (false);
		list = names.iterator ();
		while (list.hasNext ())
			{
			select.Name ((String)list.next ());
			parameter = null;
			while ((parameter = info.Find (select, criteria, parameter))
					!= null)
				{
				name = parameter.Name ();
				parameter.Name (parameter.Path_to_Name ()
					.substring (root_name_length) + name);
				try {parameter.Write ();}
				catch (Exception exception)
					{
					System.out.println
						("PVL error -\n" + exception.getMessage () + "\n");
					System.exit (PVL_ERROR);
					}
				parameter.Name (name);
				}
			}
		}
	if (! patterns.isEmpty ())
		{
		criteria = new Selection ().Name (true).Pattern_Match (true);
		list = patterns.iterator ();
		while (list.hasNext ())
			{
			select.Name ((String)list.next ());
			parameter = null;
			while ((parameter = info.Find (select, criteria, parameter))
					!= null)
				{
				name = parameter.Name ();
				parameter.Name (parameter.Path_to_Name ()
					.substring (root_name_length) + name);
				try {parameter.Write ();}
				catch (Exception exception)
					{
					System.out.println
						("PVL error -\n" + exception.getMessage () + "\n");
					System.exit (PVL_ERROR);
					}
				parameter.Name (name);
				}
			}
		}
	}
else if (GUI)
	{
	Parameter_View
		view = new Parameter_View (info);
	view.setVisible (true);
	}
else
	{
	System.out.println (String_Buffer.escape_to_special (info.Description ()));
	System.exit (exit_status);
	}
}

/**	Print the command line usage syntax.
<p>
<h3>Usage</h3>
<p>
	<b>JP2_Info</b>
		[<b>-</b>[<b><u>N</u>o_</b>]<b><u>T</u>iles</b>]
		[<b>-<u>G</u>ui</b>]
		[<b>-<u>F</u>ind</b> &lt;<i>parameter</i>&gt;]
		[<b>-<u>P</u>attern</b> &lt;<i>regex</i>&gt;]
		&lt;<i>source</i>&gt;

<h3>Description</h3>
<p>
	The contents of the specified file are scanned for JP2 file format
	boxes and JPEG2000 codestream segments. The information found is
	reported in the form of PVL parameters containing all the data
	element values.

<h3>Options</h3>
<p>
<h4><b>-</b>[<b><u>N</u>o_</b>]<b><u>T</u>iles</b></h4>
	<p><blockquote>
	Tiles segment parameters are to be included (or not).
	The default is not to include tile segment parameters.
	</blockquote><p>
<h4><b>-<u>G</u>ui</b></h4>
	<p><blockquote>
	The information PVL is provided in a GUI viewer. The default is to
	provide a terminal listing.
	</blockquote><p>
<h4><b>-<u>F</u>ind</b> &lt;<i>parameter</i>&gt;</h4>
	<p><blockquote>
	The named parameter is listed. This option may be repeated to obtain
	a listing of all parameters specified.
	</blockquote><p>
<h4><b>-<u>P</u>attern</b> &lt;<i>regex</i>&gt;</h4>
	<p><blockquote>
	The parameters with names that match the regular expression are
	listed. This option may also be repeated.
	</blockquote><p>

<h3>Exit Status</h3>
<p>
0 - Success<br>
1 - Command line syntax problem<br>
2 - I/O Exception while attemping to read the source file<br>
3 - No information was found<br>
4 - An error occurred in assembing the PVL parameters for the JP2 information<br>
5 - The source file does not have the required JP2 contents<br>
<p>
<b>N.B.</b>This method always results in a System.exit with a status
of 1.
*/
public static void Usage ()
{
System.out.println
	(
	ID + '\n' +
	"Usage: JP2_Info [<Options>] <source>\n" +
	"  The contents of the specified file are scanned\n" +
	"  for JP2 file format boxes and JPEG2000 codestream segments.\n" +
	"  The information found is reported in the form of\n" +
	"  PVL parameters containing all the data element values.\n" +
	"\n" +
	"Options -\n" +
	"  -[No_]Tiles\n" +
	"    Tiles segment parameters are to be included (or not).\n" +
	"    The default is not to include tile segment parameters.\n" +
	"  -[No_]EOC_search\n" +
	"    Search a final tile-part of unknown length for the EOC marker.\n" +
	"    The default is not to search.\n" +
	"  -Gui\n" +
	"    The information PVL is provided in a GUI viewer.\n" +
	"    The default is to provide a terminal listing.\n" +
	"  -Find <parameter>\n" +
	"    The named parameter is listed. This option may be repeated\n" +
	"    to obtain a listing of all parameters specified.\n" +
	"  -Pattern <regex>\n" +
	"    The parameters with names that match the regular expression\n" +
	"    are listed. This option may also be repeated.\n" +
	"  -Help\n" +
	"    Print this usage description and exit.\n" +
	"Exit Status -\n" +
	"  0 - Success\n" +
	"  1 - Command line syntax problem\n" +
	"  2 - I/O Exception while attemping to read the source file\n" +
	"  3 - No information was found\n" +
	"  4 - An error occurred in assembing the PVL parameters for the JP2 information\n" +
	"  5 - The source file does not have the required JP2 contents\n"
	);
System.exit (SYNTAX_ERROR);
}

}
