/*	View_Locator

PIRL CVS ID: View_Locator.java,v 1.15 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2003-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	java.awt.Window;
import	java.awt.Rectangle;
import	java.awt.Point;
import	java.awt.Insets;
import	java.awt.Toolkit;

//	For Select_LAF
import	javax.swing.UIManager;
import	javax.swing.JFrame;
import	javax.swing.JDialog;

/**	A <I>View_Locator</I> is used to manage the location of a Window
	view on the Display relative to a base location.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version 1.15
*/
public class View_Locator
{
private static final String
	ID = "PIRL.Viewers.View_Locator (1.15 2012/04/16 06:22:59)";

//	Orientation:

/**	Specifies horizontal positioning of a view relative to the left
	side of the base.
*/
public static final int		LEFT			= 1 << 0;


/**	Specifies horizontal positioning of a view relative to the right
	side of the base.
*/
public static final int		RIGHT			= 1 << 1;

/**	Selects the base side orientation policy.
*/
public static final int		SIDE			= LEFT | RIGHT;

/**	Specifies vertical positioning of a view relative to the top
	side of the base.
*/
public static final int		TOP				= LEFT;

/**	Specifies vertical positioning of a view relative to the bottom
	side of the base.
*/
public static final int		BOTTOM			= RIGHT;

/**	Specifies centered positioning of a view relative to the base.
*/
public static final int		CENTER			= LEFT | RIGHT;

/**	Specifies positioning of a view relative to the outside side of the
	base.
*/
public static final int		OUTWARD			= 1 << 2;

/**	Specifies positioning of a view relative to the inside side of the
	base.
*/
public static final int		INWARD			= 0;

/**	Selects the direction relative to the base orientation policy.
*/
public static final int		DIRECTION		= OUTWARD;

/**	The default orientation: x: LEFT | INWARD, y: TOP | INWARD.
*/
public static Point			Default_Orientation
							= new Point (LEFT | INWARD, TOP | INWARD);

private Point				Orientation		= new Point (Default_Orientation);

//	Offsets:

/**	The default horizontal and vertical offsets relative to a base
	to apply to a view when it is relocated: 20x, 20y.
*/
public static Point			Default_Offsets	= new Point (20,  20);

private Point				Offsets			= new Point (Default_Offsets);

//	Frame Margins:

/**	The default top, left, bottom and right margins to be used
	with the base dimensions when calculating the relocation
	position of a view: 0, 0, 0, 0.
*/
public static Insets		Default_Frame_Margins	= new Insets (0, 0, 0, 0);

private Insets				Frame_Margins	= Default_Frame_Margins;

//	Display Bounds:

/**	The effective display bounds: [left, right), [top, bottom).
<p>
	Unless the user specifies the <code>{@link #Display(Rectangle)
	Display}</code> it is set to the <code>{@link
	#Screen_Bounds(Rectangle) Screen_Bounds}</code> as soon as the
	latter becomes valid.
*/
private Insets				Display_Bounds	= null;

/**	The bounds of the screen display. The first view encountered is
	used to obtain its GraphicsConfiguration bounds which is taken to
	be the bounds of the screen used for all subsequent views. This is
	usually sufficient.
*/
private static Insets		Screen_Bounds	= null;

/**	The default location to position a view if it would otherwise
	overlap a side of the display: 175x, 75y.
*/
public static Point			Default_Warp_Location	= new Point (175,  75);

private Point				Warp_Location	= new Point (Default_Warp_Location);

/**	The default horizontal and/or vertical offsets to apply to the
	<code>{@link #Warp_Location() Warp_Location}</code> when it is
	moved along the corresponding axis.
*/
public static Point			Default_Warp_Offsets	= new Point (130,  20);

private Point				Warp_Offsets	= new Point (Default_Warp_Offsets);

//	The direction to move the warp location.
private static int			HORIZONTAL	= 1 << 0,
							VERTICAL	= 1 << 1;

private static Toolkit		TOOLKIT = Toolkit.getDefaultToolkit ();

private static final String	NL = System.getProperty ("line.separator");


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_SETUP			= 1 << 0,
	DEBUG_LOCATORS		= 1 << 1,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a View_Locator using another View_Locator for
	<code>{@link #Policies(View_Locator) Policies}</code>, and
	positions the view to the <code>{@link #Warp_Location()
	Warp_Location}</code>.
<p>
	@param	view	A Window to position (may be null).
	@param	locator	A View_Locator used to set the
		<code>Policies</code> of the new View_Locator. If null then
		default values will be used.
*/
public View_Locator
	(
	Window			view,
	View_Locator	locator
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> View_Locator");
Policies (locator);
if (view != null)
	{
	Display (view);
	if ((DEBUG & DEBUG_SETUP) != 0)
		System.out.println ("    View_Locator:" + NL
			+ "    Set view location to the Warp_Location - "
			+ Warp_Location);
	view.setLocation (Warp_Location);
	move_warp_location (HORIZONTAL);
	}
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< View_Locator");
}

/**	Constructs a View_Locator using default policies, and positions the
	view to the <code>{@link #Warp_Location() Warp_Location}</code>.
<p>
	@param	view	The Window to be positioned.
*/
public View_Locator
	(
	Window			view
	)
{this (view, null);}

/**	Constructs a View_Locator as a copy of another View_Locator.
<p>
	@param	locator	The View_Locator to copy.
*/
public View_Locator
	(
	View_Locator	locator
	)
{this (null, locator);}

/**	Constructs a View_Locator with the specified orientation policies.
<p>
	@param	horizontal_orientation	The horizontal <code>{@link
		#Orientation(int, int) Orientation}</code> policies.
	@param	vertical_orientation	The vertical <code>{@link
		#Orientation(int, int) Orientation}</code> policies.
*/
public View_Locator
	(
	int				horizontal_orientation,
	int				vertical_orientation
	)
{
this ();
Orientation (horizontal_orientation, vertical_orientation);
}

/**	Constructs a View_Locator using default policies.
*/
public View_Locator ()
{this (null, null);}

/*==============================================================================
	Accessors
*/
/**	Sets the view relocation polices for this View_Locator from
	the policies of another View_Locator.
<p>
	@param	policies	The View_Locator from which to copy all
		relocation parameters. If null, defaults will be applied.
	@return	This View_Locator.
*/
public View_Locator Policies
	(
	View_Locator	policies
	)
{
if (policies == null)
	{
	Orientation			= Default_Orientation;
	Offsets				= Default_Offsets;
	Warp_Location		= Default_Warp_Location;
	Warp_Offsets		= Default_Warp_Offsets;
	Frame_Margins		= Default_Frame_Margins;
	Display (Screen_Bounds);
	}
else
	{
	Orientation			= new Point (policies.Orientation);
	Offsets				= new Point (policies.Offsets);
	Warp_Location		= new Point (policies.Warp_Location);
	Warp_Offsets		= new Point (policies.Warp_Offsets);
	Frame_Margins		= new Insets
			(
			policies.Frame_Margins.top,
			policies.Frame_Margins.left,
			policies.Frame_Margins.bottom,
			policies.Frame_Margins.right
			);
	if (policies.Display_Bounds == null)
		Display_Bounds = null;
	else
		Display (policies.Display_Bounds);
	}
return this;
}

/**	Sets the boundary to use when views are relocated.
<p>
	The boundary does not need to be coincident with the real screen
	boundaries. For example, if it is desirable to allow a view to be
	relocated such that it would overlap the screen display, use a
	Rectangle that is positioned and/or sized beyond the screen bounds
	by the maximum amount of overlap to be allowed.
<p>
	@param	rectangle	The Rectangle specifying the bounds of
		the effective display. If null, then the display boundary
		will be reset to the boundary of the screen.
	@return	This View_Locator.
*/
public View_Locator Display
	(
	Rectangle	rectangle
	)
{
if (rectangle == null)
	{
	if (Screen_Bounds == null)
		Display_Bounds = null;
	else
		Display (Screen_Bounds);
	}
else
	Display_Bounds = new Insets
		(
		rectangle.y,
		rectangle.x,
		rectangle.y + Math.abs (rectangle.height),
		rectangle.x + Math.abs (rectangle.width)
		);
return this;
}

/**	Initialize the {@link #Screen() screen bounds} and
	{@link #Display(Rectangle) display bounds} from a view.
<p>
	If the Screen_Bounds was already set it is not reset.
<p>
	If the view is null or not showing on the screen the screen size
	is obtained from the AWT Toolkit. Otherwise the bounds of the
	view's GraphicsConfiguration is used.
<p>
	If the Display_Bounds has not yet been set it is set to the
	Screen_Bounds.
<p>
	@param	view	A Window view reference.
*/
private void Display
	(
	Window		view
	)
{
if (Screen_Bounds == null)
	{
	Rectangle
		screen_rectangle;
	if (view == null ||
		! view.isShowing ())
		screen_rectangle = new Rectangle (TOOLKIT.getScreenSize ());
	else
		screen_rectangle = view.getGraphicsConfiguration ().getBounds ();
	Screen_Bounds = new Insets
		(
		screen_rectangle.y,
		screen_rectangle.x,
		screen_rectangle.y + screen_rectangle.height,
		screen_rectangle.x + screen_rectangle.width
		);
	}
if (Display_Bounds == null)
	Display (Screen_Bounds);
}

/**	Calls <code>{@link #Display(Rectangle) Display}<code> with a Rectangle
	constructed from the Inset values.
<p>
	@param	display_bounds	The Insets values used to construct a
		Rectangle: x = left, y = top, width = right - left, height =
		bottom - top.
*/
private void Display
	(
	Insets		display_bounds
	)
{
if (display_bounds != null)
	Display (new Rectangle
		(
		display_bounds.left,
		display_bounds.top,
		display_bounds.right  - display_bounds.left,
		display_bounds.bottom - display_bounds.top
		));
}

/**	Gets the Rectangle for the current display bounds.
<p>
	@return	A Rectangle describing the current display bounds.
		<B>Note</B>: This will be null if the display bounds have not
		yet been determined.
*/
public Rectangle Display ()
{
if (Display_Bounds == null)
	return null;
return new Rectangle
	(
	Display_Bounds.left,
	Display_Bounds.top,
	Display_Bounds.right - Display_Bounds.left,
	Display_Bounds.bottom - Display_Bounds.top
	);
}

/**	Sets the screen boundary to use for all View_Locators.
<p>
	Normally this is automatically found by using GraphicsConfiguration
	of the first view encountered.
<p>
	The boundary does not need to be coincident with the real screen
	boundaries. For example, if it is desirable to allow a view to be
	relocated such that it would overlap the screen display, use a
	Rectangle that is positioned and/or sized beyond the screen bounds
	by the maximum amount of overlap to be allowed.
<p>
	@param	rectangle	The Rectangle specifying the bounds of
		the effective screen. If null, then the screen boundary will be
		reset using the GraphicsConfiguration of the next view
		encountered.
	@return	This View_Locator.
*/
public View_Locator Screen
	(
	Rectangle	rectangle
	)
{
if (rectangle == null)
	Screen_Bounds = null;
else
	Screen_Bounds = new Insets
		(
		rectangle.y,
		rectangle.x,
		rectangle.y + Math.abs (rectangle.height),
		rectangle.x + Math.abs (rectangle.width)
		);
return this;
}

/**	Gets the Rectangle for the current effective screen bounds.
<p>
	@return	A Rectangle describing the current effective screen bounds.
		<B>Note</B>: This will be null if the screen bounds have not
		yet been determined.
*/
public Rectangle Screen ()
{
if (Screen_Bounds == null)
	return null;
return new Rectangle
	(
	Screen_Bounds.left,
	Screen_Bounds.top,
	Screen_Bounds.right - Screen_Bounds.left,
	Screen_Bounds.bottom - Screen_Bounds.top
	);
}

/**	Sets the orientation policies when relocating a view relative
	to a base.
<p>
	The values used may be a combination, by ORing, of:
<p><ul type="DISC">
<li>x (horizontal)
<dl>
<dt><code>LEFT</code>, <code>RIGHT</code> or <code>CENTER</code>
	<DD>Locate the view relative to the left or right side or center of
	the base.
<dt><code>OUTWARD or INWARD</code>
	<DD>Locate the view towards the inside or ourside of the base.
</dl>
<li>y (vertical)
<dl>
<dt><code>TOP</code>, <code>BOTTOM</code> or<code>CENTER</code>
	<DD>Locate the view relative to the top, bottom or center of the
	base.
<dt><code>OUTWARD or INWARD</code>
	<DD>Locate the view towards the inside or ourside of the base.
</dl>
</ul>
<p>
	@param	orientation	The orientation policies value.
	@return	This View_Locator.
*/
public View_Locator Orientation (Point orientation)
{
if (orientation == null)
	Orientation = Default_Orientation;
else
	Orientation = new Point (orientation);
return this;
}

/**	Sets the orientation policies when relocating a view relative
	to a base.
<p>
	@param	x	The horizontal relocation policies.
	@param	y	The vertical relocation policies.
	@see	#Orientation(Point)
*/
public View_Locator Orientation (int x, int y)
{
Orientation = new Point (x, y);
return this;
}

/**	Gets the orientation polices.
<p>
	The values may be examined by ANDing with the SIDE and comparing with
	the LEFT/RIGHT/CENTER or TOP/BOTTOM/CENTER constants, and/or
	ANDing with the DIRECTION and comparing with the OUTWARD/INWARD
	constants.
<p>
	@return	The orientation policies as a Point of x and y values.
*/
public Point Orientation ()
{return new Point (Orientation);}

/**	Sets the horizontal orientation policies.
<p>
	@param	policies	A horizontal orientation value.
	@return	This View_Locator.
	@see	#Orientation(Point)
*/
public View_Locator Horizontal (int policies)
{
Orientation.x = policies;
return this;
}

/**	Gets the horizontal orientation policies.
<p>
	@return A horizontal orientation value.
	@see	#Orientation(Point)
*/
public int Horizontal ()
{return Orientation.x;}

/**	Sets the vertical orientation policies.
<p>
	@param	policies	Vertical orientation policies.
	@return	This View_Locator.
	@see	#Orientation(Point)
*/
public View_Locator Vertical (int policies)
{
Orientation.y = policies;
return this;
}

/**	Gets the vertical orientation policies.
<p>
	@return Either TOP or BOTTOM or-ed with either INWARD or OUTWARD.
	@see	#Orientation(Point)
*/
public int Vertical ()
{return Orientation.y;}

/**	Sets the location on the screen to which to "warp" a view if
	relocating it would cause it to overlap the display bounds.
<p>
	@param	location	The setLocation Point to use if a view is
		warped.
	@return	This View_Locator.
*/
public View_Locator Warp_Location (Point location)
{
if (location == null)
	Warp_Location = new Point (Default_Warp_Location);
else
	Warp_Location = new Point (location);
return this;
}

/**	Sets the location on the screen to which to "warp" a view if
	relocating it would cause it to overlap the display bounds.
<p>
	@param	x	The horizontal setLocation Point to use if a view is
		warped.
	@param	y	The vertical setLocation Point to use if a view is
		warped.
	@return	This View_Locator.
	@see	java.awt.Component#setLocation(Point)
*/
public View_Locator Warp_Location (int x, int y)
{
Warp_Location = new Point (x, y);
return this;
}

/**	Gets the current location to which to "warp" a view if
	relocating it would cause it to overlap the display bounds.
<p>
	@return	The Point for the current warp location.
*/
public Point Warp_Location ()
{return new Point (Warp_Location);}

/**	Sets the horizontal and vertical offsets to be applied when a view
	is relocated.
<p>
	@param	offsets	The Point containing the horizontal (x) and vertical (y)
		offsets.
	@return	This View_Locator.
	@see	#Offsets(int, int)
*/
public View_Locator Offsets (Point offsets)
{
if (offsets == null)
	Offsets = new Point (Default_Offsets);
else
	Offsets = new Point (offsets);
return this;
}

/**	Sets the horizontal and vertical offsets to be applied when a view
	is relocated.
<p>
	@param	x	The horizontal offset amount.
	@param	y	The vertical offset amount.
	@return	This View_Locator.
	@see	#Offsets(Point)
*/
public View_Locator Offsets (int x, int y)
{
Offsets = new Point (x, y);
return this;
}

/**	Gets the horizontal and vertical offsets to be applied when a view
	is relocated.
<p>
	@return	The Point containing the horizontal (x) and vertical (y)
		offsets.
*/
public Point Offsets ()
{return new Point (Offsets);}

/**	Sets the horizontal and vertical offsets to be applied when the
	warp location is automatically moved.
<p>
	@param	offsets	The Point containing the horizontal (x) and vertical (y)
		offsets.
	@return	This View_Locator.
*/
public View_Locator Warp_Offsets (Point offsets)
{
if (offsets == null)
	Warp_Offsets = new Point (Default_Warp_Offsets);
else
	Warp_Offsets = new Point (offsets);
return this;
}

/**	Sets the horizontal and vertical offsets to be applied when the
	warp location is automatically moved.
<p>
	@param	x	The horizontal offset amount.
	@param	y	The vertical offset amount.
	@return	This View_Locator.
*/
public View_Locator Warp_Offsets (int x, int y)
{
Warp_Offsets = new Point (x, y);
return this;
}

/**	Gets the horizontal and vertical offsets to be applied when the
	warp location is automatically moved.
<p>
	@return	The Point containing the horizontal (x) and vertical (y)
		offsets.
*/
public Point Warp_Offsets ()
{return new Point (Warp_Offsets);}

/**	Sets the margins around a view frame to be used when calculating a
	view relocation position.
<p>
	@param	margins	Insets providing the top, left, bottom, and right margins.
	@return	This View_Locator.
*/
public View_Locator Frame_Margins (Insets margins)
{
if (margins == null)
	Frame_Margins = Default_Frame_Margins;
else
	Frame_Margins = new Insets
		(margins.top, margins.left, margins.bottom, margins.right);
return this;
}

/**	Sets the margins around a view frame to be used when calculating a
	view relocation position.
<p>
	@param	top		The margin for the top side of a frame.
	@param	left	The margin for the left side of a frame.
	@param	bottom	The margin for the bottom side of a frame.
	@param	right	The margin for the right side of a frame.
	@return	This View_Locator.
*/
public View_Locator Frame_Margins (int top, int left, int bottom, int right)
{
Frame_Margins = new Insets (top, left, bottom, right);
return this;
}
/**	Gets the margins to be use for a view frame.
<p>
	@return	Insets providing the top, left, bottom, and right margins.
*/
public Insets Frame_Margins ()
{
return new Insets
	(Frame_Margins.top, Frame_Margins.left,
	Frame_Margins.bottom, Frame_Margins.right);
}

/**	Provides a description of the policies and parmeters of
	this View_Locator.
<p>
	@return	A descriptive String (not new-line terminated).
*/
public String toString ()
{
String
	string =
	ID + NL +
	"Horizontal Orientation: ";
if ((Orientation.x & SIDE) == CENTER)
	string += "Centered" + NL;
else
	string +=
	(((Orientation.x & SIDE)      == LEFT)    ? "Left"    : "Right")  + ' ' +
	(((Orientation.x & DIRECTION) == OUTWARD) ? "Outward" : "Inward") + NL;
string +=
	"  Vertical Orientation: ";
if ((Orientation.y & SIDE) == CENTER)
	string += "Centered" + NL;
else
	string +=
	(((Orientation.y & SIDE)      == TOP)     ? "Top"     : "Bottom") + ' ' +
	(((Orientation.y & DIRECTION) == OUTWARD) ? "Outward" : "Inward") + NL;
string +=
	"       Offsets = " + coordinates (Offsets) + NL +
	" Frame_Margins = " + coordinates (Frame_Margins) + NL +
	" Screen_Bounds = " + coordinates (Screen_Bounds) + NL +
	"Display_Bounds = " + coordinates (Display_Bounds) + NL +
	" Warp_Location = " + coordinates (Warp_Location) + NL +
	"  Warp_Offsets = " + coordinates (Warp_Offsets);
return string;
}

/*
	The coordinates methods remove the leading portion of a String
	before the first '[' character. The leading portion contains the
	class name. Only the coordinate value listing is desired.
*/
private String coordinates (Point location)
{
if (location == null)
	return "null";
String
	description = location.toString ();
return description.substring (description.indexOf ('['));
}

private String coordinates (Rectangle location)
{
if (location == null)
	return "null";
String
	description = location.toString ();
return description.substring (description.indexOf ('['));
}

private String coordinates (Insets location)
{
if (location == null)
	return "null";
String
	description = location.toString ();
return description.substring (description.indexOf ('['));
}

/*==============================================================================
	Locators
*/
/**	Relocates a view relative to a base view.
<p>
	@param	view	The Window to be relocated on the screen.
	@param	base	The Window to use as the base bounds reference.
	@see	#Orientation(Point)
	@see	#Offsets(Point)
	@see	#Display(Rectangle)
	@see	#Warp_Location(Point)
	@see	#Warp_Offsets(Point)
	@see	#Frame_Margins(Insets)
	@see	java.awt.Component#setLocation(Point)
	@see	#Relocate(Window, Rectangle)
*/
public void Relocate
	(
	Window		view,
	Window		base
	)
{
//	Ensure that the screen and display boundaries are set.
Display (base);
Relocate (view, base.getBounds ());
}

/**	Relocates a view relative to base bounds.
<p>
	The view location coordinates are adjusted. The horizontal location
	is changed according to the <code>LEFT</code>, <code>RIGHT</code> or
	<code>CENTER</code> {@link #Horizontal(int) horizontal positioning
	policies} and the vertical location is changed according to the
	<code>TOP</code>, <code>BOTTOM</code> or <code>CENTER</code> {@link
	#Vertical(int) vertical positioning policies}.
<p>
	Centering policies set the view location such that the center of the
	view is aligned with the center of the base. <b>N.B.</b>: Centering
	polices ignore the direction of the orientation and do not use
	{@link #Offsets(Point) offset} values.
<p>
	Other policies offset the view location  either <code>INWARD</code>
	or <code>OUTWARD</code> relative to the <code>LEFT</code> or
	<code>RIGHT</code> and/or <code>TOP</code> or <code>BOTTOM</code>
	sides of the base location coordinates. <code>OUTWARD</code> motion
	positions the view outside of the base adjacent to the specified
	side, plus that side's frame margin, aligned with the top (for
	horizonatal motion) or left (for vertical motion) side of the base
	before applying the offsets. <code>INWARD</code> motion positions the
	view towards the inside of the base adjacent to the specified sides
	before applying the offsets. Note that the offset values may be
	positive, negative or zero.
<p>
	For example, to produce the effect of horizontal, top aligned
	window tiling for a <code>View Window</code> relative to a
	<code>Base Window</code>:
<pre><code>
	View_Locator locator = new View_Locator ();
	locator
		.Offsets (0, 0)
		.Horizontal (View_Locator.RIGHT | View_Locator.OUTWARD)
		.Vertical   (View_Locator.TOP   | View_Locator.INWARD);
	locator.Relocate (View, Base);
</code></pre>
<p>
	After the view location coordinates have been offset the view
	boundary is checked for having been repositioned across a
	<code>{@link #Display(Rectangle) Display} </code> boundary. If it
	has, then it is "warped" to the current <code>{@link
	#Warp_Location(Point) Warp_Location}</code>. This position is
	intended to keep the view from overlapping a boundary. However, if
	the view is larger than the display bounds then it is placed
	against the left and/or top display boundary according to which
	dimensions are too large to fit. Each time a view is warped the
	current <code>Warp_Location</code> is moved horizontally by the
	<code>{@link #Warp_Offsets(Point) Warp_Offsets.y}</code> amount. If
	this new position crosses a display boundary it is wrapped around
	to the other boundary and then the vertical position is also
	offset. When the vertical position crosses a boundary it is wrapped
	around to the other boundary. The movement of the
	<code>Warp_Location</code> is done to help prevent warped views
	from being positioned directly on top of another warped view.
<p>
	Once the view relocation position has been determined the view
	is moved to that location on the screen.
<p>
	@param	view	The Window to be relocated on the screen.
	@param	base_bounds	The Rectangle to use as a relative reference.
	@see	#Orientation(Point)
	@see	#Offsets(Point)
	@see	#Display(Rectangle)
	@see	#Warp_Location(Point)
	@see	#Warp_Offsets(Point)
	@see	#Frame_Margins(Insets)
	@see	java.awt.Component#setLocation(Point)
*/
public void Relocate
	(
	Window		view,
	Rectangle	base_bounds
	)
{
//	Ensure that the screen and display boundaries are set.
Display (view);
Rectangle
	view_bounds = view.getBounds ();
Relocate (view_bounds, base_bounds);
view.setLocation (view_bounds.x, view_bounds.y);
}

/**	Relocates a view relative to a base Point.
<p>
	@param	view	The Window to be relocated on the screen.
	@param	location	The Point to use as a relative reference.
		This will be used as a Rectangle with no height or width.
	@see	#Relocate(Window, Rectangle)
*/
public void Relocate
	(
	Window		view,
	Point		location
	)
{Relocate (view, new Rectangle (location));}

private void Relocate
	(
	Rectangle	view_bounds,
	Rectangle	base_bounds
	)
{
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
		(">>> View_Locator.Relocate:" + NL
		+"                 view bounds = " + coordinates (view_bounds) + NL
		+"                 base bounds = " + coordinates (base_bounds));
//	Horizontal positioning:
view_bounds.x = base_bounds.x;
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
			("      Horizontal orientation = " + Orientation.x);
if ((Orientation.x & SIDE) == CENTER)
	{
	//	Relative to the center of the base.
	view_bounds.x += (base_bounds.width >> 1) - (view_bounds.width >> 1);
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("  Move relative to base center "
				+ (base_bounds.x + (base_bounds.width >> 1)) + NL
			+"                            to " + view_bounds.x);
	}
else
if ((Orientation.x & SIDE) == LEFT)
	{
	//	Relative to the left side of the base.
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("      Move from base left side " + base_bounds.x);
	if ((Orientation.x & DIRECTION) == INWARD)
		{
		//	Move towards the inside of the base.
		view_bounds.x += Offsets.x;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("              inward by offset " + Offsets.x + NL
			+"                            to " + view_bounds.x);
		}
	else
		{
		//	Move away from the outside of the base.
		view_bounds.x -= view_bounds.width + Offsets.x
			+ Frame_Margins.left;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("         outward by view width " + view_bounds.width + NL
			+"                      + offset " + Offsets.x + NL
			+"           + left frame margin " + Frame_Margins.left
			+"                            to " + view_bounds.x);
		}
	}
else
	{
	//	Relative to the right side of the base.
	view_bounds.x += base_bounds.width;
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("     Move from base right side " + view_bounds.x);
	if ((Orientation.x & DIRECTION) == INWARD)
		{
		//	Move towards the inside of the base.
		view_bounds.x -= view_bounds.width + Offsets.x
			+ Frame_Margins.left;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("          inward by view width " + view_bounds.width + NL
			+"                      + offset " + Offsets.x + NL
			+"           + left frame margin " + Frame_Margins.left + NL
			+"                            to " + view_bounds.x);
		}
	else
		{
		//	Move away from the outside of the base.
		view_bounds.x += Offsets.x + Frame_Margins.right;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("             outward by offset " + Offsets.x + NL
			+"          + right frame margin " + Frame_Margins.right + NL
			+"                            to " + view_bounds.x);
		}
	}

//	Vertical positioning:
view_bounds.y = base_bounds.y;
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
			("        Vertical orientation = " + Orientation.y);
if ((Orientation.x & SIDE) == CENTER)
	{
	//	Relative to the center of the base.
	view_bounds.y += (base_bounds.height >> 1) - (view_bounds.height >> 1);
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("  Move relative to base center "
				+ (base_bounds.y + (base_bounds.height >> 1)) + NL
			+"                            to " + view_bounds.y);
	}
else
if ((Orientation.y & SIDE) == TOP)
	{
	//	Relative to the top of the base.
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("       Move from base top side " + base_bounds.y);
	if ((Orientation.y & DIRECTION) == INWARD)
		{
		//	Move towards the inside of the base.
		view_bounds.y += Offsets.y;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("              inward by offset " + Offsets.y + NL
			+"                            to " + view_bounds.y);
		}
	else
		{
		//	Move away from the outside of the base.
		view_bounds.y -= view_bounds.height + Offsets.y
			+ Frame_Margins.top;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("        outward by view height " + view_bounds.height + NL
			+"                      + offset " + Offsets.y + NL
			+"            + top frame margin " + Frame_Margins.top + NL
			+"                            to " + view_bounds.y);
		}
	}
else
	{
	//	Relative to the bottom of the base.
	view_bounds.y += base_bounds.height;
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("    Move from base bottom side " + view_bounds.y);
	if ((Orientation.y & DIRECTION) == INWARD)
		{
		//	Move towards the inside of the base.
		view_bounds.y -= view_bounds.height + Offsets.y
			+ Frame_Margins.top;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("         inward by view height " + view_bounds.height + NL
			+"                      + offset " + Offsets.y + NL
			+"            + top frame margin " + Frame_Margins.top + NL
			+"                            to " + view_bounds.y);
		}
	else
		{
		//	Move away from the outside of the base.
		view_bounds.y += Offsets.y + Frame_Margins.bottom;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
			("             outward by offset " + Offsets.y + NL
			+"                + frame margin " + Frame_Margins.bottom + NL
			+"                            to " + view_bounds.y);
		}
	}
//	Warp the relocated view around the Display_Bounds.
warp (view_bounds);
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
		("<<< View_Locator.Relocate:" + NL
		+"                 view bounds = " + coordinates (view_bounds));
}

/**	If the location for the view_bounds results in an overlap with the
	Display_Bounds, move it to the Warp_Location. Then move the
	Warp_Location.
*/
private void warp
	(
	Rectangle	view_bounds
	)
{
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
		(">>> View_Locator.warp:" + NL
		+"                 view bounds = " + coordinates (view_bounds) + NL
		+"              Display_Bounds = " + coordinates (Display_Bounds));
int
	limit_x = Display_Bounds.right  - view_bounds.width,
	limit_y = Display_Bounds.bottom - view_bounds.height;
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
		("                      limits = " + limit_x + "x, " + limit_y + 'y');
if (view_bounds.x > limit_x || view_bounds.x < Display_Bounds.left ||
	view_bounds.y > limit_y || view_bounds.y < Display_Bounds.top)
	{
	if (limit_x <= Display_Bounds.left)
		{
		limit_x = Display_Bounds.left;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("    Oversized view pushed against left side.");
		}
	else
		{
		while (Warp_Location.x > limit_x)
			move_warp_location (HORIZONTAL);
		limit_x = Warp_Location.x;
		}
	if (limit_y <= Display_Bounds.top)
		{
		limit_y = Display_Bounds.top;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("    Oversized view pushed against top side.");
		}
	else
		{
		while (Warp_Location.y > limit_y)
			move_warp_location (VERTICAL);
		limit_y = Warp_Location.y;
		}
	//	Set the view to the warp location.
	view_bounds.x = limit_x;
	view_bounds.y = limit_y;

	//	Offset the next warp location horizontally
	move_warp_location (HORIZONTAL);
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
		("<<< View_Locator.warp:" + NL
		+"                 view bounds = " + coordinates (view_bounds));
	}
}

private void move_warp_location
	(
	int		direction
	)
{
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println
		(">>> View_Locator.move_warp_location: "
			+ direction + " -"
			+ (((direction & HORIZONTAL) != 0) ? " HORIZONTAL" : "")
			+ (((direction & VERTICAL) != 0) ? " VERTICAL" : "") + NL
		+"              Display_Bounds = " + coordinates (Display_Bounds) + NL
		+"               Warp_Location = " + coordinates (Warp_Location) + NL
		+"                Warp_Offsets = " + coordinates (Warp_Offsets));
Horizontal:
{
if ((direction & HORIZONTAL) != 0)
	{
	Warp_Location.x += Warp_Offsets.x;
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("             Horizontal Offset " + Warp_Offsets.x + NL
			+"                            to " + Warp_Location.x);
	if (Warp_Location.x >= Display_Bounds.right)
		{
		//	Warp from the right to the left side of the screen.
		Warp_Location.x = Display_Bounds.left;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("       Warp through right side " + Display_Bounds.right + NL
				+"                  to left side " + Display_Bounds.left);
		//	and move vertically.
		direction |= VERTICAL;
		}
	else if (Warp_Location.x < Display_Bounds.left)
		{
		//	Warp from the left to the right side of the screen.
		Warp_Location.x = Display_Bounds.right - 1;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("        Warp through left side " + Display_Bounds.left + NL
				+"                 to right side " + (Display_Bounds.right - 1));
		//	and move vertically.
		direction |= VERTICAL;
		}
	}
if ((direction & VERTICAL) != 0)
	{
	Warp_Location.y += Warp_Offsets.y;
	if ((DEBUG & DEBUG_LOCATORS) != 0)
		System.out.println
			("               Vertical Offset " + Warp_Offsets.y + NL
			+"                            to " + Warp_Location.y);
	if (Warp_Location.y >= Display_Bounds.bottom)
		{
		//	Warp from the bottom to the top of the screen.
		Warp_Location.y = Display_Bounds.top;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("      Warp through bottom side " + Display_Bounds.bottom + NL
				+"                   to top side " + Display_Bounds.top);
		if ((direction & HORIZONTAL) == 0)
			{
			//	and move horizontally.
			direction |= HORIZONTAL;
			break Horizontal;
			}
		}
	else if (Warp_Location.y < Display_Bounds.top)
		{
		//	Warp from the top to the bottom of the screen.
		Warp_Location.y = Display_Bounds.bottom - 1;
		if ((DEBUG & DEBUG_LOCATORS) != 0)
			System.out.println
				("         Warp through top side " + Display_Bounds.top + NL
				+"                to bottom side " + (Display_Bounds.bottom - 1));
		if ((direction & HORIZONTAL) == 0)
			{
			//	and move horizontally.
			direction |= HORIZONTAL;
			break Horizontal;
			}
		}
	}
}
if ((DEBUG & DEBUG_LOCATORS) != 0)
	System.out.println ("<<< View_Locator.move_warp_location: "
		+ coordinates (Warp_Location));
}

/*==============================================================================
	Look-and_Feel
*/
/**	The default Look-and-Feel for {@link #Select_LAF(String)}.
*/
public static final String
		DEFAULT_LAF		= "Java";

/**	Select a GUI Look-and_Feel.
<p>
	A set of well-known LAF styles may be selected by their short
	names: Java, which is also known as Metal, Basic or Cross-platform;
	Motif; GTK; Synth; Windows and Windows_Classic. A fully qualified
	classpath for a {@link javax.swing.LookAndFeel} may be specified and
	an attempt will be made to load it into the {@link
	UIManager#setLookAndFeel(String) User Interface Manager}. If the
	name Native is specified the LAF associated with the host system
	will be selected. If the selected LAF can not be used by the User
	Interface Manager the Native LAF will be used instead.
<p>
	New windows ({@link JFrame} and {@link JDialog}) may be requested
	to use the selected LAF, if possible. However, this hint will only be
	effective if applied before the new windows are constructed.
<p>
	@param	LAF_name	The name of the selected LAF. This may be one
		of the well known short names or a fully qualified classpath
		for a LookAndFeel class. If null or the empty String the
		{@link #DEFAULT_LAF} will be selected.
	@param	apply_to_window_frames	If true the JFrame and JDialog
		classes will have a hint applied that they should use the
		selected LAF when a new object is constructed.
	@return	An error message String. This will be null if the selected
		LAF was accepted by the User Interface Manager. Otherwise it
		will describe the reason that the selected LAF could not be
		used.
*/
public static String Select_LAF
	(
	String	LAF_name,
	boolean	apply_to_window_frames
	)
{
String
	error_message = null;
if (LAF_name == null ||
	LAF_name.length () == 0)
	LAF_name = DEFAULT_LAF;

if (LAF_name.indexOf ('.') < 0)
	{
	String
		name = LAF_name.toUpperCase ();
	switch (name.charAt (0))
		{
		case 'M':
			if (name.length () > 1 &&
				name.charAt (1) == 'O')
				{
				//	Motif
				LAF_name = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";
				break;
				}
			//	Fall through to Metal.
		case 'J':	//	Java/Metal
			LAF_name = "javax.swing.plaf.metal.MetalLookAndFeel";
			break;

		case 'B':	//	Basic
			LAF_name = "javax.swing.plaf.basic.BasicLookAndFeel";
			break;

		case 'G':	//	GTK
			LAF_name = "com.sun.java.swing.plaf.gtk.GTKLookAndFeel";
			break;

		case 'S':	//	Synth
			LAF_name = "javax.swing.plaf.synth.SynthLookAndFeel";
			break;

		case 'X':	//	XAWT
			LAF_name = "sun.awt.X11.XAWTLookAndFeel";
			break;

		case 'W':	//	Windows
			if (name.indexOf ('C') > 0)
				{
				//	Windows Classic
				LAF_name =
					"com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel";
				break;
				}
			LAF_name = "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";
			break;

		case 'C':	//	Cross-platform
			try {UIManager.setLookAndFeel
					(UIManager.getCrossPlatformLookAndFeelClassName ());}
			catch (Exception exception)
				{error_message = exception.getMessage ();}
			if (apply_to_window_frames)
				{
				JFrame.setDefaultLookAndFeelDecorated (true);
				JDialog.setDefaultLookAndFeelDecorated (true);
				}
			return error_message;

		default:
			return "Unknown LAF: " + LAF_name;

		case 'N':	//	Native
			return null;
		}
	}
try {UIManager.setLookAndFeel (LAF_name);}
catch (Exception exception)
	{error_message = exception.getMessage ();}
if (apply_to_window_frames)
	{
	JFrame.setDefaultLookAndFeelDecorated (true);
	JDialog.setDefaultLookAndFeelDecorated (true);
	}
return error_message;
}

/**	Select a GUI Look-and_Feel.
<p>
	This is the same as using {@link #Select_LAF(String, boolean)}
	with the apply_to_window_frames argument set to true.
<p>
	@param	LAF_name	The name of the selected LAF. This may be one
		of the well known short names or a fully qualified classpath
		for a LookAndFeel class. If null or the empty String the
		{@link #DEFAULT_LAF} will be selected.
	@return	An error message String. This will be null if the selected
		LAF was accepted by the User Interface Manager. Otherwise it
		will describe the reason that the selected LAF could not be
		used.
*/
public static String Select_LAF
	(
	String	LAF_name
	)
{return Select_LAF (LAF_name, true);}


}	//	End of View_Locator class.
