/*
 * Packet driver interface for Gero's bootstrap IP/TFTP and
 * Linux ethernet driver environment of the other side.
 * This is developed for the context of Proll.
 */

#include <general.h>		/* __P() */
#include <net.h>		/* ETH_ALEN for netpriv.h */
#include <romlib.h>		/* printk() */
#include "./netpriv.h"		/* Our own protos */

#define ETH_FRAME_MIN (ETH_DATA_MIN + ETH_HLEN) /* Min. bytes in frame w/o FCS*/
#define ETH_FRAME_MAX (ETH_DATA_MAX + ETH_HLEN) /* Max. bytes in frame w/o FCS*/
#define ETH_BUF_SIZE  (ETH_FRAME_MAX + ETH_FLEN) /* Buffer size includes FCS */

struct treg {
	int type;
	int (*func)();
};

#define MAXTYPES  3

static struct treg tregv[MAXTYPES];

#define MAXSKBS   35		/* hme: 32, then some */
#define SKBLN    (1536)         /* hme wants ETH_FRAME_LEN + 2, aligned to 64 */

struct skb_ent {
	char inuse;
	char queued;
	struct sk_buff skb;
};

static char skb_rbuf[64 + MAXSKBS*SKBLN];
static struct skb_ent skev[MAXSKBS];

static struct device *eth0;

static union {
	unsigned char s[ETH_BUF_SIZE + 4];
	int aligner;
} wbuf;
static struct sk_buff *rskb;
static int nqskb = 0;


void init_packet()
{
	int i;

	if (eth0 == 0) {
		printk("init_packet: no eth0\n");
		return;
	}

	memcpy(myhwaddr, eth0->dev_addr, ETH_ALEN);
	if ((*eth0->open)(eth0) != 0) {
		printk("init_packet: eth0 open error\n");
		return;
	}

	for (i = 0; i < MAXSKBS; i++) {
		skev[i].skb.allocn = i;
	}
}

unsigned char *reg_type(int ptype, int (*func)())
{
	int n;
	struct treg *tp;

	tp = tregv;
	for (n = 0; ; n++) {
		if (n >= MAXTYPES) return 0;
		if (tp->type == ptype) break;
		if (tp->func == 0) break;
		tp++;
	}
	tp->type = ptype;
	tp->func = func;
	return wbuf.s;
}

int write_packet(int leng, int type, unsigned char *dst)
{
	struct sk_buff *skb;
	unsigned char *s;
#if 0
	struct treg *tp;
#endif

#if 0
	tp = tregv;
	for (n = 0; ; n++) {
		if (n >= MAXTYPES) return -1;
		if (tp->type == ptype) break;
		tp++;
	}
#endif

	if ((skb = dev_alloc_skb(14+leng)) == 0) {
		printk("write_packet: no skb %d\n", leng);
		return -1;
	}
	s = (unsigned char *)skb->data;

	bcopy(dst, s+0, ETH_ALEN);
	bcopy(myhwaddr, s+6, ETH_ALEN);
	s[12] = type >> 8;
	s[13] = type;
	bcopy(wbuf.s, s+14, leng);
	skb->len = 14 + leng;

	if ((*eth0->hard_start_xmit)(skb, eth0) != 0) {
		printk("write_packet: tranmission error\n");
		return -1;
	}

	return 0;
}

/*
 */
void empty_buf()
{

	if (rskb != 0) {
		dev_kfree_skb(rskb);
		rskb = 0;
	} else {
/* P3 */ printk("empty_buf: already free\n");
	}
}

/*
 * Hardware driver registers with us.
 */
void ether_setup(struct device *dev) {

	if (eth0 != 0) {
		printk("Duplicate ether_setup\n");
	}
	eth0 = dev;
}

/*
 */
struct sk_buff *dev_alloc_skb(unsigned int size)
{
	int i;
	struct skb_ent *skep;

	if (size > SKBLN) {
		printk("dev_alloc_skb: too long (%d)\n", size);
		return 0;
	}

	skep = skev;
	for (i = 0; i < MAXSKBS; i++) {
		if (!skep->inuse) {
			skep->inuse = 1;
			skep->skb.data = (char *)
			    (((unsigned)skb_rbuf + 63) & (~63)) + i*SKBLN;
			skep->skb.len = 0;
			/* skep->skb.truesize = size; */
			skep->skb.dev = 0;
			skep->skb.protocol = 0;
			return &skep->skb;
		}
		skep++;
	}

	return 0;
}

void dev_kfree_skb(struct sk_buff *skb) {
	int x = skb->allocn;
	skev[x].inuse = 0;
	if (skev[x].queued) {
/* P3 */ printk("kfree_skb: %d was queued\n", x);
		skev[x].queued = 0;
	}
}

void skb_reserve(struct sk_buff *skb, unsigned int len) {
	skb->data += len;
}

/*
 * Add data to an sk_buff
 */
unsigned char *skb_put(struct sk_buff *skb, unsigned int len) {
	char *p = skb->data + skb->len;
	if ((skb->len += len) > SKBLN) {
		printk("skb_put: too long (+%d %d)\n", len, skb->len);
		return 0;
	}
	return p;
}

/*
 */
void skb_trim(struct sk_buff *skb, unsigned int len) {
	if (skb->len > len) {
		skb->len = len;
	}
}

/*
 */
void
eth_copy_and_sum(struct sk_buff *dest, unsigned char *src, int len, int base)
{
	bcopy(src, dest->data, len);
}

unsigned short eth_type_trans(struct sk_buff *skb, struct device *dev)
{
	unsigned char *s = skb->data + 12;
	return s[0] << 8 | s[1];		/* Network order word */
}

void netif_rx(struct sk_buff *skb)
{
	int i, n;
	struct treg *tp;
	struct skb_ent *skep;

	/*
	 * Queue it
	 */
	skev[skb->allocn].queued = 1;
	nqskb++;

	/*
	 * Process them
	 */
	for (i = 0, skep = skev; i < MAXSKBS; i++, skep++) {
		if (nqskb == 0 || rskb != 0)	/* Optimize || deferred */
			break;

		if (!skep->queued)
			continue;
		/*
		 * Dequeue
		 */
		--nqskb;
		skep->queued = 0;
		skb = &skep->skb;

		for (n = 0, tp = tregv; n < MAXTYPES; n++, tp++) {
			if (tp->type == skb->protocol && tp->func != 0)
				break;
		}
		if (n >= MAXTYPES) {
#if 1	/* Useful. Comment it out for busy networks. */
			printk("netif_rx: dropping type %x, no func\n",
			    skb->protocol);
#endif
			dev_kfree_skb(skb);
			continue;
		}

		rskb = skb;
		if ((*tp->func)(skb->data + ETH_HLEN,
		   skb->len - ETH_HLEN, skb->data+6) == 0) {
			dev_kfree_skb(skb);
			rskb = 0;
		}
	}

	return;
}

/*
 */
unsigned int ld4(void *ptr) {
	unsigned char *s = ptr;
	return (s[0] << 24) | (s[1] << 16) | (s[2] << 8) | s[3];
}

void st4(void *ptr, unsigned int n) {
	unsigned char *s = ptr;
	s[0] = n >> 24;  s[1] = n >> 16;  s[2] = n >> 8;  s[3] = n;
}
