/*	viewer.c
	Copyright (C) 2004-2007 Mark Tyler and Dmitry Groshev

	This file is part of rgbPaint.

	rgbPaint is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	rgbPaint is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with rgbPaint in the file COPYING.
*/

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "global.h"

#include "memory.h"
#include "png.h"
#include "mainwindow.h"
#include "viewer.h"
#include "canvas.h"
#include "mygtk.h"


///	PAN WINDOW


static GtkWidget *pan_window, *draw_pan;
static int pan_w, pan_h;
static unsigned char *pan_rgb;

void draw_pan_thumb(int x1, int y1, int x2, int y2)
{
	int i, j, k, ix, iy, zoom = 1, scale = 1;
	unsigned char *dest, *src;

	if (!pan_rgb) return;		// Needed to stop segfault

	/* Create thumbnail */
	dest = pan_rgb;
	for (i = 0; i < pan_h; i++)
	{
		iy = (i * mem_height) / pan_h;
		src = mem_img[CHN_IMAGE] + iy * mem_width * mem_img_bpp;
		if (mem_img_bpp == 3) /* RGB */
		{
			for (j = 0; j < pan_w; j++ , dest += 3)
			{
				ix = ((j * mem_width) / pan_w) * 3;
				dest[0] = src[ix + 0];
				dest[1] = src[ix + 1];
				dest[2] = src[ix + 2];
			}
		}
		else /* Indexed */
		{
			for (j = 0; j < pan_w; j++ , dest += 3)
			{
				ix = src[(j * mem_width) / pan_w];
				dest[0] = mem_pal[ix].red;
				dest[1] = mem_pal[ix].green;
				dest[2] = mem_pal[ix].blue;
			}
		}
	}


	/* !!! This uses the fact that zoom factor is either N or 1/N !!! */
	if (can_zoom < 1.0) zoom = rint(1.0 / can_zoom);
	else scale = rint(can_zoom);

	/* Canvas coords to image coords */
	x2 = ((x1 + x2) / scale) * zoom;
	y2 = ((y1 + y2) / scale) * zoom;
	x1 = (x1 / scale) * zoom;
	y1 = (y1 / scale) * zoom;
	x1 = x1 < 0 ? 0 : x1 >= mem_width ? mem_width - 1 : x1;
	x2 = x2 < 0 ? 0 : x2 >= mem_width ? mem_width - 1 : x2;
	y1 = y1 < 0 ? 0 : y1 >= mem_height ? mem_height - 1 : y1;
	y2 = y2 < 0 ? 0 : y2 >= mem_height ? mem_height - 1 : y2;

	/* Image coords to thumbnail coords */
	x1 = (x1 * pan_w) / mem_width;
	x2 = (x2 * pan_w) / mem_width;
	y1 = (y1 * pan_h) / mem_height;
	y2 = (y2 * pan_h) / mem_height;

	/* Draw the border */
	dest = src = pan_rgb + (y1 * pan_w + x1) * 3;
	j = y2 - y1;
	k = (x2 - x1) * 3;
	for (i = 0; i <= j; i++)
	{
		dest[k + 0] = dest[k + 1] = dest[k + 2] =
			dest[0] = dest[1] = dest[2] = ((i >> 2) & 1) * 255;
		dest += pan_w * 3;
	}
	j = x2 - x1;
	k = (y2 - y1) * pan_w * 3;
	for (i = 0; i <= j; i++)
	{
		src[k + 0] = src[k + 1] = src[k + 2] =
			src[0] = src[1] = src[2] = ((i >> 2) & 1) * 255;
		src += 3;
	}

	if (draw_pan) gdk_draw_rgb_image(draw_pan->window, draw_pan->style->black_gc,
		0, 0, pan_w, pan_h, GDK_RGB_DITHER_NONE, pan_rgb, pan_w * 3);
}

void pan_thumbnail()		// Create thumbnail and selection box
{
	GtkAdjustment *hori, *vert;

	while (gtk_events_pending()) gtk_main_iteration();
		// Update main window first to get new scroll positions if necessary

	hori = gtk_scrolled_window_get_hadjustment( GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );
	vert = gtk_scrolled_window_get_vadjustment( GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );

	draw_pan_thumb(hori->value, vert->value, hori->page_size, vert->page_size);
}

gint delete_pan( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	if ( pan_rgb != NULL ) free(pan_rgb);
	pan_rgb = NULL;				// Needed to stop segfault
	gtk_widget_destroy(pan_window);

	return FALSE;
}

gint key_pan( GtkWidget *widget, GdkEventKey *event )
{
	int nv_h, nv_v, arrow_key = 0;
	GtkAdjustment *hori, *vert;

	if (!check_zoom_keys_real(wtf_pressed(event)))
	{
		switch (event->keyval)
		{
			case GDK_KP_Left:
			case GDK_Left:		arrow_key = 1; break;
			case GDK_KP_Right:
			case GDK_Right:		arrow_key = 2; break;
			case GDK_KP_Up:
			case GDK_Up:		arrow_key = 3; break;
			case GDK_KP_Down:
			case GDK_Down:		arrow_key = 4; break;
		}

		/* xine-ui sends bogus keypresses so don't delete on this */
		if (!arrow_key && !XINE_FAKERY(event->keyval))
			delete_pan(NULL, NULL, NULL);
		else
		{
			hori = gtk_scrolled_window_get_hadjustment(
				GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );
			vert = gtk_scrolled_window_get_vadjustment(
				GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );

			nv_h = hori->value;
			nv_v = vert->value;

			if ( arrow_key == 1 ) nv_h -= hori->page_size/4;
			if ( arrow_key == 2 ) nv_h += hori->page_size/4;
			if ( arrow_key == 3 ) nv_v -= vert->page_size/4;
			if ( arrow_key == 4 ) nv_v += vert->page_size/4;

			if ( (nv_h + hori->page_size) > hori->upper ) nv_h = hori->upper - hori->page_size;
			if ( (nv_v + vert->page_size) > vert->upper ) nv_v = vert->upper - vert->page_size;
			mtMAX(nv_h, nv_h, 0)
			mtMAX(nv_v, nv_v, 0)
			hori->value = nv_h;
			vert->value = nv_v;
			gtk_adjustment_value_changed( hori );
			gtk_adjustment_value_changed( vert );

			pan_thumbnail();	// Update selection box
		}
	}
	else pan_thumbnail();	// Update selection box as user may have zoomed in/out

	return TRUE;
}

void pan_button(int mx, int my, int button)
{
	int nv_h, nv_v;
	float cent_x, cent_y;
	GtkAdjustment *hori, *vert;

	if ( button == 1 )	// Left click = pan window
	{
		hori = gtk_scrolled_window_get_hadjustment( GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );
		vert = gtk_scrolled_window_get_vadjustment( GTK_SCROLLED_WINDOW(scrolledwindow_canvas) );

		cent_x = ((float) mx) / pan_w;
		cent_y = ((float) my) / pan_h;

		nv_h = mem_width*can_zoom*cent_x - hori->page_size/2;
		nv_v = mem_height*can_zoom*cent_y - vert->page_size/2;

		if ( (nv_h + hori->page_size) > hori->upper ) nv_h = hori->upper - hori->page_size;
		if ( (nv_v + vert->page_size) > vert->upper ) nv_v = vert->upper - vert->page_size;
		mtMAX(nv_h, nv_h, 0)
		mtMAX(nv_v, nv_v, 0)

		hori->value = nv_h;
		vert->value = nv_v;
		gtk_adjustment_value_changed( hori );
		gtk_adjustment_value_changed( vert );

		draw_pan_thumb(nv_h, nv_v, hori->page_size, vert->page_size);
	}
	if ( button == 3 )	// Right click = kill window
		delete_pan(NULL, NULL, NULL);
}

static gint click_pan( GtkWidget *widget, GdkEventButton *event )
{
	pan_button(event->x, event->y, event->button);

	return FALSE;
}

gint pan_motion( GtkWidget *widget, GdkEventMotion *event )
{
	int x, y, button = 0;
	GdkModifierType state;

	if (event->is_hint) gdk_window_get_pointer (event->window, &x, &y, &state);
	else
	{
		x = event->x;
		y = event->y;
		state = event->state;
	}

	if (state & GDK_BUTTON1_MASK) button = 1;
	if (state & GDK_BUTTON3_MASK) button = 3;

	pan_button(x, y, button);

	return TRUE;
}

static gint expose_pan( GtkWidget *widget, GdkEventExpose *event )
{
	int w = event->area.width, h = event->area.height;

	if ( w>pan_w ) w=pan_w;
	if ( h>pan_h ) h=pan_h;

	gdk_draw_rgb_image( widget->window, widget->style->black_gc,
				event->area.x, event->area.y, w, h,
				GDK_RGB_DITHER_NONE,
				pan_rgb + 3*( event->area.x + pan_w*event->area.y ),
				pan_w*3
				);
	return FALSE;
}

void pressed_pan( GtkMenuItem *menu_item, gpointer user_data )
{
	GtkWidget *button;
	int max_pan = 128;
	float rat_x, rat_y;

	draw_pan = NULL;	// Needed by draw_pan_thumb above

	rat_x = max_pan / ((float) mem_width);
	rat_y = max_pan / ((float) mem_height);

	if ( rat_x > rat_y )
	{
		pan_w = rat_y * mem_width;
		pan_h = max_pan;
	}
	else
	{
		pan_w = max_pan;
		pan_h = rat_x * mem_height;
	}
	mtMAX(pan_w, pan_w, 1)
	mtMAX(pan_h, pan_h, 1)

	pan_rgb = grab_memory( 3*pan_w*pan_h, 0 );

	pan_thumbnail();

	pan_window = gtk_dialog_new();
	gtk_window_set_title( GTK_WINDOW(pan_window), "" );
	gtk_window_set_position( GTK_WINDOW(pan_window), GTK_WIN_POS_MOUSE );
	gtk_window_set_decorated( GTK_WINDOW(pan_window), FALSE );
	gtk_window_set_modal(GTK_WINDOW(pan_window), TRUE);
	gtk_window_set_transient_for( GTK_WINDOW(pan_window), GTK_WINDOW(main_window) );


	draw_pan = gtk_drawing_area_new();
	gtk_widget_set_usize( draw_pan, pan_w, pan_h );
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(pan_window)->vbox), draw_pan);
	gtk_widget_show( draw_pan );
	gtk_signal_connect_object( GTK_OBJECT(draw_pan), "expose_event",
		GTK_SIGNAL_FUNC (expose_pan), GTK_OBJECT(draw_pan) );
	gtk_signal_connect_object( GTK_OBJECT(draw_pan), "button_press_event",
		GTK_SIGNAL_FUNC (click_pan), GTK_OBJECT(draw_pan) );
	gtk_signal_connect_object( GTK_OBJECT(draw_pan), "motion_notify_event",
		GTK_SIGNAL_FUNC (pan_motion), GTK_OBJECT(draw_pan) );
	gtk_widget_set_events (draw_pan, GDK_BUTTON_PRESS_MASK);
	gtk_signal_connect_object (GTK_OBJECT (pan_window), "key_press_event",
		GTK_SIGNAL_FUNC (key_pan), NULL);

	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_widget_show( button );
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(delete_pan), NULL);
	gtk_box_pack_start( GTK_BOX(GTK_DIALOG(pan_window)->action_area), button, TRUE, TRUE, 0 );


	gtk_widget_set_events (draw_pan, GDK_ALL_EVENTS_MASK);

	gtk_widget_show (pan_window);
}



////	TAKE SCREENSHOT

void screen_copy_pixels( unsigned char *rgb, int w, int h )
{
	if (mem_new( w, h, 3, CMASK_IMAGE )) return;
	memcpy(mem_img[CHN_IMAGE], rgb, w * h * 3);
}


gboolean grab_screen()
{
	int width = gdk_screen_width(), height = gdk_screen_height();

	GdkPixbuf *screenshot = NULL;

	screenshot = gdk_pixbuf_get_from_drawable (NULL, gdk_get_default_root_window(), NULL,
			0, 0, 0, 0, width, height);

	if ( screenshot != NULL )
	{
		screen_copy_pixels( gdk_pixbuf_get_pixels( screenshot ), width, height );
		g_object_unref( screenshot );
	} else return FALSE;

	return TRUE;
}


////	TEXT TOOL

static GtkWidget *text_window, *text_font_window;
static char font_name[256], font_text[256];

#define PAD_SIZE 4

void initialize_text()
{
	snprintf(font_name, 250, "Sans 16" );
	snprintf(font_text, 250, _("Enter Text Here") );
}

gint render_text( GtkWidget *widget )
{
	GdkImage *t_image = NULL;
	GdkPixmap *text_pixmap;
	gboolean antialias[3] = { TRUE, FALSE, FALSE };
	unsigned char *source, *dest, *dest2, r, g, b, pix_and = 255, v;
	int width, height, i, j, /*k, m,*/ bpp, clip_bpp, back, tx = PAD_SIZE, ty = PAD_SIZE;

	PangoContext *context;
	PangoLayout *layout;
	PangoFontDescription *font_desc;

	context = gtk_widget_create_pango_context (widget);
	layout = pango_layout_new( context );
	font_desc = pango_font_description_from_string( font_name );

	pango_layout_set_text( layout, font_text, -1 );
	pango_layout_set_font_description( layout, font_desc );

	pango_layout_get_pixel_size( layout, &width, &height );
	pango_font_description_free( font_desc );

	width += PAD_SIZE*2;
	height += PAD_SIZE*2;

	text_pixmap = gdk_pixmap_new(widget->window, width, height, -1);

	gdk_draw_rectangle (text_pixmap, widget->style->white_gc, TRUE, 0, 0, width, height);
	gdk_draw_layout( text_pixmap, widget->style->black_gc, tx, ty, layout );

	t_image = gdk_image_get( text_pixmap, 0, 0, width, height );
	bpp = t_image->bpp;

	if ( bpp == 2 ) pix_and = 31;
	if ( bpp < 2 ) pix_and = 1;

	if (mem_channel == CHN_IMAGE) clip_bpp = mem_img_bpp;
	else clip_bpp = 1;

	back = 0;
	r = mem_pal[back].red;
	g = mem_pal[back].green;
	b = mem_pal[back].blue;

	if ( t_image != NULL )
	{
		free( mem_clipboard );	// Lose old clipboard
		free( mem_clip_alpha );	// Lose old clipboard alpha
		mem_clip_alpha = NULL;
		mem_clipboard = malloc( width * height * clip_bpp );
		mem_clip_w = width;
		mem_clip_h = height;
		mem_clip_bpp = clip_bpp;
		if ( !antialias[1] && mem_channel == CHN_IMAGE ) mem_clip_mask_init(255);
		else mem_clip_mask_clear();
	
		if ( mem_clipboard == NULL ||
			( !antialias[1] && mem_channel == CHN_IMAGE && mem_clip_mask == NULL) )
		{
			alert_box_stock( _("Error"), _("Not enough memory to create clipboard"),
					GTK_STOCK_OK, NULL, NULL );
			if ( mem_clipboard != NULL ) free( mem_clipboard );
			mem_clip_mask_clear();
			text_paste = FALSE;
		}
		else
		{
			text_paste = TRUE;
			for ( j=0; j<height; j++ )
			{
				source = (unsigned char *) (t_image->mem) + j * t_image->bpl;
				dest = mem_clipboard + width * j * clip_bpp;
				dest2 = mem_clip_mask + width*j;
				if ( clip_bpp == 3 )		// RGB Clipboard
				{
					if ( antialias[0] )	// Antialiased
					{
							// Antialiased without background
						for ( i=0; i<width; i++ )
						{
							v = source[0] & pix_and;
							dest2[i] = (v * 255 / pix_and) ^ 255;	// Alpha blend

							dest[0] = mem_col_A24.red;
							dest[1] = mem_col_A24.green;
							dest[2] = mem_col_A24.blue;

							source += bpp;
							dest += 3;
						}
					}
				}
			}
			gdk_image_destroy(t_image);
		}
	}
	gdk_pixmap_unref(text_pixmap);		// REMOVE PIXMAP
	g_object_unref( layout );
	g_object_unref( context );

	return TRUE;		// Success
}

static gint delete_text( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	gtk_widget_destroy( text_window );
	return FALSE;
}


static gint paste_text_ok( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	snprintf(font_name, 250, gtk_font_selection_get_font_name(
				GTK_FONT_SELECTION(text_font_window) ));
	snprintf(font_text, 250, (char *) gtk_font_selection_get_preview_text(
				GTK_FONT_SELECTION(text_font_window) ));

	render_text( widget );
	if ( mem_clipboard != NULL ) pressed_paste_centre( NULL, NULL );

	delete_text( widget, event, data );

	return FALSE;
}

void pressed_text( GtkMenuItem *menu_item, gpointer user_data )
{
	GtkWidget *vbox, *hbox, *button;
	GtkAccelGroup* ag = gtk_accel_group_new();

	text_window = gtk_dialog_new();
	gtk_window_set_title( GTK_WINDOW(text_window), _("Paste Text") );
	gtk_window_set_modal(GTK_WINDOW(text_window), TRUE);
	gtk_window_set_transient_for( GTK_WINDOW(text_window), GTK_WINDOW(main_window) );

	vbox = GTK_DIALOG(text_window)->vbox;		// Get dialog vertical box area

	text_font_window = gtk_font_selection_new ();
	gtk_widget_show(text_font_window);
	gtk_container_set_border_width (GTK_CONTAINER (text_font_window), 4);

	gtk_box_pack_start( GTK_BOX(vbox), text_font_window, TRUE, TRUE, 0 );

	hbox = GTK_DIALOG(text_window)->action_area;

	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_widget_show( button );
	gtk_widget_add_accelerator (button, "clicked", ag, GDK_Escape, 0, (GtkAccelFlags) 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(delete_text), NULL);
	gtk_box_pack_end( GTK_BOX(hbox), button, TRUE, TRUE, 5 );

	button = gtk_button_new_from_stock(GTK_STOCK_PASTE);
	gtk_widget_show( button );
	gtk_widget_add_accelerator (button, "clicked", ag, GDK_KP_Enter, 0, (GtkAccelFlags) 0);
	gtk_widget_add_accelerator (button, "clicked", ag, GDK_Return, 0, (GtkAccelFlags) 0);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(paste_text_ok), NULL);
	gtk_box_pack_end( GTK_BOX(hbox), button, TRUE, TRUE, 5 );

	gtk_widget_show(text_window);
	gtk_window_add_accel_group( GTK_WINDOW(text_window), ag );

	gtk_font_selection_set_font_name( GTK_FONT_SELECTION(text_font_window), font_name );
	gtk_font_selection_set_preview_text( GTK_FONT_SELECTION(text_font_window), font_text );

	gtk_widget_grab_focus( GTK_FONT_SELECTION(text_font_window)->preview_entry );
}
