///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/field.h"
#include "rheolef/characteristic.h"
#include "rheolef/basis_on_lattice.h"
#include "rheolef/field_element.h"
using namespace std;
namespace rheolef { 

// ---------------------------------------------------------------------------
// initializer with advection field (with some auxilliary functions)
// ---------------------------------------------------------------------------
// x1 = x0 + va
static 
meshpoint
robust_advect_1 (const geo& bg_omega, const point& x0, const point& va, bool& is_inside) {
    static const Float tol = ::sqrt(numeric_limits<Float>::epsilon());
    point x;
    Float t;
    size_t e;
    // Is the characteristic origin in the mesh ?
    bool b_trace ;
    if (norm2(va) > tol) {
        b_trace = bg_omega.trace(x0, va, x, t, e);
    } else {
	bg_omega.localize_nearest(x0, x, e);
	t = 1;
	b_trace = true;
    }
    if (! b_trace) {    
	// x is outside of the mesh:
        // then use x0 as characteristic origin
	point y;
	bg_omega.localize_nearest(x0, y, e);
	x = y;
    }
    is_inside = b_trace && (1-t < tol);
    meshpoint x1 = bg_omega.hatter(x,e);
    return x1;
}
// xN = x0 + va
// split the advection in N sub-steps and check when it goes out
static
meshpoint 
robust_advect_N (const geo& bg_omega, const point& x0, const point& v0, const field& vh, size_t n, bool& is_inside)
{
  size_t i = 0;
  point xi = x0;
  point vi = v0/Float(n);
  do {
    meshpoint hat_xi1 = robust_advect_1 (bg_omega, xi, vi, is_inside);
    if (i+1 >= n || !is_inside) return hat_xi1;
    xi = bg_omega.dehatter (hat_xi1);
    // TODO: when vh is localized on the background mesh (most of the cases)
    // gives as a guess the previous element K containing xi
    // for evaluation vh(xi): do not need any new localization
    vi = vh.vector_evaluate(xi)/Float(n);
    i++;
  } while (true);
}
static Float h_moy (geo::const_iterator_node p, const geo_element& K) {
  if (K.variant()!=geo_element::t)
     error_macro("Only quadrature on triangles supported yet.");
  Float area = fabs( vect2d( p[K[1]] -p[K[0]], p[K[2]] -p[K[0]] ))/2;
  return ::sqrt(area);
}
// in order to localize  y = x + v(x)
// use adapt n substeps such that: n*h0 = v0
// with the local mesh size h0=h_moy(K)
static inline size_t adapt_n_track_step (Float h0, const point& v0, size_t n_track_step_max) {
    if (n_track_step_max <= 1) return 1;
    size_t n = size_t(norm(v0)/h0 + 0.5);
    n = std::max(n, size_t(1));
    n = std::min(n, n_track_step_max);
    return n;
}
point
field_o_characteristic::advect (const point& x0) const
{
    // TODO: use optional control class for n_track_step
    static size_t option_n_track_step = 10;
    // TODO: when vh is localized on the background mesh (most of the cases)
    // gives as a guess the previous element K containing xi
    // for evaluation vh(xi): do not need any new localization
    point v0 = get_advection().vector_evaluate(x0);
    point x0a;
    size_t K_index;
    get_background_geo().localize_nearest(x0, x0a, K_index);
    const geo_element& K = *(get_background_geo().begin() + K_index);
    Float hK = h_moy(get_background_geo().begin_node(), K);
    size_t n = adapt_n_track_step (hK, v0, option_n_track_step);
    bool is_inside;
    meshpoint hat_x = robust_advect_N (get_background_geo(), x0, v0, get_advection(), n, is_inside);
    point x = get_background_geo().dehatter (hat_x);
    // TODO: idem if the field is located on the background mesh (when no adapt in time loop)
    return x;
}
Float 
field_o_characteristic::operator() (const point& x0) const
{
    point x = advect (x0);
    Float val = get_field().operator()(x);
    return val;
}
point
field_o_characteristic::vector_evaluate(const point& x0) const
{
    point x = advect (x0);
    point val = get_field().vector_evaluate(x);
    return val;
}
tensor
field_o_characteristic::tensor_evaluate(const point& x) const
{
  error_macro ("not yet");
}
// The sum over Omega is rapported to each reference element
// and approximated by a quadrature formulae:
//
// mgh(i) = integrate(Omega)  foX(x) phi_i(x) dx
//        = sum_K integrate(hat(K)) foX(F_K(hat_x)) hat_phi_i(hat_x) DF_K(hat_x) d hat_x
//        ~= sum_K sum_q foX(F_K(hat_xq)) hat_phi_i(hat_xq) DF_K(hat_xq) wq
//
// => each function basis has just to be evaluated on the reference element
//    and only on quadrature nodes, one time for all.
//    this approach is also used when computing biinear forms.
// TODO: a class "field_element" as "form_element" and "field_assembly" ?
//
// TODO: assembly elementary mass matrix between P2 and P2b with center
//  => eval only one time each node of simpson, globaly
// apply it localy to the mass matrix mk(P2,P2b) computed on the fly
//
field
riesz_representer (const space& Vh, const field_o_characteristic& foX)
{
  field_element field_e;
  quadrature_option_type qopt;
  qopt.set_family (quadrature_option_type::gauss_lobatto);
  field_e.initialize (Vh, qopt);
  field mgh (Vh, 0.);
#ifdef TODO
  field_assembly (mgh, field_e, foX);
#endif // TODO
  error_macro ("not yet");
  return mgh;
}
}// namespace rheolef
