///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:geo
NAME: @code{geo} - plot a finite element mesh 
@pindex geo
@cindex plotting
@cindex plotting mesh
@cindex mesh graphic representation
@clindex geo
SYNOPSIS:
  @example
        geo @var{options} @var{mesh}[.geo[.gz]]
  @end example
DESCRIPTION:       
  @noindent
  Plot or upgrade a finite element mesh.
EXAMPLES:
  Plot a mesh:
  @example
        geo square.geo
        geo box.geo
        geo box.geo -full
  @end example
  Plot a mesh into a file:
  @example
        geo square.geo -image-format png
  @end example
  Convert from a old geo file format to the new one:
  @example
        geo -upgrade - < square-old.geo > square.geo
  @end example
  See below for the geo file format scpecification.
  The old file format does not contains edges and faces connectivity in 3d geometries,
  or edges connectivity in 2d geometries.
  The converter add it automatically into the upgraded file format.
  Conversely, the old file format is usefull when combined with a
  translator from another file format that do not provides edges and faces connectivity.

INPUT FILE SPECIFICATION:
  @table @code
  @itemx @var{filename}
        specifies the name of the file containing
        the input mesh.
        The ".geo" suffix extension is assumed.
  @itemx -
        read mesh on standard input instead on a file.
  @itemx -name
        when mesh comes from standard input, the mesh name
        is not known and is set to "output" by default.
        This option allows to change this default.
        Useful when dealing with output formats (graphic, format conversion)
        that creates auxilliary files, based on this name.
@cindex RHEOPATH environment variable
  @itemx -I@var{dir} 
  @itemx -I @var{dir} 
        Add @var{dir} to the rheolef file search path.
        This mechanism initializes a search path given by the environment variable @samp{RHEOPATH}.
        If the environment variable @samp{RHEOPATH} is not set, the default value is the current directory.
  @itemx -check
        Check that element orientation is positive.
  @end table

INPUT FORMAT OPTIONS:
@toindex @code{bamg}
@fiindex @file{.bamg} mesh file
@toindex @code{vtk}
@fiindex @file{.vtk} mesh file
  @table @code
  @itemx -if @var{format}
  @itemx -input-format @var{format}
        load mesh in @var{format} file format,
        instead of plotting it.
        Supported output formats are: @code{geo}, @code{bamg}, @code{vtk}.
        When loading from a file, the corresponding suffix extension is assumed.
  @end table

RENDER SPECIFICATION:
@cindex graphic render
@toindex @code{gnuplot}
  @table @code
  @itemx -gnuplot
        use gnuplot tool.
        This is the default.
@toindex @code{mayavi}
  @itemx -mayavi
        use @code{mayavi} tool.
  @end table

RENDER OPTIONS:
  @table @code
  @itemx -lattice
  @itemx -nolattice
  	When using a high order geometry, the lattice inside any element appears.
	Default is on;
  @itemx -subdivide int
  	When using a high order geometry, the number of points per edge used to draw
	a curved element. Default value is the mesh order.
  @itemx -full
  @itemx -nofull
  	All internal edges appears, for 3d meshes.
	Default is off;
  @itemx -fill
  @itemx -nofill
        Fill mesh faces using light effects, when available.
@cindex @code{stereo 3D rendering}
  @itemx -stereo
  @itemx -[no]stereo
        Rendering mode suitable for red-blue anaglyph 3D stereoscopic glasses.
        Option only available with @code{mayavi}.
  @itemx -cut
  @itemx -nocut
        cut by plane and clip (with @code{mayavi} only).
  @end table

OUTPUT FILE FORMAT OPTIONS:
  @table @code
  @itemx -geo
        output mesh on standard output stream in geo text file format,
        instead of plotting it.
  @itemx -upgrade
        Convert from a old geo file format to the new one.
@toindex @code{gmsh}
@fiindex @file{.gmsh} mesh file
  @itemx -gmsh
        output mesh on standard output stream in gmsh text file format,
        instead of plotting it.
@cindex graphic render
@cindex image file format
@cindex graphic render
@fiindex @file{.png} image 
@fiindex @file{.gif} image
@fiindex @file{.jpg} image
@fiindex @file{.ps} image
@fiindex @file{.pdf} image
  @itemx -image-format @code{string}
        The argument is any valid image format, such as
        bitmap @code{png}, @code{jpg}, @code{gif} or vectorial @code{ps} or @code{pdf}
	image file formats, and that could be handled by the corresponding render.
	The output file is e.g. @file{@emph{basename}.png} when @emph{basename}
	is the name of the mesh, or can be set with the @code{-name} option.
  @end table

OTHERS OPTIONS:
  @table @code
    @itemx -add-boundary
        check for a domain named "boundary"; If this domain
  	does not exists, extract the boundary of the geometry
	and append it to the domain list.
@toindex @code{bamg}
	This command is usefull for mesh converted from 
	generators, as @code{bamg}, that cannot have more than
	one domain specification per boundary edge.

    @itemx -verbose
        print messages related to graphic files created and
        command system calls (this is the default).

    @itemx -noverbose
        does not print previous messages.

    @itemx -clean
        clear temporary graphic files (this is the default).
    
    @itemx -noclean
        does not clear temporary graphic files.

    @itemx -execute
        execute graphic command (this is the default).
    
    @itemx -noexecute
        does not execute graphic command. Generates only 
        graphic files. This is usefull in conjuction with the
        "-noclean" command.

  @itemx -check
  @itemx -dump
        used for debug purpose.
  @end table

INQUIRE OPTIONS:
  @table @code
  @itemx -min-element-measure
  @itemx -max-element-measure
        print the smallest (resp. largest) element measure and then exit.
  @end table

GEO FILE FORMAT:       
	TODO

AUTHOR: Pierre.Saramito@imag.fr
DATE: 16 september 2011
End:
*/

# include "rheolef.h"
//# include "rheolef/iorheo.h"
//# include "rheolef/iofem.h"
//# include "rheolef/rheostream.h"
using namespace rheolef;
void usage() {
      std::cerr
           << "geo: usage:" << std::endl
           << "geo "
           << "{-|file[.geo[.gz]]}"
           << "[-Idir|-I dir] "
           << "[-check] "
           << "[-upgrade] "
           << "[-if {geo,bamg,vtk}] "
           << "[-geo|-gmsh|-gnuplot|-mayavi] "
           << "[-name string] "
           << "[-[no]full|-[no]fill|-[no]lattice] "
           << "[-subdivide int] "
           << "[-[no]stereo|-[no]full|-[no]cut] "
           << "[-image-format string] "
           << "[-add-boundary] "
           << "[-[min|max]-elment-measure] "
           << "[-[no]clean] [-[no]execute] [-[no]verbose] "
           << std::endl;
      exit (1);
}
void set_input_format (idiststream& in, std::string input_format) 
{
       if (input_format == "bamg") in.is() >> bamg;
  else if (input_format == "vtk")  in.is() >> vtk;
  else if (input_format == "geo")  in.is() >> rheo;
  else {
      std::cerr << "geo: invalid input format \""<<input_format<<"\"" << std::endl;
      usage();
  }
}
typedef enum {
        no_render,
        gnuplot_render,
        mayavi_render,
        plotmtv_render,
        vtk_render,
        gmsh_render,
        atom_render,
        x3d_render,
        file_render
} render_type;

template <class T>
void add_boundary (const geo_basic<T,sequential>& omega) 
{
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef std::pair<size_type,size_type>              pair_t;
  typedef heap_allocator<size_type>                   alloc_t;
  typedef geo_element_auto<alloc_t>                   element_t;
  typedef array<element_t, sequential, alloc_t>       e_array_t;

  if (omega.have_domain_indirect ("boundary")) return;
  size_type map_dim = omega.map_dimension();
  if (map_dim == 0) return;

  std::vector<index_set> ball[4];
  // compute bdry_tmp:
  // 1) vertice-to-element map:
  ball[map_dim].resize (omega.n_vertex());
  for (size_type ie = 0, ne = omega.size(map_dim); ie < ne; ie++) {
    const geo_element& K = omega.get_geo_element (map_dim, ie);
    for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
      ball [map_dim][K[iloc]].insert(ie);
    }
  }
  // 2) scan sides, count adjacency and list bdry side indexes
  size_type sub_map_dim = map_dim-1;
  std::list<size_type, alloc_t> iside;
  for (size_type is = 0, ns = omega.size(sub_map_dim); is < ns; is++) {
    const geo_element& S = omega.get_geo_element (sub_map_dim, is);
    index_set adj = ball [map_dim][S[0]];
    for (size_type iloc = 1, nloc = S.size(); iloc < nloc; iloc++) {
      adj.inplace_intersection (ball [map_dim][S[iloc]]);
    }
    if (adj.size() != 1) continue;
    // here S is a boundary side:
    iside.push_back (is);
  }

  // 3) copy sides to a bdry array
  alloc_t alloc;
  element_t dummy_elt (reference_element::p, 0, alloc);
  e_array_t bdry_tmp (iside.size(), dummy_elt, alloc);
  size_type ibdr = 0;
  for (typename std::list<size_type, alloc_t>::const_iterator iter = iside.begin(), last = iside.end();	
	iter != last; ++iter, ++ibdr) {
    size_type is = *iter;
    bdry_tmp [ibdr] = omega.get_geo_element (sub_map_dim, is);
  } 
  // 4) insert the bdry array into the mesh
  domain_indirect_basic<sequential> bdry (bdry_tmp, omega, ball);
  bdry.set_name ("boundary");
  omega.insert_domain_indirect (bdry);
}
Float extrema_element_measure (const geo_basic<Float,sequential>& omega, bool is_min)
{
  space_basic<Float,sequential> Xh (omega, "P0");
  form_basic<Float,sequential>  m (Xh, Xh, "mass");
  field_basic<Float,sequential> one (Xh, 1);
  return is_min ? (m*one).min_abs() : (m*one).max_abs();
}
int main(int argc, char**argv) {
    environment distributed(argc, argv);
    check_macro (communicator().size() == 1, "geo: command may be used as mono-process only");
    // ----------------------------
    // default options
    // ----------------------------
    bool do_upgrade  = false ;
    bool do_check    = false ;
    bool do_add_bdry = false ;
    bool show_min_element_measure = false ;
    bool show_max_element_measure = false ;
    std::string name = "output";
    dout.os() << verbose; bool bverbose = true;
    render_type render = no_render;
    geo_basic<Float,sequential> omega;
    dout.os() << lattice;
    dout.os() << nofill;
    dout.os() << grid;
#ifdef TO_CLEAN
    dout.os() << setorigin (point(0,0,0));
    dout.os() << setnormal (point(1,0,0));
#endif // TO_CLEAN
    std::string filename;
    std::string input_format = "geo";
    // ----------------------------
    // scan the command line
    // ----------------------------
    for (int i = 1; i < argc; i++) {

	// general options:
             if (strcmp (argv[i], "-clean") == 0)     dout.os() << clean;
        else if (strcmp (argv[i], "-noclean") == 0)   dout.os() << noclean;
        else if (strcmp (argv[i], "-execute") == 0)   dout.os() << execute;
        else if (strcmp (argv[i], "-noexecute") == 0) dout.os() << noexecute;
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; dout.os() << verbose; }
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; dout.os() << noverbose; }
        else if (strcmp (argv[i], "-add-boundary") == 0) { do_add_bdry = true; }
        else if (strcmp (argv[i], "-min-element-measure") == 0)  { show_min_element_measure = true; }
        else if (strcmp (argv[i], "-max-element-measure") == 0)  { show_max_element_measure = true; }
        else if (strcmp (argv[i], "-I") == 0)         {
            if (i == argc-1) { std::cerr << "geo -I: option argument missing" << std::endl; usage(); }
            append_dir_to_rheo_path (argv[++i]);
        }
        else if (argv [i][0] == '-' && argv [i][1] == 'I') { append_dir_to_rheo_path (argv[i]+2); }
	// output file option:
        else if (strcmp (argv[i], "-geo") == 0)       { dout.os() << rheo;     render = file_render; }
        else if (strcmp (argv[i], "-gmsh") == 0)      { dout.os() << gmsh;     render = gmsh_render; }

	// render spec:
        else if (strcmp (argv[i], "-gnuplot") == 0)   { dout.os() << gnuplot;  render = gnuplot_render; }
        else if (strcmp (argv[i], "-mayavi") == 0)   { dout.os() << mayavi;  render = mayavi_render; }

	// render option:
        else if (strcmp (argv[i], "-full") == 0)      { dout.os() << full; }
        else if (strcmp (argv[i], "-nofull") == 0)    { dout.os() << nofull; }
        else if (strcmp (argv[i], "-fill") == 0)      { dout.os() << fill; }
        else if (strcmp (argv[i], "-nofill") == 0)    { dout.os() << nofill; }
        else if (strcmp (argv[i], "-stereo") == 0)    { dout.os() << stereo << mayavi; render = mayavi_render; }
        else if (strcmp (argv[i], "-nostereo") == 0)  { dout.os() << nostereo; }
        else if (strcmp (argv[i], "-cut") == 0)       { dout.os() << cut; }
        else if (strcmp (argv[i], "-nocut") == 0)     { dout.os() << nocut; }
        else if (strcmp (argv[i], "-lattice") == 0)   { dout.os() << lattice; }
        else if (strcmp (argv[i], "-nolattice") == 0) { dout.os() << nolattice; }
        else if (strcmp (argv[i], "-subdivide") == 0) {
            if (i == argc-1) { std::cerr << "geo -subdivide: option argument missing" << std::endl; usage(); }
            size_t nsub = atoi(argv[++i]);
	    dout.os() << setsubdivide (nsub);
	}
        else if (strcmp (argv[i], "-image-format") == 0) {
            if (i == argc-1) { std::cerr << "geo -image-format: option argument missing" << std::endl; usage(); }
            std::string format = argv[++i];
	    if (format == "jpeg")       format = "jpg";
	    if (format == "postscript") format = "ps";
	    dout.os() << setimage_format(format);
        }

	// input options:
        else if (strcmp (argv[i], "-upgrade") == 0)   { do_upgrade = true; dout.os() << rheo; render = file_render; }
        else if (strcmp (argv[i], "-check") == 0)   { do_check = true; }
        else if (strcmp (argv[i], "-name") == 0) {
            if (i == argc-1) { std::cerr << "geo -name: option argument missing" << std::endl; usage(); }
            name = argv[++i];
        }
        else if (strcmp (argv[i], "-if") == 0 ||
                 strcmp (argv[i], "-input-format") == 0) {
            if (i == argc-1) { std::cerr << "geo "<<argv[i]<<": option argument missing" << std::endl; usage(); }
            input_format = argv[++i];
        }
        else if (strcmp (argv[i], "-") == 0) {
	    filename = "-"; // geo on stdin
	}
        else if (argv[i][0] != '-') {
	    filename = argv[i]; // input on file	
        }
        else { usage(); }
    }
    // ----------------------------
    // geo treatment
    // ----------------------------
    if (filename == "") {
      std::cerr << "geo: no input file specified" << std::endl;
      usage();
    } else if (filename == "-") {
      // geo on stdin
      if (do_upgrade) std::cin >> upgrade;
      std::string thename;
      if (name != "output") thename = name;
      std::cin >> setbasename(thename);
      set_input_format (din, input_format);
      din >> omega;
      dout.os() << setbasename(name) << reader_on_stdin;
    } else {
      // input geo on file
      std::string suffix = input_format;
      if (name == "output") {
        name = get_basename(delete_suffix (delete_suffix (filename, "gz"), suffix));
      }
      idiststream ips;
      ips.open (filename, suffix);
      check_macro(ips.good(), "\"" << filename << "[."<<suffix<<"[.gz]]\" not found.");
      if (do_upgrade) ips.is() >> upgrade;
      std::string root_name = delete_suffix (delete_suffix(filename, "gz"), suffix);
      std::string name = get_basename (root_name);
      ips.is() >> setbasename(name);
      set_input_format (ips, input_format);
      ips >> omega;
      dout.os() << setbasename(name);
    }
    if (do_add_bdry) {
      add_boundary (omega);
    }
    if (do_check) {
        check_macro (omega.check(bverbose), "geo check failed");
    }
    bool show = show_min_element_measure || show_max_element_measure;
    if (show) {
      if (show_min_element_measure) {
        dout << "min_element_measure = " << extrema_element_measure (omega, true) << std::endl;
      }
      if (show_max_element_measure) {
        dout << "max_element_measure = " << extrema_element_measure (omega, false) << std::endl;
      }
      return 0;
    }
    if (render == no_render) { dout.os() << gnuplot; }
    dout << omega;
}
