///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/grad.h"
#include "rheolef/piola.h"

namespace rheolef {
using namespace std;
using namespace ublas;

template<class T, class M>
void 
grad<T,M>::operator() (const geo_element& K, ublas::matrix<T>& m) const
{
  check_macro (! base::is_on_band(), "unsupported banded level set extension");

  size_type map_d = K.dimension();
  size_type d = base::coordinate_dimension();
  check_macro (d == map_d,
    "unsupported `grad' form on element type `" << K.name()
    << " in " << d << "D geometry");
  reference_element hat_K = K;
  size_type nx = base::get_first_basis().size(hat_K);
  size_type ny = base::get_second_basis().size(hat_K);
  m.resize (map_d*ny, nx);
  m.clear();
  std::vector<size_type> dis_inod;
  base::_omega.dis_inod (K, dis_inod);
  typename quadrature<T>::const_iterator
    first = base::_quad.begin(hat_K),
    last  = base::_quad.end  (hat_K);
  tensor_basic<T> DF, invDF, invDFt_wq;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (base::_omega, base::_piola_table, K, dis_inod, q, DF);
    invDF = inv(DF,map_d);
    T wq = base::is_weighted() ? base::weight(K,q) : T(1);
    wq *= base::weight_coordinate_system (K, dis_inod, q);
    wq *= rheolef::det_jacobian_piola_transformation (DF, base::coordinate_dimension(), K.dimension());
    wq *= (*first).w;
    invDFt_wq = trans(invDF) * wq;
    
    typename base::quad_const_iterator
      phi            = base::_by_table.begin(hat_K, q),
      last_phi       = base::_by_table.end  (hat_K, q);
    typename base::quad_const_iterator_grad
      first_grad_psi = base::_bx_table.begin_grad (hat_K, q),
      last_grad_psi  = base::_bx_table.end_grad   (hat_K, q);
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      typename base::quad_const_iterator_grad grad_psi = first_grad_psi;
      T phi_i = *phi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point_basic<T>& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < map_d; k++) {
	  T sum = 0;
          for (size_type l = 0; l < map_d; l++) {
            sum += phi_i * invDFt_wq(k,l) * grad_psi_j[l];
	  }
	  m(i+ny*k,j) += sum;
	}
      }
    }
  }
}
template<class T, class M>
void
grad<T,M>::initialize () const
{
  base::set_n_derivative(1);

  check_macro (
	space_constant::vector == base::get_second_space().valued_tag(),
	"unsupported non-vectorial second space for `grad' form");
  check_macro (
	space_constant::scalar == base::get_first_space().valued_tag(),
	"unsupported non-scalar first space for `grad' form");
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class grad<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class grad<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
