///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// mayavi vtk visualization
//
// author: Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 23 oct 2011
//
#include "rheolef/field.h"
#include "rheolef/piola.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"
#include "rheolef/iofem.h"
using namespace std;
namespace rheolef { 

// ----------------------------------------------------------------------------
// python utils
// ----------------------------------------------------------------------------
template<class T>
static
std::string
python (const point_basic<T>& x, size_t d=3)
{
    std::ostringstream os;
    os << "(" << x[0];
    for (size_t i = 1; i < d; i++)
      os << ", " << x[i];
    os << ")" << flush;
    string buffer = os.str();
    return buffer;
}
// ----------------------------------------------------------------------------
// field puts
// ----------------------------------------------------------------------------
// extern:
template <class T> odiststream& field_put_vtk (odiststream&, const field_basic<T,sequential>&);

template <class T>
odiststream&
visu_vtk_mayavi (odiststream& ops, const field_basic<T,sequential>& uh)
{
  //
  // 1) prerequises
  //
  using namespace std;
  typedef typename field_basic<T,sequential>::float_type float_type;
  typedef typename geo_basic<float_type,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  ostream& os = ops.os();
  bool verbose  = iorheo::getverbose(os);
  bool clean    = iorheo::getclean(os);
  bool execute  = iorheo::getexecute(os);
  bool fill      = iorheo::getfill(os);    // isocontours or color fill
  bool elevation = iorheo::getelevation(os);
  bool color   = iorheo::getcolor(os);
  bool gray    = iorheo::getgray(os);
  bool black_and_white = iorheo::getblack_and_white(os);
  bool stereo    = iorheo::getstereo(os);
  bool volume    = iorheo::getvolume(os);
  bool iso       = iorheo::getiso(os);
  bool cut       = iorheo::getcut(os);
  bool grid       = iorheo::getgrid(os);
  string format   = iorheo::getimage_format(os);
  string basename = iorheo::getbasename(os);
  string valued = uh.get_space().valued();
  bool is_scalar = (valued == "scalar");
  bool velocity    = iorheo::getvelocity(os);
  bool deformation = iorheo::getdeformation(os);
  Float vscale = iorheo::getvectorscale(os);
  size_type n_isovalue = iorheo::getn_isovalue(os);
  size_type n_isovalue_negative = iorheo::getn_isovalue_negative(os);
  point_basic<float_type> origin  = iofem::getorigin(os);
  point_basic<float_type> normal  = iofem::getnormal(os);
  if (black_and_white) {
    fill = false;
  }
  string outfile_fmt = "";
  string tmp = get_tmpdir() + "/";
  if (!clean) tmp = "";

  const geo_basic<float_type,sequential>& omega = uh.get_geo();
  size_type dim     = omega.dimension();
  size_type map_dim = omega.map_dimension();
  size_type nv      = omega.sizes().ownership_by_dimension[0].size();
  size_type nedg    = omega.sizes().ownership_by_dimension[1].size();
  size_type nfac    = omega.sizes().ownership_by_dimension[2].size();
  size_type nvol    = omega.sizes().ownership_by_dimension[3].size();
  size_type ne      = omega.sizes().ownership_by_dimension[map_dim].size();

  const numbering<float_type,sequential>& fem = uh.get_space().get_numbering();
  basis_on_pointset<float_type> b (fem.get_basis(), omega.get_piola_basis());
  //
  // 2) output data
  //
  string filelist;
  string filename = tmp+basename + ".vtk";
  filelist = filelist + " " + filename;
  ofstream vtk_os (filename.c_str());
  odiststream vtk (vtk_os);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  field_put_vtk (vtk, uh);
  vtk.close();
  //
  // 3) create python data file
  //
  std::string py_name = filename = tmp+basename + ".py";
  filelist = filelist + " " + filename;
  ofstream py (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  string style = (valued == "tensor" ? "tensor" : (velocity ? "velocity" : "deformation"));
  string label = (is_scalar ? basename : (style + " module"));
  py << "#!/usr/bin/env mayavi2" << endl
     << "# This is a mayavi script for the visualization of " << basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
  // mayavi2 python library
  py << "from sys import version_info" << endl
     << "# mayavi2.py file has changed from directory since python 2.7:" << endl
     << "if version_info[0] >= 2 and version_info[1] >= 7 and version_info[2] >= 2:" << endl
     << "    from mayavi.scripts import mayavi2" << endl
     << "else:" << endl
     << "    from enthought.mayavi.scripts import mayavi2" << endl
     << "mayavi2.standalone(globals())     # start mayvi2 graphic interface" << endl
     << "from mayavi2_rheolef import *     # load rheolef specific functions" << endl
     << endl
     ;
  py << "opt = {                 	\\" << endl
     << "    'label'   : '" << label << "',   \\" << endl
     << "    'axis'    : 1," << endl
     << "    'color'   : '" << (color ? "color" : (gray ? "gray" : "black_and_white")) << "'," << endl
     << "    'stereo'  : " << stereo << ",		\\" << endl
     ;
  if (valued == "vector") {
     py << "    'style'   : '" << style << "',		\\" << endl
     ;
  }
  py << "    'scale'   : " << vscale << ",		\\" << endl
     << "    'elevation'  : " << elevation   << ",		\\" << endl
     ;
  if (valued == "scalar") {
     py << "    'volume'     : " << volume   << ",		\\" << endl
        << "    'iso'        : " << iso   << ",		\\" << endl
        << "    'n_isovalue' : " << n_isovalue   << ",		\\" << endl
        << "    'n_isovalue_negative' : " << n_isovalue_negative   << ",		\\" << endl
        << "    'cut'        : " << cut   << ",		\\" << endl
        << "    'origin'     : " << python(origin) << ", 		\\" << endl
        << "    'normal'     : " << python(normal) << ",  		\\" << endl
	;
  }
  py << "    'grid'    : " << grid   << ",		\\" << endl
     << "    'fill'    : " << fill   << ",		\\" << endl
     << "    'view_1d' : 0,				\\" << endl
     << "    'view_2d' : " << (dim == 2) << ",		\\" << endl
     << "    'view_map' : " << (dim > map_dim) << "		\\" << endl
     << "    }" << endl
     << endl
     ;
  py << "mayavi2_field_" << valued << "(mayavi, \"" << tmp+basename << "\", opt)" << endl
     << endl
     ;
  py.close();
  //
  // 3) run pyton
  //
  int status = 0;
  string command;
  if (execute) {
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " mayavi2 -x " + py_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // 4) clear vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + filelist;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  return ops;
}
// --------------------------------------------------------------------
// plane cut : via vtk-mayavi script
// --------------------------------------------------------------------
template <class T> idiststream& geo_get_vtk (idiststream&, geo_basic<T,sequential>&);

template <class T>
field_basic<T,sequential>
plane_cut (
  const field_basic<T,sequential>& uh,
  const point_basic<T>&            origin,
  const point_basic<T>&            normal)
{
  //
  // 1) prerequises
  //
  using namespace std;
  typedef typename field_basic<T,sequential>::float_type float_type;
  typedef typename geo_basic<float_type,sequential>::size_type size_type;
  typedef typename geo_basic<float_type,sequential>::node_type node_type;
  typedef point_basic<size_type>                      ilat;
  ostream& os = std::cout;
  bool verbose  = iorheo::getverbose(os);
  bool clean    = iorheo::getclean(os);
  bool execute  = iorheo::getexecute(os);
  string basename = iorheo::getbasename(os);
  string tmp = get_tmpdir() + "/";
  if (!clean) tmp = "";
#ifdef TO_CLEAN
  point_basic<float_type> origin  = iofem::getorigin(os);
  point_basic<float_type> normal  = iofem::getnormal(os);
#endif // TO_CLEAN
  //
  // 2) output data
  //
  string filelist;
  string vtk_name = tmp+basename + ".vtk";
  filelist = filelist + " " + vtk_name;
  ofstream vtk_os (vtk_name.c_str());
  odiststream vtk (vtk_os);
  if (verbose) clog << "! file \"" << vtk_name << "\" created.\n";
  field_put_vtk (vtk, uh);
  vtk.close();
  //
  // create python script
  //
  //
  // 3) create python data file
  //
  string      py_name = tmp+basename + "-cut.py";
  string vtk_cut_name = tmp+basename + "-cut.vtk";
  filelist = filelist + " " + py_name + " " + vtk_cut_name;
  ofstream py (py_name.c_str());
  if (verbose) clog << "! file \"" << py_name << "\" created.\n";
  py << "#!/usr/bin/env mayavi2" << endl
     << "# This is a mayavi script for the visualization of " << basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
  // mayavi2 python library
  py << "from sys import version_info" << endl
     << "# mayavi2.py file has changed from directory since python 2.7:" << endl
     << "if version_info[0] >= 2 and version_info[1] >= 7 and version_info[2] >= 2:" << endl
     << "    from mayavi.scripts import mayavi2" << endl
     << "else:" << endl
     << "    from enthought.mayavi.scripts import mayavi2" << endl
     << "mayavi2.standalone(globals())     # start mayvi2 graphic interface" << endl
     << "from mayavi2_rheolef import *     # load rheolef specific functions" << endl
     << endl
     << setprecision(numeric_limits<Float>::digits10)
     << "input  = \"" << vtk_name     << "\"" << endl
     << "output = \"" << vtk_cut_name << "\"" << endl
     << "origin = " << python(origin) << endl
     << "normal = " << python(normal) << endl
     << "tol    = " << sqrt(numeric_limits<Float>::epsilon()) << endl
     << "plane_cutter_on_file(input, output, origin, normal, tol)" << endl
	;
  py.close();
  //
  // 3) run pyton
  //
  int status = 0;
  string command;
  if (execute) {
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " python " + py_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //  
  // 4) load vtk cutted mesh
  //
  ifstream vtk_polydata (vtk_cut_name.c_str());
  check_macro (vtk_polydata, "field: vtk polydata file \"" << vtk_cut_name << "\" not found.");
  if (verbose) clog << "! load `" << vtk_cut_name << "'." << endl;
  idiststream ips_vtk_polydata (vtk_polydata);
  geo_basic<T,sequential> gamma_cut; 
  geo_get_vtk (ips_vtk_polydata, gamma_cut);
  gamma_cut.set_name (basename + "-cut");
  check_macro (gamma_cut.n_node() > 0, "empty mesh & plane intersection: HINT check normal and origin");
  //
  // clean
  //
  if (clean) {
      command = "/bin/rm -f " + filelist;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // project geometry from 2d/3d on plane in 1d/2d
  // 
  //   1D) x := dot(OM, tangent)
  //
  //   2D) x1 := dot(OM, t1)
  //       x2 := dot(OM, t2)
  //
  bool use_projection = true;
  if (use_projection) {
    switch (uh.get_geo().dimension()) {
      case 2: {
        // project in 1D space
        gamma_cut.set_dimension (1);
        point_basic<T> tangent = point_basic<T>(normal[1], -normal[0]);
	array<node_type,sequential> node = gamma_cut.get_nodes();
        for (size_type i = 0, n = node.size(); i < n; i++) {
	  node[i] = point_basic<T> (dot(node[i], tangent), 0);
        }
        gamma_cut.set_nodes (node); 
        break;
      }
      case 3: {
        gamma_cut.set_dimension (2);
        point_basic<T> t1 = point_basic<T>(1, 0, 0);
        if (norm(vect(t1, normal)) == Float(0)) t1 = point_basic<T>(0, 1, 0);
        t1 = t1 - dot(t1,normal)*normal;
        t1 = t1 / norm(t1);
        point_basic<T> t2 = vect(normal,t1);
	array<node_type,sequential> node = gamma_cut.get_nodes();
        for (size_type i = 0, n = node.size(); i < n; i++) {
	  node[i] = point_basic<T> (dot(node[i] - origin, t1), dot(node[i] - origin, t2), 0);
        }
        gamma_cut.set_nodes (node); 
        break;
      }
      default: error_macro ("unexpected dimension="<< uh.get_geo().dimension());
    }
  }
  // -------------------------------
  // get field values
  // polydata header:
  //     POINT_DATA <n_node>
  //     SCALARS scalars float
  //     LOOKUP_TABLE default
  // -------------------------------
  scatch (vtk_polydata, "POINT_DATA");
  scatch (vtk_polydata, "SCALARS");
  scatch (vtk_polydata, "default");
  space_basic<T,sequential> V_cut (gamma_cut, "P1");
  field_basic<T,sequential> u_cut (V_cut);
  u_cut.set_u().get_values (ips_vtk_polydata);
  return u_cut;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template odiststream& visu_vtk_mayavi<Float>  (odiststream&, const field_basic<Float,sequential>&);
template field_basic<Float,sequential> plane_cut (const field_basic<Float,sequential>&, const point_basic<Float>&, const point_basic<Float>&);

}// namespace rheolef
