# ifndef _SKIT_CG_H
# define _SKIT_CG_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*D:cg
NAME: @code{cg} -- conjugate gradient algorithm. 
@findex cg
@cindex conjugate gradient algorithm
@cindex iterative solver
SYNOPSIS:
    @example
	template <class Matrix, class Vector, class Preconditioner, class Real>
    	int cg (const Matrix &A, Vector &x, const Vector &b,
    		const Preconditioner &M, int &max_iter, Real &tol);
    @end example

EXAMPLE:
  @noindent
  The simplest call to 'cg' has the folling form:
    @example
  	int status = cg(a, x, b, EYE, 100, 1e-7);
    @end example

DESCRIPTION:       
  @noindent
  @code{cg} solves the symmetric positive definite linear
  system Ax=b using the Conjugate Gradient method.

  @noindent
  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:
  @table @code
    @itemx x
	approximate solution to Ax = b

    @itemx max_iter
	the number of iterations performed before the tolerance was reached

    @itemx tol
	the residual after the final iteration
  @end table

NOTE: 

  @noindent
  @code{cg} is an iterative template routine.

  @noindent
  @code{cg} follows the algorithm described on p. 15 in

  @quotation
	Templates for the Solution of Linear Systems: Building Blocks for Iterative Methods, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, 
	@url{ftp.netlib.org/templates/templates.ps}.
  @end quotation

  @noindent
  The present implementation is inspired from 
  @code{IML++ 1.2} iterative method library,
  @url{http://math.nist.gov/iml++}.

AUTHOR: 

    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France

DATE: 
    
    28 february 1997

METHODS: @cg
End:
*/
// ========== [ method implementation ] ====================================

namespace rheolef { 

template < class Matrix, class Vector, class Preconditioner, class Real>
int cg(const Matrix &A, Vector &x, const Vector &b,
       const Preconditioner &M, int &max_iter, Real &tol)
{
    Real resid;
    Vector p, q;
    Real  alpha, beta, rho, rho_1 = 0;

    Real normb = norm(b);
    Vector r = b - A*x;

    if (normb == Real(0)) 
        normb = 1;
  
    if ((resid = norm(r) / normb) <= tol) {
        tol = resid;
        max_iter = 0;
        return 0;
    }
    for (int i = 1; i <= max_iter; i++) {
    
        if (i == 1) {
	    Vector z = M.solve(r);
            rho = dot(r, z);
            p = z;
        } else {
	    Vector z = M.solve(r);
            rho = dot(r, z);
            beta = rho / rho_1;
            p = z + beta * p;
        }
        q = A*p;
        alpha = rho / dot(p, q);
    
        x += alpha * p;
        r -= alpha * q;
    
        if ((resid = norm(r) / normb) <= tol) {
          tol = resid;
          max_iter = i;
          return 0;     
        }
        rho_1 = rho;
    }
    tol = resid;
    return 1;
}
}// namespace rheolef
# endif // _SKIT_CG_H
