///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
# include "rheolef/asr.h"
# include "rheolef/csr.h"
using namespace std;
namespace rheolef {

// extern:
void read_mm_header(idiststream&);

// ----------------------------------------------------------------------------
// class member functions
// ----------------------------------------------------------------------------
template <class T>
asr_seq_rep<T>::asr_seq_rep(const distributor& row_ownership, const distributor& col_ownership)
  : vector<row_type>(row_ownership.size()),
    _row_ownership (row_ownership),
    _col_ownership (col_ownership),
    _nnz (0) 
{
}
template <class T>
void
asr_seq_rep<T>::resize(const distributor& row_ownership, const distributor& col_ownership)
{
  vector<row_type>::resize (row_ownership.size());
  _row_ownership = row_ownership;
  _col_ownership = col_ownership;
  _nnz = 0; 
}
template <class T>
asr_seq_rep<T>::asr_seq_rep (size_type loc_nrow, size_type loc_ncol)
  : vector<row_type>(),
    _row_ownership (distributor::decide, communicator(), loc_nrow),
    _col_ownership (distributor::decide, communicator(), loc_ncol),
    _nnz(0) 
{
  vector<row_type>::resize (_row_ownership.size());
}
template <class T>
void
asr_seq_rep<T>::resize (size_type loc_nrow, size_type loc_ncol)
{
  _row_ownership = distributor (distributor::decide, communicator(), loc_nrow);
  _col_ownership = distributor (distributor::decide, communicator(), loc_ncol);
  vector<row_type>::resize (_row_ownership.size());
  _nnz = 0; 
}
template <class T>
asr_seq_rep<T>::asr_seq_rep(const asr_seq_rep<T>& a) 
  : vector<row_type>(a),
    _row_ownership (a._row_ownership),
    _col_ownership (a._col_ownership),
    _nnz(a._nnz)
{
    fatal_macro ("physical copy of asr");
}
template <class T>
asr_seq_rep<T>::asr_seq_rep(const csr_seq_rep<T>& a)
  : vector<row_type>(a.nrow()),
    _row_ownership (a.row_ownership()),
    _col_ownership (a.col_ownership()),
    _nnz (a.nnz())
{
    a.to_asr(*this);
}
template <class T>
idiststream&
asr_seq_rep<T>::get (idiststream& ips) 
{
    // matrix market format:
    //   %%mm_header
    //   nrow ncol nnz
    //   {i, j, aij}*
    // we suppose nrow ncol already readed
    // and the matrix has good dimensions
    //  size_type nrow;
    // 	s >> nrow >> _ncol;
    //  vector<row_type>::resize(nrow,numeric_limits<T>::max());
    read_mm_header(ips);
    istream& is = ips.is();
    size_type n_row, n_col, n_nz;
    is >> n_row >> n_col >> n_nz;
    resize (n_row, n_col);
    size_type i, j;
    T aij;
    for (size_type p = 0; p < n_nz; p++) {
        is >> i >> j >> aij;
	// 1..N convention -> 0..N-1
	i--; j--;
	dis_entry(i,j) += aij;
    }
    return ips;
}
template <class T>
odiststream&
asr_seq_rep<T>::put (odiststream& ops, size_type istart) const
{
    std::ostream& os = ops.os();
    os << setprecision(std::numeric_limits<T>::digits10)
       << "%%MatrixMarket matrix coordinate real general" << std::endl
       << nrow() << " " << ncol() << " " << nnz() << endl;
    if (nnz() == 0) return ops;
    for (size_type i = 0; i < nrow(); i++) {
        typename row_type::const_iterator iter = base::operator[](i).begin();
        typename row_type::const_iterator last = base::operator[](i).end();
	while (iter != last) {
  	    os << i+istart+1 << " " 
	       << (*iter).first+1 << " " 
	       << (*iter).second << endl;
	    iter++;
	}
    }
    return ops;
}
template <class T>
void
asr_seq_rep<T>::dump (const string& name, size_type istart) const
{ 
	char iname [256];
	sprintf (iname, "%s.mtx", name.c_str());
        cerr << "! file \"" << iname << "\" created." << endl << flush;
	ofstream s(iname);
	s << nrow() << " " << ncol() << " " << nnz() << endl;
	odiststream ps(s);
	put(ps, istart);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class asr_seq_rep<Float>;
} // namespace rheolef
