///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// direct solver interface
//
#include "rheolef/solver.h"
#include "solver_wrapper.h"
#include "solver_mumps.h"
#include "solver_trilinos_ifpack.h"
#include "solver_no_trilinos_ifpack.h"
#include "solver_pastix.h"

namespace rheolef {

template<class T, class M>
solver_rep<T,M>::solver_rep ()
 : _ptr(0)
{
}
template<class T, class M>
solver_rep<T,M>::solver_rep (const csr<T,M>& a, const solver_option_type& opt)
 : _ptr(0)
{
  typedef solver_wrapper_rep<T,M> rep;
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
void
solver_rep<T,M>::build_lu (const csr<T,M>& a, const solver_option_type& opt)
{
#ifdef _RHEOLEF_HAVE_MUMPS
  typedef solver_mumps_rep<T,M> rep;
#elif defined(_RHEOLEF_HAVE_PASTIX)
  typedef solver_pastix_rep<T,M> rep;
  opt.iterative = 0;
#else // ! _RHEOLEF_HAVE_TRILINOS && !_RHEOLEF_HAVE_PASTIX
  typedef solver_no_trilinos_ifpack_rep<T,M> rep;
#endif // ! _RHEOLEF_HAVE_MUMPS
  if (_ptr) delete_macro (_ptr);
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
void
solver_rep<T,M>::build_ilu (const csr<T,M>& a, const solver_option_type& opt)
{
#ifdef _RHEOLEF_HAVE_TRILINOS
  typedef solver_trilinos_ifpack_rep<T,M> rep;
#elif defined(_RHEOLEF_HAVE_PASTIX)
  typedef solver_pastix_rep<T,M> rep;
  opt.iterative = 1;
#else // ! _RHEOLEF_HAVE_TRILINOS && !_RHEOLEF_HAVE_PASTIX
  typedef solver_no_trilinos_ifpack_rep<T,M> rep;
#endif
  if (_ptr) delete_macro (_ptr);
  _ptr = new_macro (rep(a,opt));
}
template<class T, class M>
solver_rep<T,M>::~solver_rep ()
{
  if (_ptr) delete_macro (_ptr);
  _ptr = 0;
}
template<class T, class M>
void 
solver_rep<T,M>::update_values (const csr<T,M>& a)
{
  _ptr->update_values (a);
}
template<class T, class M>
vec<T,M>
solver_rep<T,M>::solve (const vec<T,M>& b) const
{
  return _ptr->solve (b);
}
template<class T, class M>
vec<T,M>
solver_rep<T,M>::trans_solve (const vec<T,M>& b) const
{
  return _ptr->trans_solve (b);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class solver_rep<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class solver_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
