///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// solver wrapper: direct or iterative
//
#include "solver_wrapper.h"
#include "rheolef/pcg.h"
#include "rheolef/pgmres.h"
#include "rheolef/vec_expr_ops.h"
#include <boost/numeric/ublas/matrix.hpp>

namespace rheolef {
using namespace std;

template<class T, class M>
solver_wrapper_rep<T,M>::solver_wrapper_rep (const csr<T,M>& a, const solver_option_type& opt)
 : solver_abstract_rep<T,M>(opt),
  _is_direct (a.pattern_dimension() <= 2),
  _is_sym    (a.is_symmetric()),
  _sa(),
  _a()
{
  if (_is_direct) {
    if (_is_sym) { _sa = ldlt (a,opt); }
    else         { _sa = lu   (a,opt); }
  } else {
    if (_is_sym) { _sa = ic0  (a,opt); }
    else         { _sa = ilu0 (a,opt); }
    _a = a;
  }
}
template<class T, class M>
void
solver_wrapper_rep<T,M>::update_values (const csr<T,M>& a) 
{
  _sa.update_values (a);
  if (!_is_direct) { _a = a; }
}
template<class T, class M>
vec<T,M>
solver_wrapper_rep<T,M>::solve (const vec<T,M>& b) const
{
  if (_is_direct) { return _sa.solve (b); }

  T tol = std::numeric_limits<T>::epsilon();
  T tol_reached = tol;
  size_type max_iter = 1000; // TODO: use _opt to get tol & max_iter
  vec<T,M> x (b.ownership(), 0);
  if (_is_sym) {
    int status = pcg (_a, x, b, _sa, max_iter, tol_reached, &derr);
  } else {
    size_type m = 6;
    boost::numeric::ublas::matrix<T> h(m+1,m+1);
    //boost::numeric::ublas::vector<T> dummy(m);
    vec<T,sequential> dummy(m,0.0);
    trace_macro ("pgmres...");
    int status = pgmres (_a, x, b, _sa, h, dummy, m, max_iter, tol_reached);
    trace_macro ("pgmres done");
  }
  check_macro (tol_reached < tol,
	"solver: precision "<<tol<<" not reached: get "<<tol_reached
	<< " after " << max_iter << " iterations");
  return x;
}
template<class T, class M>
vec<T,M>
solver_wrapper_rep<T,M>::trans_solve (const vec<T,M>& b) const
{
  if (_is_direct) { return _sa.trans_solve (b); }

  error_macro ("iterative trans_solve: not yet");
  // TODO: wrap a as a*x return a.trans_mult(x) and the same with _sa.solve(b)
  return vec<T,M>();
}
template<class T, class M>
solver_wrapper_rep<T,M>::~solver_wrapper_rep()
{
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

template class solver_wrapper_rep<Float,sequential>;
#ifdef _RHEOLEF_HAVE_MPI
template class solver_wrapper_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
