///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// direct solver LDL^T test
//
// usage:
//     ldlt_tst | octave -q
//
//  machine-precision independent computation:
//     ldlt_tst -ndigit 12 | octave -q
//
#include "rheolef/ssk.h"
#include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

Float eps1 = 1e-7;
template <class T>
inline T my_round (const T& x) { return eps1*T(int(x/eps1+0.5)); }

bool do_trunc = false;

void 
check (const char* expr)
{
    static int i = 0;
    cout << "error" << i++ << " = ";
    if (!do_trunc) {
        cout << "norm(" << expr << ")" << endl;
    } else {
        cout << "eps1*round(norm(" << expr << ")/eps1)" << endl;
    }
}
int main(int argc, char** argv)
{
    int digits10 = numeric_limits<Float>::digits10;
    cout.unsetf(ios::fixed);
    cout.unsetf(ios::floatfield);

    // avoid machine-dependent output digits in non-regression mode:
    if (argc == 3) {
 	do_trunc = true;
        digits10 = atoi(argv[2]);
	cout << "eps1=sqrt(10^(-" << digits10 << "));\n";
    }
    eps1 = sqrt(pow(Float(10), Float(-digits10)));
    cout << setprecision(digits10);

    // example from Theodor and Lascaux, tome 1, page 303
    const unsigned int n = 5;
    dns<Float> LL(n,n);
    for (unsigned int i = 0; i < n; i++)
      for (unsigned int j = 0; j < n; j++)
        LL(i,j) = 0;
    for (unsigned int i = 0; i < n; i++)
      LL(i,i) = 1;
    LL(1,0) = -1.3;
    LL(2,1) = 2.4;
    LL(3,0) = 1.7;
    LL(3,2) = 1.5;
    LL(4,2) = 0.4;
    LL(4,3) = -2.4;
    csr<Float> L(LL);
    cout << "L = " << ml << L << ";\n";

    vec<Float> DD(n);
    DD(0) = 1;
    DD(1) = 2;
    DD(2) = 4;
    DD(3) = 1;
    DD(4) = 5;
    basic_diag<Float> D(DD);
    cout << "D = " << ml << DD << ";\n";

    csr<Float> A = L*D*trans(L);
    cout << "A = " << ml << A << ";\n";

    ssk<Float> LDLt = ldlt(A);
    
#ifdef NOT_PORTED
    // test: convert csr to ssk and go back
    // pb: csr2ssk and ssk2csr are not ported to spooles
    //     chevron data structure.
    ssk<Float> A_SSK = ssk<Float>(A);
    csr<Float> A1(A_SSK);
    cout << "A0 = " << ml << A1 << ";\n";
    cout << "A1=A0+A0'-diag(diag(A0))\n";
    cout << "LDLt = " << ml << csr<Float>(LDLt) << ";\n";
    cout << "L1 = tril(LDLt)-diag(diag(LDLt))+eye(" << n << ")\n";
    check ("L-L1");
    cout << "D1 = diag(LDLt)"  << ";\n";
    check ("D-D1");
#endif // NOT_PORTED

    vec<Float> xe(n);
    xe = 1;
    cout << "xe = " << ml << xe << ";\n";

    vec<Float> b = A*xe;
    cout << "b = " << ml << b << ";\n";
    cout << "b1 = A*xe"  << ";\n";
    check ("b-b1");

    vec<Float> x(n);
    LDLt.solve(b,x);
    cout << "x = " << ml << x << ";\n";
    cout << "x1 = inv(A)*b"  << ";\n";
    check ("x-x1");
    check ("A*x-b");
    Float residu = norm(A*x-b);
    cout << "residu = " << double(my_round(residu)) << ";\n";
    check ("residu");

#ifdef BUG
    // test empty matrix size(0,0)
    cerr << "test empty matrix size(0,0)..." << endl;
    csr<Float> c0;
    ssk<Float> a0 = ssk<Float>(c0);
    cout << "null_0 = " << ml << csr<Float>(a0) << ";\n";

    // test empty matrix size(5,5)
    cerr << "test empty matrix size(5,5)..." << endl;
    csr<Float> c5(5,5);
    ssk<Float>  a5 = ssk<Float>(c5);
    cout << "null_5 = " << ml << csr<Float>(a5) << ";\n";
#endif // BUG

    return 0;
}

