///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/geo_element.h"

namespace rheolef {

// --------------------------------------------------------------------------
// i/o
// --------------------------------------------------------------------------
char
skip_blancs_and_tabs (std::istream& is)
{
    if (!is.good()) return 0;
    char c = is.peek();
    while (is.good() && (c == ' ' || c == '\t')) {
	is >> c;
    } while (is.good() && (c == ' ' || c == '\t'));
    return c;
}
void
geo_element::get (std::istream& is)
{
    geo_element& K = *this; // for more readability
    typedef geo_element::size_type size_type;
    extern char skip_blancs_and_tabs (std::istream&);
    char c;
    is >> std::ws >> c;
    if (!isdigit(c)) {
      // have an element type: 'e', 't' etc
      geo_element::variant_type variant = reference_element::variant (c);
      check_macro (variant != reference_element::max_variant, "undefined element variant `"<<c<<"'");
      size_type order = 1;
      char c2;
      is >> std::ws >> c2;
      if (c2 == 'p') {
	// have a high order spec as "p2"
	is >> order;
      } else {
        is.unget(); // or is.putback(c); ?
      }
      K.reset (variant, order);
      for (size_type i = 0, n = K.n_node(); i < n; i++) {
	is >> K[i];
      }
      return;
    }
    // here, starts with a digit: triangle (e.g. "21 22 23"), edge (e.g. "12 13") or point (e.g. "11")
    // end of element input: with end-of-line or non-digit input char
    // Note: with 4 vertices, there is an ambiguity between quadrangle and thetraedra
    size_type tmp [3];
    size_type nvertex = 0;
    while (is.good() && isdigit(c) && nvertex < 3) {
        is.unget(); // or is.putback(c); ?
        is >> tmp [nvertex];
        nvertex++;
        c = skip_blancs_and_tabs (is);
    }
    size_type variant = reference_element::max_variant;
    switch (nvertex) {
     case 1: variant = reference_element::p; break;
     case 2: variant = reference_element::e; break;
     case 3: variant = reference_element::t; break;
     default: error_macro ("unexpected element with "<<nvertex<<" vertices");
    }
    K.reset (variant, 1);
    for (size_type i = 0, n = K.n_node(); i < n; i++) {
      K[i] = tmp[i];
    }
    return;
}
void
geo_element::put (std::ostream& os) const
{
  typedef geo_element::size_type size_type;
  os << name() << "\tp" << order();
  for (size_type loc_inod = 0, loc_nnod = n_node(); loc_inod < loc_nnod; loc_inod++) {
    os << " " << operator[](loc_inod);
  }
}
// --------------------------------------------------------------------------
/// @brief return orientation and shift between *this element and S
/** assume that vertices of *this and S match
 * this routine is used for domain elements identification
 *  shift     : such that vertex (*this)[0] == S[shift] matches
 *  orient=+1 : when (*this)[1] == S[shift+1]
 * return true when all is ok (does not check fully that elements match)
 */
bool
geo_element::get_orientation_and_shift (
    const geo_element& S,
    orientation_type&  orient,
    shift_type&        shift) const
{
    check_macro (S.dimension() == dimension(),
	"get_orientation_and_shift: elements have different dimensions "<<dimension()<<" and "<<S.dimension());
    check_macro (S.size() == size(),
	"get_orientation_and_shift: elements have different sizes "<<size()<<" and "<<S.size());
    if (S.dimension() == 0) {
      orient = 1;
      shift = 0;
      return true;
    }
    if (S.dimension() == 1) {
      orient = (operator[](0) == S[0]) ? 1 : -1;
      shift = 0;
      return true;
    }
    if (S.dimension() == 2) {
      size_type n = size();
      // 3d face: triangle or quadrangle
      for (shift = 0; shift < shift_type(n); shift++) {
        if (operator[](0) == S[shift]) break;
      }
      if (shift == shift_type(n)) {
        orient = 0;
        shift = std::numeric_limits<shift_type>::max();
        return false;
      }
      orient = (operator[](1) == S[(shift+1)%n]) ? 1 : -1;
      return true;
    }
    // S.dimension() == 3: TODO volumic domain: tetra can be rotated ?
    orient = 1;
    shift = 0;
    return true;
}
// if edge (dis_iv0,dis_iv1) has the same orientation as current edge
// assume that current geo_element is an edge that contains the same vertices
geo_element::orientation_type
geo_element::get_edge_orientation (
    size_type dis_iv0, size_type dis_iv1) const
{
    return (operator[](0) == dis_iv0) ? 1 : -1; 
}
// for a triangular side (dis_iv0,dis_iv1,dis_iv2)
// assume that current geo_element is a triangle that contains the same vertices
void
geo_element::get_orientation_and_shift (
    size_type dis_iv0, size_type dis_iv1, size_type dis_iv2,
    orientation_type&  orient,
    shift_type&        shift) const
{
    if         (operator[](0) == dis_iv0) {
      orient = (operator[](1) == dis_iv1) ? 1 : -1; 
      shift = 0;
    } else  if (operator[](0) == dis_iv1) {
      orient = (operator[](1) == dis_iv2) ? 1 : -1; 
      shift = 1;
    } else {
      orient = (operator[](1) == dis_iv0) ? 1 : -1; 
      shift = 2;
    }
}
// for a quadrangular side (dis_iv0,dis_iv1,dis_iv2,dis_iv3)
// assume that current geo_element is a quadrangular that contains the same vertices
void
geo_element::get_orientation_and_shift (
    size_type dis_iv0, size_type dis_iv1, size_type dis_iv2, size_type dis_iv3,
    orientation_type&  orient,
    shift_type&        shift) const
{
    if         (operator[](0) == dis_iv0) {
      orient = (operator[](1) == dis_iv1) ? 1 : -1; 
      shift = 0;
    } else  if (operator[](0) == dis_iv1) {
      orient = (operator[](1) == dis_iv2) ? 1 : -1; 
      shift = 1;
    } else  if (operator[](0) == dis_iv2) {
      orient = (operator[](1) == dis_iv3) ? 1 : -1; 
      shift = 2;
    } else {
      orient = (operator[](1) == dis_iv0) ? 1 : -1; 
      shift = 3;
    }
}
// let K=(*this) one element and S one side of K
// S could have the good or the opposite orientation on K:
// gives its sign, local side index and node shift (in 3D)
// Note: assume that S is a valid side of K
geo_element::orientation_type
geo_element::get_side_informations (
  const geo_element& S,
  size_type& loc_isid,
  size_type& shift) const
{
  loc_isid = shift = 0;
  check_macro (S.dimension() + 1 == dimension(),
	"get_side_orientation: side have unexpected dimension "<<S.dimension()<<": "
	    <<dimension()<<"-1 was expected");
  const geo_element& K = *this;
  size_type side_dim = S.dimension();
  for (size_type loc_nsid = K.n_subgeo(side_dim); loc_isid < loc_nsid; loc_isid++) {
    size_type sid_nloc = K.subgeo_size (side_dim, loc_isid);
    if (sid_nloc != S.size()) continue;
    for (shift = 0; shift < sid_nloc; shift++) {
      size_type loc_jv = K.subgeo_local_vertex (side_dim, loc_isid, shift);
      if (K[loc_jv] != S[0]) continue;
      // one node matches with S[0] on K: loc_isid and shift
      // check others nodes in a first rotation direction:
      bool matches = true;
      for (size_type sid_kloc = 1; sid_kloc < sid_nloc; sid_kloc++) {
        size_type loc_kv = K.subgeo_local_vertex (side_dim, loc_isid, (shift+sid_kloc) % sid_nloc);
        if (K[loc_kv] != S[sid_kloc]) { matches = false; break; }
      }
      if (matches) {
        if (side_dim == 1 && shift == 1) {
          // shift=1 for an edge means a change of orientation of this edge (for DG)
          shift = 0;
	  return -1; 
        }
	return 1; 
      }
      // check others nodes in the opposite rotation direction:
      matches = true;
      for (size_type sid_kloc = 1; sid_kloc < sid_nloc; sid_kloc++) {
        size_type loc_kv = K.subgeo_local_vertex (side_dim, loc_isid, (shift+sid_nloc-sid_kloc) % sid_nloc);
        if (K[loc_kv] != S[sid_kloc]) { matches = false; break; }
      }
      if (matches) { return -1; }
    }
  }
  fatal_macro ("get_side_orientation: side is not part of the element");
  return 0; // not reached
}
void
geo_element::get_side_informations (
  const geo_element& S,
  side_information_type& sid) const
{
  sid.orient   = get_side_informations (S, sid.loc_isid, sid.shift);
  sid.dim      = S.dimension();
  sid.n_vertex = subgeo_size (sid.dim, sid.loc_isid);
  sid.hat.set_variant (sid.n_vertex, sid.dim);
}
geo_element::orientation_type
geo_element::get_side_orientation (const geo_element& S) const
{
  size_type loc_isid, shift;
  return get_side_informations (S, loc_isid, shift);
}
// =========================================================================
// fix rotation and orientation on a 2d edge or 3d face
// =========================================================================
// --------------
// 1) edges
// --------------
geo_element::size_type
geo_element::fix_edge_indirect (
      orientation_type  orient,
      size_type         order,
      size_type         loc_iedg_j)
{
  if (orient > 0) return loc_iedg_j;
  return order - 2 - loc_iedg_j;
}
geo_element::size_type
geo_element::fix_edge_indirect (
  const geo_element& K,
  size_type          loc_iedg,
  size_type          loc_iedg_j, 
  size_type          order)
{
  // shift and/or inverse-orient the lattice(i,j)
  if (K.dimension() < 2 || order <= 2) {
    return loc_iedg_j;
  }
  return fix_edge_indirect (K.edge_indirect(loc_iedg).orientation(), order, loc_iedg_j);
}
// --------------
// 2) triangles
// --------------
void
geo_element::loc_tri_inod2lattice (
  size_type               loc_tri_inod,
  size_type               order,
  point_basic<size_type>& ij_lattice)
{
  // search (i,j) integers such that
  //     loc_tri_inod = (n_face_node - (j1-1)*j1/2) + i-1, where j1=order-1-j
  // first, search j1 integer such that for i=1:
  //     i - 1 = 0 = loc_tri_inod - (n_face_node - (j1-1)*j1/2);
  // <=> j1^2 - j1 - 2*(n_face_node - loc_tri_inod) = 0
  size_type n_face_node = (order-1)*(order-2)/2;
  double a = 1;
  double b = -1;
  double c = -2*int(n_face_node - loc_tri_inod);
  double delta = b*b - 4*a*c;
  check_macro (delta >= 0, "loc_tri_inod2lattice(loc_tri_inod="<<loc_tri_inod<<",order="<<order<<"): unexpected delta="<<delta<<" < 0");
  double j1_plus = (-b + sqrt(delta))/(2*a);
  //   j1_minus = (-b - sqrt(delta))/(2*a); is negative always
  size_type j = floor (order - j1_plus);
  size_type j1 = order - j;
  size_type i = loc_tri_inod - (n_face_node - (j1-1)*j1/2) + 1 ;
  ij_lattice = point_basic<size_type>(i,j);
}
geo_element::size_type
geo_element::fix_triangle_indirect (
      orientation_type  orient,
      shift_type        shift,
      size_type         order,
      size_type         loc_itri_j)
{
  // 1) compute lattice coord (i,j) on this face, from current loc_itri_j
  point_basic<size_type> ij_lattice;
  loc_tri_inod2lattice (loc_itri_j, order, ij_lattice);
  size_type i = ij_lattice[0];
  size_type j = ij_lattice[1];
  // 2) then shift and/or inverse-orient (i,j)
  size_type coord[3];
  coord [0] = i;
  coord [1] = j;
  coord [2] = order - i - j;
  size_type i_fix = coord [shift%3];
  size_type j_fix = coord [(shift+1)%3];
  if (orient < 0) std::swap (i_fix, j_fix);
  // 3) re-compute the fixed loc_itri_j face index
  size_type loc_tri_nnod = (order-1)*(order-2)/2;
  size_type j1_fix = order - j_fix;
  size_type loc_itri_j_fix = (loc_tri_nnod - (j1_fix-1)*j1_fix/2) + (i_fix-1);
  return loc_itri_j_fix;
}
geo_element::size_type
geo_element::fix_triangle_indirect (
  const geo_element& K,
  size_type          loc_ifac,
  size_type          loc_itri_j, 
  size_type          order)
{
  // shift and/or inverse-orient the lattice(i,j)
  if (K.dimension() < 3 || 
      (K.face_indirect(loc_ifac).orientation() > 0 && K.face_indirect(loc_ifac).shift() == 0)) {
    return loc_itri_j;
  }
  return fix_triangle_indirect (
      K.face_indirect(loc_ifac).orientation(),
      K.face_indirect(loc_ifac).shift(),
      order,
      loc_itri_j);
}
// --------------
// 3) quadrangles
// --------------
void
geo_element::loc_qua_inod2lattice (
  size_type               loc_qua_inod,
  size_type               order,
  point_basic<size_type>& ij_lattice)
{
  // search (i,j) integers in [0:order-1[ such that
  //     loc_qua_inod = (order-1)*(j-1) + (i-1);
  ij_lattice[0] = (loc_qua_inod % (order-1)) + 1;
  ij_lattice[1] = (loc_qua_inod / (order-1)) + 1;
}
geo_element::size_type
geo_element::fix_quadrangle_indirect (
      orientation_type  orient,
      shift_type        shift,
      size_type         order,
      size_type         loc_iqua_j)
{
  // 1) compute lattice coord (i,j) on this face, from current loc_itri_j
  point_basic<size_type> ij_lattice;
  loc_qua_inod2lattice (loc_iqua_j, order, ij_lattice);
  size_type i = ij_lattice[0];
  size_type j = ij_lattice[1];
  // 2) then shift and/or inverse-orient (i,j)
  size_type coord [4];
  coord [0] = i;
  coord [1] = j;
  coord [2] = order - i;
  coord [3] = order - j;
  size_type i_fix = coord [shift%4];
  size_type j_fix = coord [(shift+1)%4];
  if (orient < 0) std::swap (i_fix, j_fix);
  // 3) re-compute the fixed loc_iqua_j face index
  size_type loc_iqua_j_fix = (order-1)*(j_fix-1) + (i_fix-1);
  return loc_iqua_j_fix;
}
geo_element::size_type
geo_element::fix_quadrangle_indirect (
  const geo_element& K,
  size_type          loc_ifac,
  size_type          loc_iqua_j, 
  size_type          order)
{
  // shift and/or inverse-orient the lattice(i,j)
  if (K.dimension() < 3 || order <= 2 ||
      (K.face_indirect(loc_ifac).orientation() > 0 && K.face_indirect(loc_ifac).shift() == 0)) {
    return loc_iqua_j;
  }
  return fix_quadrangle_indirect (
      K.face_indirect(loc_ifac).orientation(),
      K.face_indirect(loc_ifac).shift(),
      order,
      loc_iqua_j);
}

} // namespace rheolef
