#ifndef _RHEOLEF_SOLVER_OPTION_TYPE_H
#define _RHEOLEF_SOLVER_OPTION_TYPE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// options for solvers
//
#include "rheolef/csr.h"

namespace rheolef {

/*Class:solver
NAME:   @code{solver_option_type} - options for direct or interative solvers
@clindex solver
DESCRIPTION:
  @noindent
  This class implements a set of options for direct or iterative solvers,
  as implemented by the @ref{solver class}. An instance of this class
  can be supplied to the @code{solver} constructor.

AUTHOR: Pierre.Saramito@imag.fr
DATE:   4 march 2011
end:
*/
//<verbatim:
class solver_option_type {
public:
  typedef std::size_t size_type;

  static const long int  decide = -1;    // indicate the solver will try to choose the best method

// data:

  mutable long int   iterative;          // indicate if we want to use iterative solver
  Float 	     tol;		 // tolerance when using iterative methode
  size_type	     max_iter;		 // maximum number of iteration when using iterative method
  size_type	     n_refinement;	 // number of iterative refinement, when using direct method (umfpack only support it)
  bool               compute_determinant; // compute also det, when using direct method
  std::string        prefered_library;   // e.g. "umfpack", when available
  size_type          verbose_level;      // 0, 1, 2, 3
  bool               do_check;
  bool               force_seq;		 // (still buggy; when using direct) restrict to diagonal blocs per process (e.g. mumps solver)
  size_type          level_of_fill;      // (in development) Level of fill [1:5] for incomplete factorisation
  size_type          amalgamation;       // (in development) Level of amalgamation [10:70] for Kass                         
  size_type          ooc;                // (in development) out-of-core limit (Mo/percent depending on compilation options) 

// allocator with default values:

  solver_option_type ()
   : iterative     (decide),
#if defined(_RHEOLEF_HAVE_QD) || defined(_RHEOLEF_HAVE_FLOAT128)
     tol           (1e6*std::numeric_limits<Float>::epsilon()),
#else
     tol           (std::numeric_limits<Float>::epsilon()),
#endif
     max_iter      (100000),
     n_refinement  (2),
     compute_determinant(false),
     prefered_library(),
     verbose_level (0),
     do_check      (false),
     force_seq     (false),
     level_of_fill (1),
     amalgamation  (10),
     ooc           (20000)
  {
  }
  solver_option_type (const solver_option_type&);
  solver_option_type& operator= (const solver_option_type&);
};
//>verbatim:

// ---------------------------------------------------------------
// inlined
// ---------------------------------------------------------------
inline
solver_option_type::solver_option_type (const solver_option_type& x)
   : iterative     (x.iterative),
     tol           (x.tol),
     max_iter      (x.max_iter),
     n_refinement  (x.n_refinement),
     compute_determinant(x.compute_determinant),
     verbose_level (x.verbose_level),
     do_check      (x.do_check),
     force_seq     (x.force_seq),
     level_of_fill (x.level_of_fill),
     amalgamation  (x.amalgamation),
     ooc           (x.ooc)
{
}
inline
solver_option_type& 
solver_option_type::operator= (const solver_option_type& x)
{
     iterative     = x.iterative;
     tol           = x.tol;
     max_iter      = x.max_iter;
     n_refinement  = x.n_refinement;
     compute_determinant = x.compute_determinant;
     verbose_level = x.verbose_level;
     do_check      = x.do_check;
     force_seq     = x.force_seq;
     level_of_fill = x.level_of_fill;
     amalgamation  = x.amalgamation;
     ooc           = x.ooc;
     return *this;
}

} // namespace rheolef
#endif // _RHEOLEF_SOLVER_OPTION_TYPE_H
