/*
  Copyright (c) 2010 Fizians SAS. <http://www.fizians.com>
  This file is part of Rozofs.

  Rozofs is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation, version 2.

  Rozofs is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see
  <http://www.gnu.org/licenses/>.
 */
#ifndef UMA_DBG_API_H
#define UMA_DBG_API_H

#include <stdio.h>

#include <rozofs/common/types.h>
#include <rozofs/common/log.h>
#include <rozofs/core/rozofs_string.h>

#include "uma_dbg_msgHeader.h"
#include "ruc_common.h"
#include "rozofs_string.h"
#include "ruc_buffer_api.h"
#include "uma_tcp_main_api.h"


#define   UMA_DBG_OPTION_HIDE           (1<<0)
#define   UMA_DBG_OPTION_RESET          (1<<1)

/*
** Max length of the user payload when answering to a debug command
*/
#define UMA_DBG_MAX_SEND_SIZE (1024*384)
extern char uma_dbg_temporary_buffer[];
extern uint32_t uma_dbg_do_not_send;
extern char   * uma_gdb_system_name;
extern char     rcvCmdBuffer[];

/*__________________________________________________________________________
 */
/**
*  Display bytes with correct unit 
*  @param value         Value in bytes to display
*  @param value_string  String where to format the value
*/
static inline int uma_dbg_byte2String(uint64_t value, char * value_string) {
  uint64_t   modulo=0;
  char     * pt = value_string;
  
  if (value<1000) {
    pt += rozofs_u64_append(pt,value);
    pt += rozofs_string_append(pt," Bytes");
    return (pt-value_string);  		    
  }
  
  if (value<1000000) {
  
    if (value>99000) {
      pt += rozofs_u64_append(pt,value/1000);
      pt += rozofs_string_append(pt," KB");
      return (pt-value_string);    		    
    }
    
    modulo = (value % 1000) / 100;
    pt += rozofs_u64_append(pt,value/1000);
    *pt++ = '.';
    pt += rozofs_u32_padded_append(pt,1,rozofs_zero,modulo);
    pt += rozofs_string_append(pt," KB");
    return (pt-value_string); 
  }
  
  if (value<1000000000) {
  
    if (value>99000000) {
      pt += rozofs_u64_append(pt,value/1000000);
      pt += rozofs_string_append(pt," MB");
      return (pt-value_string);    		    
    }
    
    modulo = (value % 1000000) / 100000;
    pt += rozofs_u64_append(pt,value/1000000);
    *pt++ = '.';
    pt += rozofs_u32_padded_append(pt,1,rozofs_zero,modulo);
    pt += rozofs_string_append(pt," MB");
    return (pt-value_string);     
  }
    
  if (value<1000000000000) {
  
    if (value>99000000000) {
      pt += rozofs_u64_append(pt,value/1000000000);
      pt += rozofs_string_append(pt," GB");
      return (pt-value_string);    		    
    }
    
    modulo = (value % 1000000000) / 100000000;
    pt += rozofs_u64_append(pt,value/1000000000);
    *pt++ = '.';
    pt += rozofs_u32_padded_append(pt,1,rozofs_zero,modulo);
    pt += rozofs_string_append(pt," GB");
    return (pt-value_string);     
  }  
  
  if (value>99000000000000) {
    pt += rozofs_u64_append(pt,value/1000000000000);
    pt += rozofs_string_append(pt," PB");
    return (pt-value_string);     
  }  

  modulo = (value % 1000000000000) / 100000000000;
  pt += rozofs_u64_append(pt,value/1000000000000);
  *pt++ = '.';
  pt += rozofs_u32_padded_append(pt,1,rozofs_zero,modulo);      
  pt += rozofs_string_append(pt," PB");
  return (pt-value_string);   
}
/*__________________________________________________________________________
*  Add a thread in the thread table
** @param tid    The thread identifier
** @param name   The function of the tread
*/
void uma_dbg_thread_add_self(char * name);
/*__________________________________________________________________________
*  Remove a thread from the thread table
*/
void uma_dbg_thread_remove_self(void) ;
/*__________________________________________________________________________
*  Get the thread name of a thread
** @param tid    The thread identifier
** @retval the name of the thread or NULL
*/
char * uma_dbg_thread_get_name(pthread_t tid);

/*__________________________________________________________________________
 */
/**
*  Format an ASCII dump
* @param mem   memory area to dump
* @param len   size to dump mem on
* @param p     where to output the dump
*
*  return the address of the end of the dump 
*/
char * uma_dbg_hexdump(void *mem, unsigned int len, char * p);

/*__________________________________________________________________________
 */
/**
*  Return a temporary buffer where one can format a response
*/
static inline char * uma_dbg_get_buffer() {return uma_dbg_temporary_buffer;}
/*__________________________________________________________________________
 */
/**
*  Return the size of the temporary buffer where one can format a response
*/
static inline int uma_dbg_get_buffer_len() {return UMA_DBG_MAX_SEND_SIZE;}
/*
**--------------------------------------------------------------------------
**  #SYNOPSIS
**   Read and execute a rozodiag command file if it exist
**
**
**   IN:
**       The command file name to execute
**
**   OUT : none
**
**
**--------------------------------------------------------------------------
*/
void uma_dbg_process_command_file(char * command_file_name);
/*-----------------------------------------------------------------------------
**
**   Send back a diagnostic response
**
**  @param tcpCnxRef   TCP connection reference
**  @param bufRef      reference of the received buffer that will be used to respond
**  @param end         whether this is the last buffer of the response 
**  @param string      A pre-formated string ontaining the reponse 
**
**----------------------------------------------------------------------------
*/
static inline void uma_dbg_send(uint32_t tcpCnxRef, void  *bufRef, uint8_t end, char *string) {
  UMA_MSGHEADER_S *pHead;
  char            *pChar;
  uint32_t           len;

  /* 
  ** May be in a specific process such as counter reset
  ** and so do not send any thing
  */
  if (uma_dbg_do_not_send) return;
  
  /* Retrieve the buffer payload */
  if ((pHead = (UMA_MSGHEADER_S *)ruc_buf_getPayload(bufRef)) == NULL) {
    severe( "ruc_buf_getPayload(%p)", bufRef );
    /* Let's tell the caller fsm that the message is sent */
    return;
  }
  pChar = (char*) (pHead+1);
  
  pChar += rozofs_string_append(pChar,"____[");
  pChar += rozofs_string_append(pChar,uma_gdb_system_name);
  pChar += rozofs_string_append(pChar,"]__[");  
  pChar += rozofs_string_append(pChar,rcvCmdBuffer);
  pChar += rozofs_string_append(pChar,"]____\n");  
  pChar += rozofs_string_append(pChar,string);
  
  len = pChar - (char*)pHead;
  len ++;

  if (len > UMA_DBG_MAX_SEND_SIZE)
  {
    severe("debug response exceeds buffer length %u/%u",len,(int)UMA_DBG_MAX_SEND_SIZE);
  }

  pHead->len = htonl(len-sizeof(UMA_MSGHEADER_S));
  pHead->end = end;

  ruc_buf_setPayloadLen(bufRef,len);
  uma_tcp_sendSocket(tcpCnxRef,bufRef,0);
}
/*
   The function uma_dbg_addTopic enables to declare a new topic to
   the debug module. You have to give :

   1) topic :
   The topic is a string that has to be unic among the list of all
   topics knwon by the debug module. If you give a topic name that
   already exist, the debug module process an ERRLOG. So change
   the topic names.

   2) funct :
   A function with a specific prototype that will be called when
   the debug module receives a command beginning with the
   corresponding topic name.


   For instance : you declare a topic named "capicone"
   uma_dbg_addTopic("capicone", capicone_debug_entry);

   When on debug session is entered the following command
   line
     capicone     234 0x56    lulu         0o0
   the function capicone_debug_entry is called with

   - argv[] :
     argv[0] = "capicone"
     argv[1] = "234"
     argv[2] = "0x56"
     argv[3] = "lulu"
     argv[4] = "0o0"
     argv[5]...argv[40] = NULL

   - tcpRef : the reference of the debug session that the command
              comes from

   - bufRef : a reference a buffer in which  Y O U   H A V E  to
     put the response (note that the buffer is 2048 bytes length)

   To send a response, you are given an API uma_dbg_send which
   prototype is a bit like printf :
   - tcpCnxRef : is the reference of the debug session
   - bufRef : is the buffer for the response
   - end : is true when your response is complete, and false
           when a extra response buffer is to be sent. It is
	   up to you to get an other buffer.
   Examples :
   If you are happy
     uma_dbg_send(tcpCnxRef,bufRef,TRUE,"I am happy");
   or if you think that argv[1 has a bad value
     uma_dbg_send(tcpCnxRef,bufRef,TRUE,"2nd parameter has a bad value \"%s\"", argv[1]);
   or
     uma_dbg_send(tcpCnxRef,bufRef,TRUE,"IP addres is %u.%u.%u.%u",
                  ip>>24 & 0xFF, ip>>16 & 0xFF, ip>>8 & 0xFF, ip & 0xFF);

 */
 //64BITS typedef void (*uma_dbg_topic_function_t)(char * argv[], uint32_t tcpRef, uint32 bufRef);
typedef void (*uma_dbg_topic_function_t)(char * argv[], uint32_t tcpRef, void *bufRef);
typedef void (*uma_dbg_manual_function_t)(char * pt);


/*
** Adding a topic
*/
void uma_dbg_addTopicAndMan(char * topic, uma_dbg_topic_function_t funct, uma_dbg_manual_function_t man, uint16_t option);
// For compatibility with older interface
#define uma_dbg_addTopic_option(topic, funct, opt) uma_dbg_addTopicAndMan(topic, funct, NULL, opt);
#define uma_dbg_addTopic(topic, funct) uma_dbg_addTopicAndMan(topic, funct, NULL, 0);


void uma_dbg_hide_topic(char * topic);
void uma_dbg_init(uint32_t nbElements, uint32_t ipAddr, uint16_t serverPort) ;
//64BITS void uma_dbg_send(uint32_t tcpCnxRef, uint32 bufRef, uint8_t end, char *fmt, ... );
void uma_dbg_send_format(uint32_t tcpCnxRef, void *bufRef, uint8_t end, char *fmt, ... ); 
void uma_dbg_set_name( char * system_name) ;

//64BITS typedef uint32_t (*uma_dbg_catcher_function_t)(uint32 tcpRef, uint32 bufRef);
typedef uint32_t (*uma_dbg_catcher_function_t)(uint32_t tcpRef, void *bufRef);

//64BITS uint32_t uma_dbg_catcher_DFT(uint32 tcpRef, uint32 bufRef);
uint32_t uma_dbg_catcher_DFT(uint32_t tcpRef, void *bufRef);
void uma_dbg_setCatcher(uma_dbg_catcher_function_t funct);
/*__________________________________________________________________________
 */
/**
*  Run a system command and return the result 
*/
int uma_dbg_run_system_cmd(char * cmd, char *result, int len);

/*__________________________________________________________________________
 */
/**
*  Declare the path where to serach for core files
*/
void uma_dbg_declare_core_dir(char * path);
/*__________________________________________________________________________
*  Record syslog name
*
* @param name The syslog name
*/
void uma_dbg_record_syslog_name(char * name);
#endif
