/*
  Copyright (c) 2010 Fizians SAS. <http://www.fizians.com>
  This file is part of Rozofs.

  Rozofs is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation, version 2.

  Rozofs is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see
  <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <inttypes.h>

#include <rozofs/common/list.h>
#include <rozofs/common/log.h>
#include <rozofs/common/profile.h>
#include <rozofs/rpc/epproto.h>

#include "config.h"
#include "monitor.h"
#include "econfig.h"

DECLARE_PROFILING(epp_profiler_t);

#define HEADER "\
# This file was generated by exportd(8) version: %s.\n\
# All changes to this file will be lost.\n\n"

int monitor_initialize() {
    int status = -1;
    char path[FILENAME_MAX];
    DEBUG_FUNCTION;

    sprintf(path, "%s%s", DAEMON_PID_DIRECTORY, "exportd");
    if (access(path, X_OK) != 0) {
        if (mkdir(path, S_IRWXU) != 0) {
            severe("can't create %s", path);
            goto out;
        }
    }
    status = 0;
out:
    return status;
}

void monitor_release() {
    //XXX should clean MONITOR_DIRECTORY
    return;
}

int monitor_volume(volume_t *volume) {
    int status = -1;
    int fd = -1;
    char path[FILENAME_MAX];
    volume_stat_t vstat;
    list_t *p, *q;
    volume_t clone;
    uint32_t nb_storages = 0;
    int local_site = export_get_local_site_number();

    volume_initialize(&clone, 0, 0, 0, 0);
    if (volume_safe_copy(&clone, volume) != 0) {
        severe("can't clone volume: %d", volume->vid);
        goto out;
    }

    sprintf(path, "%s%s%d", DAEMON_PID_DIRECTORY, "exportd/volume_", clone.vid);
    if ((fd = open(path, O_WRONLY | O_CREAT | O_TRUNC, S_IRWXU | S_IROTH)) < 0) {
        severe("can't open %s", path);
        goto out;
    }

    gprofiler.vstats[gprofiler.nb_volumes].vid = clone.vid;
    gprofiler.vstats[gprofiler.nb_volumes].georep = clone.georep;

    dprintf(fd, HEADER, VERSION);
    dprintf(fd, "volume: %u\n", clone.vid);
    dprintf(fd, "  georep: %s\n", clone.georep?"YES":"NO");
    dprintf(fd, "  multi site: %s\n", clone.multi_site?"YES":"NO");

    //XXX TO CHANGE
    volume_stat(&clone,&vstat);
    gprofiler.vstats[gprofiler.nb_volumes].bsize = vstat.bsize;
    gprofiler.vstats[gprofiler.nb_volumes].bfree = vstat.bfree;
    gprofiler.vstats[gprofiler.nb_volumes].blocks = vstat.blocks;

    dprintf(fd, "  bsize: %u\n", vstat.bsize);
    dprintf(fd, "  bfree: %"PRIu64"\n", vstat.bfree);
    dprintf(fd, "  blocks: %"PRIu64"\n", vstat.blocks);
    if (vstat.blocks == 0) {
      dprintf(fd, "  %cfree: 0\n",'%');
    }
    else {
      dprintf(fd, "  %cfree: %"PRIu64"\n", '%', vstat.bfree*100/vstat.blocks);
    }
    dprintf(fd, "  nb_clusters: %d\n", list_size(&clone.clusters));

    list_for_each_forward(p, &clone.clusters) {
        cluster_t *cluster = list_entry(p, cluster_t, list);
        dprintf(fd, "    cluster: %u\n", cluster->cid);
        dprintf(fd, "      nb_storages: %d\n", list_size((&cluster->storages[local_site])));
        dprintf(fd, "      size: %"PRIu64"\n", cluster->size);
        dprintf(fd, "      free: %"PRIu64"\n", cluster->free);
	if (cluster->size==0) {
          dprintf(fd, "      %cfree: 0\n",'%');	   
	}
	else {
          dprintf(fd, "      %cfree: %"PRIu64"\n",'%', cluster->free*100/cluster->size);     
	}   
	
	if (clone.georep) {
	  dprintf(fd, "        site: %d\n", local_site); 
	}
        list_for_each_forward(q, (&cluster->storages[local_site])) {
            volume_storage_t *storage = list_entry(q, volume_storage_t, list);

            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].cid = cluster->cid;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].sid = storage->sid;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].status = storage->status;
	    strcpy(gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].host,storage->host);
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].size = storage->stat.size;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].free = storage->stat.free;
	    if (clone.georep) 
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = local_site;
	    else  
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = storage->siteNum;
            nb_storages++;

            dprintf(fd, "          storage: %u\n", storage->sid);
	    dprintf(fd, "            host: %s\n", storage->host);
	    if (clone.multi_site) {
            dprintf(fd, "            site: %d\n", storage->siteNum);
	    }  
            dprintf(fd, "            status: %s\n", (storage->status==1)?"UP":"DOWN");
            dprintf(fd, "            size: %"PRIu64"\n", storage->stat.size);
            dprintf(fd, "            free: %"PRIu64"\n", storage->stat.free);
	    if (storage->stat.size==0) {
              dprintf(fd, "            %cfree: 0\n",'%');	   
	    }
	    else {
              dprintf(fd, "            %cfree: %"PRIu64"\n",'%', storage->stat.free*100/storage->stat.size);	   
	    }   
        }
	
	if (clone.georep) {
	  dprintf(fd, "        site: %d\n", 1-local_site); 
          list_for_each_forward(q, (&cluster->storages[1-local_site])) {
              volume_storage_t *storage = list_entry(q, volume_storage_t, list);

              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].cid = cluster->cid;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].sid = storage->sid;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].status = storage->status;
	      strcpy(gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].host,storage->host);
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].size = storage->stat.size;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].free = storage->stat.free;
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = 1-local_site;
              nb_storages++;

              dprintf(fd, "          storage: %u\n", storage->sid);
	      dprintf(fd, "            host: %s\n", storage->host);
	      if (clone.multi_site) {
              dprintf(fd, "            site: %d\n", storage->siteNum);
	      }  
              dprintf(fd, "            status: %s\n", (storage->status==1)?"UP":"DOWN");
              dprintf(fd, "            size: %"PRIu64"\n", storage->stat.size);
              dprintf(fd, "            free: %"PRIu64"\n", storage->stat.free);
	      if (storage->stat.size==0) {
        	dprintf(fd, "            %cfree: 0\n",'%');	   
	      }
	      else {
        	dprintf(fd, "            %cfree: %"PRIu64"\n",'%', storage->stat.free*100/storage->stat.size);	   
	      }   
          }
	}  
    }
    
    gprofiler.vstats[gprofiler.nb_volumes].nb_storages = nb_storages;
 
    

    // Free the clone volume
    p = NULL;
    q = NULL;

    list_for_each_forward_safe(p, q, &clone.clusters) {
        cluster_t *entry = list_entry(p, cluster_t, list);
        list_remove(p);
        cluster_release(entry);
        free(entry);
    }

    status = 0;
out:
    if (fd > 0) close(fd);
    return status;
}
int monitor_volume_slave(volume_t *volume) {
    int status = -1;
    volume_stat_t vstat;
    list_t *p, *q;
    volume_t clone;
    uint32_t nb_storages = 0;
    int local_site = export_get_local_site_number();

    volume_initialize(&clone, 0, 0,0,0);
    if (volume_safe_copy(&clone, volume) != 0) {
        severe("can't clone volume: %d", volume->vid);
        goto out;
    }

    gprofiler.vstats[gprofiler.nb_volumes].vid = clone.vid;
    gprofiler.vstats[gprofiler.nb_volumes].georep = clone.georep;


    //XXX TO CHANGE
    volume_stat(&clone,&vstat);
    gprofiler.vstats[gprofiler.nb_volumes].bsize = vstat.bsize;
    gprofiler.vstats[gprofiler.nb_volumes].bfree = vstat.bfree;
    gprofiler.vstats[gprofiler.nb_volumes].blocks = vstat.blocks;

    list_for_each_forward(p, &clone.clusters) {
        cluster_t *cluster = list_entry(p, cluster_t, list);
     list_for_each_forward(q, (&cluster->storages[local_site])) {
            volume_storage_t *storage = list_entry(q, volume_storage_t, list);

            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].cid = cluster->cid;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].sid = storage->sid;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].status = storage->status;
	    strcpy(gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].host,storage->host);
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].size = storage->stat.size;
            gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].free = storage->stat.free;
	    if (clone.georep) 
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = local_site;
	    else  
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = storage->siteNum;
            nb_storages++;
        }
	
	if (clone.georep) {
          list_for_each_forward(q, (&cluster->storages[1-local_site])) {
              volume_storage_t *storage = list_entry(q, volume_storage_t, list);

              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].cid = cluster->cid;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].sid = storage->sid;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].status = storage->status;
	      strcpy(gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].host,storage->host);
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].size = storage->stat.size;
              gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].free = storage->stat.free;
	      gprofiler.vstats[gprofiler.nb_volumes].sstats[nb_storages].site = 1-local_site;
              nb_storages++;
          }
	}  
    }
    
    gprofiler.vstats[gprofiler.nb_volumes].nb_storages = nb_storages;
    

    // Free the clone volume
    p = NULL;
    q = NULL;

    list_for_each_forward_safe(p, q, &clone.clusters) {
        cluster_t *entry = list_entry(p, cluster_t, list);
        list_remove(p);
        cluster_release(entry);
        free(entry);
    }

    status = 0;
out:
    return status;
}
int monitor_export(export_t *export) {
    int status = -1;
    int fd = -1;
    char path[FILENAME_MAX];
    ep_statfs_t estat;
    uint64_t exceed = 0;
    DEBUG_FUNCTION;

    sprintf(path, "%s%s%d", DAEMON_PID_DIRECTORY, "exportd/export_", export->eid);
    if ((fd = open(path, O_WRONLY | O_CREAT | O_TRUNC, S_IRWXU | S_IROTH)) < 0) {
        severe("can't open %s", path);
        goto out;
    }

    if (export_stat(export, &estat) != 0) {
        severe("can't stat export: %"PRIu32"", export->eid);
        goto out;
    }

    gprofiler.estats[gprofiler.nb_exports].eid = export->eid;
    strcpy(gprofiler.estats[gprofiler.nb_exports].path, export->root);
    gprofiler.estats[gprofiler.nb_exports].vid = export->volume->vid;
    gprofiler.estats[gprofiler.nb_exports].bsize = estat.bsize;
    gprofiler.estats[gprofiler.nb_exports].blocks = estat.blocks;
    gprofiler.estats[gprofiler.nb_exports].bfree = estat.bfree;
    gprofiler.estats[gprofiler.nb_exports].files = estat.files;
    gprofiler.estats[gprofiler.nb_exports].ffree = estat.ffree;

    dprintf(fd, HEADER, VERSION);
    dprintf(fd, "export: %"PRIu32"\n", export->eid);
    dprintf(fd, "volume: %u\n", export->volume->vid);
    dprintf(fd, "root: %s\n", export->root);
    dprintf(fd, "squota: %"PRIu64"\n", export->squota);
    dprintf(fd, "hquota: %"PRIu64"\n", export->hquota);
    dprintf(fd, "bsize: %u\n", estat.bsize);
    dprintf(fd, "blocks: %"PRIu64"\n", estat.blocks);
    dprintf(fd, "bfree: %"PRIu64"\n", estat.bfree);
    dprintf(fd, "files: %"PRIu64"\n", estat.files);
    dprintf(fd, "ffree: %"PRIu64"\n", estat.ffree);
    if (export->squota > 0) {
        exceed = estat.blocks - estat.bfree > export->squota ?
                estat.blocks - estat.bfree - export->squota : 0;
    }
    dprintf(fd, "squota_exceed: %"PRIu64"\n", exceed);

    status = 0;
out:
    if (fd > 0) close(fd);
    return status;
}
