require 'prometheus/client/uses_value_type'
require 'prometheus/client/helper/json_parser'

module Prometheus
  module Client
    module Formats
      # Text format is human readable mainly used for manual inspection.
      module Text
        MEDIA_TYPE = 'text/plain'.freeze
        VERSION = '0.0.4'.freeze
        CONTENT_TYPE = "#{MEDIA_TYPE}; version=#{VERSION}".freeze

        METRIC_LINE = '%s%s %s'.freeze
        TYPE_LINE = '# TYPE %s %s'.freeze
        HELP_LINE = '# HELP %s %s'.freeze

        LABEL = '%s="%s"'.freeze
        SEPARATOR = ','.freeze
        DELIMITER = "\n".freeze

        REGEX = { doc: /[\n\\]/, label: /[\n\\"]/ }.freeze
        REPLACE = { "\n" => '\n', '\\' => '\\\\', '"' => '\"' }.freeze

        class << self
          def marshal(registry)
            lines = []

            registry.metrics.each do |metric|
              lines << format(HELP_LINE, metric.name, escape(metric.docstring))
              lines << format(TYPE_LINE, metric.name, metric.type)
              metric.values.each do |label_set, value|
                representation(metric, label_set, value) { |l| lines << l }
              end
            end

            # there must be a trailing delimiter
            (lines << nil).join(DELIMITER)
          end

          def marshal_multiprocess(path = Prometheus::Client.configuration.multiprocess_files_dir)
            metrics = load_metrics(path)

            lines = []
            merge_metrics(metrics).each do |name, metric|
              lines << format(HELP_LINE, name, escape(metric[:help]))
              lines << format(TYPE_LINE, name, metric[:type])

              metric[:samples].each do |metric_name, labels, value|
                lines << metric(metric_name, format_labels(labels), value)
              end
            end
            (lines << nil).join(DELIMITER)
          end

          private

          def merge_metrics(metrics)
            metrics.each_value do |metric|
              metric[:samples] = merge_samples(metric[:samples], metric[:type], metric[:multiprocess_mode]).map do |(name, labels), value|
                [name, labels.to_h, value]
              end
            end
          end

          def merge_samples(raw_samples, metric_type, multiprocess_mode)
            samples = {}
            raw_samples.each do |name, labels, value|
              without_pid = labels.reject { |l| l[0] == 'pid' }

              case metric_type
                when :gauge
                  case multiprocess_mode
                    when 'min'
                      s = samples.fetch([name, without_pid], value)
                      samples[[name, without_pid]] = [s, value].min
                    when 'max'
                      s = samples.fetch([name, without_pid], value)
                      samples[[name, without_pid]] = [s, value].max
                    when 'livesum'
                      s = samples.fetch([name, without_pid], 0.0)
                      samples[[name, without_pid]] = s + value
                    else # all/liveall
                      samples[[name, labels]] = value
                  end
                else
                  # Counter, Histogram and Summary.
                  s = samples.fetch([name, without_pid], 0.0)
                  samples[[name, without_pid]] = s + value
              end
            end

            samples
          end

          def load_metrics(path)
            metrics = {}
            Dir.glob(File.join(path, '*.db')).sort.each do |f|
              parts = File.basename(f, '.db').split('_')
              type = parts[0].to_sym

              MmapedDict.read_all_values(f).each do |key, value|
                metric_name, name, labelnames, labelvalues = Helper::JsonParser.load(key)
                metric = metrics.fetch(metric_name,
                                       metric_name: metric_name,
                                       help: 'Multiprocess metric',
                                       type: type,
                                       samples: []
                )
                if type == :gauge
                  pid = parts[2..-1].join('_')
                  metric[:multiprocess_mode] = parts[1]
                  metric[:samples] += [[name, labelnames.zip(labelvalues) + [['pid', pid]], value]]
                else
                  # The duplicates and labels are fixed in the next for.
                  metric[:samples] += [[name, labelnames.zip(labelvalues), value]]
                end
                metrics[metric_name] = metric
              end
            end
            metrics
          end

          def representation(metric, label_set, value, &block)
            set = metric.base_labels.merge(label_set)

            if metric.type == :summary
              summary(metric.name, set, value, &block)
            elsif metric.type == :histogram
              histogram(metric.name, set, value, &block)
            else
              yield metric(metric.name, format_labels(set), value.get)
            end
          end

          def summary(name, set, value)
            value.get.each do |q, v|
              yield metric(name, format_labels(set.merge(quantile: q)), v)
            end

            l = format_labels(set)
            yield metric("#{name}_sum", l, value.get.sum)
            yield metric("#{name}_count", l, value.get.total)
          end

          def histogram(name, set, value)
            value.get.each do |q, v|
              yield metric(name, format_labels(set.merge(le: q)), v)
            end
            yield metric(name, format_labels(set.merge(le: '+Inf')), value.get.total)

            l = format_labels(set)
            yield metric("#{name}_sum", l, value.get.sum)
            yield metric("#{name}_count", l, value.get.total)
          end

          def metric(name, labels, value)
            format(METRIC_LINE, name, labels, value)
          end

          def format_labels(set)
            return if set.empty?

            strings = set.each_with_object([]) do |(key, value), memo|
              memo << format(LABEL, key, escape(value, :label))
            end

            "{#{strings.join(SEPARATOR)}}"
          end

          def escape(string, format = :doc)
            string.to_s.gsub(REGEX[format], REPLACE)
          end
        end
      end
    end
  end
end
