/*
 * Unix SMB/CIFS implementation.
 *
 * HRESULT Error definitions
 *
 * Copyright (C) Noel Power <noel.power@suse.com> 2014
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _HRESULT_H_
#define _HRESULT_H_

#if defined(HAVE_IMMEDIATE_STRUCTURES)
typedef struct {uint32_t h;} HRESULT;
#define HRES_ERROR(x) ((HRESULT) { x })
#define HRES_ERROR_V(x) ((x).h)
#else
typedef uint32_t HRESULT;
#define HRES_ERROR(x) (x)
#define HRES_ERROR_V(x) (x)
#endif

#define HRES_IS_OK(x) (HRES_ERROR_V(x) == 0)
#define HRES_IS_EQUAL(x,y) (HRES_ERROR_V(x) == HRES_ERROR_V(y))


/*
 * The following error codes are autogenerated from [MS-ERREF]
 * see http://msdn.microsoft.com/en-us/library/cc704587.aspx
 */

#define HRES_STG_S_CONVERTED				  HRES_ERROR(0x00030200)
#define HRES_STG_S_BLOCK				  HRES_ERROR(0x00030201)
#define HRES_STG_S_RETRYNOW				  HRES_ERROR(0x00030202)
#define HRES_STG_S_MONITORING				  HRES_ERROR(0x00030203)
#define HRES_STG_S_MULTIPLEOPENS			  HRES_ERROR(0x00030204)
#define HRES_STG_S_CONSOLIDATIONFAILED			  HRES_ERROR(0x00030205)
#define HRES_STG_S_CANNOTCONSOLIDATE			  HRES_ERROR(0x00030206)
#define HRES_OLE_S_USEREG				  HRES_ERROR(0x00040000)
#define HRES_OLE_S_STATIC				  HRES_ERROR(0x00040001)
#define HRES_OLE_S_MAC_CLIPFORMAT			  HRES_ERROR(0x00040002)
#define HRES_DRAGDROP_S_DROP				  HRES_ERROR(0x00040100)
#define HRES_DRAGDROP_S_CANCEL				  HRES_ERROR(0x00040101)
#define HRES_DRAGDROP_S_USEDEFAULTCURSORS		  HRES_ERROR(0x00040102)
#define HRES_DATA_S_SAMEFORMATETC			  HRES_ERROR(0x00040130)
#define HRES_VIEW_S_ALREADY_FROZEN			  HRES_ERROR(0x00040140)
#define HRES_CACHE_S_FORMATETC_NOTSUPPORTED		  HRES_ERROR(0x00040170)
#define HRES_CACHE_S_SAMECACHE				  HRES_ERROR(0x00040171)
#define HRES_CACHE_S_SOMECACHES_NOTUPDATED		  HRES_ERROR(0x00040172)
#define HRES_OLEOBJ_S_INVALIDVERB			  HRES_ERROR(0x00040180)
#define HRES_OLEOBJ_S_CANNOT_DOVERB_NOW			  HRES_ERROR(0x00040181)
#define HRES_OLEOBJ_S_INVALIDHWND			  HRES_ERROR(0x00040182)
#define HRES_INPLACE_S_TRUNCATED			  HRES_ERROR(0x000401A0)
#define HRES_CONVERT10_S_NO_PRESENTATION		  HRES_ERROR(0x000401C0)
#define HRES_MK_S_REDUCED_TO_SELF			  HRES_ERROR(0x000401E2)
#define HRES_MK_S_ME					  HRES_ERROR(0x000401E4)
#define HRES_MK_S_HIM					  HRES_ERROR(0x000401E5)
#define HRES_MK_S_US					  HRES_ERROR(0x000401E6)
#define HRES_MK_S_MONIKERALREADYREGISTERED		  HRES_ERROR(0x000401E7)
#define HRES_EVENT_S_SOME_SUBSCRIBERS_FAILED		  HRES_ERROR(0x00040200)
#define HRES_EVENT_S_NOSUBSCRIBERS			  HRES_ERROR(0x00040202)
#define HRES_SCHED_S_TASK_READY				  HRES_ERROR(0x00041300)
#define HRES_SCHED_S_TASK_RUNNING			  HRES_ERROR(0x00041301)
#define HRES_SCHED_S_TASK_DISABLED			  HRES_ERROR(0x00041302)
#define HRES_SCHED_S_TASK_HAS_NOT_RUN			  HRES_ERROR(0x00041303)
#define HRES_SCHED_S_TASK_NO_MORE_RUNS			  HRES_ERROR(0x00041304)
#define HRES_SCHED_S_TASK_NOT_SCHEDULED			  HRES_ERROR(0x00041305)
#define HRES_SCHED_S_TASK_TERMINATED			  HRES_ERROR(0x00041306)
#define HRES_SCHED_S_TASK_NO_VALID_TRIGGERS		  HRES_ERROR(0x00041307)
#define HRES_SCHED_S_EVENT_TRIGGER			  HRES_ERROR(0x00041308)
#define HRES_SCHED_S_SOME_TRIGGERS_FAILED		  HRES_ERROR(0x0004131B)
#define HRES_SCHED_S_BATCH_LOGON_PROBLEM		  HRES_ERROR(0x0004131C)
#define HRES_XACT_S_ASYNC				  HRES_ERROR(0x0004D000)
#define HRES_XACT_S_READONLY				  HRES_ERROR(0x0004D002)
#define HRES_XACT_S_SOMENORETAIN			  HRES_ERROR(0x0004D003)
#define HRES_XACT_S_OKINFORM				  HRES_ERROR(0x0004D004)
#define HRES_XACT_S_MADECHANGESCONTENT			  HRES_ERROR(0x0004D005)
#define HRES_XACT_S_MADECHANGESINFORM			  HRES_ERROR(0x0004D006)
#define HRES_XACT_S_ALLNORETAIN				  HRES_ERROR(0x0004D007)
#define HRES_XACT_S_ABORTING				  HRES_ERROR(0x0004D008)
#define HRES_XACT_S_SINGLEPHASE				  HRES_ERROR(0x0004D009)
#define HRES_XACT_S_LOCALLY_OK				  HRES_ERROR(0x0004D00A)
#define HRES_XACT_S_LASTRESOURCEMANAGER			  HRES_ERROR(0x0004D010)
#define HRES_CO_S_NOTALLINTERFACES			  HRES_ERROR(0x00080012)
#define HRES_CO_S_MACHINENAMENOTFOUND			  HRES_ERROR(0x00080013)
#define HRES_SEC_I_CONTINUE_NEEDED			  HRES_ERROR(0x00090312)
#define HRES_SEC_I_COMPLETE_NEEDED			  HRES_ERROR(0x00090313)
#define HRES_SEC_I_COMPLETE_AND_CONTINUE		  HRES_ERROR(0x00090314)
#define HRES_SEC_I_LOCAL_LOGON				  HRES_ERROR(0x00090315)
#define HRES_SEC_I_CONTEXT_EXPIRED			  HRES_ERROR(0x00090317)
#define HRES_SEC_I_INCOMPLETE_CREDENTIALS		  HRES_ERROR(0x00090320)
#define HRES_SEC_I_RENEGOTIATE				  HRES_ERROR(0x00090321)
#define HRES_SEC_I_NO_LSA_CONTEXT			  HRES_ERROR(0x00090323)
#define HRES_SEC_I_SIGNATURE_NEEDED			  HRES_ERROR(0x0009035C)
#define HRES_CRYPT_I_NEW_PROTECTION_REQUIRED		  HRES_ERROR(0x00091012)
#define HRES_NS_S_CALLPENDING				  HRES_ERROR(0x000D0000)
#define HRES_NS_S_CALLABORTED				  HRES_ERROR(0x000D0001)
#define HRES_NS_S_STREAM_TRUNCATED			  HRES_ERROR(0x000D0002)
#define HRES_NS_S_REBUFFERING				  HRES_ERROR(0x000D0BC8)
#define HRES_NS_S_DEGRADING_QUALITY			  HRES_ERROR(0x000D0BC9)
#define HRES_NS_S_TRANSCRYPTOR_EOF			  HRES_ERROR(0x000D0BDB)
#define HRES_NS_S_WMP_UI_VERSIONMISMATCH		  HRES_ERROR(0x000D0FE8)
#define HRES_NS_S_WMP_EXCEPTION				  HRES_ERROR(0x000D0FE9)
#define HRES_NS_S_WMP_LOADED_GIF_IMAGE			  HRES_ERROR(0x000D1040)
#define HRES_NS_S_WMP_LOADED_PNG_IMAGE			  HRES_ERROR(0x000D1041)
#define HRES_NS_S_WMP_LOADED_BMP_IMAGE			  HRES_ERROR(0x000D1042)
#define HRES_NS_S_WMP_LOADED_JPG_IMAGE			  HRES_ERROR(0x000D1043)
#define HRES_NS_S_WMG_FORCE_DROP_FRAME			  HRES_ERROR(0x000D104F)
#define HRES_NS_S_WMR_ALREADYRENDERED			  HRES_ERROR(0x000D105F)
#define HRES_NS_S_WMR_PINTYPEPARTIALMATCH		  HRES_ERROR(0x000D1060)
#define HRES_NS_S_WMR_PINTYPEFULLMATCH			  HRES_ERROR(0x000D1061)
#define HRES_NS_S_WMG_ADVISE_DROP_FRAME			  HRES_ERROR(0x000D1066)
#define HRES_NS_S_WMG_ADVISE_DROP_TO_KEYFRAME		  HRES_ERROR(0x000D1067)
#define HRES_NS_S_NEED_TO_BUY_BURN_RIGHTS		  HRES_ERROR(0x000D10DB)
#define HRES_NS_S_WMPCORE_PLAYLISTCLEARABORT		  HRES_ERROR(0x000D10FE)
#define HRES_NS_S_WMPCORE_PLAYLISTREMOVEITEMABORT	  HRES_ERROR(0x000D10FF)
#define HRES_NS_S_WMPCORE_PLAYLIST_CREATION_PENDING	  HRES_ERROR(0x000D1102)
#define HRES_NS_S_WMPCORE_MEDIA_VALIDATION_PENDING	  HRES_ERROR(0x000D1103)
#define HRES_NS_S_WMPCORE_PLAYLIST_REPEAT_SECONDARY_SEGMENTS_IGNORED HRES_ERROR(0x000D1104)
#define HRES_NS_S_WMPCORE_COMMAND_NOT_AVAILABLE		  HRES_ERROR(0x000D1105)
#define HRES_NS_S_WMPCORE_PLAYLIST_NAME_AUTO_GENERATED	  HRES_ERROR(0x000D1106)
#define HRES_NS_S_WMPCORE_PLAYLIST_IMPORT_MISSING_ITEMS   HRES_ERROR(0x000D1107)
#define HRES_NS_S_WMPCORE_PLAYLIST_COLLAPSED_TO_SINGLE_MEDIA HRES_ERROR(0x000D1108)
#define HRES_NS_S_WMPCORE_MEDIA_CHILD_PLAYLIST_OPEN_PENDING HRES_ERROR(0x000D1109)
#define HRES_NS_S_WMPCORE_MORE_NODES_AVAIABLE		  HRES_ERROR(0x000D110A)
#define HRES_NS_S_WMPBR_SUCCESS				  HRES_ERROR(0x000D1135)
#define HRES_NS_S_WMPBR_PARTIALSUCCESS			  HRES_ERROR(0x000D1136)
#define HRES_NS_S_WMPEFFECT_TRANSPARENT			  HRES_ERROR(0x000D1144)
#define HRES_NS_S_WMPEFFECT_OPAQUE			  HRES_ERROR(0x000D1145)
#define HRES_NS_S_OPERATION_PENDING			  HRES_ERROR(0x000D114E)
#define HRES_NS_S_TRACK_BUY_REQUIRES_ALBUM_PURCHASE	  HRES_ERROR(0x000D1359)
#define HRES_NS_S_NAVIGATION_COMPLETE_WITH_ERRORS	  HRES_ERROR(0x000D135E)
#define HRES_NS_S_TRACK_ALREADY_DOWNLOADED		  HRES_ERROR(0x000D1361)
#define HRES_NS_S_PUBLISHING_POINT_STARTED_WITH_FAILED_SINKS HRES_ERROR(0x000D1519)
#define HRES_NS_S_DRM_LICENSE_ACQUIRED			  HRES_ERROR(0x000D2726)
#define HRES_NS_S_DRM_INDIVIDUALIZED			  HRES_ERROR(0x000D2727)
#define HRES_NS_S_DRM_MONITOR_CANCELLED			  HRES_ERROR(0x000D2746)
#define HRES_NS_S_DRM_ACQUIRE_CANCELLED			  HRES_ERROR(0x000D2747)
#define HRES_NS_S_DRM_BURNABLE_TRACK			  HRES_ERROR(0x000D276E)
#define HRES_NS_S_DRM_BURNABLE_TRACK_WITH_PLAYLIST_RESTRICTION HRES_ERROR(0x000D276F)
#define HRES_NS_S_DRM_NEEDS_INDIVIDUALIZATION		  HRES_ERROR(0x000D27DE)
#define HRES_NS_S_REBOOT_RECOMMENDED			  HRES_ERROR(0x000D2AF8)
#define HRES_NS_S_REBOOT_REQUIRED			  HRES_ERROR(0x000D2AF9)
#define HRES_NS_S_EOSRECEDING				  HRES_ERROR(0x000D2F09)
#define HRES_NS_S_CHANGENOTICE				  HRES_ERROR(0x000D2F0D)
#define HRES_ERROR_FLT_IO_COMPLETE			  HRES_ERROR(0x001F0001)
#define HRES_ERROR_GRAPHICS_MODE_NOT_PINNED		  HRES_ERROR(0x00262307)
#define HRES_ERROR_GRAPHICS_NO_PREFERRED_MODE		  HRES_ERROR(0x0026231E)
#define HRES_ERROR_GRAPHICS_DATASET_IS_EMPTY		  HRES_ERROR(0x0026234B)
#define HRES_ERROR_GRAPHICS_NO_MORE_ELEMENTS_IN_DATASET   HRES_ERROR(0x0026234C)
#define HRES_ERROR_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_PINNED HRES_ERROR(0x00262351)
#define HRES_PLA_S_PROPERTY_IGNORED			  HRES_ERROR(0x00300100)
#define HRES_ERROR_NDIS_INDICATION_REQUIRED		  HRES_ERROR(0x00340001)
#define HRES_TRK_S_OUT_OF_SYNC				  HRES_ERROR(0x0DEAD100)
#define HRES_TRK_VOLUME_NOT_FOUND			  HRES_ERROR(0x0DEAD102)
#define HRES_TRK_VOLUME_NOT_OWNED			  HRES_ERROR(0x0DEAD103)
#define HRES_TRK_S_NOTIFICATION_QUOTA_EXCEEDED		  HRES_ERROR(0x0DEAD107)
#define HRES_NS_I_TIGER_START				  HRES_ERROR(0x400D004F)
#define HRES_NS_I_CUB_START				  HRES_ERROR(0x400D0051)
#define HRES_NS_I_CUB_RUNNING				  HRES_ERROR(0x400D0052)
#define HRES_NS_I_DISK_START				  HRES_ERROR(0x400D0054)
#define HRES_NS_I_DISK_REBUILD_STARTED			  HRES_ERROR(0x400D0056)
#define HRES_NS_I_DISK_REBUILD_FINISHED			  HRES_ERROR(0x400D0057)
#define HRES_NS_I_DISK_REBUILD_ABORTED			  HRES_ERROR(0x400D0058)
#define HRES_NS_I_LIMIT_FUNNELS				  HRES_ERROR(0x400D0059)
#define HRES_NS_I_START_DISK				  HRES_ERROR(0x400D005A)
#define HRES_NS_I_STOP_DISK				  HRES_ERROR(0x400D005B)
#define HRES_NS_I_STOP_CUB				  HRES_ERROR(0x400D005C)
#define HRES_NS_I_KILL_USERSESSION			  HRES_ERROR(0x400D005D)
#define HRES_NS_I_KILL_CONNECTION			  HRES_ERROR(0x400D005E)
#define HRES_NS_I_REBUILD_DISK				  HRES_ERROR(0x400D005F)
#define HRES_MCMADM_I_NO_EVENTS				  HRES_ERROR(0x400D0069)
#define HRES_NS_I_LOGGING_FAILED			  HRES_ERROR(0x400D006E)
#define HRES_NS_I_LIMIT_BANDWIDTH			  HRES_ERROR(0x400D0070)
#define HRES_NS_I_CUB_UNFAIL_LINK			  HRES_ERROR(0x400D0191)
#define HRES_NS_I_RESTRIPE_START			  HRES_ERROR(0x400D0193)
#define HRES_NS_I_RESTRIPE_DONE				  HRES_ERROR(0x400D0194)
#define HRES_NS_I_RESTRIPE_DISK_OUT			  HRES_ERROR(0x400D0196)
#define HRES_NS_I_RESTRIPE_CUB_OUT			  HRES_ERROR(0x400D0197)
#define HRES_NS_I_DISK_STOP				  HRES_ERROR(0x400D0198)
#define HRES_NS_I_PLAYLIST_CHANGE_RECEDING		  HRES_ERROR(0x400D14BE)
#define HRES_NS_I_RECONNECTED				  HRES_ERROR(0x400D2EFF)
#define HRES_NS_I_NOLOG_STOP				  HRES_ERROR(0x400D2F01)
#define HRES_NS_I_EXISTING_PACKETIZER			  HRES_ERROR(0x400D2F03)
#define HRES_NS_I_MANUAL_PROXY				  HRES_ERROR(0x400D2F04)
#define HRES_ERROR_GRAPHICS_DRIVER_MISMATCH		  HRES_ERROR(0x40262009)
#define HRES_ERROR_GRAPHICS_UNKNOWN_CHILD_STATUS	  HRES_ERROR(0x4026242F)
#define HRES_ERROR_GRAPHICS_LEADLINK_START_DEFERRED	  HRES_ERROR(0x40262437)
#define HRES_ERROR_GRAPHICS_POLLING_TOO_FREQUENTLY	  HRES_ERROR(0x40262439)
#define HRES_ERROR_GRAPHICS_START_DEFERRED		  HRES_ERROR(0x4026243A)
#define HRES_E_PENDING					  HRES_ERROR(0x8000000A)
#define HRES_E_NOTIMPL					  HRES_ERROR(0x80004001)
#define HRES_E_NOINTERFACE				  HRES_ERROR(0x80004002)
#define HRES_E_POINTER					  HRES_ERROR(0x80004003)
#define HRES_E_ABORT					  HRES_ERROR(0x80004004)
#define HRES_E_FAIL					  HRES_ERROR(0x80004005)
#define HRES_CO_E_INIT_TLS				  HRES_ERROR(0x80004006)
#define HRES_CO_E_INIT_SHARED_ALLOCATOR			  HRES_ERROR(0x80004007)
#define HRES_CO_E_INIT_MEMORY_ALLOCATOR			  HRES_ERROR(0x80004008)
#define HRES_CO_E_INIT_CLASS_CACHE			  HRES_ERROR(0x80004009)
#define HRES_CO_E_INIT_RPC_CHANNEL			  HRES_ERROR(0x8000400A)
#define HRES_CO_E_INIT_TLS_SET_CHANNEL_CONTROL		  HRES_ERROR(0x8000400B)
#define HRES_CO_E_INIT_TLS_CHANNEL_CONTROL		  HRES_ERROR(0x8000400C)
#define HRES_CO_E_INIT_UNACCEPTED_USER_ALLOCATOR	  HRES_ERROR(0x8000400D)
#define HRES_CO_E_INIT_SCM_MUTEX_EXISTS			  HRES_ERROR(0x8000400E)
#define HRES_CO_E_INIT_SCM_FILE_MAPPING_EXISTS		  HRES_ERROR(0x8000400F)
#define HRES_CO_E_INIT_SCM_MAP_VIEW_OF_FILE		  HRES_ERROR(0x80004010)
#define HRES_CO_E_INIT_SCM_EXEC_FAILURE			  HRES_ERROR(0x80004011)
#define HRES_CO_E_INIT_ONLY_SINGLE_THREADED		  HRES_ERROR(0x80004012)
#define HRES_CO_E_CANT_REMOTE				  HRES_ERROR(0x80004013)
#define HRES_CO_E_BAD_SERVER_NAME			  HRES_ERROR(0x80004014)
#define HRES_CO_E_WRONG_SERVER_IDENTITY			  HRES_ERROR(0x80004015)
#define HRES_CO_E_OLE1DDE_DISABLED			  HRES_ERROR(0x80004016)
#define HRES_CO_E_RUNAS_SYNTAX				  HRES_ERROR(0x80004017)
#define HRES_CO_E_CREATEPROCESS_FAILURE			  HRES_ERROR(0x80004018)
#define HRES_CO_E_RUNAS_CREATEPROCESS_FAILURE		  HRES_ERROR(0x80004019)
#define HRES_CO_E_RUNAS_LOGON_FAILURE			  HRES_ERROR(0x8000401A)
#define HRES_CO_E_LAUNCH_PERMSSION_DENIED		  HRES_ERROR(0x8000401B)
#define HRES_CO_E_START_SERVICE_FAILURE			  HRES_ERROR(0x8000401C)
#define HRES_CO_E_REMOTE_COMMUNICATION_FAILURE		  HRES_ERROR(0x8000401D)
#define HRES_CO_E_SERVER_START_TIMEOUT			  HRES_ERROR(0x8000401E)
#define HRES_CO_E_CLSREG_INCONSISTENT			  HRES_ERROR(0x8000401F)
#define HRES_CO_E_IIDREG_INCONSISTENT			  HRES_ERROR(0x80004020)
#define HRES_CO_E_NOT_SUPPORTED				  HRES_ERROR(0x80004021)
#define HRES_CO_E_RELOAD_DLL				  HRES_ERROR(0x80004022)
#define HRES_CO_E_MSI_ERROR				  HRES_ERROR(0x80004023)
#define HRES_CO_E_ATTEMPT_TO_CREATE_OUTSIDE_CLIENT_CONTEXT HRES_ERROR(0x80004024)
#define HRES_CO_E_SERVER_PAUSED				  HRES_ERROR(0x80004025)
#define HRES_CO_E_SERVER_NOT_PAUSED			  HRES_ERROR(0x80004026)
#define HRES_CO_E_CLASS_DISABLED			  HRES_ERROR(0x80004027)
#define HRES_CO_E_CLRNOTAVAILABLE			  HRES_ERROR(0x80004028)
#define HRES_CO_E_ASYNC_WORK_REJECTED			  HRES_ERROR(0x80004029)
#define HRES_CO_E_SERVER_INIT_TIMEOUT			  HRES_ERROR(0x8000402A)
#define HRES_CO_E_NO_SECCTX_IN_ACTIVATE			  HRES_ERROR(0x8000402B)
#define HRES_CO_E_TRACKER_CONFIG			  HRES_ERROR(0x80004030)
#define HRES_CO_E_THREADPOOL_CONFIG			  HRES_ERROR(0x80004031)
#define HRES_CO_E_SXS_CONFIG				  HRES_ERROR(0x80004032)
#define HRES_CO_E_MALFORMED_SPN				  HRES_ERROR(0x80004033)
#define HRES_E_UNEXPECTED				  HRES_ERROR(0x8000FFFF)
#define HRES_RPC_E_CALL_REJECTED			  HRES_ERROR(0x80010001)
#define HRES_RPC_E_CALL_CANCELED			  HRES_ERROR(0x80010002)
#define HRES_RPC_E_CANTPOST_INSENDCALL			  HRES_ERROR(0x80010003)
#define HRES_RPC_E_CANTCALLOUT_INASYNCCALL		  HRES_ERROR(0x80010004)
#define HRES_RPC_E_CANTCALLOUT_INEXTERNALCALL		  HRES_ERROR(0x80010005)
#define HRES_RPC_E_CONNECTION_TERMINATED		  HRES_ERROR(0x80010006)
#define HRES_RPC_E_SERVER_DIED				  HRES_ERROR(0x80010007)
#define HRES_RPC_E_CLIENT_DIED				  HRES_ERROR(0x80010008)
#define HRES_RPC_E_INVALID_DATAPACKET			  HRES_ERROR(0x80010009)
#define HRES_RPC_E_CANTTRANSMIT_CALL			  HRES_ERROR(0x8001000A)
#define HRES_RPC_E_CLIENT_CANTMARSHAL_DATA		  HRES_ERROR(0x8001000B)
#define HRES_RPC_E_CLIENT_CANTUNMARSHAL_DATA		  HRES_ERROR(0x8001000C)
#define HRES_RPC_E_SERVER_CANTMARSHAL_DATA		  HRES_ERROR(0x8001000D)
#define HRES_RPC_E_SERVER_CANTUNMARSHAL_DATA		  HRES_ERROR(0x8001000E)
#define HRES_RPC_E_INVALID_DATA				  HRES_ERROR(0x8001000F)
#define HRES_RPC_E_INVALID_PARAMETER			  HRES_ERROR(0x80010010)
#define HRES_RPC_E_CANTCALLOUT_AGAIN			  HRES_ERROR(0x80010011)
#define HRES_RPC_E_SERVER_DIED_DNE			  HRES_ERROR(0x80010012)
#define HRES_RPC_E_SYS_CALL_FAILED			  HRES_ERROR(0x80010100)
#define HRES_RPC_E_OUT_OF_RESOURCES			  HRES_ERROR(0x80010101)
#define HRES_RPC_E_ATTEMPTED_MULTITHREAD		  HRES_ERROR(0x80010102)
#define HRES_RPC_E_NOT_REGISTERED			  HRES_ERROR(0x80010103)
#define HRES_RPC_E_FAULT				  HRES_ERROR(0x80010104)
#define HRES_RPC_E_SERVERFAULT				  HRES_ERROR(0x80010105)
#define HRES_RPC_E_CHANGED_MODE				  HRES_ERROR(0x80010106)
#define HRES_RPC_E_INVALIDMETHOD			  HRES_ERROR(0x80010107)
#define HRES_RPC_E_DISCONNECTED				  HRES_ERROR(0x80010108)
#define HRES_RPC_E_RETRY				  HRES_ERROR(0x80010109)
#define HRES_RPC_E_SERVERCALL_RETRYLATER		  HRES_ERROR(0x8001010A)
#define HRES_RPC_E_SERVERCALL_REJECTED			  HRES_ERROR(0x8001010B)
#define HRES_RPC_E_INVALID_CALLDATA			  HRES_ERROR(0x8001010C)
#define HRES_RPC_E_CANTCALLOUT_ININPUTSYNCCALL		  HRES_ERROR(0x8001010D)
#define HRES_RPC_E_WRONG_THREAD				  HRES_ERROR(0x8001010E)
#define HRES_RPC_E_THREAD_NOT_INIT			  HRES_ERROR(0x8001010F)
#define HRES_RPC_E_VERSION_MISMATCH			  HRES_ERROR(0x80010110)
#define HRES_RPC_E_INVALID_HEADER			  HRES_ERROR(0x80010111)
#define HRES_RPC_E_INVALID_EXTENSION			  HRES_ERROR(0x80010112)
#define HRES_RPC_E_INVALID_IPID				  HRES_ERROR(0x80010113)
#define HRES_RPC_E_INVALID_OBJECT			  HRES_ERROR(0x80010114)
#define HRES_RPC_S_CALLPENDING				  HRES_ERROR(0x80010115)
#define HRES_RPC_S_WAITONTIMER				  HRES_ERROR(0x80010116)
#define HRES_RPC_E_CALL_COMPLETE			  HRES_ERROR(0x80010117)
#define HRES_RPC_E_UNSECURE_CALL			  HRES_ERROR(0x80010118)
#define HRES_RPC_E_TOO_LATE				  HRES_ERROR(0x80010119)
#define HRES_RPC_E_NO_GOOD_SECURITY_PACKAGES		  HRES_ERROR(0x8001011A)
#define HRES_RPC_E_ACCESS_DENIED			  HRES_ERROR(0x8001011B)
#define HRES_RPC_E_REMOTE_DISABLED			  HRES_ERROR(0x8001011C)
#define HRES_RPC_E_INVALID_OBJREF			  HRES_ERROR(0x8001011D)
#define HRES_RPC_E_NO_CONTEXT				  HRES_ERROR(0x8001011E)
#define HRES_RPC_E_TIMEOUT				  HRES_ERROR(0x8001011F)
#define HRES_RPC_E_NO_SYNC				  HRES_ERROR(0x80010120)
#define HRES_RPC_E_FULLSIC_REQUIRED			  HRES_ERROR(0x80010121)
#define HRES_RPC_E_INVALID_STD_NAME			  HRES_ERROR(0x80010122)
#define HRES_CO_E_FAILEDTOIMPERSONATE			  HRES_ERROR(0x80010123)
#define HRES_CO_E_FAILEDTOGETSECCTX			  HRES_ERROR(0x80010124)
#define HRES_CO_E_FAILEDTOOPENTHREADTOKEN		  HRES_ERROR(0x80010125)
#define HRES_CO_E_FAILEDTOGETTOKENINFO			  HRES_ERROR(0x80010126)
#define HRES_CO_E_TRUSTEEDOESNTMATCHCLIENT		  HRES_ERROR(0x80010127)
#define HRES_CO_E_FAILEDTOQUERYCLIENTBLANKET		  HRES_ERROR(0x80010128)
#define HRES_CO_E_FAILEDTOSETDACL			  HRES_ERROR(0x80010129)
#define HRES_CO_E_ACCESSCHECKFAILED			  HRES_ERROR(0x8001012A)
#define HRES_CO_E_NETACCESSAPIFAILED			  HRES_ERROR(0x8001012B)
#define HRES_CO_E_WRONGTRUSTEENAMESYNTAX		  HRES_ERROR(0x8001012C)
#define HRES_CO_E_INVALIDSID				  HRES_ERROR(0x8001012D)
#define HRES_CO_E_CONVERSIONFAILED			  HRES_ERROR(0x8001012E)
#define HRES_CO_E_NOMATCHINGSIDFOUND			  HRES_ERROR(0x8001012F)
#define HRES_CO_E_LOOKUPACCSIDFAILED			  HRES_ERROR(0x80010130)
#define HRES_CO_E_NOMATCHINGNAMEFOUND			  HRES_ERROR(0x80010131)
#define HRES_CO_E_LOOKUPACCNAMEFAILED			  HRES_ERROR(0x80010132)
#define HRES_CO_E_SETSERLHNDLFAILED			  HRES_ERROR(0x80010133)
#define HRES_CO_E_FAILEDTOGETWINDIR			  HRES_ERROR(0x80010134)
#define HRES_CO_E_PATHTOOLONG				  HRES_ERROR(0x80010135)
#define HRES_CO_E_FAILEDTOGENUUID			  HRES_ERROR(0x80010136)
#define HRES_CO_E_FAILEDTOCREATEFILE			  HRES_ERROR(0x80010137)
#define HRES_CO_E_FAILEDTOCLOSEHANDLE			  HRES_ERROR(0x80010138)
#define HRES_CO_E_EXCEEDSYSACLLIMIT			  HRES_ERROR(0x80010139)
#define HRES_CO_E_ACESINWRONGORDER			  HRES_ERROR(0x8001013A)
#define HRES_CO_E_INCOMPATIBLESTREAMVERSION		  HRES_ERROR(0x8001013B)
#define HRES_CO_E_FAILEDTOOPENPROCESSTOKEN		  HRES_ERROR(0x8001013C)
#define HRES_CO_E_DECODEFAILED				  HRES_ERROR(0x8001013D)
#define HRES_CO_E_ACNOTINITIALIZED			  HRES_ERROR(0x8001013F)
#define HRES_CO_E_CANCEL_DISABLED			  HRES_ERROR(0x80010140)
#define HRES_RPC_E_UNEXPECTED				  HRES_ERROR(0x8001FFFF)
#define HRES_DISP_E_UNKNOWNINTERFACE			  HRES_ERROR(0x80020001)
#define HRES_DISP_E_MEMBERNOTFOUND			  HRES_ERROR(0x80020003)
#define HRES_DISP_E_PARAMNOTFOUND			  HRES_ERROR(0x80020004)
#define HRES_DISP_E_TYPEMISMATCH			  HRES_ERROR(0x80020005)
#define HRES_DISP_E_UNKNOWNNAME				  HRES_ERROR(0x80020006)
#define HRES_DISP_E_NONAMEDARGS				  HRES_ERROR(0x80020007)
#define HRES_DISP_E_BADVARTYPE				  HRES_ERROR(0x80020008)
#define HRES_DISP_E_EXCEPTION				  HRES_ERROR(0x80020009)
#define HRES_DISP_E_OVERFLOW				  HRES_ERROR(0x8002000A)
#define HRES_DISP_E_BADINDEX				  HRES_ERROR(0x8002000B)
#define HRES_DISP_E_UNKNOWNLCID				  HRES_ERROR(0x8002000C)
#define HRES_DISP_E_ARRAYISLOCKED			  HRES_ERROR(0x8002000D)
#define HRES_DISP_E_BADPARAMCOUNT			  HRES_ERROR(0x8002000E)
#define HRES_DISP_E_PARAMNOTOPTIONAL			  HRES_ERROR(0x8002000F)
#define HRES_DISP_E_BADCALLEE				  HRES_ERROR(0x80020010)
#define HRES_DISP_E_NOTACOLLECTION			  HRES_ERROR(0x80020011)
#define HRES_DISP_E_DIVBYZERO				  HRES_ERROR(0x80020012)
#define HRES_DISP_E_BUFFERTOOSMALL			  HRES_ERROR(0x80020013)
#define HRES_TYPE_E_BUFFERTOOSMALL			  HRES_ERROR(0x80028016)
#define HRES_TYPE_E_FIELDNOTFOUND			  HRES_ERROR(0x80028017)
#define HRES_TYPE_E_INVDATAREAD				  HRES_ERROR(0x80028018)
#define HRES_TYPE_E_UNSUPFORMAT				  HRES_ERROR(0x80028019)
#define HRES_TYPE_E_REGISTRYACCESS			  HRES_ERROR(0x8002801C)
#define HRES_TYPE_E_LIBNOTREGISTERED			  HRES_ERROR(0x8002801D)
#define HRES_TYPE_E_UNDEFINEDTYPE			  HRES_ERROR(0x80028027)
#define HRES_TYPE_E_QUALIFIEDNAMEDISALLOWED		  HRES_ERROR(0x80028028)
#define HRES_TYPE_E_INVALIDSTATE			  HRES_ERROR(0x80028029)
#define HRES_TYPE_E_WRONGTYPEKIND			  HRES_ERROR(0x8002802A)
#define HRES_TYPE_E_ELEMENTNOTFOUND			  HRES_ERROR(0x8002802B)
#define HRES_TYPE_E_AMBIGUOUSNAME			  HRES_ERROR(0x8002802C)
#define HRES_TYPE_E_NAMECONFLICT			  HRES_ERROR(0x8002802D)
#define HRES_TYPE_E_UNKNOWNLCID				  HRES_ERROR(0x8002802E)
#define HRES_TYPE_E_DLLFUNCTIONNOTFOUND			  HRES_ERROR(0x8002802F)
#define HRES_TYPE_E_BADMODULEKIND			  HRES_ERROR(0x800288BD)
#define HRES_TYPE_E_SIZETOOBIG				  HRES_ERROR(0x800288C5)
#define HRES_TYPE_E_DUPLICATEID				  HRES_ERROR(0x800288C6)
#define HRES_TYPE_E_INVALIDID				  HRES_ERROR(0x800288CF)
#define HRES_TYPE_E_TYPEMISMATCH			  HRES_ERROR(0x80028CA0)
#define HRES_TYPE_E_OUTOFBOUNDS				  HRES_ERROR(0x80028CA1)
#define HRES_TYPE_E_IOERROR				  HRES_ERROR(0x80028CA2)
#define HRES_TYPE_E_CANTCREATETMPFILE			  HRES_ERROR(0x80028CA3)
#define HRES_TYPE_E_CANTLOADLIBRARY			  HRES_ERROR(0x80029C4A)
#define HRES_TYPE_E_INCONSISTENTPROPFUNCS		  HRES_ERROR(0x80029C83)
#define HRES_TYPE_E_CIRCULARTYPE			  HRES_ERROR(0x80029C84)
#define HRES_STG_E_INVALIDFUNCTION			  HRES_ERROR(0x80030001)
#define HRES_STG_E_FILENOTFOUND				  HRES_ERROR(0x80030002)
#define HRES_STG_E_PATHNOTFOUND				  HRES_ERROR(0x80030003)
#define HRES_STG_E_TOOMANYOPENFILES			  HRES_ERROR(0x80030004)
#define HRES_STG_E_ACCESSDENIED				  HRES_ERROR(0x80030005)
#define HRES_STG_E_INVALIDHANDLE			  HRES_ERROR(0x80030006)
#define HRES_STG_E_INSUFFICIENTMEMORY			  HRES_ERROR(0x80030008)
#define HRES_STG_E_INVALIDPOINTER			  HRES_ERROR(0x80030009)
#define HRES_STG_E_NOMOREFILES				  HRES_ERROR(0x80030012)
#define HRES_STG_E_DISKISWRITEPROTECTED			  HRES_ERROR(0x80030013)
#define HRES_STG_E_SEEKERROR				  HRES_ERROR(0x80030019)
#define HRES_STG_E_WRITEFAULT				  HRES_ERROR(0x8003001D)
#define HRES_STG_E_READFAULT				  HRES_ERROR(0x8003001E)
#define HRES_STG_E_SHAREVIOLATION			  HRES_ERROR(0x80030020)
#define HRES_STG_E_LOCKVIOLATION			  HRES_ERROR(0x80030021)
#define HRES_STG_E_FILEALREADYEXISTS			  HRES_ERROR(0x80030050)
#define HRES_STG_E_INVALIDPARAMETER			  HRES_ERROR(0x80030057)
#define HRES_STG_E_MEDIUMFULL				  HRES_ERROR(0x80030070)
#define HRES_STG_E_PROPSETMISMATCHED			  HRES_ERROR(0x800300F0)
#define HRES_STG_E_ABNORMALAPIEXIT			  HRES_ERROR(0x800300FA)
#define HRES_STG_E_INVALIDHEADER			  HRES_ERROR(0x800300FB)
#define HRES_STG_E_INVALIDNAME				  HRES_ERROR(0x800300FC)
#define HRES_STG_E_UNKNOWN				  HRES_ERROR(0x800300FD)
#define HRES_STG_E_UNIMPLEMENTEDFUNCTION		  HRES_ERROR(0x800300FE)
#define HRES_STG_E_INVALIDFLAG				  HRES_ERROR(0x800300FF)
#define HRES_STG_E_INUSE				  HRES_ERROR(0x80030100)
#define HRES_STG_E_NOTCURRENT				  HRES_ERROR(0x80030101)
#define HRES_STG_E_REVERTED				  HRES_ERROR(0x80030102)
#define HRES_STG_E_CANTSAVE				  HRES_ERROR(0x80030103)
#define HRES_STG_E_OLDFORMAT				  HRES_ERROR(0x80030104)
#define HRES_STG_E_OLDDLL				  HRES_ERROR(0x80030105)
#define HRES_STG_E_SHAREREQUIRED			  HRES_ERROR(0x80030106)
#define HRES_STG_E_NOTFILEBASEDSTORAGE			  HRES_ERROR(0x80030107)
#define HRES_STG_E_EXTANTMARSHALLINGS			  HRES_ERROR(0x80030108)
#define HRES_STG_E_DOCFILECORRUPT			  HRES_ERROR(0x80030109)
#define HRES_STG_E_BADBASEADDRESS			  HRES_ERROR(0x80030110)
#define HRES_STG_E_DOCFILETOOLARGE			  HRES_ERROR(0x80030111)
#define HRES_STG_E_NOTSIMPLEFORMAT			  HRES_ERROR(0x80030112)
#define HRES_STG_E_INCOMPLETE				  HRES_ERROR(0x80030201)
#define HRES_STG_E_TERMINATED				  HRES_ERROR(0x80030202)
#define HRES_STG_E_STATUS_COPY_PROTECTION_FAILURE	  HRES_ERROR(0x80030305)
#define HRES_STG_E_CSS_AUTHENTICATION_FAILURE		  HRES_ERROR(0x80030306)
#define HRES_STG_E_CSS_KEY_NOT_PRESENT			  HRES_ERROR(0x80030307)
#define HRES_STG_E_CSS_KEY_NOT_ESTABLISHED		  HRES_ERROR(0x80030308)
#define HRES_STG_E_CSS_SCRAMBLED_SECTOR			  HRES_ERROR(0x80030309)
#define HRES_STG_E_CSS_REGION_MISMATCH			  HRES_ERROR(0x8003030A)
#define HRES_STG_E_RESETS_EXHAUSTED			  HRES_ERROR(0x8003030B)
#define HRES_OLE_E_OLEVERB				  HRES_ERROR(0x80040000)
#define HRES_OLE_E_ADVF					  HRES_ERROR(0x80040001)
#define HRES_OLE_E_ENUM_NOMORE				  HRES_ERROR(0x80040002)
#define HRES_OLE_E_ADVISENOTSUPPORTED			  HRES_ERROR(0x80040003)
#define HRES_OLE_E_NOCONNECTION				  HRES_ERROR(0x80040004)
#define HRES_OLE_E_NOTRUNNING				  HRES_ERROR(0x80040005)
#define HRES_OLE_E_NOCACHE				  HRES_ERROR(0x80040006)
#define HRES_OLE_E_BLANK				  HRES_ERROR(0x80040007)
#define HRES_OLE_E_CLASSDIFF				  HRES_ERROR(0x80040008)
#define HRES_OLE_E_CANT_GETMONIKER			  HRES_ERROR(0x80040009)
#define HRES_OLE_E_CANT_BINDTOSOURCE			  HRES_ERROR(0x8004000A)
#define HRES_OLE_E_STATIC				  HRES_ERROR(0x8004000B)
#define HRES_OLE_E_PROMPTSAVECANCELLED			  HRES_ERROR(0x8004000C)
#define HRES_OLE_E_INVALIDRECT				  HRES_ERROR(0x8004000D)
#define HRES_OLE_E_WRONGCOMPOBJ				  HRES_ERROR(0x8004000E)
#define HRES_OLE_E_INVALIDHWND				  HRES_ERROR(0x8004000F)
#define HRES_OLE_E_NOT_INPLACEACTIVE			  HRES_ERROR(0x80040010)
#define HRES_OLE_E_CANTCONVERT				  HRES_ERROR(0x80040011)
#define HRES_OLE_E_NOSTORAGE				  HRES_ERROR(0x80040012)
#define HRES_DV_E_FORMATETC				  HRES_ERROR(0x80040064)
#define HRES_DV_E_DVTARGETDEVICE			  HRES_ERROR(0x80040065)
#define HRES_DV_E_STGMEDIUM				  HRES_ERROR(0x80040066)
#define HRES_DV_E_STATDATA				  HRES_ERROR(0x80040067)
#define HRES_DV_E_LINDEX				  HRES_ERROR(0x80040068)
#define HRES_DV_E_TYMED					  HRES_ERROR(0x80040069)
#define HRES_DV_E_CLIPFORMAT				  HRES_ERROR(0x8004006A)
#define HRES_DV_E_DVASPECT				  HRES_ERROR(0x8004006B)
#define HRES_DV_E_DVTARGETDEVICE_SIZE			  HRES_ERROR(0x8004006C)
#define HRES_DV_E_NOIVIEWOBJECT				  HRES_ERROR(0x8004006D)
#define HRES_DRAGDROP_E_NOTREGISTERED			  HRES_ERROR(0x80040100)
#define HRES_DRAGDROP_E_ALREADYREGISTERED		  HRES_ERROR(0x80040101)
#define HRES_DRAGDROP_E_INVALIDHWND			  HRES_ERROR(0x80040102)
#define HRES_CLASS_E_NOAGGREGATION			  HRES_ERROR(0x80040110)
#define HRES_CLASS_E_CLASSNOTAVAILABLE			  HRES_ERROR(0x80040111)
#define HRES_CLASS_E_NOTLICENSED			  HRES_ERROR(0x80040112)
#define HRES_VIEW_E_DRAW				  HRES_ERROR(0x80040140)
#define HRES_REGDB_E_READREGDB				  HRES_ERROR(0x80040150)
#define HRES_REGDB_E_WRITEREGDB				  HRES_ERROR(0x80040151)
#define HRES_REGDB_E_KEYMISSING				  HRES_ERROR(0x80040152)
#define HRES_REGDB_E_INVALIDVALUE			  HRES_ERROR(0x80040153)
#define HRES_REGDB_E_CLASSNOTREG			  HRES_ERROR(0x80040154)
#define HRES_REGDB_E_IIDNOTREG				  HRES_ERROR(0x80040155)
#define HRES_REGDB_E_BADTHREADINGMODEL			  HRES_ERROR(0x80040156)
#define HRES_CAT_E_CATIDNOEXIST				  HRES_ERROR(0x80040160)
#define HRES_CAT_E_NODESCRIPTION			  HRES_ERROR(0x80040161)
#define HRES_CS_E_PACKAGE_NOTFOUND			  HRES_ERROR(0x80040164)
#define HRES_CS_E_NOT_DELETABLE				  HRES_ERROR(0x80040165)
#define HRES_CS_E_CLASS_NOTFOUND			  HRES_ERROR(0x80040166)
#define HRES_CS_E_INVALID_VERSION			  HRES_ERROR(0x80040167)
#define HRES_CS_E_NO_CLASSSTORE				  HRES_ERROR(0x80040168)
#define HRES_CS_E_OBJECT_NOTFOUND			  HRES_ERROR(0x80040169)
#define HRES_CS_E_OBJECT_ALREADY_EXISTS			  HRES_ERROR(0x8004016A)
#define HRES_CS_E_INVALID_PATH				  HRES_ERROR(0x8004016B)
#define HRES_CS_E_NETWORK_ERROR				  HRES_ERROR(0x8004016C)
#define HRES_CS_E_ADMIN_LIMIT_EXCEEDED			  HRES_ERROR(0x8004016D)
#define HRES_CS_E_SCHEMA_MISMATCH			  HRES_ERROR(0x8004016E)
#define HRES_CS_E_INTERNAL_ERROR			  HRES_ERROR(0x8004016F)
#define HRES_CACHE_E_NOCACHE_UPDATED			  HRES_ERROR(0x80040170)
#define HRES_OLEOBJ_E_NOVERBS				  HRES_ERROR(0x80040180)
#define HRES_OLEOBJ_E_INVALIDVERB			  HRES_ERROR(0x80040181)
#define HRES_INPLACE_E_NOTUNDOABLE			  HRES_ERROR(0x800401A0)
#define HRES_INPLACE_E_NOTOOLSPACE			  HRES_ERROR(0x800401A1)
#define HRES_CONVERT10_E_OLESTREAM_GET			  HRES_ERROR(0x800401C0)
#define HRES_CONVERT10_E_OLESTREAM_PUT			  HRES_ERROR(0x800401C1)
#define HRES_CONVERT10_E_OLESTREAM_FMT			  HRES_ERROR(0x800401C2)
#define HRES_CONVERT10_E_OLESTREAM_BITMAP_TO_DIB	  HRES_ERROR(0x800401C3)
#define HRES_CONVERT10_E_STG_FMT			  HRES_ERROR(0x800401C4)
#define HRES_CONVERT10_E_STG_NO_STD_STREAM		  HRES_ERROR(0x800401C5)
#define HRES_CONVERT10_E_STG_DIB_TO_BITMAP		  HRES_ERROR(0x800401C6)
#define HRES_CLIPBRD_E_CANT_OPEN			  HRES_ERROR(0x800401D0)
#define HRES_CLIPBRD_E_CANT_EMPTY			  HRES_ERROR(0x800401D1)
#define HRES_CLIPBRD_E_CANT_SET				  HRES_ERROR(0x800401D2)
#define HRES_CLIPBRD_E_BAD_DATA				  HRES_ERROR(0x800401D3)
#define HRES_CLIPBRD_E_CANT_CLOSE			  HRES_ERROR(0x800401D4)
#define HRES_MK_E_CONNECTMANUALLY			  HRES_ERROR(0x800401E0)
#define HRES_MK_E_EXCEEDEDDEADLINE			  HRES_ERROR(0x800401E1)
#define HRES_MK_E_NEEDGENERIC				  HRES_ERROR(0x800401E2)
#define HRES_MK_E_UNAVAILABLE				  HRES_ERROR(0x800401E3)
#define HRES_MK_E_SYNTAX				  HRES_ERROR(0x800401E4)
#define HRES_MK_E_NOOBJECT				  HRES_ERROR(0x800401E5)
#define HRES_MK_E_INVALIDEXTENSION			  HRES_ERROR(0x800401E6)
#define HRES_MK_E_INTERMEDIATEINTERFACENOTSUPPORTED	  HRES_ERROR(0x800401E7)
#define HRES_MK_E_NOTBINDABLE				  HRES_ERROR(0x800401E8)
#define HRES_MK_E_NOTBOUND				  HRES_ERROR(0x800401E9)
#define HRES_MK_E_CANTOPENFILE				  HRES_ERROR(0x800401EA)
#define HRES_MK_E_MUSTBOTHERUSER			  HRES_ERROR(0x800401EB)
#define HRES_MK_E_NOINVERSE				  HRES_ERROR(0x800401EC)
#define HRES_MK_E_NOSTORAGE				  HRES_ERROR(0x800401ED)
#define HRES_MK_E_NOPREFIX				  HRES_ERROR(0x800401EE)
#define HRES_MK_E_ENUMERATION_FAILED			  HRES_ERROR(0x800401EF)
#define HRES_CO_E_NOTINITIALIZED			  HRES_ERROR(0x800401F0)
#define HRES_CO_E_ALREADYINITIALIZED			  HRES_ERROR(0x800401F1)
#define HRES_CO_E_CANTDETERMINECLASS			  HRES_ERROR(0x800401F2)
#define HRES_CO_E_CLASSSTRING				  HRES_ERROR(0x800401F3)
#define HRES_CO_E_IIDSTRING				  HRES_ERROR(0x800401F4)
#define HRES_CO_E_APPNOTFOUND				  HRES_ERROR(0x800401F5)
#define HRES_CO_E_APPSINGLEUSE				  HRES_ERROR(0x800401F6)
#define HRES_CO_E_ERRORINAPP				  HRES_ERROR(0x800401F7)
#define HRES_CO_E_DLLNOTFOUND				  HRES_ERROR(0x800401F8)
#define HRES_CO_E_ERRORINDLL				  HRES_ERROR(0x800401F9)
#define HRES_CO_E_WRONGOSFORAPP				  HRES_ERROR(0x800401FA)
#define HRES_CO_E_OBJNOTREG				  HRES_ERROR(0x800401FB)
#define HRES_CO_E_OBJISREG				  HRES_ERROR(0x800401FC)
#define HRES_CO_E_OBJNOTCONNECTED			  HRES_ERROR(0x800401FD)
#define HRES_CO_E_APPDIDNTREG				  HRES_ERROR(0x800401FE)
#define HRES_CO_E_RELEASED				  HRES_ERROR(0x800401FF)
#define HRES_EVENT_E_ALL_SUBSCRIBERS_FAILED		  HRES_ERROR(0x80040201)
#define HRES_EVENT_E_QUERYSYNTAX			  HRES_ERROR(0x80040203)
#define HRES_EVENT_E_QUERYFIELD				  HRES_ERROR(0x80040204)
#define HRES_EVENT_E_INTERNALEXCEPTION			  HRES_ERROR(0x80040205)
#define HRES_EVENT_E_INTERNALERROR			  HRES_ERROR(0x80040206)
#define HRES_EVENT_E_INVALID_PER_USER_SID		  HRES_ERROR(0x80040207)
#define HRES_EVENT_E_USER_EXCEPTION			  HRES_ERROR(0x80040208)
#define HRES_EVENT_E_TOO_MANY_METHODS			  HRES_ERROR(0x80040209)
#define HRES_EVENT_E_MISSING_EVENTCLASS			  HRES_ERROR(0x8004020A)
#define HRES_EVENT_E_NOT_ALL_REMOVED			  HRES_ERROR(0x8004020B)
#define HRES_EVENT_E_COMPLUS_NOT_INSTALLED		  HRES_ERROR(0x8004020C)
#define HRES_EVENT_E_CANT_MODIFY_OR_DELETE_UNCONFIGURED_OBJECT HRES_ERROR(0x8004020D)
#define HRES_EVENT_E_CANT_MODIFY_OR_DELETE_CONFIGURED_OBJECT HRES_ERROR(0x8004020E)
#define HRES_EVENT_E_INVALID_EVENT_CLASS_PARTITION	  HRES_ERROR(0x8004020F)
#define HRES_EVENT_E_PER_USER_SID_NOT_LOGGED_ON		  HRES_ERROR(0x80040210)
#define HRES_SCHED_E_TRIGGER_NOT_FOUND			  HRES_ERROR(0x80041309)
#define HRES_SCHED_E_TASK_NOT_READY			  HRES_ERROR(0x8004130A)
#define HRES_SCHED_E_TASK_NOT_RUNNING			  HRES_ERROR(0x8004130B)
#define HRES_SCHED_E_SERVICE_NOT_INSTALLED		  HRES_ERROR(0x8004130C)
#define HRES_SCHED_E_CANNOT_OPEN_TASK			  HRES_ERROR(0x8004130D)
#define HRES_SCHED_E_INVALID_TASK			  HRES_ERROR(0x8004130E)
#define HRES_SCHED_E_ACCOUNT_INFORMATION_NOT_SET	  HRES_ERROR(0x8004130F)
#define HRES_SCHED_E_ACCOUNT_NAME_NOT_FOUND		  HRES_ERROR(0x80041310)
#define HRES_SCHED_E_ACCOUNT_DBASE_CORRUPT		  HRES_ERROR(0x80041311)
#define HRES_SCHED_E_NO_SECURITY_SERVICES		  HRES_ERROR(0x80041312)
#define HRES_SCHED_E_UNKNOWN_OBJECT_VERSION		  HRES_ERROR(0x80041313)
#define HRES_SCHED_E_UNSUPPORTED_ACCOUNT_OPTION		  HRES_ERROR(0x80041314)
#define HRES_SCHED_E_SERVICE_NOT_RUNNING		  HRES_ERROR(0x80041315)
#define HRES_SCHED_E_UNEXPECTEDNODE			  HRES_ERROR(0x80041316)
#define HRES_SCHED_E_NAMESPACE				  HRES_ERROR(0x80041317)
#define HRES_SCHED_E_INVALIDVALUE			  HRES_ERROR(0x80041318)
#define HRES_SCHED_E_MISSINGNODE			  HRES_ERROR(0x80041319)
#define HRES_SCHED_E_MALFORMEDXML			  HRES_ERROR(0x8004131A)
#define HRES_SCHED_E_TOO_MANY_NODES			  HRES_ERROR(0x8004131D)
#define HRES_SCHED_E_PAST_END_BOUNDARY			  HRES_ERROR(0x8004131E)
#define HRES_SCHED_E_ALREADY_RUNNING			  HRES_ERROR(0x8004131F)
#define HRES_SCHED_E_USER_NOT_LOGGED_ON			  HRES_ERROR(0x80041320)
#define HRES_SCHED_E_INVALID_TASK_HASH			  HRES_ERROR(0x80041321)
#define HRES_SCHED_E_SERVICE_NOT_AVAILABLE		  HRES_ERROR(0x80041322)
#define HRES_SCHED_E_SERVICE_TOO_BUSY			  HRES_ERROR(0x80041323)
#define HRES_SCHED_E_TASK_ATTEMPTED			  HRES_ERROR(0x80041324)
#define HRES_XACT_E_ALREADYOTHERSINGLEPHASE		  HRES_ERROR(0x8004D000)
#define HRES_XACT_E_CANTRETAIN				  HRES_ERROR(0x8004D001)
#define HRES_XACT_E_COMMITFAILED			  HRES_ERROR(0x8004D002)
#define HRES_XACT_E_COMMITPREVENTED			  HRES_ERROR(0x8004D003)
#define HRES_XACT_E_HEURISTICABORT			  HRES_ERROR(0x8004D004)
#define HRES_XACT_E_HEURISTICCOMMIT			  HRES_ERROR(0x8004D005)
#define HRES_XACT_E_HEURISTICDAMAGE			  HRES_ERROR(0x8004D006)
#define HRES_XACT_E_HEURISTICDANGER			  HRES_ERROR(0x8004D007)
#define HRES_XACT_E_ISOLATIONLEVEL			  HRES_ERROR(0x8004D008)
#define HRES_XACT_E_NOASYNC				  HRES_ERROR(0x8004D009)
#define HRES_XACT_E_NOENLIST				  HRES_ERROR(0x8004D00A)
#define HRES_XACT_E_NOISORETAIN				  HRES_ERROR(0x8004D00B)
#define HRES_XACT_E_NORESOURCE				  HRES_ERROR(0x8004D00C)
#define HRES_XACT_E_NOTCURRENT				  HRES_ERROR(0x8004D00D)
#define HRES_XACT_E_NOTRANSACTION			  HRES_ERROR(0x8004D00E)
#define HRES_XACT_E_NOTSUPPORTED			  HRES_ERROR(0x8004D00F)
#define HRES_XACT_E_UNKNOWNRMGRID			  HRES_ERROR(0x8004D010)
#define HRES_XACT_E_WRONGSTATE				  HRES_ERROR(0x8004D011)
#define HRES_XACT_E_WRONGUOW				  HRES_ERROR(0x8004D012)
#define HRES_XACT_E_XTIONEXISTS				  HRES_ERROR(0x8004D013)
#define HRES_XACT_E_NOIMPORTOBJECT			  HRES_ERROR(0x8004D014)
#define HRES_XACT_E_INVALIDCOOKIE			  HRES_ERROR(0x8004D015)
#define HRES_XACT_E_INDOUBT				  HRES_ERROR(0x8004D016)
#define HRES_XACT_E_NOTIMEOUT				  HRES_ERROR(0x8004D017)
#define HRES_XACT_E_ALREADYINPROGRESS			  HRES_ERROR(0x8004D018)
#define HRES_XACT_E_ABORTED				  HRES_ERROR(0x8004D019)
#define HRES_XACT_E_LOGFULL				  HRES_ERROR(0x8004D01A)
#define HRES_XACT_E_TMNOTAVAILABLE			  HRES_ERROR(0x8004D01B)
#define HRES_XACT_E_CONNECTION_DOWN			  HRES_ERROR(0x8004D01C)
#define HRES_XACT_E_CONNECTION_DENIED			  HRES_ERROR(0x8004D01D)
#define HRES_XACT_E_REENLISTTIMEOUT			  HRES_ERROR(0x8004D01E)
#define HRES_XACT_E_TIP_CONNECT_FAILED			  HRES_ERROR(0x8004D01F)
#define HRES_XACT_E_TIP_PROTOCOL_ERROR			  HRES_ERROR(0x8004D020)
#define HRES_XACT_E_TIP_PULL_FAILED			  HRES_ERROR(0x8004D021)
#define HRES_XACT_E_DEST_TMNOTAVAILABLE			  HRES_ERROR(0x8004D022)
#define HRES_XACT_E_TIP_DISABLED			  HRES_ERROR(0x8004D023)
#define HRES_XACT_E_NETWORK_TX_DISABLED			  HRES_ERROR(0x8004D024)
#define HRES_XACT_E_PARTNER_NETWORK_TX_DISABLED		  HRES_ERROR(0x8004D025)
#define HRES_XACT_E_XA_TX_DISABLED			  HRES_ERROR(0x8004D026)
#define HRES_XACT_E_UNABLE_TO_READ_DTC_CONFIG		  HRES_ERROR(0x8004D027)
#define HRES_XACT_E_UNABLE_TO_LOAD_DTC_PROXY		  HRES_ERROR(0x8004D028)
#define HRES_XACT_E_ABORTING				  HRES_ERROR(0x8004D029)
#define HRES_XACT_E_CLERKNOTFOUND			  HRES_ERROR(0x8004D080)
#define HRES_XACT_E_CLERKEXISTS				  HRES_ERROR(0x8004D081)
#define HRES_XACT_E_RECOVERYINPROGRESS			  HRES_ERROR(0x8004D082)
#define HRES_XACT_E_TRANSACTIONCLOSED			  HRES_ERROR(0x8004D083)
#define HRES_XACT_E_INVALIDLSN				  HRES_ERROR(0x8004D084)
#define HRES_XACT_E_REPLAYREQUEST			  HRES_ERROR(0x8004D085)
#define HRES_XACT_E_CONNECTION_REQUEST_DENIED		  HRES_ERROR(0x8004D100)
#define HRES_XACT_E_TOOMANY_ENLISTMENTS			  HRES_ERROR(0x8004D101)
#define HRES_XACT_E_DUPLICATE_GUID			  HRES_ERROR(0x8004D102)
#define HRES_XACT_E_NOTSINGLEPHASE			  HRES_ERROR(0x8004D103)
#define HRES_XACT_E_RECOVERYALREADYDONE			  HRES_ERROR(0x8004D104)
#define HRES_XACT_E_PROTOCOL				  HRES_ERROR(0x8004D105)
#define HRES_XACT_E_RM_FAILURE				  HRES_ERROR(0x8004D106)
#define HRES_XACT_E_RECOVERY_FAILED			  HRES_ERROR(0x8004D107)
#define HRES_XACT_E_LU_NOT_FOUND			  HRES_ERROR(0x8004D108)
#define HRES_XACT_E_DUPLICATE_LU			  HRES_ERROR(0x8004D109)
#define HRES_XACT_E_LU_NOT_CONNECTED			  HRES_ERROR(0x8004D10A)
#define HRES_XACT_E_DUPLICATE_TRANSID			  HRES_ERROR(0x8004D10B)
#define HRES_XACT_E_LU_BUSY				  HRES_ERROR(0x8004D10C)
#define HRES_XACT_E_LU_NO_RECOVERY_PROCESS		  HRES_ERROR(0x8004D10D)
#define HRES_XACT_E_LU_DOWN				  HRES_ERROR(0x8004D10E)
#define HRES_XACT_E_LU_RECOVERING			  HRES_ERROR(0x8004D10F)
#define HRES_XACT_E_LU_RECOVERY_MISMATCH		  HRES_ERROR(0x8004D110)
#define HRES_XACT_E_RM_UNAVAILABLE			  HRES_ERROR(0x8004D111)
#define HRES_CONTEXT_E_ABORTED				  HRES_ERROR(0x8004E002)
#define HRES_CONTEXT_E_ABORTING				  HRES_ERROR(0x8004E003)
#define HRES_CONTEXT_E_NOCONTEXT			  HRES_ERROR(0x8004E004)
#define HRES_CONTEXT_E_WOULD_DEADLOCK			  HRES_ERROR(0x8004E005)
#define HRES_CONTEXT_E_SYNCH_TIMEOUT			  HRES_ERROR(0x8004E006)
#define HRES_CONTEXT_E_OLDREF				  HRES_ERROR(0x8004E007)
#define HRES_CONTEXT_E_ROLENOTFOUND			  HRES_ERROR(0x8004E00C)
#define HRES_CONTEXT_E_TMNOTAVAILABLE			  HRES_ERROR(0x8004E00F)
#define HRES_CO_E_ACTIVATIONFAILED			  HRES_ERROR(0x8004E021)
#define HRES_CO_E_ACTIVATIONFAILED_EVENTLOGGED		  HRES_ERROR(0x8004E022)
#define HRES_CO_E_ACTIVATIONFAILED_CATALOGERROR		  HRES_ERROR(0x8004E023)
#define HRES_CO_E_ACTIVATIONFAILED_TIMEOUT		  HRES_ERROR(0x8004E024)
#define HRES_CO_E_INITIALIZATIONFAILED			  HRES_ERROR(0x8004E025)
#define HRES_CONTEXT_E_NOJIT				  HRES_ERROR(0x8004E026)
#define HRES_CONTEXT_E_NOTRANSACTION			  HRES_ERROR(0x8004E027)
#define HRES_CO_E_THREADINGMODEL_CHANGED		  HRES_ERROR(0x8004E028)
#define HRES_CO_E_NOIISINTRINSICS			  HRES_ERROR(0x8004E029)
#define HRES_CO_E_NOCOOKIES				  HRES_ERROR(0x8004E02A)
#define HRES_CO_E_DBERROR				  HRES_ERROR(0x8004E02B)
#define HRES_CO_E_NOTPOOLED				  HRES_ERROR(0x8004E02C)
#define HRES_CO_E_NOTCONSTRUCTED			  HRES_ERROR(0x8004E02D)
#define HRES_CO_E_NOSYNCHRONIZATION			  HRES_ERROR(0x8004E02E)
#define HRES_CO_E_ISOLEVELMISMATCH			  HRES_ERROR(0x8004E02F)
#define HRES_CO_E_CALL_OUT_OF_TX_SCOPE_NOT_ALLOWED	  HRES_ERROR(0x8004E030)
#define HRES_CO_E_EXIT_TRANSACTION_SCOPE_NOT_CALLED	  HRES_ERROR(0x8004E031)
#define HRES_E_ACCESSDENIED				  HRES_ERROR(0x80070005)
#define HRES_E_OUTOFMEMORY				  HRES_ERROR(0x8007000E)
#define HRES_ERROR_NOT_SUPPORTED			  HRES_ERROR(0x80070032)
#define HRES_E_INVALIDARG				  HRES_ERROR(0x80070057)
#define HRES_CO_E_CLASS_CREATE_FAILED			  HRES_ERROR(0x80080001)
#define HRES_CO_E_SCM_ERROR				  HRES_ERROR(0x80080002)
#define HRES_CO_E_SCM_RPC_FAILURE			  HRES_ERROR(0x80080003)
#define HRES_CO_E_BAD_PATH				  HRES_ERROR(0x80080004)
#define HRES_CO_E_SERVER_EXEC_FAILURE			  HRES_ERROR(0x80080005)
#define HRES_CO_E_OBJSRV_RPC_FAILURE			  HRES_ERROR(0x80080006)
#define HRES_MK_E_NO_NORMALIZED				  HRES_ERROR(0x80080007)
#define HRES_CO_E_SERVER_STOPPING			  HRES_ERROR(0x80080008)
#define HRES_MEM_E_INVALID_ROOT				  HRES_ERROR(0x80080009)
#define HRES_MEM_E_INVALID_LINK				  HRES_ERROR(0x80080010)
#define HRES_MEM_E_INVALID_SIZE				  HRES_ERROR(0x80080011)
#define HRES_CO_E_MISSING_DISPLAYNAME			  HRES_ERROR(0x80080015)
#define HRES_CO_E_RUNAS_VALUE_MUST_BE_AAA		  HRES_ERROR(0x80080016)
#define HRES_CO_E_ELEVATION_DISABLED			  HRES_ERROR(0x80080017)
#define HRES_NTE_BAD_UID				  HRES_ERROR(0x80090001)
#define HRES_NTE_BAD_HASH				  HRES_ERROR(0x80090002)
#define HRES_NTE_BAD_KEY				  HRES_ERROR(0x80090003)
#define HRES_NTE_BAD_LEN				  HRES_ERROR(0x80090004)
#define HRES_NTE_BAD_DATA				  HRES_ERROR(0x80090005)
#define HRES_NTE_BAD_SIGNATURE				  HRES_ERROR(0x80090006)
#define HRES_NTE_BAD_VER				  HRES_ERROR(0x80090007)
#define HRES_NTE_BAD_ALGID				  HRES_ERROR(0x80090008)
#define HRES_NTE_BAD_FLAGS				  HRES_ERROR(0x80090009)
#define HRES_NTE_BAD_TYPE				  HRES_ERROR(0x8009000A)
#define HRES_NTE_BAD_KEY_STATE				  HRES_ERROR(0x8009000B)
#define HRES_NTE_BAD_HASH_STATE				  HRES_ERROR(0x8009000C)
#define HRES_NTE_NO_KEY					  HRES_ERROR(0x8009000D)
#define HRES_NTE_NO_MEMORY				  HRES_ERROR(0x8009000E)
#define HRES_NTE_EXISTS					  HRES_ERROR(0x8009000F)
#define HRES_NTE_PERM					  HRES_ERROR(0x80090010)
#define HRES_NTE_NOT_FOUND				  HRES_ERROR(0x80090011)
#define HRES_NTE_DOUBLE_ENCRYPT				  HRES_ERROR(0x80090012)
#define HRES_NTE_BAD_PROVIDER				  HRES_ERROR(0x80090013)
#define HRES_NTE_BAD_PROV_TYPE				  HRES_ERROR(0x80090014)
#define HRES_NTE_BAD_PUBLIC_KEY				  HRES_ERROR(0x80090015)
#define HRES_NTE_BAD_KEYSET				  HRES_ERROR(0x80090016)
#define HRES_NTE_PROV_TYPE_NOT_DEF			  HRES_ERROR(0x80090017)
#define HRES_NTE_PROV_TYPE_ENTRY_BAD			  HRES_ERROR(0x80090018)
#define HRES_NTE_KEYSET_NOT_DEF				  HRES_ERROR(0x80090019)
#define HRES_NTE_KEYSET_ENTRY_BAD			  HRES_ERROR(0x8009001A)
#define HRES_NTE_PROV_TYPE_NO_MATCH			  HRES_ERROR(0x8009001B)
#define HRES_NTE_SIGNATURE_FILE_BAD			  HRES_ERROR(0x8009001C)
#define HRES_NTE_PROVIDER_DLL_FAIL			  HRES_ERROR(0x8009001D)
#define HRES_NTE_PROV_DLL_NOT_FOUND			  HRES_ERROR(0x8009001E)
#define HRES_NTE_BAD_KEYSET_PARAM			  HRES_ERROR(0x8009001F)
#define HRES_NTE_FAIL					  HRES_ERROR(0x80090020)
#define HRES_NTE_SYS_ERR				  HRES_ERROR(0x80090021)
#define HRES_NTE_SILENT_CONTEXT				  HRES_ERROR(0x80090022)
#define HRES_NTE_TOKEN_KEYSET_STORAGE_FULL		  HRES_ERROR(0x80090023)
#define HRES_NTE_TEMPORARY_PROFILE			  HRES_ERROR(0x80090024)
#define HRES_NTE_FIXEDPARAMETER				  HRES_ERROR(0x80090025)
#define HRES_NTE_INVALID_HANDLE				  HRES_ERROR(0x80090026)
#define HRES_NTE_INVALID_PARAMETER			  HRES_ERROR(0x80090027)
#define HRES_NTE_BUFFER_TOO_SMALL			  HRES_ERROR(0x80090028)
#define HRES_NTE_NOT_SUPPORTED				  HRES_ERROR(0x80090029)
#define HRES_NTE_NO_MORE_ITEMS				  HRES_ERROR(0x8009002A)
#define HRES_NTE_BUFFERS_OVERLAP			  HRES_ERROR(0x8009002B)
#define HRES_NTE_DECRYPTION_FAILURE			  HRES_ERROR(0x8009002C)
#define HRES_NTE_INTERNAL_ERROR				  HRES_ERROR(0x8009002D)
#define HRES_NTE_UI_REQUIRED				  HRES_ERROR(0x8009002E)
#define HRES_NTE_HMAC_NOT_SUPPORTED			  HRES_ERROR(0x8009002F)
#define HRES_SEC_E_INSUFFICIENT_MEMORY			  HRES_ERROR(0x80090300)
#define HRES_SEC_E_INVALID_HANDLE			  HRES_ERROR(0x80090301)
#define HRES_SEC_E_UNSUPPORTED_FUNCTION			  HRES_ERROR(0x80090302)
#define HRES_SEC_E_TARGET_UNKNOWN			  HRES_ERROR(0x80090303)
#define HRES_SEC_E_INTERNAL_ERROR			  HRES_ERROR(0x80090304)
#define HRES_SEC_E_SECPKG_NOT_FOUND			  HRES_ERROR(0x80090305)
#define HRES_SEC_E_NOT_OWNER				  HRES_ERROR(0x80090306)
#define HRES_SEC_E_CANNOT_INSTALL			  HRES_ERROR(0x80090307)
#define HRES_SEC_E_INVALID_TOKEN			  HRES_ERROR(0x80090308)
#define HRES_SEC_E_CANNOT_PACK				  HRES_ERROR(0x80090309)
#define HRES_SEC_E_QOP_NOT_SUPPORTED			  HRES_ERROR(0x8009030A)
#define HRES_SEC_E_NO_IMPERSONATION			  HRES_ERROR(0x8009030B)
#define HRES_SEC_E_LOGON_DENIED				  HRES_ERROR(0x8009030C)
#define HRES_SEC_E_UNKNOWN_CREDENTIALS			  HRES_ERROR(0x8009030D)
#define HRES_SEC_E_NO_CREDENTIALS			  HRES_ERROR(0x8009030E)
#define HRES_SEC_E_MESSAGE_ALTERED			  HRES_ERROR(0x8009030F)
#define HRES_SEC_E_OUT_OF_SEQUENCE			  HRES_ERROR(0x80090310)
#define HRES_SEC_E_NO_AUTHENTICATING_AUTHORITY		  HRES_ERROR(0x80090311)
#define HRES_SEC_E_BAD_PKGID				  HRES_ERROR(0x80090316)
#define HRES_SEC_E_CONTEXT_EXPIRED			  HRES_ERROR(0x80090317)
#define HRES_SEC_E_INCOMPLETE_MESSAGE			  HRES_ERROR(0x80090318)
#define HRES_SEC_E_INCOMPLETE_CREDENTIALS		  HRES_ERROR(0x80090320)
#define HRES_SEC_E_BUFFER_TOO_SMALL			  HRES_ERROR(0x80090321)
#define HRES_SEC_E_WRONG_PRINCIPAL			  HRES_ERROR(0x80090322)
#define HRES_SEC_E_TIME_SKEW				  HRES_ERROR(0x80090324)
#define HRES_SEC_E_UNTRUSTED_ROOT			  HRES_ERROR(0x80090325)
#define HRES_SEC_E_ILLEGAL_MESSAGE			  HRES_ERROR(0x80090326)
#define HRES_SEC_E_CERT_UNKNOWN				  HRES_ERROR(0x80090327)
#define HRES_SEC_E_CERT_EXPIRED				  HRES_ERROR(0x80090328)
#define HRES_SEC_E_ENCRYPT_FAILURE			  HRES_ERROR(0x80090329)
#define HRES_SEC_E_DECRYPT_FAILURE			  HRES_ERROR(0x80090330)
#define HRES_SEC_E_ALGORITHM_MISMATCH			  HRES_ERROR(0x80090331)
#define HRES_SEC_E_SECURITY_QOS_FAILED			  HRES_ERROR(0x80090332)
#define HRES_SEC_E_UNFINISHED_CONTEXT_DELETED		  HRES_ERROR(0x80090333)
#define HRES_SEC_E_NO_TGT_REPLY				  HRES_ERROR(0x80090334)
#define HRES_SEC_E_NO_IP_ADDRESSES			  HRES_ERROR(0x80090335)
#define HRES_SEC_E_WRONG_CREDENTIAL_HANDLE		  HRES_ERROR(0x80090336)
#define HRES_SEC_E_CRYPTO_SYSTEM_INVALID		  HRES_ERROR(0x80090337)
#define HRES_SEC_E_MAX_REFERRALS_EXCEEDED		  HRES_ERROR(0x80090338)
#define HRES_SEC_E_MUST_BE_KDC				  HRES_ERROR(0x80090339)
#define HRES_SEC_E_STRONG_CRYPTO_NOT_SUPPORTED		  HRES_ERROR(0x8009033A)
#define HRES_SEC_E_TOO_MANY_PRINCIPALS			  HRES_ERROR(0x8009033B)
#define HRES_SEC_E_NO_PA_DATA				  HRES_ERROR(0x8009033C)
#define HRES_SEC_E_PKINIT_NAME_MISMATCH			  HRES_ERROR(0x8009033D)
#define HRES_SEC_E_SMARTCARD_LOGON_REQUIRED		  HRES_ERROR(0x8009033E)
#define HRES_SEC_E_SHUTDOWN_IN_PROGRESS			  HRES_ERROR(0x8009033F)
#define HRES_SEC_E_KDC_INVALID_REQUEST			  HRES_ERROR(0x80090340)
#define HRES_SEC_E_KDC_UNABLE_TO_REFER			  HRES_ERROR(0x80090341)
#define HRES_SEC_E_KDC_UNKNOWN_ETYPE			  HRES_ERROR(0x80090342)
#define HRES_SEC_E_UNSUPPORTED_PREAUTH			  HRES_ERROR(0x80090343)
#define HRES_SEC_E_DELEGATION_REQUIRED			  HRES_ERROR(0x80090345)
#define HRES_SEC_E_BAD_BINDINGS				  HRES_ERROR(0x80090346)
#define HRES_SEC_E_MULTIPLE_ACCOUNTS			  HRES_ERROR(0x80090347)
#define HRES_SEC_E_NO_KERB_KEY				  HRES_ERROR(0x80090348)
#define HRES_SEC_E_CERT_WRONG_USAGE			  HRES_ERROR(0x80090349)
#define HRES_SEC_E_DOWNGRADE_DETECTED			  HRES_ERROR(0x80090350)
#define HRES_SEC_E_SMARTCARD_CERT_REVOKED		  HRES_ERROR(0x80090351)
#define HRES_SEC_E_ISSUING_CA_UNTRUSTED			  HRES_ERROR(0x80090352)
#define HRES_SEC_E_REVOCATION_OFFLINE_C			  HRES_ERROR(0x80090353)
#define HRES_SEC_E_PKINIT_CLIENT_FAILURE		  HRES_ERROR(0x80090354)
#define HRES_SEC_E_SMARTCARD_CERT_EXPIRED		  HRES_ERROR(0x80090355)
#define HRES_SEC_E_NO_S4U_PROT_SUPPORT			  HRES_ERROR(0x80090356)
#define HRES_SEC_E_CROSSREALM_DELEGATION_FAILURE	  HRES_ERROR(0x80090357)
#define HRES_SEC_E_REVOCATION_OFFLINE_KDC		  HRES_ERROR(0x80090358)
#define HRES_SEC_E_ISSUING_CA_UNTRUSTED_KDC		  HRES_ERROR(0x80090359)
#define HRES_SEC_E_KDC_CERT_EXPIRED			  HRES_ERROR(0x8009035A)
#define HRES_SEC_E_KDC_CERT_REVOKED			  HRES_ERROR(0x8009035B)
#define HRES_SEC_E_INVALID_PARAMETER			  HRES_ERROR(0x8009035D)
#define HRES_SEC_E_DELEGATION_POLICY			  HRES_ERROR(0x8009035E)
#define HRES_SEC_E_POLICY_NLTM_ONLY			  HRES_ERROR(0x8009035F)
#define HRES_CRYPT_E_MSG_ERROR				  HRES_ERROR(0x80091001)
#define HRES_CRYPT_E_UNKNOWN_ALGO			  HRES_ERROR(0x80091002)
#define HRES_CRYPT_E_OID_FORMAT				  HRES_ERROR(0x80091003)
#define HRES_CRYPT_E_INVALID_MSG_TYPE			  HRES_ERROR(0x80091004)
#define HRES_CRYPT_E_UNEXPECTED_ENCODING		  HRES_ERROR(0x80091005)
#define HRES_CRYPT_E_AUTH_ATTR_MISSING			  HRES_ERROR(0x80091006)
#define HRES_CRYPT_E_HASH_VALUE				  HRES_ERROR(0x80091007)
#define HRES_CRYPT_E_INVALID_INDEX			  HRES_ERROR(0x80091008)
#define HRES_CRYPT_E_ALREADY_DECRYPTED			  HRES_ERROR(0x80091009)
#define HRES_CRYPT_E_NOT_DECRYPTED			  HRES_ERROR(0x8009100A)
#define HRES_CRYPT_E_RECIPIENT_NOT_FOUND		  HRES_ERROR(0x8009100B)
#define HRES_CRYPT_E_CONTROL_TYPE			  HRES_ERROR(0x8009100C)
#define HRES_CRYPT_E_ISSUER_SERIALNUMBER		  HRES_ERROR(0x8009100D)
#define HRES_CRYPT_E_SIGNER_NOT_FOUND			  HRES_ERROR(0x8009100E)
#define HRES_CRYPT_E_ATTRIBUTES_MISSING			  HRES_ERROR(0x8009100F)
#define HRES_CRYPT_E_STREAM_MSG_NOT_READY		  HRES_ERROR(0x80091010)
#define HRES_CRYPT_E_STREAM_INSUFFICIENT_DATA		  HRES_ERROR(0x80091011)
#define HRES_CRYPT_E_BAD_LEN				  HRES_ERROR(0x80092001)
#define HRES_CRYPT_E_BAD_ENCODE				  HRES_ERROR(0x80092002)
#define HRES_CRYPT_E_FILE_ERROR				  HRES_ERROR(0x80092003)
#define HRES_CRYPT_E_NOT_FOUND				  HRES_ERROR(0x80092004)
#define HRES_CRYPT_E_EXISTS				  HRES_ERROR(0x80092005)
#define HRES_CRYPT_E_NO_PROVIDER			  HRES_ERROR(0x80092006)
#define HRES_CRYPT_E_SELF_SIGNED			  HRES_ERROR(0x80092007)
#define HRES_CRYPT_E_DELETED_PREV			  HRES_ERROR(0x80092008)
#define HRES_CRYPT_E_NO_MATCH				  HRES_ERROR(0x80092009)
#define HRES_CRYPT_E_UNEXPECTED_MSG_TYPE		  HRES_ERROR(0x8009200A)
#define HRES_CRYPT_E_NO_KEY_PROPERTY			  HRES_ERROR(0x8009200B)
#define HRES_CRYPT_E_NO_DECRYPT_CERT			  HRES_ERROR(0x8009200C)
#define HRES_CRYPT_E_BAD_MSG				  HRES_ERROR(0x8009200D)
#define HRES_CRYPT_E_NO_SIGNER				  HRES_ERROR(0x8009200E)
#define HRES_CRYPT_E_PENDING_CLOSE			  HRES_ERROR(0x8009200F)
#define HRES_CRYPT_E_REVOKED				  HRES_ERROR(0x80092010)
#define HRES_CRYPT_E_NO_REVOCATION_DLL			  HRES_ERROR(0x80092011)
#define HRES_CRYPT_E_NO_REVOCATION_CHECK		  HRES_ERROR(0x80092012)
#define HRES_CRYPT_E_REVOCATION_OFFLINE			  HRES_ERROR(0x80092013)
#define HRES_CRYPT_E_NOT_IN_REVOCATION_DATABASE		  HRES_ERROR(0x80092014)
#define HRES_CRYPT_E_INVALID_NUMERIC_STRING		  HRES_ERROR(0x80092020)
#define HRES_CRYPT_E_INVALID_PRINTABLE_STRING		  HRES_ERROR(0x80092021)
#define HRES_CRYPT_E_INVALID_IA5_STRING			  HRES_ERROR(0x80092022)
#define HRES_CRYPT_E_INVALID_X500_STRING		  HRES_ERROR(0x80092023)
#define HRES_CRYPT_E_NOT_CHAR_STRING			  HRES_ERROR(0x80092024)
#define HRES_CRYPT_E_FILERESIZED			  HRES_ERROR(0x80092025)
#define HRES_CRYPT_E_SECURITY_SETTINGS			  HRES_ERROR(0x80092026)
#define HRES_CRYPT_E_NO_VERIFY_USAGE_DLL		  HRES_ERROR(0x80092027)
#define HRES_CRYPT_E_NO_VERIFY_USAGE_CHECK		  HRES_ERROR(0x80092028)
#define HRES_CRYPT_E_VERIFY_USAGE_OFFLINE		  HRES_ERROR(0x80092029)
#define HRES_CRYPT_E_NOT_IN_CTL				  HRES_ERROR(0x8009202A)
#define HRES_CRYPT_E_NO_TRUSTED_SIGNER			  HRES_ERROR(0x8009202B)
#define HRES_CRYPT_E_MISSING_PUBKEY_PARA		  HRES_ERROR(0x8009202C)
#define HRES_CRYPT_E_OSS_ERROR				  HRES_ERROR(0x80093000)
#define HRES_OSS_MORE_BUF				  HRES_ERROR(0x80093001)
#define HRES_OSS_NEGATIVE_UINTEGER			  HRES_ERROR(0x80093002)
#define HRES_OSS_PDU_RANGE				  HRES_ERROR(0x80093003)
#define HRES_OSS_MORE_INPUT				  HRES_ERROR(0x80093004)
#define HRES_OSS_DATA_ERROR				  HRES_ERROR(0x80093005)
#define HRES_OSS_BAD_ARG				  HRES_ERROR(0x80093006)
#define HRES_OSS_BAD_VERSION				  HRES_ERROR(0x80093007)
#define HRES_OSS_OUT_MEMORY				  HRES_ERROR(0x80093008)
#define HRES_OSS_PDU_MISMATCH				  HRES_ERROR(0x80093009)
#define HRES_OSS_LIMITED				  HRES_ERROR(0x8009300A)
#define HRES_OSS_BAD_PTR				  HRES_ERROR(0x8009300B)
#define HRES_OSS_BAD_TIME				  HRES_ERROR(0x8009300C)
#define HRES_OSS_INDEFINITE_NOT_SUPPORTED		  HRES_ERROR(0x8009300D)
#define HRES_OSS_MEM_ERROR				  HRES_ERROR(0x8009300E)
#define HRES_OSS_BAD_TABLE				  HRES_ERROR(0x8009300F)
#define HRES_OSS_TOO_LONG				  HRES_ERROR(0x80093010)
#define HRES_OSS_CONSTRAINT_VIOLATED			  HRES_ERROR(0x80093011)
#define HRES_OSS_FATAL_ERROR				  HRES_ERROR(0x80093012)
#define HRES_OSS_ACCESS_SERIALIZATION_ERROR		  HRES_ERROR(0x80093013)
#define HRES_OSS_NULL_TBL				  HRES_ERROR(0x80093014)
#define HRES_OSS_NULL_FCN				  HRES_ERROR(0x80093015)
#define HRES_OSS_BAD_ENCRULES				  HRES_ERROR(0x80093016)
#define HRES_OSS_UNAVAIL_ENCRULES			  HRES_ERROR(0x80093017)
#define HRES_OSS_CANT_OPEN_TRACE_WINDOW			  HRES_ERROR(0x80093018)
#define HRES_OSS_UNIMPLEMENTED				  HRES_ERROR(0x80093019)
#define HRES_OSS_OID_DLL_NOT_LINKED			  HRES_ERROR(0x8009301A)
#define HRES_OSS_CANT_OPEN_TRACE_FILE			  HRES_ERROR(0x8009301B)
#define HRES_OSS_TRACE_FILE_ALREADY_OPEN		  HRES_ERROR(0x8009301C)
#define HRES_OSS_TABLE_MISMATCH				  HRES_ERROR(0x8009301D)
#define HRES_OSS_TYPE_NOT_SUPPORTED			  HRES_ERROR(0x8009301E)
#define HRES_OSS_REAL_DLL_NOT_LINKED			  HRES_ERROR(0x8009301F)
#define HRES_OSS_REAL_CODE_NOT_LINKED			  HRES_ERROR(0x80093020)
#define HRES_OSS_OUT_OF_RANGE				  HRES_ERROR(0x80093021)
#define HRES_OSS_COPIER_DLL_NOT_LINKED			  HRES_ERROR(0x80093022)
#define HRES_OSS_CONSTRAINT_DLL_NOT_LINKED		  HRES_ERROR(0x80093023)
#define HRES_OSS_COMPARATOR_DLL_NOT_LINKED		  HRES_ERROR(0x80093024)
#define HRES_OSS_COMPARATOR_CODE_NOT_LINKED		  HRES_ERROR(0x80093025)
#define HRES_OSS_MEM_MGR_DLL_NOT_LINKED			  HRES_ERROR(0x80093026)
#define HRES_OSS_PDV_DLL_NOT_LINKED			  HRES_ERROR(0x80093027)
#define HRES_OSS_PDV_CODE_NOT_LINKED			  HRES_ERROR(0x80093028)
#define HRES_OSS_API_DLL_NOT_LINKED			  HRES_ERROR(0x80093029)
#define HRES_OSS_BERDER_DLL_NOT_LINKED			  HRES_ERROR(0x8009302A)
#define HRES_OSS_PER_DLL_NOT_LINKED			  HRES_ERROR(0x8009302B)
#define HRES_OSS_OPEN_TYPE_ERROR			  HRES_ERROR(0x8009302C)
#define HRES_OSS_MUTEX_NOT_CREATED			  HRES_ERROR(0x8009302D)
#define HRES_OSS_CANT_CLOSE_TRACE_FILE			  HRES_ERROR(0x8009302E)
#define HRES_CRYPT_E_ASN1_ERROR				  HRES_ERROR(0x80093100)
#define HRES_CRYPT_E_ASN1_INTERNAL			  HRES_ERROR(0x80093101)
#define HRES_CRYPT_E_ASN1_EOD				  HRES_ERROR(0x80093102)
#define HRES_CRYPT_E_ASN1_CORRUPT			  HRES_ERROR(0x80093103)
#define HRES_CRYPT_E_ASN1_LARGE				  HRES_ERROR(0x80093104)
#define HRES_CRYPT_E_ASN1_CONSTRAINT			  HRES_ERROR(0x80093105)
#define HRES_CRYPT_E_ASN1_MEMORY			  HRES_ERROR(0x80093106)
#define HRES_CRYPT_E_ASN1_OVERFLOW			  HRES_ERROR(0x80093107)
#define HRES_CRYPT_E_ASN1_BADPDU			  HRES_ERROR(0x80093108)
#define HRES_CRYPT_E_ASN1_BADARGS			  HRES_ERROR(0x80093109)
#define HRES_CRYPT_E_ASN1_BADREAL			  HRES_ERROR(0x8009310A)
#define HRES_CRYPT_E_ASN1_BADTAG			  HRES_ERROR(0x8009310B)
#define HRES_CRYPT_E_ASN1_CHOICE			  HRES_ERROR(0x8009310C)
#define HRES_CRYPT_E_ASN1_RULE				  HRES_ERROR(0x8009310D)
#define HRES_CRYPT_E_ASN1_UTF8				  HRES_ERROR(0x8009310E)
#define HRES_CRYPT_E_ASN1_PDU_TYPE			  HRES_ERROR(0x80093133)
#define HRES_CRYPT_E_ASN1_NYI				  HRES_ERROR(0x80093134)
#define HRES_CRYPT_E_ASN1_EXTENDED			  HRES_ERROR(0x80093201)
#define HRES_CRYPT_E_ASN1_NOEOD				  HRES_ERROR(0x80093202)
#define HRES_CERTSRV_E_BAD_REQUESTSUBJECT		  HRES_ERROR(0x80094001)
#define HRES_CERTSRV_E_NO_REQUEST			  HRES_ERROR(0x80094002)
#define HRES_CERTSRV_E_BAD_REQUESTSTATUS		  HRES_ERROR(0x80094003)
#define HRES_CERTSRV_E_PROPERTY_EMPTY			  HRES_ERROR(0x80094004)
#define HRES_CERTSRV_E_INVALID_CA_CERTIFICATE		  HRES_ERROR(0x80094005)
#define HRES_CERTSRV_E_SERVER_SUSPENDED			  HRES_ERROR(0x80094006)
#define HRES_CERTSRV_E_ENCODING_LENGTH			  HRES_ERROR(0x80094007)
#define HRES_CERTSRV_E_ROLECONFLICT			  HRES_ERROR(0x80094008)
#define HRES_CERTSRV_E_RESTRICTEDOFFICER		  HRES_ERROR(0x80094009)
#define HRES_CERTSRV_E_KEY_ARCHIVAL_NOT_CONFIGURED	  HRES_ERROR(0x8009400A)
#define HRES_CERTSRV_E_NO_VALID_KRA			  HRES_ERROR(0x8009400B)
#define HRES_CERTSRV_E_BAD_REQUEST_KEY_ARCHIVAL		  HRES_ERROR(0x8009400C)
#define HRES_CERTSRV_E_NO_CAADMIN_DEFINED		  HRES_ERROR(0x8009400D)
#define HRES_CERTSRV_E_BAD_RENEWAL_CERT_ATTRIBUTE	  HRES_ERROR(0x8009400E)
#define HRES_CERTSRV_E_NO_DB_SESSIONS			  HRES_ERROR(0x8009400F)
#define HRES_CERTSRV_E_ALIGNMENT_FAULT			  HRES_ERROR(0x80094010)
#define HRES_CERTSRV_E_ENROLL_DENIED			  HRES_ERROR(0x80094011)
#define HRES_CERTSRV_E_TEMPLATE_DENIED			  HRES_ERROR(0x80094012)
#define HRES_CERTSRV_E_DOWNLEVEL_DC_SSL_OR_UPGRADE	  HRES_ERROR(0x80094013)
#define HRES_CERTSRV_E_UNSUPPORTED_CERT_TYPE		  HRES_ERROR(0x80094800)
#define HRES_CERTSRV_E_NO_CERT_TYPE			  HRES_ERROR(0x80094801)
#define HRES_CERTSRV_E_TEMPLATE_CONFLICT		  HRES_ERROR(0x80094802)
#define HRES_CERTSRV_E_SUBJECT_ALT_NAME_REQUIRED	  HRES_ERROR(0x80094803)
#define HRES_CERTSRV_E_ARCHIVED_KEY_REQUIRED		  HRES_ERROR(0x80094804)
#define HRES_CERTSRV_E_SMIME_REQUIRED			  HRES_ERROR(0x80094805)
#define HRES_CERTSRV_E_BAD_RENEWAL_SUBJECT		  HRES_ERROR(0x80094806)
#define HRES_CERTSRV_E_BAD_TEMPLATE_VERSION		  HRES_ERROR(0x80094807)
#define HRES_CERTSRV_E_TEMPLATE_POLICY_REQUIRED		  HRES_ERROR(0x80094808)
#define HRES_CERTSRV_E_SIGNATURE_POLICY_REQUIRED	  HRES_ERROR(0x80094809)
#define HRES_CERTSRV_E_SIGNATURE_COUNT			  HRES_ERROR(0x8009480A)
#define HRES_CERTSRV_E_SIGNATURE_REJECTED		  HRES_ERROR(0x8009480B)
#define HRES_CERTSRV_E_ISSUANCE_POLICY_REQUIRED		  HRES_ERROR(0x8009480C)
#define HRES_CERTSRV_E_SUBJECT_UPN_REQUIRED		  HRES_ERROR(0x8009480D)
#define HRES_CERTSRV_E_SUBJECT_DIRECTORY_GUID_REQUIRED	  HRES_ERROR(0x8009480E)
#define HRES_CERTSRV_E_SUBJECT_DNS_REQUIRED		  HRES_ERROR(0x8009480F)
#define HRES_CERTSRV_E_ARCHIVED_KEY_UNEXPECTED		  HRES_ERROR(0x80094810)
#define HRES_CERTSRV_E_KEY_LENGTH			  HRES_ERROR(0x80094811)
#define HRES_CERTSRV_E_SUBJECT_EMAIL_REQUIRED		  HRES_ERROR(0x80094812)
#define HRES_CERTSRV_E_UNKNOWN_CERT_TYPE		  HRES_ERROR(0x80094813)
#define HRES_CERTSRV_E_CERT_TYPE_OVERLAP		  HRES_ERROR(0x80094814)
#define HRES_CERTSRV_E_TOO_MANY_SIGNATURES		  HRES_ERROR(0x80094815)
#define HRES_CERTSRV_E_RENEWAL_BAD_PUBLIC_KEY		  HRES_ERROR(0x80094816)
#define HRES_CERTSRV_E_INVALID_EK			  HRES_ERROR(0x80094817)
#define HRES_CERTSRV_E_KEY_ATTESTATION			  HRES_ERROR(0x8009481A)
#define HRES_XENROLL_E_KEY_NOT_EXPORTABLE		  HRES_ERROR(0x80095000)
#define HRES_XENROLL_E_CANNOT_ADD_ROOT_CERT		  HRES_ERROR(0x80095001)
#define HRES_XENROLL_E_RESPONSE_KA_HASH_NOT_FOUND	  HRES_ERROR(0x80095002)
#define HRES_XENROLL_E_RESPONSE_UNEXPECTED_KA_HASH	  HRES_ERROR(0x80095003)
#define HRES_XENROLL_E_RESPONSE_KA_HASH_MISMATCH	  HRES_ERROR(0x80095004)
#define HRES_XENROLL_E_KEYSPEC_SMIME_MISMATCH		  HRES_ERROR(0x80095005)
#define HRES_TRUST_E_SYSTEM_ERROR			  HRES_ERROR(0x80096001)
#define HRES_TRUST_E_NO_SIGNER_CERT			  HRES_ERROR(0x80096002)
#define HRES_TRUST_E_COUNTER_SIGNER			  HRES_ERROR(0x80096003)
#define HRES_TRUST_E_CERT_SIGNATURE			  HRES_ERROR(0x80096004)
#define HRES_TRUST_E_TIME_STAMP				  HRES_ERROR(0x80096005)
#define HRES_TRUST_E_BAD_DIGEST				  HRES_ERROR(0x80096010)
#define HRES_TRUST_E_BASIC_CONSTRAINTS			  HRES_ERROR(0x80096019)
#define HRES_TRUST_E_FINANCIAL_CRITERIA			  HRES_ERROR(0x8009601E)
#define HRES_MSSIPOTF_E_OUTOFMEMRANGE			  HRES_ERROR(0x80097001)
#define HRES_MSSIPOTF_E_CANTGETOBJECT			  HRES_ERROR(0x80097002)
#define HRES_MSSIPOTF_E_NOHEADTABLE			  HRES_ERROR(0x80097003)
#define HRES_MSSIPOTF_E_BAD_MAGICNUMBER			  HRES_ERROR(0x80097004)
#define HRES_MSSIPOTF_E_BAD_OFFSET_TABLE		  HRES_ERROR(0x80097005)
#define HRES_MSSIPOTF_E_TABLE_TAGORDER			  HRES_ERROR(0x80097006)
#define HRES_MSSIPOTF_E_TABLE_LONGWORD			  HRES_ERROR(0x80097007)
#define HRES_MSSIPOTF_E_BAD_FIRST_TABLE_PLACEMENT	  HRES_ERROR(0x80097008)
#define HRES_MSSIPOTF_E_TABLES_OVERLAP			  HRES_ERROR(0x80097009)
#define HRES_MSSIPOTF_E_TABLE_PADBYTES			  HRES_ERROR(0x8009700A)
#define HRES_MSSIPOTF_E_FILETOOSMALL			  HRES_ERROR(0x8009700B)
#define HRES_MSSIPOTF_E_TABLE_CHECKSUM			  HRES_ERROR(0x8009700C)
#define HRES_MSSIPOTF_E_FILE_CHECKSUM			  HRES_ERROR(0x8009700D)
#define HRES_MSSIPOTF_E_FAILED_POLICY			  HRES_ERROR(0x80097010)
#define HRES_MSSIPOTF_E_FAILED_HINTS_CHECK		  HRES_ERROR(0x80097011)
#define HRES_MSSIPOTF_E_NOT_OPENTYPE			  HRES_ERROR(0x80097012)
#define HRES_MSSIPOTF_E_FILE				  HRES_ERROR(0x80097013)
#define HRES_MSSIPOTF_E_CRYPT				  HRES_ERROR(0x80097014)
#define HRES_MSSIPOTF_E_BADVERSION			  HRES_ERROR(0x80097015)
#define HRES_MSSIPOTF_E_DSIG_STRUCTURE			  HRES_ERROR(0x80097016)
#define HRES_MSSIPOTF_E_PCONST_CHECK			  HRES_ERROR(0x80097017)
#define HRES_MSSIPOTF_E_STRUCTURE			  HRES_ERROR(0x80097018)
#define HRES_ERROR_CRED_REQUIRES_CONFIRMATION		  HRES_ERROR(0x80097019)
#define HRES_TRUST_E_PROVIDER_UNKNOWN			  HRES_ERROR(0x800B0001)
#define HRES_TRUST_E_ACTION_UNKNOWN			  HRES_ERROR(0x800B0002)
#define HRES_TRUST_E_SUBJECT_FORM_UNKNOWN		  HRES_ERROR(0x800B0003)
#define HRES_TRUST_E_SUBJECT_NOT_TRUSTED		  HRES_ERROR(0x800B0004)
#define HRES_DIGSIG_E_ENCODE				  HRES_ERROR(0x800B0005)
#define HRES_DIGSIG_E_DECODE				  HRES_ERROR(0x800B0006)
#define HRES_DIGSIG_E_EXTENSIBILITY			  HRES_ERROR(0x800B0007)
#define HRES_DIGSIG_E_CRYPTO				  HRES_ERROR(0x800B0008)
#define HRES_PERSIST_E_SIZEDEFINITE			  HRES_ERROR(0x800B0009)
#define HRES_PERSIST_E_SIZEINDEFINITE			  HRES_ERROR(0x800B000A)
#define HRES_PERSIST_E_NOTSELFSIZING			  HRES_ERROR(0x800B000B)
#define HRES_TRUST_E_NOSIGNATURE			  HRES_ERROR(0x800B0100)
#define HRES_CERT_E_EXPIRED				  HRES_ERROR(0x800B0101)
#define HRES_CERT_E_VALIDITYPERIODNESTING		  HRES_ERROR(0x800B0102)
#define HRES_CERT_E_ROLE				  HRES_ERROR(0x800B0103)
#define HRES_CERT_E_PATHLENCONST			  HRES_ERROR(0x800B0104)
#define HRES_CERT_E_CRITICAL				  HRES_ERROR(0x800B0105)
#define HRES_CERT_E_PURPOSE				  HRES_ERROR(0x800B0106)
#define HRES_CERT_E_ISSUERCHAINING			  HRES_ERROR(0x800B0107)
#define HRES_CERT_E_MALFORMED				  HRES_ERROR(0x800B0108)
#define HRES_CERT_E_UNTRUSTEDROOT			  HRES_ERROR(0x800B0109)
#define HRES_CERT_E_CHAINING				  HRES_ERROR(0x800B010A)
#define HRES_TRUST_E_FAIL				  HRES_ERROR(0x800B010B)
#define HRES_CERT_E_REVOKED				  HRES_ERROR(0x800B010C)
#define HRES_CERT_E_UNTRUSTEDTESTROOT			  HRES_ERROR(0x800B010D)
#define HRES_CERT_E_REVOCATION_FAILURE			  HRES_ERROR(0x800B010E)
#define HRES_CERT_E_CN_NO_MATCH				  HRES_ERROR(0x800B010F)
#define HRES_CERT_E_WRONG_USAGE				  HRES_ERROR(0x800B0110)
#define HRES_TRUST_E_EXPLICIT_DISTRUST			  HRES_ERROR(0x800B0111)
#define HRES_CERT_E_UNTRUSTEDCA				  HRES_ERROR(0x800B0112)
#define HRES_CERT_E_INVALID_POLICY			  HRES_ERROR(0x800B0113)
#define HRES_CERT_E_INVALID_NAME			  HRES_ERROR(0x800B0114)
#define HRES_NS_W_SERVER_BANDWIDTH_LIMIT		  HRES_ERROR(0x800D0003)
#define HRES_NS_W_FILE_BANDWIDTH_LIMIT			  HRES_ERROR(0x800D0004)
#define HRES_NS_W_UNKNOWN_EVENT				  HRES_ERROR(0x800D0060)
#define HRES_NS_I_CATATONIC_FAILURE			  HRES_ERROR(0x800D0199)
#define HRES_NS_I_CATATONIC_AUTO_UNFAIL			  HRES_ERROR(0x800D019A)
#define HRES_SPAPI_E_EXPECTED_SECTION_NAME		  HRES_ERROR(0x800F0000)
#define HRES_SPAPI_E_BAD_SECTION_NAME_LINE		  HRES_ERROR(0x800F0001)
#define HRES_SPAPI_E_SECTION_NAME_TOO_LONG		  HRES_ERROR(0x800F0002)
#define HRES_SPAPI_E_GENERAL_SYNTAX			  HRES_ERROR(0x800F0003)
#define HRES_SPAPI_E_WRONG_INF_STYLE			  HRES_ERROR(0x800F0100)
#define HRES_SPAPI_E_SECTION_NOT_FOUND			  HRES_ERROR(0x800F0101)
#define HRES_SPAPI_E_LINE_NOT_FOUND			  HRES_ERROR(0x800F0102)
#define HRES_SPAPI_E_NO_BACKUP				  HRES_ERROR(0x800F0103)
#define HRES_SPAPI_E_NO_ASSOCIATED_CLASS		  HRES_ERROR(0x800F0200)
#define HRES_SPAPI_E_CLASS_MISMATCH			  HRES_ERROR(0x800F0201)
#define HRES_SPAPI_E_DUPLICATE_FOUND			  HRES_ERROR(0x800F0202)
#define HRES_SPAPI_E_NO_DRIVER_SELECTED			  HRES_ERROR(0x800F0203)
#define HRES_SPAPI_E_KEY_DOES_NOT_EXIST			  HRES_ERROR(0x800F0204)
#define HRES_SPAPI_E_INVALID_DEVINST_NAME		  HRES_ERROR(0x800F0205)
#define HRES_SPAPI_E_INVALID_CLASS			  HRES_ERROR(0x800F0206)
#define HRES_SPAPI_E_DEVINST_ALREADY_EXISTS		  HRES_ERROR(0x800F0207)
#define HRES_SPAPI_E_DEVINFO_NOT_REGISTERED		  HRES_ERROR(0x800F0208)
#define HRES_SPAPI_E_INVALID_REG_PROPERTY		  HRES_ERROR(0x800F0209)
#define HRES_SPAPI_E_NO_INF				  HRES_ERROR(0x800F020A)
#define HRES_SPAPI_E_NO_SUCH_DEVINST			  HRES_ERROR(0x800F020B)
#define HRES_SPAPI_E_CANT_LOAD_CLASS_ICON		  HRES_ERROR(0x800F020C)
#define HRES_SPAPI_E_INVALID_CLASS_INSTALLER		  HRES_ERROR(0x800F020D)
#define HRES_SPAPI_E_DI_DO_DEFAULT			  HRES_ERROR(0x800F020E)
#define HRES_SPAPI_E_DI_NOFILECOPY			  HRES_ERROR(0x800F020F)
#define HRES_SPAPI_E_INVALID_HWPROFILE			  HRES_ERROR(0x800F0210)
#define HRES_SPAPI_E_NO_DEVICE_SELECTED			  HRES_ERROR(0x800F0211)
#define HRES_SPAPI_E_DEVINFO_LIST_LOCKED		  HRES_ERROR(0x800F0212)
#define HRES_SPAPI_E_DEVINFO_DATA_LOCKED		  HRES_ERROR(0x800F0213)
#define HRES_SPAPI_E_DI_BAD_PATH			  HRES_ERROR(0x800F0214)
#define HRES_SPAPI_E_NO_CLASSINSTALL_PARAMS		  HRES_ERROR(0x800F0215)
#define HRES_SPAPI_E_FILEQUEUE_LOCKED			  HRES_ERROR(0x800F0216)
#define HRES_SPAPI_E_BAD_SERVICE_INSTALLSECT		  HRES_ERROR(0x800F0217)
#define HRES_SPAPI_E_NO_CLASS_DRIVER_LIST		  HRES_ERROR(0x800F0218)
#define HRES_SPAPI_E_NO_ASSOCIATED_SERVICE		  HRES_ERROR(0x800F0219)
#define HRES_SPAPI_E_NO_DEFAULT_DEVICE_INTERFACE	  HRES_ERROR(0x800F021A)
#define HRES_SPAPI_E_DEVICE_INTERFACE_ACTIVE		  HRES_ERROR(0x800F021B)
#define HRES_SPAPI_E_DEVICE_INTERFACE_REMOVED		  HRES_ERROR(0x800F021C)
#define HRES_SPAPI_E_BAD_INTERFACE_INSTALLSECT		  HRES_ERROR(0x800F021D)
#define HRES_SPAPI_E_NO_SUCH_INTERFACE_CLASS		  HRES_ERROR(0x800F021E)
#define HRES_SPAPI_E_INVALID_REFERENCE_STRING		  HRES_ERROR(0x800F021F)
#define HRES_SPAPI_E_INVALID_MACHINENAME		  HRES_ERROR(0x800F0220)
#define HRES_SPAPI_E_REMOTE_COMM_FAILURE		  HRES_ERROR(0x800F0221)
#define HRES_SPAPI_E_MACHINE_UNAVAILABLE		  HRES_ERROR(0x800F0222)
#define HRES_SPAPI_E_NO_CONFIGMGR_SERVICES		  HRES_ERROR(0x800F0223)
#define HRES_SPAPI_E_INVALID_PROPPAGE_PROVIDER		  HRES_ERROR(0x800F0224)
#define HRES_SPAPI_E_NO_SUCH_DEVICE_INTERFACE		  HRES_ERROR(0x800F0225)
#define HRES_SPAPI_E_DI_POSTPROCESSING_REQUIRED		  HRES_ERROR(0x800F0226)
#define HRES_SPAPI_E_INVALID_COINSTALLER		  HRES_ERROR(0x800F0227)
#define HRES_SPAPI_E_NO_COMPAT_DRIVERS			  HRES_ERROR(0x800F0228)
#define HRES_SPAPI_E_NO_DEVICE_ICON			  HRES_ERROR(0x800F0229)
#define HRES_SPAPI_E_INVALID_INF_LOGCONFIG		  HRES_ERROR(0x800F022A)
#define HRES_SPAPI_E_DI_DONT_INSTALL			  HRES_ERROR(0x800F022B)
#define HRES_SPAPI_E_INVALID_FILTER_DRIVER		  HRES_ERROR(0x800F022C)
#define HRES_SPAPI_E_NON_WINDOWS_NT_DRIVER		  HRES_ERROR(0x800F022D)
#define HRES_SPAPI_E_NON_WINDOWS_DRIVER			  HRES_ERROR(0x800F022E)
#define HRES_SPAPI_E_NO_CATALOG_FOR_OEM_INF		  HRES_ERROR(0x800F022F)
#define HRES_SPAPI_E_DEVINSTALL_QUEUE_NONNATIVE		  HRES_ERROR(0x800F0230)
#define HRES_SPAPI_E_NOT_DISABLEABLE			  HRES_ERROR(0x800F0231)
#define HRES_SPAPI_E_CANT_REMOVE_DEVINST		  HRES_ERROR(0x800F0232)
#define HRES_SPAPI_E_INVALID_TARGET			  HRES_ERROR(0x800F0233)
#define HRES_SPAPI_E_DRIVER_NONNATIVE			  HRES_ERROR(0x800F0234)
#define HRES_SPAPI_E_IN_WOW64				  HRES_ERROR(0x800F0235)
#define HRES_SPAPI_E_SET_SYSTEM_RESTORE_POINT		  HRES_ERROR(0x800F0236)
#define HRES_SPAPI_E_INCORRECTLY_COPIED_INF		  HRES_ERROR(0x800F0237)
#define HRES_SPAPI_E_SCE_DISABLED			  HRES_ERROR(0x800F0238)
#define HRES_SPAPI_E_UNKNOWN_EXCEPTION			  HRES_ERROR(0x800F0239)
#define HRES_SPAPI_E_PNP_REGISTRY_ERROR			  HRES_ERROR(0x800F023A)
#define HRES_SPAPI_E_REMOTE_REQUEST_UNSUPPORTED		  HRES_ERROR(0x800F023B)
#define HRES_SPAPI_E_NOT_AN_INSTALLED_OEM_INF		  HRES_ERROR(0x800F023C)
#define HRES_SPAPI_E_INF_IN_USE_BY_DEVICES		  HRES_ERROR(0x800F023D)
#define HRES_SPAPI_E_DI_FUNCTION_OBSOLETE		  HRES_ERROR(0x800F023E)
#define HRES_SPAPI_E_NO_AUTHENTICODE_CATALOG		  HRES_ERROR(0x800F023F)
#define HRES_SPAPI_E_AUTHENTICODE_DISALLOWED		  HRES_ERROR(0x800F0240)
#define HRES_SPAPI_E_AUTHENTICODE_TRUSTED_PUBLISHER	  HRES_ERROR(0x800F0241)
#define HRES_SPAPI_E_AUTHENTICODE_TRUST_NOT_ESTABLISHED   HRES_ERROR(0x800F0242)
#define HRES_SPAPI_E_AUTHENTICODE_PUBLISHER_NOT_TRUSTED   HRES_ERROR(0x800F0243)
#define HRES_SPAPI_E_SIGNATURE_OSATTRIBUTE_MISMATCH	  HRES_ERROR(0x800F0244)
#define HRES_SPAPI_E_ONLY_VALIDATE_VIA_AUTHENTICODE	  HRES_ERROR(0x800F0245)
#define HRES_SPAPI_E_DEVICE_INSTALLER_NOT_READY		  HRES_ERROR(0x800F0246)
#define HRES_SPAPI_E_DRIVER_STORE_ADD_FAILED		  HRES_ERROR(0x800F0247)
#define HRES_SPAPI_E_DEVICE_INSTALL_BLOCKED		  HRES_ERROR(0x800F0248)
#define HRES_SPAPI_E_DRIVER_INSTALL_BLOCKED		  HRES_ERROR(0x800F0249)
#define HRES_SPAPI_E_WRONG_INF_TYPE			  HRES_ERROR(0x800F024A)
#define HRES_SPAPI_E_FILE_HASH_NOT_IN_CATALOG		  HRES_ERROR(0x800F024B)
#define HRES_SPAPI_E_DRIVER_STORE_DELETE_FAILED		  HRES_ERROR(0x800F024C)
#define HRES_SPAPI_E_UNRECOVERABLE_STACK_OVERFLOW	  HRES_ERROR(0x800F0300)
#define HRES_SPAPI_E_ERROR_NOT_INSTALLED		  HRES_ERROR(0x800F1000)
#define HRES_SCARD_F_INTERNAL_ERROR			  HRES_ERROR(0x80100001)
#define HRES_SCARD_E_CANCELLED				  HRES_ERROR(0x80100002)
#define HRES_SCARD_E_INVALID_HANDLE			  HRES_ERROR(0x80100003)
#define HRES_SCARD_E_INVALID_PARAMETER			  HRES_ERROR(0x80100004)
#define HRES_SCARD_E_INVALID_TARGET			  HRES_ERROR(0x80100005)
#define HRES_SCARD_E_NO_MEMORY				  HRES_ERROR(0x80100006)
#define HRES_SCARD_F_WAITED_TOO_LONG			  HRES_ERROR(0x80100007)
#define HRES_SCARD_E_INSUFFICIENT_BUFFER		  HRES_ERROR(0x80100008)
#define HRES_SCARD_E_UNKNOWN_READER			  HRES_ERROR(0x80100009)
#define HRES_SCARD_E_TIMEOUT				  HRES_ERROR(0x8010000A)
#define HRES_SCARD_E_SHARING_VIOLATION			  HRES_ERROR(0x8010000B)
#define HRES_SCARD_E_NO_SMARTCARD			  HRES_ERROR(0x8010000C)
#define HRES_SCARD_E_UNKNOWN_CARD			  HRES_ERROR(0x8010000D)
#define HRES_SCARD_E_CANT_DISPOSE			  HRES_ERROR(0x8010000E)
#define HRES_SCARD_E_PROTO_MISMATCH			  HRES_ERROR(0x8010000F)
#define HRES_SCARD_E_NOT_READY				  HRES_ERROR(0x80100010)
#define HRES_SCARD_E_INVALID_VALUE			  HRES_ERROR(0x80100011)
#define HRES_SCARD_E_SYSTEM_CANCELLED			  HRES_ERROR(0x80100012)
#define HRES_SCARD_F_COMM_ERROR				  HRES_ERROR(0x80100013)
#define HRES_SCARD_F_UNKNOWN_ERROR			  HRES_ERROR(0x80100014)
#define HRES_SCARD_E_INVALID_ATR			  HRES_ERROR(0x80100015)
#define HRES_SCARD_E_NOT_TRANSACTED			  HRES_ERROR(0x80100016)
#define HRES_SCARD_E_READER_UNAVAILABLE			  HRES_ERROR(0x80100017)
#define HRES_SCARD_P_SHUTDOWN				  HRES_ERROR(0x80100018)
#define HRES_SCARD_E_PCI_TOO_SMALL			  HRES_ERROR(0x80100019)
#define HRES_SCARD_E_READER_UNSUPPORTED			  HRES_ERROR(0x8010001A)
#define HRES_SCARD_E_DUPLICATE_READER			  HRES_ERROR(0x8010001B)
#define HRES_SCARD_E_CARD_UNSUPPORTED			  HRES_ERROR(0x8010001C)
#define HRES_SCARD_E_NO_SERVICE				  HRES_ERROR(0x8010001D)
#define HRES_SCARD_E_SERVICE_STOPPED			  HRES_ERROR(0x8010001E)
#define HRES_SCARD_E_UNEXPECTED				  HRES_ERROR(0x8010001F)
#define HRES_SCARD_E_ICC_INSTALLATION			  HRES_ERROR(0x80100020)
#define HRES_SCARD_E_ICC_CREATEORDER			  HRES_ERROR(0x80100021)
#define HRES_SCARD_E_UNSUPPORTED_FEATURE		  HRES_ERROR(0x80100022)
#define HRES_SCARD_E_DIR_NOT_FOUND			  HRES_ERROR(0x80100023)
#define HRES_SCARD_E_FILE_NOT_FOUND			  HRES_ERROR(0x80100024)
#define HRES_SCARD_E_NO_DIR				  HRES_ERROR(0x80100025)
#define HRES_SCARD_E_NO_FILE				  HRES_ERROR(0x80100026)
#define HRES_SCARD_E_NO_ACCESS				  HRES_ERROR(0x80100027)
#define HRES_SCARD_E_WRITE_TOO_MANY			  HRES_ERROR(0x80100028)
#define HRES_SCARD_E_BAD_SEEK				  HRES_ERROR(0x80100029)
#define HRES_SCARD_E_INVALID_CHV			  HRES_ERROR(0x8010002A)
#define HRES_SCARD_E_UNKNOWN_RES_MNG			  HRES_ERROR(0x8010002B)
#define HRES_SCARD_E_NO_SUCH_CERTIFICATE		  HRES_ERROR(0x8010002C)
#define HRES_SCARD_E_CERTIFICATE_UNAVAILABLE		  HRES_ERROR(0x8010002D)
#define HRES_SCARD_E_NO_READERS_AVAILABLE		  HRES_ERROR(0x8010002E)
#define HRES_SCARD_E_COMM_DATA_LOST			  HRES_ERROR(0x8010002F)
#define HRES_SCARD_E_NO_KEY_CONTAINER			  HRES_ERROR(0x80100030)
#define HRES_SCARD_E_SERVER_TOO_BUSY			  HRES_ERROR(0x80100031)
#define HRES_SCARD_W_UNSUPPORTED_CARD			  HRES_ERROR(0x80100065)
#define HRES_SCARD_W_UNRESPONSIVE_CARD			  HRES_ERROR(0x80100066)
#define HRES_SCARD_W_UNPOWERED_CARD			  HRES_ERROR(0x80100067)
#define HRES_SCARD_W_RESET_CARD				  HRES_ERROR(0x80100068)
#define HRES_SCARD_W_REMOVED_CARD			  HRES_ERROR(0x80100069)
#define HRES_SCARD_W_SECURITY_VIOLATION			  HRES_ERROR(0x8010006A)
#define HRES_SCARD_W_WRONG_CHV				  HRES_ERROR(0x8010006B)
#define HRES_SCARD_W_CHV_BLOCKED			  HRES_ERROR(0x8010006C)
#define HRES_SCARD_W_EOF				  HRES_ERROR(0x8010006D)
#define HRES_SCARD_W_CANCELLED_BY_USER			  HRES_ERROR(0x8010006E)
#define HRES_SCARD_W_CARD_NOT_AUTHENTICATED		  HRES_ERROR(0x8010006F)
#define HRES_COMADMIN_E_OBJECTERRORS			  HRES_ERROR(0x80110401)
#define HRES_COMADMIN_E_OBJECTINVALID			  HRES_ERROR(0x80110402)
#define HRES_COMADMIN_E_KEYMISSING			  HRES_ERROR(0x80110403)
#define HRES_COMADMIN_E_ALREADYINSTALLED		  HRES_ERROR(0x80110404)
#define HRES_COMADMIN_E_APP_FILE_WRITEFAIL		  HRES_ERROR(0x80110407)
#define HRES_COMADMIN_E_APP_FILE_READFAIL		  HRES_ERROR(0x80110408)
#define HRES_COMADMIN_E_APP_FILE_VERSION		  HRES_ERROR(0x80110409)
#define HRES_COMADMIN_E_BADPATH				  HRES_ERROR(0x8011040A)
#define HRES_COMADMIN_E_APPLICATIONEXISTS		  HRES_ERROR(0x8011040B)
#define HRES_COMADMIN_E_ROLEEXISTS			  HRES_ERROR(0x8011040C)
#define HRES_COMADMIN_E_CANTCOPYFILE			  HRES_ERROR(0x8011040D)
#define HRES_COMADMIN_E_NOUSER				  HRES_ERROR(0x8011040F)
#define HRES_COMADMIN_E_INVALIDUSERIDS			  HRES_ERROR(0x80110410)
#define HRES_COMADMIN_E_NOREGISTRYCLSID			  HRES_ERROR(0x80110411)
#define HRES_COMADMIN_E_BADREGISTRYPROGID		  HRES_ERROR(0x80110412)
#define HRES_COMADMIN_E_AUTHENTICATIONLEVEL		  HRES_ERROR(0x80110413)
#define HRES_COMADMIN_E_USERPASSWDNOTVALID		  HRES_ERROR(0x80110414)
#define HRES_COMADMIN_E_CLSIDORIIDMISMATCH		  HRES_ERROR(0x80110418)
#define HRES_COMADMIN_E_REMOTEINTERFACE			  HRES_ERROR(0x80110419)
#define HRES_COMADMIN_E_DLLREGISTERSERVER		  HRES_ERROR(0x8011041A)
#define HRES_COMADMIN_E_NOSERVERSHARE			  HRES_ERROR(0x8011041B)
#define HRES_COMADMIN_E_DLLLOADFAILED			  HRES_ERROR(0x8011041D)
#define HRES_COMADMIN_E_BADREGISTRYLIBID		  HRES_ERROR(0x8011041E)
#define HRES_COMADMIN_E_APPDIRNOTFOUND			  HRES_ERROR(0x8011041F)
#define HRES_COMADMIN_E_REGISTRARFAILED			  HRES_ERROR(0x80110423)
#define HRES_COMADMIN_E_COMPFILE_DOESNOTEXIST		  HRES_ERROR(0x80110424)
#define HRES_COMADMIN_E_COMPFILE_LOADDLLFAIL		  HRES_ERROR(0x80110425)
#define HRES_COMADMIN_E_COMPFILE_GETCLASSOBJ		  HRES_ERROR(0x80110426)
#define HRES_COMADMIN_E_COMPFILE_CLASSNOTAVAIL		  HRES_ERROR(0x80110427)
#define HRES_COMADMIN_E_COMPFILE_BADTLB			  HRES_ERROR(0x80110428)
#define HRES_COMADMIN_E_COMPFILE_NOTINSTALLABLE		  HRES_ERROR(0x80110429)
#define HRES_COMADMIN_E_NOTCHANGEABLE			  HRES_ERROR(0x8011042A)
#define HRES_COMADMIN_E_NOTDELETEABLE			  HRES_ERROR(0x8011042B)
#define HRES_COMADMIN_E_SESSION				  HRES_ERROR(0x8011042C)
#define HRES_COMADMIN_E_COMP_MOVE_LOCKED		  HRES_ERROR(0x8011042D)
#define HRES_COMADMIN_E_COMP_MOVE_BAD_DEST		  HRES_ERROR(0x8011042E)
#define HRES_COMADMIN_E_REGISTERTLB			  HRES_ERROR(0x80110430)
#define HRES_COMADMIN_E_SYSTEMAPP			  HRES_ERROR(0x80110433)
#define HRES_COMADMIN_E_COMPFILE_NOREGISTRAR		  HRES_ERROR(0x80110434)
#define HRES_COMADMIN_E_COREQCOMPINSTALLED		  HRES_ERROR(0x80110435)
#define HRES_COMADMIN_E_SERVICENOTINSTALLED		  HRES_ERROR(0x80110436)
#define HRES_COMADMIN_E_PROPERTYSAVEFAILED		  HRES_ERROR(0x80110437)
#define HRES_COMADMIN_E_OBJECTEXISTS			  HRES_ERROR(0x80110438)
#define HRES_COMADMIN_E_COMPONENTEXISTS			  HRES_ERROR(0x80110439)
#define HRES_COMADMIN_E_REGFILE_CORRUPT			  HRES_ERROR(0x8011043B)
#define HRES_COMADMIN_E_PROPERTY_OVERFLOW		  HRES_ERROR(0x8011043C)
#define HRES_COMADMIN_E_NOTINREGISTRY			  HRES_ERROR(0x8011043E)
#define HRES_COMADMIN_E_OBJECTNOTPOOLABLE		  HRES_ERROR(0x8011043F)
#define HRES_COMADMIN_E_APPLID_MATCHES_CLSID		  HRES_ERROR(0x80110446)
#define HRES_COMADMIN_E_ROLE_DOES_NOT_EXIST		  HRES_ERROR(0x80110447)
#define HRES_COMADMIN_E_START_APP_NEEDS_COMPONENTS	  HRES_ERROR(0x80110448)
#define HRES_COMADMIN_E_REQUIRES_DIFFERENT_PLATFORM	  HRES_ERROR(0x80110449)
#define HRES_COMADMIN_E_CAN_NOT_EXPORT_APP_PROXY	  HRES_ERROR(0x8011044A)
#define HRES_COMADMIN_E_CAN_NOT_START_APP		  HRES_ERROR(0x8011044B)
#define HRES_COMADMIN_E_CAN_NOT_EXPORT_SYS_APP		  HRES_ERROR(0x8011044C)
#define HRES_COMADMIN_E_CANT_SUBSCRIBE_TO_COMPONENT	  HRES_ERROR(0x8011044D)
#define HRES_COMADMIN_E_EVENTCLASS_CANT_BE_SUBSCRIBER	  HRES_ERROR(0x8011044E)
#define HRES_COMADMIN_E_LIB_APP_PROXY_INCOMPATIBLE	  HRES_ERROR(0x8011044F)
#define HRES_COMADMIN_E_BASE_PARTITION_ONLY		  HRES_ERROR(0x80110450)
#define HRES_COMADMIN_E_START_APP_DISABLED		  HRES_ERROR(0x80110451)
#define HRES_COMADMIN_E_CAT_DUPLICATE_PARTITION_NAME	  HRES_ERROR(0x80110457)
#define HRES_COMADMIN_E_CAT_INVALID_PARTITION_NAME	  HRES_ERROR(0x80110458)
#define HRES_COMADMIN_E_CAT_PARTITION_IN_USE		  HRES_ERROR(0x80110459)
#define HRES_COMADMIN_E_FILE_PARTITION_DUPLICATE_FILES	  HRES_ERROR(0x8011045A)
#define HRES_COMADMIN_E_CAT_IMPORTED_COMPONENTS_NOT_ALLOWED HRES_ERROR(0x8011045B)
#define HRES_COMADMIN_E_AMBIGUOUS_APPLICATION_NAME	  HRES_ERROR(0x8011045C)
#define HRES_COMADMIN_E_AMBIGUOUS_PARTITION_NAME	  HRES_ERROR(0x8011045D)
#define HRES_COMADMIN_E_REGDB_NOTINITIALIZED		  HRES_ERROR(0x80110472)
#define HRES_COMADMIN_E_REGDB_NOTOPEN			  HRES_ERROR(0x80110473)
#define HRES_COMADMIN_E_REGDB_SYSTEMERR			  HRES_ERROR(0x80110474)
#define HRES_COMADMIN_E_REGDB_ALREADYRUNNING		  HRES_ERROR(0x80110475)
#define HRES_COMADMIN_E_MIG_VERSIONNOTSUPPORTED		  HRES_ERROR(0x80110480)
#define HRES_COMADMIN_E_MIG_SCHEMANOTFOUND		  HRES_ERROR(0x80110481)
#define HRES_COMADMIN_E_CAT_BITNESSMISMATCH		  HRES_ERROR(0x80110482)
#define HRES_COMADMIN_E_CAT_UNACCEPTABLEBITNESS		  HRES_ERROR(0x80110483)
#define HRES_COMADMIN_E_CAT_WRONGAPPBITNESS		  HRES_ERROR(0x80110484)
#define HRES_COMADMIN_E_CAT_PAUSE_RESUME_NOT_SUPPORTED	  HRES_ERROR(0x80110485)
#define HRES_COMADMIN_E_CAT_SERVERFAULT			  HRES_ERROR(0x80110486)
#define HRES_COMQC_E_APPLICATION_NOT_QUEUED		  HRES_ERROR(0x80110600)
#define HRES_COMQC_E_NO_QUEUEABLE_INTERFACES		  HRES_ERROR(0x80110601)
#define HRES_COMQC_E_QUEUING_SERVICE_NOT_AVAILABLE	  HRES_ERROR(0x80110602)
#define HRES_COMQC_E_NO_IPERSISTSTREAM			  HRES_ERROR(0x80110603)
#define HRES_COMQC_E_BAD_MESSAGE			  HRES_ERROR(0x80110604)
#define HRES_COMQC_E_UNAUTHENTICATED			  HRES_ERROR(0x80110605)
#define HRES_COMQC_E_UNTRUSTED_ENQUEUER			  HRES_ERROR(0x80110606)
#define HRES_MSDTC_E_DUPLICATE_RESOURCE			  HRES_ERROR(0x80110701)
#define HRES_COMADMIN_E_OBJECT_PARENT_MISSING		  HRES_ERROR(0x80110808)
#define HRES_COMADMIN_E_OBJECT_DOES_NOT_EXIST		  HRES_ERROR(0x80110809)
#define HRES_COMADMIN_E_APP_NOT_RUNNING			  HRES_ERROR(0x8011080A)
#define HRES_COMADMIN_E_INVALID_PARTITION		  HRES_ERROR(0x8011080B)
#define HRES_COMADMIN_E_SVCAPP_NOT_POOLABLE_OR_RECYCLABLE HRES_ERROR(0x8011080D)
#define HRES_COMADMIN_E_USER_IN_SET			  HRES_ERROR(0x8011080E)
#define HRES_COMADMIN_E_CANTRECYCLELIBRARYAPPS		  HRES_ERROR(0x8011080F)
#define HRES_COMADMIN_E_CANTRECYCLESERVICEAPPS		  HRES_ERROR(0x80110811)
#define HRES_COMADMIN_E_PROCESSALREADYRECYCLED		  HRES_ERROR(0x80110812)
#define HRES_COMADMIN_E_PAUSEDPROCESSMAYNOTBERECYCLED	  HRES_ERROR(0x80110813)
#define HRES_COMADMIN_E_CANTMAKEINPROCSERVICE		  HRES_ERROR(0x80110814)
#define HRES_COMADMIN_E_PROGIDINUSEBYCLSID		  HRES_ERROR(0x80110815)
#define HRES_COMADMIN_E_DEFAULT_PARTITION_NOT_IN_SET	  HRES_ERROR(0x80110816)
#define HRES_COMADMIN_E_RECYCLEDPROCESSMAYNOTBEPAUSED	  HRES_ERROR(0x80110817)
#define HRES_COMADMIN_E_PARTITION_ACCESSDENIED		  HRES_ERROR(0x80110818)
#define HRES_COMADMIN_E_PARTITION_MSI_ONLY		  HRES_ERROR(0x80110819)
#define HRES_COMADMIN_E_LEGACYCOMPS_NOT_ALLOWED_IN_1_0_FORMAT HRES_ERROR(0x8011081A)
#define HRES_COMADMIN_E_LEGACYCOMPS_NOT_ALLOWED_IN_NONBASE_PARTITIONS HRES_ERROR(0x8011081B)
#define HRES_COMADMIN_E_COMP_MOVE_SOURCE		  HRES_ERROR(0x8011081C)
#define HRES_COMADMIN_E_COMP_MOVE_DEST			  HRES_ERROR(0x8011081D)
#define HRES_COMADMIN_E_COMP_MOVE_PRIVATE		  HRES_ERROR(0x8011081E)
#define HRES_COMADMIN_E_BASEPARTITION_REQUIRED_IN_SET	  HRES_ERROR(0x8011081F)
#define HRES_COMADMIN_E_CANNOT_ALIAS_EVENTCLASS		  HRES_ERROR(0x80110820)
#define HRES_COMADMIN_E_PRIVATE_ACCESSDENIED		  HRES_ERROR(0x80110821)
#define HRES_COMADMIN_E_SAFERINVALID			  HRES_ERROR(0x80110822)
#define HRES_COMADMIN_E_REGISTRY_ACCESSDENIED		  HRES_ERROR(0x80110823)
#define HRES_COMADMIN_E_PARTITIONS_DISABLED		  HRES_ERROR(0x80110824)
#define HRES_ERROR_FLT_NO_HANDLER_DEFINED		  HRES_ERROR(0x801F0001)
#define HRES_ERROR_FLT_CONTEXT_ALREADY_DEFINED		  HRES_ERROR(0x801F0002)
#define HRES_ERROR_FLT_INVALID_ASYNCHRONOUS_REQUEST	  HRES_ERROR(0x801F0003)
#define HRES_ERROR_FLT_DISALLOW_FAST_IO			  HRES_ERROR(0x801F0004)
#define HRES_ERROR_FLT_INVALID_NAME_REQUEST		  HRES_ERROR(0x801F0005)
#define HRES_ERROR_FLT_NOT_SAFE_TO_POST_OPERATION	  HRES_ERROR(0x801F0006)
#define HRES_ERROR_FLT_NOT_INITIALIZED			  HRES_ERROR(0x801F0007)
#define HRES_ERROR_FLT_FILTER_NOT_READY			  HRES_ERROR(0x801F0008)
#define HRES_ERROR_FLT_POST_OPERATION_CLEANUP		  HRES_ERROR(0x801F0009)
#define HRES_ERROR_FLT_INTERNAL_ERROR			  HRES_ERROR(0x801F000A)
#define HRES_ERROR_FLT_DELETING_OBJECT			  HRES_ERROR(0x801F000B)
#define HRES_ERROR_FLT_MUST_BE_NONPAGED_POOL		  HRES_ERROR(0x801F000C)
#define HRES_ERROR_FLT_DUPLICATE_ENTRY			  HRES_ERROR(0x801F000D)
#define HRES_ERROR_FLT_CBDQ_DISABLED			  HRES_ERROR(0x801F000E)
#define HRES_ERROR_FLT_DO_NOT_ATTACH			  HRES_ERROR(0x801F000F)
#define HRES_ERROR_FLT_DO_NOT_DETACH			  HRES_ERROR(0x801F0010)
#define HRES_ERROR_FLT_INSTANCE_ALTITUDE_COLLISION	  HRES_ERROR(0x801F0011)
#define HRES_ERROR_FLT_INSTANCE_NAME_COLLISION		  HRES_ERROR(0x801F0012)
#define HRES_ERROR_FLT_FILTER_NOT_FOUND			  HRES_ERROR(0x801F0013)
#define HRES_ERROR_FLT_VOLUME_NOT_FOUND			  HRES_ERROR(0x801F0014)
#define HRES_ERROR_FLT_INSTANCE_NOT_FOUND		  HRES_ERROR(0x801F0015)
#define HRES_ERROR_FLT_CONTEXT_ALLOCATION_NOT_FOUND	  HRES_ERROR(0x801F0016)
#define HRES_ERROR_FLT_INVALID_CONTEXT_REGISTRATION	  HRES_ERROR(0x801F0017)
#define HRES_ERROR_FLT_NAME_CACHE_MISS			  HRES_ERROR(0x801F0018)
#define HRES_ERROR_FLT_NO_DEVICE_OBJECT			  HRES_ERROR(0x801F0019)
#define HRES_ERROR_FLT_VOLUME_ALREADY_MOUNTED		  HRES_ERROR(0x801F001A)
#define HRES_ERROR_FLT_ALREADY_ENLISTED			  HRES_ERROR(0x801F001B)
#define HRES_ERROR_FLT_CONTEXT_ALREADY_LINKED		  HRES_ERROR(0x801F001C)
#define HRES_ERROR_FLT_NO_WAITER_FOR_REPLY		  HRES_ERROR(0x801F0020)
#define HRES_ERROR_HUNG_DISPLAY_DRIVER_THREAD		  HRES_ERROR(0x80260001)
#define HRES_ERROR_MONITOR_NO_DESCRIPTOR		  HRES_ERROR(0x80261001)
#define HRES_ERROR_MONITOR_UNKNOWN_DESCRIPTOR_FORMAT	  HRES_ERROR(0x80261002)
#define HRES_DWM_E_COMPOSITIONDISABLED			  HRES_ERROR(0x80263001)
#define HRES_DWM_E_REMOTING_NOT_SUPPORTED		  HRES_ERROR(0x80263002)
#define HRES_DWM_E_NO_REDIRECTION_SURFACE_AVAILABLE	  HRES_ERROR(0x80263003)
#define HRES_DWM_E_NOT_QUEUING_PRESENTS			  HRES_ERROR(0x80263004)
#define HRES_TPM_E_ERROR_MASK				  HRES_ERROR(0x80280000)
#define HRES_TPM_E_AUTHFAIL				  HRES_ERROR(0x80280001)
#define HRES_TPM_E_BADINDEX				  HRES_ERROR(0x80280002)
#define HRES_TPM_E_BAD_PARAMETER			  HRES_ERROR(0x80280003)
#define HRES_TPM_E_AUDITFAILURE				  HRES_ERROR(0x80280004)
#define HRES_TPM_E_CLEAR_DISABLED			  HRES_ERROR(0x80280005)
#define HRES_TPM_E_DEACTIVATED				  HRES_ERROR(0x80280006)
#define HRES_TPM_E_DISABLED				  HRES_ERROR(0x80280007)
#define HRES_TPM_E_DISABLED_CMD				  HRES_ERROR(0x80280008)
#define HRES_TPM_E_FAIL					  HRES_ERROR(0x80280009)
#define HRES_TPM_E_BAD_ORDINAL				  HRES_ERROR(0x8028000A)
#define HRES_TPM_E_INSTALL_DISABLED			  HRES_ERROR(0x8028000B)
#define HRES_TPM_E_INVALID_KEYHANDLE			  HRES_ERROR(0x8028000C)
#define HRES_TPM_E_KEYNOTFOUND				  HRES_ERROR(0x8028000D)
#define HRES_TPM_E_INAPPROPRIATE_ENC			  HRES_ERROR(0x8028000E)
#define HRES_TPM_E_MIGRATEFAIL				  HRES_ERROR(0x8028000F)
#define HRES_TPM_E_INVALID_PCR_INFO			  HRES_ERROR(0x80280010)
#define HRES_TPM_E_NOSPACE				  HRES_ERROR(0x80280011)
#define HRES_TPM_E_NOSRK				  HRES_ERROR(0x80280012)
#define HRES_TPM_E_NOTSEALED_BLOB			  HRES_ERROR(0x80280013)
#define HRES_TPM_E_OWNER_SET				  HRES_ERROR(0x80280014)
#define HRES_TPM_E_RESOURCES				  HRES_ERROR(0x80280015)
#define HRES_TPM_E_SHORTRANDOM				  HRES_ERROR(0x80280016)
#define HRES_TPM_E_SIZE					  HRES_ERROR(0x80280017)
#define HRES_TPM_E_WRONGPCRVAL				  HRES_ERROR(0x80280018)
#define HRES_TPM_E_BAD_PARAM_SIZE			  HRES_ERROR(0x80280019)
#define HRES_TPM_E_SHA_THREAD				  HRES_ERROR(0x8028001A)
#define HRES_TPM_E_SHA_ERROR				  HRES_ERROR(0x8028001B)
#define HRES_TPM_E_FAILEDSELFTEST			  HRES_ERROR(0x8028001C)
#define HRES_TPM_E_AUTH2FAIL				  HRES_ERROR(0x8028001D)
#define HRES_TPM_E_BADTAG				  HRES_ERROR(0x8028001E)
#define HRES_TPM_E_IOERROR				  HRES_ERROR(0x8028001F)
#define HRES_TPM_E_ENCRYPT_ERROR			  HRES_ERROR(0x80280020)
#define HRES_TPM_E_DECRYPT_ERROR			  HRES_ERROR(0x80280021)
#define HRES_TPM_E_INVALID_AUTHHANDLE			  HRES_ERROR(0x80280022)
#define HRES_TPM_E_NO_ENDORSEMENT			  HRES_ERROR(0x80280023)
#define HRES_TPM_E_INVALID_KEYUSAGE			  HRES_ERROR(0x80280024)
#define HRES_TPM_E_WRONG_ENTITYTYPE			  HRES_ERROR(0x80280025)
#define HRES_TPM_E_INVALID_POSTINIT			  HRES_ERROR(0x80280026)
#define HRES_TPM_E_INAPPROPRIATE_SIG			  HRES_ERROR(0x80280027)
#define HRES_TPM_E_BAD_KEY_PROPERTY			  HRES_ERROR(0x80280028)
#define HRES_TPM_E_BAD_MIGRATION			  HRES_ERROR(0x80280029)
#define HRES_TPM_E_BAD_SCHEME				  HRES_ERROR(0x8028002A)
#define HRES_TPM_E_BAD_DATASIZE				  HRES_ERROR(0x8028002B)
#define HRES_TPM_E_BAD_MODE				  HRES_ERROR(0x8028002C)
#define HRES_TPM_E_BAD_PRESENCE				  HRES_ERROR(0x8028002D)
#define HRES_TPM_E_BAD_VERSION				  HRES_ERROR(0x8028002E)
#define HRES_TPM_E_NO_WRAP_TRANSPORT			  HRES_ERROR(0x8028002F)
#define HRES_TPM_E_AUDITFAIL_UNSUCCESSFUL		  HRES_ERROR(0x80280030)
#define HRES_TPM_E_AUDITFAIL_SUCCESSFUL			  HRES_ERROR(0x80280031)
#define HRES_TPM_E_NOTRESETABLE				  HRES_ERROR(0x80280032)
#define HRES_TPM_E_NOTLOCAL				  HRES_ERROR(0x80280033)
#define HRES_TPM_E_BAD_TYPE				  HRES_ERROR(0x80280034)
#define HRES_TPM_E_INVALID_RESOURCE			  HRES_ERROR(0x80280035)
#define HRES_TPM_E_NOTFIPS				  HRES_ERROR(0x80280036)
#define HRES_TPM_E_INVALID_FAMILY			  HRES_ERROR(0x80280037)
#define HRES_TPM_E_NO_NV_PERMISSION			  HRES_ERROR(0x80280038)
#define HRES_TPM_E_REQUIRES_SIGN			  HRES_ERROR(0x80280039)
#define HRES_TPM_E_KEY_NOTSUPPORTED			  HRES_ERROR(0x8028003A)
#define HRES_TPM_E_AUTH_CONFLICT			  HRES_ERROR(0x8028003B)
#define HRES_TPM_E_AREA_LOCKED				  HRES_ERROR(0x8028003C)
#define HRES_TPM_E_BAD_LOCALITY				  HRES_ERROR(0x8028003D)
#define HRES_TPM_E_READ_ONLY				  HRES_ERROR(0x8028003E)
#define HRES_TPM_E_PER_NOWRITE				  HRES_ERROR(0x8028003F)
#define HRES_TPM_E_FAMILYCOUNT				  HRES_ERROR(0x80280040)
#define HRES_TPM_E_WRITE_LOCKED				  HRES_ERROR(0x80280041)
#define HRES_TPM_E_BAD_ATTRIBUTES			  HRES_ERROR(0x80280042)
#define HRES_TPM_E_INVALID_STRUCTURE			  HRES_ERROR(0x80280043)
#define HRES_TPM_E_KEY_OWNER_CONTROL			  HRES_ERROR(0x80280044)
#define HRES_TPM_E_BAD_COUNTER				  HRES_ERROR(0x80280045)
#define HRES_TPM_E_NOT_FULLWRITE			  HRES_ERROR(0x80280046)
#define HRES_TPM_E_CONTEXT_GAP				  HRES_ERROR(0x80280047)
#define HRES_TPM_E_MAXNVWRITES				  HRES_ERROR(0x80280048)
#define HRES_TPM_E_NOOPERATOR				  HRES_ERROR(0x80280049)
#define HRES_TPM_E_RESOURCEMISSING			  HRES_ERROR(0x8028004A)
#define HRES_TPM_E_DELEGATE_LOCK			  HRES_ERROR(0x8028004B)
#define HRES_TPM_E_DELEGATE_FAMILY			  HRES_ERROR(0x8028004C)
#define HRES_TPM_E_DELEGATE_ADMIN			  HRES_ERROR(0x8028004D)
#define HRES_TPM_E_TRANSPORT_NOTEXCLUSIVE		  HRES_ERROR(0x8028004E)
#define HRES_TPM_E_OWNER_CONTROL			  HRES_ERROR(0x8028004F)
#define HRES_TPM_E_DAA_RESOURCES			  HRES_ERROR(0x80280050)
#define HRES_TPM_E_DAA_INPUT_DATA0			  HRES_ERROR(0x80280051)
#define HRES_TPM_E_DAA_INPUT_DATA1			  HRES_ERROR(0x80280052)
#define HRES_TPM_E_DAA_ISSUER_SETTINGS			  HRES_ERROR(0x80280053)
#define HRES_TPM_E_DAA_TPM_SETTINGS			  HRES_ERROR(0x80280054)
#define HRES_TPM_E_DAA_STAGE				  HRES_ERROR(0x80280055)
#define HRES_TPM_E_DAA_ISSUER_VALIDITY			  HRES_ERROR(0x80280056)
#define HRES_TPM_E_DAA_WRONG_W				  HRES_ERROR(0x80280057)
#define HRES_TPM_E_BAD_HANDLE				  HRES_ERROR(0x80280058)
#define HRES_TPM_E_BAD_DELEGATE				  HRES_ERROR(0x80280059)
#define HRES_TPM_E_BADCONTEXT				  HRES_ERROR(0x8028005A)
#define HRES_TPM_E_TOOMANYCONTEXTS			  HRES_ERROR(0x8028005B)
#define HRES_TPM_E_MA_TICKET_SIGNATURE			  HRES_ERROR(0x8028005C)
#define HRES_TPM_E_MA_DESTINATION			  HRES_ERROR(0x8028005D)
#define HRES_TPM_E_MA_SOURCE				  HRES_ERROR(0x8028005E)
#define HRES_TPM_E_MA_AUTHORITY				  HRES_ERROR(0x8028005F)
#define HRES_TPM_E_PERMANENTEK				  HRES_ERROR(0x80280061)
#define HRES_TPM_E_BAD_SIGNATURE			  HRES_ERROR(0x80280062)
#define HRES_TPM_E_NOCONTEXTSPACE			  HRES_ERROR(0x80280063)
#define HRES_TPM_E_COMMAND_BLOCKED			  HRES_ERROR(0x80280400)
#define HRES_TPM_E_INVALID_HANDLE			  HRES_ERROR(0x80280401)
#define HRES_TPM_E_DUPLICATE_VHANDLE			  HRES_ERROR(0x80280402)
#define HRES_TPM_E_EMBEDDED_COMMAND_BLOCKED		  HRES_ERROR(0x80280403)
#define HRES_TPM_E_EMBEDDED_COMMAND_UNSUPPORTED		  HRES_ERROR(0x80280404)
#define HRES_TPM_E_RETRY				  HRES_ERROR(0x80280800)
#define HRES_TPM_E_NEEDS_SELFTEST			  HRES_ERROR(0x80280801)
#define HRES_TPM_E_DOING_SELFTEST			  HRES_ERROR(0x80280802)
#define HRES_TPM_E_DEFEND_LOCK_RUNNING			  HRES_ERROR(0x80280803)
#define HRES_TBS_E_INTERNAL_ERROR			  HRES_ERROR(0x80284001)
#define HRES_TBS_E_BAD_PARAMETER			  HRES_ERROR(0x80284002)
#define HRES_TBS_E_INVALID_OUTPUT_POINTER		  HRES_ERROR(0x80284003)
#define HRES_TBS_E_INVALID_CONTEXT			  HRES_ERROR(0x80284004)
#define HRES_TBS_E_INSUFFICIENT_BUFFER			  HRES_ERROR(0x80284005)
#define HRES_TBS_E_IOERROR				  HRES_ERROR(0x80284006)
#define HRES_TBS_E_INVALID_CONTEXT_PARAM		  HRES_ERROR(0x80284007)
#define HRES_TBS_E_SERVICE_NOT_RUNNING			  HRES_ERROR(0x80284008)
#define HRES_TBS_E_TOO_MANY_TBS_CONTEXTS		  HRES_ERROR(0x80284009)
#define HRES_TBS_E_TOO_MANY_RESOURCES			  HRES_ERROR(0x8028400A)
#define HRES_TBS_E_SERVICE_START_PENDING		  HRES_ERROR(0x8028400B)
#define HRES_TBS_E_PPI_NOT_SUPPORTED			  HRES_ERROR(0x8028400C)
#define HRES_TBS_E_COMMAND_CANCELED			  HRES_ERROR(0x8028400D)
#define HRES_TBS_E_BUFFER_TOO_LARGE			  HRES_ERROR(0x8028400E)
#define HRES_TPMAPI_E_INVALID_STATE			  HRES_ERROR(0x80290100)
#define HRES_TPMAPI_E_NOT_ENOUGH_DATA			  HRES_ERROR(0x80290101)
#define HRES_TPMAPI_E_TOO_MUCH_DATA			  HRES_ERROR(0x80290102)
#define HRES_TPMAPI_E_INVALID_OUTPUT_POINTER		  HRES_ERROR(0x80290103)
#define HRES_TPMAPI_E_INVALID_PARAMETER			  HRES_ERROR(0x80290104)
#define HRES_TPMAPI_E_OUT_OF_MEMORY			  HRES_ERROR(0x80290105)
#define HRES_TPMAPI_E_BUFFER_TOO_SMALL			  HRES_ERROR(0x80290106)
#define HRES_TPMAPI_E_INTERNAL_ERROR			  HRES_ERROR(0x80290107)
#define HRES_TPMAPI_E_ACCESS_DENIED			  HRES_ERROR(0x80290108)
#define HRES_TPMAPI_E_AUTHORIZATION_FAILED		  HRES_ERROR(0x80290109)
#define HRES_TPMAPI_E_INVALID_CONTEXT_HANDLE		  HRES_ERROR(0x8029010A)
#define HRES_TPMAPI_E_TBS_COMMUNICATION_ERROR		  HRES_ERROR(0x8029010B)
#define HRES_TPMAPI_E_TPM_COMMAND_ERROR			  HRES_ERROR(0x8029010C)
#define HRES_TPMAPI_E_MESSAGE_TOO_LARGE			  HRES_ERROR(0x8029010D)
#define HRES_TPMAPI_E_INVALID_ENCODING			  HRES_ERROR(0x8029010E)
#define HRES_TPMAPI_E_INVALID_KEY_SIZE			  HRES_ERROR(0x8029010F)
#define HRES_TPMAPI_E_ENCRYPTION_FAILED			  HRES_ERROR(0x80290110)
#define HRES_TPMAPI_E_INVALID_KEY_PARAMS		  HRES_ERROR(0x80290111)
#define HRES_TPMAPI_E_INVALID_MIGRATION_AUTHORIZATION_BLOB HRES_ERROR(0x80290112)
#define HRES_TPMAPI_E_INVALID_PCR_INDEX			  HRES_ERROR(0x80290113)
#define HRES_TPMAPI_E_INVALID_DELEGATE_BLOB		  HRES_ERROR(0x80290114)
#define HRES_TPMAPI_E_INVALID_CONTEXT_PARAMS		  HRES_ERROR(0x80290115)
#define HRES_TPMAPI_E_INVALID_KEY_BLOB			  HRES_ERROR(0x80290116)
#define HRES_TPMAPI_E_INVALID_PCR_DATA			  HRES_ERROR(0x80290117)
#define HRES_TPMAPI_E_INVALID_OWNER_AUTH		  HRES_ERROR(0x80290118)
#define HRES_TBSIMP_E_BUFFER_TOO_SMALL			  HRES_ERROR(0x80290200)
#define HRES_TBSIMP_E_CLEANUP_FAILED			  HRES_ERROR(0x80290201)
#define HRES_TBSIMP_E_INVALID_CONTEXT_HANDLE		  HRES_ERROR(0x80290202)
#define HRES_TBSIMP_E_INVALID_CONTEXT_PARAM		  HRES_ERROR(0x80290203)
#define HRES_TBSIMP_E_TPM_ERROR				  HRES_ERROR(0x80290204)
#define HRES_TBSIMP_E_HASH_BAD_KEY			  HRES_ERROR(0x80290205)
#define HRES_TBSIMP_E_DUPLICATE_VHANDLE			  HRES_ERROR(0x80290206)
#define HRES_TBSIMP_E_INVALID_OUTPUT_POINTER		  HRES_ERROR(0x80290207)
#define HRES_TBSIMP_E_INVALID_PARAMETER			  HRES_ERROR(0x80290208)
#define HRES_TBSIMP_E_RPC_INIT_FAILED			  HRES_ERROR(0x80290209)
#define HRES_TBSIMP_E_SCHEDULER_NOT_RUNNING		  HRES_ERROR(0x8029020A)
#define HRES_TBSIMP_E_COMMAND_CANCELED			  HRES_ERROR(0x8029020B)
#define HRES_TBSIMP_E_OUT_OF_MEMORY			  HRES_ERROR(0x8029020C)
#define HRES_TBSIMP_E_LIST_NO_MORE_ITEMS		  HRES_ERROR(0x8029020D)
#define HRES_TBSIMP_E_LIST_NOT_FOUND			  HRES_ERROR(0x8029020E)
#define HRES_TBSIMP_E_NOT_ENOUGH_SPACE			  HRES_ERROR(0x8029020F)
#define HRES_TBSIMP_E_NOT_ENOUGH_TPM_CONTEXTS		  HRES_ERROR(0x80290210)
#define HRES_TBSIMP_E_COMMAND_FAILED			  HRES_ERROR(0x80290211)
#define HRES_TBSIMP_E_UNKNOWN_ORDINAL			  HRES_ERROR(0x80290212)
#define HRES_TBSIMP_E_RESOURCE_EXPIRED			  HRES_ERROR(0x80290213)
#define HRES_TBSIMP_E_INVALID_RESOURCE			  HRES_ERROR(0x80290214)
#define HRES_TBSIMP_E_NOTHING_TO_UNLOAD			  HRES_ERROR(0x80290215)
#define HRES_TBSIMP_E_HASH_TABLE_FULL			  HRES_ERROR(0x80290216)
#define HRES_TBSIMP_E_TOO_MANY_TBS_CONTEXTS		  HRES_ERROR(0x80290217)
#define HRES_TBSIMP_E_TOO_MANY_RESOURCES		  HRES_ERROR(0x80290218)
#define HRES_TBSIMP_E_PPI_NOT_SUPPORTED			  HRES_ERROR(0x80290219)
#define HRES_TBSIMP_E_TPM_INCOMPATIBLE			  HRES_ERROR(0x8029021A)
#define HRES_TPM_E_PPI_ACPI_FAILURE			  HRES_ERROR(0x80290300)
#define HRES_TPM_E_PPI_USER_ABORT			  HRES_ERROR(0x80290301)
#define HRES_TPM_E_PPI_BIOS_FAILURE			  HRES_ERROR(0x80290302)
#define HRES_TPM_E_PPI_NOT_SUPPORTED			  HRES_ERROR(0x80290303)
#define HRES_PLA_E_DCS_NOT_FOUND			  HRES_ERROR(0x80300002)
#define HRES_PLA_E_TOO_MANY_FOLDERS			  HRES_ERROR(0x80300045)
#define HRES_PLA_E_NO_MIN_DISK				  HRES_ERROR(0x80300070)
#define HRES_PLA_E_DCS_IN_USE				  HRES_ERROR(0x803000AA)
#define HRES_PLA_E_DCS_ALREADY_EXISTS			  HRES_ERROR(0x803000B7)
#define HRES_PLA_E_PROPERTY_CONFLICT			  HRES_ERROR(0x80300101)
#define HRES_PLA_E_DCS_SINGLETON_REQUIRED		  HRES_ERROR(0x80300102)
#define HRES_PLA_E_CREDENTIALS_REQUIRED			  HRES_ERROR(0x80300103)
#define HRES_PLA_E_DCS_NOT_RUNNING			  HRES_ERROR(0x80300104)
#define HRES_PLA_E_CONFLICT_INCL_EXCL_API		  HRES_ERROR(0x80300105)
#define HRES_PLA_E_NETWORK_EXE_NOT_VALID		  HRES_ERROR(0x80300106)
#define HRES_PLA_E_EXE_ALREADY_CONFIGURED		  HRES_ERROR(0x80300107)
#define HRES_PLA_E_EXE_PATH_NOT_VALID			  HRES_ERROR(0x80300108)
#define HRES_PLA_E_DC_ALREADY_EXISTS			  HRES_ERROR(0x80300109)
#define HRES_PLA_E_DCS_START_WAIT_TIMEOUT		  HRES_ERROR(0x8030010A)
#define HRES_PLA_E_DC_START_WAIT_TIMEOUT		  HRES_ERROR(0x8030010B)
#define HRES_PLA_E_REPORT_WAIT_TIMEOUT			  HRES_ERROR(0x8030010C)
#define HRES_PLA_E_NO_DUPLICATES			  HRES_ERROR(0x8030010D)
#define HRES_PLA_E_EXE_FULL_PATH_REQUIRED		  HRES_ERROR(0x8030010E)
#define HRES_PLA_E_INVALID_SESSION_NAME			  HRES_ERROR(0x8030010F)
#define HRES_PLA_E_PLA_CHANNEL_NOT_ENABLED		  HRES_ERROR(0x80300110)
#define HRES_PLA_E_TASKSCHED_CHANNEL_NOT_ENABLED	  HRES_ERROR(0x80300111)
#define HRES_FVE_E_LOCKED_VOLUME			  HRES_ERROR(0x80310000)
#define HRES_FVE_E_NOT_ENCRYPTED			  HRES_ERROR(0x80310001)
#define HRES_FVE_E_NO_TPM_BIOS				  HRES_ERROR(0x80310002)
#define HRES_FVE_E_NO_MBR_METRIC			  HRES_ERROR(0x80310003)
#define HRES_FVE_E_NO_BOOTSECTOR_METRIC			  HRES_ERROR(0x80310004)
#define HRES_FVE_E_NO_BOOTMGR_METRIC			  HRES_ERROR(0x80310005)
#define HRES_FVE_E_WRONG_BOOTMGR			  HRES_ERROR(0x80310006)
#define HRES_FVE_E_SECURE_KEY_REQUIRED			  HRES_ERROR(0x80310007)
#define HRES_FVE_E_NOT_ACTIVATED			  HRES_ERROR(0x80310008)
#define HRES_FVE_E_ACTION_NOT_ALLOWED			  HRES_ERROR(0x80310009)
#define HRES_FVE_E_AD_SCHEMA_NOT_INSTALLED		  HRES_ERROR(0x8031000A)
#define HRES_FVE_E_AD_INVALID_DATATYPE			  HRES_ERROR(0x8031000B)
#define HRES_FVE_E_AD_INVALID_DATASIZE			  HRES_ERROR(0x8031000C)
#define HRES_FVE_E_AD_NO_VALUES				  HRES_ERROR(0x8031000D)
#define HRES_FVE_E_AD_ATTR_NOT_SET			  HRES_ERROR(0x8031000E)
#define HRES_FVE_E_AD_GUID_NOT_FOUND			  HRES_ERROR(0x8031000F)
#define HRES_FVE_E_BAD_INFORMATION			  HRES_ERROR(0x80310010)
#define HRES_FVE_E_TOO_SMALL				  HRES_ERROR(0x80310011)
#define HRES_FVE_E_SYSTEM_VOLUME			  HRES_ERROR(0x80310012)
#define HRES_FVE_E_FAILED_WRONG_FS			  HRES_ERROR(0x80310013)
#define HRES_FVE_E_FAILED_BAD_FS			  HRES_ERROR(0x80310014)
#define HRES_FVE_E_NOT_SUPPORTED			  HRES_ERROR(0x80310015)
#define HRES_FVE_E_BAD_DATA				  HRES_ERROR(0x80310016)
#define HRES_FVE_E_VOLUME_NOT_BOUND			  HRES_ERROR(0x80310017)
#define HRES_FVE_E_TPM_NOT_OWNED			  HRES_ERROR(0x80310018)
#define HRES_FVE_E_NOT_DATA_VOLUME			  HRES_ERROR(0x80310019)
#define HRES_FVE_E_AD_INSUFFICIENT_BUFFER		  HRES_ERROR(0x8031001A)
#define HRES_FVE_E_CONV_READ				  HRES_ERROR(0x8031001B)
#define HRES_FVE_E_CONV_WRITE				  HRES_ERROR(0x8031001C)
#define HRES_FVE_E_KEY_REQUIRED				  HRES_ERROR(0x8031001D)
#define HRES_FVE_E_CLUSTERING_NOT_SUPPORTED		  HRES_ERROR(0x8031001E)
#define HRES_FVE_E_VOLUME_BOUND_ALREADY			  HRES_ERROR(0x8031001F)
#define HRES_FVE_E_OS_NOT_PROTECTED			  HRES_ERROR(0x80310020)
#define HRES_FVE_E_PROTECTION_DISABLED			  HRES_ERROR(0x80310021)
#define HRES_FVE_E_RECOVERY_KEY_REQUIRED		  HRES_ERROR(0x80310022)
#define HRES_FVE_E_FOREIGN_VOLUME			  HRES_ERROR(0x80310023)
#define HRES_FVE_E_OVERLAPPED_UPDATE			  HRES_ERROR(0x80310024)
#define HRES_FVE_E_TPM_SRK_AUTH_NOT_ZERO		  HRES_ERROR(0x80310025)
#define HRES_FVE_E_FAILED_SECTOR_SIZE			  HRES_ERROR(0x80310026)
#define HRES_FVE_E_FAILED_AUTHENTICATION		  HRES_ERROR(0x80310027)
#define HRES_FVE_E_NOT_OS_VOLUME			  HRES_ERROR(0x80310028)
#define HRES_FVE_E_AUTOUNLOCK_ENABLED			  HRES_ERROR(0x80310029)
#define HRES_FVE_E_WRONG_BOOTSECTOR			  HRES_ERROR(0x8031002A)
#define HRES_FVE_E_WRONG_SYSTEM_FS			  HRES_ERROR(0x8031002B)
#define HRES_FVE_E_POLICY_PASSWORD_REQUIRED		  HRES_ERROR(0x8031002C)
#define HRES_FVE_E_CANNOT_SET_FVEK_ENCRYPTED		  HRES_ERROR(0x8031002D)
#define HRES_FVE_E_CANNOT_ENCRYPT_NO_KEY		  HRES_ERROR(0x8031002E)
#define HRES_FVE_E_BOOTABLE_CDDVD			  HRES_ERROR(0x80310030)
#define HRES_FVE_E_PROTECTOR_EXISTS			  HRES_ERROR(0x80310031)
#define HRES_FVE_E_RELATIVE_PATH			  HRES_ERROR(0x80310032)
#define HRES_FWP_E_CALLOUT_NOT_FOUND			  HRES_ERROR(0x80320001)
#define HRES_FWP_E_CONDITION_NOT_FOUND			  HRES_ERROR(0x80320002)
#define HRES_FWP_E_FILTER_NOT_FOUND			  HRES_ERROR(0x80320003)
#define HRES_FWP_E_LAYER_NOT_FOUND			  HRES_ERROR(0x80320004)
#define HRES_FWP_E_PROVIDER_NOT_FOUND			  HRES_ERROR(0x80320005)
#define HRES_FWP_E_PROVIDER_CONTEXT_NOT_FOUND		  HRES_ERROR(0x80320006)
#define HRES_FWP_E_SUBLAYER_NOT_FOUND			  HRES_ERROR(0x80320007)
#define HRES_FWP_E_NOT_FOUND				  HRES_ERROR(0x80320008)
#define HRES_FWP_E_ALREADY_EXISTS			  HRES_ERROR(0x80320009)
#define HRES_FWP_E_IN_USE				  HRES_ERROR(0x8032000A)
#define HRES_FWP_E_DYNAMIC_SESSION_IN_PROGRESS		  HRES_ERROR(0x8032000B)
#define HRES_FWP_E_WRONG_SESSION			  HRES_ERROR(0x8032000C)
#define HRES_FWP_E_NO_TXN_IN_PROGRESS			  HRES_ERROR(0x8032000D)
#define HRES_FWP_E_TXN_IN_PROGRESS			  HRES_ERROR(0x8032000E)
#define HRES_FWP_E_TXN_ABORTED				  HRES_ERROR(0x8032000F)
#define HRES_FWP_E_SESSION_ABORTED			  HRES_ERROR(0x80320010)
#define HRES_FWP_E_INCOMPATIBLE_TXN			  HRES_ERROR(0x80320011)
#define HRES_FWP_E_TIMEOUT				  HRES_ERROR(0x80320012)
#define HRES_FWP_E_NET_EVENTS_DISABLED			  HRES_ERROR(0x80320013)
#define HRES_FWP_E_INCOMPATIBLE_LAYER			  HRES_ERROR(0x80320014)
#define HRES_FWP_E_KM_CLIENTS_ONLY			  HRES_ERROR(0x80320015)
#define HRES_FWP_E_LIFETIME_MISMATCH			  HRES_ERROR(0x80320016)
#define HRES_FWP_E_BUILTIN_OBJECT			  HRES_ERROR(0x80320017)
#define HRES_FWP_E_TOO_MANY_BOOTTIME_FILTERS		  HRES_ERROR(0x80320018)
#define HRES_FWP_E_NOTIFICATION_DROPPED			  HRES_ERROR(0x80320019)
#define HRES_FWP_E_TRAFFIC_MISMATCH			  HRES_ERROR(0x8032001A)
#define HRES_FWP_E_INCOMPATIBLE_SA_STATE		  HRES_ERROR(0x8032001B)
#define HRES_FWP_E_NULL_POINTER				  HRES_ERROR(0x8032001C)
#define HRES_FWP_E_INVALID_ENUMERATOR			  HRES_ERROR(0x8032001D)
#define HRES_FWP_E_INVALID_FLAGS			  HRES_ERROR(0x8032001E)
#define HRES_FWP_E_INVALID_NET_MASK			  HRES_ERROR(0x8032001F)
#define HRES_FWP_E_INVALID_RANGE			  HRES_ERROR(0x80320020)
#define HRES_FWP_E_INVALID_INTERVAL			  HRES_ERROR(0x80320021)
#define HRES_FWP_E_ZERO_LENGTH_ARRAY			  HRES_ERROR(0x80320022)
#define HRES_FWP_E_NULL_DISPLAY_NAME			  HRES_ERROR(0x80320023)
#define HRES_FWP_E_INVALID_ACTION_TYPE			  HRES_ERROR(0x80320024)
#define HRES_FWP_E_INVALID_WEIGHT			  HRES_ERROR(0x80320025)
#define HRES_FWP_E_MATCH_TYPE_MISMATCH			  HRES_ERROR(0x80320026)
#define HRES_FWP_E_TYPE_MISMATCH			  HRES_ERROR(0x80320027)
#define HRES_FWP_E_OUT_OF_BOUNDS			  HRES_ERROR(0x80320028)
#define HRES_FWP_E_RESERVED				  HRES_ERROR(0x80320029)
#define HRES_FWP_E_DUPLICATE_CONDITION			  HRES_ERROR(0x8032002A)
#define HRES_FWP_E_DUPLICATE_KEYMOD			  HRES_ERROR(0x8032002B)
#define HRES_FWP_E_ACTION_INCOMPATIBLE_WITH_LAYER	  HRES_ERROR(0x8032002C)
#define HRES_FWP_E_ACTION_INCOMPATIBLE_WITH_SUBLAYER	  HRES_ERROR(0x8032002D)
#define HRES_FWP_E_CONTEXT_INCOMPATIBLE_WITH_LAYER	  HRES_ERROR(0x8032002E)
#define HRES_FWP_E_CONTEXT_INCOMPATIBLE_WITH_CALLOUT	  HRES_ERROR(0x8032002F)
#define HRES_FWP_E_INCOMPATIBLE_AUTH_METHOD		  HRES_ERROR(0x80320030)
#define HRES_FWP_E_INCOMPATIBLE_DH_GROUP		  HRES_ERROR(0x80320031)
#define HRES_FWP_E_EM_NOT_SUPPORTED			  HRES_ERROR(0x80320032)
#define HRES_FWP_E_NEVER_MATCH				  HRES_ERROR(0x80320033)
#define HRES_FWP_E_PROVIDER_CONTEXT_MISMATCH		  HRES_ERROR(0x80320034)
#define HRES_FWP_E_INVALID_PARAMETER			  HRES_ERROR(0x80320035)
#define HRES_FWP_E_TOO_MANY_SUBLAYERS			  HRES_ERROR(0x80320036)
#define HRES_FWP_E_CALLOUT_NOTIFICATION_FAILED		  HRES_ERROR(0x80320037)
#define HRES_FWP_E_INCOMPATIBLE_AUTH_CONFIG		  HRES_ERROR(0x80320038)
#define HRES_FWP_E_INCOMPATIBLE_CIPHER_CONFIG		  HRES_ERROR(0x80320039)
#define HRES_ERROR_NDIS_INTERFACE_CLOSING		  HRES_ERROR(0x80340002)
#define HRES_ERROR_NDIS_BAD_VERSION			  HRES_ERROR(0x80340004)
#define HRES_ERROR_NDIS_BAD_CHARACTERISTICS		  HRES_ERROR(0x80340005)
#define HRES_ERROR_NDIS_ADAPTER_NOT_FOUND		  HRES_ERROR(0x80340006)
#define HRES_ERROR_NDIS_OPEN_FAILED			  HRES_ERROR(0x80340007)
#define HRES_ERROR_NDIS_DEVICE_FAILED			  HRES_ERROR(0x80340008)
#define HRES_ERROR_NDIS_MULTICAST_FULL			  HRES_ERROR(0x80340009)
#define HRES_ERROR_NDIS_MULTICAST_EXISTS		  HRES_ERROR(0x8034000A)
#define HRES_ERROR_NDIS_MULTICAST_NOT_FOUND		  HRES_ERROR(0x8034000B)
#define HRES_ERROR_NDIS_REQUEST_ABORTED			  HRES_ERROR(0x8034000C)
#define HRES_ERROR_NDIS_RESET_IN_PROGRESS		  HRES_ERROR(0x8034000D)
#define HRES_ERROR_NDIS_INVALID_PACKET			  HRES_ERROR(0x8034000F)
#define HRES_ERROR_NDIS_INVALID_DEVICE_REQUEST		  HRES_ERROR(0x80340010)
#define HRES_ERROR_NDIS_ADAPTER_NOT_READY		  HRES_ERROR(0x80340011)
#define HRES_ERROR_NDIS_INVALID_LENGTH			  HRES_ERROR(0x80340014)
#define HRES_ERROR_NDIS_INVALID_DATA			  HRES_ERROR(0x80340015)
#define HRES_ERROR_NDIS_BUFFER_TOO_SHORT		  HRES_ERROR(0x80340016)
#define HRES_ERROR_NDIS_INVALID_OID			  HRES_ERROR(0x80340017)
#define HRES_ERROR_NDIS_ADAPTER_REMOVED			  HRES_ERROR(0x80340018)
#define HRES_ERROR_NDIS_UNSUPPORTED_MEDIA		  HRES_ERROR(0x80340019)
#define HRES_ERROR_NDIS_GROUP_ADDRESS_IN_USE		  HRES_ERROR(0x8034001A)
#define HRES_ERROR_NDIS_FILE_NOT_FOUND			  HRES_ERROR(0x8034001B)
#define HRES_ERROR_NDIS_ERROR_READING_FILE		  HRES_ERROR(0x8034001C)
#define HRES_ERROR_NDIS_ALREADY_MAPPED			  HRES_ERROR(0x8034001D)
#define HRES_ERROR_NDIS_RESOURCE_CONFLICT		  HRES_ERROR(0x8034001E)
#define HRES_ERROR_NDIS_MEDIA_DISCONNECTED		  HRES_ERROR(0x8034001F)
#define HRES_ERROR_NDIS_INVALID_ADDRESS			  HRES_ERROR(0x80340022)
#define HRES_ERROR_NDIS_PAUSED				  HRES_ERROR(0x8034002A)
#define HRES_ERROR_NDIS_INTERFACE_NOT_FOUND		  HRES_ERROR(0x8034002B)
#define HRES_ERROR_NDIS_UNSUPPORTED_REVISION		  HRES_ERROR(0x8034002C)
#define HRES_ERROR_NDIS_INVALID_PORT			  HRES_ERROR(0x8034002D)
#define HRES_ERROR_NDIS_INVALID_PORT_STATE		  HRES_ERROR(0x8034002E)
#define HRES_ERROR_NDIS_NOT_SUPPORTED			  HRES_ERROR(0x803400BB)
#define HRES_ERROR_NDIS_DOT11_AUTO_CONFIG_ENABLED	  HRES_ERROR(0x80342000)
#define HRES_ERROR_NDIS_DOT11_MEDIA_IN_USE		  HRES_ERROR(0x80342001)
#define HRES_ERROR_NDIS_DOT11_POWER_STATE_INVALID	  HRES_ERROR(0x80342002)
#define HRES_TRK_E_NOT_FOUND				  HRES_ERROR(0x8DEAD01B)
#define HRES_TRK_E_VOLUME_QUOTA_EXCEEDED		  HRES_ERROR(0x8DEAD01C)
#define HRES_TRK_SERVER_TOO_BUSY			  HRES_ERROR(0x8DEAD01E)
#define HRES_ERROR_AUDITING_DISABLED			  HRES_ERROR(0xC0090001)
#define HRES_ERROR_ALL_SIDS_FILTERED			  HRES_ERROR(0xC0090002)
#define HRES_ERROR_BIZRULES_NOT_ENABLED			  HRES_ERROR(0xC0090003)
#define HRES_NS_E_NOCONNECTION				  HRES_ERROR(0xC00D0005)
#define HRES_NS_E_CANNOTCONNECT				  HRES_ERROR(0xC00D0006)
#define HRES_NS_E_CANNOTDESTROYTITLE			  HRES_ERROR(0xC00D0007)
#define HRES_NS_E_CANNOTRENAMETITLE			  HRES_ERROR(0xC00D0008)
#define HRES_NS_E_CANNOTOFFLINEDISK			  HRES_ERROR(0xC00D0009)
#define HRES_NS_E_CANNOTONLINEDISK			  HRES_ERROR(0xC00D000A)
#define HRES_NS_E_NOREGISTEREDWALKER			  HRES_ERROR(0xC00D000B)
#define HRES_NS_E_NOFUNNEL				  HRES_ERROR(0xC00D000C)
#define HRES_NS_E_NO_LOCALPLAY				  HRES_ERROR(0xC00D000D)
#define HRES_NS_E_NETWORK_BUSY				  HRES_ERROR(0xC00D000E)
#define HRES_NS_E_TOO_MANY_SESS				  HRES_ERROR(0xC00D000F)
#define HRES_NS_E_ALREADY_CONNECTED			  HRES_ERROR(0xC00D0010)
#define HRES_NS_E_INVALID_INDEX				  HRES_ERROR(0xC00D0011)
#define HRES_NS_E_PROTOCOL_MISMATCH			  HRES_ERROR(0xC00D0012)
#define HRES_NS_E_TIMEOUT				  HRES_ERROR(0xC00D0013)
#define HRES_NS_E_NET_WRITE				  HRES_ERROR(0xC00D0014)
#define HRES_NS_E_NET_READ				  HRES_ERROR(0xC00D0015)
#define HRES_NS_E_DISK_WRITE				  HRES_ERROR(0xC00D0016)
#define HRES_NS_E_DISK_READ				  HRES_ERROR(0xC00D0017)
#define HRES_NS_E_FILE_WRITE				  HRES_ERROR(0xC00D0018)
#define HRES_NS_E_FILE_READ				  HRES_ERROR(0xC00D0019)
#define HRES_NS_E_FILE_NOT_FOUND			  HRES_ERROR(0xC00D001A)
#define HRES_NS_E_FILE_EXISTS				  HRES_ERROR(0xC00D001B)
#define HRES_NS_E_INVALID_NAME				  HRES_ERROR(0xC00D001C)
#define HRES_NS_E_FILE_OPEN_FAILED			  HRES_ERROR(0xC00D001D)
#define HRES_NS_E_FILE_ALLOCATION_FAILED		  HRES_ERROR(0xC00D001E)
#define HRES_NS_E_FILE_INIT_FAILED			  HRES_ERROR(0xC00D001F)
#define HRES_NS_E_FILE_PLAY_FAILED			  HRES_ERROR(0xC00D0020)
#define HRES_NS_E_SET_DISK_UID_FAILED			  HRES_ERROR(0xC00D0021)
#define HRES_NS_E_INDUCED				  HRES_ERROR(0xC00D0022)
#define HRES_NS_E_CCLINK_DOWN				  HRES_ERROR(0xC00D0023)
#define HRES_NS_E_INTERNAL				  HRES_ERROR(0xC00D0024)
#define HRES_NS_E_BUSY					  HRES_ERROR(0xC00D0025)
#define HRES_NS_E_UNRECOGNIZED_STREAM_TYPE		  HRES_ERROR(0xC00D0026)
#define HRES_NS_E_NETWORK_SERVICE_FAILURE		  HRES_ERROR(0xC00D0027)
#define HRES_NS_E_NETWORK_RESOURCE_FAILURE		  HRES_ERROR(0xC00D0028)
#define HRES_NS_E_CONNECTION_FAILURE			  HRES_ERROR(0xC00D0029)
#define HRES_NS_E_SHUTDOWN				  HRES_ERROR(0xC00D002A)
#define HRES_NS_E_INVALID_REQUEST			  HRES_ERROR(0xC00D002B)
#define HRES_NS_E_INSUFFICIENT_BANDWIDTH		  HRES_ERROR(0xC00D002C)
#define HRES_NS_E_NOT_REBUILDING			  HRES_ERROR(0xC00D002D)
#define HRES_NS_E_LATE_OPERATION			  HRES_ERROR(0xC00D002E)
#define HRES_NS_E_INVALID_DATA				  HRES_ERROR(0xC00D002F)
#define HRES_NS_E_FILE_BANDWIDTH_LIMIT			  HRES_ERROR(0xC00D0030)
#define HRES_NS_E_OPEN_FILE_LIMIT			  HRES_ERROR(0xC00D0031)
#define HRES_NS_E_BAD_CONTROL_DATA			  HRES_ERROR(0xC00D0032)
#define HRES_NS_E_NO_STREAM				  HRES_ERROR(0xC00D0033)
#define HRES_NS_E_STREAM_END				  HRES_ERROR(0xC00D0034)
#define HRES_NS_E_SERVER_NOT_FOUND			  HRES_ERROR(0xC00D0035)
#define HRES_NS_E_DUPLICATE_NAME			  HRES_ERROR(0xC00D0036)
#define HRES_NS_E_DUPLICATE_ADDRESS			  HRES_ERROR(0xC00D0037)
#define HRES_NS_E_BAD_MULTICAST_ADDRESS			  HRES_ERROR(0xC00D0038)
#define HRES_NS_E_BAD_ADAPTER_ADDRESS			  HRES_ERROR(0xC00D0039)
#define HRES_NS_E_BAD_DELIVERY_MODE			  HRES_ERROR(0xC00D003A)
#define HRES_NS_E_INVALID_CHANNEL			  HRES_ERROR(0xC00D003B)
#define HRES_NS_E_INVALID_STREAM			  HRES_ERROR(0xC00D003C)
#define HRES_NS_E_INVALID_ARCHIVE			  HRES_ERROR(0xC00D003D)
#define HRES_NS_E_NOTITLES				  HRES_ERROR(0xC00D003E)
#define HRES_NS_E_INVALID_CLIENT			  HRES_ERROR(0xC00D003F)
#define HRES_NS_E_INVALID_BLACKHOLE_ADDRESS		  HRES_ERROR(0xC00D0040)
#define HRES_NS_E_INCOMPATIBLE_FORMAT			  HRES_ERROR(0xC00D0041)
#define HRES_NS_E_INVALID_KEY				  HRES_ERROR(0xC00D0042)
#define HRES_NS_E_INVALID_PORT				  HRES_ERROR(0xC00D0043)
#define HRES_NS_E_INVALID_TTL				  HRES_ERROR(0xC00D0044)
#define HRES_NS_E_STRIDE_REFUSED			  HRES_ERROR(0xC00D0045)
#define HRES_NS_E_MMSAUTOSERVER_CANTFINDWALKER		  HRES_ERROR(0xC00D0046)
#define HRES_NS_E_MAX_BITRATE				  HRES_ERROR(0xC00D0047)
#define HRES_NS_E_LOGFILEPERIOD				  HRES_ERROR(0xC00D0048)
#define HRES_NS_E_MAX_CLIENTS				  HRES_ERROR(0xC00D0049)
#define HRES_NS_E_LOG_FILE_SIZE				  HRES_ERROR(0xC00D004A)
#define HRES_NS_E_MAX_FILERATE				  HRES_ERROR(0xC00D004B)
#define HRES_NS_E_WALKER_UNKNOWN			  HRES_ERROR(0xC00D004C)
#define HRES_NS_E_WALKER_SERVER				  HRES_ERROR(0xC00D004D)
#define HRES_NS_E_WALKER_USAGE				  HRES_ERROR(0xC00D004E)
#define HRES_NS_E_TIGER_FAIL				  HRES_ERROR(0xC00D0050)
#define HRES_NS_E_CUB_FAIL				  HRES_ERROR(0xC00D0053)
#define HRES_NS_E_DISK_FAIL				  HRES_ERROR(0xC00D0055)
#define HRES_NS_E_MAX_FUNNELS_ALERT			  HRES_ERROR(0xC00D0060)
#define HRES_NS_E_ALLOCATE_FILE_FAIL			  HRES_ERROR(0xC00D0061)
#define HRES_NS_E_PAGING_ERROR				  HRES_ERROR(0xC00D0062)
#define HRES_NS_E_BAD_BLOCK0_VERSION			  HRES_ERROR(0xC00D0063)
#define HRES_NS_E_BAD_DISK_UID				  HRES_ERROR(0xC00D0064)
#define HRES_NS_E_BAD_FSMAJOR_VERSION			  HRES_ERROR(0xC00D0065)
#define HRES_NS_E_BAD_STAMPNUMBER			  HRES_ERROR(0xC00D0066)
#define HRES_NS_E_PARTIALLY_REBUILT_DISK		  HRES_ERROR(0xC00D0067)
#define HRES_NS_E_ENACTPLAN_GIVEUP			  HRES_ERROR(0xC00D0068)
#define HRES_MCMADM_E_REGKEY_NOT_FOUND			  HRES_ERROR(0xC00D006A)
#define HRES_NS_E_NO_FORMATS				  HRES_ERROR(0xC00D006B)
#define HRES_NS_E_NO_REFERENCES				  HRES_ERROR(0xC00D006C)
#define HRES_NS_E_WAVE_OPEN				  HRES_ERROR(0xC00D006D)
#define HRES_NS_E_CANNOTCONNECTEVENTS			  HRES_ERROR(0xC00D006F)
#define HRES_NS_E_NO_DEVICE				  HRES_ERROR(0xC00D0071)
#define HRES_NS_E_NO_SPECIFIED_DEVICE			  HRES_ERROR(0xC00D0072)
#define HRES_NS_E_MONITOR_GIVEUP			  HRES_ERROR(0xC00D00C8)
#define HRES_NS_E_REMIRRORED_DISK			  HRES_ERROR(0xC00D00C9)
#define HRES_NS_E_INSUFFICIENT_DATA			  HRES_ERROR(0xC00D00CA)
#define HRES_NS_E_ASSERT				  HRES_ERROR(0xC00D00CB)
#define HRES_NS_E_BAD_ADAPTER_NAME			  HRES_ERROR(0xC00D00CC)
#define HRES_NS_E_NOT_LICENSED				  HRES_ERROR(0xC00D00CD)
#define HRES_NS_E_NO_SERVER_CONTACT			  HRES_ERROR(0xC00D00CE)
#define HRES_NS_E_TOO_MANY_TITLES			  HRES_ERROR(0xC00D00CF)
#define HRES_NS_E_TITLE_SIZE_EXCEEDED			  HRES_ERROR(0xC00D00D0)
#define HRES_NS_E_UDP_DISABLED				  HRES_ERROR(0xC00D00D1)
#define HRES_NS_E_TCP_DISABLED				  HRES_ERROR(0xC00D00D2)
#define HRES_NS_E_HTTP_DISABLED				  HRES_ERROR(0xC00D00D3)
#define HRES_NS_E_LICENSE_EXPIRED			  HRES_ERROR(0xC00D00D4)
#define HRES_NS_E_TITLE_BITRATE				  HRES_ERROR(0xC00D00D5)
#define HRES_NS_E_EMPTY_PROGRAM_NAME			  HRES_ERROR(0xC00D00D6)
#define HRES_NS_E_MISSING_CHANNEL			  HRES_ERROR(0xC00D00D7)
#define HRES_NS_E_NO_CHANNELS				  HRES_ERROR(0xC00D00D8)
#define HRES_NS_E_INVALID_INDEX2			  HRES_ERROR(0xC00D00D9)
#define HRES_NS_E_CUB_FAIL_LINK				  HRES_ERROR(0xC00D0190)
#define HRES_NS_E_BAD_CUB_UID				  HRES_ERROR(0xC00D0192)
#define HRES_NS_E_GLITCH_MODE				  HRES_ERROR(0xC00D0195)
#define HRES_NS_E_NO_MEDIA_PROTOCOL			  HRES_ERROR(0xC00D019B)
#define HRES_NS_E_NOTHING_TO_DO				  HRES_ERROR(0xC00D07F1)
#define HRES_NS_E_NO_MULTICAST				  HRES_ERROR(0xC00D07F2)
#define HRES_NS_E_INVALID_INPUT_FORMAT			  HRES_ERROR(0xC00D0BB8)
#define HRES_NS_E_MSAUDIO_NOT_INSTALLED			  HRES_ERROR(0xC00D0BB9)
#define HRES_NS_E_UNEXPECTED_MSAUDIO_ERROR		  HRES_ERROR(0xC00D0BBA)
#define HRES_NS_E_INVALID_OUTPUT_FORMAT			  HRES_ERROR(0xC00D0BBB)
#define HRES_NS_E_NOT_CONFIGURED			  HRES_ERROR(0xC00D0BBC)
#define HRES_NS_E_PROTECTED_CONTENT			  HRES_ERROR(0xC00D0BBD)
#define HRES_NS_E_LICENSE_REQUIRED			  HRES_ERROR(0xC00D0BBE)
#define HRES_NS_E_TAMPERED_CONTENT			  HRES_ERROR(0xC00D0BBF)
#define HRES_NS_E_LICENSE_OUTOFDATE			  HRES_ERROR(0xC00D0BC0)
#define HRES_NS_E_LICENSE_INCORRECT_RIGHTS		  HRES_ERROR(0xC00D0BC1)
#define HRES_NS_E_AUDIO_CODEC_NOT_INSTALLED		  HRES_ERROR(0xC00D0BC2)
#define HRES_NS_E_AUDIO_CODEC_ERROR			  HRES_ERROR(0xC00D0BC3)
#define HRES_NS_E_VIDEO_CODEC_NOT_INSTALLED		  HRES_ERROR(0xC00D0BC4)
#define HRES_NS_E_VIDEO_CODEC_ERROR			  HRES_ERROR(0xC00D0BC5)
#define HRES_NS_E_INVALIDPROFILE			  HRES_ERROR(0xC00D0BC6)
#define HRES_NS_E_INCOMPATIBLE_VERSION			  HRES_ERROR(0xC00D0BC7)
#define HRES_NS_E_OFFLINE_MODE				  HRES_ERROR(0xC00D0BCA)
#define HRES_NS_E_NOT_CONNECTED				  HRES_ERROR(0xC00D0BCB)
#define HRES_NS_E_TOO_MUCH_DATA				  HRES_ERROR(0xC00D0BCC)
#define HRES_NS_E_UNSUPPORTED_PROPERTY			  HRES_ERROR(0xC00D0BCD)
#define HRES_NS_E_8BIT_WAVE_UNSUPPORTED			  HRES_ERROR(0xC00D0BCE)
#define HRES_NS_E_NO_MORE_SAMPLES			  HRES_ERROR(0xC00D0BCF)
#define HRES_NS_E_INVALID_SAMPLING_RATE			  HRES_ERROR(0xC00D0BD0)
#define HRES_NS_E_MAX_PACKET_SIZE_TOO_SMALL		  HRES_ERROR(0xC00D0BD1)
#define HRES_NS_E_LATE_PACKET				  HRES_ERROR(0xC00D0BD2)
#define HRES_NS_E_DUPLICATE_PACKET			  HRES_ERROR(0xC00D0BD3)
#define HRES_NS_E_SDK_BUFFERTOOSMALL			  HRES_ERROR(0xC00D0BD4)
#define HRES_NS_E_INVALID_NUM_PASSES			  HRES_ERROR(0xC00D0BD5)
#define HRES_NS_E_ATTRIBUTE_READ_ONLY			  HRES_ERROR(0xC00D0BD6)
#define HRES_NS_E_ATTRIBUTE_NOT_ALLOWED			  HRES_ERROR(0xC00D0BD7)
#define HRES_NS_E_INVALID_EDL				  HRES_ERROR(0xC00D0BD8)
#define HRES_NS_E_DATA_UNIT_EXTENSION_TOO_LARGE		  HRES_ERROR(0xC00D0BD9)
#define HRES_NS_E_CODEC_DMO_ERROR			  HRES_ERROR(0xC00D0BDA)
#define HRES_NS_E_FEATURE_DISABLED_BY_GROUP_POLICY	  HRES_ERROR(0xC00D0BDC)
#define HRES_NS_E_FEATURE_DISABLED_IN_SKU		  HRES_ERROR(0xC00D0BDD)
#define HRES_NS_E_NO_CD					  HRES_ERROR(0xC00D0FA0)
#define HRES_NS_E_CANT_READ_DIGITAL			  HRES_ERROR(0xC00D0FA1)
#define HRES_NS_E_DEVICE_DISCONNECTED			  HRES_ERROR(0xC00D0FA2)
#define HRES_NS_E_DEVICE_NOT_SUPPORT_FORMAT		  HRES_ERROR(0xC00D0FA3)
#define HRES_NS_E_SLOW_READ_DIGITAL			  HRES_ERROR(0xC00D0FA4)
#define HRES_NS_E_MIXER_INVALID_LINE			  HRES_ERROR(0xC00D0FA5)
#define HRES_NS_E_MIXER_INVALID_CONTROL			  HRES_ERROR(0xC00D0FA6)
#define HRES_NS_E_MIXER_INVALID_VALUE			  HRES_ERROR(0xC00D0FA7)
#define HRES_NS_E_MIXER_UNKNOWN_MMRESULT		  HRES_ERROR(0xC00D0FA8)
#define HRES_NS_E_USER_STOP				  HRES_ERROR(0xC00D0FA9)
#define HRES_NS_E_MP3_FORMAT_NOT_FOUND			  HRES_ERROR(0xC00D0FAA)
#define HRES_NS_E_CD_READ_ERROR_NO_CORRECTION		  HRES_ERROR(0xC00D0FAB)
#define HRES_NS_E_CD_READ_ERROR				  HRES_ERROR(0xC00D0FAC)
#define HRES_NS_E_CD_SLOW_COPY				  HRES_ERROR(0xC00D0FAD)
#define HRES_NS_E_CD_COPYTO_CD				  HRES_ERROR(0xC00D0FAE)
#define HRES_NS_E_MIXER_NODRIVER			  HRES_ERROR(0xC00D0FAF)
#define HRES_NS_E_REDBOOK_ENABLED_WHILE_COPYING		  HRES_ERROR(0xC00D0FB0)
#define HRES_NS_E_CD_REFRESH				  HRES_ERROR(0xC00D0FB1)
#define HRES_NS_E_CD_DRIVER_PROBLEM			  HRES_ERROR(0xC00D0FB2)
#define HRES_NS_E_WONT_DO_DIGITAL			  HRES_ERROR(0xC00D0FB3)
#define HRES_NS_E_WMPXML_NOERROR			  HRES_ERROR(0xC00D0FB4)
#define HRES_NS_E_WMPXML_ENDOFDATA			  HRES_ERROR(0xC00D0FB5)
#define HRES_NS_E_WMPXML_PARSEERROR			  HRES_ERROR(0xC00D0FB6)
#define HRES_NS_E_WMPXML_ATTRIBUTENOTFOUND		  HRES_ERROR(0xC00D0FB7)
#define HRES_NS_E_WMPXML_PINOTFOUND			  HRES_ERROR(0xC00D0FB8)
#define HRES_NS_E_WMPXML_EMPTYDOC			  HRES_ERROR(0xC00D0FB9)
#define HRES_NS_E_WMP_PATH_ALREADY_IN_LIBRARY		  HRES_ERROR(0xC00D0FBA)
#define HRES_NS_E_WMP_FILESCANALREADYSTARTED		  HRES_ERROR(0xC00D0FBE)
#define HRES_NS_E_WMP_HME_INVALIDOBJECTID		  HRES_ERROR(0xC00D0FBF)
#define HRES_NS_E_WMP_MF_CODE_EXPIRED			  HRES_ERROR(0xC00D0FC0)
#define HRES_NS_E_WMP_HME_NOTSEARCHABLEFORITEMS		  HRES_ERROR(0xC00D0FC1)
#define HRES_NS_E_WMP_ADDTOLIBRARY_FAILED		  HRES_ERROR(0xC00D0FC7)
#define HRES_NS_E_WMP_WINDOWSAPIFAILURE			  HRES_ERROR(0xC00D0FC8)
#define HRES_NS_E_WMP_RECORDING_NOT_ALLOWED		  HRES_ERROR(0xC00D0FC9)
#define HRES_NS_E_DEVICE_NOT_READY			  HRES_ERROR(0xC00D0FCA)
#define HRES_NS_E_DAMAGED_FILE				  HRES_ERROR(0xC00D0FCB)
#define HRES_NS_E_MPDB_GENERIC				  HRES_ERROR(0xC00D0FCC)
#define HRES_NS_E_FILE_FAILED_CHECKS			  HRES_ERROR(0xC00D0FCD)
#define HRES_NS_E_MEDIA_LIBRARY_FAILED			  HRES_ERROR(0xC00D0FCE)
#define HRES_NS_E_SHARING_VIOLATION			  HRES_ERROR(0xC00D0FCF)
#define HRES_NS_E_NO_ERROR_STRING_FOUND			  HRES_ERROR(0xC00D0FD0)
#define HRES_NS_E_WMPOCX_NO_REMOTE_CORE			  HRES_ERROR(0xC00D0FD1)
#define HRES_NS_E_WMPOCX_NO_ACTIVE_CORE			  HRES_ERROR(0xC00D0FD2)
#define HRES_NS_E_WMPOCX_NOT_RUNNING_REMOTELY		  HRES_ERROR(0xC00D0FD3)
#define HRES_NS_E_WMPOCX_NO_REMOTE_WINDOW		  HRES_ERROR(0xC00D0FD4)
#define HRES_NS_E_WMPOCX_ERRORMANAGERNOTAVAILABLE	  HRES_ERROR(0xC00D0FD5)
#define HRES_NS_E_PLUGIN_NOTSHUTDOWN			  HRES_ERROR(0xC00D0FD6)
#define HRES_NS_E_WMP_CANNOT_FIND_FOLDER		  HRES_ERROR(0xC00D0FD7)
#define HRES_NS_E_WMP_STREAMING_RECORDING_NOT_ALLOWED	  HRES_ERROR(0xC00D0FD8)
#define HRES_NS_E_WMP_PLUGINDLL_NOTFOUND		  HRES_ERROR(0xC00D0FD9)
#define HRES_NS_E_NEED_TO_ASK_USER			  HRES_ERROR(0xC00D0FDA)
#define HRES_NS_E_WMPOCX_PLAYER_NOT_DOCKED		  HRES_ERROR(0xC00D0FDB)
#define HRES_NS_E_WMP_EXTERNAL_NOTREADY			  HRES_ERROR(0xC00D0FDC)
#define HRES_NS_E_WMP_MLS_STALE_DATA			  HRES_ERROR(0xC00D0FDD)
#define HRES_NS_E_WMP_UI_SUBCONTROLSNOTSUPPORTED	  HRES_ERROR(0xC00D0FDE)
#define HRES_NS_E_WMP_UI_VERSIONMISMATCH		  HRES_ERROR(0xC00D0FDF)
#define HRES_NS_E_WMP_UI_NOTATHEMEFILE			  HRES_ERROR(0xC00D0FE0)
#define HRES_NS_E_WMP_UI_SUBELEMENTNOTFOUND		  HRES_ERROR(0xC00D0FE1)
#define HRES_NS_E_WMP_UI_VERSIONPARSE			  HRES_ERROR(0xC00D0FE2)
#define HRES_NS_E_WMP_UI_VIEWIDNOTFOUND			  HRES_ERROR(0xC00D0FE3)
#define HRES_NS_E_WMP_UI_PASSTHROUGH			  HRES_ERROR(0xC00D0FE4)
#define HRES_NS_E_WMP_UI_OBJECTNOTFOUND			  HRES_ERROR(0xC00D0FE5)
#define HRES_NS_E_WMP_UI_SECONDHANDLER			  HRES_ERROR(0xC00D0FE6)
#define HRES_NS_E_WMP_UI_NOSKININZIP			  HRES_ERROR(0xC00D0FE7)
#define HRES_NS_E_WMP_URLDOWNLOADFAILED			  HRES_ERROR(0xC00D0FEA)
#define HRES_NS_E_WMPOCX_UNABLE_TO_LOAD_SKIN		  HRES_ERROR(0xC00D0FEB)
#define HRES_NS_E_WMP_INVALID_SKIN			  HRES_ERROR(0xC00D0FEC)
#define HRES_NS_E_WMP_SENDMAILFAILED			  HRES_ERROR(0xC00D0FED)
#define HRES_NS_E_WMP_LOCKEDINSKINMODE			  HRES_ERROR(0xC00D0FEE)
#define HRES_NS_E_WMP_FAILED_TO_SAVE_FILE		  HRES_ERROR(0xC00D0FEF)
#define HRES_NS_E_WMP_SAVEAS_READONLY			  HRES_ERROR(0xC00D0FF0)
#define HRES_NS_E_WMP_FAILED_TO_SAVE_PLAYLIST		  HRES_ERROR(0xC00D0FF1)
#define HRES_NS_E_WMP_FAILED_TO_OPEN_WMD		  HRES_ERROR(0xC00D0FF2)
#define HRES_NS_E_WMP_CANT_PLAY_PROTECTED		  HRES_ERROR(0xC00D0FF3)
#define HRES_NS_E_SHARING_STATE_OUT_OF_SYNC		  HRES_ERROR(0xC00D0FF4)
#define HRES_NS_E_WMPOCX_REMOTE_PLAYER_ALREADY_RUNNING	  HRES_ERROR(0xC00D0FFA)
#define HRES_NS_E_WMP_RBC_JPGMAPPINGIMAGE		  HRES_ERROR(0xC00D1004)
#define HRES_NS_E_WMP_JPGTRANSPARENCY			  HRES_ERROR(0xC00D1005)
#define HRES_NS_E_WMP_INVALID_MAX_VAL			  HRES_ERROR(0xC00D1009)
#define HRES_NS_E_WMP_INVALID_MIN_VAL			  HRES_ERROR(0xC00D100A)
#define HRES_NS_E_WMP_CS_JPGPOSITIONIMAGE		  HRES_ERROR(0xC00D100E)
#define HRES_NS_E_WMP_CS_NOTEVENLYDIVISIBLE		  HRES_ERROR(0xC00D100F)
#define HRES_NS_E_WMPZIP_NOTAZIPFILE			  HRES_ERROR(0xC00D1018)
#define HRES_NS_E_WMPZIP_CORRUPT			  HRES_ERROR(0xC00D1019)
#define HRES_NS_E_WMPZIP_FILENOTFOUND			  HRES_ERROR(0xC00D101A)
#define HRES_NS_E_WMP_IMAGE_FILETYPE_UNSUPPORTED	  HRES_ERROR(0xC00D1022)
#define HRES_NS_E_WMP_IMAGE_INVALID_FORMAT		  HRES_ERROR(0xC00D1023)
#define HRES_NS_E_WMP_GIF_UNEXPECTED_ENDOFFILE		  HRES_ERROR(0xC00D1024)
#define HRES_NS_E_WMP_GIF_INVALID_FORMAT		  HRES_ERROR(0xC00D1025)
#define HRES_NS_E_WMP_GIF_BAD_VERSION_NUMBER		  HRES_ERROR(0xC00D1026)
#define HRES_NS_E_WMP_GIF_NO_IMAGE_IN_FILE		  HRES_ERROR(0xC00D1027)
#define HRES_NS_E_WMP_PNG_INVALIDFORMAT			  HRES_ERROR(0xC00D1028)
#define HRES_NS_E_WMP_PNG_UNSUPPORTED_BITDEPTH		  HRES_ERROR(0xC00D1029)
#define HRES_NS_E_WMP_PNG_UNSUPPORTED_COMPRESSION	  HRES_ERROR(0xC00D102A)
#define HRES_NS_E_WMP_PNG_UNSUPPORTED_FILTER		  HRES_ERROR(0xC00D102B)
#define HRES_NS_E_WMP_PNG_UNSUPPORTED_INTERLACE		  HRES_ERROR(0xC00D102C)
#define HRES_NS_E_WMP_PNG_UNSUPPORTED_BAD_CRC		  HRES_ERROR(0xC00D102D)
#define HRES_NS_E_WMP_BMP_INVALID_BITMASK		  HRES_ERROR(0xC00D102E)
#define HRES_NS_E_WMP_BMP_TOPDOWN_DIB_UNSUPPORTED	  HRES_ERROR(0xC00D102F)
#define HRES_NS_E_WMP_BMP_BITMAP_NOT_CREATED		  HRES_ERROR(0xC00D1030)
#define HRES_NS_E_WMP_BMP_COMPRESSION_UNSUPPORTED	  HRES_ERROR(0xC00D1031)
#define HRES_NS_E_WMP_BMP_INVALID_FORMAT		  HRES_ERROR(0xC00D1032)
#define HRES_NS_E_WMP_JPG_JERR_ARITHCODING_NOTIMPL	  HRES_ERROR(0xC00D1033)
#define HRES_NS_E_WMP_JPG_INVALID_FORMAT		  HRES_ERROR(0xC00D1034)
#define HRES_NS_E_WMP_JPG_BAD_DCTSIZE			  HRES_ERROR(0xC00D1035)
#define HRES_NS_E_WMP_JPG_BAD_VERSION_NUMBER		  HRES_ERROR(0xC00D1036)
#define HRES_NS_E_WMP_JPG_BAD_PRECISION			  HRES_ERROR(0xC00D1037)
#define HRES_NS_E_WMP_JPG_CCIR601_NOTIMPL		  HRES_ERROR(0xC00D1038)
#define HRES_NS_E_WMP_JPG_NO_IMAGE_IN_FILE		  HRES_ERROR(0xC00D1039)
#define HRES_NS_E_WMP_JPG_READ_ERROR			  HRES_ERROR(0xC00D103A)
#define HRES_NS_E_WMP_JPG_FRACT_SAMPLE_NOTIMPL		  HRES_ERROR(0xC00D103B)
#define HRES_NS_E_WMP_JPG_IMAGE_TOO_BIG			  HRES_ERROR(0xC00D103C)
#define HRES_NS_E_WMP_JPG_UNEXPECTED_ENDOFFILE		  HRES_ERROR(0xC00D103D)
#define HRES_NS_E_WMP_JPG_SOF_UNSUPPORTED		  HRES_ERROR(0xC00D103E)
#define HRES_NS_E_WMP_JPG_UNKNOWN_MARKER		  HRES_ERROR(0xC00D103F)
#define HRES_NS_E_WMP_FAILED_TO_OPEN_IMAGE		  HRES_ERROR(0xC00D1044)
#define HRES_NS_E_WMP_DAI_SONGTOOSHORT			  HRES_ERROR(0xC00D1049)
#define HRES_NS_E_WMG_RATEUNAVAILABLE			  HRES_ERROR(0xC00D104A)
#define HRES_NS_E_WMG_PLUGINUNAVAILABLE			  HRES_ERROR(0xC00D104B)
#define HRES_NS_E_WMG_CANNOTQUEUE			  HRES_ERROR(0xC00D104C)
#define HRES_NS_E_WMG_PREROLLLICENSEACQUISITIONNOTALLOWED HRES_ERROR(0xC00D104D)
#define HRES_NS_E_WMG_UNEXPECTEDPREROLLSTATUS		  HRES_ERROR(0xC00D104E)
#define HRES_NS_E_WMG_INVALID_COPP_CERTIFICATE		  HRES_ERROR(0xC00D1051)
#define HRES_NS_E_WMG_COPP_SECURITY_INVALID		  HRES_ERROR(0xC00D1052)
#define HRES_NS_E_WMG_COPP_UNSUPPORTED			  HRES_ERROR(0xC00D1053)
#define HRES_NS_E_WMG_INVALIDSTATE			  HRES_ERROR(0xC00D1054)
#define HRES_NS_E_WMG_SINKALREADYEXISTS			  HRES_ERROR(0xC00D1055)
#define HRES_NS_E_WMG_NOSDKINTERFACE			  HRES_ERROR(0xC00D1056)
#define HRES_NS_E_WMG_NOTALLOUTPUTSRENDERED		  HRES_ERROR(0xC00D1057)
#define HRES_NS_E_WMG_FILETRANSFERNOTALLOWED		  HRES_ERROR(0xC00D1058)
#define HRES_NS_E_WMR_UNSUPPORTEDSTREAM			  HRES_ERROR(0xC00D1059)
#define HRES_NS_E_WMR_PINNOTFOUND			  HRES_ERROR(0xC00D105A)
#define HRES_NS_E_WMR_WAITINGONFORMATSWITCH		  HRES_ERROR(0xC00D105B)
#define HRES_NS_E_WMR_NOSOURCEFILTER			  HRES_ERROR(0xC00D105C)
#define HRES_NS_E_WMR_PINTYPENOMATCH			  HRES_ERROR(0xC00D105D)
#define HRES_NS_E_WMR_NOCALLBACKAVAILABLE		  HRES_ERROR(0xC00D105E)
#define HRES_NS_E_WMR_SAMPLEPROPERTYNOTSET		  HRES_ERROR(0xC00D1062)
#define HRES_NS_E_WMR_CANNOT_RENDER_BINARY_STREAM	  HRES_ERROR(0xC00D1063)
#define HRES_NS_E_WMG_LICENSE_TAMPERED			  HRES_ERROR(0xC00D1064)
#define HRES_NS_E_WMR_WILLNOT_RENDER_BINARY_STREAM	  HRES_ERROR(0xC00D1065)
#define HRES_NS_E_WMX_UNRECOGNIZED_PLAYLIST_FORMAT	  HRES_ERROR(0xC00D1068)
#define HRES_NS_E_ASX_INVALIDFORMAT			  HRES_ERROR(0xC00D1069)
#define HRES_NS_E_ASX_INVALIDVERSION			  HRES_ERROR(0xC00D106A)
#define HRES_NS_E_ASX_INVALID_REPEAT_BLOCK		  HRES_ERROR(0xC00D106B)
#define HRES_NS_E_ASX_NOTHING_TO_WRITE			  HRES_ERROR(0xC00D106C)
#define HRES_NS_E_URLLIST_INVALIDFORMAT			  HRES_ERROR(0xC00D106D)
#define HRES_NS_E_WMX_ATTRIBUTE_DOES_NOT_EXIST		  HRES_ERROR(0xC00D106E)
#define HRES_NS_E_WMX_ATTRIBUTE_ALREADY_EXISTS		  HRES_ERROR(0xC00D106F)
#define HRES_NS_E_WMX_ATTRIBUTE_UNRETRIEVABLE		  HRES_ERROR(0xC00D1070)
#define HRES_NS_E_WMX_ITEM_DOES_NOT_EXIST		  HRES_ERROR(0xC00D1071)
#define HRES_NS_E_WMX_ITEM_TYPE_ILLEGAL			  HRES_ERROR(0xC00D1072)
#define HRES_NS_E_WMX_ITEM_UNSETTABLE			  HRES_ERROR(0xC00D1073)
#define HRES_NS_E_WMX_PLAYLIST_EMPTY			  HRES_ERROR(0xC00D1074)
#define HRES_NS_E_MLS_SMARTPLAYLIST_FILTER_NOT_REGISTERED HRES_ERROR(0xC00D1075)
#define HRES_NS_E_WMX_INVALID_FORMAT_OVER_NESTING	  HRES_ERROR(0xC00D1076)
#define HRES_NS_E_WMPCORE_NOSOURCEURLSTRING		  HRES_ERROR(0xC00D107C)
#define HRES_NS_E_WMPCORE_COCREATEFAILEDFORGITOBJECT	  HRES_ERROR(0xC00D107D)
#define HRES_NS_E_WMPCORE_FAILEDTOGETMARSHALLEDEVENTHANDLERINTERFACE HRES_ERROR(0xC00D107E)
#define HRES_NS_E_WMPCORE_BUFFERTOOSMALL		  HRES_ERROR(0xC00D107F)
#define HRES_NS_E_WMPCORE_UNAVAILABLE			  HRES_ERROR(0xC00D1080)
#define HRES_NS_E_WMPCORE_INVALIDPLAYLISTMODE		  HRES_ERROR(0xC00D1081)
#define HRES_NS_E_WMPCORE_ITEMNOTINPLAYLIST		  HRES_ERROR(0xC00D1086)
#define HRES_NS_E_WMPCORE_PLAYLISTEMPTY			  HRES_ERROR(0xC00D1087)
#define HRES_NS_E_WMPCORE_NOBROWSER			  HRES_ERROR(0xC00D1088)
#define HRES_NS_E_WMPCORE_UNRECOGNIZED_MEDIA_URL	  HRES_ERROR(0xC00D1089)
#define HRES_NS_E_WMPCORE_GRAPH_NOT_IN_LIST		  HRES_ERROR(0xC00D108A)
#define HRES_NS_E_WMPCORE_PLAYLIST_EMPTY_OR_SINGLE_MEDIA  HRES_ERROR(0xC00D108B)
#define HRES_NS_E_WMPCORE_ERRORSINKNOTREGISTERED	  HRES_ERROR(0xC00D108C)
#define HRES_NS_E_WMPCORE_ERRORMANAGERNOTAVAILABLE	  HRES_ERROR(0xC00D108D)
#define HRES_NS_E_WMPCORE_WEBHELPFAILED			  HRES_ERROR(0xC00D108E)
#define HRES_NS_E_WMPCORE_MEDIA_ERROR_RESUME_FAILED	  HRES_ERROR(0xC00D108F)
#define HRES_NS_E_WMPCORE_NO_REF_IN_ENTRY		  HRES_ERROR(0xC00D1090)
#define HRES_NS_E_WMPCORE_WMX_LIST_ATTRIBUTE_NAME_EMPTY   HRES_ERROR(0xC00D1091)
#define HRES_NS_E_WMPCORE_WMX_LIST_ATTRIBUTE_NAME_ILLEGAL HRES_ERROR(0xC00D1092)
#define HRES_NS_E_WMPCORE_WMX_LIST_ATTRIBUTE_VALUE_EMPTY  HRES_ERROR(0xC00D1093)
#define HRES_NS_E_WMPCORE_WMX_LIST_ATTRIBUTE_VALUE_ILLEGAL HRES_ERROR(0xC00D1094)
#define HRES_NS_E_WMPCORE_WMX_LIST_ITEM_ATTRIBUTE_NAME_EMPTY HRES_ERROR(0xC00D1095)
#define HRES_NS_E_WMPCORE_WMX_LIST_ITEM_ATTRIBUTE_NAME_ILLEGAL HRES_ERROR(0xC00D1096)
#define HRES_NS_E_WMPCORE_WMX_LIST_ITEM_ATTRIBUTE_VALUE_EMPTY HRES_ERROR(0xC00D1097)
#define HRES_NS_E_WMPCORE_LIST_ENTRY_NO_REF		  HRES_ERROR(0xC00D1098)
#define HRES_NS_E_WMPCORE_MISNAMED_FILE			  HRES_ERROR(0xC00D1099)
#define HRES_NS_E_WMPCORE_CODEC_NOT_TRUSTED		  HRES_ERROR(0xC00D109A)
#define HRES_NS_E_WMPCORE_CODEC_NOT_FOUND		  HRES_ERROR(0xC00D109B)
#define HRES_NS_E_WMPCORE_CODEC_DOWNLOAD_NOT_ALLOWED	  HRES_ERROR(0xC00D109C)
#define HRES_NS_E_WMPCORE_ERROR_DOWNLOADING_PLAYLIST	  HRES_ERROR(0xC00D109D)
#define HRES_NS_E_WMPCORE_FAILED_TO_BUILD_PLAYLIST	  HRES_ERROR(0xC00D109E)
#define HRES_NS_E_WMPCORE_PLAYLIST_ITEM_ALTERNATE_NONE	  HRES_ERROR(0xC00D109F)
#define HRES_NS_E_WMPCORE_PLAYLIST_ITEM_ALTERNATE_EXHAUSTED HRES_ERROR(0xC00D10A0)
#define HRES_NS_E_WMPCORE_PLAYLIST_ITEM_ALTERNATE_NAME_NOT_FOUND HRES_ERROR(0xC00D10A1)
#define HRES_NS_E_WMPCORE_PLAYLIST_ITEM_ALTERNATE_MORPH_FAILED HRES_ERROR(0xC00D10A2)
#define HRES_NS_E_WMPCORE_PLAYLIST_ITEM_ALTERNATE_INIT_FAILED HRES_ERROR(0xC00D10A3)
#define HRES_NS_E_WMPCORE_MEDIA_ALTERNATE_REF_EMPTY	  HRES_ERROR(0xC00D10A4)
#define HRES_NS_E_WMPCORE_PLAYLIST_NO_EVENT_NAME	  HRES_ERROR(0xC00D10A5)
#define HRES_NS_E_WMPCORE_PLAYLIST_EVENT_ATTRIBUTE_ABSENT HRES_ERROR(0xC00D10A6)
#define HRES_NS_E_WMPCORE_PLAYLIST_EVENT_EMPTY		  HRES_ERROR(0xC00D10A7)
#define HRES_NS_E_WMPCORE_PLAYLIST_STACK_EMPTY		  HRES_ERROR(0xC00D10A8)
#define HRES_NS_E_WMPCORE_CURRENT_MEDIA_NOT_ACTIVE	  HRES_ERROR(0xC00D10A9)
#define HRES_NS_E_WMPCORE_USER_CANCEL			  HRES_ERROR(0xC00D10AB)
#define HRES_NS_E_WMPCORE_PLAYLIST_REPEAT_EMPTY		  HRES_ERROR(0xC00D10AC)
#define HRES_NS_E_WMPCORE_PLAYLIST_REPEAT_START_MEDIA_NONE HRES_ERROR(0xC00D10AD)
#define HRES_NS_E_WMPCORE_PLAYLIST_REPEAT_END_MEDIA_NONE  HRES_ERROR(0xC00D10AE)
#define HRES_NS_E_WMPCORE_INVALID_PLAYLIST_URL		  HRES_ERROR(0xC00D10AF)
#define HRES_NS_E_WMPCORE_MISMATCHED_RUNTIME		  HRES_ERROR(0xC00D10B0)
#define HRES_NS_E_WMPCORE_PLAYLIST_IMPORT_FAILED_NO_ITEMS HRES_ERROR(0xC00D10B1)
#define HRES_NS_E_WMPCORE_VIDEO_TRANSFORM_FILTER_INSERTION HRES_ERROR(0xC00D10B2)
#define HRES_NS_E_WMPCORE_MEDIA_UNAVAILABLE		  HRES_ERROR(0xC00D10B3)
#define HRES_NS_E_WMPCORE_WMX_ENTRYREF_NO_REF		  HRES_ERROR(0xC00D10B4)
#define HRES_NS_E_WMPCORE_NO_PLAYABLE_MEDIA_IN_PLAYLIST   HRES_ERROR(0xC00D10B5)
#define HRES_NS_E_WMPCORE_PLAYLIST_EMPTY_NESTED_PLAYLIST_SKIPPED_ITEMS HRES_ERROR(0xC00D10B6)
#define HRES_NS_E_WMPCORE_BUSY				  HRES_ERROR(0xC00D10B7)
#define HRES_NS_E_WMPCORE_MEDIA_CHILD_PLAYLIST_UNAVAILABLE HRES_ERROR(0xC00D10B8)
#define HRES_NS_E_WMPCORE_MEDIA_NO_CHILD_PLAYLIST	  HRES_ERROR(0xC00D10B9)
#define HRES_NS_E_WMPCORE_FILE_NOT_FOUND		  HRES_ERROR(0xC00D10BA)
#define HRES_NS_E_WMPCORE_TEMP_FILE_NOT_FOUND		  HRES_ERROR(0xC00D10BB)
#define HRES_NS_E_WMDM_REVOKED				  HRES_ERROR(0xC00D10BC)
#define HRES_NS_E_DDRAW_GENERIC				  HRES_ERROR(0xC00D10BD)
#define HRES_NS_E_DISPLAY_MODE_CHANGE_FAILED		  HRES_ERROR(0xC00D10BE)
#define HRES_NS_E_PLAYLIST_CONTAINS_ERRORS		  HRES_ERROR(0xC00D10BF)
#define HRES_NS_E_CHANGING_PROXY_NAME			  HRES_ERROR(0xC00D10C0)
#define HRES_NS_E_CHANGING_PROXY_PORT			  HRES_ERROR(0xC00D10C1)
#define HRES_NS_E_CHANGING_PROXY_EXCEPTIONLIST		  HRES_ERROR(0xC00D10C2)
#define HRES_NS_E_CHANGING_PROXYBYPASS			  HRES_ERROR(0xC00D10C3)
#define HRES_NS_E_CHANGING_PROXY_PROTOCOL_NOT_FOUND	  HRES_ERROR(0xC00D10C4)
#define HRES_NS_E_GRAPH_NOAUDIOLANGUAGE			  HRES_ERROR(0xC00D10C5)
#define HRES_NS_E_GRAPH_NOAUDIOLANGUAGESELECTED		  HRES_ERROR(0xC00D10C6)
#define HRES_NS_E_CORECD_NOTAMEDIACD			  HRES_ERROR(0xC00D10C7)
#define HRES_NS_E_WMPCORE_MEDIA_URL_TOO_LONG		  HRES_ERROR(0xC00D10C8)
#define HRES_NS_E_WMPFLASH_CANT_FIND_COM_SERVER		  HRES_ERROR(0xC00D10C9)
#define HRES_NS_E_WMPFLASH_INCOMPATIBLEVERSION		  HRES_ERROR(0xC00D10CA)
#define HRES_NS_E_WMPOCXGRAPH_IE_DISALLOWS_ACTIVEX_CONTROLS HRES_ERROR(0xC00D10CB)
#define HRES_NS_E_NEED_CORE_REFERENCE			  HRES_ERROR(0xC00D10CC)
#define HRES_NS_E_MEDIACD_READ_ERROR			  HRES_ERROR(0xC00D10CD)
#define HRES_NS_E_IE_DISALLOWS_ACTIVEX_CONTROLS		  HRES_ERROR(0xC00D10CE)
#define HRES_NS_E_FLASH_PLAYBACK_NOT_ALLOWED		  HRES_ERROR(0xC00D10CF)
#define HRES_NS_E_UNABLE_TO_CREATE_RIP_LOCATION		  HRES_ERROR(0xC00D10D0)
#define HRES_NS_E_WMPCORE_SOME_CODECS_MISSING		  HRES_ERROR(0xC00D10D1)
#define HRES_NS_E_WMP_RIP_FAILED			  HRES_ERROR(0xC00D10D2)
#define HRES_NS_E_WMP_FAILED_TO_RIP_TRACK		  HRES_ERROR(0xC00D10D3)
#define HRES_NS_E_WMP_ERASE_FAILED			  HRES_ERROR(0xC00D10D4)
#define HRES_NS_E_WMP_FORMAT_FAILED			  HRES_ERROR(0xC00D10D5)
#define HRES_NS_E_WMP_CANNOT_BURN_NON_LOCAL_FILE	  HRES_ERROR(0xC00D10D6)
#define HRES_NS_E_WMP_FILE_TYPE_CANNOT_BURN_TO_AUDIO_CD   HRES_ERROR(0xC00D10D7)
#define HRES_NS_E_WMP_FILE_DOES_NOT_FIT_ON_CD		  HRES_ERROR(0xC00D10D8)
#define HRES_NS_E_WMP_FILE_NO_DURATION			  HRES_ERROR(0xC00D10D9)
#define HRES_NS_E_PDA_FAILED_TO_BURN			  HRES_ERROR(0xC00D10DA)
#define HRES_NS_E_FAILED_DOWNLOAD_ABORT_BURN		  HRES_ERROR(0xC00D10DC)
#define HRES_NS_E_WMPCORE_DEVICE_DRIVERS_MISSING	  HRES_ERROR(0xC00D10DD)
#define HRES_NS_E_WMPIM_USEROFFLINE			  HRES_ERROR(0xC00D1126)
#define HRES_NS_E_WMPIM_USERCANCELED			  HRES_ERROR(0xC00D1127)
#define HRES_NS_E_WMPIM_DIALUPFAILED			  HRES_ERROR(0xC00D1128)
#define HRES_NS_E_WINSOCK_ERROR_STRING			  HRES_ERROR(0xC00D1129)
#define HRES_NS_E_WMPBR_NOLISTENER			  HRES_ERROR(0xC00D1130)
#define HRES_NS_E_WMPBR_BACKUPCANCEL			  HRES_ERROR(0xC00D1131)
#define HRES_NS_E_WMPBR_RESTORECANCEL			  HRES_ERROR(0xC00D1132)
#define HRES_NS_E_WMPBR_ERRORWITHURL			  HRES_ERROR(0xC00D1133)
#define HRES_NS_E_WMPBR_NAMECOLLISION			  HRES_ERROR(0xC00D1134)
#define HRES_NS_E_WMPBR_DRIVE_INVALID			  HRES_ERROR(0xC00D1137)
#define HRES_NS_E_WMPBR_BACKUPRESTOREFAILED		  HRES_ERROR(0xC00D1138)
#define HRES_NS_E_WMP_CONVERT_FILE_FAILED		  HRES_ERROR(0xC00D1158)
#define HRES_NS_E_WMP_CONVERT_NO_RIGHTS_ERRORURL	  HRES_ERROR(0xC00D1159)
#define HRES_NS_E_WMP_CONVERT_NO_RIGHTS_NOERRORURL	  HRES_ERROR(0xC00D115A)
#define HRES_NS_E_WMP_CONVERT_FILE_CORRUPT		  HRES_ERROR(0xC00D115B)
#define HRES_NS_E_WMP_CONVERT_PLUGIN_UNAVAILABLE_ERRORURL HRES_ERROR(0xC00D115C)
#define HRES_NS_E_WMP_CONVERT_PLUGIN_UNAVAILABLE_NOERRORURL HRES_ERROR(0xC00D115D)
#define HRES_NS_E_WMP_CONVERT_PLUGIN_UNKNOWN_FILE_OWNER   HRES_ERROR(0xC00D115E)
#define HRES_NS_E_DVD_DISC_COPY_PROTECT_OUTPUT_NS	  HRES_ERROR(0xC00D1160)
#define HRES_NS_E_DVD_DISC_COPY_PROTECT_OUTPUT_FAILED	  HRES_ERROR(0xC00D1161)
#define HRES_NS_E_DVD_NO_SUBPICTURE_STREAM		  HRES_ERROR(0xC00D1162)
#define HRES_NS_E_DVD_COPY_PROTECT			  HRES_ERROR(0xC00D1163)
#define HRES_NS_E_DVD_AUTHORING_PROBLEM			  HRES_ERROR(0xC00D1164)
#define HRES_NS_E_DVD_INVALID_DISC_REGION		  HRES_ERROR(0xC00D1165)
#define HRES_NS_E_DVD_COMPATIBLE_VIDEO_CARD		  HRES_ERROR(0xC00D1166)
#define HRES_NS_E_DVD_MACROVISION			  HRES_ERROR(0xC00D1167)
#define HRES_NS_E_DVD_SYSTEM_DECODER_REGION		  HRES_ERROR(0xC00D1168)
#define HRES_NS_E_DVD_DISC_DECODER_REGION		  HRES_ERROR(0xC00D1169)
#define HRES_NS_E_DVD_NO_VIDEO_STREAM			  HRES_ERROR(0xC00D116A)
#define HRES_NS_E_DVD_NO_AUDIO_STREAM			  HRES_ERROR(0xC00D116B)
#define HRES_NS_E_DVD_GRAPH_BUILDING			  HRES_ERROR(0xC00D116C)
#define HRES_NS_E_DVD_NO_DECODER			  HRES_ERROR(0xC00D116D)
#define HRES_NS_E_DVD_PARENTAL				  HRES_ERROR(0xC00D116E)
#define HRES_NS_E_DVD_CANNOT_JUMP			  HRES_ERROR(0xC00D116F)
#define HRES_NS_E_DVD_DEVICE_CONTENTION			  HRES_ERROR(0xC00D1170)
#define HRES_NS_E_DVD_NO_VIDEO_MEMORY			  HRES_ERROR(0xC00D1171)
#define HRES_NS_E_DVD_CANNOT_COPY_PROTECTED		  HRES_ERROR(0xC00D1172)
#define HRES_NS_E_DVD_REQUIRED_PROPERTY_NOT_SET		  HRES_ERROR(0xC00D1173)
#define HRES_NS_E_DVD_INVALID_TITLE_CHAPTER		  HRES_ERROR(0xC00D1174)
#define HRES_NS_E_NO_CD_BURNER				  HRES_ERROR(0xC00D1176)
#define HRES_NS_E_DEVICE_IS_NOT_READY			  HRES_ERROR(0xC00D1177)
#define HRES_NS_E_PDA_UNSUPPORTED_FORMAT		  HRES_ERROR(0xC00D1178)
#define HRES_NS_E_NO_PDA				  HRES_ERROR(0xC00D1179)
#define HRES_NS_E_PDA_UNSPECIFIED_ERROR			  HRES_ERROR(0xC00D117A)
#define HRES_NS_E_MEMSTORAGE_BAD_DATA			  HRES_ERROR(0xC00D117B)
#define HRES_NS_E_PDA_FAIL_SELECT_DEVICE		  HRES_ERROR(0xC00D117C)
#define HRES_NS_E_PDA_FAIL_READ_WAVE_FILE		  HRES_ERROR(0xC00D117D)
#define HRES_NS_E_IMAPI_LOSSOFSTREAMING			  HRES_ERROR(0xC00D117E)
#define HRES_NS_E_PDA_DEVICE_FULL			  HRES_ERROR(0xC00D117F)
#define HRES_NS_E_FAIL_LAUNCH_ROXIO_PLUGIN		  HRES_ERROR(0xC00D1180)
#define HRES_NS_E_PDA_DEVICE_FULL_IN_SESSION		  HRES_ERROR(0xC00D1181)
#define HRES_NS_E_IMAPI_MEDIUM_INVALIDTYPE		  HRES_ERROR(0xC00D1182)
#define HRES_NS_E_PDA_MANUALDEVICE			  HRES_ERROR(0xC00D1183)
#define HRES_NS_E_PDA_PARTNERSHIPNOTEXIST		  HRES_ERROR(0xC00D1184)
#define HRES_NS_E_PDA_CANNOT_CREATE_ADDITIONAL_SYNC_RELATIONSHIP HRES_ERROR(0xC00D1185)
#define HRES_NS_E_PDA_NO_TRANSCODE_OF_DRM		  HRES_ERROR(0xC00D1186)
#define HRES_NS_E_PDA_TRANSCODECACHEFULL		  HRES_ERROR(0xC00D1187)
#define HRES_NS_E_PDA_TOO_MANY_FILE_COLLISIONS		  HRES_ERROR(0xC00D1188)
#define HRES_NS_E_PDA_CANNOT_TRANSCODE			  HRES_ERROR(0xC00D1189)
#define HRES_NS_E_PDA_TOO_MANY_FILES_IN_DIRECTORY	  HRES_ERROR(0xC00D118A)
#define HRES_NS_E_PROCESSINGSHOWSYNCWIZARD		  HRES_ERROR(0xC00D118B)
#define HRES_NS_E_PDA_TRANSCODE_NOT_PERMITTED		  HRES_ERROR(0xC00D118C)
#define HRES_NS_E_PDA_INITIALIZINGDEVICES		  HRES_ERROR(0xC00D118D)
#define HRES_NS_E_PDA_OBSOLETE_SP			  HRES_ERROR(0xC00D118E)
#define HRES_NS_E_PDA_TITLE_COLLISION			  HRES_ERROR(0xC00D118F)
#define HRES_NS_E_PDA_DEVICESUPPORTDISABLED		  HRES_ERROR(0xC00D1190)
#define HRES_NS_E_PDA_NO_LONGER_AVAILABLE		  HRES_ERROR(0xC00D1191)
#define HRES_NS_E_PDA_ENCODER_NOT_RESPONDING		  HRES_ERROR(0xC00D1192)
#define HRES_NS_E_PDA_CANNOT_SYNC_FROM_LOCATION		  HRES_ERROR(0xC00D1193)
#define HRES_NS_E_WMP_PROTOCOL_PROBLEM			  HRES_ERROR(0xC00D1194)
#define HRES_NS_E_WMP_NO_DISK_SPACE			  HRES_ERROR(0xC00D1195)
#define HRES_NS_E_WMP_LOGON_FAILURE			  HRES_ERROR(0xC00D1196)
#define HRES_NS_E_WMP_CANNOT_FIND_FILE			  HRES_ERROR(0xC00D1197)
#define HRES_NS_E_WMP_SERVER_INACCESSIBLE		  HRES_ERROR(0xC00D1198)
#define HRES_NS_E_WMP_UNSUPPORTED_FORMAT		  HRES_ERROR(0xC00D1199)
#define HRES_NS_E_WMP_DSHOW_UNSUPPORTED_FORMAT		  HRES_ERROR(0xC00D119A)
#define HRES_NS_E_WMP_PLAYLIST_EXISTS			  HRES_ERROR(0xC00D119B)
#define HRES_NS_E_WMP_NONMEDIA_FILES			  HRES_ERROR(0xC00D119C)
#define HRES_NS_E_WMP_INVALID_ASX			  HRES_ERROR(0xC00D119D)
#define HRES_NS_E_WMP_ALREADY_IN_USE			  HRES_ERROR(0xC00D119E)
#define HRES_NS_E_WMP_IMAPI_FAILURE			  HRES_ERROR(0xC00D119F)
#define HRES_NS_E_WMP_WMDM_FAILURE			  HRES_ERROR(0xC00D11A0)
#define HRES_NS_E_WMP_CODEC_NEEDED_WITH_4CC		  HRES_ERROR(0xC00D11A1)
#define HRES_NS_E_WMP_CODEC_NEEDED_WITH_FORMATTAG	  HRES_ERROR(0xC00D11A2)
#define HRES_NS_E_WMP_MSSAP_NOT_AVAILABLE		  HRES_ERROR(0xC00D11A3)
#define HRES_NS_E_WMP_WMDM_INTERFACEDEAD		  HRES_ERROR(0xC00D11A4)
#define HRES_NS_E_WMP_WMDM_NOTCERTIFIED			  HRES_ERROR(0xC00D11A5)
#define HRES_NS_E_WMP_WMDM_LICENSE_NOTEXIST		  HRES_ERROR(0xC00D11A6)
#define HRES_NS_E_WMP_WMDM_LICENSE_EXPIRED		  HRES_ERROR(0xC00D11A7)
#define HRES_NS_E_WMP_WMDM_BUSY				  HRES_ERROR(0xC00D11A8)
#define HRES_NS_E_WMP_WMDM_NORIGHTS			  HRES_ERROR(0xC00D11A9)
#define HRES_NS_E_WMP_WMDM_INCORRECT_RIGHTS		  HRES_ERROR(0xC00D11AA)
#define HRES_NS_E_WMP_IMAPI_GENERIC			  HRES_ERROR(0xC00D11AB)
#define HRES_NS_E_WMP_IMAPI_DEVICE_NOTPRESENT		  HRES_ERROR(0xC00D11AD)
#define HRES_NS_E_WMP_IMAPI_DEVICE_BUSY			  HRES_ERROR(0xC00D11AE)
#define HRES_NS_E_WMP_IMAPI_LOSS_OF_STREAMING		  HRES_ERROR(0xC00D11AF)
#define HRES_NS_E_WMP_SERVER_UNAVAILABLE		  HRES_ERROR(0xC00D11B0)
#define HRES_NS_E_WMP_FILE_OPEN_FAILED			  HRES_ERROR(0xC00D11B1)
#define HRES_NS_E_WMP_VERIFY_ONLINE			  HRES_ERROR(0xC00D11B2)
#define HRES_NS_E_WMP_SERVER_NOT_RESPONDING		  HRES_ERROR(0xC00D11B3)
#define HRES_NS_E_WMP_DRM_CORRUPT_BACKUP		  HRES_ERROR(0xC00D11B4)
#define HRES_NS_E_WMP_DRM_LICENSE_SERVER_UNAVAILABLE	  HRES_ERROR(0xC00D11B5)
#define HRES_NS_E_WMP_NETWORK_FIREWALL			  HRES_ERROR(0xC00D11B6)
#define HRES_NS_E_WMP_NO_REMOVABLE_MEDIA		  HRES_ERROR(0xC00D11B7)
#define HRES_NS_E_WMP_PROXY_CONNECT_TIMEOUT		  HRES_ERROR(0xC00D11B8)
#define HRES_NS_E_WMP_NEED_UPGRADE			  HRES_ERROR(0xC00D11B9)
#define HRES_NS_E_WMP_AUDIO_HW_PROBLEM			  HRES_ERROR(0xC00D11BA)
#define HRES_NS_E_WMP_INVALID_PROTOCOL			  HRES_ERROR(0xC00D11BB)
#define HRES_NS_E_WMP_INVALID_LIBRARY_ADD		  HRES_ERROR(0xC00D11BC)
#define HRES_NS_E_WMP_MMS_NOT_SUPPORTED			  HRES_ERROR(0xC00D11BD)
#define HRES_NS_E_WMP_NO_PROTOCOLS_SELECTED		  HRES_ERROR(0xC00D11BE)
#define HRES_NS_E_WMP_GOFULLSCREEN_FAILED		  HRES_ERROR(0xC00D11BF)
#define HRES_NS_E_WMP_NETWORK_ERROR			  HRES_ERROR(0xC00D11C0)
#define HRES_NS_E_WMP_CONNECT_TIMEOUT			  HRES_ERROR(0xC00D11C1)
#define HRES_NS_E_WMP_MULTICAST_DISABLED		  HRES_ERROR(0xC00D11C2)
#define HRES_NS_E_WMP_SERVER_DNS_TIMEOUT		  HRES_ERROR(0xC00D11C3)
#define HRES_NS_E_WMP_PROXY_NOT_FOUND			  HRES_ERROR(0xC00D11C4)
#define HRES_NS_E_WMP_TAMPERED_CONTENT			  HRES_ERROR(0xC00D11C5)
#define HRES_NS_E_WMP_OUTOFMEMORY			  HRES_ERROR(0xC00D11C6)
#define HRES_NS_E_WMP_AUDIO_CODEC_NOT_INSTALLED		  HRES_ERROR(0xC00D11C7)
#define HRES_NS_E_WMP_VIDEO_CODEC_NOT_INSTALLED		  HRES_ERROR(0xC00D11C8)
#define HRES_NS_E_WMP_IMAPI_DEVICE_INVALIDTYPE		  HRES_ERROR(0xC00D11C9)
#define HRES_NS_E_WMP_DRM_DRIVER_AUTH_FAILURE		  HRES_ERROR(0xC00D11CA)
#define HRES_NS_E_WMP_NETWORK_RESOURCE_FAILURE		  HRES_ERROR(0xC00D11CB)
#define HRES_NS_E_WMP_UPGRADE_APPLICATION		  HRES_ERROR(0xC00D11CC)
#define HRES_NS_E_WMP_UNKNOWN_ERROR			  HRES_ERROR(0xC00D11CD)
#define HRES_NS_E_WMP_INVALID_KEY			  HRES_ERROR(0xC00D11CE)
#define HRES_NS_E_WMP_CD_ANOTHER_USER			  HRES_ERROR(0xC00D11CF)
#define HRES_NS_E_WMP_DRM_NEEDS_AUTHORIZATION		  HRES_ERROR(0xC00D11D0)
#define HRES_NS_E_WMP_BAD_DRIVER			  HRES_ERROR(0xC00D11D1)
#define HRES_NS_E_WMP_ACCESS_DENIED			  HRES_ERROR(0xC00D11D2)
#define HRES_NS_E_WMP_LICENSE_RESTRICTS			  HRES_ERROR(0xC00D11D3)
#define HRES_NS_E_WMP_INVALID_REQUEST			  HRES_ERROR(0xC00D11D4)
#define HRES_NS_E_WMP_CD_STASH_NO_SPACE			  HRES_ERROR(0xC00D11D5)
#define HRES_NS_E_WMP_DRM_NEW_HARDWARE			  HRES_ERROR(0xC00D11D6)
#define HRES_NS_E_WMP_DRM_INVALID_SIG			  HRES_ERROR(0xC00D11D7)
#define HRES_NS_E_WMP_DRM_CANNOT_RESTORE		  HRES_ERROR(0xC00D11D8)
#define HRES_NS_E_WMP_BURN_DISC_OVERFLOW		  HRES_ERROR(0xC00D11D9)
#define HRES_NS_E_WMP_DRM_GENERIC_LICENSE_FAILURE	  HRES_ERROR(0xC00D11DA)
#define HRES_NS_E_WMP_DRM_NO_SECURE_CLOCK		  HRES_ERROR(0xC00D11DB)
#define HRES_NS_E_WMP_DRM_NO_RIGHTS			  HRES_ERROR(0xC00D11DC)
#define HRES_NS_E_WMP_DRM_INDIV_FAILED			  HRES_ERROR(0xC00D11DD)
#define HRES_NS_E_WMP_SERVER_NONEWCONNECTIONS		  HRES_ERROR(0xC00D11DE)
#define HRES_NS_E_WMP_MULTIPLE_ERROR_IN_PLAYLIST	  HRES_ERROR(0xC00D11DF)
#define HRES_NS_E_WMP_IMAPI2_ERASE_FAIL			  HRES_ERROR(0xC00D11E0)
#define HRES_NS_E_WMP_IMAPI2_ERASE_DEVICE_BUSY		  HRES_ERROR(0xC00D11E1)
#define HRES_NS_E_WMP_DRM_COMPONENT_FAILURE		  HRES_ERROR(0xC00D11E2)
#define HRES_NS_E_WMP_DRM_NO_DEVICE_CERT		  HRES_ERROR(0xC00D11E3)
#define HRES_NS_E_WMP_SERVER_SECURITY_ERROR		  HRES_ERROR(0xC00D11E4)
#define HRES_NS_E_WMP_AUDIO_DEVICE_LOST			  HRES_ERROR(0xC00D11E5)
#define HRES_NS_E_WMP_IMAPI_MEDIA_INCOMPATIBLE		  HRES_ERROR(0xC00D11E6)
#define HRES_NS_E_SYNCWIZ_DEVICE_FULL			  HRES_ERROR(0xC00D11EE)
#define HRES_NS_E_SYNCWIZ_CANNOT_CHANGE_SETTINGS	  HRES_ERROR(0xC00D11EF)
#define HRES_NS_E_TRANSCODE_DELETECACHEERROR		  HRES_ERROR(0xC00D11F0)
#define HRES_NS_E_CD_NO_BUFFERS_READ			  HRES_ERROR(0xC00D11F8)
#define HRES_NS_E_CD_EMPTY_TRACK_QUEUE			  HRES_ERROR(0xC00D11F9)
#define HRES_NS_E_CD_NO_READER				  HRES_ERROR(0xC00D11FA)
#define HRES_NS_E_CD_ISRC_INVALID			  HRES_ERROR(0xC00D11FB)
#define HRES_NS_E_CD_MEDIA_CATALOG_NUMBER_INVALID	  HRES_ERROR(0xC00D11FC)
#define HRES_NS_E_SLOW_READ_DIGITAL_WITH_ERRORCORRECTION  HRES_ERROR(0xC00D11FD)
#define HRES_NS_E_CD_SPEEDDETECT_NOT_ENOUGH_READS	  HRES_ERROR(0xC00D11FE)
#define HRES_NS_E_CD_QUEUEING_DISABLED			  HRES_ERROR(0xC00D11FF)
#define HRES_NS_E_WMP_DRM_ACQUIRING_LICENSE		  HRES_ERROR(0xC00D1202)
#define HRES_NS_E_WMP_DRM_LICENSE_EXPIRED		  HRES_ERROR(0xC00D1203)
#define HRES_NS_E_WMP_DRM_LICENSE_NOTACQUIRED		  HRES_ERROR(0xC00D1204)
#define HRES_NS_E_WMP_DRM_LICENSE_NOTENABLED		  HRES_ERROR(0xC00D1205)
#define HRES_NS_E_WMP_DRM_LICENSE_UNUSABLE		  HRES_ERROR(0xC00D1206)
#define HRES_NS_E_WMP_DRM_LICENSE_CONTENT_REVOKED	  HRES_ERROR(0xC00D1207)
#define HRES_NS_E_WMP_DRM_LICENSE_NOSAP			  HRES_ERROR(0xC00D1208)
#define HRES_NS_E_WMP_DRM_UNABLE_TO_ACQUIRE_LICENSE	  HRES_ERROR(0xC00D1209)
#define HRES_NS_E_WMP_LICENSE_REQUIRED			  HRES_ERROR(0xC00D120A)
#define HRES_NS_E_WMP_PROTECTED_CONTENT			  HRES_ERROR(0xC00D120B)
#define HRES_NS_E_WMP_POLICY_VALUE_NOT_CONFIGURED	  HRES_ERROR(0xC00D122A)
#define HRES_NS_E_PDA_CANNOT_SYNC_FROM_INTERNET		  HRES_ERROR(0xC00D1234)
#define HRES_NS_E_PDA_CANNOT_SYNC_INVALID_PLAYLIST	  HRES_ERROR(0xC00D1235)
#define HRES_NS_E_PDA_FAILED_TO_SYNCHRONIZE_FILE	  HRES_ERROR(0xC00D1236)
#define HRES_NS_E_PDA_SYNC_FAILED			  HRES_ERROR(0xC00D1237)
#define HRES_NS_E_PDA_DELETE_FAILED			  HRES_ERROR(0xC00D1238)
#define HRES_NS_E_PDA_FAILED_TO_RETRIEVE_FILE		  HRES_ERROR(0xC00D1239)
#define HRES_NS_E_PDA_DEVICE_NOT_RESPONDING		  HRES_ERROR(0xC00D123A)
#define HRES_NS_E_PDA_FAILED_TO_TRANSCODE_PHOTO		  HRES_ERROR(0xC00D123B)
#define HRES_NS_E_PDA_FAILED_TO_ENCRYPT_TRANSCODED_FILE   HRES_ERROR(0xC00D123C)
#define HRES_NS_E_PDA_CANNOT_TRANSCODE_TO_AUDIO		  HRES_ERROR(0xC00D123D)
#define HRES_NS_E_PDA_CANNOT_TRANSCODE_TO_VIDEO		  HRES_ERROR(0xC00D123E)
#define HRES_NS_E_PDA_CANNOT_TRANSCODE_TO_IMAGE		  HRES_ERROR(0xC00D123F)
#define HRES_NS_E_PDA_RETRIEVED_FILE_FILENAME_TOO_LONG	  HRES_ERROR(0xC00D1240)
#define HRES_NS_E_PDA_CEWMDM_DRM_ERROR			  HRES_ERROR(0xC00D1241)
#define HRES_NS_E_INCOMPLETE_PLAYLIST			  HRES_ERROR(0xC00D1242)
#define HRES_NS_E_PDA_SYNC_RUNNING			  HRES_ERROR(0xC00D1243)
#define HRES_NS_E_PDA_SYNC_LOGIN_ERROR			  HRES_ERROR(0xC00D1244)
#define HRES_NS_E_PDA_TRANSCODE_CODEC_NOT_FOUND		  HRES_ERROR(0xC00D1245)
#define HRES_NS_E_CANNOT_SYNC_DRM_TO_NON_JANUS_DEVICE	  HRES_ERROR(0xC00D1246)
#define HRES_NS_E_CANNOT_SYNC_PREVIOUS_SYNC_RUNNING	  HRES_ERROR(0xC00D1247)
#define HRES_NS_E_WMP_HWND_NOTFOUND			  HRES_ERROR(0xC00D125C)
#define HRES_NS_E_BKGDOWNLOAD_WRONG_NO_FILES		  HRES_ERROR(0xC00D125D)
#define HRES_NS_E_BKGDOWNLOAD_COMPLETECANCELLEDJOB	  HRES_ERROR(0xC00D125E)
#define HRES_NS_E_BKGDOWNLOAD_CANCELCOMPLETEDJOB	  HRES_ERROR(0xC00D125F)
#define HRES_NS_E_BKGDOWNLOAD_NOJOBPOINTER		  HRES_ERROR(0xC00D1260)
#define HRES_NS_E_BKGDOWNLOAD_INVALIDJOBSIGNATURE	  HRES_ERROR(0xC00D1261)
#define HRES_NS_E_BKGDOWNLOAD_FAILED_TO_CREATE_TEMPFILE   HRES_ERROR(0xC00D1262)
#define HRES_NS_E_BKGDOWNLOAD_PLUGIN_FAILEDINITIALIZE	  HRES_ERROR(0xC00D1263)
#define HRES_NS_E_BKGDOWNLOAD_PLUGIN_FAILEDTOMOVEFILE	  HRES_ERROR(0xC00D1264)
#define HRES_NS_E_BKGDOWNLOAD_CALLFUNCFAILED		  HRES_ERROR(0xC00D1265)
#define HRES_NS_E_BKGDOWNLOAD_CALLFUNCTIMEOUT		  HRES_ERROR(0xC00D1266)
#define HRES_NS_E_BKGDOWNLOAD_CALLFUNCENDED		  HRES_ERROR(0xC00D1267)
#define HRES_NS_E_BKGDOWNLOAD_WMDUNPACKFAILED		  HRES_ERROR(0xC00D1268)
#define HRES_NS_E_BKGDOWNLOAD_FAILEDINITIALIZE		  HRES_ERROR(0xC00D1269)
#define HRES_NS_E_INTERFACE_NOT_REGISTERED_IN_GIT	  HRES_ERROR(0xC00D126A)
#define HRES_NS_E_BKGDOWNLOAD_INVALID_FILE_NAME		  HRES_ERROR(0xC00D126B)
#define HRES_NS_E_IMAGE_DOWNLOAD_FAILED			  HRES_ERROR(0xC00D128E)
#define HRES_NS_E_WMP_UDRM_NOUSERLIST			  HRES_ERROR(0xC00D12C0)
#define HRES_NS_E_WMP_DRM_NOT_ACQUIRING			  HRES_ERROR(0xC00D12C1)
#define HRES_NS_E_WMP_BSTR_TOO_LONG			  HRES_ERROR(0xC00D12F2)
#define HRES_NS_E_WMP_AUTOPLAY_INVALID_STATE		  HRES_ERROR(0xC00D12FC)
#define HRES_NS_E_WMP_COMPONENT_REVOKED			  HRES_ERROR(0xC00D1306)
#define HRES_NS_E_CURL_NOTSAFE				  HRES_ERROR(0xC00D1324)
#define HRES_NS_E_CURL_INVALIDCHAR			  HRES_ERROR(0xC00D1325)
#define HRES_NS_E_CURL_INVALIDHOSTNAME			  HRES_ERROR(0xC00D1326)
#define HRES_NS_E_CURL_INVALIDPATH			  HRES_ERROR(0xC00D1327)
#define HRES_NS_E_CURL_INVALIDSCHEME			  HRES_ERROR(0xC00D1328)
#define HRES_NS_E_CURL_INVALIDURL			  HRES_ERROR(0xC00D1329)
#define HRES_NS_E_CURL_CANTWALK				  HRES_ERROR(0xC00D132B)
#define HRES_NS_E_CURL_INVALIDPORT			  HRES_ERROR(0xC00D132C)
#define HRES_NS_E_CURLHELPER_NOTADIRECTORY		  HRES_ERROR(0xC00D132D)
#define HRES_NS_E_CURLHELPER_NOTAFILE			  HRES_ERROR(0xC00D132E)
#define HRES_NS_E_CURL_CANTDECODE			  HRES_ERROR(0xC00D132F)
#define HRES_NS_E_CURLHELPER_NOTRELATIVE		  HRES_ERROR(0xC00D1330)
#define HRES_NS_E_CURL_INVALIDBUFFERSIZE		  HRES_ERROR(0xC00D1331)
#define HRES_NS_E_SUBSCRIPTIONSERVICE_PLAYBACK_DISALLOWED HRES_ERROR(0xC00D1356)
#define HRES_NS_E_CANNOT_BUY_OR_DOWNLOAD_FROM_MULTIPLE_SERVICES HRES_ERROR(0xC00D1357)
#define HRES_NS_E_CANNOT_BUY_OR_DOWNLOAD_CONTENT	  HRES_ERROR(0xC00D1358)
#define HRES_NS_E_NOT_CONTENT_PARTNER_TRACK		  HRES_ERROR(0xC00D135A)
#define HRES_NS_E_TRACK_DOWNLOAD_REQUIRES_ALBUM_PURCHASE  HRES_ERROR(0xC00D135B)
#define HRES_NS_E_TRACK_DOWNLOAD_REQUIRES_PURCHASE	  HRES_ERROR(0xC00D135C)
#define HRES_NS_E_TRACK_PURCHASE_MAXIMUM_EXCEEDED	  HRES_ERROR(0xC00D135D)
#define HRES_NS_E_SUBSCRIPTIONSERVICE_LOGIN_FAILED	  HRES_ERROR(0xC00D135F)
#define HRES_NS_E_SUBSCRIPTIONSERVICE_DOWNLOAD_TIMEOUT	  HRES_ERROR(0xC00D1360)
#define HRES_NS_E_CONTENT_PARTNER_STILL_INITIALIZING	  HRES_ERROR(0xC00D1362)
#define HRES_NS_E_OPEN_CONTAINING_FOLDER_FAILED		  HRES_ERROR(0xC00D1363)
#define HRES_NS_E_ADVANCEDEDIT_TOO_MANY_PICTURES	  HRES_ERROR(0xC00D136A)
#define HRES_NS_E_REDIRECT				  HRES_ERROR(0xC00D1388)
#define HRES_NS_E_STALE_PRESENTATION			  HRES_ERROR(0xC00D1389)
#define HRES_NS_E_NAMESPACE_WRONG_PERSIST		  HRES_ERROR(0xC00D138A)
#define HRES_NS_E_NAMESPACE_WRONG_TYPE			  HRES_ERROR(0xC00D138B)
#define HRES_NS_E_NAMESPACE_NODE_CONFLICT		  HRES_ERROR(0xC00D138C)
#define HRES_NS_E_NAMESPACE_NODE_NOT_FOUND		  HRES_ERROR(0xC00D138D)
#define HRES_NS_E_NAMESPACE_BUFFER_TOO_SMALL		  HRES_ERROR(0xC00D138E)
#define HRES_NS_E_NAMESPACE_TOO_MANY_CALLBACKS		  HRES_ERROR(0xC00D138F)
#define HRES_NS_E_NAMESPACE_DUPLICATE_CALLBACK		  HRES_ERROR(0xC00D1390)
#define HRES_NS_E_NAMESPACE_CALLBACK_NOT_FOUND		  HRES_ERROR(0xC00D1391)
#define HRES_NS_E_NAMESPACE_NAME_TOO_LONG		  HRES_ERROR(0xC00D1392)
#define HRES_NS_E_NAMESPACE_DUPLICATE_NAME		  HRES_ERROR(0xC00D1393)
#define HRES_NS_E_NAMESPACE_EMPTY_NAME			  HRES_ERROR(0xC00D1394)
#define HRES_NS_E_NAMESPACE_INDEX_TOO_LARGE		  HRES_ERROR(0xC00D1395)
#define HRES_NS_E_NAMESPACE_BAD_NAME			  HRES_ERROR(0xC00D1396)
#define HRES_NS_E_NAMESPACE_WRONG_SECURITY		  HRES_ERROR(0xC00D1397)
#define HRES_NS_E_CACHE_ARCHIVE_CONFLICT		  HRES_ERROR(0xC00D13EC)
#define HRES_NS_E_CACHE_ORIGIN_SERVER_NOT_FOUND		  HRES_ERROR(0xC00D13ED)
#define HRES_NS_E_CACHE_ORIGIN_SERVER_TIMEOUT		  HRES_ERROR(0xC00D13EE)
#define HRES_NS_E_CACHE_NOT_BROADCAST			  HRES_ERROR(0xC00D13EF)
#define HRES_NS_E_CACHE_CANNOT_BE_CACHED		  HRES_ERROR(0xC00D13F0)
#define HRES_NS_E_CACHE_NOT_MODIFIED			  HRES_ERROR(0xC00D13F1)
#define HRES_NS_E_CANNOT_REMOVE_PUBLISHING_POINT	  HRES_ERROR(0xC00D1450)
#define HRES_NS_E_CANNOT_REMOVE_PLUGIN			  HRES_ERROR(0xC00D1451)
#define HRES_NS_E_WRONG_PUBLISHING_POINT_TYPE		  HRES_ERROR(0xC00D1452)
#define HRES_NS_E_UNSUPPORTED_LOAD_TYPE			  HRES_ERROR(0xC00D1453)
#define HRES_NS_E_INVALID_PLUGIN_LOAD_TYPE_CONFIGURATION  HRES_ERROR(0xC00D1454)
#define HRES_NS_E_INVALID_PUBLISHING_POINT_NAME		  HRES_ERROR(0xC00D1455)
#define HRES_NS_E_TOO_MANY_MULTICAST_SINKS		  HRES_ERROR(0xC00D1456)
#define HRES_NS_E_PUBLISHING_POINT_INVALID_REQUEST_WHILE_STARTED HRES_ERROR(0xC00D1457)
#define HRES_NS_E_MULTICAST_PLUGIN_NOT_ENABLED		  HRES_ERROR(0xC00D1458)
#define HRES_NS_E_INVALID_OPERATING_SYSTEM_VERSION	  HRES_ERROR(0xC00D1459)
#define HRES_NS_E_PUBLISHING_POINT_REMOVED		  HRES_ERROR(0xC00D145A)
#define HRES_NS_E_INVALID_PUSH_PUBLISHING_POINT_START_REQUEST HRES_ERROR(0xC00D145B)
#define HRES_NS_E_UNSUPPORTED_LANGUAGE			  HRES_ERROR(0xC00D145C)
#define HRES_NS_E_WRONG_OS_VERSION			  HRES_ERROR(0xC00D145D)
#define HRES_NS_E_PUBLISHING_POINT_STOPPED		  HRES_ERROR(0xC00D145E)
#define HRES_NS_E_PLAYLIST_ENTRY_ALREADY_PLAYING	  HRES_ERROR(0xC00D14B4)
#define HRES_NS_E_EMPTY_PLAYLIST			  HRES_ERROR(0xC00D14B5)
#define HRES_NS_E_PLAYLIST_PARSE_FAILURE		  HRES_ERROR(0xC00D14B6)
#define HRES_NS_E_PLAYLIST_UNSUPPORTED_ENTRY		  HRES_ERROR(0xC00D14B7)
#define HRES_NS_E_PLAYLIST_ENTRY_NOT_IN_PLAYLIST	  HRES_ERROR(0xC00D14B8)
#define HRES_NS_E_PLAYLIST_ENTRY_SEEK			  HRES_ERROR(0xC00D14B9)
#define HRES_NS_E_PLAYLIST_RECURSIVE_PLAYLISTS		  HRES_ERROR(0xC00D14BA)
#define HRES_NS_E_PLAYLIST_TOO_MANY_NESTED_PLAYLISTS	  HRES_ERROR(0xC00D14BB)
#define HRES_NS_E_PLAYLIST_SHUTDOWN			  HRES_ERROR(0xC00D14BC)
#define HRES_NS_E_PLAYLIST_END_RECEDING			  HRES_ERROR(0xC00D14BD)
#define HRES_NS_E_DATAPATH_NO_SINK			  HRES_ERROR(0xC00D1518)
#define HRES_NS_E_INVALID_PUSH_TEMPLATE			  HRES_ERROR(0xC00D151A)
#define HRES_NS_E_INVALID_PUSH_PUBLISHING_POINT		  HRES_ERROR(0xC00D151B)
#define HRES_NS_E_CRITICAL_ERROR			  HRES_ERROR(0xC00D151C)
#define HRES_NS_E_NO_NEW_CONNECTIONS			  HRES_ERROR(0xC00D151D)
#define HRES_NS_E_WSX_INVALID_VERSION			  HRES_ERROR(0xC00D151E)
#define HRES_NS_E_HEADER_MISMATCH			  HRES_ERROR(0xC00D151F)
#define HRES_NS_E_PUSH_DUPLICATE_PUBLISHING_POINT_NAME	  HRES_ERROR(0xC00D1520)
#define HRES_NS_E_NO_SCRIPT_ENGINE			  HRES_ERROR(0xC00D157C)
#define HRES_NS_E_PLUGIN_ERROR_REPORTED			  HRES_ERROR(0xC00D157D)
#define HRES_NS_E_SOURCE_PLUGIN_NOT_FOUND		  HRES_ERROR(0xC00D157E)
#define HRES_NS_E_PLAYLIST_PLUGIN_NOT_FOUND		  HRES_ERROR(0xC00D157F)
#define HRES_NS_E_DATA_SOURCE_ENUMERATION_NOT_SUPPORTED   HRES_ERROR(0xC00D1580)
#define HRES_NS_E_MEDIA_PARSER_INVALID_FORMAT		  HRES_ERROR(0xC00D1581)
#define HRES_NS_E_SCRIPT_DEBUGGER_NOT_INSTALLED		  HRES_ERROR(0xC00D1582)
#define HRES_NS_E_FEATURE_REQUIRES_ENTERPRISE_SERVER	  HRES_ERROR(0xC00D1583)
#define HRES_NS_E_WIZARD_RUNNING			  HRES_ERROR(0xC00D1584)
#define HRES_NS_E_INVALID_LOG_URL			  HRES_ERROR(0xC00D1585)
#define HRES_NS_E_INVALID_MTU_RANGE			  HRES_ERROR(0xC00D1586)
#define HRES_NS_E_INVALID_PLAY_STATISTICS		  HRES_ERROR(0xC00D1587)
#define HRES_NS_E_LOG_NEED_TO_BE_SKIPPED		  HRES_ERROR(0xC00D1588)
#define HRES_NS_E_HTTP_TEXT_DATACONTAINER_SIZE_LIMIT_EXCEEDED HRES_ERROR(0xC00D1589)
#define HRES_NS_E_PORT_IN_USE				  HRES_ERROR(0xC00D158A)
#define HRES_NS_E_PORT_IN_USE_HTTP			  HRES_ERROR(0xC00D158B)
#define HRES_NS_E_HTTP_TEXT_DATACONTAINER_INVALID_SERVER_RESPONSE HRES_ERROR(0xC00D158C)
#define HRES_NS_E_ARCHIVE_REACH_QUOTA			  HRES_ERROR(0xC00D158D)
#define HRES_NS_E_ARCHIVE_ABORT_DUE_TO_BCAST		  HRES_ERROR(0xC00D158E)
#define HRES_NS_E_ARCHIVE_GAP_DETECTED			  HRES_ERROR(0xC00D158F)
#define HRES_NS_E_AUTHORIZATION_FILE_NOT_FOUND		  HRES_ERROR(0xC00D1590)
#define HRES_NS_E_BAD_MARKIN				  HRES_ERROR(0xC00D1B58)
#define HRES_NS_E_BAD_MARKOUT				  HRES_ERROR(0xC00D1B59)
#define HRES_NS_E_NOMATCHING_MEDIASOURCE		  HRES_ERROR(0xC00D1B5A)
#define HRES_NS_E_UNSUPPORTED_SOURCETYPE		  HRES_ERROR(0xC00D1B5B)
#define HRES_NS_E_TOO_MANY_AUDIO			  HRES_ERROR(0xC00D1B5C)
#define HRES_NS_E_TOO_MANY_VIDEO			  HRES_ERROR(0xC00D1B5D)
#define HRES_NS_E_NOMATCHING_ELEMENT			  HRES_ERROR(0xC00D1B5E)
#define HRES_NS_E_MISMATCHED_MEDIACONTENT		  HRES_ERROR(0xC00D1B5F)
#define HRES_NS_E_CANNOT_DELETE_ACTIVE_SOURCEGROUP	  HRES_ERROR(0xC00D1B60)
#define HRES_NS_E_AUDIODEVICE_BUSY			  HRES_ERROR(0xC00D1B61)
#define HRES_NS_E_AUDIODEVICE_UNEXPECTED		  HRES_ERROR(0xC00D1B62)
#define HRES_NS_E_AUDIODEVICE_BADFORMAT			  HRES_ERROR(0xC00D1B63)
#define HRES_NS_E_VIDEODEVICE_BUSY			  HRES_ERROR(0xC00D1B64)
#define HRES_NS_E_VIDEODEVICE_UNEXPECTED		  HRES_ERROR(0xC00D1B65)
#define HRES_NS_E_INVALIDCALL_WHILE_ENCODER_RUNNING	  HRES_ERROR(0xC00D1B66)
#define HRES_NS_E_NO_PROFILE_IN_SOURCEGROUP		  HRES_ERROR(0xC00D1B67)
#define HRES_NS_E_VIDEODRIVER_UNSTABLE			  HRES_ERROR(0xC00D1B68)
#define HRES_NS_E_VIDCAPSTARTFAILED			  HRES_ERROR(0xC00D1B69)
#define HRES_NS_E_VIDSOURCECOMPRESSION			  HRES_ERROR(0xC00D1B6A)
#define HRES_NS_E_VIDSOURCESIZE				  HRES_ERROR(0xC00D1B6B)
#define HRES_NS_E_ICMQUERYFORMAT			  HRES_ERROR(0xC00D1B6C)
#define HRES_NS_E_VIDCAPCREATEWINDOW			  HRES_ERROR(0xC00D1B6D)
#define HRES_NS_E_VIDCAPDRVINUSE			  HRES_ERROR(0xC00D1B6E)
#define HRES_NS_E_NO_MEDIAFORMAT_IN_SOURCE		  HRES_ERROR(0xC00D1B6F)
#define HRES_NS_E_NO_VALID_OUTPUT_STREAM		  HRES_ERROR(0xC00D1B70)
#define HRES_NS_E_NO_VALID_SOURCE_PLUGIN		  HRES_ERROR(0xC00D1B71)
#define HRES_NS_E_NO_ACTIVE_SOURCEGROUP			  HRES_ERROR(0xC00D1B72)
#define HRES_NS_E_NO_SCRIPT_STREAM			  HRES_ERROR(0xC00D1B73)
#define HRES_NS_E_INVALIDCALL_WHILE_ARCHIVAL_RUNNING	  HRES_ERROR(0xC00D1B74)
#define HRES_NS_E_INVALIDPACKETSIZE			  HRES_ERROR(0xC00D1B75)
#define HRES_NS_E_PLUGIN_CLSID_INVALID			  HRES_ERROR(0xC00D1B76)
#define HRES_NS_E_UNSUPPORTED_ARCHIVETYPE		  HRES_ERROR(0xC00D1B77)
#define HRES_NS_E_UNSUPPORTED_ARCHIVEOPERATION		  HRES_ERROR(0xC00D1B78)
#define HRES_NS_E_ARCHIVE_FILENAME_NOTSET		  HRES_ERROR(0xC00D1B79)
#define HRES_NS_E_SOURCEGROUP_NOTPREPARED		  HRES_ERROR(0xC00D1B7A)
#define HRES_NS_E_PROFILE_MISMATCH			  HRES_ERROR(0xC00D1B7B)
#define HRES_NS_E_INCORRECTCLIPSETTINGS			  HRES_ERROR(0xC00D1B7C)
#define HRES_NS_E_NOSTATSAVAILABLE			  HRES_ERROR(0xC00D1B7D)
#define HRES_NS_E_NOTARCHIVING				  HRES_ERROR(0xC00D1B7E)
#define HRES_NS_E_INVALIDCALL_WHILE_ENCODER_STOPPED	  HRES_ERROR(0xC00D1B7F)
#define HRES_NS_E_NOSOURCEGROUPS			  HRES_ERROR(0xC00D1B80)
#define HRES_NS_E_INVALIDINPUTFPS			  HRES_ERROR(0xC00D1B81)
#define HRES_NS_E_NO_DATAVIEW_SUPPORT			  HRES_ERROR(0xC00D1B82)
#define HRES_NS_E_CODEC_UNAVAILABLE			  HRES_ERROR(0xC00D1B83)
#define HRES_NS_E_ARCHIVE_SAME_AS_INPUT			  HRES_ERROR(0xC00D1B84)
#define HRES_NS_E_SOURCE_NOTSPECIFIED			  HRES_ERROR(0xC00D1B85)
#define HRES_NS_E_NO_REALTIME_TIMECOMPRESSION		  HRES_ERROR(0xC00D1B86)
#define HRES_NS_E_UNSUPPORTED_ENCODER_DEVICE		  HRES_ERROR(0xC00D1B87)
#define HRES_NS_E_UNEXPECTED_DISPLAY_SETTINGS		  HRES_ERROR(0xC00D1B88)
#define HRES_NS_E_NO_AUDIODATA				  HRES_ERROR(0xC00D1B89)
#define HRES_NS_E_INPUTSOURCE_PROBLEM			  HRES_ERROR(0xC00D1B8A)
#define HRES_NS_E_WME_VERSION_MISMATCH			  HRES_ERROR(0xC00D1B8B)
#define HRES_NS_E_NO_REALTIME_PREPROCESS		  HRES_ERROR(0xC00D1B8C)
#define HRES_NS_E_NO_REPEAT_PREPROCESS			  HRES_ERROR(0xC00D1B8D)
#define HRES_NS_E_CANNOT_PAUSE_LIVEBROADCAST		  HRES_ERROR(0xC00D1B8E)
#define HRES_NS_E_DRM_PROFILE_NOT_SET			  HRES_ERROR(0xC00D1B8F)
#define HRES_NS_E_DUPLICATE_DRMPROFILE			  HRES_ERROR(0xC00D1B90)
#define HRES_NS_E_INVALID_DEVICE			  HRES_ERROR(0xC00D1B91)
#define HRES_NS_E_SPEECHEDL_ON_NON_MIXEDMODE		  HRES_ERROR(0xC00D1B92)
#define HRES_NS_E_DRM_PASSWORD_TOO_LONG			  HRES_ERROR(0xC00D1B93)
#define HRES_NS_E_DEVCONTROL_FAILED_SEEK		  HRES_ERROR(0xC00D1B94)
#define HRES_NS_E_INTERLACE_REQUIRE_SAMESIZE		  HRES_ERROR(0xC00D1B95)
#define HRES_NS_E_TOO_MANY_DEVICECONTROL		  HRES_ERROR(0xC00D1B96)
#define HRES_NS_E_NO_MULTIPASS_FOR_LIVEDEVICE		  HRES_ERROR(0xC00D1B97)
#define HRES_NS_E_MISSING_AUDIENCE			  HRES_ERROR(0xC00D1B98)
#define HRES_NS_E_AUDIENCE_CONTENTTYPE_MISMATCH		  HRES_ERROR(0xC00D1B99)
#define HRES_NS_E_MISSING_SOURCE_INDEX			  HRES_ERROR(0xC00D1B9A)
#define HRES_NS_E_NUM_LANGUAGE_MISMATCH			  HRES_ERROR(0xC00D1B9B)
#define HRES_NS_E_LANGUAGE_MISMATCH			  HRES_ERROR(0xC00D1B9C)
#define HRES_NS_E_VBRMODE_MISMATCH			  HRES_ERROR(0xC00D1B9D)
#define HRES_NS_E_INVALID_INPUT_AUDIENCE_INDEX		  HRES_ERROR(0xC00D1B9E)
#define HRES_NS_E_INVALID_INPUT_LANGUAGE		  HRES_ERROR(0xC00D1B9F)
#define HRES_NS_E_INVALID_INPUT_STREAM			  HRES_ERROR(0xC00D1BA0)
#define HRES_NS_E_EXPECT_MONO_WAV_INPUT			  HRES_ERROR(0xC00D1BA1)
#define HRES_NS_E_INPUT_WAVFORMAT_MISMATCH		  HRES_ERROR(0xC00D1BA2)
#define HRES_NS_E_RECORDQ_DISK_FULL			  HRES_ERROR(0xC00D1BA3)
#define HRES_NS_E_NO_PAL_INVERSE_TELECINE		  HRES_ERROR(0xC00D1BA4)
#define HRES_NS_E_ACTIVE_SG_DEVICE_DISCONNECTED		  HRES_ERROR(0xC00D1BA5)
#define HRES_NS_E_ACTIVE_SG_DEVICE_CONTROL_DISCONNECTED   HRES_ERROR(0xC00D1BA6)
#define HRES_NS_E_NO_FRAMES_SUBMITTED_TO_ANALYZER	  HRES_ERROR(0xC00D1BA7)
#define HRES_NS_E_INPUT_DOESNOT_SUPPORT_SMPTE		  HRES_ERROR(0xC00D1BA8)
#define HRES_NS_E_NO_SMPTE_WITH_MULTIPLE_SOURCEGROUPS	  HRES_ERROR(0xC00D1BA9)
#define HRES_NS_E_BAD_CONTENTEDL			  HRES_ERROR(0xC00D1BAA)
#define HRES_NS_E_INTERLACEMODE_MISMATCH		  HRES_ERROR(0xC00D1BAB)
#define HRES_NS_E_NONSQUAREPIXELMODE_MISMATCH		  HRES_ERROR(0xC00D1BAC)
#define HRES_NS_E_SMPTEMODE_MISMATCH			  HRES_ERROR(0xC00D1BAD)
#define HRES_NS_E_END_OF_TAPE				  HRES_ERROR(0xC00D1BAE)
#define HRES_NS_E_NO_MEDIA_IN_AUDIENCE			  HRES_ERROR(0xC00D1BAF)
#define HRES_NS_E_NO_AUDIENCES				  HRES_ERROR(0xC00D1BB0)
#define HRES_NS_E_NO_AUDIO_COMPAT			  HRES_ERROR(0xC00D1BB1)
#define HRES_NS_E_INVALID_VBR_COMPAT			  HRES_ERROR(0xC00D1BB2)
#define HRES_NS_E_NO_PROFILE_NAME			  HRES_ERROR(0xC00D1BB3)
#define HRES_NS_E_INVALID_VBR_WITH_UNCOMP		  HRES_ERROR(0xC00D1BB4)
#define HRES_NS_E_MULTIPLE_VBR_AUDIENCES		  HRES_ERROR(0xC00D1BB5)
#define HRES_NS_E_UNCOMP_COMP_COMBINATION		  HRES_ERROR(0xC00D1BB6)
#define HRES_NS_E_MULTIPLE_AUDIO_CODECS			  HRES_ERROR(0xC00D1BB7)
#define HRES_NS_E_MULTIPLE_AUDIO_FORMATS		  HRES_ERROR(0xC00D1BB8)
#define HRES_NS_E_AUDIO_BITRATE_STEPDOWN		  HRES_ERROR(0xC00D1BB9)
#define HRES_NS_E_INVALID_AUDIO_PEAKRATE		  HRES_ERROR(0xC00D1BBA)
#define HRES_NS_E_INVALID_AUDIO_PEAKRATE_2		  HRES_ERROR(0xC00D1BBB)
#define HRES_NS_E_INVALID_AUDIO_BUFFERMAX		  HRES_ERROR(0xC00D1BBC)
#define HRES_NS_E_MULTIPLE_VIDEO_CODECS			  HRES_ERROR(0xC00D1BBD)
#define HRES_NS_E_MULTIPLE_VIDEO_SIZES			  HRES_ERROR(0xC00D1BBE)
#define HRES_NS_E_INVALID_VIDEO_BITRATE			  HRES_ERROR(0xC00D1BBF)
#define HRES_NS_E_VIDEO_BITRATE_STEPDOWN		  HRES_ERROR(0xC00D1BC0)
#define HRES_NS_E_INVALID_VIDEO_PEAKRATE		  HRES_ERROR(0xC00D1BC1)
#define HRES_NS_E_INVALID_VIDEO_PEAKRATE_2		  HRES_ERROR(0xC00D1BC2)
#define HRES_NS_E_INVALID_VIDEO_WIDTH			  HRES_ERROR(0xC00D1BC3)
#define HRES_NS_E_INVALID_VIDEO_HEIGHT			  HRES_ERROR(0xC00D1BC4)
#define HRES_NS_E_INVALID_VIDEO_FPS			  HRES_ERROR(0xC00D1BC5)
#define HRES_NS_E_INVALID_VIDEO_KEYFRAME		  HRES_ERROR(0xC00D1BC6)
#define HRES_NS_E_INVALID_VIDEO_IQUALITY		  HRES_ERROR(0xC00D1BC7)
#define HRES_NS_E_INVALID_VIDEO_CQUALITY		  HRES_ERROR(0xC00D1BC8)
#define HRES_NS_E_INVALID_VIDEO_BUFFER			  HRES_ERROR(0xC00D1BC9)
#define HRES_NS_E_INVALID_VIDEO_BUFFERMAX		  HRES_ERROR(0xC00D1BCA)
#define HRES_NS_E_INVALID_VIDEO_BUFFERMAX_2		  HRES_ERROR(0xC00D1BCB)
#define HRES_NS_E_INVALID_VIDEO_WIDTH_ALIGN		  HRES_ERROR(0xC00D1BCC)
#define HRES_NS_E_INVALID_VIDEO_HEIGHT_ALIGN		  HRES_ERROR(0xC00D1BCD)
#define HRES_NS_E_MULTIPLE_SCRIPT_BITRATES		  HRES_ERROR(0xC00D1BCE)
#define HRES_NS_E_INVALID_SCRIPT_BITRATE		  HRES_ERROR(0xC00D1BCF)
#define HRES_NS_E_MULTIPLE_FILE_BITRATES		  HRES_ERROR(0xC00D1BD0)
#define HRES_NS_E_INVALID_FILE_BITRATE			  HRES_ERROR(0xC00D1BD1)
#define HRES_NS_E_SAME_AS_INPUT_COMBINATION		  HRES_ERROR(0xC00D1BD2)
#define HRES_NS_E_SOURCE_CANNOT_LOOP			  HRES_ERROR(0xC00D1BD3)
#define HRES_NS_E_INVALID_FOLDDOWN_COEFFICIENTS		  HRES_ERROR(0xC00D1BD4)
#define HRES_NS_E_DRMPROFILE_NOTFOUND			  HRES_ERROR(0xC00D1BD5)
#define HRES_NS_E_INVALID_TIMECODE			  HRES_ERROR(0xC00D1BD6)
#define HRES_NS_E_NO_AUDIO_TIMECOMPRESSION		  HRES_ERROR(0xC00D1BD7)
#define HRES_NS_E_NO_TWOPASS_TIMECOMPRESSION		  HRES_ERROR(0xC00D1BD8)
#define HRES_NS_E_TIMECODE_REQUIRES_VIDEOSTREAM		  HRES_ERROR(0xC00D1BD9)
#define HRES_NS_E_NO_MBR_WITH_TIMECODE			  HRES_ERROR(0xC00D1BDA)
#define HRES_NS_E_INVALID_INTERLACEMODE			  HRES_ERROR(0xC00D1BDB)
#define HRES_NS_E_INVALID_INTERLACE_COMPAT		  HRES_ERROR(0xC00D1BDC)
#define HRES_NS_E_INVALID_NONSQUAREPIXEL_COMPAT		  HRES_ERROR(0xC00D1BDD)
#define HRES_NS_E_INVALID_SOURCE_WITH_DEVICE_CONTROL	  HRES_ERROR(0xC00D1BDE)
#define HRES_NS_E_CANNOT_GENERATE_BROADCAST_INFO_FOR_QUALITYVBR HRES_ERROR(0xC00D1BDF)
#define HRES_NS_E_EXCEED_MAX_DRM_PROFILE_LIMIT		  HRES_ERROR(0xC00D1BE0)
#define HRES_NS_E_DEVICECONTROL_UNSTABLE		  HRES_ERROR(0xC00D1BE1)
#define HRES_NS_E_INVALID_PIXEL_ASPECT_RATIO		  HRES_ERROR(0xC00D1BE2)
#define HRES_NS_E_AUDIENCE__LANGUAGE_CONTENTTYPE_MISMATCH HRES_ERROR(0xC00D1BE3)
#define HRES_NS_E_INVALID_PROFILE_CONTENTTYPE		  HRES_ERROR(0xC00D1BE4)
#define HRES_NS_E_TRANSFORM_PLUGIN_NOT_FOUND		  HRES_ERROR(0xC00D1BE5)
#define HRES_NS_E_TRANSFORM_PLUGIN_INVALID		  HRES_ERROR(0xC00D1BE6)
#define HRES_NS_E_EDL_REQUIRED_FOR_DEVICE_MULTIPASS	  HRES_ERROR(0xC00D1BE7)
#define HRES_NS_E_INVALID_VIDEO_WIDTH_FOR_INTERLACED_ENCODING HRES_ERROR(0xC00D1BE8)
#define HRES_NS_E_MARKIN_UNSUPPORTED			  HRES_ERROR(0xC00D1BE9)
#define HRES_NS_E_DRM_INVALID_APPLICATION		  HRES_ERROR(0xC00D2711)
#define HRES_NS_E_DRM_LICENSE_STORE_ERROR		  HRES_ERROR(0xC00D2712)
#define HRES_NS_E_DRM_SECURE_STORE_ERROR		  HRES_ERROR(0xC00D2713)
#define HRES_NS_E_DRM_LICENSE_STORE_SAVE_ERROR		  HRES_ERROR(0xC00D2714)
#define HRES_NS_E_DRM_SECURE_STORE_UNLOCK_ERROR		  HRES_ERROR(0xC00D2715)
#define HRES_NS_E_DRM_INVALID_CONTENT			  HRES_ERROR(0xC00D2716)
#define HRES_NS_E_DRM_UNABLE_TO_OPEN_LICENSE		  HRES_ERROR(0xC00D2717)
#define HRES_NS_E_DRM_INVALID_LICENSE			  HRES_ERROR(0xC00D2718)
#define HRES_NS_E_DRM_INVALID_MACHINE			  HRES_ERROR(0xC00D2719)
#define HRES_NS_E_DRM_ENUM_LICENSE_FAILED		  HRES_ERROR(0xC00D271B)
#define HRES_NS_E_DRM_INVALID_LICENSE_REQUEST		  HRES_ERROR(0xC00D271C)
#define HRES_NS_E_DRM_UNABLE_TO_INITIALIZE		  HRES_ERROR(0xC00D271D)
#define HRES_NS_E_DRM_UNABLE_TO_ACQUIRE_LICENSE		  HRES_ERROR(0xC00D271E)
#define HRES_NS_E_DRM_INVALID_LICENSE_ACQUIRED		  HRES_ERROR(0xC00D271F)
#define HRES_NS_E_DRM_NO_RIGHTS				  HRES_ERROR(0xC00D2720)
#define HRES_NS_E_DRM_KEY_ERROR				  HRES_ERROR(0xC00D2721)
#define HRES_NS_E_DRM_ENCRYPT_ERROR			  HRES_ERROR(0xC00D2722)
#define HRES_NS_E_DRM_DECRYPT_ERROR			  HRES_ERROR(0xC00D2723)
#define HRES_NS_E_DRM_LICENSE_INVALID_XML		  HRES_ERROR(0xC00D2725)
#define HRES_NS_E_DRM_NEEDS_INDIVIDUALIZATION		  HRES_ERROR(0xC00D2728)
#define HRES_NS_E_DRM_ALREADY_INDIVIDUALIZED		  HRES_ERROR(0xC00D2729)
#define HRES_NS_E_DRM_ACTION_NOT_QUERIED		  HRES_ERROR(0xC00D272A)
#define HRES_NS_E_DRM_ACQUIRING_LICENSE			  HRES_ERROR(0xC00D272B)
#define HRES_NS_E_DRM_INDIVIDUALIZING			  HRES_ERROR(0xC00D272C)
#define HRES_NS_E_BACKUP_RESTORE_FAILURE		  HRES_ERROR(0xC00D272D)
#define HRES_NS_E_BACKUP_RESTORE_BAD_REQUEST_ID		  HRES_ERROR(0xC00D272E)
#define HRES_NS_E_DRM_PARAMETERS_MISMATCHED		  HRES_ERROR(0xC00D272F)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_LICENSE_OBJECT	  HRES_ERROR(0xC00D2730)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_INDI_OBJECT	  HRES_ERROR(0xC00D2731)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_ENCRYPT_OBJECT	  HRES_ERROR(0xC00D2732)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_DECRYPT_OBJECT	  HRES_ERROR(0xC00D2733)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_PROPERTIES_OBJECT  HRES_ERROR(0xC00D2734)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_BACKUP_OBJECT	  HRES_ERROR(0xC00D2735)
#define HRES_NS_E_DRM_INDIVIDUALIZE_ERROR		  HRES_ERROR(0xC00D2736)
#define HRES_NS_E_DRM_LICENSE_OPEN_ERROR		  HRES_ERROR(0xC00D2737)
#define HRES_NS_E_DRM_LICENSE_CLOSE_ERROR		  HRES_ERROR(0xC00D2738)
#define HRES_NS_E_DRM_GET_LICENSE_ERROR			  HRES_ERROR(0xC00D2739)
#define HRES_NS_E_DRM_QUERY_ERROR			  HRES_ERROR(0xC00D273A)
#define HRES_NS_E_DRM_REPORT_ERROR			  HRES_ERROR(0xC00D273B)
#define HRES_NS_E_DRM_GET_LICENSESTRING_ERROR		  HRES_ERROR(0xC00D273C)
#define HRES_NS_E_DRM_GET_CONTENTSTRING_ERROR		  HRES_ERROR(0xC00D273D)
#define HRES_NS_E_DRM_MONITOR_ERROR			  HRES_ERROR(0xC00D273E)
#define HRES_NS_E_DRM_UNABLE_TO_SET_PARAMETER		  HRES_ERROR(0xC00D273F)
#define HRES_NS_E_DRM_INVALID_APPDATA			  HRES_ERROR(0xC00D2740)
#define HRES_NS_E_DRM_INVALID_APPDATA_VERSION		  HRES_ERROR(0xC00D2741)
#define HRES_NS_E_DRM_BACKUP_EXISTS			  HRES_ERROR(0xC00D2742)
#define HRES_NS_E_DRM_BACKUP_CORRUPT			  HRES_ERROR(0xC00D2743)
#define HRES_NS_E_DRM_BACKUPRESTORE_BUSY		  HRES_ERROR(0xC00D2744)
#define HRES_NS_E_BACKUP_RESTORE_BAD_DATA		  HRES_ERROR(0xC00D2745)
#define HRES_NS_E_DRM_LICENSE_UNUSABLE			  HRES_ERROR(0xC00D2748)
#define HRES_NS_E_DRM_INVALID_PROPERTY			  HRES_ERROR(0xC00D2749)
#define HRES_NS_E_DRM_SECURE_STORE_NOT_FOUND		  HRES_ERROR(0xC00D274A)
#define HRES_NS_E_DRM_CACHED_CONTENT_ERROR		  HRES_ERROR(0xC00D274B)
#define HRES_NS_E_DRM_INDIVIDUALIZATION_INCOMPLETE	  HRES_ERROR(0xC00D274C)
#define HRES_NS_E_DRM_DRIVER_AUTH_FAILURE		  HRES_ERROR(0xC00D274D)
#define HRES_NS_E_DRM_NEED_UPGRADE_MSSAP		  HRES_ERROR(0xC00D274E)
#define HRES_NS_E_DRM_REOPEN_CONTENT			  HRES_ERROR(0xC00D274F)
#define HRES_NS_E_DRM_DRIVER_DIGIOUT_FAILURE		  HRES_ERROR(0xC00D2750)
#define HRES_NS_E_DRM_INVALID_SECURESTORE_PASSWORD	  HRES_ERROR(0xC00D2751)
#define HRES_NS_E_DRM_APPCERT_REVOKED			  HRES_ERROR(0xC00D2752)
#define HRES_NS_E_DRM_RESTORE_FRAUD			  HRES_ERROR(0xC00D2753)
#define HRES_NS_E_DRM_HARDWARE_INCONSISTENT		  HRES_ERROR(0xC00D2754)
#define HRES_NS_E_DRM_SDMI_TRIGGER			  HRES_ERROR(0xC00D2755)
#define HRES_NS_E_DRM_SDMI_NOMORECOPIES			  HRES_ERROR(0xC00D2756)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_HEADER_OBJECT	  HRES_ERROR(0xC00D2757)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_KEYS_OBJECT	  HRES_ERROR(0xC00D2758)
#define HRES_NS_E_DRM_LICENSE_NOTACQUIRED		  HRES_ERROR(0xC00D2759)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_CODING_OBJECT	  HRES_ERROR(0xC00D275A)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_STATE_DATA_OBJECT  HRES_ERROR(0xC00D275B)
#define HRES_NS_E_DRM_BUFFER_TOO_SMALL			  HRES_ERROR(0xC00D275C)
#define HRES_NS_E_DRM_UNSUPPORTED_PROPERTY		  HRES_ERROR(0xC00D275D)
#define HRES_NS_E_DRM_ERROR_BAD_NET_RESP		  HRES_ERROR(0xC00D275E)
#define HRES_NS_E_DRM_STORE_NOTALLSTORED		  HRES_ERROR(0xC00D275F)
#define HRES_NS_E_DRM_SECURITY_COMPONENT_SIGNATURE_INVALID HRES_ERROR(0xC00D2760)
#define HRES_NS_E_DRM_INVALID_DATA			  HRES_ERROR(0xC00D2761)
#define HRES_NS_E_DRM_POLICY_DISABLE_ONLINE		  HRES_ERROR(0xC00D2762)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_AUTHENTICATION_OBJECT HRES_ERROR(0xC00D2763)
#define HRES_NS_E_DRM_NOT_CONFIGURED			  HRES_ERROR(0xC00D2764)
#define HRES_NS_E_DRM_DEVICE_ACTIVATION_CANCELED	  HRES_ERROR(0xC00D2765)
#define HRES_NS_E_BACKUP_RESTORE_TOO_MANY_RESETS	  HRES_ERROR(0xC00D2766)
#define HRES_NS_E_DRM_DEBUGGING_NOT_ALLOWED		  HRES_ERROR(0xC00D2767)
#define HRES_NS_E_DRM_OPERATION_CANCELED		  HRES_ERROR(0xC00D2768)
#define HRES_NS_E_DRM_RESTRICTIONS_NOT_RETRIEVED	  HRES_ERROR(0xC00D2769)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_PLAYLIST_OBJECT	  HRES_ERROR(0xC00D276A)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_PLAYLIST_BURN_OBJECT HRES_ERROR(0xC00D276B)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_DEVICE_REGISTRATION_OBJECT HRES_ERROR(0xC00D276C)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_METERING_OBJECT	  HRES_ERROR(0xC00D276D)
#define HRES_NS_E_DRM_TRACK_EXCEEDED_PLAYLIST_RESTICTION  HRES_ERROR(0xC00D2770)
#define HRES_NS_E_DRM_TRACK_EXCEEDED_TRACKBURN_RESTRICTION HRES_ERROR(0xC00D2771)
#define HRES_NS_E_DRM_UNABLE_TO_GET_DEVICE_CERT		  HRES_ERROR(0xC00D2772)
#define HRES_NS_E_DRM_UNABLE_TO_GET_SECURE_CLOCK	  HRES_ERROR(0xC00D2773)
#define HRES_NS_E_DRM_UNABLE_TO_SET_SECURE_CLOCK	  HRES_ERROR(0xC00D2774)
#define HRES_NS_E_DRM_UNABLE_TO_GET_SECURE_CLOCK_FROM_SERVER HRES_ERROR(0xC00D2775)
#define HRES_NS_E_DRM_POLICY_METERING_DISABLED		  HRES_ERROR(0xC00D2776)
#define HRES_NS_E_DRM_TRANSFER_CHAINED_LICENSES_UNSUPPORTED HRES_ERROR(0xC00D2777)
#define HRES_NS_E_DRM_SDK_VERSIONMISMATCH		  HRES_ERROR(0xC00D2778)
#define HRES_NS_E_DRM_LIC_NEEDS_DEVICE_CLOCK_SET	  HRES_ERROR(0xC00D2779)
#define HRES_NS_E_LICENSE_HEADER_MISSING_URL		  HRES_ERROR(0xC00D277A)
#define HRES_NS_E_DEVICE_NOT_WMDRM_DEVICE		  HRES_ERROR(0xC00D277B)
#define HRES_NS_E_DRM_INVALID_APPCERT			  HRES_ERROR(0xC00D277C)
#define HRES_NS_E_DRM_PROTOCOL_FORCEFUL_TERMINATION_ON_PETITION HRES_ERROR(0xC00D277D)
#define HRES_NS_E_DRM_PROTOCOL_FORCEFUL_TERMINATION_ON_CHALLENGE HRES_ERROR(0xC00D277E)
#define HRES_NS_E_DRM_CHECKPOINT_FAILED			  HRES_ERROR(0xC00D277F)
#define HRES_NS_E_DRM_BB_UNABLE_TO_INITIALIZE		  HRES_ERROR(0xC00D2780)
#define HRES_NS_E_DRM_UNABLE_TO_LOAD_HARDWARE_ID	  HRES_ERROR(0xC00D2781)
#define HRES_NS_E_DRM_UNABLE_TO_OPEN_DATA_STORE		  HRES_ERROR(0xC00D2782)
#define HRES_NS_E_DRM_DATASTORE_CORRUPT			  HRES_ERROR(0xC00D2783)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_INMEMORYSTORE_OBJECT HRES_ERROR(0xC00D2784)
#define HRES_NS_E_DRM_STUBLIB_REQUIRED			  HRES_ERROR(0xC00D2785)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_CERTIFICATE_OBJECT HRES_ERROR(0xC00D2786)
#define HRES_NS_E_DRM_MIGRATION_TARGET_NOT_ONLINE	  HRES_ERROR(0xC00D2787)
#define HRES_NS_E_DRM_INVALID_MIGRATION_IMAGE		  HRES_ERROR(0xC00D2788)
#define HRES_NS_E_DRM_MIGRATION_TARGET_STATES_CORRUPTED   HRES_ERROR(0xC00D2789)
#define HRES_NS_E_DRM_MIGRATION_IMPORTER_NOT_AVAILABLE	  HRES_ERROR(0xC00D278A)
#define HRES_NS_DRM_E_MIGRATION_UPGRADE_WITH_DIFF_SID	  HRES_ERROR(0xC00D278B)
#define HRES_NS_DRM_E_MIGRATION_SOURCE_MACHINE_IN_USE	  HRES_ERROR(0xC00D278C)
#define HRES_NS_DRM_E_MIGRATION_TARGET_MACHINE_LESS_THAN_LH HRES_ERROR(0xC00D278D)
#define HRES_NS_DRM_E_MIGRATION_IMAGE_ALREADY_EXISTS	  HRES_ERROR(0xC00D278E)
#define HRES_NS_E_DRM_HARDWAREID_MISMATCH		  HRES_ERROR(0xC00D278F)
#define HRES_NS_E_INVALID_DRMV2CLT_STUBLIB		  HRES_ERROR(0xC00D2790)
#define HRES_NS_E_DRM_MIGRATION_INVALID_LEGACYV2_DATA	  HRES_ERROR(0xC00D2791)
#define HRES_NS_E_DRM_MIGRATION_LICENSE_ALREADY_EXISTS	  HRES_ERROR(0xC00D2792)
#define HRES_NS_E_DRM_MIGRATION_INVALID_LEGACYV2_SST_PASSWORD HRES_ERROR(0xC00D2793)
#define HRES_NS_E_DRM_MIGRATION_NOT_SUPPORTED		  HRES_ERROR(0xC00D2794)
#define HRES_NS_E_DRM_UNABLE_TO_CREATE_MIGRATION_IMPORTER_OBJECT HRES_ERROR(0xC00D2795)
#define HRES_NS_E_DRM_CHECKPOINT_MISMATCH		  HRES_ERROR(0xC00D2796)
#define HRES_NS_E_DRM_CHECKPOINT_CORRUPT		  HRES_ERROR(0xC00D2797)
#define HRES_NS_E_REG_FLUSH_FAILURE			  HRES_ERROR(0xC00D2798)
#define HRES_NS_E_HDS_KEY_MISMATCH			  HRES_ERROR(0xC00D2799)
#define HRES_NS_E_DRM_MIGRATION_OPERATION_CANCELLED	  HRES_ERROR(0xC00D279A)
#define HRES_NS_E_DRM_MIGRATION_OBJECT_IN_USE		  HRES_ERROR(0xC00D279B)
#define HRES_NS_E_DRM_MALFORMED_CONTENT_HEADER		  HRES_ERROR(0xC00D279C)
#define HRES_NS_E_DRM_LICENSE_EXPIRED			  HRES_ERROR(0xC00D27D8)
#define HRES_NS_E_DRM_LICENSE_NOTENABLED		  HRES_ERROR(0xC00D27D9)
#define HRES_NS_E_DRM_LICENSE_APPSECLOW			  HRES_ERROR(0xC00D27DA)
#define HRES_NS_E_DRM_STORE_NEEDINDI			  HRES_ERROR(0xC00D27DB)
#define HRES_NS_E_DRM_STORE_NOTALLOWED			  HRES_ERROR(0xC00D27DC)
#define HRES_NS_E_DRM_LICENSE_APP_NOTALLOWED		  HRES_ERROR(0xC00D27DD)
#define HRES_NS_E_DRM_LICENSE_CERT_EXPIRED		  HRES_ERROR(0xC00D27DF)
#define HRES_NS_E_DRM_LICENSE_SECLOW			  HRES_ERROR(0xC00D27E0)
#define HRES_NS_E_DRM_LICENSE_CONTENT_REVOKED		  HRES_ERROR(0xC00D27E1)
#define HRES_NS_E_DRM_DEVICE_NOT_REGISTERED		  HRES_ERROR(0xC00D27E2)
#define HRES_NS_E_DRM_LICENSE_NOSAP			  HRES_ERROR(0xC00D280A)
#define HRES_NS_E_DRM_LICENSE_NOSVP			  HRES_ERROR(0xC00D280B)
#define HRES_NS_E_DRM_LICENSE_NOWDM			  HRES_ERROR(0xC00D280C)
#define HRES_NS_E_DRM_LICENSE_NOTRUSTEDCODEC		  HRES_ERROR(0xC00D280D)
#define HRES_NS_E_DRM_SOURCEID_NOT_SUPPORTED		  HRES_ERROR(0xC00D280E)
#define HRES_NS_E_DRM_NEEDS_UPGRADE_TEMPFILE		  HRES_ERROR(0xC00D283D)
#define HRES_NS_E_DRM_NEED_UPGRADE_PD			  HRES_ERROR(0xC00D283E)
#define HRES_NS_E_DRM_SIGNATURE_FAILURE			  HRES_ERROR(0xC00D283F)
#define HRES_NS_E_DRM_LICENSE_SERVER_INFO_MISSING	  HRES_ERROR(0xC00D2840)
#define HRES_NS_E_DRM_BUSY				  HRES_ERROR(0xC00D2841)
#define HRES_NS_E_DRM_PD_TOO_MANY_DEVICES		  HRES_ERROR(0xC00D2842)
#define HRES_NS_E_DRM_INDIV_FRAUD			  HRES_ERROR(0xC00D2843)
#define HRES_NS_E_DRM_INDIV_NO_CABS			  HRES_ERROR(0xC00D2844)
#define HRES_NS_E_DRM_INDIV_SERVICE_UNAVAILABLE		  HRES_ERROR(0xC00D2845)
#define HRES_NS_E_DRM_RESTORE_SERVICE_UNAVAILABLE	  HRES_ERROR(0xC00D2846)
#define HRES_NS_E_DRM_CLIENT_CODE_EXPIRED		  HRES_ERROR(0xC00D2847)
#define HRES_NS_E_DRM_NO_UPLINK_LICENSE			  HRES_ERROR(0xC00D2848)
#define HRES_NS_E_DRM_INVALID_KID			  HRES_ERROR(0xC00D2849)
#define HRES_NS_E_DRM_LICENSE_INITIALIZATION_ERROR	  HRES_ERROR(0xC00D284A)
#define HRES_NS_E_DRM_CHAIN_TOO_LONG			  HRES_ERROR(0xC00D284C)
#define HRES_NS_E_DRM_UNSUPPORTED_ALGORITHM		  HRES_ERROR(0xC00D284D)
#define HRES_NS_E_DRM_LICENSE_DELETION_ERROR		  HRES_ERROR(0xC00D284E)
#define HRES_NS_E_DRM_INVALID_CERTIFICATE		  HRES_ERROR(0xC00D28A0)
#define HRES_NS_E_DRM_CERTIFICATE_REVOKED		  HRES_ERROR(0xC00D28A1)
#define HRES_NS_E_DRM_LICENSE_UNAVAILABLE		  HRES_ERROR(0xC00D28A2)
#define HRES_NS_E_DRM_DEVICE_LIMIT_REACHED		  HRES_ERROR(0xC00D28A3)
#define HRES_NS_E_DRM_UNABLE_TO_VERIFY_PROXIMITY	  HRES_ERROR(0xC00D28A4)
#define HRES_NS_E_DRM_MUST_REGISTER			  HRES_ERROR(0xC00D28A5)
#define HRES_NS_E_DRM_MUST_APPROVE			  HRES_ERROR(0xC00D28A6)
#define HRES_NS_E_DRM_MUST_REVALIDATE			  HRES_ERROR(0xC00D28A7)
#define HRES_NS_E_DRM_INVALID_PROXIMITY_RESPONSE	  HRES_ERROR(0xC00D28A8)
#define HRES_NS_E_DRM_INVALID_SESSION			  HRES_ERROR(0xC00D28A9)
#define HRES_NS_E_DRM_DEVICE_NOT_OPEN			  HRES_ERROR(0xC00D28AA)
#define HRES_NS_E_DRM_DEVICE_ALREADY_REGISTERED		  HRES_ERROR(0xC00D28AB)
#define HRES_NS_E_DRM_UNSUPPORTED_PROTOCOL_VERSION	  HRES_ERROR(0xC00D28AC)
#define HRES_NS_E_DRM_UNSUPPORTED_ACTION		  HRES_ERROR(0xC00D28AD)
#define HRES_NS_E_DRM_CERTIFICATE_SECURITY_LEVEL_INADEQUATE HRES_ERROR(0xC00D28AE)
#define HRES_NS_E_DRM_UNABLE_TO_OPEN_PORT		  HRES_ERROR(0xC00D28AF)
#define HRES_NS_E_DRM_BAD_REQUEST			  HRES_ERROR(0xC00D28B0)
#define HRES_NS_E_DRM_INVALID_CRL			  HRES_ERROR(0xC00D28B1)
#define HRES_NS_E_DRM_ATTRIBUTE_TOO_LONG		  HRES_ERROR(0xC00D28B2)
#define HRES_NS_E_DRM_EXPIRED_LICENSEBLOB		  HRES_ERROR(0xC00D28B3)
#define HRES_NS_E_DRM_INVALID_LICENSEBLOB		  HRES_ERROR(0xC00D28B4)
#define HRES_NS_E_DRM_INCLUSION_LIST_REQUIRED		  HRES_ERROR(0xC00D28B5)
#define HRES_NS_E_DRM_DRMV2CLT_REVOKED			  HRES_ERROR(0xC00D28B6)
#define HRES_NS_E_DRM_RIV_TOO_SMALL			  HRES_ERROR(0xC00D28B7)
#define HRES_NS_E_OUTPUT_PROTECTION_LEVEL_UNSUPPORTED	  HRES_ERROR(0xC00D2904)
#define HRES_NS_E_COMPRESSED_DIGITAL_VIDEO_PROTECTION_LEVEL_UNSUPPORTED HRES_ERROR(0xC00D2905)
#define HRES_NS_E_UNCOMPRESSED_DIGITAL_VIDEO_PROTECTION_LEVEL_UNSUPPORTED HRES_ERROR(0xC00D2906)
#define HRES_NS_E_ANALOG_VIDEO_PROTECTION_LEVEL_UNSUPPORTED HRES_ERROR(0xC00D2907)
#define HRES_NS_E_COMPRESSED_DIGITAL_AUDIO_PROTECTION_LEVEL_UNSUPPORTED HRES_ERROR(0xC00D2908)
#define HRES_NS_E_UNCOMPRESSED_DIGITAL_AUDIO_PROTECTION_LEVEL_UNSUPPORTED HRES_ERROR(0xC00D2909)
#define HRES_NS_E_OUTPUT_PROTECTION_SCHEME_UNSUPPORTED	  HRES_ERROR(0xC00D290A)
#define HRES_NS_E_REBOOT_RECOMMENDED			  HRES_ERROR(0xC00D2AFA)
#define HRES_NS_E_REBOOT_REQUIRED			  HRES_ERROR(0xC00D2AFB)
#define HRES_NS_E_SETUP_INCOMPLETE			  HRES_ERROR(0xC00D2AFC)
#define HRES_NS_E_SETUP_DRM_MIGRATION_FAILED		  HRES_ERROR(0xC00D2AFD)
#define HRES_NS_E_SETUP_IGNORABLE_FAILURE		  HRES_ERROR(0xC00D2AFE)
#define HRES_NS_E_SETUP_DRM_MIGRATION_FAILED_AND_IGNORABLE_FAILURE HRES_ERROR(0xC00D2AFF)
#define HRES_NS_E_SETUP_BLOCKED				  HRES_ERROR(0xC00D2B00)
#define HRES_NS_E_UNKNOWN_PROTOCOL			  HRES_ERROR(0xC00D2EE0)
#define HRES_NS_E_REDIRECT_TO_PROXY			  HRES_ERROR(0xC00D2EE1)
#define HRES_NS_E_INTERNAL_SERVER_ERROR			  HRES_ERROR(0xC00D2EE2)
#define HRES_NS_E_BAD_REQUEST				  HRES_ERROR(0xC00D2EE3)
#define HRES_NS_E_ERROR_FROM_PROXY			  HRES_ERROR(0xC00D2EE4)
#define HRES_NS_E_PROXY_TIMEOUT				  HRES_ERROR(0xC00D2EE5)
#define HRES_NS_E_SERVER_UNAVAILABLE			  HRES_ERROR(0xC00D2EE6)
#define HRES_NS_E_REFUSED_BY_SERVER			  HRES_ERROR(0xC00D2EE7)
#define HRES_NS_E_INCOMPATIBLE_SERVER			  HRES_ERROR(0xC00D2EE8)
#define HRES_NS_E_MULTICAST_DISABLED			  HRES_ERROR(0xC00D2EE9)
#define HRES_NS_E_INVALID_REDIRECT			  HRES_ERROR(0xC00D2EEA)
#define HRES_NS_E_ALL_PROTOCOLS_DISABLED		  HRES_ERROR(0xC00D2EEB)
#define HRES_NS_E_MSBD_NO_LONGER_SUPPORTED		  HRES_ERROR(0xC00D2EEC)
#define HRES_NS_E_PROXY_NOT_FOUND			  HRES_ERROR(0xC00D2EED)
#define HRES_NS_E_CANNOT_CONNECT_TO_PROXY		  HRES_ERROR(0xC00D2EEE)
#define HRES_NS_E_SERVER_DNS_TIMEOUT			  HRES_ERROR(0xC00D2EEF)
#define HRES_NS_E_PROXY_DNS_TIMEOUT			  HRES_ERROR(0xC00D2EF0)
#define HRES_NS_E_CLOSED_ON_SUSPEND			  HRES_ERROR(0xC00D2EF1)
#define HRES_NS_E_CANNOT_READ_PLAYLIST_FROM_MEDIASERVER   HRES_ERROR(0xC00D2EF2)
#define HRES_NS_E_SESSION_NOT_FOUND			  HRES_ERROR(0xC00D2EF3)
#define HRES_NS_E_REQUIRE_STREAMING_CLIENT		  HRES_ERROR(0xC00D2EF4)
#define HRES_NS_E_PLAYLIST_ENTRY_HAS_CHANGED		  HRES_ERROR(0xC00D2EF5)
#define HRES_NS_E_PROXY_ACCESSDENIED			  HRES_ERROR(0xC00D2EF6)
#define HRES_NS_E_PROXY_SOURCE_ACCESSDENIED		  HRES_ERROR(0xC00D2EF7)
#define HRES_NS_E_NETWORK_SINK_WRITE			  HRES_ERROR(0xC00D2EF8)
#define HRES_NS_E_FIREWALL				  HRES_ERROR(0xC00D2EF9)
#define HRES_NS_E_MMS_NOT_SUPPORTED			  HRES_ERROR(0xC00D2EFA)
#define HRES_NS_E_SERVER_ACCESSDENIED			  HRES_ERROR(0xC00D2EFB)
#define HRES_NS_E_RESOURCE_GONE				  HRES_ERROR(0xC00D2EFC)
#define HRES_NS_E_NO_EXISTING_PACKETIZER		  HRES_ERROR(0xC00D2EFD)
#define HRES_NS_E_BAD_SYNTAX_IN_SERVER_RESPONSE		  HRES_ERROR(0xC00D2EFE)
#define HRES_NS_E_RESET_SOCKET_CONNECTION		  HRES_ERROR(0xC00D2F00)
#define HRES_NS_E_TOO_MANY_HOPS				  HRES_ERROR(0xC00D2F02)
#define HRES_NS_E_TOO_MUCH_DATA_FROM_SERVER		  HRES_ERROR(0xC00D2F05)
#define HRES_NS_E_CONNECT_TIMEOUT			  HRES_ERROR(0xC00D2F06)
#define HRES_NS_E_PROXY_CONNECT_TIMEOUT			  HRES_ERROR(0xC00D2F07)
#define HRES_NS_E_SESSION_INVALID			  HRES_ERROR(0xC00D2F08)
#define HRES_NS_E_PACKETSINK_UNKNOWN_FEC_STREAM		  HRES_ERROR(0xC00D2F0A)
#define HRES_NS_E_PUSH_CANNOTCONNECT			  HRES_ERROR(0xC00D2F0B)
#define HRES_NS_E_INCOMPATIBLE_PUSH_SERVER		  HRES_ERROR(0xC00D2F0C)
#define HRES_NS_E_END_OF_PLAYLIST			  HRES_ERROR(0xC00D32C8)
#define HRES_NS_E_USE_FILE_SOURCE			  HRES_ERROR(0xC00D32C9)
#define HRES_NS_E_PROPERTY_NOT_FOUND			  HRES_ERROR(0xC00D32CA)
#define HRES_NS_E_PROPERTY_READ_ONLY			  HRES_ERROR(0xC00D32CC)
#define HRES_NS_E_TABLE_KEY_NOT_FOUND			  HRES_ERROR(0xC00D32CD)
#define HRES_NS_E_INVALID_QUERY_OPERATOR		  HRES_ERROR(0xC00D32CF)
#define HRES_NS_E_INVALID_QUERY_PROPERTY		  HRES_ERROR(0xC00D32D0)
#define HRES_NS_E_PROPERTY_NOT_SUPPORTED		  HRES_ERROR(0xC00D32D2)
#define HRES_NS_E_SCHEMA_CLASSIFY_FAILURE		  HRES_ERROR(0xC00D32D4)
#define HRES_NS_E_METADATA_FORMAT_NOT_SUPPORTED		  HRES_ERROR(0xC00D32D5)
#define HRES_NS_E_METADATA_NO_EDITING_CAPABILITY	  HRES_ERROR(0xC00D32D6)
#define HRES_NS_E_METADATA_CANNOT_SET_LOCALE		  HRES_ERROR(0xC00D32D7)
#define HRES_NS_E_METADATA_LANGUAGE_NOT_SUPORTED	  HRES_ERROR(0xC00D32D8)
#define HRES_NS_E_METADATA_NO_RFC1766_NAME_FOR_LOCALE	  HRES_ERROR(0xC00D32D9)
#define HRES_NS_E_METADATA_NOT_AVAILABLE		  HRES_ERROR(0xC00D32DA)
#define HRES_NS_E_METADATA_CACHE_DATA_NOT_AVAILABLE	  HRES_ERROR(0xC00D32DB)
#define HRES_NS_E_METADATA_INVALID_DOCUMENT_TYPE	  HRES_ERROR(0xC00D32DC)
#define HRES_NS_E_METADATA_IDENTIFIER_NOT_AVAILABLE	  HRES_ERROR(0xC00D32DD)
#define HRES_NS_E_METADATA_CANNOT_RETRIEVE_FROM_OFFLINE_CACHE HRES_ERROR(0xC00D32DE)
#define HRES_ERROR_MONITOR_INVALID_DESCRIPTOR_CHECKSUM	  HRES_ERROR(0xC0261003)
#define HRES_ERROR_MONITOR_INVALID_STANDARD_TIMING_BLOCK  HRES_ERROR(0xC0261004)
#define HRES_ERROR_MONITOR_WMI_DATABLOCK_REGISTRATION_FAILED HRES_ERROR(0xC0261005)
#define HRES_ERROR_MONITOR_INVALID_SERIAL_NUMBER_MONDSC_BLOCK HRES_ERROR(0xC0261006)
#define HRES_ERROR_MONITOR_INVALID_USER_FRIENDLY_MONDSC_BLOCK HRES_ERROR(0xC0261007)
#define HRES_ERROR_MONITOR_NO_MORE_DESCRIPTOR_DATA	  HRES_ERROR(0xC0261008)
#define HRES_ERROR_MONITOR_INVALID_DETAILED_TIMING_BLOCK  HRES_ERROR(0xC0261009)
#define HRES_ERROR_GRAPHICS_NOT_EXCLUSIVE_MODE_OWNER	  HRES_ERROR(0xC0262000)
#define HRES_ERROR_GRAPHICS_INSUFFICIENT_DMA_BUFFER	  HRES_ERROR(0xC0262001)
#define HRES_ERROR_GRAPHICS_INVALID_DISPLAY_ADAPTER	  HRES_ERROR(0xC0262002)
#define HRES_ERROR_GRAPHICS_ADAPTER_WAS_RESET		  HRES_ERROR(0xC0262003)
#define HRES_ERROR_GRAPHICS_INVALID_DRIVER_MODEL	  HRES_ERROR(0xC0262004)
#define HRES_ERROR_GRAPHICS_PRESENT_MODE_CHANGED	  HRES_ERROR(0xC0262005)
#define HRES_ERROR_GRAPHICS_PRESENT_OCCLUDED		  HRES_ERROR(0xC0262006)
#define HRES_ERROR_GRAPHICS_PRESENT_DENIED		  HRES_ERROR(0xC0262007)
#define HRES_ERROR_GRAPHICS_CANNOTCOLORCONVERT		  HRES_ERROR(0xC0262008)
#define HRES_ERROR_GRAPHICS_NO_VIDEO_MEMORY		  HRES_ERROR(0xC0262100)
#define HRES_ERROR_GRAPHICS_CANT_LOCK_MEMORY		  HRES_ERROR(0xC0262101)
#define HRES_ERROR_GRAPHICS_ALLOCATION_BUSY		  HRES_ERROR(0xC0262102)
#define HRES_ERROR_GRAPHICS_TOO_MANY_REFERENCES		  HRES_ERROR(0xC0262103)
#define HRES_ERROR_GRAPHICS_TRY_AGAIN_LATER		  HRES_ERROR(0xC0262104)
#define HRES_ERROR_GRAPHICS_TRY_AGAIN_NOW		  HRES_ERROR(0xC0262105)
#define HRES_ERROR_GRAPHICS_ALLOCATION_INVALID		  HRES_ERROR(0xC0262106)
#define HRES_ERROR_GRAPHICS_UNSWIZZLING_APERTURE_UNAVAILABLE HRES_ERROR(0xC0262107)
#define HRES_ERROR_GRAPHICS_UNSWIZZLING_APERTURE_UNSUPPORTED HRES_ERROR(0xC0262108)
#define HRES_ERROR_GRAPHICS_CANT_EVICT_PINNED_ALLOCATION  HRES_ERROR(0xC0262109)
#define HRES_ERROR_GRAPHICS_INVALID_ALLOCATION_USAGE	  HRES_ERROR(0xC0262110)
#define HRES_ERROR_GRAPHICS_CANT_RENDER_LOCKED_ALLOCATION HRES_ERROR(0xC0262111)
#define HRES_ERROR_GRAPHICS_ALLOCATION_CLOSED		  HRES_ERROR(0xC0262112)
#define HRES_ERROR_GRAPHICS_INVALID_ALLOCATION_INSTANCE   HRES_ERROR(0xC0262113)
#define HRES_ERROR_GRAPHICS_INVALID_ALLOCATION_HANDLE	  HRES_ERROR(0xC0262114)
#define HRES_ERROR_GRAPHICS_WRONG_ALLOCATION_DEVICE	  HRES_ERROR(0xC0262115)
#define HRES_ERROR_GRAPHICS_ALLOCATION_CONTENT_LOST	  HRES_ERROR(0xC0262116)
#define HRES_ERROR_GRAPHICS_GPU_EXCEPTION_ON_DEVICE	  HRES_ERROR(0xC0262200)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_TOPOLOGY	  HRES_ERROR(0xC0262300)
#define HRES_ERROR_GRAPHICS_VIDPN_TOPOLOGY_NOT_SUPPORTED  HRES_ERROR(0xC0262301)
#define HRES_ERROR_GRAPHICS_VIDPN_TOPOLOGY_CURRENTLY_NOT_SUPPORTED HRES_ERROR(0xC0262302)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN		  HRES_ERROR(0xC0262303)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE  HRES_ERROR(0xC0262304)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET  HRES_ERROR(0xC0262305)
#define HRES_ERROR_GRAPHICS_VIDPN_MODALITY_NOT_SUPPORTED  HRES_ERROR(0xC0262306)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_SOURCEMODESET   HRES_ERROR(0xC0262308)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_TARGETMODESET   HRES_ERROR(0xC0262309)
#define HRES_ERROR_GRAPHICS_INVALID_FREQUENCY		  HRES_ERROR(0xC026230A)
#define HRES_ERROR_GRAPHICS_INVALID_ACTIVE_REGION	  HRES_ERROR(0xC026230B)
#define HRES_ERROR_GRAPHICS_INVALID_TOTAL_REGION	  HRES_ERROR(0xC026230C)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_SOURCE_MODE HRES_ERROR(0xC0262310)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEO_PRESENT_TARGET_MODE HRES_ERROR(0xC0262311)
#define HRES_ERROR_GRAPHICS_PINNED_MODE_MUST_REMAIN_IN_SET HRES_ERROR(0xC0262312)
#define HRES_ERROR_GRAPHICS_PATH_ALREADY_IN_TOPOLOGY	  HRES_ERROR(0xC0262313)
#define HRES_ERROR_GRAPHICS_MODE_ALREADY_IN_MODESET	  HRES_ERROR(0xC0262314)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEOPRESENTSOURCESET HRES_ERROR(0xC0262315)
#define HRES_ERROR_GRAPHICS_INVALID_VIDEOPRESENTTARGETSET HRES_ERROR(0xC0262316)
#define HRES_ERROR_GRAPHICS_SOURCE_ALREADY_IN_SET	  HRES_ERROR(0xC0262317)
#define HRES_ERROR_GRAPHICS_TARGET_ALREADY_IN_SET	  HRES_ERROR(0xC0262318)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_PRESENT_PATH	  HRES_ERROR(0xC0262319)
#define HRES_ERROR_GRAPHICS_NO_RECOMMENDED_VIDPN_TOPOLOGY HRES_ERROR(0xC026231A)
#define HRES_ERROR_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGESET HRES_ERROR(0xC026231B)
#define HRES_ERROR_GRAPHICS_INVALID_MONITOR_FREQUENCYRANGE HRES_ERROR(0xC026231C)
#define HRES_ERROR_GRAPHICS_FREQUENCYRANGE_NOT_IN_SET	  HRES_ERROR(0xC026231D)
#define HRES_ERROR_GRAPHICS_FREQUENCYRANGE_ALREADY_IN_SET HRES_ERROR(0xC026231F)
#define HRES_ERROR_GRAPHICS_STALE_MODESET		  HRES_ERROR(0xC0262320)
#define HRES_ERROR_GRAPHICS_INVALID_MONITOR_SOURCEMODESET HRES_ERROR(0xC0262321)
#define HRES_ERROR_GRAPHICS_INVALID_MONITOR_SOURCE_MODE   HRES_ERROR(0xC0262322)
#define HRES_ERROR_GRAPHICS_NO_RECOMMENDED_FUNCTIONAL_VIDPN HRES_ERROR(0xC0262323)
#define HRES_ERROR_GRAPHICS_MODE_ID_MUST_BE_UNIQUE	  HRES_ERROR(0xC0262324)
#define HRES_ERROR_GRAPHICS_EMPTY_ADAPTER_MONITOR_MODE_SUPPORT_INTERSECTION HRES_ERROR(0xC0262325)
#define HRES_ERROR_GRAPHICS_VIDEO_PRESENT_TARGETS_LESS_THAN_SOURCES HRES_ERROR(0xC0262326)
#define HRES_ERROR_GRAPHICS_PATH_NOT_IN_TOPOLOGY	  HRES_ERROR(0xC0262327)
#define HRES_ERROR_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_SOURCE HRES_ERROR(0xC0262328)
#define HRES_ERROR_GRAPHICS_ADAPTER_MUST_HAVE_AT_LEAST_ONE_TARGET HRES_ERROR(0xC0262329)
#define HRES_ERROR_GRAPHICS_INVALID_MONITORDESCRIPTORSET  HRES_ERROR(0xC026232A)
#define HRES_ERROR_GRAPHICS_INVALID_MONITORDESCRIPTOR	  HRES_ERROR(0xC026232B)
#define HRES_ERROR_GRAPHICS_MONITORDESCRIPTOR_NOT_IN_SET  HRES_ERROR(0xC026232C)
#define HRES_ERROR_GRAPHICS_MONITORDESCRIPTOR_ALREADY_IN_SET HRES_ERROR(0xC026232D)
#define HRES_ERROR_GRAPHICS_MONITORDESCRIPTOR_ID_MUST_BE_UNIQUE HRES_ERROR(0xC026232E)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_TARGET_SUBSET_TYPE HRES_ERROR(0xC026232F)
#define HRES_ERROR_GRAPHICS_RESOURCES_NOT_RELATED	  HRES_ERROR(0xC0262330)
#define HRES_ERROR_GRAPHICS_SOURCE_ID_MUST_BE_UNIQUE	  HRES_ERROR(0xC0262331)
#define HRES_ERROR_GRAPHICS_TARGET_ID_MUST_BE_UNIQUE	  HRES_ERROR(0xC0262332)
#define HRES_ERROR_GRAPHICS_NO_AVAILABLE_VIDPN_TARGET	  HRES_ERROR(0xC0262333)
#define HRES_ERROR_GRAPHICS_MONITOR_COULD_NOT_BE_ASSOCIATED_WITH_ADAPTER HRES_ERROR(0xC0262334)
#define HRES_ERROR_GRAPHICS_NO_VIDPNMGR			  HRES_ERROR(0xC0262335)
#define HRES_ERROR_GRAPHICS_NO_ACTIVE_VIDPN		  HRES_ERROR(0xC0262336)
#define HRES_ERROR_GRAPHICS_STALE_VIDPN_TOPOLOGY	  HRES_ERROR(0xC0262337)
#define HRES_ERROR_GRAPHICS_MONITOR_NOT_CONNECTED	  HRES_ERROR(0xC0262338)
#define HRES_ERROR_GRAPHICS_SOURCE_NOT_IN_TOPOLOGY	  HRES_ERROR(0xC0262339)
#define HRES_ERROR_GRAPHICS_INVALID_PRIMARYSURFACE_SIZE   HRES_ERROR(0xC026233A)
#define HRES_ERROR_GRAPHICS_INVALID_VISIBLEREGION_SIZE	  HRES_ERROR(0xC026233B)
#define HRES_ERROR_GRAPHICS_INVALID_STRIDE		  HRES_ERROR(0xC026233C)
#define HRES_ERROR_GRAPHICS_INVALID_PIXELFORMAT		  HRES_ERROR(0xC026233D)
#define HRES_ERROR_GRAPHICS_INVALID_COLORBASIS		  HRES_ERROR(0xC026233E)
#define HRES_ERROR_GRAPHICS_INVALID_PIXELVALUEACCESSMODE  HRES_ERROR(0xC026233F)
#define HRES_ERROR_GRAPHICS_TARGET_NOT_IN_TOPOLOGY	  HRES_ERROR(0xC0262340)
#define HRES_ERROR_GRAPHICS_NO_DISPLAY_MODE_MANAGEMENT_SUPPORT HRES_ERROR(0xC0262341)
#define HRES_ERROR_GRAPHICS_VIDPN_SOURCE_IN_USE		  HRES_ERROR(0xC0262342)
#define HRES_ERROR_GRAPHICS_CANT_ACCESS_ACTIVE_VIDPN	  HRES_ERROR(0xC0262343)
#define HRES_ERROR_GRAPHICS_INVALID_PATH_IMPORTANCE_ORDINAL HRES_ERROR(0xC0262344)
#define HRES_ERROR_GRAPHICS_INVALID_PATH_CONTENT_GEOMETRY_TRANSFORMATION HRES_ERROR(0xC0262345)
#define HRES_ERROR_GRAPHICS_PATH_CONTENT_GEOMETRY_TRANSFORMATION_NOT_SUPPORTED HRES_ERROR(0xC0262346)
#define HRES_ERROR_GRAPHICS_INVALID_GAMMA_RAMP		  HRES_ERROR(0xC0262347)
#define HRES_ERROR_GRAPHICS_GAMMA_RAMP_NOT_SUPPORTED	  HRES_ERROR(0xC0262348)
#define HRES_ERROR_GRAPHICS_MULTISAMPLING_NOT_SUPPORTED   HRES_ERROR(0xC0262349)
#define HRES_ERROR_GRAPHICS_MODE_NOT_IN_MODESET		  HRES_ERROR(0xC026234A)
#define HRES_ERROR_GRAPHICS_INVALID_VIDPN_TOPOLOGY_RECOMMENDATION_REASON HRES_ERROR(0xC026234D)
#define HRES_ERROR_GRAPHICS_INVALID_PATH_CONTENT_TYPE	  HRES_ERROR(0xC026234E)
#define HRES_ERROR_GRAPHICS_INVALID_COPYPROTECTION_TYPE   HRES_ERROR(0xC026234F)
#define HRES_ERROR_GRAPHICS_UNASSIGNED_MODESET_ALREADY_EXISTS HRES_ERROR(0xC0262350)
#define HRES_ERROR_GRAPHICS_INVALID_SCANLINE_ORDERING	  HRES_ERROR(0xC0262352)
#define HRES_ERROR_GRAPHICS_TOPOLOGY_CHANGES_NOT_ALLOWED  HRES_ERROR(0xC0262353)
#define HRES_ERROR_GRAPHICS_NO_AVAILABLE_IMPORTANCE_ORDINALS HRES_ERROR(0xC0262354)
#define HRES_ERROR_GRAPHICS_INCOMPATIBLE_PRIVATE_FORMAT   HRES_ERROR(0xC0262355)
#define HRES_ERROR_GRAPHICS_INVALID_MODE_PRUNING_ALGORITHM HRES_ERROR(0xC0262356)
#define HRES_ERROR_GRAPHICS_SPECIFIED_CHILD_ALREADY_CONNECTED HRES_ERROR(0xC0262400)
#define HRES_ERROR_GRAPHICS_CHILD_DESCRIPTOR_NOT_SUPPORTED HRES_ERROR(0xC0262401)
#define HRES_ERROR_GRAPHICS_NOT_A_LINKED_ADAPTER	  HRES_ERROR(0xC0262430)
#define HRES_ERROR_GRAPHICS_LEADLINK_NOT_ENUMERATED	  HRES_ERROR(0xC0262431)
#define HRES_ERROR_GRAPHICS_CHAINLINKS_NOT_ENUMERATED	  HRES_ERROR(0xC0262432)
#define HRES_ERROR_GRAPHICS_ADAPTER_CHAIN_NOT_READY	  HRES_ERROR(0xC0262433)
#define HRES_ERROR_GRAPHICS_CHAINLINKS_NOT_STARTED	  HRES_ERROR(0xC0262434)
#define HRES_ERROR_GRAPHICS_CHAINLINKS_NOT_POWERED_ON	  HRES_ERROR(0xC0262435)
#define HRES_ERROR_GRAPHICS_INCONSISTENT_DEVICE_LINK_STATE HRES_ERROR(0xC0262436)
#define HRES_ERROR_GRAPHICS_NOT_POST_DEVICE_DRIVER	  HRES_ERROR(0xC0262438)
#define HRES_ERROR_GRAPHICS_OPM_NOT_SUPPORTED		  HRES_ERROR(0xC0262500)
#define HRES_ERROR_GRAPHICS_COPP_NOT_SUPPORTED		  HRES_ERROR(0xC0262501)
#define HRES_ERROR_GRAPHICS_UAB_NOT_SUPPORTED		  HRES_ERROR(0xC0262502)
#define HRES_ERROR_GRAPHICS_OPM_INVALID_ENCRYPTED_PARAMETERS HRES_ERROR(0xC0262503)
#define HRES_ERROR_GRAPHICS_OPM_PARAMETER_ARRAY_TOO_SMALL HRES_ERROR(0xC0262504)
#define HRES_ERROR_GRAPHICS_OPM_NO_VIDEO_OUTPUTS_EXIST	  HRES_ERROR(0xC0262505)
#define HRES_ERROR_GRAPHICS_PVP_NO_DISPLAY_DEVICE_CORRESPONDS_TO_NAME HRES_ERROR(0xC0262506)
#define HRES_ERROR_GRAPHICS_PVP_DISPLAY_DEVICE_NOT_ATTACHED_TO_DESKTOP HRES_ERROR(0xC0262507)
#define HRES_ERROR_GRAPHICS_PVP_MIRRORING_DEVICES_NOT_SUPPORTED HRES_ERROR(0xC0262508)
#define HRES_ERROR_GRAPHICS_OPM_INVALID_POINTER		  HRES_ERROR(0xC026250A)
#define HRES_ERROR_GRAPHICS_OPM_INTERNAL_ERROR		  HRES_ERROR(0xC026250B)
#define HRES_ERROR_GRAPHICS_OPM_INVALID_HANDLE		  HRES_ERROR(0xC026250C)
#define HRES_ERROR_GRAPHICS_PVP_NO_MONITORS_CORRESPOND_TO_DISPLAY_DEVICE HRES_ERROR(0xC026250D)
#define HRES_ERROR_GRAPHICS_PVP_INVALID_CERTIFICATE_LENGTH HRES_ERROR(0xC026250E)
#define HRES_ERROR_GRAPHICS_OPM_SPANNING_MODE_ENABLED	  HRES_ERROR(0xC026250F)
#define HRES_ERROR_GRAPHICS_OPM_THEATER_MODE_ENABLED	  HRES_ERROR(0xC0262510)
#define HRES_ERROR_GRAPHICS_PVP_HFS_FAILED		  HRES_ERROR(0xC0262511)
#define HRES_ERROR_GRAPHICS_OPM_INVALID_SRM		  HRES_ERROR(0xC0262512)
#define HRES_ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_HDCP HRES_ERROR(0xC0262513)
#define HRES_ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_ACP HRES_ERROR(0xC0262514)
#define HRES_ERROR_GRAPHICS_OPM_OUTPUT_DOES_NOT_SUPPORT_CGMSA HRES_ERROR(0xC0262515)
#define HRES_ERROR_GRAPHICS_OPM_HDCP_SRM_NEVER_SET	  HRES_ERROR(0xC0262516)
#define HRES_ERROR_GRAPHICS_OPM_RESOLUTION_TOO_HIGH	  HRES_ERROR(0xC0262517)
#define HRES_ERROR_GRAPHICS_OPM_ALL_HDCP_HARDWARE_ALREADY_IN_USE HRES_ERROR(0xC0262518)
#define HRES_ERROR_GRAPHICS_OPM_VIDEO_OUTPUT_NO_LONGER_EXISTS HRES_ERROR(0xC0262519)
#define HRES_ERROR_GRAPHICS_OPM_SESSION_TYPE_CHANGE_IN_PROGRESS HRES_ERROR(0xC026251A)
#define HRES_ERROR_GRAPHICS_I2C_NOT_SUPPORTED		  HRES_ERROR(0xC0262580)
#define HRES_ERROR_GRAPHICS_I2C_DEVICE_DOES_NOT_EXIST	  HRES_ERROR(0xC0262581)
#define HRES_ERROR_GRAPHICS_I2C_ERROR_TRANSMITTING_DATA   HRES_ERROR(0xC0262582)
#define HRES_ERROR_GRAPHICS_I2C_ERROR_RECEIVING_DATA	  HRES_ERROR(0xC0262583)
#define HRES_ERROR_GRAPHICS_DDCCI_VCP_NOT_SUPPORTED	  HRES_ERROR(0xC0262584)
#define HRES_ERROR_GRAPHICS_DDCCI_INVALID_DATA		  HRES_ERROR(0xC0262585)
#define HRES_ERROR_GRAPHICS_DDCCI_MONITOR_RETURNED_INVALID_TIMING_STATUS_BYTE HRES_ERROR(0xC0262586)
#define HRES_ERROR_GRAPHICS_MCA_INVALID_CAPABILITIES_STRING HRES_ERROR(0xC0262587)
#define HRES_ERROR_GRAPHICS_MCA_INTERNAL_ERROR		  HRES_ERROR(0xC0262588)
#define HRES_ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_COMMAND HRES_ERROR(0xC0262589)
#define HRES_ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_LENGTH  HRES_ERROR(0xC026258A)
#define HRES_ERROR_GRAPHICS_DDCCI_INVALID_MESSAGE_CHECKSUM HRES_ERROR(0xC026258B)
#define HRES_ERROR_GRAPHICS_PMEA_INVALID_MONITOR	  HRES_ERROR(0xC02625D6)
#define HRES_ERROR_GRAPHICS_PMEA_INVALID_D3D_DEVICE	  HRES_ERROR(0xC02625D7)
#define HRES_ERROR_GRAPHICS_DDCCI_CURRENT_CURRENT_VALUE_GREATER_THAN_MAXIMUM_VALUE HRES_ERROR(0xC02625D8)
#define HRES_ERROR_GRAPHICS_MCA_INVALID_VCP_VERSION	  HRES_ERROR(0xC02625D9)
#define HRES_ERROR_GRAPHICS_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION HRES_ERROR(0xC02625DA)
#define HRES_ERROR_GRAPHICS_MCA_MCCS_VERSION_MISMATCH	  HRES_ERROR(0xC02625DB)
#define HRES_ERROR_GRAPHICS_MCA_UNSUPPORTED_MCCS_VERSION  HRES_ERROR(0xC02625DC)
#define HRES_ERROR_GRAPHICS_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED HRES_ERROR(0xC02625DE)
#define HRES_ERROR_GRAPHICS_MCA_UNSUPPORTED_COLOR_TEMPERATURE HRES_ERROR(0xC02625DF)
#define HRES_ERROR_GRAPHICS_ONLY_CONSOLE_SESSION_SUPPORTED HRES_ERROR(0xC02625E0)

const char *hresult_errstr_const(HRESULT err_code);

#define FACILITY_WIN32 0x0007
#define WIN32_FROM_HRESULT(x) (HRES_ERROR_V(x) == 0 ? HRES_ERROR_V(x) : ~((FACILITY_WIN32 << 16) | 0x80000000) & HRES_ERROR_V(x))
#define HRESULT_IS_LIKELY_WERR(x) ((HRES_ERROR_V(x) & 0xFFFF0000) == 0x80070000)



#endif /*_HRESULT_H_*/
