// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iostream>
#include <sstream>
#include <iomanip>
using namespace std;

#include <tcl.h>
#include <tk.h>
#include <Xlib.h>

#include <tkMacOSX.h>
#include <tkMacOSXInt.h>
#include <tkMacOSXFont.h>
#include <tkMacOSXPrivate.h>

#include "vector.h"

#define RGBFLOATRED(c)   ((c).red   / 65535.0)
#define RGBFLOATGREEN(c) ((c).green / 65535.0)
#define RGBFLOATBLUE(c)  ((c).blue  / 65535.0)

char* XImageData(XImage* ximage)
{
  GWorldPtr gw = TkMacOSXGetDrawablePort((Pixmap)(ximage->obdata));
  PixMapHandle ph = GetGWorldPixMap(gw);
  PixMap* pm = *ph;
  return GetPixBaseAddr(ph);
}

void DrawRotString(Display* display, Drawable drawable, GC gc, Tk_Font font, 
		   double angle, const Vector& v, const char* text,
		   Widget* parent)
{
  MacDrawable* macWin = (MacDrawable*)drawable;

  TkMacOSXDrawingContext dc;
  TkMacOSXSetupDrawingContext(drawable, gc, 1, &dc);

  // undo the xform in TkMacOSXSetupDrawingContext
  CGContextConcatCTM(dc.context, CGAffineTransformMake(1.0, 0.0, 0.0,
		 -1.0, 0.0, dc.portBounds.bottom - dc.portBounds.top));

  CGAffineTransform mm = CGAffineTransformMakeRotation(angle);
  CGContextSetTextMatrix(dc.context, mm);

  // map to window space
  Vector center = 
    Vector((dc.portBounds.right-dc.portBounds.left)/2.,
	   (dc.portBounds.bottom-dc.portBounds.top)/2.);
  Vector vv = v *
    Translate(-center) *
    Translate(macWin->xOff, macWin->yOff) *
    FlipY() *
    Translate(center);

  // draw it
  TkFont* fontPtr = (TkFont*)font;
  Tcl_DString ds;
  Tcl_DStringInit(&ds);
  int size = Tk_PostscriptFontName(font, &ds);
  CGContextSelectFont(dc.context, Tcl_DStringValue(&ds), 
		      size, kCGEncodingMacRoman);
  Tcl_DStringFree(&ds);
  CGContextShowTextAtPoint (dc.context, vv[0], vv[1], text, strlen(text));

  // done
  TkMacOSXRestoreDrawingContext(&dc);
}

extern "C" {
  void XDrawString(Display* display, Drawable drawable, GC gc, 
			  int x, int y, _Xconst char* str, int length)
  {
    DrawRotString(display, drawable, gc, (Tk_Font)gc->font, 0, 
		  Vector(x,y), str);
    //Tk_DrawChars(display, drawable, gc, (Tk_Font)gc->font, str, length, x, y);
  }

  // Pixmap Suite
  Pixmap XCreatePixmap(Display* display, Drawable drawable, 
			      unsigned int width, unsigned int height,
			      unsigned int depth)
  {
    Tk_GetPixmap(display, drawable, width, height, depth);
  }

  // XImage Suite
  XImage* XXCreateImage(Display* display, Visual* visual,
			unsigned int depth, int format, int offset,
			char* data, unsigned int width, unsigned int height,
			int bitmap_pad, int bytes_per_line)
  {
    XImage* ximage = XCreateImage(display, visual, depth, format, offset,
				  data, width, height, 
				  bitmap_pad, bytes_per_line);

    GWorldPtr gw = (GWorldPtr)data;
    PixMapHandle ph = GetGWorldPixMap(gw);
    PixMap* pm = *ph;

    // now correct it
    ximage->bytes_per_line = (pm->rowBytes & 0x3fff);

    switch (depth) {
    case 1:
      ximage->bits_per_pixel = 1;
      ximage->bitmap_unit = 8;
      ximage->bitmap_pad = 1;
      break;
    case 8:
      ximage->bits_per_pixel = 8;
      ximage->bitmap_unit = 8;
      ximage->bitmap_pad = 8;
      break;
    case 15:
      ximage->bits_per_pixel = 16;
      ximage->bitmap_unit = 32;
      ximage->bitmap_pad = 32;
      break;
    case 16:
      ximage->bits_per_pixel = 16;
      ximage->bitmap_unit = 32;
      ximage->bitmap_pad = 32;
      break;
    case 24:
      ximage->bits_per_pixel = 24;
      ximage->bitmap_unit = 32;
      ximage->bitmap_pad = 32;
      break;
    case 32:
      ximage->bits_per_pixel = 32;
      ximage->bitmap_unit = 32;
      ximage->bitmap_pad = 32;
      break;
    }

    switch (pm->pixelFormat) {
    case k16BE555PixelFormat: // 0x10
      ximage->bitmap_bit_order = MSBFirst;
      ximage->byte_order = MSBFirst;
      break;
    case k16LE555PixelFormat: // 'L555'
      ximage->bitmap_bit_order = LSBFirst;
      ximage->byte_order = LSBFirst;
      break;
    case k16BE565PixelFormat: // 'B565'
      ximage->bitmap_bit_order = MSBFirst;
      ximage->byte_order = MSBFirst;
      break;
    case k16LE565PixelFormat: // 'L565'
      ximage->bitmap_bit_order = LSBFirst;
      ximage->byte_order = LSBFirst;
      break;

    case k24RGBPixelFormat: // 0x18
      ximage->bitmap_bit_order = MSBFirst;
      ximage->byte_order = MSBFirst;
      break;
    case k24BGRPixelFormat: // '24BG'
      ximage->bitmap_bit_order = LSBFirst;
      ximage->byte_order = LSBFirst;
      break;

    case k32ARGBPixelFormat: // 0x20
      ximage->bitmap_bit_order = MSBFirst;
      ximage->byte_order = MSBFirst;
      break;
    case k32BGRAPixelFormat: // 'BGRA'
      ximage->bitmap_bit_order = LSBFirst;
      ximage->byte_order = LSBFirst;
      break;
    case k32ABGRPixelFormat: // 'ABGR'
      ximage->bitmap_bit_order = MSBFirst;
      ximage->byte_order = MSBFirst;
      break;
    case k32RGBAPixelFormat: // 'RGBA'
      ximage->bitmap_bit_order = LSBFirst;
      ximage->byte_order = LSBFirst;
      break;

    default:
      cerr << "MacOSX Internal Error: XXCreateImage: pixelFormat " << hex << pm->pixelFormat 
	   << " not implemented" << endl;
      break;
    }

    return ximage;
  }

  XImage *XXGetImage(Display *display, Drawable drawable, 
		     int x, int y,
		     unsigned int width, unsigned int height,
		     unsigned long plane_mask, int format)
  { 
    if (width ==0 || height ==0) {
      cerr << "MacOSX Internal Error: XXGetImage: width/height is zero" << endl;
      return NULL;
    }

    if (format != ZPixmap) {
      cerr << "MacOSX Internal Error: XXGetImage: illegal format" << endl;
      return NULL;
    }

    if (!TkMacOSXGetDrawablePort(drawable)) {
      cerr << "MacOSX Internal Error: XXGetImage: bad drawable" << endl;
      return NULL;
    }

    // get pixmap
    Pixmap pixmap = Tk_GetPixmap(display, drawable, width, height, 32);
    if (!pixmap) {
      cerr << "MacOSX Internal Error: XXGetImage: bad pixmap" << endl;
      return NULL;
    }
 
    // get gc
    GC gc;
    Tk_Window win = (Tk_Window)((MacDrawable *) drawable)->winPtr;
    if (win) {
      XGCValues values;
      gc = Tk_GetGC(win, 0, &values);
    } 
    else
      gc = XCreateGC(display, pixmap, 0, NULL);
 
    // populate pixmap
    XCopyArea(display, drawable, pixmap, gc, x, y, width, height, 0, 0);

    if (win)
      Tk_FreeGC(display, gc);
    else
      XFreeGC(display, gc);

    GWorldPtr gw = TkMacOSXGetDrawablePort(pixmap);
    PixMapHandle ph = GetGWorldPixMap(gw);
    PixMap* pm = *ph;

    int depth = pm->pixelSize;
    int bitmap_pad = depth; // will be reset
    int bytes_per_line = width; // will be reset
    XImage* image = XXCreateImage(display, NULL, depth, format, 0,
				     (char*)TkMacOSXGetDrawablePort(pixmap),
				     width, height, 
				     bitmap_pad, bytes_per_line);
    // needed by tk
    image->obdata = (XPointer)pixmap;

    return image;
  }

  void XSetClipRectangles(Display* display, GC gc, int clip_x_origin,
			  int  clip_y_origin, XRectangle* r,
			  int  n, int ordering)
  {
    // clear it
    TkpClipMask *clip_mask = (TkpClipMask*)gc->clip_mask;
    if (clip_mask && clip_mask->type == TKP_CLIP_REGION)
      TkpReleaseRegion(clip_mask->value.region);

    // new region
    TkRegion clipRgn = TkCreateRegion();

    // clip origin not used, but set anyways
    gc->clip_x_origin = clip_x_origin;
    gc->clip_y_origin = clip_y_origin;

    for (int ii =0; ii<n; ii++)
      TkUnionRectWithRegion(&r[ii], clipRgn, clipRgn);

    // and set it
    TkSetRegion(display, gc, clipRgn);
  }

  Bool XXQueryPointer(Display* display, Window w, 
		      Window* root_return, Window* child_return, 
		      int* root_x_return, int* root_y_return, 
		      int* win_x_return, int* win_y_return, 
		      unsigned int* mask_return)
  {
    XQueryPointer(display, w, root_return, child_return, 
		  root_x_return, root_y_return, 
		  win_x_return, win_y_return, mask_return);

    // flag root for XXWarpPointer
    *root_return = 1;
    *child_return = NULL;
  }

  void XXWarpPointer(Display* display, Window src_w, Window dest_w,
		     int src_x, int src_y,
		     unsigned int src_width, unsigned int src_height,
		     int dest_x, int dest_y)
  {
    // we have a special mode, root=1, from XXQueryPointer
    CGPoint pt;

    if (dest_w == None) {
      Point ppt;
      GetGlobalMouse(&ppt);
      pt.x = dest_x+ppt.h;
      pt.y = dest_y+ppt.v;
    }
    else if (dest_w == 1) {
      pt.x = dest_x;
      pt.y = dest_y;
    }
    else {
      Point ppt;
      MacDrawable* macPtr = (MacDrawable*)dest_w;
      ppt.h = dest_x+macPtr->xOff;
      ppt.v = dest_y+macPtr->yOff;

      CGrafPtr port = TkMacOSXGetDrawablePort(dest_w);
      QDLocalToGlobalPoint(port, &ppt);
      pt.x = ppt.h;
      pt.y = ppt.v;
    }

    // we can't use the following, we want a mouse event generated.
    //      CGWarpMouseCursorPosition(pt);
    // and we can't use the following, its buggy as all get out
    //      CGPostMouseEvent(pt, true, 1, false);

    CGEventSourceRef src = CGEventSourceCreate(kCGEventSourceStateCombinedSessionState);
    CGEventRef ev = CGEventCreateMouseEvent(src,kCGEventMouseMoved,pt,kCGMouseButtonLeft);
    // bug in tiger, reset event type
    CGEventSetType(ev,kCGEventMouseMoved);
    CGEventPost(kCGSessionEventTap,ev);
    CFRelease(ev);
    CFRelease(src);
  }
}
