// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "rotstr.h"
#include "vector.h"
#include "widget.h"

#include <tk.h>
#include <X11/Xutil.h>

void XDrawRotString(Display* display, Drawable drawable, GC gc, 
		   Vector& v, double angle, const char* text, 
		   Tk_Font font, Widget* parent)
{
  if (angle>-.01 && angle<+.01) {
    Tk_DrawChars(display, drawable, gc, font, text, strlen(text), v[0], v[1]);
    return;
  }

  // this is a very crude "green screen" method.
  // since the text is rendered with an anti-alias method, the text color
  // will vary from the initial color to the mask color.
  // so to avoid a 'dark' ring around the text, we calculate a mask value, 
  // from the middle of the background image.

  int width = parent->getWidth();
  int height = parent->getHeight();
  int depth = parent->getDepth();

  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(font, &metrics);
  int srcw = Tk_TextWidth(font, text, strlen(text))+4;
  int srch = metrics.linespace+2;
  BBox dstbb;
  {
    // calculate bounding box
    Vector corner = (Vector(srcw,srch)-Vector(1,1))/2;
    int ww = Tk_TextWidth(font, text, strlen(text));
    int hh = metrics.ascent-metrics.descent;
    Matrix mx = Translate(ww/2.,-hh/2.) * Rotate(angle) * Translate(v);
    dstbb = BBox(v,v);
    dstbb.bound(-corner * mx);
    dstbb.bound( corner * mx);
    dstbb.bound(Vector( corner[0],-corner[1]) * mx);
    dstbb.bound(Vector(-corner[0], corner[1]) * mx);
  }

  // clip bounding box against window
  BBox dsttbb = intersect(dstbb,BBox(0,0,width,height));

  int dstx = dsttbb.ll[0];
  int dsty = dsttbb.ll[1];
  int dstw = dsttbb.size()[0];
  int dsth = dsttbb.size()[1];
  if (dstw<=0 || dsth<=0)
    return;

  // get background image into dstxmap
  Pixmap dstpmap = XCreatePixmap(display, drawable, dstw, dsth, depth);
  XCopyArea(display, drawable, dstpmap, gc, dstx, dsty, dstw, dsth, 0, 0);

  XImage* dstxmap = XGetImage(display, dstpmap, 0, 0, dstw, dsth,
			      AllPlanes, ZPixmap);
  unsigned char* dst = (unsigned char*)dstxmap->data;
  int dstl = dstxmap->bytes_per_line;
  int dstb = dstxmap->bits_per_pixel/8;

  // calc mask
  unsigned char mask[4];
  {
    switch (dstb) {
    case 1:
    case 2:
      // We don't want to decode, so just take the first value in the bg
      memcpy(mask,dst,dstb);
    case 3:
    case 4:
      // average the background values over the extent
      {
	int cnt =0;
	unsigned char* dst = (unsigned char*)dstxmap->data;
 
	unsigned long mk[4];
	for (int kk=0; kk<3; kk++) 
	  mk[kk] =0;
 
	for (int jj=0; jj<dsth; jj++) {
	  // the line may be padded at the end
	  unsigned char* dptr = dst + jj*dstl;
 
	  for (int ii=0; ii<dstw; ii++, dptr+=dstb) {
	    // the edges can be unprediciable, don't use
	    if (dstx+ii > 0 && dstx+ii < width 
		&& dsty+jj > 0 && dsty+jj < height) {
	      for (int kk=0; kk<dstb; kk++)
		mk[kk] += *(dptr+kk);
	      cnt++;
	    }
	  }
	}
 
	for (int kk=0; kk<dstb; kk++) {
	  if (mk[kk] && cnt)
	    mk[kk] /= cnt;
	  mask[kk] = mk[kk];
	}
      }
      break;
    }
  }
  
  // create srcpmap
  Pixmap srcpmap = XCreatePixmap(display, drawable, srcw, srch, depth);
  {
    XImage* srcxmap = XGetImage(display, srcpmap, 0, 0, srcw, srch,
				AllPlanes, ZPixmap);
    unsigned char* src = (unsigned char*)srcxmap->data;
    int srcl = srcxmap->bytes_per_line;
    int srcb = srcxmap->bits_per_pixel/8;

    // fill srcpmap with mask value(s)
    for (int jj=0; jj<srch; jj++) {
      // the line may be padded at the end
      unsigned char* sptr = src + jj*srcl;
      for (int ii=0; ii<srcw; ii++, sptr+=srcb)
	memcpy(sptr,mask,srcb);
    }
    XPutImage(display, srcpmap, gc, srcxmap, 0, 0, 0, 0, srcw, srch);

    // render text into srcpmap
    XSetBackground(display,gc,parent->getColor("black"));
    Tk_DrawChars(display, srcpmap, gc, font, text, strlen(text), 
    		 2, metrics.ascent);

    XDestroyImage(srcxmap);
  }

  // get rendered text into srcxmap
  {
    XImage* srcxmap = XGetImage(display, srcpmap, 0, 0, srcw, srch, 
				AllPlanes, ZPixmap);
    unsigned char* src = (unsigned char*)srcxmap->data;
    int srcl = srcxmap->bytes_per_line;
    int srcb = srcxmap->bits_per_pixel/8;

    // rotate text from srcxmap to dstxmap
    Matrix m = 
      Translate(-Vector(srcw,srch)/2) *
      Rotate(angle) *
      Translate(Vector(dstw,dsth)/2) *
      Translate((dstbb.center()-dsttbb.center()));
    double* mm = (m.invert()).mm();

    for (int jj=0; jj<dsth; jj++) {
      // the line may be padded at the end
      unsigned char* dptr = dst + jj*dstl;

      for (int ii=0; ii<dstw; ii++, dptr+=dstb) {
	double xx = ii*mm[0] + jj*mm[3] + mm[6];
	double yy = ii*mm[1] + jj*mm[4] + mm[7];

	if (xx >= 0 && xx < srcw && yy >= 0 && yy < srch) {
	  unsigned char* sptr = src + ((int)yy)*srcl + ((int)xx)*srcb;
	  switch (dstb) {
	  case 1:
	    if (*sptr != mask[0])
	      memcpy(dptr,sptr,dstb);
	    break;
	  case 2:
	    if (*(sptr+0) != mask[0] || 
		*(sptr+1) != mask[1])
	      memcpy(dptr,sptr,dstb);
	    break;
	  case 3:
	    if (*(sptr+0) != mask[0] || 
		*(sptr+1) != mask[1] || 
		*(sptr+2) != mask[2])
	      memcpy(dptr,sptr,dstb);
	  case 4:
	    if (*(sptr+0) != mask[0] || 
		*(sptr+1) != mask[1] || 
		*(sptr+2) != mask[2])
	      memcpy(dptr,sptr,dstb);
	    break;
	  }
	}
      }
    }
    XDestroyImage(srcxmap);
  }

  // put rotated text into dstpmap and copy into drawable
  XPutImage(display, dstpmap, gc, dstxmap, 0, 0, 0, 0, dstw, dsth);
  XCopyArea(display, dstpmap, drawable, gc, 0, 0, dstw, dsth, dstx, dsty);

  // clean up
  XFreePixmap(display, srcpmap);
  XFreePixmap(display, dstpmap);
  XDestroyImage(dstxmap);
}
