# the list config class - a simple container

module Schleuder
  class ListConfig < Storage

    # Options and their defaults
    # If you want to change the defaults, edit conf/default-list.conf

    # Emailaddress of the list
    schleuder_attr :myaddr, ''

    # Realname of this list address (mainly used for gpg key)
    schleuder_attr :myname, ''

    # Listadmin's emailaddress(es). Must be an array.
    schleuder_attr :admins, []

    # Default mime setting
    schleuder_attr :default_mime, 'MIME'

    # The gpg password
    schleuder_attr :gpg_password, nil

    # The fingerprint of the key used for this list
    schleuder_attr :key_fingerprint, nil

    # Wether sending emails in the clear is allowed or not.
    schleuder_attr :send_encrypted_only, false

    # Wether to accept only incoming emails that are encrypted
    schleuder_attr :receive_encrypted_only, false

    # Wether to accept only emails that are validly signed
    schleuder_attr :receive_signed_only, false

    # Wether to accept only emails that are validly signed by a list-member's key
    schleuder_attr :receive_authenticated_only, false

    # Wether to accept only emails that are validly signed by a list-admin's key
    schleuder_attr :receive_admin_only, false

    # Whether to accept only emails that are sent from a members address.
    # NOTE: better rely on :receive_authenticated_only and ignore that option.
    schleuder_attr :receive_from_member_emailaddresses_only, false

    # Wether to keep the msgid or not
    schleuder_attr :keep_msgid, true

    # Footer for outgoing mails
    schleuder_attr :public_footer, ''

    # Subject prefix for incoming (signed) mails from listmembers
    schleuder_attr :prefix, ''

    # Subject prefix for incoming mails
    schleuder_attr :prefix_in, ''

    # Subject prefix for outgoing mails
    schleuder_attr :prefix_out, ''

    # The log_level (ERROR || WARN || INFO || DEBUG)
    schleuder_attr :log_level, 'ERROR'

    # Log to SYSLOG?
    schleuder_attr :log_syslog, false

    # Log to IO (writing into STDIN of another process/executable)
    schleuder_attr :log_io, false

    # Log to a file? If the path doesn't start with a slash the list-dir will
    # be prefixed.
    schleuder_attr :log_file, 'list.log'

    # Which headers from original mail to include into the internal meta data
    schleuder_attr :headers_to_meta, [:from, :to, :cc, :date]

    # Restrict specific plugins to admin
    schleuder_attr :keywords_admin_only,  ['SAVE-MEMBERS', 'DEL-KEY']

    # Notify admin if these keywords triggered commands.
    schleuder_attr :keywords_admin_notify, [ 'X-ADD-KEY' ]

    # Drop any bounces (incoming email not passing the receive_*_only-rules)
    schleuder_attr :bounces_drop_all, false

    # Drop bounces if they match one of these headers. Must be a hash, keys and values are case insensitive.
    schleuder_attr :bounces_drop_on_headers,  {'x-spam-flag' => 'yes'}

    # Send a notice to admin(s) on bouncing or dropping
    schleuder_attr :bounces_notify_admin, true

    # Include RFC-compliant List-* Headers into member mails
    schleuder_attr :include_list_headers, true

    # Include OpenPGP-Header
    schleuder_attr :include_openpgp_header, true
    # Preferred way to receive emails to note in OpenPGP-Header ('sign'|'encrypt'|'signencrypt'|'unprotected'|'none')
    # 'none' to not include a preference
    # default: 'signencrypt'
    schleuder_attr :openpgp_header_preference, 'signencrypt'

    # If we want to dump the original incoming mail.
    # ATTENTION: this stores the incoming e-mail on disk!
    schleuder_attr :dump_incoming_mail, false

    # Maximum size of message allowed on the list in kilobyte. All others will be bounced.
    schleuder_attr :max_message_size, 10240 # 10MB

    # Whether to archive messages sent to list members or not.
    # default: false
    schleuder_attr :archive, false

    ### END OF CONFIG OPTIONS

    def initialize(config_file, fromfile=true)
      # First Overload with default-list.conf then load our config
      overload_from_file!(Schleuder.config.lists_default_conf)
      # overload with config_file
      super(config_file, fromfile)

      # load admins as members
      self.admins = self.admins
      # compress fingerprint
      self.key_fingerprint = self.key_fingerprint
    end

    def key_fingerprint=(fpr)
      schleuder_attributes['key_fingerprint'] = Schleuder::Utils.compress_fingerprint(fpr)
    end

    def admins=(ary)
      schleuder_attributes['admins'] = Array(ary).collect { |mem|
        if mem.kind_of?(Member)
          mem
        else
          if mem.kind_of?(Hash) && mem.has_key?("email")
            Member.new(mem)
          else
            Schleuder.log.error "Wrong input: #{mem.inspect} is not suitable data for a Member."
            nil
          end
        end
      }.compact
    end
  end
end
