/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */


#include "common/system.h"
#include "common/events.h"

#include "graphics/cursorman.h"
#include "graphics/palette.h"

#include "queen/display.h"
#include "queen/input.h"
#include "queen/logic.h"
#include "queen/queen.h"
#include "queen/resource.h"

namespace Queen {

Display::Display(QueenEngine *vm, OSystem *system)
	: _fullscreen(true), _horizontalScroll(0), _bdWidth(0), _bdHeight(0),
	_system(system), _vm(vm), _rnd("queenDisplay") {

	initFont();

	_screenBuf   = new uint8[ SCREEN_W * SCREEN_H ];
	_panelBuf    = new uint8[ PANEL_W * PANEL_H ];
	_backdropBuf = new uint8[ BACKDROP_W * BACKDROP_H ];
	memset(_screenBuf,   0, SCREEN_W * SCREEN_H);
	memset(_panelBuf,    0, PANEL_W * PANEL_H);
	memset(_backdropBuf, 0, BACKDROP_W * BACKDROP_H);

	_fullRefresh = 1;
	_dirtyBlocksWidth  = SCREEN_W / D_BLOCK_W;
	_dirtyBlocksHeight = SCREEN_H / D_BLOCK_H;
	_dirtyBlocks = new uint8[_dirtyBlocksWidth * _dirtyBlocksHeight];
	memset(_dirtyBlocks, 0, _dirtyBlocksWidth * _dirtyBlocksHeight);

	_pal.room   = new uint8[ 256 * 3 ];
	_pal.screen = new uint8[ 256 * 3 ];
	_pal.panel  = new uint8[ 112 * 3 ];
	memset(_pal.room,   0, 256 * 3);
	memset(_pal.screen, 0, 256 * 3);
	memset(_pal.panel,  0, 112 * 3);
	_pal.dirtyMin = 0;
	_pal.dirtyMax = 255;
	_pal.scrollable = true;

	_imageExt = (_vm->resource()->getPlatform() == Common::kPlatformAmiga) ? "LBM" : "PCX";

	_curTextColor = 0;
	memset(_texts, 0, sizeof(_texts));

	memset(&_dynalum, 0, sizeof(_dynalum));

	setupInkColors();
}

Display::~Display() {
	delete[] _backdropBuf;
	delete[] _panelBuf;
	delete[] _screenBuf;

	delete[] _dirtyBlocks;

	delete[] _pal.room;
	delete[] _pal.screen;
	delete[] _pal.panel;

	delete[] _dynalum.mskBuf;
	delete[] _dynalum.lumBuf;
}

void Display::dynalumInit(const char *roomName, uint16 roomNum) {
	debug(9, "Display::dynalumInit(%s, %d)", roomName, roomNum);

	_dynalum.valid = false;
	delete[] _dynalum.mskBuf;
	_dynalum.mskBuf = NULL;
	delete[] _dynalum.lumBuf;
	_dynalum.lumBuf = NULL;

	if (!isPalFadingDisabled(roomNum)) {
		char filename[20];
		sprintf(filename, "%s.MSK", roomName);
		if (_vm->resource()->fileExists(filename)) {
			_dynalum.mskBuf = (uint8 *)_vm->resource()->loadFile(filename, 0, &_dynalum.mskSize);
			sprintf(filename, "%s.LUM", roomName);
			if (_vm->resource()->fileExists(filename)) {
				_dynalum.lumBuf = (int8 *)_vm->resource()->loadFile(filename, 0, &_dynalum.lumSize);
				_dynalum.valid = true;
				_dynalum.prevColMask = 0xFF;
			}
		}
	}
}

void Display::dynalumUpdate(int16 x, int16 y) {
	if (!_dynalum.valid)
		return;

	if (x < 0) {
		x = 0;
	} else if (x > _bdWidth) {
		x = _bdWidth;
	}
	if (y < 0) {
		y = 0;
	} else if (y > ROOM_ZONE_HEIGHT - 1) {
		y = ROOM_ZONE_HEIGHT - 1;
	}

	uint32 offset = (y / 4) * 160 + (x / 4);
	assert(offset < _dynalum.mskSize);

	uint8 colMask = _dynalum.mskBuf[offset];
	debug(9, "Display::dynalumUpdate(%d, %d) - colMask = %d", x, y, colMask);
	if (colMask != _dynalum.prevColMask) {
		for (int i = 144; i < 160; ++i) {
			for (int j = 0; j < 3; ++j) {
				int16 c = (int16)(_pal.room[i * 3 + j] + _dynalum.lumBuf[colMask * 3 + j] * 4);
				if (c < 0) {
					c = 0;
				} else if (c > 255) {
					c = 255;
				}
				_pal.screen[i * 3 + j] = (uint8)c;
			}
		}
		_pal.dirtyMin = MIN(_pal.dirtyMin, 144);
		_pal.dirtyMax = MAX(_pal.dirtyMax, 159);
		_dynalum.prevColMask = colMask;
	}
}

void Display::palSet(const uint8 *pal, int start, int end, bool updateScreen) {
	debug(9, "Display::palSet(%d, %d)", start, end);
	const int numColors = end - start + 1;
	assert(numColors <= 256);
	_system->getPaletteManager()->setPalette(pal + start * 3, start, numColors);
	if (updateScreen) {
		_vm->input()->delay(20);
	}
}

void Display::palSetJoeDress() {
	if (_vm->resource()->getPlatform() == Common::kPlatformPC) {
		memcpy(_pal.room + 144 * 3, _palJoeDress, 16 * 3);
		memcpy(_pal.screen + 144 * 3, _palJoeDress, 16 * 3);
		palSet(_pal.screen, 144, 159, true);
	}
}

void Display::palSetJoeNormal() {
	if (_vm->resource()->getPlatform() == Common::kPlatformPC) {
		memcpy(_pal.room + 144 * 3, _palJoeClothes, 16 * 3);
		memcpy(_pal.screen + 144 * 3, _palJoeClothes, 16 * 3);
		palSet(_pal.screen, 144, 159, true);
	}
}

void Display::palSetPanel() {
	memcpy(_pal.room + 144 * 3, _pal.panel, (256 - 144) * 3);
	memcpy(_pal.screen + 144 * 3, _pal.panel, (256 - 144) * 3);
}

void Display::palFadeIn(uint16 roomNum, bool dynalum, int16 dynaX, int16 dynaY) {
	debug(9, "Display::palFadeIn(%d)", roomNum);
	int n = getNumColorsForRoom(roomNum);
	memcpy(_pal.screen, _pal.room, n * 3);
	if (!isPalFadingDisabled(roomNum)) {
		if (dynalum) {
			dynalumUpdate(dynaX, dynaY);
		}
		uint8 tempPal[256 * 3];
		for (int i = 0; i <= FADE_SPEED; ++i) {
			for (int j = 0; j < n * 3; ++j) {
				tempPal[j] = _pal.screen[j] * i / FADE_SPEED;
			}
			palSet(tempPal, 0, n - 1, true);
		}
	}
	_pal.dirtyMin = 0;
	_pal.dirtyMax = n - 1;
	_pal.scrollable = true;
}

void Display::palFadeOut(uint16 roomNum) {
	debug(9, "Display::palFadeOut(%d)", roomNum);
	_pal.scrollable = false;
	int n = getNumColorsForRoom(roomNum);
	if (isPalFadingDisabled(roomNum)) {
		memset(_pal.screen, 0, n * 3);
		palSet(_pal.screen, 0, n - 1, true);
	} else {
		uint8 tempPal[256 * 3];
		memcpy(tempPal, _pal.screen, n * 3);
		for (int i = FADE_SPEED; i >= 0; --i) {
			for (int j = 0; j < n * 3; ++j) {
				_pal.screen[j] = tempPal[j] * i / FADE_SPEED;
			}
			palSet(_pal.screen, 0, n - 1, true);
		}
	}
}

void Display::palGreyPanel() {
	debug(9, "Display::palGreyPanel()");
	uint8 tempPal[256 * 3];
	for (int i = 224 * 3; i < 256 * 3; i += 3) {
		tempPal[i] = tempPal[i + 1] = tempPal[i + 2] = _pal.screen[i + 1] * 2 / 3;
	}
	palSet(tempPal, 224, 255, true);
}

void Display::palScroll(int start, int end) {
	debug(9, "Display::palScroll(%d, %d)", start, end);

	uint8 *palEnd = _pal.screen + end * 3;
	uint8 *palStart = _pal.screen + start * 3;

	uint8 r = *palEnd++;
	uint8 g = *palEnd++;
	uint8 b = *palEnd;

	int n = (end - start) * 3;
	while (n--) {
		*palEnd = *(palEnd - 3);
		--palEnd;
	}

	*palStart++ = r;
	*palStart++ = g;
	*palStart   = b;
}

void Display::palSetAmigaColor(uint8 color, uint16 rgb) {
	uint8 b = rgb & 0xF; rgb >>= 4;
	uint8 g = rgb & 0xF; rgb >>= 4;
	uint8 r = rgb & 0xF;
	_pal.room[color * 3] = (r << 4) | r;
	_pal.room[color * 3 + 1] = (g << 4) | g;
	_pal.room[color * 3 + 2] = (b << 4) | b;
}

void Display::palCustomColors(uint16 roomNum) {
	debug(9, "Display::palCustomColors(%d)", roomNum);
	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		switch (roomNum) {
		case 28:
			palSetAmigaColor(27, 0xC60);
			palSetAmigaColor(28, 0xA30);
			palSetAmigaColor(29, 0x810);
			palSetAmigaColor(30, 0x600);
			break;
		case 29:
			palSetAmigaColor(27, 0x58B);
			palSetAmigaColor(28, 0x369);
			palSetAmigaColor(29, 0x158);
			palSetAmigaColor(30, 0x046);
			break;
		case 30:
			palSetAmigaColor(27, 0x5A4);
			palSetAmigaColor(28, 0x384);
			palSetAmigaColor(29, 0x171);
			palSetAmigaColor(30, 0x056);
			break;
		case 31:
			palSetAmigaColor(27, 0xDA4);
			palSetAmigaColor(28, 0xB83);
			palSetAmigaColor(29, 0x873);
			palSetAmigaColor(30, 0x652);
			break;
		case 45:
			palSetAmigaColor(20, 0xA58);
			palSetAmigaColor(21, 0x845);
			break;
		}
		return;
	}
	switch (roomNum) {
	case 31:
		for (int i = 72; i < 84; i++) {
			_pal.room[i * 3 + 1] = _pal.room[i * 3 + 1] * 90 / 100;
			_pal.room[i * 3 + 2] = _pal.room[i * 3 + 2] * 70 / 100;
		}
		break;
	case 29:
		for (int i = 72; i < 84; i++) {
			_pal.room[i * 3 + 1] = _pal.room[i * 3 + 1] * 60 / 100;
			_pal.room[i * 3 + 2] = _pal.room[i * 3 + 2] * 60 / 100;
		}
		break;
	case 30:
		for (int i = 72; i < 84; i++) {
			_pal.room[i * 3 + 0] = _pal.room[i * 3 + 0] * 60 / 100;
			_pal.room[i * 3 + 1] = _pal.room[i * 3 + 1] * 80 / 100;
		}
		break;
	case 28:
		for (int i = 72; i < 84; i++) {
			_pal.room[i * 3 + 0] = _pal.room[i * 3 + 0] * 80 / 100;
			_pal.room[i * 3 + 2] = _pal.room[i * 3 + 1] * 60 / 100;
		}
		break;
	}
}

void Display::palCustomScroll(uint16 roomNum) {
	debug(9, "Display::palCustomScroll(%d)", roomNum);
	static int16 scrollx = 0;

	if (!_pal.scrollable) {
		return;
	}

	int hiPal = 0;
	int loPal = 255;
	int i;

	++scrollx;

	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		switch (roomNum) {
		case 4:
			if ((scrollx & 1) == 0) {
				palScroll(24, 26);
				loPal = 24;
				hiPal = 26;
			}
			break;
		case 74:
			palScroll(28, 31);
			loPal = 28;
			hiPal = 31;
			break;
		}
		_pal.dirtyMin = MIN(_pal.dirtyMin, loPal);
		_pal.dirtyMax = MAX(_pal.dirtyMax, hiPal);
		return;
	}

	switch (roomNum) {
	case 123: {
			static int16 j = 0, jdir = 2;
			for (i = 96; i < 111; ++i) {
				_pal.screen[i * 3 + 0] = MIN(255, _pal.room[i * 3 + 0] + j * 8);
				_pal.screen[i * 3 + 1] = MIN(255, _pal.room[i * 3 + 1] + j * 4);
			}
			j += jdir;
			if (j <= 0 || j >= 18) {
				jdir = -jdir;
			}
			loPal = 96;
			hiPal = 111;
		}
		break;
	case 124: {
			static int16 j = 0,jdir = 2;
			for (i = 80; i < 144; ++i) {
				_pal.screen[i * 3 + 0] = MIN(255, _pal.room[i * 3 + 0] + j * 8);
				_pal.screen[i * 3 + 1] = MIN(255, _pal.room[i * 3 + 1] + j * 4);
			}
			j += jdir;
			if (j <= 0 || j >= 14) {
				jdir = -jdir;
				if (_rnd.getRandomNumber(1)) {
					if (ABS(jdir) == 1) {
						jdir *= 2;
					} else {
						jdir /= 2;
					}
				}
			}
			loPal = 80;
			hiPal = 143;
		}
		break;
	case 125:
		palScroll(32, 63);
		palScroll(64, 95);
		loPal = 32;
		hiPal = 95;
		break;
	case 100:
		if (scrollx & 1) {
			palScroll(128, 132);
			palScroll(133, 137);
			palScroll(138, 143);
			loPal = 128;
			hiPal = 143;
		}
		break;
	case 102:
		if (scrollx & 1) {
			palScroll(112, 127);
			loPal = 112;
			hiPal = 127;
		}
		break;
	case 62:
		if (scrollx & 1) {
			palScroll(108, 119);
			loPal = 108;
			hiPal = 119;
		}
		break;
	case 25:
		palScroll(116, 123);
		loPal = 116;
		hiPal = 123;
		break;
	case 59:
		if (scrollx & 1) {
			palScroll(56, 63);
			loPal = 56;
			hiPal = 63;
		}
		break;
	case 39:
		palScroll(112, 143);
		loPal = 112;
		hiPal = 143;
		break;
	case 74:
		palScroll(28, 31);
		palScroll(88, 91);
		palScroll(92, 95);
		palScroll(128, 135);
		if (scrollx & 1) {
			palScroll(136, 143);
		}
		loPal = 28;
		hiPal = 143;
		break;
	case 40:
		if (scrollx & 1) {
			palScroll(96, 103);
		}
		if (scrollx & 3) {
			palScroll(104, 107);
		}
		loPal = 96;
		hiPal = 107;
		break;
	case 97:
		if (scrollx & 1) {
			palScroll(96, 107);
			palScroll(108, 122);
			loPal = 96;
			hiPal = 122;
		}
		break;
	case 55:
		palScroll(128, 143);
		loPal = 128;
		hiPal = 143;
		break;
	case 57:
		palScroll(128, 143);
		if (scrollx & 1) {
			palScroll(96, 103);
		}
		loPal = 96;
		hiPal = 143;
		break;
	case 76:
		palScroll(88, 95);
		loPal = 88;
		hiPal = 95;
		break;
	case 2:
		if (scrollx & 1) {
			palScroll(120, 127);
			loPal = 120;
			hiPal = 127;
		}
		break;
	case 3:
	case 5:
		if (scrollx & 1) {
			palScroll(128, 135);
			palScroll(136, 143);
			loPal = 128;
			hiPal = 143;
		}
		break;
	case 7:
		if (scrollx & 1) {
			palScroll(119, 127);
			loPal = 119;
			hiPal = 127;
		}
		break;
	case 42:
		if (scrollx & 1) {
			palScroll(118, 127);
			palScroll(136, 143);
			loPal = 118;
			hiPal = 143;
		}
		break;
	case 4:
		if (scrollx & 1) {
			palScroll(32,47);
		}
		palScroll(64, 70);
		palScroll(71, 79);
		loPal = 32;
		hiPal = 79;
		break;
	case 8:
		if (scrollx & 1) {
			palScroll(120, 127);
		}
		loPal = 120;
		hiPal = 127;
		break;
	case 12:
	case 64:
		if (scrollx & 1) {
			palScroll(112, 119);
			palScroll(120, 127);
			loPal = 112;
			hiPal = 127;
		}
		break;
	case 49:
		palScroll(101, 127);
		loPal = 101;
		hiPal = 127;
		break;
	}
	_pal.dirtyMin = MIN(_pal.dirtyMin, loPal);
	_pal.dirtyMax = MAX(_pal.dirtyMax, hiPal);
}

void Display::palCustomFlash() {
	uint8 tempPal[256 * 3];
	memset(tempPal, 255, 17 * 3);
	memset(tempPal + 17 * 3, 0, 67 * 3);
	memset(tempPal + 67 * 3, 255, 172 * 3);
	// set flash palette
	palSet(tempPal, 0, 255, true);
	// restore original palette
	palSet(_pal.screen, 0, 255, true);
}

void Display::palCustomLightsOff(uint16 roomNum) {
	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		memset(_pal.screen, 0, 31 * 3);
		_pal.screen[31 * 3] = 15;
		_pal.screen[31 * 3 + 1] = 15;
		_pal.screen[31 * 3 + 2] = 0;
		palSet(_pal.screen, 0, 31, true);
	} else {
		const int end = 223;
		const int start = (roomNum == ROOM_FLODA_FRONTDESK) ? 32 : 16;
		const int n = end - start + 1;
		memset(_pal.screen + start * 3, 0, n * 3);
		palSet(_pal.screen, start, end, true);
	}
	_pal.scrollable = false;
}

void Display::palCustomLightsOn(uint16 roomNum) {
	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		memcpy(_pal.screen, _pal.room, 32 * 3);
		palSet(_pal.screen, 0, 31, true);
	} else {
		const int end = 223;
		const int start = (roomNum == ROOM_FLODA_FRONTDESK) ? 32 : 0;
		const int n = end - start + 1;
		memcpy(_pal.screen + start * 3, _pal.room + start * 3, n * 3);
		palSet(_pal.screen, start, end, true);
	}
	_pal.dirtyMin = 0;
	_pal.dirtyMax = 223;
	_pal.scrollable = true;
}

int Display::getNumColorsForRoom(uint16 room) const {
	int n = 224;
	if (room >= 114 && room <= 125) {
		n = 256;
	}
	return n;
}

bool Display::isPalFadingDisabled(uint16 room) const {
	// introduction rooms don't fade palette
	return (room >= 90 && room <= 94) || (room >= 115 && room <= 125);
}

void Display::screenMode(int comPanel, bool inCutaway) {
	debug(6, "Display::screenMode(%d, %d)", comPanel, inCutaway);

	if (comPanel == 2 && inCutaway) {
		fullscreen((_bdHeight == GAME_SCREEN_HEIGHT));
	} else if (comPanel == 1) {
		fullscreen(false);
	}
}

void Display::prepareUpdate() {
	int h = GAME_SCREEN_HEIGHT;
	if (!_fullscreen) {
		h = ROOM_ZONE_HEIGHT;
		memcpy(_screenBuf + SCREEN_W * ROOM_ZONE_HEIGHT, _panelBuf, PANEL_W * PANEL_H);
	}
	uint8 *dst = _screenBuf;
	const uint8 *src = _backdropBuf + _horizontalScroll;

	while (h--) {
		memcpy(dst, src, SCREEN_W);
		dst += SCREEN_W;
		src += BACKDROP_W;
	}
}

void Display::update(bool dynalum, int16 dynaX, int16 dynaY) {
	drawTexts();
	if (_pal.scrollable && dynalum) {
		dynalumUpdate(dynaX, dynaY);
	}
	if (_pal.dirtyMin != 144 || _pal.dirtyMax != 144) {
		palSet(_pal.screen, _pal.dirtyMin, _pal.dirtyMax);
		_pal.dirtyMin = 144;
		_pal.dirtyMax = 144;
	}
	// uncomment this line to disable the dirty blocks rendering
//	_fullRefresh = 1;
	if (_fullRefresh) {
		_system->copyRectToScreen(_screenBuf, SCREEN_W, 0, 0, SCREEN_W, SCREEN_H);
		_system->updateScreen();
		--_fullRefresh;
		if (_fullRefresh) {
			memset(_dirtyBlocks, 0, _dirtyBlocksWidth * _dirtyBlocksHeight);
		}
		debug(9, "Display::update() - Full blit (%d)", _fullRefresh);
	} else {
		uint16 count = 0;
		uint8 *scrBuf = _screenBuf;
		uint8 *dbBuf = _dirtyBlocks;
		for (int j = 0; j < _dirtyBlocksHeight; ++j) {
			uint16 accW = 0;
			for (int i = 0; i < _dirtyBlocksWidth; ++i) {
				if (dbBuf[i] != 0) {
					--dbBuf[i];
					++accW;
				} else if (accW != 0) {
					int x = (i - accW) * D_BLOCK_W;
					_system->copyRectToScreen(scrBuf + x, SCREEN_W, x, j * D_BLOCK_H, accW * D_BLOCK_W, D_BLOCK_H);
					accW = 0;
					++count;
				}
			}
			if (accW != 0) {
				int x = (_dirtyBlocksWidth - accW) * D_BLOCK_W;
				_system->copyRectToScreen(scrBuf + x, SCREEN_W, x, j * D_BLOCK_H, accW * D_BLOCK_W, D_BLOCK_H);
				++count;
			}
			dbBuf += _dirtyBlocksWidth;
			scrBuf += SCREEN_W * D_BLOCK_H;
		}
		if (count != 0) {
			_system->updateScreen();
		}
		debug(9, "Display::update() - Dirtyblocks blit (%d)", count);
	}
}

void Display::setupPanel() {
	uint16 panelWidth, panelHeight;

	uint32 dataSize;
	char dataName[20];
	sprintf(dataName, "PANEL.%s", _imageExt);
	uint8 *data = _vm->resource()->loadFile(dataName, 0, &dataSize);

	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		decodeLBM(data, dataSize, _panelBuf, PANEL_W, &panelWidth, &panelHeight, _pal.panel, 0, 32, 144);
	} else {
		WRITE_LE_UINT16(data + 14, PANEL_H - 10);
		decodePCX(data, dataSize, _panelBuf + PANEL_W * 10, PANEL_W, &panelWidth, &panelHeight, _pal.panel, 144, 256);
	}
	palSetPanel();

	delete[] data;
}

void Display::setupNewRoom(const char *name, uint16 room) {
	dynalumInit(name, room);

	uint32 dataSize;
	char dataName[20];
	sprintf(dataName, "%s.%s", name, _imageExt);
	uint8 *data = _vm->resource()->loadFile(dataName, 0, &dataSize);

	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		decodeLBM(data, dataSize, _backdropBuf, BACKDROP_W, &_bdWidth, &_bdHeight, _pal.room, 0, 32);
		if (_bdHeight < BACKDROP_H) {
			memset(_backdropBuf + _bdHeight * BACKDROP_W, 0, (BACKDROP_H - _bdHeight) * BACKDROP_W);
		}
	} else {
		int n = getNumColorsForRoom(room);
		if (n != 256) {
			n = 144;
		}
		decodePCX(data, dataSize, _backdropBuf, BACKDROP_W, &_bdWidth, &_bdHeight, _pal.room, 0, n);
	}

	delete[] data;
	palCustomColors(room);
	forceFullRefresh();
}

void Display::drawBobSprite(const uint8 *data, uint16 x, uint16 y, uint16 w, uint16 h, uint16 pitch, bool xflip) {
	blit(_screenBuf, SCREEN_W, x, y, data, pitch, w, h, xflip, true);
	setDirtyBlock(xflip ? (x - w + 1) : x, y, w, h);
}

void Display::drawBobPasteDown(const uint8 *data, uint16 x, uint16 y, uint16 w, uint16 h) {
	blit(_backdropBuf, BACKDROP_W, x, y, data, w, w, h, false, true);
}

void Display::drawInventoryItem(const uint8 *data, uint16 x, uint16 y, uint16 w, uint16 h) {
	if (data != NULL) {
		if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
			uint8 *dst = _panelBuf + y * PANEL_W + x;
			for (int j = 0; j < h; ++j) {
				for (int i = 0; i < w; ++i) {
					dst[i] = 144 + *data++;
				}
				dst += PANEL_W;
			}
		} else {
			blit(_panelBuf, PANEL_W, x, y, data, w, w, h, false, false);
		}
	} else {
		fill(_panelBuf, PANEL_W, x, y, w, h, getInkColor(INK_BG_PANEL));
	}
	setDirtyBlock(x, 150 + y, w, h);
}

void Display::blit(uint8 *dstBuf, uint16 dstPitch, uint16 x, uint16 y, const uint8 *srcBuf, uint16 srcPitch, uint16 w, uint16 h, bool xflip, bool masked) {
	assert(w <= dstPitch);
	dstBuf += dstPitch * y + x;

	if (!masked) { // Unmasked always unflipped
		while (h--) {
			memcpy(dstBuf, srcBuf, w);
			srcBuf += srcPitch;
			dstBuf += dstPitch;
		}
	} else if (!xflip) { // Masked bitmap unflipped
		while (h--) {
			for (int i = 0; i < w; ++i) {
				uint8 b = *(srcBuf + i);
				if (b != 0) {
					*(dstBuf + i) = b;
				}
			}
			srcBuf += srcPitch;
			dstBuf += dstPitch;
		}
	} else { // Masked bitmap flipped
		while (h--) {
			for (int i = 0; i < w; ++i) {
				uint8 b = *(srcBuf + i);
				if (b != 0) {
					*(dstBuf - i) = b;
				}
			}
			srcBuf += srcPitch;
			dstBuf += dstPitch;
		}
	}
}

void Display::fill(uint8 *dstBuf, uint16 dstPitch, uint16 x, uint16 y, uint16 w, uint16 h, uint8 color) {
	assert(w <= dstPitch);
	dstBuf += dstPitch * y + x;
	while (h--) {
		memset(dstBuf, color, w);
		dstBuf += dstPitch;
	}
}

void Display::decodePCX(const uint8 *src, uint32 srcSize, uint8 *dst, uint16 dstPitch, uint16 *w, uint16 *h, uint8 *pal, uint16 palStart, uint16 palEnd) {
	*w = READ_LE_UINT16(src + 12);
	*h = READ_LE_UINT16(src + 14);

	assert(palStart <= palEnd && palEnd <= 256);
	const uint8 *palData = src + srcSize - 768;
	memcpy(pal, palData + palStart * 3, (palEnd - palStart) * 3);

	src += 128;
	for (int y = 0; y < *h; ++y) {
		uint8 *p = dst;
		while (p < dst + *w) {
			uint8 col = *src++;
			if ((col & 0xC0) == 0xC0) {
				uint8 len = col & 0x3F;
				memset(p, *src++, len);
				p += len;
			} else {
				*p++ = col;
			}
		}
		dst += dstPitch;
	}
}

void Display::decodeLBM(const uint8 *src, uint32 srcSize, uint8 *dst, uint16 dstPitch, uint16 *w, uint16 *h, uint8 *pal, uint16 palStart, uint16 palEnd, uint8 colorBase) {
	int planeCount = 0, planePitch = 0;
	const uint8 *srcEnd = src + srcSize;
	src += 12;
	while (src < srcEnd) {
		uint32 type = READ_BE_UINT32(src);
		uint32 size = READ_BE_UINT32(src + 4);
		src += 8;
		switch (type) {
		case MKTAG('B','M','H','D'): {
				*w = READ_BE_UINT16(src + 0);
				*h = READ_BE_UINT16(src + 2);
				planeCount = src[8];
				planePitch = ((*w + 15) >> 4) * 2;
			}
			break;
		case MKTAG('C','M','A','P'): {
				assert(palStart <= palEnd && palEnd <= size / 3);
				memcpy(pal, src + palStart * 3, (palEnd - palStart) * 3);
			}
			break;
		case MKTAG('B','O','D','Y'): {
				uint32 planarSize = (*h) * planeCount * planePitch;
				uint8 *planarBuf = new uint8[planarSize];
				uint8 *dstPlanar = planarBuf;
				for (int y = 0; y < *h; ++y) {
					for (int p = 0; p < planeCount; ++p) {
						const uint8 *end = dstPlanar + planePitch;
						while (dstPlanar < end) {
							int code = (int8)*src++;
							if (code != -128) {
								if (code < 0) {
									code = -code + 1;
									memset(dstPlanar, *src++, code);
								} else {
									++code;
									memcpy(dstPlanar, src, code);
									src += code;
								}
								dstPlanar += code;
							}
						}
					}
				}
				src = planarBuf;
				for (int y = 0; y < *h; ++y) {
					for (int x = 0; x < *w / 8; ++x) {
						for (int b = 0; b < 8; ++b) {
							const uint8 mask = (1 << (7 - b));
							uint8 color = 0;
							for (int p = 0; p < planeCount; ++p) {
								if (src[planePitch * p + x] & mask) {
									color |= 1 << p;
								}
							}
							dst[x * 8 + b] = colorBase + color;
						}
					}
					src += planeCount * planePitch;
					dst += dstPitch;
				}
				delete[] planarBuf;
			}
			return;
		}
		src += size;
	}
}

void Display::horizontalScrollUpdate(int16 xCamera) {
	debug(9, "Display::horizontalScrollUpdate(%d)", xCamera);
	if (_bdWidth <= 320) {
		horizontalScroll(0);
	} else {
		if (xCamera > 160 && xCamera < 480) {
			horizontalScroll(xCamera - 160);
		} else if (xCamera >= 480) {
			horizontalScroll(320);
		} else {
			horizontalScroll(0);
		}
	}
}

void Display::horizontalScroll(int16 scroll) {
	if (_horizontalScroll != scroll) {
		_fullRefresh = 2;
		_horizontalScroll = scroll;
	}
}

void Display::setDirtyBlock(uint16 x, uint16 y, uint16 w, uint16 h) {
	if (_fullRefresh < 2) {
		if (x >= SCREEN_W) {
			return;
		} else if (x + w > SCREEN_W) {
			w = SCREEN_W - x;
		}
		if (y >= SCREEN_H) {
			return;
		} else if (y + h > SCREEN_H) {
			h = SCREEN_H - y;
		}
		uint16 ex = (x + w - 1) / D_BLOCK_W;
		uint16 ey = (y + h - 1) / D_BLOCK_H;
		x /= D_BLOCK_W;
		y /= D_BLOCK_H;
		uint8 *p = _dirtyBlocks + _dirtyBlocksWidth * y + x;
		for (; y <= ey; ++y) {
			memset(p, 2, ex - x + 1);
			p += _dirtyBlocksWidth;
		}
	}
}

void Display::setMouseCursor(uint8 *buf, uint16 w, uint16 h) {
	CursorMan.replaceCursor(buf, w, h, 1, 1, 0);
}

void Display::showMouseCursor(bool show) {
	CursorMan.showMouse(show);
}

void Display::initFont() {
	switch (_vm->resource()->getLanguage()) {
	case Common::HE_ISR:
		_font = _fontHebrew;
		break;
	case Common::RU_RUS:
		_font = _fontRussian;
		break;
	case Common::GR_GRE:
		_font = _fontGreek;
		break;
	default:
		_font = _fontRegular;
		break;
	}
	// calculate font justification sizes
	for (int i = 0; i < 256; ++i) {
		_charWidth[i] = 0;
		for (int y = 0; y < 8; ++y) {
			uint8 c = _font[i * 8 + y];
			for (int x = 0; x < 8; ++x) {
				if ((c & (0x80 >> x)) && (x > _charWidth[i])) {
					_charWidth[i] = x;
				}
			}
		}
		_charWidth[i] += 2;
	}
	_charWidth[0x20] = 4;
	--_charWidth[0x5E];
}

void Display::setText(uint16 x, uint16 y, const char *text, bool outlined) {
	if (y < GAME_SCREEN_HEIGHT) {
		if (x == 0) x = 1;
		if (y == 0) y = 1;
		TextSlot *pts = &_texts[y];
		pts->x = x;
		pts->color = _curTextColor;
		pts->outlined = outlined;
		pts->text = text;
	}
}

void Display::setTextCentered(uint16 y, const char *text, bool outlined) {
	int len = strlen(text);
	int16 x;
	while ((x = (GAME_SCREEN_WIDTH - textWidth(text, len)) / 2) <= 0) {
		++text;
		len -= 2;
	}
	assert(y < GAME_SCREEN_HEIGHT);
	TextSlot *pts = &_texts[y];
	pts->x = x;
	pts->color = _curTextColor;
	pts->outlined = outlined;
	pts->text = Common::String(text, len);
}

void Display::drawTexts() {
	for (int y = GAME_SCREEN_HEIGHT - 1; y > 0; --y) {
		const TextSlot *pts = &_texts[y];
		if (!pts->text.empty()) {
			drawText(pts->x, y, pts->color, pts->text.c_str(), pts->outlined);
		}
	}
}

void Display::clearTexts(uint16 y1, uint16 y2) {
	assert(y1 <= y2 && y2 < GAME_SCREEN_HEIGHT);
	while (y1 <= y2) {
		_texts[y1].text.clear();
		++y1;
	}
}

void Display::setupInkColors() {
	memset(_inkColors, 0, sizeof(_inkColors));
	if (_vm->resource()->getPlatform() == Common::kPlatformAmiga) {
		_inkColors[INK_BG_PANEL] = 144 + 2;
		_inkColors[INK_JOURNAL] = 144 + 16;
		_inkColors[INK_PINNACLE_ROOM] = 144 + 14;
		_inkColors[INK_CMD_SELECT] = 144 + 17;
		_inkColors[INK_CMD_NORMAL] = 144 + 1;
		_inkColors[INK_TALK_NORMAL] = 144 + 1;
		_inkColors[INK_JOE] = 144 + 17;
		_inkColors[INK_OUTLINED_TEXT] = 0;
	} else {
		_inkColors[INK_BG_PANEL] = 226;
		_inkColors[INK_JOURNAL] = 248;
		_inkColors[INK_PINNACLE_ROOM] = 243;
		_inkColors[INK_CMD_SELECT] = 255;
		_inkColors[INK_CMD_NORMAL] = 225;
		_inkColors[INK_TALK_NORMAL] = 7;
		_inkColors[INK_JOE] = 14;
		_inkColors[INK_OUTLINED_TEXT] = 16;
	}
}

void Display::setFocusRect(const Common::Rect& rect) {
	_system->setFocusRectangle(rect);
}

int Display::textCenterX(const char *text) const {
	return (GAME_SCREEN_WIDTH - textWidth(text)) / 2;
}

uint16 Display::textWidth(const char *text) const {
	return textWidth(text, strlen(text));
}

uint16 Display::textWidth(const char *text, uint16 len) const {
	assert(len <= strlen(text));
	uint16 width = 0;
	for (uint16 i = 0; i < len; ++i) {
		width += _charWidth[ (uint8)text[i] ];
	}
	return width;
}

void Display::drawChar(uint16 x, uint16 y, uint8 color, const uint8 *chr) {
	uint8 *dstBuf = _screenBuf + SCREEN_W * y + x;
	for (int j = 0; j < 8; ++j) {
		uint8 *p = dstBuf;
		uint8 c = *chr++;
		if (c != 0) {
			for (int i = 0; i < 8; ++i) {
				if (c & 0x80) {
					*p = color;
				}
				++p;
				c <<= 1;
			}
		}
		dstBuf += SCREEN_W;
	}
}

void Display::drawText(uint16 x, uint16 y, uint8 color, const char *text, bool outlined) {
	static const int dx[] = { -1, 0, 1, 1, 1, 0, -1, -1 };
	static const int dy[] = { -1, -1, -1, 0, 1, 1, 1, 0 };
	const uint8 *str = (const uint8 *)text;
	uint16 xs = x;
	while (*str && x < SCREEN_W) {
		const uint8 ch = *str++;
		const uint8 *ftch = _font + ch * 8;
		if (outlined) {
			for (int i = 0; i < 8; ++i) {
				drawChar(x + dx[i], y + dy[i], getInkColor(INK_OUTLINED_TEXT), ftch);
			}
		}
		drawChar(x, y, color, ftch);
		x += _charWidth[ch];
	}
	setDirtyBlock(xs - 1, y - 1, x - xs + 2, 8 + 2);
}

void Display::drawBox(int16 x1, int16 y1, int16 x2, int16 y2, uint8 col) {
	int i;
	for (i = y1; i <= y2; ++i) {
		_screenBuf[i * SCREEN_W + x1] = _screenBuf[i * SCREEN_W + x2] = col;
	}
	setDirtyBlock(x1, y1, 1, y2 - y1);
	setDirtyBlock(x2, y1, 1, y2 - y1);
	for (i = x1; i <= x2; ++i) {
		_screenBuf[y1 * SCREEN_W + i] = _screenBuf[y2 * SCREEN_W + i] = col;
	}
	setDirtyBlock(x1, y1, x2 - x1, 1);
	setDirtyBlock(x1, y2, x2 - x1, 1);
}

void Display::shake(bool reset) {
	_system->setShakePos(reset ? 0 : 3);
}

void Display::blankScreen() {
	static int current = 0;
	typedef void (Display::*BlankerEffect)();
	static const BlankerEffect effects[] = {
		&Display::blankScreenEffect1,
		&Display::blankScreenEffect2,
		&Display::blankScreenEffect3
	};
	(this->*effects[current])();
	current = (current + 1) % ARRAYSIZE(effects);
	forceFullRefresh();
}

void Display::blankScreenEffect1() {
	uint8 buf[32 * 32];
	while (_vm->input()->idleTime() >= Input::DELAY_SCREEN_BLANKER) {
		for (int i = 0; i < 2; ++i) {
			int x = _rnd.getRandomNumber(SCREEN_W - 32 - 2) + 1;
			int y = _rnd.getRandomNumber(SCREEN_H - 32 - 2) + 1;
			const uint8 *p = _screenBuf + SCREEN_W * y + x;
			for (int j = 0; j < 32; ++j) {
				memcpy(buf + j * 32, p, 32);
				p += SCREEN_W;
			}
			if (_rnd.getRandomNumber(1)) {
				++x;
			} else {
				--x;
			}
			if (_rnd.getRandomNumber(1)) {
				++y;
			} else {
				--y;
			}
			_system->copyRectToScreen(buf, 32, x, y, 32, 32);
			_vm->input()->delay(10);
		}
	}
}

void Display::blankScreenEffect2() {
	while (_vm->input()->idleTime() >= Input::DELAY_SCREEN_BLANKER) {
		int x = _rnd.getRandomNumber(SCREEN_W - 2);
		int y = _rnd.getRandomNumber(SCREEN_H - 2);
		uint8 *p = _screenBuf + y * SCREEN_W + x;
		uint8 c = 0;
		switch (_rnd.getRandomNumber(3)) {
		case 0:
			c = *p;
			break;
		case 1:
			c = *(p + 1);
			break;
		case 2:
			c = *(p + SCREEN_W);
			break;
		case 3:
			c = *(p + SCREEN_W + 1);
			break;
		}
		memset(p, c, 2);
		memset(p + SCREEN_W, c, 2);
		_system->copyRectToScreen(p, SCREEN_W, x, y, 2, 2);
		_vm->input()->delay(10);
	}
}

void Display::blankScreenEffect3() {
	uint32 i = 0;
	while (_vm->input()->idleTime() >= Input::DELAY_SCREEN_BLANKER) {
		if (i > 4000000) {
			memset(_screenBuf, 0, SCREEN_W * SCREEN_H);
			_system->copyRectToScreen(_screenBuf, SCREEN_W, 0, 0, SCREEN_W, SCREEN_H);
		} else {
			int x = _rnd.getRandomNumber(SCREEN_W - 2);
			int y = _rnd.getRandomNumber(SCREEN_H - 2);
			uint8 *p = _screenBuf + SCREEN_W * y + x;
			int sum = *p + *(p + 1) + *(p + SCREEN_W) + *(p + SCREEN_W + 1);
			uint8 c = (uint8)(sum / 4);
			memset(p, c, 2);
			memset(p + SCREEN_W, c, 2);
			++i;
			_system->copyRectToScreen(p, SCREEN_W, x, y, 2, 2);
		}
		_vm->input()->delay(10);
	}
}

const uint8 Display::_fontRegular[] = {
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0x00,
	0xD8, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00,
	0x30, 0x7C, 0xC0, 0x78, 0x0C, 0xF8, 0x30, 0x00, 0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00,
	0x38, 0x6C, 0x68, 0x36, 0xDC, 0xCC, 0x76, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0xC0, 0xC0, 0x60, 0x30, 0x00, 0xC0, 0x60, 0x30, 0x30, 0x30, 0x60, 0xC0, 0x00,
	0x00, 0x6C, 0x38, 0xFE, 0x38, 0x6C, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x00,
	0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x30, 0x70, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0x0C, 0x78, 0xC0, 0xC0, 0xFC, 0x00, 0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00,
	0x1C, 0x3C, 0x6C, 0xCC, 0xFC, 0x0C, 0x0C, 0x00, 0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00,
	0x78, 0xCC, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00, 0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00, 0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0xCC, 0x78, 0x00,
	0x00, 0xC0, 0xC0, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x00, 0x60, 0x60, 0x00, 0x00, 0x60, 0x60, 0xC0,
	0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00, 0x00, 0x00, 0xFC, 0x00, 0xFC, 0x00, 0x00, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x30, 0x60, 0xC0, 0x00, 0x78, 0xCC, 0x0C, 0x18, 0x30, 0x00, 0x30, 0x00,
	0x6C, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x38, 0x7C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xCC, 0xCC, 0xF8, 0x00, 0x78, 0xCC, 0xC0, 0xC0, 0xC0, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xFC, 0x00,
	0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xC0, 0xDC, 0xCC, 0xCC, 0x7C, 0x00,
	0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00,
	0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00, 0xC6, 0xCC, 0xD8, 0xF8, 0xD8, 0xCC, 0xC6, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xFC, 0x00, 0x82, 0xC6, 0xEE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00,
	0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xDC, 0x78, 0x0C,
	0xF8, 0xCC, 0xCC, 0xF8, 0xD8, 0xCC, 0xCC, 0x00, 0x78, 0xCC, 0xC0, 0x78, 0x0C, 0xCC, 0x78, 0x00,
	0xFC, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0xC6, 0xC6, 0x6C, 0x6C, 0x38, 0x38, 0x10, 0x00, 0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0xEE, 0xC6, 0x00,
	0xC6, 0x6C, 0x38, 0x10, 0x38, 0x6C, 0xC6, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x00,
	0xFC, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xFC, 0x00, 0xF0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xF0, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x03, 0x00, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x30, 0xF0, 0x00,
	0xE8, 0x4D, 0x4A, 0x48, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00,
	0xC0, 0xC0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7C, 0x00,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00,
	0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x7C, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0x38, 0x6C, 0x60, 0xF8, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x78,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x0C, 0x00, 0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0x78, 0xC0, 0xC0, 0xCC, 0xD8, 0xF0, 0xD8, 0xCC, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0xCC, 0xEE, 0xD6, 0xC6, 0xC6, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x0C,
	0x00, 0x00, 0xF8, 0xCC, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0x7C, 0xC0, 0x78, 0x0C, 0x78, 0x00,
	0x30, 0x30, 0xFC, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00, 0x00, 0x00, 0xC6, 0xD6, 0xD6, 0x6C, 0x6C, 0x00,
	0x00, 0x00, 0xCC, 0x78, 0x30, 0x78, 0xCC, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0xE0,
	0x00, 0x00, 0xFC, 0x18, 0x30, 0x60, 0xFC, 0x00, 0x38, 0x60, 0x60, 0xC0, 0x60, 0x60, 0x38, 0x00,
	0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0x00, 0xE0, 0x30, 0x30, 0x18, 0x30, 0x30, 0xE0, 0x00,
	0x38, 0x44, 0xBA, 0xAA, 0xBA, 0x44, 0x38, 0x00, 0x00, 0x98, 0x30, 0x60, 0xC8, 0x98, 0x30, 0x00,
	0x1E, 0x30, 0x60, 0x60, 0x30, 0x1E, 0x0C, 0x18, 0x00, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x0C, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x18, 0x66, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1E, 0x30, 0x60, 0x60, 0x30, 0x1E, 0x0C, 0x18,
	0x18, 0x66, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x66, 0x00, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x30, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x00, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x18, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x30, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x30, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x00, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x30, 0x18, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x18, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x30, 0x18, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x18, 0x30, 0x78, 0x0C, 0x7C, 0xCC, 0x7C, 0x00, 0x0C, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x18, 0x30, 0x00, 0x78, 0xCC, 0xCC, 0x78, 0x00, 0x18, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0x71, 0x8E, 0x00, 0x7C, 0x66, 0x66, 0x66, 0x00, 0x71, 0xCE, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0x00,
	0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00, 0x3C, 0x60, 0x3C, 0x66, 0x3C, 0x06, 0x3C, 0x00,
	0x18, 0x00, 0x18, 0x30, 0x60, 0x66, 0x3C, 0x00, 0x3F, 0x40, 0x4E, 0x58, 0x4E, 0x40, 0x3F, 0x00,
	0x1C, 0xA4, 0xC4, 0xBC, 0x80, 0xFE, 0x00, 0x00, 0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00,
	0x3E, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x81, 0xB9, 0xA5, 0xB9, 0xA5, 0x81, 0x7E, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x78, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00,
	0xF0, 0x18, 0x30, 0x60, 0xF8, 0x00, 0x00, 0x00, 0xF0, 0x18, 0x30, 0x18, 0xF0, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xFE, 0xC0,
	0x3E, 0x7A, 0x7A, 0x3A, 0x0A, 0x0A, 0x0A, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0x60, 0xE0, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x38, 0x44, 0x44, 0x38, 0x00, 0x7C, 0x00, 0x00, 0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00,
	0x40, 0xC6, 0x4C, 0x58, 0x32, 0x66, 0xCF, 0x02, 0x40, 0xC6, 0x4C, 0x58, 0x3E, 0x62, 0xC4, 0x0E,
	0xC0, 0x23, 0x66, 0x2C, 0xD9, 0x33, 0x67, 0x01, 0x18, 0x00, 0x18, 0x30, 0x60, 0x66, 0x3C, 0x00,
	0x60, 0x30, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x0C, 0x18, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x38, 0xC6, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x71, 0x8E, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x6C, 0x00, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x38, 0x44, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x1F, 0x3C, 0x3C, 0x6F, 0x7C, 0xCC, 0xCF, 0x00, 0x1E, 0x30, 0x60, 0x60, 0x30, 0x1E, 0x0C, 0x18,
	0x60, 0x30, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00, 0x18, 0x30, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00,
	0x30, 0xCC, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00, 0xCC, 0x00, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00,
	0x60, 0x30, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00, 0x18, 0x30, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x30, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00, 0xCC, 0x00, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x78, 0x6C, 0x66, 0xF6, 0x66, 0x6C, 0x78, 0x00, 0x71, 0xCE, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0x00,
	0x30, 0x18, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x0C, 0x18, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0x18, 0x66, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0xC3, 0x3C, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00, 0x00,
	0x3F, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0xFC, 0x00, 0x30, 0x18, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x0C, 0x18, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x18, 0x24, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x66, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x06, 0x08, 0xC3, 0x66, 0x3C, 0x18, 0x18, 0x00,
	0x60, 0x60, 0x7E, 0x63, 0x7E, 0x60, 0x60, 0x00, 0x3C, 0x66, 0x66, 0x6C, 0x66, 0x66, 0x6C, 0x60,
	0x30, 0x18, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x0C, 0x18, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x18, 0x66, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x71, 0x8E, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x66, 0x00, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x18, 0x24, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x00, 0x00, 0x7E, 0x1B, 0x7F, 0xD8, 0x77, 0x00, 0x00, 0x00, 0x3C, 0x60, 0x60, 0x60, 0x3C, 0x18,
	0x30, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x0C, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x18, 0x66, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x66, 0x00, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x30, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x0C, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x18, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x60, 0xFC, 0x18, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x00, 0x7C, 0x66, 0x66, 0x66, 0x00,
	0x30, 0x18, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x0C, 0x18, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x18, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x00, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x00, 0x18, 0x00, 0x7E, 0x00, 0x18, 0x00, 0x00,
	0x00, 0x02, 0x7C, 0xCE, 0xD6, 0xE6, 0x7C, 0x80, 0x30, 0x18, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x0C, 0x18, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x18, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x00, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x0C, 0x18, 0x00, 0x66, 0x66, 0x3C, 0x18, 0x30,
	0x60, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x60, 0x60, 0x00, 0x66, 0x00, 0x66, 0x66, 0x3C, 0x18, 0x30
};

const uint8 Display::_fontGreek[] = {
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0x00,
	0xD8, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00,
	0x30, 0x7C, 0xC0, 0x78, 0x0C, 0xF8, 0x30, 0x00, 0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00,
	0x38, 0x6C, 0x68, 0x36, 0xDC, 0xCC, 0x76, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0xC0, 0xC0, 0x60, 0x30, 0x00, 0xC0, 0x60, 0x30, 0x30, 0x30, 0x60, 0xC0, 0x00,
	0x00, 0x6C, 0x38, 0xFE, 0x38, 0x6C, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x00,
	0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x30, 0x70, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0x0C, 0x78, 0xC0, 0xC0, 0xFC, 0x00, 0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00,
	0x1C, 0x3C, 0x6C, 0xCC, 0xFC, 0x0C, 0x0C, 0x00, 0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00,
	0x78, 0xCC, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00, 0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00, 0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0xCC, 0x78, 0x00,
	0x00, 0xC0, 0xC0, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x00, 0x60, 0x60, 0x00, 0x00, 0x60, 0x60, 0xC0,
	0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00, 0x00, 0x00, 0xFC, 0x00, 0xFC, 0x00, 0x00, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x30, 0x60, 0xC0, 0x00, 0x78, 0xCC, 0x0C, 0x18, 0x30, 0x00, 0x30, 0x00,
	0x6C, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x38, 0x7C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xCC, 0xCC, 0xF8, 0x00, 0x78, 0xCC, 0xC0, 0xC0, 0xC0, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xFC, 0x00,
	0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xC0, 0xDC, 0xCC, 0xCC, 0x7C, 0x00,
	0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00,
	0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00, 0xC6, 0xCC, 0xD8, 0xF8, 0xD8, 0xCC, 0xC6, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xFC, 0x00, 0x82, 0xC6, 0xEE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00,
	0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xDC, 0x78, 0x0C,
	0xF8, 0xCC, 0xCC, 0xF8, 0xD8, 0xCC, 0xCC, 0x00, 0x78, 0xCC, 0xC0, 0x78, 0x0C, 0xCC, 0x78, 0x00,
	0xFC, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0xC6, 0xC6, 0x6C, 0x6C, 0x38, 0x38, 0x10, 0x00, 0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0xEE, 0xC6, 0x00,
	0xC6, 0x6C, 0x38, 0x10, 0x38, 0x6C, 0xC6, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x00,
	0xFC, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xFC, 0x00, 0xF0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xF0, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x03, 0x00, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x30, 0xF0, 0x00,
	0xE8, 0x4D, 0x4A, 0x48, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00,
	0xC0, 0xC0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7C, 0x00,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00,
	0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x7C, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0x38, 0x6C, 0x60, 0xF8, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x78,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x0C, 0x00, 0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0x78, 0xC0, 0xC0, 0xCC, 0xD8, 0xF0, 0xD8, 0xCC, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0xCC, 0xEE, 0xD6, 0xC6, 0xC6, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x0C,
	0x00, 0x00, 0xF8, 0xCC, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0x7C, 0xC0, 0x78, 0x0C, 0x78, 0x00,
	0x30, 0x30, 0xFC, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00, 0x00, 0x00, 0xC6, 0xD6, 0xD6, 0x6C, 0x6C, 0x00,
	0x00, 0x00, 0xCC, 0x78, 0x30, 0x78, 0xCC, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0xE0,
	0x00, 0x00, 0xFC, 0x18, 0x30, 0x60, 0xFC, 0x00, 0x38, 0x60, 0x60, 0xC0, 0x60, 0x60, 0x38, 0x00,
	0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0x00, 0xE0, 0x30, 0x30, 0x18, 0x30, 0x30, 0xE0, 0x00,
	0x38, 0x44, 0xBA, 0xAA, 0xBA, 0x44, 0x38, 0x00, 0x00, 0x98, 0x30, 0x60, 0xC8, 0x98, 0x30, 0x00,
	0x38, 0x64, 0xF0, 0x60, 0xF0, 0x64, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x5A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x9C, 0xBE, 0x63, 0x63, 0x7F, 0x63, 0x63, 0x00, 0x38, 0x6C, 0x60, 0xF0, 0x60, 0x60, 0xFC, 0x00,
	0x42, 0x3C, 0x66, 0x3C, 0x42, 0x00, 0x00, 0x00, 0xC3, 0x66, 0x3C, 0x18, 0x3C, 0x18, 0x18, 0x00,
	0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00, 0x3C, 0x60, 0x3C, 0x66, 0x3C, 0x06, 0x3C, 0x00,
	0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 0x3C, 0x00, 0x3F, 0x40, 0x4E, 0x58, 0x4E, 0x40, 0x3F, 0x00,
	0x1C, 0xA4, 0xC4, 0xBC, 0x80, 0xFE, 0x00, 0x00, 0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00,
	0x3E, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x81, 0xB9, 0xA5, 0xB9, 0xA5, 0x81, 0x7E, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x78, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00,
	0xF0, 0x18, 0x30, 0x60, 0xF8, 0x00, 0x00, 0x00, 0xF0, 0x18, 0x30, 0x18, 0xF0, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xFE, 0xC0,
	0x3E, 0x7A, 0x7A, 0x3A, 0x0A, 0x0A, 0x0A, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0xBF, 0xB0, 0x30, 0x3C, 0x30, 0x30, 0x3F, 0x00, 0xB3, 0xB3, 0x33, 0x3F, 0x33, 0x33, 0x33, 0x00,
	0xBC, 0x98, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00,
	0x9E, 0xB3, 0x33, 0x33, 0x33, 0x33, 0x1E, 0x00, 0x40, 0xC6, 0x4C, 0x58, 0x3E, 0x62, 0xC4, 0x0E,
	0xB3, 0xB3, 0x33, 0x1E, 0x0C, 0x0C, 0x0C, 0x00, 0x9E, 0xB3, 0x33, 0x33, 0x1E, 0x00, 0x3F, 0x00,
	0xB4, 0x00, 0x30, 0x30, 0x30, 0x30, 0x18, 0x00, 0x38, 0x7C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xCC, 0xCC, 0xF8, 0x00, 0xFC, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x38, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xFE, 0x00, 0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xFC, 0x00,
	0xFC, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xFC, 0x00, 0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00,
	0x78, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0x78, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00,
	0xC6, 0xCC, 0xD8, 0xF8, 0xD8, 0xCC, 0xC6, 0x00, 0x38, 0x7C, 0xC6, 0xC6, 0xC6, 0xC6, 0xC6, 0x00,
	0x82, 0xC6, 0xEE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00, 0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00,
	0xFC, 0xFC, 0x00, 0x78, 0x00, 0xFC, 0xFC, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0xFC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0xC0, 0x00,
	0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00, 0xFC, 0x60, 0x30, 0x18, 0x30, 0x60, 0xFC, 0x00,
	0xFC, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x00,
	0x10, 0x7C, 0xD6, 0xD6, 0xD6, 0x7C, 0x10, 0x00, 0xC6, 0x6C, 0x38, 0x10, 0x38, 0x6C, 0xC6, 0x00,
	0xC6, 0xD6, 0xD6, 0xD6, 0x7C, 0x10, 0x10, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0xFC, 0x00,
	0xBD, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3C, 0x00, 0xB4, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x00,
	0x30, 0x00, 0x7A, 0xCC, 0xCC, 0xCC, 0x7A, 0x00, 0x30, 0x00, 0x78, 0xCC, 0x60, 0xCC, 0x78, 0x00,
	0x30, 0x00, 0xB8, 0xEC, 0xCC, 0xCC, 0xCC, 0x0C, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x00,
	0xB4, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x7A, 0xCC, 0xCC, 0xCC, 0x7A, 0x00,
	0x70, 0xD8, 0xD8, 0xF0, 0xD8, 0xD8, 0xF0, 0xC0, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30,
	0xF8, 0x60, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x78, 0xCC, 0x60, 0xCC, 0x78, 0x00,
	0xF8, 0x18, 0x30, 0x60, 0x60, 0x30, 0x18, 0x30, 0x00, 0x00, 0xB8, 0xEC, 0xCC, 0xCC, 0xCC, 0x0C,
	0x70, 0xD8, 0xD8, 0xF8, 0xD8, 0xD8, 0x70, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x60, 0x00,
	0x00, 0x00, 0xCC, 0xD8, 0xF0, 0xD8, 0xCC, 0x00, 0xF0, 0x18, 0x3C, 0x6C, 0xCC, 0xCC, 0xCC, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xFA, 0xC0, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00,
	0xF8, 0x18, 0x30, 0x18, 0x30, 0x18, 0x08, 0x10, 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xFC, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xF8, 0xC0,
	0x00, 0x00, 0x7C, 0xE0, 0x78, 0x1C, 0xF8, 0x00, 0x00, 0x00, 0x7E, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xFC, 0x30, 0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0x4C, 0xD6, 0xD6, 0xD6, 0x7C, 0x10, 0x00, 0x00, 0xCC, 0xCC, 0x30, 0x30, 0xCC, 0xCC,
	0x00, 0x00, 0x54, 0xD6, 0xD6, 0xD6, 0x7C, 0x10, 0x00, 0x00, 0x6C, 0xC6, 0xD6, 0xFE, 0x6C, 0x00,
	0x90, 0x00, 0x60, 0x60, 0x60, 0x60, 0x30, 0x00, 0x48, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x30, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x18, 0x00, 0x6C, 0xC6, 0xD6, 0xFE, 0x6C, 0x00, 0x00, 0x00, 0x18, 0x3C, 0x3C, 0x18, 0x00, 0x00
};

const uint8 Display::_fontHebrew[] = {
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00, 0xF8, 0xB0, 0xB0, 0x80, 0xB0, 0xB0, 0xC0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0x00,
	0xD8, 0xD8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6C, 0x6C, 0xFE, 0x6C, 0xFE, 0x6C, 0x6C, 0x00,
	0x30, 0x7C, 0xC0, 0x78, 0x0C, 0xF8, 0x30, 0x00, 0x00, 0xC6, 0xCC, 0x18, 0x30, 0x66, 0xC6, 0x00,
	0x38, 0x6C, 0x68, 0x36, 0xDC, 0xCC, 0x76, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0xC0, 0xC0, 0x60, 0x30, 0x00, 0xC0, 0x60, 0x30, 0x30, 0x30, 0x60, 0xC0, 0x00,
	0x00, 0x6C, 0x38, 0xFE, 0x38, 0x6C, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x60, 0xC0, 0x00, 0x00, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0x00,
	0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00, 0x30, 0x70, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0x0C, 0x78, 0xC0, 0xC0, 0xFC, 0x00, 0x78, 0xCC, 0x0C, 0x38, 0x0C, 0xCC, 0x78, 0x00,
	0x1C, 0x3C, 0x6C, 0xCC, 0xFC, 0x0C, 0x0C, 0x00, 0xFC, 0xC0, 0xF8, 0x0C, 0x0C, 0xCC, 0x78, 0x00,
	0x78, 0xCC, 0xC0, 0xF8, 0xCC, 0xCC, 0x78, 0x00, 0xFC, 0xCC, 0x0C, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x78, 0xCC, 0xCC, 0x78, 0xCC, 0xCC, 0x78, 0x00, 0x78, 0xCC, 0xCC, 0x7C, 0x0C, 0xCC, 0x78, 0x00,
	0x00, 0xC0, 0xC0, 0x00, 0x00, 0xC0, 0xC0, 0x00, 0x00, 0x60, 0x60, 0x00, 0x00, 0x60, 0x60, 0xC0,
	0x18, 0x30, 0x60, 0xC0, 0x60, 0x30, 0x18, 0x00, 0x00, 0x00, 0xFC, 0x00, 0xFC, 0x00, 0x00, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x30, 0x60, 0xC0, 0x00, 0x78, 0xCC, 0x0C, 0x18, 0x30, 0x00, 0x30, 0x00,
	0x6C, 0xFE, 0xFE, 0xFE, 0x7C, 0x38, 0x10, 0x00, 0x38, 0x7C, 0xC6, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xCC, 0xCC, 0xF8, 0x00, 0x78, 0xCC, 0xC0, 0xC0, 0xC0, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xFC, 0x00,
	0xFC, 0xC0, 0xC0, 0xF0, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xC0, 0xDC, 0xCC, 0xCC, 0x7C, 0x00,
	0xCC, 0xCC, 0xCC, 0xFC, 0xCC, 0xCC, 0xCC, 0x00, 0xF0, 0x60, 0x60, 0x60, 0x60, 0x60, 0xF0, 0x00,
	0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0xCC, 0x78, 0x00, 0xC6, 0xCC, 0xD8, 0xF8, 0xD8, 0xCC, 0xC6, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xFC, 0x00, 0x82, 0xC6, 0xEE, 0xFE, 0xD6, 0xC6, 0xC6, 0x00,
	0xC6, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0xC6, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0xC0, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0xCC, 0xDC, 0x78, 0x0C,
	0xF8, 0xCC, 0xCC, 0xF8, 0xD8, 0xCC, 0xCC, 0x00, 0x78, 0xCC, 0xC0, 0x78, 0x0C, 0xCC, 0x78, 0x00,
	0xFC, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0xC6, 0xC6, 0x6C, 0x6C, 0x38, 0x38, 0x10, 0x00, 0xC6, 0xC6, 0xC6, 0xD6, 0xFE, 0xEE, 0xC6, 0x00,
	0xC6, 0x6C, 0x38, 0x10, 0x38, 0x6C, 0xC6, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x00,
	0xFC, 0x0C, 0x18, 0x30, 0x60, 0xC0, 0xFC, 0x00, 0xF0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xF0, 0x00,
	0xC0, 0x60, 0x30, 0x18, 0x0C, 0x06, 0x03, 0x00, 0xF0, 0x30, 0x30, 0x30, 0x30, 0x30, 0xF0, 0x00,
	0xE8, 0x4D, 0x4A, 0x48, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00,
	0xC0, 0xC0, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0C, 0x7C, 0xCC, 0x7C, 0x00,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xF8, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xC0, 0xCC, 0x78, 0x00,
	0x0C, 0x0C, 0x7C, 0xCC, 0xCC, 0xCC, 0x7C, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xFC, 0xC0, 0x78, 0x00,
	0x38, 0x6C, 0x60, 0xF8, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x78,
	0xC0, 0xC0, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00,
	0x0C, 0x00, 0x0C, 0x0C, 0x0C, 0x0C, 0xCC, 0x78, 0xC0, 0xC0, 0xCC, 0xD8, 0xF0, 0xD8, 0xCC, 0x00,
	0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0xCC, 0xEE, 0xD6, 0xC6, 0xC6, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xCC, 0xCC, 0x00, 0x00, 0x00, 0x78, 0xCC, 0xCC, 0xCC, 0x78, 0x00,
	0x00, 0x00, 0xF8, 0xCC, 0xCC, 0xF8, 0xC0, 0xC0, 0x00, 0x00, 0x7C, 0xCC, 0xCC, 0x7C, 0x0C, 0x0C,
	0x00, 0x00, 0xF8, 0xCC, 0xC0, 0xC0, 0xC0, 0x00, 0x00, 0x00, 0x7C, 0xC0, 0x78, 0x0C, 0x78, 0x00,
	0x30, 0x30, 0xFC, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0x7C, 0x00,
	0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0x00, 0x00, 0x00, 0xC6, 0xD6, 0xD6, 0x6C, 0x6C, 0x00,
	0x00, 0x00, 0xCC, 0x78, 0x30, 0x78, 0xCC, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0x78, 0x30, 0xE0,
	0x00, 0x00, 0xFC, 0x18, 0x30, 0x60, 0xFC, 0x00, 0x38, 0x60, 0x60, 0xC0, 0x60, 0x60, 0x38, 0x00,
	0xC0, 0xC0, 0xC0, 0x00, 0xC0, 0xC0, 0xC0, 0x00, 0xE0, 0x30, 0x30, 0x18, 0x30, 0x30, 0xE0, 0x00,
	0x38, 0x44, 0xBA, 0xAA, 0xBA, 0x44, 0x38, 0x00, 0x00, 0x98, 0x30, 0x60, 0xC8, 0x98, 0x30, 0x00,
	0xCC, 0x66, 0x76, 0xBC, 0x98, 0x8C, 0xE6, 0x00, 0xFC, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0xFE, 0x00,
	0x78, 0x18, 0x18, 0x18, 0x38, 0x78, 0xD8, 0x00, 0xFE, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0xFE, 0x06, 0x06, 0xC6, 0xC6, 0xC6, 0xC6, 0x00, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x7C, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0xFE, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00,
	0xDC, 0x66, 0xE6, 0xC6, 0xC6, 0xC6, 0x7C, 0x00, 0xF0, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFE, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0xF8, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0xF8, 0x00,
	0xC0, 0xFE, 0x06, 0x06, 0x0C, 0x18, 0x18, 0x00, 0xFE, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7E, 0x00,
	0xFC, 0x76, 0x66, 0x66, 0x66, 0x66, 0x6E, 0x00, 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00, 0xFE, 0x66, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0xEE, 0x66, 0x66, 0x66, 0x66, 0x6C, 0xF8, 0x00, 0xFE, 0xC6, 0xC6, 0xF6, 0x06, 0x06, 0x06, 0x06,
	0xFE, 0xC6, 0xC6, 0xFE, 0x06, 0x06, 0xFE, 0x00, 0xFE, 0x66, 0x6C, 0x78, 0x60, 0x60, 0x60, 0x60,
	0xEE, 0x66, 0x3C, 0x18, 0x0C, 0x06, 0xFE, 0x00, 0xFE, 0x06, 0x0E, 0xD8, 0xF0, 0xF0, 0xC0, 0xC0,
	0xFC, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x00, 0xEE, 0xCA, 0xCA, 0xCA, 0xCA, 0xCA, 0x7C, 0x00,
	0xFF, 0x67, 0x67, 0x67, 0x67, 0x67, 0xE7, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x00, 0x0C, 0x3E, 0x6C, 0x3E, 0x0C, 0x00, 0x00, 0x38, 0x6C, 0x60, 0xF0, 0x60, 0x60, 0xFC, 0x00,
	0x42, 0x3C, 0x66, 0x3C, 0x42, 0x00, 0x00, 0x00, 0xC3, 0x66, 0x3C, 0x18, 0x3C, 0x18, 0x18, 0x00,
	0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00, 0x3C, 0x60, 0x3C, 0x66, 0x3C, 0x06, 0x3C, 0x00,
	0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3F, 0x40, 0x4E, 0x58, 0x4E, 0x40, 0x3F, 0x00,
	0x1C, 0xA4, 0xC4, 0xBC, 0x80, 0xFE, 0x00, 0x00, 0x00, 0x33, 0x66, 0xCC, 0x66, 0x33, 0x00, 0x00,
	0x3E, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7E, 0x00, 0x00, 0x00, 0x00,
	0x81, 0xB9, 0xA5, 0xB9, 0xA5, 0x81, 0x7E, 0x00, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x78, 0xCC, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0xFC, 0x30, 0x30, 0x00, 0xFC, 0x00,
	0xF0, 0x18, 0x30, 0x60, 0xF8, 0x00, 0x00, 0x00, 0xF0, 0x18, 0x30, 0x18, 0xF0, 0x00, 0x00, 0x00,
	0x30, 0x60, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xCC, 0xCC, 0xCC, 0xCC, 0xFE, 0xC0,
	0x3E, 0x7A, 0x7A, 0x3A, 0x0A, 0x0A, 0x0A, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0x60, 0xE0, 0x60, 0x60, 0x60, 0x00, 0x00, 0x00,
	0x38, 0x44, 0x44, 0x38, 0x00, 0x7C, 0x00, 0x00, 0x00, 0xCC, 0x66, 0x33, 0x66, 0xCC, 0x00, 0x00,
	0x40, 0xC6, 0x4C, 0x58, 0x32, 0x66, 0xCF, 0x02, 0x40, 0xC6, 0x4C, 0x58, 0x3E, 0x62, 0xC4, 0x0E,
	0xC0, 0x23, 0x66, 0x2C, 0xD9, 0x33, 0x67, 0x01, 0x18, 0x00, 0x18, 0x30, 0x60, 0x66, 0x3C, 0x00,
	0x60, 0x30, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x0C, 0x18, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x38, 0xC6, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x71, 0x8E, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x6C, 0x00, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00, 0x38, 0x44, 0x7C, 0xC6, 0xFE, 0xC6, 0xC6, 0x00,
	0x1F, 0x3C, 0x3C, 0x6F, 0x7C, 0xCC, 0xCF, 0x00, 0x1E, 0x30, 0x60, 0x60, 0x30, 0x1E, 0x0C, 0x18,
	0x60, 0x30, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00, 0x18, 0x30, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00,
	0x30, 0xCC, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00, 0xCC, 0x00, 0xFC, 0xC0, 0xF0, 0xC0, 0xFC, 0x00,
	0x60, 0x30, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00, 0x18, 0x30, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x30, 0xCC, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00, 0xCC, 0x00, 0x78, 0x30, 0x30, 0x30, 0x78, 0x00,
	0x78, 0x6C, 0x66, 0xF6, 0x66, 0x6C, 0x78, 0x00, 0x71, 0xCE, 0xE6, 0xF6, 0xDE, 0xCE, 0xC6, 0x00,
	0x30, 0x18, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x0C, 0x18, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0x18, 0x66, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x3C, 0x66, 0x66, 0x66, 0x3C, 0x00,
	0xC3, 0x3C, 0x66, 0x66, 0x66, 0x66, 0x3C, 0x00, 0x00, 0xC6, 0x6C, 0x38, 0x6C, 0xC6, 0x00, 0x00,
	0x3F, 0x66, 0x6E, 0x7E, 0x76, 0x66, 0xFC, 0x00, 0x30, 0x18, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x0C, 0x18, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x18, 0x24, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x66, 0x00, 0x66, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x06, 0x08, 0xC3, 0x66, 0x3C, 0x18, 0x18, 0x00,
	0x60, 0x60, 0x7E, 0x63, 0x7E, 0x60, 0x60, 0x00, 0x3C, 0x66, 0x66, 0x6C, 0x66, 0x66, 0x6C, 0x60,
	0x30, 0x18, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x0C, 0x18, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x18, 0x66, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x71, 0x8E, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x66, 0x00, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00, 0x18, 0x24, 0x3C, 0x06, 0x3E, 0x66, 0x3E, 0x00,
	0x00, 0x00, 0x7E, 0x1B, 0x7F, 0xD8, 0x77, 0x00, 0x00, 0x00, 0x3C, 0x60, 0x60, 0x60, 0x3C, 0x18,
	0x30, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x0C, 0x18, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x18, 0x66, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00, 0x66, 0x00, 0x3C, 0x66, 0x7E, 0x60, 0x3C, 0x00,
	0x30, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x0C, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x18, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x66, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x60, 0xFC, 0x18, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x00, 0x7C, 0x66, 0x66, 0x66, 0x00,
	0x30, 0x18, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x0C, 0x18, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x18, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x71, 0x8E, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00,
	0x00, 0x66, 0x00, 0x3C, 0x66, 0x66, 0x3C, 0x00, 0x00, 0x18, 0x00, 0x7E, 0x00, 0x18, 0x00, 0x00,
	0x00, 0x02, 0x7C, 0xCE, 0xD6, 0xE6, 0x7C, 0x80, 0x30, 0x18, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x0C, 0x18, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x18, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00,
	0x00, 0x66, 0x00, 0x66, 0x66, 0x66, 0x3E, 0x00, 0x0C, 0x18, 0x00, 0x66, 0x66, 0x3C, 0x18, 0x30,
	0x60, 0x60, 0x7C, 0x66, 0x66, 0x7C, 0x60, 0x60, 0x00, 0x66, 0x00, 0x66, 0x66, 0x3C, 0x18, 0x30
};

const uint8 Display::_fontRussian[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e,
	0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e, 0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00,
	0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00, 0x38, 0x7c, 0x38, 0xfe, 0xfe, 0xd6, 0x10, 0x38,
	0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x10, 0x38, 0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00,
	0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00,
	0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff, 0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78,
	0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0,
	0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0, 0x18, 0xdb, 0x3c, 0xe7, 0xe7, 0x3c, 0xdb, 0x18,
	0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00, 0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00,
	0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00,
	0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00, 0x3e, 0x63, 0x38, 0x6c, 0x6c, 0x38, 0xcc, 0x78,
	0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff,
	0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
	0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00,
	0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00, 0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00,
	0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00, 0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x38, 0x38, 0x00, 0x38, 0x00,
	0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00,
	0x18, 0x3e, 0x60, 0x3c, 0x06, 0x7c, 0x18, 0x00, 0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00,
	0x70, 0xd8, 0xd8, 0x70, 0xdb, 0xce, 0x7b, 0x00, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0c, 0x18, 0x30, 0x30, 0x30, 0x18, 0x0c, 0x00, 0x30, 0x18, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00,
	0x18, 0xdb, 0x3c, 0xff, 0x3c, 0xdb, 0x18, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x1c, 0x38, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x00,
	0x7c, 0xc6, 0xce, 0xd6, 0xe6, 0xc6, 0x7c, 0x00, 0x18, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x7c, 0xc6, 0x06, 0x1c, 0x70, 0xc0, 0xfe, 0x00, 0x7c, 0xc6, 0x06, 0x3c, 0x06, 0xc6, 0x7c, 0x00,
	0x0e, 0x1e, 0x36, 0x66, 0xff, 0x06, 0x06, 0x00, 0xfe, 0xc0, 0xfc, 0x06, 0x06, 0xc6, 0x7c, 0x00,
	0x3c, 0x60, 0xc0, 0xfc, 0xc6, 0xc6, 0x7c, 0x00, 0xfe, 0xc6, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x00,
	0x7c, 0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0x7c, 0x00, 0x7c, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00,
	0x00, 0x00, 0x38, 0x38, 0x00, 0x38, 0x38, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x38, 0x38, 0x70,
	0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x7e, 0x00, 0x00, 0x00,
	0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0x06, 0x1c, 0x30, 0x00, 0x30, 0x00,
	0x3e, 0x63, 0xcb, 0xd7, 0xce, 0x60, 0x3e, 0x00, 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00,
	0xfc, 0xc6, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c, 0x00,
	0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xfc, 0x00, 0xfe, 0xc0, 0xc0, 0xf8, 0xc0, 0xc0, 0xfe, 0x00,
	0xfe, 0xc0, 0xc0, 0xf8, 0xc0, 0xc0, 0xc0, 0x00, 0x7c, 0xc6, 0xc0, 0xde, 0xc6, 0xc6, 0x7c, 0x00,
	0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x66, 0x3c, 0x00, 0xc6, 0xcc, 0xd8, 0xf0, 0xd8, 0xcc, 0xc6, 0x00,
	0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0xc3, 0xe7, 0xff, 0xdb, 0xdb, 0xc3, 0xc3, 0x00,
	0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
	0xfc, 0xc6, 0xc6, 0xfc, 0xc0, 0xc0, 0xc0, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xda, 0xcc, 0x76, 0x00,
	0xfc, 0xc6, 0xc6, 0xfc, 0xd8, 0xcc, 0xc6, 0x00, 0x7c, 0xc6, 0xc0, 0x7c, 0x06, 0xc6, 0x7c, 0x00,
	0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
	0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x38, 0x38, 0x00, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0xe7, 0xc3, 0x00,
	0xc6, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0xc6, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x00,
	0xfe, 0x06, 0x0c, 0x38, 0x60, 0xc0, 0xfe, 0x00, 0x3e, 0x30, 0x30, 0x30, 0x30, 0x30, 0x3e, 0x00,
	0x00, 0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x00, 0x7c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x7c, 0x00,
	0x18, 0x3c, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
	0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x06, 0x7e, 0xc6, 0x7e, 0x00,
	0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x3e, 0x60, 0x60, 0x60, 0x3e, 0x00,
	0x06, 0x06, 0x7e, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0x7e, 0x00,
	0x1c, 0x36, 0x30, 0xfc, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x7e, 0x06, 0x7c,
	0xc0, 0xc0, 0xdc, 0xe6, 0xc6, 0xc6, 0xc6, 0x00, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00,
	0x06, 0x00, 0x06, 0x06, 0x06, 0x06, 0x66, 0x3c, 0x60, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0x00,
	0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x66, 0xdb, 0xdb, 0xdb, 0xdb, 0x00,
	0x00, 0x00, 0x7c, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
	0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xfc, 0xc0, 0xc0, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x7e, 0x06, 0x06,
	0x00, 0x00, 0xdc, 0xe6, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x7e, 0xc0, 0x7c, 0x06, 0xfc, 0x00,
	0x18, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00,
	0x00, 0x00, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xdb, 0xdb, 0x66, 0x00,
	0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xce, 0x76, 0x06, 0x7c,
	0x00, 0x00, 0xfe, 0x0c, 0x38, 0x60, 0xfe, 0x00, 0x0e, 0x18, 0x18, 0x70, 0x18, 0x18, 0x0e, 0x00,
	0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x1e, 0x18, 0x18, 0x70, 0x00,
	0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xee, 0xc6, 0xfe, 0xfe, 0x00,
	0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0xfc, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xfc, 0x00,
	0xfc, 0xc6, 0xc6, 0xfc, 0xc6, 0xc6, 0xfc, 0x00, 0xfe, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0xc0, 0x00,
	0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xfc, 0x00, 0xfe, 0xc0, 0xc0, 0xf8, 0xc0, 0xc0, 0xfe, 0x00,
	0xd6, 0xd6, 0xd6, 0x7c, 0xd6, 0xd6, 0xd6, 0x00, 0x7c, 0xc6, 0x06, 0x1c, 0x06, 0xc6, 0x7c, 0x00,
	0xc6, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0xc6, 0x00, 0xd6, 0xd6, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0x00,
	0xc6, 0xcc, 0xd8, 0xf0, 0xd8, 0xcc, 0xc6, 0x00, 0x3c, 0x66, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00,
	0xc3, 0xe7, 0xff, 0xdb, 0xdb, 0xc3, 0xc3, 0x00, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00,
	0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00,
	0xfc, 0xc6, 0xc6, 0xc6, 0xfc, 0xc0, 0xc0, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c, 0x00,
	0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x7c, 0x00,
	0x10, 0x7c, 0xd6, 0xd6, 0xd6, 0x7c, 0x10, 0x00, 0xc6, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0xc6, 0x00,
	0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xff, 0x00, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x06, 0x00,
	0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xfe, 0x00, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xd6, 0xff, 0x01,
	0xe0, 0x60, 0x60, 0x7c, 0x66, 0x66, 0x7c, 0x00, 0xc6, 0xc6, 0xc6, 0xf6, 0xde, 0xde, 0xf6, 0x00,
	0xc0, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xfc, 0x00, 0x7c, 0xc6, 0x06, 0x3e, 0x06, 0xc6, 0x7c, 0x00,
	0xcc, 0xd6, 0xd6, 0xf6, 0xd6, 0xd6, 0xcc, 0x00, 0x7e, 0xc6, 0xc6, 0x7e, 0x36, 0x66, 0xc6, 0x00,
	0x00, 0x00, 0x7c, 0x06, 0x7e, 0xc6, 0x7e, 0x00, 0x3e, 0x60, 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x00,
	0x00, 0x00, 0x7c, 0x66, 0x7c, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x7e, 0x60, 0x60, 0x60, 0x60, 0x00,
	0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0xff, 0x81, 0x00, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0x7e, 0x00,
	0x00, 0x00, 0xd6, 0xd6, 0x7c, 0xd6, 0xd6, 0x00, 0x00, 0x00, 0x3c, 0x46, 0x1c, 0x46, 0x3c, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x6e, 0x76, 0x66, 0x00, 0x00, 0x18, 0x66, 0x66, 0x6e, 0x76, 0x66, 0x00,
	0x00, 0x00, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0xc6, 0x00,
	0x00, 0x00, 0xe7, 0xdb, 0xdb, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x66, 0x66, 0x7e, 0x66, 0x66, 0x00,
	0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x7e, 0x66, 0x66, 0x66, 0x66, 0x00,
	0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa,
	0xdb, 0x77, 0xdb, 0xee, 0xdb, 0x77, 0xdb, 0xee, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18,
	0x66, 0x66, 0x66, 0x66, 0xe6, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66, 0x66, 0x66,
	0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x66, 0x66, 0xe6, 0x06, 0xe6, 0x66, 0x66, 0x66,
	0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0xfe, 0x06, 0xe6, 0x66, 0x66, 0x66,
	0x66, 0x66, 0xe6, 0x06, 0xfe, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0xfe, 0x00, 0x00, 0x00,
	0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18,
	0x18, 0x18, 0x1e, 0x18, 0x1e, 0x18, 0x18, 0x18, 0x66, 0x66, 0x66, 0x66, 0x67, 0x66, 0x66, 0x66,
	0x66, 0x66, 0x67, 0x60, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x60, 0x67, 0x66, 0x66, 0x66,
	0x66, 0x66, 0xe7, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xe7, 0x66, 0x66, 0x66,
	0x66, 0x66, 0x67, 0x60, 0x67, 0x66, 0x66, 0x66, 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
	0x66, 0x66, 0xe7, 0x00, 0xe7, 0x66, 0x66, 0x66, 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00,
	0x66, 0x66, 0x66, 0x66, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x7f, 0x00, 0x00, 0x00,
	0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18,
	0x00, 0x00, 0x00, 0x00, 0x7f, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0xff, 0x66, 0x66, 0x66,
	0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
	0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xfc, 0xc6, 0xc6, 0xfc, 0xc0, 0xc0, 0x00, 0x00, 0x3e, 0x60, 0x60, 0x60, 0x3e, 0x00,
	0x00, 0x00, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x3e, 0x06, 0x7c,
	0x00, 0x00, 0x7c, 0xd6, 0xd6, 0xd6, 0x7c, 0x10, 0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00,
	0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x7f, 0x00, 0x00, 0x00, 0x66, 0x66, 0x3e, 0x06, 0x06, 0x00,
	0x00, 0x00, 0xd6, 0xd6, 0xd6, 0xd6, 0xfe, 0x00, 0x00, 0x00, 0xd6, 0xd6, 0xd6, 0xd6, 0xff, 0x01,
	0x00, 0x00, 0x70, 0x30, 0x3c, 0x36, 0x3c, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xf6, 0xde, 0xf6, 0x00,
	0x00, 0x00, 0x60, 0x60, 0x7c, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x7c, 0x06, 0x7e, 0x06, 0x7c, 0x00,
	0x00, 0x00, 0xcc, 0xd6, 0xf6, 0xd6, 0xcc, 0x00, 0x00, 0x00, 0x3e, 0x66, 0x3e, 0x36, 0x66, 0x00,
	0x48, 0xfc, 0xc0, 0xf8, 0xc0, 0xc0, 0xfc, 0x00, 0x28, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0x7e, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x03, 0x06, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x60, 0x30, 0x18,
	0x18, 0x30, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x18, 0x0c, 0x06, 0x03, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x04, 0x02, 0xff, 0x02, 0x04, 0x00, 0x00, 0x00, 0x20, 0x40, 0xff, 0x40, 0x20, 0x00,
	0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00,
	0x00, 0x18, 0x00, 0xff, 0x00, 0x18, 0x00, 0x00, 0x10, 0x10, 0x7c, 0x10, 0x10, 0x00, 0x7c, 0x00,
	0x00, 0x8e, 0x88, 0xc9, 0xa9, 0x98, 0x88, 0x00, 0x00, 0x00, 0x42, 0x3c, 0x42, 0x3c, 0x42, 0x00,
	0x00, 0x00, 0x1c, 0x1c, 0x1c, 0x1c, 0x00, 0x00, 0x51, 0x8a, 0x51, 0x8a, 0x51, 0x8a, 0x51, 0x9a
};

const uint8 Display::_palJoeClothes[] = {
	0x00, 0x00, 0x00, 0x60, 0x60, 0x60, 0x87, 0x87, 0x87, 0xB0, 0xB0, 0xB0, 0xDA, 0xDA, 0xDA, 0x43,
	0x34, 0x20,	0x77, 0x33, 0x1F, 0xA3, 0x43, 0x27, 0x80, 0x45, 0x45, 0x9E, 0x5D, 0x5B, 0xB9, 0x78,
	0x75, 0xDF, 0x97, 0x91,	0x17, 0x27, 0x63, 0x1F, 0x3F, 0x83, 0x27, 0x5B, 0xA7, 0x98, 0xD4, 0xFF
};

const uint8 Display::_palJoeDress[] = {
	0x00, 0x00, 0x00, 0x50, 0x50, 0x50, 0x70, 0x70, 0x70, 0x90, 0x90, 0x90, 0xC6, 0xC6, 0xC6, 0xFF,
	0xFF, 0xFF,	0x30, 0x30, 0x90, 0x47, 0x49, 0xD0, 0x40, 0x24, 0x00, 0x79, 0x34, 0x0B, 0xB2, 0x3D,
	0x22, 0xED, 0x42, 0x42,	0x80, 0x45, 0x45, 0xA3, 0x5F, 0x5F, 0xC8, 0x7C, 0x7C, 0xEC, 0x9C, 0x9C
};

} // End of namespace Queen
