/* $Id: udp.c,v 1.4 2003/05/30 00:42:40 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ip.h"
#include "udp.h"
#include "misc.h"


/* 
 * Data structures
 */

typedef struct {
	uint8_t src_port[2] ;
	uint8_t dest_port[2];
	uint8_t length[2]   ;
	uint8_t checksum[2] ;
} udp_header_t;

typedef struct udp_handler_node_struct {
	int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port);
	void *data;
	uint32_t src_ip;
	uint32_t dest_ip;
	uint16_t src_port;
	uint16_t dest_port;
	struct udp_handler_node_struct *next;
} udp_handler_node_t;



/* 
 * Static data
 */

static udp_handler_node_t *handler_chain;



/* 
 * Static functions
 */


/*
 * Take the udp packet in mbuf and check it for validity.  The
 * ip's are used in the pseudoheader for checksum generation.
 *
 * A packet is considered valid iff: 
 *   - The udp length matches the amout of data in the buffer
 *   - The checksum checks out
 */

static int is_valid_udp_packet(mbuf_t *mbuf, uint32_t src_ip, uint32_t dest_ip)
{
	int retval = 0;
	uint32_t checksum, packet_checksum;
	int udp_length;
	int chunks;
	udp_header_t *udp_header;
	int i;

	udp_header = (udp_header_t *)mbuf->front_ptr;
	if (mbuf->size < sizeof(udp_header_t)) {
		goto out;
	}

	udp_length = get_uint16_field(udp_header->length);
	if (udp_length > mbuf->size) {
		goto out;
	}
	
	packet_checksum = get_uint16_field(udp_header->checksum);
	
	if (!packet_checksum) {
		/* No checksum included, so it's considered valid */
		retval = 1;
		goto out;
	}
	
	/* pseudo header stuff.  What a pain */
	checksum = (src_ip>>16) + (src_ip & 0xffff) + (dest_ip>>16) + (dest_ip & 0xffff);
	checksum += IP_PROT_UDP;
	checksum += udp_length;
	
	chunks = udp_length >> 1;
	for (i = 0; i < chunks; i++) {
		checksum += get_uint16_field(((uint8_t *)udp_header) + (i << 1));
	}
	if (udp_length & 1) {
		checksum += ((uint16_t)(((uint8_t *)udp_header)[udp_length-1]))<<8;
	}
	
	while (checksum >> 16) {
		checksum = (checksum >> 16) + (checksum & 0xffff);
	}
	
	if ((checksum == 0xffff)
	    || (!checksum && (packet_checksum == 0xffff))) {
		retval = 1;
	}
 out:
	return retval;
}


/*
 * Handler used to grab and categorize UDP packets from the IP layer. 
 * 
 * Whenever we have handlers for udp packets registered, this function
 * is registered with the IP layer to capture all UDP packets.  When
 * called, all header information from the IP header onward has been
 * stripped. 
 * 
 * All we do here is check to make sure the packet is valid, and, 
 * if it is, check for a handler in our chain that matches 
 * the ip/port tuple for this packet.  For any handler that 
 * matches the tuple, we call the handler function.  If the
 * handler function returns nonzero, we designate the packet
 * handled, and return such to the IP layer.  Otherwise, 
 * we keep searching the chain. 
 * 
 * If no registered handler handles the packet the packet 
 * is designated unhandled. 
 *
 * Returns 1 if the packet is handled, 0 otherwise.
 */

static int udp_handler(mbuf_t *buf, void *data, uint32_t src_ip, uint32_t dest_ip)
{
	int retval = 0;
	udp_header_t *header;
	
	if (is_valid_udp_packet(buf, src_ip, dest_ip)) {
		udp_handler_node_t *tmp;
		uint16_t src_port;
		uint16_t dest_port;
		uint16_t udp_len;
		
		header = (udp_header_t *)buf->front_ptr;
		
		src_port = get_uint16_field(header->src_port);
		dest_port = get_uint16_field(header->dest_port);
		udp_len = get_uint16_field(header->length);

		mbuf_trim_head(buf, sizeof(udp_header_t));

		for (tmp = handler_chain; tmp != 0; tmp = tmp->next) {
			if ((!tmp->src_ip || (tmp->src_ip == src_ip))
			    && (!tmp->dest_ip || (tmp->dest_ip == dest_ip))
			    && (!tmp->src_port || (tmp->src_port == src_port))
			    && (!tmp->dest_port || (tmp->dest_port == dest_port))) {
				if ((*(tmp->handler))(buf, tmp->data, src_ip, dest_ip, src_port, dest_port)) {
					retval = 1;
					break;
				}
			}
		}
	}
	return retval;
}


/* 
 * Exported functions.  See header file for semantics 
 */

void udp_send_packet(mbuf_t *mbuf, uint32_t src_ip, uint32_t dest_ip, uint8_t ip_flags, uint16_t src_port, uint16_t dest_port)
{
	uint32_t checksum;
	int chunks;
	int i;
	udp_header_t *header;

	if (mbuf_prepend_space(mbuf, sizeof(udp_header_t)) != MBUF_SUCCESS) {
		lib_die("Out of memory");
	}
	header = (udp_header_t *)mbuf->front_ptr;
	set_uint16_field(header->src_port, src_port);
	set_uint16_field(header->dest_port, dest_port);
	set_uint16_field(header->length, mbuf->size);
	set_uint16_field(header->checksum, 0);

	checksum = (dest_ip >> 16) + (dest_ip & 0xffff) + (src_ip >> 16) + (src_ip & 0xffff) 
		+ IP_PROT_UDP + mbuf->size;

	chunks = mbuf->size >> 1;
	for (i = 0; i < chunks; i++) {
		checksum += get_uint16_field(mbuf->front_ptr + (i<<1));
	}
	if (mbuf->size & 1) {
		/* Zero pad the last byte, if not aligned */
		checksum += ((uint16_t)(mbuf->front_ptr[mbuf->size-1])) << 8;
	}
	
	while(checksum >> 16) {
		checksum = (checksum >> 16) + (checksum & 0xffff);
	}
	
	if (checksum != 0xffff) {
		checksum = (~checksum) & 0xffff;
	}
	set_uint16_field(header->checksum, checksum);

	ip_send_packet(mbuf, IP_PROT_UDP, src_ip, dest_ip, ip_flags);
}

void udp_add_handler(int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port), 
		     void *data, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port)
{
	udp_handler_node_t *tmp = lib_malloc(sizeof(udp_handler_node_t));
	if (!tmp) {
		lib_die("Out of memory");
	}
	if (!handler_chain) {
		/* This is the first thing we're handling.  Set up an ip handler */
		ip_add_handler(udp_handler, 0, IP_PROT_UDP);
	}
	tmp->next = handler_chain;
	handler_chain = tmp;
	tmp->handler = handler;
	tmp->data = data;
	tmp->src_ip = src_ip;
	tmp->dest_ip = dest_ip;
	tmp->src_port= src_port;
	tmp->dest_port = dest_port;
}

void udp_del_handler(int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port), 
		     void *data, uint32_t src_ip, uint32_t dest_ip, uint16_t src_port, uint16_t dest_port)
{
	udp_handler_node_t *prev = 0;
	udp_handler_node_t *tmp = handler_chain;
	
	while (tmp) {
		if ((tmp->handler == handler) 
		    && (tmp->data == data)
		    && (tmp->src_ip == src_ip)
		    && (tmp->dest_ip == dest_ip)
		    && (tmp->src_port == src_port)
		    && (tmp->dest_port == dest_port)) {
			if (prev) {
				prev->next = tmp->next;
			} else {
				handler_chain = tmp->next;
			}
			lib_free(tmp);
			if (handler_chain == 0) {
				/* Not listening for anything.  Remove our
				   ip hook */
				ip_del_handler(udp_handler, 0, IP_PROT_UDP);
			}
			return;
		}
		prev = tmp;
		tmp = tmp->next;
	}
	lib_die("udp_del_handler didn't find a matching handler\n");	
}

