# encoding: utf-8
=begin

 * Name: SiSU

 * Description: a framework for document structuring, publishing and search

 * Author: Ralph Amissah

 * Copyright: (C) 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006,
   2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014 Ralph Amissah,
   All Rights Reserved.

 * License: GPL 3 or later:

   SiSU, a framework for document structuring, publishing and search

   Copyright (C) Ralph Amissah

   This program is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program. If not, see <http://www.gnu.org/licenses/>.

   If you have Internet connection, the latest version of the GPL should be
   available at these locations:
   <http://www.fsf.org/licensing/licenses/gpl.html>
   <http://www.gnu.org/licenses/gpl.html>

   <http://www.sisudoc.org/sisu/en/manifest/gpl.fsf.html>

 * SiSU uses:
   * Standard SiSU markup syntax,
   * Standard SiSU meta-markup syntax, and the
   * Standard SiSU object citation numbering and system

 * Hompages:
   <http://www.jus.uio.no/sisu>
   <http://www.sisudoc.org>

 * Git
   <http://git.sisudoc.org/gitweb/?p=code/sisu.git;a=summary>
   <http://git.sisudoc.org/gitweb/?p=code/sisu.git;a=blob;f=lib/sisu/v6/se_file_op.rb;hb=HEAD>

 * Ralph Amissah
   <ralph@amissah.com>
   <ralph.amissah@gmail.com>

 ** Description: system environment, resource control and configuration details

=end
module SiSU_Info_File
  require_relative 'utils'                                 # utils.rb
  require_relative 'se_info_env'                           # se_info_env.rb
  begin
    require 'fileutils'
      include FileUtils::Verbose
  rescue LoadError
    SiSU_Utils::CodeMarker.new(__LINE__,__FILE__,:fuchsia).
      error('fileutils NOT FOUND (LoadError)')
  end
  class InfoFile < SiSU_Info_Env::InfoEnv                  # se_info_env.rb
    #todo unify with FileOp
    def initialize(fns)
      begin
        super(fns)
        @fns=fns
        @env=SiSU_Env::InfoEnv.new(@fns)
        m=/((.+?)(?:\~\w\w(?:_\w\w)?)?)\.((?:-|ssm\.)?sst|ssm)$/
        @fnn,@fnb,@fnt=@fns[m,1],@fns[m,2],@fns[m,3]
      rescue
        SiSU_Screen::Ansi.new('',$!,$@).rescue do
          __LINE__.to_s + ':' + __FILE__
        end
      ensure
      end
    end
    def basefilename #Remove if possible
      m=/(.+?)\.(?:(?:-|ssm\.)?sst|ssm)$/m
      @fns[m,1]
    end
    def make_file(path,filename)
      (File.writable?("#{path}/.")) \
      ? File.new("#{path}/#{filename}",'w+')
      : (SiSU_Screen::Ansi.new(
          '',
          "*WARN* is the file or directory writable?, could not create #{filename}"
        ).warn)
    end
    def touch_file(path,filename)
      if File.writable?("#{path}/.");
        FileUtils::touch("#{path}/#{filename}")
      else
        SiSU_Screen::Ansi.new(
          '',
          "*WARN* is the file or directory writable?, could not create #{filename}"
        ).warn
      end
    end
    def make_path(path)
      FileUtils::mkdir_p(path) unless FileTest.directory?(path)
    end
    def marshal
      def ao_content
        @env.processing_path.ao + '/' \
          + @fns + '.content.rbm'
      end
      def ao_idx_sst_rel_html_seg
        @env.processing_path.ao + '/' \
          + @fns + '.idx_sst.rbm'
      end
      def ao_idx_sst_rel #used by tex & odf
        @env.processing_path.ao + '/' \
          + @fns + '.idx_raw.rbm'
      end
      def ao_idx_html
        @env.processing_path.ao + '/' \
          + @fns + '.idx_html.rbm'
      end
      def ao_idx_xhtml
        @env.processing_path.ao + '/' \
          + @fns + '.idx_xhtml.rbm'
      end
      def ao_metadata
        @env.processing_path.ao + '/' \
          + @fns + '.metadata.rbm'
      end
      def ao_map_nametags
        @env.processing_path.ao + '/' \
          + @fns + '.map_name_tags.rbm'
      end
      def ao_map_ocn_htmlseg
        @env.processing_path.ao + '/' \
          + @fns + '.map_ocn_htmlseg.rbm'
      end
      def html_tune
        @env.processing_path.tune + '/' \
          + @fns + '.marshal_tune'
      end
      def xhtml_tune
        @env.processing_path.tune + '/' \
          + @fns + '.marshal_tune'
      end
      self
    end
    def write_file_processing
      def html_tune
        File.new("#{@env.processing_path.tune}/#{@fns}.tune",'w+')
      end
      self
    end
    def mkdir #check moved from FileOp, existing mkdir
      def processing
        def ao
          FileUtils::mkdir_p(@env.processing_path.ao) \
            unless FileTest.directory?(@env.processing_path.ao)
        end
        def tune
          FileUtils::mkdir_p(@env.processing_path.tune) \
            unless FileTest.directory?(@env.processing_path.tune)
        end
        self
      end
    end
  end
end
module SiSU_File_Op
  require_relative 'constants'                             # constants.rb
  require_relative 'utils'                                 # utils.rb
  begin
    require 'fileutils'
      include FileUtils::Verbose
  rescue LoadError
    SiSU_Utils::CodeMarker.new(__LINE__,__FILE__,:fuchsia).
      error('fileutils NOT FOUND (LoadError)')
  end
  class FileOp < SiSU_Info_File::InfoFile
    #todo unify with CreateFile
    def initialize(md,fno='')
      begin
        @md,@fno=md,fno
        @env=SiSU_Env::InfoEnv.new(@md.fns)
      rescue
        SiSU_Screen::Ansi.new(md.opt.selections.str,$!,$@).rescue do
          __LINE__.to_s + ':' + __FILE__
        end
      ensure
      end
    end
    def output_dir_structure
      SiSU_Env::ProcessingSettings.new(@md).output_dir_structure
    end
    def mkdir_initialize                # not used but consider using
      FileUtils::mkdir_p(output_path.base.dir) \
        unless FileTest.directory?(output_path.base.dir)
      FileUtils::mkdir_p("#{output_path.base.dir}/#{@md.fnb}") \
        unless FileTest.directory?("#{output_path.base.dir}/#{@md.fnb}")
      FileUtils::mkdir_p("#{output_path.base.dir}/#{@env.path.style}") \
        unless FileTest.directory?("#{output_path.base.dir}/#{@env.path.style}")
      FileUtils::mkdir_p(@env.processing_path.ao) \
        unless FileTest.directory?(@env.processing_path.ao)
      FileUtils::mkdir_p(@env.processing_path.tune) \
        unless FileTest.directory?(@env.processing_path.tune)
    end
    def path_rel_links
      def html_scroll_2
        if output_dir_structure.by_language_code?
          '../../'
        elsif output_dir_structure.by_filetype?
          '../'
        else
          '../'
        end
      end
      def html_seg_2
        if output_dir_structure.by_language_code?
          '../../../'
        elsif output_dir_structure.by_filetype?
          '../../'
        else
          '../'
        end
      end
      def html_scroll_1
        if output_dir_structure.by_language_code?
          '../'
        elsif output_dir_structure.by_filetype?
          '../'
        else
          './'
        end
      end
      def html_seg_1
        if output_dir_structure.by_language_code?
          '../../'
        elsif output_dir_structure.by_filetype?
          '../../'
        else
          './'
        end
      end
      def default_output_css
        if (@md.opt.opt_act[:dump][:bool] \
        &&  @md.opt.opt_act[:dump][:inst]) \
        || (@md.opt.opt_act[:redirect][:bool] \
        &&  @md.opt.opt_act[:redirect][:inst])
          './'
        elsif output_dir_structure.by_language_code?
          '../../'
        elsif output_dir_structure.by_filetype?
          '../'
        else
          '../'
        end
      end
      def html_scroll_css
        default_output_css
      end
      def xhtml_css
        default_output_css
      end
      def xml_css
        default_output_css
      end
      def html_seg_css
        if output_dir_structure.by_language_code?
          '../../../'
        elsif output_dir_structure.by_filetype?
          '../../'
        else
          '../'
        end
      end
      def manifest_css
        if output_dir_structure.by_language_code?
          '../../_sisu/css'
        elsif output_dir_structure.by_filetype?
          ''
        else
          '../'
        end
      end
      self
    end
    def mkdir
      def output
        def base
          FileUtils::mkdir_p(output_path.base.dir) \
            unless FileTest.directory?(output_path.base.dir)
        end
        def css
          FileUtils::mkdir_p("#{output_path.base.dir}/#{@env.path.style}") \
            unless FileTest.directory?("#{output_path.base.dir}/#{@env.path.style}")
        end
        def epub
          path=output_path.epub.dir
          make_path(path)
        end
        self
      end
      self
    end
    def mkfile #consider using more
      path="#{output_path.base.dir}/#{@md.fnb}"
      make_path(path)
      filename=@fno
      make_file(path,filename)
    end
    def mkfile_pwd
      path=Dir.pwd
      filename=@fno
      make_file(path,filename)
    end
    def write_file
      def txt
        path=output_path.txt.dir
        make_path(path)
        fn=base_filename.txt
        make_file(path,fn)
      end
      def textile
        path=output_path.textile.dir
        make_path(path)
        fn=base_filename.textile
        make_file(path,fn)
      end
      def asciidoc
        path=output_path.asciidoc.dir
        make_path(path)
        fn=base_filename.asciidoc
        make_file(path,fn)
      end
      def markdown
        path=output_path.markdown.dir
        make_path(path)
        fn=base_filename.markdown
        make_file(path,fn)
      end
      def rst
        path=output_path.rst.dir
        make_path(path)
        fn=base_filename.rst
        make_file(path,fn)
      end
      def html_scroll
        pth=output_path.html.dir
        make_path(pth)
        p_fn=place_file.html_scroll.dir
        @@filename_html_scroll=File.new(p_fn,'w+')
      end
      def html_seg_index
        pth=((output_dir_structure.by_filename?) \
        || (output_dir_structure.dump?)) \
        ? "#{output_path.html.dir}"
        : "#{output_path.html.dir}/#{@md.fnb}"
        make_path(pth)
        p_fn=place_file.html_seg_index.dir
        @@filename_html_index=File.new(p_fn,'w+')
      end
      def html_segtoc
        pth=((output_dir_structure.by_filename?) \
        || (output_dir_structure.dump?) \
        || (output_dir_structure.redirect?)) \
        ? "#{output_path.html.dir}"
        : "#{output_path.html.dir}/#{@md.fnb}"
        make_path(pth)
        p_fn=place_file.html_segtoc.dir
        @@filename_html_index=File.new(p_fn,'w+')
      end
      def xhtml
        path=output_path.xhtml.dir
        make_path(path)
        fn=base_filename.xhtml
        make_file(path,fn)
      end
      def xml_sax
        path=output_path.xml.dir
        make_path(path)
        fn=base_filename.xml_sax
        make_file(path,fn)
      end
      def xml_dom
        path=output_path.xml.dir
        make_path(path)
        fn=base_filename.xml_dom
        make_file(path,fn)
      end
      def xml_docbook_book
        path=output_path.xml_docbook_book.dir
        make_path(path)
        fn=base_filename.xml_docbook_book
        make_file(path,fn)
      end
      def xml_fictionbook
        path=output_path.xml_fictionbook.dir
        make_path(path)
        fn=base_filename.xml_fictionbook
        make_file(path,fn)
      end
      def xml_scaffold_structure_sisu
        path=output_path.xml_scaffold_structure_sisu.dir
        make_path(path)
        fn=base_filename.xml_scaffold_structure_sisu
        make_file(path,fn)
      end
      def xml_scaffold_structure_collapse
        path=output_path.xml_scaffold_structure_collapse.dir
        make_path(path)
        fn=base_filename.xml_scaffold_structure_collapse
        make_file(path,fn)
      end
      def manpage
        path=output_path.manpage.dir
        make_path(path)
        fn=base_filename.manpage
        make_file(path,fn)
      end
      def texinfo
        path=output_path.texinfo.dir
        make_path(path)
        fn=base_filename.texinfo
        make_file(path,fn)
      end
      def info
        path=output_path.texinfo.dir
        make_path(path)
        fn=base_filename.info
        make_file(path,fn)
      end
      def hash_digest
        path=output_path.hash_digest.dir
        make_path(path)
        fn=base_filename.hash_digest
        make_file(path,fn)
      end
      def qrcode
        path=output_path.qrcode.dir
        make_path(path)
        fn=base_filename.qrcode
        make_file(path,fn)
      end
      def manifest
        path=output_path.manifest.dir
        make_path(path)
        fn=base_filename.manifest
        make_file(path,fn)
      end
      def manifest_txt
        path=output_path.manifest.dir
        make_path(path)
        fn=base_filename.manifest_txt
        make_file(path,fn)
      end
      def pot
        path=output_path.pot.dir
        make_path(path)
        fn=base_filename.pot
        make_file(path,fn)
      end
      def po
        path=output_path.po.dir
        make_path(path)
        fn=base_filename.po
        make_file(path,fn)
      end
      self
    end
    def place_file
      def txt
        def dir
          output_path.txt.dir + '/' \
          + base_filename.txt
        end
        def rel
          output_path.txt.rel + '/' \
          + base_filename.txt
        end
        self
      end
      def textile
        def dir
          output_path.textile.dir + '/' \
          + base_filename.textile
        end
        def rel
          output_path.textile.rel + '/' \
          + base_filename.textile
        end
        self
      end
      def asciidoc
        def dir
          output_path.asciidoc.dir + '/' \
          + base_filename.asciidoc
        end
        def rel
          output_path.asciidoc.rel + '/' \
          + base_filename.asciidoc
        end
        self
      end
      def markdown
        def dir
          output_path.markdown.dir + '/' \
          + base_filename.markdown
        end
        def rel
          output_path.markdown.rel + '/' \
          + base_filename.markdown
        end
        self
      end
      def rst
        def dir
          output_path.rst.dir + '/' \
          + base_filename.rst
        end
        def rel
          output_path.rst.rel + '/' \
          + base_filename.rst
        end
        self
      end
      def html_scroll
        def dir
          output_path.html_scroll.dir + '/' \
          + base_filename.html_scroll
        end
        def rel
          output_path.html_scroll.rel + '/' \
          + base_filename.html_scroll
        end
        self
      end
      def html_seg_index
        def dir
          output_path.html_seg.dir + '/' \
          + base_filename.html_seg_index
        end
        def rel
          output_path.html_seg.rel + '/' \
          + base_filename.html_seg_index
        end
        self
      end
      def html_segtoc
        def dir
          output_path.html_seg.dir + '/' \
          + base_filename.html_segtoc
        end
        def rel
          output_path.html_seg.rel + '/' \
          + base_filename.html_segtoc
        end
        self
      end
      def html_book_index
        def dir
          output_path.html_seg.dir + '/' \
          + base_filename.html_book_index
        end
        def rel
          output_path.html_seg.rel + '/' \
          + base_filename.html_book_index
        end
        self
      end
      def html_concordance
        def dir
          output_path.html_seg.dir + '/' \
          + base_filename.html_concordance
        end
        def rel
          output_path.html_seg.rel + '/' \
          + base_filename.html_concordance
        end
        self
      end
      def odt
        def dir
          output_path.odt.dir + '/' \
          + base_filename.odt
        end
        def rel
          output_path.odt.rel + '/' \
          + base_filename.odt
        end
        self
      end
      def epub
        def dir
          output_path.epub.dir + '/' \
          + base_filename.epub
        end
        def rel
          output_path.epub.rel + '/' \
          + base_filename.epub
        end
        self
      end
      def pdf_p
        STDERR.puts 'ERROR not available due to multiple page format sizes'
      end
      def pdf_l
        STDERR.puts 'ERROR not available due to multiple page format sizes'
      end
      def xhtml
        def dir
          output_path.xhtml.dir + '/' \
          + base_filename.xhtml
        end
        def rel
          output_path.xhtml.rel + '/' \
          + base_filename.xhtml
        end
        self
      end
      def xml_sax
        def dir
          output_path.xml.dir + '/' \
          + base_filename.xml_sax
        end
        def rel
          output_path.xml.rel + '/' \
          + base_filename.xml_sax
        end
        self
      end
      def xml_dom
        def dir
          output_path.xml.dir + '/' \
          + base_filename.xml_dom
        end
        def rel
          output_path.xml.rel + '/' \
          + base_filename.xml_dom
        end
        self
      end
      def xml_docbook_book
        def dir
          output_path.xml_docbook.dir + '/' \
          + base_filename.xml_docbook_book
        end
        def rel
          output_path.xml_docbook.rel + '/' \
          + base_filename.xml_docbook_book
        end
        self
      end
      def xml_fictionbook
        def dir
          output_path.xml_fictionbook.dir + '/' \
          + base_filename.xml_fictionbook
        end
        def rel
          output_path.xml_fictionbook.rel + '/' \
          + base_filename.xml_fictionbook
        end
        self
      end
      def xml_scaffold_structure_sisu
        def dir
          output_path.xml.dir + '/' \
          + base_filename.xml_scaffold_structure_sisu
        end
        def rel
          output_path.xml.rel + '/' \
          + base_filename.xml_scaffold_structure_sisu
        end
        self
      end
      def xml_scaffold_structure_collapse
        def dir
          output_path.xml.dir + '/' \
          + base_filename.xml_scaffold_structure_collapse
        end
        def rel
          output_path.xml.rel + '/' \
          + base_filename.xml_scaffold_structure_collapse
        end
        self
      end
      def sqlite_discrete
        def dir
          output_path.sqlite_discrete.dir + '/' \
          + base_filename.sqlite_discrete
        end
        def rel
          output_path.sqlite_discrete.rel + '/' \
          + base_filename.sqlite_discrete
        end
        self
      end
      def hash_digest
        def dir
          output_path.hash_digest.dir + '/' \
          + base_filename.hash_digest
        end
        def rel
          output_path.hash_digest.rel + '/' \
          + base_filename.hash_digest
        end
        self
      end
      def src
        def dir
          output_path.src.dir + '/' \
            + base_filename.src
        end
        def rel
          output_path.src.rel + '/' \
          + base_filename.src
        end
        self
      end
      def sisupod
        def dir
          output_path.sisupod.dir + '/' \
          + base_filename.sisupod
        end
        def rel
          output_path.sisupod.rel + '/' \
          + base_filename.sisupod
        end
        self
      end
      def po
        def dir
          output_path.po.dir + '/' \
          + base_filename.po
        end
        def rel
          output_path.po.rel + '/' \
          + base_filename.po
        end
        self
      end
      def pot
        def dir
          output_path.pot.dir + '/' \
          + base_filename.pot
        end
        def rel
          output_path.pot.rel + '/' \
          + base_filename.pot
        end
        self
      end
      def po_git
        def dir
          output_path.po_git + '/' \
          + base_filename.po
        end
        def rel
          #output_path.po_git + '/' + base_filename.po
        end
        self
      end
      def pot_git
        def dir
          output_path.pot_git + '/' \
          + base_filename.pot
        end
        def rel
          #output_path.pot_git + '/' + base_filename.pot
        end
        self
      end
      def manpage
        def dir
          output_path.manpage.dir + '/' \
          + base_filename.manpage
        end
        def rel
          output_path.manpage.rel + '/' \
          + base_filename.manpage
        end
        self
      end
      def texinfo
        def dir
          output_path.texinfo.dir + '/' \
          + base_filename.texinfo
        end
        def rel
          output_path.texinfo.rel + '/' \
          + base_filename.texinfo
        end
        self
      end
      def info
        def dir
          output_path.texinfo.dir + '/' \
          + base_filename.info
        end
        def rel
          output_path.texinfo.rel + '/' \
          + base_filename.info
        end
        self
      end
      def qrcode_title
        def dir
          output_path.qrcode.dir + '/' \
          + base_filename.qrcode_title
        end
        def rel
          output_path.qrcode.rel + '/' \
          + base_filename.qrcode_title
        end
        self
      end
      def qrcode_md
        def dir
          output_path.qrcode.dir + '/' \
          + base_filename.qrcode_md
        end
        def rel
          output_path.qrcode.rel + '/' \
          + base_filename.qrcode_md
        end
        self
      end
      def manifest
        def dir
          output_path.manifest.dir + '/' \
            + base_filename.manifest
        end
        def rel
          output_path.manifest.rel + '/' \
            + base_filename.manifest
        end
        self
      end
      self
    end
    def base_filename
      def i18n(f)
        f=default_hash.merge(f)
        f[:lng] ||=@md.lang_code_insert
        f[:fn] + f[:lng] + f[:ft]
      end
      def default_hash
        {
          fn: @md.fnb,
          lng: @md.lang_code_insert,
        }
      end
      def default_hash_build(fh,sfx)
        if fh.is_a?(Hash)
          fh[:fn] ||=@md.fnb
          fh[:lng] ||= @md.lang_code_insert
          fh[:ft]=sfx
          fh
        else
          {
            fn: @md.fnb,
            lng: @md.lang_code_insert,
            ft: sfx,
          }
        end
      end
      def lang_code?(lng)
        (output_dir_structure.by_language_code?) \
        ? ''
        : (lng ||=@md.lang_code_insert)
      end
      def txt(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'plain',
            ft: fh[:ft],
            lng: fh[:lng],
           }
         else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def textile(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt_textile])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'plain',
            ft: fh[:ft],
            lng: fh[:lng],
           }
         else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def asciidoc(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt_asciidoc])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'plain',
            ft: fh[:ft],
            lng: fh[:lng],
           }
         else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def markdown(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt_markdown])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'plain',
            ft: fh[:ft],
            lng: fh[:lng],
           }
         else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def rst(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt_rst])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'plain',
            ft: fh[:ft],
            lng: fh[:lng],
           }
         else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def html_scroll(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def html_seg_index(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: 'index',
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def html_segtoc(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fnh=if output_dir_structure.dump_or_redirect?
          {
            fn: fh[:fn] + '.toc',
            ft: fh[:ft],
          }
        else
          {
            fn: 'toc',
            ft: fh[:ft],
            lng: lang_code?(fh[:lng]),
          }
        end
        i18n(fnh)
      end
      def html_seg(fh)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def html_book_index(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: 'book_index',
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def html_concordance(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.dump_or_redirect?
          {
            fn: 'concordance',
            ft: fh[:ft],
          }
        else
          {
            fn: 'concordance',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xhtml(fh=nil)
        fh=default_hash_build(fh,Sfx[:xhtml])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def epub(fh=nil)
        fh=default_hash_build(fh,Sfx[:epub])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: fh[:fn],
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def odt(fh=nil)
        fh=default_hash_build(fh,Sfx[:odt])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'opendocument',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_sax(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_sax])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_dom(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_dom])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_docbook_book(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_docbook_book])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_fictionbook(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_fictionbook])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_scaffold_structure_sisu(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_scaffold_structure_sisu])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def xml_scaffold_structure_collapse(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml_scaffold_structure_collapse])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'scroll',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def pdf_p(fh=nil)
        fh=default_hash_build(fh,Sfx[:pdf])
        fh[:lng]=lang_code?(fh[:lng])
        if output_dir_structure.by_filename?
          'portrait' + fh[:lng] + '.'
        else
          fh[:fn] + '.portrait' + fh[:lng] + '.'
        end
      end
      def pdf_l(fh=nil)
        fh=default_hash_build(fh,Sfx[:pdf])
        fh[:lng]=lang_code?(fh[:lng])
        if output_dir_structure.by_filename?
          'landscape' + fh[:lng] + '.'
        else
          fh[:fn] + '.landscape' + fh[:lng] + '.'
        end
      end
      def pdf_p_a4(fh=nil)
        pdf_p(fh) + @md.fn[:pdf_p_a4]
      end
      def pdf_p_a5(fh=nil)
        pdf_p(fh) + @md.fn[:pdf_p_a5]
      end
      def pdf_p_b5(fh=nil)
        pdf_p(fh) + @md.fn[:pdf_p_b5]
      end
      def pdf_p_letter(fh=nil)
        pdf_p(fh) + @md.fn[:pdf_p_letter]
      end
      def pdf_p_legal(fh=nil)
        pdf_p(fh) + @md.fn[:pdf_p_legal]
      end
      def pdf_l_a4(fh=nil)
        pdf_l(fh) + @md.fn[:pdf_l_a4]
      end
      def pdf_l_a5(fh=nil)
        pdf_l(fh) + @md.fn[:pdf_l_a5]
      end
      def pdf_l_b5(fh=nil)
        pdf_l(fh) + @md.fn[:pdf_l_b5]
      end
      def pdf_l_letter(fh=nil)
        pdf_l(fh) + @md.fn[:pdf_l_letter]
      end
      def pdf_l_legal(fh=nil)
        pdf_l(fh) + @md.fn[:pdf_l_legal]
      end
      def manpage(fh=nil)
        fh=default_hash_build(fh,Sfx[:manpage])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: fh[:fn],
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def info(fh=nil)
        fh=default_hash_build(fh,Sfx[:info])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: fh[:fn],
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def texinfo(fh=nil)
        fh=default_hash_build(fh,Sfx[:texinfo])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: fh[:fn],
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def sqlite_discrete(fh=nil)
        fh=default_hash_build(fh,Sfx[:sql])
        fh[:lng]=lang_code?(fh[:lng])
        fnh={
          fn: fh[:fn],
          ft: fh[:ft],
          lng: fh[:lng],
        }
        i18n(fnh)
      end
      def hash_digest(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_language_code?
          {
            fn: fh[:fn] + '.hash_digest',
            ft: fh[:ft],
          }
        elsif output_dir_structure.by_filetype?
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: 'digest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def sitemap(fh=nil)
        fh=default_hash_build(fh,Sfx[:xml])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_language_code?
          {
            fn: fh[:fn] + '.sitemap',
            ft: fh[:ft],
          }
        elsif output_dir_structure.by_filetype?
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: 'sitemap',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def qrcode_title(fh=nil)
        fh=default_hash_build(fh,'.title.png')
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'sisu_manifest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def qrcode_md #check name below
        fh=default_hash_build(fh,'.md.png')
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'sisu_manifest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def manifest_txt(fh=nil)
        fh=default_hash_build(fh,Sfx[:txt])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.by_filename?
          {
            fn: 'sisu_manifest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def manifest(fh=nil)
        fh=default_hash_build(fh,Sfx[:html])
        fh[:lng]=lang_code?(fh[:lng])
        fnh=if output_dir_structure.dump_or_redirect?
          {
            fn: fh[:fn] + '.manifest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        elsif output_dir_structure.by_filename?
          {
            fn: 'sisu_manifest',
            ft: fh[:ft],
            lng: fh[:lng],
          }
        else
          {
            fn: fh[:fn],
            ft: fh[:ft],
            lng: fh[:lng],
          }
        end
        i18n(fnh)
      end
      def src
        @md.fno
      end
      def po #check
        (@fno.empty?) \
        ? (@md.fn[:po])
        : (@fno + '.po')
      end
      def pot
        (@fno.empty?) \
        ? (@md.fn[:pot])
        : (@fno + '.pot')
      end
      def sisupod
        (@md.fns =~/\.ssm\.sst$/) \
        ? @md.fns.gsub(/(?:\~\S{2,3})?\.ssm\.sst$/,'.ssm.txz')
        : @md.fns.gsub(/(?:\~\S{2,3})?(\.sst)$/,'\1.txz')
      end
      self
    end
    def set_path(ft)
      @ft=ft
      def dir
        def abc
          if output_dir_structure.redirect?
            @md.opt.opt_act[:redirect][:inst] + '/' + @md.fnb
          elsif output_dir_structure.dump?
            @md.opt.opt_act[:dump][:inst]
          elsif output_dir_structure.by_language_code?
            output_path.base.dir + '/' + @md.opt.lng + '/' + @ft
          elsif output_dir_structure.by_filetype?
            output_path.base.dir + '/' + @ft
          else
            output_path.base.dir + '/' + @md.fnb
          end
        end
        def ab
          if output_dir_structure.redirect?
            @md.opt.opt_act[:redirect][:inst] + '/' + @md.fnb
          elsif output_dir_structure.dump?
            @md.opt.opt_act[:dump][:inst]
          elsif output_dir_structure.by_language_code?
            output_path.base.dir + '/' + @md.opt.lng + '/' + @ft
          else
            output_path.base.dir + '/' + @ft
          end
        end
        def ab_src
          if output_dir_structure.redirect?
            @md.opt.opt_act[:redirect][:inst] + '/' + @md.fnb
          elsif output_dir_structure.dump?
            @md.opt.opt_act[:dump][:inst]
          else
            output_path.base.dir + '/' \
              + @ft + '/' \
              + @md.opt.fng + '/' \
              + Gt[:sisupod] + '/' \
              + Gt[:doc] + '/' \
              + @md.opt.lng
          end
        end
        def ab_pod
          if output_dir_structure.redirect?
            @md.opt.opt_act[:redirect][:inst] + '/' + @md.fnb
          elsif output_dir_structure.dump?
            @md.opt.opt_act[:dump][:inst]
          else
            output_path.base.dir + '/' + @ft
          end
        end
        self
      end
      def url
        def abc
          if output_dir_structure.by_language_code?
            output_path.base.url + '/' + @md.opt.lng + '/' + @ft
          elsif output_dir_structure.by_filetype?
            output_path.base.url + '/' + @ft
          else
            output_path.base.url + '/' + @md.fnb
          end
        end
        def ab
          if output_dir_structure.by_language_code?
            output_path.base.url + '/' + @md.opt.lng + '/' + @ft
          else
            output_path.base.url + '/' + @ft
          end
        end
        def ab_src
          output_path.base.url + '/' \
            + @ft + '/' \
            + @md.opt.fng + '/' \
            + Gt[:sisupod] + '/' \
            + Gt[:doc] + '/' \
            + @md.opt.lng
        end
        def ab_pod
          output_path.base.url + '/' + @ft
        end
        self
      end
      def rel
        def abc
          if output_dir_structure.by_language_code?
            @md.opt.lng + '/' + @ft
          elsif output_dir_structure.by_filetype?
            @ft
          else
            @md.fnb
          end
        end
        def ab
          if output_dir_structure.by_language_code?
            @md.opt.lng + '/' + @ft
          else
            @ft
          end
        end
        def ab_src
          @ft
        end
        def ab_pod
          @ft
        end
        self
      end
      def rel_sm
        def abc
          if output_dir_structure.by_language_code?
            @md.opt.lng + '/' + @ft
          elsif output_dir_structure.by_filetype?
            @ft
          else
            @md.fnb
          end
        end
        def ab
          if output_dir_structure.dump_or_redirect?
            '.'
          elsif output_dir_structure.by_language_code? \
          or output_dir_structure.by_filetype?
            '../' + @ft
          else '.'
          end
        end
        def ab_src
          locate="#{@ft}/#{@md.opt.fng}/#{Gt[:sisupod]}/#{Gt[:doc]}/#{@md.opt.lng}"
          if output_dir_structure.dump_or_redirect?
            '.'
          elsif output_dir_structure.by_language_code?
            '../../' + locate
          else
            '../' + locate
          end
        end
        def ab_pod
          if output_dir_structure.dump_or_redirect?
            '.'
          elsif output_dir_structure.by_language_code?
            '../../' + @ft
          else
            '../' + @ft
          end
        end
        self
      end
      def rcp
        def abc
          if output_dir_structure.by_language_code?
            output_path.stub.rcp + '/' + @md.opt.lng + '/' + @ft
          elsif output_dir_structure.by_filetype?
            output_path.stub.rcp + '/' + @ft
          else
            output_path.stub.rcp + '/' + @md.fnb
          end
        end
        def ab
          if output_dir_structure.by_language_code?
            output_path.stub.rcp + '/' + @md.opt.lng + '/' + @ft
          else
            output_path.stub.rcp + '/' + @ft
          end
        end
        self
      end
      self
    end
    def output_path
      def web_base
        def dir
          @env.path.webserv
        end
        def url
          #"#{@env.url.root}"
        end
        def rel
          '.'
        end
        def rcp
          '.'
        end
        self
      end
      def stub
        def dir
          @md.opt.f_pth[:pth_stub]
        end
        #def url
        #  "#{@env.url.root}"
        #end
        def rel
          './' + @md.opt.f_pth[:pth_stub]
        end
        def rcp
          @md.opt.f_pth[:pth_stub]
        end
        self
      end
      def webserver_path
        if output_dir_structure.dump?
          @md.opt.opt_act[:dump][:inst]
        elsif output_dir_structure.redirect?
          @md.opt.opt_act[:redirect][:inst]
        else
          @env.path.webserv
        end
      end
      def base
        def dir
          webserver_path + '/' + @md.opt.f_pth[:pth_stub]
        end
        def url
          @env.url.webserv + '/' + @md.opt.f_pth[:pth_stub]
        end
        def rel
          './' + @md.opt.f_pth[:pth_stub]
        end
        def rcp
          './' + @md.opt.f_pth[:pth_stub]
        end
        self
      end
      def sisugit
        def dir
          output_path.base.dir + '/git'
        end
        def url
          output_path.base.url + '/git'
        end
        def rel
          output_path.base.rel + '/git'
        end
        def rcp
          output_path.base.rcp + '/git'
        end
        self
      end
      #def pod
      #  ft='pod'
      #  path=set_path(ft).dir.ab
      #end
      def src
        def ft
          Gt[:src]
        end
        def dir
          set_path(ft).dir.ab_src
        end
        def url
          set_path(ft).url.ab_src
        end
        def rel
          set_path(ft).rel.ab_src
        end
        def rcp
          set_path(ft).rcp.ab_src
        end
        def rel_sm
          set_path(ft).rel_sm.ab_src
        end
        self
      end
      def sisupod
        def ft
          Gt[:src]
        end
        def dir
          set_path(ft).dir.ab_pod
        end
        def url
          set_path(ft).url.ab_pod
        end
        def rel
          set_path(ft).rel.ab_pod
        end
        def rcp
          set_path(ft).rcp.ab_pod
        end
        def rel_sm
          set_path(ft).rel_sm.ab_pod
        end
        self
      end
      def po
        def dir
          output_path.base.dir + '/' \
            + Gt[:src] + '/' \
            + @md.opt.fng + '/po4a/po/' \
            + @md.opt.lng
        end
        def url
          output_path.base.url + '/po4a/' \
            + @md.fnb + '/po/' \
            + @md.opt.lng
        end
        self
      end
      def pot
        def dir
         output_path.base.dir + '/' \
           + Gt[:src] + '/' \
           + @md.opt.fng + '/po4a/pot'
        end
        def url
          output_path.base.url + '/po4a/' \
            + @md.fnb + '/pot'
        end
        def rcp
          #p "#{output_path.base.dir}/po4a/#{@md.fnb}/pot"
        end
        self
      end
      def po_git # consider !!!
        def ft
          Gt[:po]
        end
        def dir
          pth=@env.processing_path.git + '/' \
            + @md.fnb + '/' \
            + ft + '/' \
            + @md.opt.lng
          FileUtils::mkdir_p(pth) unless FileTest.directory?(pth)
          pth
        end
        self
      end
      def pot_git # consider !!!
        def ft
          Gt[:pot]
        end
        def dir
          @env.processing_path.git + '/' \
            + @md.fnb + '/' \
            + ft
        end
        self
      end
      def md_harvest
        manifest
        self
      end
      def txt
        def ft
          'txt'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def textile
        def ft
          'textile' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def asciidoc
        def ft
          'asciidoc' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def markdown
        def ft
          'markdown' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def rst
        def ft
          'rst' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def html_scroll
        def ft
          'html'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          if output_dir_structure.dump_or_redirect?
            './image'
          elsif output_dir_structure.by_language_code?
            '../../_sisu/image'
          elsif output_dir_structure.by_filetype?
            '../_sisu/image'
          else
            '../_sisu/image'
          end
        end
        self
      end
      def html_seg
        def ft
          'html/' + @md.fnb
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          if output_dir_structure.dump_or_redirect?
            './image'
          elsif output_dir_structure.by_language_code?
            '../../../_sisu/image'
          elsif output_dir_structure.by_filetype?
            '../../_sisu/image'
          else
            '../_sisu/image'
          end
        end
        self
      end
      def html_concordance
        html_seg
        self
      end
      def html
        def ft
          'html'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).url.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          if output_dir_structure.by_language_code?
            '../../_sisu/image'
          elsif output_dir_structure.by_filetype?
            '../_sisu/image'
          else
            '../_sisu/image'
          end
        end
        self
      end
      def xhtml
        def ft
          'xhtml'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def epub
        def ft
          'epub'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          './image'
        end
        self
      end
      def odt
        def ft
          'odt'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def xml
        def ft
          'xml'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_sax
        xml
        self
      end
      def xml_dom
        xml
        self
      end
      def xml_docbook
        def ft
          'docbook'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_docbook_article
        def ft
          'docbook' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_docbook_book
        def ft
          'docbook'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_fictionbook
        def ft
          'fictionbook' \
            + DEVELOPER[:under_construction]
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_scaffold_structure_sisu
        def ft
          'sisu.scaffold.xml'
          #'xml'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def xml_scaffold_structure_collapse
        def ft
          'collapsed.scaffold.xml'
          #'xml'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        def rel_image
          '../../_sisu/image'
        end
        self
      end
      def pdf
        def ft
          'pdf'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def sqlite_discrete
        def ft
          'sql'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def hash_digest
        def ft
          'hashes'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def manifest
        def ft
          'manifest'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rel_image
          if output_dir_structure.dump_or_redirect?
            './image'
          elsif output_dir_structure.by_language_code?
            '../../_sisu/image'
          elsif output_dir_structure.by_filetype?
            '../_sisu/image'
          else
            '../_sisu/image'
          end
        end
        def rcp
          set_path(ft).rcp.abc
        end
        self
      end
      def qrcode
        def ft
          'manifest/qrcode'
        end
        def dir
          set_path(ft).dir.abc
        end
        def url
          set_path(ft).url.abc
        end
        def rel
          set_path(ft).rel.abc
        end
        def rcp
          set_path(ft).rcp.abc
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def harvest
        def ft
          'site_metadata'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        def rel_sm
          if output_dir_structure.by_language_code?
            ''
          elsif output_dir_structure.by_filetype?
            ''
          else
            ''
          end
        end
        self
      end
      def manpage
        def ft
          'man'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def texinfo
        def ft
          'texinfo'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        def rel_sm
          set_path(ft).rel_sm.ab
        end
        self
      end
      def sitemaps
        def ft
          'sitemaps'
        end
        def dir
          set_path(ft).dir.ab
        end
        def url
          set_path(ft).url.ab
        end
        def rel
          set_path(ft).rel.ab
        end
        def rcp
          set_path(ft).rcp.ab
        end
        self
      end
      def sqlite #check url
        def dir
          output_path.base.dir
        end
        def url
          output_path.base.url
        end
        def rel
          output_path.base.rel
        end
        def rcp
          output_path.base.rcp
        end
        self
      end
      #def cgi
      #end
      def css
        @d='_sisu/css'
        def dir
          output_path.base.dir + '/' + @d
        end
        def url
          output_path.base.url + '/' + @d
        end
        def rel
          @d
          #output_path.base.rel + '/' + @d
        end
        def rcp
          output_path.stub.rcp + '/' + @d
        end
        self
      end
      def images
        @d='_sisu/image'
        def dir
          output_path.base.dir + '/' + @d
        end
        def url
          output_path.base.url + '/' + @d
        end
        def rel
          @d
          #output_path.base.rel + '/' + @d
        end
        def rcp
          output_path.stub.rcp + '/' + @d
        end
        self
      end
      def images_external
        @d='_sisu/image_external'
        def dir
          output_path.base.dir + '/' + @d
        end
        def url
          output_path.base.url + '/' + @d
        end
        def rel
          output_path.base.rel + '/' + @d
        end
        def rcp
          output_path.base.rcp + '/' + @d
        end
        self
      end
      #def css
      #  #"#{@env.path.output}/#{@env.path.style}"
      #end
      self
    end
  end
end
module SiSU_Create_File
  require_relative 'constants'                             # constants.rb
  require_relative 'utils'                                 # utils.rb
  require_relative 'se_info_env'                           # se_info_env.rb
  class CreateFile < SiSU_Info_Env::InfoEnv                # se_info_env.rb
    #todo unify with FileOp
    def initialize(fns)
      begin
        super(fns)
        @env=SiSU_Env::InfoEnv.new(fns)
      rescue
        SiSU_Screen::Ansi.new('',$!,$@).rescue do
          __LINE__.to_s + ':' + __FILE__
        end
      ensure
      end
    end
    def html_root
      #@env.path.output
    end
    def mkdir_pdf
      Dir.mkdir(@env.processing_path.tex) \
        unless FileTest.directory?(@env.processing_path.tex)
    end
    def file_generic(output_file='')
      fn=@env.path.output + '/' \
      + @fnb + '/' \
      + output_file
      File.new(fn,'w+')
    end
    def file_error
      fn='/tmp/errorlog.sisu'
      File.new(fn,'w+')
    end
    def file_txt
      fn=@env.processing_path.ao + '/' \
      + @fns + '.txt'
      File.new(fn,'w+')
    end
    def file_debug
      fn=@env.processing_path.ao + '/' \
      + @fns + '.debug.txt'
      File.new(fn,'w+')
    end
    def metaverse
      def file_meta
        fn=@env.processing_path.ao + '/' \
        + @fns + '.meta'
        File.new(fn,'w+')
      end
      def file_meta_idx_html
        fn=@env.processing_path.ao + '/' \
        + @fns + '.idx.html'
        File.new(fn,'w+')
      end
      self
    end
    def file_note
      fn=Dir.pwd + '/' \
      + @fns + '.fn'
      File.new(fn,'w+')
    end
    def meta
      @env.processing_path.ao + '/' \
      + @fns + '.meta'
    end
    def file_semantic
      fn='./semantic.yaml'
      @@filename_semantic=File.new(fn,'w+')
    end
    def file_rss
      fn='./semantic.xml'
      @@filename_rss=File.new(fn,'w+')
    end
    def epub
      @pth=@env.processing_path.epub
      def xhtml_index
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/index.xhtml'
        File.new(fn,'w+')
      end
      def xhtml_cover_image
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/cover_image.xhtml'
        File.new(fn,'w+')
      end
      def xhtml_segtoc
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/toc.xhtml'
        File.new(fn,'w+')
      end
      def mimetype  #fixed application/epub+zip ~/grotto/theatre/dbld/builds/epub_sample/mimetype
        File.new("#{@pth}/mimetype",'w')
      end
      def metadata #variable matadata ~/grotto/theatre/dbld/builds/epub_sample/metadata.opf
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/' \
        + Ep[:f_opf]
        File.new(fn,'w')
      end
      def toc_ncx  #variable toc ~/grotto/theatre/dbld/builds/epub_sample/toc.ncx
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/' \
        + Ep[:f_ncx]
        File.new(fn,'w')
      end
      def metainf_cont #variable content ~/grotto/theatre/dbld/builds/epub_sample/META-INF/container.xml
        fn=@pth + '/META-INF/container.xml'
        File.new(fn,'w')
      end
      def xhtml_css #fixed epub xhtml css
        fn=@pth + '/' \
        + Ep[:d_oebps] + '/css/xhtml.css'
        File.new(fn,'w')
      end
      self
    end
    def file_texinfo
      fn=@env.processing_path.texinfo + '/' \
      + @fnb + '.texinfo'
      File.new(fn,'w+')
    end
  end
end
module SiSU_Filename_Lang
  require_relative 'constants'                             # constants.rb
  require_relative 'utils'                                 # utils.rb
  class FilenameLanguageCodeInsert
    def initialize(opt,lng=nil)
      @opt=opt
      @lng=lng ||=opt.lng
    end
    def language_code_insert
      if @opt.dir_structure_by ==:language \
      or ((@opt.dir_structure_by ==:filetype \
      || @opt.dir_structure_by ==:filename) \
      and (@opt.lingual ==:mono \
      && @lng == @opt.act[:default_language][:code]))
        ''
      elsif (@opt.dir_structure_by ==:filetype \
      || @opt.dir_structure_by ==:filename) \
      and not @opt.lingual ==:mono
        '.' + @lng
      else
        '.' + @lng
      end
    end
  end
end
__END__
