-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with E_Strings;

separate (DAG.BuildGraph)
procedure ModelProcedureCall is

   PreConNode, PostConNode : STree.SyntaxNode;
   ProcedureNameCell       : Cells.Cell;
   ProcedureSym            : Dictionary.Symbol;
   PrefixSym               : Dictionary.Symbol := Dictionary.NullSymbol;
   SubstitutionTable       : Cells.Cell;

   -- The interface of a procedure can have two views. An abstract view and a
   -- refined view which are denoted by the type Dictionary.Abstractions.
   -- A refined view can be defined by data refinement or proof refinement.
   -- Data refinement occurs when an own variable is refined into constituent
   -- parts and may also involve proof refinement where the pre and post
   -- conditions are in terms of the constituents.
   -- However, proof refinement may occur without data refinement to facilitate
   -- defining pre and post conditions in terms of the complete view of a
   -- private type.
   -- Hence the introduction of these two variables.
   --   Constraint_View gives the view to be used for the pre and post
   --   conditions,
   --   Data_View gives the view to be used for the parameters, globals and
   --   derives annotations and from this the view of the imports and exports of
   --   the subprogram.
   Constraint_View, Data_View : Dictionary.Abstractions;

   ----------------------------------------------------------

   procedure BuildSubstitutionTable (Node : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     DoAssumeLocalRvalues;
   --#        in     LineNmbr;
   --#        in     LoopStack;
   --#        in     LScope;
   --#        in     OutputFile;
   --#        in     ProcedureSym;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out FlowHeap;
   --#        in out Graph.Table;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.State;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --#           out SubstitutionTable;
   --# derives CheckStack,
   --#         ContainsReals,
   --#         Dictionary.Dict,
   --#         FlowHeap,
   --#         Graph.Table,
   --#         KindOfStackedCheck,
   --#         LexTokenManager.State,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         StmtStack.S,
   --#         VCGFailure,
   --#         VCGHeap                    from *,
   --#                                         CheckStack,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         DoAssumeLocalRvalues,
   --#                                         FlowHeap,
   --#                                         Graph.Table,
   --#                                         KindOfStackedCheck,
   --#                                         LexTokenManager.State,
   --#                                         LineNmbr,
   --#                                         LoopStack,
   --#                                         LScope,
   --#                                         Node,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         ShortCircuitStack,
   --#                                         StmtStack.S,
   --#                                         STree.Table,
   --#                                         VCGHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CheckStack,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         DoAssumeLocalRvalues,
   --#                                         ErrorHandler.Error_Context,
   --#                                         FlowHeap,
   --#                                         Graph.Table,
   --#                                         KindOfStackedCheck,
   --#                                         LexTokenManager.State,
   --#                                         LineNmbr,
   --#                                         LoopStack,
   --#                                         LScope,
   --#                                         Node,
   --#                                         OutputFile,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         ShortCircuitStack,
   --#                                         SPARK_IO.File_Sys,
   --#                                         StmtStack.S,
   --#                                         STree.Table,
   --#                                         VCGHeap &
   --#         SubstitutionTable          from VCGHeap;
   is
      NameArgListNode : STree.SyntaxNode;

      function NotModeOut (ParamNumber : Positive) return Boolean
      --# global in Dictionary.Dict;
      --#        in ProcedureSym;
      is
      begin
         return Dictionary.GetSubprogramParameterMode (Dictionary.GetSubprogramParameter (ProcedureSym, ParamNumber)) /=
           Dictionary.OutMode;
      end NotModeOut;

      -----------------

      procedure DoPositionalAssociation
      --# global in     CommandLineData.Content;
      --#        in     DoAssumeLocalRvalues;
      --#        in     LineNmbr;
      --#        in     LoopStack;
      --#        in     LScope;
      --#        in     NameArgListNode;
      --#        in     OutputFile;
      --#        in     ProcedureSym;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     SubstitutionTable;
      --#        in out CheckStack;
      --#        in out ContainsReals;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out FlowHeap;
      --#        in out Graph.Table;
      --#        in out KindOfStackedCheck;
      --#        in out LexTokenManager.State;
      --#        in out ShortCircuitStack;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out StmtStack.S;
      --#        in out VCGFailure;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ContainsReals,
      --#         Dictionary.Dict,
      --#         FlowHeap,
      --#         Graph.Table,
      --#         KindOfStackedCheck,
      --#         LexTokenManager.State,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage,
      --#         StmtStack.S,
      --#         VCGFailure,
      --#         VCGHeap                    from *,
      --#                                         CheckStack,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         DoAssumeLocalRvalues,
      --#                                         FlowHeap,
      --#                                         Graph.Table,
      --#                                         KindOfStackedCheck,
      --#                                         LexTokenManager.State,
      --#                                         LineNmbr,
      --#                                         LoopStack,
      --#                                         LScope,
      --#                                         NameArgListNode,
      --#                                         ProcedureSym,
      --#                                         Scope,
      --#                                         ShortCircuitStack,
      --#                                         StmtStack.S,
      --#                                         STree.Table,
      --#                                         SubstitutionTable,
      --#                                         VCGHeap &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CheckStack,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         DoAssumeLocalRvalues,
      --#                                         ErrorHandler.Error_Context,
      --#                                         FlowHeap,
      --#                                         Graph.Table,
      --#                                         KindOfStackedCheck,
      --#                                         LexTokenManager.State,
      --#                                         LineNmbr,
      --#                                         LoopStack,
      --#                                         LScope,
      --#                                         NameArgListNode,
      --#                                         OutputFile,
      --#                                         ProcedureSym,
      --#                                         Scope,
      --#                                         ShortCircuitStack,
      --#                                         SPARK_IO.File_Sys,
      --#                                         StmtStack.S,
      --#                                         STree.Table,
      --#                                         SubstitutionTable,
      --#                                         VCGHeap;
      is
         -- Builds a table using the facilities of CLists.  Each element of the list is a cell with
         -- a parameter number as its contents.  The C pointer of each list element points to a
         -- A DAG representing the actual parameter.  If there is a constraining index involved
         -- (i.e. the actual parameter is a constrained subtype of an unconstrained array) then
         -- a cell of kind ConstrainingIndex is placed between the parameter list cell and the
         -- expression DAG itself.  The structure is a bit like this:

         -- PNA experiment.  In the case of an actual parameter that is a named subtype of an
         --                  unconstrained array type then the constraint node will hold the
         --                  array subtype symbol rather than the first index subtype as originally
         --                  implemented.  Note that in the special case of a string literal actual parameter
         --                  then the constraint will still have an index type (a subtype of positive).

         --
         -- SubstitutionTable -> 1 -> 2 -> 3 -> etc
         --                      |    |    |
         --                      v    v    v
         --                     dag  dag  constraint
         --                                |
         --                                v
         --                               dag

         ParamCounter            : Positive;
         TryNode                 : STree.SyntaxNode;
         ListElement, Expression : Cells.Cell;
         ConstraintCell          : Cells.Cell;
         ConstraintIndex         : Dictionary.Symbol;

         -----------------

      begin -- DoPositionalAssociation
         ParamCounter := 1;
         TryNode      := NameArgListNode;
         while STree.Syntax_Node_Type (Node => TryNode) /= SP_Symbols.expression loop
            TryNode := STree.Child_Node (Current_Node => TryNode);
         end loop;
         -- TryNode is now bottommost expression ie. first parameter
         while TryNode /= STree.NullNode loop
            ConstraintCell := Cells.Null_Cell;
            Cells.Create_Cell (VCGHeap, ListElement);
            Cells.Set_Natural_Value (VCGHeap, ListElement, ParamCounter);

            BuildExpnDAG
              (OutputFile,
               TryNode,
               LScope,
               Scope,
               LineNmbr,
               True,
               DoAssumeLocalRvalues and then NotModeOut (ParamCounter),
               LoopStack,
               FlowHeap,
               VCGHeap,
               ContainsReals,
               VCGFailure,
               ShortCircuitStack,
               CheckStack,
               KindOfStackedCheck,
               -- to get
               Expression);

            -- Constraining index (if there is one) will have been planted at expression node by wffs

            ConstraintIndex := STree.NodeSymbol (TryNode);
            if ConstraintIndex /= Dictionary.NullSymbol then
               CreateCellKind (ConstraintCell, VCGHeap, Cells.Constraining_Index);
               Cells.Set_Symbol_Value (VCGHeap, ConstraintCell, ConstraintIndex);
               -- PNA unchnaged name at present, but "Index" is misleading now
            end if;

            -- We may need to convert the actual parameter by inserting some inherit
            -- derefences in front of it; conversion is required if we have called
            -- an inherited root function.  The parameter in this case must be an
            -- object.
            ConvertTaggedActualIfNecessary (ProcedureSym, VCGHeap, Expression);

            -- Link constraint (if any) and DAG into linked list of parameters
            if Cells.Is_Null_Cell (ConstraintCell) then
               SetAuxPtr (ListElement, Expression, VCGHeap);
            else
               SetAuxPtr (ListElement, ConstraintCell, VCGHeap);
               SetAuxPtr (ConstraintCell, Expression, VCGHeap);
            end if;
            Clists.InsertCell (VCGHeap, ListElement, SubstitutionTable);
            ParamCounter := ParamCounter + 1;
            TryNode      := STree.Next_Sibling (Current_Node => STree.Parent_Node (Current_Node => TryNode));
         end loop;
      end DoPositionalAssociation;

      -------------------------------------------------------------

      procedure DoNamedAssociation
      --# global in     CommandLineData.Content;
      --#        in     DoAssumeLocalRvalues;
      --#        in     LineNmbr;
      --#        in     LoopStack;
      --#        in     LScope;
      --#        in     NameArgListNode;
      --#        in     OutputFile;
      --#        in     ProcedureSym;
      --#        in     Scope;
      --#        in     STree.Table;
      --#        in     SubstitutionTable;
      --#        in out CheckStack;
      --#        in out ContainsReals;
      --#        in out Dictionary.Dict;
      --#        in out ErrorHandler.Error_Context;
      --#        in out FlowHeap;
      --#        in out Graph.Table;
      --#        in out KindOfStackedCheck;
      --#        in out LexTokenManager.State;
      --#        in out ShortCircuitStack;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out StmtStack.S;
      --#        in out VCGFailure;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ContainsReals,
      --#         Dictionary.Dict,
      --#         FlowHeap,
      --#         Graph.Table,
      --#         KindOfStackedCheck,
      --#         LexTokenManager.State,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage,
      --#         StmtStack.S,
      --#         VCGFailure,
      --#         VCGHeap                    from *,
      --#                                         CheckStack,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         DoAssumeLocalRvalues,
      --#                                         FlowHeap,
      --#                                         Graph.Table,
      --#                                         KindOfStackedCheck,
      --#                                         LexTokenManager.State,
      --#                                         LineNmbr,
      --#                                         LoopStack,
      --#                                         LScope,
      --#                                         NameArgListNode,
      --#                                         ProcedureSym,
      --#                                         Scope,
      --#                                         ShortCircuitStack,
      --#                                         StmtStack.S,
      --#                                         STree.Table,
      --#                                         SubstitutionTable,
      --#                                         VCGHeap &
      --#         ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CheckStack,
      --#                                         CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         DoAssumeLocalRvalues,
      --#                                         ErrorHandler.Error_Context,
      --#                                         FlowHeap,
      --#                                         Graph.Table,
      --#                                         KindOfStackedCheck,
      --#                                         LexTokenManager.State,
      --#                                         LineNmbr,
      --#                                         LoopStack,
      --#                                         LScope,
      --#                                         NameArgListNode,
      --#                                         OutputFile,
      --#                                         ProcedureSym,
      --#                                         Scope,
      --#                                         ShortCircuitStack,
      --#                                         SPARK_IO.File_Sys,
      --#                                         StmtStack.S,
      --#                                         STree.Table,
      --#                                         SubstitutionTable,
      --#                                         VCGHeap;
      is
         -- See comment in DoPositionalAssociation above for the purpose of this procedure.
         ParamCounter            : Positive;
         TryNode                 : STree.SyntaxNode;
         ListElement, Expression : Cells.Cell;
         ConstraintCell          : Cells.Cell;
         ConstraintIndex         : Dictionary.Symbol;

         -------------------------------------------------------

         procedure GetParamNumber (Name    : in     LexTokenManager.Lex_String;
                                   ProcSym : in     Dictionary.Symbol;
                                   ParamNo :    out Positive)
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.State;
         --# derives ParamNo from Dictionary.Dict,
         --#                      LexTokenManager.State,
         --#                      Name,
         --#                      ProcSym;
         is
            It  : Dictionary.Iterator;
            Sym : Dictionary.Symbol;

         begin
            It := Dictionary.FirstSubprogramParameter (ProcSym);
            SystemErrors.RT_Assert
              (C       => not Dictionary.IsNullIterator (It),
               Sys_Err => SystemErrors.Precondition_Failure,
               Msg     => "Can't find first parameter in BuildGraph.ModelProcedureCall.GetParamNumber");
            loop
               Sym := Dictionary.CurrentSymbol (It);
               exit when LexTokenManager.Lex_String_Case_Insensitive_Compare
                 (Lex_Str1 => Dictionary.GetSimpleName (Sym),
                  Lex_Str2 => Name) =
                 LexTokenManager.Str_Eq;
               It := Dictionary.NextSymbol (It);
               exit when Dictionary.IsNullIterator (It);
            end loop;
            ParamNo := Dictionary.GetSubprogramParameterNumber (Sym);
         end GetParamNumber;

         --------------------------------------------------------------

      begin -- DoNamedAssociation
         TryNode := NameArgListNode;
         while STree.Syntax_Node_Type (Node => TryNode) /= SP_Symbols.simple_name loop
            TryNode := STree.Child_Node (Current_Node => TryNode);
         end loop;
         -- TryNode is now simple_name of first parameter
         while TryNode /= STree.NullNode loop
            ConstraintCell := Cells.Null_Cell;
            Cells.Create_Cell (VCGHeap, ListElement);
            GetParamNumber
              (STree.Node_Lex_String (Node => STree.Child_Node (Current_Node => TryNode)),
               ProcedureSym,
               -- to get
               ParamCounter);
            Cells.Set_Natural_Value (VCGHeap, ListElement, ParamCounter);
            BuildExpnDAG
              (OutputFile,
               STree.Next_Sibling (Current_Node => TryNode),
               LScope,
               Scope,
               LineNmbr,
               True,
               DoAssumeLocalRvalues and then NotModeOut (ParamCounter), -- assume rvalues only for in/inout params
               LoopStack,
               FlowHeap,
               VCGHeap,
               ContainsReals,
               VCGFailure,
               ShortCircuitStack,
               CheckStack,
               KindOfStackedCheck,
               -- to get
               Expression);

            -- Constraining index (if there is one) will have been planted at expression node by wffs
            ConstraintIndex := STree.NodeSymbol (STree.Next_Sibling (Current_Node => TryNode));
            if ConstraintIndex /= Dictionary.NullSymbol then
               CreateCellKind (ConstraintCell, VCGHeap, Cells.Constraining_Index);
               Cells.Set_Symbol_Value (VCGHeap, ConstraintCell, ConstraintIndex);
            end if;

            -- We may need to convert the actual parameter by inserting some inherit
            -- derefences in front of it; conversion is required if we have called
            -- an inherited root function.  The parameter in this case must be an
            -- object.
            ConvertTaggedActualIfNecessary (ProcedureSym, VCGHeap, Expression);

            -- Link constraint (if any) and DAG into linked list of parameters
            if Cells.Is_Null_Cell (ConstraintCell) then
               SetAuxPtr (ListElement, Expression, VCGHeap);
            else
               SetAuxPtr (ListElement, ConstraintCell, VCGHeap);
               SetAuxPtr (ConstraintCell, Expression, VCGHeap);
            end if;
            Clists.InsertCell (VCGHeap, ListElement, SubstitutionTable);
            TryNode := STree.Next_Sibling (Current_Node => STree.Parent_Node (Current_Node => TryNode));
         end loop;
      end DoNamedAssociation;

      -------------------------------------------------------------

   begin  -- BuildSubstitutionTable

      -- Node is procedure_call_statement
      Clists.CreateList (VCGHeap, SubstitutionTable);
      NameArgListNode :=
        STree.Next_Sibling (Current_Node => STree.Child_Node (Current_Node => STree.Child_Node (Current_Node => Node)));
      if NameArgListNode /= STree.NullNode then
         if STree.Syntax_Node_Type (Node => STree.Child_Node (Current_Node => NameArgListNode)) =
           SP_Symbols.positional_argument_association then
            DoPositionalAssociation;
         else
            DoNamedAssociation;
         end if;
      end if;
   end BuildSubstitutionTable;

   ----------------------------------------------------------

   function GetExpressionCell (Parameter : Positive) return Cells.Cell
   --# global in SubstitutionTable;
   --#        in VCGHeap;
   is
      -- For a particular parameter number, find the DAG associated with the actual parameter
      -- expression.  This will be found in the substitution table built by procedure BuildSubstitutionTable
      Try : Cells.Cell;
   begin
      Try := Clists.FirstCell (VCGHeap, SubstitutionTable);
      while Cells.Get_Natural_Value (VCGHeap, Try) /= Parameter loop
         Try := Clists.NextCell (VCGHeap, Try);
      end loop;
      Try := AuxPtr (VCGHeap, Try);
      if Cells.Get_Kind (VCGHeap, Try) = Cells.Constraining_Index then
         -- expression is one lower in the data strucure - see comment in DoPositionalAssociation above
         Try := AuxPtr (VCGHeap, Try);
      end if;
      return Try;
   end GetExpressionCell;

   ----------------------------------------------------------

   function GetConstraintCell (Parameter : Positive) return Cells.Cell
   --# global in SubstitutionTable;
   --#        in VCGHeap;
   is
      -- This is similar to GetExpressionCell but returns the constraining index cell if there is one otherwise
      -- it returns the actual parameter expression's DAG.
      Try : Cells.Cell;
   begin
      Try := Clists.FirstCell (VCGHeap, SubstitutionTable);
      while Cells.Get_Natural_Value (VCGHeap, Try) /= Parameter loop
         Try := Clists.NextCell (VCGHeap, Try);
      end loop;
      return AuxPtr (VCGHeap, Try);
   end GetConstraintCell;

   ----------------------------------------------------------
   -- There are several places below where we need to traverse a DAG that contains
   -- record field selections and/or array element references.  The following two functions
   -- simplify this search process

   function IsSelector (TheCell : Cells.Cell) return Boolean
   --# global in VCGHeap;
   is
   begin
      return Cells.Get_Kind (VCGHeap, TheCell) = Cells.Field_Access_Function
        or else Cells.Get_Kind (VCGHeap, TheCell) = Cells.Element_Function;
   end IsSelector;

   -- a field selector function has the form:
   -- fld_name --- expression
   --
   -- an array element function has the form
   -- element --- , --- index
   --             |
   --          expression
   function ArgumentOfSelector (TheCell : Cells.Cell) return Cells.Cell
   --# global in VCGHeap;
   is
      Result : Cells.Cell;
   begin
      if Cells.Get_Kind (VCGHeap, TheCell) = Cells.Field_Access_Function then
         -- for a record field access, the expression is to the right
         Result := RightPtr (VCGHeap, TheCell);
      else -- must be element function because of precondition

         -- for an array access, we have a comma to the right and the expression to the left of this
         Result := LeftPtr (VCGHeap, RightPtr (VCGHeap, TheCell));
      end if;
      return Result;
   end ArgumentOfSelector;

   ----------------------------------------------------------
   -- Given a symbol for an export which is either the symbol of a global
   -- variable or the symbol of a formal parameter, this procedure returns
   -- the symbol of the entire variable being exported and the DAG that
   -- represents the actual parameter expression.
   -- If the export is a global then EntireActualSym = FormalOrGlobalSym and
   -- ActualDAG = Cells.Null_Cell.  For parameters the substitution table is used
   -- to obtain the returned results.
   --
   procedure GetExportDetails
     (FormalOrGlobalSym : in     Dictionary.Symbol;
      EntireActualSym   :    out Dictionary.Symbol;
      ActualDAG         :    out Cells.Cell)
   --# global in Dictionary.Dict;
   --#        in PrefixSym;
   --#        in ProcedureSym;
   --#        in SubstitutionTable;
   --#        in VCGHeap;
   --# derives ActualDAG       from Dictionary.Dict,
   --#                              FormalOrGlobalSym,
   --#                              ProcedureSym,
   --#                              SubstitutionTable,
   --#                              VCGHeap &
   --#         EntireActualSym from Dictionary.Dict,
   --#                              FormalOrGlobalSym,
   --#                              PrefixSym,
   --#                              ProcedureSym,
   --#                              SubstitutionTable,
   --#                              VCGHeap;
   is
      ActualDAGLocal       : Cells.Cell;
      EntireActualSymLocal : Dictionary.Symbol;

      ----------------------------------------------------------

      function SubstituteProtectedTypeSelfReference (Sym, PrefixSymbol : Dictionary.Symbol) return Dictionary.Symbol
      --# global in Dictionary.Dict;
      is
         Result : Dictionary.Symbol;
      begin
         -- if Sym is the implicitly-declared own variable of a protected type
         -- then we must replace it with the "current instance of the protected object"
         -- before checking whether it is visible.
         -- Background: given protected type PT its operations will globally reference and
         -- derive PT meaning, in this case, "myself".
         -- If an object PO of type PT (or a subtype of PT) is declared then calls to its
         -- operations will take the form PO.Op and the calling environment will be annotated
         -- in terms of PO.  Therefore, when checking that the globals necessary for the call
         -- PO.Op are visible (for example), we need to replace all references to PT into
         -- references to PO before making the check.  The Prefix Symbol of the call is the
         -- symbol we need to substitute in.
         Result := Sym;
         if PrefixSymbol /= Dictionary.NullSymbol
           and then Dictionary.IsOwnVariable (Sym)
           and then Dictionary.IsProtectedType (Dictionary.GetOwner (Sym)) then
            Result := PrefixSymbol;
         end if;
         return Result;
      end SubstituteProtectedTypeSelfReference;

   begin  -- GetExportDetails

      -- Debug.PrintSym ("Call to GetExportDetails with ", FormalOrGlobalSym);
      if Dictionary.IsFormalParameter (ProcedureSym, FormalOrGlobalSym) then
         ActualDAGLocal := GetExpressionCell (Dictionary.GetSubprogramParameterNumber (FormalOrGlobalSym));

         ActualDAG := ActualDAGLocal;
         while IsSelector (ActualDAGLocal) loop
            ActualDAGLocal := ArgumentOfSelector (ActualDAGLocal);
         end loop;
         EntireActualSymLocal := Cells.Get_Symbol_Value (VCGHeap, ActualDAGLocal);

      else
         EntireActualSymLocal := FormalOrGlobalSym;
         ActualDAG            := Cells.Null_Cell;
      end if;
      -- if the export is from a protected procedure it may ba type name (representing "this") rather
      -- than the protected object itself; the following substitutes the PO name.
      EntireActualSym := SubstituteProtectedTypeSelfReference (EntireActualSymLocal, PrefixSym);
      -- Debug.PrintSym ("Returning from GetExportDetails with ", EntireActualSym);
   end GetExportDetails;

   ----------------------------------------------------------
   procedure SubstituteImport (Sym    : in     Dictionary.Symbol;
                               Change :    out Boolean;
                               Result :    out Cells.Cell)
   --# global in Dictionary.Dict;
   --#        in ProcedureSym;
   --#        in SubstitutionTable;
   --#        in VCGHeap;
   --# derives Change from Dictionary.Dict,
   --#                     ProcedureSym,
   --#                     Sym &
   --#         Result from Dictionary.Dict,
   --#                     ProcedureSym,
   --#                     SubstitutionTable,
   --#                     Sym,
   --#                     VCGHeap;
   is
      -- Given the Symbol of an import, replaces it with the matching actual parameter expression
      -- (if a formal parameter) otherwise does nothing.  Change is set to True if a substitution has been made
      LResult : Cells.Cell;
   begin
      if Dictionary.IsFormalParameter (ProcedureSym, Sym) then
         Change  := True;
         LResult := GetExpressionCell (Dictionary.GetSubprogramParameterNumber (Sym));
      else
         LResult := Cells.Null_Cell;
         Change  := False;
      end if;
      Result := LResult;
   end SubstituteImport;

   ---------------------------------------------------------------------

   procedure SubstituteImportConstraint (Sym    : in     Dictionary.Symbol;
                                         Change :    out Boolean;
                                         Result :    out Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in     ProcedureSym;
   --#        in     SubstitutionTable;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives Change                from Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    Sym &
   --#         Result                from Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    Sym,
   --#                                    VCGHeap &
   --#         Statistics.TableUsage from *,
   --#                                    Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    Sym,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    SubstitutionTable,
   --#                                    Sym;
   is
      -- Similar to SubstituteImport above but returns the constraining index type associated with
      -- a constrained actual parameter associated with an unconstrained formal parameter

      LResult       : Cells.Cell;
      ConstraintSym : Dictionary.Symbol;

      ObjectSym      : Dictionary.Symbol;
      ArrayDimension : Positive;
   begin
      -- Debug.PrintMsg ("In SubstituteImportConstraint", True);
      -- Debug.PrintSym ("Sym passed in is ", Sym);

      -- The Sym passed to this routine will be a Dictionary.ParameterConstraintSymbol.
      -- From this we can obtain the object itself and the dimesnion of that object that appears
      -- in the expression we may be making substitutions to.
      ObjectSym := Dictionary.GetParameterAssociatedWithParameterConstraint (Sym);
      -- Debug.PrintSym ("Object sym is ", ObjectSym);
      ArrayDimension := Dictionary.GetSubprogramParameterConstraintDimension (Sym);

      if Dictionary.IsFormalParameter (ProcedureSym, ObjectSym) then
         Change := True;
         Cells.Create_Cell (VCGHeap, LResult);
         Cells.Copy_Contents (VCGHeap, GetConstraintCell (Dictionary.GetSubprogramParameterNumber (ObjectSym)), LResult);
         -- LResult contains either:
         -- (1) an array subtype symbol in the case where the actual paramater is of a constrained
         --     array subtype
         -- (2) a scalar index type symbol in the case of a string literal being passed to string
         -- (3) a symbol of a subprogram parameter in the case where the actual parameter is also
         --     an unconstrained array and no constraint has been planted (this final behaviour occurs
         --     because GetConstraintCell returns the actual parameter DAG if no constraint is present)
         ConstraintSym := Cells.Get_Symbol_Value (VCGHeap, LResult);
         -- Debug.PrintSym ("Constraint sym obtained from syntax tree ", ConstraintSym);
         if Dictionary.IsSubprogramParameter (ConstraintSym) then
            -- Case 3.  We substitute "actual__index__subtype__n" for "formal__index__subtype__n"
            -- Debug.PrintMsg ("Case 3", True);
            Cells.Set_Symbol_Value (VCGHeap, LResult, Dictionary.GetSubprogramParameterConstraint (ConstraintSym, ArrayDimension));

         elsif Dictionary.TypeIsArray (ConstraintSym) then
            -- Case 2. We substitute array index n of constraining subtype for "formal__index__subtype__n"
            -- Debug.PrintMsg ("Case 2", True);
            Cells.Set_Symbol_Value (VCGHeap, LResult, Dictionary.GetArrayIndex (ConstraintSym, ArrayDimension));
         else
            -- Case 1. we already have the constraining index directly
            -- Debug.PrintMsg ("Case 1", True);
            null;
         end if;
      else
         LResult := Cells.Null_Cell;
         Change  := False;
      end if;
      -- Debug.PrintSym ("Substituted symbol is ", Cells.Get_Symbol_Value (VCGHeap, LResult));
      -- Debug.PrintBool ("Change is ", Change);
      Result := LResult;
   end SubstituteImportConstraint;

   ---------------------------------------------------------------------

   ---------------------------------------------------------------------
   -- procedure to create modelling vars for procedure export
   procedure ConvertCellToExportCell (CellName : in Cells.Cell)
   --# global in     Dictionary.Dict;
   --#        in     PrefixSym;
   --#        in     ProcedureSym;
   --#        in     SubprogramCalls;
   --#        in     SubstitutionTable;
   --#        in out LexTokenManager.State;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives LexTokenManager.State from *,
   --#                                    SubprogramCalls &
   --#         Statistics.TableUsage from *,
   --#                                    CellName,
   --#                                    Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    SubstitutionTable,
   --#                                    VCGHeap &
   --#         VCGHeap               from *,
   --#                                    CellName,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    PrefixSym,
   --#                                    ProcedureSym,
   --#                                    SubprogramCalls,
   --#                                    SubstitutionTable;
   is
      CountStr                           : LexTokenManager.Lex_String;
      CellNameLocal, TempCell, ExportDAG : Cells.Cell;
      ExportSym                          : Dictionary.Symbol;

   begin
      -- if the cell supplied is simply a variable we convert the cell sort to
      -- a procedure export variable adding in the necessary call counter.  If
      -- it is the top of a DAG containing record field references we convert the
      -- cell which represents the entire record variable after making a copy
      -- of the actual DAG.
      LexTokenManager.Insert_Nat (N       => SubprogramCalls,
                                  Lex_Str => CountStr);
      GetExportDetails (Cells.Get_Symbol_Value (VCGHeap, CellName), ExportSym, ExportDAG);

      -- see if there is an Export DAG representing a fld access of a record
      -- variable or an array element function.
      if not Cells.Is_Null_Cell (ExportDAG)
        and then (Cells.Get_Kind (VCGHeap, ExportDAG) = Cells.Field_Access_Function
                    or else Cells.Get_Kind (VCGHeap, ExportDAG) = Cells.Element_Function) then
         -- field access or array element access found we must substitute input node with
         -- the export DAG and converting the entire variable concerned to an export
         -- variable.
         Structures.CopyStructure (VCGHeap, ExportDAG, CellNameLocal);
         Cells.Copy_Contents (VCGHeap, CellNameLocal, CellName);
         TempCell := CellName;
         while IsSelector (TempCell) loop
            TempCell := ArgumentOfSelector (TempCell);
         end loop;
         -- TempCell now points at entire variable cell
         Cells.Set_Kind (VCGHeap, TempCell, Cells.Procedure_Export);
         Cells.Set_Lex_Str (VCGHeap, TempCell, CountStr);

      else
         --  no FieldAccess found so export is an entire variable.
         --  Just convert cell directly to an export cell as before
         --  and put in the symbol of the actual export
         Cells.Set_Kind (VCGHeap, CellName, Cells.Procedure_Export);
         Cells.Set_Symbol_Value (VCGHeap, CellName, ExportSym);
         Cells.Set_Lex_Str (VCGHeap, CellName, CountStr);
      end if;

   end ConvertCellToExportCell;

   ---------------------------------------------------------------------

   -- The view of the precondition, abstract or refined is determined by the parameter Constraint_View
   procedure ModelPrecondition (Constraint_View : in Dictionary.Abstractions)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LoopStack;
   --#        in     PreConNode;
   --#        in     ProcedureSym;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in     SubstitutionTable;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ErrorHandler.Error_Context;
   --#        in out KindOfStackedCheck;
   --#        in out LexTokenManager.State;
   --#        in out ShortCircuitStack;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ContainsReals,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap                    from *,
   --#                                         CheckStack,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PreConNode,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         ShortCircuitStack,
   --#                                         STree.Table,
   --#                                         SubstitutionTable,
   --#                                         VCGHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PreConNode,
   --#                                         ProcedureSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         VCGHeap &
   --#         KindOfStackedCheck         from *,
   --#                                         Dictionary.Dict,
   --#                                         PreConNode,
   --#                                         ProcedureSym &
   --#         LexTokenManager.State      from *,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Dictionary.Dict,
   --#                                         LoopStack,
   --#                                         PreConNode,
   --#                                         ProcedureSym,
   --#                                         STree.Table,
   --#                                         VCGHeap;
   is
      -- The following declaration is employed in constructing assume statement:
      DAGCell                   : Cells.Cell;
      GenericOrActualSubprogram : Dictionary.Symbol;
      ConstraintNode            : STree.SyntaxNode;

      --------------------------------------------

      procedure SubstituteParameters (ConstraintRoot : in Cells.Cell)
      -- replace formal parameters by actual ones in a subprogram constraint;
      --# global in     Dictionary.Dict;
      --#        in     ProcedureSym;
      --#        in     SubstitutionTable;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ConstraintRoot,
      --#                                    Dictionary.Dict,
      --#                                    ProcedureSym,
      --#                                    SubstitutionTable,
      --#                                    VCGHeap;
      is
         Subs, P : Cells.Cell;
         S       : CStacks.Stack;
         VarSym  : Dictionary.Symbol;
         Change  : Boolean;

         function IsLeaf (Node : Cells.Cell) return Boolean
         --# global in VCGHeap;
         is
         begin
            return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
         end IsLeaf;

      begin -- SubstituteParameters
            -- DAG traversal algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
         CStacks.CreateStack (S);
         Subs := Cells.Null_Cell; -- to avoid conditional DFA later
         P    := ConstraintRoot;
         loop
            loop
               exit when Cells.Is_Null_Cell (P);
               CStacks.Push (VCGHeap, P, S);
               if IsLeaf (P) then
                  P := Cells.Null_Cell;
               else
                  P := LeftPtr (VCGHeap, P);
               end if;
            end loop;
            exit when CStacks.IsEmpty (S);
            P := CStacks.Top (VCGHeap, S);
            CStacks.Pop (VCGHeap, S);
            if IsLeaf (P) then
               Change := False;
               VarSym := Cells.Get_Symbol_Value (VCGHeap, P);
               if Cells.Get_Kind (VCGHeap, P) = Cells.Reference then
                  SubstituteImport (VarSym,
                                    -- to get
                                    Change, Subs);
               elsif Cells.Get_Kind (VCGHeap, P) = Cells.Unconstrained_Attribute_Prefix then
                  SubstituteImportConstraint (VarSym,
                                              -- to get
                                              Change, Subs);
               end if;
               if Change then
                  Cells.Copy_Contents (VCGHeap, Subs, P);
               end if;
               P := Cells.Null_Cell;
            else
               P := RightPtr (VCGHeap, P);
            end if;
         end loop;
      end SubstituteParameters;

      procedure CheckTypeOfActualImportParams
      --# global in     Dictionary.Dict;
      --#        in     ProcedureSym;
      --#        in     Scope;
      --#        in     SubstitutionTable;
      --#        in out CheckStack;
      --#        in out ContainsReals;
      --#        in out ShortCircuitStack;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives CheckStack,
      --#         ContainsReals,
      --#         ShortCircuitStack,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    CheckStack,
      --#                                    Dictionary.Dict,
      --#                                    ProcedureSym,
      --#                                    Scope,
      --#                                    ShortCircuitStack,
      --#                                    SubstitutionTable,
      --#                                    VCGHeap;
      is
         ParamElement : Cells.Cell;
         ParamNum     : Natural;
         ActualParam  : Cells.Cell;
         FormalParam  : Dictionary.Symbol;
      begin
         ParamNum     := 1;
         ParamElement := Clists.FirstCell (VCGHeap, SubstitutionTable);

         while not Cells.Is_Null_Cell (ParamElement) loop
            ActualParam := AuxPtr (VCGHeap, ParamElement);
            FormalParam := Dictionary.GetSubprogramParameter (ProcedureSym, ParamNum);
            -- Formal parameters are always defined in the abstract view and cannot be refined
            -- and so we should always consider the abstract view.
            if Dictionary.IsImport (Dictionary.IsAbstract, ProcedureSym, FormalParam) then
               CheckConstraintRunTimeError
                 (Dictionary.GetType (FormalParam),
                  ActualParam,
                  Scope,
                  VCGHeap,
                  ShortCircuitStack,
                  CheckStack,
                  ContainsReals);
            end if;

            ParamElement := Clists.NextCell (VCGHeap, ParamElement);
            ParamNum     := ParamNum + 1;
         end loop;
      end CheckTypeOfActualImportParams;

      --------------------------------------------

   begin -- ModelPreCondition
      if Dictionary.IsInstantiation (ProcedureSym) then
         -- for instantiations we go and get the constraint of the original generic
         GenericOrActualSubprogram := Dictionary.GetGenericOfInstantiation (ProcedureSym);
         ConstraintNode            :=
           STree.RefToNode (Dictionary.GetPrecondition (Dictionary.IsAbstract, GenericOrActualSubprogram));

         if ConstraintNode /= STree.NullNode then
            BuildAnnotationExpnDAG
              (ConstraintNode,
               Dictionary.LocalScope (GenericOrActualSubprogram),
               -- Ensure we use the right view of the precondition determined by the value of Constraint_View
               Constraint_View = Dictionary.IsAbstract,
               LoopStack,
               VCGHeap,
               -- to get
               DAGCell);
            -- then replace all generic formal symbols with their instantiated equivalent
            InstantiateParameters (DAGCell, ProcedureSym);
            SubstituteParameters (DAGCell);
            StackCheckStatement (DAGCell, VCGHeap, CheckStack);
            KindOfStackedCheck := Graph.Precon_Check;
         end if;

      else -- not generic instantiation so use the constraint node directly
         GenericOrActualSubprogram := ProcedureSym;
         ConstraintNode            := PreConNode;
         if ConstraintNode /= STree.NullNode then
            BuildAnnotationExpnDAG
              (ConstraintNode,
               Dictionary.LocalScope (GenericOrActualSubprogram),
               -- Ensure we use the right view of the precondition determined by the value of Constraint_View
               Constraint_View = Dictionary.IsAbstract,
               LoopStack,
               VCGHeap,
               -- to get
               DAGCell);
            SubstituteParameters (DAGCell);
            StackCheckStatement (DAGCell, VCGHeap, CheckStack);
            KindOfStackedCheck := Graph.Precon_Check;
         end if;
      end if;
      CheckTypeOfActualImportParams;
   end ModelPrecondition;

   --------------------------------------------------------------------------

   -- Constraint_View gives the view of the postcondition to be used and Data_View gives the view of
   -- the Imports and Exports to be used
   procedure ModelPostcondition (Constraint_View, Data_View : in Dictionary.Abstractions)
   --# global in     CommandLineData.Content;
   --#        in     LoopStack;
   --#        in     OutputFile;
   --#        in     PostConNode;
   --#        in     PrefixSym;
   --#        in     ProcedureSym;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in     SubprogramCalls;
   --#        in     SubstitutionTable;
   --#        in out ContainsReals;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGFailure;
   --#        in out VCGHeap;
   --# derives ContainsReals,
   --#         Dictionary.Dict,
   --#         VCGFailure                 from *,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Data_View,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PostConNode,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         VCGHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PostConNode,
   --#                                         ProcedureSym,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         VCGHeap &
   --#         Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap                    from CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Data_View,
   --#                                         Dictionary.Dict,
   --#                                         Graph.Table,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PostConNode,
   --#                                         PrefixSym,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         StmtStack.S,
   --#                                         STree.Table,
   --#                                         SubprogramCalls,
   --#                                         SubstitutionTable,
   --#                                         VCGHeap &
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Data_View,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         PostConNode,
   --#                                         PrefixSym,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         SubprogramCalls,
   --#                                         SubstitutionTable,
   --#                                         VCGHeap &
   --#         SPARK_IO.File_Sys          from *,
   --#                                         CommandLineData.Content,
   --#                                         Constraint_View,
   --#                                         Data_View,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         LoopStack,
   --#                                         OutputFile,
   --#                                         PostConNode,
   --#                                         ProcedureSym,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         VCGHeap;
   is
      DAGRoot, StmtCell         : Cells.Cell;
      StmtLabel                 : Labels.Label;
      GenericOrActualSubprogram : Dictionary.Symbol;
      ConstraintNode            : STree.SyntaxNode;

      --------------------------------------------

      -- ConstraintRoot gives the root node of the postcondition.
      -- Data_View gives the view to be used for imports and exports of the procedure.
      procedure SubstituteParameters (ConstraintRoot : in out Cells.Cell;
                                      Data_View      : in     Dictionary.Abstractions)
      -- replace formal parameters by actual ones in a subprogram constraint;
      --# global in     Dictionary.Dict;
      --#        in     PrefixSym;
      --#        in     ProcedureSym;
      --#        in     Scope;
      --#        in     SubprogramCalls;
      --#        in     SubstitutionTable;
      --#        in out LexTokenManager.State;
      --#        in out Statistics.TableUsage;
      --#        in out VCGHeap;
      --# derives ConstraintRoot,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         VCGHeap               from *,
      --#                                    ConstraintRoot,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    LexTokenManager.State,
      --#                                    PrefixSym,
      --#                                    ProcedureSym,
      --#                                    Scope,
      --#                                    SubprogramCalls,
      --#                                    SubstitutionTable,
      --#                                    VCGHeap;
      is
         Subs, P  : Cells.Cell;
         S        : CStacks.Stack;
         VarSym   : Dictionary.Symbol;
         Tilded   : Boolean;
         Change   : Boolean;
         HypStack : CStacks.Stack;

         -- procedure, generates hypothesis that all export record
         -- fields are unchanged other than that appearing as exported actual
         -- parameter.
         -- Data_View determines the view of the imports and exports of the procedure.
         procedure AddInRecordFieldAndArrayElementPreservation (Data_View : in Dictionary.Abstractions)
         --# global in     Dictionary.Dict;
         --#        in     PrefixSym;
         --#        in     ProcedureSym;
         --#        in     Scope;
         --#        in     SubprogramCalls;
         --#        in     SubstitutionTable;
         --#        in out HypStack;
         --#        in out LexTokenManager.State;
         --#        in out Statistics.TableUsage;
         --#        in out VCGHeap;
         --# derives HypStack,
         --#         LexTokenManager.State,
         --#         Statistics.TableUsage,
         --#         VCGHeap               from *,
         --#                                    Data_View,
         --#                                    Dictionary.Dict,
         --#                                    HypStack,
         --#                                    LexTokenManager.State,
         --#                                    ProcedureSym,
         --#                                    Scope,
         --#                                    SubprogramCalls,
         --#                                    SubstitutionTable,
         --#                                    VCGHeap &
         --#         null                  from PrefixSym;
         is
            It                   : Dictionary.Iterator;
            ExportSym, UnusedSym : Dictionary.Symbol;
            TempCell             : Cells.Cell;
            OpCell               : Cells.Cell;
            ConvertedExportDAG   : Cells.Cell;
            ExportDAG            : Cells.Cell;
            PtrToExportDAG       : Cells.Cell;
            Ptr                  : Cells.Cell;
            EntireExport         : Cells.Cell;
            UpdateCell           : Cells.Cell;
            S                    : CStacks.Stack;

            procedure ConvertExportDAG
              (ExportDAG          : in     Cells.Cell;
               ConvertedExportDAG :    out Cells.Cell;
               EntireExport       :    out Cells.Cell)
            --# global in     SubprogramCalls;
            --#        in out LexTokenManager.State;
            --#        in out Statistics.TableUsage;
            --#        in out VCGHeap;
            --# derives ConvertedExportDAG,
            --#         EntireExport          from ExportDAG,
            --#                                    VCGHeap &
            --#         LexTokenManager.State from *,
            --#                                    SubprogramCalls &
            --#         Statistics.TableUsage from *,
            --#                                    ExportDAG,
            --#                                    VCGHeap &
            --#         VCGHeap               from *,
            --#                                    ExportDAG,
            --#                                    LexTokenManager.State,
            --#                                    SubprogramCalls;
            is
               TempCell, ConvertedExportDAGLocal : Cells.Cell;
               CountStr                          : LexTokenManager.Lex_String;
            begin -- ConvertExportDAG
               Structures.CopyStructure (VCGHeap, ExportDAG, ConvertedExportDAGLocal);
               TempCell := ConvertedExportDAGLocal;
               while IsSelector (TempCell) loop
                  TempCell := ArgumentOfSelector (TempCell);
               end loop;
               -- TempCell now points at entire variable cell
               Cells.Set_Kind (VCGHeap, TempCell, Cells.Procedure_Export);
               LexTokenManager.Insert_Nat (N       => SubprogramCalls,
                                           Lex_Str => CountStr);
               Cells.Set_Lex_Str (VCGHeap, TempCell, CountStr);
               -- TempCell now points at converted entire export
               EntireExport       := TempCell;
               ConvertedExportDAG := ConvertedExportDAGLocal;
            end ConvertExportDAG;

            -------------------

            procedure AddInSiblingHypotheses (ExportDAG, ConvertedExportDAG : in Cells.Cell)
            --# global in     Dictionary.Dict;
            --#        in     Scope;
            --#        in out HypStack;
            --#        in out Statistics.TableUsage;
            --#        in out VCGHeap;
            --# derives HypStack,
            --#         Statistics.TableUsage,
            --#         VCGHeap               from *,
            --#                                    ConvertedExportDAG,
            --#                                    Dictionary.Dict,
            --#                                    ExportDAG,
            --#                                    HypStack,
            --#                                    Scope,
            --#                                    VCGHeap;
            is
               ExportDAGLocal, ConvertedExportDAGLocal : Cells.Cell;

               procedure DoOneLevelOfSiblings (ExportDAG, ConvertedExportDAG : in Cells.Cell)
               --# global in     Dictionary.Dict;
               --#        in     Scope;
               --#        in out HypStack;
               --#        in out Statistics.TableUsage;
               --#        in out VCGHeap;
               --# derives HypStack,
               --#         Statistics.TableUsage,
               --#         VCGHeap               from *,
               --#                                    ConvertedExportDAG,
               --#                                    Dictionary.Dict,
               --#                                    ExportDAG,
               --#                                    HypStack,
               --#                                    Scope,
               --#                                    VCGHeap;
               is
                  It                      : Dictionary.Iterator;
                  Sym, SymOfExportedField : Dictionary.Symbol;
                  AndCell                 : Cells.Cell;
                  OpCell                  : Cells.Cell;
                  TempCell                : Cells.Cell;
                  ExportDAGCopy           : Cells.Cell;
                  LocalExportDAG          : Cells.Cell;
                  ConvertedExportDAGCopy  : Cells.Cell;
                  CurrentRecord           : Dictionary.Symbol;

               begin -- DoOneLevelOfSiblings
                     -- Debug.PrintMsg ("Entering DoOneLevelOfSiblings", True);
                  CurrentRecord := Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap, RightPtr (VCGHeap, ExportDAG)));
                  -- In the array of record case, once we have passed the indexing expression, we
                  -- no longer have symbols for the record fields.  This is indicated at this point
                  -- by CurrentRecord being a NullSymbol.  In that case we can't produce any useful
                  -- hypotheses and we skip the whole remaining body
                  -- Debug.PrintSym ("Current record = ", CurrentRecord);
                  if CurrentRecord /= Dictionary.NullSymbol then
                     -- suppress unchnaged field hypotheses for record fields that are
                     -- private here
                     if not Dictionary.IsPrivateType (CurrentRecord, Scope) then
                        -- or else IsPredefinedTimeType not needed here as
                        -- definitely a record -- CFR 1743
                        -- Debug.PrintMsg ("Not private", True);
                        -- the symbol of the field being exported can be found at the top
                        -- of the export DAG
                        SymOfExportedField := Cells.Get_Symbol_Value (VCGHeap, ExportDAG);
                        -- Debug.PrintSym ("SymOfExportedField = ", SymOfExportedField);
                        -- make local copy of ExportDAG and convert its rightmostr cell to a RefCell
                        Structures.CopyStructure (VCGHeap, ExportDAG, LocalExportDAG);
                        TempCell := LocalExportDAG;
                        while IsSelector (TempCell) loop
                           TempCell := ArgumentOfSelector (TempCell);
                        end loop;
                        -- TempCell now points at entire variable cell
                        Cells.Set_Kind (VCGHeap, TempCell, Cells.Reference);

                        -- iterate through all sibling field of that being exported, type
                        -- of siblings is found one cell down from top of export dag
                        --It := Dictionary.FirstRecordComponent
                        --   (Dictionary.GetType (Cells.Get_Symbol_Value (VCGHeap,
                        --                                        RightPtr (ExportDAG))));
                        It := Dictionary.FirstRecordComponent (CurrentRecord);
                        while not Dictionary.IsNullIterator (It) loop
                           Sym := Dictionary.CurrentSymbol (It);
                           -- Debug.PrintSym ("Considering Sym in loop ", Sym);
                           if Sym /= SymOfExportedField then
                              -- Debug.PrintMsg ("Adding hypothesis", True);
                              -- need to build an "unchanged" hypothesis for this case
                              Structures.CopyStructure (VCGHeap, LocalExportDAG, ExportDAGCopy);
                              Structures.CopyStructure (VCGHeap, ConvertedExportDAG, ConvertedExportDAGCopy);
                              -- substitute current field into top of each DAG
                              Cells.Set_Symbol_Value (VCGHeap, ExportDAGCopy, Sym);
                              Cells.Set_Lex_Str (VCGHeap, ExportDAGCopy, Dictionary.GetSimpleName (Sym));
                              Cells.Set_Symbol_Value (VCGHeap, ConvertedExportDAGCopy, Sym);
                              Cells.Set_Lex_Str (VCGHeap, ConvertedExportDAGCopy, Dictionary.GetSimpleName (Sym));
                              -- create an equals operator to asert them equal
                              CreateOpCell (OpCell, VCGHeap, SP_Symbols.equals);
                              -- assert equality
                              SetRightArgument (OpCell, ExportDAGCopy, VCGHeap);
                              SetLeftArgument (OpCell, ConvertedExportDAGCopy, VCGHeap);

                              --  CFR 1744: The HypStack may only contain a null
                              --  cell if this is the first term of the hypothesis.
                              if Cells.Is_Null_Cell (CStacks.Top (VCGHeap, HypStack)) then
                                 -- Debug.PrintMsg ("Is a null cell", True);
                                 -- Start of a new hypothesis
                                 CStacks.Push (VCGHeap, OpCell, HypStack);
                              else
                                 -- Debug.PrintMsg ("Not a null cell", True);
                                 -- Not the start of the hypothesis:
                                 -- 'and' result on to hypothesis stack
                                 CreateOpCell (AndCell, VCGHeap, SP_Symbols.RWand);
                                 SetRightArgument (AndCell, OpCell, VCGHeap);
                                 SetLeftArgument (AndCell, CStacks.Top (VCGHeap, HypStack), VCGHeap);
                                 CStacks.Pop (VCGHeap, HypStack);
                                 CStacks.Push (VCGHeap, AndCell, HypStack);
                              end if;
                           end if;

                           It := Dictionary.NextSymbol (It);
                        end loop;
                     end if;
                  end if;
               end DoOneLevelOfSiblings;

            begin -- AddInSiblingHypotheses
               ExportDAGLocal          := ExportDAG;
               ConvertedExportDAGLocal := ConvertedExportDAG;
               while IsSelector (ExportDAGLocal) loop
                  if Cells.Get_Kind (VCGHeap, ExportDAGLocal) = Cells.Field_Access_Function then
                     DoOneLevelOfSiblings (ExportDAGLocal, ConvertedExportDAGLocal);
                  end if;
                  ExportDAGLocal          := ArgumentOfSelector (ExportDAGLocal);
                  ConvertedExportDAGLocal := ArgumentOfSelector (ConvertedExportDAGLocal);
               end loop;
            end AddInSiblingHypotheses;

            -------------------

         begin -- AddInRecordFieldAndArrayElementPreservation
            CStacks.CreateStack (S);
            It := Dictionary.FirstExport (Data_View, ProcedureSym);
            while not Dictionary.IsNullIterator (It) loop
               ExportSym := Dictionary.CurrentSymbol (It);
               -- Debug.PrintSym ("ExportSym = ", ExportSym);
               -- Debug.PrintScope ("Scope = ", Scope);
               --# accept F, 10, UnusedSym, "UnusedSym unused here";
               GetExportDetails (ExportSym,
                                 -- to get
                                 UnusedSym, PtrToExportDAG);
               --# end accept;

               if IsSelector (PtrToExportDAG) then
                  -- Debug.PrintMsg ("it's a record field", True);
                  -- a record field is exported so extra hypothesis is needed
                  Structures.CopyStructure (VCGHeap, PtrToExportDAG, ExportDAG);
                  ConvertExportDAG (PtrToExportDAG,
                                    -- to get
                                    ConvertedExportDAG, EntireExport);

                  -- create hypotheses that each sibling of exported field is
                  -- unchanged
                  AddInSiblingHypotheses (ExportDAG, ConvertedExportDAG);

                  -- now continue to produce "catch-all" hypothesis that all
                  -- unchanged record fields (not just sibling fields) are
                  -- unchanged.

                  -- we need to find bottom right of export DAG and convert cell
                  -- there to a RefCell so that it is preserved as initial value
                  TempCell := ExportDAG;
                  while IsSelector (TempCell) loop
                     TempCell := ArgumentOfSelector (TempCell);
                  end loop;
                  -- TempCell now points at entire variable cell
                  Cells.Set_Kind (VCGHeap, TempCell, Cells.Reference);

                  -- using ingredients ExportDAG, ConvertedExportDAG and EntireExport
                  -- we can now build the hypothesis described in S.P0468.53.27
                  CStacks.Push (VCGHeap, ConvertedExportDAG, S);

                  Ptr := ExportDAG;
                  loop
                     -- Need decision here on record or array handling.
                     -- We know we are looping over either FldFunctions or ElementFunctions
                     if Cells.Get_Kind (VCGHeap, Ptr) = Cells.Field_Access_Function then
                        -- handling record field
                        -- create upf_ function cell
                        CreateUpfCell
                          (UpdateCell,
                           VCGHeap,
                           Cells.Get_Symbol_Value (VCGHeap, Ptr),
                           Cells.Get_Lex_Str (VCGHeap, Ptr));
                     else
                        -- handling an array element
                        -- create update function cell
                        CreateCellKind (UpdateCell, VCGHeap, Cells.Update_Function);
                     end if;

                     -- create comma cell
                     CreateOpCell (OpCell, VCGHeap, SP_Symbols.comma);
                     -- and link it to arguments
                     SetRightArgument (OpCell, CStacks.Top (VCGHeap, S), VCGHeap);
                     SetLeftArgument (OpCell, RightPtr (VCGHeap, Ptr), VCGHeap);

                     -- now link update cell to arguments
                     SetRightArgument (UpdateCell, OpCell, VCGHeap);

                     -- remove old expression from stack and replace by new one
                     CStacks.Pop (VCGHeap, S);
                     CStacks.Push (VCGHeap, UpdateCell, S);

                     -- move down unconverted exportDAG
                     Ptr := ArgumentOfSelector (Ptr);

                     exit when not IsSelector (Ptr);
                  end loop;

                  -- TOS is the RHS of the desired hypothesis, we must set it equal
                  -- to converted entire export
                  CreateOpCell (OpCell, VCGHeap, SP_Symbols.equals);
                  SetRightArgument (OpCell, CStacks.Top (VCGHeap, S), VCGHeap);
                  SetLeftArgument (OpCell, EntireExport, VCGHeap);
                  -- remove old expression from stack and replace by new one
                  CStacks.Pop (VCGHeap, S);
                  CStacks.Push (VCGHeap, OpCell, S);

                  -- TOS is the desired new hypothesis, needs anding into current one
                  CreateOpCell (OpCell, VCGHeap, SP_Symbols.RWand);
                  SetRightArgument (OpCell, CStacks.Top (VCGHeap, S), VCGHeap);
                  SetLeftArgument (OpCell, CStacks.Top (VCGHeap, HypStack), VCGHeap);
                  CStacks.Pop (VCGHeap, HypStack);
                  CStacks.Push (VCGHeap, OpCell, HypStack);

                  -- clean up local stack
                  CStacks.Pop (VCGHeap, S);

               end if;
               It := Dictionary.NextSymbol (It);
            end loop;
            --# accept F, 35, PrefixSym, "Coupled to UnusedSym only" &
            --#        F, 33, UnusedSym, "UnusedSym unused here";
         end AddInRecordFieldAndArrayElementPreservation;

         -----------------

         function IsLeaf (Node : Cells.Cell) return Boolean
         --# global in VCGHeap;
         is
         begin
            return Cells.Is_Null_Cell (RightPtr (VCGHeap, Node));
         end IsLeaf;

         -----------------

         procedure SubstituteTildedParameter (TheCell : in Cells.Cell)
         --# global in     Dictionary.Dict;
         --#        in     PrefixSym;
         --#        in     ProcedureSym;
         --#        in     SubstitutionTable;
         --#        in out VCGHeap;
         --# derives VCGHeap from *,
         --#                      Dictionary.Dict,
         --#                      PrefixSym,
         --#                      ProcedureSym,
         --#                      SubstitutionTable,
         --#                      TheCell;
         is
            ExportDAG : Cells.Cell;
            ExportSym : Dictionary.Symbol;

         begin -- SubstituteTildedParameter
            GetExportDetails (Cells.Get_Symbol_Value (VCGHeap, TheCell), ExportSym, ExportDAG);
            -- see if the DAG represents a field access or array element
            if not Cells.Is_Null_Cell (ExportDAG)
              and then (Cells.Get_Kind (VCGHeap, ExportDAG) = Cells.Field_Access_Function
                          or else Cells.Get_Kind (VCGHeap, ExportDAG) = Cells.Element_Function) then
               -- there is a field access, substitute DAG for cell contents
               Cells.Copy_Contents (VCGHeap, ExportDAG, TheCell);
            else -- no field access, just substitute symbol of actual
               Cells.Set_Symbol_Value (VCGHeap, TheCell, ExportSym);
            end if;
         end SubstituteTildedParameter;

         -----------------

      begin -- SubstituteParameters

         -- HypStack is used to ease joining of extra hypotheses concerning
         -- exportation of record fields to the main hypothesis where the called
         -- procedure's post-condition is assumed.

         CStacks.CreateStack (HypStack);
         CStacks.Push (VCGHeap, ConstraintRoot, HypStack);

         -- stack S is used for the
         -- DAG traversal algorithm of D.E. Knuth, Fundamental Algorithms, p.317;
         CStacks.CreateStack (S);
         P := ConstraintRoot;
         loop
            loop
               exit when Cells.Is_Null_Cell (P);
               CStacks.Push (VCGHeap, P, S);
               if IsLeaf (P) then
                  P := Cells.Null_Cell;
               else
                  P := LeftPtr (VCGHeap, P);
               end if;
            end loop;
            exit when CStacks.IsEmpty (S);
            P := CStacks.Top (VCGHeap, S);
            CStacks.Pop (VCGHeap, S);
            if IsLeaf (P) then

               if Cells.Get_Kind (VCGHeap, P) = Cells.Reference
                 or else Cells.Get_Kind (VCGHeap, P) = Cells.Unconstrained_Attribute_Prefix then

                  VarSym := Cells.Get_Symbol_Value (VCGHeap, P);

                  Tilded := Cells.Get_Op_Symbol (VCGHeap, P) = SP_Symbols.tilde;

                  if Dictionary.IsExport (Data_View, ProcedureSym, VarSym) then

                     if Cells.Get_Kind (VCGHeap, P) = Cells.Unconstrained_Attribute_Prefix then
                        -- Although VarSym is an export, it is used here as an attribute prefix
                        -- and we don't want to convert it to an export cell or do anything to
                        -- the tilde, rather we want to replace the variable name with a
                        -- constraining type mark
                        SubstituteImportConstraint (VarSym,
                                                    -- to get
                                                    Change, Subs);
                        if Change then
                           Cells.Copy_Contents (VCGHeap, Subs, P);
                        end if;
                     else
                        -- not an unconstrained attribute prefix - handle as before
                        if Tilded then
                           if Dictionary.IsGlobalVariable (Data_View, ProcedureSym, VarSym) then
                              Cells.Set_Op_Symbol (VCGHeap, P, SP_Symbols.RWnull);
                           else   -- must be a parameter
                              Cells.Set_Op_Symbol (VCGHeap, P, SP_Symbols.RWnull);
                              SubstituteTildedParameter (P);
                           end if;
                        else -- not Tilded, but still an export
                           ConvertCellToExportCell (P);
                        end if;
                     end if;
                  else -- process imports

                     --  In a post-condition, a reference to an
                     --  attribute of an unconstrained array parameter
                     --  appears as am imported
                     --  SubprogramParameterConstraint here here, so
                     if Dictionary.IsSubprogramParameter (VarSym) or Dictionary.IsSubprogramParameterConstraint (VarSym) then

                        if Cells.Get_Kind (VCGHeap, P) = Cells.Reference then
                           SubstituteImport (VarSym,
                                             -- to get
                                             Change, Subs);
                           if Change then
                              Cells.Copy_Contents (VCGHeap, Subs, P);
                           end if;
                        elsif Cells.Get_Kind (VCGHeap, P) = Cells.Unconstrained_Attribute_Prefix then
                           SubstituteImportConstraint (VarSym,
                                                       -- to get
                                                       Change, Subs);
                           if Change then
                              Cells.Copy_Contents (VCGHeap, Subs, P);
                           end if;
                        end if;
                     end if;
                  end if;
               end if;
               P := Cells.Null_Cell;
            else
               P := RightPtr (VCGHeap, P);
            end if;
         end loop;

         -- where record fields or array elements are exported we need to create a hypothis
         -- in terms of the entire export and, for records, assert that all
         -- unaffected fields are unchanged.
         AddInRecordFieldAndArrayElementPreservation (Data_View => Data_View);

         --# accept F, 10, HypStack, "HypStack unused here";
         CStacks.PopOff (VCGHeap, HypStack, ConstraintRoot);
         --# end accept;

      end SubstituteParameters;

      -- Data_View gives the view of the imports and exports of the procedure to be used.
      procedure AssumeTypeOfActualExports (DAGRoot   : in out Cells.Cell;
                                           Data_View : in     Dictionary.Abstractions)
      --# global in     CommandLineData.Content;
      --#        in     OutputFile;
      --#        in     ProcedureSym;
      --#        in     Scope;
      --#        in out ContainsReals;
      --#        in out Dictionary.Dict;
      --#        in out LexTokenManager.State;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --#        in out VCGFailure;
      --#        in out VCGHeap;
      --# derives ContainsReals,
      --#         DAGRoot,
      --#         Dictionary.Dict,
      --#         LexTokenManager.State,
      --#         Statistics.TableUsage,
      --#         VCGFailure,
      --#         VCGHeap               from *,
      --#                                    CommandLineData.Content,
      --#                                    DAGRoot,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    LexTokenManager.State,
      --#                                    ProcedureSym,
      --#                                    Scope,
      --#                                    VCGHeap &
      --#         SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    DAGRoot,
      --#                                    Data_View,
      --#                                    Dictionary.Dict,
      --#                                    LexTokenManager.State,
      --#                                    OutputFile,
      --#                                    ProcedureSym,
      --#                                    Scope,
      --#                                    VCGHeap;
      is
         It        : Dictionary.Iterator;
         ExportVar : Dictionary.Symbol;
         Type_Sym  : Dictionary.Symbol;
      begin
         -- Debug.PrintMsg ("Into AssumeTypeOfActualExports", True);
         It := Dictionary.FirstExport (Data_View, ProcedureSym);
         while not Dictionary.IsNullIterator (It) loop
            -- Debug.PrintMsg ("Looping...", True);
            ExportVar := Dictionary.CurrentSymbol (It);
            -- Debug.PrintSym ("Exported Var: ", ExportVar);
            -- Debug.PrintScope ("Scope: ", Scope);
            if IsDirectlyVisible (ExportVar, Scope) and then not Dictionary.IsOwnVariableOrConstituentWithMode (ExportVar) then
               -- Debug.PrintMsg ("Is directly visible", True);
               Type_Sym := Dictionary.GetType (ExportVar);
               if not Dictionary.IsPrivateType (Type_Sym, Scope) or else Dictionary.IsPredefinedTimeType (Type_Sym) then
                  -- Debug.PrintMsg ("Is not a private type", True);
                  ConjoinParamConstraint (Type_Sym, ExportVar, Dictionary.NullSymbol, DAGRoot);
               end if;
            end if;
            It := Dictionary.NextSymbol (It);
         end loop;
      end AssumeTypeOfActualExports;

      --------------------------------------------

   begin -- ModelPostcondition
         -- Debug.PrintMsg ("Into ModelPostcondition", True);
      if Dictionary.IsInstantiation (ProcedureSym) then
         -- for instantiations we go and get the constraint of the original generic
         GenericOrActualSubprogram := Dictionary.GetGenericOfInstantiation (ProcedureSym);
         ConstraintNode            :=
           STree.RefToNode (Dictionary.GetPostcondition (Dictionary.IsAbstract, GenericOrActualSubprogram));

         if ConstraintNode /= STree.NullNode then
            BuildAnnotationExpnDAG
              (ConstraintNode,
               Dictionary.LocalScope (GenericOrActualSubprogram),
               -- Ensure the correct view of the postcondition is used determined by the value of Constraint_View
               Constraint_View = Dictionary.IsAbstract,
               LoopStack,
               VCGHeap,
               -- to get
               DAGRoot);

            InstantiateParameters (DAGRoot, ProcedureSym);
         else
            DAGRoot := Cells.Null_Cell;
         end if;

      else -- not generic instantiation so use the constraint node directly
         GenericOrActualSubprogram := ProcedureSym;
         ConstraintNode            := PostConNode;
         if ConstraintNode /= STree.NullNode then
            BuildAnnotationExpnDAG
              (ConstraintNode,
               Dictionary.LocalScope (GenericOrActualSubprogram),
               -- Ensure the correct view of the postcondition is used determined by the value of Constraint_View
               Constraint_View = Dictionary.IsAbstract,
               LoopStack,
               VCGHeap,
               -- to get
               DAGRoot);
         else
            DAGRoot := Cells.Null_Cell;
         end if;
      end if;

      -- Debug.PrintDAG ("Postcondition DAG (before AssumeType) is: ", DAGRoot, VCGHeap, Scope);

      -- Ensure the correct view of the exports is used determined by the value of Data_View
      AssumeTypeOfActualExports (DAGRoot   => DAGRoot,
                                 Data_View => Data_View);

      -- Debug.PrintDAG ("Postcondition DAG (after  AssumeType) is: ", DAGRoot, VCGHeap, Scope);

      -- CFR 1744: Call SubstituteParameters even if DAGRoot is a null cell
      -- in order to generate hypothesis for invariant fields of composite
      -- objects where an exported variable of a private type is a component
      -- of the composite object.
      -- Ensure the correct view of imports and exports is used determined by the value of Data_View
      SubstituteParameters (ConstraintRoot => DAGRoot,
                            Data_View      => Data_View);

      -- Debug.PrintDAG ("Postcondition DAG (after SubstituteParameters) is: ", DAGRoot, VCGHeap, Scope);

      PrepareLabel (VCGHeap, StmtLabel, StmtCell);
      SetRightArgument (StmtCell, DAGRoot, VCGHeap);
      Chain (StmtLabel, VCGHeap);

   end ModelPostcondition;

   procedure CheckTypeOfActualExportParams
   --# global in     Dictionary.Dict;
   --#        in     ProcedureSym;
   --#        in     Scope;
   --#        in     SubstitutionTable;
   --#        in out CheckStack;
   --#        in out ContainsReals;
   --#        in out ShortCircuitStack;
   --#        in out Statistics.TableUsage;
   --#        in out VCGHeap;
   --# derives CheckStack,
   --#         ContainsReals,
   --#         ShortCircuitStack,
   --#         Statistics.TableUsage,
   --#         VCGHeap               from *,
   --#                                    CheckStack,
   --#                                    Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    Scope,
   --#                                    ShortCircuitStack,
   --#                                    SubstitutionTable,
   --#                                    VCGHeap;
   is
      ParamElement    : Cells.Cell;
      ParamNum        : Natural;
      ActualParamCell : Cells.Cell;
      ActualParamSym  : Dictionary.Symbol;
      ActualParamType : Dictionary.Symbol;
      FormalParamSym  : Dictionary.Symbol;

      -- This subprogram is added merely as a temporary measure
      -- to flag up presence of additional VCs being generated
      -- as a result of the bug under SEPR 2124.
      -- To be removed in a future release of the Examiner.
      procedure SEPR2124Warning
      --# derives ;
      is
         --# hide SEPR2124Warning;
      begin
         -- Only output warning in case of different types in use
         if Dictionary.GetType (FormalParamSym) /= ActualParamType then
            ErrorHandler.Semantic_Warning
              (Err_Num  => 420,
               Position => LexTokenManager.Token_Position'(Start_Line_No => LexTokenManager.Line_Numbers (LineNmbr),
                                                           Start_Pos     => 0),
               Id_Str   => LexTokenManager.Null_String);
         end if;
      end SEPR2124Warning;

   begin
      ParamNum     := 1;
      ParamElement := Clists.FirstCell (VCGHeap, SubstitutionTable);

      while not Cells.Is_Null_Cell (ParamElement) loop
         FormalParamSym  := Dictionary.GetSubprogramParameter (ProcedureSym, ParamNum);
         ActualParamCell := AuxPtr (VCGHeap, ParamElement);

         -- Subprogram parameters cannot be refined they are always abstract
         if Dictionary.IsExport (Dictionary.IsAbstract, ProcedureSym, FormalParamSym) then
            --            Debug.PrintMsg  ("Formal param is an export", True);

            -- We need to work out the subtype of the actual parameter
            -- here for the case where the actual is a subtype of the
            -- formal parameter, and so requires a range check.
            if Cells.Get_Kind (VCGHeap, ActualParamCell) = Cells.Field_Access_Function then

               ActualParamSym  := Cells.Get_Symbol_Value (VCGHeap, ActualParamCell);
               ActualParamType := Dictionary.GetType (ActualParamSym);

               SEPR2124Warning;

            elsif Cells.Get_Kind (VCGHeap, ActualParamCell) = Cells.Element_Function then

               -- BuildExpnDAG.ProcessNameArgumentList will have planted
               -- the type of the array component here as the SymValue of
               -- the ActualParamCell, so
               ActualParamType := Cells.Get_Symbol_Value (VCGHeap, ActualParamCell);

               SEPR2124Warning;

            else -- not an array element or record field selector, so must be a simple variable
               ActualParamSym  := Cells.Get_Symbol_Value (VCGHeap, ActualParamCell);
               ActualParamType := Dictionary.GetType (ActualParamSym);

            end if;

            --            Debug.PrintSym  ("Formal param is      ", FormalParamSym);
            --            Debug.PrintSym  ("Formal param type    ", Dictionary.GetType (FormalParamSym));
            --            Debug.PrintDAG  ("Actual param DAG  is ", ActualParamCell, VCGHeap, Scope);
            --            Debug.PrintSym  ("Actual param type is ", ActualParamType);

            CheckConstraintRunTimeError
              (ActualParamType,
               ActualParamCell,
               Scope,
               VCGHeap,
               ShortCircuitStack,
               CheckStack,
               ContainsReals);

         end if;
         ParamElement := Clists.NextCell (VCGHeap, ParamElement);
         ParamNum     := ParamNum + 1;
      end loop;
   end CheckTypeOfActualExportParams;

   ----------------------------------------------------------------
   -- Data_View gives the view of the imports and exports to be used.
   procedure BuildVCSAssignments (Data_View : in Dictionary.Abstractions)
   --# global in     Dictionary.Dict;
   --#        in     PrefixSym;
   --#        in     ProcedureSym;
   --#        in     SubprogramCalls;
   --#        in     SubstitutionTable;
   --#        in out Graph.Table;
   --#        in out LexTokenManager.State;
   --#        in out Statistics.TableUsage;
   --#        in out StmtStack.S;
   --#        in out VCGHeap;
   --# derives Graph.Table,
   --#         StmtStack.S,
   --#         VCGHeap               from Data_View,
   --#                                    Dictionary.Dict,
   --#                                    Graph.Table,
   --#                                    LexTokenManager.State,
   --#                                    PrefixSym,
   --#                                    ProcedureSym,
   --#                                    StmtStack.S,
   --#                                    SubprogramCalls,
   --#                                    SubstitutionTable,
   --#                                    VCGHeap &
   --#         LexTokenManager.State from *,
   --#                                    Data_View,
   --#                                    Dictionary.Dict,
   --#                                    ProcedureSym,
   --#                                    SubprogramCalls &
   --#         Statistics.TableUsage from *,
   --#                                    Data_View,
   --#                                    Dictionary.Dict,
   --#                                    LexTokenManager.State,
   --#                                    PrefixSym,
   --#                                    ProcedureSym,
   --#                                    SubprogramCalls,
   --#                                    SubstitutionTable,
   --#                                    VCGHeap;

   is
      It                       : Dictionary.Iterator;
      ExportSym                : Dictionary.Symbol;
      ExportCell, AssignedCell : Cells.Cell;
      ExportsFound             : Boolean;
      ModList                  : Cells.Cell;
      StmtPair                 : Cells.Cell;
      StmtLabel                : Labels.Label;
      EntireExportSym          : Dictionary.Symbol;
      UnusedExportDAG          : Cells.Cell;

   begin
      -- Debug.PrintMsg ("In BuildVCSAssignments", True);
      ExportsFound := False;
      Clists.CreateList (VCGHeap, ModList);
      It := Dictionary.FirstExport (Data_View, ProcedureSym);

      --# accept F, 10, UnusedExportDAG, "UnusedExportDAG unused here" &
      --#        F, 33, UnusedExportDAG, "UnusedExportDAG unused here";
      while not Dictionary.IsNullIterator (It) loop
         ExportSym := Dictionary.CurrentSymbol (It);
         if ExportSym /= Dictionary.GetNullVariable then --don't model data sinks
            ExportsFound := True;
            GetExportDetails (ExportSym,
                              -- to get
                              EntireExportSym, UnusedExportDAG);

            -- Debug.PrintSym ("Exportsym passed to GetExportDetails is ", ExportSym);
            -- Debug.PrintSym ("EntireExportsym returned by GetExportDetails is ", EntireExportSym);

            CreateModifiedCell (ExportCell, VCGHeap, EntireExportSym);
            Clists.InsertCell (VCGHeap, ExportCell, ModList);
            Cells.Create_Cell (VCGHeap, AssignedCell);
            Cells.Set_Symbol_Value (VCGHeap, AssignedCell, EntireExportSym);
            ConvertCellToExportCell (AssignedCell);
            SetRightArgument (ExportCell, AssignedCell, VCGHeap);
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;

      if ExportsFound then
         PrepareLabel (VCGHeap, StmtLabel, StmtPair);
         SetAuxPtr (StmtPair, ModList, VCGHeap);
      else
         CreateUnitLabel (StmtLabel, VCGHeap);
      end if;
      Chain (StmtLabel, VCGHeap);
   end BuildVCSAssignments;

   -----------------------------------------------------------------------

begin -- ModelProcedureCall
      -- Debug.PrintMsg ("Into ModelProcedureCall", True);
   SubprogramCalls := SubprogramCalls + 1;
   -- Node represents procedure_call_statement;

   -- If we are calling an inherited root operation then the proc_call node will
   -- have been seeded with the actual procedure that got called.
   ProcedureSym := STree.NodeSymbol (Node);
   if ProcedureSym /= Dictionary.NullSymbol and then Dictionary.IsVariable (ProcedureSym) then
      -- We have found something in the syntax tree but it is not a subprogram.  In that
      -- case it must be a protected object prefix to a protected op call.  We need this so
      -- we can subtitute instances of PT in annotations with PO.  PO is what we have just found.
      PrefixSym    := ProcedureSym;
      ProcedureSym := Dictionary.NullSymbol;
   end if;

   if ProcedureSym = Dictionary.NullSymbol then
      -- no root proc symbol found so carry on and find a procedure name locally

      BuildExpnDAG
        (OutputFile,
         STree.Child_Node (Current_Node => STree.Child_Node (Current_Node => Node)),
         LScope,
         Scope,
         LineNmbr,
         True, --PNA observation, 8/3/4, why is RTC turned on when all we are doing is identifying proc name?
         False,
         LoopStack,
         FlowHeap,
         VCGHeap,
         ContainsReals,
         VCGFailure,
         ShortCircuitStack,
         CheckStack,
         KindOfStackedCheck,
         -- to get
         ProcedureNameCell);

      ProcedureSym := Cells.Get_Symbol_Value (VCGHeap, ProcedureNameCell);
   end if;

   -- First obtain what the view of the pre and post conditions is -
   -- abstract or refined?
   Constraint_View := Dictionary.GetConstraintAbstraction (ProcedureSym, LScope);

   if Constraint_View = Dictionary.IsRefined then
      -- The view of the pre and post condition is refined.
      -- Determine whether the view of the data should be
      -- Abstract (Proof Refinement only) or
      -- Refined  (Data Refinement is present).
      if Dictionary.IsNullIterator (Dictionary.FirstGlobalVariable (Dictionary.IsRefined, ProcedureSym)) then
         -- No Data Refinement.
         Data_View := Dictionary.IsAbstract;
      else
         -- Data Refinement is present.
         Data_View := Dictionary.IsRefined;
      end if;
   else
      -- Only proof refinement is present.
      Data_View := Dictionary.IsAbstract;
   end if;

   BuildSubstitutionTable (Node);

   -- Get the correct view of the pre and post conditions.
   PreConNode  := STree.RefToNode (Dictionary.GetPrecondition (Constraint_View, ProcedureSym));
   PostConNode := STree.RefToNode (Dictionary.GetPostcondition (Constraint_View, ProcedureSym));

   ModelPrecondition (Constraint_View => Constraint_View);
   UnStackRtcs (LineNmbr, VCGHeap, CheckStack, KindOfStackedCheck);
   ModelPostcondition (Constraint_View => Constraint_View,
                       Data_View       => Data_View);
   BuildVCSAssignments (Data_View => Data_View);

   CheckTypeOfActualExportParams;
   UnStackRtcs (LineNmbr, VCGHeap, CheckStack, KindOfStackedCheck);

   Clists.DisposeOfList (VCGHeap, SubstitutionTable);
end ModelProcedureCall;
