-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .SIV file                                            --
--                                                                            --
--------------------------------------------------------------------------------

with SPARK_XML;

separate (VCS)
procedure AnalyseSimplifiedVCFile
  (Report_File        : in     SPARK_IO.File_Type;
   Filename           : in     E_Strings.T;
   VC_File_Date_Time  : in     E_Strings.T;
   SIV_File_Date_Time :    out E_Strings.T;
   Error_In_SIV_File  :    out Boolean;
   File_Error         :    out E_Strings.T) is

   Bad_File_Format                       : Boolean            := False;
   Dummy_Close_Status                    : SPARK_IO.File_Status;
   File_Line                             : E_Strings.T;
   Finished_With_File                    : Boolean;
   Open_Status                           : SPARK_IO.File_Status;
   Process_Success                       : Boolean;
   Read_Line_Success                     : Boolean;
   Simplified_VC_File                    : SPARK_IO.File_Type := SPARK_IO.Null_File;
   File_Status                           : File_Status_T;
   Simplification_Date_Time              : E_Strings.T;
   VC_Generation_Date_Time_From_SIV_File : E_Strings.T;
   Trimmed_Line                          : E_Strings.T;
   Current_VC_Name                       : E_Strings.T;

   -------------------------------------------------------------------------
   -- NOTE, this procedure also removes the comma inserted in the string
   -- by the simplifier
   --
   procedure Extract_Dates_And_Times_From_Simplified_VC_File
     (Simplified_VC_File       : in     SPARK_IO.File_Type;
      VC_Generation_Date_Time  :    out E_Strings.T;
      Simplification_Date_Time :    out E_Strings.T;
      File_Status              :    out File_Status_T)
   --# global in out SPARK_IO.File_Sys;
   --# derives File_Status,
   --#         Simplification_Date_Time,
   --#         SPARK_IO.File_Sys,
   --#         VC_Generation_Date_Time  from Simplified_VC_File,
   --#                                       SPARK_IO.File_Sys;
   is
      File_Line        : E_Strings.T;
      Trimmed_Line     : E_Strings.T;
      Gen_Date_Time    : E_Strings.T;
      Simp_Date_Time   : E_Strings.T;
      Subprogram_Found : Boolean := False;
   begin
      File_Status              := Not_Corrupt;
      VC_Generation_Date_Time  := E_Strings.Empty_String;
      Simplification_Date_Time := E_Strings.Empty_String;

      --Check for completly empty file.
      E_Strings.Get_Line (File  => Simplified_VC_File,
                          E_Str => File_Line);
      if E_Strings.Is_Empty (E_Str => File_Line) and SPARK_IO.End_Of_File (Simplified_VC_File) then
         File_Status := Corrupt_Empty_File;
      else
         --Keep on reading from this file, until the desired information is retrieved
         --or the end of the fikle is reached.
         loop
            Trimmed_Line := E_Strings.Trim (File_Line);

            -- find date
            -- (There is an implicit assumption that the date, if present, will appear
            --  before the subprogram name.)
            -- When the Examiner is in plain_output mode, the DATE line doesn't appear.
            if E_Strings.Eq1_String (E_Str => E_Strings.Section (Trimmed_Line, 1, 7),
                                     Str   => "CREATED") then
               -- extract the VC generation date and time from the string
               Gen_Date_Time :=
                 E_Strings.Section (Trimmed_Line, SIV_File_VC_Generation_Date_Start_Column, SIV_File_VC_Generation_Date_Length);
               E_Strings.Append_String (E_Str => Gen_Date_Time,
                                        Str   => " ");
               E_Strings.Append_Examiner_String
                 (E_Str1 => Gen_Date_Time,
                  E_Str2 => E_Strings.Section
                    (Trimmed_Line,
                     SIV_File_VC_Generation_Time_Start_Column,
                     SIV_File_VC_Generation_Time_Length));
               VC_Generation_Date_Time := Gen_Date_Time;

               -- extract the simplification date and time from the string
               Simp_Date_Time :=
                 E_Strings.Section (Trimmed_Line, SIV_File_Simplification_Date_Start_Column, SIV_File_Simplification_Date_Length);
               E_Strings.Append_String (E_Str => Simp_Date_Time,
                                        Str   => " ");
               E_Strings.Append_Examiner_String
                 (E_Str1 => Simp_Date_Time,
                  E_Str2 => E_Strings.Section
                    (Trimmed_Line,
                     SIV_File_Simplification_Time_Start_Column,
                     SIV_File_Simplification_Time_Length));

               Simplification_Date_Time := Simp_Date_Time;
            end if;

            Subprogram_Found := Is_Valid_Subprogram (Trimmed_Line);

            exit when (Subprogram_Found or SPARK_IO.End_Of_File (Simplified_VC_File));
            E_Strings.Get_Line (File  => Simplified_VC_File,
                                E_Str => File_Line);
         end loop;
      end if;

      if (File_Status = Not_Corrupt) and not Subprogram_Found then
         File_Status := Corrupt_Unknown_Subprogram;
      end if;

      -- if date has not been found must be in plain output mode
      -- The above is a false assumption -- the file may just be corrupt. However, the
      -- effect below of setting the string as unknown date is reasonable for both cases.
      if E_Strings.Is_Empty (E_Str => VC_Generation_Date_Time) then
         E_Strings.Append_String (E_Str => VC_Generation_Date_Time,
                                  Str   => Unknown_VCG_Date);
         E_Strings.Append_String (E_Str => Simplification_Date_Time,
                                  Str   => Unknown_SIV_Date);
      end if;
   end Extract_Dates_And_Times_From_Simplified_VC_File;

   -------------------------------------------------------------------------
   -- look at the next non-blank line to see whether it starts
   -- "*** true". If so the VC has been discharged. Otherwise, increment
   -- the counter of undischarged VCs, and set the flag that an undischarged
   -- VC has been found
   procedure Process_New_Simplified_VC_Line
     (Simplified_VC_File : in     SPARK_IO.File_Type;
      VC_Name            : in     E_Strings.T;
      Success            :    out Boolean)
   --# global in out FatalErrors.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out VCHeap.State;
   --# derives FatalErrors.State,
   --#         VCHeap.State      from *,
   --#                                Simplified_VC_File,
   --#                                SPARK_IO.File_Sys,
   --#                                VCHeap.State,
   --#                                VC_Name &
   --#         SPARK_IO.File_Sys,
   --#         Success           from Simplified_VC_File,
   --#                                SPARK_IO.File_Sys;
   is
      File_Line         : E_Strings.T;
      Read_Line_Success : Boolean;
   begin
      Read_Next_Non_Blank_Line (File      => Simplified_VC_File,
                                Success   => Read_Line_Success,
                                File_Line => File_Line);
      if not Read_Line_Success then
         Success := False;
      else
         Success := True;
         if E_Strings.Eq1_String (E_Str => E_Strings.Section (File_Line, 1, 8),
                                  Str   => "*** true") then
            if E_Strings.Eq1_String (E_Str => E_Strings.Section (File_Line, 15, 15),
                                     Str   => "* contradiction") then
               VCHeap.Set_VC_State (VC_Name, VCDetails.VC_Proved_By_Contradiction);
            elsif E_Strings.Eq1_String (E_Str => E_Strings.Section (File_Line, 15, 14),
                                        Str   => "* proved using") then
               VCHeap.Set_VC_State (VC_Name, VCDetails.VC_Proved_Using_User_Proof_Rules);
            elsif VCHeap.Get_VC_State (VC_Name) /= VCDetails.VC_Proved_By_Examiner then
               VCHeap.Set_VC_State (VC_Name, VCDetails.VC_Proved_By_Inference);

            end if;
         elsif VCHeap.Get_VC_State (VC_Name) /= VCDetails.VC_Proved_By_Examiner then
            -- The VC is undischarged if it has not been discharged by the
            -- Examiner.
            VCHeap.Set_VC_State (VC_Name, VCDetails.VC_Undischarged);
         end if;
      end if;
   end Process_New_Simplified_VC_Line;

   --------------------------------------------------------------------------

begin -- AnalyseSimplifiedVCFile

   -- open simplified VC file
   E_Strings.Open
     (File         => Simplified_VC_File,
      Mode_Of_File => SPARK_IO.In_File,
      Name_Of_File => Filename,
      Form_Of_File => "",
      Status       => Open_Status);
   if Open_Status /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.Could_Not_Open_Input_File, E_Strings.Empty_String);
   end if;

   --No errors, until discover otherwise.
   Error_In_SIV_File := False;
   File_Error        := E_Strings.Empty_String;

   Extract_Dates_And_Times_From_Simplified_VC_File
     (Simplified_VC_File       => Simplified_VC_File,
      VC_Generation_Date_Time  => VC_Generation_Date_Time_From_SIV_File,
      Simplification_Date_Time => Simplification_Date_Time,
      File_Status              => File_Status);
   case File_Status is
      when Not_Corrupt =>
         null;
      when Corrupt_Empty_File =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "************* SIV file corrupt: empty file ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         File_Error        := SPARK_XML.X_Str (Str => "SIV file corrupt: empty file");
         Error_In_SIV_File := True;
      when Corrupt_Unknown_Subprogram =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "************* SIV file corrupt: missing subprogram name ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         File_Error        := SPARK_XML.X_Str (Str => "SIV file corrupt: missing subprogram name");
         Error_In_SIV_File := True;
   end case;

   --Only continue working on this file if an error has not been seen.
   --(Previously POGS would attempt to work with corrupt files. This feature has the
   -- capacity to produce confusing and wrong results.)
   if not (Error_In_SIV_File) then
      if CommandLine.Data.IgnoreDates
        or else E_Strings.Eq_String (E_Str1 => VC_Generation_Date_Time_From_SIV_File,
                                     E_Str2 => VC_File_Date_Time) then

         if not CommandLine.Data.IgnoreDates and not CommandLine.Data.XML then
            SPARK_IO.New_Line (Report_File, 1);
            SPARK_IO.Put_String (Report_File, "VCs simplified ", 0);
            E_Strings.Put_Line (File  => Report_File,
                                E_Str => Simplification_Date_Time);
         end if;

         -- find first non blank line
         -- if we get to the end of the file first, flag a fatal error
         Read_Next_Non_Blank_Line (File      => Simplified_VC_File,
                                   Success   => Read_Line_Success,
                                   File_Line => File_Line);

         if not Read_Line_Success then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output, "************* SIV file corrupt: no data beyond header ************", 0);
            SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
            Bad_File_Format := True;
            File_Error      := SPARK_XML.X_Str (Str => "SIV file corrupt: no data beyond header");
         else
            Finished_With_File := False;
            File_Error         := E_Strings.Empty_String;

            -- process file line-by-line
            -- on entry to the loop there is already a valid line in the
            -- File_Line buffer
            Current_VC_Name := E_Strings.Empty_String;
            while not Finished_With_File loop
               -- examine line and act accordingly
               if Is_New_Range_Line (Line => File_Line) then
                  Append_Next_Line_From_File (Line => File_Line,
                                              File => Simplified_VC_File);

               elsif Is_New_VC_Line (Line => File_Line) then

                  Trimmed_Line    := E_Strings.Trim (File_Line);
                  Current_VC_Name :=
                    E_Strings.Section
                    (E_Str     => Trimmed_Line,
                     Start_Pos => 1,
                     Length    => E_Strings.Get_Length (E_Str => Trimmed_Line) - 1);

                  Process_New_Simplified_VC_Line
                    (Simplified_VC_File => Simplified_VC_File,
                     VC_Name            => Current_VC_Name,
                     Success            => Process_Success);

                  if not Process_Success then
                     --# accept F, 41, "Expression is stable but cheap";
                     if not CommandLine.Data.XML then
                        SPARK_IO.Put_String (Report_File, "*** Warning: Bad format in simplified VC file ***", 0);
                     end if;
                     --# end accept;
                     Finished_With_File := True;
                     Bad_File_Format    := True;
                     File_Error         := SPARK_XML.X_Str (Str => "Warning: Bad format in simplified VC file");
                  end if;
               end if;

               if not Finished_With_File then
                  -- read next line and check if VC has been proved false
                  Read_Next_Non_Blank_Line (File      => Simplified_VC_File,
                                            Success   => Read_Line_Success,
                                            File_Line => File_Line);
                  if Is_Trivially_False_VC (Line => File_Line) then
                     VCHeap.Set_VC_State (Current_VC_Name, VCDetails.VC_False);
                  end if;
                  -- if unsuccessful then check EOF
                  -- and set Finished_With_File accordingly
                  if not Read_Line_Success then
                     if SPARK_IO.End_Of_File (Simplified_VC_File) then
                        Finished_With_File := True;
                     else
                        FatalErrors.Process (FatalErrors.Problem_Reading_File, E_Strings.Empty_String);
                     end if;
                  end if;
               end if;
            end loop;

         end if;

      else
         -- SIV file is out of date
         if not CommandLine.Data.XML then
            SPARK_IO.New_Line (Report_File, 1);
            SPARK_IO.Put_Line (Report_File, "*** Warning: Simplified VC file out of date ***", 0);
            SPARK_IO.Put_String (Report_File, "VCs Generated: ", 0);
            E_Strings.Put_String (File  => Report_File,
                                  E_Str => VC_File_Date_Time);
            SPARK_IO.New_Line (Report_File, 1);

            SPARK_IO.Put_String (Report_File, "SIV File Date: ", 0);
            E_Strings.Put_String (File  => Report_File,
                                  E_Str => VC_Generation_Date_Time_From_SIV_File);
            SPARK_IO.New_Line (Report_File, 1);
         end if;

         Bad_File_Format := True;
         File_Error      := SPARK_XML.X_Str (Str => "Warning: Simplified VC file out of date");
      end if;
   end if;

   --# accept F, 10, Dummy_Close_Status, "Dummy_Close_Status unused here" &
   --#        F, 10, Simplified_VC_File, "Simplified_VC_File unused here";
   SPARK_IO.Close (Simplified_VC_File, Dummy_Close_Status);
   --# end accept;

   SIV_File_Date_Time := Simplification_Date_Time;
   --Either an error being raised, or a 'Bad_File_Format'
   --being detected is an error.
   Error_In_SIV_File := Error_In_SIV_File or Bad_File_Format;

   --# accept F, 33, Dummy_Close_Status, "Dummy_Close_Status unused here";
end AnalyseSimplifiedVCFile;
