-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with LexTokenStacks;

separate (Dictionary)
procedure GenerateSimpleName (Item      : in     Symbol;
                              Separator : in     String;
                              Name      :    out E_Strings.T) is

   function Fetch_Simple_Name (Item : Symbol) return E_Strings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Item_Local : Symbol;
   begin
      Item_Local := Item;
      if IsType (Item_Local) and then TypeIsAccess (Item_Local) then
         Item_Local := DeReference (Item_Local);
      end if;
      return LexTokenManager.Lex_String_To_String (Lex_Str => GetSimpleName (Item_Local));
   end Fetch_Simple_Name;

   --------------------------------------------------------------------------------

   -- Each protected own variable has an associated implicit in stream which is used for
   -- volatile flow analysis of shared protected state.  The names of these should never
   -- apepar in Examienr output; however, if they are needed for diagnostic reasons they
   -- can be constructed by this function.  For a stream associated with P we return P__in.
   function Get_Implicit_Protected_In_Stream_Name (Item : Symbol) return E_Strings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Name : E_Strings.T;
   begin
      Name := Fetch_Simple_Name (Item => GetOwnVariableOfProtectedImplicitInStream (Item));
      E_Strings.Append_String (E_Str => Name,
                               Str   => "__in");
      return Name;
   end Get_Implicit_Protected_In_Stream_Name;

   --------------------------------------------------------------------------------

   function Get_Loop_Name (The_Loop : Symbol) return E_Strings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      --# hide Get_Loop_Name;

      Name : E_Strings.T;

      function Get_Loop_Number (The_Loop : Symbol) return Positive
      --# global in Dict;
      is
         Loops  : Iterator;
         Number : Positive;
      begin

         Loops  := FirstLoop (GetEnclosingCompilationUnit (LocalScope (The_Loop)));
         Number := 1;

         loop
            exit when CurrentSymbol (Loops) = The_Loop;
            Loops  := NextSymbol (Loops);
            Number := Number + 1;
         end loop;

         return Number;

      end Get_Loop_Number;

      --------------------------------------------------------------------------------

      function Image (Number : Positive) return String is
         Signed_Image : constant String := Positive'Image (Number);
      begin
         return Signed_Image (2 .. Signed_Image'Length);
      end Image;

   begin

      if LoopHasName (The_Loop) then
         Name := Fetch_Simple_Name (Item => The_Loop);
      else
         Name := E_Strings.Copy_String (Str => "LOOP__");
         E_Strings.Append_String (E_Str => Name,
                                  Str   => Image (Number => Get_Loop_Number (The_Loop => The_Loop)));
      end if;

      return Name;

   end Get_Loop_Name;

   --------------------------------------------------------------------------------

   procedure Get_Package_Name (The_Package : in     Symbol;
                               Separator   : in     String;
                               Name        :    out E_Strings.T)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Name from Dict,
   --#                   LexTokenManager.State,
   --#                   Separator,
   --#                   The_Package;
   is
      Package_Local : Symbol;
      Current_Token : LexTokenManager.Lex_String;
      Stack         : LexTokenStacks.Stacks;
      Local_Name    : E_Strings.T;
   begin
      Local_Name    := E_Strings.Empty_String;
      Package_Local := The_Package;

      LexTokenStacks.Clear (Stack);
      loop
         LexTokenStacks.Push (Stack, GetSimpleName (Package_Local));
         Package_Local := RawDict.GetPackageParent (Package_Local);
         exit when Package_Local = NullSymbol;
      end loop;

      loop
         LexTokenStacks.Pop (Stack, Current_Token);
         E_Strings.Append_Examiner_String
           (E_Str1 => Local_Name,
            E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => Current_Token));
         exit when LexTokenStacks.IsEmpty (Stack);
         E_Strings.Append_String (E_Str => Local_Name,
                                  Str   => Separator);
      end loop;
      Name := Local_Name;
   end Get_Package_Name;

   --------------------------------------------------------------------------------

   procedure Get_Record_Variable_Name (The_Record : in     Symbol;
                                       Separator  : in     String;
                                       Name       :    out E_Strings.T)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Name from Dict,
   --#                   LexTokenManager.State,
   --#                   Separator,
   --#                   The_Record;
   is
      Record_Local  : Symbol;
      Current_Token : LexTokenManager.Lex_String;
      Stack         : LexTokenStacks.Stacks;
      Local_Name    : E_Strings.T;
   begin
      Local_Name   := E_Strings.Empty_String;
      Record_Local := The_Record;

      LexTokenStacks.Clear (Stack);
      loop
         -- we want to ignore any inherited fields for name generation purposes
         if not (IsRecordSubcomponent (Record_Local)
                   and then RecordComponentIsInherited (RawDict.GetSubcomponentComponent (Record_Local))) then
            LexTokenStacks.Push (Stack, GetSimpleName (Record_Local));
         end if;
         exit when not IsRecordSubcomponent (Record_Local); --entire record var

         Record_Local := GetEnclosingObject (Record_Local);
      end loop;

      loop
         LexTokenStacks.Pop (Stack, Current_Token);
         E_Strings.Append_Examiner_String
           (E_Str1 => Local_Name,
            E_Str2 => LexTokenManager.Lex_String_To_String (Lex_Str => Current_Token));
         exit when LexTokenStacks.IsEmpty (Stack);
         E_Strings.Append_String (E_Str => Local_Name,
                                  Str   => Separator);
      end loop;
      Name := Local_Name;
   end Get_Record_Variable_Name;

   --------------------------------------------------------------------------------

   function Get_Loop_Entry_Variable_Name (The_Loop_Entry_Var : Symbol) return E_Strings.T
   --# global in Dict;
   --#        in LexTokenManager.State;
   is
      Result    : E_Strings.T;
      Loop_Name : E_Strings.T;
   begin
      -- Loop on entry variable names are constructed from the original variable name
      -- and the associated loop name
      Result    := Fetch_Simple_Name (Item => The_Loop_Entry_Var);
      Loop_Name := Get_Loop_Name (The_Loop => RawDict.GetLoopEntryVariableTheLoop (The_Loop_Entry_Var));
      E_Strings.Append_String (E_Str => Result,
                               Str   => "__entry__");
      E_Strings.Append_Examiner_String (E_Str1 => Result,
                                        E_Str2 => Loop_Name);
      return Result;
   end Get_Loop_Entry_Variable_Name;

   --------------------------------------------------------------------------------

   procedure Get_Parameter_Constraint_Name (Item : in     Symbol;
                                            Name :    out E_Strings.T)
   --# global in Dict;
   --#        in LexTokenManager.State;
   --# derives Name from Dict,
   --#                   Item,
   --#                   LexTokenManager.State;
   is
      -- because of temp use of 'image
      Local_Name : E_Strings.T;
   begin
      Local_Name := Fetch_Simple_Name (Item => Item);
      -- above line will return the name of the formal parameter associated with the constraint
      E_Strings.Append_String (E_Str => Local_Name,
                               Str   => "__index__subtype__");
      E_Strings.Append_Examiner_String
        (E_Str1 => Local_Name,
         E_Str2 => Maths.ValueToString (Num => Maths.IntegerToValue (I => GetSubprogramParameterConstraintDimension (Item))));

      Name := Local_Name;
   end Get_Parameter_Constraint_Name;

begin --GenerateSimpleName

   if IsLoop (Item) then
      Name := Get_Loop_Name (The_Loop => Item);
   elsif IsPackage (Item) then
      Get_Package_Name (The_Package => Item,
                        Separator   => Separator,
                        Name        => Name);
   elsif IsRecordSubcomponent (Item) then
      Get_Record_Variable_Name (The_Record => Item,
                                Separator  => Separator,
                                Name       => Name);
   elsif RawDict.GetSymbolDiscriminant (Item) = ProtectedImplicitInStreamSymbol then
      Name := Get_Implicit_Protected_In_Stream_Name (Item => Item);

   elsif RawDict.GetSymbolDiscriminant (Item) = LoopEntryVariableSymbol then
      Name := Get_Loop_Entry_Variable_Name (The_Loop_Entry_Var => Item);

   elsif RawDict.GetSymbolDiscriminant (Item) = ParameterConstraintSymbol then
      Get_Parameter_Constraint_Name (Item => Item,
                                     Name => Name);

   else
      Name := Fetch_Simple_Name (Item => Item);
   end if;

end GenerateSimpleName;
