/************************************************************************

The L1 theory of the galilean satellites
by Valery Lainey can be found at
ftp://ftp.imcce.fr/pub/ephem/satel/galilean/L1

I (Johannes Gajdosik) have just taken the Fortran code and data
obtained from above and rearranged it into this piece of software.

I can neither allow nor forbid the usage of the L1 theory.
The copyright notice below covers not the work of Valery Lainey
but just my work, that is the compilation of the L1 theory
into the software supplied in this file.


Copyright (c) 2005 Johannes Gajdosik

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This is an implementation of the L1 theory, V1.1.
My modifications are:
1) do not calculate constant terms at runtime but beforehand
2) unite terms with the same frequencies
3) rearrange the terms so that calculation of the argument becomes easier
4) substitute so that the independent variable becomes T=jd-2433282.5
5) artificially blend out the usage of polynomials outside [-695+t0,695+t0]
   in order to avoid a discontinuity

****************************************************************/

#include "l1.h"
#include "calc_interpolated_elements.h"
#include "elliptic_to_rectangular.h"

#include <math.h>
#include <string.h> /* memcpy */


struct L1Term {
  double phase,frequency,amplitude;
};

struct L1TermList {
  const struct L1Term *terms;
  int size;
};

struct L1Body {
  const char *name;
  double mu,l;
  const double cheb_coef[9*5];
  const double constants[6];
  const struct L1TermList lists[4];
};

static const struct L1Term l1_io_0[40] = {
  {     4.936971269662929, 1.358483651813734e-02,     0.0000000013655244},
  {     4.318879753523476, 1.303413843350877e-02,     0.0000000003280812},
  {     4.318879692863362, 1.303413843323839e-02,     0.0000000003280812},
  {     5.776230083696149, 1.557111725795875e-02,     0.0000000001706288},
  {     5.776230180265387, 1.557111725834464e-02,     0.0000000001706288},
  {     1.493615407699053, 1.293892891234463e-02,     0.0000000001855996},
  {     1.493615489707281, 1.293892891261108e-02,     0.0000000001855996},
  {     0.198719243572640, 2.467531551030581e-02,     0.0000000003014418},
  {     1.907986021537059, 3.050651153320071e-03,     0.0000000001449720},
  {     5.088989063845719, 2.292942571803521e-02,     0.0000000000545492},
  {     2.488042064840510, 2.511961071886558e-02,     0.0000000000326102},
  {     3.639290231885086, 3.564459165624139e+00,     0.0000000762024588},
  {     1.819648778297500, 1.782229577756764e+00,     0.0000000172337652},
  {     0.995547070485000, 7.128918331248277e+00,     0.0000000180900324},
  {     3.476022511126086, 8.020033111279874e+00,     0.0000000094794086},
  {     4.634700512911087, 1.069337743620875e+01,     0.0000000092196266},
  {     2.815055958829500, 8.911147863507288e+00,     0.0000000101726080},
  {     1.158674645271500, 2.673344370426625e+00,     0.0000000058581604},
  {     0.171224688618000, 1.247560707968448e+01,     0.0000000034892754},
  {     3.617031125424586, 6.350132082671720e+00,     0.0000000030842852},
  {     5.458960182461587, 5.346688718104374e+00,     0.0000000036218148},
  {     4.292519743915586, 1.192210212315995e+01,     0.0000000020794650},
  {     0.489778750861000, 1.281321698327768e+01,     0.0000000011682572},
  {     2.472795329432000, 1.013987254731953e+01,     0.0000000008031976},
  {     4.329737719804585, 7.100204488649648e+00,     0.0000000007014118},
  {     0.653129619279000, 8.357642973670989e+00,     0.0000000007309510},
  {     5.425217941794086, 9.525198124007579e+00,     0.0000000005753088},
  {     5.116174218429586, 6.575413448118254e+00,     0.0000000004359548},
  {     3.296999891121586, 4.793183845928428e+00,     0.0000000003228344},
  {     1.799472337577500, 3.175066346181017e+00,     0.0000000003432980},
  {     2.978320571599000, 4.455573931753644e+00,     0.0000000001655832},
  {     4.455620193332086, 7.466528180315160e+00,     0.0000000001527992},
  {     1.477346040837000, 3.010954258455435e+00,     0.0000000001612910},
  {     5.071877861964587, 3.551425545660377e+00,     0.0000000001707670},
  {     4.632329727516586, 4.468609210818232e+00,     0.0000000001523312},
  {     5.332168066546586, 7.098754944908187e+00,     0.0000000001188688},
  {     0.950314972153000, 1.270026416534344e+01,     0.0000000001129258},
  {     5.941275870875086, 1.228724668691016e+00,     0.0000000000986086},
  {     3.368283422860086, 5.973671126673010e+00,     0.0000000000857194},
  {     0.481234139262000, 1.795814557653498e+00,     0.0000000000877720},
};

static const struct L1Term l1_io_1[46] = {
  {     0.224569991851467, 1.358483658304878e-02,     0.0000962629174333},
  {     0.224569946198471, 1.358483658275507e-02,     0.0000962629174333},
  {     5.889675968665478, 1.303413843242723e-02,     0.0000970803596076},
  {     3.478797969600659, 3.050648671575486e-03,     0.0000898817416500},
  {     3.064411994986507, 1.293892891154528e-02,     0.0000553101050262},
  {     1.063837150558219, 1.557111722129952e-02,     0.0000418078870490},
  {     0.569447763489052, 1.450097748894791e-03,     0.0000186178298694},
  {     0.569447764235179, 1.450097748934951e-03,     0.0000186178298694},
  {     1.769515784072581, 2.467531550739867e-02,     0.0000234440533016},
  {     4.391143425869121, 9.519618999652977e-05,     0.0000080156582120},
  {     4.391143425815649, 9.519618997170757e-05,     0.0000080156582120},
  {     5.559625884260257, 5.498007890313636e-04,     0.0000041114727746},
  {     5.559625884225639, 5.498007890308824e-04,     0.0000041114727746},
  {     0.728712205995279, 4.445680518495960e-04,     0.0000043608559052},
  {     0.728712207380696, 4.445680521141367e-04,     0.0000043608559052},
  {     1.493663997667284, 6.482674962353211e-04,     0.0000037682740860},
  {     1.493663996541898, 6.482674963109011e-04,     0.0000037682740860},
  {     3.066002181736643, 2.900129099234684e-03,     0.0000026598151336},
  {     3.066002183922500, 2.900129099355159e-03,     0.0000026598151336},
  {     5.211815935681586, 3.564459104960471e+00,     0.0000503584426150},
  {     3.390444221031586, 1.782229577756764e+00,     0.0000444412770116},
  {     2.729470598014000, 2.673344370426625e+00,     0.0000109014269320},
  {     2.566011851738000, 7.128918331248277e+00,     0.0000119049755698},
  {     5.046820014308086, 8.020033111279874e+00,     0.0000057575824778},
  {     4.385846277590586, 8.911147863507288e+00,     0.0000061452803962},
  {     0.211262261618000, 3.055483387780041e-03,     0.0000023908706663},
  {     0.211277724557000, 3.055482560568297e-03,     0.0000023908713552},
  {     0.387713436606500, 2.292862594121051e-02,     0.0000045554015322},
  {     6.205504042701087, 1.069337743620875e+01,     0.0000051181206936},
  {     0.013492534637000, 1.567711247818633e-02,     0.0000021602067349},
  {     0.013492473587000, 1.567711248209791e-02,     0.0000021602067349},
  {     0.746568951200000, 5.346688718104374e+00,     0.0000037684098282},
  {     3.789203954929086, 2.515548916550717e-02,     0.0000031403738247},
  {     1.567931706749000, 6.041417160387329e-04,     0.0000008114937768},
  {     1.567928771616000, 6.041419175960708e-04,     0.0000008114933753},
  {     5.565595482699586, 2.542696883441216e-03,     0.0000012168267714},
  {     5.565390277403586, 2.542709795311175e-03,     0.0000012168291116},
  {     5.187696189195586, 6.350132082671720e+00,     0.0000020289901692},
  {     1.741664418607500, 1.247560707968448e+01,     0.0000018552431038},
  {     3.275009838206586, 1.358967489812801e-02,     0.0000009332719352},
  {     3.275009843305586, 1.358967489754016e-02,     0.0000009332719348},
  {     6.146147910884586, 4.569242920804141e-03,     0.0000001301198829},
  {     6.146147973389586, 4.569242952636889e-03,     0.0000001301198573},
  {     5.099743141813586, 3.765868037914274e-03,     0.0000004035364904},
  {     5.099738145003586, 3.765868360205259e-03,     0.0000004035364083},
  {     3.042608902565000, 1.435846001232353e-02,     0.0000013160987603},
};

static const struct L1Term l1_io_2[23] = {
  {     4.089939635618272,-1.290686414665676e-02,     0.0041510849668155},
  {     2.125628703511182, 1.272741656696607e-04,     0.0000352747346169},
  {     5.583561992607979, 3.206575113588943e-05,     0.0000198194483636},
  {     0.441372126933203, 2.664253354771139e-03,     0.0000146399842989},
  {     5.887665756035155, 6.779610093550867e-04,     0.0000052220588690},
  {     5.436070258064092,-2.649170024124419e-02,     0.0000022044764663},
  {     5.855785804221171, 2.773123667986205e-03,     0.0000015410375360},
  {     5.997705336538921,-9.856616995689868e-03,     0.0000012193607962},
  {     1.446188898563000, 3.551552294980162e+00,     0.0006260521444113},
  {     2.768134001457000, 1.769322707946172e+00,     0.0000096819265753},
  {     0.450761187889000,-3.577366026002207e+00,     0.0000098749504021},
  {     3.429107402393586, 8.782079195152743e-01,     0.0000083063168209},
  {     4.253351377085586,-4.468480820057747e+00,     0.0000045951340101},
  {     5.074075247830586, 7.116011804891830e+00,     0.0000059689735869},
  {     4.914320333957586,-5.359595618434683e+00,     0.0000046538995236},
  {     3.094673734781000,-7.141825164091565e+00,     0.0000037405126681},
  {     2.270416702740000,-1.795136444582530e+00,     0.0000037711061757},
  {     4.112404291459587,-2.798579795458930e+00,     0.0000018698303790},
  {     1.275017772430000,-8.924054779978736e+00,     0.0000013214613496},
  {     1.972514861645000, 3.765422737898159e-01,     0.0000012707585609},
  {     0.124242374879000, 5.333781846063283e+00,     0.0000011886104747},
  {     0.689291076961000,-2.729322550009891e-03,     0.0000007689215742},
  {     2.390352831189000, 2.519492181877683e-03,     0.0000008742035177},
};

static const struct L1Term l1_io_3[14] = {
  {     2.796421972262301,-2.315096098002346e-03,     0.0003142172466014},
  {     1.047706187943049,-5.692063819573326e-04,     0.0000904169207946},
  {     3.336886177393517,-1.249130719694158e-04,     0.0000164452324013},
  {     5.972020238073122,-3.056116472064809e-05,     0.0000055424829091},
  {     0.848987368454367,-2.524452190063377e-02,     0.0000035856270353},
  {     5.560377095159332, 2.900368144577740e-03,     0.0000024180760140},
  {     3.426559514581584,-1.450059335316387e-03,     0.0000008673084930},
  {     2.241870649990240,-2.349863229870366e-02,     0.0000003176227277},
  {     4.556949902706002, 4.350465430403529e-03,     0.0000003152816608},
  {     1.763329212023187, 1.450133913858279e-03,     0.0000002338676726},
  {     4.842931998487813,-2.568881653244262e-02,     0.0000001754553689},
  {     5.754334714440532,-2.581366097973931e-02,     0.0000001286319583},
  {     1.150359242663171,-2.900147139783979e-03,     0.0000000967213304},
  {     4.074596685132587,-3.250675731907161e-02,     0.0000000000692310},
};

static const struct L1Term l1_europa_0[37] = {
  {     1.177276578613870, 1.303413830886066e-02,     0.0000000181310770},
  {     4.635246315103281, 1.293892882068561e-02,     0.0000000095367130},
  {     4.937673897162706, 1.358483301703257e-02,     0.0000000071939342},
  {     2.634643550907048, 1.557111712675731e-02,     0.0000000016464798},
  {     5.049578507084870, 3.050649783815080e-03,     0.0000000010251826},
  {     5.629533861681669, 2.511960973067428e-02,     0.0000000006957184},
  {     3.340315686762771, 2.467531551127487e-02,     0.0000000005817914},
  {     5.812695353374450, 2.597306713875762e-02,     0.0000000003397378},
  {     4.718111635715633, 2.524444183019983e-02,     0.0000000003792178},
  {     1.947144118910063, 2.292942491975595e-02,     0.0000000002538154},
  {     2.354547673191147, 2.606827709954982e-02,     0.0000000003159492},
  {     1.819645606290000, 1.782229577756764e+00,     0.0000004324367498},
  {     4.300272670479086, 2.673344370426625e+00,     0.0000001603614750},
  {     2.317355432932000, 5.346688718104374e+00,     0.0000001019146786},
  {     5.622213904945586, 8.911147888783818e-01,     0.0000000924734786},
  {     2.978330742899000, 4.455573931753644e+00,     0.0000000511509000},
  {     0.497691978790000, 3.564459165624139e+00,     0.0000000523665800},
  {     5.956040701803087, 8.911147863507288e+00,     0.0000000272859938},
  {     4.136258231316587, 7.128918331248277e+00,     0.0000000311907780},
  {     1.656394146385500, 6.237803539842242e+00,     0.0000000174960544},
  {     6.260843436356586, 2.785672921154949e+00,     0.0000000232225828},
  {     1.500536365979000, 1.069337725421775e+01,     0.0000000122874072},
  {     3.313061217653586, 1.247560707968448e+01,     0.0000000084863836},
  {     6.248874599604086, 4.178509428046355e+00,     0.0000000069122354},
  {     0.334349746004500, 8.020033111279874e+00,     0.0000000061377568},
  {     1.152376951765000, 1.192210212315995e+01,     0.0000000045343054},
  {     0.345978042928000, 3.535769222753918e+00,     0.0000000044574684},
  {     6.271965520209586, 1.392836463665070e+00,     0.0000000042350072},
  {     5.613896859751086, 1.013987254731953e+01,     0.0000000028783772},
  {     0.995871908751000, 9.041498929737958e-01,     0.0000000024354662},
  {     6.237905056035586, 5.571345867010661e+00,     0.0000000021573570},
  {     5.295896589090586, 9.802262705466383e+00,     0.0000000022532940},
  {     3.779109380028586, 8.357644085371536e+00,     0.0000000016530062},
  {     3.273238819524086, 1.769195144071588e+00,     0.0000000011589838},
  {     1.686281234612500, 3.534296144322957e+00,     0.0000000007614982},
  {     5.911271718795586, 2.409221457483074e+00,     0.0000000007104494},
  {     1.972390619301000, 6.575413585742917e+00,     0.0000000010203510},
};

static const struct L1Term l1_europa_1[46] = {
  {     2.748072985127734, 1.303413830804950e-02,     0.0008576433172936},
  {     6.206042155402145, 1.293892881961980e-02,     0.0004549582875090},
  {     0.225314741917063, 1.358483286723702e-02,     0.0003074250079334},
  {     0.569489012050155, 1.450097893384534e-03,     0.0000917031775902},
  {     0.569489012343467, 1.450097893347720e-03,     0.0000917031775902},
  {     0.337190578636981, 3.051012128689309e-03,     0.0001982386144784},
  {     4.205435626520602, 1.557111708469567e-02,     0.0000632289777196},
  {     1.258190389436086, 2.988574915090479e-05,     0.0000174586375148},
  {     1.258190390003000, 2.988574903752128e-05,     0.0000174586375134},
  {     0.659751997608950, 1.249523377356387e-04,     0.0000162854547323},
  {     0.659751997617150, 1.249523377385212e-04,     0.0000162854547324},
  {     6.238235401865971, 2.900131552216926e-03,     0.0000102913236930},
  {     6.238235399661924, 2.900131552096451e-03,     0.0000102913236930},
  {     0.917145067551655, 2.511960972565299e-02,     0.0000168028316254},
  {     4.911112175101827, 2.467531551030581e-02,     0.0000140713155600},
  {     4.489153977198828, 6.493040371814837e-04,     0.0000141628733606},
  {     2.858093676287215, 5.529787193149190e-04,     0.0000100746080234},
  {     3.764894412909473, 3.023321923191210e-03,     0.0000106598617621},
  {     4.071832143192552, 9.347832247393878e-05,     0.0000047325683320},
  {     4.071832143214682, 9.347832247226989e-05,     0.0000047325683325},
  {     1.100458390519143, 2.597306924635157e-02,     0.0000078429703340},
  {     0.005721925039605, 2.524444168211706e-02,     0.0000091108073323},
  {     0.248853137660000, 1.782229577756764e+00,     0.0003248939825174},
  {     0.909825056364000, 8.911147888783818e-01,     0.0001434383188452},
  {     5.871068746433586, 2.673344370426625e+00,     0.0000771939140944},
  {     2.068494534904000, 3.564459165624139e+00,     0.0000436574731410},
  {     3.888148529365586, 5.346688718104374e+00,     0.0000446766477388},
  {     4.549127462407086, 4.455573931753644e+00,     0.0000192706087556},
  {     5.708236423986586, 7.128918331248277e+00,     0.0000131946915760},
  {     1.160449904077500, 8.911151023074472e+00,     0.0000097414019416},
  {     3.108492565509000, 1.567639331506651e-02,     0.0000043860283834},
  {     3.108492540844000, 1.567639331663100e-02,     0.0000043860283834},
  {     4.629421495642586, 4.325287255860915e-04,     0.0000037783019709},
  {     4.629426998359586, 4.325281834936648e-04,     0.0000037783070079},
  {     1.548440639408000, 2.785672921154949e+00,     0.0000104011727738},
  {     3.529923386229586, 2.292856741249285e-02,     0.0000066580990752},
  {     3.227185713463586, 6.237803539842242e+00,     0.0000058131135230},
  {     3.432563660625086, 2.609305838466956e-02,     0.0000065854142774},
  {     2.385773248504000, 2.548121633990641e-03,     0.0000027860432638},
  {     2.385773134668000, 2.548121600674440e-03,     0.0000027860415206},
  {     1.559634116074000, 1.392836460577475e+00,     0.0000048198508906},
  {     3.051294950265000, 1.069337798218179e+01,     0.0000042728431266},
  {     6.063323317190586, 3.016443578777988e-03,     0.0000021087772652},
  {     6.063326404038587, 3.016443399773415e-03,     0.0000021087773173},
  {     3.532654385236586, 2.521965820225456e-02,     0.0000018853812260},
  {     3.532654376330586, 2.521965820326684e-02,     0.0000018853812260},
};

static const struct L1Term l1_europa_2[41] = {
  {     0.948346997389059,-1.290686414665676e-02,     0.0093589104136341},
  {     2.743516829216055, 6.779734300917777e-04,     0.0001980963564781},
  {     2.125628929943986, 1.272741840737257e-04,     0.0002139036390350},
  {     5.583994371124913, 3.205661489829892e-05,     0.0001210388158965},
  {     6.054257618717536,-2.594100240450487e-02,     0.0000139052321679},
  {     2.596285588195975,-2.584579292787240e-02,     0.0000073925894084},
  {     2.293742862387695,-2.649169672503697e-02,     0.0000051968296512},
  {     5.909726518566586, 1.769322707946172e+00,     0.0002988994545555},
  {     1.446188770927000, 3.551552294980162e+00,     0.0000823525166369},
  {     1.609453836719000,-9.040216580884552e-01,     0.0000837042048393},
  {     5.412030084622586,-1.795136449711322e+00,     0.0000278946698536},
  {     3.429104897570586, 8.782079195152743e-01,     0.0000315906532820},
  {     3.592372683290586,-3.577366026002207e+00,     0.0000294503681314},
  {     2.931395664158000,-2.686251242238948e+00,     0.0000144958688621},
  {     5.932076111633586,-1.016350216012791e+00,     0.0000108374431350},
  {     1.772880355220000,-5.359595618434683e+00,     0.0000082175838585},
  {     6.225293638347586,-7.141824839379420e+00,     0.0000062618381566},
  {     4.253337137108586,-4.468480820057747e+00,     0.0000041298266970},
  {     1.973436581069000, 3.765422106060407e-01,     0.0000043507065743},
  {     3.120261383685000, 4.442723075715851e+00,     0.0000042081682285},
  {     4.415073322123586,-8.924054653229724e+00,     0.0000027357551003},
  {     2.107163763597000, 2.660437500205714e+00,     0.0000036991221930},
  {     5.848419525410586,-2.409184340145380e+00,     0.0000020163445050},
  {     3.897559286261586, 1.580695318046377e-02,     0.0000026854901206},
  {     2.596961125563000,-1.070628432888433e+01,     0.0000017894118824},
  {     1.943899853407000, 7.116011482522718e+00,     0.0000023074479953},
  {     2.604869046210000, 6.224896681878776e+00,     0.0000018159137522},
  {     1.882700621195000,-1.435121070413662e-02,     0.0000011726743714},
  {     3.265849247810586, 5.333781846063283e+00,     0.0000016518864520},
  {     5.825415163815586, 6.823660907511857e-04,     0.0000018506530067},
  {     5.575302565324586,-6.250710366541258e+00,     0.0000013196935928},
  {     6.001407591182586,-1.766476914981084e+00,     0.0000014426949422},
  {     3.037342396932000,-1.145358884795799e-02,     0.0000015660692561},
  {     2.168360257491000,-1.287180323293973e-02,     0.0000005646670312},
  {     6.131229888257586, 2.009755324388130e-03,     0.0000007495662748},
  {     0.668283437067000,-2.727162779379118e-03,     0.0000007569857746},
  {     5.982532958241586, 1.425759504491192e-03,     0.0000009550285338},
  {     2.713933181549000,-1.992493278328030e-03,     0.0000007091149133},
  {     4.430977594715586,-3.272492347779120e-03,     0.0000002004455524},
  {     3.383487199815586, 4.460933767881522e-03,     0.0000001623489363},
  {     4.535695340655586, 3.926990817208134e-03,     0.0000001058862562},
};

static const struct L1Term l1_europa_3[24] = {
  {     1.047706317001854,-5.692064054047047e-04,     0.0040404917832303},
  {     3.336885786362735,-1.249130730713476e-04,     0.0002200421034564},
  {     5.971993096807454,-3.056160225452175e-05,     0.0000590282470983},
  {     5.938090491493398,-2.315096612380823e-03,     0.0000105030331400},
  {     3.990580616746044,-2.524452190164805e-02,     0.0000102943248250},
  {     5.560373031227549, 2.900367671313766e-03,     0.0000072600013020},
  {     0.284805154959940,-1.450057919685263e-03,     0.0000018391258758},
  {     4.842997492970255,-2.568881513871077e-02,     0.0000014880605763},
  {     3.791704507719480, 3.469617068314515e-03,     0.0000008828196274},
  {     1.763943031904567, 1.450135215764323e-03,     0.0000008714042768},
  {     4.556850666664978, 4.350464141008628e-03,     0.0000008536188044},
  {     5.754211725450162,-2.581376870262505e-02,     0.0000006846214331},
  {     5.383469432185194,-2.349863236637365e-02,     0.0000004471826348},
  {     2.207820131500654,-2.578317090601877e-02,     0.0000003034392168},
  {     2.053356838149028,-2.019323693190186e-03,     0.0000001792048645},
  {     3.185886850191700, 8.808605651694719e-04,     0.0000001799083735},
  {     3.838710286255186,-5.379508584724946e-04,     0.0000001062153531},
  {     2.787089536861053, 4.919731657972428e-03,     0.0000001098546626},
  {     1.439213487322675,-5.995945940555366e-04,     0.0000001083128732},
  {     3.555376872968097, 5.800558781273577e-03,     0.0000000768496749},
  {     1.502468480636942, 3.025321902922990e-03,     0.0000000692273841},
  {     0.136213196685887,-4.443041360187748e-04,     0.0000000676969224},
  {     6.150942371631842,-1.360328720068605e-02,     0.0000000621559952},
  {     4.052956952638587,-3.251086990093566e-02,     0.0000000000608298},
};

static const struct L1Term l1_ganymede_0[37] = {
  {     1.795689963930872, 1.358483493794050e-02,     0.0000000109535132},
  {     4.318866575067722, 1.303413828533705e-02,     0.0000000112067460},
  {     1.493663064768192, 1.293892879936955e-02,     0.0000000057842684},
  {     3.340309354863841, 2.467531549383199e-02,     0.0000000006819260},
  {     2.488000300392696, 2.511961019665066e-02,     0.0000000004433776},
  {     2.671035049815109, 2.597306860752402e-02,     0.0000000002836658},
  {     1.158674594378500, 2.673344370426625e+00,     0.0000013930299110},
  {     5.622214570144586, 8.911147888783818e-01,     0.0000006449829346},
  {     1.299592404339000, 1.003443345672872e+00,     0.0000002298059520},
  {     1.819651005956500, 1.782229577756764e+00,     0.0000001221434370},
  {     1.948670852981000, 1.505165046152930e+00,     0.0000001095798176},
  {     0.649785092858000, 5.017216696313782e-01,     0.0000000701435616},
  {     2.599205084699500, 2.006886694550801e+00,     0.0000000547868566},
  {     0.497708209841000, 3.564459165624139e+00,     0.0000000363221428},
  {     5.458946213782586, 5.346688718104374e+00,     0.0000000394635858},
  {     3.249001076179586, 2.508608372194823e+00,     0.0000000281244968},
  {     6.119923526117086, 4.455573931753644e+00,     0.0000000207924698},
  {     2.012339224075000, 1.753515735038438e+00,     0.0000000290949364},
  {     3.898824401243586, 3.010330041826201e+00,     0.0000000146896774},
  {     4.797989502688086, 6.237803539842242e+00,     0.0000000119930042},
  {     0.967007208325000, 6.269963377697709e-01,     0.0000000099867772},
  {     4.548637298168086, 3.512051718268327e+00,     0.0000000077668260},
  {     0.334410719597500, 8.020033111279874e+00,     0.0000000066346638},
  {     0.161404499137500, 1.253136166156572e-01,     0.0000000074143972},
  {     3.014796788641000, 1.752066209379324e+00,     0.0000000049395106},
  {     1.323585106216000, 1.128738614471043e+00,     0.0000000055768352},
  {     5.198455807371587, 4.013773414742063e+00,     0.0000000041439704},
  {     1.497090958510500, 1.069337725421775e+01,     0.0000000036862062},
  {     3.749365837602586, 8.780866918016850e-01,     0.0000000033617538},
  {     4.137030077788586, 7.128918331248277e+00,     0.0000000040765630},
  {     2.266819680012500, 1.630439448567333e+00,     0.0000000033348284},
  {     3.474524629117086, 1.795264812030685e+00,     0.0000000025754698},
  {     1.960483834895500, 1.123285451319658e-01,     0.0000000024363084},
  {     2.916664806503000, 2.132161076533328e+00,     0.0000000020032676},
  {     4.139420220609086, 9.041497836838437e-01,     0.0000000018115706},
  {     5.848274571420586, 4.515495095190510e+00,     0.0000000022265432},
  {     1.874821204178000, 8.911213709350575e+00,     0.0000000014535006},
};

static const struct L1Term l1_ganymede_1[40] = {
  {     0.569502392743459, 1.450097843836871e-03,     0.0001155398943113},
  {     0.569502392904042, 1.450097843950659e-03,     0.0001155398943113},
  {     5.889663600434602, 1.303413828263318e-02,     0.0000914317982059},
  {     5.889663551477412, 1.303413828236280e-02,     0.0000914317982059},
  {     3.366513910377989, 1.358483481252341e-02,     0.0000756189389102},
  {     3.366513977192199, 1.358483481281712e-02,     0.0000756189389102},
  {     3.064457611036154, 1.293892879857020e-02,     0.0000955478069849},
  {     1.221958417224527, 2.988087370216886e-05,     0.0000271961236501},
  {     1.221958417402599, 2.988087381555233e-05,     0.0000271961236540},
  {     0.658783828622433, 1.249527829223785e-04,     0.0000244626823237},
  {     0.658783828631151, 1.249527829177814e-04,     0.0000244626823247},
  {     4.120474976177235, 6.516407355555155e-04,     0.0000077362945254},
  {     4.120474976113769, 6.516407355528122e-04,     0.0000077362945254},
  {     3.478660054990111, 3.050767822667433e-03,     0.0000169767346458},
  {     4.363820153673164, 5.566368140678420e-04,     0.0000135654458738},
  {     3.785265431705326, 9.242697748875326e-05,     0.0000037731001815},
  {     3.785265431557370, 9.242697748917474e-05,     0.0000037731001804},
  {     6.265025662502943, 2.900130973239494e-03,     0.0000032076570609},
  {     6.265025663242703, 2.900130973279653e-03,     0.0000032076570609},
  {     4.911104313708385, 2.467531550158438e-02,     0.0000052570245720},
  {     4.057010234996053, 4.483462238592540e-04,     0.0000056564973024},
  {     5.871062330575587, 2.673344370426625e+00,     0.0001163720969778},
  {     4.051417386548087, 8.911147888783818e-01,     0.0000815246854464},
  {     2.870388622006000, 1.003443345672872e+00,     0.0000801219679602},
  {     2.220584006016000, 5.017216704326426e-01,     0.0000607017260182},
  {     3.390447645294086, 1.782229577756764e+00,     0.0000427574981536},
  {     3.520633597109586, 1.505165006490341e+00,     0.0000307360417826},
  {     4.169993625488086, 2.006886694550801e+00,     0.0000134648621904},
  {     1.759435083055500, 1.253082718119475e-01,     0.0000145268863648},
  {     2.068506076335000, 3.564459165624139e+00,     0.0000087955125170},
  {     0.746555718634000, 5.346688718104374e+00,     0.0000095524017320},
  {     4.819808572193086, 2.508608372194823e+00,     0.0000063221625942},
  {     3.582852484268586, 1.753515764400782e+00,     0.0000071146195958},
  {     5.183307669641087, 4.363323133976446e-04,     0.0000046565198769},
  {     2.536966837075000, 6.269971273750473e-01,     0.0000047004229470},
  {     1.407535446449500, 4.455573931753644e+00,     0.0000047020767994},
  {     3.531270776153086, 1.123285428225707e-01,     0.0000042349322008},
  {     0.973793744157000, 1.707457650159172e-03,     0.0000032577733689},
  {     3.832758803081586, 2.514666393972801e-02,     0.0000019377870959},
  {     3.832758803745586, 2.514666393872159e-02,     0.0000019377870959},
};

static const struct L1Term l1_ganymede_2[49] = {
  {     2.125629594296477, 1.272741302888879e-04,     0.0014289811307319},
  {     5.583633000278704, 3.206434109929470e-05,     0.0007710931226760},
  {     4.089939663676184,-1.290686414665676e-02,     0.0005925911780766},
  {     2.912662235744477,-2.594100241521497e-02,     0.0000070808673396},
  {     5.939436031355292, 6.777425870293163e-04,     0.0000070804404020},
  {     2.713829450845914, 2.773132967187178e-03,     0.0000048299146941},
  {     5.737887016354762,-2.584579295551464e-02,     0.0000038142769361},
  {     5.560800670596301, 1.450189296775174e-03,     0.0000022651772374},
  {     5.538575925817732, 2.868333902884076e-03,     0.0000023143850535},
  {     5.433477423083559,-2.649168789654619e-02,     0.0000007268381420},
  {     5.271368366954587,-1.252354407610591e-01,     0.0002045597496146},
  {     0.287431567198000, 8.782079244251990e-01,     0.0001785118648258},
  {     1.446212727724000, 3.551552294980162e+00,     0.0001131999784893},
  {     5.411835052601586,-1.795136439453738e+00,     0.0000658778169210},
  {     5.909679220467586, 1.769322712928523e+00,     0.0000497058888328},
  {     4.750898147481586,-9.040216502842435e-01,     0.0000316384926978},
  {     4.621732126831587,-6.269571234183975e-01,     0.0000287801237327},
  {     2.779471484341000, 3.764862399167294e-01,     0.0000181744317896},
  {     3.972019139951586,-1.128678804105747e+00,     0.0000105558175161},
  {     1.111461108666000,-4.468480788278769e+00,     0.0000057610853129},
  {     4.556761229712587,-8.753076941691317e-01,     0.0000061046181888},
  {     6.072772975850587,-2.686251219269880e+00,     0.0000057310334964},
  {     3.322298022924586,-1.630400483203856e+00,     0.0000046610005483},
  {     1.586656801126000, 1.881651292059337e+00,     0.0000034982417330},
  {     0.450524745204000,-3.577366005634277e+00,     0.0000030091617315},
  {     2.204580355873000, 2.512257683511147e-01,     0.0000024732926446},
  {     2.672542463618000,-2.132122165476585e+00,     0.0000022247695560},
  {     2.235037411637000, 2.383373019267311e+00,     0.0000020947921969},
  {     4.704947702792587,-2.504961383471171e-01,     0.0000024416432533},
  {     4.078773097655587, 1.379929604182191e+00,     0.0000014042712722},
  {     1.772399331347000,-5.359595572717003e+00,     0.0000011180389240},
  {     3.548820047544586,-8.738575908927385e-01,     0.0000010371714944},
  {     2.022753854104000,-2.633843845433241e+00,     0.0000011076384510},
  {     2.886194141381000, 2.885094641620094e+00,     0.0000011932531874},
  {     2.107149695705000, 2.660437500205714e+00,     0.0000007178049665},
  {     4.059613418491587,-7.522179355699680e-01,     0.0000006908412319},
  {     3.535953029515586, 3.386816325850993e+00,     0.0000006659820028},
  {     2.301347989676000, 2.631723535815786e+00,     0.0000006772314920},
  {     6.235861841953586,-7.141825164091565e+00,     0.0000008993128501},
  {     5.085653543011587, 7.116011401930445e+00,     0.0000006286307601},
  {     0.743089169080000, 4.249653553444271e-03,     0.0000006784151570},
  {     1.372931645641000,-3.135565516587292e+00,     0.0000005660807396},
  {     0.785273916249000, 4.442667065855896e+00,     0.0000006339665249},
  {     5.644334161083586, 6.224900197155575e+00,     0.0000006128705113},
  {     2.433337444677000,-6.250710366541258e+00,     0.0000005206551413},
  {     5.076699678365586,-9.815169557449957e+00,     0.0000004583970422},
  {     0.754124274965000, 1.580709749570035e-02,     0.0000003466029660},
  {     3.093485444125000, 1.756337305843437e+00,     0.0000004577854173},
  {     1.395067854967000, 1.678667735288600e-04,     0.0000004718481418},
};

static const struct L1Term l1_ganymede_3[17] = {
  {     3.336886279643515,-1.249130705808571e-04,     0.0015932721570848},
  {     5.972078984993627,-3.056101770621953e-05,     0.0003513347911037},
  {     4.189298830024249,-5.692063212368414e-04,     0.0001441929255483},
  {     5.560404119741996, 2.900366501121089e-03,     0.0000157303527750},
  {     0.284776537096548,-1.450055448684063e-03,     0.0000025161319881},
  {     1.765262856076923, 1.450138392651262e-03,     0.0000020438305183},
  {     4.556897734077857, 4.350462159038006e-03,     0.0000017939612784},
  {     0.848988726208994,-2.524452190063377e-02,     0.0000013614276895},
  {     1.502522946673032, 3.025321458825885e-03,     0.0000008996109017},
  {     5.938792663009366,-2.315096564509435e-03,     0.0000008702078430},
  {     1.701360384894123,-2.568881601149987e-02,     0.0000004371144064},
  {     5.153446624221260, 2.933059686445946e-03,     0.0000001926397869},
  {     2.612785906861475,-2.581364299331308e-02,     0.0000002174259374},
  {     3.555472701935102, 5.800557776835659e-03,     0.0000001589279656},
  {     4.338250108044157,-1.575012498384486e-03,     0.0000001432228753},
  {     5.346846514319586,-2.578279742601696e-02,     0.0000000926213408},
  {     4.576421330689586,-3.261161471680143e-02,     0.0000000000106902},
};

static const struct L1Term l1_callisto_0[23] = {
  {     5.689407195800122, 6.516502165363889e-04,     0.0000000001238821},
  {     5.689407195941634, 6.516502165390923e-04,     0.0000000001238821},
  {     6.001425980182368, 5.563954266128718e-04,     0.0000000000937288},
  {     6.001425980276814, 5.563954266508102e-04,     0.0000000000937288},
  {     1.808423583970000, 3.175066041335860e+00,     0.0000027580210652},
  {     0.649657760137000, 5.017216816503444e-01,     0.0000035952049470},
  {     6.271890828573587, 1.392836469840261e+00,     0.0000012874896172},
  {     4.440657688683586, 1.003443369710804e+00,     0.0000004173729106},
  {     0.714288688361000, 7.500722586912998e-01,     0.0000002790757718},
  {     5.090580754632086, 1.505165046152930e+00,     0.0000001998252258},
  {     5.740309532653586, 2.006886726601377e+00,     0.0000001001149838},
  {     0.106887217341500, 2.508608402242239e+00,     0.0000000513967092},
  {     1.721959555734500, 7.486221659360567e-01,     0.0000000475687992},
  {     0.150827135054000, 3.764591707052528e-01,     0.0000000348242240},
  {     4.357351927207587, 6.269823879873727e-01,     0.0000000239106346},
  {     0.208389567476000, 3.010349123257764e+00,     0.0000000263234638},
  {     5.167297349010086, 1.253067807304936e-01,     0.0000000283840630},
  {     1.507540463385000, 2.798610908676766e+00,     0.0000000219977422},
  {     3.119143533262500, 2.785672933505331e+00,     0.0000000171144478},
  {     1.406515126072500, 3.512051757530283e+00,     0.0000000141956834},
  {     4.999981401905086, 1.128704257915200e+00,     0.0000000120003630},
  {     5.487313880061587, 6.739523859312739e+00,     0.0000000108418904},
  {     5.977263051687586, 1.016381159041193e+00,     0.0000000108218254},
};

static const struct L1Term l1_callisto_1[27] = {
  {     0.569624392172443, 1.450097932313169e-03,     0.0002793020061912},
  {     0.569624392465698, 1.450097932276356e-03,     0.0002793020061912},
  {     4.306680816494765, 2.972965062307501e-05,     0.0001902906934088},
  {     4.306680816654067, 2.972965069866374e-05,     0.0001902906934113},
  {     3.800467791671927, 1.249401172511456e-04,     0.0000167518242157},
  {     3.800467791886845, 1.249401172282642e-04,     0.0000167518242264},
  {     6.175488817572635, 2.900133940521357e-03,     0.0000092970467736},
  {     6.175488816122332, 2.900133940441040e-03,     0.0000092970467736},
  {     0.976949254931839, 6.516504478063490e-04,     0.0000085219011443},
  {     0.976949254803742, 6.516504478056732e-04,     0.0000085219011444},
  {     1.304155277683059, 5.564606906691821e-04,     0.0000075696916557},
  {     1.304155277735127, 5.564606906317277e-04,     0.0000075696916557},
  {     0.906837882382209, 9.316625672198218e-05,     0.0000058058699386},
  {     0.906837882259694, 9.316625673132085e-05,     0.0000058058699344},
  {     5.362185506310587, 5.017216724358036e-01,     0.0002205152863262},
  {     0.237682433315500, 3.175066041335860e+00,     0.0001877895151158},
  {     4.701215105146086, 1.392836463665070e+00,     0.0000766916975242},
  {     6.011980600630586, 1.003443345672872e+00,     0.0000747056855106},
  {     2.283688674382000, 7.500723697232807e-01,     0.0000388323297366},
  {     0.378597607197000, 1.505165020913100e+00,     0.0000293032677938},
  {     1.028401022243500, 2.006886694550801e+00,     0.0000129927896682},
  {     4.902903788593086, 1.253079007501147e-01,     0.0000148825637256},
  {     1.987215709322000, 1.655051374188395e-03,     0.0000032693721243},
  {     1.987215709307000, 1.655051374192755e-03,     0.0000032693721243},
  {     3.285699493062586, 7.486228616656904e-01,     0.0000066211702894},
  {     1.678207462150000, 2.508608372194823e+00,     0.0000061580798140},
  {     5.678510696756586, 6.269971661671169e-01,     0.0000046797140778},
};

static const struct L1Term l1_callisto_2[45] = {
  {     5.583607157693351, 3.206509914393900e-05,     0.0073755808467977},
  {     5.267332440148252, 1.272744128508833e-04,     0.0001561131605348},
  {     5.539035084599682, 2.868340822827979e-03,     0.0000540660842731},
  {     5.560671996478980, 1.450183749077938e-03,     0.0000090411191759},
  {     4.538619202078515, 4.319183203226879e-03,     0.0000060406575087},
  {     2.418533859388985,-3.158488613807020e-05,     0.0000028551622596},
  {     3.401291481398445, 1.418202555329188e-03,     0.0000026588026505},
  {     4.578953439281555, 1.482242915300486e-03,     0.0000016317841395},
  {     3.443165191194147,-1.418028444788184e-03,     0.0000016159095087},
  {     5.855368141690349, 2.773137309294145e-03,     0.0000012029942283},
  {     5.014742085348139, 6.835386423102111e-04,     0.0000008869382117},
  {     4.496366437229220, 2.902332511122542e-03,     0.0000004518928658},
  {     5.306114446513249, 5.884547448229826e-04,     0.0000004739086661},
  {     3.537704696769715, 5.768534027128792e-03,     0.0000005321318002},
  {     5.920983156515586, 3.764862419470324e-01,     0.0002065924169942},
  {     0.287440062412000, 8.782079244251990e-01,     0.0001589869764021},
  {     1.446213430046000, 3.551552294980162e+00,     0.0001486043380971},
  {     5.909680328578586, 1.769322712928523e+00,     0.0000635073108731},
  {     4.112551758424586,-2.798579795458930e+00,     0.0000599351698525},
  {     1.480222294735000,-6.269571252951934e-01,     0.0000489596900866},
  {     5.206697523915587,-3.735860173449657e-01,     0.0000333682283528},
  {     5.270762340277586,-1.252354244860152e-01,     0.0000292325461337},
  {     5.932269789699586,-1.016350227520926e+00,     0.0000295832427279},
  {     0.830451671341000,-1.128678804105747e+00,     0.0000183551029746},
  {     0.180738718447000,-1.630400483203856e+00,     0.0000081987970452},
  {     5.814175879052586,-2.132122165476585e+00,     0.0000039403527376},
  {     4.199095666765587,-3.721359265672032e-01,     0.0000056895636122},
  {     1.559247963578000,-2.504960288928830e-01,     0.0000040434854859},
  {     0.352077722692000, 1.126558501852482e+00,     0.0000036901291978},
  {     5.141859545789586, 2.512311790884658e-01,     0.0000019322089806},
  {     5.164394621664586,-2.633843845433241e+00,     0.0000019711212463},
  {     4.078341574555586, 1.379929616304742e+00,     0.0000018673159813},
  {     4.728373702472586, 1.881651286424278e+00,     0.0000016695689644},
  {     2.801758550422000,-2.409186686503694e+00,     0.0000014034621874},
  {     0.916517243491000,-7.522178950452509e-01,     0.0000011758260607},
  {     6.079603342769587, 7.529452084377499e-01,     0.0000016838424078},
  {     5.378078785475586, 2.383372965022849e+00,     0.0000010798624964},
  {     4.514579857013587,-3.135565516587292e+00,     0.0000010108880552},
  {     1.911818107846000,-5.016928157568171e-01,     0.0000008876681807},
  {     3.077495182201000,-3.750361593421911e-01,     0.0000008194699011},
  {     3.191001606319586,-3.706858488172629e-01,     0.0000006728737059},
  {     1.359571973412000, 1.125108413556394e+00,     0.0000006297345982},
  {     5.140216129902586, 7.116009548308659e+00,     0.0000006128899757},
  {     4.411831264147587,-2.422124613167147e+00,     0.0000007093782158},
  {     0.270180657271000,-1.253939666677100e+00,     0.0000005580987049},
};

static const struct L1Term l1_callisto_3[17] = {
  {     5.972173677355479,-3.056125525186985e-05,     0.0022453891791894},
  {     0.195281976995034,-1.249130997158616e-04,     0.0002604479450559},
  {     5.560413774187417, 2.900376885067736e-03,     0.0000332112143230},
  {     0.284882297059426,-1.450057176193737e-03,     0.0000049727136261},
  {     4.189283499205679,-5.692029885714521e-04,     0.0000049416729114},
  {     1.768427374670601, 1.450134452467177e-03,     0.0000043945193428},
  {     4.556768052987046, 4.350464540704399e-03,     0.0000037630501589},
  {     5.151028719135752, 2.931305137674350e-03,     0.0000030823418750},
  {     0.686160228640311,-1.480873100162566e-03,     0.0000004637177865},
  {     1.805541761860624, 1.419544543202132e-03,     0.0000004719790711},
  {     4.153668446290737, 4.381612682288441e-03,     0.0000003467132626},
  {     4.644436033056316, 3.025313016227793e-03,     0.0000003497224175},
  {     3.554939168620919, 5.800537903205751e-03,     0.0000003324412570},
  {     6.125168715140077, 2.880826487278455e-03,     0.0000001945374351},
  {     1.190077323740297,-2.900106852472988e-03,     0.0000001727743329},
  {     3.091990817069338, 1.480767909272152e-03,     0.0000001485176585},
  {     4.061622576182586,-3.272492347489368e-02,     0.0000000000666922},
};

static const struct L1Body l1_bodies[4] = {
  {
    "io",2.82489428433814e-07, 3.551552286182400,
    {
       4.13349410167037e-05,
      -6.67905577476840e-08,
      -2.23775040402638e-08,
      -1.45617464908851e-07,
      -1.58558513521631e-07,
      -5.46986318473484e-06,
      -7.58579668623469e-09,
      -5.78177122641661e-10,
      -1.62905002759586e-07,
      -1.26321744097649e-07,
       5.94965634346142e-05,
      -1.73800453537518e-09,
      -5.35142525086474e-09,
      -3.68431020234090e-09,
       6.51758518535435e-09,
      -1.10076009434021e-05,
      -3.02381800276162e-09,
       1.02942385189608e-10,
       7.77164290527916e-10,
       2.62070675646012e-09,
      -5.75048718532862e-06,
      -7.21047445746571e-09,
      -8.49040607452073e-09,
       1.59932283427177e-09,
       2.59238370832140e-09,
       5.74994024502453e-06,
      -3.92259073275152e-10,
      -2.52840670456370e-09,
      -2.28491296590504e-09,
      -7.26318061740534e-10,
       1.78506935927118e-06,
      -4.16879673796892e-09,
      -6.13405545173217e-09,
       7.03033944342130e-10,
      -1.18191642174809e-09,
      -2.40677412003469e-06,
      -3.29083173926104e-09,
      -3.43800069555589e-09,
      -2.89289190860300e-11,
       5.90816038543625e-10,
      -1.38513209122323e-06,
       2.88537255009037e-09,
      -3.59984191898440e-09,
       4.86451562749639e-10,
       1.23053999757825e-09,
    },
    {
           0.0028210960212903,
           1.4462132960212235,
           0.0000000000000000,
           0.0000000000000000,
          -0.0000131331625731,
           0.0000116708509463,
    },
    {
      {l1_io_0,40},
      {l1_io_1,46},
      {l1_io_2,23},
      {l1_io_3,14}
    }
  },
  {
    "europa",2.82483274392893e-07, 1.769322711123470,
    {
      -1.17345718523364e-04,
      -5.01776992634625e-07,
      -1.30512480107910e-07,
      -1.37352383697128e-06,
      -1.49207848327093e-06,
       2.59036121717763e-05,
      -5.80438343691240e-08,
      -1.82656275716032e-07,
      -1.54365304913777e-06,
      -1.19708851225481e-06,
      -1.73835841495123e-04,
       3.80875695145265e-08,
       2.78016374834476e-08,
      -3.49546762061882e-08,
       5.67146658521585e-08,
       3.80673357683932e-05,
      -2.10657418625019e-07,
      -1.58858363289506e-07,
       2.86789414864130e-09,
       2.68814727457189e-08,
      -1.03485888270120e-07,
       3.12238251072476e-07,
       2.31850746815330e-08,
       1.75452766852069e-08,
       2.61897719058852e-08,
       1.17376776995138e-05,
      -1.72444072203130e-07,
      -1.10691155968765e-07,
      -1.97744662243578e-08,
      -5.11649608698392e-09,
       3.19509608989603e-06,
       2.35985832098386e-07,
      -3.69048459423356e-08,
       3.43169676541053e-09,
      -1.27841777400320e-08,
      -4.83167496026174e-06,
       6.13727340553035e-08,
       6.78631069587260e-09,
      -1.30924950693160e-10,
       1.19534263567129e-08,
       4.47094431509368e-07,
      -1.58131098756712e-07,
       1.90777145516695e-08,
       1.35338072425105e-08,
       1.15749543781328e-08,
    },
    {
           0.0044871037804314,
          -0.3735263437471362,
           0.0000000000000000,
           0.0000000000000000,
          -0.0001240982981784,
           0.0001106352686586,
    },
    {
      {l1_europa_0,37},
      {l1_europa_1,46},
      {l1_europa_2,41},
      {l1_europa_3,24}
    }
  },
  {
    "ganymede",2.82498184184723e-07, 0.878207923589328,
    {
      -2.01249945111849e-04,
      -5.58987296395410e-08,
      -5.35221008486725e-08,
      -7.09841479811065e-06,
      -7.69880546608065e-06,
       4.22870514937427e-05,
      -2.78967292638245e-07,
      -2.23668695371038e-07,
      -7.93995493982020e-06,
      -6.12022141299496e-06,
      -2.88258753042082e-04,
      -7.04645336679784e-08,
      -2.18527598721759e-07,
      -2.08882224662785e-07,
       3.15217558237908e-07,
       6.23338370193608e-05,
      -2.38693040681653e-07,
      -2.00040954734845e-07,
       2.09762683478441e-08,
       1.48179418616661e-07,
       1.78692873660163e-06,
       1.22869992713930e-08,
      -3.19586844323525e-07,
       8.35070484335148e-08,
       1.41150094231365e-07,
       1.44147643545141e-05,
      -1.53370399783488e-07,
      -2.22086683443490e-07,
      -1.07729670354460e-07,
      -2.24063936978287e-08,
       6.58944353306297e-06,
       4.47318789843126e-08,
      -2.98609138042077e-07,
       3.06301230195125e-08,
      -3.33311708439601e-08,
      -6.19139500170464e-06,
      -1.65802124252890e-08,
      -1.29217448782639e-07,
       1.12449433356374e-08,
       1.07863442857316e-08,
       7.82954615405540e-07,
      -8.28946056921499e-08,
      -1.50654590880644e-07,
       7.08786417409988e-09,
       6.62499119729457e-08,
    },
    {
           0.0071566594572575,
           0.2874089391143348,
          -0.0000023025209427,
          -0.0000007355571437,
          -0.0006368849625749,
           0.0005679034494627,
    },
    {
      {l1_ganymede_0,37},
      {l1_ganymede_1,40},
      {l1_ganymede_2,49},
      {l1_ganymede_3,17}
    }
  },
  {
    "callisto",2.82492144889909e-07, 0.376486233433828,
    {
       1.53102173532307e-04,
      -9.06550129921080e-08,
      -7.03913087149460e-07,
      -3.20468327586225e-05,
      -3.48034717955846e-05,
      -1.68321429486551e-04,
      -1.26014618556329e-06,
      -1.35394137498700e-06,
      -3.57948862513955e-05,
      -2.76636397672750e-05,
       2.02182016979082e-04,
      -3.82306332383069e-07,
      -3.20084097458594e-07,
      -8.87261624128964e-07,
       1.38328020437413e-06,
      -1.37641833741129e-04,
      -6.67102945644653e-07,
      -6.20618627039507e-07,
       1.64506004407617e-07,
       6.51631405812798e-07,
      -5.30404402536766e-05,
      -5.05214970641581e-08,
      -6.11742588608485e-07,
       4.77415826267116e-07,
       6.20307849865976e-07,
      -1.10474926063654e-05,
       1.53918794369926e-07,
      -2.76537195728238e-07,
      -3.59030239515192e-07,
      -2.66081185506962e-07,
      -1.05634674903822e-05,
       9.28279781805284e-08,
      -1.33810977431205e-07,
       1.17406290303369e-07,
      -2.09003718083271e-07,
       3.11526543050599e-05,
       3.66048038576706e-07,
       2.89677483700145e-07,
       8.95051737231057e-08,
      -1.46372661194868e-07,
       4.66495863544146e-06,
       5.68038769950531e-08,
       4.70713328715423e-08,
       9.71875658565188e-09,
      -1.22390974721364e-07,
    },
    {
           0.0125879701715314,
          -0.3620341291375704,
          -0.0000194025740559,
          -0.0000037352051335,
          -0.0028677883416141,
           0.0025571551250530,
    },
    {
      {l1_callisto_0,23},
      {l1_callisto_1,27},
      {l1_callisto_2,45},
      {l1_callisto_3,17}
    }
  },
};

static void CalcL1Elem(const double t,const int body,double elem[6]) {
  int j;
  const struct L1Body *const bp = l1_bodies + body;
  const double *cheb = bp->cheb_coef;
  double use_polynomials;

  memcpy(elem,bp->constants,6*sizeof(double));

  for (j=0;j<2;j++) {
    const struct L1Term *const begin = bp->lists[j].terms;
    const struct L1Term *p = begin + bp->lists[j].size;
    while (--p >= begin) {
      const double d = p->phase + t*p->frequency;
      elem[j] += p->amplitude * cos(d);
    }
  }
  for (j=2;j<4;j++) {
    const struct L1Term *const begin = bp->lists[j].terms;
    const struct L1Term *p = begin + bp->lists[j].size;
    while (--p >= begin) {
      const double d = p->phase + t*p->frequency;
      elem[2*j-2] += p->amplitude * cos(d);
      elem[2*j-1] += p->amplitude * sin(d);
    }
  }

  for (j=0;j<5;j++) {
    elem[j] += (*cheb++);
  }
  
    /* This is an artificial hack. Valery Lainey recommends
       not using polynoials outside of [-700,700].
       I do the following:
       inside  [-695,695] use polynomials
       outside [-705,705] do not use polynomials
       inbetween avoid jumping moons by partly using polynomials */
  use_polynomials = (705*365.25 - fabs(t)) / (10*365.25);
  if (use_polynomials > 0) {
    int i;
    const double a = -819.727638594856;
    const double b = 812.72180699036;
    const double x = (t / (0.5*365.25) - (b + a)) / (b - a);
    double ti2 = 1.0;
    double ti1 = x;
    if (use_polynomials >= 1.0) use_polynomials = 1.0;
    for (j=0;j<5;j++) {
      elem[j] += use_polynomials * ((*cheb++) * x);
    }
    for (i=2;i<9;i++) {
      const double ti = 2.0 * x * ti1 - ti2;
      ti2 = ti1;
      ti1 = ti;
      for (j=0;j<5;j++) {
        elem[j] += use_polynomials * ((*cheb++) * ti);
      }
    }
  }

  elem[1] += t*bp->l;
}


/*
const double L1toJ2000[9] = {
   9.994327653386544723e-01, 3.039594289062820484e-02,-1.449945596633516053e-02,
  -3.367710746976384242e-02, 9.020579123528089974e-01,-4.302991694091006926e-01,
   0.000000000000000000e+00, 4.305433885422950373e-01, 9.025698812737539884e-01
};
*/

const double L1toVsop87[9] = {
   9.994327815023905713e-01, 3.039550993390781261e-02,-1.449924943755843383e-02,
  -3.089770442223671880e-02, 9.988822846893227815e-01,-3.577028369016394015e-02,
   1.339578739122566807e-02, 3.619798764705610479e-02, 9.992548516622136737e-01
};


static double t_0[4] = {-1e100,-1e100,-1e100,-1e100};
static double t_1[4] = {-1e100,-1e100,-1e100,-1e100};
static double t_2[4] = {-1e100,-1e100,-1e100,-1e100};
static double l1_elem_0[4*6];
static double l1_elem_1[4*6];
static double l1_elem_2[4*6];

/* 1 day: */
#define DELTA_T 1.0

static double l1_jd0[4] = {-1e100,-1e100,-1e100,-1e100};
static double l1_elem[4*6];

static int ugly_static_parameter_body = -1;
static void CalcUglyStaticL1Elem(double t,double elem[6]) {
  CalcL1Elem(t,ugly_static_parameter_body,elem);
}

void GetL1Coor(double jd,int body,double *xyz) {
  GetL1OsculatingCoor(jd,jd,body,xyz);
}

void GetL1OsculatingCoor(const double jd0,const double jd,
                         const int body,double *xyz) {
  double x[3];
  if (jd0 != l1_jd0[body]) {
    const double t0 = jd0 - 2433282.5;
    l1_jd0[body] = jd0;
    ugly_static_parameter_body = body;
    CalcInterpolatedElements(t0,l1_elem+(body*6),6,
                             &CalcUglyStaticL1Elem,DELTA_T,
                             t_0+body,l1_elem_0+(body*6),
                             t_1+body,l1_elem_1+(body*6),
                             t_2+body,l1_elem_2+(body*6));
  }
  EllipticToRectangularA(l1_bodies[body].mu,l1_elem+(body*6),jd-jd0,x);
  xyz[0] = L1toVsop87[0]*x[0]+L1toVsop87[1]*x[1]+L1toVsop87[2]*x[2];
  xyz[1] = L1toVsop87[3]*x[0]+L1toVsop87[4]*x[1]+L1toVsop87[5]*x[2];
  xyz[2] = L1toVsop87[6]*x[0]+L1toVsop87[7]*x[1]+L1toVsop87[8]*x[2];
}
