/*
 * Copyright (C) 2013-2018 Canonical, Ltd.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * This code is a complete clean re-write of the stress tool by
 * Colin Ian King <colin.king@canonical.com> and attempts to be
 * backwardly compatible with the stress tool by Amos Waterland
 * <apw@rossby.metr.ou.edu> but has more stress tests and more
 * functionality.
 *
 */
#include "stress-ng.h"

#if defined(HAVE_LIB_PTHREAD) && defined(__linux__)

#define PROC_BUF_SZ		(4096)
#define MAX_READ_THREADS	(4)

typedef struct ctxt {
	const args_t *args;
	const char *path;
	char *badbuf;
	bool writeable;
} ctxt_t;

static sigset_t set;
static pthread_spinlock_t lock;
static char *proc_path;

/*
 *  stress_proc_rw()
 *	read a proc file
 */
static inline void stress_proc_rw(
	const ctxt_t *ctxt,
	int32_t loops)
{
	int fd;
	ssize_t ret, i = 0;
	char buffer[PROC_BUF_SZ];
	char *path;
	const double threshold = 0.2;
	off_t pos;

	while (loops == -1 || loops > 0) {
		double t_start;
		bool timeout = false;


		ret = pthread_spin_lock(&lock);
		if (ret)
			return;
		path = (char *)proc_path;
		(void)pthread_spin_unlock(&lock);

		if (!path || !g_keep_stressing_flag)
			break;

		t_start = time_now();

		if ((fd = open(path, O_RDONLY | O_NONBLOCK)) < 0)
			return;

		if (time_now() - t_start > threshold) {
			timeout = true;
			(void)close(fd);
			goto next;
		}

		/*
		 *  Multiple randomly sized reads
		 */
		while (i < (4096 * PROC_BUF_SZ)) {
			ssize_t sz = 1 + (mwc32() % sizeof(buffer));
			if (!g_keep_stressing_flag)
				break;
			ret = read(fd, buffer, sz);
			if (ret < 0)
				break;
			if (ret < sz)
				break;
			i += sz;

			if (time_now() - t_start > threshold) {
				timeout = true;
				(void)close(fd);
				goto next;
			}
		}
		(void)close(fd);

		if ((fd = open(path, O_RDONLY | O_NONBLOCK)) < 0)
			return;

		if (time_now() - t_start > threshold) {
			timeout = true;
			(void)close(fd);
			goto next;
		}
		/*
		 *  Zero sized reads
		 */
		ret = read(fd, buffer, 0);
		if (ret < 0)
			goto err;
		/*
		 *  Bad read buffer, should fail!
		 */
		if (ctxt->badbuf) {
			ret = read(fd, ctxt->badbuf, PROC_BUF_SZ);
			if (ret == 0)
				goto err;
		}

		if (time_now() - t_start > threshold) {
			timeout = true;
			(void)close(fd);
			goto next;
		}

		/*
		 *  Seek and read
		 */
		pos = lseek(fd, 0, SEEK_SET);
		if (pos == (off_t)-1)
			goto err;
		pos = lseek(fd, 1, SEEK_CUR);
		if (pos == (off_t)-1)
			goto err;
		pos = lseek(fd, 0, SEEK_END);
		if (pos == (off_t)-1)
			goto err;
		pos = lseek(fd, 1, SEEK_SET);
		if (pos == (off_t)-1)
			goto err;

		if (time_now() - t_start > threshold) {
			timeout = true;
			(void)close(fd);
			goto next;
		}

		ret = read(fd, buffer, 1);
		(void)ret;
err:
		(void)close(fd);
		if (time_now() - t_start > threshold) {
			timeout = true;
			goto next;
		}

		if (ctxt->writeable) {
			/*
			 *  Zero sized writes
			 */
			if ((fd = open(path, O_WRONLY | O_NONBLOCK)) < 0)
				return;
			ret = write(fd, buffer, 0);
			(void)ret;
			(void)close(fd);

			/*
			 *  Write using badbuf, expect it to fail
			 */
			if (ctxt->badbuf) {
				if ((fd = open(path, O_WRONLY | O_NONBLOCK)) < 0)
					return;
				ret = write(fd, ctxt->badbuf, PROC_BUF_SZ);
				(void)ret;
				(void)close(fd);
			}
		}
next:
		if (loops > 0) {
			if (timeout)
				break;
			loops--;
		}
	}
}

/*
 *  stress_proc_rw_thread
 *	keep exercising a procfs entry until
 *	controlling thread triggers an exit
 */
static void *stress_proc_rw_thread(void *ctxt_ptr)
{
	static void *nowt = NULL;
	uint8_t stack[SIGSTKSZ + STACK_ALIGNMENT];
	ctxt_t *ctxt = (ctxt_t *)ctxt_ptr;

	/*
	 *  Block all signals, let controlling thread
	 *  handle these
	 */
	(void)sigprocmask(SIG_BLOCK, &set, NULL);

	/*
	 *  According to POSIX.1 a thread should have
	 *  a distinct alternative signal stack.
	 *  However, we block signals in this thread
	 *  so this is probably just totally unncessary.
	 */
	(void)memset(stack, 0, sizeof(stack));
	if (stress_sigaltstack(stack, SIGSTKSZ) < 0)
		return &nowt;

	while (g_keep_stressing_flag)
		stress_proc_rw(ctxt, -1);

	return &nowt;
}

/*
 *  stress_proc_dir()
 *	read directory
 */
static void stress_proc_dir(
	const ctxt_t *ctxt,
	const char *path,
	const bool recurse,
	const int depth)
{
	DIR *dp;
	struct dirent *d;
	const args_t *args = ctxt->args;

	if (!g_keep_stressing_flag)
		return;

	/* Don't want to go too deep */
	if (depth > 20)
		return;

	dp = opendir(path);
	if (dp == NULL)
		return;

	while ((d = readdir(dp)) != NULL) {
		int ret;
		char filename[PATH_MAX];
		char tmp[PATH_MAX];

		if (!g_keep_stressing_flag)
			break;
		if (is_dot_filename(d->d_name))
			continue;

		(void)snprintf(tmp, sizeof(tmp), "%s/%s", path, d->d_name);
		switch (d->d_type) {
		case DT_DIR:
			if (!recurse)
				continue;

			inc_counter(args);
			stress_proc_dir(ctxt, tmp, recurse, depth + 1);
			break;
		case DT_REG:
			ret = pthread_spin_lock(&lock);
			if (!ret) {
				strncpy(filename, tmp, sizeof(filename));
				proc_path = filename;
				(void)pthread_spin_unlock(&lock);
				stress_proc_rw(ctxt, 8);
				inc_counter(args);
			}
			break;
		default:
			break;
		}
	}
	(void)closedir(dp);
}

/*
 *  stress_procfs
 *	stress reading all of /proc
 */
int stress_procfs(const args_t *args)
{
	size_t i;
	pthread_t pthreads[MAX_READ_THREADS];
	int rc, ret[MAX_READ_THREADS];
	ctxt_t ctxt;

	(void)sigfillset(&set);

	proc_path = "/proc/self";

	ctxt.args = args;
	ctxt.writeable = (geteuid() != 0);

	rc = pthread_spin_init(&lock, PTHREAD_PROCESS_PRIVATE);
	if (rc) {
		pr_inf("%s: pthread_spin_init failed, errno=%d (%s)\n",
			args->name, rc, strerror(rc));
		return EXIT_NO_RESOURCE;
	}

	(void)memset(ret, 0, sizeof(ret));

	ctxt.badbuf = mmap(NULL, PROC_BUF_SZ, PROT_READ,
			MAP_SHARED | MAP_ANONYMOUS, -1, 0);
	if (ctxt.badbuf == MAP_FAILED) {
		pr_inf("%s: mmap failed: errno=%d (%s)\n",
			args->name, errno, strerror(errno));
		return EXIT_NO_RESOURCE;
	}

	for (i = 0; i < MAX_READ_THREADS; i++) {
		ret[i] = pthread_create(&pthreads[i], NULL,
				stress_proc_rw_thread, &ctxt);
	}

	do {
		stress_proc_dir(&ctxt, "/proc", false, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/self", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/sys", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/sysvipc", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/fs", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/bus", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/irq", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/scsi", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/tty", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/driver", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/tty", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/self", true, 0);
		inc_counter(args);
		if (!keep_stressing())
			break;

		stress_proc_dir(&ctxt, "/proc/thread_self", true, 0);
		inc_counter(args);
	} while (keep_stressing());

	proc_path = NULL;

	for (i = 0; i < MAX_READ_THREADS; i++) {
		if (ret[i] == 0)
			pthread_join(pthreads[i], NULL);
	}
	(void)munmap(ctxt.badbuf, PROC_BUF_SZ);
	(void)pthread_spin_destroy(&lock);

	return EXIT_SUCCESS;
}
#else
int stress_procfs(const args_t *args)
{
	return stress_not_implemented(args);
}
#endif
