// Copyright (C) 2009 Martin Sandve Alnes
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program solves a 1D nonlinear hyperelasticity model.

#include <iostream>
#include <dolfin.h>
#include <cmath>
#include "generated_code/HyperElasticity1D.h"

using namespace dolfin;
using namespace HyperElasticity1D;


const double eps = DOLFIN_EPS;
const double K_VALUE = 1.0; 
const double B_VALUE = 0.1;

class Weight: public Expression
{
public:
  double value; 
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = value;  
  }
};

class Source: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    const double dx = x[0] - 0.5;
    const double b = B_VALUE;
    const double K = K_VALUE;
    // Matching solution u = sin(x) - x:
    values[0] = -K*exp((1.0/4.0)*std::pow( std::pow(cos(x[0]),2.0)-1.0,2.0)*b)*( std::pow(cos(x[0]),2.0)-1.0)*sin(x[0])*b+-2.0*K*exp((1.0/4.0)*std::pow( std::pow(cos(x[0]),2.0)-1.0,2.0)*b)*std::pow(cos(x[0]),2.0)*sin(x[0])*b-K*exp((1.0/4.0)*std::pow( std::pow(cos(x[0]),2.0)-1.0,2.0)*b)*std::pow(cos(x[0]),2.0)*std::pow( std::pow(cos(x[0]),2.0)-1.0,2.0)*sin(x[0])*(b*b);
  }
};

class Solution: public Expression 
{
public:
  void eval(Array<double>& values, const Array<double>& x) const
  {
    values[0] = sin(x[0]) - x[0];
  }
};

class DirichletBoundary: public SubDomain
{
  bool inside(const Array<double>& x, bool on_boundary) const
  {
    return x[0] < DOLFIN_EPS || x[0] > 1.0-DOLFIN_EPS;
  }
};

int main()
{
    // Geometry
    unsigned n = 20;
    UnitInterval mesh(n);
    
    // Function spaces
    Form_J::TrialSpace V(mesh);
    CoefficientSpace_K C(mesh);
    
    // Coefficient functions
    Weight K;
    K.value = K_VALUE;
    Weight b;
    b.value = B_VALUE;
    Source g;
    Solution usol;
    
    // Solution function
    Function u(V);
    
    // Forms
    Form_J  a(V, V);
    Form_F  L(V);
    
    // Attach coefficients to forms
    a.u = u;
    a.K = K;
    a.b = b;
    L.u = u;
    L.K = K;
    L.b = b;
    L.g = g;
    
    // Boundary conditions
    DirichletBoundary boundary;
    DirichletBC bc(V, usol, boundary);

    // Solve variational problem
    solve(a==L, u, bc); 

    // Compute F at u
    Vector F;
    assemble(F, L);
    //problem.F(F, u.vector());
    cout << "Norm of F = " << F.norm("l2") << endl;
    cout << "Min  of F = " << F.min() << endl;
    cout << "Max  of F = " << F.max() << endl;
    
    // Compute e_h = usol_h - e
    Function usol_h(V);
    usol_h = usol; 
    Function e_h(V);
    e_h.vector() = usol_h.vector();
    e_h.vector() -= u.vector();
    cout << "Norm of e_h = " << e_h.vector().norm("l2") << endl;
    cout << "Min  of e_h = " << e_h.vector().min() << endl;
    cout << "Max  of e_h = " << e_h.vector().max() << endl;
    
    // Write solutions to file
    File ufile("u.pvd");
    ufile << u;
    File usolfile("usol.pvd");
    usolfile << usol_h;
    File efile("e.pvd");
    efile << e_h;
    
    //plot(u);
    //plot(usol_h);
}
