
from sfc import *

def define_linear_elasticity(itg):
    I     = Id(itg.nsd)
    GinvT = itg.GinvT()

    # get coefficients
    lambd, mu = itg.coefficients()

    # for all trial functions u
    for j, u in enumerate(itg.v_basis(1)):
        Du = grad(u, GinvT)
        DuT = Du.transpose()
        E = (Du + DuT) / 2
        sigma = lambd * Du * I + 2*mu*E

        # for all test functions v
        for i, v in enumerate(itg.v_basis(0)):
            Dv = grad(v, GinvT)
            integrand = inner(sigma, Dv)
            itg.set_A((i, j), integrand)

def define_linear_elasticity_traction(itg):
    I     = Id(itg.nsd)
    GinvT = itg.GinvT()

    # get coefficients
    lambd, mu, t = itg.coefficients()

    # for all test functions v
    for i, v in enumerate(itg.v_basis(0)):
        integrand = inner(t, v)
        itg.set_A(i, integrand)

def define_linear_elasticity_pressure(itg):
    I     = Id(itg.nsd)
    GinvT = itg.GinvT()
    n     = itg.n()

    # get coefficients
    lambd, mu, p = itg.coefficients()

    # for all test functions v
    for i, v in enumerate(itg.v_basis(0)):
        integrand = inner(p*n, v)
        itg.set_A(i, integrand)

polygon = "tetrahedron"
vfe1 = VectorElement("CG", polygon, 1)

v = TestFunction(vfe1)
u = TrialFunction(vfe1)

lambd = Constant(polygon, "lambd")
mu    = Constant(polygon, "mu")
p     = Constant(polygon, "p")
t     = Function(vfe1, "t")

# the momentum form object
a = Form(name = "elasticity",
         basisfunctions = [v, u],
         coefficients = [lambd, mu])
itg = a.add_cell_integral()
define_linear_elasticity(itg)

# the traction form object
b = Form(name = "elasticity_traction",
         basisfunctions = [v],
         coefficients = [lambd, mu, t])
itg = b.add_exterior_facet_integral()
define_linear_elasticity_traction(itg)

# the pressure form object
c = Form(name = "elasticity_pressure",
         basisfunctions = [v],
         coefficients = [lambd, mu, p])
itg = c.add_exterior_facet_integral()
define_linear_elasticity_pressure(itg)


# generate code and compile extension module
a_compiled = compile_form(a)
b_compiled = compile_form(b)
c_compiled = compile_form(c)

# assemble, solve and plot with PyDOLFIN
from dolfin import *
n = 10
mesh = UnitCube(n, n, n)

lambd = Function(mesh, 0.3)
mu    = Function(mesh, 0.3)

class Pressure(cpp_Function):
    def __init__(self, mesh):
        cpp_Function.__init__(self, mesh)
    def eval(self, v, x):
        v[0] = 10000.0*x[0]
    def rank(self):
        return 0

p = Pressure(mesh)
#t = Function(mesh, (1.0, 1.0, 1.0))

A, dms = assemble(a_compiled, mesh, coefficients=[lambd, mu], return_dofmaps = True)
b = assemble(c_compiled, mesh, coefficients=[lambd, mu, p])
#b = assemble(b_compiled, mesh, coefficients=[lambd, mu, t])

uvec = Vector()
u = cpp_Function(mesh, uvec, dms.sub(1), a_compiled, 1)
solve(A, u.vector(), b)

plot(u)

