#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module handles finding quadrature rules based on certain properties.
"""

# Copyright (C) 2008 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-08-12
# Last changed: 2008-08-12

from sfc.quadrature.QuadRule import QuadRule
from sfc.quadrature.quad_tables import load_rule
from ufl.geometry import domain2dim

def find_quadrature_rule(polygon, order, family="gauss"):
    if polygon == "point":
        return QuadRule(polygon, 1, 0, [(0.0,)], [1.0], "Single point evaluation rule.")

    if family == "gauss":
        if polygon in ("triangle", "tetrahedron"):
            family = "economical_gauss"
        else:
            family = "composite_gauss"

    if family == "economical_gauss":
        return load_rule(polygon, order, family)

    if family == "composite_gauss":

        rule = load_rule("interval", order, "gauss")
        #print "rule = ", rule
        
        nsd = domain2dim[polygon]
        if nsd == 1:
            return rule

        # construct composite rule from 1D formula
        assert polygon in ("quadrilateral", "hexahedron") 
        points  = []
        weights = []
        rr = range(rule.num_points)
        if nsd == 2:
            for i in rr:
                for j in rr:
                    points.append( (rule.points[i][0], rule.points[j][0]) )
                    weights.append( rule.weights[i]*rule.weights[j] )
        elif nsd == 3:
            for i in rr:
                for j in rr:
                    for k in rr:
                        points.append( (rule.points[i][0], rule.points[j][0], rule.points[k][0]) )
                        weights.append( rule.weights[i]*rule.weights[j]*rule.weights[k] )
        comment = "Computed composite_gauss rule on %s of order %d." % (polygon, order)
        return QuadRule(polygon, nsd, order, points, weights, comment)

    raise RuntimeError("Found no rule for a %s with order %d in family %s" % (polygon, order, family))


if __name__ == '__main__':
    print "No test here."

